import struct
import os
import sys

try:
    import cPickle as pickle
except ImportError:
    import pickle


class TransferFile(object):
    """Object used to transfer a file between Communicators"""
    def __init__(self, filename):
        self._filename = filename


class HeartBeat:
    pass


class DataType:
    desc = "generic data"

    def recv(self, buffer):
        pass

    def send(self, obj):
        pass

    def __str__(self):
        return self.desc


if sys.version_info[0] >= 3:
    def _raw_byte_code(s):
        # Cannot simply use b'foo' since that is invalid syntax in Python 2.3
        return bytes(s, encoding='ascii')
else:
    def _raw_byte_code(s):
        return s


# Number of bytes used to pack the length of a string
_slen_size = struct.calcsize('!i')


def _pack_string(s):
    return struct.pack('!i', len(s)) + s


def _unpack_string(buffer):
    slen, = struct.unpack_from('!i', buffer, 1)
    if 1 + _slen_size + slen > len(buffer):
        raise EOFError()
    return buffer[1 + _slen_size: 1 + _slen_size + slen], _slen_size + slen


if sys.version_info[0] >= 3:
    # In Python 3, strings are Unicode, so they must be encoded as bytes
    # to travel across the network
    class NetString(DataType):
        code = _raw_byte_code('S')
        obj = str
        desc = "string"

        def recv(self, buffer):
            obj, sz = _unpack_string(buffer)
            return (obj.decode('UTF8'), buffer[1+sz:])

        def send(self, obj):
            return self.code + _pack_string(obj.encode('UTF8'))
else:
    class NetString(DataType):
        code = _raw_byte_code('S')
        obj = str
        desc = "string"

        def recv(self, buffer):
            obj, sz = _unpack_string(buffer)
            return (obj, buffer[1+sz:])

        def send(self, obj):
            return self.code + _pack_string(obj)


class NetInteger(DataType):
    code = _raw_byte_code('I')
    obj = int
    desc = "integer"
    _calcsize = struct.calcsize('!i')

    def recv(self, buffer):
        obj, = struct.unpack_from('!i', buffer, 1)
        return (obj, buffer[1+self._calcsize:])

    def send(self, obj):
        return self.code + struct.pack('!i', obj)


class NetFloat(DataType):
    code = _raw_byte_code('F')
    obj = float
    desc = "floating point"
    _calcsize = struct.calcsize('!f')

    def recv(self, buffer):
        obj, = struct.unpack_from('!f', buffer, 1)
        return (obj, buffer[1+self._calcsize:])

    def send(self, obj):
        return self.code + struct.pack('!f', obj)


class NetPickle(DataType):
    code = _raw_byte_code('P')
    obj = None

    desc = "Python pickled object"

    def recv(self, buffer):
        obj, sz = _unpack_string(buffer)
        return (pickle.loads(obj), buffer[1+sz:])

    def send(self, obj):
        try:
            return self.code + _pack_string(pickle.dumps(obj, -1))
        # Python < 2.5 can fail trying to send Inf or NaN floats in binary
        # mode, so fall back to the old protocol in this case:
        except SystemError:
            return self.code + _pack_string(pickle.dumps(obj, 0))


class NetFile(DataType):
    code = _raw_byte_code('f')
    obj = TransferFile
    desc = "Transferred file"

    def recv(self, buffer):
        filename, buffer = NetString.recv(buffer[1:])
        filelen, buffer = NetInteger.recv(buffer)
        if len(buffer) < filelen:
            raise IndexError("File not completely read")
        filename = os.path.basename(filename)
        f = open(filename, 'wb')
        f.write(buffer[:filelen])
        return TransferFile(filename), buffer[filelen:]

    def send(self, obj):
        filename = obj._filename
        f = open(filename, 'rb')
        buf = f.read()
        return (self.code + NetString.send(filename)
                + NetInteger.send(len(buf)) + buf)


class NetCommand(NetString):
    code = _raw_byte_code('C')
    desc = "command"


typemap = {}
cmdmap = {}
for name in dir():
    var = eval(name)
    try:
        if issubclass(var, DataType) and var is not DataType:
            exec("%s = %s()" % (name, name))
            var = eval(name)
            if sys.version_info[0] >= 3:
                # In Python 3, b'foo'[0] returns an int, not b'f'
                cmdmap[ord(var.code)] = var
            else:
                cmdmap[var.code] = var
            typemap[var.obj] = var
    except TypeError:
        pass
