"""Functions to assess a model or selection. Each should take a selection
   object, and return one of:
   1. A modeller.terms.QAMetric instance; or
   2. A tuple containing the score name and the score itself.
   The first is preferable as it provides more information which can be
   used to populate relevant mmCIF tables.
"""

import modeller.terms


class _GA341Score(modeller.terms.ModelQAMetric):
    metric_type = 'normalized score'
    name = 'GA341 score'
    description = 'GA341 score'


class _DOPEScore(modeller.terms.SelectionQAMetric):
    metric_type = 'energy'
    name = 'DOPE score'
    description = 'Discrete Optimized Protein Energy'


class _DOPEHRScore(modeller.terms.SelectionQAMetric):
    metric_type = 'energy'
    name = 'DOPE-HR score'
    description = 'Discrete Optimized Protein Energy, high resolution'


class _NormalizedDOPEScore(modeller.terms.ModelQAMetric):
    metric_type = 'zscore'
    name = 'Normalized DOPE score'
    description = 'Normalized DOPE score'


def GA341(atmsel):
    """Returns the GA341 score of the given model."""
    mdl = atmsel.get_model()
    return _GA341Score(mdl.assess_ga341())


def DOPE(atmsel):
    """Returns the DOPE score of the given model."""
    return _DOPEScore(atmsel.assess_dope())


def DOPEHR(atmsel):
    """Returns the DOPE-HR score of the given model."""
    return _DOPEHRScore(atmsel.assess_dopehr())


def normalized_dope(atmsel):
    """Returns the normalized DOPE score of the given model."""
    mdl = atmsel.get_model()
    return _NormalizedDOPEScore(mdl.assess_normalized_dope())
