// -*- c++ -*-

/*

 Copyright 2019-2020 Alain Dargelas

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */

/*
 * File:   ElaboratorListener.h
 * Author: alaindargelas
 *
 * Created on May 6, 2020, 10:03 PM
 */

#include <map>
#include <stack>
#include <iostream>

#include "headers/VpiListener.h"
#include "headers/clone_tree.h"

// Since there is a lot of implementation happening inside this
// header, we need to include this.
// TODO: move implementation in ElaboratorListener.cpp
#include "headers/Serializer.h"

namespace UHDM {

class ElaboratorListener : public VpiListener {
  friend function;
  friend gen_scope_array;
public:

  ElaboratorListener (Serializer* serializer, bool debug = false) : serializer_(serializer), debug_(debug) {}
  void uniquifyTypespec(bool uniquify) { uniquifyTypespec_ = uniquify; }
  bool uniquifyTypespec() { return uniquifyTypespec_; }
  bool isFunctionCall(const std::string& name, const expr* prefix);

  bool isTaskCall(const std::string& name, const expr* prefix);

  // Bind to a net in the current instance
  net* bindNet(const std::string& name);

  // Bind to a net or parameter in the current instance
  any* bindAny(const std::string& name);

  // Bind to a param in the current instance
  any* bindParam(const std::string& name);

  // Bind to a function or task in the current scope
  any* bindTaskFunc(const std::string& name, const class_var* prefix = nullptr);

  void scheduleTaskFuncBinding(tf_call* clone) { scheduledTfCallBinding_.push_back(clone); }
    
protected:
  typedef std::map<std::string, const BaseClass*> ComponentMap;

  void leaveDesign(const design* object, const BaseClass* parent, vpiHandle handle, vpiHandle parentHandle) {
    design* root = (design*) object;
    root->VpiElaborated(true);
  }

  void enterModule(const module* object, const BaseClass* parent,
                   vpiHandle handle, vpiHandle parentHandle) override {
    module* inst = (module*) object;
    bool topLevelModule         = object->VpiTopModule();
    const std::string& instName = object->VpiName();
    const std::string& defName  = object->VpiDefName();
    bool flatModule             = (instName == "") && ((object->VpiParent() == 0) ||
                                                       ((object->VpiParent() != 0) && (object->VpiParent()->VpiType() != vpiModule)));
                                  // false when it is a module in a hierachy tree
    if (debug_)
      std::cout << "Module: " << defName << " (" << instName << ") Flat:"  << flatModule << ", Top:" << topLevelModule << std::endl;

    if (flatModule) {
      // Flat list of module (unelaborated)
      flatComponentMap_.insert(std::make_pair(object->VpiDefName(), object));
    } else {
      // Hierachical module list (elaborated)
      inHierarchy_ = true;

      // Collect instance elaborated nets
      ComponentMap netMap;
      if (object->Nets()) {
        for (net* net : *object->Nets()) {
          netMap.insert(std::make_pair(net->VpiName(), net));
        }
      }

      // Collect instance parameters, defparams
      ComponentMap paramMap;
      if (object->Parameters()) {
        for (any* param : *object->Parameters()) {
          paramMap.insert(std::make_pair(param->VpiName(), param));
        }
      }
      if (object->Def_params()) {
        for (def_param* param : *object->Def_params()) {
          paramMap.insert(std::make_pair(param->VpiName(), param));
        }
      }
      if (object->Variables()) {
        for (variables* var : *object->Variables()) {
          paramMap.insert(std::make_pair(var->VpiName(), var));
        }
      }

      // Collect func and task declaration
      ComponentMap funcMap;
      if (object->Task_funcs()) {
        for (task_func* var : *object->Task_funcs()) {
          funcMap.insert(std::make_pair(var->VpiName(), var));
        }
      }

      // Check if Module instance has a definition, collect enums
      ComponentMap::iterator itrDef = flatComponentMap_.find(defName);
      if (itrDef != flatComponentMap_.end()) {
        const BaseClass* comp = (*itrDef).second;
        int compType = comp->VpiType();
        switch (compType) {
          case vpiModule: {
            module* defMod = (module*) comp;
            if (defMod->Typespecs()) {
              for (typespec* tps : *defMod->Typespecs()) {
                if (tps->UhdmType() == uhdmenum_typespec) {
                  enum_typespec* etps = (enum_typespec*)tps;
                  for (enum_const* econst : * etps->Enum_consts()) {
                    paramMap.insert(std::make_pair(econst->VpiName(), econst));
                  }
                }
              }
            }
          }
        }
      }

      // Push instance context on the stack
      instStack_.push_back(std::make_pair(object, std::make_tuple(netMap, paramMap, funcMap)));

      // Check if Module instance has a definition
      if (itrDef != flatComponentMap_.end()) {
        const BaseClass* comp = (*itrDef).second;
        int compType = comp->VpiType();
        switch (compType) {
        case vpiModule: {
          module* defMod = (module*) comp;

          if (auto obj = defMod->Instance_array()) {
            auto* stmt = obj->DeepClone(serializer_, this, defMod);
            stmt->VpiParent(inst);
            inst->Instance_array(stmt);
          }
          if (auto vec = defMod->Process()) {
            auto clone_vec = serializer_->MakeProcess_stmtVec();
            inst->Process(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Primitives()) {
            auto clone_vec = serializer_->MakePrimitiveVec();
            inst->Primitives(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Primitive_arrays()) {
            auto clone_vec = serializer_->MakePrimitive_arrayVec();
            inst->Primitive_arrays(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto obj = defMod->Global_clocking()) {
            auto* stmt = obj->DeepClone(serializer_, this, defMod);
            stmt->VpiParent(inst);
            inst->Global_clocking(stmt);
          }
          if (auto obj = defMod->Default_clocking()) {
            auto* stmt = obj->DeepClone(serializer_, this, defMod);
            stmt->VpiParent(inst);
            inst->Default_clocking(stmt);
          }
          if (auto obj = defMod->Module_array()) {
            auto* stmt = obj->DeepClone(serializer_, this, defMod);
            stmt->VpiParent(inst);
            inst->Module_array(stmt);
          }
          if (auto vec = defMod->Interfaces()) {
            auto clone_vec = serializer_->MakeInterfaceVec();
            inst->Interfaces(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Interface_arrays()) {
            auto clone_vec = serializer_->MakeInterface_arrayVec();
            inst->Interface_arrays(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = inst->Cont_assigns()) {
            auto clone_vec = serializer_->MakeCont_assignVec();
            inst->Cont_assigns(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
         }
          if (auto vec = defMod->Cont_assigns()) {
            if (inst->Cont_assigns() == nullptr) {
              auto clone_vec = serializer_->MakeCont_assignVec();
              inst->Cont_assigns(clone_vec);
            }
            auto clone_vec = inst->Cont_assigns();
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Modules()) {
            auto clone_vec = serializer_->MakeModuleVec();
            inst->Modules(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Module_arrays()) {
            auto clone_vec = serializer_->MakeModule_arrayVec();
            inst->Module_arrays(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Mod_paths()) {
            auto clone_vec = serializer_->MakeMod_pathVec();
            inst->Mod_paths(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Tchks()) {
            auto clone_vec = serializer_->MakeTchkVec();
            inst->Tchks(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Def_params()) {
            auto clone_vec = serializer_->MakeDef_paramVec();
            inst->Def_params(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Io_decls()) {
            auto clone_vec = serializer_->MakeIo_declVec();
            inst->Io_decls(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Alias_stmts()) {
            auto clone_vec = serializer_->MakeAlias_stmtVec();
            inst->Alias_stmts(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Clocking_blocks()) {
            auto clone_vec = serializer_->MakeClocking_blockVec();
            inst->Clocking_blocks(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = inst->Gen_scope_arrays()) {
            auto clone_vec = serializer_->MakeGen_scope_arrayVec();
            inst->Gen_scope_arrays(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
         }
          if (auto vec = defMod->Gen_scope_arrays()) {
            if (inst->Gen_scope_arrays() == nullptr) {
              auto clone_vec = serializer_->MakeGen_scope_arrayVec();
              inst->Gen_scope_arrays(clone_vec);
            }
            auto clone_vec = inst->Gen_scope_arrays();
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto obj = defMod->Expr_dist()) {
            auto* stmt = obj->DeepClone(serializer_, this, defMod);
            stmt->VpiParent(inst);
            inst->Expr_dist(stmt);
          }
          if (auto vec = defMod->Task_funcs()) {
            auto clone_vec = serializer_->MakeTask_funcVec();
            inst->Task_funcs(clone_vec);
            for (auto obj : *vec) {
              enterTask_func(obj, defMod, nullptr, nullptr);
              auto* tf = obj->DeepClone(serializer_, this, inst);
              ComponentMap& funcMap = std::get<2>((instStack_.at(instStack_.size()-2)).second);
              funcMap.insert(std::make_pair(tf->VpiName(), tf));
              leaveTask_func(obj, defMod, nullptr, nullptr);
              tf->VpiParent(inst);
              clone_vec->push_back(tf);
            }
          }
          if (auto obj = defMod->Instance()) {
            auto* stmt = obj->DeepClone(serializer_, this, defMod);
            stmt->VpiParent(inst);
            inst->Instance(stmt);
          }
          if (auto vec = defMod->Programs()) {
            auto clone_vec = serializer_->MakeProgramVec();
            inst->Programs(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Program_arrays()) {
            auto clone_vec = serializer_->MakeProgramVec();
            inst->Program_arrays(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Array_nets()) {
            auto clone_vec = serializer_->MakeArray_netVec();
            inst->Array_nets(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Spec_params()) {
            auto clone_vec = serializer_->MakeSpec_paramVec();
            inst->Spec_params(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Class_defns()) {
            auto clone_vec = serializer_->MakeClass_defnVec();
            inst->Class_defns(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto obj = defMod->Module()) {
            auto* stmt = obj->DeepClone(serializer_, this, defMod);
            stmt->VpiParent(inst);
            inst->Module(stmt);
          }
          if (auto vec = defMod->Assertions()) {
            auto clone_vec = serializer_->MakeAnyVec();
            inst->Assertions(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Concurrent_assertions()) {
            auto clone_vec = serializer_->MakeConcurrent_assertionsVec();
            inst->Concurrent_assertions(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Variables()) {
            auto clone_vec = serializer_->MakeVariablesVec();
            inst->Variables(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Scopes()) {
            auto clone_vec = serializer_->MakeScopeVec();
            inst->Scopes(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Typespecs()) {
            auto clone_vec = serializer_->MakeTypespecVec();
            inst->Typespecs(clone_vec);
            for (auto obj : *vec) {
              if (uniquifyTypespec()) {
                auto* stmt = obj->DeepClone(serializer_, this, inst);
                stmt->VpiParent(inst);
                clone_vec->push_back(stmt);
              } else {
                auto* stmt = obj;
                clone_vec->push_back(stmt);
              }
            }
          }
          if (auto vec = defMod->Property_decls()) {
            auto clone_vec = serializer_->MakeProperty_declVec();
            inst->Property_decls(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Sequence_decls()) {
            auto clone_vec = serializer_->MakeSequence_declVec();
            inst->Sequence_decls(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Named_events()) {
            auto clone_vec = serializer_->MakeNamed_eventVec();
            inst->Named_events(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Named_event_arrays()) {
            auto clone_vec = serializer_->MakeNamed_event_arrayVec();
            inst->Named_event_arrays(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Virtual_interface_vars()) {
            auto clone_vec = serializer_->MakeVirtual_interface_varVec();
            inst->Virtual_interface_vars(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Logic_vars()) {
            auto clone_vec = serializer_->MakeLogic_varVec();
            inst->Logic_vars(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Array_vars()) {
            auto clone_vec = serializer_->MakeArray_varVec();
            inst->Array_vars(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Array_var_mems()) {
            auto clone_vec = serializer_->MakeArray_varVec();
            inst->Array_var_mems(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Let_decls()) {
            auto clone_vec = serializer_->MakeLet_declVec();
            inst->Let_decls(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Attributes()) {
            auto clone_vec = serializer_->MakeAttributeVec();
            inst->Attributes(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = defMod->Instance_items()) {
            auto clone_vec = serializer_->MakeAnyVec();
            inst->Instance_items(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, inst);
              stmt->VpiParent(inst);
              clone_vec->push_back(stmt);
            }
          }


          break;
        }
        default:
          break;
        }
      }

    }
  }

  void leaveModule(const module* object, const BaseClass* parent,
                   vpiHandle handle, vpiHandle parentHandle) override {
    for (tf_call* call : scheduledTfCallBinding_) {
      if (call->UhdmType() == uhdmfunc_call) {
        if (function* f = dynamic_cast<function*>(bindTaskFunc(call->VpiName()))) {
          ((func_call*)call)->Function(f);
        }
      } else {
        if (task* f = dynamic_cast<task*>(bindTaskFunc(call->VpiName()))) {
          ((task_call*)call)->Task(f);
        }
      }
    }
    scheduledTfCallBinding_.clear();
    if (inHierarchy_) {
      instStack_.pop_back();
      if (instStack_.empty()) {
        inHierarchy_ = false;
      }
    }
  }

  void enterClass_defn(const class_defn* object, const BaseClass* parent,
                   vpiHandle handle, vpiHandle parentHandle) override {
    class_defn* cl = (class_defn*) object;

    // Collect instance elaborated nets
    ComponentMap varMap;
    if (object->Variables()) {
      for (variables* var : *object->Variables()) {
        varMap.insert(std::make_pair(var->VpiName(), var));
      }
    }

    // Collect instance parameters, defparams
    ComponentMap paramMap;
    if (object->Parameters()) {
      for (any* param : *object->Parameters()) {
        paramMap.insert(std::make_pair(param->VpiName(), param));
      }
    }

    // Collect func and task declaration
    ComponentMap funcMap;
    if (object->Task_funcs()) {
      for (task_func* var : *object->Task_funcs()) {
        funcMap.insert(std::make_pair(var->VpiName(), var));
      }
    }

    // Push class defn context on the stack
    // Class context is going to be pushed in case of:
    //   - imbricated classes
    //   - inheriting classes (Through the extends relation)
    instStack_.push_back(std::make_pair(object, std::make_tuple(varMap, paramMap, funcMap)));

          if (auto vec = cl->Task_funcs()) {
            auto clone_vec = serializer_->MakeTask_funcVec();
            cl->Task_funcs(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, cl);
              stmt->VpiParent(cl);
              clone_vec->push_back(stmt);
            }
          }
          if (auto obj = cl->Extends()) {
            auto* stmt = obj->DeepClone(serializer_, this, cl);
            stmt->VpiParent(cl);
            cl->Extends(stmt);
          }
          if (auto vec = cl->Constraints()) {
            auto clone_vec = serializer_->MakeConstraintVec();
            cl->Constraints(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, cl);
              stmt->VpiParent(cl);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = cl->Deriveds()) {
            auto clone_vec = serializer_->MakeClass_defnVec();
            cl->Deriveds(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj;
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = cl->Class_typespecs()) {
            auto clone_vec = serializer_->MakeClass_typespecVec();
            cl->Class_typespecs(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, cl);
              stmt->VpiParent(cl);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = cl->Concurrent_assertions()) {
            auto clone_vec = serializer_->MakeConcurrent_assertionsVec();
            cl->Concurrent_assertions(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, cl);
              stmt->VpiParent(cl);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = cl->Variables()) {
            auto clone_vec = serializer_->MakeVariablesVec();
            cl->Variables(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, cl);
              stmt->VpiParent(cl);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = cl->Scopes()) {
            auto clone_vec = serializer_->MakeScopeVec();
            cl->Scopes(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, cl);
              stmt->VpiParent(cl);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = cl->Typespecs()) {
            auto clone_vec = serializer_->MakeTypespecVec();
            cl->Typespecs(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, cl);
              stmt->VpiParent(cl);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = cl->Property_decls()) {
            auto clone_vec = serializer_->MakeProperty_declVec();
            cl->Property_decls(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, cl);
              stmt->VpiParent(cl);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = cl->Sequence_decls()) {
            auto clone_vec = serializer_->MakeSequence_declVec();
            cl->Sequence_decls(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, cl);
              stmt->VpiParent(cl);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = cl->Named_events()) {
            auto clone_vec = serializer_->MakeNamed_eventVec();
            cl->Named_events(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, cl);
              stmt->VpiParent(cl);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = cl->Named_event_arrays()) {
            auto clone_vec = serializer_->MakeNamed_event_arrayVec();
            cl->Named_event_arrays(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, cl);
              stmt->VpiParent(cl);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = cl->Virtual_interface_vars()) {
            auto clone_vec = serializer_->MakeVirtual_interface_varVec();
            cl->Virtual_interface_vars(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, cl);
              stmt->VpiParent(cl);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = cl->Logic_vars()) {
            auto clone_vec = serializer_->MakeLogic_varVec();
            cl->Logic_vars(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, cl);
              stmt->VpiParent(cl);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = cl->Array_vars()) {
            auto clone_vec = serializer_->MakeArray_varVec();
            cl->Array_vars(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, cl);
              stmt->VpiParent(cl);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = cl->Array_var_mems()) {
            auto clone_vec = serializer_->MakeArray_varVec();
            cl->Array_var_mems(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, cl);
              stmt->VpiParent(cl);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = cl->Param_assigns()) {
            auto clone_vec = serializer_->MakeParam_assignVec();
            cl->Param_assigns(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, cl);
              stmt->VpiParent(cl);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = cl->Let_decls()) {
            auto clone_vec = serializer_->MakeLet_declVec();
            cl->Let_decls(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, cl);
              stmt->VpiParent(cl);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = cl->Attributes()) {
            auto clone_vec = serializer_->MakeAttributeVec();
            cl->Attributes(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, cl);
              stmt->VpiParent(cl);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = cl->Parameters()) {
            auto clone_vec = serializer_->MakeAnyVec();
            cl->Parameters(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, cl);
              stmt->VpiParent(cl);
              clone_vec->push_back(stmt);
            }
          }
          if (auto vec = cl->Instance_items()) {
            auto clone_vec = serializer_->MakeAnyVec();
            cl->Instance_items(clone_vec);
            for (auto obj : *vec) {
              auto* stmt = obj->DeepClone(serializer_, this, cl);
              stmt->VpiParent(cl);
              clone_vec->push_back(stmt);
            }
          }


  }

  void leaveClass_defn(const class_defn* object, const BaseClass* parent,
                   vpiHandle handle, vpiHandle parentHandle) override {
    instStack_.pop_back();
  }

  void enterVariables(const variables* object, const BaseClass* parent,
                   vpiHandle handle, vpiHandle parentHandle) override;

  void leaveVariables(const variables* object, const BaseClass* parent,
                   vpiHandle handle, vpiHandle parentHandle) override;


  void enterTask_func(const task_func* object, const BaseClass* parent,
		   vpiHandle handle, vpiHandle parentHandle);

  void leaveTask_func(const task_func* object, const BaseClass* parent,
		   vpiHandle handle, vpiHandle parentHandle);

  void enterGen_scope(const gen_scope* object, const BaseClass* parent,
		      vpiHandle handle, vpiHandle parentHandle);
  
  void leaveGen_scope(const gen_scope* object, const BaseClass* parent,
		      vpiHandle handle, vpiHandle parentHandle);
 
private:

  // Instance context stack
  typedef std::vector<std::pair<const BaseClass*, std::tuple<ComponentMap, ComponentMap, ComponentMap>>> InstStack;
  InstStack instStack_;

  // Flat list of components (modules, udps, interfaces)
  ComponentMap flatComponentMap_;

  Serializer* serializer_ = nullptr;
  bool inHierarchy_ = false;
  bool debug_ = false;
  bool uniquifyTypespec_ = true;
  std::vector<tf_call*> scheduledTfCallBinding_;
};

};
