# -*- coding: utf-8 -*-
"""Hooks for pygments syntax highlighting."""
import os
import re
import string
import builtins
from warnings import warn
from collections import ChainMap
from collections.abc import MutableMapping

from pygments.lexer import inherit, bygroups, using, this
from pygments.token import (Keyword, Name, Comment, String, Error, Number,
                            Operator, Generic, Whitespace, Token)
from pygments.lexers.shell import BashLexer
from pygments.lexers.agile import PythonLexer
from pygments.style import Style
from pygments.styles import get_style_by_name
import pygments.util

from xonsh.tools import (ON_WINDOWS, intensify_colors_for_cmd_exe,
                         expand_gray_colors_for_cmd_exe)

class XonshSubprocLexer(BashLexer):
    """Lexer for xonsh subproc mode."""

    name = 'Xonsh subprocess lexer'

    tokens = {'root': [(r'`[^`]*?`', String.Backtick), inherit, ]}


ROOT_TOKENS = [(r'\?', Keyword),
               (r'\$\w+', Name.Variable),
               (r'\$\{', Keyword, ('pymode', )),
               (r'[\!\$]\(', Keyword, ('subproc', )),
               (r'[\!\$]\[', Keyword, ('subproc', )),
               (r'@\$\(', Keyword, ('subproc', )),
               (r'@\(', Keyword, ('pymode', )),
               inherit, ]

PYMODE_TOKENS = [(r'(.+)(\))', bygroups(using(this), Keyword), '#pop'),
                 (r'(.+)(\})', bygroups(using(this), Keyword), '#pop'), ]

SUBPROC_TOKENS = [
    (r'(.+)(\))', bygroups(using(XonshSubprocLexer), Keyword), '#pop'),
    (r'(.+)(\])', bygroups(using(XonshSubprocLexer), Keyword), '#pop'),
]


class XonshLexer(PythonLexer):
    """Xonsh console lexer for pygments."""

    name = 'Xonsh lexer'
    aliases = ['xonsh', 'xsh']
    filenames = ['*.xsh', '*xonshrc']

    tokens = {
        'root': list(ROOT_TOKENS),
        'pymode': PYMODE_TOKENS,
        'subproc': SUBPROC_TOKENS,
    }


class XonshConsoleLexer(PythonLexer):
    """Xonsh console lexer for pygments."""

    name = 'Xonsh console lexer'
    aliases = ['xonshcon']
    filenames = []

    tokens = {
        'root': [(r'^(>>>|\.\.\.) ', Generic.Prompt),
                 (r'\n(>>>|\.\.\.)', Generic.Prompt),
                 (r'\n(?![>.][>.][>.] )([^\n]*)', Generic.Output),
                 (r'\n(?![>.][>.][>.] )(.*?)$', Generic.Output)] + ROOT_TOKENS,
        'pymode': PYMODE_TOKENS,
        'subproc': SUBPROC_TOKENS,
    }

# XonshLexer & XonshSubprocLexer have to reference each other
XonshSubprocLexer.tokens['root'] = [
    (r'(\$\{)(.*)(\})', bygroups(Keyword, using(XonshLexer), Keyword)),
    (r'(@\()(.+)(\))', bygroups(Keyword, using(XonshLexer), Keyword)),
] + XonshSubprocLexer.tokens['root']


#
# Colors and Styles
#

Color = Token.Color  # alias to new color token namespace

RE_BACKGROUND = re.compile('(BG#|BGHEX|BACKGROUND)')

def norm_name(name):
    """Normalizes a color name."""
    return name.replace('#', 'HEX').replace('BGHEX', 'BACKGROUND_HEX')

def color_by_name(name, fg=None, bg=None):
    """Converts a color name to a color token, foreground name,
    and background name.  Will take into consideration current foreground
    and background colors, if provided.

    Parameters
    ----------
    name : str
        Color name.
    fg : str, optional
        Foreground color name.
    bg : str, optional
        Background color name.

    Returns
    -------
    tok : Token
        Pygments Token.Color subclass
    fg : str or None
        New computed foreground color name.
    bg : str or None
        New computed background color name.
    """
    name = name.upper()
    if name == 'NO_COLOR':
        return Color.NO_COLOR, None, None
    m = RE_BACKGROUND.search(name)
    if m is None:  # must be foreground color
        fg = norm_name(name)
    else:
        bg = norm_name(name)
    # assmble token
    if fg is None and bg is None:
        tokname = 'NO_COLOR'
    elif fg is None:
        tokname = bg
    elif bg is None:
        tokname = fg
    else:
        tokname = fg + '__' + bg
    tok = getattr(Color, tokname)
    return tok, fg, bg


def code_by_name(name, styles):
    """Converts a token name into a pygments-style color code.

    Parameters
    ----------
    name : str
        Color token name.
    styles : Mapping
        Mapping for looking up non-hex colors

    Returns
    -------
    code : str
        Pygments style color code.
    """
    fg, _, bg = name.lower().partition('__')
    if fg.startswith('background_'):
        fg, bg = bg, fg
    codes = []
    # foreground color
    if len(fg) == 0:
        pass
    elif 'hex' in fg:
        for p in fg.split('_'):
            codes.append('#'+p[3:] if p.startswith('hex') else p)
    else:
        fgtok = getattr(Color, fg.upper())
        if fgtok in styles:
            codes.append(styles[fgtok])
        else:
            codes += fg.split('_')
    # background color
    if len(bg) == 0:
        pass
    elif bg.startswith('background_hex'):
        codes.append('bg:#'+bg[14:])
    else:
        bgtok = getattr(Color, bg.upper())
        if bgtok in styles:
            codes.append(styles[bgtok])
        else:
            codes.append(bg.replace('background_', 'bg:'))
    code = ' '.join(codes)
    return code


def partial_color_tokenize(template):
    """Tokenizes a template string containing colors. Will return a list
    of tuples mapping the token to the string which has that color.
    These sub-strings maybe templates themselves.
    """
    if hasattr(builtins, '__xonsh_shell__'):
        styles = __xonsh_shell__.shell.styler.styles
    else:
        styles = None
    color = Color.NO_COLOR
    try:
        toks, color = _partial_color_tokenize_main(template, styles)
    except:
        toks = [(Color.NO_COLOR, template)]
    if styles is not None:
        styles[color]  # ensure color is available
    return toks


def _partial_color_tokenize_main(template, styles):
    formatter = string.Formatter()
    bopen = '{'
    bclose = '}'
    colon = ':'
    expl = '!'
    color = Color.NO_COLOR
    fg = bg = None
    value = ''
    toks = []
    for literal, field, spec, conv in formatter.parse(template):
        if field is None:
            value += literal
        elif field in KNOWN_COLORS or '#' in field:
            value += literal
            next_color, fg, bg = color_by_name(field, fg, bg)
            if next_color is not color:
                if len(value) > 0:
                    toks.append((color, value))
                    if styles is not None:
                        styles[color]  # ensure color is available
                color = next_color
                value = ''
        elif field is not None:
            parts = [literal, bopen, field]
            if conv is not None and len(conv) > 0:
                parts.append(expl)
                parts.append(conv)
            if spec is not None and len(spec) > 0:
                parts.append(colon)
                parts.append(spec)
            parts.append(bclose)
            value += ''.join(parts)
        else:
            value += literal
    toks.append((color, value))
    return toks, color


class CompoundColorMap(MutableMapping):
    """Looks up color tokes by name, potentailly generating the value
    from the lookup.
    """

    def __init__(self, styles, *args, **kwargs):
        self.styles = styles
        self.colors = dict(*args, **kwargs)

    def __getitem__(self, key):
        if key in self.colors:
            return self.colors[key]
        if key in self.styles:
            value = self.styles[key]
            self[key] = value
            return value
        if key is Color:
            raise KeyError
        pre, _, name = str(key).rpartition('.')
        if pre != 'Token.Color':
            raise KeyError
        value = code_by_name(name, self.styles)
        self[key] = value
        return value

    def __setitem__(self, key, value):
        self.colors[key] = value

    def __delitem__(self, key):
        del self.colors[key]

    def __iter__(self):
        yield from self.colors.keys()

    def __len__(self):
        return len(self.colors)


class XonshStyle(Style):
    """A xonsh pygments style that will dispatch to the correct color map
    by using a ChainMap.  The style_name property may be used to reset
    the current style.
    """

    def __init__(self, style_name='default'):
        """
        Parameters
        ----------
        style_name : str, optional
            The style name to initialize with.
        """
        self.trap = {}  # for traping custom colors set by user
        self._smap = {}
        self._style_name = ''
        self.style_name = style_name
        super().__init__()

    @property
    def style_name(self):
        return self._style_name

    @style_name.setter
    def style_name(self, value):
        if self._style_name == value:
            return
        if value in STYLES:
            cmap = STYLES[value]
        else:
            warn('Could not find style {0!r}, using default'.format(value),
                 RuntimeWarning)
            cmap = DEFAULT_STYLE
        try:
            self._smap = get_style_by_name(value)().styles.copy()
        except (ImportError, pygments.util.ClassNotFound):
            self._smap = XONSH_BASE_STYLE.copy()
        compound = CompoundColorMap(ChainMap(self.trap, cmap, PTK_STYLE, self._smap))
        self.styles = ChainMap(self.trap, cmap, PTK_STYLE, self._smap, compound)
        self._style_name = value
        if ON_WINDOWS:
            self.enhance_colors_for_cmd_exe()

    @style_name.deleter
    def style_name(self):
        self._style_name = ''

    def enhance_colors_for_cmd_exe(self):
        """ Enhance colors when using cmd.exe on windows.
            When using the default style all blue and dark red colors
            are changed to CYAN and intence red.
        """
        env = builtins.__xonsh_env__
        # Ensure we are not using ConEmu
        if 'CONEMUANSI' not in env:
            # Auto suggest needs to be a darker shade to be distinguishable
            # from the default color
            self.styles[Token.AutoSuggestion] = '#444444'
            if env.get('INTENSIFY_COLORS_ON_WIN', False):
                self._smap.update(expand_gray_colors_for_cmd_exe(self._smap))
                self._smap.update(intensify_colors_for_cmd_exe(self._smap))


def xonsh_style_proxy(styler):
    """Factory for a proxy class to a xonsh style."""
    class XonshStyleProxy(Style):
        """Simple proxy class to fool prompt toolkit."""

        target = styler
        styles = styler.styles

        def __new__(cls, *args, **kwargs):
            return cls.target

    return XonshStyleProxy


PTK_STYLE = {
    Token.Menu.Completions.Completion.Current: 'bg:#00aaaa #000000',
    Token.Menu.Completions.Completion: 'bg:#008888 #ffffff',
    Token.Menu.Completions.ProgressButton: 'bg:#003333',
    Token.Menu.Completions.ProgressBar: 'bg:#00aaaa',
    Token.AutoSuggestion: '#666666',
    Token.Aborted: '#888888',
}

XONSH_BASE_STYLE = {
    Whitespace: '#008080',
    Comment: 'underline',
    Comment.Preproc: 'underline',
    Keyword: 'bold',
    Keyword.Pseudo: '#008000',
    Keyword.Type: '',
    Operator: '#008080',
    Operator.Word: 'bold',
    Name.Builtin: '',
    Name.Function: '#000080',
    Name.Class: 'bold',
    Name.Namespace: 'bold',
    Name.Exception: 'bold',
    Name.Variable: '#008080',
    Name.Constant: '#800000',
    Name.Label: '#808000',
    Name.Entity: 'bold',
    Name.Attribute: '#008080',
    Name.Tag: 'bold',
    Name.Decorator: '#008080',
    String: '',
    String.Doc: 'underline',
    String.Interpol: 'bold',
    String.Escape: 'bold',
    String.Regex: '',
    String.Symbol: '',
    String.Other: '#008000',
    Number: '#800000',
    Generic.Heading: 'bold',
    Generic.Subheading: 'bold',
    Generic.Deleted: '#800000',
    Generic.Inserted: '#008000',
    Generic.Error: 'bold',
    Generic.Emph: 'underline',
    Generic.Prompt: 'bold',
    Generic.Output: '#008080',
    Generic.Traceback: '#800000',
    Error: '#800000',
}

KNOWN_COLORS = frozenset([
    'BACKGROUND_BLACK',
    'BACKGROUND_BLUE',
    'BACKGROUND_CYAN',
    'BACKGROUND_GREEN',
    'BACKGROUND_INTENSE_BLACK',
    'BACKGROUND_INTENSE_BLUE',
    'BACKGROUND_INTENSE_CYAN',
    'BACKGROUND_INTENSE_GREEN',
    'BACKGROUND_INTENSE_PURPLE',
    'BACKGROUND_INTENSE_RED',
    'BACKGROUND_INTENSE_WHITE',
    'BACKGROUND_INTENSE_YELLOW',
    'BACKGROUND_PURPLE',
    'BACKGROUND_RED',
    'BACKGROUND_WHITE',
    'BACKGROUND_YELLOW',
    'BLACK',
    'BLUE',
    'BOLD_BLACK',
    'BOLD_BLUE',
    'BOLD_CYAN',
    'BOLD_GREEN',
    'BOLD_INTENSE_BLACK',
    'BOLD_INTENSE_BLUE',
    'BOLD_INTENSE_CYAN',
    'BOLD_INTENSE_GREEN',
    'BOLD_INTENSE_PURPLE',
    'BOLD_INTENSE_RED',
    'BOLD_INTENSE_WHITE',
    'BOLD_INTENSE_YELLOW',
    'BOLD_PURPLE',
    'BOLD_RED',
    'BOLD_UNDERLINE_BLACK',
    'BOLD_UNDERLINE_BLUE',
    'BOLD_UNDERLINE_CYAN',
    'BOLD_UNDERLINE_GREEN',
    'BOLD_UNDERLINE_INTENSE_BLACK',
    'BOLD_UNDERLINE_INTENSE_BLUE',
    'BOLD_UNDERLINE_INTENSE_CYAN',
    'BOLD_UNDERLINE_INTENSE_GREEN',
    'BOLD_UNDERLINE_INTENSE_PURPLE',
    'BOLD_UNDERLINE_INTENSE_RED',
    'BOLD_UNDERLINE_INTENSE_WHITE',
    'BOLD_UNDERLINE_INTENSE_YELLOW',
    'BOLD_UNDERLINE_PURPLE',
    'BOLD_UNDERLINE_RED',
    'BOLD_UNDERLINE_WHITE',
    'BOLD_UNDERLINE_YELLOW',
    'BOLD_WHITE',
    'BOLD_YELLOW',
    'CYAN',
    'GREEN',
    'INTENSE_BLACK',
    'INTENSE_BLUE',
    'INTENSE_CYAN',
    'INTENSE_GREEN',
    'INTENSE_PURPLE',
    'INTENSE_RED',
    'INTENSE_WHITE',
    'INTENSE_YELLOW',
    'NO_COLOR',
    'PURPLE',
    'RED',
    'UNDERLINE_BLACK',
    'UNDERLINE_BLUE',
    'UNDERLINE_CYAN',
    'UNDERLINE_GREEN',
    'UNDERLINE_INTENSE_BLACK',
    'UNDERLINE_INTENSE_BLUE',
    'UNDERLINE_INTENSE_CYAN',
    'UNDERLINE_INTENSE_GREEN',
    'UNDERLINE_INTENSE_PURPLE',
    'UNDERLINE_INTENSE_RED',
    'UNDERLINE_INTENSE_WHITE',
    'UNDERLINE_INTENSE_YELLOW',
    'UNDERLINE_PURPLE',
    'UNDERLINE_RED',
    'UNDERLINE_WHITE',
    'UNDERLINE_YELLOW',
    'WHITE',
    'YELLOW',
])


def _expand_style(cmap):
    """Expands a style in order to more quickly make color map changes."""
    for key, val in list(cmap.items()):
        if key is Color.NO_COLOR:
            continue
        _, _, key = str(key).rpartition('.')
        cmap[getattr(Color, 'BOLD_'+key)] = 'bold ' + val
        cmap[getattr(Color, 'UNDERLINE_'+key)] = 'underline ' + val
        cmap[getattr(Color, 'BOLD_UNDERLINE_'+key)] = 'bold underline ' + val
        if val == 'noinherit':
            cmap[getattr(Color, 'BACKGROUND_'+key)] = val
        else:
            cmap[getattr(Color, 'BACKGROUND_'+key)] = 'bg:' + val


BW_STYLE = {
    Color.BLACK: 'noinherit',
    Color.BLUE: 'noinherit',
    Color.CYAN: 'noinherit',
    Color.GREEN: 'noinherit',
    Color.INTENSE_BLACK: 'noinherit',
    Color.INTENSE_BLUE: 'noinherit',
    Color.INTENSE_CYAN: 'noinherit',
    Color.INTENSE_GREEN: 'noinherit',
    Color.INTENSE_PURPLE: 'noinherit',
    Color.INTENSE_RED: 'noinherit',
    Color.INTENSE_WHITE: 'noinherit',
    Color.INTENSE_YELLOW: 'noinherit',
    Color.NO_COLOR: 'noinherit',
    Color.PURPLE: 'noinherit',
    Color.RED: 'noinherit',
    Color.WHITE: 'noinherit',
    Color.YELLOW: 'noinherit',
}
_expand_style(BW_STYLE)


if hasattr(pygments.style, 'ansicolors'):
    DEFAULT_STYLE = {
        Color.BLACK: '#ansidarkgray',
        Color.BLUE: '#ansiblue',
        Color.CYAN: '#ansiturquoise',
        Color.GREEN: '#ansigreen',
        Color.INTENSE_BLACK: '#ansiblack',
        Color.INTENSE_BLUE: '#ansidarkblue',
        Color.INTENSE_CYAN: '#ansiteal',
        Color.INTENSE_GREEN: '#ansidarkgreen',
        Color.INTENSE_PURPLE: '#ansipurple',
        Color.INTENSE_RED: '#ansidarkred',
        Color.INTENSE_WHITE: '#ansilightgray',
        Color.INTENSE_YELLOW: '#ansibrown',
        Color.NO_COLOR: 'noinherit',
        Color.PURPLE: '#ansifuchsia',
        Color.RED: '#ansired',
        Color.WHITE: '#ansiwhite',
        Color.YELLOW: '#ansiyellow',
    }
elif ON_WINDOWS and 'CONEMUANSI' not in os.environ:
    # These colors must match the color specification
    # in prompt_toolkit, so the colors are converted 
    # correctly when using cmd.exe
    DEFAULT_STYLE = {
        Color.BLACK: '#000000',
        Color.BLUE: '#0000AA',
        Color.CYAN: '#00AAAA',
        Color.GREEN: '#00AA00',
        Color.INTENSE_BLACK: '#444444',
        Color.INTENSE_BLUE: '#4444FF',
        Color.INTENSE_CYAN: '#44FFFF',
        Color.INTENSE_GREEN: '#44FF44',
        Color.INTENSE_PURPLE: '#FF44FF',
        Color.INTENSE_RED: '#FF4444',
        Color.INTENSE_WHITE: '#888888',
        Color.INTENSE_YELLOW: '#FFFF44',
        Color.NO_COLOR: 'noinherit',
        Color.PURPLE: '#AA00AA',
        Color.RED: '#AA0000',
        Color.WHITE: '#FFFFFF',
        Color.YELLOW: '#AAAA00',
        }
else:
    DEFAULT_STYLE = {
        Color.BLACK: '#000000',
        Color.BLUE: '#0000AA',
        Color.CYAN: '#00AAAA',
        Color.GREEN: '#00AA00',
        Color.INTENSE_BLACK: '#555555',
        Color.INTENSE_BLUE: '#0000FF',
        Color.INTENSE_CYAN: '#55FFFF',
        Color.INTENSE_GREEN: '#00FF00',
        Color.INTENSE_PURPLE: '#FF00FF',
        Color.INTENSE_RED: '#FF0000',
        Color.INTENSE_WHITE: '#aaaaaa',
        Color.INTENSE_YELLOW: '#FFFF55',
        Color.NO_COLOR: 'noinherit',
        Color.PURPLE: '#AA00AA',
        Color.RED: '#AA0000',
        Color.WHITE: '#ffffff',
        Color.YELLOW: '#ffff00',
    }

_expand_style(DEFAULT_STYLE)

MONOKAI_STYLE = {
    Color.BLACK: '#1e0010',
    Color.BLUE: '#6666ef',
    Color.CYAN: '#66d9ef',
    Color.GREEN: '#2ee22e',
    Color.INTENSE_BLACK: '#5e5e5e',
    Color.INTENSE_BLUE: '#2626d7',
    Color.INTENSE_CYAN: '#2ed9d9',
    Color.INTENSE_GREEN: '#a6e22e',
    Color.INTENSE_PURPLE: '#ae81ff',
    Color.INTENSE_RED: '#f92672',
    Color.INTENSE_WHITE: '#f8f8f2',
    Color.INTENSE_YELLOW: '#e6db74',
    Color.NO_COLOR: 'noinherit',
    Color.PURPLE: '#960050',
    Color.RED: '#AF0000',
    Color.WHITE: '#d7d7d7',
    Color.YELLOW: '#e2e22e',
}
_expand_style(MONOKAI_STYLE)

#############################################################
#############   Auto-generated below this line   ############
#############################################################

ALGOL_STYLE = {
    Color.BACKGROUND_BLACK: 'bg:#666',
    Color.BACKGROUND_BLUE: 'bg:#666',
    Color.BACKGROUND_CYAN: 'bg:#666',
    Color.BACKGROUND_GREEN: 'bg:#666',
    Color.BACKGROUND_INTENSE_BLACK: 'bg:#666',
    Color.BACKGROUND_INTENSE_BLUE: 'bg:#888',
    Color.BACKGROUND_INTENSE_CYAN: 'bg:#888',
    Color.BACKGROUND_INTENSE_GREEN: 'bg:#888',
    Color.BACKGROUND_INTENSE_PURPLE: 'bg:#888',
    Color.BACKGROUND_INTENSE_RED: 'bg:#FF0000',
    Color.BACKGROUND_INTENSE_WHITE: 'bg:#888',
    Color.BACKGROUND_INTENSE_YELLOW: 'bg:#888',
    Color.BACKGROUND_PURPLE: 'bg:#666',
    Color.BACKGROUND_RED: 'bg:#FF0000',
    Color.BACKGROUND_WHITE: 'bg:#888',
    Color.BACKGROUND_YELLOW: 'bg:#FF0000',
    Color.BLACK: '#666',
    Color.BLUE: '#666',
    Color.BOLD_BLACK: 'bold #666',
    Color.BOLD_BLUE: 'bold #666',
    Color.BOLD_CYAN: 'bold #666',
    Color.BOLD_GREEN: 'bold #666',
    Color.BOLD_INTENSE_BLACK: 'bold #666',
    Color.BOLD_INTENSE_BLUE: 'bold #888',
    Color.BOLD_INTENSE_CYAN: 'bold #888',
    Color.BOLD_INTENSE_GREEN: 'bold #888',
    Color.BOLD_INTENSE_PURPLE: 'bold #888',
    Color.BOLD_INTENSE_RED: 'bold #FF0000',
    Color.BOLD_INTENSE_WHITE: 'bold #888',
    Color.BOLD_INTENSE_YELLOW: 'bold #888',
    Color.BOLD_PURPLE: 'bold #666',
    Color.BOLD_RED: 'bold #FF0000',
    Color.BOLD_UNDERLINE_BLACK: 'bold underline #666',
    Color.BOLD_UNDERLINE_BLUE: 'bold underline #666',
    Color.BOLD_UNDERLINE_CYAN: 'bold underline #666',
    Color.BOLD_UNDERLINE_GREEN: 'bold underline #666',
    Color.BOLD_UNDERLINE_INTENSE_BLACK: 'bold underline #666',
    Color.BOLD_UNDERLINE_INTENSE_BLUE: 'bold underline #888',
    Color.BOLD_UNDERLINE_INTENSE_CYAN: 'bold underline #888',
    Color.BOLD_UNDERLINE_INTENSE_GREEN: 'bold underline #888',
    Color.BOLD_UNDERLINE_INTENSE_PURPLE: 'bold underline #888',
    Color.BOLD_UNDERLINE_INTENSE_RED: 'bold underline #FF0000',
    Color.BOLD_UNDERLINE_INTENSE_WHITE: 'bold underline #888',
    Color.BOLD_UNDERLINE_INTENSE_YELLOW: 'bold underline #888',
    Color.BOLD_UNDERLINE_PURPLE: 'bold underline #666',
    Color.BOLD_UNDERLINE_RED: 'bold underline #FF0000',
    Color.BOLD_UNDERLINE_WHITE: 'bold underline #888',
    Color.BOLD_UNDERLINE_YELLOW: 'bold underline #FF0000',
    Color.BOLD_WHITE: 'bold #888',
    Color.BOLD_YELLOW: 'bold #FF0000',
    Color.CYAN: '#666',
    Color.GREEN: '#666',
    Color.INTENSE_BLACK: '#666',
    Color.INTENSE_BLUE: '#888',
    Color.INTENSE_CYAN: '#888',
    Color.INTENSE_GREEN: '#888',
    Color.INTENSE_PURPLE: '#888',
    Color.INTENSE_RED: '#FF0000',
    Color.INTENSE_WHITE: '#888',
    Color.INTENSE_YELLOW: '#888',
    Color.NO_COLOR: 'noinherit',
    Color.PURPLE: '#666',
    Color.RED: '#FF0000',
    Color.UNDERLINE_BLACK: 'underline #666',
    Color.UNDERLINE_BLUE: 'underline #666',
    Color.UNDERLINE_CYAN: 'underline #666',
    Color.UNDERLINE_GREEN: 'underline #666',
    Color.UNDERLINE_INTENSE_BLACK: 'underline #666',
    Color.UNDERLINE_INTENSE_BLUE: 'underline #888',
    Color.UNDERLINE_INTENSE_CYAN: 'underline #888',
    Color.UNDERLINE_INTENSE_GREEN: 'underline #888',
    Color.UNDERLINE_INTENSE_PURPLE: 'underline #888',
    Color.UNDERLINE_INTENSE_RED: 'underline #FF0000',
    Color.UNDERLINE_INTENSE_WHITE: 'underline #888',
    Color.UNDERLINE_INTENSE_YELLOW: 'underline #888',
    Color.UNDERLINE_PURPLE: 'underline #666',
    Color.UNDERLINE_RED: 'underline #FF0000',
    Color.UNDERLINE_WHITE: 'underline #888',
    Color.UNDERLINE_YELLOW: 'underline #FF0000',
    Color.WHITE: '#888',
    Color.YELLOW: '#FF0000',
}

ALGOL_NU_STYLE = {
    Color.BACKGROUND_BLACK: 'bg:#666',
    Color.BACKGROUND_BLUE: 'bg:#666',
    Color.BACKGROUND_CYAN: 'bg:#666',
    Color.BACKGROUND_GREEN: 'bg:#666',
    Color.BACKGROUND_INTENSE_BLACK: 'bg:#666',
    Color.BACKGROUND_INTENSE_BLUE: 'bg:#888',
    Color.BACKGROUND_INTENSE_CYAN: 'bg:#888',
    Color.BACKGROUND_INTENSE_GREEN: 'bg:#888',
    Color.BACKGROUND_INTENSE_PURPLE: 'bg:#888',
    Color.BACKGROUND_INTENSE_RED: 'bg:#FF0000',
    Color.BACKGROUND_INTENSE_WHITE: 'bg:#888',
    Color.BACKGROUND_INTENSE_YELLOW: 'bg:#888',
    Color.BACKGROUND_PURPLE: 'bg:#666',
    Color.BACKGROUND_RED: 'bg:#FF0000',
    Color.BACKGROUND_WHITE: 'bg:#888',
    Color.BACKGROUND_YELLOW: 'bg:#FF0000',
    Color.BLACK: '#666',
    Color.BLUE: '#666',
    Color.BOLD_BLACK: 'bold #666',
    Color.BOLD_BLUE: 'bold #666',
    Color.BOLD_CYAN: 'bold #666',
    Color.BOLD_GREEN: 'bold #666',
    Color.BOLD_INTENSE_BLACK: 'bold #666',
    Color.BOLD_INTENSE_BLUE: 'bold #888',
    Color.BOLD_INTENSE_CYAN: 'bold #888',
    Color.BOLD_INTENSE_GREEN: 'bold #888',
    Color.BOLD_INTENSE_PURPLE: 'bold #888',
    Color.BOLD_INTENSE_RED: 'bold #FF0000',
    Color.BOLD_INTENSE_WHITE: 'bold #888',
    Color.BOLD_INTENSE_YELLOW: 'bold #888',
    Color.BOLD_PURPLE: 'bold #666',
    Color.BOLD_RED: 'bold #FF0000',
    Color.BOLD_UNDERLINE_BLACK: 'bold underline #666',
    Color.BOLD_UNDERLINE_BLUE: 'bold underline #666',
    Color.BOLD_UNDERLINE_CYAN: 'bold underline #666',
    Color.BOLD_UNDERLINE_GREEN: 'bold underline #666',
    Color.BOLD_UNDERLINE_INTENSE_BLACK: 'bold underline #666',
    Color.BOLD_UNDERLINE_INTENSE_BLUE: 'bold underline #888',
    Color.BOLD_UNDERLINE_INTENSE_CYAN: 'bold underline #888',
    Color.BOLD_UNDERLINE_INTENSE_GREEN: 'bold underline #888',
    Color.BOLD_UNDERLINE_INTENSE_PURPLE: 'bold underline #888',
    Color.BOLD_UNDERLINE_INTENSE_RED: 'bold underline #FF0000',
    Color.BOLD_UNDERLINE_INTENSE_WHITE: 'bold underline #888',
    Color.BOLD_UNDERLINE_INTENSE_YELLOW: 'bold underline #888',
    Color.BOLD_UNDERLINE_PURPLE: 'bold underline #666',
    Color.BOLD_UNDERLINE_RED: 'bold underline #FF0000',
    Color.BOLD_UNDERLINE_WHITE: 'bold underline #888',
    Color.BOLD_UNDERLINE_YELLOW: 'bold underline #FF0000',
    Color.BOLD_WHITE: 'bold #888',
    Color.BOLD_YELLOW: 'bold #FF0000',
    Color.CYAN: '#666',
    Color.GREEN: '#666',
    Color.INTENSE_BLACK: '#666',
    Color.INTENSE_BLUE: '#888',
    Color.INTENSE_CYAN: '#888',
    Color.INTENSE_GREEN: '#888',
    Color.INTENSE_PURPLE: '#888',
    Color.INTENSE_RED: '#FF0000',
    Color.INTENSE_WHITE: '#888',
    Color.INTENSE_YELLOW: '#888',
    Color.NO_COLOR: 'noinherit',
    Color.PURPLE: '#666',
    Color.RED: '#FF0000',
    Color.UNDERLINE_BLACK: 'underline #666',
    Color.UNDERLINE_BLUE: 'underline #666',
    Color.UNDERLINE_CYAN: 'underline #666',
    Color.UNDERLINE_GREEN: 'underline #666',
    Color.UNDERLINE_INTENSE_BLACK: 'underline #666',
    Color.UNDERLINE_INTENSE_BLUE: 'underline #888',
    Color.UNDERLINE_INTENSE_CYAN: 'underline #888',
    Color.UNDERLINE_INTENSE_GREEN: 'underline #888',
    Color.UNDERLINE_INTENSE_PURPLE: 'underline #888',
    Color.UNDERLINE_INTENSE_RED: 'underline #FF0000',
    Color.UNDERLINE_INTENSE_WHITE: 'underline #888',
    Color.UNDERLINE_INTENSE_YELLOW: 'underline #888',
    Color.UNDERLINE_PURPLE: 'underline #666',
    Color.UNDERLINE_RED: 'underline #FF0000',
    Color.UNDERLINE_WHITE: 'underline #888',
    Color.UNDERLINE_YELLOW: 'underline #FF0000',
    Color.WHITE: '#888',
    Color.YELLOW: '#FF0000',
}

AUTUMN_STYLE = {
    Color.BACKGROUND_BLACK: 'bg:#000080',
    Color.BACKGROUND_BLUE: 'bg:#0000aa',
    Color.BACKGROUND_CYAN: 'bg:#00aaaa',
    Color.BACKGROUND_GREEN: 'bg:#00aa00',
    Color.BACKGROUND_INTENSE_BLACK: 'bg:#555555',
    Color.BACKGROUND_INTENSE_BLUE: 'bg:#1e90ff',
    Color.BACKGROUND_INTENSE_CYAN: 'bg:#1e90ff',
    Color.BACKGROUND_INTENSE_GREEN: 'bg:#4c8317',
    Color.BACKGROUND_INTENSE_PURPLE: 'bg:#FAA',
    Color.BACKGROUND_INTENSE_RED: 'bg:#aa5500',
    Color.BACKGROUND_INTENSE_WHITE: 'bg:#bbbbbb',
    Color.BACKGROUND_INTENSE_YELLOW: 'bg:#FAA',
    Color.BACKGROUND_PURPLE: 'bg:#800080',
    Color.BACKGROUND_RED: 'bg:#aa0000',
    Color.BACKGROUND_WHITE: 'bg:#aaaaaa',
    Color.BACKGROUND_YELLOW: 'bg:#aa5500',
    Color.BLACK: '#000080',
    Color.BLUE: '#0000aa',
    Color.BOLD_BLACK: 'bold #000080',
    Color.BOLD_BLUE: 'bold #0000aa',
    Color.BOLD_CYAN: 'bold #00aaaa',
    Color.BOLD_GREEN: 'bold #00aa00',
    Color.BOLD_INTENSE_BLACK: 'bold #555555',
    Color.BOLD_INTENSE_BLUE: 'bold #1e90ff',
    Color.BOLD_INTENSE_CYAN: 'bold #1e90ff',
    Color.BOLD_INTENSE_GREEN: 'bold #4c8317',
    Color.BOLD_INTENSE_PURPLE: 'bold #FAA',
    Color.BOLD_INTENSE_RED: 'bold #aa5500',
    Color.BOLD_INTENSE_WHITE: 'bold #bbbbbb',
    Color.BOLD_INTENSE_YELLOW: 'bold #FAA',
    Color.BOLD_PURPLE: 'bold #800080',
    Color.BOLD_RED: 'bold #aa0000',
    Color.BOLD_UNDERLINE_BLACK: 'bold underline #000080',
    Color.BOLD_UNDERLINE_BLUE: 'bold underline #0000aa',
    Color.BOLD_UNDERLINE_CYAN: 'bold underline #00aaaa',
    Color.BOLD_UNDERLINE_GREEN: 'bold underline #00aa00',
    Color.BOLD_UNDERLINE_INTENSE_BLACK: 'bold underline #555555',
    Color.BOLD_UNDERLINE_INTENSE_BLUE: 'bold underline #1e90ff',
    Color.BOLD_UNDERLINE_INTENSE_CYAN: 'bold underline #1e90ff',
    Color.BOLD_UNDERLINE_INTENSE_GREEN: 'bold underline #4c8317',
    Color.BOLD_UNDERLINE_INTENSE_PURPLE: 'bold underline #FAA',
    Color.BOLD_UNDERLINE_INTENSE_RED: 'bold underline #aa5500',
    Color.BOLD_UNDERLINE_INTENSE_WHITE: 'bold underline #bbbbbb',
    Color.BOLD_UNDERLINE_INTENSE_YELLOW: 'bold underline #FAA',
    Color.BOLD_UNDERLINE_PURPLE: 'bold underline #800080',
    Color.BOLD_UNDERLINE_RED: 'bold underline #aa0000',
    Color.BOLD_UNDERLINE_WHITE: 'bold underline #aaaaaa',
    Color.BOLD_UNDERLINE_YELLOW: 'bold underline #aa5500',
    Color.BOLD_WHITE: 'bold #aaaaaa',
    Color.BOLD_YELLOW: 'bold #aa5500',
    Color.CYAN: '#00aaaa',
    Color.GREEN: '#00aa00',
    Color.INTENSE_BLACK: '#555555',
    Color.INTENSE_BLUE: '#1e90ff',
    Color.INTENSE_CYAN: '#1e90ff',
    Color.INTENSE_GREEN: '#4c8317',
    Color.INTENSE_PURPLE: '#FAA',
    Color.INTENSE_RED: '#aa5500',
    Color.INTENSE_WHITE: '#bbbbbb',
    Color.INTENSE_YELLOW: '#FAA',
    Color.NO_COLOR: 'noinherit',
    Color.PURPLE: '#800080',
    Color.RED: '#aa0000',
    Color.UNDERLINE_BLACK: 'underline #000080',
    Color.UNDERLINE_BLUE: 'underline #0000aa',
    Color.UNDERLINE_CYAN: 'underline #00aaaa',
    Color.UNDERLINE_GREEN: 'underline #00aa00',
    Color.UNDERLINE_INTENSE_BLACK: 'underline #555555',
    Color.UNDERLINE_INTENSE_BLUE: 'underline #1e90ff',
    Color.UNDERLINE_INTENSE_CYAN: 'underline #1e90ff',
    Color.UNDERLINE_INTENSE_GREEN: 'underline #4c8317',
    Color.UNDERLINE_INTENSE_PURPLE: 'underline #FAA',
    Color.UNDERLINE_INTENSE_RED: 'underline #aa5500',
    Color.UNDERLINE_INTENSE_WHITE: 'underline #bbbbbb',
    Color.UNDERLINE_INTENSE_YELLOW: 'underline #FAA',
    Color.UNDERLINE_PURPLE: 'underline #800080',
    Color.UNDERLINE_RED: 'underline #aa0000',
    Color.UNDERLINE_WHITE: 'underline #aaaaaa',
    Color.UNDERLINE_YELLOW: 'underline #aa5500',
    Color.WHITE: '#aaaaaa',
    Color.YELLOW: '#aa5500',
}

BORLAND_STYLE = {
    Color.BACKGROUND_BLACK: 'bg:#000000',
    Color.BACKGROUND_BLUE: 'bg:#000080',
    Color.BACKGROUND_CYAN: 'bg:#008080',
    Color.BACKGROUND_GREEN: 'bg:#008800',
    Color.BACKGROUND_INTENSE_BLACK: 'bg:#555555',
    Color.BACKGROUND_INTENSE_BLUE: 'bg:#0000FF',
    Color.BACKGROUND_INTENSE_CYAN: 'bg:#ddffdd',
    Color.BACKGROUND_INTENSE_GREEN: 'bg:#888888',
    Color.BACKGROUND_INTENSE_PURPLE: 'bg:#e3d2d2',
    Color.BACKGROUND_INTENSE_RED: 'bg:#FF0000',
    Color.BACKGROUND_INTENSE_WHITE: 'bg:#ffdddd',
    Color.BACKGROUND_INTENSE_YELLOW: 'bg:#e3d2d2',
    Color.BACKGROUND_PURPLE: 'bg:#800080',
    Color.BACKGROUND_RED: 'bg:#aa0000',
    Color.BACKGROUND_WHITE: 'bg:#aaaaaa',
    Color.BACKGROUND_YELLOW: 'bg:#a61717',
    Color.BLACK: '#000000',
    Color.BLUE: '#000080',
    Color.BOLD_BLACK: 'bold #000000',
    Color.BOLD_BLUE: 'bold #000080',
    Color.BOLD_CYAN: 'bold #008080',
    Color.BOLD_GREEN: 'bold #008800',
    Color.BOLD_INTENSE_BLACK: 'bold #555555',
    Color.BOLD_INTENSE_BLUE: 'bold #0000FF',
    Color.BOLD_INTENSE_CYAN: 'bold #ddffdd',
    Color.BOLD_INTENSE_GREEN: 'bold #888888',
    Color.BOLD_INTENSE_PURPLE: 'bold #e3d2d2',
    Color.BOLD_INTENSE_RED: 'bold #FF0000',
    Color.BOLD_INTENSE_WHITE: 'bold #ffdddd',
    Color.BOLD_INTENSE_YELLOW: 'bold #e3d2d2',
    Color.BOLD_PURPLE: 'bold #800080',
    Color.BOLD_RED: 'bold #aa0000',
    Color.BOLD_UNDERLINE_BLACK: 'bold underline #000000',
    Color.BOLD_UNDERLINE_BLUE: 'bold underline #000080',
    Color.BOLD_UNDERLINE_CYAN: 'bold underline #008080',
    Color.BOLD_UNDERLINE_GREEN: 'bold underline #008800',
    Color.BOLD_UNDERLINE_INTENSE_BLACK: 'bold underline #555555',
    Color.BOLD_UNDERLINE_INTENSE_BLUE: 'bold underline #0000FF',
    Color.BOLD_UNDERLINE_INTENSE_CYAN: 'bold underline #ddffdd',
    Color.BOLD_UNDERLINE_INTENSE_GREEN: 'bold underline #888888',
    Color.BOLD_UNDERLINE_INTENSE_PURPLE: 'bold underline #e3d2d2',
    Color.BOLD_UNDERLINE_INTENSE_RED: 'bold underline #FF0000',
    Color.BOLD_UNDERLINE_INTENSE_WHITE: 'bold underline #ffdddd',
    Color.BOLD_UNDERLINE_INTENSE_YELLOW: 'bold underline #e3d2d2',
    Color.BOLD_UNDERLINE_PURPLE: 'bold underline #800080',
    Color.BOLD_UNDERLINE_RED: 'bold underline #aa0000',
    Color.BOLD_UNDERLINE_WHITE: 'bold underline #aaaaaa',
    Color.BOLD_UNDERLINE_YELLOW: 'bold underline #a61717',
    Color.BOLD_WHITE: 'bold #aaaaaa',
    Color.BOLD_YELLOW: 'bold #a61717',
    Color.CYAN: '#008080',
    Color.GREEN: '#008800',
    Color.INTENSE_BLACK: '#555555',
    Color.INTENSE_BLUE: '#0000FF',
    Color.INTENSE_CYAN: '#ddffdd',
    Color.INTENSE_GREEN: '#888888',
    Color.INTENSE_PURPLE: '#e3d2d2',
    Color.INTENSE_RED: '#FF0000',
    Color.INTENSE_WHITE: '#ffdddd',
    Color.INTENSE_YELLOW: '#e3d2d2',
    Color.NO_COLOR: 'noinherit',
    Color.PURPLE: '#800080',
    Color.RED: '#aa0000',
    Color.UNDERLINE_BLACK: 'underline #000000',
    Color.UNDERLINE_BLUE: 'underline #000080',
    Color.UNDERLINE_CYAN: 'underline #008080',
    Color.UNDERLINE_GREEN: 'underline #008800',
    Color.UNDERLINE_INTENSE_BLACK: 'underline #555555',
    Color.UNDERLINE_INTENSE_BLUE: 'underline #0000FF',
    Color.UNDERLINE_INTENSE_CYAN: 'underline #ddffdd',
    Color.UNDERLINE_INTENSE_GREEN: 'underline #888888',
    Color.UNDERLINE_INTENSE_PURPLE: 'underline #e3d2d2',
    Color.UNDERLINE_INTENSE_RED: 'underline #FF0000',
    Color.UNDERLINE_INTENSE_WHITE: 'underline #ffdddd',
    Color.UNDERLINE_INTENSE_YELLOW: 'underline #e3d2d2',
    Color.UNDERLINE_PURPLE: 'underline #800080',
    Color.UNDERLINE_RED: 'underline #aa0000',
    Color.UNDERLINE_WHITE: 'underline #aaaaaa',
    Color.UNDERLINE_YELLOW: 'underline #a61717',
    Color.WHITE: '#aaaaaa',
    Color.YELLOW: '#a61717',
}

COLORFUL_STYLE = {
    Color.BACKGROUND_BLACK: 'bg:#000',
    Color.BACKGROUND_BLUE: 'bg:#00C',
    Color.BACKGROUND_CYAN: 'bg:#0e84b5',
    Color.BACKGROUND_GREEN: 'bg:#00A000',
    Color.BACKGROUND_INTENSE_BLACK: 'bg:#555',
    Color.BACKGROUND_INTENSE_BLUE: 'bg:#33B',
    Color.BACKGROUND_INTENSE_CYAN: 'bg:#bbbbbb',
    Color.BACKGROUND_INTENSE_GREEN: 'bg:#888',
    Color.BACKGROUND_INTENSE_PURPLE: 'bg:#FAA',
    Color.BACKGROUND_INTENSE_RED: 'bg:#D42',
    Color.BACKGROUND_INTENSE_WHITE: 'bg:#fff0ff',
    Color.BACKGROUND_INTENSE_YELLOW: 'bg:#FAA',
    Color.BACKGROUND_PURPLE: 'bg:#800080',
    Color.BACKGROUND_RED: 'bg:#A00000',
    Color.BACKGROUND_WHITE: 'bg:#bbbbbb',
    Color.BACKGROUND_YELLOW: 'bg:#A60',
    Color.BLACK: '#000',
    Color.BLUE: '#00C',
    Color.BOLD_BLACK: 'bold #000',
    Color.BOLD_BLUE: 'bold #00C',
    Color.BOLD_CYAN: 'bold #0e84b5',
    Color.BOLD_GREEN: 'bold #00A000',
    Color.BOLD_INTENSE_BLACK: 'bold #555',
    Color.BOLD_INTENSE_BLUE: 'bold #33B',
    Color.BOLD_INTENSE_CYAN: 'bold #bbbbbb',
    Color.BOLD_INTENSE_GREEN: 'bold #888',
    Color.BOLD_INTENSE_PURPLE: 'bold #FAA',
    Color.BOLD_INTENSE_RED: 'bold #D42',
    Color.BOLD_INTENSE_WHITE: 'bold #fff0ff',
    Color.BOLD_INTENSE_YELLOW: 'bold #FAA',
    Color.BOLD_PURPLE: 'bold #800080',
    Color.BOLD_RED: 'bold #A00000',
    Color.BOLD_UNDERLINE_BLACK: 'bold underline #000',
    Color.BOLD_UNDERLINE_BLUE: 'bold underline #00C',
    Color.BOLD_UNDERLINE_CYAN: 'bold underline #0e84b5',
    Color.BOLD_UNDERLINE_GREEN: 'bold underline #00A000',
    Color.BOLD_UNDERLINE_INTENSE_BLACK: 'bold underline #555',
    Color.BOLD_UNDERLINE_INTENSE_BLUE: 'bold underline #33B',
    Color.BOLD_UNDERLINE_INTENSE_CYAN: 'bold underline #bbbbbb',
    Color.BOLD_UNDERLINE_INTENSE_GREEN: 'bold underline #888',
    Color.BOLD_UNDERLINE_INTENSE_PURPLE: 'bold underline #FAA',
    Color.BOLD_UNDERLINE_INTENSE_RED: 'bold underline #D42',
    Color.BOLD_UNDERLINE_INTENSE_WHITE: 'bold underline #fff0ff',
    Color.BOLD_UNDERLINE_INTENSE_YELLOW: 'bold underline #FAA',
    Color.BOLD_UNDERLINE_PURPLE: 'bold underline #800080',
    Color.BOLD_UNDERLINE_RED: 'bold underline #A00000',
    Color.BOLD_UNDERLINE_WHITE: 'bold underline #bbbbbb',
    Color.BOLD_UNDERLINE_YELLOW: 'bold underline #A60',
    Color.BOLD_WHITE: 'bold #bbbbbb',
    Color.BOLD_YELLOW: 'bold #A60',
    Color.CYAN: '#0e84b5',
    Color.GREEN: '#00A000',
    Color.INTENSE_BLACK: '#555',
    Color.INTENSE_BLUE: '#33B',
    Color.INTENSE_CYAN: '#bbbbbb',
    Color.INTENSE_GREEN: '#888',
    Color.INTENSE_PURPLE: '#FAA',
    Color.INTENSE_RED: '#D42',
    Color.INTENSE_WHITE: '#fff0ff',
    Color.INTENSE_YELLOW: '#FAA',
    Color.NO_COLOR: 'noinherit',
    Color.PURPLE: '#800080',
    Color.RED: '#A00000',
    Color.UNDERLINE_BLACK: 'underline #000',
    Color.UNDERLINE_BLUE: 'underline #00C',
    Color.UNDERLINE_CYAN: 'underline #0e84b5',
    Color.UNDERLINE_GREEN: 'underline #00A000',
    Color.UNDERLINE_INTENSE_BLACK: 'underline #555',
    Color.UNDERLINE_INTENSE_BLUE: 'underline #33B',
    Color.UNDERLINE_INTENSE_CYAN: 'underline #bbbbbb',
    Color.UNDERLINE_INTENSE_GREEN: 'underline #888',
    Color.UNDERLINE_INTENSE_PURPLE: 'underline #FAA',
    Color.UNDERLINE_INTENSE_RED: 'underline #D42',
    Color.UNDERLINE_INTENSE_WHITE: 'underline #fff0ff',
    Color.UNDERLINE_INTENSE_YELLOW: 'underline #FAA',
    Color.UNDERLINE_PURPLE: 'underline #800080',
    Color.UNDERLINE_RED: 'underline #A00000',
    Color.UNDERLINE_WHITE: 'underline #bbbbbb',
    Color.UNDERLINE_YELLOW: 'underline #A60',
    Color.WHITE: '#bbbbbb',
    Color.YELLOW: '#A60',
}

EMACS_STYLE = {
    Color.BACKGROUND_BLACK: 'bg:#008000',
    Color.BACKGROUND_BLUE: 'bg:#000080',
    Color.BACKGROUND_CYAN: 'bg:#04D',
    Color.BACKGROUND_GREEN: 'bg:#00A000',
    Color.BACKGROUND_INTENSE_BLACK: 'bg:#666666',
    Color.BACKGROUND_INTENSE_BLUE: 'bg:#04D',
    Color.BACKGROUND_INTENSE_CYAN: 'bg:#bbbbbb',
    Color.BACKGROUND_INTENSE_GREEN: 'bg:#00BB00',
    Color.BACKGROUND_INTENSE_PURPLE: 'bg:#AA22FF',
    Color.BACKGROUND_INTENSE_RED: 'bg:#D2413A',
    Color.BACKGROUND_INTENSE_WHITE: 'bg:#bbbbbb',
    Color.BACKGROUND_INTENSE_YELLOW: 'bg:#bbbbbb',
    Color.BACKGROUND_PURPLE: 'bg:#800080',
    Color.BACKGROUND_RED: 'bg:#A00000',
    Color.BACKGROUND_WHITE: 'bg:#bbbbbb',
    Color.BACKGROUND_YELLOW: 'bg:#BB6622',
    Color.BLACK: '#008000',
    Color.BLUE: '#000080',
    Color.BOLD_BLACK: 'bold #008000',
    Color.BOLD_BLUE: 'bold #000080',
    Color.BOLD_CYAN: 'bold #04D',
    Color.BOLD_GREEN: 'bold #00A000',
    Color.BOLD_INTENSE_BLACK: 'bold #666666',
    Color.BOLD_INTENSE_BLUE: 'bold #04D',
    Color.BOLD_INTENSE_CYAN: 'bold #bbbbbb',
    Color.BOLD_INTENSE_GREEN: 'bold #00BB00',
    Color.BOLD_INTENSE_PURPLE: 'bold #AA22FF',
    Color.BOLD_INTENSE_RED: 'bold #D2413A',
    Color.BOLD_INTENSE_WHITE: 'bold #bbbbbb',
    Color.BOLD_INTENSE_YELLOW: 'bold #bbbbbb',
    Color.BOLD_PURPLE: 'bold #800080',
    Color.BOLD_RED: 'bold #A00000',
    Color.BOLD_UNDERLINE_BLACK: 'bold underline #008000',
    Color.BOLD_UNDERLINE_BLUE: 'bold underline #000080',
    Color.BOLD_UNDERLINE_CYAN: 'bold underline #04D',
    Color.BOLD_UNDERLINE_GREEN: 'bold underline #00A000',
    Color.BOLD_UNDERLINE_INTENSE_BLACK: 'bold underline #666666',
    Color.BOLD_UNDERLINE_INTENSE_BLUE: 'bold underline #04D',
    Color.BOLD_UNDERLINE_INTENSE_CYAN: 'bold underline #bbbbbb',
    Color.BOLD_UNDERLINE_INTENSE_GREEN: 'bold underline #00BB00',
    Color.BOLD_UNDERLINE_INTENSE_PURPLE: 'bold underline #AA22FF',
    Color.BOLD_UNDERLINE_INTENSE_RED: 'bold underline #D2413A',
    Color.BOLD_UNDERLINE_INTENSE_WHITE: 'bold underline #bbbbbb',
    Color.BOLD_UNDERLINE_INTENSE_YELLOW: 'bold underline #bbbbbb',
    Color.BOLD_UNDERLINE_PURPLE: 'bold underline #800080',
    Color.BOLD_UNDERLINE_RED: 'bold underline #A00000',
    Color.BOLD_UNDERLINE_WHITE: 'bold underline #bbbbbb',
    Color.BOLD_UNDERLINE_YELLOW: 'bold underline #BB6622',
    Color.BOLD_WHITE: 'bold #bbbbbb',
    Color.BOLD_YELLOW: 'bold #BB6622',
    Color.CYAN: '#04D',
    Color.GREEN: '#00A000',
    Color.INTENSE_BLACK: '#666666',
    Color.INTENSE_BLUE: '#04D',
    Color.INTENSE_CYAN: '#bbbbbb',
    Color.INTENSE_GREEN: '#00BB00',
    Color.INTENSE_PURPLE: '#AA22FF',
    Color.INTENSE_RED: '#D2413A',
    Color.INTENSE_WHITE: '#bbbbbb',
    Color.INTENSE_YELLOW: '#bbbbbb',
    Color.NO_COLOR: 'noinherit',
    Color.PURPLE: '#800080',
    Color.RED: '#A00000',
    Color.UNDERLINE_BLACK: 'underline #008000',
    Color.UNDERLINE_BLUE: 'underline #000080',
    Color.UNDERLINE_CYAN: 'underline #04D',
    Color.UNDERLINE_GREEN: 'underline #00A000',
    Color.UNDERLINE_INTENSE_BLACK: 'underline #666666',
    Color.UNDERLINE_INTENSE_BLUE: 'underline #04D',
    Color.UNDERLINE_INTENSE_CYAN: 'underline #bbbbbb',
    Color.UNDERLINE_INTENSE_GREEN: 'underline #00BB00',
    Color.UNDERLINE_INTENSE_PURPLE: 'underline #AA22FF',
    Color.UNDERLINE_INTENSE_RED: 'underline #D2413A',
    Color.UNDERLINE_INTENSE_WHITE: 'underline #bbbbbb',
    Color.UNDERLINE_INTENSE_YELLOW: 'underline #bbbbbb',
    Color.UNDERLINE_PURPLE: 'underline #800080',
    Color.UNDERLINE_RED: 'underline #A00000',
    Color.UNDERLINE_WHITE: 'underline #bbbbbb',
    Color.UNDERLINE_YELLOW: 'underline #BB6622',
    Color.WHITE: '#bbbbbb',
    Color.YELLOW: '#BB6622',
}

FRIENDLY_STYLE = {
    Color.BACKGROUND_BLACK: 'bg:#007020',
    Color.BACKGROUND_BLUE: 'bg:#000080',
    Color.BACKGROUND_CYAN: 'bg:#0e84b5',
    Color.BACKGROUND_GREEN: 'bg:#00A000',
    Color.BACKGROUND_INTENSE_BLACK: 'bg:#555555',
    Color.BACKGROUND_INTENSE_BLUE: 'bg:#70a0d0',
    Color.BACKGROUND_INTENSE_CYAN: 'bg:#60add5',
    Color.BACKGROUND_INTENSE_GREEN: 'bg:#40a070',
    Color.BACKGROUND_INTENSE_PURPLE: 'bg:#bb60d5',
    Color.BACKGROUND_INTENSE_RED: 'bg:#d55537',
    Color.BACKGROUND_INTENSE_WHITE: 'bg:#fff0f0',
    Color.BACKGROUND_INTENSE_YELLOW: 'bg:#bbbbbb',
    Color.BACKGROUND_PURPLE: 'bg:#800080',
    Color.BACKGROUND_RED: 'bg:#A00000',
    Color.BACKGROUND_WHITE: 'bg:#bbbbbb',
    Color.BACKGROUND_YELLOW: 'bg:#c65d09',
    Color.BLACK: '#007020',
    Color.BLUE: '#000080',
    Color.BOLD_BLACK: 'bold #007020',
    Color.BOLD_BLUE: 'bold #000080',
    Color.BOLD_CYAN: 'bold #0e84b5',
    Color.BOLD_GREEN: 'bold #00A000',
    Color.BOLD_INTENSE_BLACK: 'bold #555555',
    Color.BOLD_INTENSE_BLUE: 'bold #70a0d0',
    Color.BOLD_INTENSE_CYAN: 'bold #60add5',
    Color.BOLD_INTENSE_GREEN: 'bold #40a070',
    Color.BOLD_INTENSE_PURPLE: 'bold #bb60d5',
    Color.BOLD_INTENSE_RED: 'bold #d55537',
    Color.BOLD_INTENSE_WHITE: 'bold #fff0f0',
    Color.BOLD_INTENSE_YELLOW: 'bold #bbbbbb',
    Color.BOLD_PURPLE: 'bold #800080',
    Color.BOLD_RED: 'bold #A00000',
    Color.BOLD_UNDERLINE_BLACK: 'bold underline #007020',
    Color.BOLD_UNDERLINE_BLUE: 'bold underline #000080',
    Color.BOLD_UNDERLINE_CYAN: 'bold underline #0e84b5',
    Color.BOLD_UNDERLINE_GREEN: 'bold underline #00A000',
    Color.BOLD_UNDERLINE_INTENSE_BLACK: 'bold underline #555555',
    Color.BOLD_UNDERLINE_INTENSE_BLUE: 'bold underline #70a0d0',
    Color.BOLD_UNDERLINE_INTENSE_CYAN: 'bold underline #60add5',
    Color.BOLD_UNDERLINE_INTENSE_GREEN: 'bold underline #40a070',
    Color.BOLD_UNDERLINE_INTENSE_PURPLE: 'bold underline #bb60d5',
    Color.BOLD_UNDERLINE_INTENSE_RED: 'bold underline #d55537',
    Color.BOLD_UNDERLINE_INTENSE_WHITE: 'bold underline #fff0f0',
    Color.BOLD_UNDERLINE_INTENSE_YELLOW: 'bold underline #bbbbbb',
    Color.BOLD_UNDERLINE_PURPLE: 'bold underline #800080',
    Color.BOLD_UNDERLINE_RED: 'bold underline #A00000',
    Color.BOLD_UNDERLINE_WHITE: 'bold underline #bbbbbb',
    Color.BOLD_UNDERLINE_YELLOW: 'bold underline #c65d09',
    Color.BOLD_WHITE: 'bold #bbbbbb',
    Color.BOLD_YELLOW: 'bold #c65d09',
    Color.CYAN: '#0e84b5',
    Color.GREEN: '#00A000',
    Color.INTENSE_BLACK: '#555555',
    Color.INTENSE_BLUE: '#70a0d0',
    Color.INTENSE_CYAN: '#60add5',
    Color.INTENSE_GREEN: '#40a070',
    Color.INTENSE_PURPLE: '#bb60d5',
    Color.INTENSE_RED: '#d55537',
    Color.INTENSE_WHITE: '#fff0f0',
    Color.INTENSE_YELLOW: '#bbbbbb',
    Color.NO_COLOR: 'noinherit',
    Color.PURPLE: '#800080',
    Color.RED: '#A00000',
    Color.UNDERLINE_BLACK: 'underline #007020',
    Color.UNDERLINE_BLUE: 'underline #000080',
    Color.UNDERLINE_CYAN: 'underline #0e84b5',
    Color.UNDERLINE_GREEN: 'underline #00A000',
    Color.UNDERLINE_INTENSE_BLACK: 'underline #555555',
    Color.UNDERLINE_INTENSE_BLUE: 'underline #70a0d0',
    Color.UNDERLINE_INTENSE_CYAN: 'underline #60add5',
    Color.UNDERLINE_INTENSE_GREEN: 'underline #40a070',
    Color.UNDERLINE_INTENSE_PURPLE: 'underline #bb60d5',
    Color.UNDERLINE_INTENSE_RED: 'underline #d55537',
    Color.UNDERLINE_INTENSE_WHITE: 'underline #fff0f0',
    Color.UNDERLINE_INTENSE_YELLOW: 'underline #bbbbbb',
    Color.UNDERLINE_PURPLE: 'underline #800080',
    Color.UNDERLINE_RED: 'underline #A00000',
    Color.UNDERLINE_WHITE: 'underline #bbbbbb',
    Color.UNDERLINE_YELLOW: 'underline #c65d09',
    Color.WHITE: '#bbbbbb',
    Color.YELLOW: '#c65d09',
}

FRUITY_STYLE = {
    Color.BACKGROUND_BLACK: 'bg:#0f140f',
    Color.BACKGROUND_BLUE: 'bg:#0086d2',
    Color.BACKGROUND_CYAN: 'bg:#0086d2',
    Color.BACKGROUND_GREEN: 'bg:#008800',
    Color.BACKGROUND_INTENSE_BLACK: 'bg:#444444',
    Color.BACKGROUND_INTENSE_BLUE: 'bg:#0086f7',
    Color.BACKGROUND_INTENSE_CYAN: 'bg:#0086f7',
    Color.BACKGROUND_INTENSE_GREEN: 'bg:#888888',
    Color.BACKGROUND_INTENSE_PURPLE: 'bg:#ff0086',
    Color.BACKGROUND_INTENSE_RED: 'bg:#fb660a',
    Color.BACKGROUND_INTENSE_WHITE: 'bg:#ffffff',
    Color.BACKGROUND_INTENSE_YELLOW: 'bg:#cdcaa9',
    Color.BACKGROUND_PURPLE: 'bg:#ff0086',
    Color.BACKGROUND_RED: 'bg:#ff0007',
    Color.BACKGROUND_WHITE: 'bg:#cdcaa9',
    Color.BACKGROUND_YELLOW: 'bg:#fb660a',
    Color.BLACK: '#0f140f',
    Color.BLUE: '#0086d2',
    Color.BOLD_BLACK: 'bold #0f140f',
    Color.BOLD_BLUE: 'bold #0086d2',
    Color.BOLD_CYAN: 'bold #0086d2',
    Color.BOLD_GREEN: 'bold #008800',
    Color.BOLD_INTENSE_BLACK: 'bold #444444',
    Color.BOLD_INTENSE_BLUE: 'bold #0086f7',
    Color.BOLD_INTENSE_CYAN: 'bold #0086f7',
    Color.BOLD_INTENSE_GREEN: 'bold #888888',
    Color.BOLD_INTENSE_PURPLE: 'bold #ff0086',
    Color.BOLD_INTENSE_RED: 'bold #fb660a',
    Color.BOLD_INTENSE_WHITE: 'bold #ffffff',
    Color.BOLD_INTENSE_YELLOW: 'bold #cdcaa9',
    Color.BOLD_PURPLE: 'bold #ff0086',
    Color.BOLD_RED: 'bold #ff0007',
    Color.BOLD_UNDERLINE_BLACK: 'bold underline #0f140f',
    Color.BOLD_UNDERLINE_BLUE: 'bold underline #0086d2',
    Color.BOLD_UNDERLINE_CYAN: 'bold underline #0086d2',
    Color.BOLD_UNDERLINE_GREEN: 'bold underline #008800',
    Color.BOLD_UNDERLINE_INTENSE_BLACK: 'bold underline #444444',
    Color.BOLD_UNDERLINE_INTENSE_BLUE: 'bold underline #0086f7',
    Color.BOLD_UNDERLINE_INTENSE_CYAN: 'bold underline #0086f7',
    Color.BOLD_UNDERLINE_INTENSE_GREEN: 'bold underline #888888',
    Color.BOLD_UNDERLINE_INTENSE_PURPLE: 'bold underline #ff0086',
    Color.BOLD_UNDERLINE_INTENSE_RED: 'bold underline #fb660a',
    Color.BOLD_UNDERLINE_INTENSE_WHITE: 'bold underline #ffffff',
    Color.BOLD_UNDERLINE_INTENSE_YELLOW: 'bold underline #cdcaa9',
    Color.BOLD_UNDERLINE_PURPLE: 'bold underline #ff0086',
    Color.BOLD_UNDERLINE_RED: 'bold underline #ff0007',
    Color.BOLD_UNDERLINE_WHITE: 'bold underline #cdcaa9',
    Color.BOLD_UNDERLINE_YELLOW: 'bold underline #fb660a',
    Color.BOLD_WHITE: 'bold #cdcaa9',
    Color.BOLD_YELLOW: 'bold #fb660a',
    Color.CYAN: '#0086d2',
    Color.GREEN: '#008800',
    Color.INTENSE_BLACK: '#444444',
    Color.INTENSE_BLUE: '#0086f7',
    Color.INTENSE_CYAN: '#0086f7',
    Color.INTENSE_GREEN: '#888888',
    Color.INTENSE_PURPLE: '#ff0086',
    Color.INTENSE_RED: '#fb660a',
    Color.INTENSE_WHITE: '#ffffff',
    Color.INTENSE_YELLOW: '#cdcaa9',
    Color.NO_COLOR: 'noinherit',
    Color.PURPLE: '#ff0086',
    Color.RED: '#ff0007',
    Color.UNDERLINE_BLACK: 'underline #0f140f',
    Color.UNDERLINE_BLUE: 'underline #0086d2',
    Color.UNDERLINE_CYAN: 'underline #0086d2',
    Color.UNDERLINE_GREEN: 'underline #008800',
    Color.UNDERLINE_INTENSE_BLACK: 'underline #444444',
    Color.UNDERLINE_INTENSE_BLUE: 'underline #0086f7',
    Color.UNDERLINE_INTENSE_CYAN: 'underline #0086f7',
    Color.UNDERLINE_INTENSE_GREEN: 'underline #888888',
    Color.UNDERLINE_INTENSE_PURPLE: 'underline #ff0086',
    Color.UNDERLINE_INTENSE_RED: 'underline #fb660a',
    Color.UNDERLINE_INTENSE_WHITE: 'underline #ffffff',
    Color.UNDERLINE_INTENSE_YELLOW: 'underline #cdcaa9',
    Color.UNDERLINE_PURPLE: 'underline #ff0086',
    Color.UNDERLINE_RED: 'underline #ff0007',
    Color.UNDERLINE_WHITE: 'underline #cdcaa9',
    Color.UNDERLINE_YELLOW: 'underline #fb660a',
    Color.WHITE: '#cdcaa9',
    Color.YELLOW: '#fb660a',
}

IGOR_STYLE = {
    Color.BACKGROUND_BLACK: 'bg:#009C00',
    Color.BACKGROUND_BLUE: 'bg:#0000FF',
    Color.BACKGROUND_CYAN: 'bg:#007575',
    Color.BACKGROUND_GREEN: 'bg:#009C00',
    Color.BACKGROUND_INTENSE_BLACK: 'bg:#007575',
    Color.BACKGROUND_INTENSE_BLUE: 'bg:#0000FF',
    Color.BACKGROUND_INTENSE_CYAN: 'bg:#007575',
    Color.BACKGROUND_INTENSE_GREEN: 'bg:#009C00',
    Color.BACKGROUND_INTENSE_PURPLE: 'bg:#CC00A3',
    Color.BACKGROUND_INTENSE_RED: 'bg:#C34E00',
    Color.BACKGROUND_INTENSE_WHITE: 'bg:#CC00A3',
    Color.BACKGROUND_INTENSE_YELLOW: 'bg:#C34E00',
    Color.BACKGROUND_PURPLE: 'bg:#CC00A3',
    Color.BACKGROUND_RED: 'bg:#C34E00',
    Color.BACKGROUND_WHITE: 'bg:#CC00A3',
    Color.BACKGROUND_YELLOW: 'bg:#C34E00',
    Color.BLACK: '#009C00',
    Color.BLUE: '#0000FF',
    Color.BOLD_BLACK: 'bold #009C00',
    Color.BOLD_BLUE: 'bold #0000FF',
    Color.BOLD_CYAN: 'bold #007575',
    Color.BOLD_GREEN: 'bold #009C00',
    Color.BOLD_INTENSE_BLACK: 'bold #007575',
    Color.BOLD_INTENSE_BLUE: 'bold #0000FF',
    Color.BOLD_INTENSE_CYAN: 'bold #007575',
    Color.BOLD_INTENSE_GREEN: 'bold #009C00',
    Color.BOLD_INTENSE_PURPLE: 'bold #CC00A3',
    Color.BOLD_INTENSE_RED: 'bold #C34E00',
    Color.BOLD_INTENSE_WHITE: 'bold #CC00A3',
    Color.BOLD_INTENSE_YELLOW: 'bold #C34E00',
    Color.BOLD_PURPLE: 'bold #CC00A3',
    Color.BOLD_RED: 'bold #C34E00',
    Color.BOLD_UNDERLINE_BLACK: 'bold underline #009C00',
    Color.BOLD_UNDERLINE_BLUE: 'bold underline #0000FF',
    Color.BOLD_UNDERLINE_CYAN: 'bold underline #007575',
    Color.BOLD_UNDERLINE_GREEN: 'bold underline #009C00',
    Color.BOLD_UNDERLINE_INTENSE_BLACK: 'bold underline #007575',
    Color.BOLD_UNDERLINE_INTENSE_BLUE: 'bold underline #0000FF',
    Color.BOLD_UNDERLINE_INTENSE_CYAN: 'bold underline #007575',
    Color.BOLD_UNDERLINE_INTENSE_GREEN: 'bold underline #009C00',
    Color.BOLD_UNDERLINE_INTENSE_PURPLE: 'bold underline #CC00A3',
    Color.BOLD_UNDERLINE_INTENSE_RED: 'bold underline #C34E00',
    Color.BOLD_UNDERLINE_INTENSE_WHITE: 'bold underline #CC00A3',
    Color.BOLD_UNDERLINE_INTENSE_YELLOW: 'bold underline #C34E00',
    Color.BOLD_UNDERLINE_PURPLE: 'bold underline #CC00A3',
    Color.BOLD_UNDERLINE_RED: 'bold underline #C34E00',
    Color.BOLD_UNDERLINE_WHITE: 'bold underline #CC00A3',
    Color.BOLD_UNDERLINE_YELLOW: 'bold underline #C34E00',
    Color.BOLD_WHITE: 'bold #CC00A3',
    Color.BOLD_YELLOW: 'bold #C34E00',
    Color.CYAN: '#007575',
    Color.GREEN: '#009C00',
    Color.INTENSE_BLACK: '#007575',
    Color.INTENSE_BLUE: '#0000FF',
    Color.INTENSE_CYAN: '#007575',
    Color.INTENSE_GREEN: '#009C00',
    Color.INTENSE_PURPLE: '#CC00A3',
    Color.INTENSE_RED: '#C34E00',
    Color.INTENSE_WHITE: '#CC00A3',
    Color.INTENSE_YELLOW: '#C34E00',
    Color.NO_COLOR: 'noinherit',
    Color.PURPLE: '#CC00A3',
    Color.RED: '#C34E00',
    Color.UNDERLINE_BLACK: 'underline #009C00',
    Color.UNDERLINE_BLUE: 'underline #0000FF',
    Color.UNDERLINE_CYAN: 'underline #007575',
    Color.UNDERLINE_GREEN: 'underline #009C00',
    Color.UNDERLINE_INTENSE_BLACK: 'underline #007575',
    Color.UNDERLINE_INTENSE_BLUE: 'underline #0000FF',
    Color.UNDERLINE_INTENSE_CYAN: 'underline #007575',
    Color.UNDERLINE_INTENSE_GREEN: 'underline #009C00',
    Color.UNDERLINE_INTENSE_PURPLE: 'underline #CC00A3',
    Color.UNDERLINE_INTENSE_RED: 'underline #C34E00',
    Color.UNDERLINE_INTENSE_WHITE: 'underline #CC00A3',
    Color.UNDERLINE_INTENSE_YELLOW: 'underline #C34E00',
    Color.UNDERLINE_PURPLE: 'underline #CC00A3',
    Color.UNDERLINE_RED: 'underline #C34E00',
    Color.UNDERLINE_WHITE: 'underline #CC00A3',
    Color.UNDERLINE_YELLOW: 'underline #C34E00',
    Color.WHITE: '#CC00A3',
    Color.YELLOW: '#C34E00',
}

LOVELACE_STYLE = {
    Color.BACKGROUND_BLACK: 'bg:#444444',
    Color.BACKGROUND_BLUE: 'bg:#2838b0',
    Color.BACKGROUND_CYAN: 'bg:#289870',
    Color.BACKGROUND_GREEN: 'bg:#388038',
    Color.BACKGROUND_INTENSE_BLACK: 'bg:#666666',
    Color.BACKGROUND_INTENSE_BLUE: 'bg:#2838b0',
    Color.BACKGROUND_INTENSE_CYAN: 'bg:#888888',
    Color.BACKGROUND_INTENSE_GREEN: 'bg:#289870',
    Color.BACKGROUND_INTENSE_PURPLE: 'bg:#a848a8',
    Color.BACKGROUND_INTENSE_RED: 'bg:#b83838',
    Color.BACKGROUND_INTENSE_WHITE: 'bg:#888888',
    Color.BACKGROUND_INTENSE_YELLOW: 'bg:#a89028',
    Color.BACKGROUND_PURPLE: 'bg:#a848a8',
    Color.BACKGROUND_RED: 'bg:#c02828',
    Color.BACKGROUND_WHITE: 'bg:#888888',
    Color.BACKGROUND_YELLOW: 'bg:#b85820',
    Color.BLACK: '#444444',
    Color.BLUE: '#2838b0',
    Color.BOLD_BLACK: 'bold #444444',
    Color.BOLD_BLUE: 'bold #2838b0',
    Color.BOLD_CYAN: 'bold #289870',
    Color.BOLD_GREEN: 'bold #388038',
    Color.BOLD_INTENSE_BLACK: 'bold #666666',
    Color.BOLD_INTENSE_BLUE: 'bold #2838b0',
    Color.BOLD_INTENSE_CYAN: 'bold #888888',
    Color.BOLD_INTENSE_GREEN: 'bold #289870',
    Color.BOLD_INTENSE_PURPLE: 'bold #a848a8',
    Color.BOLD_INTENSE_RED: 'bold #b83838',
    Color.BOLD_INTENSE_WHITE: 'bold #888888',
    Color.BOLD_INTENSE_YELLOW: 'bold #a89028',
    Color.BOLD_PURPLE: 'bold #a848a8',
    Color.BOLD_RED: 'bold #c02828',
    Color.BOLD_UNDERLINE_BLACK: 'bold underline #444444',
    Color.BOLD_UNDERLINE_BLUE: 'bold underline #2838b0',
    Color.BOLD_UNDERLINE_CYAN: 'bold underline #289870',
    Color.BOLD_UNDERLINE_GREEN: 'bold underline #388038',
    Color.BOLD_UNDERLINE_INTENSE_BLACK: 'bold underline #666666',
    Color.BOLD_UNDERLINE_INTENSE_BLUE: 'bold underline #2838b0',
    Color.BOLD_UNDERLINE_INTENSE_CYAN: 'bold underline #888888',
    Color.BOLD_UNDERLINE_INTENSE_GREEN: 'bold underline #289870',
    Color.BOLD_UNDERLINE_INTENSE_PURPLE: 'bold underline #a848a8',
    Color.BOLD_UNDERLINE_INTENSE_RED: 'bold underline #b83838',
    Color.BOLD_UNDERLINE_INTENSE_WHITE: 'bold underline #888888',
    Color.BOLD_UNDERLINE_INTENSE_YELLOW: 'bold underline #a89028',
    Color.BOLD_UNDERLINE_PURPLE: 'bold underline #a848a8',
    Color.BOLD_UNDERLINE_RED: 'bold underline #c02828',
    Color.BOLD_UNDERLINE_WHITE: 'bold underline #888888',
    Color.BOLD_UNDERLINE_YELLOW: 'bold underline #b85820',
    Color.BOLD_WHITE: 'bold #888888',
    Color.BOLD_YELLOW: 'bold #b85820',
    Color.CYAN: '#289870',
    Color.GREEN: '#388038',
    Color.INTENSE_BLACK: '#666666',
    Color.INTENSE_BLUE: '#2838b0',
    Color.INTENSE_CYAN: '#888888',
    Color.INTENSE_GREEN: '#289870',
    Color.INTENSE_PURPLE: '#a848a8',
    Color.INTENSE_RED: '#b83838',
    Color.INTENSE_WHITE: '#888888',
    Color.INTENSE_YELLOW: '#a89028',
    Color.NO_COLOR: 'noinherit',
    Color.PURPLE: '#a848a8',
    Color.RED: '#c02828',
    Color.UNDERLINE_BLACK: 'underline #444444',
    Color.UNDERLINE_BLUE: 'underline #2838b0',
    Color.UNDERLINE_CYAN: 'underline #289870',
    Color.UNDERLINE_GREEN: 'underline #388038',
    Color.UNDERLINE_INTENSE_BLACK: 'underline #666666',
    Color.UNDERLINE_INTENSE_BLUE: 'underline #2838b0',
    Color.UNDERLINE_INTENSE_CYAN: 'underline #888888',
    Color.UNDERLINE_INTENSE_GREEN: 'underline #289870',
    Color.UNDERLINE_INTENSE_PURPLE: 'underline #a848a8',
    Color.UNDERLINE_INTENSE_RED: 'underline #b83838',
    Color.UNDERLINE_INTENSE_WHITE: 'underline #888888',
    Color.UNDERLINE_INTENSE_YELLOW: 'underline #a89028',
    Color.UNDERLINE_PURPLE: 'underline #a848a8',
    Color.UNDERLINE_RED: 'underline #c02828',
    Color.UNDERLINE_WHITE: 'underline #888888',
    Color.UNDERLINE_YELLOW: 'underline #b85820',
    Color.WHITE: '#888888',
    Color.YELLOW: '#b85820',
}

MANNI_STYLE = {
    Color.BACKGROUND_BLACK: 'bg:#000000',
    Color.BACKGROUND_BLUE: 'bg:#000099',
    Color.BACKGROUND_CYAN: 'bg:#009999',
    Color.BACKGROUND_GREEN: 'bg:#00CC00',
    Color.BACKGROUND_INTENSE_BLACK: 'bg:#555555',
    Color.BACKGROUND_INTENSE_BLUE: 'bg:#9999FF',
    Color.BACKGROUND_INTENSE_CYAN: 'bg:#00CCFF',
    Color.BACKGROUND_INTENSE_GREEN: 'bg:#99CC66',
    Color.BACKGROUND_INTENSE_PURPLE: 'bg:#CC00FF',
    Color.BACKGROUND_INTENSE_RED: 'bg:#FF6600',
    Color.BACKGROUND_INTENSE_WHITE: 'bg:#FFCCCC',
    Color.BACKGROUND_INTENSE_YELLOW: 'bg:#FFCC33',
    Color.BACKGROUND_PURPLE: 'bg:#CC00FF',
    Color.BACKGROUND_RED: 'bg:#AA0000',
    Color.BACKGROUND_WHITE: 'bg:#AAAAAA',
    Color.BACKGROUND_YELLOW: 'bg:#CC3300',
    Color.BLACK: '#000000',
    Color.BLUE: '#000099',
    Color.BOLD_BLACK: 'bold #000000',
    Color.BOLD_BLUE: 'bold #000099',
    Color.BOLD_CYAN: 'bold #009999',
    Color.BOLD_GREEN: 'bold #00CC00',
    Color.BOLD_INTENSE_BLACK: 'bold #555555',
    Color.BOLD_INTENSE_BLUE: 'bold #9999FF',
    Color.BOLD_INTENSE_CYAN: 'bold #00CCFF',
    Color.BOLD_INTENSE_GREEN: 'bold #99CC66',
    Color.BOLD_INTENSE_PURPLE: 'bold #CC00FF',
    Color.BOLD_INTENSE_RED: 'bold #FF6600',
    Color.BOLD_INTENSE_WHITE: 'bold #FFCCCC',
    Color.BOLD_INTENSE_YELLOW: 'bold #FFCC33',
    Color.BOLD_PURPLE: 'bold #CC00FF',
    Color.BOLD_RED: 'bold #AA0000',
    Color.BOLD_UNDERLINE_BLACK: 'bold underline #000000',
    Color.BOLD_UNDERLINE_BLUE: 'bold underline #000099',
    Color.BOLD_UNDERLINE_CYAN: 'bold underline #009999',
    Color.BOLD_UNDERLINE_GREEN: 'bold underline #00CC00',
    Color.BOLD_UNDERLINE_INTENSE_BLACK: 'bold underline #555555',
    Color.BOLD_UNDERLINE_INTENSE_BLUE: 'bold underline #9999FF',
    Color.BOLD_UNDERLINE_INTENSE_CYAN: 'bold underline #00CCFF',
    Color.BOLD_UNDERLINE_INTENSE_GREEN: 'bold underline #99CC66',
    Color.BOLD_UNDERLINE_INTENSE_PURPLE: 'bold underline #CC00FF',
    Color.BOLD_UNDERLINE_INTENSE_RED: 'bold underline #FF6600',
    Color.BOLD_UNDERLINE_INTENSE_WHITE: 'bold underline #FFCCCC',
    Color.BOLD_UNDERLINE_INTENSE_YELLOW: 'bold underline #FFCC33',
    Color.BOLD_UNDERLINE_PURPLE: 'bold underline #CC00FF',
    Color.BOLD_UNDERLINE_RED: 'bold underline #AA0000',
    Color.BOLD_UNDERLINE_WHITE: 'bold underline #AAAAAA',
    Color.BOLD_UNDERLINE_YELLOW: 'bold underline #CC3300',
    Color.BOLD_WHITE: 'bold #AAAAAA',
    Color.BOLD_YELLOW: 'bold #CC3300',
    Color.CYAN: '#009999',
    Color.GREEN: '#00CC00',
    Color.INTENSE_BLACK: '#555555',
    Color.INTENSE_BLUE: '#9999FF',
    Color.INTENSE_CYAN: '#00CCFF',
    Color.INTENSE_GREEN: '#99CC66',
    Color.INTENSE_PURPLE: '#CC00FF',
    Color.INTENSE_RED: '#FF6600',
    Color.INTENSE_WHITE: '#FFCCCC',
    Color.INTENSE_YELLOW: '#FFCC33',
    Color.NO_COLOR: 'noinherit',
    Color.PURPLE: '#CC00FF',
    Color.RED: '#AA0000',
    Color.UNDERLINE_BLACK: 'underline #000000',
    Color.UNDERLINE_BLUE: 'underline #000099',
    Color.UNDERLINE_CYAN: 'underline #009999',
    Color.UNDERLINE_GREEN: 'underline #00CC00',
    Color.UNDERLINE_INTENSE_BLACK: 'underline #555555',
    Color.UNDERLINE_INTENSE_BLUE: 'underline #9999FF',
    Color.UNDERLINE_INTENSE_CYAN: 'underline #00CCFF',
    Color.UNDERLINE_INTENSE_GREEN: 'underline #99CC66',
    Color.UNDERLINE_INTENSE_PURPLE: 'underline #CC00FF',
    Color.UNDERLINE_INTENSE_RED: 'underline #FF6600',
    Color.UNDERLINE_INTENSE_WHITE: 'underline #FFCCCC',
    Color.UNDERLINE_INTENSE_YELLOW: 'underline #FFCC33',
    Color.UNDERLINE_PURPLE: 'underline #CC00FF',
    Color.UNDERLINE_RED: 'underline #AA0000',
    Color.UNDERLINE_WHITE: 'underline #AAAAAA',
    Color.UNDERLINE_YELLOW: 'underline #CC3300',
    Color.WHITE: '#AAAAAA',
    Color.YELLOW: '#CC3300',
}

MURPHY_STYLE = {
    Color.BACKGROUND_BLACK: 'bg:#000',
    Color.BACKGROUND_BLUE: 'bg:#000080',
    Color.BACKGROUND_CYAN: 'bg:#0e84b5',
    Color.BACKGROUND_GREEN: 'bg:#00A000',
    Color.BACKGROUND_INTENSE_BLACK: 'bg:#555',
    Color.BACKGROUND_INTENSE_BLUE: 'bg:#66f',
    Color.BACKGROUND_INTENSE_CYAN: 'bg:#5ed',
    Color.BACKGROUND_INTENSE_GREEN: 'bg:#5ed',
    Color.BACKGROUND_INTENSE_PURPLE: 'bg:#e9e',
    Color.BACKGROUND_INTENSE_RED: 'bg:#f84',
    Color.BACKGROUND_INTENSE_WHITE: 'bg:#eee',
    Color.BACKGROUND_INTENSE_YELLOW: 'bg:#fc8',
    Color.BACKGROUND_PURPLE: 'bg:#800080',
    Color.BACKGROUND_RED: 'bg:#A00000',
    Color.BACKGROUND_WHITE: 'bg:#bbbbbb',
    Color.BACKGROUND_YELLOW: 'bg:#c65d09',
    Color.BLACK: '#000',
    Color.BLUE: '#000080',
    Color.BOLD_BLACK: 'bold #000',
    Color.BOLD_BLUE: 'bold #000080',
    Color.BOLD_CYAN: 'bold #0e84b5',
    Color.BOLD_GREEN: 'bold #00A000',
    Color.BOLD_INTENSE_BLACK: 'bold #555',
    Color.BOLD_INTENSE_BLUE: 'bold #66f',
    Color.BOLD_INTENSE_CYAN: 'bold #5ed',
    Color.BOLD_INTENSE_GREEN: 'bold #5ed',
    Color.BOLD_INTENSE_PURPLE: 'bold #e9e',
    Color.BOLD_INTENSE_RED: 'bold #f84',
    Color.BOLD_INTENSE_WHITE: 'bold #eee',
    Color.BOLD_INTENSE_YELLOW: 'bold #fc8',
    Color.BOLD_PURPLE: 'bold #800080',
    Color.BOLD_RED: 'bold #A00000',
    Color.BOLD_UNDERLINE_BLACK: 'bold underline #000',
    Color.BOLD_UNDERLINE_BLUE: 'bold underline #000080',
    Color.BOLD_UNDERLINE_CYAN: 'bold underline #0e84b5',
    Color.BOLD_UNDERLINE_GREEN: 'bold underline #00A000',
    Color.BOLD_UNDERLINE_INTENSE_BLACK: 'bold underline #555',
    Color.BOLD_UNDERLINE_INTENSE_BLUE: 'bold underline #66f',
    Color.BOLD_UNDERLINE_INTENSE_CYAN: 'bold underline #5ed',
    Color.BOLD_UNDERLINE_INTENSE_GREEN: 'bold underline #5ed',
    Color.BOLD_UNDERLINE_INTENSE_PURPLE: 'bold underline #e9e',
    Color.BOLD_UNDERLINE_INTENSE_RED: 'bold underline #f84',
    Color.BOLD_UNDERLINE_INTENSE_WHITE: 'bold underline #eee',
    Color.BOLD_UNDERLINE_INTENSE_YELLOW: 'bold underline #fc8',
    Color.BOLD_UNDERLINE_PURPLE: 'bold underline #800080',
    Color.BOLD_UNDERLINE_RED: 'bold underline #A00000',
    Color.BOLD_UNDERLINE_WHITE: 'bold underline #bbbbbb',
    Color.BOLD_UNDERLINE_YELLOW: 'bold underline #c65d09',
    Color.BOLD_WHITE: 'bold #bbbbbb',
    Color.BOLD_YELLOW: 'bold #c65d09',
    Color.CYAN: '#0e84b5',
    Color.GREEN: '#00A000',
    Color.INTENSE_BLACK: '#555',
    Color.INTENSE_BLUE: '#66f',
    Color.INTENSE_CYAN: '#5ed',
    Color.INTENSE_GREEN: '#5ed',
    Color.INTENSE_PURPLE: '#e9e',
    Color.INTENSE_RED: '#f84',
    Color.INTENSE_WHITE: '#eee',
    Color.INTENSE_YELLOW: '#fc8',
    Color.NO_COLOR: 'noinherit',
    Color.PURPLE: '#800080',
    Color.RED: '#A00000',
    Color.UNDERLINE_BLACK: 'underline #000',
    Color.UNDERLINE_BLUE: 'underline #000080',
    Color.UNDERLINE_CYAN: 'underline #0e84b5',
    Color.UNDERLINE_GREEN: 'underline #00A000',
    Color.UNDERLINE_INTENSE_BLACK: 'underline #555',
    Color.UNDERLINE_INTENSE_BLUE: 'underline #66f',
    Color.UNDERLINE_INTENSE_CYAN: 'underline #5ed',
    Color.UNDERLINE_INTENSE_GREEN: 'underline #5ed',
    Color.UNDERLINE_INTENSE_PURPLE: 'underline #e9e',
    Color.UNDERLINE_INTENSE_RED: 'underline #f84',
    Color.UNDERLINE_INTENSE_WHITE: 'underline #eee',
    Color.UNDERLINE_INTENSE_YELLOW: 'underline #fc8',
    Color.UNDERLINE_PURPLE: 'underline #800080',
    Color.UNDERLINE_RED: 'underline #A00000',
    Color.UNDERLINE_WHITE: 'underline #bbbbbb',
    Color.UNDERLINE_YELLOW: 'underline #c65d09',
    Color.WHITE: '#bbbbbb',
    Color.YELLOW: '#c65d09',
}

NATIVE_STYLE = {
    Color.BACKGROUND_BLACK: 'bg:#520000',
    Color.BACKGROUND_BLUE: 'bg:#3677a9',
    Color.BACKGROUND_CYAN: 'bg:#24909d',
    Color.BACKGROUND_GREEN: 'bg:#589819',
    Color.BACKGROUND_INTENSE_BLACK: 'bg:#666666',
    Color.BACKGROUND_INTENSE_BLUE: 'bg:#447fcf',
    Color.BACKGROUND_INTENSE_CYAN: 'bg:#40ffff',
    Color.BACKGROUND_INTENSE_GREEN: 'bg:#6ab825',
    Color.BACKGROUND_INTENSE_PURPLE: 'bg:#e3d2d2',
    Color.BACKGROUND_INTENSE_RED: 'bg:#cd2828',
    Color.BACKGROUND_INTENSE_WHITE: 'bg:#ffffff',
    Color.BACKGROUND_INTENSE_YELLOW: 'bg:#ed9d13',
    Color.BACKGROUND_PURPLE: 'bg:#666666',
    Color.BACKGROUND_RED: 'bg:#a61717',
    Color.BACKGROUND_WHITE: 'bg:#aaaaaa',
    Color.BACKGROUND_YELLOW: 'bg:#a61717',
    Color.BLACK: '#520000',
    Color.BLUE: '#3677a9',
    Color.BOLD_BLACK: 'bold #520000',
    Color.BOLD_BLUE: 'bold #3677a9',
    Color.BOLD_CYAN: 'bold #24909d',
    Color.BOLD_GREEN: 'bold #589819',
    Color.BOLD_INTENSE_BLACK: 'bold #666666',
    Color.BOLD_INTENSE_BLUE: 'bold #447fcf',
    Color.BOLD_INTENSE_CYAN: 'bold #40ffff',
    Color.BOLD_INTENSE_GREEN: 'bold #6ab825',
    Color.BOLD_INTENSE_PURPLE: 'bold #e3d2d2',
    Color.BOLD_INTENSE_RED: 'bold #cd2828',
    Color.BOLD_INTENSE_WHITE: 'bold #ffffff',
    Color.BOLD_INTENSE_YELLOW: 'bold #ed9d13',
    Color.BOLD_PURPLE: 'bold #666666',
    Color.BOLD_RED: 'bold #a61717',
    Color.BOLD_UNDERLINE_BLACK: 'bold underline #520000',
    Color.BOLD_UNDERLINE_BLUE: 'bold underline #3677a9',
    Color.BOLD_UNDERLINE_CYAN: 'bold underline #24909d',
    Color.BOLD_UNDERLINE_GREEN: 'bold underline #589819',
    Color.BOLD_UNDERLINE_INTENSE_BLACK: 'bold underline #666666',
    Color.BOLD_UNDERLINE_INTENSE_BLUE: 'bold underline #447fcf',
    Color.BOLD_UNDERLINE_INTENSE_CYAN: 'bold underline #40ffff',
    Color.BOLD_UNDERLINE_INTENSE_GREEN: 'bold underline #6ab825',
    Color.BOLD_UNDERLINE_INTENSE_PURPLE: 'bold underline #e3d2d2',
    Color.BOLD_UNDERLINE_INTENSE_RED: 'bold underline #cd2828',
    Color.BOLD_UNDERLINE_INTENSE_WHITE: 'bold underline #ffffff',
    Color.BOLD_UNDERLINE_INTENSE_YELLOW: 'bold underline #ed9d13',
    Color.BOLD_UNDERLINE_PURPLE: 'bold underline #666666',
    Color.BOLD_UNDERLINE_RED: 'bold underline #a61717',
    Color.BOLD_UNDERLINE_WHITE: 'bold underline #aaaaaa',
    Color.BOLD_UNDERLINE_YELLOW: 'bold underline #a61717',
    Color.BOLD_WHITE: 'bold #aaaaaa',
    Color.BOLD_YELLOW: 'bold #a61717',
    Color.CYAN: '#24909d',
    Color.GREEN: '#589819',
    Color.INTENSE_BLACK: '#666666',
    Color.INTENSE_BLUE: '#447fcf',
    Color.INTENSE_CYAN: '#40ffff',
    Color.INTENSE_GREEN: '#6ab825',
    Color.INTENSE_PURPLE: '#e3d2d2',
    Color.INTENSE_RED: '#cd2828',
    Color.INTENSE_WHITE: '#ffffff',
    Color.INTENSE_YELLOW: '#ed9d13',
    Color.NO_COLOR: 'noinherit',
    Color.PURPLE: '#666666',
    Color.RED: '#a61717',
    Color.UNDERLINE_BLACK: 'underline #520000',
    Color.UNDERLINE_BLUE: 'underline #3677a9',
    Color.UNDERLINE_CYAN: 'underline #24909d',
    Color.UNDERLINE_GREEN: 'underline #589819',
    Color.UNDERLINE_INTENSE_BLACK: 'underline #666666',
    Color.UNDERLINE_INTENSE_BLUE: 'underline #447fcf',
    Color.UNDERLINE_INTENSE_CYAN: 'underline #40ffff',
    Color.UNDERLINE_INTENSE_GREEN: 'underline #6ab825',
    Color.UNDERLINE_INTENSE_PURPLE: 'underline #e3d2d2',
    Color.UNDERLINE_INTENSE_RED: 'underline #cd2828',
    Color.UNDERLINE_INTENSE_WHITE: 'underline #ffffff',
    Color.UNDERLINE_INTENSE_YELLOW: 'underline #ed9d13',
    Color.UNDERLINE_PURPLE: 'underline #666666',
    Color.UNDERLINE_RED: 'underline #a61717',
    Color.UNDERLINE_WHITE: 'underline #aaaaaa',
    Color.UNDERLINE_YELLOW: 'underline #a61717',
    Color.WHITE: '#aaaaaa',
    Color.YELLOW: '#a61717',
}

PARAISO_DARK_STYLE = {
    Color.BACKGROUND_BLACK: 'bg:#776e71',
    Color.BACKGROUND_BLUE: 'bg:#815ba4',
    Color.BACKGROUND_CYAN: 'bg:#06b6ef',
    Color.BACKGROUND_GREEN: 'bg:#48b685',
    Color.BACKGROUND_INTENSE_BLACK: 'bg:#776e71',
    Color.BACKGROUND_INTENSE_BLUE: 'bg:#815ba4',
    Color.BACKGROUND_INTENSE_CYAN: 'bg:#5bc4bf',
    Color.BACKGROUND_INTENSE_GREEN: 'bg:#48b685',
    Color.BACKGROUND_INTENSE_PURPLE: 'bg:#e7e9db',
    Color.BACKGROUND_INTENSE_RED: 'bg:#ef6155',
    Color.BACKGROUND_INTENSE_WHITE: 'bg:#e7e9db',
    Color.BACKGROUND_INTENSE_YELLOW: 'bg:#fec418',
    Color.BACKGROUND_PURPLE: 'bg:#815ba4',
    Color.BACKGROUND_RED: 'bg:#ef6155',
    Color.BACKGROUND_WHITE: 'bg:#5bc4bf',
    Color.BACKGROUND_YELLOW: 'bg:#f99b15',
    Color.BLACK: '#776e71',
    Color.BLUE: '#815ba4',
    Color.BOLD_BLACK: 'bold #776e71',
    Color.BOLD_BLUE: 'bold #815ba4',
    Color.BOLD_CYAN: 'bold #06b6ef',
    Color.BOLD_GREEN: 'bold #48b685',
    Color.BOLD_INTENSE_BLACK: 'bold #776e71',
    Color.BOLD_INTENSE_BLUE: 'bold #815ba4',
    Color.BOLD_INTENSE_CYAN: 'bold #5bc4bf',
    Color.BOLD_INTENSE_GREEN: 'bold #48b685',
    Color.BOLD_INTENSE_PURPLE: 'bold #e7e9db',
    Color.BOLD_INTENSE_RED: 'bold #ef6155',
    Color.BOLD_INTENSE_WHITE: 'bold #e7e9db',
    Color.BOLD_INTENSE_YELLOW: 'bold #fec418',
    Color.BOLD_PURPLE: 'bold #815ba4',
    Color.BOLD_RED: 'bold #ef6155',
    Color.BOLD_UNDERLINE_BLACK: 'bold underline #776e71',
    Color.BOLD_UNDERLINE_BLUE: 'bold underline #815ba4',
    Color.BOLD_UNDERLINE_CYAN: 'bold underline #06b6ef',
    Color.BOLD_UNDERLINE_GREEN: 'bold underline #48b685',
    Color.BOLD_UNDERLINE_INTENSE_BLACK: 'bold underline #776e71',
    Color.BOLD_UNDERLINE_INTENSE_BLUE: 'bold underline #815ba4',
    Color.BOLD_UNDERLINE_INTENSE_CYAN: 'bold underline #5bc4bf',
    Color.BOLD_UNDERLINE_INTENSE_GREEN: 'bold underline #48b685',
    Color.BOLD_UNDERLINE_INTENSE_PURPLE: 'bold underline #e7e9db',
    Color.BOLD_UNDERLINE_INTENSE_RED: 'bold underline #ef6155',
    Color.BOLD_UNDERLINE_INTENSE_WHITE: 'bold underline #e7e9db',
    Color.BOLD_UNDERLINE_INTENSE_YELLOW: 'bold underline #fec418',
    Color.BOLD_UNDERLINE_PURPLE: 'bold underline #815ba4',
    Color.BOLD_UNDERLINE_RED: 'bold underline #ef6155',
    Color.BOLD_UNDERLINE_WHITE: 'bold underline #5bc4bf',
    Color.BOLD_UNDERLINE_YELLOW: 'bold underline #f99b15',
    Color.BOLD_WHITE: 'bold #5bc4bf',
    Color.BOLD_YELLOW: 'bold #f99b15',
    Color.CYAN: '#06b6ef',
    Color.GREEN: '#48b685',
    Color.INTENSE_BLACK: '#776e71',
    Color.INTENSE_BLUE: '#815ba4',
    Color.INTENSE_CYAN: '#5bc4bf',
    Color.INTENSE_GREEN: '#48b685',
    Color.INTENSE_PURPLE: '#e7e9db',
    Color.INTENSE_RED: '#ef6155',
    Color.INTENSE_WHITE: '#e7e9db',
    Color.INTENSE_YELLOW: '#fec418',
    Color.NO_COLOR: 'noinherit',
    Color.PURPLE: '#815ba4',
    Color.RED: '#ef6155',
    Color.UNDERLINE_BLACK: 'underline #776e71',
    Color.UNDERLINE_BLUE: 'underline #815ba4',
    Color.UNDERLINE_CYAN: 'underline #06b6ef',
    Color.UNDERLINE_GREEN: 'underline #48b685',
    Color.UNDERLINE_INTENSE_BLACK: 'underline #776e71',
    Color.UNDERLINE_INTENSE_BLUE: 'underline #815ba4',
    Color.UNDERLINE_INTENSE_CYAN: 'underline #5bc4bf',
    Color.UNDERLINE_INTENSE_GREEN: 'underline #48b685',
    Color.UNDERLINE_INTENSE_PURPLE: 'underline #e7e9db',
    Color.UNDERLINE_INTENSE_RED: 'underline #ef6155',
    Color.UNDERLINE_INTENSE_WHITE: 'underline #e7e9db',
    Color.UNDERLINE_INTENSE_YELLOW: 'underline #fec418',
    Color.UNDERLINE_PURPLE: 'underline #815ba4',
    Color.UNDERLINE_RED: 'underline #ef6155',
    Color.UNDERLINE_WHITE: 'underline #5bc4bf',
    Color.UNDERLINE_YELLOW: 'underline #f99b15',
    Color.WHITE: '#5bc4bf',
    Color.YELLOW: '#f99b15',
}

PARAISO_LIGHT_STYLE = {
    Color.BACKGROUND_BLACK: 'bg:#2f1e2e',
    Color.BACKGROUND_BLUE: 'bg:#2f1e2e',
    Color.BACKGROUND_CYAN: 'bg:#06b6ef',
    Color.BACKGROUND_GREEN: 'bg:#48b685',
    Color.BACKGROUND_INTENSE_BLACK: 'bg:#2f1e2e',
    Color.BACKGROUND_INTENSE_BLUE: 'bg:#815ba4',
    Color.BACKGROUND_INTENSE_CYAN: 'bg:#5bc4bf',
    Color.BACKGROUND_INTENSE_GREEN: 'bg:#48b685',
    Color.BACKGROUND_INTENSE_PURPLE: 'bg:#815ba4',
    Color.BACKGROUND_INTENSE_RED: 'bg:#ef6155',
    Color.BACKGROUND_INTENSE_WHITE: 'bg:#5bc4bf',
    Color.BACKGROUND_INTENSE_YELLOW: 'bg:#fec418',
    Color.BACKGROUND_PURPLE: 'bg:#815ba4',
    Color.BACKGROUND_RED: 'bg:#2f1e2e',
    Color.BACKGROUND_WHITE: 'bg:#8d8687',
    Color.BACKGROUND_YELLOW: 'bg:#f99b15',
    Color.BLACK: '#2f1e2e',
    Color.BLUE: '#2f1e2e',
    Color.BOLD_BLACK: 'bold #2f1e2e',
    Color.BOLD_BLUE: 'bold #2f1e2e',
    Color.BOLD_CYAN: 'bold #06b6ef',
    Color.BOLD_GREEN: 'bold #48b685',
    Color.BOLD_INTENSE_BLACK: 'bold #2f1e2e',
    Color.BOLD_INTENSE_BLUE: 'bold #815ba4',
    Color.BOLD_INTENSE_CYAN: 'bold #5bc4bf',
    Color.BOLD_INTENSE_GREEN: 'bold #48b685',
    Color.BOLD_INTENSE_PURPLE: 'bold #815ba4',
    Color.BOLD_INTENSE_RED: 'bold #ef6155',
    Color.BOLD_INTENSE_WHITE: 'bold #5bc4bf',
    Color.BOLD_INTENSE_YELLOW: 'bold #fec418',
    Color.BOLD_PURPLE: 'bold #815ba4',
    Color.BOLD_RED: 'bold #2f1e2e',
    Color.BOLD_UNDERLINE_BLACK: 'bold underline #2f1e2e',
    Color.BOLD_UNDERLINE_BLUE: 'bold underline #2f1e2e',
    Color.BOLD_UNDERLINE_CYAN: 'bold underline #06b6ef',
    Color.BOLD_UNDERLINE_GREEN: 'bold underline #48b685',
    Color.BOLD_UNDERLINE_INTENSE_BLACK: 'bold underline #2f1e2e',
    Color.BOLD_UNDERLINE_INTENSE_BLUE: 'bold underline #815ba4',
    Color.BOLD_UNDERLINE_INTENSE_CYAN: 'bold underline #5bc4bf',
    Color.BOLD_UNDERLINE_INTENSE_GREEN: 'bold underline #48b685',
    Color.BOLD_UNDERLINE_INTENSE_PURPLE: 'bold underline #815ba4',
    Color.BOLD_UNDERLINE_INTENSE_RED: 'bold underline #ef6155',
    Color.BOLD_UNDERLINE_INTENSE_WHITE: 'bold underline #5bc4bf',
    Color.BOLD_UNDERLINE_INTENSE_YELLOW: 'bold underline #fec418',
    Color.BOLD_UNDERLINE_PURPLE: 'bold underline #815ba4',
    Color.BOLD_UNDERLINE_RED: 'bold underline #2f1e2e',
    Color.BOLD_UNDERLINE_WHITE: 'bold underline #8d8687',
    Color.BOLD_UNDERLINE_YELLOW: 'bold underline #f99b15',
    Color.BOLD_WHITE: 'bold #8d8687',
    Color.BOLD_YELLOW: 'bold #f99b15',
    Color.CYAN: '#06b6ef',
    Color.GREEN: '#48b685',
    Color.INTENSE_BLACK: '#2f1e2e',
    Color.INTENSE_BLUE: '#815ba4',
    Color.INTENSE_CYAN: '#5bc4bf',
    Color.INTENSE_GREEN: '#48b685',
    Color.INTENSE_PURPLE: '#815ba4',
    Color.INTENSE_RED: '#ef6155',
    Color.INTENSE_WHITE: '#5bc4bf',
    Color.INTENSE_YELLOW: '#fec418',
    Color.NO_COLOR: 'noinherit',
    Color.PURPLE: '#815ba4',
    Color.RED: '#2f1e2e',
    Color.UNDERLINE_BLACK: 'underline #2f1e2e',
    Color.UNDERLINE_BLUE: 'underline #2f1e2e',
    Color.UNDERLINE_CYAN: 'underline #06b6ef',
    Color.UNDERLINE_GREEN: 'underline #48b685',
    Color.UNDERLINE_INTENSE_BLACK: 'underline #2f1e2e',
    Color.UNDERLINE_INTENSE_BLUE: 'underline #815ba4',
    Color.UNDERLINE_INTENSE_CYAN: 'underline #5bc4bf',
    Color.UNDERLINE_INTENSE_GREEN: 'underline #48b685',
    Color.UNDERLINE_INTENSE_PURPLE: 'underline #815ba4',
    Color.UNDERLINE_INTENSE_RED: 'underline #ef6155',
    Color.UNDERLINE_INTENSE_WHITE: 'underline #5bc4bf',
    Color.UNDERLINE_INTENSE_YELLOW: 'underline #fec418',
    Color.UNDERLINE_PURPLE: 'underline #815ba4',
    Color.UNDERLINE_RED: 'underline #2f1e2e',
    Color.UNDERLINE_WHITE: 'underline #8d8687',
    Color.UNDERLINE_YELLOW: 'underline #f99b15',
    Color.WHITE: '#8d8687',
    Color.YELLOW: '#f99b15',
}

PASTIE_STYLE = {
    Color.BACKGROUND_BLACK: 'bg:#000000',
    Color.BACKGROUND_BLUE: 'bg:#0000DD',
    Color.BACKGROUND_CYAN: 'bg:#0066bb',
    Color.BACKGROUND_GREEN: 'bg:#008800',
    Color.BACKGROUND_INTENSE_BLACK: 'bg:#555555',
    Color.BACKGROUND_INTENSE_BLUE: 'bg:#3333bb',
    Color.BACKGROUND_INTENSE_CYAN: 'bg:#ddffdd',
    Color.BACKGROUND_INTENSE_GREEN: 'bg:#22bb22',
    Color.BACKGROUND_INTENSE_PURPLE: 'bg:#e3d2d2',
    Color.BACKGROUND_INTENSE_RED: 'bg:#dd7700',
    Color.BACKGROUND_INTENSE_WHITE: 'bg:#fff0ff',
    Color.BACKGROUND_INTENSE_YELLOW: 'bg:#e3d2d2',
    Color.BACKGROUND_PURPLE: 'bg:#bb0066',
    Color.BACKGROUND_RED: 'bg:#aa0000',
    Color.BACKGROUND_WHITE: 'bg:#bbbbbb',
    Color.BACKGROUND_YELLOW: 'bg:#aa6600',
    Color.BLACK: '#000000',
    Color.BLUE: '#0000DD',
    Color.BOLD_BLACK: 'bold #000000',
    Color.BOLD_BLUE: 'bold #0000DD',
    Color.BOLD_CYAN: 'bold #0066bb',
    Color.BOLD_GREEN: 'bold #008800',
    Color.BOLD_INTENSE_BLACK: 'bold #555555',
    Color.BOLD_INTENSE_BLUE: 'bold #3333bb',
    Color.BOLD_INTENSE_CYAN: 'bold #ddffdd',
    Color.BOLD_INTENSE_GREEN: 'bold #22bb22',
    Color.BOLD_INTENSE_PURPLE: 'bold #e3d2d2',
    Color.BOLD_INTENSE_RED: 'bold #dd7700',
    Color.BOLD_INTENSE_WHITE: 'bold #fff0ff',
    Color.BOLD_INTENSE_YELLOW: 'bold #e3d2d2',
    Color.BOLD_PURPLE: 'bold #bb0066',
    Color.BOLD_RED: 'bold #aa0000',
    Color.BOLD_UNDERLINE_BLACK: 'bold underline #000000',
    Color.BOLD_UNDERLINE_BLUE: 'bold underline #0000DD',
    Color.BOLD_UNDERLINE_CYAN: 'bold underline #0066bb',
    Color.BOLD_UNDERLINE_GREEN: 'bold underline #008800',
    Color.BOLD_UNDERLINE_INTENSE_BLACK: 'bold underline #555555',
    Color.BOLD_UNDERLINE_INTENSE_BLUE: 'bold underline #3333bb',
    Color.BOLD_UNDERLINE_INTENSE_CYAN: 'bold underline #ddffdd',
    Color.BOLD_UNDERLINE_INTENSE_GREEN: 'bold underline #22bb22',
    Color.BOLD_UNDERLINE_INTENSE_PURPLE: 'bold underline #e3d2d2',
    Color.BOLD_UNDERLINE_INTENSE_RED: 'bold underline #dd7700',
    Color.BOLD_UNDERLINE_INTENSE_WHITE: 'bold underline #fff0ff',
    Color.BOLD_UNDERLINE_INTENSE_YELLOW: 'bold underline #e3d2d2',
    Color.BOLD_UNDERLINE_PURPLE: 'bold underline #bb0066',
    Color.BOLD_UNDERLINE_RED: 'bold underline #aa0000',
    Color.BOLD_UNDERLINE_WHITE: 'bold underline #bbbbbb',
    Color.BOLD_UNDERLINE_YELLOW: 'bold underline #aa6600',
    Color.BOLD_WHITE: 'bold #bbbbbb',
    Color.BOLD_YELLOW: 'bold #aa6600',
    Color.CYAN: '#0066bb',
    Color.GREEN: '#008800',
    Color.INTENSE_BLACK: '#555555',
    Color.INTENSE_BLUE: '#3333bb',
    Color.INTENSE_CYAN: '#ddffdd',
    Color.INTENSE_GREEN: '#22bb22',
    Color.INTENSE_PURPLE: '#e3d2d2',
    Color.INTENSE_RED: '#dd7700',
    Color.INTENSE_WHITE: '#fff0ff',
    Color.INTENSE_YELLOW: '#e3d2d2',
    Color.NO_COLOR: 'noinherit',
    Color.PURPLE: '#bb0066',
    Color.RED: '#aa0000',
    Color.UNDERLINE_BLACK: 'underline #000000',
    Color.UNDERLINE_BLUE: 'underline #0000DD',
    Color.UNDERLINE_CYAN: 'underline #0066bb',
    Color.UNDERLINE_GREEN: 'underline #008800',
    Color.UNDERLINE_INTENSE_BLACK: 'underline #555555',
    Color.UNDERLINE_INTENSE_BLUE: 'underline #3333bb',
    Color.UNDERLINE_INTENSE_CYAN: 'underline #ddffdd',
    Color.UNDERLINE_INTENSE_GREEN: 'underline #22bb22',
    Color.UNDERLINE_INTENSE_PURPLE: 'underline #e3d2d2',
    Color.UNDERLINE_INTENSE_RED: 'underline #dd7700',
    Color.UNDERLINE_INTENSE_WHITE: 'underline #fff0ff',
    Color.UNDERLINE_INTENSE_YELLOW: 'underline #e3d2d2',
    Color.UNDERLINE_PURPLE: 'underline #bb0066',
    Color.UNDERLINE_RED: 'underline #aa0000',
    Color.UNDERLINE_WHITE: 'underline #bbbbbb',
    Color.UNDERLINE_YELLOW: 'underline #aa6600',
    Color.WHITE: '#bbbbbb',
    Color.YELLOW: '#aa6600',
}

PERLDOC_STYLE = {
    Color.BACKGROUND_BLACK: 'bg:#000080',
    Color.BACKGROUND_BLUE: 'bg:#000080',
    Color.BACKGROUND_CYAN: 'bg:#1e889b',
    Color.BACKGROUND_GREEN: 'bg:#00aa00',
    Color.BACKGROUND_INTENSE_BLACK: 'bg:#555555',
    Color.BACKGROUND_INTENSE_BLUE: 'bg:#B452CD',
    Color.BACKGROUND_INTENSE_CYAN: 'bg:#bbbbbb',
    Color.BACKGROUND_INTENSE_GREEN: 'bg:#228B22',
    Color.BACKGROUND_INTENSE_PURPLE: 'bg:#B452CD',
    Color.BACKGROUND_INTENSE_RED: 'bg:#CD5555',
    Color.BACKGROUND_INTENSE_WHITE: 'bg:#e3d2d2',
    Color.BACKGROUND_INTENSE_YELLOW: 'bg:#e3d2d2',
    Color.BACKGROUND_PURPLE: 'bg:#8B008B',
    Color.BACKGROUND_RED: 'bg:#aa0000',
    Color.BACKGROUND_WHITE: 'bg:#a7a7a7',
    Color.BACKGROUND_YELLOW: 'bg:#cb6c20',
    Color.BLACK: '#000080',
    Color.BLUE: '#000080',
    Color.BOLD_BLACK: 'bold #000080',
    Color.BOLD_BLUE: 'bold #000080',
    Color.BOLD_CYAN: 'bold #1e889b',
    Color.BOLD_GREEN: 'bold #00aa00',
    Color.BOLD_INTENSE_BLACK: 'bold #555555',
    Color.BOLD_INTENSE_BLUE: 'bold #B452CD',
    Color.BOLD_INTENSE_CYAN: 'bold #bbbbbb',
    Color.BOLD_INTENSE_GREEN: 'bold #228B22',
    Color.BOLD_INTENSE_PURPLE: 'bold #B452CD',
    Color.BOLD_INTENSE_RED: 'bold #CD5555',
    Color.BOLD_INTENSE_WHITE: 'bold #e3d2d2',
    Color.BOLD_INTENSE_YELLOW: 'bold #e3d2d2',
    Color.BOLD_PURPLE: 'bold #8B008B',
    Color.BOLD_RED: 'bold #aa0000',
    Color.BOLD_UNDERLINE_BLACK: 'bold underline #000080',
    Color.BOLD_UNDERLINE_BLUE: 'bold underline #000080',
    Color.BOLD_UNDERLINE_CYAN: 'bold underline #1e889b',
    Color.BOLD_UNDERLINE_GREEN: 'bold underline #00aa00',
    Color.BOLD_UNDERLINE_INTENSE_BLACK: 'bold underline #555555',
    Color.BOLD_UNDERLINE_INTENSE_BLUE: 'bold underline #B452CD',
    Color.BOLD_UNDERLINE_INTENSE_CYAN: 'bold underline #bbbbbb',
    Color.BOLD_UNDERLINE_INTENSE_GREEN: 'bold underline #228B22',
    Color.BOLD_UNDERLINE_INTENSE_PURPLE: 'bold underline #B452CD',
    Color.BOLD_UNDERLINE_INTENSE_RED: 'bold underline #CD5555',
    Color.BOLD_UNDERLINE_INTENSE_WHITE: 'bold underline #e3d2d2',
    Color.BOLD_UNDERLINE_INTENSE_YELLOW: 'bold underline #e3d2d2',
    Color.BOLD_UNDERLINE_PURPLE: 'bold underline #8B008B',
    Color.BOLD_UNDERLINE_RED: 'bold underline #aa0000',
    Color.BOLD_UNDERLINE_WHITE: 'bold underline #a7a7a7',
    Color.BOLD_UNDERLINE_YELLOW: 'bold underline #cb6c20',
    Color.BOLD_WHITE: 'bold #a7a7a7',
    Color.BOLD_YELLOW: 'bold #cb6c20',
    Color.CYAN: '#1e889b',
    Color.GREEN: '#00aa00',
    Color.INTENSE_BLACK: '#555555',
    Color.INTENSE_BLUE: '#B452CD',
    Color.INTENSE_CYAN: '#bbbbbb',
    Color.INTENSE_GREEN: '#228B22',
    Color.INTENSE_PURPLE: '#B452CD',
    Color.INTENSE_RED: '#CD5555',
    Color.INTENSE_WHITE: '#e3d2d2',
    Color.INTENSE_YELLOW: '#e3d2d2',
    Color.NO_COLOR: 'noinherit',
    Color.PURPLE: '#8B008B',
    Color.RED: '#aa0000',
    Color.UNDERLINE_BLACK: 'underline #000080',
    Color.UNDERLINE_BLUE: 'underline #000080',
    Color.UNDERLINE_CYAN: 'underline #1e889b',
    Color.UNDERLINE_GREEN: 'underline #00aa00',
    Color.UNDERLINE_INTENSE_BLACK: 'underline #555555',
    Color.UNDERLINE_INTENSE_BLUE: 'underline #B452CD',
    Color.UNDERLINE_INTENSE_CYAN: 'underline #bbbbbb',
    Color.UNDERLINE_INTENSE_GREEN: 'underline #228B22',
    Color.UNDERLINE_INTENSE_PURPLE: 'underline #B452CD',
    Color.UNDERLINE_INTENSE_RED: 'underline #CD5555',
    Color.UNDERLINE_INTENSE_WHITE: 'underline #e3d2d2',
    Color.UNDERLINE_INTENSE_YELLOW: 'underline #e3d2d2',
    Color.UNDERLINE_PURPLE: 'underline #8B008B',
    Color.UNDERLINE_RED: 'underline #aa0000',
    Color.UNDERLINE_WHITE: 'underline #a7a7a7',
    Color.UNDERLINE_YELLOW: 'underline #cb6c20',
    Color.WHITE: '#a7a7a7',
    Color.YELLOW: '#cb6c20',
}

RRT_STYLE = {
    Color.BACKGROUND_BLACK: 'bg:#ff0000',
    Color.BACKGROUND_BLUE: 'bg:#87ceeb',
    Color.BACKGROUND_CYAN: 'bg:#87ceeb',
    Color.BACKGROUND_GREEN: 'bg:#00ff00',
    Color.BACKGROUND_INTENSE_BLACK: 'bg:#87ceeb',
    Color.BACKGROUND_INTENSE_BLUE: 'bg:#87ceeb',
    Color.BACKGROUND_INTENSE_CYAN: 'bg:#7fffd4',
    Color.BACKGROUND_INTENSE_GREEN: 'bg:#00ff00',
    Color.BACKGROUND_INTENSE_PURPLE: 'bg:#ee82ee',
    Color.BACKGROUND_INTENSE_RED: 'bg:#ff0000',
    Color.BACKGROUND_INTENSE_WHITE: 'bg:#e5e5e5',
    Color.BACKGROUND_INTENSE_YELLOW: 'bg:#eedd82',
    Color.BACKGROUND_PURPLE: 'bg:#ee82ee',
    Color.BACKGROUND_RED: 'bg:#ff0000',
    Color.BACKGROUND_WHITE: 'bg:#87ceeb',
    Color.BACKGROUND_YELLOW: 'bg:#ff0000',
    Color.BLACK: '#ff0000',
    Color.BLUE: '#87ceeb',
    Color.BOLD_BLACK: 'bold #ff0000',
    Color.BOLD_BLUE: 'bold #87ceeb',
    Color.BOLD_CYAN: 'bold #87ceeb',
    Color.BOLD_GREEN: 'bold #00ff00',
    Color.BOLD_INTENSE_BLACK: 'bold #87ceeb',
    Color.BOLD_INTENSE_BLUE: 'bold #87ceeb',
    Color.BOLD_INTENSE_CYAN: 'bold #7fffd4',
    Color.BOLD_INTENSE_GREEN: 'bold #00ff00',
    Color.BOLD_INTENSE_PURPLE: 'bold #ee82ee',
    Color.BOLD_INTENSE_RED: 'bold #ff0000',
    Color.BOLD_INTENSE_WHITE: 'bold #e5e5e5',
    Color.BOLD_INTENSE_YELLOW: 'bold #eedd82',
    Color.BOLD_PURPLE: 'bold #ee82ee',
    Color.BOLD_RED: 'bold #ff0000',
    Color.BOLD_UNDERLINE_BLACK: 'bold underline #ff0000',
    Color.BOLD_UNDERLINE_BLUE: 'bold underline #87ceeb',
    Color.BOLD_UNDERLINE_CYAN: 'bold underline #87ceeb',
    Color.BOLD_UNDERLINE_GREEN: 'bold underline #00ff00',
    Color.BOLD_UNDERLINE_INTENSE_BLACK: 'bold underline #87ceeb',
    Color.BOLD_UNDERLINE_INTENSE_BLUE: 'bold underline #87ceeb',
    Color.BOLD_UNDERLINE_INTENSE_CYAN: 'bold underline #7fffd4',
    Color.BOLD_UNDERLINE_INTENSE_GREEN: 'bold underline #00ff00',
    Color.BOLD_UNDERLINE_INTENSE_PURPLE: 'bold underline #ee82ee',
    Color.BOLD_UNDERLINE_INTENSE_RED: 'bold underline #ff0000',
    Color.BOLD_UNDERLINE_INTENSE_WHITE: 'bold underline #e5e5e5',
    Color.BOLD_UNDERLINE_INTENSE_YELLOW: 'bold underline #eedd82',
    Color.BOLD_UNDERLINE_PURPLE: 'bold underline #ee82ee',
    Color.BOLD_UNDERLINE_RED: 'bold underline #ff0000',
    Color.BOLD_UNDERLINE_WHITE: 'bold underline #87ceeb',
    Color.BOLD_UNDERLINE_YELLOW: 'bold underline #ff0000',
    Color.BOLD_WHITE: 'bold #87ceeb',
    Color.BOLD_YELLOW: 'bold #ff0000',
    Color.CYAN: '#87ceeb',
    Color.GREEN: '#00ff00',
    Color.INTENSE_BLACK: '#87ceeb',
    Color.INTENSE_BLUE: '#87ceeb',
    Color.INTENSE_CYAN: '#7fffd4',
    Color.INTENSE_GREEN: '#00ff00',
    Color.INTENSE_PURPLE: '#ee82ee',
    Color.INTENSE_RED: '#ff0000',
    Color.INTENSE_WHITE: '#e5e5e5',
    Color.INTENSE_YELLOW: '#eedd82',
    Color.NO_COLOR: 'noinherit',
    Color.PURPLE: '#ee82ee',
    Color.RED: '#ff0000',
    Color.UNDERLINE_BLACK: 'underline #ff0000',
    Color.UNDERLINE_BLUE: 'underline #87ceeb',
    Color.UNDERLINE_CYAN: 'underline #87ceeb',
    Color.UNDERLINE_GREEN: 'underline #00ff00',
    Color.UNDERLINE_INTENSE_BLACK: 'underline #87ceeb',
    Color.UNDERLINE_INTENSE_BLUE: 'underline #87ceeb',
    Color.UNDERLINE_INTENSE_CYAN: 'underline #7fffd4',
    Color.UNDERLINE_INTENSE_GREEN: 'underline #00ff00',
    Color.UNDERLINE_INTENSE_PURPLE: 'underline #ee82ee',
    Color.UNDERLINE_INTENSE_RED: 'underline #ff0000',
    Color.UNDERLINE_INTENSE_WHITE: 'underline #e5e5e5',
    Color.UNDERLINE_INTENSE_YELLOW: 'underline #eedd82',
    Color.UNDERLINE_PURPLE: 'underline #ee82ee',
    Color.UNDERLINE_RED: 'underline #ff0000',
    Color.UNDERLINE_WHITE: 'underline #87ceeb',
    Color.UNDERLINE_YELLOW: 'underline #ff0000',
    Color.WHITE: '#87ceeb',
    Color.YELLOW: '#ff0000',
}

TANGO_STYLE = {
    Color.BACKGROUND_BLACK: 'bg:#000000',
    Color.BACKGROUND_BLUE: 'bg:#0000cf',
    Color.BACKGROUND_CYAN: 'bg:#3465a4',
    Color.BACKGROUND_GREEN: 'bg:#00A000',
    Color.BACKGROUND_INTENSE_BLACK: 'bg:#204a87',
    Color.BACKGROUND_INTENSE_BLUE: 'bg:#5c35cc',
    Color.BACKGROUND_INTENSE_CYAN: 'bg:#f8f8f8',
    Color.BACKGROUND_INTENSE_GREEN: 'bg:#4e9a06',
    Color.BACKGROUND_INTENSE_PURPLE: 'bg:#f8f8f8',
    Color.BACKGROUND_INTENSE_RED: 'bg:#ef2929',
    Color.BACKGROUND_INTENSE_WHITE: 'bg:#f8f8f8',
    Color.BACKGROUND_INTENSE_YELLOW: 'bg:#c4a000',
    Color.BACKGROUND_PURPLE: 'bg:#800080',
    Color.BACKGROUND_RED: 'bg:#a40000',
    Color.BACKGROUND_WHITE: 'bg:#f8f8f8',
    Color.BACKGROUND_YELLOW: 'bg:#8f5902',
    Color.BLACK: '#000000',
    Color.BLUE: '#0000cf',
    Color.BOLD_BLACK: 'bold #000000',
    Color.BOLD_BLUE: 'bold #0000cf',
    Color.BOLD_CYAN: 'bold #3465a4',
    Color.BOLD_GREEN: 'bold #00A000',
    Color.BOLD_INTENSE_BLACK: 'bold #204a87',
    Color.BOLD_INTENSE_BLUE: 'bold #5c35cc',
    Color.BOLD_INTENSE_CYAN: 'bold #f8f8f8',
    Color.BOLD_INTENSE_GREEN: 'bold #4e9a06',
    Color.BOLD_INTENSE_PURPLE: 'bold #f8f8f8',
    Color.BOLD_INTENSE_RED: 'bold #ef2929',
    Color.BOLD_INTENSE_WHITE: 'bold #f8f8f8',
    Color.BOLD_INTENSE_YELLOW: 'bold #c4a000',
    Color.BOLD_PURPLE: 'bold #800080',
    Color.BOLD_RED: 'bold #a40000',
    Color.BOLD_UNDERLINE_BLACK: 'bold underline #000000',
    Color.BOLD_UNDERLINE_BLUE: 'bold underline #0000cf',
    Color.BOLD_UNDERLINE_CYAN: 'bold underline #3465a4',
    Color.BOLD_UNDERLINE_GREEN: 'bold underline #00A000',
    Color.BOLD_UNDERLINE_INTENSE_BLACK: 'bold underline #204a87',
    Color.BOLD_UNDERLINE_INTENSE_BLUE: 'bold underline #5c35cc',
    Color.BOLD_UNDERLINE_INTENSE_CYAN: 'bold underline #f8f8f8',
    Color.BOLD_UNDERLINE_INTENSE_GREEN: 'bold underline #4e9a06',
    Color.BOLD_UNDERLINE_INTENSE_PURPLE: 'bold underline #f8f8f8',
    Color.BOLD_UNDERLINE_INTENSE_RED: 'bold underline #ef2929',
    Color.BOLD_UNDERLINE_INTENSE_WHITE: 'bold underline #f8f8f8',
    Color.BOLD_UNDERLINE_INTENSE_YELLOW: 'bold underline #c4a000',
    Color.BOLD_UNDERLINE_PURPLE: 'bold underline #800080',
    Color.BOLD_UNDERLINE_RED: 'bold underline #a40000',
    Color.BOLD_UNDERLINE_WHITE: 'bold underline #f8f8f8',
    Color.BOLD_UNDERLINE_YELLOW: 'bold underline #8f5902',
    Color.BOLD_WHITE: 'bold #f8f8f8',
    Color.BOLD_YELLOW: 'bold #8f5902',
    Color.CYAN: '#3465a4',
    Color.GREEN: '#00A000',
    Color.INTENSE_BLACK: '#204a87',
    Color.INTENSE_BLUE: '#5c35cc',
    Color.INTENSE_CYAN: '#f8f8f8',
    Color.INTENSE_GREEN: '#4e9a06',
    Color.INTENSE_PURPLE: '#f8f8f8',
    Color.INTENSE_RED: '#ef2929',
    Color.INTENSE_WHITE: '#f8f8f8',
    Color.INTENSE_YELLOW: '#c4a000',
    Color.NO_COLOR: 'noinherit',
    Color.PURPLE: '#800080',
    Color.RED: '#a40000',
    Color.UNDERLINE_BLACK: 'underline #000000',
    Color.UNDERLINE_BLUE: 'underline #0000cf',
    Color.UNDERLINE_CYAN: 'underline #3465a4',
    Color.UNDERLINE_GREEN: 'underline #00A000',
    Color.UNDERLINE_INTENSE_BLACK: 'underline #204a87',
    Color.UNDERLINE_INTENSE_BLUE: 'underline #5c35cc',
    Color.UNDERLINE_INTENSE_CYAN: 'underline #f8f8f8',
    Color.UNDERLINE_INTENSE_GREEN: 'underline #4e9a06',
    Color.UNDERLINE_INTENSE_PURPLE: 'underline #f8f8f8',
    Color.UNDERLINE_INTENSE_RED: 'underline #ef2929',
    Color.UNDERLINE_INTENSE_WHITE: 'underline #f8f8f8',
    Color.UNDERLINE_INTENSE_YELLOW: 'underline #c4a000',
    Color.UNDERLINE_PURPLE: 'underline #800080',
    Color.UNDERLINE_RED: 'underline #a40000',
    Color.UNDERLINE_WHITE: 'underline #f8f8f8',
    Color.UNDERLINE_YELLOW: 'underline #8f5902',
    Color.WHITE: '#f8f8f8',
    Color.YELLOW: '#8f5902',
}

TRAC_STYLE = {
    Color.BACKGROUND_BLACK: 'bg:#000000',
    Color.BACKGROUND_BLUE: 'bg:#000080',
    Color.BACKGROUND_CYAN: 'bg:#009999',
    Color.BACKGROUND_GREEN: 'bg:#808000',
    Color.BACKGROUND_INTENSE_BLACK: 'bg:#555555',
    Color.BACKGROUND_INTENSE_BLUE: 'bg:#445588',
    Color.BACKGROUND_INTENSE_CYAN: 'bg:#ddffdd',
    Color.BACKGROUND_INTENSE_GREEN: 'bg:#999988',
    Color.BACKGROUND_INTENSE_PURPLE: 'bg:#e3d2d2',
    Color.BACKGROUND_INTENSE_RED: 'bg:#bb8844',
    Color.BACKGROUND_INTENSE_WHITE: 'bg:#ffdddd',
    Color.BACKGROUND_INTENSE_YELLOW: 'bg:#e3d2d2',
    Color.BACKGROUND_PURPLE: 'bg:#800080',
    Color.BACKGROUND_RED: 'bg:#aa0000',
    Color.BACKGROUND_WHITE: 'bg:#aaaaaa',
    Color.BACKGROUND_YELLOW: 'bg:#808000',
    Color.BLACK: '#000000',
    Color.BLUE: '#000080',
    Color.BOLD_BLACK: 'bold #000000',
    Color.BOLD_BLUE: 'bold #000080',
    Color.BOLD_CYAN: 'bold #009999',
    Color.BOLD_GREEN: 'bold #808000',
    Color.BOLD_INTENSE_BLACK: 'bold #555555',
    Color.BOLD_INTENSE_BLUE: 'bold #445588',
    Color.BOLD_INTENSE_CYAN: 'bold #ddffdd',
    Color.BOLD_INTENSE_GREEN: 'bold #999988',
    Color.BOLD_INTENSE_PURPLE: 'bold #e3d2d2',
    Color.BOLD_INTENSE_RED: 'bold #bb8844',
    Color.BOLD_INTENSE_WHITE: 'bold #ffdddd',
    Color.BOLD_INTENSE_YELLOW: 'bold #e3d2d2',
    Color.BOLD_PURPLE: 'bold #800080',
    Color.BOLD_RED: 'bold #aa0000',
    Color.BOLD_UNDERLINE_BLACK: 'bold underline #000000',
    Color.BOLD_UNDERLINE_BLUE: 'bold underline #000080',
    Color.BOLD_UNDERLINE_CYAN: 'bold underline #009999',
    Color.BOLD_UNDERLINE_GREEN: 'bold underline #808000',
    Color.BOLD_UNDERLINE_INTENSE_BLACK: 'bold underline #555555',
    Color.BOLD_UNDERLINE_INTENSE_BLUE: 'bold underline #445588',
    Color.BOLD_UNDERLINE_INTENSE_CYAN: 'bold underline #ddffdd',
    Color.BOLD_UNDERLINE_INTENSE_GREEN: 'bold underline #999988',
    Color.BOLD_UNDERLINE_INTENSE_PURPLE: 'bold underline #e3d2d2',
    Color.BOLD_UNDERLINE_INTENSE_RED: 'bold underline #bb8844',
    Color.BOLD_UNDERLINE_INTENSE_WHITE: 'bold underline #ffdddd',
    Color.BOLD_UNDERLINE_INTENSE_YELLOW: 'bold underline #e3d2d2',
    Color.BOLD_UNDERLINE_PURPLE: 'bold underline #800080',
    Color.BOLD_UNDERLINE_RED: 'bold underline #aa0000',
    Color.BOLD_UNDERLINE_WHITE: 'bold underline #aaaaaa',
    Color.BOLD_UNDERLINE_YELLOW: 'bold underline #808000',
    Color.BOLD_WHITE: 'bold #aaaaaa',
    Color.BOLD_YELLOW: 'bold #808000',
    Color.CYAN: '#009999',
    Color.GREEN: '#808000',
    Color.INTENSE_BLACK: '#555555',
    Color.INTENSE_BLUE: '#445588',
    Color.INTENSE_CYAN: '#ddffdd',
    Color.INTENSE_GREEN: '#999988',
    Color.INTENSE_PURPLE: '#e3d2d2',
    Color.INTENSE_RED: '#bb8844',
    Color.INTENSE_WHITE: '#ffdddd',
    Color.INTENSE_YELLOW: '#e3d2d2',
    Color.NO_COLOR: 'noinherit',
    Color.PURPLE: '#800080',
    Color.RED: '#aa0000',
    Color.UNDERLINE_BLACK: 'underline #000000',
    Color.UNDERLINE_BLUE: 'underline #000080',
    Color.UNDERLINE_CYAN: 'underline #009999',
    Color.UNDERLINE_GREEN: 'underline #808000',
    Color.UNDERLINE_INTENSE_BLACK: 'underline #555555',
    Color.UNDERLINE_INTENSE_BLUE: 'underline #445588',
    Color.UNDERLINE_INTENSE_CYAN: 'underline #ddffdd',
    Color.UNDERLINE_INTENSE_GREEN: 'underline #999988',
    Color.UNDERLINE_INTENSE_PURPLE: 'underline #e3d2d2',
    Color.UNDERLINE_INTENSE_RED: 'underline #bb8844',
    Color.UNDERLINE_INTENSE_WHITE: 'underline #ffdddd',
    Color.UNDERLINE_INTENSE_YELLOW: 'underline #e3d2d2',
    Color.UNDERLINE_PURPLE: 'underline #800080',
    Color.UNDERLINE_RED: 'underline #aa0000',
    Color.UNDERLINE_WHITE: 'underline #aaaaaa',
    Color.UNDERLINE_YELLOW: 'underline #808000',
    Color.WHITE: '#aaaaaa',
    Color.YELLOW: '#808000',
}

VIM_STYLE = {
    Color.BACKGROUND_BLACK: 'bg:#000080',
    Color.BACKGROUND_BLUE: 'bg:#000080',
    Color.BACKGROUND_CYAN: 'bg:#00cdcd',
    Color.BACKGROUND_GREEN: 'bg:#00cd00',
    Color.BACKGROUND_INTENSE_BLACK: 'bg:#666699',
    Color.BACKGROUND_INTENSE_BLUE: 'bg:#3399cc',
    Color.BACKGROUND_INTENSE_CYAN: 'bg:#00cdcd',
    Color.BACKGROUND_INTENSE_GREEN: 'bg:#00cd00',
    Color.BACKGROUND_INTENSE_PURPLE: 'bg:#cd00cd',
    Color.BACKGROUND_INTENSE_RED: 'bg:#FF0000',
    Color.BACKGROUND_INTENSE_WHITE: 'bg:#cccccc',
    Color.BACKGROUND_INTENSE_YELLOW: 'bg:#cdcd00',
    Color.BACKGROUND_PURPLE: 'bg:#cd00cd',
    Color.BACKGROUND_RED: 'bg:#cd0000',
    Color.BACKGROUND_WHITE: 'bg:#cccccc',
    Color.BACKGROUND_YELLOW: 'bg:#cd0000',
    Color.BLACK: '#000080',
    Color.BLUE: '#000080',
    Color.BOLD_BLACK: 'bold #000080',
    Color.BOLD_BLUE: 'bold #000080',
    Color.BOLD_CYAN: 'bold #00cdcd',
    Color.BOLD_GREEN: 'bold #00cd00',
    Color.BOLD_INTENSE_BLACK: 'bold #666699',
    Color.BOLD_INTENSE_BLUE: 'bold #3399cc',
    Color.BOLD_INTENSE_CYAN: 'bold #00cdcd',
    Color.BOLD_INTENSE_GREEN: 'bold #00cd00',
    Color.BOLD_INTENSE_PURPLE: 'bold #cd00cd',
    Color.BOLD_INTENSE_RED: 'bold #FF0000',
    Color.BOLD_INTENSE_WHITE: 'bold #cccccc',
    Color.BOLD_INTENSE_YELLOW: 'bold #cdcd00',
    Color.BOLD_PURPLE: 'bold #cd00cd',
    Color.BOLD_RED: 'bold #cd0000',
    Color.BOLD_UNDERLINE_BLACK: 'bold underline #000080',
    Color.BOLD_UNDERLINE_BLUE: 'bold underline #000080',
    Color.BOLD_UNDERLINE_CYAN: 'bold underline #00cdcd',
    Color.BOLD_UNDERLINE_GREEN: 'bold underline #00cd00',
    Color.BOLD_UNDERLINE_INTENSE_BLACK: 'bold underline #666699',
    Color.BOLD_UNDERLINE_INTENSE_BLUE: 'bold underline #3399cc',
    Color.BOLD_UNDERLINE_INTENSE_CYAN: 'bold underline #00cdcd',
    Color.BOLD_UNDERLINE_INTENSE_GREEN: 'bold underline #00cd00',
    Color.BOLD_UNDERLINE_INTENSE_PURPLE: 'bold underline #cd00cd',
    Color.BOLD_UNDERLINE_INTENSE_RED: 'bold underline #FF0000',
    Color.BOLD_UNDERLINE_INTENSE_WHITE: 'bold underline #cccccc',
    Color.BOLD_UNDERLINE_INTENSE_YELLOW: 'bold underline #cdcd00',
    Color.BOLD_UNDERLINE_PURPLE: 'bold underline #cd00cd',
    Color.BOLD_UNDERLINE_RED: 'bold underline #cd0000',
    Color.BOLD_UNDERLINE_WHITE: 'bold underline #cccccc',
    Color.BOLD_UNDERLINE_YELLOW: 'bold underline #cd0000',
    Color.BOLD_WHITE: 'bold #cccccc',
    Color.BOLD_YELLOW: 'bold #cd0000',
    Color.CYAN: '#00cdcd',
    Color.GREEN: '#00cd00',
    Color.INTENSE_BLACK: '#666699',
    Color.INTENSE_BLUE: '#3399cc',
    Color.INTENSE_CYAN: '#00cdcd',
    Color.INTENSE_GREEN: '#00cd00',
    Color.INTENSE_PURPLE: '#cd00cd',
    Color.INTENSE_RED: '#FF0000',
    Color.INTENSE_WHITE: '#cccccc',
    Color.INTENSE_YELLOW: '#cdcd00',
    Color.NO_COLOR: 'noinherit',
    Color.PURPLE: '#cd00cd',
    Color.RED: '#cd0000',
    Color.UNDERLINE_BLACK: 'underline #000080',
    Color.UNDERLINE_BLUE: 'underline #000080',
    Color.UNDERLINE_CYAN: 'underline #00cdcd',
    Color.UNDERLINE_GREEN: 'underline #00cd00',
    Color.UNDERLINE_INTENSE_BLACK: 'underline #666699',
    Color.UNDERLINE_INTENSE_BLUE: 'underline #3399cc',
    Color.UNDERLINE_INTENSE_CYAN: 'underline #00cdcd',
    Color.UNDERLINE_INTENSE_GREEN: 'underline #00cd00',
    Color.UNDERLINE_INTENSE_PURPLE: 'underline #cd00cd',
    Color.UNDERLINE_INTENSE_RED: 'underline #FF0000',
    Color.UNDERLINE_INTENSE_WHITE: 'underline #cccccc',
    Color.UNDERLINE_INTENSE_YELLOW: 'underline #cdcd00',
    Color.UNDERLINE_PURPLE: 'underline #cd00cd',
    Color.UNDERLINE_RED: 'underline #cd0000',
    Color.UNDERLINE_WHITE: 'underline #cccccc',
    Color.UNDERLINE_YELLOW: 'underline #cd0000',
    Color.WHITE: '#cccccc',
    Color.YELLOW: '#cd0000',
}

VS_STYLE = {
    Color.BACKGROUND_BLACK: 'bg:#008000',
    Color.BACKGROUND_BLUE: 'bg:#0000ff',
    Color.BACKGROUND_CYAN: 'bg:#2b91af',
    Color.BACKGROUND_GREEN: 'bg:#008000',
    Color.BACKGROUND_INTENSE_BLACK: 'bg:#2b91af',
    Color.BACKGROUND_INTENSE_BLUE: 'bg:#2b91af',
    Color.BACKGROUND_INTENSE_CYAN: 'bg:#2b91af',
    Color.BACKGROUND_INTENSE_GREEN: 'bg:#2b91af',
    Color.BACKGROUND_INTENSE_PURPLE: 'bg:#2b91af',
    Color.BACKGROUND_INTENSE_RED: 'bg:#FF0000',
    Color.BACKGROUND_INTENSE_WHITE: 'bg:#2b91af',
    Color.BACKGROUND_INTENSE_YELLOW: 'bg:#2b91af',
    Color.BACKGROUND_PURPLE: 'bg:#a31515',
    Color.BACKGROUND_RED: 'bg:#a31515',
    Color.BACKGROUND_WHITE: 'bg:#2b91af',
    Color.BACKGROUND_YELLOW: 'bg:#a31515',
    Color.BLACK: '#008000',
    Color.BLUE: '#0000ff',
    Color.BOLD_BLACK: 'bold #008000',
    Color.BOLD_BLUE: 'bold #0000ff',
    Color.BOLD_CYAN: 'bold #2b91af',
    Color.BOLD_GREEN: 'bold #008000',
    Color.BOLD_INTENSE_BLACK: 'bold #2b91af',
    Color.BOLD_INTENSE_BLUE: 'bold #2b91af',
    Color.BOLD_INTENSE_CYAN: 'bold #2b91af',
    Color.BOLD_INTENSE_GREEN: 'bold #2b91af',
    Color.BOLD_INTENSE_PURPLE: 'bold #2b91af',
    Color.BOLD_INTENSE_RED: 'bold #FF0000',
    Color.BOLD_INTENSE_WHITE: 'bold #2b91af',
    Color.BOLD_INTENSE_YELLOW: 'bold #2b91af',
    Color.BOLD_PURPLE: 'bold #a31515',
    Color.BOLD_RED: 'bold #a31515',
    Color.BOLD_UNDERLINE_BLACK: 'bold underline #008000',
    Color.BOLD_UNDERLINE_BLUE: 'bold underline #0000ff',
    Color.BOLD_UNDERLINE_CYAN: 'bold underline #2b91af',
    Color.BOLD_UNDERLINE_GREEN: 'bold underline #008000',
    Color.BOLD_UNDERLINE_INTENSE_BLACK: 'bold underline #2b91af',
    Color.BOLD_UNDERLINE_INTENSE_BLUE: 'bold underline #2b91af',
    Color.BOLD_UNDERLINE_INTENSE_CYAN: 'bold underline #2b91af',
    Color.BOLD_UNDERLINE_INTENSE_GREEN: 'bold underline #2b91af',
    Color.BOLD_UNDERLINE_INTENSE_PURPLE: 'bold underline #2b91af',
    Color.BOLD_UNDERLINE_INTENSE_RED: 'bold underline #FF0000',
    Color.BOLD_UNDERLINE_INTENSE_WHITE: 'bold underline #2b91af',
    Color.BOLD_UNDERLINE_INTENSE_YELLOW: 'bold underline #2b91af',
    Color.BOLD_UNDERLINE_PURPLE: 'bold underline #a31515',
    Color.BOLD_UNDERLINE_RED: 'bold underline #a31515',
    Color.BOLD_UNDERLINE_WHITE: 'bold underline #2b91af',
    Color.BOLD_UNDERLINE_YELLOW: 'bold underline #a31515',
    Color.BOLD_WHITE: 'bold #2b91af',
    Color.BOLD_YELLOW: 'bold #a31515',
    Color.CYAN: '#2b91af',
    Color.GREEN: '#008000',
    Color.INTENSE_BLACK: '#2b91af',
    Color.INTENSE_BLUE: '#2b91af',
    Color.INTENSE_CYAN: '#2b91af',
    Color.INTENSE_GREEN: '#2b91af',
    Color.INTENSE_PURPLE: '#2b91af',
    Color.INTENSE_RED: '#FF0000',
    Color.INTENSE_WHITE: '#2b91af',
    Color.INTENSE_YELLOW: '#2b91af',
    Color.NO_COLOR: 'noinherit',
    Color.PURPLE: '#a31515',
    Color.RED: '#a31515',
    Color.UNDERLINE_BLACK: 'underline #008000',
    Color.UNDERLINE_BLUE: 'underline #0000ff',
    Color.UNDERLINE_CYAN: 'underline #2b91af',
    Color.UNDERLINE_GREEN: 'underline #008000',
    Color.UNDERLINE_INTENSE_BLACK: 'underline #2b91af',
    Color.UNDERLINE_INTENSE_BLUE: 'underline #2b91af',
    Color.UNDERLINE_INTENSE_CYAN: 'underline #2b91af',
    Color.UNDERLINE_INTENSE_GREEN: 'underline #2b91af',
    Color.UNDERLINE_INTENSE_PURPLE: 'underline #2b91af',
    Color.UNDERLINE_INTENSE_RED: 'underline #FF0000',
    Color.UNDERLINE_INTENSE_WHITE: 'underline #2b91af',
    Color.UNDERLINE_INTENSE_YELLOW: 'underline #2b91af',
    Color.UNDERLINE_PURPLE: 'underline #a31515',
    Color.UNDERLINE_RED: 'underline #a31515',
    Color.UNDERLINE_WHITE: 'underline #2b91af',
    Color.UNDERLINE_YELLOW: 'underline #a31515',
    Color.WHITE: '#2b91af',
    Color.YELLOW: '#a31515',
}

XCODE_STYLE = {
    Color.BACKGROUND_BLACK: 'bg:#000000',
    Color.BACKGROUND_BLUE: 'bg:#1C01CE',
    Color.BACKGROUND_CYAN: 'bg:#3F6E75',
    Color.BACKGROUND_GREEN: 'bg:#177500',
    Color.BACKGROUND_INTENSE_BLACK: 'bg:#3F6E75',
    Color.BACKGROUND_INTENSE_BLUE: 'bg:#2300CE',
    Color.BACKGROUND_INTENSE_CYAN: 'bg:#3F6E75',
    Color.BACKGROUND_INTENSE_GREEN: 'bg:#3F6E75',
    Color.BACKGROUND_INTENSE_PURPLE: 'bg:#A90D91',
    Color.BACKGROUND_INTENSE_RED: 'bg:#C41A16',
    Color.BACKGROUND_INTENSE_WHITE: 'bg:#3F6E75',
    Color.BACKGROUND_INTENSE_YELLOW: 'bg:#836C28',
    Color.BACKGROUND_PURPLE: 'bg:#A90D91',
    Color.BACKGROUND_RED: 'bg:#C41A16',
    Color.BACKGROUND_WHITE: 'bg:#3F6E75',
    Color.BACKGROUND_YELLOW: 'bg:#836C28',
    Color.BLACK: '#000000',
    Color.BLUE: '#1C01CE',
    Color.BOLD_BLACK: 'bold #000000',
    Color.BOLD_BLUE: 'bold #1C01CE',
    Color.BOLD_CYAN: 'bold #3F6E75',
    Color.BOLD_GREEN: 'bold #177500',
    Color.BOLD_INTENSE_BLACK: 'bold #3F6E75',
    Color.BOLD_INTENSE_BLUE: 'bold #2300CE',
    Color.BOLD_INTENSE_CYAN: 'bold #3F6E75',
    Color.BOLD_INTENSE_GREEN: 'bold #3F6E75',
    Color.BOLD_INTENSE_PURPLE: 'bold #A90D91',
    Color.BOLD_INTENSE_RED: 'bold #C41A16',
    Color.BOLD_INTENSE_WHITE: 'bold #3F6E75',
    Color.BOLD_INTENSE_YELLOW: 'bold #836C28',
    Color.BOLD_PURPLE: 'bold #A90D91',
    Color.BOLD_RED: 'bold #C41A16',
    Color.BOLD_UNDERLINE_BLACK: 'bold underline #000000',
    Color.BOLD_UNDERLINE_BLUE: 'bold underline #1C01CE',
    Color.BOLD_UNDERLINE_CYAN: 'bold underline #3F6E75',
    Color.BOLD_UNDERLINE_GREEN: 'bold underline #177500',
    Color.BOLD_UNDERLINE_INTENSE_BLACK: 'bold underline #3F6E75',
    Color.BOLD_UNDERLINE_INTENSE_BLUE: 'bold underline #2300CE',
    Color.BOLD_UNDERLINE_INTENSE_CYAN: 'bold underline #3F6E75',
    Color.BOLD_UNDERLINE_INTENSE_GREEN: 'bold underline #3F6E75',
    Color.BOLD_UNDERLINE_INTENSE_PURPLE: 'bold underline #A90D91',
    Color.BOLD_UNDERLINE_INTENSE_RED: 'bold underline #C41A16',
    Color.BOLD_UNDERLINE_INTENSE_WHITE: 'bold underline #3F6E75',
    Color.BOLD_UNDERLINE_INTENSE_YELLOW: 'bold underline #836C28',
    Color.BOLD_UNDERLINE_PURPLE: 'bold underline #A90D91',
    Color.BOLD_UNDERLINE_RED: 'bold underline #C41A16',
    Color.BOLD_UNDERLINE_WHITE: 'bold underline #3F6E75',
    Color.BOLD_UNDERLINE_YELLOW: 'bold underline #836C28',
    Color.BOLD_WHITE: 'bold #3F6E75',
    Color.BOLD_YELLOW: 'bold #836C28',
    Color.CYAN: '#3F6E75',
    Color.GREEN: '#177500',
    Color.INTENSE_BLACK: '#3F6E75',
    Color.INTENSE_BLUE: '#2300CE',
    Color.INTENSE_CYAN: '#3F6E75',
    Color.INTENSE_GREEN: '#3F6E75',
    Color.INTENSE_PURPLE: '#A90D91',
    Color.INTENSE_RED: '#C41A16',
    Color.INTENSE_WHITE: '#3F6E75',
    Color.INTENSE_YELLOW: '#836C28',
    Color.NO_COLOR: 'noinherit',
    Color.PURPLE: '#A90D91',
    Color.RED: '#C41A16',
    Color.UNDERLINE_BLACK: 'underline #000000',
    Color.UNDERLINE_BLUE: 'underline #1C01CE',
    Color.UNDERLINE_CYAN: 'underline #3F6E75',
    Color.UNDERLINE_GREEN: 'underline #177500',
    Color.UNDERLINE_INTENSE_BLACK: 'underline #3F6E75',
    Color.UNDERLINE_INTENSE_BLUE: 'underline #2300CE',
    Color.UNDERLINE_INTENSE_CYAN: 'underline #3F6E75',
    Color.UNDERLINE_INTENSE_GREEN: 'underline #3F6E75',
    Color.UNDERLINE_INTENSE_PURPLE: 'underline #A90D91',
    Color.UNDERLINE_INTENSE_RED: 'underline #C41A16',
    Color.UNDERLINE_INTENSE_WHITE: 'underline #3F6E75',
    Color.UNDERLINE_INTENSE_YELLOW: 'underline #836C28',
    Color.UNDERLINE_PURPLE: 'underline #A90D91',
    Color.UNDERLINE_RED: 'underline #C41A16',
    Color.UNDERLINE_WHITE: 'underline #3F6E75',
    Color.UNDERLINE_YELLOW: 'underline #836C28',
    Color.WHITE: '#3F6E75',
    Color.YELLOW: '#836C28',
}

STYLES = {
    'algol': ALGOL_STYLE,
    'algol_nu': ALGOL_NU_STYLE,
    'autumn': AUTUMN_STYLE,
    'borland': BORLAND_STYLE,
    'bw': BW_STYLE,
    'colorful': COLORFUL_STYLE,
    'default': DEFAULT_STYLE,
    'emacs': EMACS_STYLE,
    'friendly': FRIENDLY_STYLE,
    'fruity': FRUITY_STYLE,
    'igor': IGOR_STYLE,
    'lovelace': LOVELACE_STYLE,
    'manni': MANNI_STYLE,
    'monokai': MONOKAI_STYLE,
    'murphy': MURPHY_STYLE,
    'native': NATIVE_STYLE,
    'paraiso-dark': PARAISO_DARK_STYLE,
    'paraiso-light': PARAISO_LIGHT_STYLE,
    'pastie': PASTIE_STYLE,
    'perldoc': PERLDOC_STYLE,
    'rrt': RRT_STYLE,
    'tango': TANGO_STYLE,
    'trac': TRAC_STYLE,
    'vim': VIM_STYLE,
    'vs': VS_STYLE,
    'xcode': XCODE_STYLE,
}
