"""Tools for helping with ANSI color codes."""
import re
import string
from warnings import warn

RE_BACKGROUND = re.compile('(bg|bg#|bghex|background)')


def partial_color_format(template, style='default', cmap=None, hide=False):
    """Formats a template string but only with respect to the colors.
    Another template string is returned, with the color values filled in.

    Parameters
    ----------
    template : str
        The template string, potentially with color names.
    style : str, optional
        Sytle name to look up color map from.
    cmap : dict, optional
        A color map to use, this will prevent the color map from being
        looked up via the style name.
    hide : bool, optional
        Whether to wrap the color codes in the \\001 and \\002 escape
        codes, so that the color codes are not counted against line
        length.

    Returns
    -------
    A template string with the color values filled in.
    """
    try:
        return _partial_color_format_main(template, style=style, cmap=cmap, hide=hide)
    except:
        return template


def _partial_color_format_main(template, style='default', cmap=None, hide=False):
    if cmap is not None:
        pass
    elif style in STYLES:
        cmap = STYLES[style]
    else:
        warn('Could not find color style {0!r}, using default.'.format(style),
             RuntimeWarning)
        cmap = DEFAULT_STYLE
    formatter = string.Formatter()
    esc = ('\001' if hide else '') + '\033['
    m = 'm' + ('\002' if hide else '')
    bopen = '{'
    bclose = '}'
    colon = ':'
    expl = '!'
    toks = []
    for literal, field, spec, conv in formatter.parse(template):
        toks.append(literal)
        if field is None:
            pass
        elif field in cmap:
            toks.extend([esc, cmap[field], m])
        elif '#' in field:
            field = field.lower()
            pre, _, post = field.partition('#')
            f_or_b = '38' if RE_BACKGROUND.search(pre) is None else '48'
            rgb, _, post = post.partition('_')
            c256, _ = rgb_to_256(rgb)
            color = f_or_b + ';5;' + c256
            mods = pre + '_' + post
            if 'underline' in mods:
                color = '4;' + color
            if 'bold' in mods:
                color = '1;' + color
            toks.extend([esc, color, m])
        elif field is not None:
            toks.append(bopen)
            toks.append(field)
            if conv is not None and len(conv) > 0:
                toks.append(expl)
                toks.append(conv)
            if spec is not None and len(spec) > 0:
                toks.append(colon)
                toks.append(spec)
            toks.append(bclose)
    return ''.join(toks)


RGB_256 = {
    '000000': '16',
    '00005f': '17',
    '000080': '04',
    '000087': '18',
    '0000af': '19',
    '0000d7': '20',
    '0000ff': '21',
    '005f00': '22',
    '005f5f': '23',
    '005f87': '24',
    '005faf': '25',
    '005fd7': '26',
    '005fff': '27',
    '008000': '02',
    '008080': '06',
    '008700': '28',
    '00875f': '29',
    '008787': '30',
    '0087af': '31',
    '0087d7': '32',
    '0087ff': '33',
    '00af00': '34',
    '00af5f': '35',
    '00af87': '36',
    '00afaf': '37',
    '00afd7': '38',
    '00afff': '39',
    '00d700': '40',
    '00d75f': '41',
    '00d787': '42',
    '00d7af': '43',
    '00d7d7': '44',
    '00d7ff': '45',
    '00ff00': '46',
    '00ff5f': '47',
    '00ff87': '48',
    '00ffaf': '49',
    '00ffd7': '50',
    '00ffff': '51',
    '080808': '232',
    '121212': '233',
    '1c1c1c': '234',
    '262626': '235',
    '303030': '236',
    '3a3a3a': '237',
    '444444': '238',
    '4e4e4e': '239',
    '585858': '240',
    '5f0000': '52',
    '5f005f': '53',
    '5f0087': '54',
    '5f00af': '55',
    '5f00d7': '56',
    '5f00ff': '57',
    '5f5f00': '58',
    '5f5f5f': '59',
    '5f5f87': '60',
    '5f5faf': '61',
    '5f5fd7': '62',
    '5f5fff': '63',
    '5f8700': '64',
    '5f875f': '65',
    '5f8787': '66',
    '5f87af': '67',
    '5f87d7': '68',
    '5f87ff': '69',
    '5faf00': '70',
    '5faf5f': '71',
    '5faf87': '72',
    '5fafaf': '73',
    '5fafd7': '74',
    '5fafff': '75',
    '5fd700': '76',
    '5fd75f': '77',
    '5fd787': '78',
    '5fd7af': '79',
    '5fd7d7': '80',
    '5fd7ff': '81',
    '5fff00': '82',
    '5fff5f': '83',
    '5fff87': '84',
    '5fffaf': '85',
    '5fffd7': '86',
    '5fffff': '87',
    '626262': '241',
    '6c6c6c': '242',
    '767676': '243',
    '800000': '01',
    '800080': '05',
    '808000': '03',
    '808080': '244',
    '870000': '88',
    '87005f': '89',
    '870087': '90',
    '8700af': '91',
    '8700d7': '92',
    '8700ff': '93',
    '875f00': '94',
    '875f5f': '95',
    '875f87': '96',
    '875faf': '97',
    '875fd7': '98',
    '875fff': '99',
    '878700': '100',
    '87875f': '101',
    '878787': '102',
    '8787af': '103',
    '8787d7': '104',
    '8787ff': '105',
    '87af00': '106',
    '87af5f': '107',
    '87af87': '108',
    '87afaf': '109',
    '87afd7': '110',
    '87afff': '111',
    '87d700': '112',
    '87d75f': '113',
    '87d787': '114',
    '87d7af': '115',
    '87d7d7': '116',
    '87d7ff': '117',
    '87ff00': '118',
    '87ff5f': '119',
    '87ff87': '120',
    '87ffaf': '121',
    '87ffd7': '122',
    '87ffff': '123',
    '8a8a8a': '245',
    '949494': '246',
    '9e9e9e': '247',
    'a8a8a8': '248',
    'af0000': '124',
    'af005f': '125',
    'af0087': '126',
    'af00af': '127',
    'af00d7': '128',
    'af00ff': '129',
    'af5f00': '130',
    'af5f5f': '131',
    'af5f87': '132',
    'af5faf': '133',
    'af5fd7': '134',
    'af5fff': '135',
    'af8700': '136',
    'af875f': '137',
    'af8787': '138',
    'af87af': '139',
    'af87d7': '140',
    'af87ff': '141',
    'afaf00': '142',
    'afaf5f': '143',
    'afaf87': '144',
    'afafaf': '145',
    'afafd7': '146',
    'afafff': '147',
    'afd700': '148',
    'afd75f': '149',
    'afd787': '150',
    'afd7af': '151',
    'afd7d7': '152',
    'afd7ff': '153',
    'afff00': '154',
    'afff5f': '155',
    'afff87': '156',
    'afffaf': '157',
    'afffd7': '158',
    'afffff': '159',
    'b2b2b2': '249',
    'bcbcbc': '250',
    'c0c0c0': '07',
    'c6c6c6': '251',
    'd0d0d0': '252',
    'd70000': '160',
    'd7005f': '161',
    'd70087': '162',
    'd700af': '163',
    'd700d7': '164',
    'd700ff': '165',
    'd75f00': '166',
    'd75f5f': '167',
    'd75f87': '168',
    'd75faf': '169',
    'd75fd7': '170',
    'd75fff': '171',
    'd78700': '172',
    'd7875f': '173',
    'd78787': '174',
    'd787af': '175',
    'd787d7': '176',
    'd787ff': '177',
    'd7af00': '178',
    'd7af5f': '179',
    'd7af87': '180',
    'd7afaf': '181',
    'd7afd7': '182',
    'd7afff': '183',
    'd7d700': '184',
    'd7d75f': '185',
    'd7d787': '186',
    'd7d7af': '187',
    'd7d7d7': '188',
    'd7d7ff': '189',
    'd7ff00': '190',
    'd7ff5f': '191',
    'd7ff87': '192',
    'd7ffaf': '193',
    'd7ffd7': '194',
    'd7ffff': '195',
    'dadada': '253',
    'e4e4e4': '254',
    'eeeeee': '255',
    'ff0000': '196',
    'ff005f': '197',
    'ff0087': '198',
    'ff00af': '199',
    'ff00d7': '200',
    'ff00ff': '201',
    'ff5f00': '202',
    'ff5f5f': '203',
    'ff5f87': '204',
    'ff5faf': '205',
    'ff5fd7': '206',
    'ff5fff': '207',
    'ff8700': '208',
    'ff875f': '209',
    'ff8787': '210',
    'ff87af': '211',
    'ff87d7': '212',
    'ff87ff': '213',
    'ffaf00': '214',
    'ffaf5f': '215',
    'ffaf87': '216',
    'ffafaf': '217',
    'ffafd7': '218',
    'ffafff': '219',
    'ffd700': '220',
    'ffd75f': '221',
    'ffd787': '222',
    'ffd7af': '223',
    'ffd7d7': '224',
    'ffd7ff': '225',
    'ffff00': '226',
    'ffff5f': '227',
    'ffff87': '228',
    'ffffaf': '229',
    'ffffd7': '230',
    'ffffff': '231',
}

RE_RGB3 = re.compile(r'(.)(.)(.)')
RE_RGB6 = re.compile(r'(..)(..)(..)')

def rgb_to_ints(rgb):
    """Converts an RGB string into a tuple of ints."""
    if len(rgb) == 6:
        return tuple([int(h, 16) for h in RE_RGB6.split(rgb)[1:4]])
    else:
        return tuple([int(h*2, 16) for h in RE_RGB3.split(rgb)[1:4]])


def rgb_to_256(rgb):
    """Find the closest ANSI 256 approximation to the given RGB value.
    Thanks to Micah Elliott (http://MicahElliott.com) for colortrans.py
    """
    rgb = rgb.lstrip('#')
    if len(rgb) == 0:
        return '0', '000000'
    incs = (0x00, 0x5f, 0x87, 0xaf, 0xd7, 0xff)
    # Break 6-char RGB code into 3 integer vals.
    parts = rgb_to_ints(rgb)
    res = []
    for part in parts:
        i = 0
        while i < len(incs)-1:
            s, b = incs[i], incs[i+1]  # smaller, bigger
            if s <= part <= b:
                s1 = abs(s - part)
                b1 = abs(b - part)
                if s1 < b1: closest = s
                else: closest = b
                res.append(closest)
                break
            i += 1
    res = ''.join([('%02.x' % i) for i in res])
    equiv = RGB_256[res]
    return equiv, res


def color_style_names():
    """Returns an iterable of all ANSI color style names."""
    return STYLES.keys()


def color_style(style='default'):
    """Returns the current color map."""
    if style in STYLES:
        cmap = STYLES[style]
    else:
        warn('Could not find color style {0!r}, using default.'.format(style),
             RuntimeWarning)
        cmap = DEFAULT_STYLE
    return cmap


def _expand_style(cmap):
    """Expands a style in order to more quickly make color map changes."""
    for key, val in list(cmap.items()):
        if key == 'NO_COLOR':
            continue
        elif len(val) == 0:
            cmap['BOLD_'+key] = '1'
            cmap['UNDERLINE_'+key] = '4'
            cmap['BOLD_UNDERLINE_'+key] = '1;4'
            cmap['BACKGROUND_'+key] = val
        else:
            cmap['BOLD_'+key] = '1;' + val
            cmap['UNDERLINE_'+key] = '4;' + val
            cmap['BOLD_UNDERLINE_'+key] = '1;4;' + val
            cmap['BACKGROUND_'+key] = val.replace('38', '48', 1)


BW_STYLE = {
    'BLACK': '',
    'BLUE': '',
    'CYAN': '',
    'GREEN': '',
    'INTENSE_BLACK': '',
    'INTENSE_BLUE': '',
    'INTENSE_CYAN': '',
    'INTENSE_GREEN': '',
    'INTENSE_PURPLE': '',
    'INTENSE_RED': '',
    'INTENSE_WHITE': '',
    'INTENSE_YELLOW': '',
    'NO_COLOR': '0',
    'PURPLE': '',
    'RED': '',
    'WHITE': '',
    'YELLOW': '',
}
_expand_style(BW_STYLE)

DEFAULT_STYLE = {
    # Reset
    'NO_COLOR': '0',  # Text Reset
    # Regular Colors
    'BLACK': '0;30',  # BLACK
    'RED': '0;31',  # RED
    'GREEN': '0;32',  # GREEN
    'YELLOW': '0;33',  # YELLOW
    'BLUE': '0;34',  # BLUE
    'PURPLE': '0;35',  # PURPLE
    'CYAN': '0;36',  # CYAN
    'WHITE': '0;37',  # WHITE
    # Bold
    'BOLD_BLACK': '1;30',  # BLACK
    'BOLD_RED': '1;31',  # RED
    'BOLD_GREEN': '1;32',  # GREEN
    'BOLD_YELLOW': '1;33',  # YELLOW
    'BOLD_BLUE': '1;34',  # BLUE
    'BOLD_PURPLE': '1;35',  # PURPLE
    'BOLD_CYAN': '1;36',  # CYAN
    'BOLD_WHITE': '1;37',  # WHITE
    # Underline
    'UNDERLINE_BLACK': '4;30',  # BLACK
    'UNDERLINE_RED': '4;31',  # RED
    'UNDERLINE_GREEN': '4;32',  # GREEN
    'UNDERLINE_YELLOW': '4;33',  # YELLOW
    'UNDERLINE_BLUE': '4;34',  # BLUE
    'UNDERLINE_PURPLE': '4;35',  # PURPLE
    'UNDERLINE_CYAN': '4;36',  # CYAN
    'UNDERLINE_WHITE': '4;37',  # WHITE
    # Bold, Underline
    'BOLD_UNDERLINE_BLACK': '1;4;30',  # BLACK
    'BOLD_UNDERLINE_RED': '1;4;31',  # RED
    'BOLD_UNDERLINE_GREEN': '1;4;32',  # GREEN
    'BOLD_UNDERLINE_YELLOW': '1;4;33',  # YELLOW
    'BOLD_UNDERLINE_BLUE': '1;4;34',  # BLUE
    'BOLD_UNDERLINE_PURPLE': '1;4;35',  # PURPLE
    'BOLD_UNDERLINE_CYAN': '1;4;36',  # CYAN
    'BOLD_UNDERLINE_WHITE': '1;4;37',  # WHITE
    # Background
    'BACKGROUND_BLACK': '40',  # BLACK
    'BACKGROUND_RED': '41',  # RED
    'BACKGROUND_GREEN': '42',  # GREEN
    'BACKGROUND_YELLOW': '43',  # YELLOW
    'BACKGROUND_BLUE': '44',  # BLUE
    'BACKGROUND_PURPLE': '45',  # PURPLE
    'BACKGROUND_CYAN': '46',  # CYAN
    'BACKGROUND_WHITE': '47',  # WHITE
    # High Intensity
    'INTENSE_BLACK': '0;90',  # BLACK
    'INTENSE_RED': '0;91',  # RED
    'INTENSE_GREEN': '0;92',  # GREEN
    'INTENSE_YELLOW': '0;93',  # YELLOW
    'INTENSE_BLUE': '0;94',  # BLUE
    'INTENSE_PURPLE': '0;95',  # PURPLE
    'INTENSE_CYAN': '0;96',  # CYAN
    'INTENSE_WHITE': '0;97',  # WHITE
    # Bold High Intensity
    'BOLD_INTENSE_BLACK': '1;90',  # BLACK
    'BOLD_INTENSE_RED': '1;91',  # RED
    'BOLD_INTENSE_GREEN': '1;92',  # GREEN
    'BOLD_INTENSE_YELLOW': '1;93',  # YELLOW
    'BOLD_INTENSE_BLUE': '1;94',  # BLUE
    'BOLD_INTENSE_PURPLE': '1;95',  # PURPLE
    'BOLD_INTENSE_CYAN': '1;96',  # CYAN
    'BOLD_INTENSE_WHITE': '1;97',  # WHITE
    # Underline High Intensity
    'UNDERLINE_INTENSE_BLACK': '4;90',  # BLACK
    'UNDERLINE_INTENSE_RED': '4;91',  # RED
    'UNDERLINE_INTENSE_GREEN': '4;92',  # GREEN
    'UNDERLINE_INTENSE_YELLOW': '4;93',  # YELLOW
    'UNDERLINE_INTENSE_BLUE': '4;94',  # BLUE
    'UNDERLINE_INTENSE_PURPLE': '4;95',  # PURPLE
    'UNDERLINE_INTENSE_CYAN': '4;96',  # CYAN
    'UNDERLINE_INTENSE_WHITE': '4;97',  # WHITE
    # Bold Underline High Intensity
    'BOLD_UNDERLINE_INTENSE_BLACK': '1;4;90',  # BLACK
    'BOLD_UNDERLINE_INTENSE_RED': '1;4;91',  # RED
    'BOLD_UNDERLINE_INTENSE_GREEN': '1;4;92',  # GREEN
    'BOLD_UNDERLINE_INTENSE_YELLOW': '1;4;93',  # YELLOW
    'BOLD_UNDERLINE_INTENSE_BLUE': '1;4;94',  # BLUE
    'BOLD_UNDERLINE_INTENSE_PURPLE': '1;4;95',  # PURPLE
    'BOLD_UNDERLINE_INTENSE_CYAN': '1;4;96',  # CYAN
    'BOLD_UNDERLINE_INTENSE_WHITE': '1;4;97',  # WHITE
    # High Intensity backgrounds
    'BACKGROUND_INTENSE_BLACK': '0;100',  # BLACK
    'BACKGROUND_INTENSE_RED': '0;101',  # RED
    'BACKGROUND_INTENSE_GREEN': '0;102',  # GREEN
    'BACKGROUND_INTENSE_YELLOW': '0;103',  # YELLOW
    'BACKGROUND_INTENSE_BLUE': '0;104',  # BLUE
    'BACKGROUND_INTENSE_PURPLE': '0;105',  # PURPLE
    'BACKGROUND_INTENSE_CYAN': '0;106',  # CYAN
    'BACKGROUND_INTENSE_WHITE': '0;107',  # WHITE
}

MONOKAI_STYLE = {
    'NO_COLOR': '0',
    'BLACK': '38;5;16',
    'BLUE': '38;5;63',
    'CYAN': '38;5;81',
    'GREEN': '38;5;40',
    'PURPLE': '38;5;89',
    'RED': '38;5;124',
    'WHITE': '38;5;188',
    'YELLOW': '38;5;184',
    'INTENSE_BLACK': '38;5;59',
    'INTENSE_BLUE': '38;5;20',
    'INTENSE_CYAN': '38;5;44',
    'INTENSE_GREEN': '38;5;148',
    'INTENSE_PURPLE': '38;5;141',
    'INTENSE_RED': '38;5;197',
    'INTENSE_WHITE': '38;5;15',
    'INTENSE_YELLOW': '38;5;186',
}
_expand_style(MONOKAI_STYLE)

#############################################################
#############   Auto-generated below this line   ############
#############################################################

ALGOL_STYLE = {
    'BACKGROUND_BLACK': '48;5;59',
    'BACKGROUND_BLUE': '48;5;59',
    'BACKGROUND_CYAN': '48;5;59',
    'BACKGROUND_GREEN': '48;5;59',
    'BACKGROUND_INTENSE_BLACK': '48;5;59',
    'BACKGROUND_INTENSE_BLUE': '48;5;102',
    'BACKGROUND_INTENSE_CYAN': '48;5;102',
    'BACKGROUND_INTENSE_GREEN': '48;5;102',
    'BACKGROUND_INTENSE_PURPLE': '48;5;102',
    'BACKGROUND_INTENSE_RED': '48;5;09',
    'BACKGROUND_INTENSE_WHITE': '48;5;102',
    'BACKGROUND_INTENSE_YELLOW': '48;5;102',
    'BACKGROUND_PURPLE': '48;5;59',
    'BACKGROUND_RED': '48;5;09',
    'BACKGROUND_WHITE': '48;5;102',
    'BACKGROUND_YELLOW': '48;5;09',
    'BLACK': '38;5;59',
    'BLUE': '38;5;59',
    'BOLD_BLACK': '1;38;5;59',
    'BOLD_BLUE': '1;38;5;59',
    'BOLD_CYAN': '1;38;5;59',
    'BOLD_GREEN': '1;38;5;59',
    'BOLD_INTENSE_BLACK': '1;38;5;59',
    'BOLD_INTENSE_BLUE': '1;38;5;102',
    'BOLD_INTENSE_CYAN': '1;38;5;102',
    'BOLD_INTENSE_GREEN': '1;38;5;102',
    'BOLD_INTENSE_PURPLE': '1;38;5;102',
    'BOLD_INTENSE_RED': '1;38;5;09',
    'BOLD_INTENSE_WHITE': '1;38;5;102',
    'BOLD_INTENSE_YELLOW': '1;38;5;102',
    'BOLD_PURPLE': '1;38;5;59',
    'BOLD_RED': '1;38;5;09',
    'BOLD_UNDERLINE_BLACK': '1;4;38;5;59',
    'BOLD_UNDERLINE_BLUE': '1;4;38;5;59',
    'BOLD_UNDERLINE_CYAN': '1;4;38;5;59',
    'BOLD_UNDERLINE_GREEN': '1;4;38;5;59',
    'BOLD_UNDERLINE_INTENSE_BLACK': '1;4;38;5;59',
    'BOLD_UNDERLINE_INTENSE_BLUE': '1;4;38;5;102',
    'BOLD_UNDERLINE_INTENSE_CYAN': '1;4;38;5;102',
    'BOLD_UNDERLINE_INTENSE_GREEN': '1;4;38;5;102',
    'BOLD_UNDERLINE_INTENSE_PURPLE': '1;4;38;5;102',
    'BOLD_UNDERLINE_INTENSE_RED': '1;4;38;5;09',
    'BOLD_UNDERLINE_INTENSE_WHITE': '1;4;38;5;102',
    'BOLD_UNDERLINE_INTENSE_YELLOW': '1;4;38;5;102',
    'BOLD_UNDERLINE_PURPLE': '1;4;38;5;59',
    'BOLD_UNDERLINE_RED': '1;4;38;5;09',
    'BOLD_UNDERLINE_WHITE': '1;4;38;5;102',
    'BOLD_UNDERLINE_YELLOW': '1;4;38;5;09',
    'BOLD_WHITE': '1;38;5;102',
    'BOLD_YELLOW': '1;38;5;09',
    'CYAN': '38;5;59',
    'GREEN': '38;5;59',
    'INTENSE_BLACK': '38;5;59',
    'INTENSE_BLUE': '38;5;102',
    'INTENSE_CYAN': '38;5;102',
    'INTENSE_GREEN': '38;5;102',
    'INTENSE_PURPLE': '38;5;102',
    'INTENSE_RED': '38;5;09',
    'INTENSE_WHITE': '38;5;102',
    'INTENSE_YELLOW': '38;5;102',
    'NO_COLOR': '0',
    'PURPLE': '38;5;59',
    'RED': '38;5;09',
    'UNDERLINE_BLACK': '4;38;5;59',
    'UNDERLINE_BLUE': '4;38;5;59',
    'UNDERLINE_CYAN': '4;38;5;59',
    'UNDERLINE_GREEN': '4;38;5;59',
    'UNDERLINE_INTENSE_BLACK': '4;38;5;59',
    'UNDERLINE_INTENSE_BLUE': '4;38;5;102',
    'UNDERLINE_INTENSE_CYAN': '4;38;5;102',
    'UNDERLINE_INTENSE_GREEN': '4;38;5;102',
    'UNDERLINE_INTENSE_PURPLE': '4;38;5;102',
    'UNDERLINE_INTENSE_RED': '4;38;5;09',
    'UNDERLINE_INTENSE_WHITE': '4;38;5;102',
    'UNDERLINE_INTENSE_YELLOW': '4;38;5;102',
    'UNDERLINE_PURPLE': '4;38;5;59',
    'UNDERLINE_RED': '4;38;5;09',
    'UNDERLINE_WHITE': '4;38;5;102',
    'UNDERLINE_YELLOW': '4;38;5;09',
    'WHITE': '38;5;102',
    'YELLOW': '38;5;09',
}

ALGOL_NU_STYLE = {
    'BACKGROUND_BLACK': '48;5;59',
    'BACKGROUND_BLUE': '48;5;59',
    'BACKGROUND_CYAN': '48;5;59',
    'BACKGROUND_GREEN': '48;5;59',
    'BACKGROUND_INTENSE_BLACK': '48;5;59',
    'BACKGROUND_INTENSE_BLUE': '48;5;102',
    'BACKGROUND_INTENSE_CYAN': '48;5;102',
    'BACKGROUND_INTENSE_GREEN': '48;5;102',
    'BACKGROUND_INTENSE_PURPLE': '48;5;102',
    'BACKGROUND_INTENSE_RED': '48;5;09',
    'BACKGROUND_INTENSE_WHITE': '48;5;102',
    'BACKGROUND_INTENSE_YELLOW': '48;5;102',
    'BACKGROUND_PURPLE': '48;5;59',
    'BACKGROUND_RED': '48;5;09',
    'BACKGROUND_WHITE': '48;5;102',
    'BACKGROUND_YELLOW': '48;5;09',
    'BLACK': '38;5;59',
    'BLUE': '38;5;59',
    'BOLD_BLACK': '1;38;5;59',
    'BOLD_BLUE': '1;38;5;59',
    'BOLD_CYAN': '1;38;5;59',
    'BOLD_GREEN': '1;38;5;59',
    'BOLD_INTENSE_BLACK': '1;38;5;59',
    'BOLD_INTENSE_BLUE': '1;38;5;102',
    'BOLD_INTENSE_CYAN': '1;38;5;102',
    'BOLD_INTENSE_GREEN': '1;38;5;102',
    'BOLD_INTENSE_PURPLE': '1;38;5;102',
    'BOLD_INTENSE_RED': '1;38;5;09',
    'BOLD_INTENSE_WHITE': '1;38;5;102',
    'BOLD_INTENSE_YELLOW': '1;38;5;102',
    'BOLD_PURPLE': '1;38;5;59',
    'BOLD_RED': '1;38;5;09',
    'BOLD_UNDERLINE_BLACK': '1;4;38;5;59',
    'BOLD_UNDERLINE_BLUE': '1;4;38;5;59',
    'BOLD_UNDERLINE_CYAN': '1;4;38;5;59',
    'BOLD_UNDERLINE_GREEN': '1;4;38;5;59',
    'BOLD_UNDERLINE_INTENSE_BLACK': '1;4;38;5;59',
    'BOLD_UNDERLINE_INTENSE_BLUE': '1;4;38;5;102',
    'BOLD_UNDERLINE_INTENSE_CYAN': '1;4;38;5;102',
    'BOLD_UNDERLINE_INTENSE_GREEN': '1;4;38;5;102',
    'BOLD_UNDERLINE_INTENSE_PURPLE': '1;4;38;5;102',
    'BOLD_UNDERLINE_INTENSE_RED': '1;4;38;5;09',
    'BOLD_UNDERLINE_INTENSE_WHITE': '1;4;38;5;102',
    'BOLD_UNDERLINE_INTENSE_YELLOW': '1;4;38;5;102',
    'BOLD_UNDERLINE_PURPLE': '1;4;38;5;59',
    'BOLD_UNDERLINE_RED': '1;4;38;5;09',
    'BOLD_UNDERLINE_WHITE': '1;4;38;5;102',
    'BOLD_UNDERLINE_YELLOW': '1;4;38;5;09',
    'BOLD_WHITE': '1;38;5;102',
    'BOLD_YELLOW': '1;38;5;09',
    'CYAN': '38;5;59',
    'GREEN': '38;5;59',
    'INTENSE_BLACK': '38;5;59',
    'INTENSE_BLUE': '38;5;102',
    'INTENSE_CYAN': '38;5;102',
    'INTENSE_GREEN': '38;5;102',
    'INTENSE_PURPLE': '38;5;102',
    'INTENSE_RED': '38;5;09',
    'INTENSE_WHITE': '38;5;102',
    'INTENSE_YELLOW': '38;5;102',
    'NO_COLOR': '0',
    'PURPLE': '38;5;59',
    'RED': '38;5;09',
    'UNDERLINE_BLACK': '4;38;5;59',
    'UNDERLINE_BLUE': '4;38;5;59',
    'UNDERLINE_CYAN': '4;38;5;59',
    'UNDERLINE_GREEN': '4;38;5;59',
    'UNDERLINE_INTENSE_BLACK': '4;38;5;59',
    'UNDERLINE_INTENSE_BLUE': '4;38;5;102',
    'UNDERLINE_INTENSE_CYAN': '4;38;5;102',
    'UNDERLINE_INTENSE_GREEN': '4;38;5;102',
    'UNDERLINE_INTENSE_PURPLE': '4;38;5;102',
    'UNDERLINE_INTENSE_RED': '4;38;5;09',
    'UNDERLINE_INTENSE_WHITE': '4;38;5;102',
    'UNDERLINE_INTENSE_YELLOW': '4;38;5;102',
    'UNDERLINE_PURPLE': '4;38;5;59',
    'UNDERLINE_RED': '4;38;5;09',
    'UNDERLINE_WHITE': '4;38;5;102',
    'UNDERLINE_YELLOW': '4;38;5;09',
    'WHITE': '38;5;102',
    'YELLOW': '38;5;09',
}

AUTUMN_STYLE = {
    'BACKGROUND_BLACK': '48;5;18',
    'BACKGROUND_BLUE': '48;5;19',
    'BACKGROUND_CYAN': '48;5;37',
    'BACKGROUND_GREEN': '48;5;34',
    'BACKGROUND_INTENSE_BLACK': '48;5;59',
    'BACKGROUND_INTENSE_BLUE': '48;5;33',
    'BACKGROUND_INTENSE_CYAN': '48;5;33',
    'BACKGROUND_INTENSE_GREEN': '48;5;64',
    'BACKGROUND_INTENSE_PURPLE': '48;5;217',
    'BACKGROUND_INTENSE_RED': '48;5;130',
    'BACKGROUND_INTENSE_WHITE': '48;5;145',
    'BACKGROUND_INTENSE_YELLOW': '48;5;217',
    'BACKGROUND_PURPLE': '48;5;90',
    'BACKGROUND_RED': '48;5;124',
    'BACKGROUND_WHITE': '48;5;145',
    'BACKGROUND_YELLOW': '48;5;130',
    'BLACK': '38;5;18',
    'BLUE': '38;5;19',
    'BOLD_BLACK': '1;38;5;18',
    'BOLD_BLUE': '1;38;5;19',
    'BOLD_CYAN': '1;38;5;37',
    'BOLD_GREEN': '1;38;5;34',
    'BOLD_INTENSE_BLACK': '1;38;5;59',
    'BOLD_INTENSE_BLUE': '1;38;5;33',
    'BOLD_INTENSE_CYAN': '1;38;5;33',
    'BOLD_INTENSE_GREEN': '1;38;5;64',
    'BOLD_INTENSE_PURPLE': '1;38;5;217',
    'BOLD_INTENSE_RED': '1;38;5;130',
    'BOLD_INTENSE_WHITE': '1;38;5;145',
    'BOLD_INTENSE_YELLOW': '1;38;5;217',
    'BOLD_PURPLE': '1;38;5;90',
    'BOLD_RED': '1;38;5;124',
    'BOLD_UNDERLINE_BLACK': '1;4;38;5;18',
    'BOLD_UNDERLINE_BLUE': '1;4;38;5;19',
    'BOLD_UNDERLINE_CYAN': '1;4;38;5;37',
    'BOLD_UNDERLINE_GREEN': '1;4;38;5;34',
    'BOLD_UNDERLINE_INTENSE_BLACK': '1;4;38;5;59',
    'BOLD_UNDERLINE_INTENSE_BLUE': '1;4;38;5;33',
    'BOLD_UNDERLINE_INTENSE_CYAN': '1;4;38;5;33',
    'BOLD_UNDERLINE_INTENSE_GREEN': '1;4;38;5;64',
    'BOLD_UNDERLINE_INTENSE_PURPLE': '1;4;38;5;217',
    'BOLD_UNDERLINE_INTENSE_RED': '1;4;38;5;130',
    'BOLD_UNDERLINE_INTENSE_WHITE': '1;4;38;5;145',
    'BOLD_UNDERLINE_INTENSE_YELLOW': '1;4;38;5;217',
    'BOLD_UNDERLINE_PURPLE': '1;4;38;5;90',
    'BOLD_UNDERLINE_RED': '1;4;38;5;124',
    'BOLD_UNDERLINE_WHITE': '1;4;38;5;145',
    'BOLD_UNDERLINE_YELLOW': '1;4;38;5;130',
    'BOLD_WHITE': '1;38;5;145',
    'BOLD_YELLOW': '1;38;5;130',
    'CYAN': '38;5;37',
    'GREEN': '38;5;34',
    'INTENSE_BLACK': '38;5;59',
    'INTENSE_BLUE': '38;5;33',
    'INTENSE_CYAN': '38;5;33',
    'INTENSE_GREEN': '38;5;64',
    'INTENSE_PURPLE': '38;5;217',
    'INTENSE_RED': '38;5;130',
    'INTENSE_WHITE': '38;5;145',
    'INTENSE_YELLOW': '38;5;217',
    'NO_COLOR': '0',
    'PURPLE': '38;5;90',
    'RED': '38;5;124',
    'UNDERLINE_BLACK': '4;38;5;18',
    'UNDERLINE_BLUE': '4;38;5;19',
    'UNDERLINE_CYAN': '4;38;5;37',
    'UNDERLINE_GREEN': '4;38;5;34',
    'UNDERLINE_INTENSE_BLACK': '4;38;5;59',
    'UNDERLINE_INTENSE_BLUE': '4;38;5;33',
    'UNDERLINE_INTENSE_CYAN': '4;38;5;33',
    'UNDERLINE_INTENSE_GREEN': '4;38;5;64',
    'UNDERLINE_INTENSE_PURPLE': '4;38;5;217',
    'UNDERLINE_INTENSE_RED': '4;38;5;130',
    'UNDERLINE_INTENSE_WHITE': '4;38;5;145',
    'UNDERLINE_INTENSE_YELLOW': '4;38;5;217',
    'UNDERLINE_PURPLE': '4;38;5;90',
    'UNDERLINE_RED': '4;38;5;124',
    'UNDERLINE_WHITE': '4;38;5;145',
    'UNDERLINE_YELLOW': '4;38;5;130',
    'WHITE': '38;5;145',
    'YELLOW': '38;5;130',
}

BORLAND_STYLE = {
    'BACKGROUND_BLACK': '48;5;16',
    'BACKGROUND_BLUE': '48;5;18',
    'BACKGROUND_CYAN': '48;5;30',
    'BACKGROUND_GREEN': '48;5;28',
    'BACKGROUND_INTENSE_BLACK': '48;5;59',
    'BACKGROUND_INTENSE_BLUE': '48;5;21',
    'BACKGROUND_INTENSE_CYAN': '48;5;194',
    'BACKGROUND_INTENSE_GREEN': '48;5;102',
    'BACKGROUND_INTENSE_PURPLE': '48;5;188',
    'BACKGROUND_INTENSE_RED': '48;5;09',
    'BACKGROUND_INTENSE_WHITE': '48;5;224',
    'BACKGROUND_INTENSE_YELLOW': '48;5;188',
    'BACKGROUND_PURPLE': '48;5;90',
    'BACKGROUND_RED': '48;5;124',
    'BACKGROUND_WHITE': '48;5;145',
    'BACKGROUND_YELLOW': '48;5;124',
    'BLACK': '38;5;16',
    'BLUE': '38;5;18',
    'BOLD_BLACK': '1;38;5;16',
    'BOLD_BLUE': '1;38;5;18',
    'BOLD_CYAN': '1;38;5;30',
    'BOLD_GREEN': '1;38;5;28',
    'BOLD_INTENSE_BLACK': '1;38;5;59',
    'BOLD_INTENSE_BLUE': '1;38;5;21',
    'BOLD_INTENSE_CYAN': '1;38;5;194',
    'BOLD_INTENSE_GREEN': '1;38;5;102',
    'BOLD_INTENSE_PURPLE': '1;38;5;188',
    'BOLD_INTENSE_RED': '1;38;5;09',
    'BOLD_INTENSE_WHITE': '1;38;5;224',
    'BOLD_INTENSE_YELLOW': '1;38;5;188',
    'BOLD_PURPLE': '1;38;5;90',
    'BOLD_RED': '1;38;5;124',
    'BOLD_UNDERLINE_BLACK': '1;4;38;5;16',
    'BOLD_UNDERLINE_BLUE': '1;4;38;5;18',
    'BOLD_UNDERLINE_CYAN': '1;4;38;5;30',
    'BOLD_UNDERLINE_GREEN': '1;4;38;5;28',
    'BOLD_UNDERLINE_INTENSE_BLACK': '1;4;38;5;59',
    'BOLD_UNDERLINE_INTENSE_BLUE': '1;4;38;5;21',
    'BOLD_UNDERLINE_INTENSE_CYAN': '1;4;38;5;194',
    'BOLD_UNDERLINE_INTENSE_GREEN': '1;4;38;5;102',
    'BOLD_UNDERLINE_INTENSE_PURPLE': '1;4;38;5;188',
    'BOLD_UNDERLINE_INTENSE_RED': '1;4;38;5;09',
    'BOLD_UNDERLINE_INTENSE_WHITE': '1;4;38;5;224',
    'BOLD_UNDERLINE_INTENSE_YELLOW': '1;4;38;5;188',
    'BOLD_UNDERLINE_PURPLE': '1;4;38;5;90',
    'BOLD_UNDERLINE_RED': '1;4;38;5;124',
    'BOLD_UNDERLINE_WHITE': '1;4;38;5;145',
    'BOLD_UNDERLINE_YELLOW': '1;4;38;5;124',
    'BOLD_WHITE': '1;38;5;145',
    'BOLD_YELLOW': '1;38;5;124',
    'CYAN': '38;5;30',
    'GREEN': '38;5;28',
    'INTENSE_BLACK': '38;5;59',
    'INTENSE_BLUE': '38;5;21',
    'INTENSE_CYAN': '38;5;194',
    'INTENSE_GREEN': '38;5;102',
    'INTENSE_PURPLE': '38;5;188',
    'INTENSE_RED': '38;5;09',
    'INTENSE_WHITE': '38;5;224',
    'INTENSE_YELLOW': '38;5;188',
    'NO_COLOR': '0',
    'PURPLE': '38;5;90',
    'RED': '38;5;124',
    'UNDERLINE_BLACK': '4;38;5;16',
    'UNDERLINE_BLUE': '4;38;5;18',
    'UNDERLINE_CYAN': '4;38;5;30',
    'UNDERLINE_GREEN': '4;38;5;28',
    'UNDERLINE_INTENSE_BLACK': '4;38;5;59',
    'UNDERLINE_INTENSE_BLUE': '4;38;5;21',
    'UNDERLINE_INTENSE_CYAN': '4;38;5;194',
    'UNDERLINE_INTENSE_GREEN': '4;38;5;102',
    'UNDERLINE_INTENSE_PURPLE': '4;38;5;188',
    'UNDERLINE_INTENSE_RED': '4;38;5;09',
    'UNDERLINE_INTENSE_WHITE': '4;38;5;224',
    'UNDERLINE_INTENSE_YELLOW': '4;38;5;188',
    'UNDERLINE_PURPLE': '4;38;5;90',
    'UNDERLINE_RED': '4;38;5;124',
    'UNDERLINE_WHITE': '4;38;5;145',
    'UNDERLINE_YELLOW': '4;38;5;124',
    'WHITE': '38;5;145',
    'YELLOW': '38;5;124',
}

COLORFUL_STYLE = {
    'BACKGROUND_BLACK': '48;5;16',
    'BACKGROUND_BLUE': '48;5;20',
    'BACKGROUND_CYAN': '48;5;31',
    'BACKGROUND_GREEN': '48;5;34',
    'BACKGROUND_INTENSE_BLACK': '48;5;59',
    'BACKGROUND_INTENSE_BLUE': '48;5;61',
    'BACKGROUND_INTENSE_CYAN': '48;5;145',
    'BACKGROUND_INTENSE_GREEN': '48;5;102',
    'BACKGROUND_INTENSE_PURPLE': '48;5;217',
    'BACKGROUND_INTENSE_RED': '48;5;166',
    'BACKGROUND_INTENSE_WHITE': '48;5;15',
    'BACKGROUND_INTENSE_YELLOW': '48;5;217',
    'BACKGROUND_PURPLE': '48;5;90',
    'BACKGROUND_RED': '48;5;124',
    'BACKGROUND_WHITE': '48;5;145',
    'BACKGROUND_YELLOW': '48;5;130',
    'BLACK': '38;5;16',
    'BLUE': '38;5;20',
    'BOLD_BLACK': '1;38;5;16',
    'BOLD_BLUE': '1;38;5;20',
    'BOLD_CYAN': '1;38;5;31',
    'BOLD_GREEN': '1;38;5;34',
    'BOLD_INTENSE_BLACK': '1;38;5;59',
    'BOLD_INTENSE_BLUE': '1;38;5;61',
    'BOLD_INTENSE_CYAN': '1;38;5;145',
    'BOLD_INTENSE_GREEN': '1;38;5;102',
    'BOLD_INTENSE_PURPLE': '1;38;5;217',
    'BOLD_INTENSE_RED': '1;38;5;166',
    'BOLD_INTENSE_WHITE': '1;38;5;15',
    'BOLD_INTENSE_YELLOW': '1;38;5;217',
    'BOLD_PURPLE': '1;38;5;90',
    'BOLD_RED': '1;38;5;124',
    'BOLD_UNDERLINE_BLACK': '1;4;38;5;16',
    'BOLD_UNDERLINE_BLUE': '1;4;38;5;20',
    'BOLD_UNDERLINE_CYAN': '1;4;38;5;31',
    'BOLD_UNDERLINE_GREEN': '1;4;38;5;34',
    'BOLD_UNDERLINE_INTENSE_BLACK': '1;4;38;5;59',
    'BOLD_UNDERLINE_INTENSE_BLUE': '1;4;38;5;61',
    'BOLD_UNDERLINE_INTENSE_CYAN': '1;4;38;5;145',
    'BOLD_UNDERLINE_INTENSE_GREEN': '1;4;38;5;102',
    'BOLD_UNDERLINE_INTENSE_PURPLE': '1;4;38;5;217',
    'BOLD_UNDERLINE_INTENSE_RED': '1;4;38;5;166',
    'BOLD_UNDERLINE_INTENSE_WHITE': '1;4;38;5;15',
    'BOLD_UNDERLINE_INTENSE_YELLOW': '1;4;38;5;217',
    'BOLD_UNDERLINE_PURPLE': '1;4;38;5;90',
    'BOLD_UNDERLINE_RED': '1;4;38;5;124',
    'BOLD_UNDERLINE_WHITE': '1;4;38;5;145',
    'BOLD_UNDERLINE_YELLOW': '1;4;38;5;130',
    'BOLD_WHITE': '1;38;5;145',
    'BOLD_YELLOW': '1;38;5;130',
    'CYAN': '38;5;31',
    'GREEN': '38;5;34',
    'INTENSE_BLACK': '38;5;59',
    'INTENSE_BLUE': '38;5;61',
    'INTENSE_CYAN': '38;5;145',
    'INTENSE_GREEN': '38;5;102',
    'INTENSE_PURPLE': '38;5;217',
    'INTENSE_RED': '38;5;166',
    'INTENSE_WHITE': '38;5;15',
    'INTENSE_YELLOW': '38;5;217',
    'NO_COLOR': '0',
    'PURPLE': '38;5;90',
    'RED': '38;5;124',
    'UNDERLINE_BLACK': '4;38;5;16',
    'UNDERLINE_BLUE': '4;38;5;20',
    'UNDERLINE_CYAN': '4;38;5;31',
    'UNDERLINE_GREEN': '4;38;5;34',
    'UNDERLINE_INTENSE_BLACK': '4;38;5;59',
    'UNDERLINE_INTENSE_BLUE': '4;38;5;61',
    'UNDERLINE_INTENSE_CYAN': '4;38;5;145',
    'UNDERLINE_INTENSE_GREEN': '4;38;5;102',
    'UNDERLINE_INTENSE_PURPLE': '4;38;5;217',
    'UNDERLINE_INTENSE_RED': '4;38;5;166',
    'UNDERLINE_INTENSE_WHITE': '4;38;5;15',
    'UNDERLINE_INTENSE_YELLOW': '4;38;5;217',
    'UNDERLINE_PURPLE': '4;38;5;90',
    'UNDERLINE_RED': '4;38;5;124',
    'UNDERLINE_WHITE': '4;38;5;145',
    'UNDERLINE_YELLOW': '4;38;5;130',
    'WHITE': '38;5;145',
    'YELLOW': '38;5;130',
}

EMACS_STYLE = {
    'BACKGROUND_BLACK': '48;5;28',
    'BACKGROUND_BLUE': '48;5;18',
    'BACKGROUND_CYAN': '48;5;26',
    'BACKGROUND_GREEN': '48;5;34',
    'BACKGROUND_INTENSE_BLACK': '48;5;59',
    'BACKGROUND_INTENSE_BLUE': '48;5;26',
    'BACKGROUND_INTENSE_CYAN': '48;5;145',
    'BACKGROUND_INTENSE_GREEN': '48;5;34',
    'BACKGROUND_INTENSE_PURPLE': '48;5;129',
    'BACKGROUND_INTENSE_RED': '48;5;167',
    'BACKGROUND_INTENSE_WHITE': '48;5;145',
    'BACKGROUND_INTENSE_YELLOW': '48;5;145',
    'BACKGROUND_PURPLE': '48;5;90',
    'BACKGROUND_RED': '48;5;124',
    'BACKGROUND_WHITE': '48;5;145',
    'BACKGROUND_YELLOW': '48;5;130',
    'BLACK': '38;5;28',
    'BLUE': '38;5;18',
    'BOLD_BLACK': '1;38;5;28',
    'BOLD_BLUE': '1;38;5;18',
    'BOLD_CYAN': '1;38;5;26',
    'BOLD_GREEN': '1;38;5;34',
    'BOLD_INTENSE_BLACK': '1;38;5;59',
    'BOLD_INTENSE_BLUE': '1;38;5;26',
    'BOLD_INTENSE_CYAN': '1;38;5;145',
    'BOLD_INTENSE_GREEN': '1;38;5;34',
    'BOLD_INTENSE_PURPLE': '1;38;5;129',
    'BOLD_INTENSE_RED': '1;38;5;167',
    'BOLD_INTENSE_WHITE': '1;38;5;145',
    'BOLD_INTENSE_YELLOW': '1;38;5;145',
    'BOLD_PURPLE': '1;38;5;90',
    'BOLD_RED': '1;38;5;124',
    'BOLD_UNDERLINE_BLACK': '1;4;38;5;28',
    'BOLD_UNDERLINE_BLUE': '1;4;38;5;18',
    'BOLD_UNDERLINE_CYAN': '1;4;38;5;26',
    'BOLD_UNDERLINE_GREEN': '1;4;38;5;34',
    'BOLD_UNDERLINE_INTENSE_BLACK': '1;4;38;5;59',
    'BOLD_UNDERLINE_INTENSE_BLUE': '1;4;38;5;26',
    'BOLD_UNDERLINE_INTENSE_CYAN': '1;4;38;5;145',
    'BOLD_UNDERLINE_INTENSE_GREEN': '1;4;38;5;34',
    'BOLD_UNDERLINE_INTENSE_PURPLE': '1;4;38;5;129',
    'BOLD_UNDERLINE_INTENSE_RED': '1;4;38;5;167',
    'BOLD_UNDERLINE_INTENSE_WHITE': '1;4;38;5;145',
    'BOLD_UNDERLINE_INTENSE_YELLOW': '1;4;38;5;145',
    'BOLD_UNDERLINE_PURPLE': '1;4;38;5;90',
    'BOLD_UNDERLINE_RED': '1;4;38;5;124',
    'BOLD_UNDERLINE_WHITE': '1;4;38;5;145',
    'BOLD_UNDERLINE_YELLOW': '1;4;38;5;130',
    'BOLD_WHITE': '1;38;5;145',
    'BOLD_YELLOW': '1;38;5;130',
    'CYAN': '38;5;26',
    'GREEN': '38;5;34',
    'INTENSE_BLACK': '38;5;59',
    'INTENSE_BLUE': '38;5;26',
    'INTENSE_CYAN': '38;5;145',
    'INTENSE_GREEN': '38;5;34',
    'INTENSE_PURPLE': '38;5;129',
    'INTENSE_RED': '38;5;167',
    'INTENSE_WHITE': '38;5;145',
    'INTENSE_YELLOW': '38;5;145',
    'NO_COLOR': '0',
    'PURPLE': '38;5;90',
    'RED': '38;5;124',
    'UNDERLINE_BLACK': '4;38;5;28',
    'UNDERLINE_BLUE': '4;38;5;18',
    'UNDERLINE_CYAN': '4;38;5;26',
    'UNDERLINE_GREEN': '4;38;5;34',
    'UNDERLINE_INTENSE_BLACK': '4;38;5;59',
    'UNDERLINE_INTENSE_BLUE': '4;38;5;26',
    'UNDERLINE_INTENSE_CYAN': '4;38;5;145',
    'UNDERLINE_INTENSE_GREEN': '4;38;5;34',
    'UNDERLINE_INTENSE_PURPLE': '4;38;5;129',
    'UNDERLINE_INTENSE_RED': '4;38;5;167',
    'UNDERLINE_INTENSE_WHITE': '4;38;5;145',
    'UNDERLINE_INTENSE_YELLOW': '4;38;5;145',
    'UNDERLINE_PURPLE': '4;38;5;90',
    'UNDERLINE_RED': '4;38;5;124',
    'UNDERLINE_WHITE': '4;38;5;145',
    'UNDERLINE_YELLOW': '4;38;5;130',
    'WHITE': '38;5;145',
    'YELLOW': '38;5;130',
}

FRIENDLY_STYLE = {
    'BACKGROUND_BLACK': '48;5;22',
    'BACKGROUND_BLUE': '48;5;18',
    'BACKGROUND_CYAN': '48;5;31',
    'BACKGROUND_GREEN': '48;5;34',
    'BACKGROUND_INTENSE_BLACK': '48;5;59',
    'BACKGROUND_INTENSE_BLUE': '48;5;74',
    'BACKGROUND_INTENSE_CYAN': '48;5;74',
    'BACKGROUND_INTENSE_GREEN': '48;5;71',
    'BACKGROUND_INTENSE_PURPLE': '48;5;134',
    'BACKGROUND_INTENSE_RED': '48;5;167',
    'BACKGROUND_INTENSE_WHITE': '48;5;15',
    'BACKGROUND_INTENSE_YELLOW': '48;5;145',
    'BACKGROUND_PURPLE': '48;5;90',
    'BACKGROUND_RED': '48;5;124',
    'BACKGROUND_WHITE': '48;5;145',
    'BACKGROUND_YELLOW': '48;5;166',
    'BLACK': '38;5;22',
    'BLUE': '38;5;18',
    'BOLD_BLACK': '1;38;5;22',
    'BOLD_BLUE': '1;38;5;18',
    'BOLD_CYAN': '1;38;5;31',
    'BOLD_GREEN': '1;38;5;34',
    'BOLD_INTENSE_BLACK': '1;38;5;59',
    'BOLD_INTENSE_BLUE': '1;38;5;74',
    'BOLD_INTENSE_CYAN': '1;38;5;74',
    'BOLD_INTENSE_GREEN': '1;38;5;71',
    'BOLD_INTENSE_PURPLE': '1;38;5;134',
    'BOLD_INTENSE_RED': '1;38;5;167',
    'BOLD_INTENSE_WHITE': '1;38;5;15',
    'BOLD_INTENSE_YELLOW': '1;38;5;145',
    'BOLD_PURPLE': '1;38;5;90',
    'BOLD_RED': '1;38;5;124',
    'BOLD_UNDERLINE_BLACK': '1;4;38;5;22',
    'BOLD_UNDERLINE_BLUE': '1;4;38;5;18',
    'BOLD_UNDERLINE_CYAN': '1;4;38;5;31',
    'BOLD_UNDERLINE_GREEN': '1;4;38;5;34',
    'BOLD_UNDERLINE_INTENSE_BLACK': '1;4;38;5;59',
    'BOLD_UNDERLINE_INTENSE_BLUE': '1;4;38;5;74',
    'BOLD_UNDERLINE_INTENSE_CYAN': '1;4;38;5;74',
    'BOLD_UNDERLINE_INTENSE_GREEN': '1;4;38;5;71',
    'BOLD_UNDERLINE_INTENSE_PURPLE': '1;4;38;5;134',
    'BOLD_UNDERLINE_INTENSE_RED': '1;4;38;5;167',
    'BOLD_UNDERLINE_INTENSE_WHITE': '1;4;38;5;15',
    'BOLD_UNDERLINE_INTENSE_YELLOW': '1;4;38;5;145',
    'BOLD_UNDERLINE_PURPLE': '1;4;38;5;90',
    'BOLD_UNDERLINE_RED': '1;4;38;5;124',
    'BOLD_UNDERLINE_WHITE': '1;4;38;5;145',
    'BOLD_UNDERLINE_YELLOW': '1;4;38;5;166',
    'BOLD_WHITE': '1;38;5;145',
    'BOLD_YELLOW': '1;38;5;166',
    'CYAN': '38;5;31',
    'GREEN': '38;5;34',
    'INTENSE_BLACK': '38;5;59',
    'INTENSE_BLUE': '38;5;74',
    'INTENSE_CYAN': '38;5;74',
    'INTENSE_GREEN': '38;5;71',
    'INTENSE_PURPLE': '38;5;134',
    'INTENSE_RED': '38;5;167',
    'INTENSE_WHITE': '38;5;15',
    'INTENSE_YELLOW': '38;5;145',
    'NO_COLOR': '0',
    'PURPLE': '38;5;90',
    'RED': '38;5;124',
    'UNDERLINE_BLACK': '4;38;5;22',
    'UNDERLINE_BLUE': '4;38;5;18',
    'UNDERLINE_CYAN': '4;38;5;31',
    'UNDERLINE_GREEN': '4;38;5;34',
    'UNDERLINE_INTENSE_BLACK': '4;38;5;59',
    'UNDERLINE_INTENSE_BLUE': '4;38;5;74',
    'UNDERLINE_INTENSE_CYAN': '4;38;5;74',
    'UNDERLINE_INTENSE_GREEN': '4;38;5;71',
    'UNDERLINE_INTENSE_PURPLE': '4;38;5;134',
    'UNDERLINE_INTENSE_RED': '4;38;5;167',
    'UNDERLINE_INTENSE_WHITE': '4;38;5;15',
    'UNDERLINE_INTENSE_YELLOW': '4;38;5;145',
    'UNDERLINE_PURPLE': '4;38;5;90',
    'UNDERLINE_RED': '4;38;5;124',
    'UNDERLINE_WHITE': '4;38;5;145',
    'UNDERLINE_YELLOW': '4;38;5;166',
    'WHITE': '38;5;145',
    'YELLOW': '38;5;166',
}

FRUITY_STYLE = {
    'BACKGROUND_BLACK': '48;5;16',
    'BACKGROUND_BLUE': '48;5;32',
    'BACKGROUND_CYAN': '48;5;32',
    'BACKGROUND_GREEN': '48;5;28',
    'BACKGROUND_INTENSE_BLACK': '48;5;59',
    'BACKGROUND_INTENSE_BLUE': '48;5;33',
    'BACKGROUND_INTENSE_CYAN': '48;5;33',
    'BACKGROUND_INTENSE_GREEN': '48;5;102',
    'BACKGROUND_INTENSE_PURPLE': '48;5;198',
    'BACKGROUND_INTENSE_RED': '48;5;202',
    'BACKGROUND_INTENSE_WHITE': '48;5;15',
    'BACKGROUND_INTENSE_YELLOW': '48;5;187',
    'BACKGROUND_PURPLE': '48;5;198',
    'BACKGROUND_RED': '48;5;09',
    'BACKGROUND_WHITE': '48;5;187',
    'BACKGROUND_YELLOW': '48;5;202',
    'BLACK': '38;5;16',
    'BLUE': '38;5;32',
    'BOLD_BLACK': '1;38;5;16',
    'BOLD_BLUE': '1;38;5;32',
    'BOLD_CYAN': '1;38;5;32',
    'BOLD_GREEN': '1;38;5;28',
    'BOLD_INTENSE_BLACK': '1;38;5;59',
    'BOLD_INTENSE_BLUE': '1;38;5;33',
    'BOLD_INTENSE_CYAN': '1;38;5;33',
    'BOLD_INTENSE_GREEN': '1;38;5;102',
    'BOLD_INTENSE_PURPLE': '1;38;5;198',
    'BOLD_INTENSE_RED': '1;38;5;202',
    'BOLD_INTENSE_WHITE': '1;38;5;15',
    'BOLD_INTENSE_YELLOW': '1;38;5;187',
    'BOLD_PURPLE': '1;38;5;198',
    'BOLD_RED': '1;38;5;09',
    'BOLD_UNDERLINE_BLACK': '1;4;38;5;16',
    'BOLD_UNDERLINE_BLUE': '1;4;38;5;32',
    'BOLD_UNDERLINE_CYAN': '1;4;38;5;32',
    'BOLD_UNDERLINE_GREEN': '1;4;38;5;28',
    'BOLD_UNDERLINE_INTENSE_BLACK': '1;4;38;5;59',
    'BOLD_UNDERLINE_INTENSE_BLUE': '1;4;38;5;33',
    'BOLD_UNDERLINE_INTENSE_CYAN': '1;4;38;5;33',
    'BOLD_UNDERLINE_INTENSE_GREEN': '1;4;38;5;102',
    'BOLD_UNDERLINE_INTENSE_PURPLE': '1;4;38;5;198',
    'BOLD_UNDERLINE_INTENSE_RED': '1;4;38;5;202',
    'BOLD_UNDERLINE_INTENSE_WHITE': '1;4;38;5;15',
    'BOLD_UNDERLINE_INTENSE_YELLOW': '1;4;38;5;187',
    'BOLD_UNDERLINE_PURPLE': '1;4;38;5;198',
    'BOLD_UNDERLINE_RED': '1;4;38;5;09',
    'BOLD_UNDERLINE_WHITE': '1;4;38;5;187',
    'BOLD_UNDERLINE_YELLOW': '1;4;38;5;202',
    'BOLD_WHITE': '1;38;5;187',
    'BOLD_YELLOW': '1;38;5;202',
    'CYAN': '38;5;32',
    'GREEN': '38;5;28',
    'INTENSE_BLACK': '38;5;59',
    'INTENSE_BLUE': '38;5;33',
    'INTENSE_CYAN': '38;5;33',
    'INTENSE_GREEN': '38;5;102',
    'INTENSE_PURPLE': '38;5;198',
    'INTENSE_RED': '38;5;202',
    'INTENSE_WHITE': '38;5;15',
    'INTENSE_YELLOW': '38;5;187',
    'NO_COLOR': '0',
    'PURPLE': '38;5;198',
    'RED': '38;5;09',
    'UNDERLINE_BLACK': '4;38;5;16',
    'UNDERLINE_BLUE': '4;38;5;32',
    'UNDERLINE_CYAN': '4;38;5;32',
    'UNDERLINE_GREEN': '4;38;5;28',
    'UNDERLINE_INTENSE_BLACK': '4;38;5;59',
    'UNDERLINE_INTENSE_BLUE': '4;38;5;33',
    'UNDERLINE_INTENSE_CYAN': '4;38;5;33',
    'UNDERLINE_INTENSE_GREEN': '4;38;5;102',
    'UNDERLINE_INTENSE_PURPLE': '4;38;5;198',
    'UNDERLINE_INTENSE_RED': '4;38;5;202',
    'UNDERLINE_INTENSE_WHITE': '4;38;5;15',
    'UNDERLINE_INTENSE_YELLOW': '4;38;5;187',
    'UNDERLINE_PURPLE': '4;38;5;198',
    'UNDERLINE_RED': '4;38;5;09',
    'UNDERLINE_WHITE': '4;38;5;187',
    'UNDERLINE_YELLOW': '4;38;5;202',
    'WHITE': '38;5;187',
    'YELLOW': '38;5;202',
}

IGOR_STYLE = {
    'BACKGROUND_BLACK': '48;5;34',
    'BACKGROUND_BLUE': '48;5;21',
    'BACKGROUND_CYAN': '48;5;30',
    'BACKGROUND_GREEN': '48;5;34',
    'BACKGROUND_INTENSE_BLACK': '48;5;30',
    'BACKGROUND_INTENSE_BLUE': '48;5;21',
    'BACKGROUND_INTENSE_CYAN': '48;5;30',
    'BACKGROUND_INTENSE_GREEN': '48;5;34',
    'BACKGROUND_INTENSE_PURPLE': '48;5;163',
    'BACKGROUND_INTENSE_RED': '48;5;166',
    'BACKGROUND_INTENSE_WHITE': '48;5;163',
    'BACKGROUND_INTENSE_YELLOW': '48;5;166',
    'BACKGROUND_PURPLE': '48;5;163',
    'BACKGROUND_RED': '48;5;166',
    'BACKGROUND_WHITE': '48;5;163',
    'BACKGROUND_YELLOW': '48;5;166',
    'BLACK': '38;5;34',
    'BLUE': '38;5;21',
    'BOLD_BLACK': '1;38;5;34',
    'BOLD_BLUE': '1;38;5;21',
    'BOLD_CYAN': '1;38;5;30',
    'BOLD_GREEN': '1;38;5;34',
    'BOLD_INTENSE_BLACK': '1;38;5;30',
    'BOLD_INTENSE_BLUE': '1;38;5;21',
    'BOLD_INTENSE_CYAN': '1;38;5;30',
    'BOLD_INTENSE_GREEN': '1;38;5;34',
    'BOLD_INTENSE_PURPLE': '1;38;5;163',
    'BOLD_INTENSE_RED': '1;38;5;166',
    'BOLD_INTENSE_WHITE': '1;38;5;163',
    'BOLD_INTENSE_YELLOW': '1;38;5;166',
    'BOLD_PURPLE': '1;38;5;163',
    'BOLD_RED': '1;38;5;166',
    'BOLD_UNDERLINE_BLACK': '1;4;38;5;34',
    'BOLD_UNDERLINE_BLUE': '1;4;38;5;21',
    'BOLD_UNDERLINE_CYAN': '1;4;38;5;30',
    'BOLD_UNDERLINE_GREEN': '1;4;38;5;34',
    'BOLD_UNDERLINE_INTENSE_BLACK': '1;4;38;5;30',
    'BOLD_UNDERLINE_INTENSE_BLUE': '1;4;38;5;21',
    'BOLD_UNDERLINE_INTENSE_CYAN': '1;4;38;5;30',
    'BOLD_UNDERLINE_INTENSE_GREEN': '1;4;38;5;34',
    'BOLD_UNDERLINE_INTENSE_PURPLE': '1;4;38;5;163',
    'BOLD_UNDERLINE_INTENSE_RED': '1;4;38;5;166',
    'BOLD_UNDERLINE_INTENSE_WHITE': '1;4;38;5;163',
    'BOLD_UNDERLINE_INTENSE_YELLOW': '1;4;38;5;166',
    'BOLD_UNDERLINE_PURPLE': '1;4;38;5;163',
    'BOLD_UNDERLINE_RED': '1;4;38;5;166',
    'BOLD_UNDERLINE_WHITE': '1;4;38;5;163',
    'BOLD_UNDERLINE_YELLOW': '1;4;38;5;166',
    'BOLD_WHITE': '1;38;5;163',
    'BOLD_YELLOW': '1;38;5;166',
    'CYAN': '38;5;30',
    'GREEN': '38;5;34',
    'INTENSE_BLACK': '38;5;30',
    'INTENSE_BLUE': '38;5;21',
    'INTENSE_CYAN': '38;5;30',
    'INTENSE_GREEN': '38;5;34',
    'INTENSE_PURPLE': '38;5;163',
    'INTENSE_RED': '38;5;166',
    'INTENSE_WHITE': '38;5;163',
    'INTENSE_YELLOW': '38;5;166',
    'NO_COLOR': '0',
    'PURPLE': '38;5;163',
    'RED': '38;5;166',
    'UNDERLINE_BLACK': '4;38;5;34',
    'UNDERLINE_BLUE': '4;38;5;21',
    'UNDERLINE_CYAN': '4;38;5;30',
    'UNDERLINE_GREEN': '4;38;5;34',
    'UNDERLINE_INTENSE_BLACK': '4;38;5;30',
    'UNDERLINE_INTENSE_BLUE': '4;38;5;21',
    'UNDERLINE_INTENSE_CYAN': '4;38;5;30',
    'UNDERLINE_INTENSE_GREEN': '4;38;5;34',
    'UNDERLINE_INTENSE_PURPLE': '4;38;5;163',
    'UNDERLINE_INTENSE_RED': '4;38;5;166',
    'UNDERLINE_INTENSE_WHITE': '4;38;5;163',
    'UNDERLINE_INTENSE_YELLOW': '4;38;5;166',
    'UNDERLINE_PURPLE': '4;38;5;163',
    'UNDERLINE_RED': '4;38;5;166',
    'UNDERLINE_WHITE': '4;38;5;163',
    'UNDERLINE_YELLOW': '4;38;5;166',
    'WHITE': '38;5;163',
    'YELLOW': '38;5;166',
}

LOVELACE_STYLE = {
    'BACKGROUND_BLACK': '48;5;59',
    'BACKGROUND_BLUE': '48;5;25',
    'BACKGROUND_CYAN': '48;5;29',
    'BACKGROUND_GREEN': '48;5;65',
    'BACKGROUND_INTENSE_BLACK': '48;5;59',
    'BACKGROUND_INTENSE_BLUE': '48;5;25',
    'BACKGROUND_INTENSE_CYAN': '48;5;102',
    'BACKGROUND_INTENSE_GREEN': '48;5;29',
    'BACKGROUND_INTENSE_PURPLE': '48;5;133',
    'BACKGROUND_INTENSE_RED': '48;5;131',
    'BACKGROUND_INTENSE_WHITE': '48;5;102',
    'BACKGROUND_INTENSE_YELLOW': '48;5;136',
    'BACKGROUND_PURPLE': '48;5;133',
    'BACKGROUND_RED': '48;5;124',
    'BACKGROUND_WHITE': '48;5;102',
    'BACKGROUND_YELLOW': '48;5;130',
    'BLACK': '38;5;59',
    'BLUE': '38;5;25',
    'BOLD_BLACK': '1;38;5;59',
    'BOLD_BLUE': '1;38;5;25',
    'BOLD_CYAN': '1;38;5;29',
    'BOLD_GREEN': '1;38;5;65',
    'BOLD_INTENSE_BLACK': '1;38;5;59',
    'BOLD_INTENSE_BLUE': '1;38;5;25',
    'BOLD_INTENSE_CYAN': '1;38;5;102',
    'BOLD_INTENSE_GREEN': '1;38;5;29',
    'BOLD_INTENSE_PURPLE': '1;38;5;133',
    'BOLD_INTENSE_RED': '1;38;5;131',
    'BOLD_INTENSE_WHITE': '1;38;5;102',
    'BOLD_INTENSE_YELLOW': '1;38;5;136',
    'BOLD_PURPLE': '1;38;5;133',
    'BOLD_RED': '1;38;5;124',
    'BOLD_UNDERLINE_BLACK': '1;4;38;5;59',
    'BOLD_UNDERLINE_BLUE': '1;4;38;5;25',
    'BOLD_UNDERLINE_CYAN': '1;4;38;5;29',
    'BOLD_UNDERLINE_GREEN': '1;4;38;5;65',
    'BOLD_UNDERLINE_INTENSE_BLACK': '1;4;38;5;59',
    'BOLD_UNDERLINE_INTENSE_BLUE': '1;4;38;5;25',
    'BOLD_UNDERLINE_INTENSE_CYAN': '1;4;38;5;102',
    'BOLD_UNDERLINE_INTENSE_GREEN': '1;4;38;5;29',
    'BOLD_UNDERLINE_INTENSE_PURPLE': '1;4;38;5;133',
    'BOLD_UNDERLINE_INTENSE_RED': '1;4;38;5;131',
    'BOLD_UNDERLINE_INTENSE_WHITE': '1;4;38;5;102',
    'BOLD_UNDERLINE_INTENSE_YELLOW': '1;4;38;5;136',
    'BOLD_UNDERLINE_PURPLE': '1;4;38;5;133',
    'BOLD_UNDERLINE_RED': '1;4;38;5;124',
    'BOLD_UNDERLINE_WHITE': '1;4;38;5;102',
    'BOLD_UNDERLINE_YELLOW': '1;4;38;5;130',
    'BOLD_WHITE': '1;38;5;102',
    'BOLD_YELLOW': '1;38;5;130',
    'CYAN': '38;5;29',
    'GREEN': '38;5;65',
    'INTENSE_BLACK': '38;5;59',
    'INTENSE_BLUE': '38;5;25',
    'INTENSE_CYAN': '38;5;102',
    'INTENSE_GREEN': '38;5;29',
    'INTENSE_PURPLE': '38;5;133',
    'INTENSE_RED': '38;5;131',
    'INTENSE_WHITE': '38;5;102',
    'INTENSE_YELLOW': '38;5;136',
    'NO_COLOR': '0',
    'PURPLE': '38;5;133',
    'RED': '38;5;124',
    'UNDERLINE_BLACK': '4;38;5;59',
    'UNDERLINE_BLUE': '4;38;5;25',
    'UNDERLINE_CYAN': '4;38;5;29',
    'UNDERLINE_GREEN': '4;38;5;65',
    'UNDERLINE_INTENSE_BLACK': '4;38;5;59',
    'UNDERLINE_INTENSE_BLUE': '4;38;5;25',
    'UNDERLINE_INTENSE_CYAN': '4;38;5;102',
    'UNDERLINE_INTENSE_GREEN': '4;38;5;29',
    'UNDERLINE_INTENSE_PURPLE': '4;38;5;133',
    'UNDERLINE_INTENSE_RED': '4;38;5;131',
    'UNDERLINE_INTENSE_WHITE': '4;38;5;102',
    'UNDERLINE_INTENSE_YELLOW': '4;38;5;136',
    'UNDERLINE_PURPLE': '4;38;5;133',
    'UNDERLINE_RED': '4;38;5;124',
    'UNDERLINE_WHITE': '4;38;5;102',
    'UNDERLINE_YELLOW': '4;38;5;130',
    'WHITE': '38;5;102',
    'YELLOW': '38;5;130',
}

MANNI_STYLE = {
    'BACKGROUND_BLACK': '48;5;16',
    'BACKGROUND_BLUE': '48;5;18',
    'BACKGROUND_CYAN': '48;5;30',
    'BACKGROUND_GREEN': '48;5;40',
    'BACKGROUND_INTENSE_BLACK': '48;5;59',
    'BACKGROUND_INTENSE_BLUE': '48;5;105',
    'BACKGROUND_INTENSE_CYAN': '48;5;45',
    'BACKGROUND_INTENSE_GREEN': '48;5;113',
    'BACKGROUND_INTENSE_PURPLE': '48;5;165',
    'BACKGROUND_INTENSE_RED': '48;5;202',
    'BACKGROUND_INTENSE_WHITE': '48;5;224',
    'BACKGROUND_INTENSE_YELLOW': '48;5;221',
    'BACKGROUND_PURPLE': '48;5;165',
    'BACKGROUND_RED': '48;5;124',
    'BACKGROUND_WHITE': '48;5;145',
    'BACKGROUND_YELLOW': '48;5;166',
    'BLACK': '38;5;16',
    'BLUE': '38;5;18',
    'BOLD_BLACK': '1;38;5;16',
    'BOLD_BLUE': '1;38;5;18',
    'BOLD_CYAN': '1;38;5;30',
    'BOLD_GREEN': '1;38;5;40',
    'BOLD_INTENSE_BLACK': '1;38;5;59',
    'BOLD_INTENSE_BLUE': '1;38;5;105',
    'BOLD_INTENSE_CYAN': '1;38;5;45',
    'BOLD_INTENSE_GREEN': '1;38;5;113',
    'BOLD_INTENSE_PURPLE': '1;38;5;165',
    'BOLD_INTENSE_RED': '1;38;5;202',
    'BOLD_INTENSE_WHITE': '1;38;5;224',
    'BOLD_INTENSE_YELLOW': '1;38;5;221',
    'BOLD_PURPLE': '1;38;5;165',
    'BOLD_RED': '1;38;5;124',
    'BOLD_UNDERLINE_BLACK': '1;4;38;5;16',
    'BOLD_UNDERLINE_BLUE': '1;4;38;5;18',
    'BOLD_UNDERLINE_CYAN': '1;4;38;5;30',
    'BOLD_UNDERLINE_GREEN': '1;4;38;5;40',
    'BOLD_UNDERLINE_INTENSE_BLACK': '1;4;38;5;59',
    'BOLD_UNDERLINE_INTENSE_BLUE': '1;4;38;5;105',
    'BOLD_UNDERLINE_INTENSE_CYAN': '1;4;38;5;45',
    'BOLD_UNDERLINE_INTENSE_GREEN': '1;4;38;5;113',
    'BOLD_UNDERLINE_INTENSE_PURPLE': '1;4;38;5;165',
    'BOLD_UNDERLINE_INTENSE_RED': '1;4;38;5;202',
    'BOLD_UNDERLINE_INTENSE_WHITE': '1;4;38;5;224',
    'BOLD_UNDERLINE_INTENSE_YELLOW': '1;4;38;5;221',
    'BOLD_UNDERLINE_PURPLE': '1;4;38;5;165',
    'BOLD_UNDERLINE_RED': '1;4;38;5;124',
    'BOLD_UNDERLINE_WHITE': '1;4;38;5;145',
    'BOLD_UNDERLINE_YELLOW': '1;4;38;5;166',
    'BOLD_WHITE': '1;38;5;145',
    'BOLD_YELLOW': '1;38;5;166',
    'CYAN': '38;5;30',
    'GREEN': '38;5;40',
    'INTENSE_BLACK': '38;5;59',
    'INTENSE_BLUE': '38;5;105',
    'INTENSE_CYAN': '38;5;45',
    'INTENSE_GREEN': '38;5;113',
    'INTENSE_PURPLE': '38;5;165',
    'INTENSE_RED': '38;5;202',
    'INTENSE_WHITE': '38;5;224',
    'INTENSE_YELLOW': '38;5;221',
    'NO_COLOR': '0',
    'PURPLE': '38;5;165',
    'RED': '38;5;124',
    'UNDERLINE_BLACK': '4;38;5;16',
    'UNDERLINE_BLUE': '4;38;5;18',
    'UNDERLINE_CYAN': '4;38;5;30',
    'UNDERLINE_GREEN': '4;38;5;40',
    'UNDERLINE_INTENSE_BLACK': '4;38;5;59',
    'UNDERLINE_INTENSE_BLUE': '4;38;5;105',
    'UNDERLINE_INTENSE_CYAN': '4;38;5;45',
    'UNDERLINE_INTENSE_GREEN': '4;38;5;113',
    'UNDERLINE_INTENSE_PURPLE': '4;38;5;165',
    'UNDERLINE_INTENSE_RED': '4;38;5;202',
    'UNDERLINE_INTENSE_WHITE': '4;38;5;224',
    'UNDERLINE_INTENSE_YELLOW': '4;38;5;221',
    'UNDERLINE_PURPLE': '4;38;5;165',
    'UNDERLINE_RED': '4;38;5;124',
    'UNDERLINE_WHITE': '4;38;5;145',
    'UNDERLINE_YELLOW': '4;38;5;166',
    'WHITE': '38;5;145',
    'YELLOW': '38;5;166',
}

MURPHY_STYLE = {
    'BACKGROUND_BLACK': '48;5;16',
    'BACKGROUND_BLUE': '48;5;18',
    'BACKGROUND_CYAN': '48;5;31',
    'BACKGROUND_GREEN': '48;5;34',
    'BACKGROUND_INTENSE_BLACK': '48;5;59',
    'BACKGROUND_INTENSE_BLUE': '48;5;63',
    'BACKGROUND_INTENSE_CYAN': '48;5;86',
    'BACKGROUND_INTENSE_GREEN': '48;5;86',
    'BACKGROUND_INTENSE_PURPLE': '48;5;213',
    'BACKGROUND_INTENSE_RED': '48;5;209',
    'BACKGROUND_INTENSE_WHITE': '48;5;15',
    'BACKGROUND_INTENSE_YELLOW': '48;5;222',
    'BACKGROUND_PURPLE': '48;5;90',
    'BACKGROUND_RED': '48;5;124',
    'BACKGROUND_WHITE': '48;5;145',
    'BACKGROUND_YELLOW': '48;5;166',
    'BLACK': '38;5;16',
    'BLUE': '38;5;18',
    'BOLD_BLACK': '1;38;5;16',
    'BOLD_BLUE': '1;38;5;18',
    'BOLD_CYAN': '1;38;5;31',
    'BOLD_GREEN': '1;38;5;34',
    'BOLD_INTENSE_BLACK': '1;38;5;59',
    'BOLD_INTENSE_BLUE': '1;38;5;63',
    'BOLD_INTENSE_CYAN': '1;38;5;86',
    'BOLD_INTENSE_GREEN': '1;38;5;86',
    'BOLD_INTENSE_PURPLE': '1;38;5;213',
    'BOLD_INTENSE_RED': '1;38;5;209',
    'BOLD_INTENSE_WHITE': '1;38;5;15',
    'BOLD_INTENSE_YELLOW': '1;38;5;222',
    'BOLD_PURPLE': '1;38;5;90',
    'BOLD_RED': '1;38;5;124',
    'BOLD_UNDERLINE_BLACK': '1;4;38;5;16',
    'BOLD_UNDERLINE_BLUE': '1;4;38;5;18',
    'BOLD_UNDERLINE_CYAN': '1;4;38;5;31',
    'BOLD_UNDERLINE_GREEN': '1;4;38;5;34',
    'BOLD_UNDERLINE_INTENSE_BLACK': '1;4;38;5;59',
    'BOLD_UNDERLINE_INTENSE_BLUE': '1;4;38;5;63',
    'BOLD_UNDERLINE_INTENSE_CYAN': '1;4;38;5;86',
    'BOLD_UNDERLINE_INTENSE_GREEN': '1;4;38;5;86',
    'BOLD_UNDERLINE_INTENSE_PURPLE': '1;4;38;5;213',
    'BOLD_UNDERLINE_INTENSE_RED': '1;4;38;5;209',
    'BOLD_UNDERLINE_INTENSE_WHITE': '1;4;38;5;15',
    'BOLD_UNDERLINE_INTENSE_YELLOW': '1;4;38;5;222',
    'BOLD_UNDERLINE_PURPLE': '1;4;38;5;90',
    'BOLD_UNDERLINE_RED': '1;4;38;5;124',
    'BOLD_UNDERLINE_WHITE': '1;4;38;5;145',
    'BOLD_UNDERLINE_YELLOW': '1;4;38;5;166',
    'BOLD_WHITE': '1;38;5;145',
    'BOLD_YELLOW': '1;38;5;166',
    'CYAN': '38;5;31',
    'GREEN': '38;5;34',
    'INTENSE_BLACK': '38;5;59',
    'INTENSE_BLUE': '38;5;63',
    'INTENSE_CYAN': '38;5;86',
    'INTENSE_GREEN': '38;5;86',
    'INTENSE_PURPLE': '38;5;213',
    'INTENSE_RED': '38;5;209',
    'INTENSE_WHITE': '38;5;15',
    'INTENSE_YELLOW': '38;5;222',
    'NO_COLOR': '0',
    'PURPLE': '38;5;90',
    'RED': '38;5;124',
    'UNDERLINE_BLACK': '4;38;5;16',
    'UNDERLINE_BLUE': '4;38;5;18',
    'UNDERLINE_CYAN': '4;38;5;31',
    'UNDERLINE_GREEN': '4;38;5;34',
    'UNDERLINE_INTENSE_BLACK': '4;38;5;59',
    'UNDERLINE_INTENSE_BLUE': '4;38;5;63',
    'UNDERLINE_INTENSE_CYAN': '4;38;5;86',
    'UNDERLINE_INTENSE_GREEN': '4;38;5;86',
    'UNDERLINE_INTENSE_PURPLE': '4;38;5;213',
    'UNDERLINE_INTENSE_RED': '4;38;5;209',
    'UNDERLINE_INTENSE_WHITE': '4;38;5;15',
    'UNDERLINE_INTENSE_YELLOW': '4;38;5;222',
    'UNDERLINE_PURPLE': '4;38;5;90',
    'UNDERLINE_RED': '4;38;5;124',
    'UNDERLINE_WHITE': '4;38;5;145',
    'UNDERLINE_YELLOW': '4;38;5;166',
    'WHITE': '38;5;145',
    'YELLOW': '38;5;166',
}

NATIVE_STYLE = {
    'BACKGROUND_BLACK': '48;5;52',
    'BACKGROUND_BLUE': '48;5;67',
    'BACKGROUND_CYAN': '48;5;31',
    'BACKGROUND_GREEN': '48;5;64',
    'BACKGROUND_INTENSE_BLACK': '48;5;59',
    'BACKGROUND_INTENSE_BLUE': '48;5;68',
    'BACKGROUND_INTENSE_CYAN': '48;5;87',
    'BACKGROUND_INTENSE_GREEN': '48;5;70',
    'BACKGROUND_INTENSE_PURPLE': '48;5;188',
    'BACKGROUND_INTENSE_RED': '48;5;160',
    'BACKGROUND_INTENSE_WHITE': '48;5;15',
    'BACKGROUND_INTENSE_YELLOW': '48;5;214',
    'BACKGROUND_PURPLE': '48;5;59',
    'BACKGROUND_RED': '48;5;124',
    'BACKGROUND_WHITE': '48;5;145',
    'BACKGROUND_YELLOW': '48;5;124',
    'BLACK': '38;5;52',
    'BLUE': '38;5;67',
    'BOLD_BLACK': '1;38;5;52',
    'BOLD_BLUE': '1;38;5;67',
    'BOLD_CYAN': '1;38;5;31',
    'BOLD_GREEN': '1;38;5;64',
    'BOLD_INTENSE_BLACK': '1;38;5;59',
    'BOLD_INTENSE_BLUE': '1;38;5;68',
    'BOLD_INTENSE_CYAN': '1;38;5;87',
    'BOLD_INTENSE_GREEN': '1;38;5;70',
    'BOLD_INTENSE_PURPLE': '1;38;5;188',
    'BOLD_INTENSE_RED': '1;38;5;160',
    'BOLD_INTENSE_WHITE': '1;38;5;15',
    'BOLD_INTENSE_YELLOW': '1;38;5;214',
    'BOLD_PURPLE': '1;38;5;59',
    'BOLD_RED': '1;38;5;124',
    'BOLD_UNDERLINE_BLACK': '1;4;38;5;52',
    'BOLD_UNDERLINE_BLUE': '1;4;38;5;67',
    'BOLD_UNDERLINE_CYAN': '1;4;38;5;31',
    'BOLD_UNDERLINE_GREEN': '1;4;38;5;64',
    'BOLD_UNDERLINE_INTENSE_BLACK': '1;4;38;5;59',
    'BOLD_UNDERLINE_INTENSE_BLUE': '1;4;38;5;68',
    'BOLD_UNDERLINE_INTENSE_CYAN': '1;4;38;5;87',
    'BOLD_UNDERLINE_INTENSE_GREEN': '1;4;38;5;70',
    'BOLD_UNDERLINE_INTENSE_PURPLE': '1;4;38;5;188',
    'BOLD_UNDERLINE_INTENSE_RED': '1;4;38;5;160',
    'BOLD_UNDERLINE_INTENSE_WHITE': '1;4;38;5;15',
    'BOLD_UNDERLINE_INTENSE_YELLOW': '1;4;38;5;214',
    'BOLD_UNDERLINE_PURPLE': '1;4;38;5;59',
    'BOLD_UNDERLINE_RED': '1;4;38;5;124',
    'BOLD_UNDERLINE_WHITE': '1;4;38;5;145',
    'BOLD_UNDERLINE_YELLOW': '1;4;38;5;124',
    'BOLD_WHITE': '1;38;5;145',
    'BOLD_YELLOW': '1;38;5;124',
    'CYAN': '38;5;31',
    'GREEN': '38;5;64',
    'INTENSE_BLACK': '38;5;59',
    'INTENSE_BLUE': '38;5;68',
    'INTENSE_CYAN': '38;5;87',
    'INTENSE_GREEN': '38;5;70',
    'INTENSE_PURPLE': '38;5;188',
    'INTENSE_RED': '38;5;160',
    'INTENSE_WHITE': '38;5;15',
    'INTENSE_YELLOW': '38;5;214',
    'NO_COLOR': '0',
    'PURPLE': '38;5;59',
    'RED': '38;5;124',
    'UNDERLINE_BLACK': '4;38;5;52',
    'UNDERLINE_BLUE': '4;38;5;67',
    'UNDERLINE_CYAN': '4;38;5;31',
    'UNDERLINE_GREEN': '4;38;5;64',
    'UNDERLINE_INTENSE_BLACK': '4;38;5;59',
    'UNDERLINE_INTENSE_BLUE': '4;38;5;68',
    'UNDERLINE_INTENSE_CYAN': '4;38;5;87',
    'UNDERLINE_INTENSE_GREEN': '4;38;5;70',
    'UNDERLINE_INTENSE_PURPLE': '4;38;5;188',
    'UNDERLINE_INTENSE_RED': '4;38;5;160',
    'UNDERLINE_INTENSE_WHITE': '4;38;5;15',
    'UNDERLINE_INTENSE_YELLOW': '4;38;5;214',
    'UNDERLINE_PURPLE': '4;38;5;59',
    'UNDERLINE_RED': '4;38;5;124',
    'UNDERLINE_WHITE': '4;38;5;145',
    'UNDERLINE_YELLOW': '4;38;5;124',
    'WHITE': '38;5;145',
    'YELLOW': '38;5;124',
}

PARAISO_DARK_STYLE = {
    'BACKGROUND_BLACK': '48;5;95',
    'BACKGROUND_BLUE': '48;5;97',
    'BACKGROUND_CYAN': '48;5;39',
    'BACKGROUND_GREEN': '48;5;72',
    'BACKGROUND_INTENSE_BLACK': '48;5;95',
    'BACKGROUND_INTENSE_BLUE': '48;5;97',
    'BACKGROUND_INTENSE_CYAN': '48;5;79',
    'BACKGROUND_INTENSE_GREEN': '48;5;72',
    'BACKGROUND_INTENSE_PURPLE': '48;5;188',
    'BACKGROUND_INTENSE_RED': '48;5;203',
    'BACKGROUND_INTENSE_WHITE': '48;5;188',
    'BACKGROUND_INTENSE_YELLOW': '48;5;220',
    'BACKGROUND_PURPLE': '48;5;97',
    'BACKGROUND_RED': '48;5;203',
    'BACKGROUND_WHITE': '48;5;79',
    'BACKGROUND_YELLOW': '48;5;214',
    'BLACK': '38;5;95',
    'BLUE': '38;5;97',
    'BOLD_BLACK': '1;38;5;95',
    'BOLD_BLUE': '1;38;5;97',
    'BOLD_CYAN': '1;38;5;39',
    'BOLD_GREEN': '1;38;5;72',
    'BOLD_INTENSE_BLACK': '1;38;5;95',
    'BOLD_INTENSE_BLUE': '1;38;5;97',
    'BOLD_INTENSE_CYAN': '1;38;5;79',
    'BOLD_INTENSE_GREEN': '1;38;5;72',
    'BOLD_INTENSE_PURPLE': '1;38;5;188',
    'BOLD_INTENSE_RED': '1;38;5;203',
    'BOLD_INTENSE_WHITE': '1;38;5;188',
    'BOLD_INTENSE_YELLOW': '1;38;5;220',
    'BOLD_PURPLE': '1;38;5;97',
    'BOLD_RED': '1;38;5;203',
    'BOLD_UNDERLINE_BLACK': '1;4;38;5;95',
    'BOLD_UNDERLINE_BLUE': '1;4;38;5;97',
    'BOLD_UNDERLINE_CYAN': '1;4;38;5;39',
    'BOLD_UNDERLINE_GREEN': '1;4;38;5;72',
    'BOLD_UNDERLINE_INTENSE_BLACK': '1;4;38;5;95',
    'BOLD_UNDERLINE_INTENSE_BLUE': '1;4;38;5;97',
    'BOLD_UNDERLINE_INTENSE_CYAN': '1;4;38;5;79',
    'BOLD_UNDERLINE_INTENSE_GREEN': '1;4;38;5;72',
    'BOLD_UNDERLINE_INTENSE_PURPLE': '1;4;38;5;188',
    'BOLD_UNDERLINE_INTENSE_RED': '1;4;38;5;203',
    'BOLD_UNDERLINE_INTENSE_WHITE': '1;4;38;5;188',
    'BOLD_UNDERLINE_INTENSE_YELLOW': '1;4;38;5;220',
    'BOLD_UNDERLINE_PURPLE': '1;4;38;5;97',
    'BOLD_UNDERLINE_RED': '1;4;38;5;203',
    'BOLD_UNDERLINE_WHITE': '1;4;38;5;79',
    'BOLD_UNDERLINE_YELLOW': '1;4;38;5;214',
    'BOLD_WHITE': '1;38;5;79',
    'BOLD_YELLOW': '1;38;5;214',
    'CYAN': '38;5;39',
    'GREEN': '38;5;72',
    'INTENSE_BLACK': '38;5;95',
    'INTENSE_BLUE': '38;5;97',
    'INTENSE_CYAN': '38;5;79',
    'INTENSE_GREEN': '38;5;72',
    'INTENSE_PURPLE': '38;5;188',
    'INTENSE_RED': '38;5;203',
    'INTENSE_WHITE': '38;5;188',
    'INTENSE_YELLOW': '38;5;220',
    'NO_COLOR': '0',
    'PURPLE': '38;5;97',
    'RED': '38;5;203',
    'UNDERLINE_BLACK': '4;38;5;95',
    'UNDERLINE_BLUE': '4;38;5;97',
    'UNDERLINE_CYAN': '4;38;5;39',
    'UNDERLINE_GREEN': '4;38;5;72',
    'UNDERLINE_INTENSE_BLACK': '4;38;5;95',
    'UNDERLINE_INTENSE_BLUE': '4;38;5;97',
    'UNDERLINE_INTENSE_CYAN': '4;38;5;79',
    'UNDERLINE_INTENSE_GREEN': '4;38;5;72',
    'UNDERLINE_INTENSE_PURPLE': '4;38;5;188',
    'UNDERLINE_INTENSE_RED': '4;38;5;203',
    'UNDERLINE_INTENSE_WHITE': '4;38;5;188',
    'UNDERLINE_INTENSE_YELLOW': '4;38;5;220',
    'UNDERLINE_PURPLE': '4;38;5;97',
    'UNDERLINE_RED': '4;38;5;203',
    'UNDERLINE_WHITE': '4;38;5;79',
    'UNDERLINE_YELLOW': '4;38;5;214',
    'WHITE': '38;5;79',
    'YELLOW': '38;5;214',
}

PARAISO_LIGHT_STYLE = {
    'BACKGROUND_BLACK': '48;5;16',
    'BACKGROUND_BLUE': '48;5;16',
    'BACKGROUND_CYAN': '48;5;39',
    'BACKGROUND_GREEN': '48;5;72',
    'BACKGROUND_INTENSE_BLACK': '48;5;16',
    'BACKGROUND_INTENSE_BLUE': '48;5;97',
    'BACKGROUND_INTENSE_CYAN': '48;5;79',
    'BACKGROUND_INTENSE_GREEN': '48;5;72',
    'BACKGROUND_INTENSE_PURPLE': '48;5;97',
    'BACKGROUND_INTENSE_RED': '48;5;203',
    'BACKGROUND_INTENSE_WHITE': '48;5;79',
    'BACKGROUND_INTENSE_YELLOW': '48;5;220',
    'BACKGROUND_PURPLE': '48;5;97',
    'BACKGROUND_RED': '48;5;16',
    'BACKGROUND_WHITE': '48;5;102',
    'BACKGROUND_YELLOW': '48;5;214',
    'BLACK': '38;5;16',
    'BLUE': '38;5;16',
    'BOLD_BLACK': '1;38;5;16',
    'BOLD_BLUE': '1;38;5;16',
    'BOLD_CYAN': '1;38;5;39',
    'BOLD_GREEN': '1;38;5;72',
    'BOLD_INTENSE_BLACK': '1;38;5;16',
    'BOLD_INTENSE_BLUE': '1;38;5;97',
    'BOLD_INTENSE_CYAN': '1;38;5;79',
    'BOLD_INTENSE_GREEN': '1;38;5;72',
    'BOLD_INTENSE_PURPLE': '1;38;5;97',
    'BOLD_INTENSE_RED': '1;38;5;203',
    'BOLD_INTENSE_WHITE': '1;38;5;79',
    'BOLD_INTENSE_YELLOW': '1;38;5;220',
    'BOLD_PURPLE': '1;38;5;97',
    'BOLD_RED': '1;38;5;16',
    'BOLD_UNDERLINE_BLACK': '1;4;38;5;16',
    'BOLD_UNDERLINE_BLUE': '1;4;38;5;16',
    'BOLD_UNDERLINE_CYAN': '1;4;38;5;39',
    'BOLD_UNDERLINE_GREEN': '1;4;38;5;72',
    'BOLD_UNDERLINE_INTENSE_BLACK': '1;4;38;5;16',
    'BOLD_UNDERLINE_INTENSE_BLUE': '1;4;38;5;97',
    'BOLD_UNDERLINE_INTENSE_CYAN': '1;4;38;5;79',
    'BOLD_UNDERLINE_INTENSE_GREEN': '1;4;38;5;72',
    'BOLD_UNDERLINE_INTENSE_PURPLE': '1;4;38;5;97',
    'BOLD_UNDERLINE_INTENSE_RED': '1;4;38;5;203',
    'BOLD_UNDERLINE_INTENSE_WHITE': '1;4;38;5;79',
    'BOLD_UNDERLINE_INTENSE_YELLOW': '1;4;38;5;220',
    'BOLD_UNDERLINE_PURPLE': '1;4;38;5;97',
    'BOLD_UNDERLINE_RED': '1;4;38;5;16',
    'BOLD_UNDERLINE_WHITE': '1;4;38;5;102',
    'BOLD_UNDERLINE_YELLOW': '1;4;38;5;214',
    'BOLD_WHITE': '1;38;5;102',
    'BOLD_YELLOW': '1;38;5;214',
    'CYAN': '38;5;39',
    'GREEN': '38;5;72',
    'INTENSE_BLACK': '38;5;16',
    'INTENSE_BLUE': '38;5;97',
    'INTENSE_CYAN': '38;5;79',
    'INTENSE_GREEN': '38;5;72',
    'INTENSE_PURPLE': '38;5;97',
    'INTENSE_RED': '38;5;203',
    'INTENSE_WHITE': '38;5;79',
    'INTENSE_YELLOW': '38;5;220',
    'NO_COLOR': '0',
    'PURPLE': '38;5;97',
    'RED': '38;5;16',
    'UNDERLINE_BLACK': '4;38;5;16',
    'UNDERLINE_BLUE': '4;38;5;16',
    'UNDERLINE_CYAN': '4;38;5;39',
    'UNDERLINE_GREEN': '4;38;5;72',
    'UNDERLINE_INTENSE_BLACK': '4;38;5;16',
    'UNDERLINE_INTENSE_BLUE': '4;38;5;97',
    'UNDERLINE_INTENSE_CYAN': '4;38;5;79',
    'UNDERLINE_INTENSE_GREEN': '4;38;5;72',
    'UNDERLINE_INTENSE_PURPLE': '4;38;5;97',
    'UNDERLINE_INTENSE_RED': '4;38;5;203',
    'UNDERLINE_INTENSE_WHITE': '4;38;5;79',
    'UNDERLINE_INTENSE_YELLOW': '4;38;5;220',
    'UNDERLINE_PURPLE': '4;38;5;97',
    'UNDERLINE_RED': '4;38;5;16',
    'UNDERLINE_WHITE': '4;38;5;102',
    'UNDERLINE_YELLOW': '4;38;5;214',
    'WHITE': '38;5;102',
    'YELLOW': '38;5;214',
}

PASTIE_STYLE = {
    'BACKGROUND_BLACK': '48;5;16',
    'BACKGROUND_BLUE': '48;5;20',
    'BACKGROUND_CYAN': '48;5;25',
    'BACKGROUND_GREEN': '48;5;28',
    'BACKGROUND_INTENSE_BLACK': '48;5;59',
    'BACKGROUND_INTENSE_BLUE': '48;5;61',
    'BACKGROUND_INTENSE_CYAN': '48;5;194',
    'BACKGROUND_INTENSE_GREEN': '48;5;34',
    'BACKGROUND_INTENSE_PURPLE': '48;5;188',
    'BACKGROUND_INTENSE_RED': '48;5;172',
    'BACKGROUND_INTENSE_WHITE': '48;5;15',
    'BACKGROUND_INTENSE_YELLOW': '48;5;188',
    'BACKGROUND_PURPLE': '48;5;125',
    'BACKGROUND_RED': '48;5;124',
    'BACKGROUND_WHITE': '48;5;145',
    'BACKGROUND_YELLOW': '48;5;130',
    'BLACK': '38;5;16',
    'BLUE': '38;5;20',
    'BOLD_BLACK': '1;38;5;16',
    'BOLD_BLUE': '1;38;5;20',
    'BOLD_CYAN': '1;38;5;25',
    'BOLD_GREEN': '1;38;5;28',
    'BOLD_INTENSE_BLACK': '1;38;5;59',
    'BOLD_INTENSE_BLUE': '1;38;5;61',
    'BOLD_INTENSE_CYAN': '1;38;5;194',
    'BOLD_INTENSE_GREEN': '1;38;5;34',
    'BOLD_INTENSE_PURPLE': '1;38;5;188',
    'BOLD_INTENSE_RED': '1;38;5;172',
    'BOLD_INTENSE_WHITE': '1;38;5;15',
    'BOLD_INTENSE_YELLOW': '1;38;5;188',
    'BOLD_PURPLE': '1;38;5;125',
    'BOLD_RED': '1;38;5;124',
    'BOLD_UNDERLINE_BLACK': '1;4;38;5;16',
    'BOLD_UNDERLINE_BLUE': '1;4;38;5;20',
    'BOLD_UNDERLINE_CYAN': '1;4;38;5;25',
    'BOLD_UNDERLINE_GREEN': '1;4;38;5;28',
    'BOLD_UNDERLINE_INTENSE_BLACK': '1;4;38;5;59',
    'BOLD_UNDERLINE_INTENSE_BLUE': '1;4;38;5;61',
    'BOLD_UNDERLINE_INTENSE_CYAN': '1;4;38;5;194',
    'BOLD_UNDERLINE_INTENSE_GREEN': '1;4;38;5;34',
    'BOLD_UNDERLINE_INTENSE_PURPLE': '1;4;38;5;188',
    'BOLD_UNDERLINE_INTENSE_RED': '1;4;38;5;172',
    'BOLD_UNDERLINE_INTENSE_WHITE': '1;4;38;5;15',
    'BOLD_UNDERLINE_INTENSE_YELLOW': '1;4;38;5;188',
    'BOLD_UNDERLINE_PURPLE': '1;4;38;5;125',
    'BOLD_UNDERLINE_RED': '1;4;38;5;124',
    'BOLD_UNDERLINE_WHITE': '1;4;38;5;145',
    'BOLD_UNDERLINE_YELLOW': '1;4;38;5;130',
    'BOLD_WHITE': '1;38;5;145',
    'BOLD_YELLOW': '1;38;5;130',
    'CYAN': '38;5;25',
    'GREEN': '38;5;28',
    'INTENSE_BLACK': '38;5;59',
    'INTENSE_BLUE': '38;5;61',
    'INTENSE_CYAN': '38;5;194',
    'INTENSE_GREEN': '38;5;34',
    'INTENSE_PURPLE': '38;5;188',
    'INTENSE_RED': '38;5;172',
    'INTENSE_WHITE': '38;5;15',
    'INTENSE_YELLOW': '38;5;188',
    'NO_COLOR': '0',
    'PURPLE': '38;5;125',
    'RED': '38;5;124',
    'UNDERLINE_BLACK': '4;38;5;16',
    'UNDERLINE_BLUE': '4;38;5;20',
    'UNDERLINE_CYAN': '4;38;5;25',
    'UNDERLINE_GREEN': '4;38;5;28',
    'UNDERLINE_INTENSE_BLACK': '4;38;5;59',
    'UNDERLINE_INTENSE_BLUE': '4;38;5;61',
    'UNDERLINE_INTENSE_CYAN': '4;38;5;194',
    'UNDERLINE_INTENSE_GREEN': '4;38;5;34',
    'UNDERLINE_INTENSE_PURPLE': '4;38;5;188',
    'UNDERLINE_INTENSE_RED': '4;38;5;172',
    'UNDERLINE_INTENSE_WHITE': '4;38;5;15',
    'UNDERLINE_INTENSE_YELLOW': '4;38;5;188',
    'UNDERLINE_PURPLE': '4;38;5;125',
    'UNDERLINE_RED': '4;38;5;124',
    'UNDERLINE_WHITE': '4;38;5;145',
    'UNDERLINE_YELLOW': '4;38;5;130',
    'WHITE': '38;5;145',
    'YELLOW': '38;5;130',
}

PERLDOC_STYLE = {
    'BACKGROUND_BLACK': '48;5;18',
    'BACKGROUND_BLUE': '48;5;18',
    'BACKGROUND_CYAN': '48;5;31',
    'BACKGROUND_GREEN': '48;5;34',
    'BACKGROUND_INTENSE_BLACK': '48;5;59',
    'BACKGROUND_INTENSE_BLUE': '48;5;134',
    'BACKGROUND_INTENSE_CYAN': '48;5;145',
    'BACKGROUND_INTENSE_GREEN': '48;5;28',
    'BACKGROUND_INTENSE_PURPLE': '48;5;134',
    'BACKGROUND_INTENSE_RED': '48;5;167',
    'BACKGROUND_INTENSE_WHITE': '48;5;188',
    'BACKGROUND_INTENSE_YELLOW': '48;5;188',
    'BACKGROUND_PURPLE': '48;5;90',
    'BACKGROUND_RED': '48;5;124',
    'BACKGROUND_WHITE': '48;5;145',
    'BACKGROUND_YELLOW': '48;5;166',
    'BLACK': '38;5;18',
    'BLUE': '38;5;18',
    'BOLD_BLACK': '1;38;5;18',
    'BOLD_BLUE': '1;38;5;18',
    'BOLD_CYAN': '1;38;5;31',
    'BOLD_GREEN': '1;38;5;34',
    'BOLD_INTENSE_BLACK': '1;38;5;59',
    'BOLD_INTENSE_BLUE': '1;38;5;134',
    'BOLD_INTENSE_CYAN': '1;38;5;145',
    'BOLD_INTENSE_GREEN': '1;38;5;28',
    'BOLD_INTENSE_PURPLE': '1;38;5;134',
    'BOLD_INTENSE_RED': '1;38;5;167',
    'BOLD_INTENSE_WHITE': '1;38;5;188',
    'BOLD_INTENSE_YELLOW': '1;38;5;188',
    'BOLD_PURPLE': '1;38;5;90',
    'BOLD_RED': '1;38;5;124',
    'BOLD_UNDERLINE_BLACK': '1;4;38;5;18',
    'BOLD_UNDERLINE_BLUE': '1;4;38;5;18',
    'BOLD_UNDERLINE_CYAN': '1;4;38;5;31',
    'BOLD_UNDERLINE_GREEN': '1;4;38;5;34',
    'BOLD_UNDERLINE_INTENSE_BLACK': '1;4;38;5;59',
    'BOLD_UNDERLINE_INTENSE_BLUE': '1;4;38;5;134',
    'BOLD_UNDERLINE_INTENSE_CYAN': '1;4;38;5;145',
    'BOLD_UNDERLINE_INTENSE_GREEN': '1;4;38;5;28',
    'BOLD_UNDERLINE_INTENSE_PURPLE': '1;4;38;5;134',
    'BOLD_UNDERLINE_INTENSE_RED': '1;4;38;5;167',
    'BOLD_UNDERLINE_INTENSE_WHITE': '1;4;38;5;188',
    'BOLD_UNDERLINE_INTENSE_YELLOW': '1;4;38;5;188',
    'BOLD_UNDERLINE_PURPLE': '1;4;38;5;90',
    'BOLD_UNDERLINE_RED': '1;4;38;5;124',
    'BOLD_UNDERLINE_WHITE': '1;4;38;5;145',
    'BOLD_UNDERLINE_YELLOW': '1;4;38;5;166',
    'BOLD_WHITE': '1;38;5;145',
    'BOLD_YELLOW': '1;38;5;166',
    'CYAN': '38;5;31',
    'GREEN': '38;5;34',
    'INTENSE_BLACK': '38;5;59',
    'INTENSE_BLUE': '38;5;134',
    'INTENSE_CYAN': '38;5;145',
    'INTENSE_GREEN': '38;5;28',
    'INTENSE_PURPLE': '38;5;134',
    'INTENSE_RED': '38;5;167',
    'INTENSE_WHITE': '38;5;188',
    'INTENSE_YELLOW': '38;5;188',
    'NO_COLOR': '0',
    'PURPLE': '38;5;90',
    'RED': '38;5;124',
    'UNDERLINE_BLACK': '4;38;5;18',
    'UNDERLINE_BLUE': '4;38;5;18',
    'UNDERLINE_CYAN': '4;38;5;31',
    'UNDERLINE_GREEN': '4;38;5;34',
    'UNDERLINE_INTENSE_BLACK': '4;38;5;59',
    'UNDERLINE_INTENSE_BLUE': '4;38;5;134',
    'UNDERLINE_INTENSE_CYAN': '4;38;5;145',
    'UNDERLINE_INTENSE_GREEN': '4;38;5;28',
    'UNDERLINE_INTENSE_PURPLE': '4;38;5;134',
    'UNDERLINE_INTENSE_RED': '4;38;5;167',
    'UNDERLINE_INTENSE_WHITE': '4;38;5;188',
    'UNDERLINE_INTENSE_YELLOW': '4;38;5;188',
    'UNDERLINE_PURPLE': '4;38;5;90',
    'UNDERLINE_RED': '4;38;5;124',
    'UNDERLINE_WHITE': '4;38;5;145',
    'UNDERLINE_YELLOW': '4;38;5;166',
    'WHITE': '38;5;145',
    'YELLOW': '38;5;166',
}

RRT_STYLE = {
    'BACKGROUND_BLACK': '48;5;09',
    'BACKGROUND_BLUE': '48;5;117',
    'BACKGROUND_CYAN': '48;5;117',
    'BACKGROUND_GREEN': '48;5;46',
    'BACKGROUND_INTENSE_BLACK': '48;5;117',
    'BACKGROUND_INTENSE_BLUE': '48;5;117',
    'BACKGROUND_INTENSE_CYAN': '48;5;122',
    'BACKGROUND_INTENSE_GREEN': '48;5;46',
    'BACKGROUND_INTENSE_PURPLE': '48;5;213',
    'BACKGROUND_INTENSE_RED': '48;5;09',
    'BACKGROUND_INTENSE_WHITE': '48;5;188',
    'BACKGROUND_INTENSE_YELLOW': '48;5;222',
    'BACKGROUND_PURPLE': '48;5;213',
    'BACKGROUND_RED': '48;5;09',
    'BACKGROUND_WHITE': '48;5;117',
    'BACKGROUND_YELLOW': '48;5;09',
    'BLACK': '38;5;09',
    'BLUE': '38;5;117',
    'BOLD_BLACK': '1;38;5;09',
    'BOLD_BLUE': '1;38;5;117',
    'BOLD_CYAN': '1;38;5;117',
    'BOLD_GREEN': '1;38;5;46',
    'BOLD_INTENSE_BLACK': '1;38;5;117',
    'BOLD_INTENSE_BLUE': '1;38;5;117',
    'BOLD_INTENSE_CYAN': '1;38;5;122',
    'BOLD_INTENSE_GREEN': '1;38;5;46',
    'BOLD_INTENSE_PURPLE': '1;38;5;213',
    'BOLD_INTENSE_RED': '1;38;5;09',
    'BOLD_INTENSE_WHITE': '1;38;5;188',
    'BOLD_INTENSE_YELLOW': '1;38;5;222',
    'BOLD_PURPLE': '1;38;5;213',
    'BOLD_RED': '1;38;5;09',
    'BOLD_UNDERLINE_BLACK': '1;4;38;5;09',
    'BOLD_UNDERLINE_BLUE': '1;4;38;5;117',
    'BOLD_UNDERLINE_CYAN': '1;4;38;5;117',
    'BOLD_UNDERLINE_GREEN': '1;4;38;5;46',
    'BOLD_UNDERLINE_INTENSE_BLACK': '1;4;38;5;117',
    'BOLD_UNDERLINE_INTENSE_BLUE': '1;4;38;5;117',
    'BOLD_UNDERLINE_INTENSE_CYAN': '1;4;38;5;122',
    'BOLD_UNDERLINE_INTENSE_GREEN': '1;4;38;5;46',
    'BOLD_UNDERLINE_INTENSE_PURPLE': '1;4;38;5;213',
    'BOLD_UNDERLINE_INTENSE_RED': '1;4;38;5;09',
    'BOLD_UNDERLINE_INTENSE_WHITE': '1;4;38;5;188',
    'BOLD_UNDERLINE_INTENSE_YELLOW': '1;4;38;5;222',
    'BOLD_UNDERLINE_PURPLE': '1;4;38;5;213',
    'BOLD_UNDERLINE_RED': '1;4;38;5;09',
    'BOLD_UNDERLINE_WHITE': '1;4;38;5;117',
    'BOLD_UNDERLINE_YELLOW': '1;4;38;5;09',
    'BOLD_WHITE': '1;38;5;117',
    'BOLD_YELLOW': '1;38;5;09',
    'CYAN': '38;5;117',
    'GREEN': '38;5;46',
    'INTENSE_BLACK': '38;5;117',
    'INTENSE_BLUE': '38;5;117',
    'INTENSE_CYAN': '38;5;122',
    'INTENSE_GREEN': '38;5;46',
    'INTENSE_PURPLE': '38;5;213',
    'INTENSE_RED': '38;5;09',
    'INTENSE_WHITE': '38;5;188',
    'INTENSE_YELLOW': '38;5;222',
    'NO_COLOR': '0',
    'PURPLE': '38;5;213',
    'RED': '38;5;09',
    'UNDERLINE_BLACK': '4;38;5;09',
    'UNDERLINE_BLUE': '4;38;5;117',
    'UNDERLINE_CYAN': '4;38;5;117',
    'UNDERLINE_GREEN': '4;38;5;46',
    'UNDERLINE_INTENSE_BLACK': '4;38;5;117',
    'UNDERLINE_INTENSE_BLUE': '4;38;5;117',
    'UNDERLINE_INTENSE_CYAN': '4;38;5;122',
    'UNDERLINE_INTENSE_GREEN': '4;38;5;46',
    'UNDERLINE_INTENSE_PURPLE': '4;38;5;213',
    'UNDERLINE_INTENSE_RED': '4;38;5;09',
    'UNDERLINE_INTENSE_WHITE': '4;38;5;188',
    'UNDERLINE_INTENSE_YELLOW': '4;38;5;222',
    'UNDERLINE_PURPLE': '4;38;5;213',
    'UNDERLINE_RED': '4;38;5;09',
    'UNDERLINE_WHITE': '4;38;5;117',
    'UNDERLINE_YELLOW': '4;38;5;09',
    'WHITE': '38;5;117',
    'YELLOW': '38;5;09',
}

TANGO_STYLE = {
    'BACKGROUND_BLACK': '48;5;16',
    'BACKGROUND_BLUE': '48;5;20',
    'BACKGROUND_CYAN': '48;5;61',
    'BACKGROUND_GREEN': '48;5;34',
    'BACKGROUND_INTENSE_BLACK': '48;5;24',
    'BACKGROUND_INTENSE_BLUE': '48;5;62',
    'BACKGROUND_INTENSE_CYAN': '48;5;15',
    'BACKGROUND_INTENSE_GREEN': '48;5;64',
    'BACKGROUND_INTENSE_PURPLE': '48;5;15',
    'BACKGROUND_INTENSE_RED': '48;5;09',
    'BACKGROUND_INTENSE_WHITE': '48;5;15',
    'BACKGROUND_INTENSE_YELLOW': '48;5;178',
    'BACKGROUND_PURPLE': '48;5;90',
    'BACKGROUND_RED': '48;5;124',
    'BACKGROUND_WHITE': '48;5;15',
    'BACKGROUND_YELLOW': '48;5;94',
    'BLACK': '38;5;16',
    'BLUE': '38;5;20',
    'BOLD_BLACK': '1;38;5;16',
    'BOLD_BLUE': '1;38;5;20',
    'BOLD_CYAN': '1;38;5;61',
    'BOLD_GREEN': '1;38;5;34',
    'BOLD_INTENSE_BLACK': '1;38;5;24',
    'BOLD_INTENSE_BLUE': '1;38;5;62',
    'BOLD_INTENSE_CYAN': '1;38;5;15',
    'BOLD_INTENSE_GREEN': '1;38;5;64',
    'BOLD_INTENSE_PURPLE': '1;38;5;15',
    'BOLD_INTENSE_RED': '1;38;5;09',
    'BOLD_INTENSE_WHITE': '1;38;5;15',
    'BOLD_INTENSE_YELLOW': '1;38;5;178',
    'BOLD_PURPLE': '1;38;5;90',
    'BOLD_RED': '1;38;5;124',
    'BOLD_UNDERLINE_BLACK': '1;4;38;5;16',
    'BOLD_UNDERLINE_BLUE': '1;4;38;5;20',
    'BOLD_UNDERLINE_CYAN': '1;4;38;5;61',
    'BOLD_UNDERLINE_GREEN': '1;4;38;5;34',
    'BOLD_UNDERLINE_INTENSE_BLACK': '1;4;38;5;24',
    'BOLD_UNDERLINE_INTENSE_BLUE': '1;4;38;5;62',
    'BOLD_UNDERLINE_INTENSE_CYAN': '1;4;38;5;15',
    'BOLD_UNDERLINE_INTENSE_GREEN': '1;4;38;5;64',
    'BOLD_UNDERLINE_INTENSE_PURPLE': '1;4;38;5;15',
    'BOLD_UNDERLINE_INTENSE_RED': '1;4;38;5;09',
    'BOLD_UNDERLINE_INTENSE_WHITE': '1;4;38;5;15',
    'BOLD_UNDERLINE_INTENSE_YELLOW': '1;4;38;5;178',
    'BOLD_UNDERLINE_PURPLE': '1;4;38;5;90',
    'BOLD_UNDERLINE_RED': '1;4;38;5;124',
    'BOLD_UNDERLINE_WHITE': '1;4;38;5;15',
    'BOLD_UNDERLINE_YELLOW': '1;4;38;5;94',
    'BOLD_WHITE': '1;38;5;15',
    'BOLD_YELLOW': '1;38;5;94',
    'CYAN': '38;5;61',
    'GREEN': '38;5;34',
    'INTENSE_BLACK': '38;5;24',
    'INTENSE_BLUE': '38;5;62',
    'INTENSE_CYAN': '38;5;15',
    'INTENSE_GREEN': '38;5;64',
    'INTENSE_PURPLE': '38;5;15',
    'INTENSE_RED': '38;5;09',
    'INTENSE_WHITE': '38;5;15',
    'INTENSE_YELLOW': '38;5;178',
    'NO_COLOR': '0',
    'PURPLE': '38;5;90',
    'RED': '38;5;124',
    'UNDERLINE_BLACK': '4;38;5;16',
    'UNDERLINE_BLUE': '4;38;5;20',
    'UNDERLINE_CYAN': '4;38;5;61',
    'UNDERLINE_GREEN': '4;38;5;34',
    'UNDERLINE_INTENSE_BLACK': '4;38;5;24',
    'UNDERLINE_INTENSE_BLUE': '4;38;5;62',
    'UNDERLINE_INTENSE_CYAN': '4;38;5;15',
    'UNDERLINE_INTENSE_GREEN': '4;38;5;64',
    'UNDERLINE_INTENSE_PURPLE': '4;38;5;15',
    'UNDERLINE_INTENSE_RED': '4;38;5;09',
    'UNDERLINE_INTENSE_WHITE': '4;38;5;15',
    'UNDERLINE_INTENSE_YELLOW': '4;38;5;178',
    'UNDERLINE_PURPLE': '4;38;5;90',
    'UNDERLINE_RED': '4;38;5;124',
    'UNDERLINE_WHITE': '4;38;5;15',
    'UNDERLINE_YELLOW': '4;38;5;94',
    'WHITE': '38;5;15',
    'YELLOW': '38;5;94',
}

TRAC_STYLE = {
    'BACKGROUND_BLACK': '48;5;16',
    'BACKGROUND_BLUE': '48;5;18',
    'BACKGROUND_CYAN': '48;5;30',
    'BACKGROUND_GREEN': '48;5;100',
    'BACKGROUND_INTENSE_BLACK': '48;5;59',
    'BACKGROUND_INTENSE_BLUE': '48;5;60',
    'BACKGROUND_INTENSE_CYAN': '48;5;194',
    'BACKGROUND_INTENSE_GREEN': '48;5;102',
    'BACKGROUND_INTENSE_PURPLE': '48;5;188',
    'BACKGROUND_INTENSE_RED': '48;5;137',
    'BACKGROUND_INTENSE_WHITE': '48;5;224',
    'BACKGROUND_INTENSE_YELLOW': '48;5;188',
    'BACKGROUND_PURPLE': '48;5;90',
    'BACKGROUND_RED': '48;5;124',
    'BACKGROUND_WHITE': '48;5;145',
    'BACKGROUND_YELLOW': '48;5;100',
    'BLACK': '38;5;16',
    'BLUE': '38;5;18',
    'BOLD_BLACK': '1;38;5;16',
    'BOLD_BLUE': '1;38;5;18',
    'BOLD_CYAN': '1;38;5;30',
    'BOLD_GREEN': '1;38;5;100',
    'BOLD_INTENSE_BLACK': '1;38;5;59',
    'BOLD_INTENSE_BLUE': '1;38;5;60',
    'BOLD_INTENSE_CYAN': '1;38;5;194',
    'BOLD_INTENSE_GREEN': '1;38;5;102',
    'BOLD_INTENSE_PURPLE': '1;38;5;188',
    'BOLD_INTENSE_RED': '1;38;5;137',
    'BOLD_INTENSE_WHITE': '1;38;5;224',
    'BOLD_INTENSE_YELLOW': '1;38;5;188',
    'BOLD_PURPLE': '1;38;5;90',
    'BOLD_RED': '1;38;5;124',
    'BOLD_UNDERLINE_BLACK': '1;4;38;5;16',
    'BOLD_UNDERLINE_BLUE': '1;4;38;5;18',
    'BOLD_UNDERLINE_CYAN': '1;4;38;5;30',
    'BOLD_UNDERLINE_GREEN': '1;4;38;5;100',
    'BOLD_UNDERLINE_INTENSE_BLACK': '1;4;38;5;59',
    'BOLD_UNDERLINE_INTENSE_BLUE': '1;4;38;5;60',
    'BOLD_UNDERLINE_INTENSE_CYAN': '1;4;38;5;194',
    'BOLD_UNDERLINE_INTENSE_GREEN': '1;4;38;5;102',
    'BOLD_UNDERLINE_INTENSE_PURPLE': '1;4;38;5;188',
    'BOLD_UNDERLINE_INTENSE_RED': '1;4;38;5;137',
    'BOLD_UNDERLINE_INTENSE_WHITE': '1;4;38;5;224',
    'BOLD_UNDERLINE_INTENSE_YELLOW': '1;4;38;5;188',
    'BOLD_UNDERLINE_PURPLE': '1;4;38;5;90',
    'BOLD_UNDERLINE_RED': '1;4;38;5;124',
    'BOLD_UNDERLINE_WHITE': '1;4;38;5;145',
    'BOLD_UNDERLINE_YELLOW': '1;4;38;5;100',
    'BOLD_WHITE': '1;38;5;145',
    'BOLD_YELLOW': '1;38;5;100',
    'CYAN': '38;5;30',
    'GREEN': '38;5;100',
    'INTENSE_BLACK': '38;5;59',
    'INTENSE_BLUE': '38;5;60',
    'INTENSE_CYAN': '38;5;194',
    'INTENSE_GREEN': '38;5;102',
    'INTENSE_PURPLE': '38;5;188',
    'INTENSE_RED': '38;5;137',
    'INTENSE_WHITE': '38;5;224',
    'INTENSE_YELLOW': '38;5;188',
    'NO_COLOR': '0',
    'PURPLE': '38;5;90',
    'RED': '38;5;124',
    'UNDERLINE_BLACK': '4;38;5;16',
    'UNDERLINE_BLUE': '4;38;5;18',
    'UNDERLINE_CYAN': '4;38;5;30',
    'UNDERLINE_GREEN': '4;38;5;100',
    'UNDERLINE_INTENSE_BLACK': '4;38;5;59',
    'UNDERLINE_INTENSE_BLUE': '4;38;5;60',
    'UNDERLINE_INTENSE_CYAN': '4;38;5;194',
    'UNDERLINE_INTENSE_GREEN': '4;38;5;102',
    'UNDERLINE_INTENSE_PURPLE': '4;38;5;188',
    'UNDERLINE_INTENSE_RED': '4;38;5;137',
    'UNDERLINE_INTENSE_WHITE': '4;38;5;224',
    'UNDERLINE_INTENSE_YELLOW': '4;38;5;188',
    'UNDERLINE_PURPLE': '4;38;5;90',
    'UNDERLINE_RED': '4;38;5;124',
    'UNDERLINE_WHITE': '4;38;5;145',
    'UNDERLINE_YELLOW': '4;38;5;100',
    'WHITE': '38;5;145',
    'YELLOW': '38;5;100',
}

VIM_STYLE = {
    'BACKGROUND_BLACK': '48;5;18',
    'BACKGROUND_BLUE': '48;5;18',
    'BACKGROUND_CYAN': '48;5;44',
    'BACKGROUND_GREEN': '48;5;40',
    'BACKGROUND_INTENSE_BLACK': '48;5;60',
    'BACKGROUND_INTENSE_BLUE': '48;5;68',
    'BACKGROUND_INTENSE_CYAN': '48;5;44',
    'BACKGROUND_INTENSE_GREEN': '48;5;40',
    'BACKGROUND_INTENSE_PURPLE': '48;5;164',
    'BACKGROUND_INTENSE_RED': '48;5;09',
    'BACKGROUND_INTENSE_WHITE': '48;5;188',
    'BACKGROUND_INTENSE_YELLOW': '48;5;184',
    'BACKGROUND_PURPLE': '48;5;164',
    'BACKGROUND_RED': '48;5;160',
    'BACKGROUND_WHITE': '48;5;188',
    'BACKGROUND_YELLOW': '48;5;160',
    'BLACK': '38;5;18',
    'BLUE': '38;5;18',
    'BOLD_BLACK': '1;38;5;18',
    'BOLD_BLUE': '1;38;5;18',
    'BOLD_CYAN': '1;38;5;44',
    'BOLD_GREEN': '1;38;5;40',
    'BOLD_INTENSE_BLACK': '1;38;5;60',
    'BOLD_INTENSE_BLUE': '1;38;5;68',
    'BOLD_INTENSE_CYAN': '1;38;5;44',
    'BOLD_INTENSE_GREEN': '1;38;5;40',
    'BOLD_INTENSE_PURPLE': '1;38;5;164',
    'BOLD_INTENSE_RED': '1;38;5;09',
    'BOLD_INTENSE_WHITE': '1;38;5;188',
    'BOLD_INTENSE_YELLOW': '1;38;5;184',
    'BOLD_PURPLE': '1;38;5;164',
    'BOLD_RED': '1;38;5;160',
    'BOLD_UNDERLINE_BLACK': '1;4;38;5;18',
    'BOLD_UNDERLINE_BLUE': '1;4;38;5;18',
    'BOLD_UNDERLINE_CYAN': '1;4;38;5;44',
    'BOLD_UNDERLINE_GREEN': '1;4;38;5;40',
    'BOLD_UNDERLINE_INTENSE_BLACK': '1;4;38;5;60',
    'BOLD_UNDERLINE_INTENSE_BLUE': '1;4;38;5;68',
    'BOLD_UNDERLINE_INTENSE_CYAN': '1;4;38;5;44',
    'BOLD_UNDERLINE_INTENSE_GREEN': '1;4;38;5;40',
    'BOLD_UNDERLINE_INTENSE_PURPLE': '1;4;38;5;164',
    'BOLD_UNDERLINE_INTENSE_RED': '1;4;38;5;09',
    'BOLD_UNDERLINE_INTENSE_WHITE': '1;4;38;5;188',
    'BOLD_UNDERLINE_INTENSE_YELLOW': '1;4;38;5;184',
    'BOLD_UNDERLINE_PURPLE': '1;4;38;5;164',
    'BOLD_UNDERLINE_RED': '1;4;38;5;160',
    'BOLD_UNDERLINE_WHITE': '1;4;38;5;188',
    'BOLD_UNDERLINE_YELLOW': '1;4;38;5;160',
    'BOLD_WHITE': '1;38;5;188',
    'BOLD_YELLOW': '1;38;5;160',
    'CYAN': '38;5;44',
    'GREEN': '38;5;40',
    'INTENSE_BLACK': '38;5;60',
    'INTENSE_BLUE': '38;5;68',
    'INTENSE_CYAN': '38;5;44',
    'INTENSE_GREEN': '38;5;40',
    'INTENSE_PURPLE': '38;5;164',
    'INTENSE_RED': '38;5;09',
    'INTENSE_WHITE': '38;5;188',
    'INTENSE_YELLOW': '38;5;184',
    'NO_COLOR': '0',
    'PURPLE': '38;5;164',
    'RED': '38;5;160',
    'UNDERLINE_BLACK': '4;38;5;18',
    'UNDERLINE_BLUE': '4;38;5;18',
    'UNDERLINE_CYAN': '4;38;5;44',
    'UNDERLINE_GREEN': '4;38;5;40',
    'UNDERLINE_INTENSE_BLACK': '4;38;5;60',
    'UNDERLINE_INTENSE_BLUE': '4;38;5;68',
    'UNDERLINE_INTENSE_CYAN': '4;38;5;44',
    'UNDERLINE_INTENSE_GREEN': '4;38;5;40',
    'UNDERLINE_INTENSE_PURPLE': '4;38;5;164',
    'UNDERLINE_INTENSE_RED': '4;38;5;09',
    'UNDERLINE_INTENSE_WHITE': '4;38;5;188',
    'UNDERLINE_INTENSE_YELLOW': '4;38;5;184',
    'UNDERLINE_PURPLE': '4;38;5;164',
    'UNDERLINE_RED': '4;38;5;160',
    'UNDERLINE_WHITE': '4;38;5;188',
    'UNDERLINE_YELLOW': '4;38;5;160',
    'WHITE': '38;5;188',
    'YELLOW': '38;5;160',
}

VS_STYLE = {
    'BACKGROUND_BLACK': '48;5;28',
    'BACKGROUND_BLUE': '48;5;21',
    'BACKGROUND_CYAN': '48;5;31',
    'BACKGROUND_GREEN': '48;5;28',
    'BACKGROUND_INTENSE_BLACK': '48;5;31',
    'BACKGROUND_INTENSE_BLUE': '48;5;31',
    'BACKGROUND_INTENSE_CYAN': '48;5;31',
    'BACKGROUND_INTENSE_GREEN': '48;5;31',
    'BACKGROUND_INTENSE_PURPLE': '48;5;31',
    'BACKGROUND_INTENSE_RED': '48;5;09',
    'BACKGROUND_INTENSE_WHITE': '48;5;31',
    'BACKGROUND_INTENSE_YELLOW': '48;5;31',
    'BACKGROUND_PURPLE': '48;5;124',
    'BACKGROUND_RED': '48;5;124',
    'BACKGROUND_WHITE': '48;5;31',
    'BACKGROUND_YELLOW': '48;5;124',
    'BLACK': '38;5;28',
    'BLUE': '38;5;21',
    'BOLD_BLACK': '1;38;5;28',
    'BOLD_BLUE': '1;38;5;21',
    'BOLD_CYAN': '1;38;5;31',
    'BOLD_GREEN': '1;38;5;28',
    'BOLD_INTENSE_BLACK': '1;38;5;31',
    'BOLD_INTENSE_BLUE': '1;38;5;31',
    'BOLD_INTENSE_CYAN': '1;38;5;31',
    'BOLD_INTENSE_GREEN': '1;38;5;31',
    'BOLD_INTENSE_PURPLE': '1;38;5;31',
    'BOLD_INTENSE_RED': '1;38;5;09',
    'BOLD_INTENSE_WHITE': '1;38;5;31',
    'BOLD_INTENSE_YELLOW': '1;38;5;31',
    'BOLD_PURPLE': '1;38;5;124',
    'BOLD_RED': '1;38;5;124',
    'BOLD_UNDERLINE_BLACK': '1;4;38;5;28',
    'BOLD_UNDERLINE_BLUE': '1;4;38;5;21',
    'BOLD_UNDERLINE_CYAN': '1;4;38;5;31',
    'BOLD_UNDERLINE_GREEN': '1;4;38;5;28',
    'BOLD_UNDERLINE_INTENSE_BLACK': '1;4;38;5;31',
    'BOLD_UNDERLINE_INTENSE_BLUE': '1;4;38;5;31',
    'BOLD_UNDERLINE_INTENSE_CYAN': '1;4;38;5;31',
    'BOLD_UNDERLINE_INTENSE_GREEN': '1;4;38;5;31',
    'BOLD_UNDERLINE_INTENSE_PURPLE': '1;4;38;5;31',
    'BOLD_UNDERLINE_INTENSE_RED': '1;4;38;5;09',
    'BOLD_UNDERLINE_INTENSE_WHITE': '1;4;38;5;31',
    'BOLD_UNDERLINE_INTENSE_YELLOW': '1;4;38;5;31',
    'BOLD_UNDERLINE_PURPLE': '1;4;38;5;124',
    'BOLD_UNDERLINE_RED': '1;4;38;5;124',
    'BOLD_UNDERLINE_WHITE': '1;4;38;5;31',
    'BOLD_UNDERLINE_YELLOW': '1;4;38;5;124',
    'BOLD_WHITE': '1;38;5;31',
    'BOLD_YELLOW': '1;38;5;124',
    'CYAN': '38;5;31',
    'GREEN': '38;5;28',
    'INTENSE_BLACK': '38;5;31',
    'INTENSE_BLUE': '38;5;31',
    'INTENSE_CYAN': '38;5;31',
    'INTENSE_GREEN': '38;5;31',
    'INTENSE_PURPLE': '38;5;31',
    'INTENSE_RED': '38;5;09',
    'INTENSE_WHITE': '38;5;31',
    'INTENSE_YELLOW': '38;5;31',
    'NO_COLOR': '0',
    'PURPLE': '38;5;124',
    'RED': '38;5;124',
    'UNDERLINE_BLACK': '4;38;5;28',
    'UNDERLINE_BLUE': '4;38;5;21',
    'UNDERLINE_CYAN': '4;38;5;31',
    'UNDERLINE_GREEN': '4;38;5;28',
    'UNDERLINE_INTENSE_BLACK': '4;38;5;31',
    'UNDERLINE_INTENSE_BLUE': '4;38;5;31',
    'UNDERLINE_INTENSE_CYAN': '4;38;5;31',
    'UNDERLINE_INTENSE_GREEN': '4;38;5;31',
    'UNDERLINE_INTENSE_PURPLE': '4;38;5;31',
    'UNDERLINE_INTENSE_RED': '4;38;5;09',
    'UNDERLINE_INTENSE_WHITE': '4;38;5;31',
    'UNDERLINE_INTENSE_YELLOW': '4;38;5;31',
    'UNDERLINE_PURPLE': '4;38;5;124',
    'UNDERLINE_RED': '4;38;5;124',
    'UNDERLINE_WHITE': '4;38;5;31',
    'UNDERLINE_YELLOW': '4;38;5;124',
    'WHITE': '38;5;31',
    'YELLOW': '38;5;124',
}

XCODE_STYLE = {
    'BACKGROUND_BLACK': '48;5;16',
    'BACKGROUND_BLUE': '48;5;20',
    'BACKGROUND_CYAN': '48;5;60',
    'BACKGROUND_GREEN': '48;5;28',
    'BACKGROUND_INTENSE_BLACK': '48;5;60',
    'BACKGROUND_INTENSE_BLUE': '48;5;20',
    'BACKGROUND_INTENSE_CYAN': '48;5;60',
    'BACKGROUND_INTENSE_GREEN': '48;5;60',
    'BACKGROUND_INTENSE_PURPLE': '48;5;126',
    'BACKGROUND_INTENSE_RED': '48;5;160',
    'BACKGROUND_INTENSE_WHITE': '48;5;60',
    'BACKGROUND_INTENSE_YELLOW': '48;5;94',
    'BACKGROUND_PURPLE': '48;5;126',
    'BACKGROUND_RED': '48;5;160',
    'BACKGROUND_WHITE': '48;5;60',
    'BACKGROUND_YELLOW': '48;5;94',
    'BLACK': '38;5;16',
    'BLUE': '38;5;20',
    'BOLD_BLACK': '1;38;5;16',
    'BOLD_BLUE': '1;38;5;20',
    'BOLD_CYAN': '1;38;5;60',
    'BOLD_GREEN': '1;38;5;28',
    'BOLD_INTENSE_BLACK': '1;38;5;60',
    'BOLD_INTENSE_BLUE': '1;38;5;20',
    'BOLD_INTENSE_CYAN': '1;38;5;60',
    'BOLD_INTENSE_GREEN': '1;38;5;60',
    'BOLD_INTENSE_PURPLE': '1;38;5;126',
    'BOLD_INTENSE_RED': '1;38;5;160',
    'BOLD_INTENSE_WHITE': '1;38;5;60',
    'BOLD_INTENSE_YELLOW': '1;38;5;94',
    'BOLD_PURPLE': '1;38;5;126',
    'BOLD_RED': '1;38;5;160',
    'BOLD_UNDERLINE_BLACK': '1;4;38;5;16',
    'BOLD_UNDERLINE_BLUE': '1;4;38;5;20',
    'BOLD_UNDERLINE_CYAN': '1;4;38;5;60',
    'BOLD_UNDERLINE_GREEN': '1;4;38;5;28',
    'BOLD_UNDERLINE_INTENSE_BLACK': '1;4;38;5;60',
    'BOLD_UNDERLINE_INTENSE_BLUE': '1;4;38;5;20',
    'BOLD_UNDERLINE_INTENSE_CYAN': '1;4;38;5;60',
    'BOLD_UNDERLINE_INTENSE_GREEN': '1;4;38;5;60',
    'BOLD_UNDERLINE_INTENSE_PURPLE': '1;4;38;5;126',
    'BOLD_UNDERLINE_INTENSE_RED': '1;4;38;5;160',
    'BOLD_UNDERLINE_INTENSE_WHITE': '1;4;38;5;60',
    'BOLD_UNDERLINE_INTENSE_YELLOW': '1;4;38;5;94',
    'BOLD_UNDERLINE_PURPLE': '1;4;38;5;126',
    'BOLD_UNDERLINE_RED': '1;4;38;5;160',
    'BOLD_UNDERLINE_WHITE': '1;4;38;5;60',
    'BOLD_UNDERLINE_YELLOW': '1;4;38;5;94',
    'BOLD_WHITE': '1;38;5;60',
    'BOLD_YELLOW': '1;38;5;94',
    'CYAN': '38;5;60',
    'GREEN': '38;5;28',
    'INTENSE_BLACK': '38;5;60',
    'INTENSE_BLUE': '38;5;20',
    'INTENSE_CYAN': '38;5;60',
    'INTENSE_GREEN': '38;5;60',
    'INTENSE_PURPLE': '38;5;126',
    'INTENSE_RED': '38;5;160',
    'INTENSE_WHITE': '38;5;60',
    'INTENSE_YELLOW': '38;5;94',
    'NO_COLOR': '0',
    'PURPLE': '38;5;126',
    'RED': '38;5;160',
    'UNDERLINE_BLACK': '4;38;5;16',
    'UNDERLINE_BLUE': '4;38;5;20',
    'UNDERLINE_CYAN': '4;38;5;60',
    'UNDERLINE_GREEN': '4;38;5;28',
    'UNDERLINE_INTENSE_BLACK': '4;38;5;60',
    'UNDERLINE_INTENSE_BLUE': '4;38;5;20',
    'UNDERLINE_INTENSE_CYAN': '4;38;5;60',
    'UNDERLINE_INTENSE_GREEN': '4;38;5;60',
    'UNDERLINE_INTENSE_PURPLE': '4;38;5;126',
    'UNDERLINE_INTENSE_RED': '4;38;5;160',
    'UNDERLINE_INTENSE_WHITE': '4;38;5;60',
    'UNDERLINE_INTENSE_YELLOW': '4;38;5;94',
    'UNDERLINE_PURPLE': '4;38;5;126',
    'UNDERLINE_RED': '4;38;5;160',
    'UNDERLINE_WHITE': '4;38;5;60',
    'UNDERLINE_YELLOW': '4;38;5;94',
    'WHITE': '38;5;60',
    'YELLOW': '38;5;94',
}

STYLES = {
    'algol': ALGOL_STYLE,
    'algol_nu': ALGOL_NU_STYLE,
    'autumn': AUTUMN_STYLE,
    'borland': BORLAND_STYLE,
    'bw': BW_STYLE,
    'colorful': COLORFUL_STYLE,
    'default': DEFAULT_STYLE,
    'emacs': EMACS_STYLE,
    'friendly': FRIENDLY_STYLE,
    'fruity': FRUITY_STYLE,
    'igor': IGOR_STYLE,
    'lovelace': LOVELACE_STYLE,
    'manni': MANNI_STYLE,
    'monokai': MONOKAI_STYLE,
    'murphy': MURPHY_STYLE,
    'native': NATIVE_STYLE,
    'paraiso-dark': PARAISO_DARK_STYLE,
    'paraiso-light': PARAISO_LIGHT_STYLE,
    'pastie': PASTIE_STYLE,
    'perldoc': PERLDOC_STYLE,
    'rrt': RRT_STYLE,
    'tango': TANGO_STYLE,
    'trac': TRAC_STYLE,
    'vim': VIM_STYLE,
    'vs': VS_STYLE,
    'xcode': XCODE_STYLE,
}

