# encoding: latin-1
# Thomas Nagy, 2005-2018
#
"""
Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.

3. The name of the author may not be used to endorse or promote products
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
"""

import os, sys, inspect

VERSION="2.0.8"
REVISION="18a001015786bd495c6d30d57e6a9ed4"
GIT="x"
INSTALL=''
C1='#9'
C2='#7'
C3='#5'
cwd = os.getcwd()
join = os.path.join


WAF='waf'
def b(x):
	return x
if sys.hexversion>0x300000f:
	WAF='waf3'
	def b(x):
		return x.encode()

def err(m):
	print(('\033[91mError: %s\033[0m' % m))
	sys.exit(1)

def unpack_wafdir(dir, src):
	f = open(src,'rb')
	c = 'corrupt archive (%d)'
	while 1:
		line = f.readline()
		if not line: err('run waf-light from a folder containing waflib')
		if line == b('#==>\n'):
			txt = f.readline()
			if not txt: err(c % 1)
			if f.readline() != b('#<==\n'): err(c % 2)
			break
	if not txt: err(c % 3)
	txt = txt[1:-1].replace(b(C1), b('\n')).replace(b(C2), b('\r')).replace(b(C3), b('\x00'))

	import shutil, tarfile
	try: shutil.rmtree(dir)
	except OSError: pass
	try:
		for x in ('Tools', 'extras'):
			os.makedirs(join(dir, 'waflib', x))
	except OSError:
		err("Cannot unpack waf lib into %s\nMove waf in a writable directory" % dir)

	os.chdir(dir)
	tmp = 't.bz2'
	t = open(tmp,'wb')
	try: t.write(txt)
	finally: t.close()

	try:
		t = tarfile.open(tmp)
	except:
		try:
			os.system('bunzip2 t.bz2')
			t = tarfile.open('t')
			tmp = 't'
		except:
			os.chdir(cwd)
			try: shutil.rmtree(dir)
			except OSError: pass
			err("Waf cannot be unpacked, check that bzip2 support is present")

	try:
		for x in t: t.extract(x)
	finally:
		t.close()

	for x in ('Tools', 'extras'):
		os.chmod(join('waflib',x), 493)

	if sys.hexversion<0x300000f:
		sys.path = [join(dir, 'waflib')] + sys.path
		import fixpy2
		fixpy2.fixdir(dir)

	os.remove(tmp)
	os.chdir(cwd)

	try: dir = unicode(dir, 'mbcs')
	except: pass
	try:
		from ctypes import windll
		windll.kernel32.SetFileAttributesW(dir, 2)
	except:
		pass

def test(dir):
	try:
		os.stat(join(dir, 'waflib'))
		return os.path.abspath(dir)
	except OSError:
		pass

def find_lib():
	src = os.path.abspath(inspect.getfile(inspect.getmodule(err)))
	base, name = os.path.split(src)

	#devs use $WAFDIR
	w=test(os.environ.get('WAFDIR', ''))
	if w: return w

	#waf-light
	if name.endswith('waf-light'):
		w = test(base)
		if w: return w
		err('waf-light requires waflib -> export WAFDIR=/folder')

	dirname = '%s-%s-%s' % (WAF, VERSION, REVISION)
	for i in (INSTALL,'/usr','/usr/local','/opt'):
		w = test(i + '/lib/' + dirname)
		if w: return w

	#waf-local
	dir = join(base, (sys.platform != 'win32' and '.' or '') + dirname)
	w = test(dir)
	if w: return w

	#unpack
	unpack_wafdir(dir, src)
	return dir

wafdir = find_lib()
sys.path.insert(0, wafdir)

if __name__ == '__main__':

	from waflib import Scripting
	Scripting.waf_entry_point(cwd, VERSION, wafdir)

#==>
#BZh91AY&SYΎX0m_Ű 0#5(b#<{g`#5#5#5#5#5#5#5#5#5#5#5#5#5#5#5#5#5#5#5#5Mv20ĝ ԶMt:{+ݜνyuyw޺hݛ7FݷnӢ<yvv-սy=ۢ]byzcݚKtdwohvnu{sz/#5#5#5:#5<:)2ݶE;,ͽǻU#5{#5#7h)^A(]7#5iTHÅJPTTZ"|{};0Ԟؗ{v)SUI[m_OS}W^knyDsw{uF}jvnͷv{7595Ӫ=sR@ӧ*Z%{\竆Jn4n]TEzD*iz=#5:{{M4eڪwǷ->#9#5kH+ctݳ24w`3o#55۽#5;s:غWUW_{/鳘W,ٛm=o㮛nwtg;]Nsyy{wӻWkAoztk_}a_ɭeW.o#9J}aitgS}^ɳ>z_n_]Gğ5<F]7#5^y{{u^PUfDYvTvlնXE܇nnC<:#9NRv]wnݮ<#5#5s #5Uo|v/wQV5Wao:=dDvwHl}16;h20ǋ7;O#5]{wn:zo'ajKmWxizu>Ϸ׼۸.>kmΧg_M81ݻ}(XF)]޶#7mU9ჽs˖f#5`SS}{ntEzwqu`^nʛcu͆Yq|iԚ=uСM;yַa2wm7t4{J#7)oJoW#5<Ǖfzs#9@x:wZ@=vwt#7{{\m#7uHRvݷwu펚N6lʕG3u=9Ǆ>+ZFF!Ƿvm{y[odoRw5⯻/6s_8i#5 h4dё1d#5#SOҌzѦz!4&4	蚧~6?Jm'#7hz#5#5#5#5 D ș'h0GH#5#5d#7#5#5#5Iꔑ	04My#9h!= QI#5#5#7#5#5#5Dhhh#5@&#71&B'be4#7#5h#5#5& #55M=5'螦GOLSOz4#5#5h#5#5?kuvy҄Lh]tLP#Jz0F~_3#91R2R3XݏenIk}|c_T#7j	GblۣZD/uOJx{US#9jU;b!^a#5b-֚ڊڤbB" " 7PH*-EB)`I#5)I( aBGA"6"(B lB!TH,ȸBEU@DP#L͛(	A"LʄK#5R(M&hd5`6i,D"4KD"Tb()E#5bʕ$RlK+)@!e6hI(&f1j5Qh4!-10ѤR1SM[6h2&[i&IS[j[flͤɓb(6Y14Y#9(4ȣdH"EeRA%,lD0QbQ6!h$J0-#jDDib dD`"SMX31"d4RY61XK)	-)H%DL2#92P4dĂLJ#I#5SLƉ"XZb)&ƲC6"Le͂ab6	,!0#$QE$%$J4H`1lLRj,͋ZMILB!-)IKbH̢fIDfF$J3dL*bHE))`Qd6I LMDjVK3!2)ZDBJd#7T2E&iE!SKMj@bK(4FFcJLe"Th&k(Rb#Č1	&jfUMb#7LS#6,hh"I$$ccFhb2)333Yk"(&4!ъBYR(#$dȦY#FRR"bLI&Rl&H)&dF0Li4I+fIb,i#9&D%2)A0d&K65d!4HAh&U#9DlR$FQ,j4Q4`)CBLdd"h,!(JLE![jZ֍ 53FR$biE$DEh5)l#9YRiBYeM"ale(BR*!*mIF(4mcZ)RHQj6##7CIQ,TXD[3V1&RڈѓL#,Y5 -4ɓ1#96PٙdU*Rc4رdFM2IS,JJ)M254H%#j,RbKHdjkFJ&hڢب(֌*hT2ѵZLX-J)&ƍ%cTbH+jBBe3SVMlHDQdM(6e&ͪdEK"fZ#L	)XУ6fȑiIH(QEh)"FD+"Sm`BT2Z#7ɚh),hbMcdEeH$BĦbPe3Qh"dFbœd42̉j2	MF,F&!4D5Q%2)%h0Ĕ)S,FBQ	I%FB1e-P*1i"2XE%#92Z(0IF̦JlY0%6!I-EEFhb*SI,Ve@L̃LJ&l dTj#7*i,Y&1iFJ̶4iQ4ق!T6(٤%15$$E1V365#7&HM#76LR#b5e%̢#9c!d2#7mQhU( j2aXAViDԕEҚe&[2-EQ`6#)m].5EDS2E!)i%MDL#mJTY0,dkZLRVSTm#7d3J	bƦQRh$U%!RdŹFٕ(m+&H#9i&(M6e2llhL%"4ZYhV-hFRVȂFأ	F1JA2)5#ŦlME*%K6VƤ(آH3TU-3Mk1MhYm6TRJ3s)5(4X9*4X+$mI[M-Fib5mZR&cBbJFXئmJ$cl[b5mֲT[)dSSljєQdk$k#9Rfl((LU&Pb1 	?eA_Ž.qrѶ,`m	i#7M5o^2i*&\S?hû%qQB"$LT$U	։c16I(R܀#Fd[sUa;}]i.x+|cLᚱiUÇRR(lrrXiK#9cQ"Q"l1>^NMvNBonLٔ)#M^HƘ4R#9I4fS\%!HBDP]Sd1-#9lH#7I#%=yKvޥ+zYh%PcEݭal,d[BE U>VPx=1\#94m(QfR(rV?H4`Z&xo}"?=GDhC#7Xqlgzv9ST+9PStI݅,m.t(ͺyhشO5m(R{ꃋ>([<Z(1#9ilnvFݖ<wvK/,{O2^Vp=\ClƿkޞU&ѝ)McH(kl#Cz]v.Qҹs%u$d/yy]MRH:T43le'X+H~cj~g$2H_QIz`-t0Cĭ5H(#9hoK峿xx,	&%aܫ|E;ko'-ֽڝn06Y~N պ8'bjL$ux3+>5QJ~Ο]9}ݭӓ_=C["	zOM6-le$@;fiDg=Gy>劝UE5K*}+5S	ansc{6EۼJ-ҏ鮰̔C;/XAځA}([b{ztsݼ֎p"x7ɕrf4"+ QW&xڹƟ[6Ȕ~,ʌoץdLb$4#7(mA(;O[{K⾏;rǥ45FyJQܱ-׻u*$G^[=Fɷ`<ZQ`uOU!xmf3S#9d:ԖP̩zbL0Wmd.{ݘKdدG!\0YLRPŤ'mH.9<hqQLUiHN\tw'7@vRED#9(SfZjASXUXTS&e)MA"'~+cgTlH](©zB&~:`Vy>:	#V%"_g*!<synÉ7]dՔCN8. )F8QJoe7]`If`;HjJ*h84Uy]\3~		elmnsE]D#$Da0gS˶)*S&G( mGSCWmNNMS(SEoCR9Ho3ұrHD-T,xg.h G01	a`'n|~V!HKaC=Fj]ڡ{<!&;N3ww7b>6/LČݽQ%lɹZx6GzVDX}ZsJaK=j#9P.RJjs4i%e䢘":BI}Mgmʏg9~NncI[HG(Ae#7:QM{3>/D1x;)w$hJߟߖEwh	cq{p֫*#7M=@..5vch܆8N6χP{ɆoI",Q֔Σm8p΃`w(2lA7⋵uEܟq~=)ڣjꑣѣӨ|CkscEZ/uCZxp͢UqSB?*Ϸ~O1g3W_R5d6CUy47~zf:"ɂ=hDQ<7+;n2}zKH">F=/ɡjjs#9QX>wus_kQ_{μb|U:Q#gOv_wJ9UgLZ=1D}{H3kmYM]k|"ihv9bZ֘k&0nZf~R>Gۥ.PH/:(O>Yb'X.p|:k'aNMG*c*nRȻk{k.AE_߳7<~Xi6.%>ǳ}Es#9~`<ݝz[~mw`n<M?KuӌG[=sv *e+r5jpeH4Tfz3//kpf0őH"fθƱ*,P+HδcG׆FŹyu>{2C<Tm|mz85`B4q{f9#9#9p/ѻ(T^E)VtN6Z<xT4`u"Zx0)AX*0_-l\kbbD;~>,t@Q<Hh\P'uKoX[ҥ'F~ZׯGѧ<?}+=}#1{Dvz˂IiKClE]cOmnzNP^>֔4:LUU5re.SVb8`1Ք3\YghrB(7+|[q;oJCTTi$@5/y㺖k(.EU{Ur0;0*_#7/pZh\ӳ?ɖwFhz8ahcP"jf;RE=MVZ7swI#/=={+yq-W)bLQu(K)C͵Plt+|Aj1"TtI:uTU6H]mdEF0p9 j*iN#94"=ؔ4v_G#pY_<?e:hSb!$7}wT<A.hP>C#9UNG]ΛՆ@*.m++&m1ܺJZcdPu?ݸI5a9		-U0D}!je8T!_̒HQ/XN˯]S6:ЫNأ=Kc!udp"-/ʯ<>](ҩ_ŵ`|ՈrCCI#Z=9oqkÊ#YS5GT~v<#Uy{d#7C?Vq#׆,.``nlu}tzф5i%UDl\PeP/M"bgd#9$#9nK2hCbլ$i0((A.Y,N.TڇHZ7y_/d;[9O#5G,'s)ƾXegًrةc4IqT59P+Eݵ)ڰ*鳌a(t*T-_.AV"O<S0-F`vkrl@~*?7Oyp3G{]tF$#"r\XFw-yYX*#9mcUiX#H=D<iEgCt@>J!!P%Mձ-_%/ox&7I0"+hMg25YLWҵOmSd[`'k%fk$MAN7?4kuZ9jaE'LecMf"1KjX9Tv,	FVhɢW}3eChmY!͆90Vwq(f7@*j <Q9k+lၦ&35%#9U2J#5eM^;wnoȥU']eձs%ķ#75#5Wy8coP6bI$:9	;*,}DD}Z1.!0)DȀ 5p[5#9$oT#9H'_醗(O*gɭko='4l"*#5'!	O[JN+)F3{Ǝs;NΨHf:`-vF~bqR#7o[!w#f]#tc#9L6|,"dX(*-(8ȇX|5kJ;:h7Yff"EURa5FGvJCl):U_)Xor N ۤ*uO!Tq(Ŭ_y{T]#/\wF]~'Pׂ=~ɝX$\Pun1po\գ:U$w3$#H8׾=y0IC]§kp?#BK/7K\Gf\q#7LvAͧ]V#,i9:JuVaΟS	]zQ4P#9V#9(ij\]S,];:ɫNҠK=ݷFHq{t#9G:Fº{qѽYȷ̆ysH!%ʘ)#9(\{81R/lR9vm/>`=$Dnxgptcgi<*Rj̤5EC̱QV+^lou]ZK}3Ui.ʔPvj6{J\j/"s<A5^0]HDPH+v(&`NIE>;w.i`岗CSb>E D`|&m!Ơv"|M9wSn$ݧH'[1Ёv4#9NwnV|űY|Vl=u6i}%|M(MZduR2.:-v 	כ틀rX9^OJְKyNźbwyTg\ԶVX7X@\j(es[8K\6Uta@fqMa]79Ss;3.t/%*gj쑊dPMpѣL%Xb&㒱Agv\##95QprNaQHD4umuХT3!A!ЧArR	c3s>2Muj@96:#7F{(`.p94y'2U<#5yP#964?7#9"D_	tx#7rsEzC!bIF΋(nw7	-X'ڇp+eȀ 8|#5Aox|"$8&ÏR8|#95焒W|#9؏W0l_]į3!πp4`ksm1^.G6	/Ay~Ɔ#QvͰ穘;#7lQ3y*GH0#5#5&@E(~˱Z.=#7'Ͽ#7sz	? B"`j8BJ@Gq#7#9j50\Q#M$@=ݮ3G2Q}FRӾ]e޽tӔGkJ?)7gNB"$:"1ggc<Z?.Hҫz6Fc, Jh4ӻۍso??*2+o Qo?5Q:C"clݭlQ*->inHLdԎ,`7q\8oJ`!`N5ţ4b7Ѹ~oK+qL!0~$i%?tJcrL'GÆw]e0oQ6ySzG/uZqth;::'>20H{_yS>A7o[ pJYirw8qR/yoC<7D"؂,,H0Q1+E![B0HX(qLlRtU&o-;Qb=|Srw{h]o=NuUKn6ݢ*dBxIrr:(A@#9Y#56M>yyq2#5Ecw9v̋.>'<@x̭#İ>>CL˜ZH04$	"O68#q.mBsr#eo?f=,]DDq#9'O+ÙwO#7s5Nw>Mj0=m{c_foRBY $Gϖ	4LO?ڤ<uԗD;@R]spϣEv1'MO~n#7ֺgHH[#7euTe#OT/3~1㶒JBbg̑&uU	>or _P.+<#5	gɶ}; JJ>>2IJ9*lR'4#7~%iovfOdzVKWv@~%ebӚqWU:#"'D;ǡ#7zVzCI|x^ɺ,;GYRh5Ti@Ht~p7x?1OAWL-HI%zj#5So.j/ZQk_z3}+.")#9=09sP&hqBIxh:EO'buc<LЪҊC%B~@xoJQ34+cPq`dm!-َuEھY1Ò ()#7d;ȴ!JV."eҘ!^pO694X^]l8v)Ɩ1yF/Kf)8.1aKj֫i61 "zTǐZ1 qcAω֩MnSyd&wT`1`{ǧ>sӆ/-و$Фլ"(?]b5*<?;_Xn#3(Ұa+eFPD(G#RּyS8;vH·f	#7ARHvjY2V-[:P:Zڡ#|v9;"KW8+6Hz/<:jH7Yי9BWzFBJz.r>jm $(A!Kvq2<T#76dRyݡvRsz]T!rLn)eSQRq#7㮎gsE<_dlv5:CX{ZG{XOXuy#7<Gc9Pᝎ^mjGqȷXZqqSdkXX.ycv!%@t~,U(&uB_N4z$#FU6]x4tvOg01C\~_YE-WO!JgܧާogփKM<7V=kڣvѴ_!/|	X;Q@#9dEDwk2qi3i-#9\{6]N#7+Rq\  #9LFQ0W#9cjZ0Oݙ0#7jFR M :`9LAv:fMUbRs.64WppHM6?N#Zⲕ{#7<kGlp,43Bh"/&J4C|s#7dCNvy6?3ޡ-s9TT)h8g#9dU<\5lOiצhێ`R#9D4zvU"YI|/d#9us (bI~H?Jѐ#7̉\R2/H|*m~CEA'	><=\h5اH@_UikMd(-08wٮK[-LS`q3	[N``KcBMP1ZJ`!v ?*b@+J1+`PbX#9BlϏߥqǶ|7Pل#9#96LZuyn@kR(ŉc6k{ѱ,<bifcGKszbaF*E,uIeRـ4`'2{g_~owXg3Èrr#9c/bv@tVA#5x0TLzǐ'.le߻ʞ)>ɾL~*ქ=6qJɡѽڇl:mv	Cæfct;KcYӸ)&'歇!XleiWFca,CvAq`Ϯ?eKkS"dW*TJ4#7JҚq߄]&6IeA&x9+ܬ3~+i~N3N/>W|8{!΄M:7{`;F٘ri՛ob,ވ,>:;@QS0`G	R颯͘v5I	-⯫bi*~a	vzȟ,͇t;0K%kps]n7fGo;~(=#7H;[}.I#9-iKFi?O4@zttr#u_RB¥pdф)rmϱ4ڵؽ15gRb%b$L,Y.)dXU4Pȳ G;8a#7$:PRfX%Rik,EΞ5#93TE!lR%0 4B0\H;#9	a6PY#907%Iڴ6da)La`T2)ŦfoÉݞ!֒k_!lח\oQ449Q@BbYFNpv..wC9rb77_URڒjN\y:·"Q12G	ob{39"1d-+[BķXϦ&tǿXf8lZ!g͏k=znP$\GgGK%f8xu=qbM2'l/)ܶI'Zm7Mߑ%ޟw4җLuR[e(8ᄖ#s,|,$tH̡Fll>}&<A.HLfsrcɐ?Ɣh}/5רu)xXF H4e[!ٽ{j4K8e>DExȍ*Teޒz+WڎjVe%:h#仭tf?_KM`Й_aica5"2igWۅVBfnl:a/BmskO+Y|i9v(M|=_ڃhE	vil:vp? `HZz$4WK7Uc+ںS/#۝ǂ9!jO62kFnjKktܶ6Kw!CEmCłDkAdJ-~:.Z2"VEl<Z>R'bc\Q"@Y#9C{ӜPO)A6u+mzPr0~wPg2:k_//7#54jt)~c%5w6N~Q#91KOEH4|||ZK2'6*X((QFsz=NDMz{?Ǐ$g6M}PvW4+ł$S1C<!.ouq_FEs2Ilx^|ـ^1FnknYt3Ϻ໙t9=Ymx7.|BFq!	;5'40qrB<$)wGim"s-޿!@ibKQ*b0dceV|y+|<WFn5T2+\T/vH~X#w^F~ҠiAI#5:ǵÙxL>	߷{s&4)maQlkAͩ,$lbQ~Ѧ_@X,kEJ8}ojnNu#5}M%tRiN/=8#dC<`h"jSnUA8k/=,a"(`#9ϩ)~|}բk[lKNF+~U~k}R5J;YN):0tQX+z%8ESDPB'b`}sҎ*ɍm((<CKk5xCˇ9íD_%*_?zpGƌKiԿ\_Rr}eI7dw"7<Ǯ6CAB?{"\N$#76+8S:14#Kjkɉ!L</&m6kxW'#B#90Mo,*6!FHd~Ǣ~QF"(t]7}'m!Ҋ0XL0R,Y% ͺN(S'>27	/Q~T7?߭a__Jx4xCѼi1zhY=OO;ꏘ_=޶Wrd;?d\1&^ޥp,YHq.q_.%G}<֛Ic]#7]7u#7]SW;~()۠u5r+/-GY&UxS*/Ƙ1@yޥP85oh:ڌqÖp LF`{tz Cmmm4!(Μ48(5}6H@U|l&1@x.hI0z?|.j$x|(a/#9ExE?C#5Ps@{B]QwJklA.*˨#뷏lSh&v?o o+G_䰠WvLB鷉EkuHԇP?gDB'%+DtVaP?Bp\ *!^BJghk}];RvΘ(TMxCIm$2>(0BfGAEʗ#56Xo!$5OƎB$gC)1Q Hl}z>[~o>~<[b{<,9m4+~#5|u/LV_#9rbCث8!H{;II؀mBpnǲz/χ-G͢D=Y8Fx_1!1.δ:'HY:7)-(T#5]pR02˯4ס#{qp7A=b32pѻx>OT}_SϬۿAt@WFI_>~X+n1~w.Zȶ#9xx?#71iWgݯ1fߖ|c`+5DH~ݍR>IvPpbqݧѩ/%+Mhvϗ/ɷӞ^s`;nt|-r`0_CҳǦD~O	/=^y#Ѐg#>+e%~|Y0N~T 2#%[)9;N=pkn3aŲzk}Pg?ٳy	ڜIU[g\e9|ƸL6[O*:w.rӲ:GQtpQcC%5gH]wlg+]d,gu!htW$v-t1Y=Bم4f\I\{5rKsǗUɞɟ4&cHW_٦nCs<NN7Uw;f?/HIXT,4wzo|fV[P.߿d֮8~`<+|qvL8ۅi!a#SXȦcoQUW~+hjתg`U޵b3˻Z?~p04LV^kŞ_.L8\+p?D#99_7~?F##7<n_a|cy]һj#7F}v|4WRB|G,{7krpmt>'Z8{ۇ˄n#5)on;CJ8_SAn7#9yc=V|[>?zvG:;k_8S#^YG\?Rx~K=߼KEQm~>YAQ[Y7pf4r@|!b@#9@Qi@:c#7Z?3ξt҉5|:F?wk?G	Bf;)WY&[ײKma5!vDy	ɨy;{)^1|>:~!Bt3slzwQ#5P4<3nnO\4\[o`So9.7]T{ԟ8'x2W#5IUz8?g>;Ɂz<*G#ûڃ)6~x݈MDH"OxwM><hvR<c+7+gw>2_{m<E#BF=}{ruqH~l0 Lx+ m9ENqdO]U(fݯK"XNZ)014IB'#Xκ#9`jnviqt7iޜy]ZJtpF#9D~Z%ЪWF6@[?"0w-bI.>ȧ0	lZoyuOWXNQ~XrM{<WrkY(i2y`9M:ż{9+/kֱ1/1B+SכwWޢ?Nl9	8yl2uRY9)2ǋz)OInzf)_]ZhӐp#7E\<WOQGs&CIzmfϯGG/22JOE\>ggc9ˈA@6q#5AUǒ}~-9#9'Pzy!ɭloXuN198:ŁuQR=iG'wY`Yuuie6Y+WG{Hv#7[Je>da|sfl79CS@r'ml<+jq+YFI[2<$݊MOlcdc:ٌn	a|[aG p<_'A͠heE~J9#7%?Nxd`)<>o7``c`b~w_J?gR!j32S$l+nBVX5 0F4j4#7*GGjQR6ݱ!n,$UEDX-ۋ!Ywr,TthD"*KHR#9kaIc4߮3ZqbcVcF1<u;౶&wL0mioZSb:	|"8DȜP__Z]9m#5;(?v7hܩLA3ahfV^Bw!alXEL&LB$Rg}SZ7prQևb+hH::T]-&7Qh%Q&FCnDiHpV#9XTiAOqLkߍ2̚o)î\Hdh>ee;Uk.;9N_euj^Aew&eǊ	)we# DCV"oc;YVʬdskamaR$9]Qwի٧W1HPAy_8sH̠bӀf<\A\#}يYu~ʻ܎.sWI`ahaULsU3ͣvO[Yov;eě~NYgux-	J(~~ 8^TS=y3KnBmZ+`HGX)%M`562AYnWcJțI<6E|,5TQ&Q\hIn(-p#7{hag:#9!VUф`$ԛfk!#9#9)ZA0e[0 ü+muS@:MJR3qe-g8o) [ZXJc?{iޖt3Lu%iSQ"ͤJE^54AQTFv*1G]dc:\Cg>Ua MN$=?T8(aL19.hʟ<xǪ?q_֞#mGȥǧV$KH$rSU(=Wc_{5t	\<k7xt29~h]>ް錍GepgO(ѱس|[Y#5ݮ`$t2 #5A|E#MYꇝ~ʆU]̞ͤAeAlOG'@HGxPryD?8>pTn^0M96I P#9E0wQ'?g%`ƿ|~}Ynܖ[]I%ɓ?wnb쑀wnqup_B&S-ZʨUVYJ%QMC5H#7Dl˦mg*++FFdϗgu~U7|<iJTzkDzPa6^rE%FɻR0mjjsc0ORǖ3#7B}iF#7hzəZvf#C"2|uay&27y#9G/S'h,,Op@_"$b!3"8*2.πeA9CUr۶"dԝdLRbBc-!B)2LKDh,j9M+z4"6mAl=g+jiR#9(-c#7k/4PVk*-R(mnmi@OAjtS30^9#7#9TVa#5J#5B2*v<_nt1sMB㲟}QLaQ'JOd,|~;àY{f]2)}'fm^S{ANPy&TFYdHbŅUOJ كN0_}bϫHsqJ>$`'O|UdNT$dH&hwU*#70Nz$<XŵQ4ޟ@Fz+z56]hh4hQm1tB!S?dXC#9\"C!<<,ю.ihWpCGU(?ATk.&EJJZ\lnǃ˙`!I)0שs|]w@xnUFRwtZd*n8HQ3UQHiWF㐊c#7RL,498!Ƶ%k#7EeDY7#9K`H1(K}EeOLh@U廡#*m8V6#7xF^9&Dˬd( ᦑ ,LKbV4iODD{daXcLclɐ:N-nج#7B#ȭmc2&y5-ߚo%`02^ИuZ:'?<-¶Ftc	uvGm޾N2"6Dԅ¨g<S$'w{R?lSmnw5aNv`'@ҹ#5/&#9C͎@n_[5$N^UBh.zftR\cmV0Ld"I*N%c[ѯrmEu{7Ƃ##9Dj\@3"Qa,PF/`PwU3y?wC~#4d{LYr N(Pm`qmmQI؆:!&/wY!!FQÃ\YKam.#5w}VGTb͝lҁ#9C&v0!H,v#72gPrם15V!!.ÕjqzG*bdʢpJ'd&eFm3Ɇ!9ʍ(|7#7tTGK#7n47BGØ:8	)]ǂSSuvwDS[W1|"Xƃۄc-vƂ,8d6{.ZvNSvs<ٲP cftf,1+&5Sa3}\;d!hnى`ZXNnqhH'pi6aa|`s2+ߨb߮vfٶs!(l?9<db(m˧%b\N¸=>;tm͔'#t{Ddb5[mW[86(L5\m4a=#9ݶJE9s0,.qS[lI5QMOFFa&4Sn8-aixeVH6B8!e`s.gJE{fڨswҷŀzdɡNN*ӎ9`(M8"z(½qKݣUl_CvODܳl`Q}&,o89mAoIo,<|%=:BOʞrrV1n$֋q6]p4=/g&>z=0Er3O]-yiw	v{!t)\%25~ϣaWuLJ4VcM_}D`VbqԞc\qm$yMIkXj	CaCk	Q` nuq{~#7)"<ٸ73oy:;c#7'`(	$61#Z_CB5lC@|#6E[~v4}l^lglOpz8.x:F͝G4[7/AȀ)yt	uIy\ rz%=m;q}ٱ;3>]Ҽ>{9gpg,mR@rr5`CX&ѐcE?}v'ⴠf!P\K#5~9}?in)~G~fwnpEBF#5>8vHG#5Wx0nƂĘRVi@81	epRn9[R֚:"#0\7O!p7iA:9C!|pr N#vlbjvBLF3olnSԿ#7KQuQ0$ԩe1?#7ȥ%[YpF\#9Q*m#7̎Ѝ`%ӏ8{qǋwèThx/0&#7bYNQ&wFV?h0ׄ*HT6e%>mE;TOGmÃ̱|$?VXnmtGIRjQix8o_g݋5(4.ܳlrL8:Dg/\eexLkp-eǰvs/ѷ=2ROMŅ:&3F}'ƻs:ٻS@$b^݊i^׼c(igQ[rtԼl$43 RS;ݗmWp]G|R,xoPl1MskBpKmݫM2qk[pΐ#7@E7,#7|;#nD<!|{N9&ZLHDa1AF1Dtazk|<3RtZ?f'N~c߷zlBM,'3lqeLFoFy3!mAn޾BƂ>_xDRz(݆i#7Qo^7qrqBtC-|?)(6iЩϧaG`f"Bꋷ<<JHNh:dTuKo.@*'jawrk3^124W<2IASۻa$U<#9N.ZDkVj>qNtL\語mA#9<"#h,P1&aW{EҞVL	#:rEou>53%d>_ϮL#9TWp<R(Z{nQCz<m|1]ĜA*y]Os,g^@]`AOeio)Xe)էjS|T<WyՒ&6;,Ujռ2MWΰ<NVfw(V$t]2\tK[ƩvY|JTsN~RG><V38wa4It²7}LND#5 6yDiNMdd[#DŲE3;1M%Urh6wl3'h\\;÷nG	t֋15E-h%n1ݠ^VrѱTb̫PX6޽nwwAMIc<ּ6KtϋoqǙm&@׃;ZPsN'#5&uAJPR޲|.v6t|$ P7EzQ#58[(!j6/.çg=?zTBm5CبzkLڇNHDǝ/)7ףDV(~-΀*Z{f(O΢Pyܲ26Ti<ȮA!MWIٴ.b*s\1϶s;#-)H戧/6%#%wٳ+eXDEJlq=.ssqmx&Iz#z;xuiѿv+D7wӂw6	ǰ*l|m3ȚN܉WZ;nK7h}* I9T̍-nmo(u#93u?k<uc[3T:|#9n~_:0H&RfY؂"uLZgOWgLUg^>;<ִ=8D'ۖ8o&Liz=<'2	6U9x6>.,[SĉR<ZAJЅ /Q]'-6,8tVf)Η*崢}U}z)齥3fuqaD-{7[,,p+ kThg#9j&!ahr!rؗ!#^-Y:幢j5b+u7jg+(._oP?J͡>/<hUȣ׮+Q/T௻LTv«++W	ͨ853tY[7ìuUfj0N)+Ecծ(%͘rQRȽՂ#5}xsڰW,nV\%mv1kea%Bz<?Yϵ!'~QiSY'}{W6amMtidŗAM.P,`Ϊ6e_#ǟ۴	SJeGuyTRǇZui+0űΐv@.EiOҲ;:0w=Ku4{vNb.s55tpSvq}fؓaVkW6*m"$:x|Y㛥>I)hNACZeV.$*t_`pudn'3θ7=<</]*,ЕtjN5.N^bvSՋ#x#9Nmߕ/IfOJ2eul#7B!+ȹT|tI;@08)Xz뉜۹u~5L4Sϵ:5̼/\pKE);.s#7~v?3,D3<qL_𘺂APl/oe?I0$ɶm)8FQL4;6Jnic `:'JYw\%-Ç}x}E~EjH0r^ܶ@:#5ewKqcu]4a]ck|}$2gwX0qT˺+<bn<wv2	AU`4#9_HR#7}y¶=Z39g..U	+堵X=lg,yyZXt#5kk#7ۊƔ}]wKjۿm׆T~5^%P^zFׄYMޕ#9pR#9T|-ϖWlsG9Tx&p,[ږF|UC[DC$b7Eo~SP!푨v_W"8ۄ'n6)k4',+ -IW{4}ށH)#G#9	iVnoJ'0N_Qj^0\F[1#7Im*:;2ۆڲw]W1ivW#91"նzqVBǡ/6b,cΛ2xu`00xV+zK↠fP#C.moj+,#e2[Sw$6]f>^g-9PRɱ,;l꒿=}q6¬uDW^0ץe	X#7#|\v;[vjszjPuyP92`bMYcǆ0@֤u;n}c#9O{gǤ;N޳1':א('qw-cc~fCxGY#7jLvB#95S䪈JRo7I%~V$$yI\ǚ6bL.RVY"!v]XzS#5GQB I w	sJe3vմYss?!N~st?'%}nh=NGㇼG:W|9|K_dC4㪳iy{ƥ8SLg7qqۻjLgKw)ĴwJ!fStU3\Np7EůlCJ۞luUnQ)5CA#9xĸto.CƦ%5!}V}{;k,l{/JjE˿;úP)SHtx\c\Ջdi74#9[IFmWYsm[H.D(֘QhV&atocQ%QȜd-(,#7k\kijJ#vܹ,Z^6󘐴.{+e1\9?tē)NK+,hj}iq7Ti(Ȏ9/]#7;Ҁ0N·XGC>?k1$Vuw|j&Nl_lnM#5nSj^%R+K3CkO4^殍te%J=zj#5/U;yFG$fi@3e{DSAobQuI広-?]s	yp>DѨҗoQ^u-ma_UK^a^X)-8+sWfKS*X#7=2(܆UfmΓ,J;Jrk	z:p@m7PztؠHijyF0z\>'*@uv7#7rGt癮D鹡b&qA;\@V˓wQrNHFjU[nm`[KS8eY&|iIl-I*~^qiB읦w[v6ߍs/W4&#9GaK"gEaj#7(.)aA#5P*hڶfR&ska?|DB0z@zTK.YQJrP|/վLkXXsL&Z_ iO4a9#73$G,,Ӧ1@`2&%0|6	MqD<wUkmx=}D*a<6ORjrDPWqՠ;l0;.eQuo<[GlMHe'>5#7ѡ)f7Q/>XsJUYYH}E"[Q^Am0F?-oiCq輷0OmǇ;Cj8jXᬺ]<hJj:tqf/.M_LJm7!m3%jZ-'$Ŭ63B?V D:o8#?#5-\/y#7aq^Y:[H~n#0 NO`[#KGd1RɽĿ}tbW<'Yɮ>5FTx}J-Mʎ<xYURG8ؒ۷1)3mx\7mUa](?yLo YΘ^xf@:qw|L[6ٛ#9fM׃P]tV#9Լ=-DJ#9Aܩh^	HCݎfXWq7h1^,tPG,[""RIb7i5THo7M1@|]=A	cg˥)>rFud'>3R%pU:pt:*ي=pPF+ sիxc܁jb;5-/g4#BGݠ[ޗqޙomS#pEw_zHfS31:PeFBd"Oׂpgf޵D`%~q5"N޸$gÕbL*BGRvuA\,`,puҡFHe-bԔ^<؛oD._#9-W/2.#5Tk	|n]{z41п(pt#73vp)7~bvNmz|ujmd	E #5u#9#뀫|W鹫Е;K#5{ڃm_jiPBFE-,Z5W˕{F%H>N2#5x¾SȩQ!&іؽ>:P6:<}#7!qpz9#9B6{zב_W@ªςxٌUH[RGxǬ1J@%ӜSsGKtuFt8ԼT/1y:^~*[P#9%t;{=IIB<RۺY\:C;׸evDXj$m\̚F%htg(g R텆% VL0}Jl!~Cro#5\;3=_}9_YJ@#9=*#5<rEsÚ'ʍhǆ~"QoW+#7"LOW],./9;)\lIi?lޢĤZ+&-mH>1&mi Sb(g*4B2!pA,[65L5|+m__/U`hRQ#93hxCdQ3\͝.vopޮ!97XW$$PI|`c?>㓏K0Ilc#70mJ`ZH1JrD^y9[#9P3hs%ql4Ts1aK|uA[]2[p4!Qya_@sFW]|Am?@Хz;d"`lIwt6.֐󳃊ytD<=a!L$?B(5.k5p1G@(G=b~qG.w`v-#5ːo	JmQѺ]ߣoiUfr#Y{ӣ޻#拌8#9?6^<6xRV1@r8<[n\C~HL=FP]hYf@Ɇ=?E}o2A(EESц~?֙J`r#56pBeKԨtz`S!@)6 TgɑLJæhZO&Rԓ&$փNJ@-ETwmI`΀}`FkG+XOPOGc3#9k#9Jam$ ꍓo?˻+_}W=v`BLuD2#c9cho}y+'I5v8:jeHa>)]{QҢ4I]#9Y:'#5uT[þ^xf@dW}Pҳ꜊a/,gUF=c=j߱$Ꮤm<D;基FAsL(3޻1tPڤ,p*3l"={sѬi,; fR@=	*ULm284)ѭjeGCRf%GZYڀd	5)9E8Q;j^'造(&]#5ju7\]Vz苒f硱~1#7/:2s	`@3(XC`(V^~@PD"xp㏛ltbug=(v3e0~jI@TD%AcY?od#FdP1(0`ZIdє#9qW:O#9cGsSTj#9GЂDs 8#߶R©,9gϏiP;ЖQ^Dyk˪q`8nLIߧ1y*vN=#9߹_VΦSSݒHMc\B{X~0#5	8s	NIp*H(t80	ݟRǄ<-tA <79(kyc7H5|36#9tXX\\E#74/tmmTZzJqss|_(H*xCs9j3ӓ詌ajjH-te}?sFM93C!bF;f-]B82SUzN!-x3	N+3 #5nؙc,tI|hl}KL؇?f5˺c,ZpJk< Z?,{UV&ٸn;E,L1#7ғ{p'"i#<Z;#l!0M$0ݣB7"W(lPTFA	ƨvߦ<|#71.lцG]G_sBw+bZ 0"{0f$=g=gƹ-3[W,9xrݐ]dZnx0D@PJ[O":(	uEwrµO'W?`ۯ^o'}V!vW)3)w_̍"#53ɞ}"<N#p;	k̹?;ƥX}So{I#7&u/V2<psyG%,qC99|G)贇r?q,oxϿd)¥/lS`63ۦzoam7q0kl]Jdēx%(JŶCe=\bGgs8:m@@D$vѻ1+F`e(07Goa*h7v	<`t,.#7)G3ٻP(#]o2]7hc}R}qNtǍ'ŗ{<\#y1O}SrvROiD|4[oK˴Z/,:,x[@:YȏT(Txqf3<ιӧhלZT$&Gb<#Cus&$e騁fFOXOC0дufPqwSm\&{ήCxȥS z`(K4UB8QY()Sxul(paICRi".L:!<1+1x5ɝZ{pl<OP/)#9r[rۻ(;~\Hגht+wn0o%tೣ#T@#9h*^|b	)Cu%,|h,Mٓ/RJUOJ$bҕwoΝ݅(P?|z%#5k\_?bB>LV9BeIUO~j	,_P"^3qz?Γ֩ϯ,TURt@O:h2"r[7}o/>佞!ׯR8Ҧ1^yH_bP\^j{^#2ʔ{H{XT<ژ6@)WC^Mt͎9Vޒ"q,_«t&Tis&)\rd}:;66/M8/1ջZ>I!#4Ն+dy;T#4&1tovx}?ߑ{D>~VxOy.O60ri.E]11r#xf!V#9|,*y}oT9[QPTRdLnPٔ~+`=>ʊf<?E̟_xpBS~!+7`uX\ƚ\7R"_.Qօ(8܈Pfw(w05@N-(rJdBM5`#) D<s3v")Ć]aHXA2.#7P{l/#`"jK3.sXdbXpH0HԉF#5G3T*B-xGWOXx)\9#5G@ĔhRX0#94nemeW鱡Ζm4/iƟ$ԗF4"`2pv=4S-hMі!mtQKog5%wmSͳdhN`I}ZHqM86VMAAH?e<4)F*K%QDgJLCb#i8{")K	J+1N?NY=DxDUb12ĕFc25$㉷j67Z!Td+_0;AyXXΡU|Ip!4>[cBiWW	bc˃qc#9щEBd[E3MkJd(0,p4UU#L(٬#7O{rix#+n![rlb&nx&,_:"S#5`#7J),"ZbfϮwQA'iƆ^D˗Mz.A&#7Bh&DC:s53#w7P0jM"H(ʉA]V82EBQ|j84J2Q&m:ה}*L#5TKdkθq=/Hy#'@ZB;-@[/k'Y&#5;K#9NүkVrmbƮkk6V+[I^*Z׫";zUsm_:l2%y+(.k7޾yr|#5X h#7-W@Â?Wǂb<RhT1oJKDoo\H&Y<r֯OYϥA_xtkj6fAh4i̊ڇM)vJi|O Y/H3xـ6T4Y!*[5Hى%>z(觾_Hl\pI[iDZ]-֍Bj5;ּ#76m{Ev -yk'V#9<$!*BadڧVKdrjgok+d>	%;d#Dasʔ2w)0ٟ8=Nȍ;vnݱBPҙiBˤA#5Y<|aoH>FTa8('spĊ䯜I}J!:5b-r3P϶*!K"-0 [C$(?f=T4tɢ$U)	Ǘ0uex=ɃCp>Tj@d Q4yoYqGeQ4UwG6I3*q daR7]B3,Na)s+vd0=YV[vl}mcrN=SBR;e=PNPZ5g\EA"B;sm2#7;_O\n:Lzt	644+SBm񃛺I0#쟣cOfgw߁ǕR`]Q&%E1 fx?Ov2xܱMU"J)6*7Mswvܵ;fn>#9F3	|N.ˏ#o(铍`wK#7>I3cr @D&,zm"NY9$nG'9wwۚUkif#9("ۨŅR4[b	zyqƓ:.]	f<Hal'O:]G&ٵՁQU+׻MU)ii9RH\+<^j#7]=I9@}e딇;TZ&E̪8Y*ѣr^VP_zM	ErȚ;;`81RjwkvL>^(ǽVg~yveu4Up>Ｑ'u>*+A]pŝ#7LD0)16`|#9X=̻v!׻7B\QaE4*}Hk]撿`+ުxzæ՟\ŮtHxI#4gSq2GVZ29((#7 Ý]]/%I,hh|ϫ[t^8*:Q̑HD6j6@OC}U83B͉D0(Mry1C$ ^gLo6linD1A<pbR9(8>6d35v$ˉ2t횬/m:L!}vf_6	&Jh~P8y#|ۧ[Hufhk]S5iQr(Tbt9	KtNB4:vACIsq&Uq@V#9ݪ6Le621j.Ҳ"FѼ7ce7FB@#7@6:x&@vdxgnr#seprCR ~T:YiSdgW:Qv8:6 rT:or?Sb#5s<6?`щ*iKl-AxW,5"!O1n6E#7eH3q(8{sN0]5Є!?_]rq&\-AbT5,#9}֦#5(n2-A>5O%䠞c#7u#7~UK.Ty}Ew45tɧ^>'ړBzE'hP$7uv->ERsNA ҭ;BNqkNd}ѐPam7{I6mK$	+x"B{#5/l)FH)#74ц7x7wGobڲ۝trR`0x+!W:4(APQP?+Eؐ!RTqς#5#56C#5#5?=^}l%dSl&aq_g[&i|=\]?x($ES1o0F5̙יi#7v<{o1n𛗮#7~$MS~o~~vMȲs!Ojzwi9dC%	".b~-ܡi(}?c9HrsRz#7xZbkUJdD!E@o=1u]jiPdY:L>1ET;sDOG`ۯ&1Ds[Lц}j	d3eBk!yBz/cʴo`ϑ$^U(}s=nM=H[6/.PU<>~zJ뵌&:~0Nh2};!IK 2%!~۴:>~ꪮ}~X텼_mp˫:0a /aLȸ{OOa4C~))c=TPPîv_VI) ٩d};:8w%Kh"hr"zϧˬ`1y`zvrgk~ed R{^c?"?3+w0PYJ&gmY嶹c R#5#5t @+o:އO'Dô$$,O@Lzގ%X^\:dzO.|q><S#70޺䠔F#9	~J:v{^)[L|_>G-79?VŅ:Q-Έ{q^J#7$ktStgs{7<EV8yM~FVt5C<'frY@-h?Gezwo!,O?:nK^zL*bB5Z2dCD!XϚ*m( $5"\`ize<3r#9qӳb	#5/BZTPuK@9ZM#7?FGDb$+oê9aŖu[D`BWf1R	.ݠV_k¤?^ZaBZjuvdtf"b_Ч(%|._ZnhEfYѠ	sTȻ'la@؃	S99)4.Иݯ?qːs.k8BrZj	jA$P:P43#9ɷG쟝(# Fql_INb`<~iIO#7YcD<l#۱o~?GoD3ԨQ:vgnNt6er]+VDѬ7_1)Zha`	z{OR'c~ {W!msZQ]xg^p]^;ر5(QFpq2|±$uL㕢]Eϙv͠wEc[eu9pPqQ2D;<Q:<(ս\NUW^q(&W4f7섖ɗ/)$y;W^f:+{'~6{`D#:'F0,v[>oU7shhT"~ꔚQd[?(smuHx|Q`"_r^=CUx9OU#Zr9-{G5j,3y;Y[Dbv~J8X#7G\\k|h)@.Ӣ^wiq+ry}˕W)1XW#7,1o$c8&-|kTU7G;6G(Xb/~=vrǄ/⩫/tĎLͪur+9^i!>p'1$a1B#7Sn4L'R}JەyQQdDuއ$tZ<e^l1N| P7gFcm:cQFrbw|CƳQ~Jp(\xVuT=哝d#5t3UDXXx#9d;:]FK^Ĝ'6뱲gm+.T%ګ,rIk(gNڇ!]RVR/X/@kۿ֚W~?r(y#76U'rQA@1#9`]*'q1LP8szuCLH%ZWO殓{ܽQg3ܜhD8}q9≙~>%a>)-/\J"lEfL_]Y]ZD?FuG8m	luXp:UvnyZ3m'3N:rmTA6m=<axhıV<{Eks0	(!|6fUur C1ԠANTd^s,kHǾ-#5Û8);#5'@vWT.zcODovbw''3^,4d}`ďH&db6Dˉ78nd(_j+1("7%)*U%#5gNBdW.br8H;h1V|xͻ?7XR(-.\V;'xfϝY;w;;^qhVd6Yz\j?ǵ7~kMٷLA򺽢N1p_9&mZ껉m_GPY;UCf hӼAf摇qC`vb]a|+;,dc@3`V^Ǹ>\{T̀Cb׎>Fp,M!̝$-#[~C]=PDxV'7C5#9;^5&di~2..cՒSm>62i|~/4^_#9ăo+1p<+E־8:YvxVk{RJJEGqӫ2'ub)_>[ݥXz	#7^5W=4WMJ] Y]g#5Uw{_AlK`t뢄Cx%kzĖ%xQKo{.btU`]'E?o)&+vSsg6x_C}v58q0Vϴo.r˰^.{=#C\`BIp{o=a&"qFtNd4j#9ĶJg̝qK^~Yx#7Ib36WUd~U*|n}igMCHrQ:ʏ[9+|XP3,a5d-%(1tRH[Iu<HރQʥnW8C;cw)gV"L͍.vȐ#5@]Jd#5#9m{2Uo"*1	u&|$Khn,d.-H(=^Y@q{)C(.g+>vƙ3c373-PPѴx̐j	mC|םs/i|T\ /DGF~_hpZ;_"7NӌRųQ0Ԇ`=fFg-'j :Gb0u{oxRsRy@K?F޿k<XGf\}u+*TOT'?Qlڜz4p;jtR>9~Tq3:js#5#5N@Qø~^ɠcP<D#9G@د(Rqh_NM}6n+x	{$~;UϷ}hJHxR9zNcI#5W>pZgQݞ}9@.>~/&f#7k!b(U A$ECLmRJ}ϟwOO^V诉]}K$3Fx3$X_ad>İ%<-og~W$?	int$FCG)Q]{^ױ}lxrL#5~駣vF%r0&X`E:((=pVnWfA9hO7:	^Ee(ZϟTM7z	v&INr$&2(a>jCR\%AC]SP;c\{^֝9ag_<sƐzT)AY`S!}ʾ'|5Fd90P4&x$*al̒l,tgg2jbON@		M~([StM@'+׿wV_?dfhTF}[HY,fJ&%JG4gn]ߔ`᜛3~?8,6Gص-6+Oݘe;g#9"߄#9}.e6i)4V-x09CI!Z9^Aq߳QZr@]fD0*@UHuKYDDgC|"@ K#5vꚦڑ/\U#7>IRp,pvd#7y|.^G-gc"rme*o2A릋 t/QPBwڨ֣хJDJ2\d)8/>Iwo!btX~/O˘aŅ'~ԣK1i91[E{nZ~F	tϞj712[lSjjkbzC 8) GBA`=%iH{Vv&S<wյ+~&0pѢ5pUq|h̧y$B1׶-a:I;H_rvmM[bSN@b:NeUζP@aĻypWՅ.#7kS	t'/'SFR`=>x#hß_½kvAq2K@jZf\#Z2~*-o6uo(. |AAngi{9n8LZ#7)Q+9W{p9kNK6[X5tCQ;@%QO#)j	ft)jԩ"	EL#9#51 #7:op{rokxGn[l-ᰞ,EM6`qX*KqXHsX/K,t)Ī8@Ys)}멑JэL]zaN>֍zTiu	|ؼnuݘLoN6vh)@%XMzɩ(D`kf6Mr&JoY7$2ܔ`>K.#9#9drEsXL|_[O1Ndg) _J2~^ΩO~NU=c2E'PoЏq#9۪~ gݜi$~֢#_#7ː¥1Cg⮚6yb2})hn		rq7w|%qulw=d-#76VSȱj{Vn1P-E\{K+i(s!耘)`>m=K@2	N#S0{Ҿ'ϲKaX	8Rsu1l_G0A#7U.|Eh3Il[wCD UG#7={4#7l#5Lq܎4WiEX9[>?nlq$#58.30^+Xloo3#5xo|B٦+GR#9X'-n]r,UvwuG~#7H\;c~3&(vtg&{OU6<d<]~X#77:읧AK:w_vvT ^%0umqCFnDXAM@r\K2E)li~V[ob/aO4i,F$32Y+隁JrvG`w=SprǍ"tkߨobY$mVq22#73v)w`p`EѶ`vL/|_3GOI0zSy{RI'Wl6m3Xl<Y!>޵+?AL}Nm=KBa6I|a^/I'o#9#9"hc"[KV+QN7΂|#9O.u{S5M*(箈>smG-7G(} G+Ô9%(*6e|?efKՠm!U|2~itT:Q8]Qx#9wUh	x@\y_	v3;eq#5SzW/{; 3	2JN%L`Zb#9#9OUGnŰ%UGG`jzt*+h:d)E9Je'zwڛnvn(RpF[ڈhߝ]$CgfFGAcaÉJvmJĉ>q<Znm#5Aj2%_-pr6+^%aQO^;tQmD|G K]rf`@tFO5Ŵޤ[zn9_ݪhC̐6Ps-ζƃ%0Tɟ;&E%t9·:霂B7@klQ9s-#M@ѿ,m\%2[H Mu^QG>oXiя(p}HcaK#)cSY=؎!_:]eQ4`!/6_E7{LO/h]~P\#9톁}PgKwv"hq;{Mw'Pl6jN5*ڙxl`&X߷]#	Y^סNZFGV7c>stR`{>OIgzs)pF>$7C@xR#7Q6P8r߀bz	N#5#9sssu?	ql(3"޵))gn5|c-e4,80p`_MRӹNM#5<#7/p.VP{߯@tGc0_<#9#56((	#K=Y};qT|6} %X"de3	ѶmBv`}U_a9@$?=mO+uyX#5C>ˬ,mVv8)%vz<V.{u2e?nܟ|-Fg+PXƄDXʆG#5`8;^h }^AfaT*@<03_a #9ԁd#5523CH7?IKe询f3i/#51ebDX=Ptg[cy#77Xkpejh,#5fj9MBm;.lB;B:I DusFU(hiBa-MP1hì^0\}ƵaOR; ?K">$4l/rHd̐rJ}b>ab'Of-P"=OսzG޽c0nbx*oJxUU9]<-F5ovY=Cd_Sɺ #5|6s~߁G]s m*˳-_;gnEKAlO	J1}D?o~ PffA#7  |_9;q;p7kjѐX%.73]#wVɃP3rN$CpZ`!B-C7eHz1T/ʚ,'#5:9wvmZ/ITo?(VRE"k2$06rs\ciJ0@{dsw15mMYŜKthUpű%UH TS+pΛB߰02Ohc#9	+wjMb#7E@hk|;a?#9ד|l6Wug}o VCtEE(*C T2#7"Tzdtszۿ;-0E10ftäڝf46{=tU{#54emҎ<LM͆7q[	:KC"	0fLoߑ2ϳXeld>Q$a']Op	 >I>N9PakYA<rƝA%S.UyЇLfB&x<6'm8.TH	|("pCZcE)c5׎WcDt9UUW,Kyi؏׺sʙ#74yd@ݩ7c#7VCo,'20rV0J1~s'CIx?jx%,I _Ah)XZ#7뺫l<?ɮ?$}Dݻs(۲'m(#gvmHHu卣/ɸi̦(jKڽM4(0D(˚.s~yKX6P#5HEqaj~6QS2!5>@}g};$(i^}mO;HClD#9wP`J+PO,\4Y#7H6Ǆe+?=k.AT2\9fSM&ơLV1bH㨠%0c<$ޢ1	TD)[?ٮI;'Qx\Ƿ) N!#7g^aF$3	pa2P\r!??t@'igah~V8Tђ/]dSs%71-.Xf]dmffȞcX#x,e",܎I$TiRb]jjKK[m<\Ֆf=2]C$TLX	r-dwtQTڈdw~n+vc"U\nPꇫ(^f6i?L&v#7؎+s'pW%@#5W]*``P,Xݕq<SĠm	#5gw>}Z)oԁԴ;v},)z8ގ(у%>A؈[5#7ԝ2'qV1`&I" Ľ6%l|Ѣvݿ:rhY+c#7I=k]w$唔vF!cn8m/xnf<J|O呪)"\Z^2>p2=ةO@dZśΏ9NU!'H*dbg;ݩMrK0$ I*m8|m,Lo[Q~.&"'i"J;~qcpQFŶkoʺS[!Xݱ*)jB!	#7x652:^#5A6#5O($cHFBu#5H|>_AWSZ#9i#7?E/BLɟ'WzE#7[oIhLRѐN(o@cdn<<,L][C.x#7@`#5i]]) BgIE};|#9s S/$#7<S΄V 76"z'?&?9=9I D87}cb aG)ԛ0dD"Ch<?Sko-8~NP#7($ퟡc7(r($n6>0%)@Ϫ|ǐpv`#7A3%j]ME5qdYISu,W'ġOW;6<#5/AyIA/@}=:!_!j?0_HQ>4ҳʈ65OPnx)'OV7or |5=bS!uUG1d>~?_"}=j'\t+k5uJS>\FnT873LBN#9[10@ҿ1Rdn`m@g	v}Wcsc$E+Crv@#9no`=s-C!p.F%PKN/d"T|r 2	v!SjiDcHU1#BS/Ңq֡CJV.g!@UH%HY^|ݶZ3$dwW0o&R-JO<5L0dgQJivObxIv%!~SbgѱF%k#G=B rO2^v5@6#-2#5yf̪^燻x?k2/ㄫwk3Zeͻ4Xªho#7*Vu%ڕ]lc̆_t;V|h聀IOzm|^kl50q"{#9(qX 44&>ҋCKͺ|N?n(Pp;SLNew꽰wU-ߞ~ley;|#5&}}^qQh aB@a2jBAU9=6NP	?n[#xzX/jHy4:F5]˕0ޠoZQqs#9#5I&7#9@P!ER3A,#9ɬdEp$C(A$ E"QTu#9&G~|DPA5#7MK_݃}mh4=/S:ꪔKV#5#9Aх$u_bhDnMjfh|QP̉].Av]O*w%}MB|(3c!uCd"#96%3(ҾWyQZ+U6T& [uמei w#7=W_<`=S=o	AA|.YI =O5$c##5ҡ%R1#B(#7DdW&hwA:x+akA1<O)@uf3ycyו<8!oE@3좒pYվFy@#5ސf0JMVr,Ꝗo#71#7Bd LƏ+겏3݃B{lZ|a!R^"(?>vkx`DZ#9ikFg-#9Lf/J=K$$><>#7ѺfgZAaH6-!I#5f'Ǎ2G<R:#8|2F@Jh0eV t=,f_zpu֏;&9t?;BHB:$?L1zq{9l$xN)W>VɰxD+#0A-J*v2-!_	9gU0O6#5mBXΎ#~c,1eA"Vs	G~VD~ؠX+V0M&Q̷ܜ,1e;	*/v!E+$E \:zWwk՝(Rd!(6aǙ$@3#7sat㷴ӆd#5;L+c~Ang\GY:ճPS:ny'#7@YD.8,8#5	*OXES#50tF!]eEa#,?_6Sw=x8~(1D	oh.#9Ŝ#7'^xiGm 	<GIH݀6"iPRFᒫF{mِ)%{qq,1峸]gP<V	jG|)-YbewE̴a̓>}/l=^jrUZ|(9d7x.U&GY 8{.="&ϥ-z7#5ߠ7qZ`gCX9i&BAFE,1@)HM׮v|!=yvq'Ryf)gA**l7OfRjF`l"2##59#7uA#5xmCE~Xܿs*ʏ;ɠC2@'w59QX(bbnm2OwW#5[&a0>yD{K9Ʃ	g?Ƃ>{U5P4^rAmކwJ>9|ogT2N2EXWֶvƱ|[?Cn5AtAYvVSV݁sV.۩֎X!(3A>X6ͧ{XKtP>pve8؂a*,W9d!y'S. Pu	`k:ͺh#9ϟѸ'uX5X^xJ&W9K]W57Q?$lWµhы!uSn(P#7Zҫb0Y	4L}{E-zKzb{.뷝-(?Y3pJ&kE$?_#9oBO2h.|e͒Z#9V^c;&x1OǉAײ#5В]:tb4bsjayPk4H$OVqxz]-aއ?oZ=aAd8{Szρ4hɭ}ݾ8& ~M6		6*#9W͜	1$RYqmu¿[H0"!$݀ъ6Uj7ՊW758,C 2k8#7椡b[J*T0@Ip!%CGW@g`"#=ZgIM[5-#5uj6g8^OP)_x) k?">R6l VJ#0:~BM6C	p{nxpj85ۨ➲P$3ȨeJQZ#7#Sꇱ5rbLZ&aBߢ#qD"\.*X$YoHr(|Cظ"u[G#5q=?y˯g]yAZ%"!HF#73*]xD,8"	^YslgG<!<P؇ңv)rW!OQ`e+¹̕(#9u3yκf['yZa(j*}N:XX &`n'EqPbx+ ʠEvڟ},3C3C$+Z\B4*"b	#'77DL\XQ.g3]]rf?0ĞZ-	+nD~Ijpp	NqR*7wbɇ#5#5+,6/=Ox*#7r};~ft9qX/imvND\&gBj,n2CƬ%["ʙ~=cE~1j:7䄜Kr %@+hl'm`Z,'Cksp[0#9tr7F33DċHs=1uik?N|x_ݞ#9IB Gpdw#9ܗxk'@"!@5qӠjd22BSbH|ˈ~_{M7[7ҳ ~T\Ǩ9&WkƩt2Hw8y;0|E:o#5s[?17D$˗t{ב6ɾIwt#9&2pAϹ#Cmxjt3F]aCNMW#9bDMoP7S`Js(.CɖPv#9i/կ!TʏVƃHCW6G"nUw'.#<vÝpQUB9@ްѽXw,%xa7aHGSiCх!t'nW[dsv`Njp|yEj4y	&H$QL|@.H+Q0)?{B<8nov8#9&˹̡U=b7dA0\9	f6F#5G5@#5|?T]puʆL7rL7oF6n/CUa+5v8YxC}.u}C8o{#.0(totOr>7x&﨧#-5k(~wEM`uL:k*2c.l#7/L:ί7?ӾRW_Ӑxb^_vhT16:ǹ[bbBn-*wbl3VU1lZcM;;mۘ~u$׫dAˢ֐rfg)9jz>Gs#7\P%XtYQBܳ!A@M5-TCa'vp.QkKu_2ɰ6Aێ1I]'w1h]ݶXm|P䉻@gBդ,[7O]4F&@C'Jy={5{t	F#7О"? "|g||򟷸OVOd`B/W~y=z9Rdb-/H{7*v+j.>2q0\pި:ȾV$Qꤻt/M%ad46_\=>QfaNmS˜9;N<jLSEXEQrPq3E\T#56Q*sr:W<u6oc#	gy#-v!lwBr((j࣡~[璁bb1O/7FGoBmFk-U(1}U'|I|k3duX@:,鬌.qT2O^i@)RǧHr?yӠlj}H#QzbPaXaK[ON~t֥psXRp(#yd^a<^2fC 6CU{?klo;0Wdۄ(얂<LS3Aw'C}c酈Ww%nЪޠ0R,i))KU={5e[	e`yGHn4$dH?<\#5u`Z$t7.ۻw"I(]oƏ~K3Geaz},{ƛz(#9gN[RgBng_捘~']7޳FKeq(zYh|K+]=Y?#7'AM/?l#bjdf2XgC#nm	q,'@q#aF&.ym&٤$Hw"5G3:ثks6`pd^뺯#5|!!G)@u`	7q#4|R}I 9fQ#5wr2y8xe6^ʐt.IzD02[36;6fAҩZU5񝙰|7N0ό|Y<[t'?c۶8`M0#;.>1V!s!gcuJ)Ie4=w|5	,Mq5k!MYٔުZx!R#9p%4Li4#H Dj`8rl#52Ѣ f"%^{9<4yQ|^FI~þ|6&z3O(*Z	-7Jvxy7$)ٟo+Y;/ܟCwgY۸A<{Z-"a=1)UBhjAJdDA ఠC.ъ?8IB1h+rI2/+?Gfh6o{/+>q$Hډ(?RUl+CDb-]vy`H!CgP6#Xf˿uMWUNѹ<ezÅ֗6cbtr#(kaq	Pv+mK{g9.$<32% 8S=OˠZ43`Ckf<VAmM"ӒDa4T~¥I6q=^{w[w{/zY|ļfk%0c`;wmZlo#9[~{<v#Iy\uu'~l8gdeaUkSjtC1MWYِBp5N8̬ڽ^MɧSW#9`S|6H_``zR\	]!߲="} G((aMַ'yxo5nh.'$nfx2oA.Ad3w49n=!9N(CI添o;AqԦ0ѐ*T$o&Q7#7R#GoTޮfPAn|?NgQM̦v,d^8($Q	ʤ5ۦ9_W߷i<P0dvS1bי\рRTHa2}htX^|(jaEz,		ag5qeKL	,8GPF?-#t.:c]>np׈_-#3iŲY;zڨB7^ʹ홌yUM#7f(%hwɏK%s/rcXMa!oxYXj60҃$9gv1N!lzgo^n^8G{KCdO&(soVhuGsq-Dȸl`rB`A"p:c^\$D_Dv#9@6(#7&ҎX,;E)SrĴu<[ۡ F؈}D%tx3c&L?#%rɁ͍-'Რt	qs~mb$R@8iZ_1ǅYnt`Ɨ_O:E%E	I$U@섹:#7A$C~6kMqKaʣR+q oTsrR|rRSAi"4Ui:yM:Wc-fSlUeǙ2ffX12%Xgv׮}]o}>i*=66Ki=ێL<`~783;¨{J6#7jB.@mvj猟fO8T:ppknfz\ &plĹݘAr#JTXJ<µۮd45o[f4cKo;(gM&s4Bڨ,LXBdj\Bř$%c[6q#v;#7gT؊L;:zzY43hZۀy13iC߹hߡ^brԄj^ui(;c79UCȚ$lTK k@(9n.5XÉ+u֊h+)DCvcoDܘ>#%|gpHkp7K%	n#7m)	#'i0#7ղϨpFaPptֈx\6c	8J(#52fdqD{Q+`#5A#7t3Xa5WeF(aBwa{q8cMVGc#9mM_S2nd2oK/#5 a%0:ֺ.yr^0f3J#7Pos򥢩h=vv@k!}7Lo~SJ{͂]t($}C!Rb@hf;!uy^%.Qroޚz␅"o7[#9 GVJJJ:9E0I#788q[&ᨧCfaMj:,dG,)*&cHw-Z֯`3GTaQ:6#9n6';!ьS5Q8y#5>!۳01#-,6KDRoS2'%x;%tM{3zz*:#5Ĺ#7o'@z;s**.)zʒs՚>##7$gHcˮYOw+=h`-|Xq7CTMgxN~sQ1T!4Ċ	=M*$	n*,P>8ټY]dfz#9)pPeMAL05G3fåuo22zOa,6_K,:窥	se^C#9^>PaHӢR?b2,2	]g,5Q/KCӻ{ƚІ:#73ǻ07+IC`E;4l.0Z"[6BwzlOT(eu"SPXi%#5yyMCWt||/s_pn}#78	0 F儁6~?ju=ԠH,b$e T#5%~=,2<:b/)OXJBdd֚D{wt4e) CMI(#5("@Ճnx E#9#H#9uJUQXfwvq$;`%HHIK]BxR ݪ5FGҁ2z)q߁F0zI$!Yz;#9=#9I#{Q\r!&m5g"ص9).S_TE[R:)ܡ9#5wR!0! >GeE\O}R2K>疸tI.18ZItDi&H%ʛ2@"#7}~&xƏ~6>9ACf	:r6DN$W_]D Ah@XQBLS!Q:@zk7W)m^lE;:Hn\xΊ(Y#78T`,@uL^@ ecj墩i,jI_oqG /H"H Ċ##5 v<b#5{7].Q(0ijIKVlUF@00i1bò%/,MOiIL^+P1H&#7%Z9QwvKˮ\x.WLs\ݷMewhM1Z5RgCoWYYI#7l0A=.D,!c"#9ǰ=s'2vr2s22Q圏c_>A<i[ci;>Gn夢K9#5R;jy:]6¥Q0$Meo4!HgREFDd F1R318Pg8@tݪ	A/-ȼ3?i#7.#7=\/%C\&̆Mup$ hDssM,).6ɂX@ąn٥vx.S4Mz=o#9t^Aԃzx,657{U`7^	%.uuǵ#7nh8/xw;lyu<5mFOVHV>bwq%Y&Aי	gJ0Z(Q*{">IDa*T4g:'QEzh5RrN(HZ#5{";ABȨ/yk*JgT<IP0(L(^klmRPOTAlMAհ<3|%HF~IHCYoZE?(c{F$FDBD*J[$-cI6hfbiji*&TiLVDQOId>SRq#7')FDE؁aON##5Pw(~Zc\2ّg+DՁ6%̅0q5Q#7/Cy8	v,\	X<S<kD~lR*De`&JviY%%LbOyy&@b*:m[.Aꈊ.ͪ(f{^Iw8yc 2b'ʄ#9oa<xj&R0@UHr!8rQ#@R #5l[QK5cU͹[\l)i-Rkb΄"H~|%bf@#95#72#9$ #ڢE*ag0>~ʑڪ\n7|^:<"?SRAВE#Hp{O>}_ 'ByÆA$#5rv_dcS9mZM!Δ3cL#9 P{4>Þ1fnt)*;F@#5_a	4d?ig/ύ󽐐3L淥2%\MS#76]]ˣfx(;>ʯE][49Oqꇄ*l4.R^h!vA%O'%L1@~]@Td!ѤtT?y?u%u>çP<|!(2jµu?nE0P#ZOowy;ݚ(w[#9>l?,QW(:I]ow|U	5n~G]A<;c-A?W<z4r-uWH3!`o.UtdTTt!:H;#7@|-@4B|(lSiH~P~&Wxzbr!m<hCM$*I3⿮nu]dB`Dcf6 V#9#5kDFDpOz'eR𱊂qߠ%$̳ |JQKAa]JʡPyCJr4w#9!X"\fPҐ)fJ5)izހD`)BwRlQ7&B1X M}˗Pr^.ҐؒAff0_:ñ'#7;1#9<lwU#5s]$8#5	?I(#7<봲#5[`6ɻNbŐ4&ndrmEhV=g@ ^1>DE5sȱ2krϜoYmiQ0-t}(ݧD3'$tjwv<(mYe٨m.eUܾt	RIAG,()qrE-6@A[ί&ڪ#7rt|7w;rp0[wϺ-NbqV#9KvuQ\"a0ѬAdEMy[Tk/=0,Ln7N8]O(!5@o}''	nTBq4a:w(	}(DP˫|68Bϙ^!#5@#O4X=`tb6k醙p\Fvf8e#7}:Ph1EctJӞY@5k).A_#7[#7͠o.k!t݁5tdk2-q4#7!5qi/͹Bcg8 ǧs]8F)5U*X 23H}7]VD`viYTP'ݹCԊzorg" &=#5=HtZ1лȘ^GY:a>az`ޏ0avxO.Ț<R$"6AYHƕUb	9:o4{榤mّmD"w۵W_T.}{4#ٴB#"[ZP NZԠZq)Q$UU֚ZVnbFlK&l\l40D_:F)$L+#4QV8hd!í$1(e÷LC@o!D16Đ	"10$DIcX{@N`sdI\ւY3m"4Jti&i/ɬ>u}뻾:yi;(*Q> d:?^tWTz?#9;3$tPXS$'mV 4(ZGu\mEBNS9s*q#90Ewdmw'J)	ˁ#7A$f9_#9<pyl <P^1#7|ȳ#k{w=j3QB$J#NF#5U@#7)B!ck#7KBC΀>JKkQ(rQK־{.uIEF{8#7䐘ΰEXp}K?{uUh/;yB:gR{Jc=&XZcGnkY>)%&@=]7/L-n음1.X1DdxrNItҁ(i$6MLTLoѡeZl~^~4BX:f^PĨX^B}W㙻!mbPuaH;B$Y虪<S9NytDd֚4Y7!rY(n^tkn/>t#98#7PR'm,3{pwܹ8)ˀ7'q%#myUN6?]pYԦmt@1TI&>AQҸD 5Ӳ$[.x!מQ1Z1 6A7Ix8	gypK5Qz 55%O#9AHWó&{K8K(  D5+!E54Syv1bD:0#9!\L440H\Y48OH.BqnX#9ZK$6Pq,ޭku#IhSemBZ(5`R]?8z}ǔ:[Hi*z^ʐ.Al<R#7RQ#*]X<0`?		œ3	X!@X7v@Cs$q%uG&9)E9\KX6UuSibi0fh:)r؁:@H#57b\^K.#Q.ĩ[?w[<spars(HU#5;=]@1P=Nևhۙc=^4;F#57*UUzKri}qyg#7oR{tؙ.[{*)P|[ZעqX4X.Ǌ7.U1 &XQL"B-k\<#LI#9jy0.2pP"Gsng3$([X6`QmҐZFv$#lh>aDO+L{io&qxh#7Mty":cITHCIA"0CvʐTKz8138LeM<9&0!#N\*m#9.2h#7PBu9;;'KڲUJ~usZM#m#9b^nNǭ2iգO$~ၣ%iM#,JXdj#7Rgpp`uumLְ?^kpQ?j  ᭱i,RҤ:|שKEFXٰƑM!?M~38]O65!otW[0hCoHXn#7ԯՅtQɀJh`i]CKw\)Pמ{Zϗmm!]zRDśTK pdT"A/Ujl7?is;Y7p׈h"ٵ|[tاw%sS[.|@Cbmi^*噒lK$)TQՆNJ5#7	݆Nma6he]hiQQ#7h-.\Ya\x}32VpC$OPІȅ2K3IU-#7h`hɣ@Qf[]q!ό_8S&drD$&#6%Y}{\](9z3ѣP{s?"Fsu\*&m%]d<?hK*s:P|<׎ף	4\")P6D^&! ;|uX#56w875zk"#9psY\a79ݎC<tjdIʬbJ0!ЙuۍӫYd7 bP@k>-4+.b$馬	apχ|8lJ?2sZ9ߎڥ2ѩl;"iR('}+%{`!tc!(rs"ow#7(7)s`!-m:ɒ>YhkbhA$Bқ#7	zemq#5ń==,XAfU@rcGY$``B#c~F*{[9@\s&%Bb	@ d9pO`Pj#9׉qz5zv#9gQ2f쿌ȡ&ּsԃ?hQɲg"ݽM;sJn-iS^r?e)iˊŹgB( suBGw='!UwTZ\XVUTj嵶-m#9d#9 t>Zp|D#5K*cc#9fw3#5,<QŬ~ֺms$4"	L)jIE)4TZ~gRfTM*PM2%#9+M2ZJ2[,̘dȣB*,M(?uE#9K05A2hZTQD)Qc+LSljBe%2)QiVF4Yܓ7{9	#7-K*ҙH#X&A,U_cBݘ̈~Y60`.bajvNX17:qp)5cz3g#7N#yQ@gG͛lm半.F{7	ć!kEt5I]ٲ.k5؂Boᓆ~8zē}X-nwmsaR*Bkk_Uhjba(Y+	*g^#3_5af;;~g'*s0;ZiJЗ?lnCiF:-oĀCO#7}Wސyj9jw $gac1璆l)ɥgYŝ7eL4T@"&IM7ڼ"E~H*:^.f!E}U^;29m`b8_-Ŋd&eE$nF/{; ͞)ZVw(Ġvʑ6wa}lsqA^M'K;ɇ5CAؒk豣Uk| LGGM;rrߩ8w+(p:h{KC&k%a5p&)GN[JVlPuDp9#rZ+D#dF1dXx<lB/GQ3S֭F2\.[fFEts#9~Ldk<nn M) !;~dU^XN_Ѣ{A}5u=LKzp#5j+ߺ5GDQ	#5D!~3&G޷䅙=X#9)yM|6#3a1Gp=I8,#ıBTsur1#7iVyH>B#I72]vӢ+H$,"@C[+P>י|U1ue6fSly>ֈ5VePm B#u#7fciż}7&pe9zikmsƹ͋RZ3Fj\[k"-d VRNb<V3'򚩗'`2Ba+;5/0k#5V%3bxq,+SV4IVCCW{|Sg[sfjɪqƜqtx3ahwk3sd,5-alI9+FiBw>yj\6Wpi>yIOC|!fwmل:@7xN8AgTYshKF$#,0hdFҥNrhM\:&NN}a9%hF8pZl(8smmEjxDO(f#9WtU؄>u\tQ01v_fSMxg`'fTdGco&舁鳗<a̲Dﾬo|XgBӺ6Osa#M[Ή!?kI:Oxd٭Fs;^G#3ׇumZ#it7k;xV|stzVJ{v>1xXbP~*cg =*JBdtƈHyǇn5(|QA͚%)L݂RTZ4WHCْ:\#9l6?jVdͶ]{u2RVn"u[Y2HB,bMdW*w$౨wbmi.o;sk/H8i&:F=YE`c2&M1/N[$Y6i!xv!pusRrXJkMpɚ#7fP[UjtSi8gms\ZkhifZ8 ޸+"}3f;83RB#O<SS)*$|X&MxW5)ُzEk1^)PI(B2lZgW糥f<LEۮ.!\2tXEDaGFoX>)=	˧?,	!#7m7kRj~0x9VJOU!,"#7[uofnofCmObS#9D3(d2S;Ӛc+[tyÆm̆'ln0<|7]\q3·#^0T?8mFTs=GJN3/UswSY:K;ܨ7<c2.9F9+mH6-(>5#7Y%4n6̾Pbd#7qrI8tZx9I2j6gZ䦶7V2f<uVc1ܷxlLRkT/t;YV˚grWAֻ,~ZͰ#N=8gNap喪LL@9ik&	.Z(t]A#7HSp2	7KԀiR`hE8(jf+H8£z,9%+kfjWԱ/J+o뜭i𷥙^!ڬ1z;ӳmU2#7Fl@w#5 8{{Q6'M24#7#7NB[IIV:āܮM\oQ(QxqhAL#70İkI(RQ}62#8c&lpagvs*Z|D4 GFKVKRR,gVeOQGF5ᢴ#|ovADhs&L5٬fa5djfe24ҐJ5̍|!Y7Efi4IBC"ti$C[!ɅFҘoٞ1qhg#7TaՖʪ`hF6<:sOJΕbx]	Nq짡iᣳXvpPA(3323jAB"#9f8Pq|n0@?(x}xnV(K47U3}/lpxVAlL((RspMN{iʞgPiɆ<,45Uhkc7hFa5D2{X*DL S8	*'{W{=lg~4e1#7*v2S#9Q4.¹&HSzS#7b+`aQb0qҘk5.2#2h0p#7T;@L16m.Yo[.YцqMٖR'#92R+PVJ+m:ir,rYt5#79rhDejJ*e)ҬLJj1Yj&>HurV"F8\3qr`(p7v,92g#Иv .#Gi@"@1#7fSǫ$.&C19ќd͡<H Kza@sdhقɟPBI#PaL#9f(SbKط.34MѵʎI-rv{MK\Eh12ia(1*3ᢂpXn)R"!Q)$aEWJA?aHA$RE50w1T؃+G)Pb5{rQBCP}ov*%_6H`3 *r#ke9ɼY"e[࣪B22Hba|pF^Țwe-ghDG19ݴ}}O}ns3x,X4\k5xɻmjpm*{"ә#9!!ڗ_\5~vݐ4ٻ5.i"2M L#1ڹ:eܡCј>7#5m-<%!	gD`u{,#5J4^-2L͵dJJڥHXP(l	u@VL)N<ʐ:Q5	pT"Q1bo#7jbx!ӏx@#5#7=):\˫nkuݙsuv%jݑ|`Tz w;,"rGd:#5.h7#5(C.'9/p:d8#9#5u(E=?]9gu~Iw?dGkl>#GlL3Oi2.]H,J@Vm$6x	e_[cbȑ4Ġ`oQU)Q8YΑEZ*Eȸ&(<b9E%CX}L?aOߗ.\yp(KPl=NȊa%&-kDbafZ-F Țf(%(MjH=kSw)bˣ퀦QH*T`jI@#9e40Xh8RLuPjdZd<"rLU"6`gaҠryʮ3kȈ\Y2pb(`1XF'$/erKooRЊVEٔ#7hqipō#q+Bkk傡,)TF0FEAĉY#5#m&3]:˓*]2v桡P	frLCc]v$#7#7A#9R#ֈVm#e#70d@2zaCyZSPcZD!@XD&&-G\Ս,-<"v%7(IDYR!f!R,.OZ!PKe$$D39$CK'o̓n$mdl6#KV#7##9kcȸr3)M9"H#9#7VcY!U,DB!8'#78A+Xe#7l5554CT_ϸI8CBTL2!Kbgʅ2BHbPͺ7Eb2A}9n}ehNidg)x"0E	la8PѦN^VJSr@#9҄RH,ڔŮo(č('<;XB&[M"30P{!վA(ĸ2\cq02N#9#UD@J&FcS!0:! * 8o`Ãc-@jf@`+@8@B;:>%P6|ɟM8Qb{(=Gt zXOە#7QC%iB|Fדzd!+Z&#5Dd=XH`!$1EZjت+d5m4B!!|$>z5`s䋇-}#9cO_aZKޭmxtVWWrr$E#7y7ODS/&֖<_\9gMt'Yh֣Etb L^f2+#9@ns.fedbIir](zPUQ/~J4[kpNr).8<ð&4֧[OkqT)\MLO#7ԧ}(Qap]W|/IvMK#;mSTFdHt*{fI3SZ;<TY,@en-<7Ծ[&mC`{6Ąņr1VJE#5 Z`F#98h[VQm1KJJC#9\KB0u@hX=<B#7hT4I$Y,#9R#511]ĩ[D%Jv4&pаhbR(~s[v;L7&lY cVeH!	dy/X,sn]W[AȢcX1HqJ"x6+ڹ"B!#9e?ᙊ#Sh٩n6񱔭3lM"kkY5mSkZbQU%F4dS	~d,wId(u(A	gBlv.hr¨\A mvQ|,aK25|#7>샻Zt9QA}>H 4;{u祍]$$̕5,aZmO;H`&oMTq+*F-BT\Z]b5U(+v<;Ȉ`~o'2P#7z)P;3`,(\	kS)@R@T58#7;HpÁV,t盬d|;/:i(txRNGnX%`6g	#7`ډ4}p䋓MMeq27Dq nr!@9\Ƅ!b!#9&ZOXobM[P>D" "W3}}F#fM02~1.FODAV=mV*Qm$^.jw"fImȳ	hLVv4kJlcR"eR41MR.]&7f61&Ejm'8imew]5Nf]q1LմmJm-b#7T_31^ldQU%DSV7ZV4TS[yxSbVm6QT«-{:Skţ-W^w$mLiM#76WXbm$jIY(F~vӖVjco	dVv07?,2KF]%#7ieGvldFydD>)uQd=efʴb!4MP!R*BΨ	0(OoUZB#[45mWJ#9/7kkב DY+u(XgIa{JLT j,m-1Dԕ1-IF&J٦md4J+r-Efؖ%M)!Eңe)MIeF[e2%cdkF)EJT+cKQIjCH(m4#9RbJL4dfhDRͩ2HԳY2hҒjR"%P@`$SY#7%Df*laP(2-Ymbѵ)#7j5XE@BAaZ6>>LI\+jm#qJߺBT>ZG	`#7MpQ5>PWHcTS&#9DG]KTIw~&;#5<0O8n:!`f]X|yqM[o6vh3/UTߙJ7i(y9TvuǿKph7}3el!"`EF$P,ڶતb0MpRUA_g"/ת9a@D,()qȍc>rbE1S!ކ%J(>O&ԺVV6=.yg Сu6_eys+@#0˟Bq1&p*/y*\wVI=Gmd0ɭai|MTǃ^qڔtb5J5d|N9M]k1QMu`^dB/i//mpZH yw#5l[cVWښ=;mVM}#9ll\`֓^Xlo	7nE`Dj' EC9(D'ٴ1V·f75:!0ܠp.^ot^ǕtKYuD0xxD3xYMm'cʻob)6J|wmLEBBصk[C+) #C/1Q4kLP5FE,aǬcX)b,h#7uRnLWV6pPhlv(lL1PP1bqJ#9[DEJt2`jH¦?0T#5l/N򂎧oFǀHCᦎUQ&YV6#7*2,:④T`S!"#9oPeĦ>~f{x¹@#7T94ޭİh6;XAX~^Mt"CЀNu-6u飅%Lm&Dq>`q?#$)xIԱP48bd!e!3-ҍ>1ZgHRfe&ȅ8i"+*e@`(od뇿q; r"{6yt+m}Y5*k#7)2 gFBSNLzA]d6MjuAdX%3ЧK&%4z0ӟ+С!M3k^%U`BJPL1X#94ɕh69;!vgI>\UtҠ;<шUr-1EA1yLz{{\EƌVB>]0ڡD٭VK6Ҵ<=pD6q5s]a~0D#5!*8&&(j2\(Fxך=|r48zar#iʙԻa< xIQ0de+g)-'T8̈[#9Y`{!GP#5d6|CxWrCk";ihITؚ:jrvaTݴ5UAHs`%L1IÒWҊU;ȞYmdkT{dfT#`p7	#9VBĈD;rU^!CvˌpIti1#5<ܱLi*Xca"Dwjh6"*pAAX[uvuew|lC̑6sd	jI23+87پ`UQTUT5bO^U&K&X(#7qQ2I"mҼkL"8\yf<?#zθ]DDA)$ #7d!zH2&53Bve2脨WudCP|#5[WG!.;ZzJې,Jg&m)B-XKԪ,wA,7D*yMApM;u	UF=Waԕب'M;1نQYON-9`ҐI#90~29;%(vl!kT"<*d"֗jr.7dH	޼VZS3uU]xzFUW'.◎v*Y{"nT5eQ*j #7&0&B	(=LS!pAydHuOwK5N滨1ⒷxdldRPt(M0׭՚mUR@T*2![ pYeڨ2MJ殇&4EjmEWz U!X%#70P\2&jhY|1ek3-PJ@#i|1,X~1HJ۹e"FoEVt0C%RfLBñ6{78#5B>̲P6i<dE:ե(shK IχSUTQP#56z)L,Eb8}tb5Y*6j4Ƣ6|~_߈\@l,Z5GaH~#]б[Tz@ 1o;	ssq\1,Z"EYl@u,m-gd2O~oW#5lLl]N~1PMBa|UwfE	gPuԙI5q! BDG⫘"|t	,416iWq6?1#5A#ID%ᘆ녖%#7#7;ή'VkΧ\<#9i=8-\ۅP_	zyè'8H3@qt, z#5wp/]@k_kktA *"#ZzZܭƲض=5yqCŘ( bthV-ԶihSl1{щDu'QX~<tOy$=GPpwXH;$AHZ}̖-%"iy$;>Rg#5X!`≫r'p('v~;|<$H󬵺ZW_=aa{1tcҟ0"uH/rIpsq,"R}ah?cT4aB?4Q2H;EZeB|1)4O_^|d֦]i2$Y+"g#5iE[cCwDc2Yԙ%vicB}D6#LLtcHCp3XX(ґN&1en#9S#5ՅmR,ܒfJb1.fh&wHMRRJTe"&XضD!0$	ZMF`L%YloeR*X,+ET=e	6!66Er(Ě6*-ܷ5IƇ]m`$(Τƭ;D0&tgb.i4%ho7<!^Xch6!Xt0 8P85VMh	L,q31N#7ρh r# clKm$08I8	=>zX,,ۻvktkzy@Jc)LD袩IP݄b_0ԏN\Oxxfϐ#5<Ԉu	;g\!vH; #7|ԶFlj[Uuj1JLt#7EUu+أUsuKM6BT#p@"H"HdᬜXpPAHX$ ?q^e~D<%:w6Sck3\2Hfz."W"K#9I/B$dE]D*uԚyu[<{GLhQQ	@UM^-Dg$AX5]LerKmy-jT5wj+zj"^ݛWmh3%$͵jI{{^#5*${aKRwLyNVhYs#5O`4#52#	L,eqabHQUW#jTbX&1#9K`Zءrkeܰ)"XOr:@7#L8`1a#5@6#9T7Lx"11Tzߣ{5UAr	r`s"3>1*_};b=j	T6pD;B#7H%ή۸)iJUcjH׼՚m++)"[vWߤm0aþVv@7(PHw~BF͗#ƊGzAIV?l@?p#.lDаh60Y.Kw4sYEX!!6̏#9I11(djL1@*4dl6H}%.)DU#7{<S;+I$PBMH,7IMg|ωs6EZ`Y;I~ (pSSDfԠF#7!QH@f{΢zR6sLG(3$Ǵ5ղ8I'XF!qhh^#9ΆP)!X#5أf!("(@*g{)FUJ>g2DN(k!"GX߮5F`S|yMFrFub)S/i^YeR}P'Ҟb!}ʸhd#QnKF--ՔFԔd31RTkE#RWDh̩~w5cVHF@қ@fQTı֒Y-e5S@V"&VhjFl[hF0)OKIv[#7I98u8~Ģ	TjصjeyF~5o6=oU]iJ5BF(P lvv_vm1{frYT{K.d3#:0B|weQ״hB!߀{ J+!^0>QA*EŹKږ#5ք-w .@՝\!,ijClnIrz$t0EZ.WKa33 FIP#9>x_@x;gbD!G;kHE%I`ACεrw0vWaa1p0kAB#9PAA0?{ yR0+#7$2}n5,X*x<?Oiͳ؇}:uZAU|8boEy2ڍҷq8(G%z$`c8H(E/ׯ9vyI!=ӆMS:!!>fu!I5HP8*Y%	l`Ƞ`|#70u?S:p^YXTcc_yu:Nkoet:f$ >#NJBA0؋pHaS$ O>Q#7bRrԁ"qU*Ld1$^$c$#7Q`էNǾQ2gHe#9Cd (c#9dR#5a%(ύ?kM׋NS$L)}q@A5*ZSTk\;]sԽRX	S&ηƆ&Q$X4GPњ#5y2vnTQ"Q66@Qi@SCr4A Ċbx5mdvI妤Uv{{]J#5:"xTO<;y\EG1x	"wQ# b#7@Ъe1twf]#7o捸7!	d9ј@@0gN9rv\aI&I3l0lՀTEFD`S#9E)l>}'CC:27TFP,KHA@RE4A)\fsyqM>jm!6o\""|H(ueݴr˳#5)vs2jQ1H5?IdGCtdN4i9"Y)P50LK@.FVO(EEü<=S*Y\ŋZOSI'ُq)!af#uSMaK[C!$gN8foj1Pet\XIlˑm1&g=1-2D1b;nP#7t'gJnQ7F̕o^fo닓o8V$OlΘ^um	kQII(}BU\(0g3m1#	l|kEFAD͜%N] AU;bdEJE-SKo6jԹa*@a6T) Tٿ:}hv_".c#9L}{um:HsLz]g8T9؄rо-ܲ+-HK@ȆppP:!N&t(:!=.&y'S@bmvz3Ȗd׊Ιםk٭V5YfF˵י;	EuFblf֐wE#7'#$`?W?ѥ0#9!Gf[H)01iF֤&{_ô*ۻ;Jf5qc+OF&0יxFͼv|F#9ǦkRM@MƆXQl`B"ZKɶ!01^=(MoxihUa4YB22.\mt&#91r#7HF̛#75_Zhl04{xł!.B4Z^c2lb0ھI?Ngd=Q6#9Z(	D2<xreP-qKEn,]jUNF'aDL%tvt@]	Ju0"ko+4Tk%lLlZJ-.+Oxs\-B}s;1Xs+:TEUeՐ{檏p8a ,>QK>Gc#7܉PYaS뉶ꙭ|9v#ZAYCեJwH^MVnVirm&Ƚ)#>gF}{j2DnELQ!#7)07N&Evg#921۫h&UFA	oDah ,/mG#5b)#5h<0Q(	ٜq<p9ܖ.G#5B9C3dLqkD6۾e=/0cW.9;AҴDmxGV1߶៺JO0aMbzMևf˱_99 6Yb˫JD:JO*>jߧJX/s}cd٬Mao=~ }H66X#n^A2מ#7uoAz5wq$jdUL$Dd/Kjh#9HK%4N_>yF۩T#7([=W0%[)"(l}XXFL(DTX	#7Hp*!D	DOap2!C@5t(̿'xߑcj=(C  OM"r]Fa	*dއSޟ#5qn];L8#8%9q;Fѿn86#9'QMAp\!%؋(O:d j8;֬Im=wJ8SmN윚*SP,YFIUY<mks]SKiMm;v6֫I7VG2`J,)SVVyZ_k=ObJV5L-_*$02aOkvSR|nߓcP)'-ME Nb_&tVvѕ֢FpP0<Vv1"BPc|gL16f&7Ch#7{h2Q=#7oWaX*ARy6=HXĻVT|9.`acuЯ:_@ƛb V0Mbd`1>qH.HKfo_Z?,NeFͨ6e,0S{i-R\WVe#7%!&?IV5t=jW6h6#5ԂؖC<];`RÖ$u/3`E\~+#xFP5tHLA@}$.غ] 1pS~}g+*&=q#7xqГMp67͚#-PioBf#5@Ա6l^?sFw/N}1L\2LG8tfe<jp#7cs8萀MUGمg:g	Hb~#(pVZ<w]GcHgMazt4t8siuC^GBy1bÔWhj/Kjw;5WmQyPJ/-UCy Y[Ӷ{#5OHF#5y2)DNLU#5]z8r2KskuQx-XFcmFѨ4m֍XչskWXsmc[e5}sJ.EKw䙘(Qp8#9T=Mcmoۃ7rhY,Nʷ(`ўz}g7i&v<rKE1#5םv#9ڛ#7	  cLi12`ÚRg2~~j5}	lхJm= @)H(5A7S4FMSMDea}&]h$<NE!`!*2@.>Nl'	3%J#7jvD)Njg&Kju$*&%`5_J'xIx(S<J5Q5E+BKC$iPcb(@@EhVH>ypUY :F>1"-TMZ;}}{}鱎k&BE\"#9#7YW8|v)xM#9>g쩒YŇ+jcaj.ܭVǤҼEʨKL]ћ^%A\#5@UQ-TLsOB꠳R@_#5,LG#7!b9g3Q=G./#mue]Pkke4HS	I(R2>/+#86=(!"Lo5`BTa#96 -n봤kLfR4&u`\xtTf7$,ЩX4{NNҔLZ'TZ2#A HDZiH4!k|\BY=?WX.PJBMf[u\jndYm*I"iԊ#9A#98+#5+b(ibhțcBZ[TTղ,0MSU)icc#9͉,ѴLMKv4\n4DՊ0nbI5tc"1RM%qE3x`if+٫V#7#DP`7]՘6ӲaDMƆG	uV'+hi<'M"MѦ*Ul|YE#9`-A!zb|RD)NlKU󅲬ЌJDrX+27``mcVU^+"+#96Ǆm"Xf6ZIRAD-eStAdtf;h1oߗ87!2.48j+)Z(LMb0s~;[Z֙f:x{\<X;%jQPr3b4=ZߵhLbxq>V%#7U7#7-i\6ƖٍM,JX>"_QBH.>ãSp@ɏ|S01~ƚ˝xhIe~'#7.y<j@Ag,+#9ΘäVg@a##53jJf]q޲CBL;g!\"ѳk#7*\"9lT|/@`0^/S^,ĦPZ17ia&0z9En22ݱ7(i44eP0Lh((#9EDE$BE#50c)IpD`ʌ~W=WCw^'g7#o?Q:Oc&MkXկ|_i+~/s<ʨ]sx﬎Īq\Wy\<ޱۏ3]2*̗d=}JΌJQI!s36ټjgnC8!|G3WXXA1;.\-3ږ}gOq!7稺(4QX0x|_A4ɅrtDɏ.8<ǔ4?oFpu䃍"z6PgeC#7QDѢ#mɬwR;Q'?G8=Kpۍ)vgW`)L<.#5/D[UV5m.З@YH"CBTɈAAG!$"m!㥍mm#7mj:,mcW;KsJwndN]hi6+"1R\nE5MR-UkJ-hȌOQ#5/e7XQQ8ԗs:VFP,PJ쪥o<jK]mm/CWNJRXK!*KPe#7bBT#9+ES#T{lEXR3*6حh4hřjMfPJ#ZedklZR-H"mU钡>?&dL-=AFАBH"Y~V~C෯?eEoats0â#9(w-,ZjiV~Gk$bHxC$Ec/#7Szmwn6kQX4kwWiMՕfQF+RJ~7?Ma521:}2y(&	i`-EX3b"M	-UAddwP#5iπ[ZRh97#59*50l{O)k[@z}ڟ854Ƶf?uzQQ#9!DXOLa}#7Ϊ?5hlaWs$P ƑZJ!ftq$pM#9fTL8:s.Bmt2bt|VOgS2O.P7B\ɉ/"yR 3KjIςbcќ;-DVG52'qK&Ds#_⓶c,DRtvVCv+؀MgxnႠrb[3,T&6t@0S|DH@EBMQ/ծ7\oHUlLb2@AX(a\w;Qq:@A7qڽU{#9\EabQZ{qckc3mk^5ڶPfAAO7iRE阘	pR.BGdDV)1V#5̥ZADP,;Nar#5$A8)oKhvFeBl	mkVPF9.e($`!#9A**+X#7/!X$X\[bY#9K@61#5sX~7&,b AJ!<T/9#5GCʒ|n_)ű#9='xikIYgb0n8|JCZdɕl(s3t9`;K궶ݕQ[k^#5!t#$A7~|OJej1??5"}OmOBo80P$lT߻mcڗgZX<Z	dVUm;[XEj3$lmvޝ?$ǌU{{k;We_)C"KC,Q<lF@Bɜ㇆]S-$9%! .aIAP#7)Je+F,@:IJ`.M|ETP4^|xK$+9Nⵤ5Qѽ5fO#5o=C" rt5Wv]\ Dw*-A8V]^߉:҈@RIj!#90\yőv<u˛W#7(6%(fȔ	xvd)60x4rW6yq]\"[9K-[rٖ4;wRډuVNm[RŷXl7tnݙܑKNrsV4rH!#Dlw-R zS#5iDJ#5:DB<`8#5vyV7-@@K*	KbSzs2~wKGhAp	"		#7f,qխSѧrP;fáBI좥Z&}{r#9Ɋ%6#9슋g:{AL!%FRjڴ)hePFDᬠ=J	ΡI&0nm۝wu,[^Iﷁ3li|Buһ8#9н"Pg5!-%*j1DX2LlEmdmp\B(h#7"a"E7[]vm6{*.U[Ґ##5Q~)EJʽo"#7" Dhj=jj$(ﭶ6UJԫM5#5HHnه#5rS|v٪MM4ɖVB"s|B!5$hoYɶKi+[ֆi+^UV+`(0k#KS;h.}PF(S1v>Gxh5$X{X=FOwM1~F	20*~A.=s=q%g8Ai#7`DɈI8Ս]44|lDHI'p㣓Z;?g`7O#5(QN8-cFČhiRD-m6Q8" 6FMIjH<d^qksWI5rڭ#5#X!b[z%0Wn2P}]MR[I	Q6J#5;Fdz*}$(wgUP$k#5#5F\ d#9c4FXhmz6pŔ¬Z/c$XEUtܣJ:5DfS-f@@&p'JmTIN"b~#9oM#5R4RѺZ}=#5l{<IHF NAo-6TmҮ #5$("$"P1fj-΃{"nL(%k )X@R8#72#6OpQ %4Bm7Ym%$I#L+ǃ#Ʀcm!+V/6-hg>;x6pUUnf_i퓶toل#7z>xo#7g`}h7<鳵&b@*0d:4܅qiHcXPΫ)FG@AD	#9$Q1>D	NA#9p@w0mR~H7<Hp899pt@pAFcO[5P=PAPASP_R}Zx7g[E֥~*_'o5YYv]XI`	үFi#7mhe(tj_5$5%ՐC)*볦QJ"nٸlLX-jx(4X#7pC_@]M[=r}rhCP33Q.h3Gti㴑Ҹ<H;UyO97s-9nalU:A0?UNls)by~*|;[DS/r6J9K,va!"qai#O˫Fuq 0P#s0Tɍ1Zc&ъNi^~	z;~\(tB7kDe&fRviΔKi<:s`kq16S'nYjh<8ϯ9@Ѽ#73]6xԁnM^Tc􊗫0eN0:@ݙI7}h:mXo:D1!Bu00IwQ%5*rDp9aZ=ԨbFzJO[bBYt\m"v<P\AUsȏLZ\#7ʪ!ճ<?]h\̳n8RFO6&HltΒFc|E%s3E0wŋ-xdb>x5TJ"I3nmA]iVьvt+xhmn[~})hxl^K(|ڜ#7I+dj&֠l&7^=+W#7*kM/|m9YsWoAY_̄ύd{Hf)C~θhiler{J9e14g %?C(v6;u;&Ea"]p1&qXVt])I!`ц(#9`CLgPy~=NR=qUP#3[d	A;gazt(;L܉gqR4HOHK3.dqѭ#'\4U(ZjyBPL#57NR=9u#;@4AI@#5" OŃf3^XM*}Ҁ!vH$[BS }#9aHADB@VDK4nR=u|kLFdm"DocAN>ޝ<ͬ3oȶI\8^5|azkM 99ݫj9hş:h"`rw80I@CFw"W+&|&]Pѱ jrnXq;'#5}B۔`#|0(H#5N(A#;#7C:mgY :C%-#9cM)	.1k_w?g߬dSV)#5څya߳O*9ԂMCtB`@:瑮{G3WMՙ#7s}n3CEQs8#jA./7wnnbc#w¸{jWhi]0 lӵݸ:XIT@w&A瀜I^ec5Wl)+c5 <My[	F %<P	AǃHHd_Lr.DO#7;\BNy4x$4^uG(H,-#9H;9-T"w,.1V֝Dآ(4!uG:T[bhCgX~$,`r7'iUȍPɆT('G }ڂ^s$[SƘHmQk&Kf4B#9BEX<,.jI $huk<|.?e(d*CJi"P>ǘc?**F"fF`,`&&(1LhAe6FLQ4,J3aBFa'ػ4xiDO-A:$wE{a5~qJI6n[Ra2`&"Ƣ3b"(!b8#5B-qVGY<(ƅJR1H~2;eJ}/1QLBV̆L_mtjrr}1C 4co5:^Q3=K1Ȼ{cX}3tg-#Kq#7=aAŶ[XI8c'.o#9|gވYlJ2Rq@`q*)!fn@vrl)+nӒ溻WHk&C[q:1r!Xٌbm``#7_#98@T(Dذiʅ{xqd(ImFڍPixہF' ĒJ#8nqkbtD_ EirɈ"e2ƕ*<KZ^R\4Ea_g<edւ0 2rh/kRiRU`tDym#7QcIzٲ8#Bm"!tSR,w2!;mQuů=I"RX6#7#0$P D|J(&ͬ#7")QQ(eSN&x$AX#7mҨ5B'\IrfW[ 6*К uHr$*²1+Mo!)FFdLza`uet7q%*p]U\p4iR#7&ٗO'3lc0\)ZEiYBH22j4ࡰJjAjGwdðJl˃Y"]`lIHWd\CzXE#6Y<!modxZpcGsb	4Q%F+^df|,	jq042$Ң#9EPtP3cJjF(Qx10hgҠ{n:S-$KL"iި*))δ"L+b<ٯoo+h1^[b2hLMX"IcF4U/;r[xr$ΨK@,)b4"lM,v9fJ6ޘќ!4or>1L&z)i4R(!SQXi7:0&AGĞ5#WaTbl|_k7+%8h:#{DUHOID6L^}3pƥؠ@Ղ 64yH1}oM@EPBJ4-%ءFb\cI#9AD*("#514ay.RD0@pZhSVktCw#9~&߷rZ;J#9jjlf&M[4ʂQ{#5?p]ރQTR?Fx`gHc{\_C'	}0hZJ-Ј-;9RO]{ú5'Gm~9@J\(dItb$K4AaЂB?1fH؛$"{k˥R't,`#9#9F,*4d56bb!?%˒VudmҴ`a%5#5Yzsa`mvQjR,6̙es"#7u-7)ۮˍq6:2LZy8B ,jcP-R?Dt{$#7wL0|c#50ɷ#ߚW+`#7_OXdvp^#9j̒zN1߇^׻4>'@H&0VD~OU>}z#5SPk1#7b@#7<UpagK?x޴pᅆf-v9R{~Bxr#7A#6#9[GQ8@[R'b#9kbT*;z-8M~eU:emr,["wİ(#Ŧd5~^8^NgV%#7DL}w[CW5T%QI?3YosNGX&l6-d"_㡠uHr]U.J(Y6ѯinn\Ѭmƣ]-mm!llZ鶹Y5˚׍mg#F65Er堣#5a4icՔB(0+ߞ1f@AM=!b-"l@#5#5SzĈ.z8;|}k@G#T?a &OI{A'l6R	Vy|?=Km٥mw5¾4Z]RwMj!*6K[vmEA"=vEB.$E0$HDa	@#5!#5YPf:*Π<u[FV#5	x(>tP1S{ZPun@R#5@s#5Ř`"	 Ċ)lJq_:9=zL	dD,(~͒	C1$eIfԔZ)&ѡM[FkZmJi-2E$Yda/=ⰷQ m2"`xFH,,UB\ŔRR4ab܃IP#v(i$0:C#9dEbm&2"ěLd#9P^sEu՚֡* B#5!Z`((E#9"	Ml1[#7KŗE]57wNRJBy0FB*A$f2꧘l$^rHύOs'eO>gnՍd)^#5|@={p"-|SG#9SGmYx/~8D(*K6!p~?˺{GŤ`14@Ω#7ԀI6!;EAFw%uTe;jkchZzXMK#.$ܘFgn *}?æ՜vpVkUJTjcϺ6Z#76E3UiR]2)tݰo39o᙭"DC$mY<of5bUG	X~DN'cGwv}dsd<ݵcWIuD'؀;TN.c<>O^	i>h9x5oO]1yB`tzҋ^|~=ɖn*_AN˹3YϦ=Wt	Td'g8Qvmׯyhq<m_W<krj$ !=%PD	=7AKjW1&:CC#9`uq#7UX".$*ˤbXI!)l&ʕK\%rv<9gEsO^t+]ԵBw`dP1]t8sÞԦb-Ģpq//b4Slam(CN8!gK8fpm#Ş8tVeHAXGqtӜǈI	53h!&7d@@w42{O%%͵@⹩\|O#7-L$fb6ݸDn)*z/7@֪jVcd	<*΁WT`;g;ĸ>Qޙbt6YrmEVm)ަ1gI8l8a@~<xEf^`#7ScE+uIO9wmg[rQOf8h>%+\t:ζtV!m	ܝOxxC?L5bAIzL]nr\렦pb̕=g}3ufdwq=Oiy-_E+*]>I#7cd/uVӌ]$ZIϚUv{_Д#n*{esFXf!߃L'XzML"mthq.hK}"ũ[Y45펽|Nw;L"79+<<S<ˤMeq-+ԙax`^#^f*}.<z!cKz[fK]	p2T:yyc9fqʣv|m=(fBbq^lS,#7 D#9{NE(~JH:a19!6I܂}`Ihņ]ݫ#PB:-xK p%&I@#7sL(E٬AZ22qۗ:?X<Q䁒<F$u`:]3㭰eTrG6#T)@`BLݴ$ *Z#7Br]{:fv)fXG-7ʜJkؐ;@;A`:[=l++420UoK9<!~Wdw2JAMaN!yƄN;.XB$v#7#0qԀ{☢LrF`9EOg$}"VsR]IqtQr{O	[ɝu>pC:TP*҈r8p[Y̫aŝxD7-kJ{L~#7XvTSU%DC;G_u)ް(Ŀ#96}<#7\aA5Q5j8lv|;Cǒ֋N21ENce/Ma?*h9{6yfbP.7l5ds?-q<;2\q[b0V5#7m#-0ՏʮO<MJka#7"	1(G	塶IupQ|: oXW}G:7M$h<svT<aq}3WJ|K}A[S9dw۾	/)x㍰iI^j#9cQneuŉi(}v`@Z^ޢgQ΅5\${u8TΜpX"$]/#r̆bF"`(NOZuXFpo|m艠k#9y;4BYGK\p.l)rf[K#^p3(c&<afġ5EǧrTU*lȶ&tL4IFBJh"YLmm,bNYxG0Er749:F.r.G#90_2MbL#9xy_ħ_`!EfadRAb:a(F{,>°H@:>MB E͑(6:yD0}u!j/jjUI%N\˟U_VGboN(q798>[TcE@AgҡFĿC#7$!V"L=y-rEJ%XI1m@+#5$(ڹW;xiISEh40#9Гj鶣mnU 5"#9 vXlPTQFB#7zodf3$&+g,Ei/dTE7%yWx\&Y[VZ*6֮Zm7*!P&V~0}XeRcLc`#92|K m\#7hHiѳQ+^sJ݁:֜h*C<`@/>V	j!!H1WAPD`UOR;]Sx02$fP-&nI(Fi")wQXؚi+rYdۏ{m|ܮ	ܭQl(PəKlxgP̸1L+qc˚+G\sor3SFIkZ֩nLk.<#7jsղ&@Ͱ8q#7CfEkz-(ڭRեबYdԵjuN@7[bjT"$:\e	C*#94Ftq-I!d@`dmR!ҵL.e}W-3<cpu(ig\+\ؤIFtR٫#7xsT"j6lw4<`==i658/*9R5hi!F6FR#9*ѐh4lbj/mɒ+vO"֡*TV<JC[IT3vTc|fCTKkfÂV["6#NV4W#q$7̪6FdLmVlau䛠t+(eJV4U5{-#7%bmaL9+a0+I͵DL<@#7uYu EmTgz(ЫU^f>P2F"%Z0#9> iǫpfhx/D鷋fRq@iXhpf,c<FCrYwG"fNłm3slֈքD`#kS(iĘH7Yk,/#`θ090)2zk.iT{G\eSQbBpĒBxݜ	Yd/ڿce%pzB0>2|Ԉu }_eඔ*JwܪwՌlbMܔHҪ@U0ŗNq&xa_Y!#5CE}Lk8)&~d#7F+$Uf#9@Jpiɬ~?VRU+˔f\&o	s0ݻ6nlK=}cÀ)GpqHs6Vp#9(K)8A(`	Wf&SzoZI@M `ݴކmeY)*S5)(2 %UER{PՀ?e#7d#9<H]w!#U5BlcHc$b&eQ46fضMֱIF%dfE4)6IES(* Ĵ,YM1J-$l(l&$EW0DxÉچ*"gƃo99|3ݸU]N$!'@?Gu9|ʃphqAp=#7I֨#51Y^&%#7ьH)#7|~1[Ȑ"1XM{w/qp6g/(CF"Yr5޿Af>}f竺0/UjmfT7YCdzY#5ַmE$d*(RjKj+"BQ.ՖԉT,%w	xaDH ¥d[3qe%׺x] |WXb#$q2Fs˥XU˵t:%b9Il[SP+#74V#B#ɖ,h1^|R[h1F2췴)(6[dY.a#5m̄ ,ib!JQ22+!).1$L^A*x1ɟ$'&u1p7eބ.NRӸ6kv\#<CS&l@}tm[RJUj`(8d ua0DswP77V@YUH59A,ozʖZjCw'R2"$I$>!l@Cٿ0jԘiJuO5^z_͆;;HVƚ-h|ʲά)r"B28Pϝ[@,cwޅ	UDߖBzRJZK&#5]+kS3۴%L܃F\_J ~dz4x*7߭LTRmxMS_DxߜLCS@:շpx96hht,D`wM!G H ]u+N!B/{,sO9Z_%,Q`H{pNϨљW@2Sud?ALL*I!#9S6'lS=[%lN`(mrm?+@行#_I;jQɐ{èp2FׅrۏI7g̽\>%CHhMl#5DD"Քs&%zݪ(QXkx{>xʳq?퍛۪Bd\i7]	NmgIja=aE5{Yb|[k:TH%K\bUB?R)wF8ukTr7g*_}Lѡp!g<jg/Fēm8gRE$xfDĠ5#9Ewȓtt^$msc'uXN&OE;?P*xnDx-Flh>!hg;7s$ |0 av= ܃jfcI{iw%9b\<z8jD(<'b̡iݭ{,rdnup	Tf#	 gAK2ajZ#724Yz``m0`vH$5VN6܁kŪ0&l"ASrP,Ms2日XD̙gm,ks^:mNSN5(dGh˶t=ךv~zq1Ȉ0cltoK0q}3ܻqfKMBdXkGSf%	0,틳|[mKgw#9ͨ6:]<˂R炓<8ܖBmxWD7f%+*24\0ۊc@R0ô3V`CBC-TvC6d-S2Xff<xպZ%]\HP)7GFf#;#9"?Jb>xD^PML[aZ 68qa8uitYX1-NI6x#ΚFI ݶi`.J<hfBRVݥ=N˘gl62'b8dlayE!Ҟq#9t؆V}m.09pmIa&Eywݲs2*1$>j#;ϴ½ċ=7#92!z-M>pz8XdyDEfa#9ףħZ]Siج60	*s&'8qu!b#j'b+5QPb\sHfJY#5!țEpVlhSݵTEܸ6G"&M I@Xj7f$m6Cv<tҥg#79]̜h7Nf2ᾄA&1D?CᘎGh<9PZH1Mi4R	f+8chC;MP#90vWHku`u?K"@drሎgYZ7	xŻ|ZHt63:F!sUYCMH#7lZBbq-B\^(sFp Ls?f?ӓm9MK<յ>$	3HrnxKt Ơr}]7]F爺fd]vb|֣&"0_#7,-&)Vެ	#!gST&(>8oa0F	d[y*!)!@*4S&=zpښ#bHPb	vZ'5Ls4L9"@pa(a)X51MC2Aڦ,r*`sg\B<\ɲ(#9|vndpn`5-8{k$$]'Ã4~y,1{Nz;ֻS`Ebٮi%$F8uCH)5c6d26q.dƃQKYHIj(Ddo#90ARl[kFQ(L㍁LWMtl5d`k"AtlW|ǸQPP2F m2k!hE"+6-h崢;#7Z`K^]7u*Zed*D\6A;x30H"v3#7C|\hv6f [M7 t7±AB5aj[l@7cq0O#90eҌ#7IkEIuќQ )/ENyׇ&v0D#7@%s"m74(Q2.uUE$I鞁QEDEzjp==f~}zXm#7PHw@!pPQ>^ӥPNҵ~2_;EGXdA2bH[]9^QųH4vdw]#96~	0845o"f	jhI#5=)TD_b{iU7ԇ2kV	xԠʁèvk4AvM2p=+b抉<*Q*Dr"R*.߅=kD~R"i''wosFx7'֕D80f.)¦#7xU6_]UZHFsK{ZcM$SL:W־D>w~NgwHG+={ D#7Cn!#*qxA:J9(jBRD79Ũ7p#91K×uo4Ȕk;6I]:\F2;	!TMCR6lBpjZ,f}T*&+	6(6CO-4j%օg]$R#7@p7OGN;|Bawj5#5U:yG|a޹#9kqlt㹅#9Z,"J1a^@jFM+Eupq@y	NG :#9coQQ4eU=jbʯ<_?IF!_yY% b8 §ع_!ǅd>e2\RB>ɖSeLɷ(ɻXw'*@CyفBT^#7f=u012ǒdlXMuCyU##9Gdx2cF!$=jM-̭Qm!+cS+mE"߶#5B+f#H$#7a9 g$ȠK+!ŀH"b"A'cE!Jd2@Tl:*GӰ+ym6ȝq#7_O޾hd4Hn^'6Q7كBČ ko۷=(߫i6E7#5tJs<55zh$)F6Q4h"*V6Ky<]p͝ے*yyzyW)zƲmfnk!fPBYdqSU#*7)3#5s"ܒ8и?DIXs|hklL^`ITaT1Z_#7aͲva%DA@(۳KR|?>#97ćRB)jH=1#9#7oUxOv<!^85,#7,a8	JˆwW_w2x~ktHk5>lB\GcǞ܎y#76y"Wp/nRLe1i|?.u)hl.յQg3ҧb*r,1o ٦-He׬/ҟivڭЉgiLMI*Yimk0&QR%DiݶCEfԻo}61WLۢ6!7xNJf w#533'NUbat ,ɟ#aN	#5<8ZTT=={OuQT!CwcI4nbQ6Y^t*(ܮi(X"MKK%#57ʛhmNCMөEkDgE gɝZ@StwqAVRw%DMNjC)J,pw$`#5h2lgP;9rI7@SxNv8|¶5FNl[Dl  ,[,iBU(mCəu=1Sqޡa,/bhhQh]FJma&ÓiuLe;5+Qc!P8(Is%-޳!@#7#9#5R8"HFL#5GCg<CԴxxg8;\$8ɺBH#5,u#5pGc<7|O#7idJpԼI$@9d/XQQ^p	9^tQqC)R}ǷϷ#9Hn0('r6J)ZA)LV=CΝ D,Q{#5Nk:ԇOԤtLxg>s0Bۮ``b}#7dc|ߘ#7Ƅ~ڰ@mQM+_e^_>4D	#&3'uō$ ":TTKTi%Q/&k^Fchx湮!wuo<ҴE#5iZ)$d!ܠe?"#5M[1}t_u#g;+ p)+ˬ9TY#5!"1bz>	#	Q!XH6̚~hWժvb(Zm!5[@02I-e脥11.$.#-QWW7]sdSbVʱE#kL٠zԩ""ӥ@0NkBIޯxE5#7<~zTȩ,ip ,íWD{V)Q`bwhY?=^xŅ)	Rti03dp *#90"#9}7&ÖD-4zuR#54	BAyQ"ABB#9cj%mJu6W4؄,y#-S6#7#9.)]/Ca}[}|j$pm2VL7LNi,q81=@-iH~T"8Ai#7gͳ#7XLӃ#5+NelcTFԿd"ks(~ǵk61/EJiI*)lb66RѺO߬,(1Y$9,A R5D'q\c$Ѵ}д+oq|'헡wZO[WF5ڔ-FՔe)B!>bQp_WaVYLF7lmc}1R*`[xQO8<1.#5;ȫAZR~o$],Cb#3ƭX#7` ȡu-i54ReBiݕ+ƻֺdn4ݙ]svU2ZT2%65,$ISa!H6v&VE^M1&0D!l#5+@3LZnv"EH0Jro#7$(4 hFрpC?:кB!}%!lU##5dOJkM&R3E,eAT.^0cMu!	#9A6&?3CYȹ"?r$|s;g>Yn2$6LMC5O["#cԉSF"F)	e4&?VQH+R3x\Ԙ.jΆ&44H|}\{@'F#9gs.[P[ޛ@q LK2u`>1E:h!c0O]Ne!C,eOӰpq&5"gta7T	dίd9*B=PŘnPh#e#7Ju2*)921N4W|$#5u@ۃ0(C)gzC!_]%E h"Q)h`ηs=6:AqX%8d+o֍<YE#7N#72I'9BKXdrȑ(o;^F*b"=v|Էj#7ROfF^eB9ԵmpMCHUpBqpL'*7g/p"#t	m1ȤQhp`Cq5_2zKsU[lu%~ X(nc""S9ǟf$#7F9Zۊ{з{~}\.iԪ$sZ84#9:~1+Y.#5	-/P& #5 3h(F%(iW<_1POn825+yj#5A&d#5YG?/.q_?|GˢߗU?\?oO{???׫'_|>Ȁjn~8tb5p )A@\?*\5J(cu?-O#7EH^?I@s$fg϶͵#5$DrφB@;թ.6kҋR LOIET-A:7nؘ{#[sxp(/ԘaȎ#7;Lk<(@A׭Q`S<.t?;d}^eKx/0j#7̴tI202ƾR[dw#2'KJltB[5{Rodkứcc[A#46JW4a"4B)hڕ}LCB@cXC =fޚD:>m6[!u|L	Ep]$0CqϫV-9R6ws*$l ×]oES|뵙8*S:~G\2ej1&2(=yp49_$6L,<2jڵPoδcMɨ¨XSw-V<UA(7O{'`Cy}Xj+%'TXG\HD;x*YU$rF<)Yv~o}#7ax*TS+^>}Ʒ1,gS5Td^*߳d=5F0M{ᣐ#5 $Xîͅd"F(!pC(o%*ȁ${]/{9_.Vv +"`] XfxuV9Zh$,*&P#7M&mXec2%LLl)BW,TC3{&P$KZ_ƾ)^1Ġ0jPV*ëA7yr~gD܍ʩJP9x-"#A|A**US"(ϐxrg8(VA{hd2X_hm߄X]>֓sy$#5laKmK̮wCﷴed#7Tm6GH}.|y^)[f#7wo?qzvnB&'&THm VUIX6ܭoҍ^ͫz+Rh@m4"4RD-볥`XrCF$5BCCH}:gpfN[tH%Y]y߂bbRQ(;'8R$Y#5G!Jﭔz3hGSA|`C_LUbcvh;jͶ[S4Mn]LadӈBwAbK:%fd_GWgt/#9  GQalh~S{GSDdO'W#9O~PxnT$bȈmt/IC#91Bzx6x\?_ܚS2$<wdxߏ]H>p>,/G!<"4po70@B?_*N?sFjPD|mJ(qA`nGo21?ו28Fu?QkX+Q	YwMN gApB#pK8e)㗄lN*%KxAr:,D]5يb⁗XglBoq_V#5)SZD3+%ȌI%#9O@)eB	D.p!`
#<==
