# encoding: ISO8859-1
# Thomas Nagy, 2005-2016
#
"""
Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.

3. The name of the author may not be used to endorse or promote products
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
"""

import os, sys, inspect

VERSION="1.9.12"
REVISION="c07dba0d26dcbb44f99712ed56ea243a"
GIT="x"
INSTALL=''
C1='#8'
C2='#3'
C3='#.'
cwd = os.getcwd()
join = os.path.join


WAF='waf'
def b(x):
	return x
if sys.hexversion>0x300000f:
	WAF='waf3'
	def b(x):
		return x.encode()

def err(m):
	print(('\033[91mError: %s\033[0m' % m))
	sys.exit(1)

def unpack_wafdir(dir, src):
	f = open(src,'rb')
	c = 'corrupt archive (%d)'
	while 1:
		line = f.readline()
		if not line: err('run waf-light from a folder containing waflib')
		if line == b('#==>\n'):
			txt = f.readline()
			if not txt: err(c % 1)
			if f.readline() != b('#<==\n'): err(c % 2)
			break
	if not txt: err(c % 3)
	txt = txt[1:-1].replace(b(C1), b('\n')).replace(b(C2), b('\r')).replace(b(C3), b('\x00'))

	import shutil, tarfile
	try: shutil.rmtree(dir)
	except OSError: pass
	try:
		for x in ('Tools', 'extras'):
			os.makedirs(join(dir, 'waflib', x))
	except OSError:
		err("Cannot unpack waf lib into %s\nMove waf in a writable directory" % dir)

	os.chdir(dir)
	tmp = 't.bz2'
	t = open(tmp,'wb')
	try: t.write(txt)
	finally: t.close()

	try:
		t = tarfile.open(tmp)
	except:
		try:
			os.system('bunzip2 t.bz2')
			t = tarfile.open('t')
			tmp = 't'
		except:
			os.chdir(cwd)
			try: shutil.rmtree(dir)
			except OSError: pass
			err("Waf cannot be unpacked, check that bzip2 support is present")

	try:
		for x in t: t.extract(x)
	finally:
		t.close()

	for x in ('Tools', 'extras'):
		os.chmod(join('waflib',x), 493)

	if sys.hexversion<0x300000f:
		sys.path = [join(dir, 'waflib')] + sys.path
		import fixpy2
		fixpy2.fixdir(dir)

	os.remove(tmp)
	os.chdir(cwd)

	try: dir = unicode(dir, 'mbcs')
	except: pass
	try:
		from ctypes import windll
		windll.kernel32.SetFileAttributesW(dir, 2)
	except:
		pass

def test(dir):
	try:
		os.stat(join(dir, 'waflib'))
		return os.path.abspath(dir)
	except OSError:
		pass

def find_lib():
	src = os.path.abspath(inspect.getfile(inspect.getmodule(err)))
	base, name = os.path.split(src)

	#devs use $WAFDIR
	w=test(os.environ.get('WAFDIR', ''))
	if w: return w

	#waf-light
	if name.endswith('waf-light'):
		w = test(base)
		if w: return w
		err('waf-light requires waflib -> export WAFDIR=/folder')

	dirname = '%s-%s-%s' % (WAF, VERSION, REVISION)
	for i in (INSTALL,'/usr','/usr/local','/opt'):
		w = test(i + '/lib/' + dirname)
		if w: return w

	#waf-local
	dir = join(base, (sys.platform != 'win32' and '.' or '') + dirname)
	w = test(dir)
	if w: return w

	#unpack
	unpack_wafdir(dir, src)
	return dir

wafdir = find_lib()
sys.path.insert(0, wafdir)

if __name__ == '__main__':

	from waflib import Scripting
	Scripting.waf_entry_point(cwd, VERSION, wafdir)

#==>
#BZh91AY&SYtxn[' $ 0#.(b\wz#.#.#.#.#.#.#.#.#.#.#.#.#.#.#.#.#.#.#.#.#.+-6WWZգ4t}*[(%.$Lg֓޹mkqC7N/^JI]k[*fȅukk^rFfsy}kNݍ=u#.#.{#.#.'c@=U`ϣP#8pP;lN;Ε#.F+E4Uhqilv6E,4#.JEe^ءEPL*Bi#8i:Ѷ#3]qȝUonw[q햙)Zn}Nou#.m%^kw7r%˧g{|R:/-LvtEl+AWflQCXsՎwQ%P^#.#.2le{Tս#3wn滸-|7ۚkP>-06VшP:f7cܸp۲>{|>n]4(ݹm7m[lByWkg_\ׇvw\n۶+r7wR:qrݟvOmj1_>\w{AN9}o{6{뙏y׬]'^^n9Swg#.1ۗݟfjx#.6B+a۴#3[jmүrt99Wnڔ9(v^+lx#.{m#.O1ӽ{am=݌zCFYiZu"bv\!G;oqKoe歲{7nfס}ܧ2wk;=#wOWﱮM^/h`EjBvwMU_l;R;^a\jwS`#.FZHx;4z5MLe[w׶ucoF٩R@ۥh*:MuB52ti#={9:=f}#.(y9k>o{z#.tn`7[t9{{n]-vYReδ6:9cmө./o.os8{ns&'Uu7nmҮ>dkHۍ<z/_i#.F#.#@i104ѠlbQ='5A#.@L@ѩ<M54=CDѓhi#.@#.  S	4hj=O#Iz#.#.#.#.IQ&i4)&FLChڀ#.#.#.#.#.#.D#.@#.hɠ 2"xLF*x@#.4#.4#.j"h#.E=OA2z&#3)4S=O4#.#.#3#.#.?0?Զ:/b6ڴ"GٵkSD&1ZOkn[!KOʹa~[R	U"i1>/MNJnPH";nvW&4r[bnuVpBSnx6w?y1i#.R]iElUL *2*#u'IaI#.I#8b#.QUo"!וmթ-Žj֨#8(3,@J#3$&MKiJ4i6#$%5MEQ04M"&*1HҌK4ŕ+H+QM,e(I(͌[F	hK!-#%#3E66֖͉4fd,kM+Tʙ*kc*U!--L&*(F!El4X,TLb$3 ŔA#.K &QZMZBB1"Fe"$dC2JiMLLE,VJX*&PŒ,jR6ɑDP4dAF$Aa"HR))MdlDS6	LBI`Ieai02dQhDA%"	PI@b3j52VjH`D&B$%`eeMđ6DL,AA!IbX#SbƅHLeFBJd#34!RK)2E&iR2Md #3%4A,FTh(Rb"H$)RbPI#fJL̶R$Ljb#3"3YCQ0ѲE2Q,bMDC&RS$d6")(lɦ!Hc4bMK%jTe4RE$ȦhF3e,$JYIQliM#31B$MfDh`2ɔdIJlȥHMbl JBBmLE"L6)#(ԑQfQjR`QdPГ&),"%0,XCQLf![S[Z4I̣JD%)kAKbRbH*Y3EB%K,4)fdd"HJ#51FX65&LҢ#3ZCa(F5Y1e0"fj$[Q1hɥ$BPJRL3M֑͔"jERY*lf3MmDkd+$%Z)T)El"6"fbKHd`XJBbhFbUљ#3QAPXEBE0#8a%XM+JUd,[I%jmY		MY4Idb@)#mSm-)KM,Y2"jlIVYY&U̙$542ٚԲ$m&! QbFi!Z#3	Dh3)+&idH"h ԌXI%6f*	SAfJHDXjjLR,eFi0j2	&BMb$FF(FRʓI	QQHBQQa4QQiKb"*1h(*h%D)3"M,h$fJɄi,lCZњ"4TɳIM4	dLL̂̓a4(M1d$͍AfIME$2Q5ƃM,LC#*hF[4$!FѪQQEX̨4i!44HR#b-eM#8)f6bhZEZMS5b*(%Qh4d(I̊"5b2[-Ub`DS2ԣ!JJI*lJ"elmhX6,2)V)2	ō%"LIHVRdm b%)S*Q̊`fj("1FSTbL$i ɖԠhډ*Ѷ-["Vb(LRaLȘiAS6eLKbKZYmUƤ(آB)fEXmIhYĶŘlmFe5J2ƥe$lhاw$[MfVLmIMFM#c&@RdL	H),ll&4%Xش[F5Jke,!mFh2"4(6fj^u͖EE#.j"$I2Z)$b2ZIk-6?`S@#._74_[żs޲ZE\qOJ!	F#3FKBA#8hDqv/?wc9MݠW[]l3zɄ;-=WF4K0[&<aJncݮ%#)*]ղD)Rʡ&9dlffXqߠg5rḾ{ThsyS(@R,"bch\ZȔlb.ॉYm͑K%bZ8hq޲[?#8E0{"mU"-KK@X(}yYea#8p"%c`#8cJTn5!B^0xδy:0 V['ÓjrqxipC==x_b},@m=Wo΋\ۥchkQwXT[=;T2M#81XgiVW$mډ].CǊ{Zl:UXHh'bMNve#3`ϗ1VDmŨVr&22Qu(̅Z^u嚦#.QH9(abЕѰV,H1Ɵ0RhY.\}Ok*'!$&X_ r?u8k-RhAP$NѧzqW#3m~\g":Ly.fІ}\mlFbR?oO5г{VɯC[h(X!I2tj#3f<j̭ࣽtc*[0Ҳ6o>E&؛RH#3Z#/-x%tqmuQZѫ:6(2J2v-VcAc#3\ܭH(_8߸vsr9vTÙF1epҌLMMֿFHLHb>4EFP~ϕzd)cB4MD!`<}zc&梶O{<e(J7rƷu~^ݍD:MFb1dP"02ؠ+ʉBO&TXmTEڏU37fGȈh:|@/Z_#."$IZ#	TBJPHe%kP2a@QJZd)E2IuVӺR7מC]+~wm1I#8r(5a4F vHtaL2ema,R?EDAY䅈#3.@O#8oʂ\vGA%VSo,1Pa+Pޢ7;Yoӆɿ.u%jzӖ#8@DL7S_wMjڊcځ#3]WjhȈ@ɥz?Owtp(wk[hSsdzB]Dh3qxf )&A2mj:bQ9qe#8v'|SPs"n\LR1pR#q񽔴@Mbt.t}_$3T&Xzoԕ^P,8Ιߥ%'׿^'	Iё#6ˆVOUb8vZN5W\ٓ-(dڴwܚ!Ǝ򶧃٦kj<ٞD3QNyq#3hɄ#B##3;-έvT?&4t1d#3:#8#8yϊ%CF_w6%b7mE*tNLΞ8)g&JIʧZ̷j-әHAL6hkP#]IWU$JtA{Wgxx悴>+YIbdO=~1Bŭ<ifh޲#3h֩')eL/ڂoErȋW*5PS@Tqs0DGHt	5+nX>h<.փpipЯ?O{3ީE'7,R5l3@,'$u8afZPX+TQuQTN-,AT"Ǯ4-U-Q^(C˟g{#8O(H.#8#.B(T.mzDf}JNDZ!ֿc0',P>ܑm=yq10kA5~g-E]]ލ#3kL5O-n)ʳndF֤ӣf+|kV#3LMtXS$|DUr"y{tF&J>),՛K\]UYF]O:٥d|x6ۿjEc#8n`fH6{iO#8ovǧi#89tשVx&1QuV4yGa戆Yz_F3ѡW#}Z??^jϺlB}ݗ׌kaDJm몤;|ȰSJ:(rinig5N[mgƪ$y=]^qc8h!~Tsz٨Q+vzT,@ agŅ%玜:<h#8h6"Zx0)AX*0xoO{2V2~w}7ҦGG5xk;u9}UTODvOg6GnHyϳ\	2-)p"Ѯñ'yNzFUiT:Lg_"Mɏ'hf3\d\qTcb!g9`UP6dX4_PzcHc;>4|KR$F?z`_V|*HtHI&pK˴#+J:[wJu4͸QwMsϘkcp3gӣҺ@wC)/kUXqxm)Ӈ߆G&!}6r.Q_MWkdߡ΅j+#A/1m-2Mf{yij6"IMEƆ,Fp8Ed`:Ӄ捿ZD^Xۦ~'೘\'QBtE)-")C;=۟]5^5^MhPز{Pҫt;.CfM ~3PR6:N8'UKL`^!{YWCw$xNBFmnρ!ThjIHGVDy#K&< M##zydԍm9⫀&qx{_]],kX^(}}UOVy(8vm3,R})ۄ6[ҫ\%3I's]79]$IC//.=s`x_O1mL1oZUTN8ΨrR]PeP*#8Q>3#3IzY"4EC4[TC\IbFFAeGRPskGa׍ߚ,43%(!!GꖏWփ^Xtp^+Fr\P+fR'Ҕ֓j8ץREUWH*QyӰO-E~?(Rm;7ԩ0:`?/]@Ǎ٧$I˹O,\۱p.N;m8b[Խ'ۛ0`](KST"l֬mHrBof]I1y]o<<d21hJf*pu5,ֽΜtͪ']AT%n#Ln)uӃ<vl7#?cPgFOYkl)Qpֳ"49Tv,IQcweEɢuaLP1RA(Yƴbf7@WqUG_0]H6VYXlGbcZAAAϷ#{|]ꔬnv껓ljbZ,_~@_w3nv$¤osZp4(d x9Fo/X#.Aj 5Fv#3̡A뤣!`U_o_-_24*<+ue}	̗ATP	Ia:0iI՗fst3r1+փoqܚB]4팍95jkQFH&InִHg7:;GW[apᶍXg3Ζ(1"8u?T-?&^w:1/#8+՝^}D#3ٚ0m:#1a}ߢh{ pk@t?#3F1ulTJdqޔB4W:L$Pݸ|57FIN"3χXfs2A18m yKm%Gj#3͠'cHIe(ܺLz\ϏRH-@uʀ2#8*<-#.p:d5't>#3N6Go8-v@Fkp;,folEKwLk/7u#27v~|Ǐ"?q|:xCnGlqWE'8R%86c#855J1jK\e<b/yD@Jj,&sC2HصG9w*%3MS,bL@J1۝i}j֢l]HVC	IB:a6ߔ+U݃hU4^#8M٬?~5(bdb;ml#.GA".[Cۉ?˩z`уS(g2;vi2;"iiCm?+^#8#3>P-(@@(P%Go;*vGD怛D ft0dHpNoᮜtJ&6:p:^]xo;#8?ע^H1f5^#*9i_	Dꝑ:	kAX*:g_|;gl\i`2߬y=	wGØv.F+5R~wƖ\x"n/[HMy,ZTg% 4#8,`u:dJN!{O.4z`/$⎚_+g8??L(qv&#3c}XuԒ##85QojLL(#8Qˣg[#8QݒCSrU$B[*|77	;N9ʿ?[J<P#8#3ބM.M2w##Rʕ'@w(M.C7Cg2 EC륟t\v&z95P*/@?BZ/UW" #3@M_y_р	C<&ۑ8RO2^b+Ht|P#8B%u>#.R]Xz9#8fl<G7EXc^k@{t~uk-Ba8UHQaN>'fv(&u&0X@j/nnx,].kk  j~"8"G$O;-#.=Q=C;j`E	(G>U<@~ZN}@>&+cWͺ]]TtƜK,d⢚!}-lT<Ά[G9A,)j(-E<SxvP3^pS	F7ilr&aSTKMUE_D^ ҚQPSP?9SdE;zW*!&3ڈ:D@O45*z E@UύX E{)БEe4ˮk/Ͱ}'[Zٖ*G{"ylk4rm9tծqWsޝzI	YdBYxTw4}<>Gh`A&\XZsZ*?q͚pg87qo3<rz)ubQ̅,H0QV#8'0ov@ǁLܿzITNk97<?HĸX$]ya-vL,[q7b!0[[!qY9E\Ї5*=6} AC}7_#. +[Fl̿#.hL#3ik/.4öO/L2.3ehm$<WT%8I*<+y8Ύx"/Z9cuBsG٬?.&^WmB$w5.㾅~١0/-W:Z${1vƿf?Z_Yhi_WF0P<(-v%/3{(02ICcqy>(lXF|:8Z$z@sz^Ҍ~Szy[NPs\6lV߿i3[4\¾Lxv+\#8Аqؚ7U0#."RU.G{oRF;C5w_#3}nM~#.&QA0k[=ZJ}:oZl:Ap3MaY}%"ʽFDFI5VZ+N+L#rs|~R:l#3UM$XkqO&<uc:!3q5!	}x1Ӯ<U8ҍO?R\+.%#8=09sPX]DhDu8j(ԟ/nV:;#ʪ{UrZ)*3pi_W.lq+Dm&#3bJM88`-8(zNaڨkKGk1)V.da0&gJA/ʱ	nsÛZd,(e+7vM0ܧXF *w#.9!ԁO`CBS#8X勷MMc#1#Y(ap07_g9=3f5EY1QB@oگN*M^z?!T\l 鰐"too}M,ןЧA^s5p}.r#8? B#8hS`!aE5)fg=n䉶6jgfa|Ck( U^#32Q։*ȆmVGeU[e(^j*RA{#Euuh;pIU	TwuX3n~uפ,{etc>>U#3@rW#.&ȡ@*PJ#8dϚDF+Qٝ{ԜX6T3ٔm)Uj&`"Z]#gCS[vКS)go5s@9BuK>#f#sJaͭ#8(§nW$/g8Kk<܄Y:IP]}b:g!0&֙&^G\O\U;xƮ3\߆}[p]Wgƿ,%uSSii#snQ*sm&q	(^qauxed*%vvpb#EVJH,ǧlF]7Ns5e2(nYc(*у>קffWFR R%2H'R,o!6|kf#UEѡ]pB2n'%B <)_<b0eDQ]^1*jqvh~Z3dY`dw=8#.uNkҪ:͔9P)"2-^2d*ꂞn.LF#8't<nf#8P$0OsްbqH@Q	NƎ]:Zy#85[nD<Q62$?2,:@YE$X4H\T<4DqKXާ5.(-08jOTz"!scMҘAZ6T4$aC%1ZJ`!v1d?jā[A!(lĮeLZos]\7J@PلAHxZWگ>zېƔ1SE-|$׫^M4#3!ٍdXвm5V[-6-h>x	#nsB\Sэ=RU{?	9l)߸	Y(-ZQʰ*E#2#3چR`AvvO\u͕3{vR(#3U2,XϠv0)Mܙ0y94}Ty9ٜlwxLOchShL!Ҍܤxf)ZUhҒ7$֦#3|>~ߗtmdș.F4Tٍ]ԵS}qNdRp;TƋ+rNG$}j|Jtq/}<u~}7ZO\nF0<Y&b?(:-yQQpu󬤬j(=0ވ	Urvw8$'5}].#?XfISb)8WY?$D}CX*.9ԍICoh̎p!q~ɖZNVcZzVBQ}CͲ9HXBWx\Ae&	#3{ْ)h	#3rُY5Kܟ{켾F^`ec*z[5d#84vr#3SJi5$f@4#`ص^\eEc¥*D#.YPc0#82D`R5(!$.T4,	ѰLAemCDH#8<f$1ZiIva8t6;'3= (qQJ)Uew͢lZ~ў̉K^ծݙ>B|'0}a:q+M_"LGRzi;(rjV-?PPAG%i@бh)=nkYǧ3aQnmɽr˯UH#3ӉI:bgi	Un")xzlB2'GhkϓMpn깧þ4k[F0g5*}٬T(^KFcj%ɇ$s͍}J^om}dǀ&#3䁔<)6o?+ܷ&1;Ϡ~?Ȼ}kD"NmRkkjṲ5,yDy^Ro(N36Fƕ&p\[;zgI<kz|c@ZI0pk#3WUKMƛ	M\ÍҟKCdN<hd?*~Or%OLQ:N]'_L|emA"B(RWW(|@$#.#.(@|jp	dmTA~mzեtI.	ksAnܝKZC>2GPڌ%]6#kr`A!&"2Bwi"-"0lMs<\ͭ@q'O#Mƹs?uG8ψP#"}#.ͺ|*1{_9Wy5w@T_c>>z,R2NDhkƱſѻ&[TJR*N׊UE!~\>]4nut3)'+UMiGisY=NDz5"ZmdZTȤ*pDy(Rw'[ûs>?O7\WK{R4:>A㫮u(/?-5.?=/{VlLD6=l<{zeuZ:'L,ihdGR!g`vI@Xv祔dN1OՈdˠQHG1RQ21#u?^|7	^=?f)bՏ%ћFjed@b G//#37}zTfʹ $YݿY|Gv܄G'G.Xj[gfԖMb6T(Qbhb/N۵0DZ*P.wosk~^V&<q`WX3TϾ4b[Pmue6_WZ_xLH%tYEpPB5jwg(!mOii[Vlk2t")JHu d>0>@R jfD8@tag2$^ʸVVL8*	Oْ?߃V4:7La#.H @g:D0H M @eT G#.dz_\/ezt,S֊?KaV;ꁏGlptZ܏ȮzQy'E9T)nVp&:Lan[bA*SI"ò6øĖJ*']rx{P?\P#4߲Ȣa0$]@ޏ0kdE ^MtIOTa!D%IaȰD)Z|墀0R[?:z;m#3C>stc髸|7#CxqG0lGA2gctz0ùTj?i"jQ.[4.~@~5;^nP0[ 82RҹZ-S|,G!ekZ0Q:y,5]g*߲A`ˬ华쏹l_0(5X7*~Χm[h#3|S.ʜ	#.T_Ƙ1@z{Rev5oh;Ti#.rn*2qCmf3 R7uT;=粈J;l5MdR3qwY)#8g5K.۪"iGSC}0Ct _P^QE_)+\%كcۯf#3$	[?u}o)vT'PGy/n6)Edj4q`am:〟:{XVQI|:\?#8mZϖg҃US8ݞIA{Fˎ1A%|aY_u1unP_4OuN^7_ыgbNϹ"Lh*w!ϲ#.d1>(L̤s\ihX#39ŋצG}҄Im)WXP$a!i:"=ɿ'63Ύxn:'acIZZRKW>h$ǡ~Ihف$U@Hha$"!;EEGtƞ,"o		C,녓Ih	DB"]#ԯ˥efi1\x@|{ٙ_f6h÷>9o7L>9L~E_?]upgT.vp]ty(4M9^Nw~;uh߆e~OfC5Zp4</ρ"VTNHaεɩݜ%=.eg[m.#.6+R"w͛jpw-bMOD}CkG~	O=NyЀg,+ץהWw(0xS\Rt9<uE+[z'ẓv:׾&(uV~,lqZuՇc9x^NĻFe#>:ue|ƈޖBw#Ϗ3]#8.rF]Go|QcalצmS#3}];첡>~ˢ$u	j茞s6`˜+wcF5f+͖<jM_HӷUmɡPØŏmsB)jy~Mmѡ;}@r=!%b>ꅆ_uf͟y0Yq46K6}/<ܫAv,=n9U$:$v#8i_׮6{!_Ue?~_o>=+y_kYᵓafH>yd2@Ƭf+Ttq	ˑCG5;}q*$@z>f϶e_-f<9F,nw6fdCG$|}mQފ6Y~_U?;˙"!Tz}WU?bFEBx w=9<Qҕ}9^ruG?iosg+#.8QN <I}uisw7݊?~dY lNCΗH!k?"Pw$,KXS__-dE?99Xto?D\~쳍r?/gŌͫv]Sү:Nߴ#.»l'Nկ7BO{w:p[o_U:-6}>X:[~p9O3eϠ5#.P'IxZc>~7Ӛ̾z8d6sj||BM~Tl0v[,{oy-?-ǸHd2*_ifi/ǟo#.>AgҖhqG[Skh$6otB#3_"v`b4(闖W}}oGh.9.|bE˕ߔ2d9M|6L)*21uFmxt?lF&xV*6#.QQT58'Ƶ(fG4gq,H'#8My&5/#..Xd̓V5+X;^W#3z!L#`釳tE|l6,i}[#~0R:x}&RU?]fHOTI%"-@@CM#3;|'g'z}omPdw9QelPTӰoJ6ؾsJ,ܲy6I޾FG80Z.(C^?{n3:ujNcOPvE1NpkWmTլ#.5TcMD$Wn5RSKѡ|lXl]ArgW[^G_[IFP&sHײ@=oЌeƬOս?qzq(Ʈ#Ug毩ɖP{2Oz۾U8rʴ7tT<1#81J?ӈQז@R_qh-q_cHyD]&L,6/tٛf/}2_mL<y=ƌ,wgVئxݷ?$cWHmdMN!];G\c\[:Ɍm	u*tvhT3`(tIf3k?P~.c{$#3,#.:^cYe@G6<B1Rf#ZԦHV܄g*XH6]cF[XM&(mAYl4K!tٮqd0+5S֤On`aJ5kj0$M0hho-SNDح++F1;	ơ#3d`pdXTM<17}3	|"30p["=c]Q(?w~GRb֣Y۱N,;#8"UTčсjop)!#F)*f&b掍	b%4n&N<%#VA"N:~^	u4g|ȧU^t:n#8])hʜ4IR^<uxצ4h;|CȫUoL8rpcѦ;_^5.p~"!ef1YU]FӦ]펨6`ea>="IG}%ܕxt{?o_/NZ<gyi#8/+<3(;w+\>shrV\IT9tՆXocv:t}56Ku.$n?G|ŒY0{]<ëά:`Px4>PV2biyR-REm<E$#bX-#3a#3xAVbtj\R#3&Ȧ©ҋA:kTȢ-$#8AD 6ZV3#8geVsF6ZZFJ#8b#$My3#8F5ߐJEL4Kf"	;Emt)lcQ&э%ZaN)j-!ittYKk$-pADT(_#8ojXScJDD6tDZT${1QA2 <DTUNB(DaWzDuJF1(FgG	`sLN#8#8I[&O6Aiyȗkzx<%m%_;l;hh2.#tTiJvTJE]8|\{<~oDmQ3O=hƈ2\O ̐Gaɼ(HNnx;za=FFȈ~<m\fazKp6cL#(!]`BHr3?yN ;QE"|a`?hHi_oqsOKa\_<]s!>3[OƼRy/$l6f{|UI$"P@64Amw0ܡh̔#37eFjSI%*#3~<c{&כ8T0Dt3޾#8sx>ݎ{yq.q",CQ/b?˗M% W***++FF'ˏÍq;ZaVӷt2i:c<f	*X'ı9hb[$l\p٬)E7f@a0!ykEXf1C\eWVb412#*,Ww]3̼Di0FcPd;|AmbwЄ}"I5%̼c̎P]#.TVmu3+vDɩ:d:`ԀآtXF ã(JlY9NXkR6mAMՀp\J֖<yO6h0^VUF+5HAy8j#3P(xZ|] >#.&\x0d*ԄHz|R#.&EX:-ǋ#3v.]q{ЯmSXuQX	=س{SL+_rAMg5ߺh.m^zq*i(~M"*z>=[gVz2I	#8sfz8#\Y6aiUSR6`Ӎ#{u>$z"q20WH9FDa*9ǖ*fܲFt5aĄ/`+\̓(RAG-[ t#ۍi٫WJT"9m9:ARD5d$0[>o辝]v#NT	ʁs*7@p_v{#3qIvu̚	F#8FQAps,d)"W8M&BbM`~VcW[r2iq#86bn8sF5e2*FN"89p)x5I2:@g1QU;i,|~5r2!Pe֘q╏#.QPʞȱpC*Vs+|`6U97dAɎphՐ	b`Rn"eX>ƪ#3qcL}Nɥ$pgql+p+nf[X3"{uɺV&h>a3ߎHK^3XtAqgit5<v5bT)1z~b%BQ5*29죽ضx{R?Po6j b:A!=_-O&#8C	#3b4xhR/c.k_@Ǐ}?)7"CYFܛ*RpˌcՍ&طc Z>%c[#s3Fדp/]$FECvta4dFEAbk|^ڡێu_RO90i)@LpI}fڄ3kMc̨BvCV1\:l挄F!F\6+ypûD.b͝g#3Vgjݰa1#3>qL3Ȑȣt4-\/zmsF)Yٿl7x9VTr*L#3vg0b-υ	QLZ;#3()d9{˹-tmktQZ#3p6!Dy#v#fG!&Ux..^î+F%tb&D&cAqm+`*d1td;j4:1EjNSu1aLLTtى3Nb%"Äf'D!&Y)jkUc&Ύs`H<h)K9XFs܍X&+0PL$#3J#.b^k#8Qd:WJO|ĬK"l_g\4/Y"ÈƋRMRdI̧c:$hOʛ\fi{wWoIOgPCXoޢ2#8(-D_l0>2n%Woy|uy|])R_8y!D쒑\fH:a0BB`I.,WK:.{ϻXtM#34-i֖G$+z3>*Xؾ=iɧBz!DYqX#3D>_KOknu#8&IXp8%j.bTpAUJ{}ӥHIÒ'&")mA#8aH/4Mer54oE:8Ytg"BQS)#3:g7蛦*Jx8m4stagIǯ&keceh#38BjH޵`66ubEZuX/vgט#8\B+s/m3VB~f>^#8)(4#8QəB#.%XsLKvƣ>#8>3}Fdg>p]IOemjny߽*p~s+R,:(|yP'ե9۔H'9#.֯Zw(Omُvy".|Z1+3`A#8ܳ˂ߛ"HQ-(*S.?OTHf!PZؒX]wh~Zzx7$4=d~}爷=qHLIZV8oо%d<k)[I<L"#\5Rǡ9<Iۋ#.Gg?Cwq3SpA@615l;!J/pF[]&9#)U]vXp@u;vP$wqdR?-Fȥ%\2YpF\#8QU%uh&agC]1ȶ(xy>}=h@OA7Yr35VcwliTm:#8x"a$u=[qK x}}'I{h{GZn#3y>J#3=gUKC(|)*Bɨp,ÚtD(Du2I٭eϻG"|}_=JIAgń옇v	Sۜk𳂚'%Գrcx;<C"E-:yqK]EoR[lzP:#3NϖmcRI6q}\úgEA1Gk|a[RA@*#3l;#}Gz(%839y&Ʉ3ٵ)Mhg$M7#/IG`&bPxEm^6l#3t!^QK=tMaS9l7M(#4ZNH#`fgP)w!CghK*׿wr?#3AВ_Of#<v8(ևQst%B[cӫ#[ӝ6ӡS~>XI2]ɞN(b1k\uG_+	@6	wW^e擦Agظ*ZWKgΏfj?8߯.Ƀ4{#YF"#DvH7V}Y#3#.78eߕCgӿւ=)vP[)V2rbY4PW!}$3EDxzwJ>6ԍz=,A*a\_G&1O,񎶗yɚ'"iܕŽͽ߾I=N<<$ef#U[t[Bn\8ٸ?5<9Gǜe/AgQ*{Fe̸XhΕ5S3	F뤗/7G*ݒ}<= [:]_YgN`W/&gToE	b'\x8ݳ}F0ZEW)*nև'6u=h%ZW"QES}mlF,^Bg/n, ]5c'ceɎgR`mixvaبs'9U#8_]$jPRJ_DhKX!	8<Mh>'#3oU܅5zoǝ?2?=	(Bd귄^͂)&bXAS+y/<VM|<ף4V8;o%HD+nwA}'E,Y[ygڬ"aڂ0&(nٮ5*#3$](g.O)|z[#F3ldYmSmw-;ȎȻo6]Aney{C2!0۹AGkQ EB=%̴C!fȎU3e?6>>37+ce}ki`y-=B5/#8ǅv['"TABVZݾ7^X#3Cߞ`]\"y1- ҏ&:ݐ?~u͂xm/{ǚJϣʣɍyYW9"bhq/okIMhwcf^RdƗ،s{ޘ1YᩴOt=:?W~tN` ޅUH~6Yl$DQ]T9?ًBm3AW[V.9W%G_&6]hQ#3k'6%\[+LySD~8aز,!U⥓-;{xuZ5fY]'0R k0\XnVڐ9]P]6pqRlٖgE4gEuvsH~/2aus{:q}Ay5Tư׿ﲞ#3UdתT$+xs0s2Ejte_#GkqSa{vZC#.rx:6Dפav3\NzkvWtLE7:o=Kb7jߙ?Cfg}>,1.qloPy/eã<n690eYO*A`<oUc'	qH&n歩Z2ۖEpf}{Ti蹪3鬚&efwf@]TD|	(>O:>|YOŲZ'x!TEDQf,/de@ #8W\7<y>7%米ϖ:ʈUb?̿ܗ}ŗ㔨Zp(	F	6_%(lGcߎ/&Rzxnr<Iޢ.,1c%_z:<zz=갴q"eyy~gFZ2\LTw8-f?;rLh1eVu?EϚ}AHK#.nh5(|u	mܓc51ot@nr6?ԏ֦kJ>/TQp]9k|VPVx)9pSa#3x>^ɣc`+\G#`VTD7ghde'|dTSgFѪmtRoϪ\79ޢ{/*ϏS^uYB f-Su]DV`m<ci=<LKɎ\%T<M:T@ ZDPE[9Y*M4`RthjM/H'49S'@t[JTaR/Đl/=ӯlu	yqu}o[M&jp^RB"UCs%kB<P;*ipR%հq_0i*#3r?&ftg$Qjvўo[}m;¸BWkT"Ӌ>*d)q3|X~1x\K)N(vig~COj9o˾住oz|g89WW3Nw.ԛ٫axgNVWS;C(چKu^=nقhK!J.M87t|"<w(;<IĚcϞ0%'c(}m[H#8K=YS#.kSNj(w:wΨ$(ly-ѕ稆`_rrdJM+;B	H`ߢaqF{fQDPkUs%srSuC$-#e!ٻB"Bug<㻎vD_sH"|R^eLejҔx:arZ9z|,f揅N/A|nuˡYSr32#w𗗼jS;Ty&GUeIO(CJ|>Xtntŵ}'m`P}l>C@kiJbps3@4aUpy.B%0&f7_9]UwN8QyD/Ϸ}zـ߿򫦣U/.]ԭׂ2mrj8::{ʻ./ϐxt1\Yђ^*N\Qfi竱$.$(ҚePTyڭ^,0hu_+K*ݴRuTPҖm_ݨrEKޟk.|)<hg6^1$S+#8B3#31;~J#.r##37bs<LҬ4Qxo;H>~.H:{aպurj]RTߝQh:#.)qH#.ν0o){Aqhm)AƫlۋyUNꤾ3zWzthAQEp}ت|%;HZ![ou#ˏmiJr{l*G#8!(nFkGcֿEBs<8p>nRFPZֺ13-= Bh:vVcr8o9/cX4Iq#8s1uOq+wΑY:9mTJr{,s(aJDz6םTl"LN%~]g6(;xCDgxs;)ޮfȻ,Vʑ#3d4m%8ɳiWB+uWsGܰ(A5ԓձZިLSFʕ4R+w ۝,hs;!2TA6{/ntCTARgw-نZ&h2UU Wm/8?Vbs#{3^ĺ']RUU!OF4z)E+s3SQKR%#8ս-;ë<:#ER9&{.yCfrGZ-0t·&ڃH։ra`'lIڀ.f˙x|5	ZZW<;kjomØ׊#8U	C}ev}RL<M4%3|7²#89öC#.:a%[t.3Ň7!ܖgӢQJ8x>菝|	Ϲ]5UU	/DKrrY%Ӄ:ahE(.sm]91_]:t⺛(1G{"Uׯ'v/5~M;4F{XGby4Mv|_H:sý	ZfDů8u!+]S_Zej$,g1$b0FƋG7k|4|VbwQN<EI3ĞZTabq}:;	y\YǏhvQ#3ߤ\ΑJS˟oÎl5XbSIXv U҃)WG>~?Ig}7pG#8/7_.^o$(Jn;@Y!gP(+5i[35f}W)FdFꈔT\&=&+׾BysݠyD{^|V.$Rl&o_s1|1pZlQ#Z8H)*A~}elx57#TOs5))o,8:)9=wq5翯JbJ:6+?gvatT'j[+?qץsYkV9x/ƛ6%ĳˊw'´߷sSfiZ'n`G!L	z<I<m,ank9<j #8#8cs(QF"ET{*J#8Zd8UшvMpAVw)yw\[%x':ǵo2#3s۞Zt8EB'>~Ӈ^<샎_C&H8#.{&#.#8Mv8էP=K4VE(BuR0.`,a|'jz{ NUU;U9"k{y]0'0,s+^jPd9#8>U z@kJhن#8U]7?XOMZl94(yP3QpE*ʡtwa9~>WInlfTض}%	i\)"8!1;0.*ťDN:35F>=B	BݥVj5,0	B`i)_(՜[r7A#.?5yxaNsz#.p(S!=$~2A3hMa6YT?Rj̈k/;y:pe:qSTd`I1%zҿbR-I,:ȉh(Cҟuo[4}3y׳B_ݴO5cj}	9HIAb+ !LX5(尩Nj/}ǝvZ?ryC#3&#8WTb}(A97HGxHH(~7ǎ1(?U$#. '{+96*/؂ʦ#8ŤBd[Bn\TjHRʖ6|O}Nh(ua#3%#j^B;wbknތ5ϿߡХlzy1ϛ#-QRK)up냃Z#.|F,>3>~'(#8rm&Dަ(?t׫=0@_X m(1"B4-wM.;Vr@UVeX#.wa*x.%Zn6lrnq#.G~t+*qQڸB7#[sGX~`<pzK}SuN,5-mN,(0l-BmSF2/#8n{r4j'm"s!TS|4:f#3γ$Ȇb&CA"?q% eN<İ|P!lonvҒŐ>ɾ1<(>On~t<є$>>h%Gdm7_	ww/tϏZpvI5̸OdKpá4=Hй;:ri$->)]{PLtJu`H-0w/2tN`뀠*Ǟ6jTQ;a8I#.&opΙ_.Gxk>OQXij^ms+#8"]#89tp;׾V+I=%JY4"],(ԣԊy7>ϔY	#.}J:\)9#NA9fPkt{M 6N3OmŪc-#8X#3c4ݳD2ا#8U?Tq|t/.#8a E@\#Ѻ٢i^dg8p1!&#3VTyxbuAOXwISw#.B/(۷tYw4[s8 o#N_IIy pV:1Ԡ(1؂Ӟoz|lrVH@#JdCgY砽v$EM6dVMtj"E%d@=̤L0o{,o6ILUO9a\W!D(!	vPCt4D"e2^#.PDf#8	zP<bg_*)M.ZkٺpvFB{Vv^h$ͤ<%	"J3m2v8gۦS]Fjm#=2vNO~Hd39<czC	@b&r	z^WXn+|;-#3L,[*fk5l_("X\/#3}O FƜ7Lg|!DbV=vxӡY\N^?g8!۽f&d;qr[ۥgg[+Sߍ/ͣ#8nԘ&)ax<e-5hDDHe7.jj5rZp*wYKIMFJ3̪c"(#}r)U#3p3M2 VF6C^Ms^GE<EܮP!aúɴDn՞n1k1[+#8\E8UM(EnRpa,"A)zʀ[˿ˈi&\"#.r*=0}ie넯(iuExaZu+w9y2}NpȚSKdmUD&!(D%O2=&?5K-<JjTؕ#8oꃈpdsVBZ8Wұ?H#wktx>%+{|	HE9̱1{|ґLj#8,)2U^FhԘv2iM@~῍w͋JH{o3%{-U<8ksT#.EJP0_@[%.|EPa[pg%y#ˏ^?Or_d_	V^pSaxGx;eBxh	C Mq5i!PT,VucMՔ>LSXȔ#8G.6L/6EM)%=ihyv{0bI{-^)i@W((Q'S x5tJ gGbpwOE+ȃe2{"#[aa}a'W%kGAc&{8"mŊ-۩c9SH=*"x_%ID*F6 ܫ0PE%bJIQEGab)AV'`CB<*(5ॏ>k~?/Ƕ4ξ#8J#3y퓣\|sの7#쁲I֫B	OBꨊJߔLzޚz	_2JUOm)[ۭyvIHܞ!T=	$#3^yv=aAudDt>{Th>9Wi?_e*|}:z"EU e'0""JrSqNy>l;S=iJqByӫBjƬjWs\#2(H}T<Pʘ6@)QQ~Il~}91գcۭYϘv[#8H#8`.#8;=@*E#8x[q3AN~=I0m!3!y97O-CA;Ya9yblOn>._Ϫ|ͦwBa&(L{?{Esj97bW1);#87T#8TPkX)Gd往;%bqڔ2 d_<OM}#8Qy?fjة;-#.{JaMOfVxwJ>k|/&"()sGdQuEQD(2@B<؎A}ҔY`Eu#."j4tECbjRX̂%rأrR:	pjta~"l4.t#l8H#8c#8E҉FD3Cc !UhP:bOT^}`y0TҠY@2 	#3Ty!͗M+bBh(T{*h6w ؘϰ~Kf*Ѳ3%I*ۚ*8<"i5bOh<^:Nyg!& q^z2_àR4;i{o-|J0[HcR	 1Tr:8Ӳמ	Ϟmv\j{|`%aӖO6YGsV<#ƪŉq%o0esQ0ʪZen8wnhδvil2WwCgll)bvXY艙_TZJO"T+l,fо=mJA{98;K-T #Z4lm33J?8ܑ%UPF)<9㛄6<^23Kn1#.ܠ3L؆E$	xS 0UHnJ)YĴ~/N=>ng "H"sn2ܠnT7DI$4C~b`g(h&x Y6`X;6yAG"ۺhRm4qj@Az:DDA	qN+q9eɺ9Ocz]3h*=zwsz?"_CVzi"pYڞ{,)(]*Tۖ-\jbui+lmoʤ*Vlλu[}ɲRQ+F#80"rz|# #3-.pC{,ߦ#ÔZ#8_w#.ltHjy_ĂsG>fAi>>SI CZdFP<a7#3ò)5Q^58k`d^@K+Y/H3xIpAZꤥ023TkŊݰu|pi<T젉'S#l$o-F0CWDVѨMFv^}k_3vQekvͷO4߆mz'f#8<$!'@>5XdbɵOKw\D1.}=z:/ӌ#|h#RǤC~Μk9x>7	^;Ρr[1f5#8O1͔#.yq>x#^V뺂q89F1$AiJC!#bTWje,8#8Y z%#3;PaEUdaK'Iv>ED7xlS^'ݣ#3/GǢ;}EUDTȏO9/};Ur	y5ΙI*s(Tt_)#.{#8|\9Aݪj0iυ\VY/g7E#L>58Pd(ʹ'{Ʀ%0œ:\2*	!:r,kנK#3y0Sp|1.L&Ȭ#?vX؎A]`[P^q($U,&#.NQ'آi&HLʛcrG˽A{y}8!>Ik!"T}ݪm:.Y8C4Qүɨf[s	FdYէcۢg4F#.6=H>{,;+|3wt~.⮳XSM4hRW7:d%èa}|3mwi71-e.W]NM7E;41-	B%A-*,DDY7@x((Ɍ!]Lda~o9F:6{Z4oS&̬h!(8fwĳ8-Ȣz%)#.Y0UiEϦ݋nY&{㬾Nvj~Bum[Ϲ{,lòs8"_V/{#3c93~&#.N0[R_v/A}b]S"v.MGp|$U*DB0p#v=Rȱr`CN(^(b1%.K7!WmBooE]E"Ņny'0}'\CDv/Y|uPþd{!|u1|#NW#.uYMK_#31FuS`|EOf!\Bm{j0zÌxP#8U&^6inlTvJRa{ LhaBf)K-4CiX!C7g|:̳6F3%QHTv#3]{[qMCdv#8N96jUWH+U`O[FɌ2ĪXCwՊB&T}GBN7I$$	P{C2-/G#3F))_L"ϯf:#8)(޲Qƫsɻ#8ptmN)#r??BI#.>\#.7hĕsഥG]v ?~<JMJM_vώ>܆CȰ<ƃNݻ{AvB7$1|˰5Qo,rO<R(?s!IQ	\wU#.$Vb}uz,Cse~	΋۫Ҡ8#3|i.􎗫O_01xn	<w4-,>3E#3LO#3^?	-TfR$~E"3\uA t%B@>&gѼ/|ѹA:O?אּZ3+#.8&R#8#$RM@qpZ:ULNv9bi^[k!)۟3jKz9?9)AUk!!Vis|F#.#.Y#3d1  #.#.~'_:yB~35k_[x[*-w*ve±A",_?vs,KQfL7yOD[c#X?0w>Y9@~-gH)E9HL?.	VOWjǝ4J؋qG?od#3t\*H\?RLHb2#3)30	A*t%3s'zuh0,9MPn^)>>AH_PU#.{;t_FgokD k$#8TDr6	*{#3qQoѻ$}O=bdʇU5pEB6>~~UOuGkeq_jH#8$E	LC_/=d,w~^K!RE7D9.\3oO(=9?c*?uCUZUDEbrաR|rpu~ojɲ%x͙p)>-	ZG	G7Hɠr9iW@nhcv*3,)N&JQ %Եߚep1?;*|"$߼`#.rTAJ9HW:Eӻ/'i		!:Bo/}?0_`c3.#3n<'Sk5cWWl:uGNc#.9BD#8 .U)-v޿9BG0O{^XcO#Ҽie@[(Iǳ_1~38WlFyC"onW6UyB <vsJe!%s?P`_`%@ϫJmX*UgH6Xcj!Ģ @h?Gk٢C*O@x^Ν۷!ec\e5F n+ag:14,hVQg~ ÛEzgmG#85Cux>b$,P^g$6|ڭÆ_-a#3խI<T}ʮU9"e	f=üW90\fYN3=Weac OfL7'^UOHLRxI2#Hp}^/mYW'I5eU8=@><1OU&e* eDdFd@hc}17xZ+׆/9⥵#.Bx	g_A۷6ZڠK𚿞L1zdwS}`@c0nE%\TVX4h @b|pOQmAf#.HN_'eCÏ.|~#3N7益D٤ٖ4#8?֢loy;z1olģώ q#8p(BcIw\m8G_ #3h!0DQsЎrJqvRgϕ+g9ZGzHq	g{_#8$<k9lQff:{xHSetJf7oo?77$$zr)#ݺfjߌfKNÎ>MV<?is;S.猢bh+ŷENoQIsd`ZzP莋0R(kT:D#*psmGlXr;}=sinxgY|txW'PRa`WZPIoUqhB|<?=Ötݻs뾧dC|z/S-yf&s\~pOYnKC:$FAGL.G,MKlZzկ-}._nquj_i\L.-ߴQD#]sc^o$`rpJƪWB#SC088{۹(ｕwMF9yNf:]W%ʎH!7/刓UO;#5S'֯:?2W+#8o^y+Դ(>8#8<"#377!#8M$se$fr[r/T9zӨo=jyBWdU{C-<(G]ڊZk4(aT`m|s=sˉ׳l$g6&QW>V<}fiOBw.ӂbGB5'흨?4!<ڠpyUzLUq3kEYP#8q<\`}z[:R:f2K3/犊|+1#87K8{hqӬ8E]tX32v9E1~Ǻb/U':	#d)T@"%C#32j(* #(",=/%ﯶL8lS?ϝ3Ye-ocWZJ~x~vi{ojt׬#3&L@q۳5cs[fTK`:3H,T͉a@?g|wi쫇 x#8IPŅ,y4O\e<Iny<:4#yܭJ<NOV"%~#}1#3)N#3mslusarMy[Z?*-\uotpaZ䪌Hsԫms誑܃阢!_?̳||كow?Ƥ^qćU#8NӅX#3ǏCFdY!i8G?ZX^ShQ#3|x^MYW+6y&2>,M9#_ȱ`h79G<BaBc+J"aҳ]s2o*!<͊l0W5clri05xTI]:i䇒Nm)wo9.#80)tRffY{0x:wUB#8ZCw;aL8(ZL?heE.x;@Hb*};_l뺉O Ni6=xTouv|OhKx"ep+Z}De٥3qgL'J^<x}X;tgH Dl>#3#3nb.&\S/ﳑr7U I鲥XUd##8~S՟kRtaT9Ə6r(V4B2SVA'Jp#3bӵuQ9z}V룰|`B[T}S(Hol$iO$tFFi"zUj'TYQnGZ.ʬDLN[+",]C(_k\}-@'[	T#+>VaJjϨlf[LY)SA-͕GC5|;>h,g#ǅU/R0c$M8/j?b+0v@ywOa	9oRt,%^.}<|YΤc++&kU5B45RWE6at#;<HyB>9zܘb/h$s#.YN!Bx yfsv0~IXV,ZC	<=7|G6مB'?ׂj½E#	s$~N7ugtf΂ _-$w^SSDSW"sH*(?ăރH#.T60Fԩ3{	$!MVKRs?:z=>ZcȱM-w*Ja~oYrXPs-zG״HvVAa!gkF#*1"5	rc$D܈ otZ`\0`z(ı̱&N#.?`Nw9@w=mSw#.;r.࡜gpa.r%9ȐPUV6:#3$/&+H"Ax@,>LlJwwRPuO1g.7p(**K,_Z53 P#./&f!}.7-w?n9e+E?~B=0M	5Rn|uow87v-|8zEegwdy#>7MQtܕKU(=#8#B*w])Drog#Koms*ؖ%z%~%8FQÌ_upO>uxHG(R#8AX!;3uշǝS:2foNۍ"ĭ)Tq!Օ,eњ#8ʀ\P4Vԁj|56ЉPxfu2YzY۰tlͰ;b#3YѠ,m#0ƦHi:ꔪm2A<]9}[_t274=:<@K]`1!_[$޼}MCnйՒ b	yXq={9-6&9/y~ݬܬ =MBh:oA0wdOd?B{F2VK[y'YJĠQ%h怰(c;z/tF|ӱ][)]STϖÖ\pWf׫bSNGVj]3o'R[8g!G>8΅Ҹr"akVc@DG23zAuT{}X#aϷGm~_N^|4r7&I=jYbb{N	~*M&)N<#L֝}l`8TX>ؠ)OB#8%֮:A1:mH,]1Pǉo'>EFhNfeF_A+i@%M^-^;8H94oڳ:ùKJQ708:/x{59lcHfЪł5,NQB/NJ5%%X?!oQF`G9x{U"*"bʪ#X	Ex[n+7\{o8lRJ̜^~osMOB'@C#8ʰIȇ3{a_JarM`f#7#8R+=z9"{Q̤fF߱t*ؼȆ5R)YK_epy~wN6ӷD:guHVG;<Y:C='>(zAvN;!1p"vL:$P:0?#3dmWqaXgLpCk殚7	yw;4ⵇF7G58f^[e ]Ұzu/Y"U'υ9@k0#.Pq9TX͞bZ,g鵂V@x%8yJUv'Uڤ{@&_IHU#.;V`	#8xyITrh^V*<	BI'\#.Dn4hr4l ܪx#8tn+pF@4_֎4WaF*%@!謹߀Jcll0$#.8-300"xָ&/m3ytloY^+:#8z Z9\=gS'L4GV]4Pu(dfHn}X#8@ROv*Ew-jܻY͛6kÜН;.9k{*=@bU=pQ[S\Pp<f)A&3D~$]j_3|wߖ0=%eJ\96T#8P/"ѣzP\jd7i;A+_ZlR¶GN֬kPCk8_]`YV}s[|nas4y2:Ajn=h]Fko#8]jJ#.P+@p jA}()YCV0ҮQGv,-^mA'?E{հC/w8I;]m#3~+ȗIA3{_yA/7:	<x4^eE#8344<nt΃(r:|F0I^3#8#8nC5QBݠ[ѿ@/ų% y#8vQf	Z!Y]G/dY杒8#.	UX}p@r@L\,*.'{KEPUӢt:':V_i`#3KXjzr^#8^9t;7a7Cuu9#'jٕH9wffvqryޕ@;|uŬ6ܶsd 5L\+5E`5G(Ĉ+jMמGA,*RV֎uŴ^[z'.;e~1$'fǉݼcżm(D<}oWn,ƼV*w"4U}x-d#8	|قOoٿόoe>pv<c}CLRYF=[p҄^Iػ#.Z*m)qq5+E[ʾBf8P_lŧ61ۙ(UDUbb,,Ni5igUH0/P[vtR |4^9gOzr)5p#3{bNdͦ;#.g>G۹x/y/=G##rqÁ»[KR~wWԺb!4ރ(e Uoˀ;p.R#3	M_ߣ>AgӜ\G2NA|5t(	#k=0UZ}!eֿLT25ÄG$T+t|o,T,}gh'j<nsӐ>3ڼt#3`)Պ7⼝t߂{wJa@`}h~{UՄra֩9r;yQ&vStgpU}0#.<}#.jPU)S0e}1[@~E,dX 1L!MUn75$d_<@TgӸ^!d˒2#37[]^OrQ_?*P0!p-#746<ys<ͩ==#37x2,ġTt&/TBԣ#3CilZ?l#.=euxh08==Gvld_#.w;,#ܹKpqrk~mRF 5k䐻mEm(ӯsLOɒvS˶K.ExUzӎYT-uBEX|˒HFLʯW#8_pޖ+#.D3%Dqhw]y?pde-I#Gwn[#.L.\BA%Uubn\I#3/-D+eNNn(Z2[#3<'#3C˳Zf YtXI#w#.#3m5N$C~hZQt+#8tdiOwD$/覫m@$|vԏ{tmܡhM#3mQ#3癸~Å<,7RH_Ъ#3$a)!.Ҕ`B-u]̄^xlx8MBw}$d(CubO#8`w?zX#3~A{}1PU#3v[PHE #.O#. ea$I6v#8/!߆%'-=JO`q:#3h4:62`)rj9e&?DݗR$_>?2aPU@a(BR=CM!=Z._a7Kv DBc[o4t[ϫkz)A`ĄXL'n8F=NMIZLsvyB~ԅt*	*D	t:|#3w<T&vl2͇l}e*!/$R5=9)Tk]t+:i*Y%vqq#8lG˃ӈٯmzSFA=x9Cymh6<5efs(o>\=A~cꄝx??s6<ӑι1430mXĸn~B6-/Im>wx삇竲s}lI~`	$_T|m3̈N4-V(1ٵ,zD3y&ֳ`A`ioKNj"#Lg9;KX<s\>@ćLȆb:Ccw6;<H$JEzv)ޟ N,RZ@!<'pf ═jΖv#3!ntV~:\~hኆ܍YHSM&ơH( *9l((L1؋MDgy/#3cL@XlBd3vÏj/ dcfI#.23zJr.3I$G2>>4#.ڇ_kZ]&xXZ)?m+K#8{%%&L"EOfeGm)?mҭMmkfdODq]swtvm2evi#3dG^I"7Cmb3ZǫK[O3W5e޴00MB+EbWur|%ë:ƍm4sJz~#84ؐ#.u@o㚖42*hV)`ݘsxt!;ҽ556xH+UhXvUA-DCT8uKChKf㚀o!! fw{}o<:\QBPAHI߈"xc0o&I" Ľ6%lѢvÎrhv3]up8x#3Ruv_L̼J'ءHm:v7o;oL%<>&u/KᐛoY}N{"HS#3LܿZ:f+1d	#To.o`>ilS2r1wbFTG~J1_}u+1mn%r0c9tqTRR"2lCHw_4f`F@0bR!K;)2?g7j()aX8tJ_9+@,@KFB@Y%1KFA<#.Fm<٬4r}w#.HlDC#.,{x#3I$#.D?OxϏh#gei)'}>-EKQZ nD:>싖~=P}_.3#ZI$&l\(>[v	 H}9>G#.8Ì"4o?ia#3Np#3B'C E #3$H o|bfm<H_erxh6='Ӟ3:xE6MC,#8) D=h=b)ԷV#.2(DD<8<yZzI9|_`Y#8Y)I%ߖקGElkFPyW-jL1	V#$OrS>ցlah=~֯Hi#8vhu[S^E%	6FnTqxØv6`|(^=%ap6ӣ`D]2;+}.ID7U=0;4M,OjX:Pwx󛙑GY#84ʚj[0ܩVD/Uj"4G&@U?zh.d6דhpdrJTN;;(qi3<Dc *RiϛFdŮnF=vWPfr&s3 E^#'ɞ0Ȓ"Ϭ([ez;A]cD;DjP]Jg>s	O0:}y`v'w35"#n;R<0%4ʩEy|>?>QuaK lA`D#8תō(7 ,J&)JƇ{NC=~죱#8r@)i6[''"W+%ZĎP1W0Q"}Op9}G#3MOEq	!P>x/˒.a7\ӶIz#~zIЯ#3)w]iWvW䅑>{#3T:L(R!(->/1QVfKϗ&S8jza+3`L}u2KrFmb\~(Į25$ *.[#3C)	``C`.hP!#׹+jV|6#G)@:&gӐ#8Q:&CcRx`6iB,Gz/UTX,Tda#.(ݵ=ASnU46V,T";[%0b|~k-T!kOO/i^^7Ǔ}w#3̣ؔJntBDͷ]y]X)YD䞟3982]O0]tBפa<QЂdc]Sd`4jGRY-Uu_IkjޜjC%HuCq!	VA<Ğ@v׏7w4z29^vOPCn=pf;7uqξ03qC#.N C;#8#8b@1qEy:#N[11Kqq-4=e{0쏸*yRnjEP?_P̕R=_U(fa˳M'H!Zdɗ`H^-b>5/G8G	p٭}-hf	fȗ]y.tRG;#8#Y9!|?z BE]T`eBSf=,??3Czcw\JQ#8C(?{<>E:-)ƺn0o$by	mUTStla̿G _URC"??;q#3r bTP|#8UZ*!c8Pd%'"B:('v#8EL¡4CaG2Ta-ڊ#8Kݥ @`"p|}.:D>A]ݯf@tH,@Kx]ߡ \z#.?;Ϯ{ͼ@/p}ͰT7#.K@i]Oa%:'ջ`A<H#.Wc=6)zo]8;D]R",7ßJ<mtz8£67`#.*J@5FزG˙~ʒ{&h+X7LSHI:J@دm0HbOqFᒫ֬Q~̅(O9qqw#8:K˵T~mҾ{KVjXv8n*.e9e{ť1痘y9y6J^%;Ș )N$*}d#IH%q˯G yǮ	^6=toU$!N*[h#8g{rB@sT4	i&āP x \O'C:;rK80H%VEU-{ĝZlF`n PW6\#32S=[ǢJ?KJT0Q'w|#l`/l$]ߘI(oi4Xx_$^UA`n9?Y10_S{An~Sq'_ #85EAåC=AAmg'ն~_K>ef9j3_ͻ`u93*|+|Wo;ڝ@\t۩p%$D7dizf,P7R,尸7N^pLH4hp!	` t#0USph8@c֊`#.#3:}闁K^2B#. 6#36NlbAIOp~,KL)X#8!G[ΊXk!pS#.A8gsߘQjȂ9^t}^cgjriE_K#`۬+$I!b\#8OSR9\qV7t]LUY#31}0ɒZ^31;Ġk#.ВNZײQ|n:$FGi?&%D=>CWe>;B'0hq@%)rxOY:r9#31qx TtT$$MvYu~}^<KD&#F17!$'#345ZFpJg'gԔ (,mN%3MT}amc$G?'cyܯ>~5	Mhu='J'&	z~G)M[#8k@ZHnroL,|^t:hZRG_	&?O5þJ=sDJf董?	s|QnyuOR:SYCl:կVfkKޛ=11ɰXP%hrԞ'?AwP@, Yʰʁ Xt#3X	g_oY"&㥬:)Y}h	!7=gsކ e(qݍbT8&dLAqCQ=<UHpnTn?,(o-ΚYk]3-`We.EEF	L2VŰa#8a0a5<Τv\T.#8:5q2EvbmO;Ԅ(f|6A%uxQ+F%D[\$Ɨlg`?s5څ"c30ɒ#8Hc-AQ񢷩j?V4{,iAaO&l]Y7=_ay>7j:2X $/R+ QŒN7j*G?s9t'R ]0(=V#3#8KE-CncSnnQ~_	99Oy݇軸P.k;}|_]o굷#3:"bD!~kOI!N_pu	uo3QfՕcYPea+P* -@'ME9	ۙqFcqC!([~PHMþlnEcu&<gvuTʺ$<NC7f_C7o8+QHt3h#8F)Cc|o:媏7Y{eg9n0w]ӵK	KDMn]19,C8Ɯ"oAUh7ԋiE!s*f )wv/ss6LH=<]!@u]DkaAٿQΏz"#8@<0O@#rѾOgyPg^?=^4м>v#3?7eЌKت:1 `P	C|yEtwzeڊp7=@.!6 &A;z#.HOv8#8.eP#3v]r6<i >:x:aϓ8Næ#8pׁs<PɆ#3GG{+6&5쀞.۩;EGS#.sVxCMuhss\uUg_{[o#>Cy7OxD'o!{ꄰ{#8QG}%woir縎NvEߑFpZUX(꺩T#3YvMrXzo8uos,+70̈M %IGc5#.74h^44*Ԏ#3??7Q='#8dُZ5KoTx	@:ـ#8_7ظ9N&׳5L/g|^.0llZZ##8/WwRo'`|	@rWsmOt:ms=ݤn|Ȧ郡#3}Ӻi*Q`U!':xyjZ^bY0yxq$c_#xlߵP.KYVB#.f+a00Y#.| El<pHI[B|7?VqRw?`z?,sxZXG]߆)'c#<F6#(ҟxTUo{|/>zr=}=؏5;&{DJz<*Ʊ#8@#8#.*ֹ]Xw{/#3U#.0(+Yp";m[06{T)DbFs0v0Ohx:}?ZtBjTwv.s]`#3okm?ZvI#8Y҇|%dAs:Ռ7:XT7L]P>ey&a!Wflҷ׾)9Fe9*ήK#K΁-*m5%#3U	{P7dL>Wy*BDZVfd'd.%麗	C Z)ӏ+?fq3v<#8\%<n( Ļįo7GN:3nfڐ7CCƂǥMvs`UӫF6,FAEV	xQI2I/UP-(JC@6$$EZO#oiڞ9fMH-zBY;(2>>pOm>̯d~x#!^}hkM#2w{D:4)11? ?iĢ?gdP#3AXك8?/n!=~=Iw1(lMo{ 1?7.$NeoN쎳`n?#3zH%-P43)H۴	4{gSTsaĒ4>(a0l`Yny[y7-p\ĸW;Zo*0;ŇK$#.yxoyp_m(IDdb'Q#3f84`VD:Xqo#y6q|8u0ٌNJ#8pa,h eGStU|32n<_B5\XzgvM'ArJ"9+X9HwVhN,u~|ÄziP#3!i4#H Dj`8rl2V4\݄}aB°%^y4Xjy0j^ޟ^{Q2V(9d羯;.>gP(	kݯ/1!N\Zֹo=ΖP<h7W=}@	UP2hS"f!bjJdDaD2	JWJkw-ĖqSĲE7yAXԃF֕D0~{OT!QnϪI;M#{ܠC;p@-	L?]zt5UDyp5Q%N\¢KKcAjhYRD6(kaq	P/n+mK{'|\g!5Pr6Pf4#.Fr6>Gp6Knr+hi\Ť``D0O5q=#eϧ>04?E|<u#.x$c`;wmϼZlo#8[}Ymyq"Q!yk7[ CZY6oޖV7'Xb24>]ZD94#13*$]o=07FV1`s󼳴uA-kCyr(;o63ǧa8Z[q/xjî}.68jۻ.F^;&QŅ[ݷ6Nz)l*2/D%SeX6*f|Id͓DjF'Qw@uM=}EDD0AIl.ն{ERGHi3!}<'MjM])%DRN%#3:F8{OcaPfkS~F)0H<@ނBg#.<D݃`߬-k5Ɖ6Fg:ҧ(q~g">&@#3hG;af UpRH%{@	l<*%iM)#3a!M,%HɼzSA&?{?<a{>]t#y1Cap0LIķNpmD0A₫8C2#3Ȯٱ[?V#8oi)DơJ<)EvfbůCؼ['qB+&-,p&&ɝ+՝᜘٢ES	7O/#.bf~r6sP,R@7ްIYnt@h\ãbB]vhǷ2{`,a<y#F!rl(jlJ1|(xN8J9+"@Jh52I9}wi9yM:uÎF8{||k2b3.<ɗ332ٙY3$cg^#m{Cb	6mT"=o:"X5OY+͝؋ѓG˃1T:k醄 LB<ːƶ#3Ο]S7\CX53=FChYCS8AsFn3>=NHҺU-VR v#3!UǬ[w1K%Ҽ;\7h#wdQjFj\Bř$KHƶX.MgTFTC)nQF%HuNΐTU~y	Rq>flfT69UC4['"^bo?NƵyUd~ݺy'Dawpmzƛٲ38-"רc([-p,sT[!0Q37gSr&ېդAHH[YrYbƘZ᧎УnRS(/Cə}Uj2L%EV:6)R45w|1C#8ߧsD*!)6oT$)9Lg=N346d]&i-QHB2wd=vKHM#3P8aVLa8&T;	oni]5QC} T!LDCO85u{A6Lxӹ,PTv3iCw1Ȇt.qE#3:f#,7FAA41<E@$@!t5c۔%ɭGRD2#`LFZkZCKo=Q46u#8n~jVFˎ4Rw|jq("T!}0<b22(l3K&!:.Vfw#3+k@NÈ/keGf 4#3݈p(U49FTarq~OذSoVD^ϟ*-3c/k_fk#3s`h,Fﴮyˍ3d#,L;~$d`ܷ"T#X($}13x<qAE>=SH	C:f)"nvnq"J	PiS WniRj.	䃷/cc: ь;1s^V+cPk9oo}ݹZ#8Vha{QoVOBӶ=wo/@\`.EK!<"G#8#F$ wʢPsY#8#8eA1xpM؆''iAH{w!>E(+	#.}vLfl[S[򏗈#3PCȾӬ#.N(iI,MYZimq2[4̥W*>H<H ,Ҵ++'yTV||UQT]zc byBBJZޒ+ #3ڡP#.h|#3:s04#.5TM	ޝ)x2IzEI ḁ#.- * 2"DkpT! 6TKhAP7:y52AP>/F d&F}Zu) ;}hKi66- #8l73vt58VFTyea58Hq9u(1Xv:*}#8(06@H@!?:]c)d<ơ	jޚd#31|Gk^e`iޢUT[3p0Q0M:Q|E#8HB>w£UXգZ-lU d@QS.O2=I=rܧE?j$C63em^):#3a)` LpvGQfϚԤ5{wo^n\wvK˻+6\ΎFA0MƘΐ~aĥv)!0nDj{+ha#3yyj=A=$.L,F+>!Ȟ/?ɪ)$Nx%#32:;zba<3A5"դK9#.R:f>~Gǧ'Ӡ&`,:Ir=hD}\&짋h$#82#1	בzd%SS{U}s0xza&ŀSU0(ldXZRLP#.r(Mpt0SK#8KaslAL 胁v7{Ҥ#8 Lx/z"g#3aa(UX;'o#8ybE|Cǁߧ?&?3?li,VAlxwOy>ɂV	29ZxB#.K*/b/RUJDi;{QEz4<ܕN2#8U$j (b1+RuF*T9Τ#.bd=?)RPOu9ic	#!sZl7}n>+1ʩQQA"xyv#3QOGj#812J$#8ŦZilX%VLͶmLI}'+}r[|dDH22#8B*== {Qҁ֭{b*⅂a6w0V\MTs3$ \i2JrV!S$:3LB<(OeGkX%b٢miMRVJN `QEЬ&jf"ީ@' utãULޘ"n<&"za$5A8t#.JSh.ͻ(^@#.{s@bd@Y[k%3jەҙS(&#8NiYLPVYS0W#3p|C矼yiJtmMs!$A:WE;S	%h#.ìG?gx}^__EN# O;>V"!I&sbB	s1#8)DB0#8 a7w}},dHIbh^R_G2&'TKmeц3P;я:gegW{7Dr7(%C.3ݏFah4SdjT- ?a#(M.!&#1ShTO:Ξ;V#@nCjPe"םPU#[<_~rcEP(]:ʹD/;&剀&Ě˨$*Rv jN!>iu-8gpؠKь?=§DGoo AIqXԅWfJC2	ŷpol@ei)=JAr_1]PC$3C\1HX@3#.OA#.A@S	YdPyXPN\vosR8;=#83((,#89>HsCJsvVb!?O d)U*"#3P_t= sj*Q~uH/W4Gf(TV'vh^,,FطQZc@ DS:CjPo0u0G30}Y]Ei-6	Yn}z-w|}H(~}G+Y#8	2#.XJR_*lcӎk䙠xzחͻ7,b̦?J^:zA9V8ͽN`[/?&F@k,s8$%+	]n5#3j+1V`Yh~چ6P%]\N'Kk*qޣej2pVzqv϶2:c=ty`gۡU\9w\6p.z5VZtO1mxY4rzQ?"<N9bu߮x,7t4||19׉c2rK&4Z֡\b1,֊5P~#34:w/P#8QU!QJW~!VZSzZ!dx#8U|Tn~;'zn٦FZuxF	63,m4tœz#8-BMv;C{9w(`Mmj5>efU#.mCR<{9^狳q\H]pMh].;?~HgdDcl3pl1:y<!@`!x$P(Ѡ5K#31~vj%M	뫞f槠EhDBaWIjzU߳$@B'Il{r_/@KȲYN((hg@{Li__A20XLM	RK#.B_U/d*umEI %t]:nʬ.,td{oPȥzT/;`Iɮ݊0(pЩ-UYuRnR͒Ү]M&Ziv/9n0Y,Ec PErnDp#3etX#..7$#3Kb֙FQoo#3=nP(#(d87e>Pӯz<JTA@KdmJԓcA>AD|'}wtnU#. ɋFySJvU`#3,LԒp\<,:-pQ:0wc* oEo26w4*KQ43-?>+f#.A=PTD*צ9n=`<ПvN޺(>yū#8 #8aCiȔDJ#NFV)"ZK#8BFO(6Fi~P>/PHáF"H5E5*"He"٨1;Ƌp'u_)YmWSk/uہN,._P~5FInLM`f޵g!/6 @Y+&^˥껓:ts.tK_m28%޲mHŏ#.5HCpdt/WhVlw}=^Gosq#._i9Uw^:wG'6Q?տMk"#8Ct@ytLvW^.4.Vt3ˠ-ib6%#3hsf#.:3úBG![ơ-	waI)Rp;֊ܹ8)Ͽonӳ-qa	0q8Xb1O;g*k}35JM3$[.xf1Via9t?&E#	!#33 2Ӱw1-  (@kkOKf@Ȑ%2'J!cܯBfYGUdf(`#*H)FQް#8a#8FazD2'&匐A`h!bi`(oIhSak}n+^uը#.Rz[<gIxaK~#3:AiQK:o{vGf,TQ6I=.8QTW髞V~Gc$$1>p#I/I	]\79`,"o`تT儰`wz:w#.xrhk1.Y1]vuHHJ#3)acYC-&T#3g1v*v#[Oh Hxr2@0$Ĩ#.Σmmۻ^i}=	 Ϡ,`KŔ|v~QL#8@:, w&#8[%܆k]q8B\	q#.Ȕd[b6Aua]R&#.Gg1k$#l4Tq+Qm6M4*0Yԝ#8VqL1٣#.-?{CM%y`f#.w#3Rl%+THB#.L᧬;:ѮQ_6,Ham&Ӄ:>Mt&0!#4S90b3@(|un C.<'r'0\qnQgb{#3>$z6H\md7TF&F)+H:h^ bRJ4(e:1}nLkl)_Qش#3n%[c"#3@q`^z,%5cQ*Xa ҳ?ذ#3D/&)!kTW0hB-0llpW5Vw!.%LLЈSfXr8zLt$Ш=/-?$.)RK x&4ܨ7\_*2o_6ڍTͻsClFмR* mo-bܕJvY)#3Ĝ۾V⩾YJ9#8包,ԭXd_\jj j00\x!zZlS2 j2n(d$a6IG4&fý)!mʱ3mt`7 1Y6aVI#3#3ڷaѶL2[g^2ҎhI35'0ηyÿ>g#9]+؅S"yX<<x~L0	Gtmư@,ev+	9B+Z!6m/satiwx&o16zaim:#3;ˤ2Aʫn9XF[×fu_BAl͖;H	(떯S,f~pS>O㱶:|N>FXs+u5&4mb	x]ob<@-= gO7~,=CJ#32v,[D}5;H$7l xbsd@@ptYBF&@lH	 YÙ:(@79GN!blV7#.#87͸?"ġE0!gU<G1v)yϡgovSy,St#$!ioԋ	?UiIsϋkg$(=<CdDDI@.<Zň[FFQUklUmF֮knJ(2#3lGy:32+TJpz/H@ޖPJ}p	Wk#8fzOtJ"Q"FA)2S4(M(5LkH`Ŧ(d#8J(d1de2YLB*biKuTl&E$2QT(c5)*BfIR1`҂H,GvvA'hZ<	H(rt!鮛!mCHHR5#8|Z#334Tyn@wbBf,H[67#Ӵ;w1/ݸ&\(nf|$:7(m@RĝV3C2~dc_ů_]a`2@vxKw5]u#.QLQY+bLC	Eܹqaz-TͅEo]Gc=#8EhoTvFT9`FY|%{~E#peiO<=2%©EWBA_ˠoL0=m0{ݩ_kH`A%D#3=c<:Y1Giç!C853>;"eDso|%	7VM15fImݩ2^ܥEAK"TX/"p ;m},/㞜~kqspaPAJ4kT,J	t@B~Nwy%C#\#/@}Ky^	uh>)&MR?EOR?';-%+]F$pL5ZH$e	xU_ҎO%B$):q]-C;hm"ϭ(n_Svw(8@vON>-('iV5u^3@KH/F]x;{Iz5I:/@<|#.%]^EO#82}a!c,o[U=(u9nn3T<{%@4R盖m7+pC=D75J<@]#8:BYϖ}|FrE&uN^:"r?9DC8#Ȯ(}Ruzhd=>~GTfn~cM2&##x֤c#yr#8rLѵG8jĤS3Z?Z#Us2LZsemr1w4-Hih&*j]Zk2FX[s$ VHb"49#3 ?UH/QDb`>7HT#~ȭaqir2A$1{8.*n=DDE:6kL)K={p&3SZjA_/w\lьjF4X$`#8E	'"G]Ǘ^LOz8}bړ;Dcb2v.cړL XD#3";&Fy]	59XAлgV^#8̅sà0"mSR8n-'zQ	a6~x#3v(9և8Z}hZ3 ~#.EJ4\PQW`}c<^n_8AD.G0Ǉ,YqrَzɆM8Ssg/FZMDyrOO$ߍ;qxYz]DJ<F|O;*fBI#8aUo8=wg.y-v?3x:IJKX[MN:Y,8$׏3f>g9͖(wK¸;uZwۜmc##3id̼<#31e*[2[,2Ѝc[0~8XJ4٤)C ւ&qd#.֫$ruP`Q9jv?jVd&B&p▁*x5XX:>ȀHHL556mr~L-}K:o#3ئ#.r|uV!L7I"dr`w/AIoٷ7DJӸtGHv!t̿&NKCQfB-a`}B#"I"6r}9Iq-D;!ޛuh*pW츈PN8s=	nS:52Nhe!yzӗ4lhM2T\I%%)vly*k0gSm9wϢ0ܰj	a\Z*G pڽQϵQs*!YmFo%5.!#3t=FV5VߍL=nO*]cYViy]*.wte$.:j.͎یLMc9|Ul"݈jD=ٶ0u4:&쎆oqIf=VoYf3Yiln~W^砪E20aQ;);*O(*N׆*t1Vm-[H"3Vj`;-Mm}B=2A1Su`6i㔓(6fqu{ӎBt1%U'cn	[&kFLbt5PlJ#3,]+LxLM#&O	bXi:<[{#3,0S3=_GZɒq%QW/p;0m0eE	7KD^eK!I#3 #3ZKjsEԯgr0*9*	b0$Vgl]iѶ*MSb{XhH#mxVõ><xhy]HU#V#.o؃SāŶXRIKES(oC::EDm	ůC:]q#.v v;yqЃ#.؉9 pѨO&Ѱ#8DPKcQƑYѢQAvtaX6dAT6aOg	LanɗS4)A4*B)#3uVүZ0!o6T0[%XJ90Rfy,^0d4e3vY4I1[ISݨ;;tOZΕbx]Gyx\By)]aGV#34LL*&T𙉂m"S$Ş#.ˮ&_p8O-赎ZoWcO×U9+M@4FsD2c@-s2ѓRhqVL2q#3=^&[!rV.#3,j1qЎzLLmÈLH4㩇kM&JkV"U\=t&d.39ESoM2S7<BDumEJHH00ZT)h*SÄCۅLM!j̜49_MlV:ƍdwHMAR08s]M֦V\A	z$b_x\#3rhDer%r#3LLhL\d`۳b˔!bC3\j# l%b	\Hu9E"F8\I@:"<HFٍB͓VQФ!B@Fzqd.&C1c4.ܩ6X.#.2&!RX8&Ej4k#35C3RG#.P7L#8v.ft#3Ss#3HcInܤJ]PCA`.˵·r[#3	<*u3*B2e%֯[iXh'Z"nA)J#.#3w-6%%I#3c#eoUF#886jS09ZRoH}:gjb#.hi 4eB:H;>~6}ò@rvҪL>Y>OҴO˶F鏭q)%udYMӒ:Ìvb髁F*]cWI8vLM2K#.n݄d`Vzk'3fUЄ2T9&;TNW#.[byJ5)zK<R$IާЊqR$0Gl@B#8ȼ<;,7TI`:BhFUV1ӗ$#3L1q#3d!XW""#8]D4OT-#.(*يwf\t_҅KHjz w"}c̖JO:'!IL"<M4!3t_qˮ!ȀU'𫸧>JsSL˿5R;ܟ_fE|gs0nCS]ؚK7R#K"DNlHmOKK(^_ϱrRQ#8m@qf+@G2LjM)Bu0#.CT,i*0@u~XpX.LOyA$Qbkѭi#326AOod6V+O2Ltvw>*xBF16	"6I7HBMBA Cld<^'ݰ6Rx-}.xݓ;ݥ"rLU"GKRObsc*Y#349VEDB,	u(.kpDmqu)#-&RJ]+}V^ES+xҘٔ#3h#8#8ᙢG?)61Wr+7#8e؁O#821$UZ	FLW饽afc!uV1#3|yѹbb1qWN8RɭtlDNC"^ $騆3@j+2@rg!v_>@<'}$XC1$VR$bKc#3]p\im`P	xl#.rIdAAݏ;;Df/Pe&ARM#37MppkqJ6-̮Ƴ[-WBb]4ӎ,[Bѵb8"a-'C:Py B}Pfdı%@B#3bW(kF#$܌In6+#8ae;ǵ8I>.RFDL#l6[F"؛#8#8K7||hAg=件-dMN6;ecB.P@CPpHBHX)(nbC#.j*Yh		S#8S(ʐ0	؈B($!dG,bt`mq{6[00+@8@B<uK>]>D&}4E`VA ~y@}C ?(|p;'4GA>4׵14%S?&	w-mx6+[a##.V!!z3CIS9XXh#8co=}E/XD5pT+ÕbW"F'pxڙy4du*]ydY#8N(0壊ךm6388:m)]mQ^$)HtDI/.25ĦJ]8<K"nE%:>F3c3ޕ8FĆq&b܋-7	D4aJn"01C$3g}$̧Lou1UODnLOiax}5Ja/>lY-ҶじŹGc!^jot50arpQ>.	Af#./Vr!HȌ`S0AloYGNz2{<.Wമ=++`6(Đࠌ"@$#.,#8TGt!- Z\eCLpEaFD4DUc]ݫvΦvreNu˸뮸8R[ZGWɒםLvI2#.R͊B$"S2R$pJ)lԁY!)DXHmt55S	`4lHAQI# #9 [a6A;#.$LQ7;@DePC( C|GD=g{3:)a؃e3͟oݘxp>yBq<JE|-A`5Ji3÷=30fbiAt"FoŤGojIiӰ#3_gH'#.v7*+QZZdPP4feHThgG8<tdS3ۓ=5M,St"֨Q#8RrFSaht!O;yv q#3q3jŗ.Mc:T8/4^t(-4FσmC%qe1]v4i/QvI3Bæ.+--Cb*=By	ֵ:7^ؾE.a4g/HE AW!]qT~M	1?ȿb9V&V6Vש{Ҩ!Ei,Uת;qTD&FMD-bkMbR1DSL*iLRj1nKVo]iHmkT٢s6RB_YvWq*b0ffզIiՈ61m_k6$TTSjZjV4f&yxSI6MfTµͤmlxۣiuurJsuf77lvǋБk@`FHoڳN|E?qZXG8~g;zvudh&1v:vE	G[-5@)C5~29C=iTSiii+F) '׭$$F$#8DVD()i9J;yMJt@H	vٸHl#QEdDRlŖ3*MW⴮$RԥiS-dCJhJU[0Dj+6Ĵ)1RXm*6RDԙel##3mScdȖFɭM**RLkcKQIjCH(iR$ŰIBiI4RY1$"cj)bel$Y,Li4+  %P+6HK}%]!,S])@TEDM"_5F̭ckhڔnUiJmhھu{B3M(m	od#3c7Ml_`^N-YJjJy&9FoϺ$?fz/r"zop|uT$1ca><&8vFҡe{Y躪:v7ЖZqۘXVaOV-\qI>fQF"@2W|2i}/ `R0;#.>$#.m|l6}W	L?njܪ!}{S2Ͻ"8YeiOkc"C	Uߡfy6zRˉl"2x~7dDs2s+:"ZSLл\LvI`Q}5e+qJWg3s.ћ0C&G#88lR|!R#3o{cjQB7J5d{9M]qJk#3,K R]_SU/ݷiFi #.cm6kWur}D+#3Kzvt9uQ6Wtfd(уD@I`&ǁJ'W EC9)#39xPRcVqOd÷ۻ#.9Nor8zpz+z?l4&#.!$zK(U]wi!	gy>[#Jn-hɘi&Y"3Բ01C DiWfI3UFmzo$gTF':$EXdj#3#3Tg6&Y^e)0) lHL*Z<,0?#8a)V%p#.4˨wdP$~HWJ֫Zб؂w-}F:qHѪJV<rC"ԊmA+-gC1$\fa)v#3#8X^(AX;N;Chծb$>`X\+PlQ7ci]`v3޵!j7ag-#3]:2IL¼dZ|]ETϦf|9Ww2H>qTҀY4b)M\h͢+*e@`;r?8HU: #mƠlŬZi+m&C˾*Ŗ>`d{[ǢdMZ'R#8NKP)\背/L\@NEJj5e4[|d,!4CqD))<4{$(V92Sgn;TvCEShn\6HQ#3xOEyX#366OPS#7(ZF(DE2Pj56ks}+CMϮAQAq:si/{#8^NO*V'˜|@#3$E.Bb&&({8bJ!-]m^:<qz,;ǚpt0)i"2H-9ΖŦ3"9ٷQԳd1X#L͗Qon)Շ4e4WKX9[FAho2 =!D|^3h<KȞqsߞ̩'}b>o#3nq9"dd)]HZ1Q(T8o+"h|ԴъFr(fƊCb߸C ^qH"Xu1TARFBHIEvP,VF$0껓&s|scy}{jLo6#8DآEUCf-DjYh#8MeRidkBlfݟFy)\QǝxvKyփx\NKg;Ѿ=ݻIwb5bѴt>c<@̡#!M9-F#3/[=#mߟ>zvŌpHF\h̎NP.%?ؘ	8Ч$էXP$n_ u{:wEӷ)r8g D,<~o+<.Vh4aTu0L'/]#(Pw,WHREZyQwuԛ~({7UXꯥ׋Wڍ<ON2UILR&Qƥ[r(#.u[KH$@Swnz^S1HA<s$d:ܩqȧc]A3&x4#35a%kNVj2`#8Û@dCQz$0ULkA̙TIgvYQh_Ucb6ZU^W^7+K6ƪKkMKޝ'm2&h'cbг,ek3[ C#.@eLgM,G$oU02rĞlc4p3,21a`#8ك}ѕF}YC#3::>rQP#.?#UiJ>ڮ5ZTE䬆T2SΊ#8<RI%/`(yvjtQ_guW6ʹsbQFXh[fԎ.M	O6l,Z5K˖nX\wu:E-hM7ª/ Yu,o)'㷒Q3cvpdMp0E0"}#3FH#.TUR[./pߥ`;<Ua#D桠"|$LA/@pA♅e|ŔxNt@dQ+#	pf!!]	C;Gp5OW{LO#PӉhqܚP_ ㄔׯ[#+¥}9`&D!N	YGWx.HOIAd;2ۚծErX!<f%DY,13Eulސ'6LTĒip)I%(s23Zusا#3hwDrzܔu$h+#8,PPVxl)%E\4!We|]IuUynX!`9ePry#8|v<kqBfyggF#Tܒ0ӸzбR(Ho$S!֘ќ0EPݸP)U$)#8XKELС>	4b1QַEOLdЁZ]!6JvX%0c#8فQI#8Mz.F-!"<*LelԺFd.VXcaѨ3X&ұMco+;ڊo>[ְ0Ga#,HgZ(+PZ,"=DiYvsk8i#.`	9hK- :nir@R-&aboTM `IEd@#80[˔[+FbƢbLW=밹is*fdVb#8xAv6#32Fĸ`#3a1F>ث6W#8ՐH 8PYG<66"Zb#8&zY}hhiz4a:B2#8Y~@U-_e.`INq,}?*X,-Fn5vwUrMtIKC0Q(U@I`_K#3[=P*ԈuTwN4B"b#$#.Ӹny[&LIUTdE&#3clQs[nm5W7kdimMYvcu#.H"H xjN@~#8s,8@DQHX!wQab	 q|<&60orEv#3M#8Y""VVI@`F4A!#8IlJmVA#ܻ֜Gl7?&PoEA$L"XFc6Y-rܶ[62ڍRldFJu]jBSfmkyfJIj5I||xe[U|0t#.\]T1N#3wGfJ?7Vm,DEg04|#.f(bL&YXÊ _#3HH¥x]dn#8X&jd a,H~m3zoXr[,'~RAChfq<s=<(yG!#3eBV5nƙ8{N\~Z]e͊̃Ldgj{"DQP6~AxB8%Pg00@	bbBjn,Ҕƶ-#_3][4)쮢F&hm^~׷ķqW#.btom?6@Ÿ^ z	5ujDVg#8p޿t_Y/}Np :_=Ҵ\;3Yrﺘh\wT	'=4?.ebcC6Q3#8wPA!X8Q@(mmo49@|ADO/K#8$A	4SG \Ñ=kzw"{lh9mïNC.4cFGdJ#3D0|w#3GM?MgQ=i[Ѹ4~32L|ƺ[#_tb5)#3@%5XVt4BI#8~P)`L#3H"Q#.QY,Tj'P}Sms3I!!W8ƨD,0?<7#8'2%mw]OtE=49랇;ԟd5dOX{zzWL~}AqHkVƊKCj#jJ)6FL[SVͩ*5iӕ3*eݫiT6F)6ͶV+eSk6SVRhXDTT֐f֚$*6cPT's-:*T%5#.dUJ	ͭQbRW5-P*@Ia\D<{x	EwQP#8	j%<{b'Evl~R#|9P"&QHYTzD#8H>7pODCHOp\v.)x:@§9@9@	#.(#.=_#.'9!LT@q!#./\FJP4[nʀ=O0b=,.P\J8Lwv#jm#3d4S5,bb.WȿPAf r!^Ջ}F8dQ4B5-T3p6x#8:ѻy#. j/Ug*4#Z#8R#3(na<yR0ClI$ UQbx_,"4,{?It#ӯZɹ&{E?LgB<mFiA	zKmI1y$ׇ~yBCk7moY]37rd>/)&#8,i|7AX" ,D	\1Nx|r2K7|LK>_dK4\#.~$10k 6"nFaS$HFQ_W$31֤fpʦ\c+@#3H1!Z#.CAm*=Mz$LQ#.cd eMF#.c*d;qK?Gdq14da"DgJ\fxMKU&dX{u٭[[ĹMθ6uo\Pm`M8u ur%{.n.mrLK߫B)0BRD!,#3<B`$=x$#.݇>^(:}fj]bzdO,5" hU2:;jg3!ѾvSX*D`6;̈<I &mmXB"2Y26;$=+#8R%6ЌcB~&/y ޴'Hjʃ"Ů$D0BVʇYc.`k$iSxh3k[I.5Lbvgad).0dNnͧxހ%(Cږ.VO#3#8H#8֥xv"OT(9KbZZ'V/lV*4:07ŌΝ7Lr1[!$"yB#.ŨYA0WEŏDciuyr?ea>5RȖ-\m0"'TtuiSmk#8\N!(yvfĀ\\|]jK{L'[g@1%z&}{ri탌1.Z5_ʭ5^2\cG\@ᝓQ_̠#K8JF AU;bHکH^ex͘z\ D1l+p:x{)S@0`v}Ȁ׏#8_Ae#.{qٿWGzfe.*HB#.'E	|;ht9YniTNP~Lɴ?6Pu B/wz{f.] xIf#8n3Ȗd׊ΙםFx[& !0nTB#8DγqE?}jxE#3A5a?KZXb&+DlZ#.ƫMMB#3ɞ#8qβi\XBEi"4Qso/ʒ1zi9ZԓQ!"ebm%:E^M	6rzPЫ70,.6KwL:<Il܃l5@q#fM#./ڴ`i|9TD"!`1݉Ɉfl2T"crm^GT#3nzGG##8wp,uΰ-֗q^kkgXG\Pd{{dDJwvJS 2 y_myMj&Z6-%mR?C%uکݹFUs\#8jSuܛ(8~;?[rQsڦ_@PXz{fC6TySo.ßswˤO8eꊟ3܌!Cki&3#8big*~k"+/K>zME*Uiz0!B6b.pxgFj0ul.@(8dR.Hվ]2!2d3@1`X=IBJb7Ad}F-h#8%}Rg>&ǉm~dpwjxi(z}í0~Uŭ\ۇپe<>f,beN4MA@Gbd5[#8^p)<xx!Qls+%7MM!LixΚ8VXukBh>4+3OI	UObTuջA@cGC#8MB>#3f`N,k^a=Tk|	u{"!k@P#8HTAIeTIIBdf*I697_v nĂ`~E,JH(b-ubR" `L 'b#.	DOaqe B%`eΐK54t.x̼Zs!4Hjx~걀eUBgrrj%~7tӴLa<!>n蘁af}-fm Qұ-u]C쁦8F.lEՁ0#3C.!ج|=="(Me҆=ANP)7lEF R!Qƈfi)fV3ZcGn=o-̚&YlmHMsI0.Zk(cԙ[ÁR	P-lzY8cC#3D^Rx^p8tp7Heؤ8J5Z#35#a:0-YĊY)Ai㾃x{?}L<yہ5QA2po)w^/LY(|J=ڏ*'sP}AG7H9Vz>:M/A^Uu_!*֛5RڡC?-4P/&qZĒ2#.PFYA(ioư,чV+̾3׊l64&蘆HRvg][#8JB#.bax*C`:2a/aꚦR䍷R#83P<b)`Ik#3(ap~#3Ԗ"`w(_$G^)gF(.y'pw^z)@Rq~1C9U^6#3#,4̷8N!-Da" T`>Sz#.|d.xP5,C{ͶapwAӥ_LeXdbzY)0(ϐQ3I&x[anxiH@YfdlHT$x&zs]eéŁ#-]b``^B񭕣ˊ԰s4İ=z:<Ď3޹=7,X6->p:ٴɎqYHt'U;CyYw讯k?R!E^	C"qT@ٮ	ĐMv )iUUն4cjdP@jl3Os\%Ȳ4X7(@Tby8ݿv!  XbUp5Fzv+pNXM@X?%m'pSml!-`A-H(٦4XA37?'5	|х=#87WyPl!r*oS4FA7aO!7fIZ18-$rM K4ٮѠ]k9d`<9 .\::>pKpT/5Flu>0A"#3`UƫIURQCHU't1BM kMD-W녠f#3إ: ҨGc#d "DZ#86(3R72jVLE#!@S-Q@dc.Z!(T!BMKuߩvJ7bgWwmvomaf2ZB,'<՜`XUܔ{=nbN37ұKk&#i'Pcσ?_)ZA?[#.L@:g4.#8-#3:T`k#.TPoe(k|Dṗw|v^ƹoW¯>;x5rۖ!tԸZSfrk4lR]uݥSso5gvW^U[:3VKiFh,YS0Լ|8xtS76mhT+cS^yoSi;G2Ǖ#8ѭSƑ ILDC#.go=%3JKudQlw]ʹ]li*I"SRA"B7#80+#&S9`F- d@(RkQ3L!{xڱJyn٩Zj-Jcc&$,fQLv4*j,urS]ukȺ7AB+:1XV)efq`jxz0Lh<٬J0)r=˧`L5672"*aQ^0b[y{	84a>#3d6D#3LTBe,¢`#`%J,f^:5%XS!cى,m#3&\[|XQ#3JJlV)]D.B%UCX`#x6܂-&cez H]n5",n"c#3a8DFCV+nlӜP6n쒺&&L&#3ncck$2E=3Yz|l"nͽ.T?rkɡ)ڪ#3A'3J%1׳O:v\*tʦch-jGYimU%SaKАKg;I/qRUF ϵ@y~6$4-GIre#3Jyadl#+lȈ傭r\fhɆx#8,	l1$Ͳ*67hB`~lIMh+[4.R9lT|-M1R1a"<}({eZq*?"Ibn!-զԉF!$Xʫv7&#.g(cEŕQbؤE ]VLA#3HQP4bJ!$cJw=}qA9R@nN#ҞX{CNh6_ڷfMkwꯝҪgJT$K"5huU~]Xsxpn4L˗#8vg#8MQzzֺ5#8g)Rt&:cNhkmΣmU<&2G!h62u7#3,#8, ɘ㝂"pʦ0"HFt;Rzafx"Aw݃	7`(_3Ĭ35=R9!^\0\jQǻlۂc1MwR;c#}Sqck(y-nE8gMs	PAOW1nT@DIE)"#.@>j",QKp6*"CJ#.Y]L,IjYBHE#i#3Q0UXYVIU:66ΝۥUʻdLxfdZF6Twmh%lcUxUr<bm[L@)O؍Ա)ݗ+JX9#86PaeV"	Qa@Vk$B$r&(#84PD)Ic#8\I,ЁpBE{^[5@FQhJdj-E6Qk *6#QbV6-**M̡b#ZfQZ6WAʤF7|:>HB1[^{BhZdw6VΗ9xs0aPà0T&C멕6P+q:B$[ILk6h֯-_m@%7tmN-s\b6-V$c[Jm6{vk0(,U)	RJ}.Z嚙#c FOBRa2`o!a#B@bʁy8HKU@#.M#8e<,=Ѽ(S;#.MHS@{['	-#.Sysf#87\~SY%:#>,#3uQYZƍ #3"aYcn61I0A";L6&fI9oj1K7{}smtV14d|v`2؏D3BYBv2O.Pq0< 1`3$bxY%$0.,q3Ye薊RY0`ႳEC#3"ocwhb р:ٹYxEvqi|(ec@c}&f#8uJکQC}&Q`>!fĀ! HFKKHRWsrlF^RH"ȂHPU,[w%vLE}T! b ZT~PakjmVk+DQKxAQZ	mZ#8>׳H?K[{[%dɠRE6(̥[!PES'c%X@d9C8-f7hIf#3Z֭D6%Z#3a4PXPkbixL+?%f$	"\?eBXA= q(g_u<l~͟?#ENY#3=;.كva7jKQvt2 +"wc$$.	#8|ON}FS]׾O΍n׷?xzL}?ncr>Wt$r(yfPU)Dy1嵫g(UN;qyweˬk	|qƱJu;_h}hvohD%Xi|N)(dw^mtT&!QToM%P[dR8-@Uڥ!Xz1#.B'BYۧH YRZC½#ЏxzS5e8`3/%:jDje*D"Z]IBH+._҈@RԄB5D_\~{u˚ؔfȔКRxںlM2cDnUќ;yv㮻f9й]ݺEsxj9J[n]fXmDxk9jܵJXٚJ[ͻ:W$Rӻt\5ƎZ?Jl"P֞n$h^ޮz,.P=nD,G(<|EX0 ADSuxKGx7uS! #.o'g@|SշR}P~:aBAI婢!##3΀j)D#3@Q6NϏ;*HMXV-W_ek66]QAnvQCwk{6W0Z`<Bv[u)3j$"OqJbmZLHtɒK"[D@5%b0]ӪՍ&#d*6	F5X	VkvR6z>12#8#8zaP{+Hc(10|X#3/#.투b)HD"DB#.HHpݏ31M`_M[eRjmeM$EY|!#u߃lhl1m[j)Z3[e{{_7"pD@5S;]{ 8#8h.<C5$~f7	8|T20AI.KDN4@B$m%lJ<QtkL8Іbd':^[<pbm₯@'ޚxZJi.Ѣijnwyזruwcvݨ5َԗi6u^))B#0HJ	bh i#8abmo/#[5ER[ŚmE=P/"V$0F&g1EUq'⊦c(I@*gIm.#.$Qѣ41JaV(l@#8@E@5#8#P$HB6D#3M&zu"JN0B4trHjz#8_kP@RHJ#8օ"J4S|3g`AH@C**`U"@(Bs*~ ufeulչ[y[ԄFX-!3~m#.=AE>%?DKBfhaA=+tJh5}Mm>&/9~mK0kS\F7Ym%$I#L+ǃ#Ʀcm!+?Z٥EX3g#Qf#8*Uͧ5ٳL#!"xzC>#8}gz9#.n<ZHH	Q$aSˉq9"#.H#A6UQ"C"AB(b΅)w|Až{@7ԆҔF#3!`ɩgFULT :|T}_;_&ѕ+达uc[%oEՄJ2&ovt /#3ǆDP1jP#8e_IdXg΃9s8yW*\ݜBolS+E3xkLbP8i% Y`#.c0eFdKOǽ;]#3>巁R!$WA:+8P#35Tx8({rЎ260U:0?.誧6;d^i?7ڴ"Ov9ee39W2[Y{o-#3I)wίˎuIhMSb;&f1b*}lHwpY5H/"cIudwyDe	q7Gj_0rZn*G00}͔'nO<9	rAp4Ͽ]ƼJ@ɻ(ݡ_Z߿zV$VK4O%QH&#3$w g$^ۮ;\{~9/G^lcpCT1#D=%J~*hHkyG0z{lG)Uf=w^`К߈DP*#8	F| K=!p@~^^"hQK'{z#3؆f7kH#80/m旰?F-clQ`m/|F#3FR(#&S8aSà;F3IЯw#3ѴD(J>J!Mkieg֣*6^#3!H[l\!HK21̍#34H-Uc^?t&1H+Vy߻~&tr¿{^ [ߏ&3Mc#Zm47h42(&qٌVg!y&i])BB2F(ŃIFjwU-3X.%Dsլ^vc Au)晬iM5D%yk9MPsęėe;thiP򕊂ea;vI;!˖#^42Pݛ	M0^4ݞehD[	N;^>w(H#8ȑH^Gc|e~C^Й7*8F=;H};=FC&E8#8{ξbNٸwzrpЀ,d	Sյmiښ1g4A^E9E'½K$JJrrϻ بZ^G3 &85"i8 G@w*N08A0!b3[57U#3Ӑ<`$IKB1P591ibۋF0 e(GܖrMj#3@iF@t-JP<8Jo>u&Gù̉;?9q[eSB'z\{AS3if_R&|h4Q#iy5e]c\;n0H\Fe#8#.wZ_~rQꇢ|0wVqUIy	2AMFs=8*1ALЇl:d~yGTO?		Es"_n]bU]w#3sq&8 7e>Pm/~y$vr0E!RGt,_B(@"ЇLQ #3(T4!D_5C-.A,R{} 9壕KRNcsj?6#Y-TmMEMFڥ;wucCT:.M<;~Sw5~>߭yBM"$FĴlJ #&(M%Md}Y"}Qj	!#ȇ:/ǛE}JI6Rah34dҊi/x頖LBJ\33>:;:i龝D>&O?t6<j*2ٜ`f{&c\#3i5U9|QIxw0<u2hh鲊,Pۑe^ynazzo9jb4"	~?`>l IfxF"x{;vTQ hTJVqаCTi6Nă@ ָDFMls[u.KJ64$PpF@Q(V6emcB#3!1C#8д!ՌY&aY ,km2mKw[ IDg#3ki:mwe!@SD@"i9ř]..\G%a.NNUb\@kx#8F94yԗ/kz)]h,Tb8	:SE֢Xpphl#B\TTWdu%#Id"o$)OMlFah`F6E!#876XD4ҕa.Q#3QFdZ*P,2Uir|zI$T܂#8h!q@d%(Rl*6#3+\#8py\pǽe5u޵Yf46a42ֶexЉH#8ں᳐TD,ZA9RU8`e$+$!jε*I۟otYdHcVp2	B8ȇ5M2Y`[Q,Wi6Lcgy5(78yt4c0OF#38G Ҹ`w#8ՌyC#0c81`KVQ&W).AMH"pa|1PhEy,e"$v@%)$5 Q#C4Y_?hfk: \V	(2Dk{A'Sкp6[|!m實[yeo.bicvYRKG\ИxT\ѽ7*5H}hcJLYoZ̓#3*5E"51ecpke'CdGW3À͠%!M%nT.V~3#3J]#.c{F*=	8\GwEeh(1db 64xjN7A#8!F:SDJA2(,!u҂BX(ȩ0`#3#bbіHH?*lH)Je\źx\^j5w]-܎h[+m%j*jlR1`0:!K)+Pv	+oՔQD"X<lxs$GiLoS!>yއE?'IъX, BDF*Q 	r{'H#.xפ:R9	]aA`B>%jK:e1!rh,2ɖ(9ouֈk$,jX A;eQjij	0ckp#%Mزd'H}t,{4a2S[N0g,TQjR,6ԙef]䱮IoYœB,d#3/=R-Z2#3Ar)ݦ/b8	má${ARFuhH'`b\"g	; ]ВFzrd*ȔAԯM{>{ԗ@^ҏǀ(qGxO|#.i岮ϟL	tMI}ޫk䵩YzD|"s#3DR>'3F!qmD"vC:ةqC#8&&٩'O>NP-5*"P	#.>LZvRJ!ꪙKśm*R'bafZGRF+.\s<;`<Lmk, q#8AZκ^5]##RQFm6&+chMy6ŢUɮk^6y-ѬoQm,xѦOVR#	vd`9vlgJRD"#dDH`U&OR@_ި~(M?}&sRN&l5N<\۳K]ۮ;]%g]W¾Z]R	x UV-kVV۶ؕH$P[ȗnP~Ѧm,Y	EU#.	o#Dh;#8POJlGU]b#8}CBE#.L4AE4#.ə}r*ޓjVt̯H(z͡^$5~VN36H%Xť2ѤZLZ)Idh֣ZƪʦU(ѩmX-%c<vDU2"`xFH^#3#3yVR0c!""w Ĩ`\j<  VL:C#8]\ccėkα%j(6!2L)#3$ QPBT"acE	T};\#85/nfIx4`JHDlbAcj6#3%^ϯIo%aO_ϝisE)^#.|I " x~{"<~%#8:<>N>---,bR\	Tȯ>^>{ͯt*^椑l^;k#3^` i!@I"J ?:<-oϲBwRQûv]o,*l̡~tsQVNUr/ku&;W|TqOꪧ`#8%{]k8"8gf+fq	Z5IfZ[\隉]Q}oQݰxaI*/8tCR<Msl8tDr`Bluff-tC	B?n۸G<9!ML'e#301+ZY};{M{ƩϋO2r,)	-G)R0#.bt0qn~2R,y?^:`#.	ÙN-Q#3g>TS(x+*]Gy?'1[F֐5J.Jƥ#8ZuQFg֓r#."zOMSXrA	ysiaylHiKC%肄03{:t^[|hK?wcND*zmEFX@ 	iGAK:[^HꛎǗy+˨jNåw9!=9+#3e-#1=)*!p>K#.#Q㥒OPBBm?r#8j[vTBYǌfS.u^#3v-L/"~^~3<{{#3N+Kܽ#3ڷg k w22j4sYΣE@>BigH:f#{n`jㆼ=59>>={Fyh;7U*8p2zxI$$賐|JUy#tʌpYg;S1LN.Fa-6M,C$`VY+.]U"~>͐b3Z#3+:^1#31o+GfDI]^:ȇKa!ѓ6'9V1`%Q'oUJ+]M-{cKc+oPD8	%yzZEJ;?yfhq8c\R+]mbUeKI9VɃ9Uoxn8"68qN,.Rea>8}3z.a#8?\`/RX [isK#t=3DH'9+Ky)7tn5?3y	QKsv|	^:BD=|VLb:d -etwy3=)6z=LOB~{D8p%eavRӳkxC8#;tm6tCxy,Hvow95V443zE G4M놻3q҉hq).1H{1tBR-Q<ׂa;Њw(	hf<anr$zC</* "7*5-ZY+Mo!C6qݹeK#Yㄡ*H{*ՠap!a B(\L˓eoȲM.[Dۑ75E}UHZ@B%@kD3^=dэd(Ii=<5Vibb		#8OHؤ!;o=5Jn6%~\TcsD.8U#J|*=^DOZWU(s[xّǞٕL8WJA9k]m8W#$1GURz&g#rJw!(spD(]#.jdq5\KN߭JJ>]#8kY5Q$jK1Ǥ>'ਜOGhzKsGjZ&G9.bNW5lFJwAxQdw:i=c#3XĪ܎`=z_k1J5b"٘`F)X1#.)q2%fBqbNa&n#8:=3ݴS#W0B(}s~RqSMR0`T8j7<2=ÔMFfn;tD7T4)^netĴД>o/ of-'%AЪJQDaPt೅B& /	&Wl̏3#3B<0ry4:|NY&ޙyK1~\#Bbqy?&Rw)j>f?2PfaD2(R%pB#C6qllMM@У[rX,F3j<=eR5WPY.HNUքIѝo]Ĕ)LZX,,8o{Fyɉ	'2kw8&9t"1Cn@`@Q#8xxħWX!Efa)qOjD 6 e&vGfXcpA<zJg)vfDj?yA=dvĚ (Zހ:ϲ|FI%N=}E`<TpjqD@ͣG֔ȲE?ܶ,Q$ַ-wy&ikD<3C	Fm4H)^o|Zj-˒-y[U⫦m5mj5kRA#8hJl#.fEDATQ=#.oo364P3l制Ym%:Ō7xUOLPE$U#.HQ#8lvU9l:#8["ilX2;X>!zpHp4HF#34/9Subf	Y]#3Zh$E ]R"#ƞwj숌N2$fP-&nRW6CEm65$R.V6&cD0EGeH0KK%rld2dMq!K͸1MV#3DpD,#86RcXʹݻ1IkZ֩nLk.<uz5lɀLl#3`٩@]jemӦbm4YdZkČbE+!9N+mhl|}tݲs&134Ae@#.`ae DJ3l5\Z,w	R0ghnotck;ܖpoFmbL#H0x%iwr&+ll!+7#3RTVFই#3&MM0jѷŷ%WoNvOkcPux6WWkcFWtSD֨"Ko3E+Y{-D#3mn?9/*+Y4k[lmup{#8H{R8s:]xxxV1*DHA#3!eLԴ	L:m"a/pxZ1ʁ|UG$u)۴#34#3%S~w4FZxe)$Ռ鶍aZec;+bn4-\ h#.8mK=-6LZFSzPKO4FH7Yk9#3X3g#^2b}'	<gp~#3T"0Rd͗ą}1QɐRɺ_)5rv!K>:'lI@;h62)UE#g/umPYXi^GW$<$[8~~a_4lR`f/X<t'O5Y,yV8MR1fm.1z~4-ϗM15 $i69;fϏ7dyvΛ^<Gz!	:o#3*$uQB}DWfFszOj-:!vI#.2l0-fI-֦iiyv((ؠ#8*ކTOq~~Ug"E**i*#8"E&$-Silk&ZƙE&F͚TJI6IFٔBJ* i,FMHl(l&$-NL"s92NT>";݃`?/]Ԓ3XWr*=sAp=xXO`^C!,f$XLKp#R&c6>UNF(#2FvnB#3ȊW y$'i>χАgwU]-Bٻ_fOJ `|+HF6GZd[)51\l^]r>eϠ`NbPRisM]cssAtdR?Tĸ#3#UeW.t-ܑLDP4`jPErYdhe#30Do	,b1&{#8p{IיJ®!61#34iGT#3=#883D6g2#RDLu=ｃa=LH9!d9 Sx YE*1FR)"F1D)!J#1R1mv͂{?lFy:ۺU#,=4.P8Ns	LC Aآ;~V]o7κB	#8a9/$HT:W	0Eٱ0*d,$#.#8C~häLSO%Dt*]xȉ)՜_ۑY$</I$ph! oBm	U77A}"%BЉiB֢i@|c@	4eAW9@# *hgጪ(v52UQJk3y7A~pbMK.o쑛0c,L#Yv*RZ5J%9SLXlh,Xh5*-?(n}ZW0e=#3:rJ5FEe7>FE:1qַ'wɡ޻@j#80\|(B3s'LاauL{з}C$ 3:N򌌦Xx4Hu"-Lw`TLwڍ(1my\ŧ7g.zO{BhMm$E/E(LK50ĎǊE~G8.O47cJBcJf8Ɠ~UЀuiͬRvOU'jeUWoA8zduJf+Y}RA/K}-3#8p:99ٌJ2rHtԞ3tZWfN-!}qmEս8qg<g/FWjh thBl)ERLJ$M_U`2#.$*	fAβ3|=Ux0ӋZź,PY'(dO*GE|JF#q8فL!ɪlgbtXR!q z2ms9x<'d*Qɓ)!t"{߇mcTU#-*kT#rL~\h@gKB[nDbqƽ[5#.- *ؚήN$vdə	3&hxǆ3ƛSӍCMlBr0CxH\6H9x8ZrwrfKL&BdPg'R9Hr!v~eB|fztvI6`vZ\l+Dx<чۂ=Rؔ`ECmχ#3T4F%($`[3VhhPb%d$Qcm#.0\i#.zZ֩[5,<׫ot9"<\%#8!C4`9,4TtƻP1sG"#3is|+0<Ē,sa\J5,:1i/"FpF^Stk%z׃WuAoz#8b	:멲Ғm-)CX6kK"|]	~Y0E@`A	B6NWz8㨴0uC#8&BiBL]JtRnoBF:;Ϲ|3ܑ#8<f;jxaUȈTT	3۬N)..dKf]y&,g,^F&%,W*eS8b\tW4Gtܸdhj8]p,ӎu I@#b-of٢b1Np,wTn:#8#3uLb5366;pc#]04qtmNd |2tX@vܷ^b ^-0Rh9;_u60m1MRmR!312D!",ETdeٚZL0AD#9Iӧ݌w5hzN"1K.aed'gWWoݚiF/h]i猉a#.*Y}6+$#8a	szDAVQv3iT~'ӓm9MKկ#8:K?#3Λ~*dk>.ey+{tyI1XqN>Pcᮬ.rI:TǗ#30M[pʹdIY8r0Aƛ;_5XN;&ʨ#3qo׮RBP7(C*toii}c14lW` 9`sܻx;vV:0JuSΌe`d"7JgC=C/.h0سkYM!VJ`s)ع;˺xArj]NNp"+]|xPgnuۦr5̄X')ef1FؖЦO(/f.57F;Ѿɏ̇:SqbhHAsFEt`TnC]F&&gi0Cbi CT6<(+pu9s#.Y`rXm!n0L0h8@HA fd1D\ `us\Ll!$c#3T(e832EJ0p)aTh0cY(dLlE"@kJ:h30H"S; 	!ԜbStJ0PF#8E$QptMl]HbdSU!I0MLR]qFx`TT;>Dt4q,J(&HTXeB&((ň'ў3mc*j	8&,ev=o#3.($< 	9TQn׶PNXe;n٦F#x5luOhSpŁ#3;b٫jSL:4E|a#5"]t~@XlCHgS!M	47%_M`TmDA8FG9tIVADAqE1ÎwLvn;f@n|tl~	Rj'}J4HybBdEW*(~%iD&O:1zNudń8S¢#8CR+)hir5]=47wt'\1#.}Z$}r>GԧC7șW3mp6~h)2|Na!@d#37i{j(9&A$0;C NMi,bDH+$2%MNo7u	dbcc&0:1R@TUP ȘZlh}4.o#3671]eёaPH	!'}A>S/ak#.N*s#.C7%ᩚ+V8<w3323TY.!0=a5RHᯨٮbPq(WXMu8<q	+y곟\4Q?8j|$_?#.g&/Y;[7egtf٢tj5\#8LgSeLɷ(ɻxX;DmB>TLunRB^NThT1=#3h,?l#.?5	&I$bAHG!#J2gJBw_묔_bj$[5&ZƢKM`BP	4F.AKJVU*Y -DXP{52R{&8$#.0ëʇINͷ68$$AdDDvYp|HhM!01ڤ, k`ݻ^<rVl6o[޾d1=a0\DCiebkYPJUly滲ٛ; Jr$16j!,* K4(,`vj#.Hh$ddE&oHC>sDs_,Ckl.HtQ6T5B`Mw;!,B BBF&|ub^esEEbCˠ$R#8:)AF=4oU;YH{xxWa`i`;	LHWܸguus!~u{U5j|nfz;AF"W=S8rϿL;f~YKG^1\umubGft^=M:$4Tτ!rBgO޶DOHzk*VY&5Rj5f	P*[(թ5!PQ(D"Rb!7x͎jz}JJݯpI@_	>&ˠvҦP."P'+~$=^[A۳AF\2DѪ&|ڌeEUaH5#.SwIFXKw&^jT`:Ql8;&Brw睱I\UkluDZR._q¶tVnekv`h$BPŔ7eDNEH3C4XtſЦ"l#8MM5K_"1;	Do	#3	"9zYE+gB %pq$@rt#.IN %FO (2&@p]d,vTQ=3b|e#8%vZ7AU@F'HQVhQ+	:u\XT^%69ekّ7Ď~ZF;8D1 Fg0:s\D,}C^ѳh7mJ~-Wp`$1~2~U򊇘7`ؠX/m6l;j#8"κ5]ђJ}DV=qlTE4Q+ɺWbѭ6"I3Zݔ_#3E>#8sShADg/yJ((}2%iT}FϮ S`u\EaAF\g&a 0a$XH6fMfK-5_RX56i4SڻT%$0T*"g鄥ct!u<7AFzFI{fllb5mVűLųH^dE"#../5ǿ~.$JS]CGzG׶߳2ː/pHdj"@plq@s3s2E=hb"$U'6?A(x1w`P퀡bp!NۅEohB:@9Ц @E(m5][Q&nU-h@i*X1I`c:{n9ZRphPX)DCIKn1^1#CH~z}Pn^Xqcz-iH}(3CTDZ \gerפxF~lG~A-(.u"}ʯTMa¤e4@flTfF*0j#mM7Wm#3^P*Lm?H۫g#8;[$.׷^#3V0U 12aG#8i˸,@9~H]C"Zэ6Q+J2mb	o}Fwom*C#8\!R`iY8Fxb]>#ZQ;"Dn#8Fj?QEaif]7lMorRѻ[kkmMoUrT[fnʮks-^]Di*Tʙwh XH*C7-M0{UHhbLh"	X#.hQK^u;nv$XC#8h2m#8FHQEb#3"#.ImTV)>elUAT$^Z#3M&R&XB53a	#.F)v@MN4r1E_qԓ0DNfoEasĉ#3osXnS	yMC,x<ϑ.o/#LIR&N.tgZ`SYa?3^BFf1"8QHEpoQ1sTW%L&ғO?,n"w?{Cvzgt	1<xʅaK.5#aY5x<tоmGb++H4,[Z3PQ(}5r)5CL0k"ɄRCƵyO1Ea6Ar6S5JdŜe(T'h*#.v:@݆aᑚ\'"l.)ZJWJ/L+)΍N4ط:.V	x}$#3c5%	[w4a7z(	tk]8(otb1AIe*ZTmBNs_*-ڃp8QC'YѢ d_(*pBtC4MF![J^'Mᮡ!"0΁|#3"M8-3\Ԅ$?]czY{+pc)7X$$|Mr:dLY=>ZR"f-c:5it*S]A'Q'	hOER@j, xm#R#8J<L{]t33b+`$,?_{Y//?ywv{5?oG_߷O#3z;ҿΟOo?P'>#3#.H#.#.sjq"!7(T#8?f!$P\UnԿͧ}LI%uV#8o@gVHEwQC2%84Y56xK'h#3ɖ#-A܀;?J'/2äv;-0ng5Y~x#.aP5_ԇ.M*X(Ԙkx,X.$330InS#.x_0Wt[3!Y୚vqU3LifY#8`V@t<y%pc6Ҩ#D"-R4xkiqEwV$J}3l6>gÖԭPlq1Xqx0C]a g'l,rF	k+ӜE#nJ!(Cn%gCAcxs;oiH6P*f,ABh1F<01hC#3fpɪ1VI.c{64#3TM:=h4/>Pa{NZ;If	ԋG*0}J#8I#8W70UQi*J)u_~<@,gQ*#8zqD)1˳	ڙta# wˆ5#3 ,@FI#.{ᱎ62#.D~)o֨3#.a	(>"1,L-+&} :wtbP"935-[`j$ DdBD6֦6U3XeI2Y##.#8\S-#3X? #.}StU#8!hȼ2ĻR"T>]^J0Q"ۛ%(i!KLؾ?Nh~|f3;Y$tc9존&;(#3N(57qNo@x4{>HFs6C#.EM38}4Xj$>|N8loqn>>z5^Ry6kvhoV5W+%crQzWj1,WjYK*Uʡ`aNEA3F_Jq8Å(޻/*eB.7ψKXTt#81!(h6I@$g)_=mQ:#×	zE XD7;w)B5iZWܨͤ/ȣXުYhD##."yn{??D!܄yz`zgemִjRZlOQI?'9\#.(ju3="9IB-JT=*	PzAF!OG-Q!_%Awley};f|CTz;騏'R?o'tF}Ï#3;@S!z^1?lG(a#8$0J89A+t1«%DuN*p(E`t'DS8&7\r`þGw:v:/LGqSGlTK^99fad!kNK>E/	q)=~oOQl#. 6*~؆s][щ$B	U+D+OpD>OD@?"(H:<#87#.
#<==
