# encoding: ISO8859-1
# Thomas Nagy, 2005-2016

"""
Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.

3. The name of the author may not be used to endorse or promote products
   derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
"""

import os, sys, inspect

VERSION="1.8.21"
REVISION="8b4f9a91ea0de6c8eb86c8d9066b72c7"
GIT="x"
INSTALL=''
C1='#1'
C2='#/'
C3='#.'
cwd = os.getcwd()
join = os.path.join


WAF='waf'
def b(x):
	return x
if sys.hexversion>0x300000f:
	WAF='waf3'
	def b(x):
		return x.encode()

def err(m):
	print(('\033[91mError: %s\033[0m' % m))
	sys.exit(1)

def unpack_wafdir(dir, src):
	f = open(src,'rb')
	c = 'corrupt archive (%d)'
	while 1:
		line = f.readline()
		if not line: err('run waf-light from a folder containing waflib')
		if line == b('#==>\n'):
			txt = f.readline()
			if not txt: err(c % 1)
			if f.readline() != b('#<==\n'): err(c % 2)
			break
	if not txt: err(c % 3)
	txt = txt[1:-1].replace(b(C1), b('\n')).replace(b(C2), b('\r')).replace(b(C3), b('\x00'))

	import shutil, tarfile
	try: shutil.rmtree(dir)
	except OSError: pass
	try:
		for x in ('Tools', 'extras'):
			os.makedirs(join(dir, 'waflib', x))
	except OSError:
		err("Cannot unpack waf lib into %s\nMove waf in a writable directory" % dir)

	os.chdir(dir)
	tmp = 't.bz2'
	t = open(tmp,'wb')
	try: t.write(txt)
	finally: t.close()

	try:
		t = tarfile.open(tmp)
	except:
		try:
			os.system('bunzip2 t.bz2')
			t = tarfile.open('t')
			tmp = 't'
		except:
			os.chdir(cwd)
			try: shutil.rmtree(dir)
			except OSError: pass
			err("Waf cannot be unpacked, check that bzip2 support is present")

	try:
		for x in t: t.extract(x)
	finally:
		t.close()

	for x in ('Tools', 'extras'):
		os.chmod(join('waflib',x), 493)

	if sys.hexversion<0x300000f:
		sys.path = [join(dir, 'waflib')] + sys.path
		import fixpy2
		fixpy2.fixdir(dir)

	os.remove(tmp)
	os.chdir(cwd)

	try: dir = unicode(dir, 'mbcs')
	except: pass
	try:
		from ctypes import windll
		windll.kernel32.SetFileAttributesW(dir, 2)
	except:
		pass

def test(dir):
	try:
		os.stat(join(dir, 'waflib'))
		return os.path.abspath(dir)
	except OSError:
		pass

def find_lib():
	src = os.path.abspath(inspect.getfile(inspect.getmodule(err)))
	base, name = os.path.split(src)

	#devs use $WAFDIR
	w=test(os.environ.get('WAFDIR', ''))
	if w: return w

	#waf-light
	if name.endswith('waf-light'):
		w = test(base)
		if w: return w
		err('waf-light requires waflib -> export WAFDIR=/folder')

	dirname = '%s-%s-%s' % (WAF, VERSION, REVISION)
	for i in (INSTALL,'/usr','/usr/local','/opt'):
		w = test(i + '/lib/' + dirname)
		if w: return w

	#waf-local
	dir = join(base, (sys.platform != 'win32' and '.' or '') + dirname)
	w = test(dir)
	if w: return w

	#unpack
	unpack_wafdir(dir, src)
	return dir

wafdir = find_lib()
sys.path.insert(0, wafdir)

if __name__ == '__main__':

	from waflib import Scripting
	Scripting.waf_entry_point(cwd, VERSION, wafdir)

#==>
#BZh91AY&SY?llv#.Y@§%H4b<g^#.#.#.#.#.#.#.#.#.#.#.#.#.#.#.#.#.#.#.#.d\m϶[*DU]2\u#1v{uR{#/bG9M=p#.4{kAzmnwzmz]m{]z4{y#.<m;{ow7=۾#.#.qt#.FT{˘:giPuaJ#.+9J#1kimRg62UJ*:j)J@4I#.:	{140Wѵ5m{ݽ:``)}w1C_yhS->|wrov֫;v>#.5#.jRMtt-FWު4k̯vIzwn#1#.U#."A^#1;Nvm.k}o}{ֽ}@44}tO^Fsj}ow4xQy{#/w]4{v|stgA"}Ϸ|o\ֻ{{#/bޞw7}]^{n]{\רݹ{gm{qwήmcޮhGl;S}s^}yT+[oS}3mzٝ}kiص}vj:nپw#._v+d=\8P#.#1B (Y[ssEIZVw6K⪩/t[*잻|#.@#.SyMm}/j[yϮkZoU9G:7v]=z7n{烙}{nZ>u{8{=5ݘv|o+ݷ뗏vn[Ǹw_{׎_f{<k`#/tޥuzhyk_;dzܻ>tIyѧ#1yU޺}1[v{kݯ})@YQ!X\v{>J:a}PϽ#.zfݏ#1g\{gGo{w^PVh>ݝkE_vpۗ=whJsO/{76zdYz}<ӻﾕ3ӫ1MaD>}=8{v}[ON:c&WwvuwdrA #.F#.CA2h4#.ښH#.@#/#.%4!&AM&jSO=MzMb4@#.#.#. Bh#.F{M&S̔Қz~zOH#.2#.#.#.#.)"=S⟪{!iѣ@4#.h#.h#.#.#.#.B4#. #1z$Sђh44zhj	5#.MѠ=MOHS?R~A6=C@#.#.#.#.Z5ܧZ(#1|mZvE1O{Uj&ֵc_~eJ$r*1UOV*baT\u$Nz@<āx#ș3]YkiYwxý\bbn**.bf:DxV@?#/7RY-3su+ZmjѪ"#.Jd̀	!A#1  Q#.l#/PjbZommRH"Pf%M#/ER%*D"iE6T"I(ѵ5H`(Dee1`(#hKE&Z)M)42-DI0RE-6i5@,FJ#1IE#(j5ԤQ2fI4X+X43fiM,fllDKSLlM$dEHҴEHi4dh(AQFk	JDC4F&	4ؤ !3dɵ$	&HIfQ,Rd5&Ji--jjECI4T(Q!dS&*#d6IKHe$S#1,E2TfEM"!$Bi,#/T#1I%DAQ%!L؂I5%Li23$eL6h(ؐ	$VII3DiJ&HI,JZlRX1aHMM4@ZXر+!3dQEI-,5XCf̠fS[M!&EIHi)eE**(PQADęƃI!b#/̶De*cdiHRj$"4 Kd[	fC&hdIhF2ʆfi!!FM#jlfeFHJIQS"lh&M42l&$iD$)YIE`d	22Y2RR#)a#1M*6BeȔQcI3lLlPFj4#.+,QfL%4L3,!Y#.j4Cd6[jZ%#1P2RcIlFe3K*Mjdbl#1QJ4ٵmi5%J26jIQ&l"fYJBTXcc!BZESY!ac*4BLjZTILT2KVheHU+D*2VMHled%l4¶bHKKD%ƱX3hmQ5XH66,%fHZCPhՋQf6	L(ԚM%ha#/J%6؍dXi%5kiF&e$̦)&[TM)eIKTI[*e,iflmYMe-3m-eH4hDF"clbABL)RPS)PTm)XdiL(Rʈ*D#/LFf1bBR*-l,6bYYF`SD60E4c4EE41KD3`#1TX1S-@l&QHX,ͣQbDQ1m4	+MaѬIIJL̓%BD$iDŖlei+##/32lM3LD	lbVeMcI,S3&LM)$fc2&4bҚ#@Y4IMLDj#T%EQljcIbe%HS#.ʊd&YRɥ#/(i+bM#.KfZ#/a`QiFl4Thȩ"M X,թP$.V,Y*"RFT*Je)ElmPU$X62Z)5&ƭJh6-cPL!ʈDciY)`Rl,ғmfC#/IE"e-VS&#/QI$҂Jj[QlmfKYheBQbR2FPVXD%Fc4FZ(֍XҚFmV253l+dTjI5fʐdkj-acU*JFFXMM%Jgw$5IFԖIeMe#c&@eHh2LZI6Z#/%eM4C%#1lh+F-mj,[j((#/FLa*[e$l*($Q$I2$jKbjA?.i#1̐*ֆ-u)/߫\#1JdGb1(bTkMA\j#/n2?zubi!,W4LEu	*>d8~ҟ嵇4/M32FwWrGK1wMݱ8}*~/)bo-6VizwxQ>$oq0TRಮ"շN|#&9&|nEӏxG})ҊE!LuRIkfjk$Z~^wZ'^uJ]ƒ(,jckz)f,C(C@6ݖ0U=edD˖գ:d3- #1'u+9ƕKA,4sw&4Cl%Ȑs3heZA#.'^5eVǅԊH3o7w"&RZcK- ,'Ψ:WҾJ#7i!)V{/⁁-`ߕ:t0j	mM^yznQ*{JUznr2(>[ȂQU#/4XM@n##u#1/;7Wyb+44JDHT>Vme~5?7qu#/!E|6x=v]Knj/*3CK;iFƿSX`#/UZ@I5_At\^fn"Z#1n`PV3\?c9ga]ᦽ'ԵsB(t\Cv3s.6#1R<:#֬rΝLաUhJ+YyPBaH,ӭ~XD#1(*ltn1#/kX&R7h}~6)7Ժ͒hȯGxL]'ː^)_/TzXubދ޺;r면XFs2ɼk-WyJ>j6{ZPcRp2T-2,bBLd8)!%#/\+VsF	'*K@	0>r{/3`WqqIT^zEbɫ7\CŰTXbCْu#1H)MD|ftP0+mf<FDVCZmپ2\XwEOZoTZ!pYiZ'T7fMͮ[|o^B<jiQ}1۞+zEiPHZyYG&iYUIn:8U<D4dY&B6gL:#17*sz3G,m+{btAFZpݘ"*=t.T<s=(Ae\j}1UGh)x ^*.H#=0hdmLCE6(\S;=̝Pqº)'L$A8$^#¿q| `}8C@@NϾGMX5X۱T3˨e1R֡A1{.1(d\g?&qSJݾ|jϊ(GF>ebM_oX]&xͷ2ԙTpa#/wa&KBU{&C	KI'a#wR*1 xhX-"AbPqG|<ݻpdi~8')fyHly25ʕ?09bQ-N=h/VZFJZWK?G{rfztTt:]?b^*V'sc!""LkC"l\{fbM=h5/Wd4#/.\Y3p#1QHxc/T}~]z=㤐-w#pvFS!3pgO)e!io h]F'g]Rޜ5vqe$S	IhjLUZYswxwU#/vZE2ҥwі{Xaa֙^ww>Z5OL;gX®*jB#18hۯ		|ݏ4c#1Czg{)25A5{n7MMmu[u/uL1XןNCdQwiPn'7vtEzW(ĺrV"xm$ִKpa^#/E%n)/^umct`SuBjU׍.X|~F+آ9D0h!:-|6_qbL撵𾋊Ҧ̺DFU	9g~LM57BYO>U"Ă+D7;>M~U=?B_!D5!X-,#1bE,-6x.C HdVd$	Cϊ3DpqWWAؚkj}qoi_?gosR`<p7TIﺙBMyZ2T&cu|/OW5Rwj7֥q̹QS@uTdȤ*d㇔wb[;nbw/9)HnPG#/taHqy2gN-%{u,nhCr":XDK|H>z(LoD_|Ey|ojd~n1'k)Eb٤dN&cbu.y)IB#<LWK]ͳac]`EV$ΤϓL~t_e=9#161ˇ&wa?^[\T:up:E'	)qOW,QMT:'#1(EqD	F boWB#/UjjxNݾS[ҽF*w_ۦjeÅBԪƼu?#1f #1HP5#|=Vgj?Ëtdz+Ȫʺb-Z^*8)|6wAn2g~Ͼa#1+o:~82^a"܋#Jw.qehs|kE՞];gzGa<?w=WvPJnVK^¥w%0>4MҚ*XU7m1B3|#TFk8ա6F,8ϳhuۇvO#.4ߋ'u0VZh#1EKQ7lre/N7,Q3w1@s8ٕ3E;{2~J6%.ˑ81Dvz=@43u#1^FC\{pό=Do-"+M%!]6wm<ڢ6OCɬ\WgI-u|;[I#/bg+ɶ>cl<S~P႟U55@黠ؚ;j/f塒;C#1jtR T]||ax8W$X`@:JgòUw8ex|#q_Opud&Gq:M䳤>-<kOo9TOF\gġcҤC7?&Sz0(eԡ9p!a;T8.:g<d!'x4ԃc?dl#9v;x]r6XI	}i|ZYWYZvg%ژD[!u|`1-D :Ƅ[*CǜŜ}c/Qz?&{w&vusswR^!ki];[C,+7W}6]2X5X2dyyfhexUaեQC]{TOf#1άfU6tC~VȪYh5m1o%"Wg}jo*kXZ~\#/$\aǳFc>sCŊQ;_WᄰA}vq6DnʉBNwxj30*)q$"Shm!6ce- (DJԭ*ZM0]BQ9#/4KUɦ11D]<T`+*U;rv봾>I'r#/PT'MXbsZBIt^'J7V5߭M	~m5^y(Y`>5>F1<&6ϟC!-*7a0_+-Qs6yQ?8UilQ!Өi!ZGGû8ٹh3_cUBH=XF5ؤ92Ydo)߹h6ZuY!,G޴WZ)öo/чvrQXvtu+p|oR%ڢoxU֞}<fެR?wr\G!Kd'\lװܪ:>7E,8\Nk=0"	>6EWUZ׿sqK|R;վ405+h]w-v(~`#q|Ğ,8̻d;%QAvE[)M1gy$%՞1"wNe)pVf$F\9>8h/豦-98e4,Wox"IIW;:iR8faYL]<jտhvD;ҍSZ}?8u#/FZZ^<dng/I+$./L,[ɵ*ɬG+Hrpc#1"eWnisQdYjҡ5f2^DƼQLCj@!#1tO,vs;~:9e<SLɨ#1kve	!&ۏ#Ó-sߕ.Kˍs/pXR7׋kHaLa?X#/`C[q~Xos$#/敆&nB"+%mwVȐ?2-'#1yk/tlfL ue"'t2Bi;I#.8O<yYzQ8[C|*4QpŨ>tx̨ĨmqDqG9PD>I}Ԝ/831Eg|Ώ*d7붓w^O.3ʾOgr)(C=i/2Q9IC,Y˱MTN*Dt9%rXN0wPR%!Q#Y8fyVLSj5ׁ(udZW:%檜ifWh՛1ŪFC8;rbDcZ:9u^'VwYa։	ꋷ*z);\qUɧ"HrӲаf#-_o.B:U#/kc8OLqt|IiyL&BL$Ԝ\*Zj!s;y?Qbwuc/W;iw00CH`Iɍ)jCOe>篠R5kYgDL|	;K6lL5:fa?L}P̎W9v,iz=GPX0uBD"o/[3_uo[ǨɃl0 #/6\K\CF\Hhj`t!$o|t+,7aw	b aHcI[L2P#/z}E.\4loJ*	#1M~(Yw2f(FۖiU+5N, hXE[^`/?5 uAYMb[i.rg$cW[		4ɡKD卛γ]*#.ُ#.1b!5ymH<Cb:aG5qJ+|Iپ~tu^i0u#1+mQ,Q8gP,p&<+nY1/v*U:Zwe,uyikӲu:>[mukt?2A|o_vV")2~ܠqb!mwxj̮,nkߟrB2!/ҡUcrQu=HE^Y\[@cq܅j'y0mv~WlrpDLN&#1[+:ZIDVж(H(Qܸ:8*<#1jDK֤#M4C#/TrY	,Y,37V{'0#1:fH_ezƘдoXP2$j.O/))umL厗tҒ3+rydbKrТ6""^>;lt廴A"o.+1TD#.gv#/h:s:։09AEr.ac3T'᭭_ft3MV网/zOI<Mؘmbݨv|t,##/>_8C;!*N,@Hc#10HGAmOb=#1Qt<>ϩ;i#1lqMaz9js+eCMR=r6sY7ה.Ko0z>ƚ+(34ܝHu~%.yP`1#1S^?m=SFѼT&|p}|FF_Ƒhô@HxlWgXx])jc;D,H5=pISWe-	oHK[4ƲF9[jRdq-K8IE{x̰hC)9Y,K)H{"'#/x隻Aڊ+O%$vGVzI{;4Z?oߍa?j#9h9CSã<n$@;|D'9&f>Lś<#<}]m<NmM&rd	VØ_Os>öv`C㓝(,:6:#/.M)SaMFoCSLFvɑvr$6ư)	4ǐ@cN@Y٢ކ&1Í=OYhw	 =׳F.ђ"-0Ms6eU6 +BDAn0MwI	\c:gvʓ	ZIvR'r*͕mݙ%CZ~ҎVN~$;;A8DVewɏ^'»pQP\œS?	Z#1輇#/!L#1&Z5A;6m{Fg>bǻY`vybN-)qM[;	ۼjQuLѱq#//Y|<G=:z1̼-gԺp0PI^:ҟnm2po4a`	c41I8gGh^)6!-#1@λ}§ť}cy	*ڂ#/41/#oo.4*Ą~6#Ǘe%5M>lcΆ1eT`;b3#/#/6QcdZ>g/3|^?+5s@ɓpW<&܎O#1TLa<#/ƌ4pg$]VVL6OL!y]ysSĂiJH,5	~/8|4q<JaPj+,q02"W٧l"jEPmIAv T<!LaLKyNI.ИR8't%3^+tDǙ#/&?	4ye;27ښX^HXPe#/QL;5}|rp_3~};fvӲd#1`S[IU<\DHFȢ߆ӧ-dђێb`úB>G7oz78tug`]#1vXl.DVяQ2FKf@OXW_Wq̑giaq8-ix-#/#A|ڟ#/vY+4,lL_T#.`*#1-Tc"ʋ#1sFoj2׆mJwHb!V$bR^ E3&`ҟVc!~Y|~zܾ#/n	/פ[_C[&ҒhZZ,ņ"VY2Y$ h	@3RǤnzlk(]Tmѝ95l :A<"ʲ[miK*ϹWU׈%_\VSm!8;͘$	ц8zwen.k>#1<e=L*ۺOLŽWOknӟ/k@bKx칹Mo:=g}gkٓVTJ^#%KEkwV(xL>|oL]HtM3w$7|u/_&>t.kV)KTw`<El}&I42㖉G#/3z4	aQ4m̭gw뭶xӿA_ùϤ!(r$BC<E]Mpd6U6I%^CKX%Nc?P|qc'P)!%77LzhzY!,.ui49OτnĆPzXqĒ+'YGSc%3e4)&qR^*5~`ti`1',L1e7MwVx'%jl*mj2CtTicL@WUCu/nkҮp2%JhDII	`FLW,*+)I!6eى1#ʟ234$/ox`tΎkmdhbm^Ѫ5@hi@כy˚˟O"=,KQc.QtO:z2}9I]$-R#/ei*&=kiϯxO	O_r`ܹrIzaaԱo#.;J*#/|/\}WƄ!21cNAC(s:=ce6sESwmzؠGH3:wӽ_&vߋgx1<lw'&,dD9ӟў;wIK_50I؂`"@0Fq%x.*Hw-ʌc&BZ4"g?c|wNPx#."z1"NmDIb=b=^;}3C4Ϛ]7<iRp[#1BWxt$\l^<m9dohyԌ~n>\lÍ>,6EUPnTMG6}ǲ.xrlpA"B+yk#/?eF^b#dP @mʝ,%(#/Z*d0;~HYFLabpǄFNPX	wH Z0mͶF*[skt3z]$oEU'#1F2`hH8Ԝ2tIAnmi1ZiMz;XNqk001D؎sct%OtbIc:%luIu\FúkBq+~۹9uib:c`31]UbsnBh|S[99\j\QEge<χ1jIr\kgp/V"Px8ۑT4IP8	]vF^KHiΗvjFgl<6˔Ф<!ЀbԃS9yn~Wփ';wcݚc#.æ]-wtEˀt%@q?Y=9uu/w[-z\١|}Q=TؽDwrWBA1LTA;;7yf۱RkF&#/EE_P$5}8b##l(iQqκ[;m"?Ok0E֠!3A3ڟXXDAjD#h*ކjXDW,?P D018I#	Vۮ[`HC:0Xsq|sw|zY}#/N$rB'zߋ͈(iE2ԖOdĆlbTsZTm>A&,$PhQ[aw{\jaCAtc "+#1TݲxD,	X0Ͼ[{DF"wjne+,>AD}ZiW?0D	CeM#/cI|x.f1Jŏݶ)R#gە޺ֽ"nQ?KL=o,Ҋj*Pd#/TxOTh\PD0#/$hh#PFB|̓xJQ-KfCi A}Lρ#10ձBwd%r%C8K3pUکeS)X|f6`٦#.gSD)IJk>ua:QH(drٻ|3yn"=d;LUTSEQ)*ryś|I(BA`}$T"P((aK[MYg&{\[_x֨Zb($F(*GA$R=9FEbO~&>x=Dz#Nm1q^Կ#LOuLĩEj}pg氎wwv1t=H:+%o#/|e[HMAt#10U^ι8t#1.0I(9,w!H"O#1rS#/:j?vUe!ܮm.^D/Wsǻ+5Ub)QxPZ1_Uj0<&pp)ǎd0Y~lkcѪ?7UMJc_"zeS_/jM^/"Ai]Ul}o꧃ض&ǀ؈#1s9nqǌ4H1weGf+h$!3vNh)㯆@vr#v#/iD'?\Κ_߂q#1Aoe^X"N!g1#/ID"c߽KH'^ʿxa/+S7>DX܆W},vЛGT	):sH}Nf5礪gZ_<̲FUQ)%Xk3M H3~k>A!yӤsRqV`.$4}YK޽97ԧǟ5#15*^erP,DEW$دwigａ3>(2@Z&r&!O͑As^M`-0I'[w)UpRXZαG["e	`@#1FU×9m_ӓ}h?0tBc_-m7|ڏyyѨ6aQt/t>kSoi߫Xߛ~vqoW=}U?B<r"e?>:%,|)Kf;;lϲRrv^M2˅=Q>['&ѼkF ]̥d,뷧ϼshyhvlnzua/Q+v&ur_'t5ݷY*ՏqUF-qa9w~MؔχڻxN-:tU'gw^yUc4Tj"|s٦U[*T%\ɐ]H/Q/h}>U]SZSot'+7Q֐՝wc['^:ݡl*3mTbe~ۭ9Uq+Ô}$'$YƧ?6iڶV07]{RV'{۫-	93_x辫^kc]׬Mu5*#1CW_?Wd{L6bz=G*oD#/(a]SuUmV|J<wΝ˯Y-ؖO	/хvh		|#19vUpǌ}Y~˗l1vf|s9t7_Ȳ4S)ُ|S/|*acGUea1`~?,GF	||[\U]#V`^t㥫-vM<:_f)v/z~{}>o	~Û|Yi{=]7=4A+#/>?n,?LzSnP"Qލs?Tܡ!Za'S#"i"јkP[@s'~Y5{JN_oϿϻ곓Q]o ~;c=ϦX]cwn߳w2ڇᷣ^#x;ڸ7u߿Oi~o]S=o+~_ӖŴ6}|6ſ7>eyo/P|5Q#1bQT!B(Bs=n5b49b>&46Bm?haL25}v.=GӇHӅ]r_ΥU_厱z}<[ZU?w}U!oƩ\zcոUqm{,&$Ō}6gݪ`@-wn#/;t>BfKT=8+kEa#1a[,ˊbz|ݮuLK㲟7B[*z6mǯN)S:mJMhkusJ]eƚ餔LnK'鰯NOh<%L~Rg.iN~"%<b%y~㯻x11qknon|U]yPȏv}F]iK&zCM9ʙ`ĮP&5wĴ[Ilb`ݾ;xLǋa[j}%pI~:ʬE9g执-]B-i1]4U~^N0JǷ)/[.Vr/ɀ38vvv]EDx/Y붻'ܰ)zlNDRd'UWBg6l	_AWOz?XwRݳvꪵ#֕p<NW }ggt|qn!FdZ}VA䠿wÙK黍rEvJ3`3B*}F8#/ٍ(c5G27)t)67/G$#/VZ&>lT8A'gֹ/ܿА`iȂtBF9)6nBV*;GA֏uź0XG I6CcE(8#1*"jDF6[kmkq:A,Q	Ai6d.[l3fV/B"5VP)X,1m&u0$mEk181:w#/2ŭr  z-LjI+V@Zf*'3IH/nޑ뻙z-tYK`_}",d8H#/2GJfIi1P`P+EC&BeLyli'GT(2EA&хSjϐ-毓<#1_pSlW!hn,:5%-:b[:h*me2׭/=g#g':9F"kź}&-2»>W-Owsu˽+k<M^<Ǹ"\5&HVGص<?kF0H3CRO~*Adw&j\~#1m'OOovfKBi4k/ `@wO~WRMws"5-|#Y;<{*ww𬢠T{ZN0#87X{> kvxb=Ӆg,nc"V3#,2VBXpQV(.+I,`4(W7˪bL]efԂdfk(ec:h#/2R@4*w[ngnnqGB1TM42nBNI\c0mU6/eۧ#/EivyN)	u)7U圷<kU-yoWͯ2E-iv#/vJh RCf4\i4DZDQxS"0,6-B)*ҀQL+#1'x#/LLX$IHѬ	fD"u>,~(dx`T=(j*X8`c0	"YY((Fr<()y^C4֤5gvy/f18O?G;*s=Vbx	vf}Z>[xuɟy\M[ՕKP_eWP%qߔ&m?%:wv~H6f(8QoQ&^63E ُ9q(HVϢ(tf)@%)>;UB/v;+&龇ܥGܴoA믄Hm-N[3=Ҽ;>]{GO#1Q/B\RHd[L=X}盖Wop(8IBYd6Jxֺ,/Us5za3#.dpr"UF:8$ދk.ȵUP˓Rev$wIo7#/aE*41V#.Q$עxU/vFe!i?ؑ#/Tr;=N&#/B(<*fh-.#.4H*6;̵uyM*c@4Y:`q-N͘ӿe9GE=_1QVmۤ/ڞBBW&/^a=x2l7m5t/bjnQIdo5]#/plٸ9w3g6	nz-hɫ`=)"D3!{49JongMLڤldq![#.齚Fę	&3WTz=ߓgw}wxq&M0!0(Q"3aP	֙S* 4V^`c(E5]ѽ2Ԇ1+FԦ4FFۍ3CD*F&DeE*=8i"a"z#uFtpf6#/:[XBD"I]&&ʆhdg##.Gp姭TUl+Lu'Y)#/Z90Ya)(Jm<t8cy1<uu"N 4l%FN3M*Wi	5;Py+M.QeՖB#/G5Ɣ#1dFap4˂lPtZ0EUVR3/?dgNVyJX=fm!.MdBn{#1&>%?11=y֞|^߶_kEF/{#~fGu#.،OJ#1PƙWrc~l0mT*ppuKR&vR+N@"FʆmAƴ;͍s-J;u<w&5H1r]ii BF"l1c#/X:smO{~8zqŮ	\$V&.M?4\?:/ǟ*yC=f<MO?icEk(n4.P)KZ#/nϡ&Iq6B,+Ƈ'ޚ$pMhdcF1A[cZMbiw,,X6Er4)xI2:Y i588c5ɘG!a	A,D:\nR5zY,zҫz1iW2b)36V˙QMCAF)"J.Q3M>t\$^jnƫM7񁬚{e<rX'RAE[wmcDxF )<KQpC|^کN^5$Bpoz|^xKE)wj69yY5oqÅjdvД'c|h5BbZj&mEʙ:^+I:Ǿr.4_K0hvw-49'C(,U-rFdy(8j#vXH1#/Me1RaF#/26h:E`B2@m	@6L<-f+5Z;hc2a)(dJn6DZQmXV-R	i뢟uA3{LÌ_a$akٛ`31b4NlNeF'ač&S#' v2ᠢY"V.\2|֨Ewsm%\DFz4	#/]#1͜MTx/V#/5RI6K/*,8^zIHqMIC;0apqx0CF#17	ESŲnZ](;wETwZ7r0t:"nbY,b/6BuZgh8+M٬X<*{km	;fpVI#WLX8.6sY8Kd;RXݼ);9&MɒV{ejM΂93Hh	*F2^^tY`8B8zk>N=PFsFOgf2}qQS'D/91[v~EFqHGȌKf"lRx5'4BB	ȬNz}ͬG"2!CMoIO=w<VyH8ȵ,TCJ-D':I!)N锘hz.[0䣮R3Q)!#.>Eu˺tZLL k9!Lw?of˙!msDzA:/50b]2!";+]!b|)ipgƫ'X+ĝaFaώVFjlvpDVUS$GXf)^#/Ro?/=J}D4uNxhn!Icf64(mCߗitlY#1tQYR6յBű]Gu;q>Jjyd/?1b{]e,f#/ߟAGZ1ͥa4[%#1.je:;ϩr#.8nw:a!ioU쥎(3UO̘¢=\>M;7έriU*b_r))no__ZjƂ~Ueu#kv:k'^ȵ^mRs3͜`'w(Wꝙ@{~)kZzͲƏ^ɓe^եC54|`$^z5#7s#.F`׆h]s1~6uǠЦW~~%Z$CPhUWp.Tvȿ[e6$M̀<UPΫgM*ŝ'թHD.xO<{n}HЫ".L8UHɴOY^cb1Ȑwy<2[EZ$ICdBh񟦇}r"aH7>6JL3gx2f)Rce9^WKvHPLL~1IdH2TjPN0b.'qLo=#/w(@#l	D	ݪ/y7:B)29DG̚Lt/ƹ'ː豇WIA𞊗C7LG[}(RO3Lu",MG]crTqgbv:F}!+QxNޙivl-s#Dc/=ROnF}Hhrv)#1V@Lzw;3h٫i?KfXP.!oyq奦)lejz-	?=Z&N`\Ce4m$!WX(5}.Ln"ר^[5IU.0V1>Nc2^AnFx@+<mewDË¦ISt2xDNUakuܔc%baiKHkl_eoeg#4%,I n7|U#/h#1HJC~vOXȒ 7N^:j0 ۩E-q6r4Vˠ>z:K'c<m9fj%کim(H\)_#/1f(1IΫ+ҕ;+â`qA#1U3<)?I@yoFˇ'dw1#Mjjgi?YX&#/eJ|K*j?V%	BeJ"1xX;veQA~%#1di3C#/foA>!4;xp~;b1e9G{lƌ#wk+AILtG.Y*T+y<ö{u5XKdY7}5ag~ݑjaߋgXXl#1y#1qAoY0O%	LvsZ䲛W.yLGv'* {\q#/3rpVdmfCҲ.U}U򎸐ejBޡj[4a2'g.$6U#/Du=-ca}]Eǜ9]W#s'	iSW<Uѽ!ʰۡabkZCx ̵|g]Y<4Q1"v>ZmM{^,Ig$߾=,䃉tՌr=/g|Zc|6RZo֋1OXgr4lmK*g۶2+ĺ8T6+FՉ#3qkT?GE;9{䉻Iu,E@J#7ZG+tM_]UF[dɶQC[lgCGϭׯ2p̿?ڦ,%zKe`?L.ɧsSi>=7$kvƋGg7܎s.:eĢYTd-0TW#yF	$x;B[#1N~ׇfޚ#Nٺ#@h_{s:Vi\trxmAݭ<bQP>/9l|J 1]eêfZKX>1#1tTaFh.!b#1oEFQ#1e542:"!2`Qz4(Y~XFxNGِFx>WHaH,pDD(>j2|]t!}cς&8	4#/-OlhF_udMǬUwA!KQQ9JЊ$J/U٨z#/%Fë	<_*"_:?73烬^@Cw+X,l9	T)6:!8'=8%;n}4^Q[SY~aG7`?-e v(Y->d(ʥ#/N^0eNq3G߼*ӼQ՞/#15wgA0VVFnm;Ԭz7Y&#/#/8LElaefEخ\si|sezyƦRv/V]1=TEz1#4`DJI?MEկFU>=cGI/8l*lN~6Wis?>+~%z\:;9=g=98#/虣f%yTI4a!E"uQ6x|-W-46LV[^jI$5ưhǹF_?dXГ6f)D)\͢Üޫ.jl"%4sz1>3[tShJI/l.ᾭ}0x6"=0k0g/r"}Qf#+IHF6Bo'<rL<5tÛNܵ[aE)UԨʞw+[2%qwLDK7nz,w{zbu6p1¨d璶bUzz7Z2*KLoCQA]Guŧ^o5ߕV]4Z/1.LSpH.=ndԚ&I8]o'X0uK2X2u~=꣈5Jt%*~6!ocQ߉;I5jb?yL4|Z_OvipD?,ϼ yx#ӭc>޻yI<PJ3GO-Rϝg7z#;^9ɬ^zj[xDhvSn/|ߑJ8T{tb|qAO\$緭MvkºӄythCKֶ&bm[/1.re9ǍU{f'W"Dnj='EA_n3Ð+)eX6V?a&+GBSG}}wXv}*~o=FG%rO.7;Oj+鸁)Dhx@'P#/mh.)	J̌3XNmTUnYjZGaǏV|_#kUoDB#/vÏ̈LfgT.Hx`D6}zN:yóa*Gg}_-,3ǼwfT"ESco'x=>3{JU_=6Wvs2]28]\+Z%Ihߕ3ʰz>k'ohy2@y||&cBX>Ol3#/l~bC2Gc}ỮYh]BHtňЁ\].3#1Sxك]iSQ)*[&NbdւwP/FcX'$/{Q5uq/bBQ#1μc:+oBCwH5<uu:zD#/.ھ$=B;bÅJΣ+؛%5\kaɌn}<x~|W(]}}w//x̧#1gb/"뻎WxP~e9OCN=s1F!Z%.>m>U[~W)yf/f:eD.}`0_}G:zx̠^#.;ӱG:)(ϴQ]KAs/xa2Ƣ¨"=׎TRnv#$?>ߌG Th^<dx$#5$ɂ|j%awMqlE>C],<&C3\v9;zx}a/-Rda]Um7$-F8$f>[nə7?,KQX]q-c꺕mvHm?/r W-7:c^Tٔ:\VNUS<{6UUGU:]>6=0x.di]l#/#1M|sPt<8U3Tze!~%6K}Z/{kt,F'ί+A.##1Orݤժ۵B+HH|?#dPx!o+-$i}6?+V{?{[Nf}xКgaٺܥ3"'~%rE}ϩ~#1'52{J^$!0gI[UUaQ1iDc7<fGGWjmYmҫ>uC=;w7#/'Q{ˁ&?x$3}+O|c)!kymxұpfҠY@s>B+儳(eM8PyWA-dcrH[r'颊nzNϪ8̏^'_XryD{c,$|ۇa"g;jBvUg[7xN/lDm/ВʠȚQ5?< zE%}Y;@MSЎ!J_nVk־-kSPLuLHy7vA0,x/٭GV,ې4 p֦}6eL$7R=L;w]y<0;PUrW6Oo2Pˍv:z:Z t+*gq}[efXGfC9#.Ǖj	AơW&Gȳ(\bo,냲^~=y:85AV}eV+`Y~jĂd#R=wM\f!<eGu1ZM8~{r|8uu3$[ N(~DzFvy+r>`ďHskz8qѼa>4LGnx2J'C|WzCc=/3GˍkŜƉ9"~[73Wܚ	n%Qu>`_Rݔ#ۗm+FfwQ㖦2Wh|/pMdTsʣUpDJЖ1;Ζ=JvK.!};{HBg>F#.rǐЯ[|#1IA鄛܆iGZ2כ5-w4f#l#Pa]"Kc+(O]#SM.y,D+}d*&?rWⓚ:ۍĭnٵk/lhNVv0zm$k.tT'tշLuH9 c¤g~HJ1nЉIpt:-thsSGg	9O)ӟ<]aCƢߤ%H>F7,Qi:47F#/&< !uc3#/<h~]<h˷m0xԇcu-UgKrsdlk#/wgSeU/;^|CT38=i2w4ɊjRz8Nɩ6@^]\Ld[GZݐa,s0bBg	l	fԳMcU:C H`%#1Ƅd~Y9r"v~W8|.n.:S4zq-@w{ᦡ_&!+h+#qW-;ju]t7m<VܽUa4/3cJ!)ڻ#ks<jĮ7}.N+:wi#DҰ5͵BtC;O,bimb^!Ⓓo01' {'Z+H{YsܧyO5v,N+֛XԤE_F$,RPDҔͱK]~sXsuOY0cW-5U-9pB|x(3;>^H*3i CJ:oC*|'}Rjoպ{G7ֆ#7)C-UU<x"iTh{\`nV!!2=\yweKDT8IEUQ3d2*RȉhQJ	W+d2up	C	@3#/?ob:9A@R$`cEUJ^ㄠpC?xseQR(Ӣ>9,5T%7#̻DT#>#/QmlOcsyƨEa_Vz:AňW$(H6*y1IgR\cr~/:t:k9%uow\֊тݛ!߀9#1BT*4S;PrhU˷P\Zi}">vL~3T5G#-*:;)@_IH6go#hbjkEi#1;KecH_U!oF-YtiRT;#1 W:yÔaZ\;!NS|~|JOHQI9û5#[|چP]:?w5"!t7=V"HӖXσ3Q+\?፵Ng\RߖjvMS|ߒ#/B<➈<N}dt}ຣo	6F	E\]]tև(TVi'A!D7]j1f]ؤt2۬Gie#.zpن9g	hO&A6p'Hg$rLVUћM:]itkB'fUT^<(61	8،<GLu*i˅mG(b#1c=ClWW.:]RD5ݾ2|A'#/ z=,yAn%ʃVGݸIZ#.Ԃ4-;v$dt`VRIjOdiDp-^k0?;|;hubfZҥ,lWd(RpS˝	#.LХ&AK̀v~5N[yg;C;Ӆa#1#6)b;{nxԔ*MyhBl&Dn~ڢG/k?$WTHX3El$&-l\vt$ɾVܚI:L	8,9y{/rHyKAA+v?	XdB֙亠Ɠrm-՞_$Jӷ1n?cS@:ehqlV|_iưzQ=6##1TS&<CmC=w#/?\PCבڋ;.۵S0+I!P"8n.%2؈i|jbF$qZtOw]H4gUڇ7pJu:~=r6P<M,\M/|)Mt%($E(G.|7VWQNDT?ǻlF9C#1(٧#1ZWthy4xюUR_(^)|BjxGIi*8tsЌvxb!fr#/z띧W#1v_Qg)HCs`a-{b$/fLnVK~y|iq\6I*Bd`igYl7)ƗQHxVˑA&%N`Zs^Q6"_m>4^{ZcnǣЛe<Ȉ5S?{8;`+ lA)9ᙎ=TV5XgnYmA6OCN&ڋ5FRa#/#imɉAopQW͉5Fex	h;ʪHMmn7F]36>3wC,&S.:S<DFZ>VQBn0ϓO%"V		}Ks5S"-~\[Ԇ@P}upvpDL&^?QDoiJth;j+_Wr20hE--ˬrj!mCx#.]}CUr3UNDCIEXNpSHVk{2MζhYhhJ="3$91;WTmQ#.2C:arg:KM_˴æ΄C_$#1˹ٱb2B\`p6#1kZښs#/QhQP˗?.QrVYh<mkvf)`"G9tcoL`RBws큺r#1n섒PN9jcGcLZE5M({O_5gczCYXl{V"bt.MuVک:e76p5	FAtuj4zj)nl?|J흮"X HhjV{ʮLܚpHCkYq奏vգb,p&b5nѠ2,-hƾچe3MPv,-Lu^qpJMj-%)-9k9a:7q7 q5I20p6SGX+yRno#/Ir#*$<=~m|[iË<4Pdw'ykf_зIJ[%RIf]/ݷWi)*o<C©y6\S!oN*& L!e&<db	}B|vzxgqIg寷A**!O!acL?Ÿ?9M骮50SŲ`S-uf}'7unPblz`vmr鹤@:#1/8-˟'F:9<}OdTXVΔjtK0}}#1eHr<1ȨaBFTчSQK/o{'}DTLY&qݞ^󟺻Td@BjW	wHwv}OD2`_TlƣG?@f/\Z@sXWY9SK=橦Ws,MNxK'("Z5Y>&^:}hN\L^Phl&BtѯLs,W-/ɏ#.a9'%hR观AXB<#1k/e"]#.Zjn!BOT.0,\Z(_9(@4`B/*K& lXY0R`҆#1Cs !m~([gگzoRl~6\cL'wrx<g, .{W>7B䑢#/i s7	Hd8ublVBS[&Nn`LcV<##7#hWYzDYAƱpg*#/o9j+bm}#/L,ce,FD+l[`}&X"̧S]hSzöM|W(U-()NF1!4d6BgQG$Gq%o(|S8DSG/PC_#1nB7rC x	wضKzEms`^TݰC[KM	v{C_r:dVTDgS:QiLbRvqӻ7nv;S4lj!9hp*E]ٛ$LiI^TLȪ8ԩAE\	~4Y0lp{+obDa@Nt#/|:\#/#1CmК\CFȮ>p89ZNȜP|m5@"BWN0c8Eh_gomoR{t&*u8&<N2l@H=#.̏dC8棛~cY4	[[^i\!wnwJ;	QMNLC J4ׂ{E͹jEQjW6tV\a%U#1RAaGt"IR2D]3Gx:Bx!IBzA	BC!i9U^!)ķ"P"W&TiтƧ<NJO!E.QWAc}VG*4ED*0ښSn!'>Vq1cckmP"Ϟ"^A$Ų&pF@)Q02TkŊmrfU"4T8J$s!xy\LHV6$5#/DeAmcjI5sVWނ[_ݸ{v HUnUMxt@Қ*ȏBbL8LʽP4GFAJU"3N#.dw0#/׻<W'Ԃ{|R4fNBh1B|o9yGWVUDao2bs#/bpD'jQ宫lp%@d`,|ġP%D*YtmDJugu28Tٳv6VF,pCtq5ɘ#/+,[=iGeQxx%߽"N/^&rz%sjDn1C3CLi'$g#/<:Ftli{Xs$An$^|, 4NC@3qjxgۖi5C7{|RzNbaXo	HIaBu#p~%ertς>^;aΉUZj^猞w,Sr%ިT#=]XGz^>sRԜKRfs.G#/ώd\v-#10(:sϮc6XEw$5}*/-S1^*k)HF+yH:HHu#.	viÊ}g8oDgT"]S>>EC5GˋQU)0rFH$˞0)uY[xTڑB#&J;5릍%}hrJo@mf!3&sd^[\\FF$v:q&_@B	_E,iÐ3wu̝tBL?Ub>n\ҽ}DR9\ByjRJ#/u.eck;uq[aҹ2&Nip-)f0zƀsj#pd,FϴW	sns%og!S9ɿ#/kk:~oЈr'FE4=sq t͋s}om&uJN*.N#/yק|~#.KDԬP{3-z/tp|z/SBNIbu*|t_4nO^$l+FV'#11@vwlxk- 4=	xռh^dR`+z%7lzNwZk#.εYUPyM'u6Bu<ȧ#/ﷹY9֎;H75YȗQި	M5N~x!H<X1n)CdC!~Ű'QfD3Bhվϛt{d58D9JAESX1'\Tv>Ҝ_WoU##/x.'uO$dX@'7/sYZ:HQ}iDH7%k,=|t34)=dY#/^Ae{ePm0 #/3UlYW9d-C}8\hǰLa`#.Az^ovr&y$OuꢋF1#j|y"<1X81	#.y&Ú4:J: UJ"!*9LV1w1E[?8&=$216Ğ	{uƍ	~d*'A=7u#1d8&줐X"wtr̺oGs\썕z.K}x@>$~q#/^.I]fA7vUAM"p3K~3C}m!g6e|--"#1O/_Ŧ_V^&xVHP}F׺8&y>7[zOu|c @12|# \HƾsY=VHι74>՛)T"/!ߗժ|3Tsx-{arHL)yxޟ^z5#/qq-1)y&0@adخ|^pq'LZH})R{Z{f2#/RN3ݖs|Tϥ[Ͳ#ԖK-*p %}W1GY6˹!7Pn~ǫ^{mRPw]?A[YJ0Ye|+^Tl]~;u{G#.r0A(1zUWKv_ZckE@z*0`%#/4h|S8oHI"v2J3U m]ƁIUÍyPWxi/Y:ZQ©~Szt$SyPwdI?GqM{7,b#"Őti]'.ZJ\4l@{Lۇ?ƵmZmZ`Cʚ+|?_.gewwXT,G+s6fwGY#/(/<Xzaj܇ڣ9t$c	ǓSlH=oQA4wfd!,oXk^Cj`}Lz;G9͝O<K҃PىYw+u׷v G|%伽x1??~zSԗv>RoBH3AC7pVbf}06z~*;c*%XY*触)b/wwǧdNQKR,L箟>v;^ZF'+F7&NɆ6Eq@?ty'b}geG9s0ҔtUգ~圓5@#/ҎcTƋVh{/g"=}7TQEjwh t_au۠%2CC`2$Ŭ3#6VrڶDh(B|$ڼd!FIL/<adэCGsĨijkhʡRC/7⇟hj;a\9Jq002_f ݧ1L2䰱#.#d8!rfjy|1»h01|f5*(:U)#1ɾ]+D_sNg-]M#ܐ`bq`aH𰭻n,ab)}Ξq'I"KXS`]gTg,N0Ǆj&!'U)>Xj?@׌MscOݪ-&Ǉ3S	49LXE-wl3컗#13ɫBѹ7E.,M?bf3GO%g6M2}4z	74	:Q]jLr{YWelkL(Zh?2",!vp`K1tHWl49ZQDIT5(ig	n[tA3=)fcy$iA^	snM~z:W\|4PyDI[qCSş⿚dq5y{!#؋5Z^$`ңc-g!)hDEU0DTo_u1JlVeF	<h7C>7G9Q~	{`.[N^O#1:2v7>kQY܄FKk1݆WwNyUB9=7靸:6T'fhPQ[a?TĵKCpOXsqGrñ& &5楥药H9!b9>f'j; DKeC7%@uވ\'z(.3Jql~=xpfṈgbǷ_,yG}q	{*F=Щ#/W#R?#Y&<vtqaU%gCz>;1J,}.E>3#1:#1Y({OOo2 ~d*SSa:;	b4M4(Vcp!C0W3Td7ɼTxfvxfLM#/~S\!$Fvs̯T#1Ǎ"<c˭S2'iBnu=,δǋ;W$F\m[gOt\&-]9^Gw7<q}xPH'	!~w"N>|F㳒w|ʟ6CAC_dܲ7vxJ\{|UN̞,){T*TLK9vJuLELBNF48ީ3g	7d[R)ps9.i1&GƲ_WD9ϴv?!sNSKK3`$F;zV-up*a,`Jⷾ#/g&S:pGqdR\RLEyEPSQVt3<S#1 I}?,)iOo\;!ux=L;~.#qͫ[iiLJ%1STYn;_IvvHf[N$x7>mtksk۷#1|uTlMeEQqׯwQ+Hqy/KUݧ(GKVty0D `4QlZ\kv"V?itv60|DjWu}ș,	"hŉLpLkN M1$J*hR!9\#/*ciM:\ZD;z^`AGrLG;SDb/D~E2Imt{0qY`ݹ	?E׻V㏙1AP0^M.vJD "n}Bp폓ںyQ#/-@2	v8Pūa#rEqxNhs+g)gCigF>!F	=9-P$}(2wFXn(5[)5i#/	~)911-/Z}CfqNÍ=K*׾"ѻQ^7_odwmkKR&vU#/.~T	'6N!;g[uzLdROF&*[n+l);&/qKuD#1ջBj'wS5u`1ĐbdIN27 (a@Fcl,\J.X)kzf<v<5Ӹ[#1#O>G(!<ElPBjI4ry,!2,XõO!;µ0MZyi%?/qN0?LTC&k&] v]	A8uȜ#/?yc!?PxN^{ Y?|Bb4KxyD&!&ۭ1P?)"	Ǥ<ᧄD/G/}*saFQ}1\G6)D/"l`MW5QDt읗jC?2#/tbkdGyf˙v#/iqDM!-#/z2$9{S׈jA}uT]|H7P2p/	:{VvCۨfr	R~1PLo>:;`tz	#1#D*vw߿>,Qg:s4^4edע_J7PZ&lcaQ#1c7		 96Yc!>M8m5	$fcSu|&Xo)ª6c)Fh?EA޵=V<~M>%k`~HvJGT"@/å׻\iD8&]9JL=LzZdw,8AlDIB(ؿպSl~X\j݁r9GEԮVAwyKēunŶߑwB>^w5Nf뛤ԉ)]ٴ+xD	]/{G4,.9aώFxۭ$?qW-H0k['i*eDBbFIP~+0>np3Cъ%#.CT^;E`ߨ,Wג	TxǞݓ#147kGtTNVҲ3*ws><.GFhxEW7.u+luSfMZJPfe#/k8		凣9,=$otsCY<joZ89ɀy{(,@}z7GaɚDRyfcNY&l;e^=^*o2I,I?S[d>翁?#.Ln۴{9=dT<-)kƚ^K5$%Խ4t?jh29U#1Sm8ZYg#10?M{TCd9,;јg$Vc/_*ĲDԚ_|HGbAO1xHBt$"Z2d"f#c&RT%] &8B*TtSl{(	5q0#/#/oH՝qXB'Ke<=sNΜ)S>|-+T_n1#1%8BRL䓓Jl2鱼g>D5 xb22Z3	Q21=#/S`Z*DgB?[1|ߴ9|)&ؙeS7LհSM[SKrN':H調>cKllUE.0HV­Ѽ?I˯x$TƺҲ1Nkwl))w<z(<ד,j1|K}=QW5zw6]bHA=0#(*[\XmqēkIn=qy^da"#OmLz#1"Z0,T#@"	7SXQbWmhZ-#/ndwDzջ0O(#/t)`%\	x)r)L)h^Vޏy8!3]H0[0>`X'x9q}r?q#/EH]-9C9YHrE6M>)v6x@ܖFW`v47:Ltȇ}*!$>\7X3FGE4>/>'ji0І-?\weF<#73UtVf>nnUaMqn$E#/Uy})SYGƥc+mG3D؅ -f0}#1ص*0YCmkPio6b&^l4$D#13QY_lwPꎝ	lؖ<kA#.v>/7T*TXiǜ#1iL<kCw,CM	uG)הO9~ö!oǋ`&1%oR~߭2kK29wR7o(|UUq@%{ͻ+lJarӼu`.st:馅\aN&O@kfa^\"ojowѭipݣԖm,hZ9qNI=*n>8URQts#1`ՙuG-VFK$]	'V#/w;1>j`6i+UUJDFUAl#/lH9v;^xBl<*ȎC|{>pE"עEU+C=&EͲ͗G{qk5 G%Z&1rTl}\@",ѻ:L%'`K#1#Y4Tf*f=cPYl	3DB(ϻM(6޳,wOKs3a6V䝱. HJy9-6|N{y]ji[W19	NCEϼԜ.l܃죮n_̻aUK[2#1S+m*1"MZ$\""ysu}s,˳MA~פPNelM09+D_GǢwr];lwXh8G&rK2'uFOW}y0.;&=ZWisϤZFh"ڌl:^r*,-疺ݬp枴5ژ\8#1%+[PTX|KnaxhS-߾E飹9s4QzI!!#.eEusd8gɇ_=l5C[oôgc2|{{T1nYjVn)Z&*~NɽK<4j*+	>A\Gbi,o#/Ks*1]Vf\ej܄ h[ug?۝}!˷ra=܌#1z.r-S#.%X|`%'8T9Q.][.ƽw|ֳe9[6HĶʋ$8E56+.P5,-zalgWӶ8re#1Ngni	7[+!T3!)&-0Lg./gV$8CVWpSfEaX3P_+vTƎoBlMeTќDRR·^ڊG2F'v_7ONTtx7ID#1>Bkײy~+^@n:4~+TUXO"5GLH@N|;}^ܹ`I~?=1lU\Tz.)%Ydyh濶><~).z6EYG¶z|e~@1c+VDYX]i/	vZ5`yK~3;03JG.o%Xt'HX>AT$(eĲ~}Q#.?c?pHͅbÊ`-ڔowoE۝ڋ5,2?k2r.a}]SR"fñSǁ33._$0iHB檺#{*QJ1F¶A[0*[#.`=H=pӰ?xuQ{!оu*Ҕ?"EpMR:1{+ـ{qJJEL0/#vڵ/{o vuRzA`Gdއ&ޗ8uYo}@%$U͂V8tɾO'S|>샛vam)I!:<䎱UלϣC'#:gAgF;A݇k&j=M#1bxs?#.NDe-D+]tHNVq,cp,4P_8<zawㆶ(x9#.?G$6A!10#!	0RTJ=CY1UZv>&E? :/#. #/۸R}5mQ7Π9I HKM[bi\ɼ#.F¥$X:K	ԽF*V2h5hIAR#.	J?Bf^HF":?wtXM֛!T%zt$?7P7%d#17W6bHKMeR~x6݁0W}#1L@PS^cܲ7Lk8o6%!JUz|S%+Jc@<[WR> =,M#CΑ!4S)m>: (89ҋ`u	oȃwh#1Bi=eDEGe_՟vl#1Mͼf;P7k!?3%4(BJ`:748y)GgT֦˓c:_VPs2kd	#1|iWX$Dz2Β$ӾX뚨nY?s#>goˀXmz팊L-{Xw9F<7f-ГP@L3s#[z|Kz1R#ޤ0G$/?}ځʤZF})/!jh[S)(?h(K^ӫ\&-Љb3rot0ه83?˗b"m/L],*!0tQHgp]8ԂKb"YG뒴XRH]hp!;T\-%Jb	yɆ.Gv1m(y&KGQqT#/p}<HJDyܟ Nhu%,Ӵ3Pu%*βd6QA.1]e,Z[{^#/ior̲rQL$9+33R-Lq[M}*b)a?UT$!:͐ڣɹ`ajz297;{"č&ĺ#/F`h|1>!~$vH&@Mn$I0v*XYÛ溫ߍЗ}cOZ2UY!Rf#1_<17C5\0©ѬrhOo66e6=K#1u+Rt]nB7o#{zYEY.XfEJ}jAZÃ`RpaKe%CZ/Ǘ,_~V#/#/Zf+ZIJpO!I#$sƝT#1h''C;7%-4#"LTX#1=Z\bZ!}Yf#Zigy$#1pc1EDc::2k:l4a;fQFm[(Βuw+㌙B$آНo[0ʹiS<"~.3Gמ35=CEΧesٴw20ؑ#//خ:eL,"E`*g,۴<Бd֗$eciv`axy!T	bH"+,Qӳ_G'#.X>`}Ệ]c=LQr#/q,#1c1[͋d#.bR!K%(Fߦ[#/چ?Qcn'gGF#@:ђi@Y>Kb P7@7	#/Ǘ-#.G+	pyEվ@j!-l^_]s"V|XBύ=Ãhq?|[D8\ o	Wם5	c?k"}t(#.=ECX2=p}s:$hbx#19ϖQ"1b1#.~R#1 9߬}!#/NdB(R5~=͝7paެ{A{DcwDIѰ x"DG<=&98Si)x|	p-yсAR((H(!	D=p,E<dP)R	47<9ZzTa՛>)+~ߗqh?#.{!﮾M6gktXpQm=(fhɂ&)#.dI#.z@qԁk#eh=^_ʑI>SȆmF+0GM}|ф7#/mnq.\~{/"S=#/TTPٽfF2T37gUKaތwBD`sySB|uӘ?3*:L4N$AS#.pĻ!t/vmi#/Ir؈^AbN5/b_Φ#1E&)U@TiHi=J#UDĆN,Ͷżw^b{И7C>&{`D^DOs<G#1 YyTe'1ڥڪlKN=>=	Λu#.9@}xz	Ƞ'S^4ªQc<_[Gi[T1IRU?Yvaua`mÌj#1obK"f.4YLmpz^l)6 ])i5[' *ZuU[E b`Bmj|H;=҅Oh9}G#/#//9W#	tZfHCmð:PL8hB50a\]gWWlg9k/20tfHHDAfa@A@kbL|ߑHvd9˲wSmmaML`H(!Ƞ&M~X)%fJ-u:/a@K#.aF燶]!!P]aC^*3V%lѵ$5&gϒ#1:&!h8H-c4#1sɅCPH",7*ҔAv0/%;55Y^'uvd?7z/#1h~w^dEg\WOnT$ɔśjzcn+jȡ ٶלZX$2-+xrOW><LG@w#/Jg3@Y"E`_w5d1m~Jf?5V[߮+&Uu_A,Vκ)&n&$VS[[Mq	<Օvg/pEɰyѢA׹?Lkt4&Ӏ4GװlBɣ4{^w[5F#.h Ѡ}^zvJwBkނ=<ۆ@ZTSֆ\Rּ*n}kq84Ol<q44HH8F!G-#1fᇯD!׮0!iظԘ#1~#1,V#.0ᓎEmng6+vTcJ9@9#1ؤ*7Qdo n'XᙬR'&.^M8J(	!l=@C_?x!8u<#1?భJ$dPcE߭#1",ܧHb-dl7(Xij?pzƗS14$hiu?x;VײRAAJHz(RíC@u&OhŇ<d	ׂ֒UY!J AE-Gվ2%y!66p?!9u@QwhXJ*RdT.#nxj7$Qԩyc}wqx;#.;mQo]#.M:&>@^;6rYT.ʘPu[=)z\`tᆈW#/e:yiWEX``>Ç8J}%QXdYLN_qBd\z##.r"3؏TySD}MO0lW`4Fy"Y3NjHLf4&vrLڹ?L͐k9bLy"'0͐LcjsXJ^ozs0eymdg#/iz<--)#/g'#1]9/ŻLPry5G0#4PyAxg'bH#b$酼/Rzl$+]˧{pZ$sT4XiBiETbp!R؀aR9Ep?OZO!@Ǉx}I#1AB¬rw_7C茋yP~̵֚)yP_+祼9{z5PB=^&e"E.]ِ_&&@|{"׺@mL_҅Cr[|_uT6S-:	O_c>rG+#1~i=?)iIɻdzwvT@&g ^ׅS5;`uΎL,:}pfLR5CxzmNX!F1WL,FfѺ7]!bѠ#1CZjHp,%`I0Uޠ?7&F?t@1#."F#.B	^@¢y/SsL]#.LA(5.װ߫Fnרlc/^̞o#/`#.%Hh?'cm`99")po{B#0.7k-zO|;̘0oJRBm~M5c"r<b#NwÞ0%r?Mh82'{?OZ~e*7_qdYڠ0YP#1QswizrO̒36?oIO`k)vOh`}*03)}	`^'`@6~7JnW`RHWW[=!m?jggwcd	J16>BQsFPT0дaF+ioDFq#/GIVYa\2U#UI3Ta#1$ P%R^|j#..WDXy2Ǳ\v8#.֩"Saw2ϔR]?&Äx"\@^[VJsyf0uto2'脌8<LlТ<~#/;=:(]MXjgDxX20aE%>yiEXZY	bX ,qEWu06.%=}*Uyw+3v*YJ{xA &PcWlmZO_7ܢG<\?סN!_m(l}BP\:f['+yXEpd)8i`[!B#1&$$L[DSB	"Qn&1``a `I+U[Exʼd%CLb-+2f%#@a!S`IĔ-DI#a-aĪCa#1Q$	 ADa72M"#/(h|e+2ICp%Mz]4A2{aOIC'tB#/0DGY\	PDT17ZeS-l#/vUwi,wu3H8I,H["^SǾ`Mx_3~_ܢ&gpMݞa$g{/uv:pxGHҜǣ=kQ("yzXf@p^Y	Oqo,bc(#.(e4Z#1 HTy{ۈĉeJbY8$Fh2_!##k&=C@qނ<jfi۾l]%-#/a6"`MtJB#/̼?Ff	mh"cQ=r_bVH#1HHNƚJN	5bAje7]fn#Iwj߰JlFżS_65((e#.e+hG#.5[`qO8KC}8pXt8ʩj,doD rEIJOћr}bM;iGl;\3"yL5;.Icr"v[)31lc-r>ŮRa?u?S` N))O:/uc\z*`O}up|Q|_Wݪ4Lg+GYw8jvØF;c9}B}57m#1f+opp^왱t;"yY\&0~YsGw#/#1Te#1Rnpe+#*95v?^YYOOR7퍖|cɐJm$Q_=^+>_7Jm-4/4b4}S2#q:j0#ƞJe"&_#/+gG#/y<J2Vh?oh)+-綞pN1NmWCjb;5;T`W=!5X!]0΁ʴ]~tGfBC*=*xgȬ{E̹W1$æ#1#4=7@|#PӬZT,ۯVvr'vA?"Qat7h%iXNTM,`{|[b#>/J'~wm6y{BMsqibA{w>+=,{ oY,4ZlUTTČ\%ELѠԤԀ$~#mEx%ATY0YARpVR?؄vn7#/#1(VTPd>'|#_ÕT<v<`K<B5M$pw"cGAc~n*G} J.a53j53'5\7IZ/.NRI_uy5^0,A]~G~N~]I;8wJkہ8Q|EMg5ё]aCm#1teAP÷:|tLkS_wvNSZ'#/1K##1<tCA ~?}u@rm,*wGq#̉-I:f3lvaIa&h;aCBY@.v?>MkozA/BĆ`<k;#/v:V٢(rQh$å4Z-^9зmR<r^r@6*0i(/3B+b"QF@#.l&EĮdԸP.d}#/Dw@Up,_҅oaV;f<[lb|H!ے_N*?~PL U]#1l2@qx<KU=]V#/KvCD߳y~2,5BqSos;"eP]`V4'%G;r.@,11_|t΄xp=!X>{%dslp45ܩO79!ėd1kQL#/RNA#.`缚r#1A+C!T~C^goopﱯ_޴OcESXmJdp{{4~L*un3	VAӧK0[k9;i#1!'O]ꄚ~CڹI4x<\!5Y^A|'CblaVH8ѮET\Gr*}iÎh/~Z,u;tx3ȯPnW/A^I	*Xa $PV.Cf?x6UUy-U+IVu#/q|cdjcqH~?046ӊp,CzE6fZ&z#/Ʒ/aN>^џ(3,xΞZ?<$i0F#uBd#/60#/x`7[Ѱׄ(s"@6`kBF9J}L{<א/w$w♥D<YGzEPc;V;^!D.f#ԮwHI8Hu_L.ӯ5po:O߯e#/||=iwl Zݠ	v#/;f)"ƎZ&i\4jR]Ӓwze]q&Bb?o1d?"6AX1D~;؞>*ݷg$6xzk#/¹2GDL]'_цL!^\#/c]E{	I,To	-p*9賡(6ú9%mbGp8S$qyIMh5L<	=(T:P#14kH5R\E%#.#BAA>!y[wyHmUu|fS-91`Lns8#/1	K悬l:x'(אf#.g/fC#a0YaB?	pyLI#/ˈDxye9<L͞USB&J<!.~<7IZ)}|3t"PԦalHw%P64X3б9[cfDn?m/#/KG\ɽyTMeH2NC]3`J#.$Cbo9%,Ibn~:"كV@N-6l@r`a_X|g˿o2CChlUBuY_;B2oN\PqbmY'=ل[1`aadBc8ibٔ[4 XjL:6&r5liϠ-	#/yh{+8uǀC~YF=Ĺ95G@> t=Bk)A(kl'nb]w.fE#զ3k:05 $[սC{!.M'aWy\Xk#/t^#/LHr!-ې,t=K!!|FAB`WS08!UXuMR们۸;dn ʂ^0.qpL%bQp,K#8Oy~`F*ݢ"ZK࣯%<g$;,oݮ5,E AĎr>AW:֩Vܦ[V)-v$%1ZOQj6Aޟ>_SD,l#1HA'qg#h#1ÉĹG5x@YN"fؓ(*bIjbi:Gi׺5]&:{k2b3.<ɗ332ٙXffdcg^#3m׽qu3R+ѥJ=#1a"X5,m\ՕxhC,=$~d<?bhu}>CLLnd#JFمQ>3kx3v!4ά]#/7&2ʠ;053V\ɓ3>2v#4w{~9LQsb(.s߻F^ñ'M"F!\@jR9Mo[]=q6QaLmr=NS$l,G]6f#1}.vԥɀwY$L哐b׻."s_Т+^u~9|o,Ҟtim&tA3VKH!~J6BM$G6Dd>͛M{`'.P؅K'I[%HlMͶۗ!,=62"{qriEx"l		.9(LM왘2e!u;oC2N@Ji$D&qNr"$<pdp˰BYd${[!;g)D}U2#.:2.Fc`e|	Lw7L})^hL-##.wk^~s5O 1HL7֦:O)nsyZPQq08#/sH^(QHzo#1F@Cɑ1D-#.5vkq["%5X&rŃx./3S#/w\)d39 Q\d!Es |$2/P#!ecn0}+3FhǊt6窲)Wl7h	{uј˙K/Qo&kvbN=EIAd_H:rs8K%pmڭUUUZp,{C>tZf^3%4WD`Cǃ\&}5%+ڼ`0H <>=o;L/-F1riwk ˦#.\c1Mz=#CsAE>q*e5MA˳&RiU>	Y'd#/3&cp<n8	PW]Go=}v]Qb@M<H~D .4DvMWcdTEC85XN]/-7uB.mxnz#qNYnչZ#1z/#/uJ[܇mHs`Dh%BP!=PNQC4J|@WAb,pmͺu!tXtysG!)?gʹ2&VU}a,Zf1Yg>>	e$7~_KAy܀{~j#/4i3Vd4%;S2j\pK«}+37b(}lUE"f!beHNHq裾ʟU#/؀45GHG'+s^F0oS*H&"9[A/$QB	+ZfH%No~25$0m֤Dr#.}PM! ⪀%-N&K.f+1 j1	RabB@#/}GyJʏkHdkf	:Td)T)m9Dfu\&uOW!#.ml`eA#/dd#1FQs%nߛ#.,j#1p#."E}'xRU-%Mm7p@(0,H#14C"",P#/|C#7<Qbe!D4\nJ>f%UF@0g#1iB 0E)`?+%D&L$F5zǦ7.\vIywrƸmstt(nrϗwt\u|-d4Bëe)|,-|2F$HDesw"@#/$rXI?À+lezDeaH@騒i?YUY32Xv5hդn,y#/R4qU?MNW "(:EFCFc Og#J^o1- JTT&OӇ35|*0Xx)r0ɮ\;Ҡ2EVb47)%I3Lˮipx-C]pd<|vD	pTB)(jcr&>!	Q:\@#.<ǳ}<m[َN[Rjo=s^Es禳#/l"tl)gH}GJ BT(Di:vqܜNr@5J_(04@l1U2|NݡFveT'$˨ЁDÚrRH>KZ!g:lA#.wxum>opvf*Q#EPpxb׿{k~u#KZV"ȩKcTm6Vd)JJL" >I|g|;T|UdB#.0))@w$OUbq/=V`񆶊ӿ)Sf0#W4/PPdbʅ V!S!:'X#1jd#.ɂ#/.Xnhi(ȴ#/J#.h^CJșWm[M€h%#15崁UB	6wq4Ec"f!RMD._O_w/҂v#.H9?~#ڜCa,befm-EILnUͮZ)&B0Y [(T"O=cEdɑ5#/qbH *fjOsX'i|qp*	z	νRؐDVq'**O[|;mL<t#Df`:D!0O~WXp%&P*K`~x6E#.0`m#/h(c+T:NUNf*" l$FV)!wV۪HtCm㶯l_rO]ET˖%#s~̄|ۼ=ak$(L6bpĘ,A,Vluw&mo=P|~;#y@mS#/A#M8FTFۼ>O#.jݢz[|.~8x#1D/:uZ 5wHQ'}uDy4Ǫ]v?lթAl/F2{"H,o	2!cXX UfbdP(Z#7ƂBRi#e)C78AH,[TED1d 2*(@"!YdPxN6nsR8%@E-BԟP ;cS<8.&Q,~YHwG$ LLiVxEXD$|©jRjIcW[ڭ!V(&';oϏ9,F#1_C\a#.UʊHĉrCa\z2lx!.#/A3]H!0qfkvs?2pGQDfHT<0Ԁu'M$ᾄ)&UXCx.rz[B#/wo-&j9",i`x	-Y4hXq۵~Mڍ\@q2y>:K.{s{kNO{<v,Cf; FS	R<s{sNNNqؗR|t&u»-Sd멌j MaѫC?ӆۓ=3{ϙ{^&9gMGY4V:EanpBiG/Qѣr#/6@bi5ʷ۩鲀V#1[g6s!:T;|,7P5jo#kկSlwz$OjL9Ś>6h`	J0ί9Em[Lh;ǜzr6eͶ^<K*@c*]1Ko5/E\#.;#.8xtphՖ#/#/x@9NR'@w,Ð^l˗8\x"nlj=v(כcv-Oxdf"-Qja@6bq(xJMURF2r#.9D`TR{` n0dD$a	Q'#.Ͻ\s<`$5Y+cc,9$lC=#/aaph:tIF\ۢ44	aeh>uSR7#kرj6x㨯jEߛKDܹ٩Y\Yhm92 2!ou#+; I+P^t'QFQ+!kUF[#.c$d\,Sdn]K4L]SZwv]ZIJ.X	_3I2ErwQ]ø|].B` p6 rN ~[x5?TdM2M9WѢθB*	#.خfT|B1!DC"	(~:B	t.Oި#1#.My+ɍC "id{t?k`?cn̡#xAy7"A>({~\5lkiMFcۊa#/y#IDR7#1=Ypӎ`	w@,N>T:#1)h\#/FNAD@PJA`lJe[Fh#.,	[>P&`!;򘐾#.x`,BH(=.! b:cpmg)-Q5/}gnq̰[l[gIBC5g˜or0|l47=KȄ`"vk#t(IT*<H@rH^QHň2tHp45yDWߨ&LϷ;;_OBKkyogwwj"N~|1)6b(:7ITc$cj{cl륏$#/Qi͹2I'9Ri?q{T-Y%יh̑Zaw>WzI`vpbpNFϿ86*!w#./xt\pCPiAÄ䪬yfQnT!vh!8Kl3@CauF,IO:" ad NTX|;(LaP0 m՞IЗR\J0`(zD';2˹i̳pvcدBgeU"#1#.jp`o/kB4!Q!`Bma(CPcbvC1<KX#/DcB6l3v;" `!,-^]|&R5 Ĝ,$&ǃ;@Bc3!AgC*G8aNTQID,TRy>H/sh`dN9b>Z@ɩw.Nbyq{EF hf:"j^3hg"Pyg2Xd__h8ǋa];x2`(['D`BN_"v7lʺW][nKBU|Ljfӿkaey<Zdbm4{zfJ}EJ1Hp0=[zy̭>?AYT1,"xPVLC	"qc0auWU1KCIQK*xX&#.vT!Vf4a::J~Mp6c#/N5 mFqm>Ή̮4a{"P#/1.!nP^1lhw5#1p	<K2,G*gA')v@Tȍek!!"G\{Ǖ}mG^!v#l('#1$9BꆋDulC:b6&	Nؖen8!N^NH2(t%LD6kaؔlEm]oS#,JjXP~lp`a7ڠ5>Z9US!@Z#[[c8öX4ba-n`6&)$@Y1,f]CIr0U$kVY^Xި`S$24ޅI81' #K+0asT⿟L:!Ro'fʉt4)M|NqYSʕ<#/2us4267ԕ8P8_k3Y<3&C3o|o#1Attؠ5edS_T·^NGxm@=фlpkRѴ(	'51[MRC"[ȈS-UN4*%VL4bbs(4,4(40!bCVvBCHr4|z9nȇtb36j%XxZ#1Pgκnfi{c(f-;OiO,d@!vDJI@Hpobm9vLb'#1Ir`w.7#.Y#.E{ٳ1ۨ,&.Z}~"k8RI@hǱqc؆!q,czKLLGkA@gÕ.:*ľ60"d<f,pbωz#/l9*{.x8[v4A&ăYkMx<m)s#/#.`ͻ;1YQFrƐ[#.^+@!nw&3vC}!#/ ,0t@2t597)E5gY$`i!"5Xs2SZ	B"A{Y} B2DI]Atp#/7&Z.?ACמkTH:q_R2#}o&#/TH΢`fw(ˈxRB%z#1N=\K'Y42IJu!`C쭿yu.mn\;W1&b-nkjjڋo[WU$D )vF=_Ŭ&^A	>ÿ UhR6`C#|Ntի#/b	LmHfIFdMh3hn-$Qa# Rl#_Gn-FM&Sk,ى̑hhE#/$I"&D2X11AhԄ%&M4#MFbiA#Sl8ۦM"s+`rKwg8>qfD; CRrZne4&kntYsPGYS6(jg0!$Ŕ3Xp-pL|t7<f8#1:_53#߹9=0rhd#/K,Sw,70pڒks%6c67J..OX\8lI$11iaAJk	؍2"$rp:5ƨ:`!5σ_U(ǰd\h}٪V:S_ZT{o<7S~#$Ĉ΂!H	ɯv؟>n$<zul,BeRU6^ݹqw]wK=#/';t iA$2$.0]ߞV~:z;D"cS:҈#/q'OO9,&8NlÆh6HӖ׷0d;Ӕ'ó:Twin+f{NyE.KG	;0shmOy3 LQQ֟w|7OR1&v>r0^_Wr{p=vfzTzg#/DwD$n,bLق>Xz/31&{Hz牡[tGOU|l٨~#/ F}'iցۻΕ뗲)<O#/,͠L#;xXޛL5$H/F_(C4Һ53yj6V7z\ipuy	FRAAHf[|w_oTh Fx)g6vx>MBe|fbV@ׇ%.`IGc\jp]ATF$$=@!IfnLoFwD7wwzv$$Ҳ[i`ӍCc"Op2,gy:,2>LfGynqm&SRn̯9⊽f3xK4foS4b.2]y51Җ%uq+Xc֬ޞ1`99&1.Od:{*0Cj\d`bM˙[pcK1<P):	d*?(qJ[3mr⨃2N1R#pnrqVNQR8{@mj!(uڜA:X©+m3ˇ8q57&!nHChطk<o# .|q7X.õW#jrFE53{PːN;ɦrK<L6dL/,@#/	|͠'80#Z||0ny#.8Q1J!ewEY`7s<^n_;TLQˑ1YLkG)N"\lI"W*V{t[ӝq'czNy뉂y9)nwd/^y㶫	&èt9(hs؛1mn!tC]M&u0u~nD>\wl3sq//	ocq0nvN߁mCy&".Ox:n7uc9c]ֱ1DSUPIB,÷b#?$pM`D1#.5A{.#1?X G-Cn	j^L4隃RbӦQS5:&/Nτ@!	5Q&Ht<1lZRPN84bBLf:?V89nk78퀩#`%f2]#D;#v!t#vicssέ\0:-eLB#/Dgr6D]VDylʧ\6&IecE9\ FchM	U1lHe4P#1KMr6׫1>g:Gc&!G!^`mLPgŢʎnpnLއn7:b2<@,r_/XQ?	KlD=$L<zJ;iQ|qvi9!?p8ff<qŧ,#/]V]6tkm#1Sm3dJ<jh!l+EL5#/<MHk]EjlrbɱL:L	8#/Zqݷz!v(n6{~($Xgj;DԎZ-&XG0pw>2<7nN?f8E᫿noZi#LFW1=j(]&#1]-*Ub[VBR#1񃺛,1')3=ܒ]+iz ɬd%xIPفֻomE3Q[	LJfyrbi7&H0Ӱ'VjAzA*b-\$	6(PQ)5^7+'yl̰چɔSlc[|0"Z`rT662f6S59X@5GpZ	)2I*#u0A	0zvRҨ${-5.4I	ð<<JI:mʵ=RL	mm#.v+`BDjH9@׮3t-`VIBUeN<ttuM,Ҝ&tDHv:NE#/oc!s-ۜ2rBGk k3#/qQTPT@b%u)uv`KBR<1EIbws*l aJNǆZ#1`0mK#/P,,);ٝ+Fd­hԛRhC#/*z-`mbmpH0ْ٫--[*8M2MA6S2MڎlOVYcpˤ#/L#1DT˽)x<ۘLv`K;8FrM h>4328G!+)e#.yqӢs(H\!wzl_C:-!O8jX1t9BJA^͵#*z:pVm3lLCƣ(#/;4f:3ǃLװف#/&JH5L;t<>:,BK42 l>PA8P`،O] -I&\$YI&bZH.R܄!!#/Gm·<Ƀz	cn1N:qz;QuO1m"#D!CWtv{D37#/	M46#/lCEKMKl:#1'P2C#5GՂvR"CBeY]Aw9#1JMd"l*QBB͓FQȤ!n B@s7QH˷`-Xw@*.`D#/]7ԔjRG#.P78ɍ$)(Ma6\7u֚AM&wMr;*waa#/mpw*Ls CCx4XMdV	@R$XEAMK@M:nX\>#.uyyUMi)P4h(64_z~ͽ`3D^\RCB4ĶF*0[Q2B9]®V촿/>=3U#dkFXgڃ#?}VKvF]Ƨb"5f,T;!ܷrp=d.oIR:8m6L8\0>a"L20+kviL𭕙x6(B*:0bwU#.>ED|9DpHqC%%E#.;5&)5MUy|*#/VduzrL	p콜Aq*QF8g:#\x9	BpC	e^#1}PXZB19Ĩ-y"6B~ӁC<	P~٬U7T78@WF99q8!(#/27"HW9<ī?6Rk.?f#.'m8yX8?WW^yմAk/k"X-N]1K(]}(=h_;uDlDF=51GGV6jM(W2}&B'>g,8"T5=bf>5b6"%ZIЈ2KljEe5b5TR3QFڐaQ"*?2nզC n\ xGYi>T`<1{#	QV0DekK,EhOdLFqBmdyL)<Nםi=.`A҃ȞYHP Kj'A9f#/$0w䩎QKқ~&VP*yQ@vDB4+@Ƅ0iY4AcL#E")~FbPY[%ƾ+x6i^'4jռ	|CvﭣH&5FwXsސPiE+ajK+#1JB1y7eq+z#bۃMM+HnV(Κ1+qf@Y!_<_(4xyܢ6j#.ġ1q"!PKWd6W0#.X0#.48ry	{UΦpa}ݬ|6TXӰ]~{F#/a?dM#}E)Z[Ҏ:iا&ύ-h%Xjucf`hI,Q`^Fgt93jn֡suDk%Z1uR̑s44k~]gh猐n#/lIʠ\d;t#/pT2V6jtSJYL祲ӦEQgrl	#)`0K0#/1#1i\803#/#.T#1,JUA*$0ذ`	hX)a	N2hqg_#<\#.RDxYuzV$_[(H]~ds9~r@p3ł(#.`![}.L?:4v嵘ƴwISAUUT*EȪH[&cj-_YزC_HAi#87aՒ0uwۺ+ހR>orŊ*V'xwp:V"'מDiLɕzywIWJg2.X0!TcmpkdBj3L/iBE"%.%ghyiԴRTPjƺ ԥig3qodBB<p#/HD	еfHa,f63wו]RAMxS٪ی|[a0R&	B]E#/H*Sjb+vC3>uv)jdSXKK"@bptS:lH;a~Dk f #.#1	9 E#Z첩nZ֯{k4`1.@`Pô)q-\AkĊ@ZR0PJDRPҌHd,)3!#.<ieCPփ݆m`G%S#.~\]]g:ާ:Lwm&mo-LZQ2ÈxUZ1V#1P!I2R$pJ)n+ݱ[^62SI	J&֕(Z cB+#.B"={&fѩ#jP	d#.BzMxu(жja)G8PIg&dRTN/&тEeTC#f8 Xbz@aCo#R3nCP NrQb+d0¦ְj3֍"F0pU$JKFI`iϬ`&29.GnCFdH$bʒU4	TehГ"Ql[Gg9U<vdS3ɞڂ?#/T&oa&8qBP#/Tx3#1 渒֦S1B#.5F,k?c6s<jS-0" Sjv9zAPL\#/!6tx<Cͷdە&A+<\D1#.]Ci [L?m1K`π~#1#/CB*mK)DGtv[^f.	M; $"${>oVM>q&?ΛpSfXhUXTRgk^6%u#-h#,uPIH-[	BܷIo7mDVJbieM2iTPmaXxM5zvJSZ(SmRPlWܘ#jĶYkƳIVWm[RmZ$Z[Vm+Xμ4fie+afٶvxJ󫧝jmS+vTUnu16%ķ[60U~ry_*JY_+K(恛+yyx+YV:V;r)MH#/p'պ-kc!}~?|1TIa-)C4cʦQI:6mzteDiͣVRj|%?_&j**mU[]M/7V\ד)tU\T"oF#/ւRXW5)QHkR	QP	͊bI6kMфRMim2e4Tb"4U+bZQS	F42%I)BڙD[dZKZ#/FR͂i)TQ3Q%5Rj&$U)Vl%E(6RJI2)21S&Yceh2&66TmK4,iL)U,dڵ[ee5){ͭ"`:FU	^-F̶mh)7̪R&JFV8b2Rc'EryHa\#/pd<6X B'ŮNg+ʲȨjJnť#/Hz<ފ`OD7Gdz!]x&lLe>Vzc'VcB!GK#|RK߀#/};Y_dX"F$:GZquT"#5,Ub#.BJK[OAk61g&՚v#1~sI w$?BEn|ܡR$R臒LbTU}+vXɔ"M#1p\Z$j|IRqz!WӬMMЄRf_eCYdQDCq7+$>メ+@>bԈR%ҍqއĽ0g*6)nu4hv6wp#.:!Y'LԂRi #.F#.U|])R`$w"\jYD}zY,ɑl2rfr@} (Js(z+ Hʐ-crSl8	5C9g:[Ռ9[wh00uMOj8tDd{]f}#'N-#/"fimQց*/Ȅ$D18yO/]zlE:hyw7.^i_m5OĊcX2CIҲlCRLF<t+*Bc@T#/34%hPfMTf3#1)eRِ7#/&4f'l0Ƌ\Uؓi7# W#m܁$+n D`PP~"L#.@Om	Vci#.r#/C~i·h%&>>,BԂxY]; wj18'$tF#/ZC5%92vCe8 ]$DXQ;?=C[:夁?;`"΢#/w#1h5{O-z|Syz<؂ItuQ|~:80"&hC)\[TIceѭwCFV#/CZ䲶d"dAU`Z6hcH!zT|;ƢU@~HE$~P3JhFC(;#.OF$[&Ǚɗkq@#/j#.p(p8A+RSⴒB#/FRyqCLMX1[&!;Kr.TD}DdoL4*0fLsOu}FFYm#/\*CPG"(i#1qz#/hf4Wȹ%	'd#{̉ynWE^:jtp!ʄ5-C#/G;~cb y#.daJz#.\b7#1l7bg!%Dw0UtR]T]ǵFAD2H¹"QA_s-: NOYG;O*GT#.a{>9zաx^_61P*FСe~(45D	 U	˚J'Og.K4)>OCw׋+vrT.5:#:MdPMxDm}zT86-(;⁪xJ²"v@&K)"AK7ˏLJ*0᫬`)_nozA4bOgaMnoP%b"{k*&84V	J#q"%)]-]~zKzQC#.vJ7(6HcM3 n#V-"޾HA3f}kzz@]9j0n}$~Mw{Wa.ǧjs7{6Exp298E@" '@a')8gؕP$n_ tfv${E6)ŧnR<"p/e	2D+[Z-dL9<dn)Q(1(D6dÊ#1U$UmEIo1"Ҫ7vk T#1P.L#/%	2ԣN9&1D4$:UXMV!ie6uxۘ(<2$dUN}YީqϿE;5tFi-]$x@Ϲ-D(#/EL+,95rvki6ծE*i)蹬LM5}Eb(ؒ.$E.@0)kdڬZՔt2(kBi# ƿrZzs8$ȁhEX[t-?b6d$);1Cyc ~e"FRP-.!f#/֟q5D,#/j7@󒊆`X)IꮺQKTP0qwݟH̫x&vZ6HmAכ&966jk*ʋIF-JŴhume^_l"d/R^BxYbCd#1	ca["*H8	aDNũ^0tX0"͎i`\m:9hEj G"Vo-go	c?QHI	OݒEum7cfz>N2?GO~{_PvǺ.{y\}/-dB㣴f6HPDCLn|u++Ĉ`@#ə;a;#1uކvo#1X]6!2qK?ç0́A)uT>TR\бӊnDEd; $*6FWYVDXflY	xPA#..33r	$.R\4J@P" %#1QxCP<a|,=j~Xp&(;	aY|`ં;dZYl)`J5IaA}XvO=4GwjLGKLX=ׅrgIwԼ<ImQׯ̄aJגB(#/|Chb@\n*1OD\R)	i6H?e!LT#.bBfKgՆQ,WJmOf]F#Ib#.7IJQsJ#.$)&V*5l	)JM#1MU̴XDܬ%0P(թuRƖ\kEN^R<b5ʢٷoL1Fhc@Ɩ&.)u5zUjXܲkٶl ZeJR7XjbF]T(W5\@R-&فW6z%MV,d'5#]5r&ܶ-\o^ѓԻ[lp#7 ٜ-3;Ax"4	3#.hz")71cF3yl 81b4GV#/ ,!˥sCoQ$uQ,1J}2O:INpK?"{QTP?Z5Hkuwnsb8j&J##X")QUSIIX/K#/B~'8"XUsLyDc_SH'n\[ziS	YmڊL@Fj6؍j-1#.Q$EDD#.@d'vp #JA,;q޾E_DXxK&+ݯ|[.i:J'Tj9b0J@E  dfɅ?ιr[>?zI\nb1#.2?5V#1([*-CIbF""ۮf̹-[}Kk/NY5cZĢbXWUhku*l+_)o7FLI3MmZ%)m#/#/!؄HtBp&#/[FDgr	#.@ppAEvEE ^A5$#.T>,u7`E`i<ѹalR@&Fw9NfIz^Lr,Bx#/}\I#wCbd&D]_$@kRwwQ2 9k顠paX/vol#.ed#/_};pm	5Z>P=#.i~y]&׎uv$lj#1ynѥ[)WR[%XAEJ_˧0>hyg9TQA%Yuqg"H<P&OTLKӶH3l(gU-Nn}GQS"20haP1 R1SuC&#1SHa]-#`,rw#10:-TI4EG,]kߗoWo/Tr%3~?DB̈́4a*M_|6#/GMF?Ytk~X#/m̊t#.$JHT;fʊӌC2D._<ZfU?30@(20نA #/R䐨 Ϧ75ON4rF|1ЫͤDPܝTge8A;{x3IC['N)c]Z/:C??#1 AX{$"&PԚ!U6mIkE##.~i&~uX3lMXQZ`Rbcm*XSk,f%JFhYkJ4ʥ*%Ц}ߝP6S-'-#/ԐxhFA*(kEZ6rey2H?#bvC~hݵ~enUΚDdX"D)!7}[dԠj#.{?HgzН (۰`ܧ>HwaϿxEdF#/E_d#.5n֜V!#.f{&"U*PQ/j[(qPO#.b<UrAhOSD_;8æ+L314U+_lȣK@Dy#.@j1<v%%m:q4MA#/8#/w2Waaa ]I%g"pqCG`llAZ.Ԓ"A#/VMH3llB\X؁<?vG˃-3fu]^>ȧ8}[Q{N;2>$/VѐDlc2HE ^˯9wyHg]4_o5( <%Z]mҀA EfL|M/O{O#.^)#t"KQFBy	8xq#/{a/8DJ_|^Vܹr5ky#XRDELM6]J{1e-E<P4d+HJ4?{bFQ] 7Aj,R`+q97]%@T`Z; 1o)]#.M5؁#/ÊN.J|N23!#/[#($s#b]c.Z&k D)]ӸЂ,~ǯs4yȘ|#/GhJ$a6D YDDaGqa@ 07 1Nas%P0 z|lvx'tFAW0Ǎ%69ð@˭S	O$ *QdOΙiոFAaFLiNA(φ3 +C {P<$+ǦRDRj7.B#.F5)";[ۇPFVW׺DTV~kkߗTM#w~'5lA=&@uȡk4iwv54USxh5	9.ֶ66]Zn`Pk;STngnǚsxpwiM#RN	ԯ=@1j$G.1S2"|	>G7|CŗSSc?Nk#/_tǭnMvfpxiǮ([F0D`BEa޺P$>a~|G"\_n!]ZeлVXY73КF#.OX,JuF0},̞MO|P,5@熊i!|{[*uǫ!ok0\\:#1m#1b0H$6#/U1B!KhXnKCʷA媙=VAԪj_qa25hhPevu<N21xD$E6!3"{0Vae-ЄZG#Xv?cN;tBͪ&vyy1}AnQ$#1*Q^1靼U;k%je䤼kn*#1H{YgmkH;;Su6UTEU}|vؖ fvx0bL4C</n9V1a.fE31Z}W,7oFo	Q.يU `)%hA*Dh)6LH&=Bk{ZAYSE$\A<aө7#Cv6Հ5I2p`p!j#CaE"!`4.~A©rbcj	NӀ PDY)*2Iv/:n mQ츌#l=ÚZ^/R`j?D0e1+ck~5-_)5Le-%LQSZu~Vū(W]ֹQV鲒RQt'YrEͽM`ÎEEa|P5q<ҏ\4V"g/TSyAQ"9Jo1biYEg6a)ݿx."tI:d$UCJeuy5TbC&X(7k<`6!w!n{c<Ncvs}:kݜo#/Tdf:7j}f"_}~@}4̑>gf-TYtHs2#/ӘTc#1Q;h00gص}͜U;ʹ΅߭`rD|T4Z	oI#1`VCF4"8A(NJj,CO0v!iFr h0&\CΪOfv6uΗ̪GgffX/;eoE{)`Hq9ic_O޾<悡="$vOf9ncV|ڴ4iRX#1C,Nl%)dHAZoVP]P8~tRX9au	M\jt<Yn=U嵳-mx	DD>f(Dc@iXR/l22)CLtԓ	y?r@P?ZLf0WJa`n+r=T[I;ʏr,AS"BD=s~ZǓ)i*#1A(i؋.j:fv#=\MB72"]!$t#/X1PrTmQA#.J@D#M"YZ͚ͩlUu[=/2kdҺ|%LUJ[֬ `;u|0<[*L!)YL>UKd0דTY%P>˳<O6emzzsr1ۣ@)WRS$IkF#1[h#1!Qb԰mHڲ&a[:1"BPc|ux{_x>DxqkQE#/w_7#+$x6h^pWaZ @U?r`j֕2>t`blYV<N#/46"똱q~t`7UB5ٵ%d??q"vb#`u)HRAtQWVr1"Z@&bXEX~̗Hk)JFl(3̪8Lq_<<CkxU</"`-<H(?54Oji[9P+73~&xA/Y"@͐g"Ӷu`?ie+C7j]ʤBۄWR<F#/M߇#.Gb0< oqT2y.Kv2xO9֕dnVy&C<#/!E6"Ci3#dHPL#}ùaR=ED &O[)MtS4q`e3bXF>=a޺9fxbD`GBJhXslr12o(>vR	8HHH.^SCz H$!=7xA	 +e#.k#/F*[ڍfI)BI#.XB#8np+mWpHұl<{	hѢulNpCiF[jU$DHkFs0Dce,y0fuסZ\ӂd0p؂C!^Q	!i&3GʪVF#>;6,M o#1#.}OO#/a#.JPjd{Q!c(C&j}dH~4QA#/aAKgF#.ûqS}C3|ىqᰓH/~1SEt9 .\\8)N7F-vx`!ud!#1*x{W&nÚbrCU%Dm#.<T-֐#/qC*}W 4##{J"DZ#16NgJĆAi}bEyb˘!@@i/>nDU?h(eU;.콷Ŵژe&JFi?ԑ+Uыm=ů/Cӹ+L޵ČkN	i2#1g/(5XPϻ#.3ZD=d̅Qq{u}okt=~#.R;K#.946gs;_idumjuEwޚE@%RԑH[)h GSlȱkKKN=(C,Cl+ 2D*Ȩe[n*̙X$j.wz*XFS.!5#1Ƙ#/<p1LLc#$RnLV:5S"T@R!MAu2	('Y$wDUmi`?OAEEAd	A7u6y^EM2tκ̩U#,Q[YEɆc#/!6) #/U PU⴦[Xҳe31Fm&ҪlJэ#1,bQQ0e5+pzXuW3fiȷ(Tl"&@0h,yXdQ,*rXˈj=(*#Iոi	7YHb#/#atom40}-ƕE:#dbhncLlci#1t)jmLs3o\P!#1b8͝ɒa@Wza䦄bTML$EsUhLSFE邛1V1hU#%1YXQ4#/7mȂ]X44ZUl շɖ`	lC,2;h1oݗ6#oaqe&2sD	meQBbiVݧJԆHvagkc,(F{{Df<w3\V`9{3=^Y[Ge©Y2=OV|ĎX"<LE-h9oO*oIc#/)_3!ƃ)AD^[`	}Cqkꬔ%2߳Lg5RqK|XJ4oPqrŎ3dQ<KiII,RMk!1S/#/!09pzTV\貙b<Et#/6Ҵ5v慢:/BQP4KL MDFUnoR3t|bJ*M!-(B B#/Ti!DQbu	WB2Ҥ*Hu88=t=5Đ#1^!p{E?dd&MkZU*'n5ds;ɷvnghxf=ɯmciw32Qr^HONEeUx+qfYdfMK؛VJiщ7/OQyFҫy7	h""7r$(As9I\0m2r2omxD`fL [7<q:L{\#5ïds. |xX^^m:M4$Qϖ 8ؽߏoDqcg0[66J<UfygMs@DG1ܨ=p05@ޖ4bl2|.mc#1#ai.#/ʟDAQ$Iy1y#JFHgo<<Mo4۬ڮەmI2km*܌-bifdTXSnpkIm%Slijmy*eVM[tyW*6lXj8dJ^"lnLq^9\XV[e+[JPFmaY	\HMlj6<.nkD.F$V")(ODݾLXR4HL#.RV񫩖.Qk#.RXU2ت6e#/F3Ek2MSPcm3X[5z۵]߸(6{P3 m@R"B2B6_yնl</mPlʎ(x&׳+e=Q&_SM%2oͼ٣jZT[TYtwWM)s[Blm6ez]m-%DF"#/4S]#c!_F5	<҄!̓Dm2H(ҐЀo'DUbRI#/jt>a-(Pn+PwE E"f]CZj|S^r^G)*1#<$BA?X#"jbi0}WIH2Dr44(5j)АXn+p[wO|<;G3%ν\ca+GA-#/؈vCG37h\bO]P#/QGi#&tfe`tFFmY`	#1,OCӜQ-Mns$︨"@uNxp+}n+[r{cDvbL5ch_j%= "B*#.B jsbXhL	!dI6 S0>%#/p#/ d|OS%P0Y+*$`@x Z57Ѱ3aDB W*2֫ͱo6;oe5u"H\R{%%!"F@HqwiZ{ҥclh#/3S-fDlJ"Z14 A@,PP%bk`ВE-M磻\HmCB	mdDb=q(y?Ë^sx%`Ǥ*i,h v=:.كnl0Qf5mۧ~{@vİ(	Ȋ#1HBHH@uqAW/*1wd>p&~.L{SZgmld#:2'@&}ʟT(?Z/8GK1T	NgbHU8gDpÒ48~tB!Ri!@2H`b51W!GA F4ǜ-z"1Pe"\AU2zz_Hf/ʖ6͝kB	\&1k6#/@}/J(TT"4)rUK\#.n=w;(BT#.;R|04o]vTu[m;5.Y$G˘ȘfK9jn7hiKI2kJ]SV)6(d6&kIWWt΅wvxkԥyyviͳ,ly[v+XVZV5&MmxunM)lܫݥEb5R P#1QxK=PH&`mSdBt~:4;|{Ѽ#.:B@$K)K#1Qu"TvpU EHϫ]MyF6>$bo@#/;+͵*??rڢfA1QETJ)7(O#.F _ÕD#UMXZ3hɫko_5V]JjuԳTmfޮKe#.0v[u*K/P*iKhi K&R!HL#A{5ҶLb0)`#/&cHlQYUzuDmuJґf6`!DP$R#b	4'D?A))R1E	LLQ(Jl`Q#/j_KUFlSCR0 x[vxsW_5㵾VTʢSR`#@9>,Cnv	ioFҙ*Klm$mQMhkT	rጏD$E#.ޢ6Z-PINqeƇ|:t$BGZPYm}\]!#.Ac!4X?@Ѓ3)SO42ͼ]3LD`k@h?M`1P2cHí[@ZvF{ZG6	`j#/#.Ւ4#$`$M,D҄PBA#/ڲ0@Ed&0p%`PR[`6n`Z	QP4L cŁ^#/1B!V6֢ʣAY5rEbH#1`I;}3r6پz.uvû9UAbLC6K	6r1wD$,d˕c	p)\`jl^F DP	TQP$b]0fc-F`Me&Mp'?Bc#}_ۨh #MPb"H4^Ɲ<g6(T~?#/`ȑ$ 	A{8TD!P>@ 倥0i_lnwM̷-;-!4Uz#.>I"mxFd#/U-&s"&u[`BHk#nӭ]ר`1jf7dTM8Fa@>|J#/8	Ii(J#1_&KmLq7xwfZ`h'>zu=(7#.z	gr9#.m<Ij)I_K_?[vl2RF+tQ8+	 Ȗ`"@@N!#.bvL+M"-u"A$aQog~FIZR$0XVscA3UCRA@H\=@ƅsR4a&P~)/7ضnD#1ZR`>xA:9aӤ*cy@fΛ)9W*\ݜBi͘lc	Ź`&#")+6KO'Oq{rÎpx{DړL/QєsTE{N<uoeUS|uQqɳS&j+Y/MZjdDID:B4R82Ai9bm8\E$,q#/#+L,Ae!đ)1S-$þV(}A'J73l槵ϱoR'Kw+\8^PGvbЏ{vx~ G"(j$j߷xڤL焫t+Y`ݞ6fq$uB[\[lcɁ!lOdx=7F5)ՐL8ތA	0P=4D%ԍ@0ζaoA>(U햪#/eU'óMf,WZU*0=BhKn?zć:	ץ%q#./tq}Gҏ2{=N}|7a4cs2O3_vB",Un,Z#a$#/v: @z7wiV8Vdr/&U7[D 	I-0jHRCU?=l7=_KƏRr6-Ȅr&F=H- jB{pцS7ډwd_vߣb]U^d<CrX !߈h!2ocl4Iß0%YE{ҍpmUEfo΄6nWU-3ޛb$dhǥ1-H~{iZ:rRLp3CYdgYfV:VwT$$UaJ.a	g0t!6E)Dhc,_?+OP$maO5?y }"!*DBDVBCijGs|>И#/wl齑AYz;آ堽Yda{"mmzvi[y~kE@Ő! 	S5|5-RJ<G}X+ĠQCC H&9P΂qz/0BN3Qsd3#/\Ynڴl4m2+#.ilN"lTPI#.h"wD)F6R[:=M@;P'D+LRd +-3-?u`1,] UZ/i#.}vyJ4UHZX)y o,2v|譜E@B$_9AS3';b>#p3DB(Sg"۵)bj\O۹ZnMm.#zʹ#<\¤`ib̈orjcY@T"`!Q.?:܆)_Bͣo[fBBDEZmG@WXr}_qͮfY\!BBބn%ph':h(I"vr4[*_O;A[>(#}pQ@ b(j$#1|(	cts.|)G\R|bxg[!d,ETJU5MKEA H#.FtݲnxP5oo~J)*4j6U}~w?ORlFZI, Ha&j&L(B"4D6eSJBV]+ZƹȬrfSZ2_V]"lH5\o(jtf9Mm1 7Mjϧb|v}ʜw|Y)'*4<#1$ЇÒH9  B'|ƓgQ]] lZB Hjl=AQPp&0#.r2Fq|aսJ3&5Ed{(oU_Tǁ".<yriJ/$FBHq`F)ǽEIg@*DdĽI١ԁ舌QY-udh FR	*v$!$ST;BFj a˴mm7]jMn˒溻(6RQ#/0e@̸#11#/c `H+(/Ax#1{Z!r9Ǌo_4ZMi[6u-筤&CZ=#0F$i6BP#/Yc\Zn<c( Ĭ11}0ìq ֍/qx.ԠWj e*K,qCE#1ͦ7Ac/[\*-&#V,b6R<YGnƫ]Rm#/#0`P2B(&6lm exǁ1fiƵp^뫨Cvªӥ2&L @7	HʁbVfn1l Bf0Ԇ11R0iHIB=<0nٱ]lS[n04Bbmizbxfo3%X(=d>m#.0|V#a\e1FѦ]ÎF颲fDƣ62	άcC#/6ZȲ0qFoRflYn(8ۚǉ@5Wd.tJ0R<GXpI`pbq042#.Ey6e(6MEw0g/&C84!ۓ\b2،TAڭJ45+I9FV>hLr,*-@0pA4=5GK#ɜԷbApl$F#H"&p<{iAuFDCh{#1ݨ}cB=aWyahCC*5CM'MCb+[lͳcXQa5BŅ#1JBg0ԥɱmx5Ki;@4l32--ҢClm'#.tѸ(@ؘ;0fȈi- `IqHPBX(ȩ0`#//i,!#.wANr'kע7MS!n&ǅjr²Nb6PUjBQ!GYL#.&e֦k Ԇudr(TH[" \}gr5Q=4QB3<$5dQlZikn69Cc}С($;W;Rtvg"i(&B?𝄩(YZ(Ș#1% i/%J$)F4W6 R^SVա댨bP̀Ee؃ES 4Ȣ$ݩHnjne"g#/E5#.՛sZϒLaUm\J.5~4xx[{qd:2Lb١\"&5Bc玸7qwi{S&7Ԡ3`{'`!"?WMd"M%'Q(ȉ72s$cܫuQ|e@8;h0F"!#1*vuuz?~/V;^K͔Jk#.U6Z&\9\#1p=;-YjKq1{:U$#"WWҝdOʏm蝬xeXe\m/#.+)2=3tTP*"55#/%#1Ǚ>=%P΄-F(D0ȇw3&!B lQdU "*sސ0'tg#>~<GV2J"pbMA% #1$4Jd٫o~q8#.q3a2!Ls5A}=^yA<sjc۠J`e[)$zxsUxyլQ9mmxKIBPa\0ip+ʅ1I7h'-XV<aMzT@2ѕ#.Y$B#1@_+hHUȊ1nUx;JR#1RZ[k_xf̆Lk\{\f]"@;˅|@<,FVDAHE-k6Mkכ`In8bHUD$VHDT>uh:&ϝUb}C!IEaǠz#/I#.6$ə}UlJuG37(N^ٴ3b#.}1$Hԃh6RŖHlhٖDLMFm5U(dPqaQHLb#er$O#1\CU11xAGrXU@EZ7nޯuΨxKs\u	P*6A[skknS5j-vb($@YňOgk;KB$-Q*2V>HT1A(	R5#1;"RWჀq:~/LKZ*M*؂!	^kr;Kx6;9qh!Za#1P&-XAz[+1#C5yR&\mp!;DU^J8u&\_wOkmʢb`@*(*0C#/l^?H%7#zrħ4ZŐfH~q¤ǜuDEI˝OӄͻU-ݞpqߊ÷xt%Pz~,s{>I9|w:41yî\vrLR؂fUjw{_(B0;RIw%O.P0ҚrYVDuLeZP!Pg΢	a &uE&)b235ߒ*:ĶCLTvh#<+ffçe:<FL{y<Q8~<ߗTa_[uە{4,)]Ot)nX)z9|a'uT-w;W1+Eߜ{..zs]ۻ!3Otw-1/WT``ځ))m#.	#1!vvuP)TTj.( n\	!÷J,O˯^.^y䭬W]uJED>?xAz0#67ID	T|,څB+%9SlJ^ӫ+~"`A%N xFc7'ybuO_j1m\]-}azsߩBl"QlAILOPHݎwf:88mϕG̹jrKOp}p03RPrVN~~.:\S!ӏ1LWAjQGeݽ{aEn,C8#f3c"enN("PKpCCFα#/ v[.7Ă~6My8'Z3#/;=&ݻADu;n(%bLsA_#1k+XCq:ɚ9.gLXe#/N@Fz3H|r>W;Kߤ4#NG=7[8kw x%HE.˿.bwTv=.u#բ5G3*gz|b}$JW}~ze&cj߁?'kIxU#/ʔz&m& ^#/vS9;7Au]zlv@D4	\n"rNFY6F%	K:rBr~P`0NȅeHg2w8CXȔ|;ka;rm	G=Y|4ۅ0HF,Nb&5s!0%/,қyKA]ǣdf!&)p&6v(2x >Q/LQDaAuSٰZ])Hu!m`m#/6bCNXw'(L04#[w1#/U<Zld57#/pPRSz)֓UiI5]f4z~}fv nC!K*:|uf#.׀K#1`,˓Uͬ.[#/s8_5>|j	R@)&-$n	K+Cm94-9hFQPzXƢ_#/+Xam>ZG	zh%GYOUK_6t|BM9GrZx#//z0#Ϗ==6Bc+<wAŜ0+	׃au{ʾʙ'4?5yjd!sI;#)*zK]KjXͼ2`b#4@=a?<s_H^<i8ܣ';NUhAR,gv}:[>Dz2#19"kMKngLP#1zG˅Pkhae"T#/+6胝#1H#/ӇzFwU.P{CRD}<%ǲ;N><yo/CЫ?V͸Z-mO_=L'Q	{\XM#.CKQK'&vKQ'yeJ	FT),FP,Pw8WaY*zlMFyܼ23%v먮JOP}RePPx)Xc@#.x-"唅4Xm2zɝOsjmlm҂,#kd	Nҙ˔U#Uu#1y/a1CΕi5t5QvY*(Tj(C!܇3b+ת:Ӑlq	Տn0:%,q80f	Z#1гBo0hg }gQM|9zp܆v?VAr~jˠƗU<%7#18{zۍ(1@:!r`I#14PtM`:o=_Gߟ@Q<oŦ M#1]j<`R3C)F*4BWڣA5q9z]kޕTQ}n[fFҮBG5L#.5AZP*aEDQTQ077WΛDK3ܴ.46-iM Db;ŐMMx6JP#.dEI@)Z#.b5	4}Z˔*[kǎ[$\44)ێܨŃi;?ɃfΓ)4rHko'`*Kjm1qp<f)sדA")EFBEIj&Ugz6k*n"0 dʄv=HW)bLP(YfK-QHܩfB;s-724k(aR40(3։9'E+Kp) `d䨆7&еe-ncQsI2&BfW+-Ǘxַ-M24A#/(YXdmfꍭFk΍fjֺYJTX`-j6kFdĵwZܑ	fYƵ)6CVEY)-v]7J#/ c#/M.$eH!0bRoƳq:!3Uovjg&Beԩ1mvsB'mJWme@iN	~MF{;Гt4GZ5d3ll)1ZQTPFohiZ6p$n^-020x*+]L+hiHdov[0̼4Xхȹ :lLilsqxvM#1l}#3L2]5)C-U-41jGc&0(б֌fcmO-elպWdpIÙ&FԡeGUaAT*|ff	V"F"=H\cӅ:ثkǈ--94koKfR3N/m0Y%#L3	]#iM2%7`ǌ!^5E PQmZ#1P`L@L"@DN	)r>!HUǎ$|~@&F󯉆I05TTg	B)#1Q!8%:Ot/J.w)~o<7<nk8#.$5āwPNȸqؒg:}$]pW|ꠚ#.	aZ	d~ kXo#/j3	nqP	Ln2qf܍7tɒTn=Z450#1)jF5A[<3Y:t8sC'()AnZ*U$~ˠ}gWy۞N!U6I@$p@GVS*[Z4W\jZ#1#1RJ@ɒy#1#_o]|e[**i6iAhM3fiIZbV2Z٦)6Ė6fdeƔe5#/mL2TERlllifԋ&lJa!$aFDAy"ǀt9쉥X=ܧc9f}۾Au=_:ݟ"o}[gm"CŴO4W	$: Î#1vHdXn$XLKp#L6568^t5do0XNT9Zז ܀,>fᙖ;,t ~}	!~^>=LCDEM׌!k:,±-Pw%6yQ#.IB5֊>m$`ͯ.ڥRJJG֖B	_$Lh?OHԱ#/[>+w7.CQңqN2hm.ՁBE1b(:2Ժv9G161V-,LKi 1s-X1,2#1Tue	1T	K=ٯTvlӳyFp[QSTUx0<<dIcIBJQ22+(3mC3뗁";"~Hvtb"	yhDZ|I6oudRL5GWp9Y|2FۃMKFId$1A]#/!41URxFj,4!kfr`EUgk$g>ދC#/)J(FE zO2<r'ç4MdŇ"$!%&/b}^2Hn1}B>a)hK56zC3B&D!![Uq/.ZֺPAIRj8&tbd.ߜb"M,#:l^im+t{Mq-QfMLmreCcAAd]]#1?-s]u;NAB/{lut>;P69;\@)*u`m{	2/Y&rL*oޕ.c(!$ws;#1z`qp)gq#1f!J44x:7S@U`|6h=Trp(xO[yD^؍C\_3a"A/%(qswc #"XXS?n]|#g"9?SZ^/jo#1,1Q-jܯ#.y'&$50l7 1a9]vlr#.`c($˼_"H/ZD!Gl*Bk%²8_*t"YjStM".	Eq1Ï&KdF#/j$i#1`QYHm"Y1Ҵ@?@JQ$|8sd;,sg2L>D#1Px7T & ѣɄlh}"<JȀ!`^N@5M0Ѳ#/Pj@(_+$:l翑t3M*S6tԂ'@Kf(`7(Φ;9~)6ƨA&AAi\l+G=I-MF# @[T)l?2MHd̒wZx䢥P`#12YîVk%@9s+{p՝_O#"aTPRid"aB8P\`ʅPxv|9t6{xBYj|Nx=Vu+r)t%69n`g|]ҁcbXJ!1 TD2AzZVA@03@e&MFkqǖU8wG0D8!;2cygfc)j`1.1)q0c;۫a$'.VkC!Q>сF)4k7AN*w]1TΩTd:ə(P>Y#"|٢#1On=;"}2n;gDG0TɼmM:-JIúQ|:T<K)&ڊY<>bvI@<Po!F1$"E ^N%pvGHZ^x#/MRzL#0}KL&j̩Mj`B%5#/H%35|YC3"#1]mNA0lvBJ7H&4}p}Hv!CvR!g'e3[:#.\#Eu22"?LA^LL"trA< 0tfcM6vjE6X%*2eٚΚ%8Uˠb.CO!zh,04Mf-#/Ԇy{y19G>*Î8(c*lJBIphK1D<& @kc(`oV6KQU=oM^ï F+psYqBDJ7i~J":sΞUt|:W58)ЗP]KK$%mf#a8쁝#.f+u0Z4#/9JC*~KKqZlj~aEw`)u%y8gʴ[rb%}1<)_bzשzZ擛F)3+S=#1fH -o0&@17YFIinxdd#1^V1/֚Zh0ǂ#.ӵspfL.0v)m'3 oU&vNe(oGD1DI5#.'RDDMpWŒ´u&K=F+t$7Mj"21uM(IWJɯ8B68[bnӂYI2mq#.3{#.41aɰ-\`Ī@əъ"1j%nrf&"@&˩e]Zgۦպ",5DCKD*3#/BA(	1PTR	Tә.pv%$#,Ȧ;Z($a,`#D@tDDtU<U6x7jr7*l)p#.."14]>+f1'äge.*cWh1} Ohsr]cX߬C@@HH#.t =C#/jcq8VR?#mt#1.D3r#EÜ7*zSٷzMaqT&rp)&/++q+xi|J}8gua"f{'J)MuP:08mZ?/=0p/gDN6C#/"~JC\,&2d>>;΍.׀'5v&<J-Q#/BYQ&af{4UFW;aZ'jNy n٘uJ=ھ!DK .'91Dʹd{DDAm.gM#.#.&4U,"Wu]ӧ]:qmJxE<q7hJzu컵E$ѯW/7u\iκ"8G%bATB1U,he&bŰ(A8[դQ0<FHo֝	x{EҨ@G!ci|4nc/w2+V"2[e!2	lf2]ĠPup	#.w6U7; I'ΝZ#1bV$/pz{4enZ#_Dux"sι"O/X/٣Q\`Re68(f|SukTMGXH;!:H)nZx$J%^\,?Sȱq#3Ś`R*dMpͲiFL\@|w,>x5>ȋ@)#1Y[56Z(Ȃ08(40a3z\S$.aɈH+m@N*Iٕ@:W*muqÇ析gă$ &&[6'~`$Qd#.	Ó'#1RcI,<ljY)+PUHq0f`J$mDYs(ITvѳUu4julW^yve4@#1Ȥu5\	YQٯ!3#/H!ei]2,t_xޥŴ㉓#1kBqtA)B$:|(4ϕLc{` ;:'q@#.'vKXKQ4YTYp#j"ʕ"@EֻW{P9爎@ck1#.Y#.,*	B~_ˆwW_w2!LV9V?<.LFhEuȻ@}96rJpg	\?8$;Nimo\p{+Jma[[:^=AY '/-K(@)ߘϚ	/Ϣ}glrflS)DTmC5e~F[&ֹD#1$@i.<CdJA#/Oy(<O`beR[; !"4	p=P}a׵I5sU_pwwHa#/`	3iq$G-MwE`f8Ҭ* ^#1gÈs,Q#	eSųM9,	,Վ"Ex,XcHh:양D%XYRՀv[lIK#qCVZ._#@v];MrE$ ,#16\!u-c#.w"{8D`r5#1pL]k"a#1KH:j0`mlҶ8T:jM4&[$6ɴ+ G#g!)(;+v#/#1(@@1$@+9t뵽 Ĝ{"k!#/<#1#1J@#/dza7)A+q趃 H8`{H#.	^B5?PI˝ŁESkǦ[5`Oب41{p`wwyuf	fPL{>1àBh/g":?+Q0v[|14־@ZA!`}\d,)_vٻa`(-d$!	L^uʍ)%$H叧Λ5~;˵k6&UQzn$52\۩Sm0AХ)D $S`ETi?nH0>x|~r»I#.cvkj%	(EI.A@LCFY%IKELFR̛fk%(QRe4ŲbVk?;/zm4Rk7@,){}S^7]sf&MFkfQͦF=yVKMuQbwA(~wWy$W"jnQ%nD~Y@= d|dH#/W^(N`fsHȞGG!^?ֽ 0T{>{hO7 R Q4TbAK/٪pm=b oX;sGr3^#1du'h/)-[>Mjj֊VtXڵ[#/t1ғfEv1w(V`(*+ʉIl{qZC.JNy\6|9zL|p?7-)f/hG/`+^j}xxvQF̗6*?3J0wlQQD#		"M(Qђu~V^()l(>2%B~&HB@ɍ(*I]XA)u`'#Ϫ9.wRO2 	cX([%XU! 		m||߽(AHrdfm){F0acH̡Npo	}6h+IGl),ĩdhh<@ w/emͶ5nIJ=-˱\ڋZyڶѩFf[Y2?Sk20+Bpd-#.,&gs$9qPĹ3Mi6bhH#.0!_n֌KL1E0+SoM`t׏!nbFUfKF]^5[%vjfeWsJR޻l)"A,%H\8b=Uyyuu+R[diSLev	M!J(Z$DLVFê<d\ AFf; %^^TZjO,D*uV6`fݭB]Rwe57/|Mp)S5{ϻoTA*kiv$C<D! "5	#/tpIw%m$m/uÚ$V8,16ZUwO_q3&Z͠c))	HbO!|*>NYL(4=|%&6DoWtLvfЀoXK{%!$47Bx.X׈4y|xaDZkr=&w#.l"x P45x<E{l.,o'}^ݩ>pƪS0Z7;Ԕl/BlM5#.uM}+RJ5{>7W,XFYwGInu4GҭKyKR_Ա.	J#/3FJ%!2\jMF',DR]GZ<!'^Ж1u-@#/=e:*ǎx?)!G<mHHMq(ܐ0z 6ܛ/j#1ù9,5o#1vDR~ MP?5k>:e$Bt.p5\ыJΕ0Bm`q2B	/z33R+dn:bc#.#/ v_oo٧__~_w_O?W~G?~OiNcS~GXr65Ʈ	UT#1uHB@M/R9 ۄ#V7@^j;T'ldJ1F1#.qZn4~SVvC[7[!TK,8VR-!7/Lfx̧H+@=#1?݀zMy~åVc7biQlMH9#1?ѩ1LT)Ѐy#.+ON$l7֑a|e+2CXr|ݥouHA}l}86+!SԕlvBVoqE.+\hVrUbH)l1#/*|L&P4֩0ۣV0`=8F-e[zhId^/l[[W}o2ĲiJLCe`D)XG*cXwirf3}0B9):ȗd9Og#]	/ol/k-ns:C-0bQ}ںmA>	TĜ~U[ox+#.TD8O^᪚ۜ1VJ*'1kѰ]tK?~|CC80<ؙ"DBzR#1îۊwmdr5FA5wKJ#/#.|ER xlcI@Q#FRC~K*C;q{]/{cs3~VN˝dD @-tn[]6嶳E$XWT(Z#1ZjfRd+aHy##.oq-*~]EE%d]ǋ@b\%A#1V*ß|jATnU?SSv&h_z1Ax'ã8$QQtxaBfA}`]S)ͱK}įhS|[S=#/?̓(5bZhϣOoM0m O%;%͹jmZt#."l04e`cTZ5j1i!3)#.f #H(2S	ļ\tʌ%`wcƈ\JIuFI8;r.y#/#q-d,o>ISP{/Fm꬈HMt"{Ca=Oa\=bĂj;UP*Bfj%y^JzlSf5/{k"c^#1	D 1"'Du!{}6XFt뗊sNz	4.6-h1K8u0l ϑIGVgOЈ	o僾l0m=D$]U~ُ#?Bpf%W/PszKx~cg'6ַLҶNHygj=~ȟP}h%֠c{9)rsUoQ7ݽiBo3EE@K&Lyn[[.r*t8(wr}«bI%.?rһxT?#/<|:v8N`NH#8dJy'pBיXdC)?zzmfp'V|*f7#1=;<h]u-h$dK7R핍i2_/3ܑN$;#.
#<==
