# Copyright (c) 2012-2018, Mark Peek <mark@peek.org>
# All rights reserved.
#
# See LICENSE file for full license.

from . import AWSObject, AWSProperty
from .validators import (boolean, double)


class LogSetup(AWSProperty):
    props = {
        'Enable': (bool, False),
        'Types': ([str], False)
        }

    def validate(self):
        types = set(self.properties.get('Types'))
        conditionals = ['api', 'audit', 'authenticator', 'controllerManager',
                        'scheduler']
        if not (types.issubset(conditionals)):
            raise ValueError(
                '%s must be one of: %s' % (', '.join(types),
                                           ', '.join(conditionals)))


class Logging(AWSProperty):
    props = {
        'ClusterLogging': ([LogSetup], False)
    }


class ResourcesVpcConfig(AWSProperty):
    props = {
        'SecurityGroupIds': ([str], False),
        'SubnetIds': ([str], True),
    }


class Cluster(AWSObject):
    resource_type = "AWS::EKS::Cluster"

    props = {
        'Name': (str, False),
        'Logging': (Logging, False),
        'ResourcesVpcConfig': (ResourcesVpcConfig, True),
        'RoleArn': (str, True),
        'Version': (str, False),
    }


class RemoteAccess(AWSProperty):
    props = {
        'Ec2SshKey': (str, True),
        'SourceSecurityGroups': ([str], False),
    }


class ScalingConfig(AWSProperty):
    props = {
        'DesiredSize': (double, False),
        'MaxSize': (double, False),
        'MinSize': (double, False),
    }


class Nodegroup(AWSObject):
    resource_type = "AWS::EKS::Nodegroup"

    props = {
        'AmiType': (str, False),
        'ClusterName': (str, True),
        'DiskSize': (double, False),
        'ForceUpdateEnabled': (boolean, False),
        'InstanceTypes': ([str], False),
        'Labels': (dict, False),
        'NodegroupName': (str, False),
        'NodeRole': (str, True),
        'ReleaseVersion': (str, False),
        'RemoteAccess': (RemoteAccess, False),
        'ScalingConfig': (ScalingConfig, False),
        'Subnets': ([str], False),
        'Tags': (dict, False),
        'Version': (str, False),
    }
