#!/bin/bash

if [ "$(uname)" == "Darwin" ]
then
    # for Mac OSX
    export CC=clang
    export CXX=clang++
    export CONDA_FORGE_MACOSX_VERSION_MIN="10.9"
    export MACOSX_DEPLOYMENT_TARGET="${CONDA_FORGE_MACOSX_VERSION_MIN}"
    export CMAKE_OSX_DEPLOYMENT_TARGET="${CONDA_FORGE_MACOSX_VERSION_MIN}"
    export CONDA_FORGE_CFLAGS="${CONDA_FORGE_CFLAGS} -mmacosx-version-min=${CONDA_FORGE_MACOSX_VERSION_MIN}"
    export CONDA_FORGE_CXXFLAGS="${CONDA_FORGE_CXXFLAGS} -mmacosx-version-min=${CONDA_FORGE_MACOSX_VERSION_MIN}"
    export CONDA_FORGE_CXXFLAGS="${CONDA_FORGE_CXXFLAGS} -stdlib=libc++"
    export CONDA_FORGE_LDFLAGS="${CONDA_FORGE_LDFLAGS} -headerpad_max_install_names"
    export CONDA_FORGE_LDFLAGS="${CONDA_FORGE_LDFLAGS} -mmacosx-version-min=${CONDA_FORGE_MACOSX_VERSION_MIN}"
    export CONDA_FORGE_LDFLAGS="${CONDA_FORGE_LDFLAGS} -lc++"
elif [ "$(uname)" == "Linux" ]
then
    # for Linux
    export CC=gcc
    export CXX=g++
    # Boost wants to enable `float128` support on Linux by default.
    # However, we don't install `libquadmath` so it will fail to find
    # the needed headers and fail to compile things. Adding this flag
    # tells Boost not to support `float128` and avoids this search
    # process. As it has confused a few people. We have added it here.
    # The idea to add this flag was inspired by this Boost ticked.
    #
    # https://svn.boost.org/trac/boost/ticket/9240
    #
    export CONDA_FORGE_CXXFLAGS="${CONDA_FORGE_CXXFLAGS} -DBOOST_MATH_DISABLE_FLOAT128"
else
    echo "This system is unsupported by the toolchain."
    exit 1
fi

# These are set by conda-build. Move them to CONDA_FORGE_* and unset
export CONDA_FORGE_CFLAGS="${CONDA_FORGE_CFLAGS} ${CFLAGS}"
export CONDA_FORGE_CXXFLAGS="${CONDA_FORGE_CXXFLAGS} ${CXXFLAGS}"
export CONDA_FORGE_LDFLAGS="${CONDA_FORGE_LDFLAGS} ${LDFLAGS}"
unset CFLAGS
unset CXXFLAGS
unset LDFLAGS

if [ -z "$PREFIX" ]
then
    export PREFIX=$CONDA_PREFIX
fi

if [ ! -z "$ARCH" ]
then
    export CONDA_FORGE_CFLAGS="${CONDA_FORGE_CFLAGS} -m${ARCH}"
    export CONDA_FORGE_CXXFLAGS="${CONDA_FORGE_CXXFLAGS} -m${ARCH}"
fi

export CONDA_FORGE_CPPFLAGS="${CONDA_FORGE_CPPFLAGS} -I${PREFIX}/include"
export CONDA_FORGE_LDFLAGS="${CONDA_FORGE_LDFLAGS} -L${PREFIX}/lib"

export PATH="${PREFIX}/bin/conda_forge:${PATH}"
setlocal EnableDelayedExpansion

:: Copy the [de]activate scripts to %LIBRARY_PREFIX%\etc\conda\[de]activate.d.
:: This will allow them to be run on environment activation.
FOR %%F IN (activate deactivate) DO (
    IF NOT EXIST %LIBRARY_PREFIX%\etc\conda\%%F.d MKDIR %LIBRARY_PREFIX%\etc\conda\%%F.d
    COPY %RECIPE_DIR%/%%F.bat %LIBRARY_PREFIX%\etc\conda\%%F.d\toolchain3_%%F.bat
)
#!/bin/bash

# Copy the [de]activate scripts to $PREFIX/etc/conda/[de]activate.d.
# This will allow them to be run on environment activation.
for CHANGE in "activate" "deactivate"
do
    mkdir -p "${PREFIX}/etc/conda/${CHANGE}.d"
    cp "${RECIPE_DIR}/${CHANGE}.sh" "${PREFIX}/etc/conda/${CHANGE}.d/toolchain3_${CHANGE}.sh"
done

mkdir -p ${PREFIX}/bin/conda_forge
cd ${PREFIX}/bin/conda_forge
cp ${RECIPE_DIR}/conda-forge-cc cc

for l in ftn f90 fc f95 f77 gfortran gcc g++ clang clang++ ld
do
  ln -s cc $l
done
#!/bin/bash
##############################################################################
# Copyright (c) 2013-2016, Lawrence Livermore National Security, LLC.
# Produced at the Lawrence Livermore National Laboratory.
#
# This file is part of Spack.
# Created by Todd Gamblin, tgamblin@llnl.gov, All rights reserved.
# LLNL-CODE-647188
#
# For details, see https://github.com/llnl/spack
# Please also see the LICENSE file for our notice and the LGPL.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License (as
# published by the Free Software Foundation) version 2.1, February 1999.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
# conditions of the GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public
# License along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
##############################################################################
#
# Spack compiler wrapper script.
#
# Compiler commands go through this compiler wrapper in Spack builds.
# The compiler wrapper is a thin layer around the standard compilers.
# It enables several key pieces of functionality:
#
# 1. It allows Spack to swap compilers into and out of builds easily.
# 2. It adds several options to the compile line so that spack
#    packages can find their dependencies at build time and run time:
#      -I           arguments for dependency /include directories.
#      -L           arguments for dependency /lib directories.
#      -Wl,-rpath   arguments for dependency /lib directories.
#


# die()
# Prints a message and exits with error 1.
function die {
    echo "$@"
    exit 1
}

# Figure out the type of compiler, the language, and the mode so that
# the compiler script knows what to do.
#
# Possible languages are C, C++, Fortran
# 'command' is set based on the input command to $CONDA_FORGE_[CC|CXX|FORT]
#
# 'mode' is set to one of:
#    vcheck  version check
#    cpp     preprocess
#    cc      compile
#    as      assemble
#    ld      link
#    ccld    compile & link

command=$(basename "$0")
case "$command" in
    cpp)
        mode=cpp
        ;;
    cc|c89|c99)
        if [[ ! -z "$CC" ]]; then
            command="$CC"
        fi
        lang_flags=C
        ;;
    c++)
        if [[ ! -z "$CXX" ]]; then
            command="$CXX"
        fi
        lang_flags=CXX
        ;;
    ftn|f90|fc|f95|f77)
        if [[ ! -z "$FC" ]]; then
            command="$FC"
        fi
        lang_flags=F
        ;;
    gcc|clang)
        lang_flags=C
        ;;
    g++|clang++)
        lang_flags=CXX
        ;;
    gfortran)
        lang_flags=F
        ;;
    ld)
        mode=ld
        ;;
    *)
        die "Unkown compiler: $command"
        ;;
esac

# If any of the arguments below are present, then the mode is vcheck.
# In vcheck mode, nothing is added in terms of extra search paths or
# libraries.
if [[ -z $mode ]]; then
    for arg in "$@"; do
        if [[ $arg == -v || $arg == -V || $arg == --version || $arg == -dumpversion ]]; then
            mode=vcheck
            break
        fi
    done
fi

# Finish setting up the mode.
if [[ -z $mode ]]; then
    mode=ccld
    prevx=false
    for arg in "$@"; do
        if [[ $arg == -E ]]; then
            mode=cpp
            break
        elif [[ $arg == -S ]]; then
            mode=as
            break
        elif [[ $arg == -c ]]; then
            mode=cc
            break
        elif [[ $arg == -x ]]; then
            prevx=true
        elif [[ $prevx == true ]]; then
            if [[ $arg == c++-header ]]; then
                mode=cc
                lang_flags=CXX
            elif [[ $arg == c-header ]]; then
                mode=cc
                lang_flags=C
            elif [[ $arg == c++ || $arg == c++-cpp-output ]]; then
                lang_flags=CXX
            elif [[ $arg == c || $arg == cpp-output ]]; then
                lang_flags=C
            elif [[ $arg == assembler ]]; then
                mode=as
            elif [[ $arg == assembler-with-cpp ]]; then
                mode=cpp
            elif [[ $arg == f77* || $arg == f90* ]]; then
                lang_flags=F
            fi
            prevx=false
        else
            prevx=false
        fi
    done
fi

CONDA_FORGE_PREFIX="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && cd ../.. && pwd )"

add_ccache=false
if [[ -f $CONDA_FORGE_PREFIX/bin/ccache ]]; then
    if [[ $mode == ccld || $mode == cc ]]; then
        add_ccache=true
    fi
fi

#
# Filter '.' and Spack environment directories out of PATH so that
# this script doesn't just call itself
#
IFS=':' read -ra env_path <<< "$PATH"
IFS=':' read -ra conda_forge_env_dirs <<< "$CONDA_FORGE_PREFIX/bin/conda_forge"
conda_forge_env_dirs+=("" ".")
PATH=""
for dir in "${env_path[@]}"; do
    addpath=true
    for env_dir in "${conda_forge_env_dirs[@]}"; do
        if [[ $dir == $env_dir ]]; then
            addpath=false
            break
        fi
    done
    if $addpath; then
        PATH="${PATH:+$PATH:}$dir"
    fi
done
export PATH

if [[ $mode == vcheck ]]; then
    exec ${command} "$@"
fi

# Darwin's linker has a -r argument that merges object files together.
# It doesn't work with -rpath.
# This variable controls whether they are added.
add_rpaths=true
if [[ ($mode == ld || $mode == ccld) && "$(uname)" == "Darwin" ]]; then
    for arg in "$@"; do
        if [[ ($arg == -r && $mode == ld) || ($arg == -Wl,-r && $mode == ccld) ]]; then
            add_rpaths=false
            break
        fi
    done
fi

# Save original command for debug logging
input_command="$@"
args=("$@")

# Prepend cppflags, cflags, cxxflags, fcflags, fflags, and ldflags

# Add ldflags
case "$mode" in
    ld|ccld)
        args=(${CONDA_FORGE_LDFLAGS[@]} "${args[@]}") ;;
esac

# Add compiler flags.
case "$mode" in
    cc|ccld)
    # Add c, cxx, fc, and f flags
        case $lang_flags in
            C)
                args=(${CONDA_FORGE_CFLAGS[@]} "${args[@]}") ;;
            CXX)
                args=(${CONDA_FORGE_CXXFLAGS[@]} "${args[@]}") ;;
        esac
        ;;
esac

# Add cppflags
case "$mode" in
    cpp|as|cc|ccld)
        args=(${CONDA_FORGE_CPPFLAGS[@]} "${args[@]}") ;;
esac

case "$mode" in
    cc|ccld)
        # Add fortran flags
        case $lang_flags in
            F)
                args=(${CONDA_FORGE_FFLAGS[@]} "${args[@]}") ;;
        esac
        ;;
esac

if [[ "${CONDA_FORGE_ADD_RPATHS}" != "off" ]]; then
    if [[ $mode == ccld ]]; then
        $add_rpaths && args=("-Wl,-rpath,${CONDA_FORGE_PREFIX}/lib" "${args[@]}")
    elif [[ $mode == ld ]]; then
        $add_rpaths && args=("-rpath" "${CONDA_FORGE_PREFIX}/lib" "${args[@]}")
    fi
fi

full_command=("$command" "${args[@]}")

if [[ "${CONDA_FORGE_USE_CCACHE}" != "off" && $add_ccache == true ]]; then
    full_command=("ccache" "$command" "${args[@]}")
fi

if [[ $CONDA_FORGE_DEBUG == "on" ]]; then
    echo "[$mode-in ] $command $input_command"
    echo "[$mode-out] ${full_command[@]}"
fi

exec "${full_command[@]}"
#!/bin/bash

if [ "$(uname)" == "Darwin" ];
then
    # for Mac OSX
    unset CC
    unset CXX
    unset MACOSX_VERSION_MIN
    unset MACOSX_DEPLOYMENT_TARGET
    unset CMAKE_OSX_DEPLOYMENT_TARGET
    unset CONDA_FORGE_CFLAGS
    unset CONDA_FORGE_CPPFLAGS
    unset CONDA_FORGE_CXXFLAGS
    unset CONDA_FORGE_LDFLAGS
elif [ "$(uname)" == "Linux" ]
then
    # for Linux
    unset CC
    unset CXX
    unset CONDA_FORGE_CFLAGS
    unset CONDA_FORGE_CPPFLAGS
    unset CONDA_FORGE_CXXFLAGS
    unset CONDA_FORGE_LDFLAGS
else
    echo "This system is unsupported by our toolchain."
    exit 1
fi

# Read PATH and split by : to array env_path
IFS=':' read -ra env_path <<< "$PATH"

PATH=""
for dir in "${env_path[@]}"; do
    if [[ $dir != "${PREFIX}/bin/conda_forge" ]]; then
        PATH="${PATH:+$PATH:}$dir"
    fi
done

export PATH

package:
  name: toolchain3
  version: 1.1.0

build:
  number: 0

test:
  commands:
    # Verify the scripts are in-place.
    {% for state in ["activate", "deactivate"] %}
    - test -f "${PREFIX}/etc/conda/{{ state }}.d/toolchain3_{{ state }}.sh"                        # [unix]
    - if not exist %LIBRARY_PREFIX%\\etc\\conda\\{{ state }}.d\\toolchain3_{{ state }}.bat exit 1  # [win]
    {% endfor %}

about:
  home: https://github.com/conda-forge/toolchain3-feedstock
  license: BSD 3-Clause
  summary: A meta-package to enable the right toolchain.

extra:
  recipe-maintainers:
    - isuruf
    - jakirkham
    - msarahan
    - patricksnape
    - pelson
    - scopatz
