#!/usr/bin/env python
# -*- coding: utf-8 -*-

# #########################################################################
# Copyright (c) 2015-2019, UChicago Argonne, LLC. All rights reserved.    #
#                                                                         #
# Copyright 2015-2019. UChicago Argonne, LLC. This software was produced  #
# under U.S. Government contract DE-AC02-06CH11357 for Argonne National   #
# Laboratory (ANL), which is operated by UChicago Argonne, LLC for the    #
# U.S. Department of Energy. The U.S. Government has rights to use,       #
# reproduce, and distribute this software.  NEITHER THE GOVERNMENT NOR    #
# UChicago Argonne, LLC MAKES ANY WARRANTY, EXPRESS OR IMPLIED, OR        #
# ASSUMES ANY LIABILITY FOR THE USE OF THIS SOFTWARE.  If software is     #
# modified to produce derivative works, such modified software should     #
# be clearly marked, so as not to confuse it with the version available   #
# from ANL.                                                               #
#                                                                         #
# Additionally, redistribution and use in source and binary forms, with   #
# or without modification, are permitted provided that the following      #
# conditions are met:                                                     #
#                                                                         #
#     * Redistributions of source code must retain the above copyright    #
#       notice, this list of conditions and the following disclaimer.     #
#                                                                         #
#     * Redistributions in binary form must reproduce the above copyright #
#       notice, this list of conditions and the following disclaimer in   #
#       the documentation and/or other materials provided with the        #
#       distribution.                                                     #
#                                                                         #
#     * Neither the name of UChicago Argonne, LLC, Argonne National       #
#       Laboratory, ANL, the U.S. Government, nor the names of its        #
#       contributors may be used to endorse or promote products derived   #
#       from this software without specific prior written permission.     #
#                                                                         #
# THIS SOFTWARE IS PROVIDED BY UChicago Argonne, LLC AND CONTRIBUTORS     #
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT       #
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS       #
# FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL UChicago     #
# Argonne, LLC OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,        #
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,    #
# BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;        #
# LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER        #
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT      #
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN       #
# ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE         #
# POSSIBILITY OF SUCH DAMAGE.                                             #
# #########################################################################

"""
Tries the FFT implementation options in order until one imports without error.
"""

from __future__ import (absolute_import, division, print_function,
                        unicode_literals)

import os
import logging

logger = logging.getLogger(__name__)

__author__ = "Oleksandr Pavlyk"
__copyright__ = "Copyright (c) 2016, UChicago Argonne, LLC."
__docformat__ = 'restructuredtext en'
__all__ = ['fft_impl']

fft_impl = 'unset'


def import_mkl_fft():
    import mkl_fft
    fft_impl = 'mkl_fft'


def import_pyfftw():
    if os.name == 'nt':
        import pyfftw
    else:
        # Import pyfftw as soon as possible with RTLD_NOW|RTLD_DEEPBIND
        # to minimize chance of MKL overriding fftw functions
        import ctypes
        import sys
        curFlags = sys.getdlopenflags()
        sys.setdlopenflags(curFlags | ctypes.RTLD_GLOBAL)
        import pyfftw
        sys.setdlopenflags(curFlags)
        del curFlags
    fft_impl = 'pyfftw'


def import_numpy_fft():
    import numpy.fft
    fft_impl = 'numpy.fft'


fft_options = [import_mkl_fft, import_pyfftw, import_numpy_fft]
if 'TOMOPY_FFT_IMPL' in os.environ:
    # Let the user choose the implementation
    fft_impl = os.environ['TOMOPY_FFT_IMPL']
    logger.info('FFT implementation preference is {}'.format(fft_impl))
else:
    for fft in fft_options:
        try:
            fft()
            # Succesfully imported an fft library
            break
        except ImportError:
            # Try the next fft library
            pass

logger.debug('FFT implementation is {}'.format(fft_impl))
