# Copyright (c) Scanlon Materials Theory Group
# Distributed under the terms of the MIT License.

"""
Class for generating k-points along Latimer & Munro high-symmetry paths.
"""

from pymatgen.symmetry.bandstructure import HighSymmKpath

from sumo.symmetry import Kpath


class LatimerKpath(Kpath):
    r"""Class to generate k-points along paths from the Latimer & Munro method.

    More details on the paths generated by can be found in the pymatgen
    documentation. These paths are based on the work described in reference
    [lm]_.

    .. [lm] Munro, J.M., Latimer, K., Horton, M.K. et al.
        An improved symmetry-based approach to reciprocal space path selection
        in band structure calculations. npj Comput Mater 6, 112 (2020).
        doi: 10.1038/s41524-020-00383-7

    Args:
        structure (:obj:`~pymatgen.core.structure.Structure`): The structure.
        symprec (:obj:`float`, optional): The tolerance for determining the
            crystal symmetry.
    """

    def __init__(self, structure, symprec=1e-3):
        Kpath.__init__(self, structure, symprec=symprec)
        pmg_path = HighSymmKpath(self.prim, symprec=symprec, path_type="lm")
        self._kpath = pmg_path._kpath
