
# stdlib imports
import argparse
import sys
import re
import configparser
import os.path

# third party imports
import pandas as pd
import numpy as np
from libcomcat.search import get_event_by_id


# local imports
from strec.subtype import SubductionSelector
from strec.tensor import fill_tensor_from_angles, fill_tensor_from_components
from strec.utils import read_input_file, get_input_columns, render_row


def check_moment_row(row):
    # row is a pandas series object
    hasAngles = True
    hasComponents = True

    if not row.str.contains('mrr', case=False).any():
        hasComponents = False
    if not row.str.contains('mpp', case=False).any():
        hasComponents = False
    if not row.str.contains('mtt', case=False).any():
        hasComponents = False
    if not row.str.contains('mrt', case=False).any():
        hasComponents = False
    if not row.str.contains('mrp', case=False).any():
        hasComponents = False
    if not row.str.contains('mtp', case=False).any():
        hasComponents = False

    if not row.str.contains('strike', case=False).any():
        hasAngles = False
    if not row.str.contains('dip', case=False).any():
        hasAngles = False
    if not row.str.contains('rake', case=False).any():
        hasAngles = False

    return hasAngles or hasComponents


def get_moment_columns(row):
    # row is a pandas series object
    isreal = row.notnull()

    has_mtt = row.index.str.contains('mtt', case=False)
    has_mpp = row.index.str.contains('mpp', case=False)
    has_mrr = row.index.str.contains('mrr', case=False)
    has_mrt = row.index.str.contains('mrt', case=False)
    has_mrp = row.index.str.contains('mrp', case=False)
    has_mtp = row.index.str.contains('mtp', case=False)

    mtts = row[isreal & has_mtt]
    mpps = row[isreal & has_mpp]
    mrrs = row[isreal & has_mrr]
    mrts = row[isreal & has_mrt]
    mrps = row[isreal & has_mrp]
    mtps = row[isreal & has_mtp]

    mtt = mpp = mrr = mrt = mrp = mtp = np.nan
    if len(mtts):
        mtt = float(mtts[0])
    if len(mpps):
        mpp = float(mpps[0])
    if len(mrrs):
        mrr = float(mrrs[0])
    if len(mrts):
        mrt = float(mrts[0])
    if len(mrps):
        mrp = float(mrps[0])
    if len(mtps):
        mtp = float(mtps[0])

    if not np.isnan(mtt):
        tensor_params = fill_tensor_from_components(
            mrr, mtt, mpp, mrt, mrp, mtp)
        return tensor_params

    has_strike = row.index.str.contains('strike', case=False)
    has_dip = row.index.str.contains('dip', case=False)
    has_rake = row.index.str.contains('rake', case=False)
    strikes = row[isreal & has_strike]
    dips = row[isreal & has_dip]
    rakes = row[isreal & has_rake]

    strike = dip = rake = np.nan
    if len(strikes):
        strike = strikes[0]
    if len(dips):
        dip = dips[0]
    if len(rakes):
        rake = rakes[0]

    if not np.isnan(strike):
        tensor_params = fill_tensor_from_angles(strike, dip, rake)
        return tensor_params

    return None


def main(args, pparser):
    # check input arguments
    # check input arguments
    haseq = args.eqinfo is not None
    hasinput = args.input_file is not None
    hasid = args.event_id is not None
    if (haseq + hasinput + hasid) >= 2:
        print('Must choose no more than one of -e or -i or -d options.')
        sys.exit(1)

    if (hasid or hasinput) and args.moment_info:
        print('Supplying moment tensor information is unnecessary/inapplicablee when also supplying event ID or input file.')
        sys.exit(1)

    if not args.output_file and args.output_format == 'excel':
        print('You cannot specify an output format of excel without also providing an output file name.')
        sys.exit(1)

    tensor_params = None
    selector = SubductionSelector()
    if args.input_file:
        df, msg = read_input_file(args.input_file)
        if df is None:
            print('Input file %s does not appear to be either a CSV or Excel file.  Returning.' %
                  args.input_file)
            sys.exit(1)

    if args.eqinfo:
        lat, lon, depth = args.eqinfo
        d = {'lat': [lat], 'lon': [lon], 'depth': [depth]}
        df = pd.DataFrame(d)

    if args.moment_info:
        strike, dip, rake, mag = args.moment_info
        tensor_params = get_tensor_params_from_nodal(strike, dip, rake, mag)

    if args.event_id:
        try:
            detail = get_event_by_id(args.event_id)
            lat, lon, depth = detail.latitude, detail.longitude, detail.depth
        except Exception as e:
            msg = 'Could not get event information on event ID %s due to error "%s".'
            tpl = (args.event_id, str(e))
            print(msg % tpl)
            sys.exit(1)

        d = {'lat': [lat], 'lon': [lon], 'depth': [depth]}
        df = pd.DataFrame(d)

    results = []
    has_tensor = check_moment_row(df.columns) or tensor_params is not None
    ic = 0
    ic = 0
    inc = min(100, np.power(10, np.floor(np.log10(len(df))) - 1))
    if args.verbose:
        sys.stderr.write(
            'Getting event info - reporting every %i events.\n' % inc)
    for idx, row in df.iterrows():
        if ic % inc == 0 and args.verbose:
            msg = 'Getting detailed information for %i of %i events.\n'
            sys.stderr.write(msg % (ic, len(df)))
        lat, lon, depth = get_input_columns(row)
        if not tensor_params and has_tensor:
            tensor_params = get_moment_columns(row)
        else:
            tensor_params = None
        result = selector.getSubductionType(
            lat, lon, depth, eventid=args.event_id, tensor_params=tensor_params)
        tensor_params = None
        row = row.append(result)
        if args.output_file:
            results.append(row)
            tt = row[row.astype(str).str.isdigit()]
            try:
                ddf = pd.DataFrame(results)
            except:
                x = 1
        else:
            render_row(row, args.output_format, lat, lon, depth)
        ic += 1
    if args.output_file:
        df = pd.DataFrame(results)
        if args.output_format == 'csv':
            df.to_csv(args.output_file, index=False)
        else:
            df.to_excel(args.output_file, index=False)


if __name__ == '__main__':
    usage = """Determine various seismo-tectonic parameters for given input coordinates.
    GMREG - Ground Motion Regionalization Tool

The output will consist of the following "columns" of data:

Tectonic Region
  - Active : is this point in a tectonically active region? (True/False)
  - Stable : is this point in a tectonically stable region? (True/False)
  - Subduction : is this point in a subduction zone? (True/False)
  - Active Volcanic : is this point in an active volcanic zone? (True/False)

Oceanic - is this point in an oceanic region? (True/False)
Induced - is this point in an induced seismicity region? (True/False)

Each tectonic region will have a subordinate "Tectonic Domain", consisting of one of the following:

 - ACR oceanic boundary (above slab)
 - ACR deep (above slab)
 - ACR shallow (above slab)
 - SOR (above slab)
 - SOR (generic)
 - SZ (inland/back-arc)
 - SZ (on-shore)
 - SZ (outer-trench)
 - SZ (generic)
 - ACR (hot spot)
 - ACR (oceanic boundary)
 - ACR (deep)
 - ACR (shallow)
 - SCR (above slab)
 - SCR (generic)

Usage:

%(prog)s -e 3.295 95.982 30.0
%(prog)s -d official20041226005853450_30

"""
    input_file_help = '''Input files can be CSV or Excel format (%(prog)s  will attempt to determine
format automatically) and MUST include the following columns (case doesn't matter):

Lat Numeric latitude of input earthquake.
Lon Numeric longitude of input earthquake.
Depth Numeric depth of earthquake (km).

Any other columns present in the input will be copied to the output.
'''

    output_format_help = '''The output format must be one of the following:
 
pretty - A human readable pretty-printed representation of the output.
csv - A CSV output, one earthquake per line.
excel - Microsoft Excel format, one earthquake per line.
json - A JSON representation of the output

'''

    parser = argparse.ArgumentParser(
        description=usage, formatter_class=argparse.RawTextHelpFormatter)
    parser.add_argument("-e", "--eqinfo", nargs=3, metavar=('LAT', 'LON', 'DEPTH'), type=float,
                        help='lat,lon, depth of earthquake')
    parser.add_argument("-i", "--input-file", dest="input_file",
                        metavar='INPUTFILE',
                        help=input_file_help)
    parser.add_argument("-f", "--output-format", dest="output_format",
                        metavar='OUTPUT FORMAT', choices=('pretty', 'csv', 'excel', 'json'), default='pretty',
                        help=output_format_help)
    parser.add_argument("-o", "--output-file",
                        dest="output_file", help='Output filename')
    parser.add_argument("-m", "--moment-info", nargs=4, metavar=('STRIKE', 'DIP', 'RAKE', 'MAG'), type=float,
                        help='strike,dip,rake,magnitude of earthquake')
    parser.add_argument("-d", "--event-id", dest="event_id",
                        help='ComCat Event ID')
    parser.add_argument("-v", "--verbose",
                        help='Verbose output', action='store_true')
    pargs = parser.parse_args()

    main(pargs, parser)
