#! /usr/bin/env python

############################################################################
# Copyright (c) 2009 Dr. Peter Bunting, Aberystwyth University
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#
#
# Purpose:  A class to generate shell for the SPDLib library based
#           on a supplied template.
# Author: Pete Bunting
# Email: petebunting@mac.com
# Date: 01/08/2009
# Version: 1.2
#
# History:
# Version 1.0 - Created.
# Version 1.1 - Added <PATH> option with the $ template character
# Version 1.2 - Changed to use argument parser 
# Version 1.3 - Updated to use os.walk if running in recursive mode
#
#############################################################################

import os.path
import sys
from time import strftime
import argparse

class SPDLibBatchGenerator (object):

    def checkFileExtension(self, filename, extension):
        foundExtension = False;
        filenamesplit = os.path.splitext(filename)
        fileExtension = filenamesplit[1].strip()
        if(fileExtension == extension):
            foundExtension = True
        return foundExtension

    def findFilesIterative(self, filelist, directory, extension):
        # Walk through directories.
        if os.path.exists(directory):
            for dName, sdName, fList in os.walk(directory):
                for filename in fList:
                    if self.checkFileExtension(filename, extension):
                        filelist.append(os.path.join(dName, filename))
        else:
            print(directory + ' does not exist!')
    
    def findFiles(self, filelist, directory, extension):
        if os.path.exists(directory):
            if os.path.isdir(directory):
                dirFileList = os.listdir(directory)
                for filename in dirFileList:
                    if(os.path.isdir(os.path.join(directory,filename))):
                        print(os.path.join(directory,filename) + ' is a directory and therefore ignored!')
                    elif(os.path.isfile(os.path.join(directory,filename))):
                        if(self.checkFileExtension(filename, extension)):
                            filelist.append(os.path.join(directory,filename))
                    else:
                        print(filename + ' is NOT a file or directory!')
            else:
                print(directory + ' is not a directory!')
        else:
            print(directory + ' does not exist!')

    def readInputTemplate(self, inputFile):
        outline = str()
        inputXMLFile = open(inputFile, 'r')
        for eachLine in inputXMLFile:
            outline = outline + eachLine
        return outline

    def constructOutputFile(self, filelist, outline, outputfile, path):
        outputSHFile = open(outputfile, 'w')
        for file in filelist:
            basefile = os.path.basename(file)
            basename = os.path.splitext(basefile)[0]
            comment = '# ' + basename + '\n'
            outlinereplacepath = outline.replace('$PATH', path, outline.count('$PATH'))
            outlinereplaceinput = outlinereplacepath.replace('$FILEPATH', file, outlinereplacepath.count('$FILEPATH'))
            outlinereplacename = outlinereplaceinput.replace('$FILENAME', basename, outlinereplaceinput.count('$FILENAME'))
            outputSHFile.write(comment)
            outputSHFile.write(outlinereplacename)
            outputSHFile.write('\n\n')
        outputSHFile.flush()
        outputSHFile.close()

    def constructMultiOutputFile(self, filelist, outline, outputfile, path):
        outputfile = os.path.splitext(outputfile)[0]
        outputFileAll = outputfile + str("_runall.sh")
        outputSHAllFile = open(outputFileAll, 'w')
        outShCmd = ""
        for file in filelist:
            basefile = os.path.basename(file)
            basename = os.path.splitext(basefile)[0]
            outlinereplacepath = outline.replace('$PATH', path, outline.count('$PATH'))
            outlinereplaceinput = outlinereplacepath.replace('$FILEPATH', file, outlinereplacepath.count('$FILEPATH'))
            outlinereplacename = outlinereplaceinput.replace('$FILENAME', basename, outlinereplaceinput.count('$FILENAME'))
            outputFileTmp = outputfile + basename + str(".sh")
            outputSHFile = open(outputFileTmp, 'w')
            outputSHFile.write(outlinereplacename)
            outputSHFile.flush()
            outputSHFile.close()
            outShCmd = str("sh ") + outputFileTmp + str("\n")
            outputSHAllFile.write(outShCmd)
        outputSHAllFile.flush()
        outputSHAllFile.close()

        return outputFileAll

    def run(self):
        parser = argparse.ArgumentParser()
        parser.add_argument("-i", "--input", dest="xmloutlineFile", type=str, help="Input shell template")
        parser.add_argument("-o", "--output", type=str, dest="outfile", help="Output shell")
        parser.add_argument("-p", "--path", type=str, dest="path", help="Path used to replace '$PATH' in shell template.")
        parser.add_argument("-e", "--ext", type=str, dest="extension", help="File extension to search for.")
        parser.add_argument("-d", "--dir", type=str, dest="dir", help="Directory to search for files with the extension specified by '-e'.")
        parser.add_argument("-r", "--recurse", type=str, dest="recurse", default='no', help="Recurse into sub directories within input directory (yes | no).")
        parser.add_argument("-t", "--output_type", type=str, dest="outputtype", default='single', help="Output type (single - single shell file | multiple - separate shell file for each file found).")

        args = parser.parse_args()    

        if args.xmloutlineFile is None:
            print("Input shell template must be set.")
            parser.print_help()
            sys.exit()
            
        if args.outfile is None:
            print("Output shell must be set.")
            parser.print_help()
            sys.exit()

        if args.extension is None:
            print("Extension must be set.")
            parser.print_help()
            sys.exit()
        
        if args.dir is None:
            print("Directory must be set.")
            parser.print_help()
            sys.exit()
        
        if args.path is None:
            print("WARNING: No path set, assuming the same as search directory:")
            print(args.dir)
            args.path = args.dir


        filelist = list()

        if(args.recurse == 'yes'):
            self.findFilesIterative(filelist, args.dir, args.extension)
        else:
            self.findFiles(filelist, args.dir, args.extension)

        if os.path.exists(args.xmloutlineFile):
            outline = self.readInputTemplate(args.xmloutlineFile)
            if args.outputtype == 'multiple':
                shellall = self.constructMultiOutputFile(filelist, outline, args.outfile, args.path)
                print("Complete. Run script:")
                print("\t\tsh ", shellall)
            else:
                self.constructOutputFile(filelist, outline, args.outfile, args.path)
                print("Complete. Run script:")
                print("\t\tsh ", args.outfile)

        else:
            print('Input File does not exist')


    def help(self):
        print('spdbatchgen.py script generates the shell commands for the ')
        print('SPD software library from a user defined template')
        
        print('Within shell Template (--input):')
        print('\t$FILEPATH - Replaced with full file path')
        print('\t$FILENAME - Replaced with file name')
        print('\t$PATH - Replaced with path (--path)')
        print('\nThis script was distributed with version 3.3.309 of the SPDLib library.')
        print('For maintenance email spdlib-develop@lists.sourceforge.net')

if __name__ == '__main__':
    obj = SPDLibBatchGenerator()
    obj.run()
