/*
 *  SPDMetrics.h
 *  SPDLIB
 *
 *  Created by Pete Bunting on 26/02/2011.
 *  Copyright 2011 SPDLib. All rights reserved.
 *
 *  This file is part of SPDLib.
 *
 *  SPDLib is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  SPDLib is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with SPDLib.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef SPDMetrics_H
#define SPDMetrics_H

#include <iostream>
#include <vector>
#include <math.h>
#include <algorithm>

#include "gsl/gsl_statistics_double.h"

#include "gdal_priv.h"
#include "ogrsf_frmts.h"
#include "ogr_api.h"

#include "spd/cmpfit/mpfit.h"

#include <boost/cstdint.hpp>
#include <boost/numeric/conversion/cast.hpp>
#include "boost/math/special_functions/fpclassify.hpp"

#include "spd/SPDException.h"
#include "spd/SPDFile.h"
#include "spd/SPDPoint.h"
#include "spd/SPDPulse.h"
#include "spd/SPDProcessingException.h"
#include "spd/SPDPointGridIndex.h"

// mark all exported classes/functions with DllExport to have
// them exported by Visual Studio
#undef DllExport
#ifdef _MSC_VER
    #ifdef libspd_EXPORTS
        #define DllExport   __declspec( dllexport )
    #else
        #define DllExport   __declspec( dllimport )
    #endif
#else
    #define DllExport
#endif

namespace spdlib{

	class DllExport SPDMetric
	{
	public:
		SPDMetric(){};
		virtual double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException) = 0;
		virtual ~SPDMetric(){};
	};

    class DllExport SPDMetricAdd : public SPDMetric
	{
	public:
		SPDMetricAdd(std::vector<SPDMetric*> *metrics)
        {
            this->metrics = metrics;
        };
		double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            double value = 0;
            for(std::vector<SPDMetric*>::iterator iterMetrics = metrics->begin(); iterMetrics != metrics->end(); ++iterMetrics)
            {
                value += (*iterMetrics)->calcValue(pulses, spdFile, geom);
            }
            return value;
        };
		~SPDMetricAdd(){};
    private:
        std::vector<SPDMetric*> *metrics;
	};

    class DllExport SPDMetricMinus : public SPDMetric
	{
	public:
		SPDMetricMinus(SPDMetric *metric1, SPDMetric *metric2)
        {
            this->metric1 = metric1;
            this->metric2 = metric2;
        };
		double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            return metric1->calcValue(pulses, spdFile, geom) - metric2->calcValue(pulses, spdFile, geom);
        };
		~SPDMetricMinus(){};
    private:
        SPDMetric *metric1;
        SPDMetric *metric2;
	};

    class DllExport SPDMetricMultiply : public SPDMetric
	{
	public:
		SPDMetricMultiply(std::vector<SPDMetric*> *metrics)
        {
            this->metrics = metrics;
        };
		double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            double value = 0;
            for(std::vector<SPDMetric*>::iterator iterMetrics = metrics->begin(); iterMetrics != metrics->end(); ++iterMetrics)
            {
                if(iterMetrics == metrics->begin()) {
                    value = (*iterMetrics)->calcValue(pulses, spdFile, geom);
                } else {
                    value *= (*iterMetrics)->calcValue(pulses, spdFile, geom);
                }
            }
            return value;
        };
		~SPDMetricMultiply(){};
    private:
        std::vector<SPDMetric*> *metrics;
	};

    class DllExport SPDMetricDivide : public SPDMetric
	{
	public:
		SPDMetricDivide(SPDMetric *metric1, SPDMetric *metric2)
        {
            this->metric1 = metric1;
            this->metric2 = metric2;
        };
		double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            return metric1->calcValue(pulses, spdFile, geom) / metric2->calcValue(pulses, spdFile, geom);
        };
		~SPDMetricDivide(){};
    private:
        SPDMetric *metric1;
        SPDMetric *metric2;
	};

    class DllExport SPDMetricPow : public SPDMetric
	{
	public:
		SPDMetricPow(SPDMetric *metric1, SPDMetric *metric2)
        {
            this->metric1 = metric1;
            this->metric2 = metric2;
        };
		double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            return pow(metric1->calcValue(pulses, spdFile, geom), metric2->calcValue(pulses, spdFile, geom));
        };
		~SPDMetricPow(){};
    private:
        SPDMetric *metric1;
        SPDMetric *metric2;
	};

    class DllExport SPDMetricAbs : public SPDMetric
	{
	public:
		SPDMetricAbs(SPDMetric *metric)
        {
            this->metric = metric;
        };
		double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            return fabs(metric->calcValue(pulses, spdFile, geom));
        };
		~SPDMetricAbs(){};
    private:
        SPDMetric *metric;
	};

    class DllExport SPDMetricSqrt : public SPDMetric
	{
	public:
		SPDMetricSqrt(SPDMetric *metric)
        {
            this->metric = metric;
        };
		double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            return sqrt(metric->calcValue(pulses, spdFile, geom));
        };
		~SPDMetricSqrt(){};
    private:
        SPDMetric *metric;
	};

    class DllExport SPDMetricSine : public SPDMetric
	{
	public:
		SPDMetricSine(SPDMetric *metric)
        {
            this->metric = metric;
        };
		double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            return sin(metric->calcValue(pulses, spdFile, geom));
        };
		~SPDMetricSine(){};
    private:
        SPDMetric *metric;
	};

    class DllExport SPDMetricCosine : public SPDMetric
	{
	public:
		SPDMetricCosine(SPDMetric *metric)
        {
            this->metric = metric;
        };
		double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            return cos(metric->calcValue(pulses, spdFile, geom));
        };
		~SPDMetricCosine(){};
    private:
        SPDMetric *metric;
	};

    class DllExport SPDMetricTangent : public SPDMetric
	{
	public:
		SPDMetricTangent(SPDMetric *metric)
        {
            this->metric = metric;
        };
		double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            return tan(metric->calcValue(pulses, spdFile, geom));
        };
		~SPDMetricTangent(){};
    private:
        SPDMetric *metric;
	};

    class DllExport SPDMetricInvSine : public SPDMetric
	{
	public:
		SPDMetricInvSine(SPDMetric *metric)
        {
            this->metric = metric;
        };
		double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            return asin(metric->calcValue(pulses, spdFile, geom));
        };
		~SPDMetricInvSine(){};
    private:
        SPDMetric *metric;
	};

    class DllExport SPDMetricInvCos : public SPDMetric
	{
	public:
		SPDMetricInvCos(SPDMetric *metric)
        {
            this->metric = metric;
        };
		double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            return acos(metric->calcValue(pulses, spdFile, geom));
        };
		~SPDMetricInvCos(){};
    private:
        SPDMetric *metric;
	};

    class DllExport SPDMetricInvTan : public SPDMetric
	{
	public:
		SPDMetricInvTan(SPDMetric *metric)
        {
            this->metric = metric;
        };
		double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            return atan(metric->calcValue(pulses, spdFile, geom));
        };
		~SPDMetricInvTan(){};
    private:
        SPDMetric *metric;
	};

    class DllExport SPDMetricLog10 : public SPDMetric
	{
	public:
		SPDMetricLog10(SPDMetric *metric)
        {
            this->metric = metric;
        };
		double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            return log10(metric->calcValue(pulses, spdFile, geom));
        };
		~SPDMetricLog10(){};
    private:
        SPDMetric *metric;
	};

    class DllExport SPDMetricLn : public SPDMetric
	{
	public:
		SPDMetricLn(SPDMetric *metric)
        {
            this->metric = metric;
        };
		double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            return log(metric->calcValue(pulses, spdFile, geom));
        };
		~SPDMetricLn(){};
    private:
        SPDMetric *metric;
	};

    class DllExport SPDMetricExp : public SPDMetric
	{
	public:
		SPDMetricExp(SPDMetric *metric)
        {
            this->metric = metric;
        };
		double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            return exp(metric->calcValue(pulses, spdFile, geom));
        };
		~SPDMetricExp(){};
    private:
        SPDMetric *metric;
	};

    class DllExport SPDMetricPercentage : public SPDMetric
	{
	public:
		SPDMetricPercentage(SPDMetric *metric1, SPDMetric *metric2)
        {
            this->metric1 = metric1;
            this->metric2 = metric2;
        };
		double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            return (metric1->calcValue(pulses, spdFile, geom) / metric2->calcValue(pulses, spdFile, geom)) * 100;
        };
		~SPDMetricPercentage(){};
    private:
        SPDMetric *metric1;
        SPDMetric *metric2;
	};

    class DllExport SPDMetricAddConst : public SPDMetric
	{
	public:
		SPDMetricAddConst(SPDMetric *metric, double constVal)
        {
            this->metric = metric;
            this->constVal = constVal;
        };
		double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            return metric->calcValue(pulses, spdFile, geom) + constVal;
        };
		~SPDMetricAddConst(){};
    private:
        SPDMetric *metric;
        double constVal;
	};

	class DllExport SPDMetricMinusConstFrom : public SPDMetric
	{
	public:
		SPDMetricMinusConstFrom(SPDMetric *metric, double constVal)
        {
            this->metric = metric;
            this->constVal = constVal;
        };
		double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            return metric->calcValue(pulses, spdFile, geom) - constVal;
        };
		~SPDMetricMinusConstFrom(){};
    private:
        SPDMetric *metric;
        double constVal;
	};

    class DllExport SPDMetricMinusFromConst : public SPDMetric
	{
	public:
		SPDMetricMinusFromConst(SPDMetric *metric, double constVal)
        {
            this->metric = metric;
            this->constVal = constVal;
        };
		double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            return constVal - metric->calcValue(pulses, spdFile, geom);
        };
		~SPDMetricMinusFromConst(){};
    private:
        SPDMetric *metric;
        double constVal;
	};

    class DllExport SPDMetricMultiplyConst : public SPDMetric
	{
	public:
		SPDMetricMultiplyConst(SPDMetric *metric, double constVal)
        {
            this->metric = metric;
            this->constVal = constVal;
        };
		double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            return constVal * metric->calcValue(pulses, spdFile, geom);
        };
		~SPDMetricMultiplyConst(){};
    private:
        SPDMetric *metric;
        double constVal;
	};

    class DllExport SPDMetricDivideByConst : public SPDMetric
	{
	public:
		SPDMetricDivideByConst(SPDMetric *metric, double constVal)
        {
            this->metric = metric;
            this->constVal = constVal;
        };
		double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            return metric->calcValue(pulses, spdFile, geom) / constVal;
        };
		~SPDMetricDivideByConst(){};
    private:
        SPDMetric *metric;
        double constVal;
	};

    class DllExport SPDMetricDivideConstBy : public SPDMetric
	{
	public:
		SPDMetricDivideConstBy(SPDMetric *metric, double constVal)
        {
            this->metric = metric;
            this->constVal = constVal;
        };
		double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            return constVal / metric->calcValue(pulses, spdFile, geom);
        };
		~SPDMetricDivideConstBy(){};
    private:
        SPDMetric *metric;
        double constVal;
	};

    class DllExport SPDMetricPowMetricConst : public SPDMetric
	{
	public:
		SPDMetricPowMetricConst(SPDMetric *metric, double constVal)
        {
            this->metric = metric;
            this->constVal = constVal;
        };
		double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            return pow(metric->calcValue(pulses, spdFile, geom), constVal);
        };
		~SPDMetricPowMetricConst(){};
    private:
        SPDMetric *metric;
        double constVal;
	};

    class DllExport SPDMetricPowConstMetric : public SPDMetric
	{
	public:
		SPDMetricPowConstMetric(SPDMetric *metric, double constVal)
        {
            this->metric = metric;
            this->constVal = constVal;
        };
		double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            return pow(constVal, metric->calcValue(pulses, spdFile, geom));
        };
		~SPDMetricPowConstMetric(){};
    private:
        SPDMetric *metric;
        double constVal;
	};

    class DllExport SPDMetricCalc : public SPDMetric
	{
	public:
		SPDMetricCalc(int_fast16_t returnID=-1,boost::int_fast16_t classification=-1,boost::uint_fast16_t minNumReturns=1, float upThreshold=-1, float lowThreshold=-1, float heightUpThreshold=-1, float heightLowThreshold=-1)
        {
            this->returnID = returnID;
            this->classification = classification;
            this->minNumReturns = minNumReturns;
            this->upThreshold = upThreshold;
            this->lowThreshold = lowThreshold;
            this->heightUpThreshold = heightUpThreshold;
            this->heightLowThreshold = heightLowThreshold;
        }
		virtual double calcValue(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException) = 0;
		virtual ~SPDMetricCalc(){};
    protected:
        virtual std::vector<double>* getPointsValuesWithinHeightParameters(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            std::vector<double> *points = new std::vector<double>();
            // If there is waveform data and the minimum number of returns has been set to 0
            if((spdFile->getReceiveWaveformDefined() == SPD_TRUE) & (this->minNumReturns == 0))
            {
                double tmpX = 0;
                double tmpY = 0;
                double tmpH = 0;

                for(std::vector<SPDPulse*>::iterator iterPulses = pulses->begin(); iterPulses != pulses->end(); ++iterPulses)
                {
                    for(unsigned int s = 0; s < (*iterPulses)->numOfReceivedBins; s++)
                    {
                        // Get the DN of the digitised value and check
                        // this is above the noise threshold.
                        double pulseDNVal = (*iterPulses)->received[s];

                        if(pulseDNVal > (*iterPulses)->receiveWaveNoiseThreshold)
                        {
                            // Get the time of the digitised value, relative to the origin
                            double timeOffset = s * spdFile->getTemporalBinSpacing();

                            // Get the height of the digitised value using the height of the origin
                            // and the time offset within the pulse.
                            SPDConvertToCartesian((*iterPulses)->zenith, (*iterPulses)->azimuth, 
                                                (SPD_SPEED_OF_LIGHT_NS * timeOffset), (*iterPulses)->x0, (*iterPulses)->y0, (*iterPulses)->h0, &tmpX, &tmpY, &tmpH);

                            if((boost::math::isnan(this->upThreshold)) & (boost::math::isnan(this->lowThreshold)))
                            {
                                points->push_back(tmpH);
                            }
                            else if((boost::math::isnan(this->lowThreshold)) & (tmpH <= this->upThreshold))
                            {
                                points->push_back(tmpH);
                            }
                            else if((tmpH >= this->lowThreshold) & (boost::math::isnan(this->upThreshold)))
                            {
                                points->push_back(tmpH);
                            }
                            else if((tmpH >= this->lowThreshold) & (tmpH <= this->upThreshold))
                            {
                                points->push_back(tmpH);
                            }

                        }
                    }
                }
            }
            else if((spdFile->getDecomposedPtDefined() == SPD_TRUE) | (spdFile->getDiscretePtDefined() == SPD_TRUE))
            {
                for(std::vector<SPDPulse*>::iterator iterPulses = pulses->begin(); iterPulses != pulses->end(); ++iterPulses)
                {
                    if(((*iterPulses)->numberOfReturns >= minNumReturns) & ((*iterPulses)->numberOfReturns > 0))
                    {
                        if(returnID == SPD_ALL_RETURNS)
                        {
                            for(std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin(); iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & ((*iterPts)->height <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                }
                                else if(((*iterPts)->height >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                }
                                else if(((*iterPts)->height >= lowThreshold) & ((*iterPts)->height <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_LAST_RETURNS)
                        {
                            SPDPoint *pt = (*iterPulses)->pts->back();
                            if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt->height);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt->height);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt->height);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt->height);
                                }
                            }
                            else if(boost::math::isnan(lowThreshold) & (pt->height <= upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt->height);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt->height);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt->height);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt->height);
                                }
                            }
                            else if((pt->height >= lowThreshold) & boost::math::isnan(upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt->height);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt->height);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt->height);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt->height);
                                }
                            }
                            else if((pt->height >= lowThreshold) & (pt->height <= upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt->height);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt->height);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt->height);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt->height);
                                }
                            }
                        }
                        else if(returnID == SPD_FIRST_RETURNS)
                        {
                            SPDPoint *pt = (*iterPulses)->pts->front();
                            if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt->height);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt->height);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt->height);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt->height);
                                }
                            }
                            else if(boost::math::isnan(lowThreshold) & (pt->height <= upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt->height);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt->height);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt->height);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt->height);
                                }
                            }
                            else if((pt->height >= lowThreshold) & boost::math::isnan(upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt->height);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt->height);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt->height);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt->height);
                                }
                            }
                            else if((pt->height >= lowThreshold) & (pt->height <= upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt->height);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt->height);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt->height);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt->height);
                                }
                            }
                        }
                        else if(returnID == SPD_FIRSTLAST_RETURNS)
                        {
                            SPDPoint *ptFirst = (*iterPulses)->pts->front();
                            SPDPoint *ptLast = NULL;
                            if((*iterPulses)->pts->size() > 1)
                            {
                                ptLast = (*iterPulses)->pts->back();
                            }
                            if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                            {
                                // First
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(ptFirst->height);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                         (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                         (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(ptFirst->height);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (ptFirst->classification != SPD_GROUND))
                                {
                                    points->push_back(ptFirst->height);
                                }
                                else if(ptFirst->classification == classification)
                                {
                                    points->push_back(ptFirst->height);
                                }

                                // Last
                                if(ptLast != NULL)
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptLast->height);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptLast->classification == SPD_LOW_VEGETATION) |
                                             (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptLast->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptLast->height);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptLast->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptLast->height);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptLast->height);
                                    }
                                }
                            }
                            else if(boost::math::isnan(lowThreshold))
                            {
                                if(ptFirst->height <= upThreshold)
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst->height);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst->height);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst->height);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst->height);
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if(ptLast->height <= upThreshold)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast->height);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast->height);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast->height);
                                        }
                                        else if(ptLast->classification == classification)
                                        {
                                            points->push_back(ptLast->height);
                                        }
                                    }
                                }
                            }
                            else if(boost::math::isnan(upThreshold))
                            {
                                if(ptFirst->height >= lowThreshold)
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst->height);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst->height);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst->height);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst->height);
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if(ptLast->height >= lowThreshold)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast->height);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast->height);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast->height);
                                        }
                                        else if(ptLast->classification == classification)
                                        {
                                            points->push_back(ptLast->height);
                                        }
                                    }
                                }
                            }
                            else
                            {
                                if((ptFirst->height >= lowThreshold) & (ptFirst->height <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst->height);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst->height);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst->height);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst->height);
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if((ptLast->height >= lowThreshold) & (ptLast->height <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast->height);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast->height);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast->height);
                                        }
                                        else if(ptLast->classification == classification)
                                        {
                                            points->push_back(ptLast->height);
                                        }
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_NOTFIRST_RETURNS)
                        {
                            std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin();
                            ++iterPts;
                            for(; iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & ((*iterPts)->height <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                }
                                else if(((*iterPts)->height >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                }
                                else if(((*iterPts)->height >= lowThreshold) & ((*iterPts)->height <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back((*iterPts)->height);
                                    }
                                }
                            }
                        }
                        else
                        {
                            for(std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin(); iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(returnID == (*iterPts)->returnID)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->height);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->height);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->height);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->height);
                                        }
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & ((*iterPts)->height <= upThreshold))
                                {
                                    if(returnID == (*iterPts)->returnID)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->height);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->height);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->height);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->height);
                                        }
                                    }
                                }
                                else if(((*iterPts)->height >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(returnID == (*iterPts)->returnID)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->height);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->height);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->height);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->height);
                                        }
                                    }
                                }
                                else if(((*iterPts)->height >= lowThreshold) & ((*iterPts)->height <= upThreshold))
                                {
                                    if(returnID == (*iterPts)->returnID)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->height);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->height);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->height);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->height);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            else if((spdFile->getReceiveWaveformDefined() == SPD_TRUE) & this->minNumReturns != 0)
            {
                throw SPDProcessingException("There are no points and the number of returns has been set to a number > 0."
                                             " Either set 'minNumReturns' to 0 or run decomposition");
            }
            else
            {
                throw SPDProcessingException("Neither waveform or point returns have been defind.");
            }

            return points;
        };
        virtual std::vector<double>* getPointsValuesWithinZParameters(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            std::vector<double> *points = new std::vector<double>();
            if((spdFile->getReceiveWaveformDefined() == SPD_TRUE) & (this->minNumReturns == 0))
            {
                double tmpX = 0;
                double tmpY = 0;
                double tmpZ = 0;

                for(std::vector<SPDPulse*>::iterator iterPulses = pulses->begin(); iterPulses != pulses->end(); ++iterPulses)
                {
                    for(unsigned int s = 0; s < (*iterPulses)->numOfReceivedBins; s++)
                    {
                        // Get the DN of the digitised value and check
                        // this is above the noise threshold.
                        double pulseDNVal = (*iterPulses)->received[s];

                        if(pulseDNVal > (*iterPulses)->receiveWaveNoiseThreshold)
                        {
                            // Get the time of the digitised value, relative to the origin
                            double timeOffset = s * spdFile->getTemporalBinSpacing();

                            // Get the height of the digitised value using the height of the origin
                            // and the time offset within the pulse.
                            SPDConvertToCartesian((*iterPulses)->zenith, (*iterPulses)->azimuth, 
                                                (SPD_SPEED_OF_LIGHT_NS * timeOffset), (*iterPulses)->x0, (*iterPulses)->y0, (*iterPulses)->z0, &tmpX, &tmpY, &tmpZ);

                            if((boost::math::isnan(this->upThreshold)) & (boost::math::isnan(this->lowThreshold)))
                            {
                                points->push_back(tmpZ);
                            }
                            else if((boost::math::isnan(this->lowThreshold)) & (tmpZ <= this->upThreshold))
                            {
                                points->push_back(tmpZ);
                            }
                            else if((tmpZ >= this->lowThreshold) & (boost::math::isnan(this->upThreshold)))
                            {
                                points->push_back(tmpZ);
                            }
                            else if((tmpZ >= this->lowThreshold) & (tmpZ <= this->upThreshold))
                            {
                                points->push_back(tmpZ);
                            }

                        }
                    }
                }
            }
            else if((spdFile->getDecomposedPtDefined() == SPD_TRUE) | (spdFile->getDiscretePtDefined() == SPD_TRUE))
            {
                for(std::vector<SPDPulse*>::iterator iterPulses = pulses->begin(); iterPulses != pulses->end(); ++iterPulses)
                {
                    if(((*iterPulses)->numberOfReturns >= minNumReturns) & ((*iterPulses)->numberOfReturns > 0))
                    {
                        if(returnID == SPD_ALL_RETURNS)
                        {
                            for(std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin(); iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & ((*iterPts)->z <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                }
                                else if(((*iterPts)->z >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                }
                                else if(((*iterPts)->z >= lowThreshold) & ((*iterPts)->z <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_LAST_RETURNS)
                        {
                            SPDPoint *pt = (*iterPulses)->pts->back();
                            if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt->z);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt->z);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt->z);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt->z);
                                }
                            }
                            else if(boost::math::isnan(lowThreshold) & (pt->z <= upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt->z);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt->z);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt->z);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt->z);
                                }
                            }
                            else if((pt->z >= lowThreshold) & boost::math::isnan(upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt->z);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt->z);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt->z);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt->z);
                                }
                            }
                            else if((pt->z >= lowThreshold) & (pt->z <= upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt->z);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt->z);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt->z);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt->z);
                                }
                            }
                        }
                        else if(returnID == SPD_FIRST_RETURNS)
                        {
                            SPDPoint *pt = (*iterPulses)->pts->front();
                            if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt->z);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt->z);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt->z);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt->z);
                                }
                            }
                            else if(boost::math::isnan(lowThreshold) & (pt->z <= upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt->z);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt->z);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt->z);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt->z);
                                }
                            }
                            else if((pt->z >= lowThreshold) & boost::math::isnan(upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt->z);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt->z);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt->z);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt->z);
                                }
                            }
                            else if((pt->z >= lowThreshold) & (pt->z <= upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt->z);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt->z);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt->z);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt->z);
                                }
                            }
                        }
                        else if(returnID == SPD_FIRSTLAST_RETURNS)
                        {
                            SPDPoint *ptFirst = (*iterPulses)->pts->front();
                            SPDPoint *ptLast = NULL;
                            if((*iterPulses)->pts->size() > 1)
                            {
                                ptLast = (*iterPulses)->pts->back();
                            }
                            if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                            {
                                // First
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(ptFirst->z);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                         (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                         (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(ptFirst->z);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (ptFirst->classification != SPD_GROUND))
                                {
                                    points->push_back(ptFirst->z);
                                }
                                else if(ptFirst->classification == classification)
                                {
                                    points->push_back(ptFirst->z);
                                }

                                // Last
                                if(ptLast != NULL)
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptLast->z);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptLast->classification == SPD_LOW_VEGETATION) |
                                             (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptLast->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptLast->z);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptLast->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptLast->z);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptLast->z);
                                    }
                                }
                            }
                            else if(boost::math::isnan(lowThreshold))
                            {
                                if(ptFirst->z <= upThreshold)
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst->z);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst->z);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst->z);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst->z);
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if(ptLast->z <= upThreshold)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast->z);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast->z);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast->z);
                                        }
                                        else if(ptLast->classification == classification)
                                        {
                                            points->push_back(ptLast->z);
                                        }
                                    }
                                }
                            }
                            else if(boost::math::isnan(upThreshold))
                            {
                                if(ptFirst->z >= lowThreshold)
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst->z);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst->z);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst->z);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst->z);
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if(ptLast->z >= lowThreshold)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast->z);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast->z);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast->z);
                                        }
                                        else if(ptLast->classification == classification)
                                        {
                                            points->push_back(ptLast->z);
                                        }
                                    }
                                }
                            }
                            else
                            {
                                if((ptFirst->z >= lowThreshold) & (ptFirst->z <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst->z);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst->z);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst->z);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst->z);
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if((ptLast->z >= lowThreshold) & (ptLast->z <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast->z);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast->z);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast->z);
                                        }
                                        else if(ptLast->classification == classification)
                                        {
                                            points->push_back(ptLast->z);
                                        }
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_NOTFIRST_RETURNS)
                        {
                            std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin();
                            ++iterPts;
                            for(; iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & ((*iterPts)->z <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                }
                                else if(((*iterPts)->z >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                }
                                else if(((*iterPts)->z >= lowThreshold) & ((*iterPts)->z <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back((*iterPts)->z);
                                    }
                                }
                            }
                        }
                        else
                        {
                            for(std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin(); iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(returnID == (*iterPts)->returnID)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->z);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->z);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->z);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->z);
                                        }
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & ((*iterPts)->z <= upThreshold))
                                {
                                    if(returnID == (*iterPts)->returnID)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->z);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->z);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->z);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->z);
                                        }
                                    }
                                }
                                else if(((*iterPts)->z >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(returnID == (*iterPts)->returnID)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->z);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->z);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->z);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->z);
                                        }
                                    }
                                }
                                else if(((*iterPts)->z >= lowThreshold) & ((*iterPts)->z <= upThreshold))
                                {
                                    if(returnID == (*iterPts)->returnID)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->z);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->z);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->z);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->z);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            else if((spdFile->getReceiveWaveformDefined() == SPD_TRUE) & this->minNumReturns != 0)
            {
                throw SPDProcessingException("There are no points and the number of returns has been set to a number > 0."
                                             " Either set 'minNumReturns' to 0 or run decomposition");
            }
            else
            {
                throw SPDProcessingException("Neither waveform or point returns have been defind.");
            }

            return points;
        };
        virtual std::vector<double>* getPointsValuesWithinAmplitudeParameters(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            std::vector<double> *points = new std::vector<double>();
            // If there is waveform data and the minimum number of returns has been set to 0
            if((spdFile->getReceiveWaveformDefined() == SPD_TRUE) & (this->minNumReturns == 0))
            {
                // Get amplitude values for all returns of pulses in geom with amplitude falling between lowThreshold and upThreshold.
                for(std::vector<SPDPulse*>::iterator iterPulses = pulses->begin(); iterPulses != pulses->end(); ++iterPulses)
                {
                    for(unsigned int s = 0; s < (*iterPulses)->numOfReceivedBins; s++)
                    {
                        double pulseDNVal = (*iterPulses)->received[s];
                        if(pulseDNVal > (*iterPulses)->receiveWaveNoiseThreshold)
                        {
                            if((boost::math::isnan(this->upThreshold)) & (boost::math::isnan(this->lowThreshold)))
                            {
                                points->push_back(pulseDNVal);
                            }
                            else if((boost::math::isnan(this->lowThreshold)) & (pulseDNVal <= this->upThreshold))
                            {
                                points->push_back(pulseDNVal);
                            }
                            else if((pulseDNVal >= this->lowThreshold) & (boost::math::isnan(this->upThreshold)))
                            {
                                points->push_back(pulseDNVal);
                            }
                            else if((pulseDNVal >= this->lowThreshold) & (pulseDNVal <= this->upThreshold))
                            {
                                points->push_back(pulseDNVal);
                            }
                        }
                    }
                }
            }
            else if((spdFile->getDecomposedPtDefined() == SPD_TRUE) | (spdFile->getDiscretePtDefined() == SPD_TRUE))
            {
                for(std::vector<SPDPulse*>::iterator iterPulses = pulses->begin(); iterPulses != pulses->end(); ++iterPulses)
                {
                    if(((*iterPulses)->numberOfReturns >= minNumReturns) & ((*iterPulses)->numberOfReturns > 0))
                    {
                        if(returnID == SPD_ALL_RETURNS)
                        {
                            for(std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin(); iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                }
                                else if(boost::math::isnan(heightLowThreshold) & ((*iterPts)->height <= heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                }
                                else if(((*iterPts)->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                }
                                else if(((*iterPts)->height >= heightLowThreshold) & ((*iterPts)->height <= heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_LAST_RETURNS)
                        {
                            SPDPoint *pt = (*iterPulses)->pts->back();
                            if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                            }
                            else if(boost::math::isnan(heightLowThreshold) & (pt->height <= heightUpThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                            }
                            else if((pt->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                            }
                            else if((pt->height >= heightLowThreshold) & (pt->height <= heightUpThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_FIRST_RETURNS)
                        {
                            SPDPoint *pt = (*iterPulses)->pts->front();
                            if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                            }
                            else if(boost::math::isnan(heightLowThreshold) & (pt->height <= heightUpThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                            }
                            else if((pt->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                            }
                            else if((pt->height >= heightLowThreshold) & (pt->height <= heightUpThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->amplitudeReturn);
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_FIRSTLAST_RETURNS)
                        {
                            SPDPoint *ptFirst = (*iterPulses)->pts->front();
                            SPDPoint *ptLast = NULL;
                            if((*iterPulses)->pts->size() > 1)
                            {
                                ptLast = (*iterPulses)->pts->back();
                            }
                            if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                            {
                                // First
                                if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst->amplitudeReturn);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst->amplitudeReturn);
                                    }
                                }
                                else if(boost::math::isnan(heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst->amplitudeReturn);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst->amplitudeReturn);
                                    }
                                }
                                else if((ptFirst->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst->amplitudeReturn);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst->amplitudeReturn);
                                    }
                                }
                                else if((ptFirst->height >= heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst->amplitudeReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst->amplitudeReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst->amplitudeReturn);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst->amplitudeReturn);
                                    }
                                }

                                // Last
                                if(ptLast != NULL)
                                {
                                    if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast->amplitudeReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptLast->amplitudeReturn);
                                        }
                                    }
                                    else if(boost::math::isnan(heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast->amplitudeReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptLast->amplitudeReturn);
                                        }
                                    }
                                    else if((ptLast->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast->amplitudeReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptLast->amplitudeReturn);
                                        }
                                    }
                                    else if((ptLast->height >= heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast->amplitudeReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptLast->amplitudeReturn);
                                        }
                                    }
                                }
                            }
                            else if(boost::math::isnan(lowThreshold))
                            {
                                if(ptFirst->amplitudeReturn <= upThreshold)
                                {
                                    if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                    }
                                    else if(boost::math::isnan(heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                    }
                                    else if((ptFirst->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                    }
                                    else if((ptFirst->height >= heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if(ptLast->amplitudeReturn <= upThreshold)
                                    {
                                        if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                        }
                                        else if(boost::math::isnan(heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                        }
                                        else if((ptLast->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                        }
                                        else if((ptLast->height >= heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                        }
                                    }
                                }
                            }
                            else if(boost::math::isnan(upThreshold))
                            {
                                if(ptFirst->amplitudeReturn >= lowThreshold)
                                {
                                    if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                    }
                                    else if(boost::math::isnan(heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                    }
                                    else if((ptFirst->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                    }
                                    else if((ptFirst->height >= heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if(ptLast->amplitudeReturn >= lowThreshold)
                                    {
                                        if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                        }
                                        else if(boost::math::isnan(heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                        }
                                        else if((ptLast->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                        }
                                        else if((ptLast->height >= heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                        }
                                    }
                                }
                            }
                            else
                            {
                                if((ptFirst->amplitudeReturn >= lowThreshold) & (ptFirst->amplitudeReturn <= upThreshold))
                                {
                                    if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                    }
                                    else if(boost::math::isnan(heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                    }
                                    else if((ptFirst->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                    }
                                    else if((ptFirst->height >= heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst->amplitudeReturn);
                                        }
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if((ptLast->amplitudeReturn >= lowThreshold) & (ptLast->amplitudeReturn <= upThreshold))
                                    {
                                        if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                        }
                                        else if(boost::math::isnan(heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                        }
                                        else if((ptLast->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                        }
                                        else if((ptLast->height >= heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast->amplitudeReturn);
                                            }
                                        }
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_NOTFIRST_RETURNS)
                        {
                            std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin();
                            ++iterPts;
                            for(; iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                }
                                else if(boost::math::isnan(heightLowThreshold) & ((*iterPts)->height <= heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                }
                                else if(((*iterPts)->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                }
                                else if(((*iterPts)->height >= heightLowThreshold) & ((*iterPts)->height <= heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->amplitudeReturn);
                                        }
                                    }
                                }
                            }
                        }
                        else
                        {
                            for(std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin(); iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                        }
                                    }
                                }
                                else if(boost::math::isnan(heightLowThreshold) & ((*iterPts)->height <= heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                        }
                                    }
                                }
                                else if(((*iterPts)->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                        }
                                    }
                                }
                                else if(((*iterPts)->height >= heightLowThreshold) & ((*iterPts)->height <= heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->amplitudeReturn);
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            else if((spdFile->getReceiveWaveformDefined() == SPD_TRUE) & this->minNumReturns != 0)
            {
                throw SPDProcessingException("There are no points and the number of returns has been set to a number > 0."
                                             " Either set 'minNumReturns' to 0 or run decomposition");
            }
            else
            {
                throw SPDProcessingException("Neither waveform or point returns have been defind.");
            }
            return points;
        };
        virtual std::vector<double>* getPointsValuesWithinRangeParameters(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            std::vector<double> *points = new std::vector<double>();
            if((spdFile->getReceiveWaveformDefined() == SPD_TRUE) & (this->minNumReturns == 0))
            {
                throw SPDProcessingException("Waveform metrics haven't been implemented for range."
                                             " Set minNumReturns to 1 (default) to calculate using returns");
            }
            else if((spdFile->getDecomposedPtDefined() == SPD_TRUE) | (spdFile->getDiscretePtDefined() == SPD_TRUE))
            {
                for(std::vector<SPDPulse*>::iterator iterPulses = pulses->begin(); iterPulses != pulses->end(); ++iterPulses)
                {
                    if(((*iterPulses)->numberOfReturns >= minNumReturns) & ((*iterPulses)->numberOfReturns > 0))
                    {
                        if(returnID == SPD_ALL_RETURNS)
                        {
                            for(std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin(); iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & ((*iterPts)->range <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                }
                                else if(((*iterPts)->range >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                }
                                else if(((*iterPts)->range >= lowThreshold) & ((*iterPts)->range <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_LAST_RETURNS)
                        {
                            SPDPoint *pt = (*iterPulses)->pts->back();
                            if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt->range);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt->range);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt->range);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt->range);
                                }
                            }
                            else if(boost::math::isnan(lowThreshold) & (pt->range <= upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt->range);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt->range);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt->range);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt->range);
                                }
                            }
                            else if((pt->range >= lowThreshold) & boost::math::isnan(upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt->range);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt->range);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt->range);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt->range);
                                }
                            }
                            else if((pt->range >= lowThreshold) & (pt->range <= upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt->range);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt->range);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt->range);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt->range);
                                }
                            }
                        }
                        else if(returnID == SPD_FIRST_RETURNS)
                        {
                            SPDPoint *pt = (*iterPulses)->pts->front();
                            if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt->range);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt->range);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt->range);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt->range);
                                }
                            }
                            else if(boost::math::isnan(lowThreshold) & (pt->range <= upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt->range);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt->range);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt->range);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt->range);
                                }
                            }
                            else if((pt->range >= lowThreshold) & boost::math::isnan(upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt->range);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt->range);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt->range);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt->range);
                                }
                            }
                            else if((pt->range >= lowThreshold) & (pt->range <= upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt->range);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt->range);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt->range);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt->range);
                                }
                            }
                        }
                        else if(returnID == SPD_FIRSTLAST_RETURNS)
                        {
                            SPDPoint *ptFirst = (*iterPulses)->pts->front();
                            SPDPoint *ptLast = NULL;
                            if((*iterPulses)->pts->size() > 1)
                            {
                                ptLast = (*iterPulses)->pts->back();
                            }
                            if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                            {
                                // First
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(ptFirst->range);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                         (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                         (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(ptFirst->range);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (ptFirst->classification != SPD_GROUND))
                                {
                                    points->push_back(ptFirst->range);
                                }
                                else if(ptFirst->classification == classification)
                                {
                                    points->push_back(ptFirst->range);
                                }

                                // Last
                                if(ptLast != NULL)
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptLast->range);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptLast->classification == SPD_LOW_VEGETATION) |
                                             (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptLast->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptLast->range);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptLast->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptLast->range);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptLast->range);
                                    }
                                }
                            }
                            else if(boost::math::isnan(lowThreshold))
                            {
                                if(ptFirst->range <= upThreshold)
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst->range);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst->range);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst->range);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst->range);
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if(ptLast->range <= upThreshold)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast->range);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast->range);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast->range);
                                        }
                                        else if(ptLast->classification == classification)
                                        {
                                            points->push_back(ptLast->range);
                                        }
                                    }
                                }
                            }
                            else if(boost::math::isnan(upThreshold))
                            {
                                if(ptFirst->range >= lowThreshold)
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst->range);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst->range);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst->range);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst->range);
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if(ptLast->range >= lowThreshold)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast->range);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast->range);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast->range);
                                        }
                                        else if(ptLast->classification == classification)
                                        {
                                            points->push_back(ptLast->range);
                                        }
                                    }
                                }
                            }
                            else
                            {
                                if((ptFirst->range >= lowThreshold) & (ptFirst->range <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst->range);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst->range);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst->range);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst->range);
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if((ptLast->range >= lowThreshold) & (ptLast->range <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast->range);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast->range);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast->range);
                                        }
                                        else if(ptLast->classification == classification)
                                        {
                                            points->push_back(ptLast->range);
                                        }
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_NOTFIRST_RETURNS)
                        {
                            std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin();
                            ++iterPts;
                            for(; iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & ((*iterPts)->range <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                }
                                else if(((*iterPts)->range >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                }
                                else if(((*iterPts)->range >= lowThreshold) & ((*iterPts)->range <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back((*iterPts)->range);
                                    }
                                }
                            }
                        }
                        else
                        {
                            for(std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin(); iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(returnID == (*iterPts)->returnID)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->range);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->range);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->range);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->range);
                                        }
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & ((*iterPts)->range <= upThreshold))
                                {
                                    if(returnID == (*iterPts)->returnID)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->range);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->range);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->range);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->range);
                                        }
                                    }
                                }
                                else if(((*iterPts)->range >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(returnID == (*iterPts)->returnID)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->range);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->range);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->range);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->range);
                                        }
                                    }
                                }
                                else if(((*iterPts)->range >= lowThreshold) & ((*iterPts)->range <= upThreshold))
                                {
                                    if(returnID == (*iterPts)->returnID)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->range);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->range);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->range);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->range);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            else if(spdFile->getReceiveWaveformDefined() == SPD_TRUE)
            {
                throw SPDProcessingException("Use the waveform metrics for waveform only data or decomposed to retrieve points.");
            }
            else
            {
                throw SPDProcessingException("Neither waveform or point returns have been defind.");
            }
            return points;
        };
        virtual std::vector<double>* getPointsValuesWithinWidthParameters(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            std::vector<double> *points = new std::vector<double>();
            if((spdFile->getReceiveWaveformDefined() == SPD_TRUE) & (this->minNumReturns == 0))
            {
                throw SPDProcessingException("Can't calculate width metrics using only waveform data, need to use returns."
                                             " Set minNumReturns to 1 (default) to calculate using returns.");
            }
            else if((spdFile->getDecomposedPtDefined() == SPD_TRUE) | (spdFile->getDiscretePtDefined() == SPD_TRUE))
            {
                for(std::vector<SPDPulse*>::iterator iterPulses = pulses->begin(); iterPulses != pulses->end(); ++iterPulses)
                {
                    if(((*iterPulses)->numberOfReturns >= minNumReturns) & ((*iterPulses)->numberOfReturns > 0))
                    {
                        if(returnID == SPD_ALL_RETURNS)
                        {
                            for(std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin(); iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                }
                                else if(boost::math::isnan(heightLowThreshold) & ((*iterPts)->height <= heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                }
                                else if(((*iterPts)->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                }
                                else if(((*iterPts)->height >= heightLowThreshold) & ((*iterPts)->height <= heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_LAST_RETURNS)
                        {
                            SPDPoint *pt = (*iterPulses)->pts->back();
                            if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                            }
                            else if(boost::math::isnan(heightLowThreshold) & (pt->height <= heightUpThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                            }
                            else if((pt->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                            }
                            else if((pt->height >= heightLowThreshold) & (pt->height <= heightUpThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_FIRST_RETURNS)
                        {
                            SPDPoint *pt = (*iterPulses)->pts->front();
                            if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                            }
                            else if(boost::math::isnan(heightLowThreshold) & (pt->height <= heightUpThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                            }
                            else if((pt->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                            }
                            else if((pt->height >= heightLowThreshold) & (pt->height <= heightUpThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt->widthReturn);
                                    }
                                }

                            }
                        }
                        else if(returnID == SPD_FIRSTLAST_RETURNS)
                        {
                            SPDPoint *ptFirst = (*iterPulses)->pts->front();
                            SPDPoint *ptLast = NULL;
                            if((*iterPulses)->pts->size() > 1)
                            {
                                ptLast = (*iterPulses)->pts->back();
                            }
                            if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                            {
                                // First
                                if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst->widthReturn);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst->widthReturn);
                                    }
                                }
                                else if(boost::math::isnan(heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst->widthReturn);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst->widthReturn);
                                    }
                                }
                                else if((ptFirst->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst->widthReturn);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst->widthReturn);
                                    }
                                }
                                else if((ptFirst->height >= heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst->widthReturn);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst->widthReturn);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst->widthReturn);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst->widthReturn);
                                    }
                                }

                                // Last
                                if(ptLast != NULL)
                                {
                                    if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast->widthReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptLast->widthReturn);
                                        }
                                    }
                                    else if(boost::math::isnan(heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast->widthReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptLast->widthReturn);
                                        }
                                    }
                                    else if((ptLast->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast->widthReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptLast->widthReturn);
                                        }
                                    }
                                    else if((ptLast->height >= heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast->widthReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptLast->widthReturn);
                                        }
                                    }
                                }
                            }
                            else if(boost::math::isnan(lowThreshold))
                            {
                                if(ptFirst->widthReturn <= upThreshold)
                                {
                                    if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                    }
                                    else if(boost::math::isnan(heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                    }
                                    else if((ptFirst->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                    }
                                    else if((ptFirst->height >= heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if(ptLast->widthReturn <= upThreshold)
                                    {
                                        if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                        }
                                        else if(boost::math::isnan(heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                        }
                                        else if((ptLast->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                        }
                                        else if((ptLast->height >= heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                        }
                                    }
                                }
                            }
                            else if(boost::math::isnan(upThreshold))
                            {
                                if(ptFirst->widthReturn >= lowThreshold)
                                {
                                    if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                    }
                                    else if(boost::math::isnan(heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                    }
                                    else if((ptFirst->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                    }
                                    else if((ptFirst->height >= heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if(ptLast->widthReturn >= lowThreshold)
                                    {
                                        if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                        }
                                        else if(boost::math::isnan(heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                        }
                                        else if((ptLast->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                        }
                                        else if((ptLast->height >= heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                        }
                                    }
                                }
                            }
                            else
                            {
                                //First
                                if((ptFirst->widthReturn >= lowThreshold) & (ptFirst->widthReturn <= upThreshold))
                                {
                                    if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                    }
                                    else if(boost::math::isnan(heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                    }
                                    else if((ptFirst->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                    }
                                    else if((ptFirst->height >= heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst->widthReturn);
                                        }
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if((ptLast->widthReturn >= lowThreshold) & (ptLast->widthReturn <= upThreshold))
                                    {
                                        if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                        }
                                        else if(boost::math::isnan(heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                        }
                                        else if((ptLast->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                        }
                                        else if((ptLast->height >= heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast->widthReturn);
                                            }
                                        }
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_NOTFIRST_RETURNS)
                        {
                            std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin();
                            ++iterPts;
                            for(; iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                }
                                else if(boost::math::isnan(heightLowThreshold) & ((*iterPts)->height <= heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                }
                                else if(((*iterPts)->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                }
                                else if(((*iterPts)->height >= heightLowThreshold) & ((*iterPts)->height <= heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts)->widthReturn);
                                        }
                                    }
                                }
                            }
                        }
                        else
                        {
                            for(std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin(); iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                        }
                                    }
                                }
                                else if(boost::math::isnan(heightLowThreshold) & ((*iterPts)->height <= heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                        }
                                    }
                                }
                                else if(((*iterPts)->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                        }
                                    }
                                }
                                else if(((*iterPts)->height >= heightLowThreshold) & ((*iterPts)->height <= heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts)->widthReturn);
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            else if(spdFile->getReceiveWaveformDefined() == SPD_TRUE)
            {
                throw SPDProcessingException("Use the waveform metrics for waveform only data or decomposed to retrieve points.");
            }
            else
            {
                throw SPDProcessingException("Neither waveform or point returns have been defind.");
            }
            return points;
        };
        virtual std::vector<SPDPoint*>* getPointsWithinHeightParameters(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            std::vector<SPDPoint*> *points = new std::vector<SPDPoint*>();
            if((spdFile->getReceiveWaveformDefined() == SPD_TRUE) & (this->minNumReturns == 0))
            {
                throw SPDProcessingException("This metric can't be calculated from waveform data."
                                             " Set minNumReturns to 1 (default) to calculate using returns");
            }
            else if((spdFile->getDecomposedPtDefined() == SPD_TRUE) | (spdFile->getDiscretePtDefined() == SPD_TRUE))
            {
                for(std::vector<SPDPulse*>::iterator iterPulses = pulses->begin(); iterPulses != pulses->end(); ++iterPulses)
                {
                    if(((*iterPulses)->numberOfReturns >= minNumReturns) & ((*iterPulses)->numberOfReturns > 0))
                    {
                        if(returnID == SPD_ALL_RETURNS)
                        {
                            for(std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin(); iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & ((*iterPts)->height <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                }
                                else if(((*iterPts)->height >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                }
                                else if(((*iterPts)->height >= lowThreshold) & ((*iterPts)->height <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_LAST_RETURNS)
                        {
                            SPDPoint *pt = (*iterPulses)->pts->back();
                            if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt);
                                }
                            }
                            else if(boost::math::isnan(lowThreshold) & (pt->height <= upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt);
                                }
                            }
                            else if((pt->height >= lowThreshold) & boost::math::isnan(upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt);
                                }
                            }
                            else if((pt->height >= lowThreshold) & (pt->height <= upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt);
                                }
                            }
                        }
                        else if(returnID == SPD_FIRST_RETURNS)
                        {
                            SPDPoint *pt = (*iterPulses)->pts->front();
                            if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt);
                                }
                            }
                            else if(boost::math::isnan(lowThreshold) & (pt->height <= upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt);
                                }
                            }
                            else if((pt->height >= lowThreshold) & boost::math::isnan(upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt);
                                }
                            }
                            else if((pt->height >= lowThreshold) & (pt->height <= upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt);
                                }
                            }
                        }
                        else if(returnID == SPD_FIRSTLAST_RETURNS)
                        {
                            SPDPoint *ptFirst = (*iterPulses)->pts->front();
                            SPDPoint *ptLast = NULL;
                            if((*iterPulses)->pts->size() > 1)
                            {
                                ptLast = (*iterPulses)->pts->back();
                            }
                            if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                            {
                                // First
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(ptFirst);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                         (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                         (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(ptFirst);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (ptFirst->classification != SPD_GROUND))
                                {
                                    points->push_back(ptFirst);
                                }
                                else if(ptFirst->classification == classification)
                                {
                                    points->push_back(ptFirst);
                                }

                                // Last
                                if(ptLast != NULL)
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptLast);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptLast->classification == SPD_LOW_VEGETATION) |
                                             (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptLast->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptLast);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptLast->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptLast);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptLast);
                                    }
                                }
                            }
                            else if(boost::math::isnan(lowThreshold))
                            {
                                if(ptFirst->height <= upThreshold)
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if(ptLast->height <= upThreshold)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if(ptLast->classification == classification)
                                        {
                                            points->push_back(ptLast);
                                        }
                                    }
                                }
                            }
                            else if(boost::math::isnan(upThreshold))
                            {
                                if(ptFirst->height >= lowThreshold)
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if(ptLast->height >= lowThreshold)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if(ptLast->classification == classification)
                                        {
                                            points->push_back(ptLast);
                                        }
                                    }
                                }
                            }
                            else
                            {
                                if((ptFirst->height >= lowThreshold) & (ptFirst->height <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if((ptLast->height >= lowThreshold) & (ptLast->height <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if(ptLast->classification == classification)
                                        {
                                            points->push_back(ptLast);
                                        }
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_NOTFIRST_RETURNS)
                        {
                            std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin();
                            ++iterPts;
                            for(; iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & ((*iterPts)->height <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                }
                                else if(((*iterPts)->height >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                }
                                else if(((*iterPts)->height >= lowThreshold) & ((*iterPts)->height <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                }
                            }
                        }
                        else
                        {
                            for(std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin(); iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(returnID == (*iterPts)->returnID)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back(*iterPts);
                                        }
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & ((*iterPts)->height <= upThreshold))
                                {
                                    if(returnID == (*iterPts)->returnID)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back(*iterPts);
                                        }
                                    }
                                }
                                else if(((*iterPts)->height >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(returnID == (*iterPts)->returnID)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back(*iterPts);
                                        }
                                    }
                                }
                                else if(((*iterPts)->height >= lowThreshold) & ((*iterPts)->height <= upThreshold))
                                {
                                    if(returnID == (*iterPts)->returnID)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back(*iterPts);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            else if(spdFile->getReceiveWaveformDefined() == SPD_TRUE)
            {
                throw SPDProcessingException("Use the waveform metrics for waveform only data or decomposed to retrieve points.");
            }
            else
            {
                throw SPDProcessingException("Neither waveform or point returns have been defind.");
            }

            return points;
        };
        virtual std::vector<SPDPoint*>* getPointsWithinZParameters(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            std::vector<SPDPoint*> *points = new std::vector<SPDPoint*>();
            if((spdFile->getReceiveWaveformDefined() == SPD_TRUE) & (this->minNumReturns == 0))
            {
                throw SPDProcessingException("This metric can't be calculated from waveform data."
                                             " Set minNumReturns to 1 (default) to calculate using returns");
            }
            else if((spdFile->getDecomposedPtDefined() == SPD_TRUE) | (spdFile->getDiscretePtDefined() == SPD_TRUE))
            {
                for(std::vector<SPDPulse*>::iterator iterPulses = pulses->begin(); iterPulses != pulses->end(); ++iterPulses)
                {
                    if(((*iterPulses)->numberOfReturns >= minNumReturns) & ((*iterPulses)->numberOfReturns > 0))
                    {
                        if(returnID == SPD_ALL_RETURNS)
                        {
                            for(std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin(); iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & ((*iterPts)->z <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                }
                                else if(((*iterPts)->z >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                }
                                else if(((*iterPts)->z >= lowThreshold) & ((*iterPts)->z <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_LAST_RETURNS)
                        {
                            SPDPoint *pt = (*iterPulses)->pts->back();
                            if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt);
                                }
                            }
                            else if(boost::math::isnan(lowThreshold) & (pt->z <= upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt);
                                }
                            }
                            else if((pt->z >= lowThreshold) & boost::math::isnan(upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt);
                                }
                            }
                            else if((pt->z >= lowThreshold) & (pt->z <= upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt);
                                }
                            }
                        }
                        else if(returnID == SPD_FIRST_RETURNS)
                        {
                            SPDPoint *pt = (*iterPulses)->pts->front();
                            if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt);
                                }
                            }
                            else if(boost::math::isnan(lowThreshold) & (pt->z <= upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt);
                                }
                            }
                            else if((pt->z >= lowThreshold) & boost::math::isnan(upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt);
                                }
                            }
                            else if((pt->z >= lowThreshold) & (pt->z <= upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt);
                                }
                            }
                        }
                        else if(returnID == SPD_FIRSTLAST_RETURNS)
                        {
                            SPDPoint *ptFirst = (*iterPulses)->pts->front();
                            SPDPoint *ptLast = NULL;
                            if((*iterPulses)->pts->size() > 1)
                            {
                                ptLast = (*iterPulses)->pts->back();
                            }
                            if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                            {
                                // First
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(ptFirst);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                         (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                         (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(ptFirst);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (ptFirst->classification != SPD_GROUND))
                                {
                                    points->push_back(ptFirst);
                                }
                                else if(ptFirst->classification == classification)
                                {
                                    points->push_back(ptFirst);
                                }

                                // Last
                                if(ptLast != NULL)
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptLast);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptLast->classification == SPD_LOW_VEGETATION) |
                                             (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptLast->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptLast);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptLast->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptLast);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptLast);
                                    }
                                }
                            }
                            else if(boost::math::isnan(lowThreshold))
                            {
                                if(ptFirst->z <= upThreshold)
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if(ptLast->z <= upThreshold)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if(ptLast->classification == classification)
                                        {
                                            points->push_back(ptLast);
                                        }
                                    }
                                }
                            }
                            else if(boost::math::isnan(upThreshold))
                            {
                                if(ptFirst->z >= lowThreshold)
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if(ptLast->z >= lowThreshold)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if(ptLast->classification == classification)
                                        {
                                            points->push_back(ptLast);
                                        }
                                    }
                                }
                            }
                            else
                            {
                                if((ptFirst->z >= lowThreshold) & (ptFirst->z <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if((ptLast->z >= lowThreshold) & (ptLast->z <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if(ptLast->classification == classification)
                                        {
                                            points->push_back(ptLast);
                                        }
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_NOTFIRST_RETURNS)
                        {
                            std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin();
                            ++iterPts;
                            for(; iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & ((*iterPts)->z <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                }
                                else if(((*iterPts)->z >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                }
                                else if(((*iterPts)->z >= lowThreshold) & ((*iterPts)->z <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                }
                            }
                        }
                        else
                        {
                            for(std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin(); iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(returnID == (*iterPts)->returnID)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back(*iterPts);
                                        }
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & ((*iterPts)->z <= upThreshold))
                                {
                                    if(returnID == (*iterPts)->returnID)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back(*iterPts);
                                        }
                                    }
                                }
                                else if(((*iterPts)->z >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(returnID == (*iterPts)->returnID)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back(*iterPts);
                                        }
                                    }
                                }
                                else if(((*iterPts)->z >= lowThreshold) & ((*iterPts)->z <= upThreshold))
                                {
                                    if(returnID == (*iterPts)->returnID)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back(*iterPts);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            else if(spdFile->getReceiveWaveformDefined() == SPD_TRUE)
            {
                throw SPDProcessingException("Use the waveform metrics for waveform only data or decomposed to retrieve points.");
            }
            else
            {
                throw SPDProcessingException("Neither waveform or point returns have been defind.");
            }

            return points;
        };
        virtual std::vector<SPDPoint*>* getPointsWithinAmplitudeParameters(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            std::vector<SPDPoint*> *points = new std::vector<SPDPoint*>();
            if((spdFile->getReceiveWaveformDefined() == SPD_TRUE) & (this->minNumReturns == 0))
            {
                throw SPDProcessingException("This metric can't be calculated from waveform data."
                                             " Set minNumReturns to 1 (default) to calculate using returns");
            }
            else if((spdFile->getDecomposedPtDefined() == SPD_TRUE) | (spdFile->getDiscretePtDefined() == SPD_TRUE))
            {
                for(std::vector<SPDPulse*>::iterator iterPulses = pulses->begin(); iterPulses != pulses->end(); ++iterPulses)
                {
                    if(((*iterPulses)->numberOfReturns >= minNumReturns) & ((*iterPulses)->numberOfReturns > 0))
                    {
                        if(returnID == SPD_ALL_RETURNS)
                        {
                            for(std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin(); iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                }
                                else if(boost::math::isnan(heightLowThreshold) & ((*iterPts)->height <= heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                }
                                else if(((*iterPts)->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                }
                                else if(((*iterPts)->height >= heightLowThreshold) & ((*iterPts)->height <= heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_LAST_RETURNS)
                        {
                            SPDPoint *pt = (*iterPulses)->pts->back();
                            if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                            }
                            else if(boost::math::isnan(heightLowThreshold) & (pt->height <= heightUpThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                            }
                            else if((pt->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                            }
                            else if((pt->height >= heightLowThreshold) & (pt->height <= heightUpThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_FIRST_RETURNS)
                        {
                            SPDPoint *pt = (*iterPulses)->pts->front();
                            if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                            }
                            else if(boost::math::isnan(heightLowThreshold) & (pt->height <= heightUpThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                            }
                            else if((pt->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                            }
                            else if((pt->height >= heightLowThreshold) & (pt->height <= heightUpThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->amplitudeReturn >= lowThreshold) & (pt->amplitudeReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_FIRSTLAST_RETURNS)
                        {
                            SPDPoint *ptFirst = (*iterPulses)->pts->front();
                            SPDPoint *ptLast = NULL;
                            if((*iterPulses)->pts->size() > 1)
                            {
                                ptLast = (*iterPulses)->pts->back();
                            }
                            if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                            {
                                // First
                                if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                }
                                else if(boost::math::isnan(heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                }
                                else if((ptFirst->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                }
                                else if((ptFirst->height >= heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                }

                                // Last
                                if(ptLast != NULL)
                                {
                                    if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptLast);
                                        }
                                    }
                                    else if(boost::math::isnan(heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptLast);
                                        }
                                    }
                                    else if((ptLast->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptLast);
                                        }
                                    }
                                    else if((ptLast->height >= heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptLast);
                                        }
                                    }
                                }
                            }
                            else if(boost::math::isnan(lowThreshold))
                            {
                                if(ptFirst->amplitudeReturn <= upThreshold)
                                {
                                    if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                    }
                                    else if(boost::math::isnan(heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                    }
                                    else if((ptFirst->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                    }
                                    else if((ptFirst->height >= heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if(ptLast->amplitudeReturn <= upThreshold)
                                    {
                                        if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast);
                                            }
                                        }
                                        else if(boost::math::isnan(heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast);
                                            }
                                        }
                                        else if((ptLast->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast);
                                            }
                                        }
                                        else if((ptLast->height >= heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast);
                                            }
                                        }
                                    }
                                }
                            }
                            else if(boost::math::isnan(upThreshold))
                            {
                                if(ptFirst->amplitudeReturn >= lowThreshold)
                                {
                                    if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                    }
                                    else if(boost::math::isnan(heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                    }
                                    else if((ptFirst->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                    }
                                    else if((ptFirst->height >= heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if(ptLast->amplitudeReturn >= lowThreshold)
                                    {
                                        if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast);
                                            }
                                        }
                                        else if(boost::math::isnan(heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast);
                                            }
                                        }
                                        else if((ptLast->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast);
                                            }
                                        }
                                        else if((ptLast->height >= heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast);
                                            }
                                        }
                                    }
                                }
                            }
                            else
                            {
                                if((ptFirst->amplitudeReturn >= lowThreshold) & (ptFirst->amplitudeReturn <= upThreshold))
                                {
                                    if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                    }
                                    else if(boost::math::isnan(heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                    }
                                    else if((ptFirst->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                    }
                                    else if((ptFirst->height >= heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if((ptLast->amplitudeReturn >= lowThreshold) & (ptLast->amplitudeReturn <= upThreshold))
                                    {
                                        if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast);
                                            }
                                        }
                                        else if(boost::math::isnan(heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast);
                                            }
                                        }
                                        else if((ptLast->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast);
                                            }
                                        }
                                        else if((ptLast->height >= heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast);
                                            }
                                        }
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_NOTFIRST_RETURNS)
                        {
                            std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin();
                            ++iterPts;
                            for(; iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                }
                                else if(boost::math::isnan(heightLowThreshold) & ((*iterPts)->height <= heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                }
                                else if(((*iterPts)->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                }
                                else if(((*iterPts)->height >= heightLowThreshold) & ((*iterPts)->height <= heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                }
                            }
                        }
                        else
                        {
                            for(std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin(); iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                        }
                                    }
                                }
                                else if(boost::math::isnan(heightLowThreshold) & ((*iterPts)->height <= heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                        }
                                    }
                                }
                                else if(((*iterPts)->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                        }
                                    }
                                }
                                else if(((*iterPts)->height >= heightLowThreshold) & ((*iterPts)->height <= heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                        }
                                    }
                                    else if(((*iterPts)->amplitudeReturn >= lowThreshold) & ((*iterPts)->amplitudeReturn <= upThreshold))
                                    {
                                        if(returnID == (*iterPts)->returnID)
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                     ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    ((*iterPts)->classification != SPD_GROUND))
                                            {
                                                points->push_back((*iterPts));
                                            }
                                            else if((*iterPts)->classification == classification)
                                            {
                                                points->push_back((*iterPts));
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            else if(spdFile->getReceiveWaveformDefined() == SPD_TRUE)
            {
                throw SPDProcessingException("Use the waveform metrics for waveform only data or decomposed to retrieve points.");
            }
            else
            {
                throw SPDProcessingException("Neither waveform or point returns have been defind.");
            }

            return points;
        };
        virtual std::vector<SPDPoint*>* getPointsWithinRangeParameters(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            std::vector<SPDPoint*> *points = new std::vector<SPDPoint*>();
            if((spdFile->getReceiveWaveformDefined() == SPD_TRUE) & (this->minNumReturns == 0))
            {
                throw SPDProcessingException("Waveform metrics haven't been implemented for range."
                                             " Set minNumReturns to 1 (default) to calculate using returns.");
            }
            else if((spdFile->getDecomposedPtDefined() == SPD_TRUE) | (spdFile->getDiscretePtDefined() == SPD_TRUE))
            {
                for(std::vector<SPDPulse*>::iterator iterPulses = pulses->begin(); iterPulses != pulses->end(); ++iterPulses)
                {
                    if(((*iterPulses)->numberOfReturns >= minNumReturns) & ((*iterPulses)->numberOfReturns > 0))
                    {
                        if(returnID == SPD_ALL_RETURNS)
                        {
                            for(std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin(); iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & ((*iterPts)->range <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                }
                                else if(((*iterPts)->range >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                }
                                else if(((*iterPts)->range >= lowThreshold) & ((*iterPts)->range <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_LAST_RETURNS)
                        {
                            SPDPoint *pt = (*iterPulses)->pts->back();
                            if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt);
                                }
                            }
                            else if(boost::math::isnan(lowThreshold) & (pt->range <= upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt);
                                }
                            }
                            else if((pt->range >= lowThreshold) & boost::math::isnan(upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt);
                                }
                            }
                            else if((pt->range >= lowThreshold) & (pt->range <= upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt);
                                }
                            }
                        }
                        else if(returnID == SPD_FIRST_RETURNS)
                        {
                            SPDPoint *pt = (*iterPulses)->pts->front();
                            if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt);
                                }
                            }
                            else if(boost::math::isnan(lowThreshold) & (pt->range <= upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt);
                                }
                            }
                            else if((pt->range >= lowThreshold) & boost::math::isnan(upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt);
                                }
                            }
                            else if((pt->range >= lowThreshold) & (pt->range <= upThreshold))
                            {
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((pt->classification == SPD_LOW_VEGETATION) |
                                         (pt->classification == SPD_MEDIUM_VEGETATION) |
                                         (pt->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(pt);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (pt->classification != SPD_GROUND))
                                {
                                    points->push_back(pt);
                                }
                                else if(pt->classification == classification)
                                {
                                    points->push_back(pt);
                                }
                            }
                        }
                        else if(returnID == SPD_FIRSTLAST_RETURNS)
                        {
                            SPDPoint *ptFirst = (*iterPulses)->pts->front();
                            SPDPoint *ptLast = NULL;
                            if((*iterPulses)->pts->size() > 1)
                            {
                                ptLast = (*iterPulses)->pts->back();
                            }
                            if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                            {
                                // First
                                if(classification == SPD_ALL_CLASSES)
                                {
                                    points->push_back(ptFirst);
                                }
                                else if((classification == SPD_VEGETATION) &
                                        ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                         (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                         (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                {
                                    points->push_back(ptFirst);
                                }
                                else if((classification == SPD_NOT_GROUND) &
                                        (ptFirst->classification != SPD_GROUND))
                                {
                                    points->push_back(ptFirst);
                                }
                                else if(ptFirst->classification == classification)
                                {
                                    points->push_back(ptFirst);
                                }

                                // Last
                                if(ptLast != NULL)
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptLast);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptLast->classification == SPD_LOW_VEGETATION) |
                                             (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptLast->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptLast);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptLast->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptLast);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptLast);
                                    }
                                }
                            }
                            else if(boost::math::isnan(lowThreshold))
                            {
                                if(ptFirst->range <= upThreshold)
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if(ptLast->range <= upThreshold)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if(ptLast->classification == classification)
                                        {
                                            points->push_back(ptLast);
                                        }
                                    }
                                }
                            }
                            else if(boost::math::isnan(upThreshold))
                            {
                                if(ptFirst->range >= lowThreshold)
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if(ptLast->range >= lowThreshold)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if(ptLast->classification == classification)
                                        {
                                            points->push_back(ptLast);
                                        }
                                    }
                                }
                            }
                            else
                            {
                                if((ptFirst->range >= lowThreshold) & (ptFirst->range <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if((ptLast->range >= lowThreshold) & (ptLast->range <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if(ptLast->classification == classification)
                                        {
                                            points->push_back(ptLast);
                                        }
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_NOTFIRST_RETURNS)
                        {
                            std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin();
                            ++iterPts;
                            for(; iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & ((*iterPts)->range <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                }
                                else if(((*iterPts)->range >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                }
                                else if(((*iterPts)->range >= lowThreshold) & ((*iterPts)->range <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                             ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                             ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            ((*iterPts)->classification != SPD_GROUND))
                                    {
                                        points->push_back(*iterPts);
                                    }
                                    else if((*iterPts)->classification == classification)
                                    {
                                        points->push_back(*iterPts);
                                    }
                                }
                            }
                        }
                        else
                        {
                            for(std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin(); iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(returnID == (*iterPts)->returnID)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back(*iterPts);
                                        }
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & ((*iterPts)->range <= upThreshold))
                                {
                                    if(returnID == (*iterPts)->returnID)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back(*iterPts);
                                        }
                                    }
                                }
                                else if(((*iterPts)->range >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(returnID == (*iterPts)->returnID)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back(*iterPts);
                                        }
                                    }
                                }
                                else if(((*iterPts)->range >= lowThreshold) & ((*iterPts)->range <= upThreshold))
                                {
                                    if(returnID == (*iterPts)->returnID)
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back(*iterPts);
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back(*iterPts);
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
            else if(spdFile->getReceiveWaveformDefined() == SPD_TRUE)
            {
                throw SPDProcessingException("Use the waveform metrics for waveform only data or decomposed to retrieve points.");
            }
            else
            {
                throw SPDProcessingException("Neither waveform or point returns have been defind.");
            }
            return points;
        };
        virtual std::vector<SPDPoint*>* getPointsWithinWidthParameters(std::vector<SPDPulse*> *pulses, SPDFile *spdFile, OGRGeometry *geom) throw(SPDProcessingException)
        {
            std::vector<SPDPoint*> *points = new std::vector<SPDPoint*>();
            if((spdFile->getReceiveWaveformDefined() == SPD_TRUE) & (this->minNumReturns == 0))
            {
                throw SPDProcessingException("Can't calculate width metrics using only waveform data, need to use returns."
                                             " Set minNumReturns to 1 (default) to calculate using returns.");
            }
            else if((spdFile->getDecomposedPtDefined() == SPD_TRUE) | (spdFile->getDiscretePtDefined() == SPD_TRUE))
            {
                for(std::vector<SPDPulse*>::iterator iterPulses = pulses->begin(); iterPulses != pulses->end(); ++iterPulses)
                {
                    if(((*iterPulses)->numberOfReturns >= minNumReturns) & ((*iterPulses)->numberOfReturns > 0))
                    {
                        if(returnID == SPD_ALL_RETURNS)
                        {
                            for(std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin(); iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                }
                                else if(boost::math::isnan(heightLowThreshold) & ((*iterPts)->height <= heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                }
                                else if(((*iterPts)->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                }
                                else if(((*iterPts)->height >= heightLowThreshold) & ((*iterPts)->height <= heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_LAST_RETURNS)
                        {
                            SPDPoint *pt = (*iterPulses)->pts->back();
                            if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                            }
                            else if(boost::math::isnan(heightLowThreshold) & (pt->height <= heightUpThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                            }
                            else if((pt->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                            }
                            else if((pt->height >= heightLowThreshold) & (pt->height <= heightUpThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_FIRST_RETURNS)
                        {
                            SPDPoint *pt = (*iterPulses)->pts->front();
                            if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                            }
                            else if(boost::math::isnan(heightLowThreshold) & (pt->height <= heightUpThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                            }
                            else if((pt->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                            }
                            else if((pt->height >= heightLowThreshold) & (pt->height <= heightUpThreshold))
                            {
                                if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if(boost::math::isnan(lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }
                                else if((pt->widthReturn >= lowThreshold) & (pt->widthReturn <= upThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((pt->classification == SPD_LOW_VEGETATION) |
                                             (pt->classification == SPD_MEDIUM_VEGETATION) |
                                             (pt->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (pt->classification != SPD_GROUND))
                                    {
                                        points->push_back(pt);
                                    }
                                    else if(pt->classification == classification)
                                    {
                                        points->push_back(pt);
                                    }
                                }

                            }
                        }
                        else if(returnID == SPD_FIRSTLAST_RETURNS)
                        {
                            SPDPoint *ptFirst = (*iterPulses)->pts->front();
                            SPDPoint *ptLast = NULL;
                            if((*iterPulses)->pts->size() > 1)
                            {
                                ptLast = (*iterPulses)->pts->back();
                            }
                            if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                            {
                                // First
                                if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                }
                                else if(boost::math::isnan(heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                }
                                else if((ptFirst->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                }
                                else if((ptFirst->height >= heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                {
                                    if(classification == SPD_ALL_CLASSES)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_VEGETATION) &
                                            ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                             (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                             (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if((classification == SPD_NOT_GROUND) &
                                            (ptFirst->classification != SPD_GROUND))
                                    {
                                        points->push_back(ptFirst);
                                    }
                                    else if(ptFirst->classification == classification)
                                    {
                                        points->push_back(ptFirst);
                                    }
                                }

                                // Last
                                if(ptLast != NULL)
                                {
                                    if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptLast);
                                        }
                                    }
                                    else if(boost::math::isnan(heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptLast);
                                        }
                                    }
                                    else if((ptLast->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptLast);
                                        }
                                    }
                                    else if((ptLast->height >= heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                 (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptLast->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptLast->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptLast);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptLast);
                                        }
                                    }
                                }
                            }
                            else if(boost::math::isnan(lowThreshold))
                            {
                                if(ptFirst->widthReturn <= upThreshold)
                                {
                                    if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                    }
                                    else if(boost::math::isnan(heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                    }
                                    else if((ptFirst->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                    }
                                    else if((ptFirst->height >= heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if(ptLast->widthReturn <= upThreshold)
                                    {
                                        if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast);
                                            }
                                        }
                                        else if(boost::math::isnan(heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast);
                                            }
                                        }
                                        else if((ptLast->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast);
                                            }
                                        }
                                        else if((ptLast->height >= heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast);
                                            }
                                        }
                                    }
                                }
                            }
                            else if(boost::math::isnan(upThreshold))
                            {
                                if(ptFirst->widthReturn >= lowThreshold)
                                {
                                    if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                    }
                                    else if(boost::math::isnan(heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                    }
                                    else if((ptFirst->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                    }
                                    else if((ptFirst->height >= heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if(ptLast->widthReturn >= lowThreshold)
                                    {
                                        if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast);
                                            }
                                        }
                                        else if(boost::math::isnan(heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast);
                                            }
                                        }
                                        else if((ptLast->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast);
                                            }
                                        }
                                        else if((ptLast->height >= heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast);
                                            }
                                        }
                                    }
                                }
                            }
                            else
                            {
                                //First
                                if((ptFirst->widthReturn >= lowThreshold) & (ptFirst->widthReturn <= upThreshold))
                                {
                                    if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                    }
                                    else if(boost::math::isnan(heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                    }
                                    else if((ptFirst->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                    }
                                    else if((ptFirst->height >= heightLowThreshold) & (ptFirst->height <= heightUpThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                ((ptFirst->classification == SPD_LOW_VEGETATION) |
                                                 (ptFirst->classification == SPD_MEDIUM_VEGETATION) |
                                                 (ptFirst->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                (ptFirst->classification != SPD_GROUND))
                                        {
                                            points->push_back(ptFirst);
                                        }
                                        else if(ptFirst->classification == classification)
                                        {
                                            points->push_back(ptFirst);
                                        }
                                    }
                                }

                                if(ptLast != NULL)
                                {
                                    if((ptLast->widthReturn >= lowThreshold) & (ptLast->widthReturn <= upThreshold))
                                    {
                                        if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast);
                                            }
                                        }
                                        else if(boost::math::isnan(heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast);
                                            }
                                        }
                                        else if((ptLast->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast);
                                            }
                                        }
                                        else if((ptLast->height >= heightLowThreshold) & (ptLast->height <= heightUpThreshold))
                                        {
                                            if(classification == SPD_ALL_CLASSES)
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_VEGETATION) &
                                                    ((ptLast->classification == SPD_LOW_VEGETATION) |
                                                     (ptLast->classification == SPD_MEDIUM_VEGETATION) |
                                                     (ptLast->classification == SPD_HIGH_VEGETATION)))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if((classification == SPD_NOT_GROUND) &
                                                    (ptLast->classification != SPD_GROUND))
                                            {
                                                points->push_back(ptLast);
                                            }
                                            else if(ptLast->classification == classification)
                                            {
                                                points->push_back(ptLast);
                                            }
                                        }
                                    }
                                }
                            }
                        }
                        else if(returnID == SPD_NOTFIRST_RETURNS)
                        {
                            std::vector<SPDPoint*>::iterator iterPts = (*iterPulses)->pts->begin();
                            ++iterPts;
                            for(; iterPts != (*iterPulses)->pts->end(); ++iterPts)
                            {
                                if(boost::math::isnan(heightUpThreshold) & boost::math::isnan(heightLowThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                }
                                else if(boost::math::isnan(heightLowThreshold) & ((*iterPts)->height <= heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                }
                                else if(((*iterPts)->height >= heightLowThreshold) & boost::math::isnan(heightUpThreshold))
                                {
                                    if(boost::math::isnan(upThreshold) & boost::math::isnan(lowThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(boost::math::isnan(lowThreshold) & ((*iterPts)->widthReturn <= upThreshold))
                                    {
                                        if(classification == SPD_ALL_CLASSES)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_VEGETATION) &
                                                (((*iterPts)->classification == SPD_LOW_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_MEDIUM_VEGETATION) |
                                                 ((*iterPts)->classification == SPD_HIGH_VEGETATION)))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((classification == SPD_NOT_GROUND) &
                                                ((*iterPts)->classification != SPD_GROUND))
                                        {
                                            points->push_back((*iterPts));
                                        }
                                        else if((*iterPts)->classification == classification)
                                        {
                                            points->push_back((*iterPts));
                                        }
                                    }
                                    else if(((*iterPts)->widthReturn >= lowThreshold) & boost::math::isnan(upThreshold))
                 