from typing import List, Optional, Set, Union

from . import JsonObject, JsonValidator, extract_json
from .elements import BlockElement, InteractiveElement
from .objects import MarkdownTextObject, PlainTextObject, TextObject


class Block(JsonObject):
    attributes = {"block_id"}

    block_id_max_length = 255

    def __init__(self, *, subtype: str, block_id: Optional[str] = None):
        self.subtype = subtype
        self.block_id = block_id
        self.color = None

    @JsonValidator(f"block_id cannot exceed {block_id_max_length} characters")
    def block_id_length(self):
        return self.block_id is None or len(self.block_id) <= self.block_id_max_length

    def to_dict(self) -> dict:
        json = super().to_dict()
        json["type"] = self.subtype
        return json


class DividerBlock(Block):
    def __init__(self):
        """
        A simple divider - equivalent to <hr>

        https://api.slack.com/reference/messaging/blocks#divider
        """
        super().__init__(subtype="divider")


class SectionBlock(Block):
    fields_max_length = 10

    def __init__(
        self,
        *,
        text: Union[str, TextObject] = None,
        fields: List[str] = None,
        block_id: Optional[str] = None,
        accessory: Optional[BlockElement] = None,
    ):
        """
        A general purpose block capable of holding text, fields (displayed in a
        semi-tabular format) and one 'accessory' element

        https://api.slack.com/reference/messaging/blocks#section

        Args:
            text: either a plain string, or a richer TextObject
            fields: optional: a sequence of strings that will be rendered using
                MarkdownTextObjects
            block_id: ID to be used for this block - autogenerated if left blank.
                Cannot exceed 255 characters.
            accessory: an optional BlockElement to attach to this SectionBlock as
                secondary content
        """
        super().__init__(subtype="section", block_id=block_id)
        self.text = text
        self.fields = fields or []
        self.accessory = accessory

    @JsonValidator("text or fields attribute must be specified")
    def text_or_fields_populated(self):
        return self.text is not None or self.fields

    @JsonValidator(f"fields attribute cannot exceed {fields_max_length} items")
    def fields_length(self):
        return self.fields is None or len(self.fields) <= self.fields_max_length

    def to_dict(self) -> dict:
        json = super().to_dict()
        if self.text is not None:
            if isinstance(self.text, TextObject):
                json["text"] = self.text.to_dict()
            else:
                json["text"] = MarkdownTextObject.direct_from_string(self.text)
        if self.fields:
            json["fields"] = [
                MarkdownTextObject.direct_from_string(field) for field in self.fields
            ]
        if self.accessory is not None:
            json["accessory"] = extract_json(self.accessory)
        return json


class ImageBlock(Block):
    @property
    def attributes(self) -> Set[str]:
        return super().attributes.union({"alt_text", "image_url"})

    image_url_max_length = 3000
    alt_text_max_length = 2000
    title_max_length = 2000

    def __init__(
        self,
        *,
        image_url: str,
        alt_text: str,
        title: Optional[str] = None,
        block_id: Optional[str] = None,
    ):
        """
        A simple image block, designed to make those cat photos really pop.

        https://api.slack.com/reference/messaging/blocks#image

        Args:
            image_url: Publicly hosted URL to be displayed. Cannot exceed 3000
                characters.
            alt_text: Plain text summary of image. Cannot exceed 2000 characters.
            title: A title to be displayed above the image. Cannot exceed 2000
                characters.
            block_id: ID to be used for this block - autogenerated if left blank.
                Cannot exceed 255 characters.
        """
        super().__init__(subtype="image", block_id=block_id)
        self.image_url = image_url
        self.alt_text = alt_text
        self.title = title

    @JsonValidator(
        f"image_url attribute cannot exceed {image_url_max_length} characters"
    )
    def image_url_length(self):
        return len(self.image_url) <= self.image_url_max_length

    @JsonValidator(f"alt_text attribute cannot exceed {alt_text_max_length} characters")
    def alt_text_length(self):
        return len(self.alt_text) <= self.alt_text_max_length

    @JsonValidator(f"title attribute cannot exceed {title_max_length} characters")
    def title_length(self):
        return self.title is None or len(self.title) <= self.title_max_length

    def to_dict(self) -> dict:
        json = super().to_dict()
        if self.title is not None:
            json["title"] = PlainTextObject.direct_from_string(self.title)
        return json


class ActionsBlock(Block):
    elements_max_length = 5

    def __init__(
        self, *, elements: List[InteractiveElement], block_id: Optional[str] = None
    ):
        """
        A block that is used to hold interactive elements.

        https://api.slack.com/reference/messaging/blocks#actions

        Args:
            elements: Up to 5 InteractiveElement objects - buttons, date pickers, etc
            block_id: ID to be used for this block - autogenerated if left blank.
                Cannot exceed 255 characters.
        """
        super().__init__(subtype="actions", block_id=block_id)
        self.elements = elements

    @JsonValidator(f"elements attribute cannot exceed {elements_max_length} elements")
    def elements_length(self):
        return len(self.elements) <= self.elements_max_length

    def to_dict(self) -> dict:
        json = super().to_dict()
        json["elements"] = extract_json(self.elements)
        return json


class ContextBlock(Block):
    elements_max_length = 10

    def __init__(
        self,
        *,
        elements: List[Union[ImageBlock, TextObject]],
        block_id: Optional[str] = None,
    ):
        """
        Displays message context, which can include both images and text.

        https://api.slack.com/reference/messaging/blocks#context

        Args:
            elements: Up to 10 ImageElements and TextObjects
            block_id: ID to be used for this block - autogenerated if left blank.
                Cannot exceed 255 characters.
        """
        super().__init__(subtype="context", block_id=block_id)
        self.elements = elements

    @JsonValidator(f"elements attribute cannot exceed {elements_max_length} elements")
    def elements_length(self):
        return len(self.elements) <= self.elements_max_length

    def to_dict(self) -> dict:
        json = super().to_dict()
        json["elements"] = extract_json(self.elements)
        return json
