"""Routines that compute Earth nutation."""
from numpy import array, cos, fmod, sin, outer, tensordot, zeros
from .constants import ASEC2RAD, ASEC360, DEG2RAD, tau, T0

def compute_nutation(t):
    """Generate the nutation rotations for Time `t`.

    If the Julian date is scalar, a simple ``(3, 3)`` matrix is
    returned; if the date is an array of length ``n``, then an array of
    matrices is returned with dimensions ``(3, 3, n)``.

    """
    oblm, oblt, eqeq, psi, eps = earth_tilt(t)

    cobm = cos(oblm * DEG2RAD)
    sobm = sin(oblm * DEG2RAD)
    cobt = cos(oblt * DEG2RAD)
    sobt = sin(oblt * DEG2RAD)
    cpsi = cos(psi * ASEC2RAD)
    spsi = sin(psi * ASEC2RAD)

    return array(((cpsi,
                  -spsi * cobm,
                  -spsi * sobm),
                  (spsi * cobt,
                   cpsi * cobm * cobt + sobm * sobt,
                   cpsi * sobm * cobt - cobm * sobt),
                  (spsi * sobt,
                   cpsi * cobm * sobt - sobm * cobt,
                   cpsi * sobm * sobt + cobm * cobt)))

def earth_tilt(t):
    """Return a tuple of information about the earth's axis and position.

    `t` - A Time object.

    The returned tuple contains five items:

    ``mean_ob`` - Mean obliquity of the ecliptic in degrees.
    ``true_ob`` - True obliquity of the ecliptic in degrees.
    ``eq_eq`` - Equation of the equinoxes in seconds of time.
    ``d_psi`` - Nutation in longitude in arcseconds.
    ``d_eps`` - Nutation in obliquity in arcseconds.

    """
    dp, de = iau2000a(t.tt)
    c_terms = equation_of_the_equinoxes_complimentary_terms(t.tt) / ASEC2RAD

    d_psi = dp * 1e-7 + t.psi_correction
    d_eps = de * 1e-7 + t.eps_correction

    mean_ob = mean_obliquity(t.tdb)
    true_ob = mean_ob + d_eps

    mean_ob /= 3600.0
    true_ob /= 3600.0

    eq_eq = d_psi * cos(mean_ob * DEG2RAD) + c_terms
    eq_eq /= 15.0

    return mean_ob, true_ob, eq_eq, d_psi, d_eps

#

def mean_obliquity(jd_tdb):
    """Return the mean obliquity of the ecliptic in arcseconds.

    `jd_tt` - TDB time as a Julian date float, or NumPy array of floats

    """
    # Compute time in Julian centuries from epoch J2000.0.

    t = (jd_tdb - T0) / 36525.0

    # Compute the mean obliquity in arcseconds.  Use expression from the
    # reference's eq. (39) with obliquity at J2000.0 taken from eq. (37)
    # or Table 8.

    epsilon = (((( -  0.0000000434   * t
                   -  0.000000576  ) * t
                   +  0.00200340   ) * t
                   -  0.0001831    ) * t
                   - 46.836769     ) * t + 84381.406

    return epsilon

def equation_of_the_equinoxes_complimentary_terms(jd_tt):
    """Compute the complementary terms of the equation of the equinoxes.

    `jd_tt` - Terrestrial Time: Julian date float, or NumPy array of floats

    """
    # Interval between fundamental epoch J2000.0 and current date.

    t = (jd_tt - T0) / 36525.0

    # Build array for intermediate results.

    shape = getattr(jd_tt, 'shape', ())
    fa = zeros((14,) if shape == () else (14, shape[0]))

    # Mean Anomaly of the Moon.

    fa[0] = ((485868.249036 +
              (715923.2178 +
              (    31.8792 +
              (     0.051635 +
              (    -0.00024470)
              * t) * t) * t) * t) * ASEC2RAD
              + (1325.0*t % 1.0) * tau)

    # Mean Anomaly of the Sun.

    fa[1] = ((1287104.793048 +
              (1292581.0481 +
              (     -0.5532 +
              (     +0.000136 +
              (     -0.00001149)
              * t) * t) * t) * t) * ASEC2RAD
              + (99.0*t % 1.0) * tau)

    # Mean Longitude of the Moon minus Mean Longitude of the Ascending
    # Node of the Moon.

    fa[2] = (( 335779.526232 +
              ( 295262.8478 +
              (    -12.7512 +
              (     -0.001037 +
              (      0.00000417)
              * t) * t) * t) * t) * ASEC2RAD
              + (1342.0*t % 1.0) * tau)

    # Mean Elongation of the Moon from the Sun.

    fa[3] = ((1072260.703692 +
              (1105601.2090 +
              (     -6.3706 +
              (      0.006593 +
              (     -0.00003169)
              * t) * t) * t) * t) * ASEC2RAD
              + (1236.0*t % 1.0) * tau)

    # Mean Longitude of the Ascending Node of the Moon.

    fa[4] = (( 450160.398036 +
              (-482890.5431 +
              (      7.4722 +
              (      0.007702 +
              (     -0.00005939)
              * t) * t) * t) * t) * ASEC2RAD
              + (-5.0*t % 1.0) * tau)

    fa[ 5] = (4.402608842 + 2608.7903141574 * t)
    fa[ 6] = (3.176146697 + 1021.3285546211 * t)
    fa[ 7] = (1.753470314 +  628.3075849991 * t)
    fa[ 8] = (6.203480913 +  334.0612426700 * t)
    fa[ 9] = (0.599546497 +   52.9690962641 * t)
    fa[10] = (0.874016757 +   21.3299104960 * t)
    fa[11] = (5.481293872 +    7.4781598567 * t)
    fa[12] = (5.311886287 +    3.8133035638 * t)
    fa[13] = (0.024381750 +    0.00000538691 * t) * t

    fa %= tau

    # Evaluate the complementary terms.

    a = ke0_t.dot(fa)
    s0 = se0_t_0.dot(sin(a)) + se0_t_1.dot(cos(a))

    a = ke1.dot(fa)
    s1 = se1_0 * sin(a) + se1_1 * cos(a)

    c_terms = s0 + s1 * t
    c_terms *= ASEC2RAD
    return c_terms

anomaly_constant, anomaly_coefficient = array([

    # Mean anomaly of the Moon.
    (2.35555598, 8328.6914269554),

    # Mean anomaly of the Sun.
    (6.24006013, 628.301955),

    # Mean argument of the latitude of the Moon.
    (1.627905234, 8433.466158131),

    # Mean elongation of the Moon from the Sun.
    (5.198466741, 7771.3771468121),

    # Mean longitude of the ascending node of the Moon.
    (2.18243920, - 33.757045),

    # Planetary longitudes, Mercury through Neptune (Souchay et al. 1999).
    (4.402608842, 2608.7903141574),
    (3.176146697, 1021.3285546211),
    (1.753470314,  628.3075849991),
    (6.203480913,  334.0612426700),
    (0.599546497,   52.9690962641),
    (0.874016757,   21.3299104960),
    (5.481293871,    7.4781598567),
    (5.321159000,    3.8127774000),

    # General accumulated precession in longitude (gets multiplied by t).
    (0.02438175, 0.00000538691),
    ]).T

def iau2000a(jd_tt):
    """Compute Earth nutation based on the IAU 2000A nutation model.

    `jd_tt` - Terrestrial Time: Julian date float, or NumPy array of floats

    Returns a tuple ``(delta_psi, delta_epsilon)`` measured in tenths of
    a micro-arcsecond.  Each value is either a float, or a NumPy array
    with the same dimensions as the input argument.

    """
    # Interval between fundamental epoch J2000.0 and given date.

    t = (jd_tt - T0) / 36525.0

    # Compute fundamental arguments from Simon et al. (1994), in radians.

    a = fundamental_arguments(t)

    # ** Luni-solar nutation **
    # Summation of luni-solar nutation series (in reverse order).

    arg = nals_t.dot(a)
    fmod(arg, tau, out=arg)

    sarg = sin(arg)
    carg = cos(arg)

    stsc = array((sarg, t * sarg, carg)).T
    ctcs = array((carg, t * carg, sarg)).T

    dpsi = tensordot(stsc, lunisolar_longitude_coefficients)
    deps = tensordot(ctcs, lunisolar_obliquity_coefficients)

    # Compute and add in planetary components.

    if getattr(t, 'shape', ()) == ():
        a = t * anomaly_coefficient + anomaly_constant
    else:
        a = (outer(anomaly_coefficient, t).T + anomaly_constant).T
    a[-1] *= t

    fmod(a, tau, out=a)
    arg = napl_t.dot(a)
    fmod(arg, tau, out=arg)
    sc = array((sin(arg), cos(arg))).T

    dpsi += tensordot(sc, nutation_coefficients_longitude)
    deps += tensordot(sc, nutation_coefficients_obliquity)

    return dpsi, deps

#

fa0, fa1, fa2, fa3, fa4 = array([

    # Mean Anomaly of the Moon.
    (485868.249036, 1717915923.2178, 31.8792, 0.051635, - .00024470),

    # Mean Anomaly of the Sun.
    (1287104.79305,  129596581.0481, - 0.5532, 0.000136, - 0.00001149),

    # Mean Longitude of the Moon minus Mean Longitude of the Ascending
    # Node of the Moon.
    (335779.526232, 1739527262.8478, - 12.7512, -  0.001037, 0.00000417),

    # Mean Elongation of the Moon from the Sun.
    (1072260.70369, 1602961601.2090, - 6.3706, 0.006593, - 0.00003169),

    # Mean Longitude of the Ascending Node of the Moon.
    (450160.398036, - 6962890.5431, 7.4722, 0.007702, - 0.00005939),

    ]).T[:,:,None]

def fundamental_arguments(t):
    """Compute the fundamental arguments (mean elements) of Sun and Moon.

    `t` - TDB time in Julian centuries since J2000.0, as float or NumPy array

    Outputs fundamental arguments, in radians:
          a[0] = l (mean anomaly of the Moon)
          a[1] = l' (mean anomaly of the Sun)
          a[2] = F (mean argument of the latitude of the Moon)
          a[3] = D (mean elongation of the Moon from the Sun)
          a[4] = Omega (mean longitude of the Moon's ascending node);
                 from Simon section 3.4(b.3),
                 precession = 5028.8200 arcsec/cy)

    """
    a = fa4 * t
    a += fa3
    a *= t
    a += fa2
    a *= t
    a += fa1
    a *= t
    a += fa0
    fmod(a, ASEC360, out=a)
    a *= ASEC2RAD
    if getattr(t, 'shape', ()):
        return a
    return a[:,0]

# Argument coefficients for t^0.

ke0_t = array([
      (0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (0,  0,  2, -2,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (0,  0,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (0,  0,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (0,  0,  2,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (0,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (0,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (1,  0,  0,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (0,  1,  2, -2,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (0,  1,  2, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (0,  0,  4, -4,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (0,  0,  1, -1,  1,  0, -8, 12,  0,  0,  0,  0,  0,  0),
      (0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (0,  0,  2,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (1,  0,  2,  0,  3,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (1,  0,  2,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (0,  0,  2, -2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (0,  1, -2,  2, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (0,  1, -2,  2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0, -1),
      (0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (2,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (1,  0,  0, -2,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (0,  1,  2, -2,  2,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (1,  0,  0, -2, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (0,  0,  4, -2,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (0,  0,  2, -2,  4,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (1,  0, -2,  0, -3,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      (1,  0, -2,  0, -1,  0,  0,  0,  0,  0,  0,  0,  0,  0),
      ])

# Argument coefficients for t^1.

ke1 = array([0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0,  0,  0])

# Sine and cosine coefficients for t^0.

se0_t = array([
      (+2640.96e-6,          -0.39e-6),
      (  +63.52e-6,          -0.02e-6),
      (  +11.75e-6,          +0.01e-6),
      (  +11.21e-6,          +0.01e-6),
      (   -4.55e-6,          +0.00e-6),
      (   +2.02e-6,          +0.00e-6),
      (   +1.98e-6,          +0.00e-6),
      (   -1.72e-6,          +0.00e-6),
      (   -1.41e-6,          -0.01e-6),
      (   -1.26e-6,          -0.01e-6),
      (   -0.63e-6,          +0.00e-6),
      (   -0.63e-6,          +0.00e-6),
      (   +0.46e-6,          +0.00e-6),
      (   +0.45e-6,          +0.00e-6),
      (   +0.36e-6,          +0.00e-6),
      (   -0.24e-6,          -0.12e-6),
      (   +0.32e-6,          +0.00e-6),
      (   +0.28e-6,          +0.00e-6),
      (   +0.27e-6,          +0.00e-6),
      (   +0.26e-6,          +0.00e-6),
      (   -0.21e-6,          +0.00e-6),
      (   +0.19e-6,          +0.00e-6),
      (   +0.18e-6,          +0.00e-6),
      (   -0.10e-6,          +0.05e-6),
      (   +0.15e-6,          +0.00e-6),
      (   -0.14e-6,          +0.00e-6),
      (   +0.14e-6,          +0.00e-6),
      (   -0.14e-6,          +0.00e-6),
      (   +0.14e-6,          +0.00e-6),
      (   +0.13e-6,          +0.00e-6),
      (   -0.11e-6,          +0.00e-6),
      (   +0.11e-6,          +0.00e-6),
      (   +0.11e-6,          +0.00e-6),
      ])

se0_t_0, se0_t_1 = se0_t.T

# Sine and cosine coefficients for t^1.

se1 = (   -0.87e-6,          +0.00e-6)
se1_0 = se1[0]
se1_1 = se1[1]

# Luni-Solar argument multipliers:
#      L     L'    F     D     Om

nals_t = array((
      ( 0,    0,    0,    0,    1),
      ( 0,    0,    2,   -2,    2),
      ( 0,    0,    2,    0,    2),
      ( 0,    0,    0,    0,    2),
      ( 0,    1,    0,    0,    0),
      ( 0,    1,    2,   -2,    2),
      ( 1,    0,    0,    0,    0),
      ( 0,    0,    2,    0,    1),
      ( 1,    0,    2,    0,    2),
      ( 0,   -1,    2,   -2,    2),
      ( 0,    0,    2,   -2,    1),
      (-1,    0,    2,    0,    2),
      (-1,    0,    0,    2,    0),
      ( 1,    0,    0,    0,    1),
      (-1,    0,    0,    0,    1),
      (-1,    0,    2,    2,    2),
      ( 1,    0,    2,    0,    1),
      (-2,    0,    2,    0,    1),
      ( 0,    0,    0,    2,    0),
      ( 0,    0,    2,    2,    2),
      ( 0,   -2,    2,   -2,    2),
      (-2,    0,    0,    2,    0),
      ( 2,    0,    2,    0,    2),
      ( 1,    0,    2,   -2,    2),
      (-1,    0,    2,    0,    1),
      ( 2,    0,    0,    0,    0),
      ( 0,    0,    2,    0,    0),
      ( 0,    1,    0,    0,    1),
      (-1,    0,    0,    2,    1),
      ( 0,    2,    2,   -2,    2),
      ( 0,    0,   -2,    2,    0),
      ( 1,    0,    0,   -2,    1),
      ( 0,   -1,    0,    0,    1),
      (-1,    0,    2,    2,    1),
      ( 0,    2,    0,    0,    0),
      ( 1,    0,    2,    2,    2),
      (-2,    0,    2,    0,    0),
      ( 0,    1,    2,    0,    2),
      ( 0,    0,    2,    2,    1),
      ( 0,   -1,    2,    0,    2),
      ( 0,    0,    0,    2,    1),
      ( 1,    0,    2,   -2,    1),
      ( 2,    0,    2,   -2,    2),
      (-2,    0,    0,    2,    1),
      ( 2,    0,    2,    0,    1),
      ( 0,   -1,    2,   -2,    1),
      ( 0,    0,    0,   -2,    1),
      (-1,   -1,    0,    2,    0),
      ( 2,    0,    0,   -2,    1),
      ( 1,    0,    0,    2,    0),
      ( 0,    1,    2,   -2,    1),
      ( 1,   -1,    0,    0,    0),
      (-2,    0,    2,    0,    2),
      ( 3,    0,    2,    0,    2),
      ( 0,   -1,    0,    2,    0),
      ( 1,   -1,    2,    0,    2),
      ( 0,    0,    0,    1,    0),
      (-1,   -1,    2,    2,    2),
      (-1,    0,    2,    0,    0),
      ( 0,   -1,    2,    2,    2),
      (-2,    0,    0,    0,    1),
      ( 1,    1,    2,    0,    2),
      ( 2,    0,    0,    0,    1),
      (-1,    1,    0,    1,    0),
      ( 1,    1,    0,    0,    0),
      ( 1,    0,    2,    0,    0),
      (-1,    0,    2,   -2,    1),
      ( 1,    0,    0,    0,    2),
      (-1,    0,    0,    1,    0),
      ( 0,    0,    2,    1,    2),
      (-1,    0,    2,    4,    2),
      (-1,    1,    0,    1,    1),
      ( 0,   -2,    2,   -2,    1),
      ( 1,    0,    2,    2,    1),
      (-2,    0,    2,    2,    2),
      (-1,    0,    0,    0,    2),
      ( 1,    1,    2,   -2,    2),
      (-2,    0,    2,    4,    2),
      (-1,    0,    4,    0,    2),
      ( 2,    0,    2,   -2,    1),
      ( 2,    0,    2,    2,    2),
      ( 1,    0,    0,    2,    1),
      ( 3,    0,    0,    0,    0),
      ( 3,    0,    2,   -2,    2),
      ( 0,    0,    4,   -2,    2),
      ( 0,    1,    2,    0,    1),
      ( 0,    0,   -2,    2,    1),
      ( 0,    0,    2,   -2,    3),
      (-1,    0,    0,    4,    0),
      ( 2,    0,   -2,    0,    1),
      (-2,    0,    0,    4,    0),
      (-1,   -1,    0,    2,    1),
      (-1,    0,    0,    1,    1),
      ( 0,    1,    0,    0,    2),
      ( 0,    0,   -2,    0,    1),
      ( 0,   -1,    2,    0,    1),
      ( 0,    0,    2,   -1,    2),
      ( 0,    0,    2,    4,    2),
      (-2,   -1,    0,    2,    0),
      ( 1,    1,    0,   -2,    1),
      (-1,    1,    0,    2,    0),
      (-1,    1,    0,    1,    2),
      ( 1,   -1,    0,    0,    1),
      ( 1,   -1,    2,    2,    2),
      (-1,    1,    2,    2,    2),
      ( 3,    0,    2,    0,    1),
      ( 0,    1,   -2,    2,    0),
      (-1,    0,    0,   -2,    1),
      ( 0,    1,    2,    2,    2),
      (-1,   -1,    2,    2,    1),
      ( 0,   -1,    0,    0,    2),
      ( 1,    0,    2,   -4,    1),
      (-1,    0,   -2,    2,    0),
      ( 0,   -1,    2,    2,    1),
      ( 2,   -1,    2,    0,    2),
      ( 0,    0,    0,    2,    2),
      ( 1,   -1,    2,    0,    1),
      (-1,    1,    2,    0,    2),
      ( 0,    1,    0,    2,    0),
      ( 0,   -1,   -2,    2,    0),
      ( 0,    3,    2,   -2,    2),
      ( 0,    0,    0,    1,    1),
      (-1,    0,    2,    2,    0),
      ( 2,    1,    2,    0,    2),
      ( 1,    1,    0,    0,    1),
      ( 1,    1,    2,    0,    1),
      ( 2,    0,    0,    2,    0),
      ( 1,    0,   -2,    2,    0),
      (-1,    0,    0,    2,    2),
      ( 0,    1,    0,    1,    0),
      ( 0,    1,    0,   -2,    1),
      (-1,    0,    2,   -2,    2),
      ( 0,    0,    0,   -1,    1),
      (-1,    1,    0,    0,    1),
      ( 1,    0,    2,   -1,    2),
      ( 1,   -1,    0,    2,    0),
      ( 0,    0,    0,    4,    0),
      ( 1,    0,    2,    1,    2),
      ( 0,    0,    2,    1,    1),
      ( 1,    0,    0,   -2,    2),
      (-1,    0,    2,    4,    1),
      ( 1,    0,   -2,    0,    1),
      ( 1,    1,    2,   -2,    1),
      ( 0,    0,    2,    2,    0),
      (-1,    0,    2,   -1,    1),
      (-2,    0,    2,    2,    1),
      ( 4,    0,    2,    0,    2),
      ( 2,   -1,    0,    0,    0),
      ( 2,    1,    2,   -2,    2),
      ( 0,    1,    2,    1,    2),
      ( 1,    0,    4,   -2,    2),
      (-1,   -1,    0,    0,    1),
      ( 0,    1,    0,    2,    1),
      (-2,    0,    2,    4,    1),
      ( 2,    0,    2,    0,    0),
      ( 1,    0,    0,    1,    0),
      (-1,    0,    0,    4,    1),
      (-1,    0,    4,    0,    1),
      ( 2,    0,    2,    2,    1),
      ( 0,    0,    2,   -3,    2),
      (-1,   -2,    0,    2,    0),
      ( 2,    1,    0,    0,    0),
      ( 0,    0,    4,    0,    2),
      ( 0,    0,    0,    0,    3),
      ( 0,    3,    0,    0,    0),
      ( 0,    0,    2,   -4,    1),
      ( 0,   -1,    0,    2,    1),
      ( 0,    0,    0,    4,    1),
      (-1,   -1,    2,    4,    2),
      ( 1,    0,    2,    4,    2),
      (-2,    2,    0,    2,    0),
      (-2,   -1,    2,    0,    1),
      (-2,    0,    0,    2,    2),
      (-1,   -1,    2,    0,    2),
      ( 0,    0,    4,   -2,    1),
      ( 3,    0,    2,   -2,    1),
      (-2,   -1,    0,    2,    1),
      ( 1,    0,    0,   -1,    1),
      ( 0,   -2,    0,    2,    0),
      (-2,    0,    0,    4,    1),
      (-3,    0,    0,    0,    1),
      ( 1,    1,    2,    2,    2),
      ( 0,    0,    2,    4,    1),
      ( 3,    0,    2,    2,    2),
      (-1,    1,    2,   -2,    1),
      ( 2,    0,    0,   -4,    1),
      ( 0,    0,    0,   -2,    2),
      ( 2,    0,    2,   -4,    1),
      (-1,    1,    0,    2,    1),
      ( 0,    0,    2,   -1,    1),
      ( 0,   -2,    2,    2,    2),
      ( 2,    0,    0,    2,    1),
      ( 4,    0,    2,   -2,    2),
      ( 2,    0,    0,   -2,    2),
      ( 0,    2,    0,    0,    1),
      ( 1,    0,    0,   -4,    1),
      ( 0,    2,    2,   -2,    1),
      (-3,    0,    0,    4,    0),
      (-1,    1,    2,    0,    1),
      (-1,   -1,    0,    4,    0),
      (-1,   -2,    2,    2,    2),
      (-2,   -1,    2,    4,    2),
      ( 1,   -1,    2,    2,    1),
      (-2,    1,    0,    2,    0),
      (-2,    1,    2,    0,    1),
      ( 2,    1,    0,   -2,    1),
      (-3,    0,    2,    0,    1),
      (-2,    0,    2,   -2,    1),
      (-1,    1,    0,    2,    2),
      ( 0,   -1,    2,   -1,    2),
      (-1,    0,    4,   -2,    2),
      ( 0,   -2,    2,    0,    2),
      (-1,    0,    2,    1,    2),
      ( 2,    0,    0,    0,    2),
      ( 0,    0,    2,    0,    3),
      (-2,    0,    4,    0,    2),
      (-1,    0,   -2,    0,    1),
      (-1,    1,    2,    2,    1),
      ( 3,    0,    0,    0,    1),
      (-1,    0,    2,    3,    2),
      ( 2,   -1,    2,    0,    1),
      ( 0,    1,    2,    2,    1),
      ( 0,   -1,    2,    4,    2),
      ( 2,   -1,    2,    2,    2),
      ( 0,    2,   -2,    2,    0),
      (-1,   -1,    2,   -1,    1),
      ( 0,   -2,    0,    0,    1),
      ( 1,    0,    2,   -4,    2),
      ( 1,   -1,    0,   -2,    1),
      (-1,   -1,    2,    0,    1),
      ( 1,   -1,    2,   -2,    2),
      (-2,   -1,    0,    4,    0),
      (-1,    0,    0,    3,    0),
      (-2,   -1,    2,    2,    2),
      ( 0,    2,    2,    0,    2),
      ( 1,    1,    0,    2,    0),
      ( 2,    0,    2,   -1,    2),
      ( 1,    0,    2,    1,    1),
      ( 4,    0,    0,    0,    0),
      ( 2,    1,    2,    0,    1),
      ( 3,   -1,    2,    0,    2),
      (-2,    2,    0,    2,    1),
      ( 1,    0,    2,   -3,    1),
      ( 1,    1,    2,   -4,    1),
      (-1,   -1,    2,   -2,    1),
      ( 0,   -1,    0,   -1,    1),
      ( 0,   -1,    0,   -2,    1),
      (-2,    0,    0,    0,    2),
      (-2,    0,   -2,    2,    0),
      (-1,    0,   -2,    4,    0),
      ( 1,   -2,    0,    0,    0),
      ( 0,    1,    0,    1,    1),
      (-1,    2,    0,    2,    0),
      ( 1,   -1,    2,   -2,    1),
      ( 1,    2,    2,   -2,    2),
      ( 2,   -1,    2,   -2,    2),
      ( 1,    0,    2,   -1,    1),
      ( 2,    1,    2,   -2,    1),
      (-2,    0,    0,   -2,    1),
      ( 1,   -2,    2,    0,    2),
      ( 0,    1,    2,    1,    1),
      ( 1,    0,    4,   -2,    1),
      (-2,    0,    4,    2,    2),
      ( 1,    1,    2,    1,    2),
      ( 1,    0,    0,    4,    0),
      ( 1,    0,    2,    2,    0),
      ( 2,    0,    2,    1,    2),
      ( 3,    1,    2,    0,    2),
      ( 4,    0,    2,    0,    1),
      (-2,   -1,    2,    0,    0),
      ( 0,    1,   -2,    2,    1),
      ( 1,    0,   -2,    1,    0),
      ( 0,   -1,   -2,    2,    1),
      ( 2,   -1,    0,   -2,    1),
      (-1,    0,    2,   -1,    2),
      ( 1,    0,    2,   -3,    2),
      ( 0,    1,    2,   -2,    3),
      ( 0,    0,    2,   -3,    1),
      (-1,    0,   -2,    2,    1),
      ( 0,    0,    2,   -4,    2),
      (-2,    1,    0,    0,    1),
      (-1,    0,    0,   -1,    1),
      ( 2,    0,    2,   -4,    2),
      ( 0,    0,    4,   -4,    4),
      ( 0,    0,    4,   -4,    2),
      (-1,   -2,    0,    2,    1),
      (-2,    0,    0,    3,    0),
      ( 1,    0,   -2,    2,    1),
      (-3,    0,    2,    2,    2),
      (-3,    0,    2,    2,    1),
      (-2,    0,    2,    2,    0),
      ( 2,   -1,    0,    0,    1),
      (-2,    1,    2,    2,    2),
      ( 1,    1,    0,    1,    0),
      ( 0,    1,    4,   -2,    2),
      (-1,    1,    0,   -2,    1),
      ( 0,    0,    0,   -4,    1),
      ( 1,   -1,    0,    2,    1),
      ( 1,    1,    0,    2,    1),
      (-1,    2,    2,    2,    2),
      ( 3,    1,    2,   -2,    2),
      ( 0,   -1,    0,    4,    0),
      ( 2,   -1,    0,    2,    0),
      ( 0,    0,    4,    0,    1),
      ( 2,    0,    4,   -2,    2),
      (-1,   -1,    2,    4,    1),
      ( 1,    0,    0,    4,    1),
      ( 1,   -2,    2,    2,    2),
      ( 0,    0,    2,    3,    2),
      (-1,    1,    2,    4,    2),
      ( 3,    0,    0,    2,    0),
      (-1,    0,    4,    2,    2),
      ( 1,    1,    2,    2,    1),
      (-2,    0,    2,    6,    2),
      ( 2,    1,    2,    2,    2),
      (-1,    0,    2,    6,    2),
      ( 1,    0,    2,    4,    1),
      ( 2,    0,    2,    4,    2),
      ( 1,    1,   -2,    1,    0),
      (-3,    1,    2,    1,    2),
      ( 2,    0,   -2,    0,    2),
      (-1,    0,    0,    1,    2),
      (-4,    0,    2,    2,    1),
      (-1,   -1,    0,    1,    0),
      ( 0,    0,   -2,    2,    2),
      ( 1,    0,    0,   -1,    2),
      ( 0,   -1,    2,   -2,    3),
      (-2,    1,    2,    0,    0),
      ( 0,    0,    2,   -2,    4),
      (-2,   -2,    0,    2,    0),
      (-2,    0,   -2,    4,    0),
      ( 0,   -2,   -2,    2,    0),
      ( 1,    2,    0,   -2,    1),
      ( 3,    0,    0,   -4,    1),
      (-1,    1,    2,   -2,    2),
      ( 1,   -1,    2,   -4,    1),
      ( 1,    1,    0,   -2,    2),
      (-3,    0,    2,    0,    0),
      (-3,    0,    2,    0,    2),
      (-2,    0,    0,    1,    0),
      ( 0,    0,   -2,    1,    0),
      (-3,    0,    0,    2,    1),
      (-1,   -1,   -2,    2,    0),
      ( 0,    1,    2,   -4,    1),
      ( 2,    1,    0,   -4,    1),
      ( 0,    2,    0,   -2,    1),
      ( 1,    0,    0,   -3,    1),
      (-2,    0,    2,   -2,    2),
      (-2,   -1,    0,    0,    1),
      (-4,    0,    0,    2,    0),
      ( 1,    1,    0,   -4,    1),
      (-1,    0,    2,   -4,    1),
      ( 0,    0,    4,   -4,    1),
      ( 0,    3,    2,   -2,    2),
      (-3,   -1,    0,    4,    0),
      (-3,    0,    0,    4,    1),
      ( 1,   -1,   -2,    2,    0),
      (-1,   -1,    0,    2,    2),
      ( 1,   -2,    0,    0,    1),
      ( 1,   -1,    0,    0,    2),
      ( 0,    0,    0,    1,    2),
      (-1,   -1,    2,    0,    0),
      ( 1,   -2,    2,   -2,    2),
      ( 0,   -1,    2,   -1,    1),
      (-1,    0,    2,    0,    3),
      ( 1,    1,    0,    0,    2),
      (-1,    1,    2,    0,    0),
      ( 1,    2,    0,    0,    0),
      (-1,    2,    2,    0,    2),
      (-1,    0,    4,   -2,    1),
      ( 3,    0,    2,   -4,    2),
      ( 1,    2,    2,   -2,    1),
      ( 1,    0,    4,   -4,    2),
      (-2,   -1,    0,    4,    1),
      ( 0,   -1,    0,    2,    2),
      (-2,    1,    0,    4,    0),
      (-2,   -1,    2,    2,    1),
      ( 2,    0,   -2,    2,    0),
      ( 1,    0,    0,    1,    1),
      ( 0,    1,    0,    2,    2),
      ( 1,   -1,    2,   -1,    2),
      (-2,    0,    4,    0,    1),
      ( 2,    1,    0,    0,    1),
      ( 0,    1,    2,    0,    0),
      ( 0,   -1,    4,   -2,    2),
      ( 0,    0,    4,   -2,    4),
      ( 0,    2,    2,    0,    1),
      (-3,    0,    0,    6,    0),
      (-1,   -1,    0,    4,    1),
      ( 1,   -2,    0,    2,    0),
      (-1,    0,    0,    4,    2),
      (-1,   -2,    2,    2,    1),
      (-1,    0,    0,   -2,    2),
      ( 1,    0,   -2,   -2,    1),
      ( 0,    0,   -2,   -2,    1),
      (-2,    0,   -2,    0,    1),
      ( 0,    0,    0,    3,    1),
      ( 0,    0,    0,    3,    0),
      (-1,    1,    0,    4,    0),
      (-1,   -1,    2,    2,    0),
      (-2,    0,    2,    3,    2),
      ( 1,    0,    0,    2,    2),
      ( 0,   -1,    2,    1,    2),
      ( 3,   -1,    0,    0,    0),
      ( 2,    0,    0,    1,    0),
      ( 1,   -1,    2,    0,    0),
      ( 0,    0,    2,    1,    0),
      ( 1,    0,    2,    0,    3),
      ( 3,    1,    0,    0,    0),
      ( 3,   -1,    2,   -2,    2),
      ( 2,    0,    2,   -1,    1),
      ( 1,    1,    2,    0,    0),
      ( 0,    0,    4,   -1,    2),
      ( 1,    2,    2,    0,    2),
      (-2,    0,    0,    6,    0),
      ( 0,   -1,    0,    4,    1),
      (-2,   -1,    2,    4,    1),
      ( 0,   -2,    2,    2,    1),
      ( 0,   -1,    2,    2,    0),
      (-1,    0,    2,    3,    1),
      (-2,    1,    2,    4,    2),
      ( 2,    0,    0,    2,    2),
      ( 2,   -2,    2,    0,    2),
      (-1,    1,    2,    3,    2),
      ( 3,    0,    2,   -1,    2),
      ( 4,    0,    2,   -2,    1),
      (-1,    0,    0,    6,    0),
      (-1,   -2,    2,    4,    2),
      (-3,    0,    2,    6,    2),
      (-1,    0,    2,    4,    0),
      ( 3,    0,    0,    2,    1),
      ( 3,   -1,    2,    0,    1),
      ( 3,    0,    2,    0,    0),
      ( 1,    0,    4,    0,    2),
      ( 5,    0,    2,   -2,    2),
      ( 0,   -1,    2,    4,    1),
      ( 2,   -1,    2,    2,    1),
      ( 0,    1,    2,    4,    2),
      ( 1,   -1,    2,    4,    2),
      ( 3,   -1,    2,    2,    2),
      ( 3,    0,    2,    2,    1),
      ( 5,    0,    2,    0,    2),
      ( 0,    0,    2,    6,    2),
      ( 4,    0,    2,    2,    2),
      ( 0,   -1,    1,   -1,    1),
      (-1,    0,    1,    0,    3),
      ( 0,   -2,    2,   -2,    3),
      ( 1,    0,   -1,    0,    1),
      ( 2,   -2,    0,   -2,    1),
      (-1,    0,    1,    0,    2),
      (-1,    0,    1,    0,    1),
      (-1,   -1,    2,   -1,    2),
      (-2,    2,    0,    2,    2),
      (-1,    0,    1,    0,    0),
      (-4,    1,    2,    2,    2),
      (-3,    0,    2,    1,    1),
      (-2,   -1,    2,    0,    2),
      ( 1,    0,   -2,    1,    1),
      ( 2,   -1,   -2,    0,    1),
      (-4,    0,    2,    2,    0),
      (-3,    1,    0,    3,    0),
      (-1,    0,   -1,    2,    0),
      ( 0,   -2,    0,    0,    2),
      ( 0,   -2,    0,    0,    2),
      (-3,    0,    0,    3,    0),
      (-2,   -1,    0,    2,    2),
      (-1,    0,   -2,    3,    0),
      (-4,    0,    0,    4,    0),
      ( 2,    1,   -2,    0,    1),
      ( 2,   -1,    0,   -2,    2),
      ( 0,    0,    1,   -1,    0),
      (-1,    2,    0,    1,    0),
      (-2,    1,    2,    0,    2),
      ( 1,    1,    0,   -1,    1),
      ( 1,    0,    1,   -2,    1),
      ( 0,    2,    0,    0,    2),
      ( 1,   -1,    2,   -3,    1),
      (-1,    1,    2,   -1,    1),
      (-2,    0,    4,   -2,    2),
      (-2,    0,    4,   -2,    1),
      (-2,   -2,    0,    2,    1),
      (-2,    0,   -2,    4,    0),
      ( 1,    2,    2,   -4,    1),
      ( 1,    1,    2,   -4,    2),
      (-1,    2,    2,   -2,    1),
      ( 2,    0,    0,   -3,    1),
      (-1,    2,    0,    0,    1),
      ( 0,    0,    0,   -2,    0),
      (-1,   -1,    2,   -2,    2),
      (-1,    1,    0,    0,    2),
      ( 0,    0,    0,   -1,    2),
      (-2,    1,    0,    1,    0),
      ( 1,   -2,    0,   -2,    1),
      ( 1,    0,   -2,    0,    2),
      (-3,    1,    0,    2,    0),
      (-1,    1,   -2,    2,    0),
      (-1,   -1,    0,    0,    2),
      (-3,    0,    0,    2,    0),
      (-3,   -1,    0,    2,    0),
      ( 2,    0,    2,   -6,    1),
      ( 0,    1,    2,   -4,    2),
      ( 2,    0,    0,   -4,    2),
      (-2,    1,    2,   -2,    1),
      ( 0,   -1,    2,   -4,    1),
      ( 0,    1,    0,   -2,    2),
      (-1,    0,    0,   -2,    0),
      ( 2,    0,   -2,   -2,    1),
      (-4,    0,    2,    0,    1),
      (-1,   -1,    0,   -1,    1),
      ( 0,    0,   -2,    0,    2),
      (-3,    0,    0,    1,    0),
      (-1,    0,   -2,    1,    0),
      (-2,    0,   -2,    2,    1),
      ( 0,    0,   -4,    2,    0),
      (-2,   -1,   -2,    2,    0),
      ( 1,    0,    2,   -6,    1),
      (-1,    0,    2,   -4,    2),
      ( 1,    0,    0,   -4,    2),
      ( 2,    1,    2,   -4,    2),
      ( 2,    1,    2,   -4,    1),
      ( 0,    1,    4,   -4,    4),
      ( 0,    1,    4,   -4,    2),
      (-1,   -1,   -2,    4,    0),
      (-1,   -3,    0,    2,    0),
      (-1,    0,   -2,    4,    1),
      (-2,   -1,    0,    3,    0),
      ( 0,    0,   -2,    3,    0),
      (-2,    0,    0,    3,    1),
      ( 0,   -1,    0,    1,    0),
      (-3,    0,    2,    2,    0),
      ( 1,    1,   -2,    2,    0),
      (-1,    1,    0,    2,    2),
      ( 1,   -2,    2,   -2,    1),
      ( 0,    0,    1,    0,    2),
      ( 0,    0,    1,    0,    1),
      ( 0,    0,    1,    0,    0),
      (-1,    2,    0,    2,    1),
      ( 0,    0,    2,    0,    2),
      (-2,    0,    2,    0,    2),
      ( 2,    0,    0,   -1,    1),
      ( 3,    0,    0,   -2,    1),
      ( 1,    0,    2,   -2,    3),
      ( 1,    2,    0,    0,    1),
      ( 2,    0,    2,   -3,    2),
      (-1,    1,    4,   -2,    2),
      (-2,   -2,    0,    4,    0),
      ( 0,   -3,    0,    2,    0),
      ( 0,    0,   -2,    4,    0),
      (-1,   -1,    0,    3,    0),
      (-2,    0,    0,    4,    2),
      (-1,    0,    0,    3,    1),
      ( 2,   -2,    0,    0,    0),
      ( 1,   -1,    0,    1,    0),
      (-1,    0,    0,    2,    0),
      ( 0,   -2,    2,    0,    1),
      (-1,    0,    1,    2,    1),
      (-1,    1,    0,    3,    0),
      (-1,   -1,    2,    1,    2),
      ( 0,   -1,    2,    0,    0),
      (-2,    1,    2,    2,    1),
      ( 2,   -2,    2,   -2,    2),
      ( 1,    1,    0,    1,    1),
      ( 1,    0,    1,    0,    1),
      ( 1,    0,    1,    0,    0),
      ( 0,    2,    0,    2,    0),
      ( 2,   -1,    2,   -2,    1),
      ( 0,   -1,    4,   -2,    1),
      ( 0,    0,    4,   -2,    3),
      ( 0,    1,    4,   -2,    1),
      ( 4,    0,    2,   -4,    2),
      ( 2,    2,    2,   -2,    2),
      ( 2,    0,    4,   -4,    2),
      (-1,   -2,    0,    4,    0),
      (-1,   -3,    2,    2,    2),
      (-3,    0,    2,    4,    2),
      (-3,    0,    2,   -2,    1),
      (-1,   -1,    0,   -2,    1),
      (-3,    0,    0,    0,    2),
      (-3,    0,   -2,    2,    0),
      ( 0,    1,    0,   -4,    1),
      (-2,    1,    0,   -2,    1),
      (-4,    0,    0,    0,    1),
      (-1,    0,    0,   -4,    1),
      (-3,    0,    0,   -2,    1),
      ( 0,    0,    0,    3,    2),
      (-1,    1,    0,    4,    1),
      ( 1,   -2,    2,    0,    1),
      ( 0,    1,    0,    3,    0),
      (-1,    0,    2,    2,    3),
      ( 0,    0,    2,    2,    2),
      (-2,    0,    2,    2,    2),
      (-1,    1,    2,    2,    0),
      ( 3,    0,    0,    0,    2),
      ( 2,    1,    0,    1,    0),
      ( 2,   -1,    2,   -1,    2),
      ( 0,    0,    2,    0,    1),
      ( 0,    0,    3,    0,    3),
      ( 0,    0,    3,    0,    2),
      (-1,    2,    2,    2,    1),
      (-1,    0,    4,    0,    0),
      ( 1,    2,    2,    0,    1),
      ( 3,    1,    2,   -2,    1),
      ( 1,    1,    4,   -2,    2),
      (-2,   -1,    0,    6,    0),
      ( 0,   -2,    0,    4,    0),
      (-2,    0,    0,    6,    1),
      (-2,   -2,    2,    4,    2),
      ( 0,   -3,    2,    2,    2),
      ( 0,    0,    0,    4,    2),
      (-1,   -1,    2,    3,    2),
      (-2,    0,    2,    4,    0),
      ( 2,   -1,    0,    2,    1),
      ( 1,    0,    0,    3,    0),
      ( 0,    1,    0,    4,    1),
      ( 0,    1,    0,    4,    0),
      ( 1,   -1,    2,    1,    2),
      ( 0,    0,    2,    2,    3),
      ( 1,    0,    2,    2,    2),
      (-1,    0,    2,    2,    2),
      (-2,    0,    4,    2,    1),
      ( 2,    1,    0,    2,    1),
      ( 2,    1,    0,    2,    0),
      ( 2,   -1,    2,    0,    0),
      ( 1,    0,    2,    1,    0),
      ( 0,    1,    2,    2,    0),
      ( 2,    0,    2,    0,    3),
      ( 3,    0,    2,    0,    2),
      ( 1,    0,    2,    0,    2),
      ( 1,    0,    3,    0,    3),
      ( 1,    1,    2,    1,    1),
      ( 0,    2,    2,    2,    2),
      ( 2,    1,    2,    0,    0),
      ( 2,    0,    4,   -2,    1),
      ( 4,    1,    2,   -2,    2),
      (-1,   -1,    0,    6,    0),
      (-3,   -1,    2,    6,    2),
      (-1,    0,    0,    6,    1),
      (-3,    0,    2,    6,    1),
      ( 1,   -1,    0,    4,    1),
      ( 1,   -1,    0,    4,    0),
      (-2,    0,    2,    5,    2),
      ( 1,   -2,    2,    2,    1),
      ( 3,   -1,    0,    2,    0),
      ( 1,   -1,    2,    2,    0),
      ( 0,    0,    2,    3,    1),
      (-1,    1,    2,    4,    1),
      ( 0,    1,    2,    3,    2),
      (-1,    0,    4,    2,    1),
      ( 2,    0,    2,    1,    1),
      ( 5,    0,    0,    0,    0),
      ( 2,    1,    2,    1,    2),
      ( 1,    0,    4,    0,    1),
      ( 3,    1,    2,    0,    1),
      ( 3,    0,    4,   -2,    2),
      (-2,   -1,    2,    6,    2),
      ( 0,    0,    0,    6,    0),
      ( 0,   -2,    2,    4,    2),
      (-2,    0,    2,    6,    1),
      ( 2,    0,    0,    4,    1),
      ( 2,    0,    0,    4,    0),
      ( 2,   -2,    2,    2,    2),
      ( 0,    0,    2,    4,    0),
      ( 1,    0,    2,    3,    2),
      ( 4,    0,    0,    2,    0),
      ( 2,    0,    2,    2,    0),
      ( 0,    0,    4,    2,    2),
      ( 4,   -1,    2,    0,    2),
      ( 3,    0,    2,    1,    2),
      ( 2,    1,    2,    2,    1),
      ( 4,    1,    2,    0,    2),
      (-1,   -1,    2,    6,    2),
      (-1,    0,    2,    6,    1),
      ( 1,   -1,    2,    4,    1),
      ( 1,    1,    2,    4,    2),
      ( 3,    1,    2,    2,    2),
      ( 5,    0,    2,    0,    1),
      ( 2,   -1,    2,    4,    2),
      ( 2,    0,    2,    4,    1),
      ))

# Luni-Solar nutation coefficients, unit 1e-7 arcsec:
# longitude (sin, t*sin, cos), obliquity (cos, t*cos, sin)

# Each row of coefficients belongs with the corresponding row of
# fundamental-argument multipliers in 'nals_t'.

lunisolar_coefficients = array((
      (-172064161.0, -174666.0,  33386.0, 92052331.0,  9086.0, 15377.0),
      ( -13170906.0,   -1675.0, -13696.0,  5730336.0, -3015.0, -4587.0),
      (  -2276413.0,    -234.0,   2796.0,   978459.0,  -485.0,  1374.0),
      (   2074554.0,     207.0,   -698.0,  -897492.0,   470.0,  -291.0),
      (   1475877.0,   -3633.0,  11817.0,    73871.0,  -184.0, -1924.0),
      (   -516821.0,    1226.0,   -524.0,   224386.0,  -677.0,  -174.0),
      (    711159.0,      73.0,   -872.0,    -6750.0,     0.0,   358.0),
      (   -387298.0,    -367.0,    380.0,   200728.0,    18.0,   318.0),
      (   -301461.0,     -36.0,    816.0,   129025.0,   -63.0,   367.0),
      (    215829.0,    -494.0,    111.0,   -95929.0,   299.0,   132.0),
      (    128227.0,     137.0,    181.0,   -68982.0,    -9.0,    39.0),
      (    123457.0,      11.0,     19.0,   -53311.0,    32.0,    -4.0),
      (    156994.0,      10.0,   -168.0,    -1235.0,     0.0,    82.0),
      (     63110.0,      63.0,     27.0,   -33228.0,     0.0,    -9.0),
      (    -57976.0,     -63.0,   -189.0,    31429.0,     0.0,   -75.0),
      (    -59641.0,     -11.0,    149.0,    25543.0,   -11.0,    66.0),
      (    -51613.0,     -42.0,    129.0,    26366.0,     0.0,    78.0),
      (     45893.0,      50.0,     31.0,   -24236.0,   -10.0,    20.0),
      (     63384.0,      11.0,   -150.0,    -1220.0,     0.0,    29.0),
      (    -38571.0,      -1.0,    158.0,    16452.0,   -11.0,    68.0),
      (     32481.0,       0.0,      0.0,   -13870.0,     0.0,     0.0),
      (    -47722.0,       0.0,    -18.0,      477.0,     0.0,   -25.0),
      (    -31046.0,      -1.0,    131.0,    13238.0,   -11.0,    59.0),
      (     28593.0,       0.0,     -1.0,   -12338.0,    10.0,    -3.0),
      (     20441.0,      21.0,     10.0,   -10758.0,     0.0,    -3.0),
      (     29243.0,       0.0,    -74.0,     -609.0,     0.0,    13.0),
      (     25887.0,       0.0,    -66.0,     -550.0,     0.0,    11.0),
      (    -14053.0,     -25.0,     79.0,     8551.0,    -2.0,   -45.0),
      (     15164.0,      10.0,     11.0,    -8001.0,     0.0,    -1.0),
      (    -15794.0,      72.0,    -16.0,     6850.0,   -42.0,    -5.0),
      (     21783.0,       0.0,     13.0,     -167.0,     0.0,    13.0),
      (    -12873.0,     -10.0,    -37.0,     6953.0,     0.0,   -14.0),
      (    -12654.0,      11.0,     63.0,     6415.0,     0.0,    26.0),
      (    -10204.0,       0.0,     25.0,     5222.0,     0.0,    15.0),
      (     16707.0,     -85.0,    -10.0,      168.0,    -1.0,    10.0),
      (     -7691.0,       0.0,     44.0,     3268.0,     0.0,    19.0),
      (    -11024.0,       0.0,    -14.0,      104.0,     0.0,     2.0),
      (      7566.0,     -21.0,    -11.0,    -3250.0,     0.0,    -5.0),
      (     -6637.0,     -11.0,     25.0,     3353.0,     0.0,    14.0),
      (     -7141.0,      21.0,      8.0,     3070.0,     0.0,     4.0),
      (     -6302.0,     -11.0,      2.0,     3272.0,     0.0,     4.0),
      (      5800.0,      10.0,      2.0,    -3045.0,     0.0,    -1.0),
      (      6443.0,       0.0,     -7.0,    -2768.0,     0.0,    -4.0),
      (     -5774.0,     -11.0,    -15.0,     3041.0,     0.0,    -5.0),
      (     -5350.0,       0.0,     21.0,     2695.0,     0.0,    12.0),
      (     -4752.0,     -11.0,     -3.0,     2719.0,     0.0,    -3.0),
      (     -4940.0,     -11.0,    -21.0,     2720.0,     0.0,    -9.0),
      (      7350.0,       0.0,     -8.0,      -51.0,     0.0,     4.0),
      (      4065.0,       0.0,      6.0,    -2206.0,     0.0,     1.0),
      (      6579.0,       0.0,    -24.0,     -199.0,     0.0,     2.0),
      (      3579.0,       0.0,      5.0,    -1900.0,     0.0,     1.0),
      (      4725.0,       0.0,     -6.0,      -41.0,     0.0,     3.0),
      (     -3075.0,       0.0,     -2.0,     1313.0,     0.0,    -1.0),
      (     -2904.0,       0.0,     15.0,     1233.0,     0.0,     7.0),
      (      4348.0,       0.0,    -10.0,      -81.0,     0.0,     2.0),
      (     -2878.0,       0.0,      8.0,     1232.0,     0.0,     4.0),
      (     -4230.0,       0.0,      5.0,      -20.0,     0.0,    -2.0),
      (     -2819.0,       0.0,      7.0,     1207.0,     0.0,     3.0),
      (     -4056.0,       0.0,      5.0,       40.0,     0.0,    -2.0),
      (     -2647.0,       0.0,     11.0,     1129.0,     0.0,     5.0),
      (     -2294.0,       0.0,    -10.0,     1266.0,     0.0,    -4.0),
      (      2481.0,       0.0,     -7.0,    -1062.0,     0.0,    -3.0),
      (      2179.0,       0.0,     -2.0,    -1129.0,     0.0,    -2.0),
      (      3276.0,       0.0,      1.0,       -9.0,     0.0,     0.0),
      (     -3389.0,       0.0,      5.0,       35.0,     0.0,    -2.0),
      (      3339.0,       0.0,    -13.0,     -107.0,     0.0,     1.0),
      (     -1987.0,       0.0,     -6.0,     1073.0,     0.0,    -2.0),
      (     -1981.0,       0.0,      0.0,      854.0,     0.0,     0.0),
      (      4026.0,       0.0,   -353.0,     -553.0,     0.0,  -139.0),
      (      1660.0,       0.0,     -5.0,     -710.0,     0.0,    -2.0),
      (     -1521.0,       0.0,      9.0,      647.0,     0.0,     4.0),
      (      1314.0,       0.0,      0.0,     -700.0,     0.0,     0.0),
      (     -1283.0,       0.0,      0.0,      672.0,     0.0,     0.0),
      (     -1331.0,       0.0,      8.0,      663.0,     0.0,     4.0),
      (      1383.0,       0.0,     -2.0,     -594.0,     0.0,    -2.0),
      (      1405.0,       0.0,      4.0,     -610.0,     0.0,     2.0),
      (      1290.0,       0.0,      0.0,     -556.0,     0.0,     0.0),
      (     -1214.0,       0.0,      5.0,      518.0,     0.0,     2.0),
      (      1146.0,       0.0,     -3.0,     -490.0,     0.0,    -1.0),
      (      1019.0,       0.0,     -1.0,     -527.0,     0.0,    -1.0),
      (     -1100.0,       0.0,      9.0,      465.0,     0.0,     4.0),
      (      -970.0,       0.0,      2.0,      496.0,     0.0,     1.0),
      (      1575.0,       0.0,     -6.0,      -50.0,     0.0,     0.0),
      (       934.0,       0.0,     -3.0,     -399.0,     0.0,    -1.0),
      (       922.0,       0.0,     -1.0,     -395.0,     0.0,    -1.0),
      (       815.0,       0.0,     -1.0,     -422.0,     0.0,    -1.0),
      (       834.0,       0.0,      2.0,     -440.0,     0.0,     1.0),
      (      1248.0,       0.0,      0.0,     -170.0,     0.0,     1.0),
      (      1338.0,       0.0,     -5.0,      -39.0,     0.0,     0.0),
      (       716.0,       0.0,     -2.0,     -389.0,     0.0,    -1.0),
      (      1282.0,       0.0,     -3.0,      -23.0,     0.0,     1.0),
      (       742.0,       0.0,      1.0,     -391.0,     0.0,     0.0),
      (      1020.0,       0.0,    -25.0,     -495.0,     0.0,   -10.0),
      (       715.0,       0.0,     -4.0,     -326.0,     0.0,     2.0),
      (      -666.0,       0.0,     -3.0,      369.0,     0.0,    -1.0),
      (      -667.0,       0.0,      1.0,      346.0,     0.0,     1.0),
      (      -704.0,       0.0,      0.0,      304.0,     0.0,     0.0),
      (      -694.0,       0.0,      5.0,      294.0,     0.0,     2.0),
      (     -1014.0,       0.0,     -1.0,        4.0,     0.0,    -1.0),
      (      -585.0,       0.0,     -2.0,      316.0,     0.0,    -1.0),
      (      -949.0,       0.0,      1.0,        8.0,     0.0,    -1.0),
      (      -595.0,       0.0,      0.0,      258.0,     0.0,     0.0),
      (       528.0,       0.0,      0.0,     -279.0,     0.0,     0.0),
      (      -590.0,       0.0,      4.0,      252.0,     0.0,     2.0),
      (       570.0,       0.0,     -2.0,     -244.0,     0.0,    -1.0),
      (      -502.0,       0.0,      3.0,      250.0,     0.0,     2.0),
      (      -875.0,       0.0,      1.0,       29.0,     0.0,     0.0),
      (      -492.0,       0.0,     -3.0,      275.0,     0.0,    -1.0),
      (       535.0,       0.0,     -2.0,     -228.0,     0.0,    -1.0),
      (      -467.0,       0.0,      1.0,      240.0,     0.0,     1.0),
      (       591.0,       0.0,      0.0,     -253.0,     0.0,     0.0),
      (      -453.0,       0.0,     -1.0,      244.0,     0.0,    -1.0),
      (       766.0,       0.0,      1.0,        9.0,     0.0,     0.0),
      (      -446.0,       0.0,      2.0,      225.0,     0.0,     1.0),
      (      -488.0,       0.0,      2.0,      207.0,     0.0,     1.0),
      (      -468.0,       0.0,      0.0,      201.0,     0.0,     0.0),
      (      -421.0,       0.0,      1.0,      216.0,     0.0,     1.0),
      (       463.0,       0.0,      0.0,     -200.0,     0.0,     0.0),
      (      -673.0,       0.0,      2.0,       14.0,     0.0,     0.0),
      (       658.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (      -438.0,       0.0,      0.0,      188.0,     0.0,     0.0),
      (      -390.0,       0.0,      0.0,      205.0,     0.0,     0.0),
      (       639.0,     -11.0,     -2.0,      -19.0,     0.0,     0.0),
      (       412.0,       0.0,     -2.0,     -176.0,     0.0,    -1.0),
      (      -361.0,       0.0,      0.0,      189.0,     0.0,     0.0),
      (       360.0,       0.0,     -1.0,     -185.0,     0.0,    -1.0),
      (       588.0,       0.0,     -3.0,      -24.0,     0.0,     0.0),
      (      -578.0,       0.0,      1.0,        5.0,     0.0,     0.0),
      (      -396.0,       0.0,      0.0,      171.0,     0.0,     0.0),
      (       565.0,       0.0,     -1.0,       -6.0,     0.0,     0.0),
      (      -335.0,       0.0,     -1.0,      184.0,     0.0,    -1.0),
      (       357.0,       0.0,      1.0,     -154.0,     0.0,     0.0),
      (       321.0,       0.0,      1.0,     -174.0,     0.0,     0.0),
      (      -301.0,       0.0,     -1.0,      162.0,     0.0,     0.0),
      (      -334.0,       0.0,      0.0,      144.0,     0.0,     0.0),
      (       493.0,       0.0,     -2.0,      -15.0,     0.0,     0.0),
      (       494.0,       0.0,     -2.0,      -19.0,     0.0,     0.0),
      (       337.0,       0.0,     -1.0,     -143.0,     0.0,    -1.0),
      (       280.0,       0.0,     -1.0,     -144.0,     0.0,     0.0),
      (       309.0,       0.0,      1.0,     -134.0,     0.0,     0.0),
      (      -263.0,       0.0,      2.0,      131.0,     0.0,     1.0),
      (       253.0,       0.0,      1.0,     -138.0,     0.0,     0.0),
      (       245.0,       0.0,      0.0,     -128.0,     0.0,     0.0),
      (       416.0,       0.0,     -2.0,      -17.0,     0.0,     0.0),
      (      -229.0,       0.0,      0.0,      128.0,     0.0,     0.0),
      (       231.0,       0.0,      0.0,     -120.0,     0.0,     0.0),
      (      -259.0,       0.0,      2.0,      109.0,     0.0,     1.0),
      (       375.0,       0.0,     -1.0,       -8.0,     0.0,     0.0),
      (       252.0,       0.0,      0.0,     -108.0,     0.0,     0.0),
      (      -245.0,       0.0,      1.0,      104.0,     0.0,     0.0),
      (       243.0,       0.0,     -1.0,     -104.0,     0.0,     0.0),
      (       208.0,       0.0,      1.0,     -112.0,     0.0,     0.0),
      (       199.0,       0.0,      0.0,     -102.0,     0.0,     0.0),
      (      -208.0,       0.0,      1.0,      105.0,     0.0,     0.0),
      (       335.0,       0.0,     -2.0,      -14.0,     0.0,     0.0),
      (      -325.0,       0.0,      1.0,        7.0,     0.0,     0.0),
      (      -187.0,       0.0,      0.0,       96.0,     0.0,     0.0),
      (       197.0,       0.0,     -1.0,     -100.0,     0.0,     0.0),
      (      -192.0,       0.0,      2.0,       94.0,     0.0,     1.0),
      (      -188.0,       0.0,      0.0,       83.0,     0.0,     0.0),
      (       276.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (      -286.0,       0.0,      1.0,        6.0,     0.0,     0.0),
      (       186.0,       0.0,     -1.0,      -79.0,     0.0,     0.0),
      (      -219.0,       0.0,      0.0,       43.0,     0.0,     0.0),
      (       276.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (      -153.0,       0.0,     -1.0,       84.0,     0.0,     0.0),
      (      -156.0,       0.0,      0.0,       81.0,     0.0,     0.0),
      (      -154.0,       0.0,      1.0,       78.0,     0.0,     0.0),
      (      -174.0,       0.0,      1.0,       75.0,     0.0,     0.0),
      (      -163.0,       0.0,      2.0,       69.0,     0.0,     1.0),
      (      -228.0,       0.0,      0.0,        1.0,     0.0,     0.0),
      (        91.0,       0.0,     -4.0,      -54.0,     0.0,    -2.0),
      (       175.0,       0.0,      0.0,      -75.0,     0.0,     0.0),
      (      -159.0,       0.0,      0.0,       69.0,     0.0,     0.0),
      (       141.0,       0.0,      0.0,      -72.0,     0.0,     0.0),
      (       147.0,       0.0,      0.0,      -75.0,     0.0,     0.0),
      (      -132.0,       0.0,      0.0,       69.0,     0.0,     0.0),
      (       159.0,       0.0,    -28.0,      -54.0,     0.0,    11.0),
      (       213.0,       0.0,      0.0,       -4.0,     0.0,     0.0),
      (       123.0,       0.0,      0.0,      -64.0,     0.0,     0.0),
      (      -118.0,       0.0,     -1.0,       66.0,     0.0,     0.0),
      (       144.0,       0.0,     -1.0,      -61.0,     0.0,     0.0),
      (      -121.0,       0.0,      1.0,       60.0,     0.0,     0.0),
      (      -134.0,       0.0,      1.0,       56.0,     0.0,     1.0),
      (      -105.0,       0.0,      0.0,       57.0,     0.0,     0.0),
      (      -102.0,       0.0,      0.0,       56.0,     0.0,     0.0),
      (       120.0,       0.0,      0.0,      -52.0,     0.0,     0.0),
      (       101.0,       0.0,      0.0,      -54.0,     0.0,     0.0),
      (      -113.0,       0.0,      0.0,       59.0,     0.0,     0.0),
      (      -106.0,       0.0,      0.0,       61.0,     0.0,     0.0),
      (      -129.0,       0.0,      1.0,       55.0,     0.0,     0.0),
      (      -114.0,       0.0,      0.0,       57.0,     0.0,     0.0),
      (       113.0,       0.0,     -1.0,      -49.0,     0.0,     0.0),
      (      -102.0,       0.0,      0.0,       44.0,     0.0,     0.0),
      (       -94.0,       0.0,      0.0,       51.0,     0.0,     0.0),
      (      -100.0,       0.0,     -1.0,       56.0,     0.0,     0.0),
      (        87.0,       0.0,      0.0,      -47.0,     0.0,     0.0),
      (       161.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (        96.0,       0.0,      0.0,      -50.0,     0.0,     0.0),
      (       151.0,       0.0,     -1.0,       -5.0,     0.0,     0.0),
      (      -104.0,       0.0,      0.0,       44.0,     0.0,     0.0),
      (      -110.0,       0.0,      0.0,       48.0,     0.0,     0.0),
      (      -100.0,       0.0,      1.0,       50.0,     0.0,     0.0),
      (        92.0,       0.0,     -5.0,       12.0,     0.0,    -2.0),
      (        82.0,       0.0,      0.0,      -45.0,     0.0,     0.0),
      (        82.0,       0.0,      0.0,      -45.0,     0.0,     0.0),
      (       -78.0,       0.0,      0.0,       41.0,     0.0,     0.0),
      (       -77.0,       0.0,      0.0,       43.0,     0.0,     0.0),
      (         2.0,       0.0,      0.0,       54.0,     0.0,     0.0),
      (        94.0,       0.0,      0.0,      -40.0,     0.0,     0.0),
      (       -93.0,       0.0,      0.0,       40.0,     0.0,     0.0),
      (       -83.0,       0.0,     10.0,       40.0,     0.0,    -2.0),
      (        83.0,       0.0,      0.0,      -36.0,     0.0,     0.0),
      (       -91.0,       0.0,      0.0,       39.0,     0.0,     0.0),
      (       128.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (       -79.0,       0.0,      0.0,       34.0,     0.0,     0.0),
      (       -83.0,       0.0,      0.0,       47.0,     0.0,     0.0),
      (        84.0,       0.0,      0.0,      -44.0,     0.0,     0.0),
      (        83.0,       0.0,      0.0,      -43.0,     0.0,     0.0),
      (        91.0,       0.0,      0.0,      -39.0,     0.0,     0.0),
      (       -77.0,       0.0,      0.0,       39.0,     0.0,     0.0),
      (        84.0,       0.0,      0.0,      -43.0,     0.0,     0.0),
      (       -92.0,       0.0,      1.0,       39.0,     0.0,     0.0),
      (       -92.0,       0.0,      1.0,       39.0,     0.0,     0.0),
      (       -94.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        68.0,       0.0,      0.0,      -36.0,     0.0,     0.0),
      (       -61.0,       0.0,      0.0,       32.0,     0.0,     0.0),
      (        71.0,       0.0,      0.0,      -31.0,     0.0,     0.0),
      (        62.0,       0.0,      0.0,      -34.0,     0.0,     0.0),
      (       -63.0,       0.0,      0.0,       33.0,     0.0,     0.0),
      (       -73.0,       0.0,      0.0,       32.0,     0.0,     0.0),
      (       115.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (      -103.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (        63.0,       0.0,      0.0,      -28.0,     0.0,     0.0),
      (        74.0,       0.0,      0.0,      -32.0,     0.0,     0.0),
      (      -103.0,       0.0,     -3.0,        3.0,     0.0,    -1.0),
      (       -69.0,       0.0,      0.0,       30.0,     0.0,     0.0),
      (        57.0,       0.0,      0.0,      -29.0,     0.0,     0.0),
      (        94.0,       0.0,      0.0,       -4.0,     0.0,     0.0),
      (        64.0,       0.0,      0.0,      -33.0,     0.0,     0.0),
      (       -63.0,       0.0,      0.0,       26.0,     0.0,     0.0),
      (       -38.0,       0.0,      0.0,       20.0,     0.0,     0.0),
      (       -43.0,       0.0,      0.0,       24.0,     0.0,     0.0),
      (       -45.0,       0.0,      0.0,       23.0,     0.0,     0.0),
      (        47.0,       0.0,      0.0,      -24.0,     0.0,     0.0),
      (       -48.0,       0.0,      0.0,       25.0,     0.0,     0.0),
      (        45.0,       0.0,      0.0,      -26.0,     0.0,     0.0),
      (        56.0,       0.0,      0.0,      -25.0,     0.0,     0.0),
      (        88.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (       -75.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        85.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        49.0,       0.0,      0.0,      -26.0,     0.0,     0.0),
      (       -74.0,       0.0,     -3.0,       -1.0,     0.0,    -1.0),
      (       -39.0,       0.0,      0.0,       21.0,     0.0,     0.0),
      (        45.0,       0.0,      0.0,      -20.0,     0.0,     0.0),
      (        51.0,       0.0,      0.0,      -22.0,     0.0,     0.0),
      (       -40.0,       0.0,      0.0,       21.0,     0.0,     0.0),
      (        41.0,       0.0,      0.0,      -21.0,     0.0,     0.0),
      (       -42.0,       0.0,      0.0,       24.0,     0.0,     0.0),
      (       -51.0,       0.0,      0.0,       22.0,     0.0,     0.0),
      (       -42.0,       0.0,      0.0,       22.0,     0.0,     0.0),
      (        39.0,       0.0,      0.0,      -21.0,     0.0,     0.0),
      (        46.0,       0.0,      0.0,      -18.0,     0.0,     0.0),
      (       -53.0,       0.0,      0.0,       22.0,     0.0,     0.0),
      (        82.0,       0.0,      0.0,       -4.0,     0.0,     0.0),
      (        81.0,       0.0,     -1.0,       -4.0,     0.0,     0.0),
      (        47.0,       0.0,      0.0,      -19.0,     0.0,     0.0),
      (        53.0,       0.0,      0.0,      -23.0,     0.0,     0.0),
      (       -45.0,       0.0,      0.0,       22.0,     0.0,     0.0),
      (       -44.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (       -33.0,       0.0,      0.0,       16.0,     0.0,     0.0),
      (       -61.0,       0.0,      0.0,        1.0,     0.0,     0.0),
      (        28.0,       0.0,      0.0,      -15.0,     0.0,     0.0),
      (       -38.0,       0.0,      0.0,       19.0,     0.0,     0.0),
      (       -33.0,       0.0,      0.0,       21.0,     0.0,     0.0),
      (       -60.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        48.0,       0.0,      0.0,      -10.0,     0.0,     0.0),
      (        27.0,       0.0,      0.0,      -14.0,     0.0,     0.0),
      (        38.0,       0.0,      0.0,      -20.0,     0.0,     0.0),
      (        31.0,       0.0,      0.0,      -13.0,     0.0,     0.0),
      (       -29.0,       0.0,      0.0,       15.0,     0.0,     0.0),
      (        28.0,       0.0,      0.0,      -15.0,     0.0,     0.0),
      (       -32.0,       0.0,      0.0,       15.0,     0.0,     0.0),
      (        45.0,       0.0,      0.0,       -8.0,     0.0,     0.0),
      (       -44.0,       0.0,      0.0,       19.0,     0.0,     0.0),
      (        28.0,       0.0,      0.0,      -15.0,     0.0,     0.0),
      (       -51.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (       -36.0,       0.0,      0.0,       20.0,     0.0,     0.0),
      (        44.0,       0.0,      0.0,      -19.0,     0.0,     0.0),
      (        26.0,       0.0,      0.0,      -14.0,     0.0,     0.0),
      (       -60.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (        35.0,       0.0,      0.0,      -18.0,     0.0,     0.0),
      (       -27.0,       0.0,      0.0,       11.0,     0.0,     0.0),
      (        47.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (        36.0,       0.0,      0.0,      -15.0,     0.0,     0.0),
      (       -36.0,       0.0,      0.0,       20.0,     0.0,     0.0),
      (       -35.0,       0.0,      0.0,       19.0,     0.0,     0.0),
      (       -37.0,       0.0,      0.0,       19.0,     0.0,     0.0),
      (        32.0,       0.0,      0.0,      -16.0,     0.0,     0.0),
      (        35.0,       0.0,      0.0,      -14.0,     0.0,     0.0),
      (        32.0,       0.0,      0.0,      -13.0,     0.0,     0.0),
      (        65.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (        47.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (        32.0,       0.0,      0.0,      -16.0,     0.0,     0.0),
      (        37.0,       0.0,      0.0,      -16.0,     0.0,     0.0),
      (       -30.0,       0.0,      0.0,       15.0,     0.0,     0.0),
      (       -32.0,       0.0,      0.0,       16.0,     0.0,     0.0),
      (       -31.0,       0.0,      0.0,       13.0,     0.0,     0.0),
      (        37.0,       0.0,      0.0,      -16.0,     0.0,     0.0),
      (        31.0,       0.0,      0.0,      -13.0,     0.0,     0.0),
      (        49.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (        32.0,       0.0,      0.0,      -13.0,     0.0,     0.0),
      (        23.0,       0.0,      0.0,      -12.0,     0.0,     0.0),
      (       -43.0,       0.0,      0.0,       18.0,     0.0,     0.0),
      (        26.0,       0.0,      0.0,      -11.0,     0.0,     0.0),
      (       -32.0,       0.0,      0.0,       14.0,     0.0,     0.0),
      (       -29.0,       0.0,      0.0,       14.0,     0.0,     0.0),
      (       -27.0,       0.0,      0.0,       12.0,     0.0,     0.0),
      (        30.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (       -11.0,       0.0,      0.0,        5.0,     0.0,     0.0),
      (       -21.0,       0.0,      0.0,       10.0,     0.0,     0.0),
      (       -34.0,       0.0,      0.0,       15.0,     0.0,     0.0),
      (       -10.0,       0.0,      0.0,        6.0,     0.0,     0.0),
      (       -36.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -9.0,       0.0,      0.0,        4.0,     0.0,     0.0),
      (       -12.0,       0.0,      0.0,        5.0,     0.0,     0.0),
      (       -21.0,       0.0,      0.0,        5.0,     0.0,     0.0),
      (       -29.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (       -15.0,       0.0,      0.0,        3.0,     0.0,     0.0),
      (       -20.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        28.0,       0.0,      0.0,        0.0,     0.0,    -2.0),
      (        17.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (       -22.0,       0.0,      0.0,       12.0,     0.0,     0.0),
      (       -14.0,       0.0,      0.0,        7.0,     0.0,     0.0),
      (        24.0,       0.0,      0.0,      -11.0,     0.0,     0.0),
      (        11.0,       0.0,      0.0,       -6.0,     0.0,     0.0),
      (        14.0,       0.0,      0.0,       -6.0,     0.0,     0.0),
      (        24.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        18.0,       0.0,      0.0,       -8.0,     0.0,     0.0),
      (       -38.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (       -31.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (       -16.0,       0.0,      0.0,        8.0,     0.0,     0.0),
      (        29.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (       -18.0,       0.0,      0.0,       10.0,     0.0,     0.0),
      (       -10.0,       0.0,      0.0,        5.0,     0.0,     0.0),
      (       -17.0,       0.0,      0.0,       10.0,     0.0,     0.0),
      (         9.0,       0.0,      0.0,       -4.0,     0.0,     0.0),
      (        16.0,       0.0,      0.0,       -6.0,     0.0,     0.0),
      (        22.0,       0.0,      0.0,      -12.0,     0.0,     0.0),
      (        20.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (       -13.0,       0.0,      0.0,        6.0,     0.0,     0.0),
      (       -17.0,       0.0,      0.0,        9.0,     0.0,     0.0),
      (       -14.0,       0.0,      0.0,        8.0,     0.0,     0.0),
      (         0.0,       0.0,      0.0,       -7.0,     0.0,     0.0),
      (        14.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        19.0,       0.0,      0.0,      -10.0,     0.0,     0.0),
      (       -34.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (       -20.0,       0.0,      0.0,        8.0,     0.0,     0.0),
      (         9.0,       0.0,      0.0,       -5.0,     0.0,     0.0),
      (       -18.0,       0.0,      0.0,        7.0,     0.0,     0.0),
      (        13.0,       0.0,      0.0,       -6.0,     0.0,     0.0),
      (        17.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (       -12.0,       0.0,      0.0,        5.0,     0.0,     0.0),
      (        15.0,       0.0,      0.0,       -8.0,     0.0,     0.0),
      (       -11.0,       0.0,      0.0,        3.0,     0.0,     0.0),
      (        13.0,       0.0,      0.0,       -5.0,     0.0,     0.0),
      (       -18.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (       -35.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         9.0,       0.0,      0.0,       -4.0,     0.0,     0.0),
      (       -19.0,       0.0,      0.0,       10.0,     0.0,     0.0),
      (       -26.0,       0.0,      0.0,       11.0,     0.0,     0.0),
      (         8.0,       0.0,      0.0,       -4.0,     0.0,     0.0),
      (       -10.0,       0.0,      0.0,        4.0,     0.0,     0.0),
      (        10.0,       0.0,      0.0,       -6.0,     0.0,     0.0),
      (       -21.0,       0.0,      0.0,        9.0,     0.0,     0.0),
      (       -15.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         9.0,       0.0,      0.0,       -5.0,     0.0,     0.0),
      (       -29.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (       -19.0,       0.0,      0.0,       10.0,     0.0,     0.0),
      (        12.0,       0.0,      0.0,       -5.0,     0.0,     0.0),
      (        22.0,       0.0,      0.0,       -9.0,     0.0,     0.0),
      (       -10.0,       0.0,      0.0,        5.0,     0.0,     0.0),
      (       -20.0,       0.0,      0.0,       11.0,     0.0,     0.0),
      (       -20.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (       -17.0,       0.0,      0.0,        7.0,     0.0,     0.0),
      (        15.0,       0.0,      0.0,       -3.0,     0.0,     0.0),
      (         8.0,       0.0,      0.0,       -4.0,     0.0,     0.0),
      (        14.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (       -12.0,       0.0,      0.0,        6.0,     0.0,     0.0),
      (        25.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (       -13.0,       0.0,      0.0,        6.0,     0.0,     0.0),
      (       -14.0,       0.0,      0.0,        8.0,     0.0,     0.0),
      (        13.0,       0.0,      0.0,       -5.0,     0.0,     0.0),
      (       -17.0,       0.0,      0.0,        9.0,     0.0,     0.0),
      (       -12.0,       0.0,      0.0,        6.0,     0.0,     0.0),
      (       -10.0,       0.0,      0.0,        5.0,     0.0,     0.0),
      (        10.0,       0.0,      0.0,       -6.0,     0.0,     0.0),
      (       -15.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (       -22.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        28.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (        15.0,       0.0,      0.0,       -7.0,     0.0,     0.0),
      (        23.0,       0.0,      0.0,      -10.0,     0.0,     0.0),
      (        12.0,       0.0,      0.0,       -5.0,     0.0,     0.0),
      (        29.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (       -25.0,       0.0,      0.0,        1.0,     0.0,     0.0),
      (        22.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (       -18.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        15.0,       0.0,      0.0,        3.0,     0.0,     0.0),
      (       -23.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        12.0,       0.0,      0.0,       -5.0,     0.0,     0.0),
      (        -8.0,       0.0,      0.0,        4.0,     0.0,     0.0),
      (       -19.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (       -10.0,       0.0,      0.0,        4.0,     0.0,     0.0),
      (        21.0,       0.0,      0.0,       -9.0,     0.0,     0.0),
      (        23.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (       -16.0,       0.0,      0.0,        8.0,     0.0,     0.0),
      (       -19.0,       0.0,      0.0,        9.0,     0.0,     0.0),
      (       -22.0,       0.0,      0.0,       10.0,     0.0,     0.0),
      (        27.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (        16.0,       0.0,      0.0,       -8.0,     0.0,     0.0),
      (        19.0,       0.0,      0.0,       -8.0,     0.0,     0.0),
      (         9.0,       0.0,      0.0,       -4.0,     0.0,     0.0),
      (        -9.0,       0.0,      0.0,        4.0,     0.0,     0.0),
      (        -9.0,       0.0,      0.0,        4.0,     0.0,     0.0),
      (        -8.0,       0.0,      0.0,        4.0,     0.0,     0.0),
      (        18.0,       0.0,      0.0,       -9.0,     0.0,     0.0),
      (        16.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (       -10.0,       0.0,      0.0,        4.0,     0.0,     0.0),
      (       -23.0,       0.0,      0.0,        9.0,     0.0,     0.0),
      (        16.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (       -12.0,       0.0,      0.0,        6.0,     0.0,     0.0),
      (        -8.0,       0.0,      0.0,        4.0,     0.0,     0.0),
      (        30.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (        24.0,       0.0,      0.0,      -10.0,     0.0,     0.0),
      (        10.0,       0.0,      0.0,       -4.0,     0.0,     0.0),
      (       -16.0,       0.0,      0.0,        7.0,     0.0,     0.0),
      (       -16.0,       0.0,      0.0,        7.0,     0.0,     0.0),
      (        17.0,       0.0,      0.0,       -7.0,     0.0,     0.0),
      (       -24.0,       0.0,      0.0,       10.0,     0.0,     0.0),
      (       -12.0,       0.0,      0.0,        5.0,     0.0,     0.0),
      (       -24.0,       0.0,      0.0,       11.0,     0.0,     0.0),
      (       -23.0,       0.0,      0.0,        9.0,     0.0,     0.0),
      (       -13.0,       0.0,      0.0,        5.0,     0.0,     0.0),
      (       -15.0,       0.0,      0.0,        7.0,     0.0,     0.0),
      (         0.0,       0.0,  -1988.0,        0.0,     0.0, -1679.0),
      (         0.0,       0.0,    -63.0,        0.0,     0.0,   -27.0),
      (        -4.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         0.0,       0.0,      5.0,        0.0,     0.0,     4.0),
      (         5.0,       0.0,      0.0,       -3.0,     0.0,     0.0),
      (         0.0,       0.0,    364.0,        0.0,     0.0,   176.0),
      (         0.0,       0.0,  -1044.0,        0.0,     0.0,  -891.0),
      (        -3.0,       0.0,      0.0,        1.0,     0.0,     0.0),
      (         4.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (         0.0,       0.0,    330.0,        0.0,     0.0,     0.0),
      (         5.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        1.0,     0.0,     0.0),
      (        -5.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         0.0,       0.0,      5.0,        0.0,     0.0,     0.0),
      (         0.0,       0.0,      0.0,        1.0,     0.0,     0.0),
      (         4.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (         6.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         5.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (        -7.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (       -12.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         5.0,       0.0,      0.0,       -3.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (        -5.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -7.0,       0.0,      0.0,        3.0,     0.0,     0.0),
      (         7.0,       0.0,      0.0,       -4.0,     0.0,     0.0),
      (         0.0,       0.0,    -12.0,        0.0,     0.0,   -10.0),
      (         4.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (        -7.0,       0.0,      0.0,        3.0,     0.0,     0.0),
      (        -4.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        1.0,     0.0,     0.0),
      (         0.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        1.0,     0.0,     0.0),
      (         7.0,       0.0,      0.0,       -3.0,     0.0,     0.0),
      (        -4.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (         4.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (        -5.0,       0.0,      0.0,        3.0,     0.0,     0.0),
      (         5.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -5.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (         5.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (        -8.0,       0.0,      0.0,        3.0,     0.0,     0.0),
      (         9.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         6.0,       0.0,      0.0,       -3.0,     0.0,     0.0),
      (        -5.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -7.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        1.0,     0.0,     0.0),
      (         5.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (         4.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (        -5.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (         4.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (         9.0,       0.0,      0.0,       -3.0,     0.0,     0.0),
      (         4.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         4.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (        -4.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (         9.0,       0.0,      0.0,       -3.0,     0.0,     0.0),
      (        -4.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -4.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (         8.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        1.0,     0.0,     0.0),
      (         6.0,       0.0,      0.0,       -3.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        1.0,     0.0,     0.0),
      (        -7.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         9.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -4.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -5.0,       0.0,      0.0,        3.0,     0.0,     0.0),
      (       -13.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -7.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        10.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (        10.0,       0.0,     13.0,        6.0,     0.0,    -5.0),
      (         0.0,       0.0,     30.0,        0.0,     0.0,    14.0),
      (         0.0,       0.0,   -162.0,        0.0,     0.0,  -138.0),
      (         0.0,       0.0,     75.0,        0.0,     0.0,     0.0),
      (        -7.0,       0.0,      0.0,        4.0,     0.0,     0.0),
      (        -4.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (         4.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (         5.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (         5.0,       0.0,      0.0,       -3.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (        -4.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (        -5.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (         6.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         9.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         5.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -7.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        1.0,     0.0,     0.0),
      (        -4.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (         7.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -4.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         4.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -6.0,       0.0,     -3.0,        3.0,     0.0,     1.0),
      (         0.0,       0.0,     -3.0,        0.0,     0.0,    -2.0),
      (        11.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (        11.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (        -1.0,       0.0,      3.0,        3.0,     0.0,    -1.0),
      (         4.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (         0.0,       0.0,    -13.0,        0.0,     0.0,   -11.0),
      (         3.0,       0.0,      6.0,        0.0,     0.0,     0.0),
      (        -7.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         5.0,       0.0,      0.0,       -3.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        1.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         5.0,       0.0,      0.0,       -3.0,     0.0,     0.0),
      (        -7.0,       0.0,      0.0,        3.0,     0.0,     0.0),
      (         8.0,       0.0,      0.0,       -3.0,     0.0,     0.0),
      (        -4.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (        11.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        1.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (        -4.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (         8.0,       0.0,      0.0,       -4.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (        11.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -6.0,       0.0,      0.0,        3.0,     0.0,     0.0),
      (        -4.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (        -8.0,       0.0,      0.0,        4.0,     0.0,     0.0),
      (        -7.0,       0.0,      0.0,        3.0,     0.0,     0.0),
      (        -4.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (         6.0,       0.0,      0.0,       -3.0,     0.0,     0.0),
      (        -6.0,       0.0,      0.0,        3.0,     0.0,     0.0),
      (         6.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         6.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (         5.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (        -5.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (        -4.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -4.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (         4.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         6.0,       0.0,      0.0,       -3.0,     0.0,     0.0),
      (        -4.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (         0.0,       0.0,    -26.0,        0.0,     0.0,   -11.0),
      (         0.0,       0.0,    -10.0,        0.0,     0.0,    -5.0),
      (         5.0,       0.0,      0.0,       -3.0,     0.0,     0.0),
      (       -13.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (         4.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (         7.0,       0.0,      0.0,       -3.0,     0.0,     0.0),
      (         4.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         5.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (        -6.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (        -5.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (        -7.0,       0.0,      0.0,        3.0,     0.0,     0.0),
      (         5.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (        13.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -4.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         5.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (       -11.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         5.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (         4.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         4.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (        -4.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (         6.0,       0.0,      0.0,       -3.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (       -12.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         4.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -4.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        1.0,     0.0,     0.0),
      (         0.0,       0.0,     -5.0,        0.0,     0.0,    -2.0),
      (        -7.0,       0.0,      0.0,        4.0,     0.0,     0.0),
      (         6.0,       0.0,      0.0,       -3.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         5.0,       0.0,      0.0,       -3.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        1.0,     0.0,     0.0),
      (        -5.0,       0.0,      0.0,        3.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (        12.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (        -4.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (         4.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         6.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         5.0,       0.0,      0.0,       -3.0,     0.0,     0.0),
      (         4.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (        -6.0,       0.0,      0.0,        3.0,     0.0,     0.0),
      (         4.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (         6.0,       0.0,      0.0,       -3.0,     0.0,     0.0),
      (         6.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -6.0,       0.0,      0.0,        3.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (         7.0,       0.0,      0.0,       -4.0,     0.0,     0.0),
      (         4.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (        -5.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (         5.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -6.0,       0.0,      0.0,        3.0,     0.0,     0.0),
      (        -6.0,       0.0,      0.0,        3.0,     0.0,     0.0),
      (        -4.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (        10.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        -4.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (         7.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         7.0,       0.0,      0.0,       -3.0,     0.0,     0.0),
      (         4.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (        11.0,       0.0,      0.0,        0.0,     0.0,     0.0),
      (         5.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (        -6.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (         4.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (         5.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (        -4.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (        -4.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      (         4.0,       0.0,      0.0,       -2.0,     0.0,     0.0),
      (         3.0,       0.0,      0.0,       -1.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        1.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        1.0,     0.0,     0.0),
      (        -3.0,       0.0,      0.0,        2.0,     0.0,     0.0),
      ))

lunisolar_longitude_coefficients = lunisolar_coefficients[:,:3]
lunisolar_obliquity_coefficients = lunisolar_coefficients[:,3:]

# Planetary argument multipliers:
#      L   L'  F   D   Om  Me  Ve  E  Ma  Ju  Sa  Ur  Ne  pre

napl_t = array((
      ( 0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -8, 16, -4, -5,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  8,-16,  4,  5,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  2,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -4,  8, -1, -5,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  1),
      ( 0,  0,  1, -1,  1,  0,  0,  3, -8,  3,  0,  0,  0,  0),
      (-1,  0,  0,  0,  0,  0, 10, -3,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0, -2,  6, -3,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0),
      ( 0,  0,  1, -1,  1,  0,  0, -5,  8, -3,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -4,  8, -3,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  4, -8,  1,  5,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -5,  6,  4,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  1),
      ( 0,  0,  1, -1,  1,  0,  0, -1,  0,  2, -5,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -5,  0,  0,  0),
      ( 0,  0,  1, -1,  1,  0,  0, -1,  0, -2,  5,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0, -2,  5,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0, -2,  5,  0,  0,  2),
      ( 2,  0, -1, -1,  0,  0,  0,  3, -7,  0,  0,  0,  0,  0),
      ( 1,  0,  0, -2,  0,  0, 19,-21,  3,  0,  0,  0,  0,  0),
      ( 0,  0,  1, -1,  1,  0,  2, -4,  0, -3,  0,  0,  0,  0),
      ( 1,  0,  0, -1,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0),
      ( 0,  0,  1, -1,  1,  0,  0, -1,  0, -4, 10,  0,  0,  0),
      (-2,  0,  0,  2,  1,  0,  0,  2,  0,  0, -5,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  3, -7,  4,  0,  0,  0,  0,  0),
      ( 0,  0, -1,  1,  0,  0,  0,  1,  0,  1, -1,  0,  0,  0),
      (-2,  0,  0,  2,  1,  0,  0,  2,  0, -2,  0,  0,  0,  0),
      (-1,  0,  0,  0,  0,  0, 18,-16,  0,  0,  0,  0,  0,  0),
      (-2,  0,  1,  1,  2,  0,  0,  1,  0, -2,  0,  0,  0,  0),
      (-1,  0,  1, -1,  1,  0, 18,-17,  0,  0,  0,  0,  0,  0),
      (-1,  0,  0,  1,  1,  0,  0,  2, -2,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0, -8, 13,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  2, -2,  2,  0, -8, 11,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0, -8, 13,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  1, -1,  1,  0, -8, 12,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  1, -1,  1,  0,  8,-14,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  8,-13,  0,  0,  0,  0,  0,  1),
      (-2,  0,  0,  2,  1,  0,  0,  2,  0, -4,  5,  0,  0,  0),
      (-2,  0,  0,  2,  2,  0,  3, -3,  0,  0,  0,  0,  0,  0),
      (-2,  0,  0,  2,  0,  0,  0,  2,  0, -3,  1,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  3, -5,  0,  2,  0,  0,  0,  0),
      (-2,  0,  0,  2,  0,  0,  0,  2,  0, -4,  3,  0,  0,  0),
      ( 0,  0, -1,  1,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  0, -1,  2,  0,  0,  0,  0,  0),
      ( 0,  0,  1, -1,  2,  0,  0, -2,  2,  0,  0,  0,  0,  0),
      (-1,  0,  1,  0,  1,  0,  3, -5,  0,  0,  0,  0,  0,  0),
      (-1,  0,  0,  1,  0,  0,  3, -4,  0,  0,  0,  0,  0,  0),
      (-2,  0,  0,  2,  0,  0,  0,  2,  0, -2, -2,  0,  0,  0),
      (-2,  0,  2,  0,  2,  0,  0, -5,  9,  0,  0,  0,  0,  0),
      ( 0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0, -1,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0),
      ( 0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  0,  2,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  2),
      (-1,  0,  0,  1,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0),
      ( 0,  0, -1,  1,  0,  0,  0,  1,  0,  0,  2,  0,  0,  0),
      ( 0,  0,  1, -1,  2,  0,  0, -1,  0,  0,  2,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  0, -9, 17,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  2,  0, -3,  5,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  2,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0),
      ( 1,  0,  0, -2,  0,  0, 17,-16,  0, -2,  0,  0,  0,  0),
      ( 0,  0,  1, -1,  1,  0,  0, -1,  0,  1, -3,  0,  0,  0),
      (-2,  0,  0,  2,  1,  0,  0,  5, -6,  0,  0,  0,  0,  0),
      ( 0,  0, -2,  2,  0,  0,  0,  9,-13,  0,  0,  0,  0,  0),
      ( 0,  0,  1, -1,  2,  0,  0, -1,  0,  0,  1,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  1,  0,  0,  0),
      ( 0,  0, -1,  1,  0,  0,  0,  1,  0,  0,  1,  0,  0,  0),
      ( 0,  0, -2,  2,  0,  0,  5, -6,  0,  0,  0,  0,  0,  0),
      ( 0,  0, -1,  1,  1,  0,  5, -7,  0,  0,  0,  0,  0,  0),
      (-2,  0,  0,  2,  0,  0,  6, -8,  0,  0,  0,  0,  0,  0),
      ( 2,  0,  1, -3,  1,  0, -6,  7,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  2,  0,  0,  0,  0,  1,  0,  0,  0,  0),
      ( 0,  0, -1,  1,  1,  0,  0,  1,  0,  1,  0,  0,  0,  0),
      ( 0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  2,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -8, 15,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -8, 15,  0,  0,  0,  0,  1),
      ( 0,  0,  1, -1,  1,  0,  0, -9, 15,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  8,-15,  0,  0,  0,  0,  0),
      ( 1,  0, -1, -1,  0,  0,  0,  8,-15,  0,  0,  0,  0,  0),
      ( 2,  0,  0, -2,  0,  0,  2, -5,  0,  0,  0,  0,  0,  0),
      (-2,  0,  0,  2,  0,  0,  0,  2,  0, -5,  5,  0,  0,  0),
      ( 2,  0,  0, -2,  1,  0,  0, -6,  8,  0,  0,  0,  0,  0),
      ( 2,  0,  0, -2,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0),
      (-2,  0,  1,  1,  0,  0,  0,  1,  0, -3,  0,  0,  0,  0),
      (-2,  0,  1,  1,  1,  0,  0,  1,  0, -3,  0,  0,  0,  0),
      (-2,  0,  0,  2,  0,  0,  0,  2,  0, -3,  0,  0,  0,  0),
      (-2,  0,  0,  2,  0,  0,  0,  6, -8,  0,  0,  0,  0,  0),
      (-2,  0,  0,  2,  0,  0,  0,  2,  0, -1, -5,  0,  0,  0),
      (-1,  0,  0,  1,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0),
      (-1,  0,  1,  1,  1,  0,-20, 20,  0,  0,  0,  0,  0,  0),
      ( 1,  0,  0, -2,  0,  0, 20,-21,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  0,  8,-15,  0,  0,  0,  0,  0),
      ( 0,  0,  2, -2,  1,  0,  0,-10, 15,  0,  0,  0,  0,  0),
      ( 0,  0, -1,  1,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  0,  0,  0,  1,  0,  0,  0,  0),
      ( 0,  0,  1, -1,  2,  0,  0, -1,  0,  1,  0,  0,  0,  0),
      ( 0,  0,  1, -1,  1,  0,  0, -1,  0, -2,  4,  0,  0,  0),
      ( 2,  0,  0, -2,  1,  0, -6,  8,  0,  0,  0,  0,  0,  0),
      ( 0,  0, -2,  2,  1,  0,  5, -6,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  1),
      ( 0,  0,  1, -1,  1,  0,  0, -1,  0,  0, -1,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0),
      ( 0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  1,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  2),
      ( 0,  0,  2, -2,  1,  0,  0, -9, 13,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  0,  7,-13,  0,  0,  0,  0,  0),
      (-2,  0,  0,  2,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  9,-17,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -9, 17,  0,  0,  0,  0,  2),
      ( 1,  0,  0, -1,  1,  0,  0, -3,  4,  0,  0,  0,  0,  0),
      ( 1,  0,  0, -1,  1,  0, -3,  4,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  2,  0,  0, -1,  2,  0,  0,  0,  0,  0),
      ( 0,  0, -1,  1,  1,  0,  0,  0,  2,  0,  0,  0,  0,  0),
      ( 0,  0, -2,  2,  0,  1,  0, -2,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  3, -5,  0,  2,  0,  0,  0,  0),
      (-2,  0,  0,  2,  1,  0,  0,  2,  0, -3,  1,  0,  0,  0),
      (-2,  0,  0,  2,  1,  0,  3, -3,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  8,-13,  0,  0,  0,  0,  0,  0),
      ( 0,  0, -1,  1,  0,  0,  8,-12,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  2, -2,  1,  0, -8, 11,  0,  0,  0,  0,  0,  0),
      (-1,  0,  0,  1,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0),
      (-1,  0,  0,  0,  1,  0, 18,-16,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  1,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  3, -7,  4,  0,  0,  0,  0,  0),
      (-2,  0,  1,  1,  1,  0,  0, -3,  7,  0,  0,  0,  0,  0),
      ( 0,  0,  1, -1,  2,  0,  0, -1,  0, -2,  5,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  0,  0,  0, -2,  5,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  0, -4,  8, -3,  0,  0,  0,  0),
      ( 1,  0,  0,  0,  1,  0,-10,  3,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  2, -2,  1,  0,  0, -2,  0,  0,  0,  0,  0,  0),
      (-1,  0,  0,  0,  1,  0, 10, -3,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  0,  4, -8,  3,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  0,  0,  0,  2, -5,  0,  0,  0),
      ( 0,  0, -1,  1,  0,  0,  0,  1,  0,  2, -5,  0,  0,  0),
      ( 2,  0, -1, -1,  1,  0,  0,  3, -7,  0,  0,  0,  0,  0),
      (-2,  0,  0,  2,  0,  0,  0,  2,  0,  0, -5,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0, -3,  7, -4,  0,  0,  0,  0,  0),
      (-2,  0,  0,  2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0),
      ( 1,  0,  0,  0,  1,  0,-18, 16,  0,  0,  0,  0,  0,  0),
      (-2,  0,  1,  1,  1,  0,  0,  1,  0, -2,  0,  0,  0,  0),
      ( 0,  0,  1, -1,  2,  0, -8, 12,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0, -8, 13,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  1),
      ( 0,  0,  1, -1,  1,  0,  0,  0, -2,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0),
      ( 0,  0,  1, -1,  1,  0,  0, -2,  2,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -1,  2,  0,  0,  0,  0,  1),
      (-1,  0,  0,  1,  1,  0,  3, -4,  0,  0,  0,  0,  0,  0),
      (-1,  0,  0,  1,  1,  0,  0,  3, -4,  0,  0,  0,  0,  0),
      ( 0,  0,  1, -1,  1,  0,  0, -1,  0,  0, -2,  0,  0,  0),
      ( 0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  2,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  2),
      ( 0,  0,  1, -1,  0,  0,  3, -6,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0, -3,  5,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  1, -1,  2,  0, -3,  4,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  0, -2,  4,  0,  0,  0,  0,  0),
      ( 0,  0,  2, -2,  1,  0, -5,  6,  0,  0,  0,  0,  0,  0),
      ( 0,  0, -1,  1,  0,  0,  5, -7,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  5, -8,  0,  0,  0,  0,  0,  0),
      (-2,  0,  0,  2,  1,  0,  6, -8,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  0, -8, 15,  0,  0,  0,  0,  0),
      (-2,  0,  0,  2,  1,  0,  0,  2,  0, -3,  0,  0,  0,  0),
      (-2,  0,  0,  2,  1,  0,  0,  6, -8,  0,  0,  0,  0,  0),
      ( 1,  0,  0, -1,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0),
      ( 0,  0,  1, -1,  1,  0,  0, -1,  0, -1,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  1),
      ( 0,  0,  1, -1,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  2),
      ( 0,  0,  1, -1,  2,  0,  0, -1,  0,  0, -1,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  0,  0,  0,  0, -1,  0,  0,  0),
      ( 0,  0, -1,  1,  0,  0,  0,  1,  0,  0, -1,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -7, 13,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  7,-13,  0,  0,  0,  0,  0),
      ( 2,  0,  0, -2,  1,  0,  0, -5,  6,  0,  0,  0,  0,  0),
      ( 0,  0,  2, -2,  1,  0,  0, -8, 11,  0,  0,  0,  0,  0),
      ( 0,  0,  2, -2,  1, -1,  0,  2,  0,  0,  0,  0,  0,  0),
      (-2,  0,  0,  2,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0),
      ( 0,  0,  1, -1,  1,  0,  0, -1,  0,  0,  3,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  2),
      (-2,  0,  0,  2,  0,  0,  3, -3,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  2,  0,  0,  4, -8,  3,  0,  0,  0,  0),
      ( 2,  0,  0, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0),
      ( 0,  0,  1, -1,  2,  0,  0, -1,  0,  2,  0,  0,  0,  0),
      ( 0,  0,  1, -1,  2,  0,  0,  0, -2,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  0,  1, -2,  0,  0,  0,  0,  0),
      ( 0,  0, -1,  1,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0),
      ( 0,  0, -1,  1,  0,  0,  0,  1,  0,  0, -2,  0,  0,  0),
      ( 0,  0,  2, -2,  1,  0,  0, -2,  0,  0,  2,  0,  0,  0),
      ( 0,  0,  1, -1,  1,  0,  3, -6,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  1, -1,  1,  0, -3,  4,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0, -3,  5,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0, -3,  5,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  2, -2,  2,  0, -3,  3,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0, -3,  5,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  1),
      ( 0,  0,  1, -1,  1,  0,  0,  1, -4,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -2,  4,  0,  0,  0,  0,  1),
      ( 0,  0,  1, -1,  1,  0,  0, -3,  4,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -2,  4,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0, -2,  4,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -5,  8,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  2, -2,  2,  0, -5,  6,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0, -5,  8,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -5,  8,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  1, -1,  1,  0, -5,  7,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0, -5,  8,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  1, -1,  2,  0,  0, -1,  0, -1,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  0,  0,  0, -1,  0,  0,  0,  0),
      ( 0,  0, -1,  1,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0),
      ( 0,  0,  2, -2,  1,  0,  0, -2,  0,  1,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -6, 11,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  6,-11,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0, -1,  0,  4,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  1,  0, -4,  0,  0,  0,  0,  0,  0),
      ( 2,  0,  0, -2,  1,  0, -3,  3,  0,  0,  0,  0,  0,  0),
      (-2,  0,  0,  2,  0,  0,  0,  2,  0,  0, -2,  0,  0,  0),
      ( 0,  0,  2, -2,  1,  0,  0, -7,  9,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  1),
      ( 0,  0,  1, -1,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  2),
      ( 0,  0,  2, -2,  2,  0,  0, -2,  0,  2,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  5,  0,  0,  2),
      ( 0,  0,  0,  0,  1,  0,  3, -5,  0,  0,  0,  0,  0,  0),
      ( 0,  0, -1,  1,  0,  0,  3, -4,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  2, -2,  1,  0, -3,  3,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  0,  2, -4,  0,  0,  0,  0,  0),
      ( 0,  0,  2, -2,  1,  0,  0, -4,  4,  0,  0,  0,  0,  0),
      ( 0,  0,  1, -1,  2,  0, -5,  7,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -3,  6,  0,  0,  0,  0,  1),
      ( 0,  0,  1, -1,  1,  0,  0, -4,  6,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -3,  6,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0, -3,  6,  0,  0,  0,  0,  2),
      ( 0,  0, -1,  1,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  2, -3,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -5,  9,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -5,  9,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  5, -9,  0,  0,  0,  0,  0),
      ( 0,  0, -1,  1,  0,  0,  0,  1,  0, -2,  0,  0,  0,  0),
      ( 0,  0,  2, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0),
      (-2,  0,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0),
      ( 0,  0, -2,  2,  0,  0,  3, -3,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0, -6, 10,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0, -6, 10,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -2,  3,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -2,  3,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  1, -1,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  1),
      ( 0,  0,  1, -1,  1,  0,  0, -1,  0,  3,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  4, -8,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -4,  8,  0,  0,  0,  0,  2),
      ( 0,  0, -2,  2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -4,  7,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -4,  7,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  4, -7,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0, -2,  3,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  2, -2,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -5, 10,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  1,  0, -1,  2,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -3,  5,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -3,  5,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  3, -5,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  1, -1,  1,  0,  1, -3,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  1, -2,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0, -1,  2,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0, -1,  2,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -7, 11,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -7, 11,  0,  0,  0,  0,  0,  1),
      ( 0,  0, -2,  2,  0,  0,  4, -4,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  2, -3,  0,  0,  0,  0,  0),
      ( 0,  0,  2, -2,  1,  0, -4,  4,  0,  0,  0,  0,  0,  0),
      ( 0,  0, -1,  1,  0,  0,  4, -5,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0, -4,  7,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  1, -1,  1,  0, -4,  6,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0, -4,  7,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -4,  6,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -4,  6,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  1, -1,  1,  0, -4,  5,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0, -4,  6,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0,  0),
      (-2,  0,  0,  2,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0),
      ( 0,  0, -1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  1, -1,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -1,  0,  5,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -1,  3,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -7, 12,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -1,  1,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -1,  1,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  1, -1,  1,  0, -1,  0,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  1, -1,  1,  0,  1, -2,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -2,  5,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -1,  0,  4,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  1,  0, -4,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0, -1,  1,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -6, 10,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -6, 10,  0,  0,  0,  0,  0),
      ( 0,  0,  2, -2,  1,  0,  0, -3,  0,  3,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -3,  7,  0,  0,  0,  0,  2),
      (-2,  0,  0,  2,  0,  0,  4, -4,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -5,  8,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  5, -8,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -1,  0,  3,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -1,  0,  3,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  1,  0, -3,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  2, -4,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0, -2,  4,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  1, -1,  1,  0, -2,  3,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0, -2,  4,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -6,  9,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -6,  9,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  6, -9,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  0,  1,  0, -2,  0,  0,  0,  0),
      ( 0,  0,  2, -2,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -4,  6,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  4, -6,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  3, -4,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -1,  0,  2,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  1,  0, -2,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  0,  1,  0, -1,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0, -5,  9,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0, -3,  4,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -3,  4,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  3, -4,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  1,  0,  0,  2, -2,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  0, -1,  0,  2,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  1,  0,  0, -3,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  1,  0,  1, -5,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -1,  0,  1,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  1,  0, -3,  5,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0, -3,  4,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  1,  0,  0, -2,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  1,  0,  0, -1,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  0, -2,  2,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0, -8, 14,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  1,  0,  2, -5,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  5, -8,  3,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  5, -8,  3,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -1,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  3, -8,  3,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -3,  8, -3,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  1,  0, -2,  5,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -8, 12,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -8, 12,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  1,  0,  1, -2,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  1,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  1,  0,  0,  2,  0,  0,  2),
      ( 0,  0,  2, -2,  1,  0, -5,  5,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  1,  0,  1,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  3, -6,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0, -3,  6,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0, -3,  6,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -1,  4,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -5,  7,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -5,  7,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  1, -1,  1,  0, -5,  6,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  2, -2,  1,  0,  0, -1,  0,  1,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -1,  0,  1,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0, -1,  0,  3,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  1,  0,  2,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -2,  6,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  1,  0,  2, -2,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -6,  9,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  6, -9,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0, -2,  2,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  1, -1,  1,  0, -2,  1,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  2, -2,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  1,  0,  3,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -5,  7,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  5, -7,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0, -2,  2,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  1, -3,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0, -1,  3,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  1, -1,  1,  0, -1,  2,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0, -1,  3,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -7, 10,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -7, 10,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0, -4,  8,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -4,  5,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -4,  5,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  4, -5,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -2,  0,  5,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -9, 13,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -1,  5,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -2,  0,  4,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  2,  0, -4,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -2,  7,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  2,  0, -3,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0, -2,  5,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0, -2,  5,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -6,  8,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -6,  8,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  6, -8,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  1,  0,  0,  2,  0, -2,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -3,  9,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -5, 10,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -3,  3,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -3,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0, -5, 13,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  2,  0, -1,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  2,  0, -1,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -2,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -2,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  2,  0,  0, -1,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -6, 15,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -8, 15,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -3,  9, -4,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  2,  0,  2, -5,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -2,  8, -1, -5,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  6, -8,  3,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  1, -1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -6, 16, -4, -5,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -2,  8, -3,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -2,  8, -3,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  6, -8,  1,  5,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  5,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  3, -5,  4,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -8, 11,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -8, 11,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0, -8, 11,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, 11,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  2,  0,  0,  1,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  3, -3,  0,  2,  0,  0,  0,  2),
      ( 0,  0,  2, -2,  1,  0,  0,  4, -8,  3,  0,  0,  0,  0),
      ( 0,  0,  1, -1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  2, -2,  1,  0,  0, -4,  8, -3,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  1,  2,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  2,  0,  1,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -3,  7,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -5,  6,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -5,  6,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  5, -6,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  2,  0,  2,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -1,  6,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  7, -9,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  2, -1,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  6, -7,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -1,  4,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0, -1,  4,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -7,  9,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -7,  9,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  4, -3,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -4,  4,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  4, -4,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -3,  0,  5,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  1,  1,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -9, 12,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  3,  0, -4,  0,  0,  0,  0),
      ( 0,  0,  2, -2,  1,  0,  1, -1,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  7, -8,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  3,  0, -3,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  3,  0, -3,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -2,  6,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -6,  7,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  6, -7,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  3,  0, -2,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  3,  0, -2,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  5, -4,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  3, -2,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  3,  0, -1,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  3,  0, -1,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  3,  0,  0, -2,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  3,  0,  0, -1,  0,  0,  2),
      ( 0,  0,  2, -2,  1,  0,  0,  1,  0, -1,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0, -8, 16,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  3,  0,  2, -5,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  7, -8,  3,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -5, 16, -4, -5,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0, -1,  8, -3,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -8, 10,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -8, 10,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0, -8, 10,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  2,  2,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  3,  0,  1,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -3,  8,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -5,  5,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  5, -5,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  2,  0,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  7, -7,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  7, -7,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  6, -5,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  7, -8,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  5, -3,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  4, -3,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  1,  2,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -9, 11,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -9, 11,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  4,  0, -4,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  4,  0, -3,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -6,  6,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  6, -6,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  4,  0, -2,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  6, -4,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  3, -1,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  4,  0, -1,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  4,  0,  0, -2,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  5, -2,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  4,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  8, -9,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  5, -4,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  2,  1,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0, -7,  7,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  7, -7,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  4, -2,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  5,  0, -4,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  5,  0, -3,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  5,  0, -2,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  3,  0,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -8,  8,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  8, -8,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  5, -3,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  5, -3,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0, -9,  9,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0, -9,  9,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0, -9,  9,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  9, -9,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  6, -4,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  1),
      ( 0,  0,  0,  0,  0,  0,  0,  6,  0,  0,  0,  0,  0,  2),
      ( 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  2),
      ( 1,  0,  0, -2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0),
      ( 1,  0,  0, -2,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0),
      ( 1,  0,  0, -2,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0),
      ( 1,  0,  0, -2,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0),
      (-1,  0,  0,  0,  0,  0,  3, -3,  0,  0,  0,  0,  0,  0),
      (-1,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0),
      (-1,  0,  0,  2,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0),
      ( 1,  0,  0, -2,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0),
      (-2,  0,  0,  2,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0),
      (-1,  0,  0,  0,  0,  0,  0,  2,  0, -3,  0,  0,  0,  0),
      (-1,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0),
      (-1,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0),
      (-1,  0,  0,  2,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0),
      ( 1,  0, -1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0),
      (-1,  0,  0,  2,  0,  0,  0,  2,  0, -3,  0,  0,  0,  0),
      (-2,  0,  0,  0,  0,  0,  0,  2,  0, -3,  0,  0,  0,  0),
      ( 1,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0),
      (-1,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0),
      ( 1,  0,  1, -1,  1,  0,  0, -1,  0,  0,  0,  0,  0,  0),
      (-1,  0,  0,  0,  0,  0,  0,  4, -8,  3,  0,  0,  0,  0),
      (-1,  0,  0,  2,  1,  0,  0,  2,  0, -2,  0,  0,  0,  0),
      ( 0,  0,  0,  0,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0),
      (-1,  0,  0,  2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0),
      (-1,  0,  0,  2,  0,  0,  3, -3,  0,  0,  0,  0,  0,  0),
      ( 1,  0,  0, -2,  1,  0,  0, -2,  0,  2,  0,  0,  0,  0),
      ( 1,  0,  2, -2,  2,  0, -3,  3,  0,  0,  0,  0,  0,  0),
      ( 1,  0,  2, -2,  2,  0,  0, -2,  0,  2,  0,  0,  0,  0),
      ( 1,  0,  0,  0,  0,  0,  1, -1,  0,  0,  0,  0,  0,  0),
      ( 1,  0,  0,  0,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0),
      ( 0,  0,  0, -2,  0,  0,  2, -2,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0, -2,  0,  0,  0,  1,  0, -1,  0,  0,  0,  0),
      ( 0,  0,  2,  0,  2,  0, -2,  2,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  2,  0,  2,  0,  0, -1,  0,  1,  0,  0,  0,  0),
      ( 0,  0,  2,  0,  2,  0, -1,  1,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  2,  0,  2,  0, -2,  3,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  0,  2,  0,  0,  0,  2,  0, -2,  0,  0,  0,  0),
      ( 0,  0,  1,  1,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0),
      ( 1,  0,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0),
      (-1,  0,  2,  0,  2,  0, 10, -3,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0),
      ( 1,  0,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  2,  0,  2,  0,  0,  4, -8,  3,  0,  0,  0,  0),
      ( 0,  0,  2,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0),
      (-1,  0,  2,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0),
      ( 2,  0,  2, -2,  2,  0,  0, -2,  0,  3,  0,  0,  0,  0),
      ( 1,  0,  2,  0,  1,  0,  0, -2,  0,  3,  0,  0,  0,  0),
      ( 0,  0,  1,  1,  0,  0,  0,  1,  0,  0,  0,  0,  0,  0),
      (-1,  0,  2,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0),
      (-2,  0,  2,  2,  2,  0,  0,  2,  0, -2,  0,  0,  0,  0),
      ( 0,  0,  2,  0,  2,  0,  2, -3,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  2,  0,  2,  0,  1, -1,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  2,  0,  2,  0,  0,  1,  0, -1,  0,  0,  0,  0),
      ( 0,  0,  2,  0,  2,  0,  2, -2,  0,  0,  0,  0,  0,  0),
      (-1,  0,  2,  2,  2,  0,  0, -1,  0,  1,  0,  0,  0,  0),
      ( 1,  0,  2,  0,  2,  0, -1,  1,  0,  0,  0,  0,  0,  0),
      (-1,  0,  2,  2,  2,  0,  0,  2,  0, -3,  0,  0,  0,  0),
      ( 2,  0,  2,  0,  2,  0,  0,  2,  0, -3,  0,  0,  0,  0),
      ( 1,  0,  2,  0,  2,  0,  0, -4,  8, -3,  0,  0,  0,  0),
      ( 1,  0,  2,  0,  2,  0,  0,  4, -8,  3,  0,  0,  0,  0),
      ( 1,  0,  1,  1,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  2,  0,  2,  0,  0,  1,  0,  0,  0,  0,  0,  0),
      ( 2,  0,  2,  0,  1,  0,  0,  1,  0,  0,  0,  0,  0,  0),
      (-1,  0,  2,  2,  2,  0,  0,  2,  0, -2,  0,  0,  0,  0),
      (-1,  0,  2,  2,  2,  0,  3, -3,  0,  0,  0,  0,  0,  0),
      ( 1,  0,  2,  0,  2,  0,  1, -1,  0,  0,  0,  0,  0,  0),
      ( 0,  0,  2,  2,  2,  0,  0,  2,  0, -2,  0,  0,  0,  0),
      ))

# Planetary nutation coefficients, unit 1e-7 arcsec:
# longitude (sin, cos), obliquity (sin, cos)

# Each row of coefficients in 'cpl_t' belongs with the corresponding
# row of fundamental-argument multipliers in 'napl_t'.

nutation_coefficients = array((
      ( 1440.0,          0.0,          0.0,          0.0),
      (   56.0,       -117.0,        -42.0,        -40.0),
      (  125.0,        -43.0,          0.0,        -54.0),
      (    0.0,          5.0,          0.0,          0.0),
      (    3.0,         -7.0,         -3.0,          0.0),
      (    3.0,          0.0,          0.0,         -2.0),
      ( -114.0,          0.0,          0.0,         61.0),
      ( -219.0,         89.0,          0.0,          0.0),
      (   -3.0,          0.0,          0.0,          0.0),
      ( -462.0,       1604.0,          0.0,          0.0),
      (   99.0,          0.0,          0.0,        -53.0),
      (   -3.0,          0.0,          0.0,          2.0),
      (    0.0,          6.0,          2.0,          0.0),
      (    3.0,          0.0,          0.0,          0.0),
      (  -12.0,          0.0,          0.0,          0.0),
      (   14.0,       -218.0,        117.0,          8.0),
      (   31.0,       -481.0,       -257.0,        -17.0),
      ( -491.0,        128.0,          0.0,          0.0),
      (-3084.0,       5123.0,       2735.0,       1647.0),
      (-1444.0,       2409.0,      -1286.0,       -771.0),
      (   11.0,        -24.0,        -11.0,         -9.0),
      (   26.0,         -9.0,          0.0,          0.0),
      (  103.0,        -60.0,          0.0,          0.0),
      (    0.0,        -13.0,         -7.0,          0.0),
      (  -26.0,        -29.0,        -16.0,         14.0),
      (    9.0,        -27.0,        -14.0,         -5.0),
      (   12.0,          0.0,          0.0,         -6.0),
      (   -7.0,          0.0,          0.0,          0.0),
      (    0.0,         24.0,          0.0,          0.0),
      (  284.0,          0.0,          0.0,       -151.0),
      (  226.0,        101.0,          0.0,          0.0),
      (    0.0,         -8.0,         -2.0,          0.0),
      (    0.0,         -6.0,         -3.0,          0.0),
      (    5.0,          0.0,          0.0,         -3.0),
      (  -41.0,        175.0,         76.0,         17.0),
      (    0.0,         15.0,          6.0,          0.0),
      (  425.0,        212.0,       -133.0,        269.0),
      ( 1200.0,        598.0,        319.0,       -641.0),
      (  235.0,        334.0,          0.0,          0.0),
      (   11.0,        -12.0,         -7.0,         -6.0),
      (    5.0,         -6.0,          3.0,          3.0),
      (   -5.0,          0.0,          0.0,          3.0),
      (    6.0,          0.0,          0.0,         -3.0),
      (   15.0,          0.0,          0.0,          0.0),
      (   13.0,          0.0,          0.0,         -7.0),
      (   -6.0,         -9.0,          0.0,          0.0),
      (  266.0,        -78.0,          0.0,          0.0),
      ( -460.0,       -435.0,       -232.0,        246.0),
      (    0.0,         15.0,          7.0,          0.0),
      (   -3.0,          0.0,          0.0,          2.0),
      (    0.0,        131.0,          0.0,          0.0),
      (    4.0,          0.0,          0.0,          0.0),
      (    0.0,          3.0,          0.0,          0.0),
      (    0.0,          4.0,          2.0,          0.0),
      (    0.0,          3.0,          0.0,          0.0),
      (  -17.0,        -19.0,        -10.0,          9.0),
      (   -9.0,        -11.0,          6.0,         -5.0),
      (   -6.0,          0.0,          0.0,          3.0),
      (  -16.0,          8.0,          0.0,          0.0),
      (    0.0,          3.0,          0.0,          0.0),
      (   11.0,         24.0,         11.0,         -5.0),
      (   -3.0,         -4.0,         -2.0,          1.0),
      (    3.0,          0.0,          0.0,         -1.0),
      (    0.0,         -8.0,         -4.0,          0.0),
      (    0.0,          3.0,          0.0,          0.0),
      (    0.0,          5.0,          0.0,          0.0),
      (    0.0,          3.0,          2.0,          0.0),
      (   -6.0,          4.0,          2.0,          3.0),
      (   -3.0,         -5.0,          0.0,          0.0),
      (   -5.0,          0.0,          0.0,          2.0),
      (    4.0,         24.0,         13.0,         -2.0),
      (  -42.0,         20.0,          0.0,          0.0),
      (  -10.0,        233.0,          0.0,          0.0),
      (   -3.0,          0.0,          0.0,          1.0),
      (   78.0,        -18.0,          0.0,          0.0),
      (    0.0,          3.0,          1.0,          0.0),
      (    0.0,         -3.0,         -1.0,          0.0),
      (    0.0,         -4.0,         -2.0,          1.0),
      (    0.0,         -8.0,         -4.0,         -1.0),
      (    0.0,         -5.0,          3.0,          0.0),
      (   -7.0,          0.0,          0.0,          3.0),
      (  -14.0,          8.0,          3.0,          6.0),
      (    0.0,          8.0,         -4.0,          0.0),
      (    0.0,         19.0,         10.0,          0.0),
      (   45.0,        -22.0,          0.0,          0.0),
      (   -3.0,          0.0,          0.0,          0.0),
      (    0.0,         -3.0,          0.0,          0.0),
      (    0.0,          3.0,          0.0,          0.0),
      (    3.0,          5.0,          3.0,         -2.0),
      (   89.0,        -16.0,         -9.0,        -48.0),
      (    0.0,          3.0,          0.0,          0.0),
      (   -3.0,          7.0,          4.0,          2.0),
      ( -349.0,        -62.0,          0.0,          0.0),
      (  -15.0,         22.0,          0.0,          0.0),
      (   -3.0,          0.0,          0.0,          0.0),
      (  -53.0,          0.0,          0.0,          0.0),
      (    5.0,          0.0,          0.0,         -3.0),
      (    0.0,         -8.0,          0.0,          0.0),
      (   15.0,         -7.0,         -4.0,         -8.0),
      (   -3.0,          0.0,          0.0,          1.0),
      (  -21.0,        -78.0,          0.0,          0.0),
      (   20.0,        -70.0,        -37.0,        -11.0),
      (    0.0,          6.0,          3.0,          0.0),
      (    5.0,          3.0,          2.0,         -2.0),
      (  -17.0,         -4.0,         -2.0,          9.0),
      (    0.0,          6.0,          3.0,          0.0),
      (   32.0,         15.0,         -8.0,         17.0),
      (  174.0,         84.0,         45.0,        -93.0),
      (   11.0,         56.0,          0.0,          0.0),
      (  -66.0,        -12.0,         -6.0,         35.0),
      (   47.0,          8.0,          4.0,        -25.0),
      (    0.0,          8.0,          4.0,          0.0),
      (   10.0,        -22.0,        -12.0,         -5.0),
      (   -3.0,          0.0,          0.0,          2.0),
      (  -24.0,         12.0,          0.0,          0.0),
      (    5.0,         -6.0,          0.0,          0.0),
      (    3.0,          0.0,          0.0,         -2.0),
      (    4.0,          3.0,          1.0,         -2.0),
      (    0.0,         29.0,         15.0,          0.0),
      (   -5.0,         -4.0,         -2.0,          2.0),
      (    8.0,         -3.0,         -1.0,         -5.0),
      (    0.0,         -3.0,          0.0,          0.0),
      (   10.0,          0.0,          0.0,          0.0),
      (    3.0,          0.0,          0.0,         -2.0),
      (   -5.0,          0.0,          0.0,          3.0),
      (   46.0,         66.0,         35.0,        -25.0),
      (  -14.0,          7.0,          0.0,          0.0),
      (    0.0,          3.0,          2.0,          0.0),
      (   -5.0,          0.0,          0.0,          0.0),
      (  -68.0,        -34.0,        -18.0,         36.0),
      (    0.0,         14.0,          7.0,          0.0),
      (   10.0,         -6.0,         -3.0,         -5.0),
      (   -5.0,         -4.0,         -2.0,          3.0),
      (   -3.0,          5.0,          2.0,          1.0),
      (   76.0,         17.0,          9.0,        -41.0),
      (   84.0,        298.0,        159.0,        -45.0),
      (    3.0,          0.0,          0.0,         -1.0),
      (   -3.0,          0.0,          0.0,          2.0),
      (   -3.0,          0.0,          0.0,          1.0),
      (  -82.0,        292.0,        156.0,         44.0),
      (  -73.0,         17.0,          9.0,         39.0),
      (   -9.0,        -16.0,          0.0,          0.0),
      (    3.0,          0.0,         -1.0,         -2.0),
      (   -3.0,          0.0,          0.0,          0.0),
      (   -9.0,         -5.0,         -3.0,          5.0),
      ( -439.0,          0.0,          0.0,          0.0),
      (   57.0,        -28.0,        -15.0,        -30.0),
      (    0.0,         -6.0,         -3.0,          0.0),
      (   -4.0,          0.0,          0.0,          2.0),
      (  -40.0,         57.0,         30.0,         21.0),
      (   23.0,          7.0,          3.0,        -13.0),
      (  273.0,         80.0,         43.0,       -146.0),
      ( -449.0,        430.0,          0.0,          0.0),
      (   -8.0,        -47.0,        -25.0,          4.0),
      (    6.0,         47.0,         25.0,         -3.0),
      (    0.0,         23.0,         13.0,          0.0),
      (   -3.0,          0.0,          0.0,          2.0),
      (    3.0,         -4.0,         -2.0,         -2.0),
      (  -48.0,       -110.0,        -59.0,         26.0),
      (   51.0,        114.0,         61.0,        -27.0),
      ( -133.0,          0.0,          0.0,         57.0),
      (    0.0,          4.0,          0.0,          0.0),
      (  -21.0,         -6.0,         -3.0,         11.0),
      (    0.0,         -3.0,         -1.0,          0.0),
      (  -11.0,        -21.0,        -11.0,          6.0),
      (  -18.0,       -436.0,       -233.0,          9.0),
      (   35.0,         -7.0,          0.0,          0.0),
      (    0.0,          5.0,          3.0,          0.0),
      (   11.0,         -3.0,         -1.0,         -6.0),
      (   -5.0,         -3.0,         -1.0,          3.0),
      (  -53.0,         -9.0,         -5.0,         28.0),
      (    0.0,          3.0,          2.0,          1.0),
      (    4.0,          0.0,          0.0,         -2.0),
      (    0.0,         -4.0,          0.0,          0.0),
      (  -50.0,        194.0,        103.0,         27.0),
      (  -13.0,         52.0,         28.0,          7.0),
      (  -91.0,        248.0,          0.0,          0.0),
      (    6.0,         49.0,         26.0,         -3.0),
      (   -6.0,        -47.0,        -25.0,          3.0),
      (    0.0,          5.0,          3.0,          0.0),
      (   52.0,         23.0,         10.0,        -23.0),
      (   -3.0,          0.0,          0.0,          1.0),
      (    0.0,          5.0,          3.0,          0.0),
      (   -4.0,          0.0,          0.0,          0.0),
      (   -4.0,          8.0,          3.0,          2.0),
      (   10.0,          0.0,          0.0,          0.0),
      (    3.0,          0.0,          0.0,         -2.0),
      (    0.0,          8.0,          4.0,          0.0),
      (    0.0,          8.0,          4.0,          1.0),
      (   -4.0,          0.0,          0.0,          0.0),
      (   -4.0,          0.0,          0.0,          0.0),
      (   -8.0,          4.0,          2.0,          4.0),
      (    8.0,         -4.0,         -2.0,         -4.0),
      (    0.0,         15.0,          7.0,          0.0),
      ( -138.0,          0.0,          0.0,          0.0),
      (    0.0,         -7.0,         -3.0,          0.0),
      (    0.0,         -7.0,         -3.0,          0.0),
      (   54.0,          0.0,          0.0,        -29.0),
      (    0.0,         10.0,          4.0,          0.0),
      (   -7.0,          0.0,          0.0,          3.0),
      (  -37.0,         35.0,         19.0,         20.0),
      (    0.0,          4.0,          0.0,          0.0),
      (   -4.0,          9.0,          0.0,          0.0),
      (    8.0,          0.0,          0.0,         -4.0),
      (   -9.0,        -14.0,         -8.0,          5.0),
      (   -3.0,         -9.0,         -5.0,          3.0),
      ( -145.0,         47.0,          0.0,          0.0),
      (  -10.0,         40.0,         21.0,          5.0),
      (   11.0,        -49.0,        -26.0,         -7.0),
      (-2150.0,          0.0,          0.0,        932.0),
      (  -12.0,          0.0,          0.0,          5.0),
      (   85.0,          0.0,          0.0,        -37.0),
      (    4.0,          0.0,          0.0,         -2.0),
      (    3.0,          0.0,          0.0,         -2.0),
      (  -86.0,        153.0,          0.0,          0.0),
      (   -6.0,          9.0,          5.0,          3.0),
      (    9.0,        -13.0,         -7.0,         -5.0),
      (   -8.0,         12.0,          6.0,          4.0),
      (  -51.0,          0.0,          0.0,         22.0),
      (  -11.0,       -268.0,       -116.0,          5.0),
      (    0.0,         12.0,          5.0,          0.0),
      (    0.0,          7.0,          3.0,          0.0),
      (   31.0,          6.0,          3.0,        -17.0),
      (  140.0,         27.0,         14.0,        -75.0),
      (   57.0,         11.0,          6.0,        -30.0),
      (  -14.0,        -39.0,          0.0,          0.0),
      (    0.0,         -6.0,         -2.0,          0.0),
      (    4.0,         15.0,          8.0,         -2.0),
      (    0.0,          4.0,          0.0,          0.0),
      (   -3.0,          0.0,          0.0,          1.0),
      (    0.0,         11.0,          5.0,          0.0),
      (    9.0,          6.0,          0.0,          0.0),
      (   -4.0,         10.0,          4.0,          2.0),
      (    5.0,          3.0,          0.0,          0.0),
      (   16.0,          0.0,          0.0,         -9.0),
      (   -3.0,          0.0,          0.0,          0.0),
      (    0.0,          3.0,          2.0,         -1.0),
      (    7.0,          0.0,          0.0,         -3.0),
      (  -25.0,         22.0,          0.0,          0.0),
      (   42.0,        223.0,        119.0,        -22.0),
      (  -27.0,       -143.0,        -77.0,         14.0),
      (    9.0,         49.0,         26.0,         -5.0),
      (-1166.0,          0.0,          0.0,        505.0),
      (   -5.0,          0.0,          0.0,          2.0),
      (   -6.0,          0.0,          0.0,          3.0),
      (   -8.0,          0.0,          1.0,          4.0),
      (    0.0,         -4.0,          0.0,          0.0),
      (  117.0,          0.0,          0.0,        -63.0),
      (   -4.0,          8.0,          4.0,          2.0),
      (    3.0,          0.0,          0.0,         -2.0),
      (   -5.0,          0.0,          0.0,          2.0),
      (    0.0,         31.0,          0.0,          0.0),
      (   -5.0,          0.0,          1.0,          3.0),
      (    4.0,          0.0,          0.0,         -2.0),
      (   -4.0,          0.0,          0.0,          2.0),
      (  -24.0,        -13.0,         -6.0,         10.0),
      (    3.0,          0.0,          0.0,          0.0),
      (    0.0,        -32.0,        -17.0,          0.0),
      (    8.0,         12.0,          5.0,         -3.0),
      (    3.0,          0.0,          0.0,         -1.0),
      (    7.0,         13.0,          0.0,          0.0),
      (   -3.0,         16.0,          0.0,          0.0),
      (   50.0,          0.0,          0.0,        -27.0),
      (    0.0,         -5.0,         -3.0,          0.0),
      (   13.0,          0.0,          0.0,          0.0),
      (    0.0,          5.0,          3.0,          1.0),
      (   24.0,          5.0,          2.0,        -11.0),
      (    5.0,        -11.0,         -5.0,         -2.0),
      (   30.0,         -3.0,         -2.0,        -16.0),
      (   18.0,          0.0,          0.0,         -9.0),
      (    8.0,        614.0,          0.0,          0.0),
      (    3.0,         -3.0,         -1.0,         -2.0),
      (    6.0,         17.0,          9.0,         -3.0),
      (   -3.0,         -9.0,         -5.0,          2.0),
      (    0.0,          6.0,          3.0,         -1.0),
      ( -127.0,         21.0,          9.0,         55.0),
      (    3.0,          5.0,          0.0,          0.0),
      (   -6.0,        -10.0,         -4.0,          3.0),
      (    5.0,          0.0,          0.0,          0.0),
      (   16.0,          9.0,          4.0,         -7.0),
      (    3.0,          0.0,          0.0,         -2.0),
      (    0.0,         22.0,          0.0,          0.0),
      (    0.0,         19.0,         10.0,          0.0),
      (    7.0,          0.0,          0.0,         -4.0),
      (    0.0,         -5.0,         -2.0,          0.0),
      (    0.0,          3.0,          1.0,          0.0),
      (   -9.0,          3.0,          1.0,          4.0),
      (   17.0,          0.0,          0.0,         -7.0),
      (    0.0,         -3.0,         -2.0,         -1.0),
      (  -20.0,         34.0,          0.0,          0.0),
      (  -10.0,          0.0,          1.0,          5.0),
      (   -4.0,          0.0,          0.0,          2.0),
      (   22.0,        -87.0,          0.0,          0.0),
      (   -4.0,          0.0,          0.0,          2.0),
      (   -3.0,         -6.0,         -2.0,          1.0),
      (  -16.0,         -3.0,         -1.0,          7.0),
      (    0.0,         -3.0,         -2.0,          0.0),
      (    4.0,          0.0,          0.0,          0.0),
      (  -68.0,         39.0,          0.0,          0.0),
      (   27.0,          0.0,          0.0,        -14.0),
      (    0.0,         -4.0,          0.0,          0.0),
      (  -25.0,          0.0,          0.0,          0.0),
      (  -12.0,         -3.0,         -2.0,          6.0),
      (    3.0,          0.0,          0.0,         -1.0),
      (    3.0,         66.0,         29.0,         -1.0),
      (  490.0,          0.0,          0.0,       -213.0),
      (  -22.0,         93.0,         49.0,         12.0),
      (   -7.0,         28.0,         15.0,          4.0),
      (   -3.0,         13.0,          7.0,          2.0),
      (  -46.0,         14.0,          0.0,          0.0),
      (   -5.0,          0.0,          0.0,          0.0),
      (    2.0,          1.0,          0.0,          0.0),
      (    0.0,         -3.0,          0.0,          0.0),
      (  -28.0,          0.0,          0.0,         15.0),
      (    5.0,          0.0,          0.0,         -2.0),
      (    0.0,          3.0,          0.0,          0.0),
      (  -11.0,          0.0,          0.0,          5.0),
      (    0.0,          3.0,          1.0,          0.0),
      (   -3.0,          0.0,          0.0,          1.0),
      (   25.0,        106.0,         57.0,        -13.0),
      (    5.0,         21.0,         11.0,         -3.0),
      ( 1485.0,          0.0,          0.0,          0.0),
      (   -7.0,        -32.0,        -17.0,          4.0),
      (    0.0,          5.0,          3.0,          0.0),
      (   -6.0,         -3.0,         -2.0,          3.0),
      (   30.0,         -6.0,         -2.0,        -13.0),
      (   -4.0,          4.0,          0.0,          0.0),
      (  -19.0,          0.0,          0.0,         10.0),
      (    0.0,          4.0,          2.0,         -1.0),
      (    0.0,          3.0,          0.0,          0.0),
      (    4.0,          0.0,          0.0,         -2.0),
      (    0.0,         -3.0,         -1.0,          0.0),
      (   -3.0,          0.0,          0.0,          0.0),
      (    5.0,          3.0,          1.0,         -2.0),
      (    0.0,         11.0,          0.0,          0.0),
      (  118.0,          0.0,          0.0,        -52.0),
      (    0.0,         -5.0,         -3.0,          0.0),
      (  -28.0,         36.0,          0.0,          0.0),
      (    5.0,         -5.0,          0.0,          0.0),
      (   14.0,        -59.0,        -31.0,         -8.0),
      (    0.0,          9.0,          5.0,          1.0),
      ( -458.0,          0.0,          0.0,        198.0),
      (    0.0,        -45.0,        -20.0,          0.0),
      (    9.0,          0.0,          0.0,         -5.0),
      (    0.0,         -3.0,          0.0,          0.0),
      (    0.0,         -4.0,         -2.0,         -1.0),
      (   11.0,          0.0,          0.0,         -6.0),
      (    6.0,          0.0,          0.0,         -2.0),
      (  -16.0,         23.0,          0.0,          0.0),
      (    0.0,         -4.0,         -2.0,          0.0),
      (   -5.0,          0.0,          0.0,          2.0),
      ( -166.0,        269.0,          0.0,          0.0),
      (   15.0,          0.0,          0.0,         -8.0),
      (   10.0,          0.0,          0.0,         -4.0),
      (  -78.0,         45.0,          0.0,          0.0),
      (    0.0,         -5.0,         -2.0,          0.0),
      (    7.0,          0.0,          0.0,         -4.0),
      (   -5.0,        328.0,          0.0,          0.0),
      (    3.0,          0.0,          0.0,         -2.0),
      (    5.0,          0.0,          0.0,         -2.0),
      (    0.0,          3.0,          1.0,          0.0),
      (   -3.0,          0.0,          0.0,          0.0),
      (   -3.0,          0.0,          0.0,          0.0),
      (    0.0,         -4.0,         -2.0,          0.0),
      (-1223.0,        -26.0,          0.0,          0.0),
      (    0.0,          7.0,          3.0,          0.0),
      (    3.0,          0.0,          0.0,          0.0),
      (    0.0,          3.0,          2.0,          0.0),
      (   -6.0,         20.0,          0.0,          0.0),
      ( -368.0,          0.0,          0.0,          0.0),
      (  -75.0,          0.0,          0.0,          0.0),
      (   11.0,          0.0,          0.0,         -6.0),
      (    3.0,          0.0,          0.0,         -2.0),
      (   -3.0,          0.0,          0.0,          1.0),
      (  -13.0,        -30.0,          0.0,          0.0),
      (   21.0,          3.0,          0.0,          0.0),
      (   -3.0,          0.0,          0.0,          1.0),
      (   -4.0,          0.0,          0.0,          2.0),
      (    8.0,        -27.0,          0.0,          0.0),
      (  -19.0,        -11.0,          0.0,          0.0),
      (   -4.0,          0.0,          0.0,          2.0),
      (    0.0,          5.0,          2.0,          0.0),
      (   -6.0,          0.0,          0.0,          2.0),
      (   -8.0,          0.0,          0.0,          0.0),
      (   -1.0,          0.0,          0.0,          0.0),
      (  -14.0,          0.0,          0.0,          6.0),
      (    6.0,          0.0,          0.0,          0.0),
      (  -74.0,          0.0,          0.0,         32.0),
      (    0.0,         -3.0,         -1.0,          0.0),
      (    4.0,          0.0,          0.0,         -2.0),
      (    8.0,         11.0,          0.0,          0.0),
      (    0.0,          3.0,          2.0,          0.0),
      ( -262.0,          0.0,          0.0,        114.0),
      (    0.0,         -4.0,          0.0,          0.0),
      (   -7.0,          0.0,          0.0,          4.0),
      (    0.0,        -27.0,        -12.0,          0.0),
      (  -19.0,         -8.0,         -4.0,          8.0),
      (  202.0,          0.0,          0.0,        -87.0),
      (   -8.0,         35.0,         19.0,          5.0),
      (    0.0,          4.0,          2.0,          0.0),
      (   16.0,         -5.0,          0.0,          0.0),
      (    5.0,          0.0,          0.0,         -3.0),
      (    0.0,         -3.0,          0.0,          0.0),
      (    1.0,          0.0,          0.0,          0.0),
      (  -35.0,        -48.0,        -21.0,         15.0),
      (   -3.0,         -5.0,         -2.0,          1.0),
      (    6.0,          0.0,          0.0,         -3.0),
      (    3.0,          0.0,          0.0,         -1.0),
      (    0.0,         -5.0,          0.0,          0.0),
      (   12.0,         55.0,         29.0,         -6.0),
      (    0.0,          5.0,          3.0,          0.0),
      ( -598.0,          0.0,          0.0,          0.0),
      (   -3.0,        -13.0,         -7.0,          1.0),
      (   -5.0,         -7.0,         -3.0,          2.0),
      (    3.0,          0.0,          0.0,         -1.0),
      (    5.0,         -7.0,          0.0,          0.0),
      (    4.0,          0.0,          0.0,         -2.0),
      (   16.0,         -6.0,          0.0,          0.0),
      (    8.0,         -3.0,          0.0,          0.0),
      (    8.0,        -31.0,        -16.0,         -4.0),
      (    0.0,          3.0,          1.0,          0.0),
      (  113.0,          0.0,          0.0,        -49.0),
      (    0.0,        -24.0,        -10.0,          0.0),
      (    4.0,          0.0,          0.0,         -2.0),
      (   27.0,          0.0,          0.0,          0.0),
      (   -3.0,          0.0,          0.0,          1.0),
      (    0.0,         -4.0,         -2.0,          0.0),
      (    5.0,          0.0,          0.0,         -2.0),
      (    0.0,         -3.0,          0.0,          0.0),
      (  -13.0,          0.0,          0.0,          6.0),
      (    5.0,          0.0,          0.0,         -2.0),
      (  -18.0,        -10.0,         -4.0,          8.0),
      (   -4.0,        -28.0,          0.0,          0.0),
      (   -5.0,          6.0,          3.0,          2.0),
      (   -3.0,          0.0,          0.0,          1.0),
      (   -5.0,         -9.0,         -4.0,          2.0),
      (   17.0,          0.0,          0.0,         -7.0),
      (   11.0,          4.0,          0.0,          0.0),
      (    0.0,         -6.0,         -2.0,          0.0),
      (   83.0,         15.0,          0.0,          0.0),
      (   -4.0,          0.0,          0.0,          2.0),
      (    0.0,       -114.0,        -49.0,          0.0),
      (  117.0,          0.0,          0.0,        -51.0),
      (   -5.0,         19.0,         10.0,          2.0),
      (   -3.0,          0.0,          0.0,          0.0),
      (   -3.0,          0.0,          0.0,          2.0),
      (    0.0,         -3.0,         -1.0,          0.0),
      (    3.0,          0.0,          0.0,          0.0),
      (    0.0,         -6.0,         -2.0,          0.0),
      (  393.0,          3.0,          0.0,          0.0),
      (   -4.0,         21.0,         11.0,          2.0),
      (   -6.0,          0.0,         -1.0,          3.0),
      (   -3.0,          8.0,          4.0,          1.0),
      (    8.0,          0.0,          0.0,          0.0),
      (   18.0,        -29.0,        -13.0,         -8.0),
      (    8.0,         34.0,         18.0,         -4.0),
      (   89.0,          0.0,          0.0,          0.0),
      (    3.0,         12.0,          6.0,         -1.0),
      (   54.0,        -15.0,         -7.0,        -24.0),
      (    0.0,          3.0,          0.0,          0.0),
      (    3.0,          0.0,          0.0,         -1.0),
      (    0.0,         35.0,          0.0,          0.0),
      ( -154.0,        -30.0,        -13.0,         67.0),
      (   15.0,          0.0,          0.0,          0.0),
      (    0.0,          4.0,          2.0,          0.0),
      (    0.0,          9.0,          0.0,          0.0),
      (   80.0,        -71.0,        -31.0,        -35.0),
      (    0.0,        -20.0,         -9.0,          0.0),
      (   11.0,          5.0,          2.0,         -5.0),
      (   61.0,        -96.0,        -42.0,        -27.0),
      (   14.0,          9.0,          4.0,         -6.0),
      (  -11.0,         -6.0,         -3.0,          5.0),
      (    0.0,         -3.0,         -1.0,          0.0),
      (  123.0,       -415.0,       -180.0,        -53.0),
      (    0.0,          0.0,          0.0,        -35.0),
      (   -5.0,          0.0,          0.0,          0.0),
      (    7.0,        -32.0,        -17.0,         -4.0),
      (    0.0,         -9.0,         -5.0,          0.0),
      (    0.0,         -4.0,          2.0,          0.0),
      (  -89.0,          0.0,          0.0,         38.0),
      (    0.0,        -86.0,        -19.0,         -6.0),
      (    0.0,          0.0,        -19.0,          6.0),
      ( -123.0,       -416.0,       -180.0,         53.0),
      (    0.0,         -3.0,         -1.0,          0.0),
      (   12.0,         -6.0,         -3.0,         -5.0),
      (  -13.0,          9.0,          4.0,          6.0),
      (    0.0,        -15.0,         -7.0,          0.0),
      (    3.0,          0.0,          0.0,         -1.0),
      (  -62.0,        -97.0,        -42.0,         27.0),
      (  -11.0,          5.0,          2.0,          5.0),
      (    0.0,        -19.0,         -8.0,          0.0),
      (   -3.0,          0.0,          0.0,          1.0),
      (    0.0,          4.0,          2.0,          0.0),
      (    0.0,          3.0,          0.0,          0.0),
      (    0.0,          4.0,          2.0,          0.0),
      (  -85.0,        -70.0,        -31.0,         37.0),
      (  163.0,        -12.0,         -5.0,        -72.0),
      (  -63.0,        -16.0,         -7.0,         28.0),
      (  -21.0,        -32.0,        -14.0,          9.0),
      (    0.0,         -3.0,         -1.0,          0.0),
      (    3.0,          0.0,          0.0,         -2.0),
      (    0.0,          8.0,          0.0,          0.0),
      (    3.0,         10.0,          4.0,         -1.0),
      (    3.0,          0.0,          0.0,         -1.0),
      (    0.0,         -7.0,         -3.0,          0.0),
      (    0.0,         -4.0,         -2.0,          0.0),
      (    6.0,         19.0,          0.0,          0.0),
      (    5.0,       -173.0,        -75.0,         -2.0),
      (    0.0,         -7.0,         -3.0,          0.0),
      (    7.0,        -12.0,         -5.0,         -3.0),
      (   -3.0,          0.0,          0.0,          2.0),
      (    3.0,         -4.0,         -2.0,         -1.0),
      (   74.0,          0.0,          0.0,        -32.0),
      (   -3.0,         12.0,          6.0,          2.0),
      (   26.0,        -14.0,         -6.0,        -11.0),
      (   19.0,          0.0,          0.0,         -8.0),
      (    6.0,         24.0,         13.0,         -3.0),
      (   83.0,          0.0,          0.0,          0.0),
      (    0.0,        -10.0,         -5.0,          0.0),
      (   11.0,         -3.0,         -1.0,         -5.0),
      (    3.0,          0.0,          1.0,         -1.0),
      (    3.0,          0.0,          0.0,         -1.0),
      (   -4.0,          0.0,          0.0,          0.0),
      (    5.0,        -23.0,        -12.0,         -3.0),
      ( -339.0,          0.0,          0.0,        147.0),
      (    0.0,        -10.0,         -5.0,          0.0),
      (    5.0,          0.0,          0.0,          0.0),
      (    3.0,          0.0,          0.0,         -1.0),
      (    0.0,         -4.0,         -2.0,          0.0),
      (   18.0,         -3.0,          0.0,          0.0),
      (    9.0,        -11.0,         -5.0,         -4.0),
      (   -8.0,          0.0,          0.0,          4.0),
      (    3.0,          0.0,          0.0,         -1.0),
      (    0.0,          9.0,          0.0,          0.0),
      (    6.0,         -9.0,         -4.0,         -2.0),
      (   -4.0,        -12.0,          0.0,          0.0),
      (   67.0,        -91.0,        -39.0,        -29.0),
      (   30.0,        -18.0,         -8.0,        -13.0),
      (    0.0,          0.0,          0.0,          0.0),
      (    0.0,       -114.0,        -50.0,          0.0),
      (    0.0,          0.0,          0.0,         23.0),
      (  517.0,         16.0,          7.0,       -224.0),
      (    0.0,         -7.0,         -3.0,          0.0),
      (  143.0,         -3.0,         -1.0,        -62.0),
      (   29.0,          0.0,          0.0,        -13.0),
      (   -4.0,          0.0,          0.0,          2.0),
      (   -6.0,          0.0,          0.0,          3.0),
      (    5.0,         12.0,          5.0,         -2.0),
      (  -25.0,          0.0,          0.0,         11.0),
      (   -3.0,          0.0,          0.0,          1.0),
      (    0.0,          4.0,          2.0,          0.0),
      (  -22.0,         12.0,          5.0,         10.0),
      (   50.0,          0.0,          0.0,        -22.0),
      (    0.0,          7.0,          4.0,          0.0),
      (    0.0,          3.0,          1.0,          0.0),
      (   -4.0,          4.0,          2.0,          2.0),
      (   -5.0,        -11.0,         -5.0,          2.0),
      (    0.0,          4.0,          2.0,          0.0),
      (    4.0,         17.0,          9.0,         -2.0),
      (   59.0,          0.0,          0.0,          0.0),
      (    0.0,         -4.0,         -2.0,          0.0),
      (   -8.0,          0.0,          0.0,          4.0),
      (   -3.0,          0.0,          0.0,          0.0),
      (    4.0,        -15.0,         -8.0,         -2.0),
      (  370.0,         -8.0,          0.0,       -160.0),
      (    0.0,          0.0,         -3.0,          0.0),
      (    0.0,          3.0,          1.0,          0.0),
      (   -6.0,          3.0,          1.0,          3.0),
      (    0.0,          6.0,          0.0,          0.0),
      (  -10.0,          0.0,          0.0,          4.0),
      (    0.0,          9.0,          4.0,          0.0),
      (    4.0,         17.0,          7.0,         -2.0),
      (   34.0,          0.0,          0.0,        -15.0),
      (    0.0,          5.0,          3.0,          0.0),
      (   -5.0,          0.0,          0.0,          2.0),
      (  -37.0,         -7.0,         -3.0,         16.0),
      (    3.0,         13.0,          7.0,         -2.0),
      (   40.0,          0.0,          0.0,          0.0),
      (    0.0,         -3.0,         -2.0,          0.0),
      ( -184.0,         -3.0,         -1.0,         80.0),
      (   -3.0,          0.0,          0.0,          1.0),
      (   -3.0,          0.0,          0.0,          0.0),
      (    0.0,        -10.0,         -6.0,         -1.0),
      (   31.0,         -6.0,          0.0,        -13.0),
      (   -3.0,        -32.0,        -14.0,          1.0),
      (   -7.0,          0.0,          0.0,          3.0),
      (    0.0,         -8.0,         -4.0,          0.0),
      (    3.0,         -4.0,          0.0,          0.0),
      (    0.0,          4.0,          0.0,          0.0),
      (    0.0,          3.0,          1.0,          0.0),
      (   19.0,        -23.0,        -10.0,          2.0),
      (    0.0,          0.0,          0.0,        -10.0),
      (    0.0,          3.0,          2.0,          0.0),
      (    0.0,          9.0,          5.0,         -1.0),
      (   28.0,          0.0,          0.0,          0.0),
      (    0.0,         -7.0,         -4.0,          0.0),
      (    8.0,         -4.0,          0.0,         -4.0),
      (    0.0,          0.0,         -2.0,          0.0),
      (    0.0,          3.0,          0.0,          0.0),
      (   -3.0,          0.0,          0.0,          1.0),
      (   -9.0,          0.0,          1.0,          4.0),
      (    3.0,         12.0,          5.0,         -1.0),
      (   17.0,         -3.0,         -1.0,          0.0),
      (    0.0,          7.0,          4.0,          0.0),
      (   19.0,          0.0,          0.0,          0.0),
      (    0.0,         -5.0,         -3.0,          0.0),
      (   14.0,         -3.0,          0.0,         -1.0),
      (    0.0,          0.0,         -1.0,          0.0),
      (    0.0,          0.0,          0.0,         -5.0),
      (    0.0,          5.0,          3.0,          0.0),
      (   13.0,          0.0,          0.0,          0.0),
      (    0.0,         -3.0,         -2.0,          0.0),
      (    2.0,          9.0,          4.0,          3.0),
      (    0.0,          0.0,          0.0,         -4.0),
      (    8.0,          0.0,          0.0,          0.0),
      (    0.0,          4.0,          2.0,          0.0),
      (    6.0,          0.0,          0.0,         -3.0),
      (    6.0,          0.0,          0.0,          0.0),
      (    0.0,          3.0,          1.0,          0.0),
      (    5.0,          0.0,          0.0,         -2.0),
      (    3.0,          0.0,          0.0,         -1.0),
      (   -3.0,          0.0,          0.0,          0.0),
      (    6.0,          0.0,          0.0,          0.0),
      (    7.0,          0.0,          0.0,          0.0),
      (   -4.0,          0.0,          0.0,          0.0),
      (    4.0,          0.0,          0.0,          0.0),
      (    6.0,          0.0,          0.0,          0.0),
      (    0.0,         -4.0,          0.0,          0.0),
      (    0.0,         -4.0,          0.0,          0.0),
      (    5.0,          0.0,          0.0,          0.0),
      (   -3.0,          0.0,          0.0,          0.0),
      (    4.0,          0.0,          0.0,          0.0),
      (   -5.0,          0.0,          0.0,          0.0),
      (    4.0,          0.0,          0.0,          0.0),
      (    0.0,          3.0,          0.0,          0.0),
      (   13.0,          0.0,          0.0,          0.0),
      (   21.0,         11.0,          0.0,          0.0),
      (    0.0,         -5.0,          0.0,          0.0),
      (    0.0,         -5.0,         -2.0,          0.0),
      (    0.0,          5.0,          3.0,          0.0),
      (    0.0,         -5.0,          0.0,          0.0),
      (   -3.0,          0.0,          0.0,          2.0),
      (   20.0,         10.0,          0.0,          0.0),
      (  -34.0,          0.0,          0.0,          0.0),
      (  -19.0,          0.0,          0.0,          0.0),
      (    3.0,          0.0,          0.0,         -2.0),
      (   -3.0,          0.0,          0.0,          1.0),
      (   -6.0,          0.0,          0.0,          3.0),
      (   -4.0,          0.0,          0.0,          0.0),
      (    3.0,          0.0,          0.0,          0.0),
      (    3.0,          0.0,          0.0,          0.0),
      (    4.0,          0.0,          0.0,          0.0),
      (    3.0,          0.0,          0.0,         -1.0),
      (    6.0,          0.0,          0.0,         -3.0),
      (   -8.0,          0.0,          0.0,          3.0),
      (    0.0,          3.0,          1.0,          0.0),
      (   -3.0,          0.0,          0.0,          0.0),
      (    0.0,         -3.0,         -2.0,          0.0),
      (  126.0,        -63.0,        -27.0,        -55.0),
      (   -5.0,          0.0,          1.0,          2.0),
      (   -3.0,         28.0,         15.0,          2.0),
      (    5.0,          0.0,          1.0,         -2.0),
      (    0.0,          9.0,          4.0,          1.0),
      (    0.0,          9.0,          4.0,         -1.0),
      ( -126.0,        -63.0,        -27.0,         55.0),
      (    3.0,          0.0,          0.0,         -1.0),
      (   21.0,        -11.0,         -6.0,        -11.0),
      (    0.0,         -4.0,          0.0,          0.0),
      (  -21.0,        -11.0,         -6.0,         11.0),
      (   -3.0,          0.0,          0.0,          1.0),
      (    0.0,          3.0,          1.0,          0.0),
      (    8.0,          0.0,          0.0,         -4.0),
      (   -6.0,          0.0,          0.0,          3.0),
      (   -3.0,          0.0,          0.0,          1.0),
      (    3.0,          0.0,          0.0,         -1.0),
      (   -3.0,          0.0,          0.0,          1.0),
      (   -5.0,          0.0,          0.0,          2.0),
      (   24.0,        -12.0,         -5.0,        -11.0),
      (    0.0,          3.0,          1.0,          0.0),
      (    0.0,          3.0,          1.0,          0.0),
      (    0.0,          3.0,          2.0,          0.0),
      (  -24.0,        -12.0,         -5.0,         10.0),
      (    4.0,          0.0,         -1.0,         -2.0),
      (   13.0,          0.0,          0.0,         -6.0),
      (    7.0,          0.0,          0.0,         -3.0),
      (    3.0,          0.0,          0.0,         -1.0),
      (    3.0,          0.0,          0.0,         -1.0),
      ))

nutation_coefficients_longitude = nutation_coefficients[:,:2]
nutation_coefficients_obliquity = nutation_coefficients[:,2:]
