# test_transmon.py
# meant to be run with 'nose'
#
# This file is part of scqubits.
#
#    Copyright (c) 2019, Jens Koch and Peter Groszkowski
#    All rights reserved.
#
#    This source code is licensed under the BSD-style license found in the
#    LICENSE file in the root directory of this source tree.
############################################################################

import numpy as np

import scqubits as qubit
import scqubits.utils.plotting as plot

from scqubits.utils.constants import TEMPDIR


def transmon_initialize():
    transmon = qubit.Transmon(
        EJ=30.02,
        EC=1.2,
        ng=0.3,
        ncut=31
    )
    return transmon


def test_transmon_eigenvals():
    print("transmon_eigenvals()")
    transmon = transmon_initialize()
    evals_reference = np.asarray([-21.84381856, -6.17518551, 8.01366695, 20.04897106, 30.54312385, 38.7071573])
    assert np.allclose(evals_reference, transmon.eigenvals(filename=TEMPDIR + 'test'))


def test_transmon_eigenvecs():
    print("transmon_eigenvecs()")
    transmon = transmon_initialize()
    evecs_reference = np.asarray([2.82596779e-50, 4.02326328e-50, 5.90227556e-49,
                                  1.74550401e-46, 4.44702148e-44, 1.05392044e-41,
                                  2.31708003e-39, 4.71183050e-37, 8.83422455e-35,
                                  1.52185437e-32, 2.39974016e-30, 3.44943517e-28,
                                  4.49938396e-26, 5.29910511e-24, 5.60376158e-22,
                                  5.28792405e-20, 4.42164853e-18, 3.25040810e-16,
                                  2.08168790e-14, 1.14941994e-12, 5.40534586e-11,
                                  2.13384706e-09, 6.94890358e-08, 1.82700196e-06,
                                  3.77392307e-05, 5.90895221e-04, 6.67283966e-03,
                                  5.04370027e-02, 2.24184912e-01, 4.30836609e-01,
                                  -7.08223225e-02, -3.74513890e-01, 5.60284557e-01,
                                  -2.86068361e-01, -4.42560641e-01, -1.54517678e-01,
                                  -2.75089946e-02, -3.06437898e-03, -2.36400812e-04,
                                  -1.34466417e-05, -5.88883142e-07, -2.05009563e-08,
                                  -5.81536100e-10, -1.37090080e-11, -2.72948277e-13,
                                  -4.65223780e-15, -6.86635442e-17, -8.86242417e-19,
                                  -1.00893360e-20, -1.02076167e-22, -9.23913413e-25,
                                  -7.52599339e-27, -5.54674578e-29, -3.71659725e-31,
                                  -2.27396000e-33, -1.27549790e-35, -6.58293426e-38,
                                  -3.13658463e-40, -1.38399278e-42, -5.67135306e-45,
                                  -2.16194662e-47, -5.65089988e-50, -1.01063092e-50])
    _, evecs_tst = transmon.eigensys(filename=TEMPDIR + 'test')
    evecs_calculated = evecs_tst.T[3]
    assert np.allclose(evecs_reference, evecs_calculated)


def test_transmon_plot_evals_vs_paramvals_ng():
    print("transmon_plot_evals_vs_paramvals_ng()")
    transmon = transmon_initialize()
    ng_list = np.linspace(-1, 1, 100)
    transmon.plot_evals_vs_paramvals('ng', ng_list, evals_count=5, subtract_ground=True)


def test_transmon_get_spectrum_vs_paramvals():
    print("transmon_get_spectrum_vs_paramvals()")
    transmon = transmon_initialize()
    reference_evals = np.asarray([[-21.84390814, -6.17107609, 7.93584546, 20.84011423],
                                 [-21.84389934, -6.1714804, 7.94338947, 20.75062548],
                                 [-21.84387519, -6.1725889, 7.96419673, 20.52074986],
                                 [-21.8438419, -6.17411552, 7.99315565, 20.23390761],
                                 [-21.84380806, -6.17566697, 8.02295507, 19.9693665],
                                 [-21.84378236, -6.17684428, 8.04582426, 19.78288924],
                                 [-21.84377142, -6.17734523, 8.05562357, 19.70671318],
                                 [-21.84377805, -6.17704135, 8.04967422, 19.75271143],
                                 [-21.84380056, -6.17601055, 8.02960601, 19.91379499],
                                 [-21.84383314, -6.17451738, 8.00083801, 20.16307018],
                                 [-21.84386741, -6.17294561, 7.97093158, 20.45095148],
                                 [-21.84389455, -6.17170001, 7.94749717, 20.70337314],
                                 [-21.84390758, -6.17110189, 7.93632611, 20.83430272],
                                 [-21.84390314, -6.17130573, 7.94012724, 20.78887809],
                                 [-21.84388238, -6.17225886, 7.95798254, 20.58703504],
                                 [-21.84385063, -6.17371525, 7.98552832, 20.30626506],
                                 [-21.84381608, -6.17529951, 8.01586288, 20.02993252],
                                 [-21.8437876, -6.17660407, 8.04113987, 19.82005885],
                                 [-21.84377254, -6.17729388, 8.05461729, 19.71444025],
                                 [-21.84377476, -6.17719203, 8.05262247, 19.72982167],
                                 [-21.8437937, -6.17632463, 8.03570247, 19.86384589],
                                 [-21.84382448, -6.17491421, 8.00844917, 20.09477273],
                                 [-21.84385918, -6.17332313, 7.97808023, 20.37902002],
                                 [-21.84388887, -6.17196093, 7.95238686, 20.64838448],
                                 [-21.8439059, -6.17117886, 7.93776068, 20.81704842],
                                 [-21.8439059, -6.17117886, 7.93776068, 20.81704842],
                                 [-21.84388887, -6.17196093, 7.95238686, 20.64838448],
                                 [-21.84385918, -6.17332313, 7.97808023, 20.37902002],
                                 [-21.84382448, -6.17491421, 8.00844917, 20.09477273],
                                 [-21.8437937, -6.17632463, 8.03570247, 19.86384589],
                                 [-21.84377476, -6.17719203, 8.05262247, 19.72982167],
                                 [-21.84377254, -6.17729388, 8.05461729, 19.71444025],
                                 [-21.8437876, -6.17660407, 8.04113987, 19.82005885],
                                 [-21.84381608, -6.17529951, 8.01586288, 20.02993252],
                                 [-21.84385063, -6.17371525, 7.98552832, 20.30626506],
                                 [-21.84388238, -6.17225886, 7.95798254, 20.58703504],
                                 [-21.84390314, -6.17130573, 7.94012724, 20.78887809],
                                 [-21.84390758, -6.17110189, 7.93632611, 20.83430272],
                                 [-21.84389455, -6.17170001, 7.94749717, 20.70337314],
                                 [-21.84386741, -6.17294561, 7.97093158, 20.45095148],
                                 [-21.84383314, -6.17451738, 8.00083801, 20.16307018],
                                 [-21.84380056, -6.17601055, 8.02960601, 19.91379499],
                                 [-21.84377805, -6.17704135, 8.04967422, 19.75271143],
                                 [-21.84377142, -6.17734523, 8.05562357, 19.70671318],
                                 [-21.84378236, -6.17684428, 8.04582426, 19.78288924],
                                 [-21.84380806, -6.17566697, 8.02295507, 19.9693665],
                                 [-21.8438419, -6.17411552, 7.99315565, 20.23390761],
                                 [-21.84387519, -6.1725889, 7.96419673, 20.52074986],
                                 [-21.84389934, -6.1714804, 7.94338947, 20.75062548],
                                 [-21.84390814, -6.17107609, 7.93584546, 20.84011423]])
    ng_list = np.linspace(-2, 2, 50)
    calculated_evals = transmon.get_spectrum_vs_paramvals('ng', ng_list, evals_count=4, subtract_ground=False,
                                                    get_eigenstates=True)
    assert np.allclose(reference_evals, calculated_evals.energy_table)

def test_transmon_get_spectrum_vs_paramvals_filewrite():
    print("transmon_get_spectrum_vs_paramvals_filewrite()")
    transmon = transmon_initialize()
    ng_list = np.linspace(-2, 2, 50)
    calculated_evals = transmon.get_spectrum_vs_paramvals('ng', ng_list, evals_count=4, subtract_ground=False,
                                                     get_eigenstates=True, filename=TEMPDIR + 'tst_tmp')

def test_transmon_plot_evals_vs_paramvals_EJ():
    print("transmon_plot_evals_vs_paramvals_EJ()")
    transmon = transmon_initialize()
    transmon.ng = 0.3
    ej_vals = transmon.EJ * np.cos(np.linspace(-np.pi / 2, np.pi / 2, 40))
    transmon.plot_evals_vs_paramvals('EJ', ej_vals, evals_count=4, subtract_ground=False)


def test_transmon_plot_n_wavefunction():
    print("transmon_plot_n_wavefunction()")
    transmon = transmon_initialize()
    transmon.plot_n_wavefunction(esys=None, which=1, mode='real')


def test_transmon_plot_phi_wavefunction():
    print("transmon_plot_phi_wavefunction()")
    transmon = transmon_initialize()
    transmon.plot_phi_wavefunction(esys=None, which=6, mode='real')
    transmon.plot_phi_wavefunction(esys=None, which=(0,3,9), mode='abs_sqr')


def test_transmon_matrixelement_table():
    print("transmon_matrixelement_table()")
    reference_matrix = np.asarray([[2.99957402e-01 - 0.j, -9.02867276e-01 - 0.j, -1.26536183e-03 - 0.j,
                                    -5.03117823e-02 - 0.j, 7.05431423e-03 - 0.j, 7.08892467e-03 - 0.j,
                                    -1.88275461e-03 - 0.j, 6.92384041e-04 - 0.j, -1.25520380e-04 - 0.j,
                                    3.61049140e-05 - 0.j, -4.84901263e-06 - 0.j, 1.17090131e-06 - 0.j,
                                    -1.24373020e-07 - 0.j, 2.60676660e-08 - 0.j, -2.27939079e-09 - 0.j,
                                    4.23871754e-10 - 0.j],
                                   [-9.02867276e-01 + 0.j, 3.01952308e-01 - 0.j, -1.21191324e+00 - 0.j,
                                    2.52010582e-02 - 0.j, 1.09441111e-01 - 0.j, -3.89271656e-02 - 0.j,
                                    -1.29141530e-02 - 0.j, -4.68151630e-03 - 0.j, -8.49521778e-04 - 0.j,
                                    -2.44502684e-04 - 0.j, -3.28166805e-05 - 0.j, -7.92937645e-06 - 0.j,
                                    -8.41719700e-07 - 0.j, -1.76530964e-07 - 0.j, -1.54262406e-08 - 0.j,
                                    -2.87047138e-09 - 0.j],
                                   [-1.26536183e-03 + 0.j, -1.21191324e+00 + 0.j, 2.62410987e-01 - 0.j,
                                    -1.36957066e+00 - 0.j, 2.10165386e-01 - 0.j, 2.08836909e-01 - 0.j,
                                    -5.63017337e-02 - 0.j, 2.04290273e-02 - 0.j, -3.75388991e-03 - 0.j,
                                    1.06529896e-03 - 0.j, -1.45017606e-04 - 0.j, 3.45482051e-05 - 0.j,
                                    -3.71957736e-06 - 0.j, 7.69143445e-07 - 0.j, -6.81688872e-08 - 0.j,
                                    1.25066119e-08 - 0.j],
                                   [-5.03117823e-02 + 0.j, 2.52010582e-02 + 0.j, -1.36957066e+00 + 0.j,
                                    6.27078975e-01 - 0.j, 1.20862842e+00 - 0.j, -5.34406599e-01 - 0.j,
                                    -1.55118315e-01 - 0.j, -6.52284685e-02 - 0.j, -1.02134587e-02 - 0.j,
                                    -3.40712566e-03 - 0.j, -3.94542934e-04 - 0.j, -1.10495266e-04 - 0.j,
                                    -1.01196878e-05 - 0.j, -2.45994573e-06 - 0.j, -1.85464043e-07 - 0.j,
                                    -3.99998033e-08 - 0.j],
                                   [7.05431423e-03 + 0.j, 1.09441111e-01 + 0.j, 2.10165386e-01 + 0.j,
                                    1.20862842e+00 + 0.j, -1.14248443e+00 - 0.j, -7.63430508e-01 - 0.j,
                                    5.47221026e-01 - 0.j, -7.98063904e-02 - 0.j, 3.68574665e-02 - 0.j,
                                    -4.16354056e-03 - 0.j, 1.42389904e-03 - 0.j, -1.35025940e-04 - 0.j,
                                    3.65217921e-05 - 0.j, -3.00606983e-06 - 0.j, 6.69336777e-07 - 0.j,
                                    -4.88800223e-08 - 0.j],
                                   [7.08892467e-03 + 0.j, -3.89271656e-02 + 0.j, 2.08836909e-01 + 0.j,
                                    -5.34406599e-01 + 0.j, -7.63430508e-01 + 0.j, 2.56565968e+00 - 0.j,
                                    1.41343797e-01 - 0.j, 4.90119761e-01 - 0.j, 9.35398266e-03 - 0.j,
                                    2.57687455e-02 - 0.j, 3.61347596e-04 - 0.j, 8.35708471e-04 - 0.j,
                                    9.26825603e-06 - 0.j, 1.86052995e-05 - 0.j, 1.69859809e-07 - 0.j,
                                    3.02530381e-07 - 0.j],
                                   [-1.88275461e-03 + 0.j, -1.29141530e-02 + 0.j, -5.63017337e-02 + 0.j,
                                    -1.55118315e-01 + 0.j, 5.47221026e-01 + 0.j, 1.41343797e-01 + 0.j,
                                    -2.83977257e+00 - 0.j, 1.53918804e-02 - 0.j, -4.14516367e-01 - 0.j,
                                    8.03244216e-04 - 0.j, -1.60649547e-02 - 0.j, 2.60496741e-05 - 0.j,
                                    -4.12053923e-04 - 0.j, 5.79941448e-07 - 0.j, -7.55173363e-06 - 0.j,
                                    9.43010391e-09 - 0.j],
                                   [6.92384041e-04 + 0.j, -4.68151630e-03 + 0.j, 2.04290273e-02 + 0.j,
                                    -6.52284685e-02 + 0.j, -7.98063904e-02 + 0.j, 4.90119761e-01 + 0.j,
                                    1.53918804e-02 + 0.j, 3.89324992e+00 - 0.j, 1.01932534e-03 - 0.j,
                                    3.74250819e-01 - 0.j, 3.93769807e-05 - 0.j, 1.21626869e-02 - 0.j,
                                    1.00998580e-06 - 0.j, 2.70777248e-04 - 0.j, 1.85100621e-08 - 0.j,
                                    4.40295756e-06 - 0.j],
                                   [-1.25520380e-04 + 0.j, -8.49521778e-04 + 0.j, -3.75388991e-03 + 0.j,
                                    -1.02134587e-02 + 0.j, 3.68574665e-02 + 0.j, 9.35398266e-03 + 0.j,
                                    -4.14516367e-01 + 0.j, 1.01932534e-03 + 0.j, -3.93476982e+00 - 0.j,
                                    5.31950251e-05 - 0.j, -3.26789838e-01 - 0.j, 1.72514542e-06 - 0.j,
                                    -8.39193226e-03 - 0.j, 3.84067505e-08 - 0.j, -1.53799479e-04 - 0.j,
                                    6.24510715e-10 - 0.j],
                                   [3.61049140e-05 + 0.j, -2.44502684e-04 + 0.j, 1.06529896e-03 + 0.j,
                                    -3.40712566e-03 + 0.j, -4.16354056e-03 + 0.j, 2.57687455e-02 + 0.j,
                                    8.03244216e-04 + 0.j, 3.74250819e-01 + 0.j, 5.31950251e-05 + 0.j,
                                    4.95077789e+00 - 0.j, 2.05494696e-06 - 0.j, 3.01394958e-01 - 0.j,
                                    5.27076278e-08 - 0.j, 6.71571077e-03 - 0.j, 9.65975427e-10 - 0.j,
                                    1.09200461e-04 - 0.j],
                                   [-4.84901263e-06 + 0.j, -3.28166805e-05 + 0.j, -1.45017606e-04 + 0.j,
                                    -3.94542934e-04 + 0.j, 1.42389904e-03 + 0.j, 3.61347596e-04 + 0.j,
                                    -1.60649547e-02 + 0.j, 3.93769807e-05 + 0.j, -3.26789838e-01 + 0.j,
                                    2.05494696e-06 + 0.j, -4.96613473e+00 - 0.j, 6.66431180e-08 - 0.j,
                                    -2.69994737e-01 - 0.j, 1.48366948e-09 - 0.j, -4.95093648e-03 - 0.j,
                                    2.41251206e-11 - 0.j],
                                   [1.17090131e-06 + 0.j, -7.92937645e-06 + 0.j, 3.45482051e-05 + 0.j,
                                    -1.10495266e-04 + 0.j, -1.35025940e-04 + 0.j, 8.35708471e-04 + 0.j,
                                    2.60496741e-05 + 0.j, 1.21626869e-02 + 0.j, 1.72514542e-06 + 0.j,
                                    3.01394958e-01 + 0.j, 6.66431180e-08 + 0.j, 5.97293356e+00 - 0.j,
                                    1.70933884e-09 - 0.j, 2.52485755e-01 - 0.j, 3.13271416e-11 - 0.j,
                                    4.10725659e-03 - 0.j],
                                   [-1.24373020e-07 + 0.j, -8.41719700e-07 + 0.j, -3.71957736e-06 + 0.j,
                                    -1.01196878e-05 + 0.j, 3.65217921e-05 + 0.j, 9.26825603e-06 + 0.j,
                                    -4.12053923e-04 + 0.j, 1.00998580e-06 + 0.j, -8.39193226e-03 + 0.j,
                                    5.27076278e-08 + 0.j, -2.69994737e-01 + 0.j, 1.70933884e-09 + 0.j,
                                    -5.98007424e+00 - 0.j, 3.80548502e-11 - 0.j, -2.30123881e-01 - 0.j,
                                    6.18788660e-13 - 0.j],
                                   [2.60676660e-08 + 0.j, -1.76530964e-07 + 0.j, 7.69143445e-07 + 0.j,
                                    -2.45994573e-06 + 0.j, -3.00606983e-06 + 0.j, 1.86052995e-05 + 0.j,
                                    5.79941448e-07 + 0.j, 2.70777248e-04 + 0.j, 3.84067505e-08 + 0.j,
                                    6.71571077e-03 + 0.j, 1.48366948e-09 + 0.j, 2.52485755e-01 + 0.j,
                                    3.80548502e-11 + 0.j, 6.98348162e+00 - 0.j, 6.97433212e-13 - 0.j,
                                    2.17302781e-01 - 0.j],
                                   [-2.27939079e-09 + 0.j, -1.54262406e-08 + 0.j, -6.81688872e-08 + 0.j,
                                    -1.85464043e-07 + 0.j, 6.69336777e-07 + 0.j, 1.69859809e-07 + 0.j,
                                    -7.55173363e-06 + 0.j, 1.85100621e-08 + 0.j, -1.53799479e-04 + 0.j,
                                    9.65975427e-10 + 0.j, -4.95093648e-03 + 0.j, 3.13271416e-11 + 0.j,
                                    -2.30123881e-01 + 0.j, 6.97433212e-13 + 0.j, -6.98726982e+00 - 0.j,
                                    1.13405719e-14 - 0.j],
                                   [4.23871754e-10 + 0.j, -2.87047138e-09 + 0.j, 1.25066119e-08 + 0.j,
                                    -3.99998033e-08 + 0.j, -4.88800223e-08 + 0.j, 3.02530381e-07 + 0.j,
                                    9.43010391e-09 + 0.j, 4.40295756e-06 + 0.j, 6.24510715e-10 + 0.j,
                                    1.09200461e-04 + 0.j, 2.41251206e-11 + 0.j, 4.10725659e-03 + 0.j,
                                    6.18788660e-13 + 0.j, 2.17302781e-01 + 0.j, 1.13405719e-14 + 0.j,
                                    7.98917001e+00 - 0.j]])
    transmon = transmon_initialize()
    calculated_matrix = transmon.matrixelement_table('n_operator', esys=None, evals_count=16)
    assert np.allclose(reference_matrix, calculated_matrix)


def test_transmon_plot_matrixelements():
    print("transmon_plot_matrixelements()")
    transmon = transmon_initialize()
    transmon.plot_matrixelements('n_operator', evals_count=10)

def test_transmon_print_matrixelements():
    print("transmon_print_matrixelements()")
    transmon = transmon_initialize()
    mat_data = transmon.matrixelement_table('n_operator')
    plot.print_matrix(abs(mat_data))

def test_transmon_plot_matelem_vs_paramvals():
    print("transmon_plot_matelem_vs_paramvals()")
    transmon = transmon_initialize()
    transmon.ncut = 60
    transmon.EJ = 30
    ng_list = np.linspace(-2, 2, 220)
    fig, ax = transmon.plot_matelem_vs_paramvals('n_operator', 'ng', ng_list, select_elems=[(0, 0), (1, 4), (1, 0)],
                                                 filename=TEMPDIR + 'test')
