# test_flux_qubit.py
# meant to be run with 'nose'
#
# This file is part of scqubits.
#
#    Copyright (c) 2019, Jens Koch and Peter Groszkowski
#    All rights reserved.
#
#    This source code is licensed under the BSD-style license found in the
#    LICENSE file in the root directory of this source tree.
############################################################################

import numpy as np

import scqubits as qubit
import scqubits.utils.plotting as plot
from scqubits.utils.constants import TEMPDIR
from scqubits.utils.spectrum_utils import standardize_phases


def flux_qubit_initialize():
    ratio = 60.0
    alpha = 0.8

    flux_qubit = qubit.FluxQubit(
        EJ1 = 1.0, 
        EJ2 = 1.0, 
        EJ3 = alpha*1.0,
        ECJ1 = 1.0/ratio,
        ECJ2 = 1.0/ratio,
        ECJ3 = 1.0/alpha/ratio,
        ECg1 = 50.0/ratio,
        ECg2 = 50.0/ratio,
        ng1 = 0.0, 
        ng2 = 0.0, 
        flux = 0.45, 
        ncut = 10,
    )
    return flux_qubit

def test_flux_qubit_eigenvals():
    print("flux_qubit_eigenvals()")
    flux_qubit = flux_qubit_initialize()
    evals_reference = np.asarray([-1.39408879, -1.16358534, -1.11509225, -0.98817209, -0.94604519, -0.88385204])
    assert np.allclose(evals_reference, flux_qubit.eigenvals(filename=TEMPDIR + 'test'))

def test_flux_qubit_eigenvecs():
    print("flux_qubit_eigenvecs()")
    flux_qubit = flux_qubit_initialize()
    evecs_reference = np.asarray([ 4.11828514e-15+0.00000000e+00j,  5.78962906e-14-2.72004641e-15j,
        7.33952749e-13-8.18370979e-14j,  7.95699479e-12-1.20907451e-12j,
        7.24374460e-11-9.38555136e-12j,  5.31777327e-10+2.13450026e-12j,
        2.91375822e-09+9.50373788e-10j,  9.96227605e-09+1.12992660e-08j,
        6.10670370e-09+6.73015788e-08j, -1.27065625e-07+2.00365949e-07j,
       -6.37047424e-07+9.70557072e-08j, -1.19259126e-06-1.17279776e-06j,
        5.13068816e-07-3.23073793e-06j,  5.75309136e-06-1.38723539e-06j,
        4.59397438e-06+6.72155287e-06j, -6.56781426e-06+6.26203360e-06j,
       -5.54816621e-06-4.20308219e-06j,  1.95914869e-06-3.68267073e-06j,
        1.77081349e-06+4.71711975e-07j,  2.68163774e-08+6.54583031e-07j,
       -1.63461763e-07+8.04681219e-08j,  5.78981307e-14+2.71696787e-15j,
        7.88820398e-13+5.20417043e-18j,  9.65236224e-12-8.86195522e-13j,
        1.01799846e-10-1.90966795e-11j,  9.17676907e-10-2.20029686e-10j,
        6.86713031e-09-1.29829262e-09j,  4.01640713e-08+9.36712294e-10j,
        1.63920620e-07+8.37599026e-08j,  3.42016721e-07+6.87577943e-07j,
       -4.56773348e-07+2.66778543e-06j, -5.41208353e-06+4.18825177e-06j,
       -1.53009216e-05-4.47945127e-06j, -1.22693967e-05-2.88026064e-05j,
        3.29722276e-05-3.55218783e-05j,  5.99200436e-05+2.25691366e-05j,
       -9.19050655e-06+5.82994515e-05j, -4.08792866e-05+3.96854082e-06j,
       -6.91301580e-06-1.96689882e-05j,  6.45362636e-06-5.28346455e-06j,
        2.29779439e-06+1.18179807e-06j,  2.68163774e-08+6.54583031e-07j,
        7.33941553e-13+8.18436031e-14j,  9.65239780e-12+8.86207774e-13j,
        1.12237620e-10-1.95156391e-18j,  1.12241419e-09-1.54067226e-10j,
        9.69764980e-09-2.65580954e-09j,  7.14289788e-08-2.34196230e-08j,
        4.28211706e-07-9.50352736e-08j,  1.91507630e-06+2.05328454e-07j,
        5.45558612e-06+4.63886254e-06j,  5.35404805e-06+2.40743767e-05j,
       -2.47006111e-05+5.66035478e-05j, -1.17914440e-04+3.36401576e-05j,
       -1.96181254e-04-1.39503441e-04j,  2.82396413e-05-3.27712241e-04j,
        3.79552861e-04-1.04142767e-04j,  1.73770093e-04+2.56505721e-04j,
       -1.19203348e-04+1.48356117e-04j, -8.08236070e-05-2.39996054e-05j,
       -4.08993661e-06-2.93993735e-05j,  6.45362636e-06-5.28346455e-06j,
        1.77081348e-06+4.71711975e-07j,  7.95697416e-12+1.20903634e-12j,
        1.01799872e-10+1.90967086e-11j,  1.12241418e-09+1.54067240e-10j,
        1.04859159e-08-2.12503626e-17j,  8.44391287e-08-1.76299342e-08j,
        5.91950121e-07-2.34992825e-07j,  3.52048836e-06-1.52468835e-06j,
        1.64926143e-05-3.89000485e-06j,  5.44055018e-05+1.36493698e-05j,
        1.04919056e-04+1.39921988e-04j,  2.94913626e-05+4.52247915e-04j,
       -4.85495494e-04+6.42787511e-04j, -1.42318224e-03-4.58386855e-05j,
       -1.08158576e-03-1.56402988e-03j,  1.04486960e-03-1.52820096e-03j,
        1.32685192e-03+2.30239545e-04j,  1.52731512e-04+6.65893881e-04j,
       -2.10502712e-04+1.80770233e-04j, -8.08236070e-05-2.39996054e-05j,
       -6.91301580e-06-1.96689882e-05j,  1.95914869e-06-3.68267073e-06j,
        7.24374892e-11+9.38550366e-12j,  9.17676881e-10+2.20029672e-10j,
        9.69764969e-09+2.65580958e-09j,  8.44391287e-08+1.76299343e-08j,
        6.18328827e-07+4.55364912e-18j,  3.94483471e-06-1.25361048e-06j,
        2.21907257e-05-1.24997983e-05j,  1.04061085e-04-5.71446929e-05j,
        3.66464487e-04-8.23139317e-05j,  8.78778526e-04+3.83766795e-04j,
        1.26023786e-03+2.13791763e-03j, -8.28018600e-06+4.54997708e-03j,
       -5.08715578e-03+4.02229449e-03j, -8.23235515e-03-2.47206439e-03j,
       -1.53401483e-03-6.79399647e-03j,  3.48549465e-03-2.74188992e-03j,
        1.88999922e-03+6.77649040e-04j,  1.52731512e-04+6.65893881e-04j,
       -1.19203348e-04+1.48356117e-04j, -4.08792866e-05+3.96854082e-06j,
       -5.54816621e-06-4.20308219e-06j,  5.31777402e-10-2.13452617e-12j,
        6.86713035e-09+1.29829264e-09j,  7.14289788e-08+2.34196230e-08j,
        5.91950122e-07+2.34992825e-07j,  3.94483471e-06+1.25361048e-06j,
        2.20602249e-05+1.99493200e-17j,  1.09920530e-04-5.20872649e-05j,
        4.84030783e-04-3.72820502e-04j,  1.69047767e-03-1.15170517e-03j,
        4.29066678e-03-9.06684011e-04j,  8.22245188e-03+4.44654490e-03j,
        1.05559654e-02+1.62606479e-02j, -2.37944223e-03+2.49788774e-02j,
       -2.48397152e-02+1.15897031e-02j, -1.87155813e-02-1.02869732e-02j,
       -6.66221033e-04-1.08096186e-02j,  3.48549465e-03-2.74188992e-03j,
        1.32685192e-03+2.30239545e-04j,  1.73770093e-04+2.56505721e-04j,
       -9.19050655e-06+5.82994515e-05j, -6.56781426e-06+6.26203360e-06j,
        2.91375828e-09-9.50373817e-10j,  4.01640714e-08-9.36712286e-10j,
        4.28211706e-07+9.50352736e-08j,  3.52048836e-06+1.52468835e-06j,
        2.21907257e-05+1.24997983e-05j,  1.09920530e-04+5.20872649e-05j,
        4.63498753e-04+3.03576608e-18j,  1.77794396e-03-1.15995755e-03j,
        5.64060899e-03-5.67941113e-03j,  1.28485754e-02-1.17504494e-02j,
        2.45492613e-02-7.26662571e-03j,  4.77528710e-02+2.02661274e-02j,
        4.59920879e-02+6.30300367e-02j, -1.56227733e-02+6.36270301e-02j,
       -4.25731095e-02+1.48567938e-02j, -1.87155813e-02-1.02869732e-02j,
       -1.53401483e-03-6.79399647e-03j,  1.04486960e-03-1.52820096e-03j,
        3.79552861e-04-1.04142767e-04j,  5.99200436e-05+2.25691366e-05j,
        4.59397438e-06+6.72155287e-06j,  9.96227611e-09-1.12992660e-08j,
        1.63920621e-07-8.37599027e-08j,  1.91507630e-06-2.05328454e-07j,
        1.64926143e-05+3.89000485e-06j,  1.04061085e-04+5.71446929e-05j,
        4.84030783e-04+3.72820502e-04j,  1.77794396e-03+1.15995755e-03j,
        5.87540630e-03+1.69135539e-17j,  1.66358692e-02-1.21853517e-02j,
        2.96512001e-02-3.92291801e-02j,  3.22345208e-02-6.00771841e-02j,
        7.34339414e-02-4.24194468e-02j,  1.38920660e-01+4.21616430e-02j,
        7.44683571e-02+1.05044286e-01j, -1.56227733e-02+6.36270301e-02j,
       -2.48397152e-02+1.15897031e-02j, -8.23235515e-03-2.47206439e-03j,
       -1.08158576e-03-1.56402988e-03j,  2.82396413e-05-3.27712241e-04j,
        3.29722276e-05-3.55218783e-05j,  5.75309136e-06-1.38723539e-06j,
        6.10670373e-09-6.73015788e-08j,  3.42016721e-07-6.87577943e-07j,
        5.45558612e-06-4.63886254e-06j,  5.44055018e-05-1.36493698e-05j,
        3.66464487e-04+8.23139317e-05j,  1.69047767e-03+1.15170517e-03j,
        5.64060899e-03+5.67941113e-03j,  1.66358692e-02+1.21853517e-02j,
        4.76674394e-02+2.30935063e-17j,  8.76498755e-02-4.97780528e-02j,
        4.81201967e-02-1.11232683e-01j,  3.84264461e-03-1.51746918e-01j,
        1.15906130e-01-8.46106227e-02j,  1.38920660e-01+4.21616430e-02j,
        4.59920879e-02+6.30300367e-02j, -2.37944223e-03+2.49788774e-02j,
       -5.08715578e-03+4.02229449e-03j, -1.42318224e-03-4.58386855e-05j,
       -1.96181254e-04-1.39503441e-04j, -1.22693967e-05-2.88026064e-05j,
        5.13068817e-07-3.23073793e-06j, -1.27065625e-07-2.00365949e-07j,
       -4.56773348e-07-2.66778543e-06j,  5.35404805e-06-2.40743767e-05j,
        1.04919056e-04-1.39921988e-04j,  8.78778526e-04-3.83766795e-04j,
        4.29066678e-03+9.06684011e-04j,  1.28485754e-02+1.17504494e-02j,
        2.96512001e-02+3.92291801e-02j,  8.76498755e-02+4.97780528e-02j,
        2.19947689e-01-2.05998413e-16j,  2.22557876e-01-6.71285658e-02j,
        1.92764563e-02-1.45982908e-01j,  3.84264461e-03-1.51746918e-01j,
        7.34339414e-02-4.24194468e-02j,  4.77528710e-02+2.02661274e-02j,
        1.05559654e-02+1.62606479e-02j, -8.28018600e-06+4.54997708e-03j,
       -4.85495494e-04+6.42787511e-04j, -1.17914440e-04+3.36401576e-05j,
       -1.53009216e-05-4.47945127e-06j, -1.19259126e-06-1.17279776e-06j,
       -6.37047424e-07-9.70557072e-08j, -5.41208353e-06-4.18825177e-06j,
       -2.47006111e-05-5.66035478e-05j,  2.94913626e-05-4.52247915e-04j,
        1.26023786e-03-2.13791763e-03j,  8.22245188e-03-4.44654490e-03j,
        2.45492613e-02+7.26662571e-03j,  3.22345208e-02+6.00771841e-02j,
        4.81201967e-02+1.11232683e-01j,  2.22557876e-01+6.71285658e-02j,
        3.99080684e-01-3.09322880e-16j,  2.22557876e-01-6.71285658e-02j,
        4.81201967e-02-1.11232683e-01j,  3.22345208e-02-6.00771841e-02j,
        2.45492613e-02-7.26662571e-03j,  8.22245188e-03+4.44654490e-03j,
        1.26023786e-03+2.13791763e-03j,  2.94913626e-05+4.52247915e-04j,
       -2.47006111e-05+5.66035478e-05j, -5.41208353e-06+4.18825177e-06j,
       -6.37047424e-07+9.70557072e-08j, -1.19259126e-06+1.17279776e-06j,
       -1.53009216e-05+4.47945127e-06j, -1.17914440e-04-3.36401576e-05j,
       -4.85495494e-04-6.42787511e-04j, -8.28018600e-06-4.54997708e-03j,
        1.05559654e-02-1.62606479e-02j,  4.77528710e-02-2.02661274e-02j,
        7.34339414e-02+4.24194468e-02j,  3.84264461e-03+1.51746918e-01j,
        1.92764563e-02+1.45982908e-01j,  2.22557876e-01+6.71285658e-02j,
        2.19947689e-01+2.39879731e-16j,  8.76498755e-02-4.97780528e-02j,
        2.96512001e-02-3.92291801e-02j,  1.28485754e-02-1.17504494e-02j,
        4.29066678e-03-9.06684011e-04j,  8.78778526e-04+3.83766795e-04j,
        1.04919056e-04+1.39921988e-04j,  5.35404805e-06+2.40743767e-05j,
       -4.56773348e-07+2.66778543e-06j, -1.27065625e-07+2.00365949e-07j,
        5.13068817e-07+3.23073793e-06j, -1.22693967e-05+2.88026064e-05j,
       -1.96181254e-04+1.39503441e-04j, -1.42318224e-03+4.58386855e-05j,
       -5.08715578e-03-4.02229449e-03j, -2.37944223e-03-2.49788774e-02j,
        4.59920879e-02-6.30300367e-02j,  1.38920660e-01-4.21616430e-02j,
        1.15906130e-01+8.46106227e-02j,  3.84264461e-03+1.51746918e-01j,
        4.81201967e-02+1.11232683e-01j,  8.76498755e-02+4.97780528e-02j,
        4.76674394e-02+1.13339785e-16j,  1.66358692e-02-1.21853517e-02j,
        5.64060899e-03-5.67941113e-03j,  1.69047767e-03-1.15170517e-03j,
        3.66464487e-04-8.23139317e-05j,  5.44055018e-05+1.36493698e-05j,
        5.45558612e-06+4.63886254e-06j,  3.42016721e-07+6.87577943e-07j,
        6.10670370e-09+6.73015788e-08j,  5.75309136e-06+1.38723539e-06j,
        3.29722276e-05+3.55218783e-05j,  2.82396413e-05+3.27712241e-04j,
       -1.08158576e-03+1.56402988e-03j, -8.23235515e-03+2.47206439e-03j,
       -2.48397152e-02-1.15897031e-02j, -1.56227733e-02-6.36270301e-02j,
        7.44683571e-02-1.05044286e-01j,  1.38920660e-01-4.21616430e-02j,
        7.34339414e-02+4.24194468e-02j,  3.22345208e-02+6.00771841e-02j,
        2.96512001e-02+3.92291801e-02j,  1.66358692e-02+1.21853517e-02j,
        5.87540630e-03+3.86763714e-17j,  1.77794396e-03-1.15995755e-03j,
        4.84030783e-04-3.72820502e-04j,  1.04061085e-04-5.71446929e-05j,
        1.64926143e-05-3.89000485e-06j,  1.91507630e-06+2.05328454e-07j,
        1.63920620e-07+8.37599027e-08j,  9.96227614e-09+1.12992659e-08j,
        4.59397438e-06-6.72155287e-06j,  5.99200436e-05-2.25691366e-05j,
        3.79552861e-04+1.04142767e-04j,  1.04486960e-03+1.52820096e-03j,
       -1.53401483e-03+6.79399647e-03j, -1.87155813e-02+1.02869732e-02j,
       -4.25731095e-02-1.48567938e-02j, -1.56227733e-02-6.36270301e-02j,
        4.59920879e-02-6.30300367e-02j,  4.77528710e-02-2.02661274e-02j,
        2.45492613e-02+7.26662571e-03j,  1.28485754e-02+1.17504494e-02j,
        5.64060899e-03+5.67941113e-03j,  1.77794396e-03+1.15995755e-03j,
        4.63498753e-04+5.35401585e-18j,  1.09920530e-04-5.20872649e-05j,
        2.21907257e-05-1.24997983e-05j,  3.52048836e-06-1.52468835e-06j,
        4.28211706e-07-9.50352736e-08j,  4.01640714e-08+9.36712312e-10j,
        2.91375827e-09+9.50373810e-10j, -6.56781426e-06-6.26203360e-06j,
       -9.19050655e-06-5.82994515e-05j,  1.73770093e-04-2.56505721e-04j,
        1.32685192e-03-2.30239545e-04j,  3.48549465e-03+2.74188992e-03j,
       -6.66221033e-04+1.08096186e-02j, -1.87155813e-02+1.02869732e-02j,
       -2.48397152e-02-1.15897031e-02j, -2.37944223e-03-2.49788774e-02j,
        1.05559654e-02-1.62606479e-02j,  8.22245188e-03-4.44654490e-03j,
        4.29066678e-03+9.06684011e-04j,  1.69047767e-03+1.15170517e-03j,
        4.84030783e-04+3.72820502e-04j,  1.09920530e-04+5.20872649e-05j,
        2.20602249e-05+4.54569364e-19j,  3.94483471e-06-1.25361048e-06j,
        5.91950121e-07-2.34992825e-07j,  7.14289788e-08-2.34196230e-08j,
        6.86713033e-09-1.29829265e-09j,  5.31777313e-10+2.13452177e-12j,
       -5.54816621e-06+4.20308219e-06j, -4.08792866e-05-3.96854082e-06j,
       -1.19203348e-04-1.48356117e-04j,  1.52731512e-04-6.65893881e-04j,
        1.88999922e-03-6.77649040e-04j,  3.48549465e-03+2.74188992e-03j,
       -1.53401483e-03+6.79399647e-03j, -8.23235515e-03+2.47206439e-03j,
       -5.08715578e-03-4.02229449e-03j, -8.28018600e-06-4.54997708e-03j,
        1.26023786e-03-2.13791763e-03j,  8.78778526e-04-3.83766795e-04j,
        3.66464487e-04+8.23139317e-05j,  1.04061085e-04+5.71446929e-05j,
        2.21907257e-05+1.24997983e-05j,  3.94483471e-06+1.25361048e-06j,
        6.18328827e-07+3.26232092e-20j,  8.44391287e-08-1.76299342e-08j,
        9.69764967e-09-2.65580961e-09j,  9.17676896e-10-2.20029696e-10j,
        7.24374764e-11-9.38548814e-12j,  1.95914869e-06+3.68267073e-06j,
       -6.91301580e-06+1.96689882e-05j, -8.08236070e-05+2.39996054e-05j,
       -2.10502712e-04-1.80770233e-04j,  1.52731512e-04-6.65893881e-04j,
        1.32685192e-03-2.30239545e-04j,  1.04486960e-03+1.52820096e-03j,
       -1.08158576e-03+1.56402988e-03j, -1.42318224e-03+4.58386855e-05j,
       -4.85495494e-04-6.42787511e-04j,  2.94913626e-05-4.52247915e-04j,
        1.04919056e-04-1.39921988e-04j,  5.44055018e-05-1.36493698e-05j,
        1.64926143e-05+3.89000485e-06j,  3.52048836e-06+1.52468835e-06j,
        5.91950121e-07+2.34992825e-07j,  8.44391287e-08+1.76299342e-08j,
        1.04859160e-08+3.31401641e-20j,  1.12241418e-09-1.54067264e-10j,
        1.01799869e-10-1.90967189e-11j,  7.95698067e-12-1.20903146e-12j,
        1.77081349e-06-4.71711975e-07j,  6.45362636e-06+5.28346455e-06j,
       -4.08993661e-06+2.93993735e-05j, -8.08236070e-05+2.39996054e-05j,
       -1.19203348e-04-1.48356117e-04j,  1.73770093e-04-2.56505721e-04j,
        3.79552861e-04+1.04142767e-04j,  2.82396413e-05+3.27712241e-04j,
       -1.96181254e-04+1.39503441e-04j, -1.17914440e-04-3.36401576e-05j,
       -2.47006111e-05-5.66035478e-05j,  5.35404805e-06-2.40743767e-05j,
        5.45558612e-06-4.63886254e-06j,  1.91507630e-06-2.05328454e-07j,
        4.28211706e-07+9.50352736e-08j,  7.14289788e-08+2.34196230e-08j,
        9.69764967e-09+2.65580961e-09j,  1.12241418e-09+1.54067264e-10j,
        1.12237635e-10+2.45639555e-20j,  9.65236007e-12-8.86225229e-13j,
        7.33907726e-13-8.18288579e-14j,  2.68163774e-08-6.54583031e-07j,
        2.29779439e-06-1.18179807e-06j,  6.45362636e-06+5.28346455e-06j,
       -6.91301580e-06+1.96689882e-05j, -4.08792866e-05-3.96854082e-06j,
       -9.19050655e-06-5.82994515e-05j,  5.99200436e-05-2.25691366e-05j,
        3.29722276e-05+3.55218783e-05j, -1.22693967e-05+2.88026064e-05j,
       -1.53009216e-05+4.47945127e-06j, -5.41208353e-06-4.18825177e-06j,
       -4.56773348e-07-2.66778543e-06j,  3.42016721e-07-6.87577943e-07j,
        1.63920620e-07-8.37599027e-08j,  4.01640714e-08-9.36712312e-10j,
        6.86713033e-09+1.29829265e-09j,  9.17676896e-10+2.20029695e-10j,
        1.01799872e-10+1.90967190e-11j,  9.65236831e-12+8.86224403e-13j,
        7.88808255e-13+5.42101086e-19j,  5.79328252e-14-2.71701064e-15j,
       -1.63461763e-07-8.04681219e-08j,  2.68163774e-08-6.54583031e-07j,
        1.77081349e-06-4.71711975e-07j,  1.95914869e-06+3.68267073e-06j,
       -5.54816621e-06+4.20308219e-06j, -6.56781426e-06-6.26203360e-06j,
        4.59397438e-06-6.72155287e-06j,  5.75309136e-06+1.38723539e-06j,
        5.13068817e-07+3.23073793e-06j, -1.19259126e-06+1.17279776e-06j,
       -6.37047424e-07-9.70557072e-08j, -1.27065625e-07-2.00365949e-07j,
        6.10670370e-09-6.73015788e-08j,  9.96227614e-09-1.12992659e-08j,
        2.91375827e-09-9.50373810e-10j,  5.31777313e-10-2.13452177e-12j,
        7.24374756e-11+9.38548708e-12j,  7.95698132e-12+1.20902908e-12j,
        7.33909461e-13+8.18338453e-14j,  5.79224169e-14+2.72481690e-15j,
        4.09394740e-15+1.19262239e-18j])
    evecs_reference = standardize_phases(evecs_reference)

    _, evecs_tst = flux_qubit.eigensys(filename=TEMPDIR + 'test')
    evecs_calculated = evecs_tst.T[3]
    evecs_calculated = standardize_phases(evecs_calculated)
    assert np.allclose(evecs_reference, evecs_calculated)

def test_flux_qubit_plot_evals_vs_paramvals_flux():
    print("flux_qubit_plot_evals_vs_paramvals_flux()")
    flux_qubit = flux_qubit_initialize()
    flux_list = np.linspace(.46, .54, 40)
    flux_qubit.plot_evals_vs_paramvals('flux', flux_list, evals_count=4, subtract_ground=False)

def test_flux_qubit_get_spectrum_vs_paramvals():
    print("flux_qubit_get_spectrum_vs_paramvals()")
    flux_qubit = flux_qubit_initialize()
    reference_evals = np.asarray([[-1.35190341, -1.12997203, -1.07587151, -1.01071222],
       [-1.34319798, -1.12319442, -1.06777423, -1.0181728 ],
       [-1.33447602, -1.11648152, -1.05965997, -1.02596366],
       [-1.32573809, -1.10985535, -1.05152909, -1.03395824],
       [-1.3169848 , -1.10335469, -1.04338192, -1.04205408],
       [-1.30821676, -1.09705931, -1.05013271, -1.0352188 ],
       [-1.29943463, -1.09117289, -1.05796787, -1.02704011],
       [-1.29063914, -1.08638257, -1.06486105, -1.01884622],
       [-1.28183104, -1.08517255, -1.06832522, -1.01063749],
       [-1.27301115, -1.08979467, -1.06611332, -1.00241434],
       [-1.26418038, -1.09717824, -1.06130827, -0.99417716],
       [-1.25533973, -1.10537452, -1.05587979, -0.98592637],
       [-1.24649034, -1.11387504, -1.05036807, -0.97766243],
       [-1.23763355, -1.12252136, -1.0449779 , -0.96938579],
       [-1.22877101, -1.13124926, -1.03984011, -0.96109694],
       [-1.21990494, -1.14002669, -1.03508177, -0.96179244],
       [-1.2110388 , -1.14883418, -1.03085548, -0.96468426],
       [-1.20217948, -1.15765557, -1.027355  , -0.96720117],
       [-1.193348  , -1.16646432, -1.02481319, -0.9690958 ],
       [-1.1847083 , -1.17509338, -1.02346441, -0.97012361],
       [-1.1847083 , -1.17509338, -1.02346441, -0.97012361],
       [-1.193348  , -1.16646432, -1.02481319, -0.9690958 ],
       [-1.20217948, -1.15765557, -1.027355  , -0.96720117],
       [-1.2110388 , -1.14883418, -1.03085548, -0.96468426],
       [-1.21990494, -1.14002669, -1.03508177, -0.96179244],
       [-1.22877101, -1.13124926, -1.03984011, -0.96109694],
       [-1.23763355, -1.12252136, -1.0449779 , -0.96938579],
       [-1.24649034, -1.11387504, -1.05036807, -0.97766243],
       [-1.25533973, -1.10537452, -1.05587979, -0.98592637],
       [-1.26418038, -1.09717824, -1.06130827, -0.99417716],
       [-1.27301115, -1.08979467, -1.06611332, -1.00241434],
       [-1.28183104, -1.08517255, -1.06832522, -1.01063749],
       [-1.29063914, -1.08638257, -1.06486105, -1.01884622],
       [-1.29943463, -1.09117289, -1.05796787, -1.02704011],
       [-1.30821676, -1.09705931, -1.05013271, -1.0352188 ],
       [-1.3169848 , -1.10335469, -1.04338192, -1.04205408],
       [-1.32573809, -1.10985535, -1.05152909, -1.03395824],
       [-1.33447602, -1.11648152, -1.05965997, -1.02596366],
       [-1.34319798, -1.12319442, -1.06777423, -1.0181728 ],
       [-1.35190341, -1.12997203, -1.07587151, -1.01071222]])
    flux_list = np.linspace(.46, .54, 40)
    calculated_evals = flux_qubit.get_spectrum_vs_paramvals('flux', flux_list, evals_count=4, subtract_ground=False,
                                                    get_eigenstates=True)
    assert np.allclose(reference_evals, calculated_evals.energy_table)

def test_flux_qubit_get_spectrum_vs_paramvals_filewrite():
    print("flux_qubit_get_spectrum_vs_paramvals_filewrite()")
    flux_qubit = flux_qubit_initialize()
    flux_list = np.linspace(.46, .54, 40)
    calculated_evals = flux_qubit.get_spectrum_vs_paramvals('flux', flux_list, evals_count=4, subtract_ground=False,
                                                            get_eigenstates=True, filename=TEMPDIR + 'tst_tmp')

def test_flux_qubit_plot_wavefunction():
    print("flux_qubit_plot_wavefunction()")
    flux_qubit = flux_qubit_initialize()
    flux_qubit.plot_wavefunction(esys=None, which=6, mode='real')

def test_flux_qubit_matrixelement_table():
    print("flux_qubit_matrixelement_table()")
    evalcount = 16
    reference_matrix = np.asarray([[ 1.11958672e-15+3.46944695e-18j,  2.02903917e-15+1.04375810e+00j,
         7.21265604e-01-1.66672966e-15j, -2.15293349e-15-2.04477138e-01j,
         1.95187864e-16-1.26173334e-01j,  1.56042037e-01+1.44154025e-15j,
        -6.42902767e-16-1.28111133e-01j,  2.67375280e-16-9.60599265e-02j,
         2.90161242e-16-1.64807833e-02j,  3.24665495e-03+2.54035564e-16j,
         5.64890441e-03+2.78217913e-16j,  5.75636965e-16+7.47481281e-04j,
        -3.04472089e-03-9.84809928e-16j, -5.76032797e-17-1.02483330e-03j,
         3.49622670e-04-1.63721250e-16j,  2.66570912e-16+6.50521789e-03j],
       [ 2.02903917e-15-1.04375810e+00j, -6.40330300e-16-0.00000000e+00j,
        -3.43627528e-02+4.12524604e-15j, -6.21816065e-16+1.00299316e+00j,
         1.03373242e-14+7.92102952e-01j,  7.00773432e-01-3.65423813e-15j,
         1.66965981e-15+1.05803691e-02j,  3.29314780e-15+5.30744064e-01j,
         1.67321615e-15+2.44352717e-01j, -1.69185235e-02-3.64773939e-15j,
        -2.38358490e-01+3.58295177e-15j,  8.59997024e-16-8.05597875e-02j,
         5.99497227e-02+8.06133785e-16j, -4.27284628e-16-1.23877918e-01j,
         4.14188286e-02-1.52255982e-15j,  1.65677152e-15-2.61945597e-03j],
       [ 7.21265604e-01+1.66672966e-15j, -3.43627528e-02-4.12524604e-15j,
         9.26969443e-16+1.73472348e-18j, -5.65897148e-02-2.95746923e-15j,
        -7.96222500e-02+5.08271051e-15j,  4.96698866e-15+1.04028897e+00j,
         9.02436254e-01-4.98218992e-15j,  3.35192042e-01-3.61802348e-15j,
         8.82640835e-02-2.15266533e-15j, -2.09613818e-15+5.38357657e-02j,
         7.76130504e-16+5.72089735e-02j,  2.23676024e-01+7.27412949e-15j,
         6.98049051e-15-3.15572854e-01j,  8.51194655e-02+1.09802526e-15j,
         1.05051278e-15-4.72744975e-02j,  3.30790509e-02-1.95843955e-15j],
       [-2.15293349e-15+2.04477138e-01j, -6.21816065e-16-1.00299316e+00j,
        -5.65897148e-02+2.95746923e-15j,  3.95717322e-15-1.38777878e-17j,
        -1.85546907e-15-5.25394288e-01j, -7.83777367e-02+1.64406990e-15j,
        -2.35098592e-15+1.65591530e-01j, -6.41297621e-16-4.74816511e-01j,
        -1.39062482e-14-7.72515609e-01j, -4.40015699e-01+1.63779225e-14j,
        -5.02262212e-01-2.35261604e-16j, -3.02020151e-15-3.54916530e-03j,
         1.82092406e-01-5.86280130e-16j,  4.85274268e-15+2.61761834e-01j,
        -2.56902189e-01+4.23372310e-16j, -3.36390843e-15-2.31990812e-02j],
       [ 1.95187864e-16+1.26173334e-01j,  1.03373242e-14-7.92102952e-01j,
        -7.96222500e-02-5.08271051e-15j, -1.85546907e-15+5.25394288e-01j,
        -2.33730252e-15-0.00000000e+00j, -5.15476095e-02-1.67104413e-16j,
        -2.49251406e-15+4.59141773e-01j,  2.94040784e-15-1.12897586e+00j,
         1.50770498e-14-2.51076149e-01j,  4.83153296e-01+9.94245328e-15j,
        -4.02136394e-01-1.63740265e-14j,  2.68520606e-15-1.70970720e-01j,
         1.79870083e-01+2.61391583e-15j,  6.59486547e-16+4.83062793e-01j,
         1.65961951e-01+1.13111773e-14j, -4.60586933e-15-2.75728287e-01j],
       [ 1.56042037e-01-1.44154025e-15j,  7.00773432e-01+3.65423813e-15j,
         4.96698866e-15-1.04028897e+00j, -7.83777367e-02-1.64406990e-15j,
        -5.15476095e-02+1.67104413e-16j, -3.20847977e-15-0.00000000e+00j,
        -3.62349350e-02-2.59995707e-15j,  1.83561127e-02+1.16897108e-14j,
         9.72529120e-02+5.35788753e-15j, -1.28854252e-14-8.81046503e-02j,
         1.19677319e-14-1.17170433e+00j,  7.48056840e-01+9.11745192e-16j,
         7.97576423e-15+5.41757204e-01j,  4.83115854e-01-2.15889492e-15j,
        -6.68595729e-16+4.38102848e-01j,  2.64299374e-01+1.64446696e-14j],
       [-6.42902767e-16+1.28111133e-01j,  1.66965981e-15-1.05803691e-02j,
         9.02436254e-01+4.98218992e-15j, -2.35098592e-15-1.65591530e-01j,
        -2.49251406e-15-4.59141773e-01j, -3.62349350e-02+2.59995707e-15j,
        -4.94283403e-15+6.93889390e-18j, -9.63898651e-16-1.83152560e-01j,
        -1.60769202e-15+4.65641499e-01j, -3.37791561e-02-1.22346924e-14j,
         3.59831451e-01+6.38069223e-15j, -1.99102635e-14+8.17486343e-01j,
         9.52015590e-01+1.77129918e-14j,  2.93770066e-15+3.25942918e-01j,
        -1.46599295e-01+2.97930566e-15j,  8.28725983e-15+2.55746002e-01j],
       [ 2.67375280e-16+9.60599265e-02j,  3.29314780e-15-5.30744064e-01j,
         3.35192042e-01+3.61802348e-15j, -6.41297621e-16+4.74816511e-01j,
         2.94040784e-15+1.12897586e+00j,  1.83561127e-02-1.16897108e-14j,
        -9.63898651e-16+1.83152560e-01j,  9.03552294e-16-0.00000000e+00j,
        -7.41007098e-15-1.33622315e+00j, -1.34401463e-01+3.01618326e-14j,
         3.36092496e-02-1.64173386e-15j, -5.05930986e-15+2.87760466e-01j,
         2.72663488e-01+6.20547666e-15j, -1.05863717e-14+5.01733793e-01j,
         5.81029900e-01+1.55081737e-14j,  1.44610001e-14+1.72383204e-02j],
       [ 2.90161242e-16+1.64807833e-02j,  1.67321615e-15-2.44352717e-01j,
         8.82640835e-02+2.15266533e-15j, -1.39062482e-14+7.72515609e-01j,
         1.50770498e-14+2.51076149e-01j,  9.72529120e-02-5.35788753e-15j,
        -1.60769202e-15-4.65641499e-01j, -7.41007098e-15+1.33622315e+00j,
         9.61578805e-16-0.00000000e+00j,  5.11603978e-02-2.39591340e-15j,
        -1.00015140e-01-1.83185710e-15j,  1.32781234e-14-8.23302381e-01j,
        -6.04153383e-02-1.79606549e-14j, -1.19201989e-14+1.51388016e+00j,
        -1.57881677e-01-5.99907143e-15j,  1.08518553e-14-4.95101632e-01j],
       [ 3.24665495e-03-2.54035564e-16j, -1.69185235e-02+3.64773939e-15j,
        -2.09613818e-15-5.38357657e-02j, -4.40015699e-01-1.63779225e-14j,
         4.83153296e-01-9.94245328e-15j, -1.28854252e-14+8.81046503e-02j,
        -3.37791561e-02+1.22346924e-14j, -1.34401463e-01-3.01618326e-14j,
         5.11603978e-02+2.39591340e-15j, -8.45465668e-15-0.00000000e+00j,
         9.69688089e-17-3.51915710e-01j,  2.09056766e-01+2.10012687e-14j,
         3.73078607e-15+4.14307353e-02j, -7.07364737e-02-2.65016899e-14j,
         3.54277775e-15-5.66143031e-01j, -5.22339594e-01+2.03503198e-14j],
       [ 5.64890441e-03-2.78217913e-16j, -2.38358490e-01-3.58295177e-15j,
         7.76130504e-16-5.72089735e-02j, -5.02262212e-01+2.35261604e-16j,
        -4.02136394e-01+1.63740265e-14j,  1.19677319e-14+1.17170433e+00j,
         3.59831451e-01-6.38069223e-15j,  3.36092496e-02+1.64173386e-15j,
        -1.00015140e-01+1.83185710e-15j,  9.69688089e-17+3.51915710e-01j,
        -5.17506840e-16-0.00000000e+00j,  1.64176107e-01+2.38836516e-15j,
         1.92997176e-15-2.85105690e-01j, -7.10429586e-02-3.67287419e-15j,
        -1.65340411e-15+5.39224928e-01j, -3.75121961e-01+3.19915145e-14j],
       [ 5.75636965e-16-7.47481281e-04j,  8.59997024e-16+8.05597875e-02j,
         2.23676024e-01-7.27412949e-15j, -3.02020151e-15+3.54916530e-03j,
         2.68520606e-15+1.70970720e-01j,  7.48056840e-01-9.11745192e-16j,
        -1.99102635e-14-8.17486343e-01j, -5.05930986e-15-2.87760466e-01j,
         1.32781234e-14+8.23302381e-01j,  2.09056766e-01-2.10012687e-14j,
         1.64176107e-01-2.38836516e-15j, -1.12966042e-15+1.38777878e-17j,
         5.40033369e-02+2.15563718e-17j,  3.91696710e-15-7.77440029e-02j,
        -1.79960347e-01+9.04252891e-15j,  1.86199551e-14+3.55708107e-01j],
       [-3.04472089e-03+9.84809928e-16j,  5.99497227e-02-8.06133785e-16j,
         6.98049051e-15+3.15572854e-01j,  1.82092406e-01+5.86280130e-16j,
         1.79870083e-01-2.61391583e-15j,  7.97576423e-15-5.41757204e-01j,
         9.52015590e-01-1.77129918e-14j,  2.72663488e-01-6.20547666e-15j,
        -6.04153383e-02+1.79606549e-14j,  3.73078607e-15-4.14307353e-02j,
         1.92997176e-15+2.85105690e-01j,  5.40033369e-02-2.15563718e-17j,
         2.89443482e-15-2.08166817e-17j, -1.29889241e-01+1.16587560e-14j,
         8.31070172e-16-7.72748556e-01j, -4.64411866e-01+5.18663398e-15j],
       [-5.76032797e-17+1.02483330e-03j, -4.27284628e-16+1.23877918e-01j,
         8.51194655e-02-1.09802526e-15j,  4.85274268e-15-2.61761834e-01j,
         6.59486547e-16-4.83062793e-01j,  4.83115854e-01+2.15889492e-15j,
         2.93770066e-15-3.25942918e-01j, -1.05863717e-14-5.01733793e-01j,
        -1.19201989e-14-1.51388016e+00j, -7.07364737e-02+2.65016899e-14j,
        -7.10429586e-02+3.67287419e-15j,  3.91696710e-15+7.77440029e-02j,
        -1.29889241e-01-1.16587560e-14j,  3.90131612e-16-0.00000000e+00j,
        -9.06627005e-02-3.49853074e-15j,  3.61144975e-15+7.34502844e-01j],
       [ 3.49622670e-04+1.63721250e-16j,  4.14188286e-02+1.52255982e-15j,
         1.05051278e-15+4.72744975e-02j, -2.56902189e-01-4.23372310e-16j,
         1.65961951e-01-1.13111773e-14j, -6.68595729e-16-4.38102848e-01j,
        -1.46599295e-01-2.97930566e-15j,  5.81029900e-01-1.55081737e-14j,
        -1.57881677e-01+5.99907143e-15j,  3.54277775e-15+5.66143031e-01j,
        -1.65340411e-15-5.39224928e-01j, -1.79960347e-01-9.04252891e-15j,
         8.31070172e-16+7.72748556e-01j, -9.06627005e-02+3.49853074e-15j,
        -1.64603440e-15+2.77555756e-17j,  2.72285381e-01-3.44541458e-14j],
       [ 2.66570912e-16-6.50521789e-03j,  1.65677152e-15+2.61945597e-03j,
         3.30790509e-02+1.95843955e-15j, -3.36390843e-15+2.31990812e-02j,
        -4.60586933e-15+2.75728287e-01j,  2.64299374e-01-1.64446696e-14j,
         8.28725983e-15-2.55746002e-01j,  1.44610001e-14-1.72383204e-02j,
         1.08518553e-14+4.95101632e-01j, -5.22339594e-01-2.03503198e-14j,
        -3.75121961e-01-3.19915145e-14j,  1.86199551e-14-3.55708107e-01j,
        -4.64411866e-01-5.18663398e-15j,  3.61144975e-15-7.34502844e-01j,
         2.72285381e-01+3.44541458e-14j,  1.91741773e-14-0.00000000e+00j]])

    flux_qubit = flux_qubit_initialize()
    calculated_matrix = flux_qubit.matrixelement_table('n_1_operator', esys=None, evals_count=evalcount)
    assert np.allclose(np.abs(reference_matrix), np.abs(calculated_matrix))

def test_flux_qubit_plot_matrixelements():
    print("flux_qubit_plot_matrixelements()")
    flux_qubit = flux_qubit_initialize()
    flux_qubit.plot_matrixelements('n_1_operator', evals_count=10)

def test_flux_qubit_print_matrixelements():
    print("flux_qubit_print_matrixelements()")
    flux_qubit = flux_qubit_initialize()
    mat_data = flux_qubit.matrixelement_table('n_1_operator')
    plot.print_matrix(abs(mat_data))

def test_flux_qubit_plot_matelem_vs_paramvals():
    print("flux_qubit_plot_matelem_vs_paramvals()")
    flux_qubit = flux_qubit_initialize()
    flux_list = np.linspace(.49, .51, 40)
    fig, ax = flux_qubit.plot_matelem_vs_paramvals('n_1_operator', 'flux', flux_list,
                                                   select_elems=[(0, 0), (1, 4), (1, 0)], filename=TEMPDIR + 'test')
