"use strict";
/**
 * Copyright 2017 Google Inc. All rights reserved.
 * Modifications copyright (c) Microsoft Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CRPage = void 0;
const dom = require("../dom");
const helper_1 = require("../helper");
const network = require("../network");
const crConnection_1 = require("./crConnection");
const crExecutionContext_1 = require("./crExecutionContext");
const crNetworkManager_1 = require("./crNetworkManager");
const page_1 = require("../page");
const crProtocolHelper_1 = require("./crProtocolHelper");
const dialog = require("../dialog");
const path = require("path");
const crInput_1 = require("./crInput");
const crAccessibility_1 = require("./crAccessibility");
const crCoverage_1 = require("./crCoverage");
const crPdf_1 = require("./crPdf");
const console_1 = require("../console");
const sourceMap = require("../../utils/sourceMap");
const stackTrace_1 = require("../../utils/stackTrace");
const utils_1 = require("../../utils/utils");
const videoRecorder_1 = require("./videoRecorder");
const UTILITY_WORLD_NAME = '__playwright_utility_world__';
class CRPage {
    constructor(client, targetId, browserContext, opener, hasUIWindow) {
        this._sessions = new Map();
        this._initializedPage = null;
        // Holds window features for the next popup being opened via window.open,
        // until the popup target arrives. This could be racy if two oopifs
        // simultaneously call window.open with window features: the order
        // of their Page.windowOpen events is not guaranteed to match the order
        // of new popup targets.
        this._nextWindowOpenPopupFeatures = [];
        this._targetId = targetId;
        this._opener = opener;
        this.rawKeyboard = new crInput_1.RawKeyboardImpl(client, browserContext._browser._isMac);
        this.rawMouse = new crInput_1.RawMouseImpl(client);
        this.rawTouchscreen = new crInput_1.RawTouchscreenImpl(client);
        this._pdf = new crPdf_1.CRPDF(client);
        this._coverage = new crCoverage_1.CRCoverage(client);
        this._browserContext = browserContext;
        this._page = new page_1.Page(this, browserContext);
        this._mainFrameSession = new FrameSession(this, client, targetId, null);
        this._sessions.set(targetId, this._mainFrameSession);
        client.once(crConnection_1.CRSessionEvents.Disconnected, () => this._page._didDisconnect());
        if (opener && !browserContext._options.noDefaultViewport) {
            const features = opener._nextWindowOpenPopupFeatures.shift() || [];
            const viewportSize = helper_1.helper.getViewportSizeFromWindowFeatures(features);
            if (viewportSize)
                this._page._state.viewportSize = viewportSize;
        }
        this._pagePromise = this._mainFrameSession._initialize(hasUIWindow).then(() => this._initializedPage = this._page).catch(e => e);
    }
    async _forAllFrameSessions(cb) {
        await Promise.all(Array.from(this._sessions.values()).map(frame => cb(frame)));
    }
    _sessionForFrame(frame) {
        // Frame id equals target id.
        while (!this._sessions.has(frame._id)) {
            const parent = frame.parentFrame();
            if (!parent)
                throw new Error(`Frame has been detached.`);
            frame = parent;
        }
        return this._sessions.get(frame._id);
    }
    _sessionForHandle(handle) {
        const frame = handle._context.frame;
        return this._sessionForFrame(frame);
    }
    async pageOrError() {
        return this._pagePromise;
    }
    didClose() {
        for (const session of this._sessions.values())
            session.dispose();
        this._page._didClose();
        this._mainFrameSession._stopScreencast().catch(() => { });
    }
    async navigateFrame(frame, url, referrer) {
        return this._sessionForFrame(frame)._navigate(frame, url, referrer);
    }
    async exposeBinding(binding) {
        await this._forAllFrameSessions(frame => frame._initBinding(binding));
        await Promise.all(this._page.frames().map(frame => frame._evaluateExpression(binding.source, false, {}).catch(e => { })));
    }
    async updateExtraHTTPHeaders() {
        await this._forAllFrameSessions(frame => frame._updateExtraHTTPHeaders(false));
    }
    async updateGeolocation() {
        await this._forAllFrameSessions(frame => frame._updateGeolocation(false));
    }
    async updateOffline() {
        await this._forAllFrameSessions(frame => frame._updateOffline(false));
    }
    async updateHttpCredentials() {
        await this._forAllFrameSessions(frame => frame._updateHttpCredentials(false));
    }
    async setViewportSize(viewportSize) {
        utils_1.assert(this._page._state.viewportSize === viewportSize);
        await this._mainFrameSession._updateViewport();
    }
    async bringToFront() {
        await this._mainFrameSession._client.send('Page.bringToFront');
    }
    async updateEmulateMedia() {
        await this._forAllFrameSessions(frame => frame._updateEmulateMedia(false));
    }
    async updateRequestInterception() {
        await this._forAllFrameSessions(frame => frame._updateRequestInterception(false));
    }
    async setFileChooserIntercepted(enabled) {
        await this._forAllFrameSessions(frame => frame._setFileChooserIntercepted(enabled));
    }
    async opener() {
        if (!this._opener)
            return null;
        const openerPage = await this._opener.pageOrError();
        if (openerPage instanceof page_1.Page && !openerPage.isClosed())
            return openerPage;
        return null;
    }
    async reload() {
        await this._mainFrameSession._client.send('Page.reload');
    }
    async _go(delta) {
        const history = await this._mainFrameSession._client.send('Page.getNavigationHistory');
        const entry = history.entries[history.currentIndex + delta];
        if (!entry)
            return false;
        await this._mainFrameSession._client.send('Page.navigateToHistoryEntry', { entryId: entry.id });
        return true;
    }
    goBack() {
        return this._go(-1);
    }
    goForward() {
        return this._go(+1);
    }
    async evaluateOnNewDocument(source) {
        await this._forAllFrameSessions(frame => frame._evaluateOnNewDocument(source));
    }
    async closePage(runBeforeUnload) {
        if (runBeforeUnload)
            await this._mainFrameSession._client.send('Page.close');
        else
            await this._browserContext._browser._closePage(this);
    }
    canScreenshotOutsideViewport() {
        return false;
    }
    async setBackgroundColor(color) {
        await this._mainFrameSession._client.send('Emulation.setDefaultBackgroundColorOverride', { color });
    }
    async startScreencast(options) {
        await this._mainFrameSession._startScreencast(utils_1.createGuid(), options);
    }
    async stopScreencast() {
        await this._mainFrameSession._stopScreencast();
    }
    async takeScreenshot(format, documentRect, viewportRect, quality) {
        const { visualViewport } = await this._mainFrameSession._client.send('Page.getLayoutMetrics');
        if (!documentRect) {
            documentRect = {
                x: visualViewport.pageX + viewportRect.x,
                y: visualViewport.pageY + viewportRect.y,
                ...helper_1.helper.enclosingIntSize({
                    width: viewportRect.width / visualViewport.scale,
                    height: viewportRect.height / visualViewport.scale,
                })
            };
        }
        // When taking screenshots with documentRect (based on the page content, not viewport),
        // ignore current page scale.
        const clip = { ...documentRect, scale: viewportRect ? visualViewport.scale : 1 };
        const result = await this._mainFrameSession._client.send('Page.captureScreenshot', { format, quality, clip });
        return Buffer.from(result.data, 'base64');
    }
    async resetViewport() {
        await this._mainFrameSession._client.send('Emulation.setDeviceMetricsOverride', { mobile: false, width: 0, height: 0, deviceScaleFactor: 0 });
    }
    async getContentFrame(handle) {
        return this._sessionForHandle(handle)._getContentFrame(handle);
    }
    async getOwnerFrame(handle) {
        return this._sessionForHandle(handle)._getOwnerFrame(handle);
    }
    isElementHandle(remoteObject) {
        return remoteObject.subtype === 'node';
    }
    async getBoundingBox(handle) {
        return this._sessionForHandle(handle)._getBoundingBox(handle);
    }
    async scrollRectIntoViewIfNeeded(handle, rect) {
        return this._sessionForHandle(handle)._scrollRectIntoViewIfNeeded(handle, rect);
    }
    rafCountForStablePosition() {
        return 1;
    }
    async getContentQuads(handle) {
        return this._sessionForHandle(handle)._getContentQuads(handle);
    }
    async setInputFiles(handle, files) {
        await handle._evaluateInUtility(([injected, node, files]) => injected.setInputFiles(node, files), files);
    }
    async adoptElementHandle(handle, to) {
        return this._sessionForHandle(handle)._adoptElementHandle(handle, to);
    }
    async getAccessibilityTree(needle) {
        return crAccessibility_1.getAccessibilityTree(this._mainFrameSession._client, needle);
    }
    async inputActionEpilogue() {
        await this._mainFrameSession._client.send('Page.enable').catch(e => { });
    }
    async pdf(options) {
        return this._pdf.generate(options);
    }
    coverage() {
        return this._coverage;
    }
    async getFrameElement(frame) {
        let parent = frame.parentFrame();
        if (!parent)
            throw new Error('Frame has been detached.');
        const parentSession = this._sessionForFrame(parent);
        const { backendNodeId } = await parentSession._client.send('DOM.getFrameOwner', { frameId: frame._id }).catch(e => {
            if (e instanceof Error && e.message.includes('Frame with the given id was not found.'))
                stackTrace_1.rewriteErrorMessage(e, 'Frame has been detached.');
            throw e;
        });
        parent = frame.parentFrame();
        if (!parent)
            throw new Error('Frame has been detached.');
        return parentSession._adoptBackendNodeId(backendNodeId, await parent._mainContext());
    }
}
exports.CRPage = CRPage;
class FrameSession {
    constructor(crPage, client, targetId, parentSession) {
        this._contextIdToContext = new Map();
        this._eventListeners = [];
        this._firstNonInitialNavigationCommittedFulfill = () => { };
        this._firstNonInitialNavigationCommittedReject = (e) => { };
        // Marks the oopif session that remote -> local transition has happened in the parent.
        // See Target.detachedFromTarget handler for details.
        this._swappedIn = false;
        this._videoRecorder = null;
        this._screencastId = null;
        this._screencastState = 'stopped';
        this._client = client;
        this._crPage = crPage;
        this._page = crPage._page;
        this._targetId = targetId;
        this._networkManager = new crNetworkManager_1.CRNetworkManager(client, this._page, parentSession ? parentSession._networkManager : null);
        this._firstNonInitialNavigationCommittedPromise = new Promise((f, r) => {
            this._firstNonInitialNavigationCommittedFulfill = f;
            this._firstNonInitialNavigationCommittedReject = r;
        });
        client.once(crConnection_1.CRSessionEvents.Disconnected, () => {
            this._firstNonInitialNavigationCommittedReject(new Error('Page closed'));
        });
    }
    _isMainFrame() {
        return this._targetId === this._crPage._targetId;
    }
    _addRendererListeners() {
        this._eventListeners.push(...[
            helper_1.helper.addEventListener(this._client, 'Log.entryAdded', event => this._onLogEntryAdded(event)),
            helper_1.helper.addEventListener(this._client, 'Page.fileChooserOpened', event => this._onFileChooserOpened(event)),
            helper_1.helper.addEventListener(this._client, 'Page.frameAttached', event => this._onFrameAttached(event.frameId, event.parentFrameId)),
            helper_1.helper.addEventListener(this._client, 'Page.frameDetached', event => this._onFrameDetached(event.frameId)),
            helper_1.helper.addEventListener(this._client, 'Page.frameNavigated', event => this._onFrameNavigated(event.frame, false)),
            helper_1.helper.addEventListener(this._client, 'Page.frameRequestedNavigation', event => this._onFrameRequestedNavigation(event)),
            helper_1.helper.addEventListener(this._client, 'Page.frameStoppedLoading', event => this._onFrameStoppedLoading(event.frameId)),
            helper_1.helper.addEventListener(this._client, 'Page.javascriptDialogOpening', event => this._onDialog(event)),
            helper_1.helper.addEventListener(this._client, 'Page.navigatedWithinDocument', event => this._onFrameNavigatedWithinDocument(event.frameId, event.url)),
            helper_1.helper.addEventListener(this._client, 'Runtime.bindingCalled', event => this._onBindingCalled(event)),
            helper_1.helper.addEventListener(this._client, 'Runtime.consoleAPICalled', event => this._onConsoleAPI(event)),
            helper_1.helper.addEventListener(this._client, 'Runtime.exceptionThrown', exception => this._handleException(exception.exceptionDetails)),
            helper_1.helper.addEventListener(this._client, 'Runtime.executionContextCreated', event => this._onExecutionContextCreated(event.context)),
            helper_1.helper.addEventListener(this._client, 'Runtime.executionContextDestroyed', event => this._onExecutionContextDestroyed(event.executionContextId)),
            helper_1.helper.addEventListener(this._client, 'Runtime.executionContextsCleared', event => this._onExecutionContextsCleared()),
            helper_1.helper.addEventListener(this._client, 'Target.attachedToTarget', event => this._onAttachedToTarget(event)),
            helper_1.helper.addEventListener(this._client, 'Target.detachedFromTarget', event => this._onDetachedFromTarget(event)),
        ]);
    }
    _addBrowserListeners() {
        this._eventListeners.push(...[
            helper_1.helper.addEventListener(this._client, 'Inspector.targetCrashed', event => this._onTargetCrashed()),
            helper_1.helper.addEventListener(this._client, 'Page.downloadWillBegin', event => this._onDownloadWillBegin(event)),
            helper_1.helper.addEventListener(this._client, 'Page.downloadProgress', event => this._onDownloadProgress(event)),
            helper_1.helper.addEventListener(this._client, 'Page.screencastFrame', event => this._onScreencastFrame(event)),
            helper_1.helper.addEventListener(this._client, 'Page.windowOpen', event => this._onWindowOpen(event)),
        ]);
    }
    async _initialize(hasUIWindow) {
        if (hasUIWindow &&
            !this._crPage._browserContext._browser.isClank() &&
            !this._crPage._browserContext._options.noDefaultViewport) {
            const { windowId } = await this._client.send('Browser.getWindowForTarget');
            this._windowId = windowId;
        }
        let lifecycleEventsEnabled;
        if (!this._isMainFrame())
            this._addRendererListeners();
        this._addBrowserListeners();
        const promises = [
            this._client.send('Page.enable'),
            this._client.send('Page.getFrameTree').then(({ frameTree }) => {
                if (this._isMainFrame()) {
                    this._handleFrameTree(frameTree);
                    this._addRendererListeners();
                }
                const localFrames = this._isMainFrame() ? this._page.frames() : [this._page._frameManager.frame(this._targetId)];
                for (const frame of localFrames) {
                    // Note: frames might be removed before we send these.
                    this._client._sendMayFail('Page.createIsolatedWorld', {
                        frameId: frame._id,
                        grantUniveralAccess: true,
                        worldName: UTILITY_WORLD_NAME,
                    });
                    for (const binding of this._crPage._browserContext._pageBindings.values())
                        frame._evaluateExpression(binding.source, false, {}).catch(e => { });
                }
                const isInitialEmptyPage = this._isMainFrame() && this._page.mainFrame().url() === ':';
                if (isInitialEmptyPage) {
                    // Ignore lifecycle events for the initial empty page. It is never the final page
                    // hence we are going to get more lifecycle updates after the actual navigation has
                    // started (even if the target url is about:blank).
                    lifecycleEventsEnabled.then(() => {
                        this._eventListeners.push(helper_1.helper.addEventListener(this._client, 'Page.lifecycleEvent', event => this._onLifecycleEvent(event)));
                    });
                }
                else {
                    this._firstNonInitialNavigationCommittedFulfill();
                    this._eventListeners.push(helper_1.helper.addEventListener(this._client, 'Page.lifecycleEvent', event => this._onLifecycleEvent(event)));
                }
            }),
            this._client.send('Log.enable', {}),
            lifecycleEventsEnabled = this._client.send('Page.setLifecycleEventsEnabled', { enabled: true }),
            this._client.send('Runtime.enable', {}),
            this._client.send('Page.addScriptToEvaluateOnNewDocument', {
                source: sourceMap.generateSourceUrl(),
                worldName: UTILITY_WORLD_NAME,
            }),
            this._networkManager.initialize(),
            this._client.send('Target.setAutoAttach', { autoAttach: true, waitForDebuggerOnStart: true, flatten: true }),
            this._client.send('Emulation.setFocusEmulationEnabled', { enabled: true }),
        ];
        const options = this._crPage._browserContext._options;
        if (options.bypassCSP)
            promises.push(this._client.send('Page.setBypassCSP', { enabled: true }));
        if (options.ignoreHTTPSErrors)
            promises.push(this._client.send('Security.setIgnoreCertificateErrors', { ignore: true }));
        if (this._isMainFrame())
            promises.push(this._updateViewport());
        if (options.hasTouch)
            promises.push(this._client.send('Emulation.setTouchEmulationEnabled', { enabled: true }));
        if (options.javaScriptEnabled === false)
            promises.push(this._client.send('Emulation.setScriptExecutionDisabled', { value: true }));
        if (options.userAgent || options.locale)
            promises.push(this._client.send('Emulation.setUserAgentOverride', { userAgent: options.userAgent || '', acceptLanguage: options.locale }));
        if (options.locale)
            promises.push(emulateLocale(this._client, options.locale));
        if (options.timezoneId)
            promises.push(emulateTimezone(this._client, options.timezoneId));
        promises.push(this._updateGeolocation(true));
        promises.push(this._updateExtraHTTPHeaders(true));
        promises.push(this._updateRequestInterception(true));
        promises.push(this._updateOffline(true));
        promises.push(this._updateHttpCredentials(true));
        promises.push(this._updateEmulateMedia(true));
        for (const binding of this._crPage._browserContext._pageBindings.values())
            promises.push(this._initBinding(binding));
        for (const binding of this._crPage._page._pageBindings.values())
            promises.push(this._initBinding(binding));
        for (const source of this._crPage._browserContext._evaluateOnNewDocumentSources)
            promises.push(this._evaluateOnNewDocument(source));
        for (const source of this._crPage._page._evaluateOnNewDocumentSources)
            promises.push(this._evaluateOnNewDocument(source));
        if (this._isMainFrame() && this._crPage._browserContext._options.recordVideo) {
            const size = this._crPage._browserContext._options.recordVideo.size || this._crPage._browserContext._options.viewport || { width: 1280, height: 720 };
            const screencastId = utils_1.createGuid();
            const outputFile = path.join(this._crPage._browserContext._options.recordVideo.dir, screencastId + '.webm');
            promises.push(this._crPage._browserContext._ensureVideosPath().then(() => {
                return this._startScreencast(screencastId, {
                    ...size,
                    outputFile,
                });
            }));
        }
        promises.push(this._client.send('Runtime.runIfWaitingForDebugger'));
        promises.push(this._firstNonInitialNavigationCommittedPromise);
        await Promise.all(promises);
    }
    dispose() {
        helper_1.helper.removeEventListeners(this._eventListeners);
        this._networkManager.dispose();
        this._crPage._sessions.delete(this._targetId);
    }
    async _navigate(frame, url, referrer) {
        const response = await this._client.send('Page.navigate', { url, referrer, frameId: frame._id });
        if (response.errorText)
            throw new Error(`${response.errorText} at ${url}`);
        return { newDocumentId: response.loaderId };
    }
    _onLifecycleEvent(event) {
        if (event.name === 'load')
            this._page._frameManager.frameLifecycleEvent(event.frameId, 'load');
        else if (event.name === 'DOMContentLoaded')
            this._page._frameManager.frameLifecycleEvent(event.frameId, 'domcontentloaded');
    }
    _onFrameStoppedLoading(frameId) {
        this._page._frameManager.frameStoppedLoading(frameId);
    }
    _handleFrameTree(frameTree) {
        this._onFrameAttached(frameTree.frame.id, frameTree.frame.parentId || null);
        this._onFrameNavigated(frameTree.frame, true);
        if (!frameTree.childFrames)
            return;
        for (const child of frameTree.childFrames)
            this._handleFrameTree(child);
    }
    _onFrameAttached(frameId, parentFrameId) {
        const frameSession = this._crPage._sessions.get(frameId);
        if (frameSession && frameId !== this._targetId) {
            // This is a remote -> local frame transition.
            frameSession._swappedIn = true;
            const frame = this._page._frameManager.frame(frameId);
            this._page._frameManager.removeChildFramesRecursively(frame);
            return;
        }
        this._page._frameManager.frameAttached(frameId, parentFrameId);
    }
    _onFrameNavigated(framePayload, initial) {
        this._page._frameManager.frameCommittedNewDocumentNavigation(framePayload.id, framePayload.url + (framePayload.urlFragment || ''), framePayload.name || '', framePayload.loaderId, initial);
        if (!initial)
            this._firstNonInitialNavigationCommittedFulfill();
    }
    _onFrameRequestedNavigation(payload) {
        if (payload.disposition === 'currentTab')
            this._page._frameManager.frameRequestedNavigation(payload.frameId);
    }
    _onFrameNavigatedWithinDocument(frameId, url) {
        this._page._frameManager.frameCommittedSameDocumentNavigation(frameId, url);
    }
    _onFrameDetached(frameId) {
        if (this._crPage._sessions.has(frameId)) {
            // This is a local -> remote frame transtion.
            // We already got a new target and handled frame reattach - nothing to do here.
            return;
        }
        this._page._frameManager.frameDetached(frameId);
    }
    _onExecutionContextCreated(contextPayload) {
        const frame = contextPayload.auxData ? this._page._frameManager.frame(contextPayload.auxData.frameId) : null;
        if (!frame)
            return;
        const delegate = new crExecutionContext_1.CRExecutionContext(this._client, contextPayload);
        const context = new dom.FrameExecutionContext(delegate, frame);
        if (contextPayload.auxData && !!contextPayload.auxData.isDefault)
            frame._contextCreated('main', context);
        else if (contextPayload.name === UTILITY_WORLD_NAME)
            frame._contextCreated('utility', context);
        this._contextIdToContext.set(contextPayload.id, context);
    }
    _onExecutionContextDestroyed(executionContextId) {
        const context = this._contextIdToContext.get(executionContextId);
        if (!context)
            return;
        this._contextIdToContext.delete(executionContextId);
        context.frame._contextDestroyed(context);
    }
    _onExecutionContextsCleared() {
        for (const contextId of Array.from(this._contextIdToContext.keys()))
            this._onExecutionContextDestroyed(contextId);
    }
    _onAttachedToTarget(event) {
        const session = crConnection_1.CRConnection.fromSession(this._client).session(event.sessionId);
        if (event.targetInfo.type === 'iframe') {
            // Frame id equals target id.
            const targetId = event.targetInfo.targetId;
            const frame = this._page._frameManager.frame(targetId);
            this._page._frameManager.removeChildFramesRecursively(frame);
            const frameSession = new FrameSession(this._crPage, session, targetId, this);
            this._crPage._sessions.set(targetId, frameSession);
            frameSession._initialize(false).catch(e => e);
            return;
        }
        if (event.targetInfo.type !== 'worker') {
            // Ideally, detaching should resume any target, but there is a bug in the backend.
            session._sendMayFail('Runtime.runIfWaitingForDebugger').then(() => {
                this._client._sendMayFail('Target.detachFromTarget', { sessionId: event.sessionId });
            });
            return;
        }
        const url = event.targetInfo.url;
        const worker = new page_1.Worker(url);
        this._page._addWorker(event.sessionId, worker);
        session.once('Runtime.executionContextCreated', async (event) => {
            worker._createExecutionContext(new crExecutionContext_1.CRExecutionContext(session, event.context));
        });
        Promise.all([
            session._sendMayFail('Runtime.enable'),
            session._sendMayFail('Network.enable'),
            session._sendMayFail('Runtime.runIfWaitingForDebugger'),
        ]); // This might fail if the target is closed before we initialize.
        session.on('Runtime.consoleAPICalled', event => {
            const args = event.args.map(o => worker._existingExecutionContext.createHandle(o));
            this._page._addConsoleMessage(event.type, args, crProtocolHelper_1.toConsoleMessageLocation(event.stackTrace));
        });
        session.on('Runtime.exceptionThrown', exception => this._page.emit(page_1.Page.Events.PageError, crProtocolHelper_1.exceptionToError(exception.exceptionDetails)));
        // TODO: attribute workers to the right frame.
        this._networkManager.instrumentNetworkEvents(session, this._page._frameManager.frame(this._targetId));
    }
    _onDetachedFromTarget(event) {
        // This might be a worker...
        this._page._removeWorker(event.sessionId);
        // ... or an oopif.
        const childFrameSession = this._crPage._sessions.get(event.targetId);
        if (!childFrameSession)
            return;
        // Usually, we get frameAttached in this session first and mark child as swappedIn.
        if (childFrameSession._swappedIn) {
            childFrameSession.dispose();
            return;
        }
        // However, sometimes we get detachedFromTarget before frameAttached.
        // In this case we don't know wheter this is a remote frame detach,
        // or just a remote -> local transition. In the latter case, frameAttached
        // is already inflight, so let's make a safe roundtrip to ensure it arrives.
        this._client.send('Page.enable').catch(e => null).then(() => {
            // Child was not swapped in - that means frameAttached did not happen and
            // this is remote detach rather than remote -> local swap.
            if (!childFrameSession._swappedIn)
                this._page._frameManager.frameDetached(event.targetId);
            childFrameSession.dispose();
        });
    }
    _onWindowOpen(event) {
        this._crPage._nextWindowOpenPopupFeatures.push(event.windowFeatures);
    }
    async _onConsoleAPI(event) {
        if (event.executionContextId === 0) {
            // DevTools protocol stores the last 1000 console messages. These
            // messages are always reported even for removed execution contexts. In
            // this case, they are marked with executionContextId = 0 and are
            // reported upon enabling Runtime agent.
            //
            // Ignore these messages since:
            // - there's no execution context we can use to operate with message
            //   arguments
            // - these messages are reported before Playwright clients can subscribe
            //   to the 'console'
            //   page event.
            //
            // @see https://github.com/GoogleChrome/puppeteer/issues/3865
            return;
        }
        const context = this._contextIdToContext.get(event.executionContextId);
        const values = event.args.map(arg => context.createHandle(arg));
        this._page._addConsoleMessage(event.type, values, crProtocolHelper_1.toConsoleMessageLocation(event.stackTrace));
    }
    async _initBinding(binding) {
        await Promise.all([
            this._client.send('Runtime.addBinding', { name: binding.name }),
            this._client.send('Page.addScriptToEvaluateOnNewDocument', { source: binding.source })
        ]);
    }
    async _onBindingCalled(event) {
        const context = this._contextIdToContext.get(event.executionContextId);
        const pageOrError = await this._crPage.pageOrError();
        if (!(pageOrError instanceof Error))
            this._page._onBindingCalled(event.payload, context);
    }
    _onDialog(event) {
        this._page.emit(page_1.Page.Events.Dialog, new dialog.Dialog(event.type, event.message, async (accept, promptText) => {
            await this._client.send('Page.handleJavaScriptDialog', { accept, promptText });
        }, event.defaultPrompt));
    }
    _handleException(exceptionDetails) {
        this._page.emit(page_1.Page.Events.PageError, crProtocolHelper_1.exceptionToError(exceptionDetails));
    }
    async _onTargetCrashed() {
        this._client._markAsCrashed();
        this._page._didCrash();
    }
    _onLogEntryAdded(event) {
        const { level, text, args, source, url, lineNumber } = event.entry;
        if (args)
            args.map(arg => crProtocolHelper_1.releaseObject(this._client, arg.objectId));
        if (source !== 'worker') {
            const location = {
                url: url || '',
                lineNumber: lineNumber || 0,
                columnNumber: 0,
            };
            this._page.emit(page_1.Page.Events.Console, new console_1.ConsoleMessage(level, text, [], location));
        }
    }
    async _onFileChooserOpened(event) {
        const frame = this._page._frameManager.frame(event.frameId);
        const utilityContext = await frame._utilityContext();
        const handle = await this._adoptBackendNodeId(event.backendNodeId, utilityContext);
        this._page._onFileChooserOpened(handle);
    }
    _onDownloadWillBegin(payload) {
        let originPage = this._crPage._initializedPage;
        // If it's a new window download, report it on the opener page.
        if (!originPage) {
            // Resume the page creation with an error. The page will automatically close right
            // after the download begins.
            this._firstNonInitialNavigationCommittedReject(new Error('Starting new page download'));
            if (this._crPage._opener)
                originPage = this._crPage._opener._initializedPage;
        }
        if (!originPage)
            return;
        this._crPage._browserContext._browser._downloadCreated(originPage, payload.guid, payload.url, payload.suggestedFilename);
    }
    _onDownloadProgress(payload) {
        if (payload.state === 'completed')
            this._crPage._browserContext._browser._downloadFinished(payload.guid, '');
        if (payload.state === 'canceled')
            this._crPage._browserContext._browser._downloadFinished(payload.guid, 'canceled');
    }
    _onScreencastFrame(payload) {
        if (!this._videoRecorder)
            return;
        const buffer = Buffer.from(payload.data, 'base64');
        this._videoRecorder.writeFrame(buffer, payload.metadata.timestamp);
        // The target may be closed before receiving the ack.
        this._client.send('Page.screencastFrameAck', { sessionId: payload.sessionId }).catch(() => { });
    }
    async _startScreencast(screencastId, options) {
        if (this._screencastState !== 'stopped')
            throw new Error('Already started');
        const videoRecorder = await videoRecorder_1.VideoRecorder.launch(options);
        this._screencastState = 'starting';
        try {
            this._screencastState = 'started';
            this._videoRecorder = videoRecorder;
            this._screencastId = screencastId;
            this._crPage._browserContext._browser._videoStarted(this._crPage._browserContext, screencastId, options.outputFile, this._crPage.pageOrError());
            await Promise.all([
                this._client.send('Page.startScreencast', {
                    format: 'jpeg',
                    quality: 90,
                    maxWidth: options.width,
                    maxHeight: options.height,
                }),
                new Promise(f => this._client.once('Page.screencastFrame', f))
            ]);
        }
        catch (e) {
            videoRecorder.stop().catch(() => { });
            throw e;
        }
    }
    async _stopScreencast() {
        if (this._screencastState !== 'started')
            throw new Error('No screencast in progress, current state: ' + this._screencastState);
        try {
            await this._client.send('Page.stopScreencast');
        }
        finally {
            const recorder = this._videoRecorder;
            const screencastId = this._screencastId;
            this._videoRecorder = null;
            this._screencastId = null;
            this._screencastState = 'stopped';
            await recorder.stop().catch(() => { });
            this._crPage._browserContext._browser._videoFinished(screencastId);
        }
    }
    async _updateExtraHTTPHeaders(initial) {
        const headers = network.mergeHeaders([
            this._crPage._browserContext._options.extraHTTPHeaders,
            this._page._state.extraHTTPHeaders
        ]);
        if (!initial || headers.length)
            await this._client.send('Network.setExtraHTTPHeaders', { headers: utils_1.headersArrayToObject(headers, false /* lowerCase */) });
    }
    async _updateGeolocation(initial) {
        const geolocation = this._crPage._browserContext._options.geolocation;
        if (!initial || geolocation)
            await this._client.send('Emulation.setGeolocationOverride', geolocation || {});
    }
    async _updateOffline(initial) {
        const offline = !!this._crPage._browserContext._options.offline;
        if (!initial || offline)
            await this._networkManager.setOffline(offline);
    }
    async _updateHttpCredentials(initial) {
        const credentials = this._crPage._browserContext._options.httpCredentials || null;
        if (!initial || credentials)
            await this._networkManager.authenticate(credentials);
    }
    async _updateViewport() {
        if (this._crPage._browserContext._browser.isClank())
            return;
        utils_1.assert(this._isMainFrame());
        const options = this._crPage._browserContext._options;
        const viewportSize = this._page._state.viewportSize;
        if (viewportSize === null)
            return;
        const isLandscape = viewportSize.width > viewportSize.height;
        const promises = [
            this._client.send('Emulation.setDeviceMetricsOverride', {
                mobile: !!options.isMobile,
                width: viewportSize.width,
                height: viewportSize.height,
                screenWidth: viewportSize.width,
                screenHeight: viewportSize.height,
                deviceScaleFactor: options.deviceScaleFactor || 1,
                screenOrientation: isLandscape ? { angle: 90, type: 'landscapePrimary' } : { angle: 0, type: 'portraitPrimary' },
            }),
        ];
        if (this._windowId) {
            let insets = { width: 0, height: 0 };
            if (this._crPage._browserContext._browser._options.headful) {
                // TODO: popup windows have their own insets.
                insets = { width: 24, height: 88 };
                if (process.platform === 'win32')
                    insets = { width: 16, height: 88 };
                else if (process.platform === 'linux')
                    insets = { width: 8, height: 85 };
                else if (process.platform === 'darwin')
                    insets = { width: 2, height: 80 };
            }
            promises.push(this._client.send('Browser.setWindowBounds', {
                windowId: this._windowId,
                bounds: { width: viewportSize.width + insets.width, height: viewportSize.height + insets.height }
            }));
        }
        await Promise.all(promises);
    }
    async _updateEmulateMedia(initial) {
        if (this._crPage._browserContext._browser.isClank())
            return;
        const colorScheme = this._page._state.colorScheme || this._crPage._browserContext._options.colorScheme || 'light';
        const features = colorScheme ? [{ name: 'prefers-color-scheme', value: colorScheme }] : [];
        await this._client.send('Emulation.setEmulatedMedia', { media: this._page._state.mediaType || '', features });
    }
    async _updateRequestInterception(initial) {
        await this._networkManager.setRequestInterception(this._page._needsRequestInterception());
    }
    async _setFileChooserIntercepted(enabled) {
        await this._client.send('Page.setInterceptFileChooserDialog', { enabled }).catch(e => { }); // target can be closed.
    }
    async _evaluateOnNewDocument(source) {
        await this._client.send('Page.addScriptToEvaluateOnNewDocument', { source });
    }
    async _getContentFrame(handle) {
        const nodeInfo = await this._client.send('DOM.describeNode', {
            objectId: handle._objectId
        });
        if (!nodeInfo || typeof nodeInfo.node.frameId !== 'string')
            return null;
        return this._page._frameManager.frame(nodeInfo.node.frameId);
    }
    async _getOwnerFrame(handle) {
        // document.documentElement has frameId of the owner frame.
        const documentElement = await handle.evaluateHandle(node => {
            const doc = node;
            if (doc.documentElement && doc.documentElement.ownerDocument === doc)
                return doc.documentElement;
            return node.ownerDocument ? node.ownerDocument.documentElement : null;
        });
        if (!documentElement)
            return null;
        if (!documentElement._objectId)
            return null;
        const nodeInfo = await this._client.send('DOM.describeNode', {
            objectId: documentElement._objectId
        });
        const frameId = nodeInfo && typeof nodeInfo.node.frameId === 'string' ?
            nodeInfo.node.frameId : null;
        documentElement.dispose();
        return frameId;
    }
    async _getBoundingBox(handle) {
        const result = await this._client._sendMayFail('DOM.getBoxModel', {
            objectId: handle._objectId
        });
        if (!result)
            return null;
        const quad = result.model.border;
        const x = Math.min(quad[0], quad[2], quad[4], quad[6]);
        const y = Math.min(quad[1], quad[3], quad[5], quad[7]);
        const width = Math.max(quad[0], quad[2], quad[4], quad[6]) - x;
        const height = Math.max(quad[1], quad[3], quad[5], quad[7]) - y;
        const position = await this._framePosition();
        if (!position)
            return null;
        return { x: x + position.x, y: y + position.y, width, height };
    }
    async _framePosition() {
        const frame = this._page._frameManager.frame(this._targetId);
        if (!frame)
            return null;
        if (frame === this._page.mainFrame())
            return { x: 0, y: 0 };
        const element = await frame.frameElement();
        const box = await element.boundingBox();
        return box;
    }
    async _scrollRectIntoViewIfNeeded(handle, rect) {
        return await this._client.send('DOM.scrollIntoViewIfNeeded', {
            objectId: handle._objectId,
            rect,
        }).then(() => 'done').catch(e => {
            if (e instanceof Error && e.message.includes('Node does not have a layout object'))
                return 'error:notvisible';
            if (e instanceof Error && e.message.includes('Node is detached from document'))
                return 'error:notconnected';
            throw e;
        });
    }
    async _getContentQuads(handle) {
        const result = await this._client._sendMayFail('DOM.getContentQuads', {
            objectId: handle._objectId
        });
        if (!result)
            return null;
        const position = await this._framePosition();
        if (!position)
            return null;
        return result.quads.map(quad => [
            { x: quad[0] + position.x, y: quad[1] + position.y },
            { x: quad[2] + position.x, y: quad[3] + position.y },
            { x: quad[4] + position.x, y: quad[5] + position.y },
            { x: quad[6] + position.x, y: quad[7] + position.y }
        ]);
    }
    async _adoptElementHandle(handle, to) {
        const nodeInfo = await this._client.send('DOM.describeNode', {
            objectId: handle._objectId,
        });
        return this._adoptBackendNodeId(nodeInfo.node.backendNodeId, to);
    }
    async _adoptBackendNodeId(backendNodeId, to) {
        const result = await this._client._sendMayFail('DOM.resolveNode', {
            backendNodeId,
            executionContextId: to._delegate._contextId,
        });
        if (!result || result.object.subtype === 'null')
            throw new Error('Unable to adopt element handle from a different document');
        return to.createHandle(result.object).asElement();
    }
}
async function emulateLocale(session, locale) {
    try {
        await session.send('Emulation.setLocaleOverride', { locale });
    }
    catch (exception) {
        // All pages in the same renderer share locale. All such pages belong to the same
        // context and if locale is overridden for one of them its value is the same as
        // we are trying to set so it's not a problem.
        if (exception.message.includes('Another locale override is already in effect'))
            return;
        throw exception;
    }
}
async function emulateTimezone(session, timezoneId) {
    try {
        await session.send('Emulation.setTimezoneOverride', { timezoneId: timezoneId });
    }
    catch (exception) {
        if (exception.message.includes('Timezone override is already in effect'))
            return;
        if (exception.message.includes('Invalid timezone'))
            throw new Error(`Invalid timezone ID: ${timezoneId}`);
        throw exception;
    }
}
//# sourceMappingURL=crPage.js.map