"use strict";
/**
 * Copyright 2017 Google Inc. All rights reserved.
 * Modifications copyright (c) Microsoft Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.verifyLoadState = exports.Frame = void 0;
const utils_1 = require("../utils/utils");
const channelOwner_1 = require("./channelOwner");
const elementHandle_1 = require("./elementHandle");
const jsHandle_1 = require("./jsHandle");
const fs = require("fs");
const network = require("./network");
const util = require("util");
const events_1 = require("events");
const waiter_1 = require("./waiter");
const events_2 = require("./events");
const types_1 = require("./types");
const clientHelper_1 = require("./clientHelper");
const fsReadFileAsync = util.promisify(fs.readFile.bind(fs));
class Frame extends channelOwner_1.ChannelOwner {
    constructor(parent, type, guid, initializer) {
        super(parent, type, guid, initializer);
        this._parentFrame = null;
        this._url = '';
        this._name = '';
        this._detached = false;
        this._childFrames = new Set();
        this._eventEmitter = new events_1.EventEmitter();
        this._eventEmitter.setMaxListeners(0);
        this._parentFrame = Frame.fromNullable(initializer.parentFrame);
        if (this._parentFrame)
            this._parentFrame._childFrames.add(this);
        this._name = initializer.name;
        this._url = initializer.url;
        this._loadStates = new Set(initializer.loadStates);
        this._channel.on('loadstate', event => {
            if (event.add) {
                this._loadStates.add(event.add);
                this._eventEmitter.emit('loadstate', event.add);
            }
            if (event.remove)
                this._loadStates.delete(event.remove);
        });
        this._channel.on('navigated', event => {
            this._url = event.url;
            this._name = event.name;
            this._eventEmitter.emit('navigated', event);
            if (!event.error && this._page)
                this._page.emit(events_2.Events.Page.FrameNavigated, this);
        });
    }
    static from(frame) {
        return frame._object;
    }
    static fromNullable(frame) {
        return frame ? Frame.from(frame) : null;
    }
    _apiName(method) {
        return this._page._isPageCall ? 'page.' + method : 'frame.' + method;
    }
    page() {
        return this._page;
    }
    async goto(url, options = {}) {
        return this._wrapApiCall(this._apiName('goto'), async () => {
            const waitUntil = verifyLoadState('waitUntil', options.waitUntil === undefined ? 'load' : options.waitUntil);
            return network.Response.fromNullable((await this._channel.goto({ url, ...options, waitUntil })).response);
        });
    }
    _setupNavigationWaiter(options) {
        const waiter = new waiter_1.Waiter();
        waiter.rejectOnEvent(this._page, events_2.Events.Page.Close, new Error('Navigation failed because page was closed!'));
        waiter.rejectOnEvent(this._page, events_2.Events.Page.Crash, new Error('Navigation failed because page crashed!'));
        waiter.rejectOnEvent(this._page, events_2.Events.Page.FrameDetached, new Error('Navigating frame was detached!'), frame => frame === this);
        const timeout = this._page._timeoutSettings.navigationTimeout(options);
        waiter.rejectOnTimeout(timeout, `Timeout ${timeout}ms exceeded.`);
        return waiter;
    }
    async waitForNavigation(options = {}) {
        return this._wrapApiCall(this._apiName('waitForNavigation'), async () => {
            const waitUntil = verifyLoadState('waitUntil', options.waitUntil === undefined ? 'load' : options.waitUntil);
            const waiter = this._setupNavigationWaiter(options);
            const toUrl = typeof options.url === 'string' ? ` to "${options.url}"` : '';
            waiter.log(`waiting for navigation${toUrl} until "${waitUntil}"`);
            const navigatedEvent = await waiter.waitForEvent(this._eventEmitter, 'navigated', event => {
                // Any failed navigation results in a rejection.
                if (event.error)
                    return true;
                waiter.log(`  navigated to "${event.url}"`);
                return clientHelper_1.urlMatches(event.url, options.url);
            });
            if (navigatedEvent.error) {
                const e = new Error(navigatedEvent.error);
                e.stack = '';
                await waiter.waitForPromise(Promise.reject(e));
            }
            if (!this._loadStates.has(waitUntil)) {
                await waiter.waitForEvent(this._eventEmitter, 'loadstate', s => {
                    waiter.log(`  "${s}" event fired`);
                    return s === waitUntil;
                });
            }
            const request = navigatedEvent.newDocument ? network.Request.fromNullable(navigatedEvent.newDocument.request) : null;
            const response = request ? await waiter.waitForPromise(request._finalRequest().response()) : null;
            waiter.dispose();
            return response;
        });
    }
    async waitForLoadState(state = 'load', options = {}) {
        state = verifyLoadState('state', state);
        if (this._loadStates.has(state))
            return;
        return this._wrapApiCall(this._apiName('waitForLoadState'), async () => {
            const waiter = this._setupNavigationWaiter(options);
            await waiter.waitForEvent(this._eventEmitter, 'loadstate', s => {
                waiter.log(`  "${s}" event fired`);
                return s === state;
            });
            waiter.dispose();
        });
    }
    async frameElement() {
        return this._wrapApiCall(this._apiName('frameElement'), async () => {
            return elementHandle_1.ElementHandle.from((await this._channel.frameElement()).element);
        });
    }
    async evaluateHandle(pageFunction, arg) {
        jsHandle_1.assertMaxArguments(arguments.length, 2);
        return this._wrapApiCall(this._apiName('evaluateHandle'), async () => {
            const result = await this._channel.evaluateExpressionHandle({ expression: String(pageFunction), isFunction: typeof pageFunction === 'function', arg: jsHandle_1.serializeArgument(arg) });
            return jsHandle_1.JSHandle.from(result.handle);
        });
    }
    async _evaluateHandleInUtility(pageFunction, arg) {
        jsHandle_1.assertMaxArguments(arguments.length, 2);
        return this._wrapApiCall(this._apiName('_evaluateHandleInUtility'), async () => {
            const result = await this._channel.evaluateExpressionHandle({ expression: String(pageFunction), isFunction: typeof pageFunction === 'function', arg: jsHandle_1.serializeArgument(arg), world: 'utility' });
            return jsHandle_1.JSHandle.from(result.handle);
        });
    }
    async evaluate(pageFunction, arg) {
        jsHandle_1.assertMaxArguments(arguments.length, 2);
        return this._wrapApiCall(this._apiName('evaluate'), async () => {
            const result = await this._channel.evaluateExpression({ expression: String(pageFunction), isFunction: typeof pageFunction === 'function', arg: jsHandle_1.serializeArgument(arg) });
            return jsHandle_1.parseResult(result.value);
        });
    }
    async _evaluateInUtility(pageFunction, arg) {
        jsHandle_1.assertMaxArguments(arguments.length, 2);
        return this._wrapApiCall(this._apiName('evaluate'), async () => {
            const result = await this._channel.evaluateExpression({ expression: String(pageFunction), isFunction: typeof pageFunction === 'function', arg: jsHandle_1.serializeArgument(arg), world: 'utility' });
            return jsHandle_1.parseResult(result.value);
        });
    }
    async $(selector) {
        return this._wrapApiCall(this._apiName('$'), async () => {
            const result = await this._channel.querySelector({ selector });
            return elementHandle_1.ElementHandle.fromNullable(result.element);
        });
    }
    async waitForSelector(selector, options = {}) {
        return this._wrapApiCall(this._apiName('waitForSelector'), async () => {
            if (options.visibility)
                throw new Error('options.visibility is not supported, did you mean options.state?');
            if (options.waitFor && options.waitFor !== 'visible')
                throw new Error('options.waitFor is not supported, did you mean options.state?');
            const result = await this._channel.waitForSelector({ selector, ...options });
            return elementHandle_1.ElementHandle.fromNullable(result.element);
        });
    }
    async dispatchEvent(selector, type, eventInit, options = {}) {
        return this._wrapApiCall(this._apiName('dispatchEvent'), async () => {
            await this._channel.dispatchEvent({ selector, type, eventInit: jsHandle_1.serializeArgument(eventInit), ...options });
        });
    }
    async $eval(selector, pageFunction, arg) {
        jsHandle_1.assertMaxArguments(arguments.length, 3);
        return this._wrapApiCall(this._apiName('$eval'), async () => {
            const result = await this._channel.evalOnSelector({ selector, expression: String(pageFunction), isFunction: typeof pageFunction === 'function', arg: jsHandle_1.serializeArgument(arg) });
            return jsHandle_1.parseResult(result.value);
        });
    }
    async $$eval(selector, pageFunction, arg) {
        jsHandle_1.assertMaxArguments(arguments.length, 3);
        return this._wrapApiCall(this._apiName('$$eval'), async () => {
            const result = await this._channel.evalOnSelectorAll({ selector, expression: String(pageFunction), isFunction: typeof pageFunction === 'function', arg: jsHandle_1.serializeArgument(arg) });
            return jsHandle_1.parseResult(result.value);
        });
    }
    async $$(selector) {
        return this._wrapApiCall(this._apiName('$$'), async () => {
            const result = await this._channel.querySelectorAll({ selector });
            return result.elements.map(e => elementHandle_1.ElementHandle.from(e));
        });
    }
    async content() {
        return this._wrapApiCall(this._apiName('content'), async () => {
            return (await this._channel.content()).value;
        });
    }
    async setContent(html, options = {}) {
        return this._wrapApiCall(this._apiName('setContent'), async () => {
            const waitUntil = verifyLoadState('waitUntil', options.waitUntil === undefined ? 'load' : options.waitUntil);
            await this._channel.setContent({ html, ...options, waitUntil });
        });
    }
    name() {
        return this._name || '';
    }
    url() {
        return this._url;
    }
    parentFrame() {
        return this._parentFrame;
    }
    childFrames() {
        return Array.from(this._childFrames);
    }
    isDetached() {
        return this._detached;
    }
    async addScriptTag(options) {
        return this._wrapApiCall(this._apiName('addScriptTag'), async () => {
            const copy = { ...options };
            if (copy.path) {
                copy.content = (await fsReadFileAsync(copy.path)).toString();
                copy.content += '//# sourceURL=' + copy.path.replace(/\n/g, '');
            }
            return elementHandle_1.ElementHandle.from((await this._channel.addScriptTag({ ...copy })).element);
        });
    }
    async addStyleTag(options) {
        return this._wrapApiCall(this._apiName('addStyleTag'), async () => {
            const copy = { ...options };
            if (copy.path) {
                copy.content = (await fsReadFileAsync(copy.path)).toString();
                copy.content += '/*# sourceURL=' + copy.path.replace(/\n/g, '') + '*/';
            }
            return elementHandle_1.ElementHandle.from((await this._channel.addStyleTag({ ...copy })).element);
        });
    }
    async click(selector, options = {}) {
        return this._wrapApiCall(this._apiName('click'), async () => {
            return await this._channel.click({ selector, ...options });
        });
    }
    async dblclick(selector, options = {}) {
        return this._wrapApiCall(this._apiName('dblclick'), async () => {
            return await this._channel.dblclick({ selector, ...options });
        });
    }
    async tap(selector, options = {}) {
        return this._wrapApiCall(this._apiName('tap'), async () => {
            return await this._channel.tap({ selector, ...options });
        });
    }
    async fill(selector, value, options = {}) {
        return this._wrapApiCall(this._apiName('fill'), async () => {
            return await this._channel.fill({ selector, value, ...options });
        });
    }
    async focus(selector, options = {}) {
        return this._wrapApiCall(this._apiName('focus'), async () => {
            await this._channel.focus({ selector, ...options });
        });
    }
    async textContent(selector, options = {}) {
        return this._wrapApiCall(this._apiName('textContent'), async () => {
            const value = (await this._channel.textContent({ selector, ...options })).value;
            return value === undefined ? null : value;
        });
    }
    async innerText(selector, options = {}) {
        return this._wrapApiCall(this._apiName('innerText'), async () => {
            return (await this._channel.innerText({ selector, ...options })).value;
        });
    }
    async innerHTML(selector, options = {}) {
        return this._wrapApiCall(this._apiName('innerHTML'), async () => {
            return (await this._channel.innerHTML({ selector, ...options })).value;
        });
    }
    async getAttribute(selector, name, options = {}) {
        return this._wrapApiCall(this._apiName('getAttribute'), async () => {
            const value = (await this._channel.getAttribute({ selector, name, ...options })).value;
            return value === undefined ? null : value;
        });
    }
    async hover(selector, options = {}) {
        return this._wrapApiCall(this._apiName('hover'), async () => {
            await this._channel.hover({ selector, ...options });
        });
    }
    async selectOption(selector, values, options = {}) {
        return this._wrapApiCall(this._apiName('selectOption'), async () => {
            return (await this._channel.selectOption({ selector, ...elementHandle_1.convertSelectOptionValues(values), ...options })).values;
        });
    }
    async setInputFiles(selector, files, options = {}) {
        return this._wrapApiCall(this._apiName('setInputFiles'), async () => {
            await this._channel.setInputFiles({ selector, files: await elementHandle_1.convertInputFiles(files), ...options });
        });
    }
    async type(selector, text, options = {}) {
        return this._wrapApiCall(this._apiName('type'), async () => {
            await this._channel.type({ selector, text, ...options });
        });
    }
    async press(selector, key, options = {}) {
        return this._wrapApiCall(this._apiName('press'), async () => {
            await this._channel.press({ selector, key, ...options });
        });
    }
    async check(selector, options = {}) {
        return this._wrapApiCall(this._apiName('check'), async () => {
            await this._channel.check({ selector, ...options });
        });
    }
    async uncheck(selector, options = {}) {
        return this._wrapApiCall(this._apiName('uncheck'), async () => {
            await this._channel.uncheck({ selector, ...options });
        });
    }
    async waitForTimeout(timeout) {
        await new Promise(fulfill => setTimeout(fulfill, timeout));
    }
    async waitForFunction(pageFunction, arg, options = {}) {
        return this._wrapApiCall(this._apiName('waitForFunction'), async () => {
            if (typeof options.polling === 'string')
                utils_1.assert(options.polling === 'raf', 'Unknown polling option: ' + options.polling);
            const result = await this._channel.waitForFunction({
                ...options,
                pollingInterval: options.polling === 'raf' ? undefined : options.polling,
                expression: String(pageFunction),
                isFunction: typeof pageFunction === 'function',
                arg: jsHandle_1.serializeArgument(arg),
            });
            return jsHandle_1.JSHandle.from(result.handle);
        });
    }
    async title() {
        return this._wrapApiCall(this._apiName('title'), async () => {
            return (await this._channel.title()).value;
        });
    }
    async _extendInjectedScript(source, arg) {
        const result = await this._channel.extendInjectedScript({ source, arg: jsHandle_1.serializeArgument(arg) });
        return jsHandle_1.JSHandle.from(result.handle);
    }
}
exports.Frame = Frame;
function verifyLoadState(name, waitUntil) {
    if (waitUntil === 'networkidle0')
        waitUntil = 'networkidle';
    if (!types_1.kLifecycleEvents.has(waitUntil))
        throw new Error(`${name}: expected one of (load|domcontentloaded|networkidle)`);
    return waitUntil;
}
exports.verifyLoadState = verifyLoadState;
//# sourceMappingURL=frame.js.map