/*
 * Decompiled with CFR 0.152.
 */
package parquet.hadoop;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import parquet.Log;
import parquet.Preconditions;
import parquet.column.ColumnWriteStore;
import parquet.column.ParquetProperties;
import parquet.hadoop.CodecFactory;
import parquet.hadoop.ColumnChunkPageWriteStore;
import parquet.hadoop.ParquetFileWriter;
import parquet.hadoop.api.WriteSupport;
import parquet.io.ColumnIOFactory;
import parquet.io.MessageColumnIO;
import parquet.schema.MessageType;

class InternalParquetRecordWriter<T> {
    private static final Log LOG = Log.getLog(InternalParquetRecordWriter.class);
    private static final int MINIMUM_RECORD_COUNT_FOR_CHECK = 100;
    private static final int MAXIMUM_RECORD_COUNT_FOR_CHECK = 10000;
    private final ParquetFileWriter parquetFileWriter;
    private final WriteSupport<T> writeSupport;
    private final MessageType schema;
    private final Map<String, String> extraMetaData;
    private final long rowGroupSize;
    private long rowGroupSizeThreshold;
    private final int pageSize;
    private final CodecFactory.BytesCompressor compressor;
    private final boolean validating;
    private final ParquetProperties parquetProperties;
    private long recordCount = 0L;
    private long recordCountForNextMemCheck = 100L;
    private ColumnWriteStore columnStore;
    private ColumnChunkPageWriteStore pageStore;

    public InternalParquetRecordWriter(ParquetFileWriter parquetFileWriter, WriteSupport<T> writeSupport, MessageType schema, Map<String, String> extraMetaData, long rowGroupSize, int pageSize, CodecFactory.BytesCompressor compressor, int dictionaryPageSize, boolean enableDictionary, boolean validating, ParquetProperties.WriterVersion writerVersion) {
        this.parquetFileWriter = parquetFileWriter;
        this.writeSupport = Preconditions.checkNotNull(writeSupport, "writeSupport");
        this.schema = schema;
        this.extraMetaData = extraMetaData;
        this.rowGroupSize = rowGroupSize;
        this.rowGroupSizeThreshold = rowGroupSize;
        this.pageSize = pageSize;
        this.compressor = compressor;
        this.validating = validating;
        this.parquetProperties = new ParquetProperties(dictionaryPageSize, writerVersion, enableDictionary);
        this.initStore();
    }

    private void initStore() {
        this.pageStore = new ColumnChunkPageWriteStore(this.compressor, this.schema, this.pageSize);
        this.columnStore = this.parquetProperties.newColumnWriteStore(this.schema, this.pageStore, this.pageSize);
        MessageColumnIO columnIO = new ColumnIOFactory(this.validating).getColumnIO(this.schema);
        this.writeSupport.prepareForWrite(columnIO.getRecordWriter(this.columnStore));
    }

    public void close() throws IOException, InterruptedException {
        this.flushRowGroupToStore();
        WriteSupport.FinalizedWriteContext finalWriteContext = this.writeSupport.finalizeWrite();
        HashMap<String, String> finalMetadata = new HashMap<String, String>(this.extraMetaData);
        finalMetadata.putAll(finalWriteContext.getExtraMetaData());
        this.parquetFileWriter.end(finalMetadata);
    }

    public void write(T value) throws IOException, InterruptedException {
        this.writeSupport.write(value);
        ++this.recordCount;
        this.checkBlockSizeReached();
    }

    private void checkBlockSizeReached() throws IOException {
        if (this.recordCount >= this.recordCountForNextMemCheck) {
            long memSize = this.columnStore.getBufferedSize();
            if (memSize > this.rowGroupSizeThreshold) {
                LOG.info(String.format("mem size %,d > %,d: flushing %,d records to disk.", memSize, this.rowGroupSizeThreshold, this.recordCount));
                this.flushRowGroupToStore();
                this.initStore();
                this.recordCountForNextMemCheck = Math.min(Math.max(100L, this.recordCount / 2L), 10000L);
            } else {
                float recordSize = (float)memSize / (float)this.recordCount;
                this.recordCountForNextMemCheck = Math.min(Math.max(100L, (this.recordCount + (long)((float)this.rowGroupSizeThreshold / recordSize)) / 2L), this.recordCount + 10000L);
                if (Log.DEBUG) {
                    LOG.debug(String.format("Checked mem at %,d will check again at: %,d ", this.recordCount, this.recordCountForNextMemCheck));
                }
            }
        }
    }

    private void flushRowGroupToStore() throws IOException {
        LOG.info(String.format("Flushing mem columnStore to file. allocated memory: %,d", this.columnStore.getAllocatedSize()));
        if (this.columnStore.getAllocatedSize() > 3L * this.rowGroupSizeThreshold) {
            LOG.warn("Too much memory used: " + this.columnStore.memUsageString());
        }
        if (this.recordCount > 0L) {
            this.parquetFileWriter.startBlock(this.recordCount);
            this.columnStore.flush();
            this.pageStore.flushToFileWriter(this.parquetFileWriter);
            this.recordCount = 0L;
            this.parquetFileWriter.endBlock();
        }
        this.columnStore = null;
        this.pageStore = null;
    }

    long getRowGroupSizeThreshold() {
        return this.rowGroupSizeThreshold;
    }

    void setRowGroupSizeThreshold(long rowGroupSizeThreshold) {
        this.rowGroupSizeThreshold = rowGroupSizeThreshold;
    }

    MessageType getSchema() {
        return this.schema;
    }
}

