"""
The refdata module contains pure-element reference state data.
"""

from sympy import And, Piecewise, Symbol, log
from pycalphad.variables import T
from collections import OrderedDict

# SGTE 1991 Pure Element Reference for the reference structures
# Reference:
# A.T. Dinsdale, SGTE data for pure elements, Calphad, Volume 15, Issue 4, 1991, Pages 317-425, ISSN 0364-5916,
# http://dx.doi.org/10.1016/0364-5916(91)90030-N.
# http://www.sciencedirect.com/science/article/pii/036459169190030N
SGTE91Stable = OrderedDict(
[('AG',
  Piecewise((-3.98587e-7*T**3 - 0.001790585*T**2 - 23.8463314*T*log(T) + 118.202013*T - 7209.512 - 12011/T, And(T < 1234.93, T >= 298.15)), (-33.472*T*log(T) + 190.266404*T - 15095.252 + 1.411773e+29/T**9, And(T < 3000.0, T >= 1234.93)))),
 ('AL',
  Piecewise((-8.77664e-7*T**3 - 0.001884662*T**2 - 24.3671976*T*log(T) + 137.093038*T - 7976.15 + 74092/T, And(T < 700.0, T >= 298.15)), (-5.764227e-6*T**3 + 0.018531982*T**2 - 38.5844296*T*log(T) + 223.048446*T - 11276.24 + 74092/T, And(T < 933.47, T >= 700.0)), (-31.748192*T*log(T) + 188.684153*T - 11278.378 - 1.230524e+28/T**9, And(T < 2900.0, T >= 933.47)))),
 ('AM',
  Piecewise((-5.41038e-7*T**3 - 0.00559955*T**2 - 21.1868*T*log(T) + 89.645685*T - 6639.201 - 30424/T, And(T < 1329.0, T >= 298.15)), (-41.84*T*log(T) + 241.107269*T - 21702.938, And(T < 3000.0, T >= 1329.0)))),
 ('AS',
  Piecewise((-0.00271613*T**2 - 23.3144*T*log(T) + 122.211069*T - 7270.447 + 11600/T, And(T < 1090.0, T >= 298.15)), (-29.216037*T*log(T) + 163.457433*T - 10454.913, And(T < 1200.0, T >= 1090.0)))),
 ('AU',
  Piecewise((3.79625e-7*T**3 - 0.00385924*T**2 - 22.75455*T*log(T) + 106.830098*T - 6938.856 - 25097/T, And(T < 929.4, T >= 298.15)), (-1.1518713e-5*T**3 + 0.08756015*T**2 - 155.706745*T*log(T) + 1021.69543*T - 93586.481 + 10637210/T, And(T < 1337.33, T >= 929.4)), (8.923844e-6*T**3 - 0.118216828*T**2 + 263.252259*T*log(T) - 2016.37825*T + 314067.829 - 67999832/T, And(T < 1735.8, T >= 1337.33)), (-30.9616*T*log(T) + 165.272524*T - 12133.783, And(T < 3200.0, T >= 1735.8)))),
 ('BA',
  Piecewise((-9.5e-11*T**3 - 0.0018314*T**2 - 42.889*T*log(T) + 233.78606*T - 17685.226 + 705880/T, And(T < 1000.0, T >= 298.15)), (-1.051353e-6*T**3 + 0.019504772*T**2 - 94.2824199*T*log(T) + 608.188389*T - 64873.614 + 8220192/T, And(T < 2995.0, T >= 1000.0)), (-32.2*T*log(T) + 136.780042*T + 8083.889, And(T < 4000.0, T >= 2995.0)))),
 ('B',
  Piecewise((6.18878e-7*T**3 - 0.006864515*T**2 - 15.6641*T*log(T) + 107.111864*T - 7735.284 + 370843/T, And(T < 1100.0, T >= 298.15)), (-2.556e-8*T**3 - 0.00079809*T**2 - 26.6047*T*log(T) + 184.801744*T - 16649.474 + 1748270/T, And(T < 2348.0, T >= 1100.0)), (1.34719e-7*T**3 - 0.00159488*T**2 - 31.5957527*T*log(T) + 231.336244*T - 36667.582 + 11205883/T, And(T < 3000.0, T >= 2348.0)), (-31.4*T*log(T) + 222.396264*T - 21530.653, And(T < 6000.0, T >= 3000.0)))),
 ('BE',
  Piecewise((-1.60413e-7*T**3 - 0.00284715*T**2 - 21.204*T*log(T) + 137.560219*T - 8553.651 + 293690/T, And(T < 1527.0, T >= 298.15)), (-1.119065e-6*T**3 + 0.021078651*T**2 - 103.9843*T*log(T) + 772.405844*T - 121305.858 + 27251743/T, And(T < 3000.0, T >= 1527.0)))),
 ('BI',
  Piecewise((-8.381598e-6*T**3 + 0.012338888*T**2 - 28.4096529*T*log(T) + 128.418925*T - 7817.776, And(T < 544.55, T >= 298.15)), (1.3499885e-5*T**3 - 0.075311163*T**2 + 51.8556592*T*log(T) - 393.650351*T + 30208.022 - 3616168/T + 1.66145e+25/T**9, And(T < 800.0, T >= 544.55)), (-1.046e-6*T**3 + 0.0074266*T**2 - 35.9824*T*log(T) + 182.548971*T - 11045.664 + 1.66145e+25/T**9, And(T < 1200.0, T >= 800.0)), (-27.196*T*log(T) + 124.77144*T - 7581.312 + 1.66145e+25/T**9, And(T < 3000.0, T >= 1200.0)))),
 ('CA',
  Piecewise((-0.01110455*T**2 - 16.3138*T*log(T) + 72.794266*T - 4955.062 - 133574/T, And(T < 1115.0, T >= 298.15)), (-1.2438e-6*T**3 + 0.023733814*T**2 - 114.292247*T*log(T) + 799.982066*T - 107304.428 + 18245540/T, And(T < 3000.0, T >= 1115.0)))),
 ('CC',
  Piecewise((-0.0004723*T**2 - 24.3*T*log(T) + 170.73*T - 17368.441 + 2562600/T - 264300000.0/T**2 + 12000000000.0/T**3, And(T < 6000.0, T >= 298.15)))),
 ('CD',
  Piecewise((-0.006273908*T**2 - 22.0442408*T*log(T) + 99.506198*T - 7083.469 - 6966/T, And(T < 594.22, T >= 298.15)), (-8.99604e-7*T**3 + 0.008832011*T**2 - 45.1611543*T*log(T) + 256.812233*T - 20064.971 + 1241290/T, And(T < 1500.0, T >= 594.22)), (-29.7064*T*log(T) + 148.20548*T - 9027.489, And(T < 1600.0, T >= 1500.0)))),
 ('CE',
  Piecewise((-3.20773e-7*T**3 - 0.0067103*T**2 - 22.3664*T*log(T) + 84.23022*T - 7160.519 - 18117/T, And(T < 1000.0, T >= 298.15)), (-1.930297e-6*T**3 + 0.026046487*T**2 - 101.32248*T*log(T) + 659.4604*T - 79678.506 + 11531707/T, And(T < 2000.0, T >= 1000.0)), (-37.6978*T*log(T) + 190.370192*T - 14198.639, And(T < 4000.0, T >= 2000.0)))),
 ('CO',
  Piecewise((-1.7348e-7*T**3 - 0.002654739*T**2 - 25.0861*T*log(T) + 133.36601*T + 310.241 + 72527/T, And(T < 1768.0, T >= 298.15)), (-40.5*T*log(T) + 253.28374*T - 17197.666 + 9.3488e+30/T**9, And(T < 6000.0, T >= 1768.0)))),
 ('CR',
  Piecewise((-1.47721e-6*T**3 + 0.00189435*T**2 - 26.908*T*log(T) + 157.48*T - 8856.94 + 139250/T, And(T < 2180.0, T >= 298.15)), (-50*T*log(T) + 344.18*T - 34869.344 - 2.88526e+32/T**9, And(T < 6000.0, T >= 2180.0)))),
 ('CS',
  Piecewise((-0.000127907669*T**3 + 0.2029422*T**2 - 90.5212584*T*log(T) + 436.899787*T - 17373.82 + 245245/T, And(T < 301.59, T >= 200.0)), (-4.074846e-6*T**3 + 0.02043269*T**2 - 46.7273304*T*log(T) + 218.689955*T - 13553.817 + 181528/T + 7.8016e+21/T**9, And(T < 2000.0, T >= 301.59)))),
 ('CU',
  Piecewise((1.29223e-7*T**3 - 0.00265684*T**2 - 24.112392*T*log(T) + 130.485235*T - 7770.458 + 52478/T, And(T < 1357.77, T >= 298.15)), (-31.38*T*log(T) + 183.803828*T - 13542.026 + 3.64167e+29/T**9, And(T < 3200.0, T >= 1357.77)))),
 ('DY',
  Piecewise((-5.86914125e-7*T**3 - 0.000761683657*T**2 - 26.3917167*T*log(T) + 102.307412*T - 7937.16586 + 4010.90565/T, And(T < 1000.0, T >= 100.0)), (-3.49702836e-6*T**3 + 0.0166909801*T**2 - 43.8283359*T*log(T) + 214.012934*T - 13733.328 + 0.0173619874/T, And(T < 1654.15, T >= 1000.0)), (-2.76169148e-6*T**3 + 0.0578301681*T**2 - 272.123952*T*log(T) + 2032.1415*T - 404681.371 + 109616238.0/T, And(T < 3000.0, T >= 1654.15)))),
 ('ER',
  Piecewise((-9.52557e-7*T**3 + 0.000995792*T**2 - 28.3846744*T*log(T) + 116.698964*T - 8489.136 + 9581/T, And(T < 1802.0, T >= 298.15)), (-3.041405e-6*T**3 + 0.065950553*T**2 - 298.135131*T*log(T) + 2233.10212*T - 445688.206 + 123973199.0/T, And(T < 3200.0, T >= 1802.0)))),
 ('EU',
  Piecewise((-4.006564e-6*T**3 + 0.00931735*T**2 - 32.8418896*T*log(T) + 135.836737*T - 9864.965 + 102717/T, And(T < 1095.0, T >= 298.15)), (-8.809866e-6*T**3 + 0.114530917*T**2 - 309.357101*T*log(T) + 2174.73304*T - 287423.476 + 48455305/T, And(T < 1900.0, T >= 1095.0)))),
 ('FE',
  Piecewise((-5.8927e-8*T**3 - 0.00439752*T**2 - 23.5143*T*log(T) + 124.134*T + 1225.7 + 77359/T, And(T < 1811.0, T >= 298.15)), (-46*T*log(T) + 299.31255*T - 25383.581 + 2.29603e+31/T**9, And(T < 6000.0, T >= 1811.0)))),
 ('GA',
  Piecewise((-0.000118575257*T**3 + 0.227155636*T**2 - 108.228783*T*log(T) + 585.263691*T - 21312.331 + 439954/T, And(T < 302.91, T >= 200.0)), (-4.0173e-8*T**3 + 0.0001506*T**2 - 26.0692906*T*log(T) + 132.73019*T - 7055.643 - 118332/T + 1.64547e+23/T**9, And(T < 4000.0, T >= 302.91)))),
 ('GD',
  Piecewise((-3.14674076e-7*T**3 - 0.00285240521*T**2 - 24.7214131*T*log(T) + 97.13101*T - 6834.5855 - 8665.73348/T, And(T < 1000.0, T >= 200.0)), (-6.61211607e-7*T**3 - 0.00185225011*T**2 - 24.6598297*T*log(T) + 95.6919924*T - 6483.25362, And(T < 1508.15, T >= 1000.0)), (-6.39165948e-7*T**3 + 0.0150644246*T**2 - 101.800197*T*log(T) + 699.125537*T - 123124.992 + 29356890.3/T, And(T < 3600.0, T >= 1508.15)))),
 ('GE',
  Piecewise((-1.513694e-6*T**3 + 0.005568297*T**2 - 29.5337682*T*log(T) + 165.635573*T - 9486.153 + 163298/T, And(T < 900.0, T >= 298.15)), (-0.003672527*T**2 - 19.8536239*T*log(T) + 102.86087*T - 5689.239, And(T < 1211.4, T >= 900.0)), (-27.6144*T*log(T) + 156.708024*T - 9548.204 - 8.59809e+28/T**9, And(T < 3200.0, T >= 1211.4)))),
 ('HF',
  Piecewise((-4.77e-10*T**3 - 0.004146145*T**2 - 22.7075*T*log(T) + 110.744026*T - 6987.297 - 22590/T, And(T < 2506.0, T >= 298.15)), (-7.575759e-6*T**3 + 0.1735215*T**2 - 787.536383*T*log(T) + 6193.60999*T - 1446776.33 + 501742495.0/T, And(T < 3000.0, T >= 2506.0)))),
 ('HG',
  Piecewise((0.00118398213*T**3 - 2.0282337*T**2 + 618.193308*T*log(T) - 3348.19466*T + 82356.855 - 2366612/T, And(T < 234.32, T >= 200.0)), (-3.20695e-6*T**3 + 0.0097977*T**2 - 32.257*T*log(T) + 135.232291*T - 8961.207 + 6670/T, And(T < 400.0, T >= 234.32)), (-1.077802e-6*T**3 + 0.00318535*T**2 - 28.414*T*log(T) + 112.33345*T - 7970.627 - 41095/T, And(T < 700.0, T >= 400.0)), (8.737e-9*T**3 - 0.00166775*T**2 - 24.87*T*log(T) + 90.797305*T - 7161.338 - 27495/T, And(T < 2000.0, T >= 700.0)))),
 ('HO',
  Piecewise((2.375467e-6*T**3 - 0.00827315*T**2 - 23.4879*T*log(T) + 86.593171*T - 7612.429, And(T < 600.0, T >= 298.15)), (-4.829733e-6*T**3 + 0.01820065*T**2 - 39.6932*T*log(T) + 182.475691*T - 10917.688, And(T < 900.0, T >= 600.0)), (3.233133e-6*T**3 - 0.0424634*T**2 + 48.0595*T*log(T) - 421.474473*T + 46646.188 - 7185900/T, And(T < 1200.0, T >= 900.0)), (-1.112352e-6*T**3 - 0.01082725*T**2 + 8.28608*T*log(T) - 156.162846*T + 27786.061 - 6183850/T, And(T < 1703.0, T >= 1200.0)), (-6.824652e-6*T**3 + 0.139111904*T**2 - 558.950682*T*log(T) + 4248.37906*T - 825364.662 + 219952973.0/T, And(T < 3000.0, T >= 1703.0)))),
 ('IN',
  Piecewise((-2.120321e-6*T**3 - 0.00572566*T**2 - 21.8386*T*log(T) + 92.338115*T - 6978.89 - 22906/T, And(T < 429.75, T >= 298.15)), (-8.367e-8*T**3 + 0.00054607*T**2 - 27.4562*T*log(T) + 124.476588*T - 7033.516 - 211708/T + 3.53116e+22/T**9, And(T < 3800.0, T >= 429.75)))),
 ('IR',
  Piecewise((-0.003091976*T**2 - 22.7944*T*log(T) + 118.780119*T - 6936.288 - 20083/T, And(T < 1215.0, T >= 298.15)), (-4.7969e-7*T**3 - 26.085*T*log(T) + 140.066697*T - 8123.73, And(T < 2719.0, T >= 1215.0)), (1.844977e-6*T**3 - 0.047176402*T**2 + 152.498874*T*log(T) - 1258.35297*T + 290529.037 - 92987250/T, And(T < 4000.0, T >= 2719.0)))),
 ('KK',
  Piecewise((-8.4949147e-5*T**3 + 0.146211135*T**2 - 77.0571464*T*log(T) + 389.624197*T - 16112.929 + 243385/T, And(T < 336.53, T >= 200.0)), (-2.64387e-6*T**3 + 0.012167386*T**2 - 39.2885968*T*log(T) + 192.586544*T - 11122.441 + 43251/T + 1.19223e+22/T**9, And(T < 2200.0, T >= 336.53)))),
 ('LA',
  Piecewise((-0.001295165*T**2 - 26.34*T*log(T) + 120.284604*T - 7968.403, And(T < 550.0, T >= 298.15)), (6.8932e-7*T**3 - 0.008371705*T**2 - 17.1659411*T*log(T) + 59.06113*T - 3381.413 - 399448/T, And(T < 2000.0, T >= 550.0)), (-34.3088*T*log(T) + 181.390071*T - 15608.882, And(T < 4000.0, T >= 2000.0)))),
 ('LI',
  Piecewise((-1.9869816e-5*T**3 + 0.035466931*T**2 - 38.940488*T*log(T) + 217.637482*T - 10583.817 + 159994/T, And(T < 453.6, T >= 200.0)), (-0.000571066077*T**3 + 2.25832944*T**2 - 1702.88865*T*log(T) + 10547.8799*T - 559579.123 + 33885874/T, And(T < 500.0, T >= 453.6)), (-4.38058e-7*T**3 + 0.002633221*T**2 - 31.2283718*T*log(T) + 179.278285*T - 9062.994 - 102387/T, And(T < 3000.0, T >= 500.0)))),
 ('LU',
  Piecewise((-1.790717e-6*T**3 + 0.00519165*T**2 - 29.812*T*log(T) + 146.536283*T - 8788.329 + 39723/T, And(T < 700.0, T >= 298.15)), (-1.50147e-6*T**3 + 0.00371416*T**2 - 29.0095*T*log(T) + 142.327643*T - 9043.057 + 141549/T, And(T < 1700.0, T >= 700.0)), (-0.0119001*T**2 - 1.83986*T*log(T) - 46.91844*T + 6940.092, And(T < 1936.0, T >= 1700.0)), (-1.661174e-6*T**3 + 0.041800748*T**2 - 239.019502*T*log(T) + 1829.37943*T - 404023.691 + 124825465.0/T, And(T < 3700.0, T >= 1936.0)))),
 ('MG',
  Piecewise((-1.393669e-6*T**3 + 0.0004858*T**2 - 26.1849782*T*log(T) + 143.675547*T - 8367.34 + 78950/T, And(T < 923.0, T >= 298.15)), (-34.3088*T*log(T) + 204.716215*T - 14130.185 + 1.038192e+28/T**9, And(T < 3000.0, T >= 923.0)))),
 ('MN',
  Piecewise((-0.00734768*T**2 - 23.4582*T*log(T) + 130.059*T - 8115.28 + 69827/T, And(T < 1519.0, T >= 298.15)), (-48*T*log(T) + 312.2648*T - 28733.41 + 1.656847e+30/T**9, And(T < 2000.0, T >= 1519.0)))),
 ('MO',
  Piecewise((-1.30927e-10*T**4 + 5.66283e-7*T**3 - 0.003443396*T**2 - 23.56414*T*log(T) + 131.9197*T - 7746.302 + 65812/T, And(T < 2896.0, T >= 298.15)), (-42.63829*T*log(T) + 283.559746*T - 30556.41 - 4.849315e+33/T**9, And(T < 5000.0, T >= 2896.0)))),
 ('NA',
  Piecewise((-4.3638283e-5*T**3 + 0.072306633*T**2 - 51.0393608*T*log(T) + 260.548732*T - 11989.434 + 132154/T, And(T < 370.87, T >= 200.0)), (-1.70664e-6*T**3 + 0.009745854*T**2 - 38.1198801*T*log(T) + 199.619999*T - 11009.884 + 34342/T + 1.65071e+23/T**9, And(T < 2300.0, T >= 370.87)))),
 ('NB',
  Piecewise((-3.5012e-7*T**3 + 0.000203475*T**2 - 26.4711*T*log(T) + 142.045475*T - 8519.353 + 93399/T, And(T < 2750.0, T >= 298.15)), (-41.77*T*log(T) + 271.720843*T - 37669.3 + 1.528238e+32/T**9, And(T < 6000.0, T >= 2750.0)))),
 ('ND',
  Piecewise((-2.6923e-6*T**3 + 0.000556125*T**2 - 27.0858*T*log(T) + 111.10239*T - 8402.93 + 34887/T, And(T < 900.0, T >= 298.15)), (-1.802e-6*T**3 - 0.00420402*T**2 - 22.7536*T*log(T) + 83.662617*T - 6984.083, And(T < 1128.0, T >= 900.0)), (-6.048207e-6*T**3 + 0.078615997*T**2 - 238.182873*T*log(T) + 1673.04075*T - 225610.846 + 38810350/T, And(T < 1799.0, T >= 1128.0)), (-48.7854*T*log(T) + 276.257088*T - 25742.331, And(T < 1800.0, T >= 1799.0)))),
 ('NI',
  Piecewise((-0.0048407*T**2 - 22.096*T*log(T) + 117.854*T - 5179.159, And(T < 1728.0, T >= 298.15)), (-43.1*T*log(T) + 279.135*T - 27840.655 + 1.12754e+31/T**9, And(T < 3000.0, T >= 1728.0)))),
 ('N',
  Piecewise((2.681e-9*T**3 - 0.00176686*T**2 - 12.7819*T*log(T) - 9.45425*T - 3750.675 - 32374/T, And(T < 950.0, T >= 298.15)), (3.0097e-8*T**3 - 0.00065107*T**2 - 16.3699*T*log(T) + 17.2003*T - 7358.85 + 563070/T, And(T < 3350.0, T >= 950.0)), (-8.333e-9*T**3 + 0.000239754*T**2 - 20.4695*T*log(T) + 50.26*T - 16392.8 + 4596375/T, And(T < 6000.0, T >= 3350.0)))),
 ('NP',
  Piecewise((-0.04127725*T**2 + 4.0543*T*log(T) - 57.531347*T + 241.888 - 402857/T, And(T < 553.0, T >= 298.15)), (-2.483917e-6*T**3 + 0.0284592*T**2 - 102.523*T*log(T) + 664.27337*T - 57015.112 + 4796910/T, And(T < 1799.0, T >= 553.0)), (-45.3964*T*log(T) + 255.780866*T - 12092.736, And(T < 4000.0, T >= 1799.0)))),
 ('O',
  Piecewise((6.61845833e-7*T**3 - 0.005098875*T**2 - 11.1355*T*log(T) - 25.503038*T - 3480.87 - 38365/T, And(T < 1000.0, T >= 298.15)), (6.781e-9*T**3 - 0.0005957975*T**2 - 16.8138*T*log(T) + 12.659879*T - 6568.763 + 262905/T, And(T < 3300.0, T >= 1000.0)), (1.0721e-8*T**3 - 0.000425243*T**2 - 18.9536*T*log(T) + 31.259624*T - 13986.728 + 4383200/T, And(T < 6000.0, T >= 3300.0)))),
 ('OS',
  Piecewise((-0.00190427*T**2 - 23.5710242*T*log(T) + 126.369531*T - 7196.978, And(T < 3306.0, T >= 298.15)), (1.173861e-6*T**3 - 0.042489827*T**2 + 224.998034*T*log(T) - 1935.2137*T + 644910.07 - 312569031.0/T, And(T < 5500.0, T >= 3306.0)))),
 ('PA',
  Piecewise((-0.00621325*T**2 - 23.9116*T*log(T) + 111.973215*T - 7681.561, And(T < 1443.0, T >= 298.15)), (1.884933e-6*T**3 - 0.0263416*T**2 + 16.305*T*log(T) - 177.320253*T + 27955.763 - 5908900/T, And(T < 2176.0, T >= 1443.0)), (-47.2792*T*log(T) + 288.308639*T - 29949.683, And(T < 4000.0, T >= 2176.0)))),
 ('PB',
  Piecewise((-2.4395e-7*T**3 - 0.00365895*T**2 - 24.5242231*T*log(T) + 101.700244*T - 7650.085, And(T < 600.61, T >= 298.15)), (0.00154613*T**2 - 32.4913959*T*log(T) + 154.243182*T - 10531.095 + 8.05448e+25/T**9, And(T < 1200.0, T >= 600.61)), (9.8144e-8*T**3 - 0.002882943*T**2 - 18.9640637*T*log(T) + 53.139072*T + 4157.616 - 2696755/T + 8.05448e+25/T**9, And(T < 2100.0, T >= 1200.0)))),
 ('PD',
  Piecewise((-1.919875e-6*T**3 + 0.007120975*T**2 - 32.211*T*log(T) + 176.076315*T - 10204.027 + 168687/T, And(T < 900.0, T >= 298.15)), (1.91115e-7*T**3 - 0.00717522*T**2 - 13.5708*T*log(T) + 49.659892*T + 917.062 - 1112465/T, And(T < 1828.0, T >= 900.0)), (-6.2811e-8*T**3 + 0.002091396*T**2 - 54.2067086*T*log(T) + 370.102147*T - 67161.018 + 18683526/T, And(T < 4000.0, T >= 1828.0)))),
 ('P',
  Piecewise((-0.000104022667*T**3 + 0.290708*T**2 - 178.426*T*log(T) + 1026.69389*T - 43821.799 + 1632695/T, And(T < 317.3, T >= 250.0)), (-2.2829e-7*T**3 + 0.001715669*T**2 - 28.7335301*T*log(T) + 152.341487*T - 9587.448 + 172966/T, And(T < 1000.0, T >= 317.3)), (-26.326*T*log(T) + 135.876831*T - 8093.075, And(T < 3000.0, T >= 1000.0)))),
 ('PR',
  Piecewise((-2.5184333e-5*T**3 + 0.072929*T**2 - 68.9176*T*log(T) + 356.587384*T - 18803.379 + 507385/T, And(T < 500.0, T >= 298.15)), (-1.22951e-6*T**3 - 0.00497126*T**2 - 22.8909*T*log(T) + 82.427384*T - 7246.848, And(T < 800.0, T >= 500.0)), (1.5592233e-5*T**3 - 0.1288205*T**2 + 146.764*T*log(T) - 1073.55111*T + 95411.023 - 11588800/T, And(T < 1068.0, T >= 800.0)), (-3.0994702e-5*T**3 + 0.305181506*T**2 - 606.120311*T*log(T) + 4234.33311*T - 481663.131 + 70926840/T, And(T < 1204.0, T >= 1068.0)), (-42.9697*T*log(T) + 227.685155*T - 20014.678, And(T < 3800.0, T >= 1204.0)))),
 ('PT',
  Piecewise((-2.0138e-8*T**3 - 0.00248297*T**2 - 24.5526*T*log(T) + 124.388275*T - 7595.631 + 7974/T, And(T < 1300.0, T >= 298.15)), (-6.56946e-7*T**3 + 0.002321665*T**2 - 30.2527*T*log(T) + 161.529615*T - 9253.174 - 272106/T, And(T < 2041.5, T >= 1300.0)), (-7.59259e-7*T**3 + 0.020454938*T**2 - 136.192996*T*log(T) + 1019.35892*T - 222048.216 + 71539020/T, And(T < 4000.0, T >= 2041.5)))),
 ('PU',
  Piecewise((-0.02241*T**2 - 18.1258*T*log(T) + 80.301382*T - 7396.309, And(T < 400.0, T >= 298.15)), (2.63443e-7*T**3 - 0.00134493*T**2 - 42.4187*T*log(T) + 236.786603*T - 16605.962 + 579325/T, And(T < 944.0, T >= 400.0)), (-42.248*T*log(T) + 232.961553*T - 14462.156, And(T < 3000.0, T >= 944.0)))),
 ('RB',
  Piecewise((-0.000152236932*T**3 + 0.26277612*T**2 - 115.282589*T*log(T) + 583.580988*T - 21669.733 + 385754/T, And(T < 312.46, T >= 200.0)), (-4.6822e-7*T**3 + 0.000412369*T**2 - 29.1775424*T*log(T) + 117.050578*T - 7823.397 - 126310/T - 5.55029e+22/T**9, And(T < 900.0, T >= 312.46)), (-4.829082e-6*T**3 + 0.033795632*T**2 - 77.7006456*T*log(T) + 450.974149*T - 39488.142 + 3778006/T - 5.55029e+22/T**9, And(T < 1600.0, T >= 900.0)), (-8.61653e-6*T**3 + 0.08161687*T**2 - 191.262774*T*log(T) + 1287.78947*T - 159742.511 + 27738456/T - 5.55029e+22/T**9, And(T < 2100.0, T >= 1600.0)))),
 ('RE',
  Piecewise((1.92818e-7*T**3 - 0.00253505*T**2 - 24.348*T*log(T) + 128.421589*T - 7695.279 + 32915/T, And(T < 1200.0, T >= 298.15)), (-2.81835e-7*T**3 + 0.00224565*T**2 - 33.586*T*log(T) + 194.667426*T - 15775.998 + 1376270/T, And(T < 2400.0, T >= 1200.0)), (-7.88955e-7*T**3 + 0.01184945*T**2 - 67.956*T*log(T) + 462.110749*T - 70882.739 + 18075200/T, And(T < 3458.0, T >= 2400.0)), (1.053726e-6*T**3 - 0.033764567*T**2 + 140.831655*T*log(T) - 1211.37186*T + 346325.888 - 134548866.0/T, And(T < 5000.0, T >= 3458.0)), (-49.519*T*log(T) + 346.997842*T - 78564.296, And(T < 6000.0, T >= 5000.0)))),
 ('RH',
  Piecewise((-1.68032e-7*T**3 - 0.003424186*T**2 - 24.0178336*T*log(T) + 132.020923*T - 7848.828 + 55846/T, And(T < 1200.0, T >= 298.15)), (-1.512774e-6*T**3 + 0.00966345*T**2 - 48.3766632*T*log(T) + 305.771019*T - 28367.852 + 3348162/T, And(T < 2237.0, T >= 1200.0)), (-5.3978814e-5*T**3 + 1.04921361*T**2 - 3874.21058*T*log(T) + 30151.6342*T - 6237470.48 + 1880362180.0/T, And(T < 2450.0, T >= 2237.0)), (-50.58456*T*log(T) + 344.889895*T - 44863.489, And(T < 2500.0, T >= 2450.0)))),
 ('RU',
  Piecewise((1.7641e-7*T**3 - 0.004062566*T**2 - 22.9143287*T*log(T) + 127.866233*T - 7561.873 + 56377/T, And(T < 1500.0, T >= 298.15)), (-1.952433e-6*T**3 + 0.018726245*T**2 - 72.3241219*T*log(T) + 489.516214*T - 59448.103 + 11063885/T, And(T < 2607.0, T >= 1500.0)), (-0.000240245985*T**3 + 5.221639*T**2 - 21329.705*T*log(T) + 168610.517*T - 38588773 + 13082992600.0/T, And(T < 2740.0, T >= 2607.0)), (-51.8816*T*log(T) + 364.482314*T - 55768.304, And(T < 4500.0, T >= 2740.0)))),
 ('SB',
  Piecewise((-3.003415e-6*T**3 + 0.007748768*T**2 - 30.5130752*T*log(T) + 156.154689*T - 9242.858 + 100625/T, And(T < 903.78, T >= 298.15)), (-31.38*T*log(T) + 169.485872*T - 11738.83 + 1.616849e+27/T**9, And(T < 2000.0, T >= 903.78)))),
 ('SC',
  Piecewise((-1.64531e-6*T**3 + 0.00321892*T**2 - 28.1882*T*log(T) + 153.48097*T - 8689.547 + 72177/T, And(T < 800.0, T >= 298.15)), (-8.59345e-7*T**3 - 0.000573295*T**2 - 24.9132*T*log(T) + 132.759582*T - 7511.295, And(T < 1608.0, T >= 800.0)), (8.7398e-6*T**3 - 0.117529396*T**2 + 241.441051*T*log(T) - 1817.92245*T + 261143.04 - 50607159/T, And(T < 2000.0, T >= 1608.0)), (-44.2249*T*log(T) + 286.474338*T - 30515.246, And(T < 3200.0, T >= 2000.0)))),
 ('SE',
  Piecewise((-1.5318461e-5*T**3 + 0.02424314*T**2 - 33.6527*T*log(T) + 174.205877*T - 9376.371 + 102249/T, And(T < 494.0, T >= 298.15)), (-5.611026e-6*T**3 + 0.037144892*T**2 - 81.2006585*T*log(T) + 507.111538*T - 37546.134 + 2614263/T, And(T < 800.0, T >= 494.0)), (-35.1456*T*log(T) + 197.770166*T - 12193.47, And(T < 1000.0, T >= 800.0)))),
 ('SI',
  Piecewise((-3.552e-9*T**3 - 0.001912904*T**2 - 22.8317533*T*log(T) + 137.236859*T - 8162.609 + 176667/T, And(T < 1687.0, T >= 298.15)), (-27.196*T*log(T) + 167.281367*T - 9457.642 - 4.20369e+30/T**9, And(T < 3600.0, T >= 1687.0)))),
 ('SM',
  Piecewise((1.010345e-5*T**3 - 0.050254*T**2 - 1.6485*T*log(T) - 32.10748*T - 3872.013 - 82168/T, And(T < 700.0, T >= 298.15)), (-7.538383e-6*T**3 + 0.0474522*T**2 - 102.665*T*log(T) + 627.869894*T - 50078.215 + 3861770/T, And(T < 1190.0, T >= 700.0)), (2.7512152e-5*T**3 - 0.254986338*T**2 + 381.41982*T*log(T) - 2744.50976*T + 289719.819 - 40102102/T, And(T < 1345.0, T >= 1190.0)), (-50.208*T*log(T) + 282.194375*T - 23056.079, And(T < 2100.0, T >= 1345.0)))),
 ('SN',
  Piecewise((-3.192767e-6*T**3 + 0.00051185*T**2 - 25.858*T*log(T) + 122.765451*T - 7958.517 + 18440/T, And(T < 250.0, T >= 100.0)), (3.121167e-6*T**3 - 0.0188702*T**2 - 15.961*T*log(T) + 65.443315*T - 5855.135 - 61960/T, And(T < 505.08, T >= 250.0)), (2.623131e-6*T**3 - 0.016814429*T**2 - 8.2590486*T*log(T) + 4.005269*T + 2524.724 - 1081244/T - 1.2307e+25/T**9, And(T < 800.0, T >= 505.08)), (-28.4512*T*log(T) + 138.99688*T - 8256.959 - 1.2307e+25/T**9, And(T < 3000.0, T >= 800.0)))),
 ('SR',
  Piecewise((-1.67477e-7*T**3 - 0.00461225*T**2 - 23.905*T*log(T) + 107.183879*T - 7532.367 - 2055/T, And(T < 820.0, T >= 298.15)), (1.84189e-7*T**3 - 0.003251266*T**2 - 30.0905432*T*log(T) + 153.196104*T - 13380.102 + 850134/T, And(T < 3000.0, T >= 820.0)))),
 ('S',
  Piecewise((7.754333e-6*T**3 - 0.026529*T**2 - 11.007*T*log(T) + 55.417762*T - 5228.956, And(T < 368.3, T >= 298.15)), (1.402558e-6*T**3 - 0.010895125*T**2 - 17.941839*T*log(T) + 94.692922*T - 6513.769 + 39910/T, And(T < 1300.0, T >= 368.3)))),
 ('TA',
  Piecewise((1.70109e-7*T**3 - 0.002623033*T**2 - 23.7592624*T*log(T) + 119.139857*T - 7285.889 - 3293/T, And(T < 1300.0, T >= 298.15)), (-6.55136e-7*T**3 + 0.006167572*T**2 - 41.137088*T*log(T) + 243.88676*T - 22389.955 + 2429586/T, And(T < 2500.0, T >= 1300.0)), (1.95033e-7*T**3 - 0.017983376*T**2 + 78.5244752*T*log(T) - 722.59722*T + 229382.886 - 93813648/T, And(T < 3290.0, T >= 2500.0)), (-1.055148e-6*T**3 + 0.043117795*T**2 - 362.159132*T*log(T) + 2985.49125*T - 1042384.01 + 554714342.0/T, And(T < 6000.0, T >= 3290.0)))),
 ('TB',
  Piecewise((-2.5672833e-5*T**3 + 0.0832265*T**2 - 77.5006*T*log(T) + 409.309555*T - 20842.158 + 562430/T, And(T < 600.0, T >= 298.15)), (-8.05838e-7*T**3 - 0.002757005*T**2 - 25.8659*T*log(T) + 102.61162*T - 8772.606 + 172355/T, And(T < 1200.0, T >= 600.0)), (-1.067632e-6*T**3 - 0.001676335*T**2 - 25.9584*T*log(T) + 101.7776*T - 7944.942, And(T < 1562.0, T >= 1200.0)), (-2.044697e-6*T**3 + 0.041615159*T**2 - 200.215695*T*log(T) + 1456.04268*T - 265240.309 + 65043790/T, And(T < 3000.0, T >= 1562.0)))),
 ('TC',
  Piecewise((-0.002954747*T**2 - 24.3394*T*log(T) + 132.5101*T - 7947.794 + 63855/T, And(T < 2430.0, T >= 298.15)), (-47*T*log(T) + 318.286*T - 47759.99 + 6.63829e+32/T**9, And(T < 4000.0, T >= 2430.0)))),
 ('TE',
  Piecewise((-5.240417e-6*T**3 + 0.01583435*T**2 - 35.6687*T*log(T) + 183.372894*T - 10544.679 + 155015/T, And(T < 722.66, T >= 298.15)), (5.006367e-6*T**3 - 0.0362361*T**2 + 13.004*T*log(T) - 129.265373*T + 9160.595 - 1286810/T, And(T < 1150.0, T >= 722.66)), (-32.5596*T*log(T) + 174.901226*T - 12781.349, And(T < 1600.0, T >= 1150.0)))),
 ('TH',
  Piecewise((-5.2883e-7*T**3 - 0.00236725*T**2 - 24.841*T*log(T) + 117.022775*T - 7732.08 + 13010/T, And(T < 1633.0, T >= 298.15)), (2.36893e-7*T**3 - 0.00358025*T**2 - 39.107*T*log(T) + 237.654918*T - 37352.871 + 7981000/T, And(T < 2900.0, T >= 1633.0)), (-46.024*T*log(T) + 283.979845*T - 33353.313, And(T < 4000.0, T >= 2900.0)))),
 ('TI',
  Piecewise((1.06716e-7*T**3 - 0.004777975*T**2 - 23.9933*T*log(T) + 133.615208*T - 8059.921 + 72636/T, And(T < 900.0, T >= 298.15)), (-9.0876e-8*T**3 - 0.0042033*T**2 - 23.9887*T*log(T) + 132.988068*T - 7811.815 + 42680/T, And(T < 1155.0, T >= 900.0)), (2.02715e-7*T**3 - 0.0081465*T**2 - 14.9466*T*log(T) + 66.976538*T + 908.837 - 1477660/T, And(T < 1941.0, T >= 1155.0)), (-3.04747e-7*T**3 + 0.008204849*T**2 - 87.2182461*T*log(T) + 638.806871*T - 124526.786 + 36699805/T, And(T < 4000.0, T >= 1941.0)))),
 ('TL',
  Piecewise((-1.21807e-7*T**3 - 0.0033063*T**2 - 25.2274*T*log(T) + 107.140405*T - 8104.038 + 42058/T, And(T < 577.0, T >= 200.0)), (-5.19136e-7*T**3 + 0.005228106*T**2 - 38.4130658*T*log(T) + 196.771926*T - 15406.859 + 729665/T, And(T < 1800.0, T >= 577.0)))),
 ('TM',
  Piecewise((-3.831156e-6*T**3 + 0.012110965*T**2 - 34.3664974*T*log(T) + 151.037648*T - 10016.715 + 95982/T, And(T < 700.0, T >= 298.15)), (-3.96694e-7*T**3 + 0.000444753*T**2 - 32.1951269*T*log(T) + 147.957496*T - 14701.965 + 1091664/T, And(T < 1600.0, T >= 700.0)), (-0.003384563*T**2 - 25.1816969*T*log(T) + 97.98144*T - 8669.227, And(T < 1818.0, T >= 1600.0)), (1.1689185e-5*T**3 - 0.19093039*T**2 + 534.082763*T*log(T) - 4147.40063*T + 727125.608 - 180382220.0/T, And(T < 2300.0, T >= 1818.0)))),
 ('U',
  Piecewise((-4.42605e-6*T**3 + 0.00125156*T**2 - 26.9182*T*log(T) + 130.955151*T - 8407.734 + 38568/T, And(T < 955.0, T >= 298.15)), (-48.66*T*log(T) + 292.121093*T - 22521.8, And(T < 3000.0, T >= 955.0)))),
 ('V',
  Piecewise((1.2175e-7*T**3 - 0.003098*T**2 - 24.134*T*log(T) + 133.346053*T - 7930.43 + 69460/T, And(T < 790.0, T >= 298.15)), (-6.8e-7*T**3 + 6.25e-5*T**2 - 25.9*T*log(T) + 143.291093*T - 7967.842, And(T < 2183.0, T >= 790.0)), (-47.43*T*log(T) + 321.140783*T - 41689.864 + 6.44389e+31/T**9, And(T < 4000.0, T >= 2183.0)))),
 ('W',
  Piecewise((-5.33e-11*T**4 + 2.07e-7*T**3 - 0.001936*T**2 - 24.1*T*log(T) + 130.4*T - 7646.311 + 44500/T, And(T < 3695.0, T >= 298.15)), (-54*T*log(T) + 389.362335*T - 82868.801 + 1.528621e+33/T**9, And(T < 6000.0, T >= 3695.0)))),
 ('YB',
  Piecewise((-2.2242e-5*T**3 + 0.04227115*T**2 - 40.0791*T*log(T) + 189.327664*T - 9370.941, And(T < 553.0, T >= 298.15)), (-0.00256065*T**2 - 26.7591*T*log(T) + 121.065655*T - 8192.154, And(T < 1033.0, T >= 553.0)), (1.421719e-6*T**3 - 0.017961331*T**2 + 2.7623966*T*log(T) - 89.478241*T + 16034.89 - 3631462/T, And(T < 2000.0, T >= 1033.0)))),
 ('Y',
  Piecewise((-4.17561786e-7*T**3 - 0.00175716414*T**2 - 25.6656992*T*log(T) + 128.572856*T - 8011.09379 + 26911.509/T, And(T < 1000.0, T >= 100.0)), (-8.2534534e-8*T**3 - 0.0038211802*T**2 - 23.4941827*T*log(T) + 114.497104*T - 7179.74574, And(T < 1795.15, T >= 1000.0)), (-7.22513088e-8*T**3 + 0.00231774379*T**2 - 56.9527111*T*log(T) + 382.124727*T - 67480.7761 + 18077162.6/T, And(T < 3700.0, T >= 1795.15)))),
 ('ZN',
  Piecewise((-1.264963e-6*T**3 - 0.001712034*T**2 - 23.701314*T*log(T) + 118.470069*T - 7285.787, And(T < 692.68, T >= 298.15)), (-31.38*T*log(T) + 172.34566*T - 11070.559 + 4.70514e+26/T**9, And(T < 1700.0, T >= 692.68)))),
 ('ZR',
  Piecewise((-0.00437791*T**2 - 24.1618*T*log(T) + 125.64905*T - 7827.595 + 34971/T, And(T < 2128.0, T >= 130.0)), (-42.144*T*log(T) + 262.724183*T - 26085.921 - 1.342896e+31/T**9, And(T < 6000.0, T >= 2128.0))))
])

# SGTE 1991 Pure Component Reference for selected structures
# This is the chemical potential relative to the standard state (reference structure, 1 mol, 298.15 K, 101325 Pa)
# The symbol GHSER[element] refers to an entry in SGTE above
# SGTE91[(element, reference structure)] = SGTE91Stable[element]
# Reference:
# A.T. Dinsdale, SGTE data for pure elements, Calphad, Volume 15, Issue 4, 1991, Pages 317-425, ISSN 0364-5916,
# http://dx.doi.org/10.1016/0364-5916(91)90030-N.
# http://www.sciencedirect.com/science/article/pii/036459169190030N
SGTE91 = OrderedDict(
[(('AG', 'BCC_A2'), Piecewise((-1.05*T + Symbol('GHSERAG') + 3400, And(T < 3000.0, T >= 298.15)))),
 (('AG', 'CUB_A13'), Piecewise((-1.8826*T + Symbol('GHSERAG') + 3765.6, And(T < 3000.0, T >= 298.15)))),
 (('AG', 'FCC_A1'), Piecewise((Symbol('GHSERAG'), And(T < 3000.0, T >= 298.15)))),
 (('AG', 'HCP_A3'), Piecewise((0.3*T + Symbol('GHSERAG') + 300, And(T < 3000.0, T >= 298.15)))),
 (('AG', 'LIQUID'), Piecewise((-1.033905e-20*T**7 - 8.89102*T + Symbol('GHSERAG') + 11025.076, And(T < 1234.93, T >= 298.15)),
                              (-33.472*T*log(T) + 180.964656*T - 3587.111, And(T < 3000.0, T >= 1234.93)))),
 (('AL', 'BCC_A2'), Piecewise((-4.813*T + Symbol('GHSERAL') + 10083, And(T < 2900.0, T >= 298.15)))),
 (('AL', 'BCT_A5'), Piecewise((-4.813*T + Symbol('GHSERAL') + 10083, And(T < 2900.0, T >= 298.15)))),
 (('AL', 'CBCC_A12'), Piecewise((-4.813*T + Symbol('GHSERAL') + 10083.4, And(T < 2900.0, T >= 298.15)))),
 (('AL', 'CUB_A13'), Piecewise((-4.8116*T + Symbol('GHSERAL') + 10920.44, And(T < 2900.0, T >= 298.15)))),
 (('AL', 'DIAMOND_A4'), Piecewise((30*T + Symbol('GHSERAL'), And(T < 2900.0, T >= 298.15)))),
 (('AL', 'FCC_A1'), Piecewise((Symbol('GHSERAL'), And(T < 2900.0, T >= 298.15)))),
 (('AL', 'HCP_A3'), Piecewise((-1.8*T + Symbol('GHSERAL') + 5481, And(T < 2900.0, T >= 298.15)))),
 (('AL', 'LIQUID'), Piecewise((7.9337e-20*T**7 - 11.841867*T + Symbol('GHSERAL') + 11005.029, And(T < 700.0, T >= 298.15)),
                              (7.9337e-20*T**7 - 11.841867*T + Symbol('GHSERAL') + 11005.03, And(T < 933.47, T >= 700.0)),
                              (-31.748192*T*log(T) + 177.430178*T - 795.996, And(T < 2900.0, T >= 933.47)))),
 (('AM', 'BCC_A2'), Piecewise((5.0e-12*T**3 + 5.0e-8*T**2 - 4.531331*T + Symbol('GHSERAM') + 5973.805, And(T < 999.0, T >= 298.15)),
                              (2.291117e-6*T**3 - 0.0190671*T**2 - 15.8832*T*log(T) + 63.93115*T - 7800.332 + 2287195/T, And(T < 1339.0, T >= 999.0)),
                              (-39.748*T*log(T) + 219.600832*T - 13153.887, And(T < 1449.0, T >= 1339.0)),
                              (1.801717e-6*T**3 - 0.02736485*T**2 + 33.413*T*log(T) - 326.394464*T + 70352.138 - 17379450/T, And(T < 2183.6, T >= 1449.0)),
                              (-3.740241*T + Symbol('GHSERAM') + 4777.694, And(T < 3000.0, T >= 2183.6)))),
 (('AM', 'DHCP'), Piecewise((Symbol('GHSERAM'), And(T < 3000.0, T >= 298.15)))),
 (('AM', 'FCC_A1'), Piecewise((-6.64773e-7*T**3 - 0.00294694*T**2 - 23.1377*T*log(T) + 99.204329*T - 5224.899 - 18507/T, And(T < 1018.0, T >= 298.15)),
                              (-3.75233e-7*T**3 - 0.005418*T**2 - 19.4406*T*log(T) + 73.800069*T - 2935.853 - 260435/T, And(T < 1548.7, T >= 1018.0)),
                              (0.246538*T + Symbol('GHSERAM') - 476.655, And(T < 3000.0, T >= 1548.7)))),
 (('AM', 'LIQUID'), Piecewise((-14.1205*T + Symbol('GHSERAM') + 19910.7, And(T < 3000.0, T >= 298.15)))),
 (('AS', 'BCC_A2'), Piecewise((-16.1*T + Symbol('GHSERAS') + 24874, And(T < 1200.0, T >= 298.15)))),
 (('AS', 'FCC_A1'), Piecewise((-14.74*T + Symbol('GHSERAS') + 24874, And(T < 1200.0, T >= 298.15)))),
 (('AS', 'HCP_A3'), Piecewise((-14*T + Symbol('GHSERAS') + 24874, And(T < 1200.0, T >= 298.15)))),
 (('AS', 'LIQUID'), Piecewise((-22.424679*T + Symbol('GHSERAS') + 24442.9, And(T < 1200.0, T >= 298.15)))),
 (('AS', 'RHOMBOHEDRAL_A7'), Piecewise((Symbol('GHSERAS'), And(T < 1200.0, T >= 298.15)))),
 (('AU', 'BCC_A2'), Piecewise((-1.1*T + Symbol('GHSERAU') + 4250, And(T < 3200, T >= 298.15)))),
 (('AU', 'FCC_A1'), Piecewise((Symbol('GHSERAU'), And(T < 3200.0, T >= 298.15)))),
 (('AU', 'HCP_A3'), Piecewise((1.6*T + Symbol('GHSERAU') + 240.75, And(T < 3200.0, T >= 298.15)))),
 (('AU', 'LIQUID'), Piecewise((-9.385866*T + Symbol('GHSERAU') + 12552, And(T < 3200.0, T >= 298.15)))),
 (('B', 'BETA_RHOMBO_B'), Piecewise((Symbol('GHSERBB'), And(T < 6000.0, T >= 298.15)))),
 (('B', 'DIAMOND_A4'), Piecewise((Symbol('GHSERBB'), And(T < 6000.0, T >= 298.15)))),
 (('B', 'FCC_A1'), Piecewise((-12.217*T + Symbol('GHSERBB') + 43514, And(T < 6000.0, T >= 298.15)))),
 (('B', 'GRAPHITE'), Piecewise((Symbol('GHSERBB') + 5000, And(T < 6000.0, T >= 298.15)))),
 (('B', 'HCP_A3'), Piecewise((-9.706*T + Symbol('GHSERBB') + 50208, And(T < 6000.0, T >= 298.15)))),
 (('B', 'LIQUID'), Piecewise((-20.268025*T + Symbol('GHSERBB') + 48458.559, And(T < 500.0, T >= 298.15)),
                             (5.07347e-7*T**3 - 0.007095669*T**2 - 14.9827763*T*log(T) + 82.101722*T + 41119.703 + 335484/T, And(T < 2348.0, T >= 500.0)),
                             (-31.4*T*log(T) + 200.94731*T + 28842.012, And(T < 3000.0, T >= 2348.0)),
                             (-21.448954*T + Symbol('GHSERBB') + 50372.665, And(T < 6000.0, T >= 3000.0)))),
 (('BA', 'BCC_A2'), Piecewise((Symbol('GHSERBA'), And(T < 4000.0, T >= 298.15)))),
 (('BA', 'FCC_A1'), Piecewise((0.6*T + Symbol('GHSERBA') + 1800, And(T < 4000.0, T >= 298.15)))),
 (('BA', 'HCP_A3'), Piecewise((1.3*T + Symbol('GHSERBA') + 2000, And(T < 4000.0, T >= 298.15)))),
 (('BA', 'LIQUID'), Piecewise((9.91223e-7*T**3 - 0.002346416*T**2 - 43.4961089*T*log(T) + 229.540143*T - 9738.988 + 723016/T, And(T < 1000.0, T >= 298.15)),
                              (2.7e-11*T**3 + 0.002154*T**2 - 45.103*T*log(T) + 235.49642*T - 7381.093 - 365/T, And(T < 2995.0, T >= 1000.0)),
                              (-4.568042*T + Symbol('GHSERBA') + 3856.393, And(T < 4000.0, T >= 2995.0)))),
 (('BE', 'BCC_A2'), Piecewise((9.61427e-7*T**3 - 0.008672487*T**2 - 17.1727841*T*log(T) + 109.411712*T - 1076.057 + 242309/T, And(T < 1527.0, T >= 298.15)),
                              (-30*T*log(T) + 196.411689*T - 6970.378, And(T < 1560.0, T >= 1527.0)),
                              (-5.9331e-8*T**3 - 0.000103629*T**2 - 27.7823769*T*log(T) + 178.131722*T - 2609.973 - 1250847/T, And(T < 3000.0, T >= 1560.0)))),
 (('BE', 'FCC_A1'), Piecewise((-1.085*T + Symbol('GHSERBE') + 6349, And(T < 3000.0, T >= 298.15)))),
 (('BE', 'HCP_A3'), Piecewise((Symbol('GHSERBE'), And(T < 3000.0, T >= 298.15)))),
 (('BE', 'LIQUID'), Piecewise((4.15958e-7*T**3 - 0.004821347*T**2 - 20.0497038*T*log(T) + 120.362788*T + 7511.838 + 281044/T, And(T < 1560.0, T >= 298.15)),
                              (-1.117e-9*T**3 - 0.0010572*T**2 - 25.486*T*log(T) + 156.961141*T + 5364.713 + 15920/T, And(T < 3000.0, T >= 1560.0)))),
 (('BI', 'BCC_A2'), Piecewise((-13.9*T + Symbol('GHSERBI') + 11297, And(T < 3000.0, T >= 298.15)))),
 (('BI', 'BCT_A5'), Piecewise((Symbol('GHSERBI') + 4184.07, And(T < 3000.0, T >= 298.15)))),
 (('BI', 'FCC_A1'), Piecewise((-12.5*T + Symbol('GHSERBI') + 9900, And(T < 3000.0, T >= 298.15)))),
 (('BI', 'HCP_A3'), Piecewise((-11.8*T + Symbol('GHSERBI') + 9900, And(T < 3000.0, T >= 298.15)))),
 (('BI', 'LIQUID'), Piecewise((-5.9549e-19*T**7 - 20.636509*T + Symbol('GHSERBI') + 11246.066, And(T < 544.55, T >= 298.15)),
                              (-20.810418*T + Symbol('GHSERBI') + 11336.26 - 1.66145e+25/T**9, And(T < 800.0, T >= 544.55)),
                              (-20.810418*T + Symbol('GHSERBI') + 11336.259 - 1.66145e+25/T**9, And(T < 1200.0, T >= 800.0)),
                              (-27.196*T*log(T) + 103.961021*T + 3754.947, And(T < 3000.0, T >= 1200.0)))),
 (('BI', 'RHOMBOHEDRAL_A7'), Piecewise((Symbol('GHSERBI'), And(T < 3000.0, T >= 298.15)))),
 (('BI', 'TETRAGONAL_A6'), Piecewise((Symbol('GHSERBI') + 4184.07, And(T < 3000.0, T >= 298.15)))),
 (('BI', 'TET_ALPHA1'), Piecewise((Symbol('GHSERBI') + 4234, And(T < 3000.0, T >= 298.15)))),
 (('C', 'DIAMOND_A4'), Piecewise((-0.0004723*T**2 - 24.31*T*log(T) + 175.61*T - 16359.441 + 2698000/T - 261000000.0/T**2 + 11100000000.0/T**3, And(T < 6000.0, T >= 298.15)))),
 (('C', 'GRAPHITE'), Piecewise((Symbol('GHSERCC'), And(T < 6000.0, T >= 298.15)))),
 (('C', 'LIQUID'), Piecewise((-24.63*T + Symbol('GHSERCC') + 117369, And(T < 6000.0, T >= 298.15)))),
 (('CA', 'BCC_A2'), Piecewise((-4.500217e-6*T**3 + 0.0072326*T**2 - 28.2541*T*log(T) + 142.970155*T - 7020.852 + 60578/T, And(T < 716.0, T >= 298.15)),
                              (-0.0161921*T**2 - 6.276*T*log(T) + 1.999694*T + 1640.475 - 523000/T, And(T < 1115.0, T >= 716.0)),
                              (-1.704079e-6*T**3 + 0.032543127*T**2 - 143.872698*T*log(T) + 1023.54905*T - 142331.096 + 25353771/T, And(T < 3000.0, T >= 1115.0)))),
 (('CA', 'FCC_A1'), Piecewise((Symbol('GHSERCA'), And(T < 3000.0, T >= 298.15)))),
 (('CA', 'HCP_A3'), Piecewise((0.7*T + Symbol('GHSERCA') + 500, And(T < 3000.0, T >= 298.15)))),
 (('CA', 'LIQUID'), Piecewise((-10.310466*T + Symbol('GHSERCA') + 10799.908, And(T < 500.0, T >= 298.15)),
                              (3.338303e-6*T**3 - 0.02266537*T**2 - 8.9874787*T*log(T) + 18.2979*T + 7838.856 - 230193/T, And(T < 1115.0, T >= 500.0)),
                              (-35*T*log(T) + 188.9223*T - 2654.938, And(T < 3000.0, T >= 1115.0)))),
 (('CD', 'FCC_A1'), Piecewise((-0.92*T + Symbol('GHSERCD') + 892.3, And(T < 1600.0, T >= 298.15)))),
 (('CD', 'HCP_A3'), Piecewise((Symbol('GHSERCD'), And(T < 1600.0, T >= 298.15)))),
 (('CD', 'LIQUID'), Piecewise((-10.296916*T + Symbol('GHSERCD') + 6128.444, And(T < 400.0, T >= 298.15)),
                              (2.8899781e-5*T**3 - 0.115159917*T**2 + 53.1313898*T*log(T) - 371.046869*T + 21716.884 - 1271815/T, And(T < 594.22, T >= 400.0)),
                              (-29.7064*T*log(T) + 138.251107*T - 3252.303, And(T < 1500.0, T >= 594.22)),
                              (-9.954373*T + Symbol('GHSERCD') + 5775.186, And(T < 1600.0, T >= 1500.0)))),
 (('CD', 'TETRAGONAL_A6'), Piecewise((-0.92*T + Symbol('GHSERCD') + 892.3, And(T < 1600.0, T >= 298.15)))),
 (('CE', 'BCC_A2'), Piecewise((4.784299e-6*T**3 - 0.029098402*T**2 - 7.7305867*T*log(T) - 5.21501*T - 1354.69 - 196303/T, And(T < 1000.0, T >= 298.15)),
                              (-37.6142*T*log(T) + 187.449688*T - 12101.106, And(T < 1072.0, T >= 1000.0)),
                              (2.348e-9*T**3 - 5.7145e-5*T**2 - 37.4627992*T*log(T) + 186.333811*T - 11950.375 - 25897/T, And(T < 4000.0, T >= 1072.0)))),
 (('CE', 'DHCP'), Piecewise((0.56886*T + Symbol('GHSERCE') - 190, And(T < 4000.0, T >= 298.15)))),
 (('CE', 'FCC_A1'), Piecewise((Symbol('GHSERCE'), And(T < 4000.0, T >= 298.15)))),
 (('CE', 'HCP_A3'), Piecewise((Symbol('GHSERCE') + 50000, And(T < 4000.0, T >= 298.15)))),
 (('CE', 'LIQUID'), Piecewise((4.827734e-6*T**3 - 0.02936407*T**2 - 7.5383948*T*log(T) - 11.423898*T + 4117.865 - 198834/T, And(T < 1000.0, T >= 298.15)),
                              (-37.6978*T*log(T) + 183.023193*T - 6730.605, And(T < 2000.0, T >= 1000.0)),
                              (-7.346999*T + Symbol('GHSERCE') + 7468.034, And(T < 4000.0, T >= 2000.0)))),
 (('CO', 'BCC_A2'), Piecewise((-0.7138*T + Symbol('GHSERCO') + 2938, And(T < 6000.0, T >= 298.15)))),
 (('CO', 'CBCC_A12'), Piecewise((Symbol('GHSERCO') + 4155, And(T < 6000.0, T >= 298.15)))),
 (('CO', 'CUB_A13'), Piecewise((Symbol('GHSERCO') + 3155, And(T < 6000.0, T >= 298.15)))),
 (('CO', 'FCC_A1'), Piecewise((-0.615248*T + Symbol('GHSERCO') + 427.591, And(T < 1768.0, T >= 298.15)),
                              (-0.615253*T + Symbol('GHSERCO') + 427.591, And(T < 6000.0, T >= 1768.0)))),
 (('CO', 'HCP_A3'), Piecewise((Symbol('GHSERCO'), And(T < 6000.0, T >= 298.15)))),
 (('CO', 'LIQUID'), Piecewise((-2.19801e-21*T**7 - 8.931932*T + Symbol('GHSERCO') + 15085.037, And(T < 1768.0, T >= 298.15)),
                              (-40.5*T*log(T) + 243.599944*T - 846.61, And(T < 6000.0, T >= 1768.0)))),
 (('CR', 'BCC_A2'), Piecewise((Symbol('GHSERCR'), And(T < 6000.0, T >= 298.15)))),
 (('CR', 'CBCC_A12'), Piecewise((2.7196*T + Symbol('GHSERCR') + 11087, And(T < 6000.0, T >= 298.15)))),
 (('CR', 'CUB_A13'), Piecewise((0.6276*T + Symbol('GHSERCR') + 15899, And(T < 6000.0, T >= 298.15)))),
 (('CR', 'FCC_A1'), Piecewise((0.163*T + Symbol('GHSERCR') + 7284, And(T < 6000.0, T >= 298.15)))),
 (('CR', 'HCP_A3'), Piecewise((Symbol('GHSERCR') + 4438, And(T < 6000.0, T >= 298.15)))),
 (('CR', 'LAVES_C15'), Piecewise((-4.43163e-6*T**3 + 0.00568305*T**2 - 80.724*T*log(T) + 472.44*T - 11570.82 + 417750/T, And(T < 2180.0, T >= 298.15)),
                                 (-150*T*log(T) + 1032.54*T - 89608.032 - 8.65578e+32/T**9, And(T < 6000.0, T >= 2180.0)))),
 (('CR', 'LIQUID'), Piecewise((2.37615e-21*T**7 - 11.420225*T + Symbol('GHSERCR') + 24339.955, And(T < 2180.0, T >= 298.15)),
                              (-50*T*log(T) + 335.616316*T - 16459.984, And(T < 6000.0, T >= 2180.0)))),
 (('CS', 'BCC_A2'), Piecewise((Symbol('GHSERCS'), And(T < 2000.0, T >= 200.0)))),
 (('CS', 'FCC_A1'), Piecewise((1.3*T + Symbol('GHSERCS') + 500, And(T < 2000.0, T >= 200.0)))),
 (('CS', 'HCP_A3'), Piecewise((2*T + Symbol('GHSERCS') + 500, And(T < 2000.0, T >= 200.0)))),
 (('CS', 'LIQUID'), Piecewise((-3.56867e-18*T**7 - 6.931035*T + Symbol('GHSERCS') + 2091.141, And(T < 301.59, T >= 200.0)),
                              (-6.961111*T + Symbol('GHSERCS') + 2099.779 - 7.8016e+21/T**9, And(T < 2000.0, T >= 301.59)))),
 (('CU', 'BCC_A2'), Piecewise((-1.255*T + Symbol('GHSERCU') + 4017, And(T < 3200.0, T >= 298.15)))),
 (('CU', 'FCC_A1'), Piecewise((Symbol('GHSERCU'), And(T < 3200.0, T >= 298.15)))),
 (('CU', 'HCP_A3'), Piecewise((0.2*T + Symbol('GHSERCU') + 600, And(T < 3200.0, T >= 298.15)))),
 (('CU', 'LAVES_C15'), Piecewise((3.87669e-7*T**3 - 0.00797052*T**2 - 72.337176*T*log(T) + 391.455705*T - 8311.374 + 157434/T, And(T < 1357.77, T >= 298.15)),
                                 (-94.14*T*log(T) + 551.411484*T - 25626.078 + 1.092501e+30/T**9, And(T < 3200.0, T >= 1357.77)))),
 (('CU', 'LIQUID'), Piecewise((-5.8489e-21*T**7 - 9.511904*T + Symbol('GHSERCU') + 12964.735, And(T < 1357.77, T >= 298.15)),
                              (-31.38*T*log(T) + 173.881484*T - 46.545, And(T < 3200.0, T >= 1357.77)))),
 (('DY', 'BCC_A2'), Piecewise((-0.566616181*T + Symbol('GHSERDY') + 1508.1802, And(T < 1000.0, T >= 100.0)),
                              (2.04076075e-5*T**3 - 0.224042148*T**2 + 391.515418*T*log(T) - 2868.04585*T + 327500.062 - 48652656.5/T, And(T < 1654.15, T >= 1000.0)),
                              (-50.208*T*log(T) + 291.409631*T - 33708.7949, And(T < 1685.15, T >= 1654.15)),
                              (-7.7437116e-8*T**3 + 0.0015254673*T**2 - 55.2811171*T*log(T) + 330.318068*T - 40775.4966 + 1776589.32/T, And(T < 3000.0, T >= 1685.15)))),
 (('DY', 'HCP_A3'), Piecewise((Symbol('GHSERDY'), And(T < 3000.0, T >= 100.0)))),
 (('DY', 'LIQUID'), Piecewise((-7.5443643*T + Symbol('GHSERDY') + 13196.6185, And(T < 1000.0, T >= 100.0)), (1.76197799e-5*T**3 - 0.196153225*T**2 + 341.302578*T*log(T) - 2519.78614*T + 300126.971 - 43071677.5/T, And(T < 1685.15, T >= 1000.0)),
                              (-49.9151*T*log(T) + 282.205014*T - 21864.7344, And(T < 3000.0, T >= 1685.15)))),
 (('ER', 'HCP_A3'), Piecewise((Symbol('GHSERER'), And(T < 3200.0, T >= 298.15)))),
 (('ER', 'LIQUID'), Piecewise((-10.241846*T + Symbol('GHSERER') + 19382.102, And(T < 500.0, T >= 298.15)),
                              (1.316517e-6*T**3 - 0.014414687*T**2 - 12.0761776*T*log(T) + 0.355564*T + 17912.678 - 528122/T, And(T < 1802.0, T >= 500.0)),
                              (-38.702*T*log(T) + 187.623024*T + 747.131, And(T < 3200.0, T >= 1802.0)))),
 (('EU', 'BCC_A2'), Piecewise((Symbol('GHSEREU'), And(T < 1900.0, T >= 298.15)))),
 (('EU', 'LIQUID'), Piecewise((-7.175215*T + Symbol('GHSEREU') + 8382.505, And(T < 400.0, T >= 298.15)),
                              (5.452934e-6*T**3 - 0.036811218*T**2 + 4.3501554*T*log(T) - 103.688201*T + 10972.726 - 646908/T, And(T < 1095.0, T >= 299.15)),
                              (-38.11624*T*log(T) + 175.517247*T - 6890.641, And(T < 1900.0, T >= 300.15)))),
 (('FE', 'BCC_A2'), Piecewise((Symbol('GHSERFE'), And(T < 6000.0, T >= 298.15)))),
 (('FE', 'CBCC_A12'), Piecewise((Symbol('GHSERFE') + 4745, And(T < 6000.0, T >= 298.15)))),
 (('FE', 'CUB_A13'), Piecewise((Symbol('GHSERFE') + 3745, And(T < 6000.0, T >= 298.15)))),
 (('FE', 'FCC_A1'), Piecewise((0.00064*T**2 - 1.15*T*log(T) + 8.282*T + Symbol('GHSERFE') - 1462.4, And(T < 1811.0, T >= 298.15)),
                              (0.94001*T + Symbol('GHSERFE') - 1713.815 + 4.9251e+30/T**9, And(T < 6000.0, T >= 1811.0)))),
 (('FE', 'HCP_A3'), Piecewise((0.00064*T**2 - 1.15*T*log(T) + 12.591*T + Symbol('GHSERFE') - 3705.78, And(T < 1811.0, T >= 298.15)),
                              (5.24951*T + Symbol('GHSERFE') - 3957.199 + 4.9251e+30/T**9, And(T < 6000.0, T >= 1811.0)))),
 (('FE', 'LIQUID'), Piecewise((-3.67516e-21*T**7 - 6.55843*T + Symbol('GHSERFE') + 12040.17, And(T < 1811.0, T >= 298.15)),
                              (-46*T*log(T) + 291.302*T - 10838.83, And(T < 6000.0, T >= 1811.0)))),
 (('GA', 'BCC_A2'), Piecewise((-11.7*T + Symbol('GHSERGA') + 4500, And(T < 4000.0, T >= 200.0)))),
 (('GA', 'BCT_A5'), Piecewise((-9.8*T + Symbol('GHSERGA') + 3846, And(T < 4000.0, T >= 200.0)))),
 (('GA', 'FCC_A1'), Piecewise((-10.2*T + Symbol('GHSERGA') + 3800, And(T < 4000.0, T >= 200.0)))),
 (('GA', 'HCP_A3'), Piecewise((-9.5*T + Symbol('GHSERGA') + 4500, And(T < 4000.0, T >= 200.0)))),
 (('GA', 'LIQUID'), Piecewise((-7.0171e-17*T**7 - 18.073995*T + Symbol('GHSERGA') + 5491.298, And(T < 302.91, T >= 200.0)),
                              (-18.681147*T + Symbol('GHSERGA') + 5666.455 - 1.64547e+23/T**9, And(T < 4000.0, T >= 302.91)))),
 (('GA', 'ORTHORHOMBIC_GA'), Piecewise((Symbol('GHSERGA'), And(T < 4000.0, T >= 200.0)))),
 (('GA', 'TETRAGONAL_A6'), Piecewise((-10*T + Symbol('GHSERGA') + 3500, And(T < 4000.0, T >= 200.0)))),
 (('GD', 'BCC_A2'), Piecewise((1.00000001e-7*T*log(T) - 2.11184585*T + Symbol('GHSERGD') + 3233.80866, And(T < 1000.0, T >= 200.0)),
                              (1.17915728e-5*T**3 - 0.119550229*T**2 + 180.097094*T*log(T) - 1349.58873*T + 152792.743 - 22038836/T, And(T < 1508.15, T >= 1000.0)),
                              (-38.960425*T*log(T) + 202.222057*T - 15783.7618, And(T < 1586.15, T >= 1508.15)),
                              (-3.77570269e-8*T**3 + 0.000858222759*T**2 - 41.904333*T*log(T) + 224.817909*T - 19850.5562 + 995428.573/T, And(T < 3600.0, T >= 1586.15)))),
 (('GD', 'HCP_A3'), Piecewise((Symbol('GHSERGD'), And(T < 3600.0, T >= 200.0)))),
 (('GD', 'LIQUID'), Piecewise((-8.32179974*T + Symbol('GHSERGD') + 13060.0262, And(T < 1000.0, T >= 200.0)),
                              (1.06945505e-5*T**3 - 0.108247135*T**2 + 159.352082*T*log(T) - 1208.70685*T + 146262.037 - 19678357/T, And(T < 1508.15, T >= 1000.0)),
                              (-38.5075*T*log(T) + 192.336215*T - 5397.314, And(T < 3600.0, T >= 1508.15)))),
 (('GE', 'BCC_A2'), Piecewise((-23.5*T + Symbol('GHSERGE') + 34100, And(T < 3200.0, T >= 298.15)))),
 (('GE', 'DIAMOND_A4'), Piecewise((Symbol('GHSERGE'), And(T < 3200.0, T >= 298.15)))),
 (('GE', 'FCC_A1'), Piecewise((-22.3*T + Symbol('GHSERGE') + 36000, And(T < 3200.0, T >= 298.15)))),
 (('GE', 'HCP_A3'), Piecewise((-21.5*T + Symbol('GHSERGE') + 35000, And(T < 3200.0, T >= 298.15)))),
 (('GE', 'LIQUID'), Piecewise((8.56632e-21*T**7 - 30.687043*T + Symbol('GHSERGE') + 37141.49, And(T < 900.0, T >= 298.15)),
                              (8.56632e-21*T**7 - 30.687044*T + Symbol('GHSERGE') + 37141.489, And(T < 1211.4, T >= 900.0)),
                              (-27.6144*T*log(T) + 126.324186*T + 27243.473, And(T < 3200.0, T >= 1211.4)))),
 (('HF', 'BCC_A2'), Piecewise((-1.446e-10*T**4 + 8.71923e-7*T**3 - 0.004206605*T**2 - 22.8995*T*log(T) + 103.836026*T + 5370.703 - 22590/T, And(T < 2506.0, T >= 298.15)),
                              (1.3427829e-5*T**3 - 0.286857065*T**2 + 1087.61412*T*log(T) - 8624.20573*T + 1912456.77 - 610085091.0/T, And(T < 3000.0, T >= 2506.0)))),
 (('HF', 'FCC_A1'), Piecewise((-2.2*T + Symbol('GHSERHF') + 10000, And(T < 3000.0, T >= 298.15)))),
 (('HF', 'HCP_A3'), Piecewise((Symbol('GHSERHF'), And(T < 3000.0, T >= 298.15)))),
 (('HF', 'LIQUID'), Piecewise((-10.953093*T + Symbol('GHSERHF') + 27402.256, And(T < 1000.0, T >= 298.15)),
                              (1.376466e-6*T**3 - 0.021262021*T**2 + 12.116812*T*log(T) - 149.91739*T + 49731.499 - 4449699/T, And(T < 2506.0, T >= 1000.0)),
                              (-44*T*log(T) + 265.470523*T - 4247.217, And(T < 3000.0, T >= 2506.0)))),
 (('HG', 'LIQUID'), Piecewise((Symbol('GHSERHG'), And(T < 2000.0, T >= 200.0)))),
 (('HG', 'RHOMBO_A10'), Piecewise((-2.4555667e-5*T**3 + 0.01699705*T**2 - 28.847*T*log(T) + 123.274598*T - 10668.401 + 13330/T, And(T < 234.32, T >= 200.0)),
                                  (-2.28298e-7*T**3 + 0.00107555*T**2 - 30.2091*T*log(T) + 135.928158*T - 11425.394 + 35545/T, And(T < 2000.0, T >= 234.32)))),
 (('HO', 'BCC_A2'), Piecewise((-2.060316*T + Symbol('GHSERHO') + 3731.229, And(T < 600.0, T >= 298.15)),
                              (-2.060316*T + Symbol('GHSERHO') + 3731.23, And(T < 900.0, T >= 600.0)),
                              (-2.060317*T + Symbol('GHSERHO') + 3731.229, And(T < 1000.0, T >= 900.0)),
                              (1.2168911e-5*T**3 - 0.13516576*T**2 + 218.937249*T*log(T) - 1635.74067*T + 185512.056 - 26729747/T, And(T < 1703.0, T >= 1000.0)),
                              (-48.116*T*log(T) + 272.946988*T - 28867.901, And(T < 1745.0, T >= 1703.0)),
                              (-1.287517e-6*T**3 + 0.025544089*T**2 - 134.793064*T*log(T) + 939.764197*T - 152754.148 + 32050889/T, And(T < 3000.0, T >= 1745.0)))),
 (('HO', 'HCP_A3'), Piecewise((Symbol('GHSERHO'), And(T < 3000.0, T >= 298.15)))),
 (('HO', 'LIQUID'), Piecewise((-9.992194*T + Symbol('GHSERHO') + 17262.172, And(T < 1000.0, T >= 298.15)),
                              (8.008222e-6*T**3 - 0.088196514*T**2 + 127.957778*T*log(T) - 994.683024*T + 124706.283 - 15727191/T, And(T < 1703.0, T >= 1000.0)),
                              (-43.932*T*log(T) + 230.793918*T - 9809.781, And(T < 3000.0, T >= 1703.0)))),
 (('IN', 'BCC_A2'), Piecewise((-0.8*T + Symbol('GHSERIN') + 800, And(T < 3800.0, T >= 298.15)))),
 (('IN', 'FCC_A1'), Piecewise((-0.1988*T + Symbol('GHSERIN') + 123, And(T < 3800.0, T >= 298.15)))),
 (('IN', 'HCP_A3'), Piecewise((-0.6868*T + Symbol('GHSERIN') + 533, And(T < 3800.0, T >= 298.15)))),
 (('IN', 'LIQUID'), Piecewise((-5.59058e-20*T**7 - 7.63686*T + Symbol('GHSERIN') + 3282.092, And(T < 429.75, T >= 298.15)),
                              (-7.640804*T + Symbol('GHSERIN') + 3283.706 - 3.53116e+22/T**9, And(T < 3800.0, T >= 429.75)))),
 (('IN', 'TETRAGONAL_A6'), Piecewise((Symbol('GHSERIN'), And(T < 3800.0, T >= 298.15)))),
 (('IN', 'TET_ALPHA1'), Piecewise((-0.16479*T + Symbol('GHSERIN') + 193, And(T < 3800.0, T >= 298.15)))),
 (('IR', 'BCC_A2'), Piecewise((-6.9*T + Symbol('GHSERIR') + 32000, And(T < 4000.0, T >= 298.15)))),
 (('IR', 'FCC_A1'), Piecewise((Symbol('GHSERIR'), And(T < 4000.0, T >= 298.15)))),
 (('IR', 'HCP_A3'), Piecewise((-0.6*T + Symbol('GHSERIR') + 4000, And(T < 4000.0, T >= 298.15)))),
 (('IR', 'LIQUID'), Piecewise((-6.312059*T + Symbol('GHSERIR') + 23455.244, And(T < 1000.0, T >= 298.15)),
                              (2.761831e-6*T**3 - 0.04638802*T**2 + 73.9517579*T*log(T) - 587.632815*T + 102217.789 - 13382612/T, And(T < 2719.0, T >= 1000.0)),
                              (-59.418*T*log(T) + 411.234043*T - 38347.217, And(T < 4000.0, T >= 2719.0)))),
 (('K', 'BCC_A2'), Piecewise((Symbol('GHSERKK'), And(T < 2200.0, T >= 200.0)))),
 (('K', 'FCC_A1'), Piecewise((1.3*T + Symbol('GHSERKK') + 50, And(T < 2200.0, T >= 200.0)))),
 (('K', 'HCP_A3'), Piecewise((2*T + Symbol('GHSERKK') + 50, And(T < 2200.0, T >= 200.0)))),
 (('K', 'LIQUID'), Piecewise((-9.44e-19*T**7 - 6.886859*T + Symbol('GHSERKK') + 2318.096, And(T < 336.53, T >= 200.0)),
                             (-6.902217*T + Symbol('GHSERKK') + 2323.019 - 1.19223e+22/T**9, And(T < 2200.0, T >= 336.53)))),
 (('LA', 'BCC_A2'), Piecewise((-5.25865e-7*T**3 - 0.004045175*T**2 - 21.7919*T*log(T) + 88.072353*T - 3952.161, And(T < 800.0, T >= 298.15)),
                              (4.9547989e-5*T**3 - 0.387295093*T**2 + 513.440708*T*log(T) - 3565.08252*T + 321682.673 - 36581228/T, And(T < 1134.0, T >= 800.0)),
                              (-39.5388*T*log(T) + 218.492988*T - 16377.894, And(T < 1193.0, T >= 1134.0)),
                              (-4.056395e-6*T**3 + 0.053968535*T**2 - 163.413074*T*log(T) + 1123.34397*T - 136609.91 + 21167204/T, And(T < 2000.0, T >= 1193.0)),
                              (-6.553756*T + Symbol('GHSERLA') + 7402.894, And(T < 4000.0, T >= 2000.0)))),
 (('LA', 'DHCP'), Piecewise((Symbol('GHSERLA'), And(T < 4000.0, T >= 298.15)))),
 (('LA', 'FCC_A1'), Piecewise((-5.25865e-7*T**3 - 0.004045175*T**2 - 21.7919*T*log(T) + 89.878761*T - 6109.797, And(T < 1134.0, T >= 298.15)),
                              (-3.066199e-6*T**3 + 0.042032405*T**2 - 139.346741*T*log(T) + 955.878375*T - 124598.976 + 20994153/T, And(T < 2000.0, T >= 1134.0)),
                              (-2.846081*T + Symbol('GHSERLA') + 3009.496, And(T < 4000.0, T >= 2000.0)))),
 (('LA', 'LIQUID'), Piecewise((2.93775e-6*T**3 - 0.020171603*T**2 - 11.0188191*T*log(T) + 18.23012*T + 5332.653 - 133541/T, And(T < 1134.0, T >= 298.15)),
                              (-34.3088*T*log(T) + 171.018431*T - 3942.004, And(T < 2000.0, T >= 1134.0)),
                              (-10.37164*T + Symbol('GHSERLA') + 11666.878, And(T < 4000.0, T >= 2000.0)))),
 (('LI', 'BCC_A2'), Piecewise((Symbol('GHSERLI'), And(T < 3000.0, T >= 200.0)))),
 (('LI', 'FCC_A1'), Piecewise((1.3*T + Symbol('GHSERLI') - 108, And(T < 3000, T >= 200.0)))),
 (('LI', 'HCP_A3'), Piecewise((2*T + Symbol('GHSERLI') - 154, And(T < 3000.0, T >= 200.0)))),
 (('LI', 'LIQUID'), Piecewise((-5.795621*T + Symbol('GHSERLI') + 2700.205, And(T < 250.0, T >= 200.0)),
                              (6.3955671e-5*T**3 - 0.182426463*T**2 + 61.6104424*T*log(T) - 362.187078*T + 12015.027 - 559968/T, And(T < 453.6, T >= 250.0)),
                              (-4.38058e-7*T**3 + 0.002633221*T**2 - 31.2283718*T*log(T) + 172.652183*T - 6057.31 - 102387/T, And(T < 500.0, T >= 453.6)),
                              (-6.626102*T + Symbol('GHSERLI') + 3005.684, And(T < 3000.0, T >= 500.0)))),
 (('LU', 'HCP_A3'), Piecewise((Symbol('GHSERLU'), And(T < 3700.0, T >= 298.15)))),
 (('LU', 'LIQUID'), Piecewise((-4.998883*T + Symbol('GHSERLU') + 12772.12, And(T < 600.0, T >= 298.15)),
                              (2.890636e-6*T**3 - 0.034238743*T**2 + 20.9392663*T*log(T) - 198.378793*T + 30389.863 - 2398650/T, And(T < 1936.0, T >= 600.0)),
                              (-47.9068*T*log(T) + 292.091104*T - 18994.687, And(T < 3700.0, T >= 1936.0)))),
 (('MG', 'BCC_A2'), Piecewise((-2.1*T + Symbol('GHSERMG') + 3100, And(T < 3000.0, T >= 298.15)))),
 (('MG', 'CBCC_A12'), Piecewise((-3.011*T + Symbol('GHSERMG') + 4602.4, And(T < 3000.0, T >= 298.15)))),
 (('MG', 'CUB_A13'), Piecewise((-3*T + Symbol('GHSERMG') + 5000, And(T < 3000.0, T >= 298.15)))),
 (('MG', 'FCC_A1'), Piecewise((-0.9*T + Symbol('GHSERMG') + 2600, And(T < 3000.0, T >= 298.15)))),
 (('MG', 'HCP_A3'), Piecewise((Symbol('GHSERMG'), And(T < 3000.0, T >= 298.15)))),
 (('MG', 'LAVES_C15'), Piecewise((-4.181007e-6*T**3 + 0.0014574*T**2 - 78.5549346*T*log(T) + 431.026641*T - 10102.02 + 236850/T, And(T < 923.0, T >= 298.15)),
                                 (-102.9264*T*log(T) + 614.148645*T - 27390.555 + 3.11458e+28/T**9, And(T < 3000.0, T >= 923.0)))),
 (('MG', 'LIQUID'), Piecewise((-8.0176e-20*T**7 - 8.83693*T + Symbol('GHSERMG') + 8202.243, And(T < 923.0, T >= 298.15)),
                              (-34.3088*T*log(T) + 195.324057*T - 5439.869, And(T < 3000.0, T >= 923.0)))),
 (('MN', 'BCC_A2'), Piecewise((-0.00744271*T**2 - 23.7*T*log(T) + 127.85*T - 3235.3 + 60000/T, And(T < 1519.0, T >= 298.15)),
                              (-4.5605*T + Symbol('GHSERMN') + 5544.58 - 3.91695e+29/T**9, And(T < 2000.0, T >= 1519.0)))),
 (('MN', 'CBCC_A12'), Piecewise((Symbol('GHSERMN'), And(T < 2000.0, T >= 298.15)))),
 (('MN', 'CUB_A13'), Piecewise((-0.00583359*T**2 - 24.8785*T*log(T) + 135.995*T - 5800.4 + 70269/T, And(T < 1519.0, T >= 298.15)),
                               (-0.9715*T + Symbol('GHSERMN') + 442.65 + 2.310723e+30/T**9, And(T < 2000.0, T >= 1519.0)))),
 (('MN', 'FCC_A1'), Piecewise((-0.006*T**2 - 24.5177*T*log(T) + 131.884*T - 3439.3 + 69600/T, And(T < 1519.0, T >= 298.15)),
                              (-2.5984*T + Symbol('GHSERMN') + 2663.31 + 2.205113e+30/T**9, And(T < 2000.0, T >= 1519.0)))),
 (('MN', 'HCP_A3'), Piecewise((-0.006*T**2 - 24.5177*T*log(T) + 133.007*T - 4439.3 + 69600/T, And(T < 1519.0, T >= 298.15)),
                              (-1.4754*T + Symbol('GHSERMN') + 1663.31 + 2.205113e+30/T**9, And(T < 2000.0, T >= 1519.0)))),
 (('MN', 'LAVES_C14'), Piecewise((-0.02204304*T**2 - 70.3746*T*log(T) + 390.177*T - 21345.84 + 209481/T, And(T < 1519.0, T >= 298.15)),
                                 (-144*T*log(T) + 936.7944*T - 83200.23 + 4.97054e+30/T**9, And(T < 2000.0, T >= 1519.0)))),
 (('MN', 'LIQUID'), Piecewise((-4.41929e-21*T**7 - 12.6208*T + Symbol('GHSERMN') + 17859.91, And(T < 1519.0, T >= 298.15)),
                              (-48*T*log(T) + 299.036*T - 9993.9, And(T < 2000.0, T >= 1519.0)))),
 (('MO', 'BCC_A2'), Piecewise((Symbol('GHSERMO'), And(T < 5000.0, T >= 298.15)))),
 (('MO', 'FCC_A1'), Piecewise((0.63*T + Symbol('GHSERMO') + 15200, And(T < 5000.0, T >= 298.15)))),
 (('MO', 'HCP_A3'), Piecewise((Symbol('GHSERMO') + 11550, And(T < 5000.0, T >= 298.15)))),
 (('MO', 'LIQUID'), Piecewise((4.24519e-22*T**7 - 14.694912*T + Symbol('GHSERMO') + 41831.347, And(T < 2896.0, T >= 298.15)),
                              (-42.63829*T*log(T) + 271.6697*T + 3538.963, And(T < 5000.0, T >= 2896.0)))),
 (('N', 'LIQUID'), Piecewise((59.02*T + Symbol('GHSERNN') + 29950, And(T < 6000.0, T >= 298.15)))),
 (('N2', 'GAS'), Piecewise((2*Symbol('GHSERNN'), And(T < 6000.0, T >= 298.15)))),
 (('NA', 'BCC_A2'), Piecewise((Symbol('GHSERNA'), And(T < 2300.0, T >= 200.0)))),
 (('NA', 'FCC_A1'), Piecewise((1.3*T + Symbol('GHSERNA') - 50, And(T < 2300.0, T >= 200.0)))),
 (('NA', 'HCP_A3'), Piecewise((2*T + Symbol('GHSERNA') - 104, And(T < 2300.0, T >= 200.0)))),
 (('NA', 'LIQUID'), Piecewise((-2.76132e-18*T**7 - 6.95218*T + Symbol('GHSERNA') + 2581.02, And(T < 370.87, T >= 200.0)),
                              (-7.032656*T + Symbol('GHSERNA') + 2609.444 - 1.65071e+23/T**9, And(T < 2300.0, T >= 370.87)))),
 (('NB', 'BCC_A2'), Piecewise((Symbol('GHSERNB'), And(T < 6000.0, T >= 298.15)))),
 (('NB', 'FCC_A1'), Piecewise((1.7*T + Symbol('GHSERNB') + 13500, And(T < 6000.0, T >= 298.15)))),
 (('NB', 'HCP_A3'), Piecewise((2.4*T + Symbol('GHSERNB') + 10000, And(T < 6000.0, T >= 298.15)))),
 (('NB', 'LIQUID'), Piecewise((-3.06098e-23*T**7 - 10.816418*T + Symbol('GHSERNB') + 29781.555, And(T < 2750.0, T >= 298.15)),
                              (-41.77*T*log(T) + 260.756148*T - 7499.398, And(T < 6000.0, T >= 2750.0)))),
 (('ND', 'BCC_A2'), Piecewise((-0.546281*T + Symbol('GHSERND') + 1437.295, And(T < 400.0, T >= 298.15)),
                              (7.287217e-6*T**3 - 0.050479*T**2 + 14.9956777*T*log(T) - 153.033976*T + 7312.2 - 831810/T, And(T < 1128.0, T >= 400.0)),
                              (-44.5596*T*log(T) + 239.677322*T - 18030.266, And(T < 1289.0, T >= 1128.0)),
                              (1.2408421e-5*T**3 - 0.156030778*T**2 + 311.409193*T*log(T) - 2363.9199*T + 334513.017 - 64319604/T, And(T < 1800.0, T >= 1289.0)))),
 (('ND', 'DHCP'), Piecewise((Symbol('GHSERND'), And(T < 1800.0, T >= 298.15)))),
 (('ND', 'FCC_A1'), Piecewise((Symbol('GHSERND') + 500, And(T < 1128.0, T >= 298.15)),
                              (3.00000011e-7*T*log(T) + 9.99999884e-7*T + Symbol('GHSERND') + 500, And(T < 1799.0, T >= 1128.0)),
                              (Symbol('GHSERND') + 500, And(T < 1800.0, T >= 1799.0)))),
 (('ND', 'LIQUID'), Piecewise((-1.585076*T + Symbol('GHSERND') + 5051.743, And(T < 300.0, T >= 298.15)),
                              (6.860782e-6*T**3 - 0.046955463*T**2 + 5.357301*T*log(T) - 86.593963*T + 5350.01 - 374380/T, And(T < 1128.0, T >= 300.0)),
                              (-48.7854*T*log(T) + 268.625903*T - 16335.232, And(T < 1799.0, T >= 1128.0)),
                              (-7.631185*T + Symbol('GHSERND') + 9407.099, And(T < 1800.0, T >= 1799.0)))),
 (('NI', 'BCC_A2'), Piecewise((-3.556*T + Symbol('GHSERNI') + 8715.084, And(T < 3000.0, T >= 298.15)))),
 (('NI', 'CBCC_A12'), Piecewise((Symbol('GHSERNI') + 3556, And(T < 3000.0, T >= 298.15)))),
 (('NI', 'CUB_A13'), Piecewise((Symbol('GHSERNI') + 2092, And(T < 3000.0, T >= 298.15)))),
 (('NI', 'FCC_A1'), Piecewise((Symbol('GHSERNI'), And(T < 3000.0, T >= 298.15)))),
 (('NI', 'HCP_A3'), Piecewise((1.255*T + Symbol('GHSERNI') + 1046, And(T < 3000.0, T >= 298.15)))),
 (('NI', 'LIQUID'), Piecewise((-3.82318e-21*T**7 - 9.397*T + Symbol('GHSERNI') + 16414.686, And(T < 1728.0, T >= 298.15)),
                              (-43.1*T*log(T) + 268.598*T - 9549.775, And(T < 3000.0, T >= 1728.0)))),
 (('NP', 'BCC_A2'), Piecewise((5.14743e-7*T**3 - 0.00251865*T**2 - 35.177*T*log(T) + 174.911817*T - 3224.664 + 302225/T, And(T < 856.0, T >= 298.15)),
                              (-36.401*T*log(T) + 180.807719*T - 2366.486, And(T < 917.0, T >= 856.0)),
                              (2.707217e-6*T**3 - 0.0343483*T**2 + 30.7734*T*log(T) - 297.324358*T + 50882.281 - 7500100/T, And(T < 1999.0, T >= 917.0)),
                              (-1.007779*T + Symbol('GHSERNP') - 2786.95, And(T < 4000.0, T >= 1999.0)))),
 (('NP', 'LIQUID'), Piecewise((2.941883e-6*T**3 - 0.0163885*T**2 - 31.229*T*log(T) + 160.024959*T - 4627.18 + 439915/T, And(T < 917.0, T >= 298.15)),
                              (-45.3964*T*log(T) + 247.671446*T - 7415.255, And(T < 1799.0, T >= 917.0)),
                              (-8.10942*T + Symbol('GHSERNP') + 4677.481, And(T < 4000.0, T >= 1799.0)))),
 (('NP', 'ORTHO_AC'), Piecewise((Symbol('GHSERNP'), And(T < 4000.0, T >= 298.15)))),
 (('NP', 'TETRAG_AD'), Piecewise((4.98465e-6*T**3 - 0.0161186*T**2 - 34.11*T*log(T) + 183.829213*T - 10157.32 + 532825/T, And(T < 555.0, T >= 298.15)),
                                 (-39.33*T*log(T) + 207.01896*T - 7873.688, And(T < 856.0, T >= 555.0)),
                                 (1.52726e-6*T**3 - 0.01921045*T**2 - 3.4265*T*log(T) - 46.64846*T + 19027.98 - 3564640/T, And(T < 1999.0, T >= 856.0)),
                                 (0.926171*T + Symbol('GHSERNP') - 3978.084, And(T < 4000.0, T >= 1999.0)))),
 (('O', 'BCC_A2'), Piecewise((Symbol('GHSEROO') + 30000, And(T < 6000.0, T >= 298.15)))),
 (('O', 'FCC_A1'), Piecewise((Symbol('GHSEROO') + 30000, And(T < 6000.0, T >= 298.15)))),
 (('O', 'LIQUID'), Piecewise((31.44*T + Symbol('GHSEROO') - 2648.9, And(T < 6000.0, T >= 298.15)))),
 (('O2', 'GAS'), Piecewise((2*Symbol('GHSEROO'), And(T < 6000.0, T >= 298.15)))),
 (('OS', 'BCC_A2'), Piecewise((-9*T + Symbol('GHSEROS') + 43500, And(T < 5500.0, T >= 298.15)))),
 (('OS', 'FCC_A1'), Piecewise((-2.5*T + Symbol('GHSEROS') + 13000, And(T < 5500.0, T >= 298.15)))),
 (('OS', 'HCP_A3'), Piecewise((Symbol('GHSEROS'), And(T < 5500.0, T >= 298.15)))),
 (('OS', 'LIQUID'), Piecewise((-8.473743*T + Symbol('GHSEROS') + 36460.17, And(T < 1000.0, T >= 298.15)),
                              (1.014279e-6*T**3 - 0.020464464*T**2 + 19.9382156*T*log(T) - 198.324341*T + 68715.318 - 6237261/T, And(T < 3306.0, T >= 1000.0)),
                              (-50*T*log(T) + 336.874526*T - 15903.192, And(T < 5500.0, T >= 3306.0)))),
 (('P', 'BCC_A2'), Piecewise((-2.418867e-6*T**3 + 0.0034121*T**2 - 25.55*T*log(T) + 135.412002*T + 18792.241 + 160095/T, And(T < 500.0, T >= 250.0)),
                             (3.93917e-7*T**3 - 0.00957685*T**2 - 14.368*T*log(T) + 64.411737*T + 23045.079 - 141375/T, And(T < 852.35, T >= 500.0)),
                             (-6.651929e-6*T**3 + 0.067272364*T**2 - 149.449556*T*log(T) + 1012.76962*T - 74639.613 + 12495943/T, And(T < 1500.0, T >= 852.35)),
                             (4.70235*T + Symbol('GHSERPP') + 28337.756, And(T < 3000.0, T >= 1500.0)))),
 (('P', 'FCC_A1'), Piecewise((-2.418867e-6*T**3 + 0.0034121*T**2 - 25.55*T*log(T) + 135.534002*T + 10842.441 + 160095/T, And(T < 500.0, T >= 250.0)),
                             (3.93917e-7*T**3 - 0.00957685*T**2 - 14.368*T*log(T) + 64.533737*T + 15095.279 - 141375/T, And(T < 852.35, T >= 500.0)),
                             (-6.651929e-6*T**3 + 0.067272364*T**2 - 149.449556*T*log(T) + 1012.89162*T - 82589.413 + 12495943/T, And(T < 1500.0, T >= 852.35)),
                             (4.82435*T + Symbol('GHSERPP') + 20387.956, And(T < 3000.0, T >= 1500.0)))),
 (('P', 'LIQUID'), Piecewise((3.9049371e-5*T**3 - 0.002898936*T**2 - 70.7440584*T*log(T) + 434.930931*T - 26316.111 + 1141147/T, And(T < 317.3, T >= 250.0)),
                             (-26.326*T*log(T) + 133.291873*T - 7232.449, And(T < 1000.0, T >= 317.3)),
                             (-2.584958*T + Symbol('GHSERPP') + 860.626, And(T < 3000.0, T >= 1000.0)))),
 (('P', 'RED_P'), Piecewise((-2.418867e-6*T**3 + 0.0034121*T**2 - 25.55*T*log(T) + 148.672002*T - 25976.559 + 160095/T, And(T < 500.0, T >= 250.0)),
                            (3.93917e-7*T**3 - 0.00957685*T**2 - 14.368*T*log(T) + 77.671737*T - 21723.721 - 141375/T, And(T < 852.35, T >= 500.0)),
                            (-6.651929e-6*T**3 + 0.067272364*T**2 - 149.449556*T*log(T) + 1026.02962*T - 119408.413 + 12495943/T, And(T < 1500.0, T >= 852.35)),
                            (17.96235*T + Symbol('GHSERPP') - 16431.044, And(T < 3000.0, T >= 1500.0)))),
 (('P', 'WHITE_P'), Piecewise((Symbol('GHSERPP'), And(T < 3000.0, T >= 250.0)))),
 (('PA', 'BCC_A2'), Piecewise((1.337387e-6*T**3 - 0.01322095*T**2 - 18.203*T*log(T) + 71.957409*T + 781.847 - 101600/T, And(T < 1443.0, T >= 298.15)),
                              (-39.748*T*log(T) + 220.478519*T - 10955.948, And(T < 1845.0, T >= 1443.0)),
                              (3.343867e-6*T**3 - 0.0637105*T**2 + 171.108*T*log(T) - 1397.15052*T + 284495.194 - 74992000/T, And(T < 2710.0, T >= 1845.0)),
                              (-2.212452*T + Symbol('GHSERPA') + 2064.512, And(T < 4000.0, T >= 2710.0)))),
 (('PA', 'BCT_AA'), Piecewise((Symbol('GHSERPA'), And(T < 4000.0, T >= 298.15)))),
 (('PA', 'LIQUID'), Piecewise((-9.544*T + Symbol('GHSERPA') + 16181.1, And(T < 1088.0, T >= 298.15)),
                              (3.075017e-6*T**3 - 0.0372478*T**2 + 30.336*T*log(T) - 278.789916*T + 48013.96 - 5064250/T, And(T < 1845.0, T >= 1088.0)),
                              (-47.2792*T*log(T) + 277.955437*T - 12508.174, And(T < 2176.0, T >= 1845.0)),
                              (-10.353202*T + Symbol('GHSERPA') + 17441.509, And(T < 4000.0, T >= 2176.0)))),
 (('PB', 'BCC_A2'), Piecewise((-1.1*T + Symbol('GHSERPB') + 2400, And(T < 2100.0, T >= 298.15)))),
 (('PB', 'BCT_A5'), Piecewise((3.52*T + Symbol('GHSERPB') + 489, And(T < 2100.0, T >= 298.15)))),
 (('PB', 'FCC_A1'), Piecewise((Symbol('GHSERPB'), And(T < 2100.0, T >= 298.15)))),
 (('PB', 'HCP_A3'), Piecewise((T + Symbol('GHSERPB') + 300, And(T < 2100.0, T >= 298.15)))),
 (('PB', 'LIQUID'), Piecewise((-6.019e-19*T**7 - 7.750683*T + Symbol('GHSERPB') + 4672.124, And(T < 600.61, T >= 298.15)),
                              (-8.067136*T + Symbol('GHSERPB') + 4853.137 - 8.05448e+25/T**9, And(T < 1200.0, T >= 600.61)),
                              (-8.067135*T + Symbol('GHSERPB') + 4853.137 - 8.05448e+25/T**9, And(T < 2100.0, T >= 1200.0)))),
 (('PB', 'RHOMBOHEDRAL_A7'), Piecewise((3.52*T + Symbol('GHSERPB') + 4900, And(T < 2100.0, T >= 298.15)))),
 (('PB', 'TETRAGONAL_A6'), Piecewise((-2.4395e-7*T**3 - 0.003637462*T**2 - 24.5639065*T*log(T) + 105.253344*T - 7176.855, And(T < 600.61, T >= 298.15)),
                                     (0.001567619*T**2 - 32.5310793*T*log(T) + 157.796282*T - 10057.865 + 8.05448e+25/T**9, And(T < 1200.0, T >= 600.61)),
                                     (2.1489e-5*T**2 - 0.0396834*T*log(T) + 3.5531*T + Symbol('GHSERPB') + 473.23, And(T < 2100.0, T >= 1200.0)))),
 (('PB', 'TET_ALPHA1'), Piecewise((-2.4395e-7*T**3 - 0.003656801*T**2 - 24.5639065*T*log(T) + 102.153384*T - 7466.885, And(T < 600.61, T >= 298.15)),
                                  (0.001548279*T**2 - 32.5310793*T*log(T) + 154.696322*T - 10347.895 + 8.05448e+25/T**9, And(T < 1200.0, T >= 600.61)),
                                  (2.148e-6*T**2 - 0.0396834*T*log(T) + 0.45314*T + Symbol('GHSERPB') + 183.2, And(T < 2100.0, T >= 1200.0)))),
 (('PD', 'BCC_A2'), Piecewise((-1.8*T + Symbol('GHSERPD') + 10500, And(T < 4000.0, T >= 298.15)))),
 (('PD', 'FCC_A1'), Piecewise((Symbol('GHSERPD'), And(T < 4000.0, T >= 298.15)))),
 (('PD', 'HCP_A3'), Piecewise((0.1*T + Symbol('GHSERPD') + 2000, And(T < 4000.0, T >= 298.15)))),
 (('PD', 'LIQUID'), Piecewise((-5.112162*T + Symbol('GHSERPD') + 11506.758, And(T < 600.0, T >= 298.15)),
                              (2.430675e-6*T**3 - 0.027266568*T**2 + 10.8922031*T*log(T) - 116.918419*T + 23405.778 - 1853674/T, And(T < 1828.0, T >= 600.0)),
                              (-41.17*T*log(T) + 251.416903*T - 12373.637, And(T < 4000.0, T >= 1828.0)))),
 (('PR', 'BCC_A2'), Piecewise((3.542468e-6*T**3 - 0.02284377*T**2 - 13.7470527*T*log(T) + 28.274853*T - 2863.651 - 87486/T, And(T < 1068.0, T >= 298.15)),
                              (-38.451*T*log(T) + 188.657121*T - 11985.919, And(T < 1204.0, T >= 1068.0)),
                              (1.76214e-7*T**3 - 0.004106833*T**2 - 26.6824313*T*log(T) + 100.826281*T + 953.224 - 2473024/T, And(T < 3800.0, T >= 1204.0)))),
 (('PR', 'DHCP'), Piecewise((Symbol('GHSERPR'), And(T < 3800.0, T >= 298.15)))),
 (('PR', 'LIQUID'), Piecewise((5.427467e-6*T**3 - 0.035119723*T**2 - 4.7344931*T*log(T) - 29.099465*T + 3848.961 - 207406/T, And(T < 1068.0, T >= 298.15)),
                              (-42.9697*T*log(T) + 219.508805*T - 10539.574, And(T < 1204.0, T >= 1068.0)),
                              (-8.17635*T + Symbol('GHSERPR') + 9475.104, And(T < 3800.0, T >= 1204.0)))),
 (('PT', 'BCC_A2'), Piecewise((-2.4*T + Symbol('GHSERPT') + 15000, And(T < 4000.0, T >= 298.15)))),
 (('PT', 'FCC_A1'), Piecewise((Symbol('GHSERPT'), And(T < 4000.0, T >= 298.15)))),
 (('PT', 'HCP_A3'), Piecewise((0.1*T + Symbol('GHSERPT') + 2500, And(T < 4000.0, T >= 298.15)))),
 (('PT', 'LIQUID'), Piecewise((-9.275183*T + Symbol('GHSERPT') + 20114.016, And(T < 600.0, T >= 298.15)),
                              (9.31516e-7*T**3 - 0.011551507*T**2 - 12.3403769*T*log(T) + 32.94182*T + 19023.491 - 601426/T, And(T < 2041.5, T >= 600.0)),
                              (-36.5*T*log(T) + 205.858962*T + 1404.468, And(T < 4000.0, T >= 2041.5)))),
 (('PU', 'ALPHA_PU'), Piecewise((Symbol('GHSERPU'), And(T < 3000.0, T >= 298.15)))),
 (('PU', 'BCC_A2'), Piecewise((2.061667e-6*T**3 - 0.009105*T**2 - 27.094*T*log(T) + 116.603882*T - 1358.984 + 20863/T, And(T < 745.0, T >= 298.15)),
                              (-33.72*T*log(T) + 156.878957*T - 2890.817, And(T < 956.0, T >= 745.0)),
                              (1.426922e-6*T**3 - 0.02023305*T**2 + 6.921*T*log(T) - 132.788248*T + 29313.619 - 4469245/T, And(T < 2071.0, T >= 956.0)),
                              (-5.539698*T + Symbol('GHSERPU') - 938.429, And(T < 3000.0, T >= 2071.0)))),
 (('PU', 'BETA_PU'), Piecewise((-0.00653*T**2 - 27.416*T*log(T) + 123.249151*T - 4873.654, And(T < 679.5, T >= 298.15)),
                               (1.524942e-6*T**3 - 0.0154772*T**2 - 15.7351*T*log(T) + 43.566585*T + 2435.094 - 864940/T, And(T < 1464.0, T >= 679.5)),
                               (-4.739938*T + Symbol('GHSERPU') + 503.094, And(T < 3000.0, T >= 1464.0)))),
 (('PU', 'FCC_A1'), Piecewise((-0.0054035*T**2 - 28.4781*T*log(T) + 127.586536*T - 3920.781, And(T < 990.0, T >= 298.15)),
                              (1.524942e-6*T**3 - 0.0154772*T**2 - 15.7351*T*log(T) + 41.52572*T + 3528.208 - 864940/T, And(T < 1464.0, T >= 990.0)),
                              (-6.780803*T + Symbol('GHSERPU') + 1596.208, And(T < 3000.0, T >= 1464.0)))),
 (('PU', 'GAMMA_PU'), Piecewise((-2.8103833e-5*T**3 + 0.0816415*T**2 - 77.5802*T*log(T) + 419.402655*T - 16766.303 + 574825/T, And(T < 487.9, T >= 298.15)),
                                (-0.0114795*T**2 - 22.0233*T*log(T) + 88.325069*T - 2942.77, And(T < 593.9, T >= 487.9)),
                                (6.92887e-7*T**3 - 0.0070383*T**2 - 32.3405*T*log(T) + 160.314641*T - 9336.967 + 630600/T, And(T < 1179.0, T >= 593.9)),
                                (-6.829936*T + Symbol('GHSERPU') + 2026.406, And(T < 3000.0, T >= 1179.0)))),
 (('PU', 'LIQUID'), Piecewise((-12.5133*T + Symbol('GHSERPU') + 6608.1, And(T < 3000.0, T >= 298.15)))),
 (('PU', 'TETRAGONAL_A6'), Piecewise((5.166667e-6*T**3 - 0.024006*T**2 - 16.43*T*log(T) + 54.586547*T - 496.178 - 158470/T, And(T < 736.0, T >= 298.15)),
                                     (-35.56*T*log(T) + 173.35008*T - 6122.307, And(T < 757.0, T >= 736.0)),
                                     (6.59882e-7*T**3 - 0.00937295*T**2 - 19.756*T*log(T) + 63.890352*T + 3982.078 - 1112565/T, And(T < 2157.0, T >= 757.0)),
                                     (-4.905143*T + Symbol('GHSERPU') - 738.383, And(T < 3000.0, T >= 2157.0)))),
 (('RB', 'BCC_A2'), Piecewise((Symbol('GHSERRB'), And(T < 2100.0, T >= 200.0)))),
 (('RB', 'FCC_A1'), Piecewise((1.3*T + Symbol('GHSERRB') + 200, And(T < 2100.0, T >= 200.0)))),
 (('RB', 'HCP_A3'), Piecewise((2*T + Symbol('GHSERRB') + 200, And(T < 2100.0, T >= 200.0)))),
 (('RB', 'LIQUID'), Piecewise((1.44078e-17*T**7 - 7.110486*T + Symbol('GHSERRB') + 2217.552, And(T < 312.46, T >= 200.0)),
                              (-6.960316*T + Symbol('GHSERRB') + 2172.865 + 5.55029e+22/T**9, And(T < 900.0, T >= 312.46)),
                              (-6.960316*T + Symbol('GHSERRB') + 2172.866 + 5.55029e+22/T**9, And(T < 1600.0, T >= 900.0)),
                              (-6.960315*T + Symbol('GHSERRB') + 2172.865 + 5.55029e+22/T**9, And(T < 2100.0, T >= 1600.0)))),
 (('RE', 'BCC_A2'), Piecewise((-6*T + Symbol('GHSERRE') + 29200, And(T < 6000.0, T >= 298.15)))),
 (('RE', 'FCC_A1'), Piecewise((-1.5*T + Symbol('GHSERRE') + 11000, And(T < 6000.0, T >= 298.15)))),
 (('RE', 'HCP_A3'), Piecewise((Symbol('GHSERRE'), And(T < 6000.0, T >= 298.15)))),
 (('RE', 'LIQUID'), Piecewise((-6.34538*T + Symbol('GHSERRE') + 23820.883, And(T < 1200.0, T >= 298.15)),
                              (-6.345379*T + Symbol('GHSERRE') + 23820.883, And(T < 2000.0, T >= 1200.0)),
                              (3.92854e-6*T**3 - 0.08939817*T**2 + 314.178898*T*log(T) - 2527.83846*T + 568842.665 - 163100987.0/T, And(T < 3458.0, T >= 2000.0)),
                              (-49.519*T*log(T) + 335.723691*T - 39044.888, And(T < 5000.0, T >= 3458.0)),
                              (-11.274151*T + Symbol('GHSERRE') + 39519.408, And(T < 6000.0, T >= 5000.0)))),
 (('RH', 'BCC_A2'), Piecewise((-4.7*T + Symbol('GHSERRH') + 19000, And(T < 2500.0, T >= 298.15)))),
 (('RH', 'FCC_A1'), Piecewise((Symbol('GHSERRH'), And(T < 2500.0, T >= 298.15)))),
 (('RH', 'HCP_A3'), Piecewise((-0.5*T + Symbol('GHSERRH') + 3000, And(T < 2500.0, T >= 298.15)))),
 (('RH', 'LIQUID'), Piecewise((-6.92133*T + Symbol('GHSERRH') + 19092.91, And(T < 700.0, T >= 298.15)),
                              (2.100572e-6*T**3 - 0.028665357*T**2 + 15.6492377*T*log(T) - 147.926418*T + 35898.508 - 2638940/T, And(T < 2237.0, T >= 700.0)),
                              (-50.58456*T*log(T) + 332.974832*T - 18208.54, And(T < 2450.0, T >= 2237.0)),
                              (-11.915063*T + Symbol('GHSERRH') + 26654.949, And(T < 2500.0, T >= 2450.0)))),
 (('RU', 'BCC_A2'), Piecewise((-6.2*T + Symbol('GHSERRU') + 26500, And(T < 4500.0, T >= 298.15)))),
 (('RU', 'FCC_A1'), Piecewise((-2.4*T + Symbol('GHSERRU') + 12500, And(T < 4500.0, T >= 298.15)))),
 (('RU', 'HCP_A3'), Piecewise((Symbol('GHSERRU'), And(T < 4500.0, T >= 298.15)))),
 (('RU', 'LIQUID'), Piecewise((-8.398748*T + Symbol('GHSERRU') + 27480.616, And(T < 800.0, T >= 298.15)),
                              (1.667839e-6*T**3 - 0.026524167*T**2 + 19.539341*T*log(T) - 179.818561*T + 50827.232 - 3861125/T, And(T < 2607.0, T >= 800.0)),
                              (-51.8816*T*log(T) + 349.673561*T - 17161.807, And(T < 2740.0, T >= 2607.0)),
                              (-14.808753*T + Symbol('GHSERRU') + 38606.497, And(T < 4500.0, T >= 2740.0)))),
 (('S', 'BCC_A2'), Piecewise((Symbol('GHSERSS') + 105000, And(T < 1300.0, T >= 298.15)))),
 (('S', 'FCC_A1'), Piecewise((Symbol('GHSERSS') + 105000, And(T < 1300.0, T >= 298.15)))),
 (('S', 'LIQUID'), Piecewise((-2.4942e-7*T**3 - 0.018629*T**2 - 15.504*T*log(T) + 77.905686*T - 4001.549 - 113945/T, And(T < 388.36, T >= 298.15)),
                             (-0.0102214167*T**3 + 32.79275*T**2 - 19762.4*T*log(T) + 118449.601*T - 5285183.35 + 264673500.0/T, And(T < 428.15, T >= 388.36)),
                             (-0.0529973333*T**3 + 135.3045*T**2 - 57607.3*T*log(T) + 319914.094*T - 8174995.23, And(T < 432.25, T >= 428.15)),
                             (-0.00101380333*T**3 + 2.845035*T**2 - 1371.85*T*log(T) + 7758.85593*T - 219408.801, And(T < 453.15, T >= 432.25)),
                             (5.18835e-5*T**3 - 0.2531915*T**2 + 202.958*T*log(T) - 1336.35027*T + 92539.872 - 8202200/T, And(T < 717.0, T >= 453.15)),
                             (-32*T*log(T) + 176.37082*T - 6889.972, And(T < 1300.0, T >= 717.0)))),
 (('S', 'MONOCLINIC'), Piecewise((-0.0101215*T**2 - 17.318*T*log(T) + 89.000772*T - 5701.485, And(T < 388.36, T >= 298.15)),
                                 (1.118079e-6*T**3 - 0.008604142*T**2 - 21.1094347*T*log(T) + 114.512564*T - 7435.888 + 120740/T, And(T < 1300.0, T >= 388.36)))),
 (('S', 'ORTHORHOMBIC_S'), Piecewise((Symbol('GHSERSS'), And(T < 1300.0, T >= 298.15)))),
 (('SB', 'BCC_A2'), Piecewise((-15.1*T + Symbol('GHSERSB') + 19874, And(T < 2000.0, T >= 298.15)))),
 (('SB', 'BCT_A5'), Piecewise((-8*T + Symbol('GHSERSB') + 13000, And(T < 2000.0, T >= 298.15)))),
 (('SB', 'FCC_A1'), Piecewise((-13.7*T + Symbol('GHSERSB') + 19874, And(T < 2000.0, T >= 298.15)))),
 (('SB', 'HCP_A3'), Piecewise((-13*T + Symbol('GHSERSB') + 19874, And(T < 2000.0, T >= 298.15)))),
 (('SB', 'LIQUID'), Piecewise((-1.74847e-20*T**7 - 21.923164*T + Symbol('GHSERSB') + 19822.328, And(T < 903.78, T >= 298.15)),
                              (-31.38*T*log(T) + 147.455986*T + 8175.359, And(T < 2000.0, T >= 903.78)))),
 (('SB', 'RHOMBOHEDRAL_A7'), Piecewise((Symbol('GHSERSB'), And(T < 2000.0, T >= 298.15)))),
 (('SC', 'BCC_A2'), Piecewise((-1.024135*T + Symbol('GHSERSC') + 1979.728, And(T < 1000.0, T >= 298.15)),
                              (1.5637371e-5*T**3 - 0.167120107*T**2 + 276.76664*T*log(T) - 2004.05469*T + 230161.408 - 33783257/T, And(T < 1608.0, T >= 1000.0)),
                              (-44.2249*T*log(T) + 283.642312*T - 25928.011, And(T < 2000.0, T >= 1608.0)),
                              (-2.832026*T + Symbol('GHSERSC') + 4587.235, And(T < 3200.0, T >= 2000.0)))),
 (('SC', 'HCP_A3'), Piecewise((Symbol('GHSERSC'), And(T < 3200.0, T >= 298.15)))),
 (('SC', 'LIQUID'), Piecewise((2.13106e-6*T**3 - 0.020636524*T**2 - 10.7967803*T*log(T) + 45.427539*T + 6478.66 - 158106/T, And(T < 1608.0, T >= 298.15)),
                              (-44.2249*T*log(T) + 275.871695*T - 11832.111, And(T < 2000.0, T >= 1608.0)),
                              (-10.602643*T + Symbol('GHSERSC') + 18683.135, And(T < 3200.0, T >= 2000.0)))),
 (('SE', 'HEXAGONAL_A8'), Piecewise((Symbol('GHSERSE'), And(T < 1000.0, T >= 298.15)))),
 (('SE', 'LIQUID'), Piecewise((0.000119219297*T**3 - 0.390268991*T**2 + 194.107439*T*log(T) - 1178.28824*T + 50533.347 - 2224398/T, And(T < 494.0, T >= 298.15)),
                              (-35.1456*T*log(T) + 183.72559*T - 5228.304, And(T < 800.0, T >= 494.0)),
                              (-14.044576*T + Symbol('GHSERSE') + 6965.166, And(T < 1000.0, T >= 800.0)))),
 (('SI', 'BCC_A2'), Piecewise((-22.5*T + Symbol('GHSERSI') + 47000, And(T < 3600.0, T >= 298.15)))),
 (('SI', 'CBCC_A12'), Piecewise((-20.377*T + Symbol('GHSERSI') + 50208, And(T < 3600.0, T >= 298.15)))),
 (('SI', 'CUB_A13'), Piecewise((-20.377*T + Symbol('GHSERSI') + 47279, And(T < 3600.0, T >= 298.15)))),
 (('SI', 'DIAMOND_A4'), Piecewise((Symbol('GHSERSI'), And(T < 3600.0, T >= 298.15)))),
 (('SI', 'FCC_A1'), Piecewise((-21.8*T + Symbol('GHSERSI') + 51000, And(T < 3600.0, T >= 298.15)))),
 (('SI', 'HCP_A3'), Piecewise((-20.8*T + Symbol('GHSERSI') + 49200, And(T < 3600.0, T >= 298.15)))),
 (('SI', 'LIQUID'), Piecewise((2.09307e-21*T**7 - 30.099439*T + Symbol('GHSERSI') + 50696.36, And(T < 1687.0, T >= 298.15)),
                              (-27.196*T*log(T) + 137.722298*T + 40370.523, And(T < 3600.0, T >= 1687.0)))),
 (('SM', 'BCC_A2'), Piecewise((3.579527e-6*T**3 - 0.025446016*T**2 - 16.9298494*T*log(T) + 55.972523*T - 4368.72 + 94209/T, And(T < 1190.0, T >= 298.15)),
                              (-46.9445*T*log(T) + 253.121044*T - 15957.862, And(T < 1345.0, T >= 1190.0)),
                              (3.329865e-6*T**3 - 0.047314968*T**2 + 71.6856914*T*log(T) - 624.680805*T + 111191.653 - 24870276/T, And(T < 2100.0, T >= 1345.0)))),
 (('SM', 'LIQUID'), Piecewise((4.544272e-6*T**3 - 0.032418177*T**2 - 11.6968284*T*log(T) + 20.117456*T + 3468.783 + 23528/T, And(T < 1190.0, T >= 298.15)),
                              (-50.208*T*log(T) + 273.487076*T - 11728.229, And(T < 1345.0, T >= 1190.0)),
                              (-8.707299*T + Symbol('GHSERSM') + 11327.85, And(T < 2100.0, T >= 1345.0)))),
 (('SM', 'RHOMB'), Piecewise((Symbol('GHSERSM'), And(T < 2100.0, T >= 298.15)))),
 (('SN', 'BCC_A2'), Piecewise((-6*T + Symbol('GHSERSN') + 4400, And(T < 3000.0, T >= 100.0)))),
 (('SN', 'BCT_A5'), Piecewise((Symbol('GHSERSN'), And(T < 3000.0, T >= 100.0)))),
 (('SN', 'DIAMOND_A4'), Piecewise((2.7288e-6*T**3 - 0.00813975*T**2 - 22.972*T*log(T) + 114.007785*T - 9579.608 + 25615/T, And(T < 298.15, T >= 100.0)),
                                  (1.784447e-6*T**3 - 0.008575282*T**2 - 21.5750771*T*log(T) + 104.84654*T - 9063.001 - 2544/T, And(T < 800.0, T >= 298.15)),
                                  (-28.4512*T*log(T) + 147.396535*T - 10909.351, And(T < 3000.0, T >= 800.0)))),
 (('SN', 'FCC_A1'), Piecewise((3.121167e-6*T**3 - 0.0188702*T**2 - 15.961*T*log(T) + 56.983315*T - 345.135 - 61960/T, And(T < 250.0, T >= 100.0)),
                              (-8.46*T + Symbol('GHSERSN') + 5510, And(T < 3000.0, T >= 250.0)))),
 (('SN', 'HCP_A3'), Piecewise((-4.4*T + Symbol('GHSERSN') + 3900, And(T < 3000.0, T >= 100.0)))),
 (('SN', 'LIQUID'), Piecewise((1.47031e-18*T**7 - 14.087767*T + Symbol('GHSERSN') + 7103.092, And(T < 505.08, T >= 100.0)),
                              (-13.814383*T + Symbol('GHSERSN') + 6971.586 + 1.2307e+25/T**9, And(T < 800.0, T >= 505.08)),
                              (-28.4512*T*log(T) + 125.182498*T - 1285.372, And(T < 3000.0, T >= 800.0)))),
 (('SN', 'RHOMBOHEDRAL_A7'), Piecewise((Symbol('GHSERSN') + 2035, And(T < 3000.0, T >= 100.0)))),
 (('SR', 'BCC_A2'), Piecewise((2.2965e-7*T**3 - 0.003126762*T**2 - 25.6708365*T*log(T) + 116.583654*T - 6779.234 + 27649/T, And(T < 820.0, T >= 298.15)),
                              (-1.7895e-8*T**3 - 0.0019493*T**2 - 26.57*T*log(T) + 122.067301*T - 6970.594 + 16495/T, And(T < 1050.0, T >= 820.0)),
                              (5.20221e-7*T**3 - 0.009539908*T**2 - 9.7788593*T*log(T) + 0.423037*T + 8168.357 - 2414794/T, And(T < 3000.0, T >= 1050.0)))),
 (('SR', 'FCC_A1'), Piecewise((Symbol('GHSERSR'), And(T < 3000.0, T >= 298.15)))),
 (('SR', 'HCP_A3'), Piecewise((0.7*T + Symbol('GHSERSR') + 250, And(T < 3000.0, T >= 298.15)))),
 (('SR', 'LIQUID'), Piecewise((4.981237e-6*T**3 - 0.031840595*T**2 - 5.0668978*T*log(T) - 10.118994*T + 2194.997 - 265559/T, And(T < 1050.0, T >= 298.15)),
                              (-39.463*T*log(T) + 213.406219*T - 10855.29, And(T < 3000.0, T >= 1050.0)))),
 (('TA', 'BCC_A2'), Piecewise((Symbol('GHSERTA'), And(T < 6000.0, T >= 298.15)))),
 (('TA', 'FCC_A1'), Piecewise((1.7*T + Symbol('GHSERTA') + 16000, And(T < 6000.0, T >= 298.15)))),
 (('TA', 'HCP_A3'), Piecewise((2.4*T + Symbol('GHSERTA') + 12000, And(T < 6000.0, T >= 298.15)))),
 (('TA', 'LIQUID'), Piecewise((-7.578729*T + Symbol('GHSERTA') + 29160.975, And(T < 1000.0, T >= 298.15)),
                              (6.14599e-7*T**3 - 0.012330066*T**2 + 0.0279523*T*log(T) - 61.981795*T + 43884.339 - 3523338/T, And(T < 3290.0, T >= 1000.0)),
                              (-41.84*T*log(T) + 258.110873*T - 6314.543, And(T < 6000.0, T >= 3290.0)))),
 (('TB', 'BCC_A2'), Piecewise((-2.652707*T + Symbol('GHSERTB') + 4167.835, And(T < 1200.0, T >= 298.15)),
                              (3.4100235e-5*T**3 - 0.373763517*T**2 + 706.580596*T*log(T) - 5157.77779*T + 633060.245 - 103233571.0/T, And(T < 1562.0, T >= 1200.0)),
                              (-46.4842*T*log(T) + 257.388486*T - 23398.029, And(T < 3000.0, T >= 1562.0)))),
 (('TB', 'HCP_A3'), Piecewise((Symbol('GHSERTB'), And(T < 3000.0, T >= 298.15)))),
 (('TB', 'LIQUID'), Piecewise((2.17475e-6*T**3 - 0.020466105*T**2 - 14.252646*T*log(T) + 29.867521*T + 3945.831 - 160724/T, And(T < 1562.0, T >= 298.15)),
                              (-46.4842*T*log(T) + 251.16889*T - 13247.649, And(T < 3000.0, T >= 1562.0)))),
 (('TC', 'BCC_A2'), Piecewise((-4.5*T + Symbol('GHSERTC') + 18000, And(T < 4000.0, T >= 298.15)))),
 (('TC', 'FCC_A1'), Piecewise((-1.5*T + Symbol('GHSERTC') + 10000, And(T < 4000.0, T >= 298.15)))),
 (('TC', 'HCP_A3'), Piecewise((Symbol('GHSERTC'), And(T < 4000.0, T >= 298.15)))),
 (('TC', 'LIQUID'), Piecewise((-9.62385e-22*T**7 - 12.313*T + Symbol('GHSERTC') + 30402.134, And(T < 2430.0, T >= 298.15)),
                              (-47*T*log(T) + 303.7538*T - 12221.9, And(T < 4000.0, T >= 2430.0)))),
 (('TE', 'HEXAGONAL_A8'), Piecewise((Symbol('GHSERTE'), And(T < 1600.0, T >= 298.15)))),
 (('TE', 'LIQUID'), Piecewise((-9.42075e-5*T**3 + 0.2219435*T**2 - 126.318*T*log(T) + 685.877639*T - 17554.731 + 827930/T, And(T < 626.49, T >= 298.15)),
                              (-0.00130692833*T**3 + 7.09775*T**2 - 7196.41*T*log(T) + 46756.357*T - 3165763.48 + 258051000.0/T, And(T < 722.66, T >= 626.49)),
                              (1.6129733e-5*T**3 - 0.142016*T**2 + 202.743*T*log(T) - 1500.57909*T + 180326.959 - 24238450/T, And(T < 1150.0, T >= 722.66)),
                              (-26.1929273*T + Symbol('GHSERTE') + 19110.036, And(T < 1600.0, T >= 1150.0)))),
 (('TH', 'BCC_A2'), Piecewise((-5.3048e-7*T**3 + 0.00043775*T**2 - 28.244*T*log(T) + 134.279995*T - 2321.06 + 91190/T, And(T < 1633.0, T >= 298.15)),
                              (-2.536883e-6*T**3 + 0.03098*T**2 - 116.453*T*log(T) + 801.657849*T - 115978.348 + 27512600/T, And(T < 2023.0, T >= 1633.0)),
                              (1.66067e-7*T**3 - 0.00346655*T**2 - 35.813*T*log(T) + 209.523509*T - 33602.796 + 11876950/T, And(T < 3600.0, T >= 2023.0)),
                              (-0.049551*T + Symbol('GHSERTH') - 980.302, And(T < 4000.0, T >= 3600.0)))),
 (('TH', 'FCC_A1'), Piecewise((Symbol('GHSERTH'), And(T < 4000.0, T >= 298.15)))),
 (('TH', 'LIQUID'), Piecewise((-9.09067e-7*T**3 - 0.00168345*T**2 - 24.987*T*log(T) + 111.635146*T + 5031.109 + 10865/T, And(T < 1499.8, T >= 298.15)),
                              (1.210117e-6*T**3 - 0.0136421*T**2 - 24.03*T*log(T) + 128.406516*T - 15602.847 + 7111100/T, And(T < 2014.5, T >= 1499.8)),
                              (-46.024*T*log(T) + 275.750074*T - 17273.382, And(T < 2900.0, T >= 2014.5)),
                              (-8.229771*T + Symbol('GHSERTH') + 16079.931, And(T < 4000.0, T >= 2900.0)))),
 (('TI', 'BCC_A2'), Piecewise((-2.78803e-7*T**3 - 0.000663845*T**2 - 25.5768*T*log(T) + 134.71418*T - 1272.064 + 7208/T, And(T < 1155.0, T >= 298.15)),
                              (-8.4534e-7*T**3 + 0.00121707*T**2 - 22.3771*T*log(T) + 105.366379*T + 6667.385 - 2002750/T, And(T < 1941.0, T >= 1155.0)),
                              (1.228863e-6*T**3 - 0.02200832*T**2 + 19.0900905*T*log(T) - 182.426471*T + 26483.26 + 1400501/T, And(T < 4000.0, T >= 1941.0)))),
 (('TI', 'BCT_A5'), Piecewise((Symbol('GHSERTI') + 4602.2, And(T < 3000.0, T >= 298.15)))),
 (('TI', 'CBCC_A12'), Piecewise((Symbol('GHSERTI') + 4602.2, And(T < 4000.0, T >= 298.15)))),
 (('TI', 'CUB_A13'), Piecewise((Symbol('GHSERTI') + 7531.2, And(T < 4000.0, T >= 298.15)))),
 (('TI', 'DIAMOND_A4'), Piecewise((Symbol('GHSERTI') + 25000, And(T < 4000.0, T >= 298.15)))),
 (('TI', 'FCC_A1'), Piecewise((-0.1*T + Symbol('GHSERTI') + 6000, And(T < 4000.0, T >= 298.15)))),
 (('TI', 'HCP_A3'), Piecewise((Symbol('GHSERTI'), And(T < 4000.0, T >= 298.15)))),
 (('TI', 'LAVES_C14'), Piecewise((3.20148e-7*T**3 - 0.014333925*T**2 - 71.9799*T*log(T) + 400.845624*T - 9179.763 + 217908/T, And(T < 900.0, T >= 298.15)),
                                 (-2.72628e-7*T**3 - 0.0126099*T**2 - 71.9661*T*log(T) + 398.964204*T - 8435.445 + 128040/T, And(T < 1155.0, T >= 900.0)),
                                 (6.08145e-7*T**3 - 0.0244395*T**2 - 44.8398*T*log(T) + 200.929614*T + 17726.511 - 4432980/T, And(T < 1941.0, T >= 1155.0)),
                                 (-9.14241e-7*T**3 + 0.024614547*T**2 - 261.654738*T*log(T) + 1916.42061*T - 358580.358 + 110099415.0/T, And(T < 4000.0, T >= 1941.0)))),
 (('TI', 'LAVES_C15'), Piecewise((3.20148e-7*T**3 - 0.014333925*T**2 - 71.9799*T*log(T) + 400.845624*T - 9179.763 + 217908/T, And(T < 900.0, T >= 298.15)),
                                 (-2.72628e-7*T**3 - 0.0126099*T**2 - 71.9661*T*log(T) + 398.964204*T - 8435.445 + 128040/T, And(T < 1155.0, T >= 900.0)),
                                 (6.08145e-7*T**3 - 0.0244395*T**2 - 44.8398*T*log(T) + 200.929614*T + 17726.511 - 4432980/T, And(T < 1941.0, T >= 1155.0)),
                                 (-9.14241e-7*T**3 + 0.024614547*T**2 - 261.654738*T*log(T) + 1916.42061*T - 358580.358 + 110099415.0/T, And(T < 4000.0, T >= 1941.0)))),
 (('TI', 'LIQUID'), Piecewise((-6.980938*T + Symbol('GHSERTI') + 12194.415, And(T < 900.0, T >= 298.15)),
                              (-6.980938*T + Symbol('GHSERTI') + 12194.416, And(T < 1300.0, T >= 900.0)),
                              (1.2457117e-5*T**3 - 0.163409355*T**2 + 342.059267*T*log(T) - 2554.0225*T + 369519.198 - 67034516/T, And(T < 1941.0, T >= 1300.0)),
                              (-46.29*T*log(T) + 298.7367*T - 19887.066, And(T < 4000.0, T >= 1941.0)))),
 (('TL', 'BCC_A2'), Piecewise((-1.0115933e-5*T**3 + 0.0172318*T**2 - 33.0508*T*log(T) + 150.019517*T - 9194.493 + 82153/T, And(T < 577.0, T >= 200.0)),
                              (-2.359101e-6*T**3 + 0.026042993*T**2 - 79.2926704*T*log(T) + 482.633817*T - 41836.403 + 3507810/T, And(T < 1800.0, T >= 577.0)))),
 (('TL', 'FCC_A1'), Piecewise((Symbol('GHSERTL') + 310, And(T < 1800.0, T >= 200.0)))),
 (('TL', 'HCP_A3'), Piecewise((Symbol('GHSERTL'), And(T < 1800.0, T >= 200.0)))),
 (('TL', 'LIQUID'), Piecewise((1.4248046e-5*T**3 - 0.044350292*T**2 - 7.44455*T*log(T) + 0.755649*T - 946.623 - 54228/T, And(T < 577.0, T >= 200.0)),
                              (9.0e-12*T**3 - 0.00083662*T**2 - 25.8437*T*log(T) + 98.472609*T - 614.74 - 612570/T, And(T < 1800.0, T >= 577.0)))),
 (('TM', 'HCP_A3'), Piecewise((Symbol('GHSERTM'), And(T < 2300.0, T >= 298.15)))),
 (('TM', 'LIQUID'), Piecewise((-6.557671*T + Symbol('GHSERTM') + 13199.249, And(T < 600.0, T >= 298.15)),
                              (2.288172e-6*T**3 - 0.025487085*T**2 + 6.8744933*T*log(T) - 126.738485*T + 22640.028 - 1585412/T, And(T < 1818.0, T >= 600.0)),
                              (-41.37976*T*log(T) + 214.184413*T - 10090.305, And(T < 2300.0, T >= 1818.0)))),
 (('U', 'BCC_A2'), Piecewise((9.67907e-7*T**3 - 0.00835595*T**2 - 27.5152*T*log(T) + 131.5381*T - 752.767 + 204611/T, And(T < 1049.0, T >= 298.15)),
                             (-38.2836*T*log(T) + 202.685635*T - 4698.365, And(T < 3000.0, T >= 1049.0)))),
 (('U', 'LIQUID'), Piecewise((-10.3239*T + Symbol('GHSERUU') + 12355.5, And(T < 3000.0, T >= 298.15)))),
 (('U', 'ORTHORHOMBIC_A20'), Piecewise((Symbol('GHSERUU'), And(T < 3000.0, T >= 298.15)))),
 (('U', 'TETRAGONAL_U'), Piecewise((2.7889e-8*T**3 - 0.01084475*T**2 - 22.841*T*log(T) + 106.976316*T - 5156.136 + 81944/T, And(T < 941.5, T >= 298.15)),
                                   (-42.9278*T*log(T) + 244.16802*T - 14327.309, And(T < 3000.0, T >= 941.5)))),
 (('V', 'BCC_A2'), Piecewise((Symbol('GHSERVV'), And(T < 4000.0, T >= 298.15)))),
 (('V', 'CBCC_A12'), Piecewise((Symbol('GHSERVV') + 9000, And(T < 4000.0, T >= 298.15)))),
 (('V', 'CUB_A13'), Piecewise((Symbol('GHSERVV') + 10000, And(T < 4000.0, T >= 298.15)))),
 (('V', 'FCC_A1'), Piecewise((1.7*T + Symbol('GHSERVV') + 7500, And(T < 4000.0, T >= 298.15)))),
 (('V', 'HCP_A3'), Piecewise((2.4*T + Symbol('GHSERVV') + 4000, And(T < 4000.0, T >= 298.15)))),
 (('V', 'LIQUID'), Piecewise((-5.19136e-22*T**7 - 9.455552*T + Symbol('GHSERVV') + 20764.117, And(T < 2183.0, T >= 298.15)),
                             (-47.43*T*log(T) + 311.055983*T - 19617.51, And(T < 4000.0, T >= 2183.0)))),
 (('W', 'BCC_A2'), Piecewise((Symbol('GHSERWW'), And(T < 6000.0, T >= 298.15)))),
 (('W', 'FCC_A1'), Piecewise((0.63*T + Symbol('GHSERWW') + 19300, And(T < 6000.0, T >= 298.15)))),
 (('W', 'HCP_A3'), Piecewise((Symbol('GHSERWW') + 14750, And(T < 6000.0, T >= 298.15)))),
 (('W', 'LIQUID'), Piecewise((-2.713468e-24*T**7 - 14.10999*T + Symbol('GHSERWW') + 52160.584, And(T < 3695.0, T >= 298.15)),
                             (-54*T*log(T) + 375.175*T - 30436.051, And(T < 6000.0, T >= 3695.0)))),
 (('Y', 'BCC_A2'), Piecewise((9.10372497e-9*T**3 - 0.00237175965*T**2 - 25.5832578*T*log(T) + 123.667346*T - 833.658863 + 27340.0687/T, And(T < 1000.0, T >= 100.0)),
                             (-3.05012175e-7*T**3 - 0.000541757644*T**2 - 27.3038477*T*log(T) + 134.528352*T - 1297.79829, And(T < 1795.15, T >= 1000.0)),
                             (4.02944768e-7*T**3 - 0.00904576576*T**2 - 8.88296647*T*log(T) + 0.981325399*T + 15389.4975 - 2542575.96/T, And(T < 3700.0, T >= 1795.15)))),
 (('Y', 'HCP_A3'), Piecewise((Symbol('GHSERYY'), And(T < 3700.0, T >= 100.0)))),
 (('Y', 'LIQUID'), Piecewise((-8.12981167e-7*T**3 - 0.00347023463*T**2 - 24.6467508*T*log(T) + 119.41873*T + 2098.50738 + 23713.7332/T, And(T < 1000.0, T >= 100.0)),
                             (1.7595327e-6*T**3 - 0.0189533369*T**2 - 9.0681627*T*log(T) + 19.4520171*T + 7386.44846, And(T < 1795.15, T >= 1000.0)),
                             (-43.0952*T*log(T) + 257.400783*T - 12976.5957, And(T < 3700.0, T >= 1795.15)))),
 (('YB', 'BCC_A2'), Piecewise((4.743871e-6*T**3 - 0.030009694*T**2 - 3.8534432*T*log(T) - 21.293677*T - 965.99 - 334650/T, And(T < 1033.0, T >= 298.15)),
                              (-36.1079*T*log(T) + 188.313864*T - 13368.113, And(T < 1097.0, T >= 1033.0)),
                              (3.63044e-7*T**3 - 0.004743348*T**2 - 25.7402233*T*log(T) + 113.174165*T - 3911.847 - 1553668/T, And(T < 2000.0, T >= 1097.0)))),
 (('YB', 'FCC_A1'), Piecewise((Symbol('GHSERYB'), And(T < 2000.0, T >= 298.15)))),
 (('YB', 'LIQUID'), Piecewise((5.136665e-6*T**3 - 0.03250938*T**2 - 1.8061816*T*log(T) - 40.615571*T + 7030.788 - 370554/T, And(T < 1033.0, T >= 298.15)),
                              (-36.7774*T*log(T) + 186.690398*T - 6445.835, And(T < 2000.0, T >= 1033.0)))),
 (('ZN', 'BCC_A2'), Piecewise((-2.5104*T + Symbol('GHSERZN') + 2886.96, And(T < 1700.0, T >= 298.15)))),
 (('ZN', 'BCT_A5'), Piecewise((-2.5104*T + Symbol('GHSERZN') + 2886.96, And(T < 1700.0, T >= 298.15)))),
 (('ZN', 'DIAMOND_A4'), Piecewise((30*T + Symbol('GHSERZN'), And(T < 1700.0, T >= 298.15)))),
 (('ZN', 'FCC_A1'), Piecewise((-1.56968*T + Symbol('GHSERZN') + 2969.82, And(T < 1700.0, T >= 298.15)))),
 (('ZN', 'HCP_A3'), Piecewise((Symbol('GHSERZN'), And(T < 1700.0, T >= 298.15)))),
 (('ZN', 'LIQUID'), Piecewise((-3.58958e-19*T**7 - 10.29299*T + Symbol('GHSERZN') + 7157.213, And(T < 692.68, T >= 298.15)),
                              (-31.38*T*log(T) + 161.608594*T - 3620.391, And(T < 1700.0, T >= 692.68)))),
 (('ZR', 'BCC_A2'), Piecewise((-7.6143e-11*T**4 - 9.729e-9*T**3 - 0.000340084*T**2 - 25.607406*T*log(T) + 124.9457*T - 525.539 + 25233/T, And(T < 2128.0, T >= 130.0)),
                              (1.55998*T + Symbol('GHSERZR') - 4620.034 + 1.4103476e+32/T**9, And(T < 6000.0, T >= 2128.0)))),
 (('ZR', 'FCC_A1'), Piecewise((-0.9*T + Symbol('GHSERZR') + 7600, And(T < 6000.0, T >= 130.0)))),
 (('ZR', 'HCP_A3'), Piecewise((Symbol('GHSERZR'), And(T < 6000.0, T >= 130.0)))),
 (('ZR', 'LIQUID'), Piecewise((1.6275e-22*T**7 - 9.080812*T + Symbol('GHSERZR') + 18147.69, And(T < 2128.0, T >= 130.0)),
                              (-42.144*T*log(T) + 253.812609*T - 8281.26, And(T < 6000.0, T >= 2128.0)))),
 (('ZR', 'OMEGA'), Piecewise((-0.002799446*T**2 - 26.8556*T*log(T) + 144.432234*T - 8878.082 + 38376/T, And(T < 2128.0, T >= 130.0)),
                             (2.566675*T + Symbol('GHSERZR') - 3414.603 + 8.517346e+31/T**9, And(T < 6000.0, T >= 2128.0))))
])