/***************************************************************************
 *   Copyright (C) 2005 by Christophe GONZALES and Pierre-Henri WUILLEMIN  *
 *   {prenom.nom}_at_lip6.fr                                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
/** @file
 * @brief A class used by RecordCounter to detect subsets of variables
 *
 * @author Christophe GONZALES and Pierre-Henri WUILLEMIN
 */
#ifndef GUM_LEARNING_ID_SET_H
#define GUM_LEARNING_ID_SET_H

#include <cstring>
#include <initializer_list>
#include <iostream>
#include <sstream>
#include <string>
#include <type_traits>
#include <utility>
#include <vector>

#include <agrum/agrum.h>
#include <agrum/core/hashFunc.h>

namespace gum {

  namespace learning {

    /** @class IdSet
     * @ingroup learning_group
     * @brief transforms an unordered set of ids into an (increasingly) ordered
     *one
     *
     * IdSets are used by RecordCounters to detect which sets of variables are
     * included in other sets. With this knowledge, RecordCounters only parse
     *the
     * database for counting the number of observations of non-included sets of
     * variables and they subsequently deduce from these counts those for the
     * included sets.
     */
    template < typename Alloc = std::allocator< Idx > >
    class IdSet {
      public:
      // ##########################################################################
      /// @name Constructors / Destructors
      // ##########################################################################
      /// @{

      /// default constructor
      IdSet();

      /// initializer list constructor
      IdSet(const std::initializer_list< Idx > list);

      /// default constructor
      template < typename OtherAlloc >
      IdSet(const std::vector< Idx, OtherAlloc >& ids, Size sz);

      /// copy constructor
      IdSet(const IdSet< Alloc >& from);

      /// generalized copy constructor
      template < typename OtherAlloc >
      IdSet(const IdSet< OtherAlloc >& from);

      /// move constructor
      IdSet(IdSet< Alloc >&& from);

      /// destructor
      ~IdSet();

      /// @}

      // ##########################################################################
      /// @name Operators
      // ##########################################################################
      /// @{

      /// copy operator
      IdSet< Alloc >& operator=(const IdSet< Alloc >& from);

      /// generalized copy operator
      template < typename OtherAlloc >
      IdSet< Alloc >& operator=(const IdSet< OtherAlloc >& from);

      /// move operator
      IdSet< Alloc >& operator=(IdSet< Alloc >&& from);

      /// returns the id stored at a given index
      Idx operator[](Idx index) const noexcept;

      /// inserts a new element into the set (assuming it is a Boolean)
      IdSet< Alloc >& operator<<(Idx id);

      /// returns true if both sets are equal
      template < typename OtherAlloc >
      bool operator==(const IdSet< OtherAlloc >& from) const noexcept;

      /// returns true if the sets differ
      template < typename OtherAlloc >
      bool operator!=(const IdSet< OtherAlloc >& from) const noexcept;

      /// @}

      // ##########################################################################
      /// @name Accessors / Modifiers
      // ##########################################################################
      /// @{

      /// returns the set of ids contained in the object
      const std::vector< Idx, Alloc >& ids() const noexcept;

      /// returns the domain size of the id set
      Size size() const noexcept;

      /// sets the domain size of the set
      void setSize(Idx) noexcept;

      /// returns the content of the set as a string
      std::string toString() const noexcept;

      /// indicates wether the current object is a subset of 'otherset'
      template < typename OtherAlloc >
      bool isSubset(const IdSet< OtherAlloc >& otherset) const noexcept;

      /// @}

      private:
      /// the ordered set of ids
      std::vector< Idx, Alloc > __ids;

      /// the domain size of the set
      Size __size{0};
    };

    /// the display operator
    template < typename Alloc >
    std::ostream& operator<<(std::ostream& stream, const IdSet< Alloc >& idset);

  } /* namespace learning */

  /// the hash function for idSets
  template < typename Alloc >
  class HashFunc< learning::IdSet< Alloc > >
      : public HashFuncBase< learning::IdSet< Alloc > > {
    public:
    /// computes the hashed value of a key
    Size operator()(const learning::IdSet< Alloc >& key) const;
  };

  /// the hash function for pairs (idSet,Idx)
  template < typename Alloc >
  class HashFunc< std::pair< learning::IdSet< Alloc >, Idx > >
      : public HashFuncBase< std::pair< learning::IdSet< Alloc >, Idx > > {
    public:
    /// computes the hashed value of a key
    Size operator()(const std::pair< learning::IdSet< Alloc >, Idx >& key) const;
  };

  /// the hash function for pairs (idSet,pair<Idx,Idx>)
  template < typename Alloc >
  class HashFunc< std::tuple< learning::IdSet< Alloc >, Idx, Idx > >
      : public HashFuncBase< std::tuple< learning::IdSet< Alloc >, Idx, Idx > > {
    public:
    /// computes the hashed value of a key
    Size operator()(
      const std::tuple< learning::IdSet< Alloc >, Idx, Idx >& key) const;
  };

  /// the hash function for pairs (idSet,tuple<Idx,Idx,Idx>)
  template < typename Alloc >
  class HashFunc< std::tuple< learning::IdSet< Alloc >, Idx, Idx, Idx > >
      : public HashFuncBase<
          std::tuple< learning::IdSet< Alloc >, Idx, Idx, Idx > > {
    public:
    /// computes the hashed value of a key
    Size operator()(
      const std::tuple< learning::IdSet< Alloc >, Idx, Idx, Idx >& key) const;
  };

} /* namespace gum */


extern template class gum::HashFunc<
  std::tuple< gum::learning::IdSet< std::allocator< gum::Idx > > > >;
extern template class gum::HashFunc<
  std::tuple< gum::learning::IdSet< std::allocator< gum::Idx > >, gum::Idx > >;
extern template class gum::HashFunc<
  std::tuple< gum::learning::IdSet< std::allocator< gum::Idx > >,
              gum::Idx,
              gum::Idx > >;
extern template class gum::HashFunc<
  std::tuple< gum::learning::IdSet< std::allocator< gum::Idx > >,
              gum::Idx,
              gum::Idx,
              gum::Idx > >;


// always include the template implementation
#include <agrum/learning/scores_and_tests/idSet_tpl.h>

#endif /* GUM_LEARNING_ID_SET_H */
