/***************************************************************************
 *   Copyright (C) 2005 by Christophe GONZALES and Pierre-Henri WUILLEMIN  *
 *   {prenom.nom}_at_lip6.fr                                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
/**
 * @file
 * @brief Headers of the Observation class.
 *
 * @author Jean-Christophe MAGNAN and Pierre-Henri WUILLEMIN
 */

// #define  TRACE_ALGO
// =========================================================================
#ifndef GUM_OBSERVATION_H
#define GUM_OBSERVATION_H
// =========================================================================
#include <agrum/core/hashTable.h>
#include <agrum/core/smallobjectallocator/smallObjectAllocator.h>
// =========================================================================
#include <agrum/variables/discreteVariable.h>
// =========================================================================

namespace gum {

  /**
   * @class Observation
   * @headerfile observation.h <agrum/FMDP/learning/observation.h>
   * @brief
   * @ingroup fmdp_group
   *
   *
   *
   */

  class Observation {
    public:
    // ==========================================================================
    /// @name Constructor & destructor.
    // ==========================================================================
    /// @{

    // ###################################################################
    /// Default constructor
    // ###################################################################
    Observation() { GUM_CONSTRUCTOR(Observation) }

    // ###################################################################
    /// Default destructor
    // ###################################################################
    ~Observation() { GUM_DESTRUCTOR(Observation) }

    // ============================================================================
    /// Allocators and Deallocators redefinition
    // ============================================================================
    void* operator new(size_t s) {
      return SmallObjectAllocator::instance().allocate(s);
    }
    void operator delete(void* p) {
      SmallObjectAllocator::instance().deallocate(p, sizeof(Observation));
    }

    /// @}

    // ==========================================================================
    /// @name Observation Handlers
    // ==========================================================================
    /// @{

    // ###################################################################
    /**
     * Returns the modality assumed by the given variable in this observation
     *
     * @throws NotFound if variable is not in this observation
     */
    // ###################################################################
    INLINE Idx modality(const DiscreteVariable* var) const {
      return __varInst[var];
    }
    INLINE Idx rModality(const DiscreteVariable* var) const {
      return __rInst[var];
    }

    // ###################################################################
    /**
     * Sets the modality assumed by the given variable in this observation
     *
     * @throws DuplicateElement if a value has already be assigned to
     * this variable
     */
    // ###################################################################
    INLINE void setModality(const DiscreteVariable* var, Idx modality) {
      __varInst.insert(var, modality);
    }
    INLINE void setRModality(const DiscreteVariable* var, Idx modality) {
      __rInst.insert(var, modality);
    }

    // ###################################################################
    // Returns the reward obtained during this observation
    // ###################################################################
    double reward() const { return __reward; }

    // ###################################################################
    // Sets the reward obtained during this observation
    // ###################################################################
    void setReward(double reward) { __reward = reward; }

    /// @}
    ///
    std::string toString() const;

    // ==========================================================================
    /// @name Iterators on Variables
    // ==========================================================================
    /// @{

    // ###################################################################
    /// Returns an const safe iterator on the beginning of the list of
    /// variables in this observation
    // ###################################################################
    HashTableConstIteratorSafe< const DiscreteVariable*, Idx >
      cbeginVariablesSafe() const {
      return __varInst.cbeginSafe();
    }

    // ###################################################################
    /// Returns an const safe iterator on the end of the list of
    /// variables in this observation
    // ###################################################################
    HashTableConstIteratorSafe< const DiscreteVariable*, Idx >
      cendVariablesSafe() const {
      return __varInst.cendSafe();
    }

    /// @}

    private:
    /// Table giving for every variables its instantiation
    HashTable< const DiscreteVariable*, Idx > __varInst;
    HashTable< const DiscreteVariable*, Idx > __rInst;

    /// The reward associated to this transition
    double __reward;
  };

} /* namespace gum */


#endif   // GUM_OBSERVATION_H
