/***************************************************************************
*   Copyright (C) 2005 by Christophe GONZALES and Pierre-Henri WUILLEMIN  *
*   {prenom.nom}_at_lip6.fr                                               *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
***************************************************************************/
/**
* @file
* @brief algorithm for approximated computation KL divergence between BNs using
*GIBBS
*sampling
*
* @author Pierre-Henri WUILLEMIN
*
*/
#ifndef GUM_GIBBS_KL_H
#define GUM_GIBBS_KL_H

#include <agrum/BN/algorithms/divergence/KL.h>
#include <agrum/BN/samplers/GibbsSampler.h>
#include <agrum/core/approximations/approximationScheme.h>

#include <agrum/core/signal/signaler.h>

namespace gum {

  /**
  * GibbsKL computes the KL divergence betweens 2 BNs using an approximation
  *pattern
  *: GIBBS sampling.
  *
  * KL.process() computes KL(P||Q) using klPQ() and KL(Q||P) using klQP(). The
  *computations are made once. The second is for free :)
  * GibbsKL allows as well to compute in the same time the Hellinger distance
  *(\f$
  *\sqrt{\sum_i (\sqrt{p_i}-\sqrt{q_i})^2}\f$) (Kokolakis and Nanopoulos, 2001)
  * and Bhattacharya distance (Kaylath,T. 1967)
  *
  * It may happen that P*ln(P/Q) is not computable (Q=0 and P!=0). In such a
  *case, KL
  *keeps working but trace this error (errorPQ() and errorQP()). In those cases,
  *Hellinger distance approximation is under-evaluated.
  *
  * @warning : convergence and stop criteria are designed w.r.t the main
  *computation
  *: KL(P||Q). The 3 others have no guarantee.
  *
  * snippets :
  * @code
  * gum::KL base_kl(net1,net2);
  * if (base_kl.difficulty()!=KL::HEAVY) {
  *  gum::BruteForceKL kl(base_kl);
  *  std::cout<<"KL net1||net2 :"<<kl.klPQ()<<std::endl;
  * } else {
  *  gum::GibbsKL kl(base_kl);
  *  std::cout<<"KL net1||net2 :"<<kl.klPQ()<<std::endl;
  * }
  * @endcode
  */
  template <typename GUM_SCALAR>
  class GibbsKL : public KL<GUM_SCALAR>,
                  public ApproximationScheme,
                  public samplers::GibbsSampler<GUM_SCALAR> {
    public:
    /* no default constructor */

    /** constructor must give 2 BNs
     * @throw gum::OperationNotAllowed if the 2 BNs have not the same domainSize
     * or
     * compatible node sets.
     */
    GibbsKL( const IBayesNet<GUM_SCALAR>& P, const IBayesNet<GUM_SCALAR>& Q );

    /** copy constructor
     */
    GibbsKL( const KL<GUM_SCALAR>& kl );

    /** destructor */
    ~GibbsKL();

    using samplers::GibbsSampler<GUM_SCALAR>::particle;
    using samplers::GibbsSampler<GUM_SCALAR>::initParticle;
    using samplers::GibbsSampler<GUM_SCALAR>::nextParticle;
    using samplers::GibbsSampler<GUM_SCALAR>::bn;

    protected:
    void _computeKL( void );

    using KL<GUM_SCALAR>::_p;
    using KL<GUM_SCALAR>::_q;
    using KL<GUM_SCALAR>::_hellinger;
    using KL<GUM_SCALAR>::_bhattacharya;

    using KL<GUM_SCALAR>::_klPQ;
    using KL<GUM_SCALAR>::_klQP;

    using KL<GUM_SCALAR>::_errorPQ;
    using KL<GUM_SCALAR>::_errorQP;
  };


  extern template class GibbsKL<float>;
  extern template class GibbsKL<double>;


}  // namespace gum

#include <agrum/BN/algorithms/divergence/GibbsKL.tcc>

#endif  // GUM_GIBBS_KL_H
