# This file is part of Py6S.
#
# Copyright 2012 Robin Wilson and contributors listed in the CONTRIBUTORS file.
#
# Py6S is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Py6S is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with Py6S.  If not, see <http://www.gnu.org/licenses/>.

from .. import sixs_exceptions
import collections
import numpy as np


def Wavelength(start_wavelength, end_wavelength=None, filter=None):
    """Select one or more wavelengths for the 6S simulation.

    There are a number of ways to do this:

    1. Pass a single value of a wavelength in micrometres.
       The simulation will be performed for just this wavelength::

        Wavelength(0.43)

    2. Pass a start and end wavelength in micrometres.
       The simulation will be performed across this wavelength range with a
       constant filter function (spectral response function) of 1.0::

        Wavelength(0.43, 0.50)

    3. Pass a start and end wavelength, and a filter given at 2.5nm intervals.
       The simulation will be performed across this wavelength range using the
       given filter function::

        Wavelength(0.400, 0.410, [0.7, 0.9, 1.0, 0.3, 0.15])

       The filter function must include values for the start and end wavelengths,
       plus values every 2.5nm (0.0025um) in between. So, in the example above,
       there are five values given: one each for 0.400, 0.4025, 0.405, 0.4075, 0.410.

    4. Pass a constant (as defined in this class) for a pre-defined wavelength
       range::

        Wavelength(PredefinedWavelengths.LANDSAT_TM_B1)

    """
    try:
        wv_id = start_wavelength[0]
        if wv_id > 0:
            # It's one of the new predefined wavelengths that I've added
            return Wavelength(start_wavelength[1], start_wavelength[2], start_wavelength[3])
        else:
            wv_type = "%d (Chosen Band)\n" % (-1 * wv_id)
            data = None
            min_wv = start_wavelength[1]
            max_wv = start_wavelength[2]
    except:
        if end_wavelength is None:
            # It's simply a wavelength value
            if start_wavelength > PredefinedWavelengths.MAX_ALLOWABLE_WAVELENGTH or start_wavelength < PredefinedWavelengths.MIN_ALLOWABLE_WAVELENGTH:
                raise sixs_exceptions.ParameterError("wavelength", "Wavelength must be between %f and %f" % (PredefinedWavelengths.MIN_ALLOWABLE_WAVELENGTH, PredefinedWavelengths.MAX_ALLOWABLE_WAVELENGTH))
            wv_type = "-1"
            data = "%f" % start_wavelength
            min_wv = start_wavelength
            max_wv = start_wavelength
        else:
            if start_wavelength > PredefinedWavelengths.MAX_ALLOWABLE_WAVELENGTH or start_wavelength < PredefinedWavelengths.MIN_ALLOWABLE_WAVELENGTH or end_wavelength > PredefinedWavelengths.MAX_ALLOWABLE_WAVELENGTH or end_wavelength < PredefinedWavelengths.MIN_ALLOWABLE_WAVELENGTH:
                raise sixs_exceptions.ParameterError("wavelength", "Wavelength must be between %f and %f" % (PredefinedWavelengths.MIN_ALLOWABLE_WAVELENGTH, PredefinedWavelengths.MAX_ALLOWABLE_WAVELENGTH))
            min_wv = start_wavelength
            max_wv = end_wavelength
            if filter is None:
                # They haven't given a filter, so assume filter is constant at 1
                wv_type = "0 constant filter function"
                data = "%f %f" % (start_wavelength, end_wavelength)
            else:
                # Filter has been given, so we must use it.
                # We assume filter has been given at 2.5nm intervals
                wv_type = "1 User's defined filtered function"
                data = """%f %f
    %s""" % (start_wavelength, end_wavelength, " ".join(map(str, filter)))

    if data is None:
        return_string = wv_type
    else:
        return_string =  """%s
%s\n""" % (wv_type, data)

    return (return_string, min_wv, max_wv)


class PredefinedWavelengths:
    MAX_ALLOWABLE_WAVELENGTH = 4
    MIN_ALLOWABLE_WAVELENGTH = 0.2

    # New predefined wavelengths that I've added to Py6S
    # CONSTANT_NAME = (ID, Start Wavelength, End Wavelength, Filter Function)
    # Note: IDs must be > 1 for new predefined wavelengths

    # Landsat OLI
    # Taken from spreadsheet downloadable from http://landsat.gsfc.nasa.gov/?p=5779
    # Interpolated to 2.5nm intervals, as required by 6S
    LANDSAT_OLI_B1 = (1, 0.427, 0.457,
                      np.array([7.30000000e-05, 2.52450000e-03, 2.47670000e-02,
                                3.85985000e-01, 9.08749000e-01, 9.80591500e-01,
                                9.86713000e-01, 9.96568500e-01, 9.82780000e-01,
                                8.25707000e-01, 2.26412000e-01, 2.55700000e-02,
                                2.41400000e-03]))

    LANDSAT_OLI_B2 = (2, 0.436, 0.526,
                      np.array([1.00000000e-05, 1.79000000e-04, 4.55000000e-04,
                                1.63350000e-03, 6.86900000e-03, 4.28880000e-02,
                                2.71370000e-01, 7.90740500e-01, 9.03034000e-01,
                                9.04677500e-01, 8.89667000e-01, 8.79232000e-01,
                                8.79688000e-01, 8.89796500e-01, 8.48533000e-01,
                                8.36270500e-01, 8.68497000e-01, 9.11461500e-01,
                                9.31726000e-01, 9.54896500e-01, 9.56424000e-01,
                                9.83834000e-01, 9.89469000e-01, 9.68066500e-01,
                                9.88729000e-01, 9.61057500e-01, 9.66125000e-01,
                                9.82077000e-01, 9.63135000e-01, 9.98249000e-01,
                                8.44893000e-01, 1.19533500e-01, 5.32800000e-03,
                                1.32850000e-03, 5.16000000e-04, 1.17000000e-04,
                                2.30000000e-05]))

    LANDSAT_OLI_B3 = (3, 0.512, 0.610,
                      np.array([-4.60000000e-05, 1.78500000e-04, 6.48000000e-04,
                                1.57400000e-03, 3.44600000e-03, 8.73250000e-03,
                                2.55130000e-02, 9.69975000e-02, 3.53885000e-01,
                                8.03215000e-01, 9.54627000e-01, 9.60271500e-01,
                                9.69873000e-01, 9.69833500e-01, 9.77001000e-01,
                                9.95392000e-01, 9.82642000e-01, 9.71423000e-01,
                                9.46245000e-01, 9.62786000e-01, 9.66447000e-01,
                                9.64176500e-01, 9.83397000e-01, 9.70875500e-01,
                                9.78208000e-01, 9.77182500e-01, 9.69181000e-01,
                                9.81277000e-01, 9.68886000e-01, 9.80432000e-01,
                                9.04478000e-01, 6.05139000e-01, 1.90467000e-01,
                                2.47350000e-02, 2.57400000e-03, 2.39500000e-04,
                                0, 0, 0,
                                0]))

    LANDSAT_OLI_B4 = (4, 0.625, 0.690,
                      np.array([-3.42000000e-04, 1.37250000e-03, 7.19700000e-03,
                                4.86465000e-02, 2.99778000e-01, 8.34958000e-01,
                                9.50823000e-01, 9.57268000e-01, 9.84173000e-01,
                                9.83172500e-01, 9.59441000e-01, 9.54442500e-01,
                                9.81688000e-01, 9.88501500e-01, 9.76960000e-01,
                                9.88942000e-01, 9.80678000e-01, 9.66466000e-01,
                                9.66928000e-01, 7.29107000e-01, 1.23946000e-01,
                                1.25175000e-02, 1.40200000e-03, 0,
                                0, 0, 0]))

    LANDSAT_OLI_B5 = (5, 0.829, 0.899,
                      np.array([0, 7.50000000e-05, 3.14000000e-04,
                                8.52500000e-04, 2.10700000e-03, 5.90150000e-03,
                                1.73460000e-02, 6.62770000e-02, 2.49733000e-01,
                                6.63830000e-01, 9.60215000e-01, 9.76869500e-01,
                                1.00000000e+00, 9.78334000e-01, 9.57357000e-01,
                                9.50103000e-01, 9.48450000e-01, 9.53355500e-01,
                                9.69821000e-01, 8.39899500e-01, 4.48364000e-01,
                                1.37481000e-01, 3.45320000e-02, 1.00205000e-02,
                                2.94400000e-03, 9.67500000e-04, 2.41000000e-04,
                                1.55000000e-05, 0]))

    LANDSAT_OLI_B6 = (6, 1.515, 1.695,
                      np.array([0, 2.00000000e-04, 4.66000000e-04,
                                8.45000000e-04, 1.36900000e-03, 2.01550000e-03,
                                2.88100000e-03, 4.02150000e-03, 5.52800000e-03,
                                7.88900000e-03, 1.09890000e-02, 1.52755000e-02,
                                2.18310000e-02, 3.25615000e-02, 4.78640000e-02,
                                7.09490000e-02, 1.01893000e-01, 1.50884500e-01,
                                2.20261000e-01, 3.10649000e-01, 4.21470000e-01,
                                5.52234000e-01, 6.76683000e-01, 7.71509000e-01,
                                8.54065000e-01, 8.95823500e-01, 9.13009000e-01,
                                9.25058000e-01, 9.26413000e-01, 9.23818000e-01,
                                9.22828000e-01, 9.22408500e-01, 9.26605000e-01,
                                9.43438000e-01, 9.46175000e-01, 9.47297500e-01,
                                9.52859000e-01, 9.51358500e-01, 9.59047000e-01,
                                9.59191500e-01, 9.61470000e-01, 9.60494000e-01,
                                9.64703000e-01, 9.69951500e-01, 9.76906000e-01,
                                9.81271500e-01, 9.88609000e-01, 9.99010500e-01,
                                9.99642000e-01, 9.89828000e-01, 9.67125000e-01,
                                9.26719000e-01, 8.40967000e-01, 7.23103000e-01,
                                5.73232000e-01, 4.22998000e-01, 2.91752000e-01,
                                1.95988000e-01, 1.28463000e-01, 8.28380000e-02,
                                5.27520000e-02, 3.45655000e-02, 2.25040000e-02,
                                1.47195000e-02, 9.58700000e-03, 6.39600000e-03,
                                4.25700000e-03, 2.79800000e-03, 1.78100000e-03,
                                1.14200000e-03, 6.77000000e-04, 3.55000000e-04,
                                1.12000000e-04]))

    LANDSAT_OLI_B7 = (7, 2.037, 2.354,
                      np.array([0, 1.07000000e-04, 2.40000000e-04,
                                3.99000000e-04, 5.99000000e-04, 8.80500000e-04,
                                1.22200000e-03, 1.64550000e-03, 2.18700000e-03,
                                2.85300000e-03, 3.73300000e-03, 4.88200000e-03,
                                6.33700000e-03, 8.44950000e-03, 1.10050000e-02,
                                1.42985000e-02, 1.88990000e-02, 2.45090000e-02,
                                3.20710000e-02, 4.28190000e-02, 5.64290000e-02,
                                7.48955000e-02, 1.00640000e-01, 1.36480000e-01,
                                1.79714000e-01, 2.40483000e-01, 3.11347000e-01,
                                3.94832500e-01, 4.88816000e-01, 5.73971000e-01,
                                6.63067000e-01, 7.39406500e-01, 7.92667000e-01,
                                8.41172500e-01, 8.67845000e-01, 8.86269000e-01,
                                9.06527000e-01, 9.14538000e-01, 9.29693000e-01,
                                9.38975000e-01, 9.42952000e-01, 9.44181000e-01,
                                9.48776000e-01, 9.49521500e-01, 9.56635000e-01,
                                9.48258500e-01, 9.50874000e-01, 9.47049500e-01,
                                9.57717000e-01, 9.47095000e-01, 9.51641000e-01,
                                9.46800000e-01, 9.40311000e-01, 9.46466500e-01,
                                9.38737000e-01, 9.44439000e-01, 9.44482000e-01,
                                9.50472000e-01, 9.39939000e-01, 9.37156500e-01,
                                9.38955000e-01, 9.28123500e-01, 9.30508000e-01,
                                9.30946000e-01, 9.36472000e-01, 9.34327500e-01,
                                9.46217000e-01, 9.53826000e-01, 9.63135000e-01,
                                9.63944000e-01, 9.62905000e-01, 9.61607000e-01,
                                9.57814000e-01, 9.55657500e-01, 9.51706000e-01,
                                9.60275500e-01, 9.47696000e-01, 9.59807000e-01,
                                9.55750000e-01, 9.56607500e-01, 9.66786000e-01,
                                9.62823000e-01, 9.77637000e-01, 9.83457500e-01,
                                9.85056000e-01, 9.98627000e-01, 9.92469000e-01,
                                9.97947000e-01, 9.97261000e-01, 9.89437000e-01,
                                9.86037000e-01, 9.81280000e-01, 9.72794000e-01,
                                9.76369500e-01, 9.74409000e-01, 9.63698500e-01,
                                9.55095000e-01, 9.51391500e-01, 9.22405000e-01,
                                8.89264000e-01, 8.23876000e-01, 7.21272500e-01,
                                6.02539000e-01, 4.77695500e-01, 3.55569000e-01,
                                2.61452500e-01, 1.86151000e-01, 1.31725000e-01,
                                9.20290000e-02, 6.49895000e-02, 4.63320000e-02,
                                3.34235000e-02, 2.40000000e-02, 1.76250000e-02,
                                1.29300000e-02, 9.55700000e-03, 7.08800000e-03,
                                5.33100000e-03, 3.90300000e-03, 2.83800000e-03,
                                2.04700000e-03, 1.44950000e-03, 9.74000000e-04,
                                6.20000000e-04, 3.20000000e-04, 7.35000000e-05,
                                0, 0]))

    LANDSAT_OLI_B8 = (8, 0.488, 0.691,
                       np.array([2.16000000e-04, 1.30000000e-03, 3.84100000e-03,
                                 1.22590000e-02, 4.27230000e-02, 1.60137500e-01,
                                 4.72496000e-01, 7.45412500e-01, 8.31881000e-01,
                                 8.55321500e-01, 8.59640000e-01, 8.57696000e-01,
                                 8.58455000e-01, 8.58301000e-01, 8.50183000e-01,
                                 8.58223500e-01, 8.61508000e-01, 8.57683500e-01,
                                 8.79249000e-01, 8.91710500e-01, 9.06294000e-01,
                                 9.12867000e-01, 9.02939000e-01, 9.20739500e-01,
                                 9.13020000e-01, 8.85650500e-01, 8.79443000e-01,
                                 8.74179000e-01, 8.75361000e-01, 8.91665000e-01,
                                 8.74097000e-01, 8.86888500e-01, 9.03528000e-01,
                                 9.10950500e-01, 9.13190000e-01, 9.20178000e-01,
                                 9.24431000e-01, 9.29809500e-01, 9.48863000e-01,
                                 9.40543000e-01, 9.45674000e-01, 9.39380000e-01,
                                 9.46659000e-01, 9.34044500e-01, 9.40838000e-01,
                                 9.58039500e-01, 9.68241000e-01, 9.66480500e-01,
                                 9.57232000e-01, 9.47675500e-01, 9.52465000e-01,
                                 9.57481500e-01, 9.64158000e-01, 9.67366000e-01,
                                 9.77026000e-01, 9.76029500e-01, 9.69583000e-01,
                                 9.72807000e-01, 9.65780000e-01, 9.66738000e-01,
                                 9.72067000e-01, 9.79346500e-01, 9.71123000e-01,
                                 9.53377000e-01, 9.63851000e-01, 9.67101500e-01,
                                 9.70613000e-01, 9.79974500e-01, 9.88302000e-01,
                                 9.91753000e-01, 1.00000000e+00, 9.98476000e-01,
                                 9.92555000e-01, 9.85811500e-01, 9.13945000e-01,
                                 5.24376500e-01, 1.67313000e-01, 4.61755000e-02,
                                 1.51780000e-02, 6.70950000e-03, 3.22000000e-03,
                                 1.21200000e-03]))

    LANDSAT_OLI_PAN = LANDSAT_OLI_B8

    LANDSAT_OLI_B9 = (9, 1.340, 1.407,
                      np.array([0, 2.57500000e-04, 6.47000000e-04,
                                1.27400000e-03, 2.31800000e-03, 4.70450000e-03,
                                1.11240000e-02, 3.45385000e-02, 1.15351000e-01,
                                3.86681500e-01, 7.72118000e-01, 9.00941500e-01,
                                9.31247000e-01, 9.91687000e-01, 1.00000000e+00,
                                9.77080500e-01, 8.71343000e-01, 6.54888000e-01,
                                2.97920000e-01, 8.96145000e-02, 2.60840000e-02,
                                8.30650000e-03, 2.78000000e-03, 1.11900000e-03,
                                2.20000000e-04, 0, 0,
                                0]))


    # RapidEye
    # Taken from http://blackbridge.com/rapideye/upload/Spectral_Response_Curves.pdf
    # Interpolated to 2.5nm intervals, as required by 6S
    RAPIDEYE_B1 = (10, 0.420, 0.514,
                    np.array([ 0.002 ,  0.    ,  0.    ,  0.0015,  0.    ,  0.    ,  0.001 ,
                            0.0095,  0.321 ,  0.725 ,  0.74  ,  0.759 ,  0.77  ,  0.781 ,
                            0.784 ,  0.7935,  0.796 ,  0.8005,  0.806 ,  0.804 ,  0.807 ,
                            0.817 ,  0.82  ,  0.8275,  0.84  ,  0.847 ,  0.862 ,  0.8765,
                            0.886 ,  0.9105,  0.928 ,  0.9415,  0.969 ,  0.9685,  1.    ,
                            0.9875,  0.437 ,  0.019, 0.0]))

    RAPIDEYE_B2 = (11, 0.462, 0.595,
                    np.array([ 0.002 ,  0.0025,  0.002 ,  0.021 ,  0.    ,  0.    ,  0.    ,
                            0.    ,  0.    ,  0.    ,  0.    ,  0.0145,  0.    ,  0.    ,
                            0.    ,  0.    ,  0.    ,  0.    ,  0.    ,  0.001 ,  0.002 ,
                            0.01  ,  0.054 ,  0.4055,  0.868 ,  0.866 ,  0.877 ,  0.872 ,
                            0.874 ,  0.8815,  0.882 ,  0.8805,  0.886 ,  0.8955,  0.899 ,
                            0.8995,  0.91  ,  0.922 ,  0.928 ,  0.9345,  0.946 ,  0.9525,
                            0.96  ,  0.972 ,  0.976 ,  0.975 ,  0.989 ,  0.9905,  0.984 ,
                            0.997 ,  0.97  ,  0.6535,  0.039 ,  0.004 ]))

    RAPIDEYE_B3 = (12, 0.486, 0.691,
                    np.array([  2.00000000e-03,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             1.00000000e-03,   5.00000000e-04,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   5.00000000e-04,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   5.00000000e-04,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             1.00000000e-03,   4.00000000e-03,   1.80000000e-02,
                             1.43000000e-01,   7.23000000e-01,   8.57500000e-01,
                             8.65000000e-01,   8.81500000e-01,   8.82000000e-01,
                             8.93000000e-01,   9.07000000e-01,   9.12500000e-01,
                             9.18000000e-01,   9.31000000e-01,   9.44000000e-01,
                             9.52500000e-01,   9.61000000e-01,   9.67000000e-01,
                             9.73000000e-01,   9.83000000e-01,   9.91000000e-01,
                             9.87000000e-01,   9.89000000e-01,   9.99000000e-01,
                             9.81000000e-01,   9.28500000e-01,   1.74000000e-01,
                             1.70000000e-02, 0.0]))

    RAPIDEYE_B4 = (13, 0.500, 0.740,
                    np.array([  2.70000000e-02,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   5.00000000e-04,
                             0.00000000e+00,   5.00000000e-04,   1.00000000e-03,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   5.00000000e-04,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   2.20000000e-02,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   5.00000000e-04,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   1.00000000e-03,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   1.00000000e-03,
                             2.00000000e-03,   5.50000000e-03,   2.10000000e-02,
                             1.08500000e-01,   4.91000000e-01,   9.49000000e-01,
                             9.98000000e-01,   9.99500000e-01,   9.98000000e-01,
                             9.91500000e-01,   9.87000000e-01,   9.85500000e-01,
                             9.82000000e-01,   9.74000000e-01,   9.66000000e-01,
                             9.64000000e-01,   9.61000000e-01,   9.45500000e-01,
                             9.39000000e-01,   8.98000000e-01,   4.25000000e-01,
                             9.40000000e-02,   2.00000000e-02,   5.50000000e-03, 0.0]))

    RAPIDEYE_B5 = (14, 0.519, 0.866,
                    np.array([  2.00000000e-03,   5.00000000e-04,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   1.00000000e-03,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   2.00000000e-03,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   1.00000000e-03,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   4.00000000e-03,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             1.00000000e-03,   2.00000000e-03,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   1.00000000e-03,   2.00000000e-03,
                             5.00000000e-04,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   5.00000000e-04,   2.20000000e-02,
                             5.00000000e-04,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   0.00000000e+00,   0.00000000e+00,
                             0.00000000e+00,   1.00000000e-03,   1.00000000e-03,
                             2.00000000e-03,   4.50000000e-03,   9.00000000e-03,
                             1.95000000e-02,   4.60000000e-02,   1.27500000e-01,
                             3.45000000e-01,   7.49000000e-01,   9.88000000e-01,
                             9.88000000e-01,   9.69000000e-01,   9.71500000e-01,
                             9.81000000e-01,   9.82000000e-01,   9.74000000e-01,
                             9.65000000e-01,   9.60000000e-01,   9.57500000e-01,
                             9.59000000e-01,   9.59000000e-01,   9.57000000e-01,
                             9.54500000e-01,   9.50000000e-01,   9.45000000e-01,
                             9.39000000e-01,   9.34000000e-01,   9.29000000e-01,
                             9.27000000e-01,   9.26000000e-01,   9.26000000e-01,
                             9.25000000e-01,   9.19500000e-01,   9.11000000e-01,
                             9.00000000e-01,   8.91000000e-01,   8.85500000e-01,
                             8.83000000e-01,   8.78000000e-01,   8.65000000e-01,
                             8.45500000e-01,   8.38000000e-01,   8.37000000e-01,
                             7.12000000e-01,   3.63000000e-01,   1.24000000e-01,
                             4.30000000e-02,   1.60000000e-02,   7.00000000e-03,
                             3.00000000e-03], ))


    # Pleiades 1B
    # Taken from ENVI Spectral Library file
    # Interpolated to 2.5nm intervals, as required by 6S
    #
    # NOTE: There are out-of-band responses for a number of these
    # bands (ie. responses in the NIR for the blue band) and these
    # *are* included, giving strange min/max wavelengths
    #
    PLEIADES_B1 = (15, 0.438, 0.915, # Blue
                    np.array([ 0.001603,  0.001338,  0.004344,  0.011072,  0.017627,  0.023367,
                    0.057403,  0.134555,  0.223763,  0.308543,  0.461557,  0.650821,
                    0.755369,  0.74772 ,  0.716819,  0.718538,  0.756696,  0.810109,
                    0.842166,  0.823437,  0.775247,  0.752701,  0.78022 ,  0.819927,
                    0.851663,  0.860275,  0.858684,  0.865826,  0.882822,  0.904041,
                    0.919695,  0.932596,  0.950233,  0.975798,  0.994977,  1.      ,
                    0.995063,  0.980628,  0.94175 ,  0.843622,  0.671142,  0.46334 ,
                    0.288865,  0.167078,  0.09018 ,  0.050519,  0.031488,  0.023814,
                    0.021311,  0.02063 ,  0.01956 ,  0.016794,  0.011358,  0.006652,
                    0.004144,  0.00303 ,  0.002416,  0.00199 ,  0.001568,  0.001136,
                    0.001253,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.001109,  0.001987,  0.00122 ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.001119,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.001629,
                    0.0024  ,  0.001032,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.001028,  0.001563,  0.002609,
                    0.004575,  0.009062,  0.017163,  0.023793,  0.021523,  0.015914,
                    0.011956,  0.009482,  0.007869,  0.007224,  0.007242,  0.007568,
                    0.008379,  0.009864,  0.012259,  0.016267,  0.022602,  0.032027,
                    0.04443 ,  0.054669,  0.056424,  0.048004,  0.035799,  0.025834,
                    0.018887,  0.014439,  0.011679,  0.009911,  0.008647,  0.00781 ,
                    0.007227,  0.006996,  0.006923,  0.006914,  0.007021,  0.007253,
                    0.007373,  0.007505,  0.00747 ,  0.007067]))

    PLEIADES_B2 = (16, 0.490, 0.950, # Green
                    np.array([ 0.001521,  0.00413 ,  0.009954,  0.019642,  0.032548,  0.056793,
                    0.123791,  0.285909,  0.528976,  0.771625,  0.883804,  0.907957,
                    0.913146,  0.913728,  0.922484,  0.936708,  0.94976 ,  0.954499,
                    0.958582,  0.964206,  0.970527,  0.972265,  0.967518,  0.95891 ,
                    0.952449,  0.952466,  0.956048,  0.955179,  0.94899 ,  0.947145,
                    0.95445 ,  0.97106 ,  0.989818,  1.      ,  0.99536 ,  0.969822,
                    0.925304,  0.863324,  0.794828,  0.723897,  0.645327,  0.543852,
                    0.417028,  0.276671,  0.157527,  0.085607,  0.049226,  0.032724,
                    0.023793,  0.018197,  0.014062,  0.009966,  0.005845,  0.003038,
                    0.001536,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.003185,  0.014191,  0.008339,  0.002244,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.001111,  0.001938,  0.003691,
                    0.006357,  0.010271,  0.013108,  0.013261,  0.011115,  0.008366,
                    0.006564,  0.005685,  0.00538 ,  0.005623,  0.0062  ,  0.007239,
                    0.00892 ,  0.01151 ,  0.014942,  0.019269,  0.023479,  0.02644 ,
                    0.027049,  0.025545,  0.023397,  0.021395,  0.019944,  0.019253,
                    0.019074,  0.019689,  0.020694,  0.022011,  0.02323 ,  0.023757,
                    0.022986,  0.02066 ,  0.017225,  0.013292, 0.0]))


    PLEIADES_B3 = (17, 0.448, 0.940, # RED
                    np.array([ 0.001195,  0.005142,  0.008003,  0.006693,  0.010859,  0.024691,
                            0.065359,  0.122542,  0.057009,  0.021375,  0.012797,  0.006278,
                            0.002354,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                            0.      ,  0.001013,  0.      ,  0.      ,  0.      ,  0.      ,
                            0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                            0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                            0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                            0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                            0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                            0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                            0.      ,  0.      ,  0.      ,  0.      ,  0.001244,  0.002142,
                            0.003819,  0.006805,  0.012333,  0.022178,  0.041333,  0.078071,
                            0.151934,  0.277675,  0.451038,  0.629132,  0.762549,  0.832945,
                            0.857906,  0.865887,  0.869263,  0.875221,  0.885776,  0.900593,
                            0.917488,  0.93488 ,  0.947811,  0.956953,  0.96233 ,  0.964767,
                            0.962429,  0.961307,  0.962025,  0.969915,  0.981157,  0.993393,
                            1.      ,  0.980951,  0.952263,  0.913173,  0.869401,  0.825208,
                            0.783047,  0.736127,  0.673489,  0.587753,  0.480491,  0.363007,
                            0.252303,  0.162603,  0.102221,  0.064127,  0.041916,  0.028464,
                            0.020455,  0.01537 ,  0.012117,  0.009881,  0.008317,  0.007102,
                            0.006095,  0.005172,  0.004314,  0.003495,  0.002771,  0.003589,
                            0.003031,  0.002317,  0.001784,  0.001331,  0.001021,  0.      ,
                            0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                            0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                            0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                            0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                            0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                            0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                            0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                            0.      ,  0.      ,  0.      ,  0.001308,  0.001761,  0.002403,
                            0.003268,  0.004364,  0.005662,  0.00689 ,  0.007822,  0.00833 ,
                            0.008436,  0.008185,  0.008138,  0.008001,  0.007768,  0.007784,
                            0.007998,  0.008152,  0.008506,  0.008793,  0.009312,  0.009753,
                            0.010136,  0.010387,  0.010406,  0.010171,  0.009522,  0.008609,
                            0.007337]))

    PLEIADES_B4 = (18, 0.750, 0.945, # NIR
                    np.array([ 0.001687,  0.002665,  0.001225,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.001549,  0.00218 ,  0.001988,  0.001045,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.001224,  0.002266,  0.002879,
                    0.002041,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.      ,
                    0.      ,  0.      ,  0.      ,  0.      ,  0.      ,  0.001154,
                    0.001352,  0.001554,  0.001756,  0.001989,  0.002251,  0.002534,
                    0.002903,  0.005236,  0.006401,  0.008013,  0.010147,  0.013109,
                    0.017135,  0.022905,  0.030978,  0.042662,  0.05919 ,  0.083507,
                    0.117888,  0.166378,  0.231114,  0.315936,  0.417216,  0.528495,
                    0.640959,  0.747239,  0.838552,  0.908812,  0.959366,  0.988114,
                    1.      ,  0.999206,  0.989642,  0.967696,  0.951436,  0.937494,
                    0.925472,  0.915223,  0.908783,  0.902608,  0.896683,  0.892483,
                    0.885491,  0.877655,  0.867639,  0.856896,  0.847441,  0.836048,
                    0.823698,  0.813044,  0.801627,  0.792162,  0.782795,  0.776935,
                    0.771465,  0.763145,  0.75434 ,  0.745237,  0.734921,  0.725512,
                    0.714614,  0.703089,  0.691948,  0.681648,  0.67052 ,  0.659536,
                    0.647596,  0.629185,  0.611093,  0.593432,  0.576059,  0.559994,
                    0.544875,  0.530614,  0.515483,  0.497676,  0.47341 ,  0.43877 ,
                    0.393422,  0.336769,  0.274273,  0.211434,  0.153554,  0.106341,
                    0.070442,  0.045912,  0.029535,  0.019275,  0.01268 ,  0.008462]))

    PLEIADES_PAN = (19, 0.460, 0.845, # Panchromatic
                    np.array([ 0.002795,  0.005578,  0.011986,  0.022135,  0.038918,  0.066419,
                                0.106702,  0.161427,  0.227202,  0.298655,  0.367815,  0.425804,
                                0.46974 ,  0.499166,  0.519316,  0.53325 ,  0.546726,  0.560304,
                                0.573644,  0.585534,  0.595662,  0.604104,  0.612015,  0.618774,
                                0.625565,  0.631905,  0.637794,  0.642186,  0.644395,  0.644882,
                                0.644885,  0.645114,  0.646836,  0.651318,  0.659091,  0.669141,
                                0.681181,  0.693702,  0.705609,  0.715769,  0.725057,  0.732351,
                                0.738419,  0.743876,  0.749253,  0.753645,  0.758079,  0.761089,
                                0.763913,  0.765235,  0.766172,  0.766597,  0.767495,  0.768755,
                                0.772303,  0.777108,  0.783383,  0.791719,  0.801545,  0.810952,
                                0.820033,  0.829068,  0.836465,  0.842652,  0.848188,  0.851917,
                                0.855505,  0.858336,  0.862017,  0.866684,  0.87158 ,  0.878019,
                                0.884707,  0.891347,  0.898891,  0.904582,  0.910509,  0.915462,
                                0.920794,  0.925316,  0.928988,  0.932893,  0.936956,  0.94008 ,
                                0.94416 ,  0.94693 ,  0.950347,  0.952109,  0.95428 ,  0.957441,
                                0.960771,  0.966315,  0.9721  ,  0.978828,  0.984596,  0.991222,
                                0.995687,  0.999088,  1.      ,  0.996327,  0.991468,  0.986969,
                                0.981702,  0.977146,  0.974597,  0.972901,  0.973932,  0.974192,
                                0.976871,  0.979825,  0.982772,  0.984573,  0.985077,  0.983727,
                                0.982597,  0.979023,  0.974595,  0.969741,  0.964814,  0.959839,
                                0.953335,  0.948357,  0.944548,  0.939689,  0.934456,  0.929716,
                                0.924834,  0.918336,  0.911194,  0.9061  ,  0.900618,  0.895346,
                                0.890809,  0.889495,  0.888541,  0.880978,  0.870784,  0.856668,
                                0.836935,  0.810241,  0.772852,  0.718955,  0.65285 ,  0.571081,
                                0.479585,  0.384649,  0.294773,  0.216645,  0.153132,  0.105705,
                                0.07168 ,  0.049143,  0.033776,  0.025699,  0.018176,  0.013197,
                                0.012156,  0.00902 ,  0.008416,  0.00428 ,  0.003313,  0.002669,
                                0.002212,  0.001829,  0.001529,  0.001363,  0.001215]))
    # Sentinel-2A MSI spectral response functions
    # Code used to create these things in
    # https://github.com/jgomezdans/sentinel_SRF
    S2A_MSI_01 = (53, 0.43000, 0.45500,
                  np.array(
                      [0.01530, 0.27659, 0.51160, 0.60043, 0.69920, 0.82744,
                       0.99388, 0.98154, 0.95673, 0.55612, 0.05181, ]))

    S2A_MSI_02 = (54, 0.45500, 0.53000,
                  np.array(
                      [0.00903, 0.06152, 0.29220, 0.38410, 0.40016, 0.43699,
                       0.50532, 0.53212, 0.53466, 0.55070, 0.60197, 0.61734,
                       0.57586, 0.54412, 0.57168, 0.65330, 0.73840, 0.77085,
                       0.78836, 0.81658, 0.84498, 0.83295, 0.78694, 0.76370,
                       0.81003, 0.92817, 1.00000, 0.81615, 0.28699, 0.08400,
                       0.02508, ]))

    S2A_MSI_03 = (55, 0.53750, 0.58250,
                  np.array(
                      [0.00861, 0.08067, 0.45720, 0.82804, 0.89007, 0.86027,
                       0.83333, 0.86773, 0.95043, 1.00000, 0.96410, 0.86866,
                       0.80267, 0.78961, 0.83845, 0.85799, 0.50599, 0.09829,
                       0.00826, ]))

    S2A_MSI_04 = (56, 0.64750, 0.68250,
                  np.array(
                      [0.09225, 0.81775, 0.99038, 0.99545, 0.95701, 0.81417,
                       0.76998, 0.83083, 0.89627, 0.95593, 0.97240, 0.96571,
                       0.91448, 0.42297, 0.04189, ]))

    S2A_MSI_05 = (57, 0.69500, 0.71250,
                  np.array(
                      [0.04126, 0.65619, 1.00000, 0.98783, 0.94595, 0.89399,
                       0.75720, 0.11677, ]))

    S2A_MSI_06 = (58, 0.73250, 0.74750,
                  np.array(
                      [0.21536, 0.92026, 0.94604, 0.98193, 0.99670, 0.96258,
                       0.32104, ]))

    S2A_MSI_07 = (59, 0.77000, 0.79500,
                  np.array(
                      [0.01473, 0.31115, 0.89849, 0.99738, 0.96466, 0.82449,
                       0.77724, 0.80195, 0.75770, 0.45044, 0.07722, ]))

    S2A_MSI_08 = (60, 0.77500, 0.90500,
                  np.array(
                      [0.01907, 0.07184, 0.20344, 0.51876, 0.81829, 0.97723,
                       0.98521, 0.93118, 0.94128, 0.96391, 0.95901, 0.94414,
                       0.94536, 0.93249, 0.90098, 0.85268, 0.80182, 0.74676,
                       0.70867, 0.68623, 0.68265, 0.67363, 0.66081, 0.66119,
                       0.68550, 0.73149, 0.77661, 0.78771, 0.77616, 0.75260,
                       0.72059, 0.68338, 0.64934, 0.62319, 0.60432, 0.58934,
                       0.58120, 0.58135, 0.58948, 0.59906, 0.60548, 0.61609,
                       0.63744, 0.66329, 0.65992, 0.59970, 0.52641, 0.49782,
                       0.52909, 0.51696, 0.32679, 0.11911, 0.03325, ]))

    S2A_MSI_09 = (61, 0.85000, 0.88000,
                  np.array(
                      [0.02471, 0.16076, 0.58573, 0.89669, 0.92604, 0.94127,
                       0.96546, 0.98160, 0.98847, 0.96032, 0.47219, 0.07653,
                       0.00882, ]))

    S2A_MSI_10 = (62, 0.93250, 0.95750,
                  np.array(
                      [0.04186, 0.40811, 0.90180, 0.98357, 0.98700, 0.99812,
                       0.99376, 0.95641, 0.89743, 0.45044, 0.03687, ]))

    S2A_MSI_11 = (63, 1.35250, 1.39500,
                  np.array(
                      [0.00588, 0.04466, 0.27162, 0.66250, 0.88561, 0.96202,
                       0.98483, 0.99648, 0.99955, 0.99096, 0.97423, 0.94860,
                       0.90454, 0.83074, 0.58158, 0.21931, 0.05050, 0.00550, ]))

    S2A_MSI_12 = (64, 1.55500, 1.67000,
                  np.array(
                      [0.00535, 0.01581, 0.03958, 0.08800, 0.18260, 0.38021,
                       0.64717, 0.83721, 0.88703, 0.89450, 0.91653, 0.93886,
                       0.95142, 0.95732, 0.96348, 0.96960, 0.97591, 0.98023,
                       0.98141, 0.98054, 0.98274, 0.98909, 0.99329, 0.98877,
                       0.98002, 0.97120, 0.96637, 0.96674, 0.97346, 0.98555,
                       0.99548, 0.99895, 0.99880, 0.99974, 0.99981, 0.98866,
                       0.96990, 0.94992, 0.93859, 0.92011, 0.82498, 0.58795,
                       0.32371, 0.13709, 0.04619, 0.01633, 0.00652, ]))

    S2A_MSI_13 = (65, 2.08250, 2.31750,
                  np.array(
                      [0.00713, 0.00854, 0.01049, 0.01336, 0.01845, 0.02767,
                       0.04455, 0.07318, 0.12419, 0.20346, 0.30604, 0.40800,
                       0.48982, 0.54335, 0.57383, 0.59889, 0.62782, 0.66371,
                       0.70406, 0.74130, 0.77883, 0.80968, 0.83266, 0.85111,
                       0.85546, 0.85953, 0.86420, 0.86970, 0.88051, 0.88947,
                       0.89830, 0.90483, 0.90558, 0.90478, 0.90305, 0.90198,
                       0.90481, 0.90809, 0.91427, 0.92130, 0.92848, 0.93414,
                       0.93736, 0.93765, 0.94289, 0.94212, 0.93754, 0.93302,
                       0.91827, 0.90829, 0.90968, 0.92285, 0.91949, 0.92453,
                       0.93437, 0.94680, 0.95574, 0.96254, 0.96634, 0.96546,
                       0.96268, 0.95733, 0.95314, 0.95173, 0.95335, 0.96064,
                       0.97050, 0.98290, 0.99236, 0.99875, 0.99954, 0.99388,
                       0.97997, 0.95834, 0.93296, 0.90600, 0.87516, 0.84062,
                       0.80069, 0.74911, 0.66973, 0.56603, 0.45196, 0.34209,
                       0.24544, 0.16809, 0.11653, 0.08236, 0.05884, 0.04286,
                       0.03347, 0.02742, 0.02282, 0.01633, 0.00694, ]))

    # Sentinel-3A OLCI spectral response functions
    # Code used to create these things in
    # (PREFLIGHT)
    # https://github.com/jgomezdans/sentinel_SRF
    S3A_OLCI_01 = (66, 0.39250, 0.40750,
                   np.array(
                       [0.32217, 0.60993, 0.74246, 0.88513, 0.95068, 0.99827,
                        0.23903, ]))

    S3A_OLCI_02 = (67, 0.40750, 0.41750,
                   np.array([0.77881, 0.99448, 0.97237, 0.93074, 0.17144, ]))

    S3A_OLCI_03 = (68, 0.43750, 0.44750,
                   np.array([0.21358, 0.96898, 0.99506, 0.99662, 0.75555, ]))

    S3A_OLCI_04 = (69, 0.48500, 0.49500,
                   np.array([0.24172, 0.97037, 0.97933, 0.99647, 0.76159, ]))

    S3A_OLCI_05 = (70, 0.50500, 0.51500,
                   np.array([0.25057, 0.97259, 0.99015, 0.99820, 0.73490, ]))

    S3A_OLCI_06 = (71, 0.55500, 0.56500,
                   np.array([0.27069, 0.99061, 0.99625, 0.99897, 0.73516, ]))

    S3A_OLCI_07 = (72, 0.61500, 0.62500,
                   np.array([0.28674, 0.97507, 0.98591, 0.99655, 0.68738, ]))

    S3A_OLCI_08 = (73, 0.66000, 0.67000,
                   np.array([0.36783, 0.97355, 0.98640, 0.99453, 0.62637, ]))

    S3A_OLCI_09 = (74, 0.67000, 0.67750,
                   np.array([0.36955, 0.98467, 0.99410, 0.64153, ]))

    S3A_OLCI_10 = (75, 0.67750, 0.68500,
                   np.array([0.35029, 0.98023, 0.99506, 0.66193, ]))

    S3A_OLCI_11 = (76, 0.70250, 0.71500,
                   np.array([0.01932, 0.89702, 0.99309, 0.99612, 0.97993,
                             0.09583, ]))

    S3A_OLCI_12 = (77, 0.75000, 0.75750,
                   np.array([0.28469, 0.99352, 0.99288, 0.72022, ]))

    S3A_OLCI_13 = (78, 0.76000, 0.76250,
                   np.array([0.28877, 0.81261, ]))

    S3A_OLCI_14 = (79, 0.76250, 0.76750,
                   np.array([0.26409, 0.99532, 0.10865, ]))

    S3A_OLCI_15 = (80, 0.76500, 0.77000,
                   np.array([0.01674, 0.96455, 0.11181, ]))

    S3A_OLCI_16 = (81, 0.77000, 0.78750,
                   np.array(
                       [0.01378, 0.88436, 0.99013, 0.99161, 0.99680, 0.99688,
                        0.98205, 0.11870, ]))

    S3A_OLCI_17 = (82, 0.85500, 0.87500,
                   np.array(
                       [0.21133, 0.99725, 0.99219, 0.98298, 0.97348, 0.96025,
                        0.94621, 0.93225, 0.73966, ]))

    S3A_OLCI_18 = (83, 0.87750, 0.89000,
                   np.array([0.00628, 0.86036, 0.99405, 0.97826, 0.95623,
                             0.14361, ]))

    S3A_OLCI_19 = (84, 0.89250, 0.90500,
                   np.array([0.00569, 0.85567, 0.99341, 0.97455, 0.94956,
                             0.14724, ]))

    S3A_OLCI_20 = (85, 0.93000, 0.95000,
                   np.array(
                       [0.85273, 0.98430, 0.94674, 0.90965, 0.87387, 0.84706,
                        0.82485, 0.79976, 0.14646, ]))

    S3A_OLCI_21 = (86, 1.00000, 1.04000,
                   np.array(
                       [0.83262, 0.97438, 0.91402, 0.85469, 0.79628, 0.73780,
                        0.68020, 0.62362, 0.57065, 0.53786, 0.50696, 0.47701,
                        0.44798, 0.41979, 0.39202, 0.36264, 0.06327, ]))

    # Sentinel-3A SLSTR spectral response functions
    # (PREFLIGHT)
    # Code used to create these things in
    # https://github.com/jgomezdans/sentinel_SRF

    S3A_SLSTR_01 = (87, 0.54000, 0.57000,
                    np.array(
                        [0.01463, 0.14031, 0.57941, 0.95383, 0.99927, 0.98361,
                         0.98065, 0.99224, 0.86356, 0.74371, 0.25806, 0.05162,
                         0.00906, ]))

    S3A_SLSTR_02 = (88, 0.64500, 0.67500,
                    np.array(
                        [0.00868, 0.10051, 0.54166, 0.94638, 0.97952, 0.99970,
                         0.99593, 0.98108, 0.98249, 0.81191, 0.28596, 0.03422,
                         0.00557, ]))

    S3A_SLSTR_03 = (89, 0.85250, 0.88500,
                    np.array(
                        [0.01457, 0.09982, 0.45901, 0.94725, 0.99882, 0.99531,
                         0.97779, 0.89514, 0.91546, 0.92751, 0.67702, 0.17794,
                         0.02665, 0.00552, ]))

    S3A_SLSTR_04 = (90, 1.36250, 1.39000,
                    np.array(
                        [0.08117, 0.62916, 0.97652, 0.96779, 0.98766, 0.99534,
                         0.99847, 0.99996, 0.95454, 0.51479, 0.08519,
                         0.00576, ]))

    S3A_SLSTR_05 = (91, 1.56500, 1.66250,
                    np.array(
                        [0.00951, 0.02064, 0.04194, 0.08240, 0.16215, 0.30042,
                         0.49672, 0.70549, 0.85567, 0.93327, 0.96306, 0.97290,
                         0.97634, 0.98050, 0.98407, 0.98816, 0.99064, 0.99422,
                         0.99715, 0.99887, 0.99974, 0.99874, 0.99389, 0.98591,
                         0.97221, 0.95534, 0.93706, 0.92338, 0.92452, 0.93923,
                         0.95034, 0.89146, 0.68978, 0.41600, 0.21213, 0.10306,
                         0.05085, 0.02604, 0.01343, 0.00755, ]))

    S3A_SLSTR_06 = (92, 2.21750, 2.29750,
                    np.array(
                        [0.01257, 0.03348, 0.07603, 0.15844, 0.30026, 0.50042,
                         0.71448, 0.87789, 0.95880, 0.97991, 0.97956, 0.97917,
                         0.98339, 0.98979, 0.99621, 0.99969, 0.99941, 0.99572,
                         0.99031, 0.98586, 0.98433, 0.98171, 0.96554, 0.91002,
                         0.78868, 0.60765, 0.41211, 0.25045, 0.14086, 0.07530,
                         0.03827, 0.01809, 0.00721, ]))

    # Redefined MODIS TERRA and AQUA bands to a more accurate spectral
    # sampling actually measured from each satellite
    # and also added the ocean bands
    # I'm using the spectra found on
    # * TERRA: http://oceancolor.gsfc.nasa.gov/DOCS/RSR/HMODIST_RSRs.txt
    # * AQUA: http://oceancolor.gsfc.nasa.gov/DOCS/RSR/HMODISA_RSRs.txt
    # And the code that creates the bands is available here:
    # https://gist.github.com/jgomezdans/0cd6fc1537e5a76e5d3971ad167badd6
    ACCURATE_MODIS_TERRA_1 = (20, 0.61500, 0.68000,
                                      np.array(
                                          [0.02807, 0.14457, 0.37699, 0.63749,
                                           0.76990,
                                           0.78074, 0.78695, 0.79042, 0.79757,
                                           0.81003, 0.82398,
                                           0.83735, 0.85865, 0.89204, 0.91952,
                                           0.96325, 0.99380,
                                           0.99782, 0.95684, 0.83402, 0.72271,
                                           0.57700, 0.39879,
                                           0.22151, 0.09847, 0.04423, 0.01564,
                                           ]))

    ACCURATE_MODIS_TERRA_2 = (21, 0.82000, 0.89750,
                                      np.array(
                                          [0.01046, 0.02323, 0.04527, 0.08294,
                                           0.13231,
                                           0.20333, 0.29725, 0.45232, 0.64668,
                                           0.82575, 0.96775,
                                           0.99464, 0.98512, 0.97220, 0.96665,
                                           0.96893, 0.96614,
                                           0.98157, 0.98110, 0.96206, 0.91074,
                                           0.79448, 0.63300,
                                           0.40130, 0.20118, 0.10805, 0.05918,
                                           0.04157, 0.03105,
                                           0.02474, 0.01912, 0.01369,
                                           ]))

    ACCURATE_MODIS_TERRA_3 = (22, 0.45250, 0.48000,
                                      np.array(
                                          [0.02715, 0.19656, 0.66544, 0.98628,
                                           0.93702,
                                           0.85392, 0.87385, 0.92127, 0.85865,
                                           0.59029, 0.21396,
                                           0.02998,
                                           ]))

    ACCURATE_MODIS_TERRA_4 = (23, 0.54000, 0.56750,
                                      np.array(
                                          [0.04011, 0.24383, 0.65195, 0.94575,
                                           0.99606,
                                           0.99064, 0.98572, 0.97833, 0.93210,
                                           0.70581, 0.31567,
                                           0.05180,
                                           ]))

    ACCURATE_MODIS_TERRA_5 = (24, 1.21500, 1.27000,
                                       np.array(
                                           [0.01581, 0.03020, 0.05587, 0.10725,
                                            0.19486,
                                            0.33641, 0.50475, 0.70053, 0.84740,
                                            0.94598, 0.98540,
                                            0.99393, 0.97463, 0.88957, 0.76061,
                                            0.56817, 0.34327,
                                            0.20250, 0.10511, 0.05389, 0.03478,
                                            0.01910, 0.01206,
                                            ]))

    ACCURATE_MODIS_TERRA_6 = (25, 1.59750, 1.66000,
                                       np.array(
                                           [0.01510, 0.02596, 0.04589, 0.08177,
                                            0.14737,
                                            0.25125, 0.39260, 0.55170, 0.72028,
                                            0.88082, 0.98416,
                                            0.99335, 0.97750, 0.96290, 0.92923,
                                            0.86426, 0.74879,
                                            0.60357, 0.45702, 0.33580, 0.22364,
                                            0.11741, 0.06227,
                                            0.04009, 0.02624, 0.01373,
                                            ]))

    ACCURATE_MODIS_TERRA_7 = (26, 2.06000, 2.17500,
                                       np.array(
                                           [0.01090, 0.01563, 0.02103, 0.03128,
                                            0.04292,
                                            0.06585, 0.09211, 0.14750, 0.21146,
                                            0.31098, 0.42048,
                                            0.54758, 0.67979, 0.79133, 0.89692,
                                            0.94566, 0.97822,
                                            0.99110, 0.99837, 0.96789, 0.92654,
                                            0.89933, 0.87611,
                                            0.84122, 0.80300, 0.76182, 0.71977,
                                            0.68366, 0.64921,
                                            0.61062, 0.57081, 0.53225, 0.49405,
                                            0.44690, 0.39718,
                                            0.34026, 0.28131, 0.22290, 0.16465,
                                            0.12282, 0.08566,
                                            0.06288, 0.04430, 0.03179, 0.02100,
                                            0.01462, 0.01036,
                                            ]))

    ACCURATE_MODIS_TERRA_8 = (27, 0.40000, 0.42250,
                                      np.array(
                                          [0.01396, 0.13170, 0.59932, 0.67268,
                                           0.79013,
                                           0.48200, 0.89280, 0.82045, 0.24315,
                                           0.03272,
                                           ]))

    ACCURATE_MODIS_TERRA_9 = (28, 0.43500, 0.45000,
                                      np.array(
                                          [0.11465, 0.52805, 0.94626, 0.99552,
                                           0.91925,
                                           0.37027, 0.03831,
                                           ]))

    ACCURATE_MODIS_TERRA_10 = (29, 0.47750, 0.49500,
                                       np.array(
                                           [0.01416, 0.15956, 0.66826, 0.94806,
                                            0.99412,
                                            0.95485, 0.46874, 0.02232,
                                            ]))

    ACCURATE_MODIS_TERRA_11 = (30, 0.52000, 0.54000,
                                       np.array(
                                           [0.01480, 0.24145, 0.76250, 0.97399,
                                            0.99901,
                                            0.90343, 0.63643, 0.18019, 0.01435,
                                            ]))

    ACCURATE_MODIS_TERRA_12 = (31, 0.53750, 0.55500,
                                       np.array(
                                           [0.01470, 0.17968, 0.64797, 0.98082,
                                            0.99460,
                                            0.86391, 0.42490, 0.04215,
                                            ]))

    ACCURATE_MODIS_TERRA_13 = (32, 0.65750, 0.67500,
                                       np.array(
                                           [0.06338, 0.37724, 0.81862, 0.99604,
                                            0.88368,
                                            0.62629, 0.25126, 0.01371,
                                            ]))

    ACCURATE_MODIS_TERRA_14 = (33, 0.66750, 0.68750,
                                       np.array(
                                           [0.05618, 0.27689, 0.68142, 0.97341,
                                            0.98489,
                                            0.86696, 0.53632, 0.14128, 0.01578,
                                            ]))

    ACCURATE_MODIS_TERRA_15 = (34, 0.73750, 0.75500,
                                       np.array(
                                           [0.04077, 0.23050, 0.62169, 0.94550,
                                            0.97978,
                                            0.75860, 0.35374, 0.05302,
                                            ]))

    ACCURATE_MODIS_TERRA_16 = (35, 0.85250, 0.88000,
                                       np.array(
                                           [0.02770, 0.11693, 0.34382, 0.66672,
                                            0.91719,
                                            0.99906, 0.98123, 0.91350, 0.70598,
                                            0.41804, 0.10558,
                                            0.02056,
                                            ]))

    ACCURATE_MODIS_AQUA_1 = (36, 0.61500, 0.68000,
                                     np.array(
                                         [0.02807, 0.14457, 0.37699, 0.63749,
                                          0.76990,
                                          0.78074, 0.78695, 0.79042, 0.79757,
                                          0.81003, 0.82398,
                                          0.83735, 0.85865, 0.89204, 0.91952,
                                          0.96325, 0.99380,
                                          0.99782, 0.95684, 0.83402, 0.72271,
                                          0.57700, 0.39879,
                                          0.22151, 0.09847, 0.04423, 0.01564,
                                          ]))

    ACCURATE_MODIS_AQUA_2 = (37, 0.82000, 0.89750,
                                     np.array(
                                         [0.01046, 0.02323, 0.04527, 0.08294,
                                          0.13231,
                                          0.20333, 0.29725, 0.45232, 0.64668,
                                          0.82575, 0.96775,
                                          0.99464, 0.98512, 0.97220, 0.96665,
                                          0.96893, 0.96614,
                                          0.98157, 0.98110, 0.96206, 0.91074,
                                          0.79448, 0.63300,
                                          0.40130, 0.20118, 0.10805, 0.05918,
                                          0.04157, 0.03105,
                                          0.02474, 0.01912, 0.01369,
                                          ]))

    ACCURATE_MODIS_AQUA_3 = (38, 0.45250, 0.48000,
                                     np.array(
                                         [0.02715, 0.19656, 0.66544, 0.98628,
                                          0.93702,
                                          0.85392, 0.87385, 0.92127, 0.85865,
                                          0.59029, 0.21396,
                                          0.02998,
                                          ]))

    ACCURATE_MODIS_AQUA_4 = (39, 0.54000, 0.56750,
                                     np.array(
                                         [0.04011, 0.24383, 0.65195, 0.94575,
                                          0.99606,
                                          0.99064, 0.98572, 0.97833, 0.93210,
                                          0.70581, 0.31567,
                                          0.05180,
                                          ]))

    ACCURATE_MODIS_AQUA_5 = (40, 1.21500, 1.27000,
                                      np.array(
                                          [0.01581, 0.03020, 0.05587, 0.10725,
                                           0.19486,
                                           0.33641, 0.50475, 0.70053, 0.84740,
                                           0.94598, 0.98540,
                                           0.99393, 0.97463, 0.88957, 0.76061,
                                           0.56817, 0.34327,
                                           0.20250, 0.10511, 0.05389, 0.03478,
                                           0.01910, 0.01206,
                                           ]))

    ACCURATE_MODIS_AQUA_6 = (41, 1.59750, 1.66000,
                                      np.array(
                                          [0.01510, 0.02596, 0.04589, 0.08177,
                                           0.14737,
                                           0.25125, 0.39260, 0.55170, 0.72028,
                                           0.88082, 0.98416,
                                           0.99335, 0.97750, 0.96290, 0.92923,
                                           0.86426, 0.74879,
                                           0.60357, 0.45702, 0.33580, 0.22364,
                                           0.11741, 0.06227,
                                           0.04009, 0.02624, 0.01373,
                                           ]))

    ACCURATE_MODIS_AQUA_7 = (42, 2.06000, 2.17500,
                                      np.array(
                                          [0.01090, 0.01563, 0.02103, 0.03128,
                                           0.04292,
                                           0.06585, 0.09211, 0.14750, 0.21146,
                                           0.31098, 0.42048,
                                           0.54758, 0.67979, 0.79133, 0.89692,
                                           0.94566, 0.97822,
                                           0.99110, 0.99837, 0.96789, 0.92654,
                                           0.89933, 0.87611,
                                           0.84122, 0.80300, 0.76182, 0.71977,
                                           0.68366, 0.64921,
                                           0.61062, 0.57081, 0.53225, 0.49405,
                                           0.44690, 0.39718,
                                           0.34026, 0.28131, 0.22290, 0.16465,
                                           0.12282, 0.08566,
                                           0.06288, 0.04430, 0.03179, 0.02100,
                                           0.01462, 0.01036,
                                           ]))

    ACCURATE_MODIS_AQUA_8 = (43, 0.40250, 0.4225,
                                     np.array(
                                         [0.08730698,  0.5009796 ,  0.62325783,
                                          0.73623119,  0.52280358,  0.83059659,
                                          0.91752085,  0.32830619,  0.0332021
                                          ]))

    ACCURATE_MODIS_AQUA_9 = (44, 0.43250, 0.45000,
                                     np.array(
                                         [0.01902, 0.09698, 0.51329, 0.92505,
                                          0.99995,
                                          0.94153, 0.39117, 0.03671,
                                          ]))

    ACCURATE_MODIS_AQUA_10 = (45, 0.47750, 0.49500,
                                      np.array(
                                          [0.01964, 0.09723, 0.58592, 0.96018,
                                           0.98957,
                                           0.98879, 0.59678, 0.03983,
                                           ]))

    ACCURATE_MODIS_AQUA_11 = (46, 0.52000, 0.54000,
                                      np.array(
                                          [0.01215, 0.19041, 0.68607, 0.98495,
                                           0.99800,
                                           0.93087, 0.70338, 0.25414, 0.02052,
                                           ]))

    ACCURATE_MODIS_AQUA_12 = (47, 0.53750, 0.55500,
                                      np.array(
                                          [0.02111, 0.16232, 0.57148, 0.97131,
                                           0.99150,
                                           0.87360, 0.50867, 0.07715,
                                           ]))

    ACCURATE_MODIS_AQUA_13 = (48, 0.65750, 0.67500,
                                      np.array(
                                          [0.04175, 0.32634, 0.79150, 0.99988,
                                           0.89403,
                                           0.68671, 0.28090, 0.01440,
                                           ]))

    ACCURATE_MODIS_AQUA_14 = (49, 0.66750, 0.68750,
                                      np.array(
                                          [0.03422, 0.18745, 0.58587, 0.93366,
                                           0.99127,
                                           0.93329, 0.62790, 0.22864, 0.02571,
                                           ]))

    ACCURATE_MODIS_AQUA_15 = (50, 0.73500, 0.75500,
                                      np.array(
                                          [0.01266, 0.02854, 0.19037, 0.58331,
                                           0.92908,
                                           0.99234, 0.77888, 0.37822, 0.05551,
                                           ]))

    ACCURATE_MODIS_AQUA_16 = (51, 0.85250, 0.88000,
                                      np.array(
                                          [0.01859, 0.08363, 0.28236, 0.59771,
                                           0.87955,
                                           0.99942, 0.97670, 0.94504, 0.75962,
                                           0.47919, 0.14822,
                                           0.02755,
                                           ]))
    # All of the original predefined wavelengths from 6S
    # CONSTANT_NAME = (ID for Constant, Start Wavelength, End Wavelength)
    METEOSAT_VISIBLE = (-2, 0.35, 1.11)
    GOES_EAST_VISIBLE = (-3, 0.49, 0.9)
    GOES_WEST_VISIBLE = (-4, 0.49, 0.9)
    AVHRR_NOAA6_B1 = (-5, 0.55, 0.75)
    AVHRR_NOAA6_B2 = (-6, 0.69, 1.12)
    AVHRR_NOAA7_B1 = (-7, 0.5, 0.8)
    AVHRR_NOAA7_B2 = (-8, 0.64, 1.17)
    AVHRR_NOAA8_B1 = (-9, 0.54, 1.01)
    AVHRR_NOAA8_B2 = (-10, 0.68, 1.12)
    AVHRR_NOAA9_B1 = (-11, 0.53, 0.81)
    AVHRR_NOAA9_B2 = (-12, 0.68, 1.17)
    AVHRR_NOAA10_B1 = (-13, 0.53, 0.78)
    AVHRR_NOAA10_B2 = (-14, 0.6, 1.19)
    AVHRR_NOAA11_B1 = (-15, 0.54, 0.82)
    AVHRR_NOAA11_B2 = (-16, 0.6, 1.12)
    SPOT_HRV1_B1 = (-17, 0.47, 0.65)
    SPOT_HRV1_B2 = (-18, 0.6, 0.72)
    SPOT_HRV1_B3 = (-19, 0.73, 0.93)
    SPOT_HRV1_PAN = (-20, 0.47, 0.79)
    SPOT_HRV2_B1 = (-21, 0.47, 0.65)
    SPOT_HRV2_B2 = (-22, 0.59, 0.73)
    SPOT_HRV2_B3 = (-23, 0.74, 0.94)
    SPOT_HRV2_PAN = (-24, 0.47, 0.79)
    LANDSAT_TM_B1 = (-25, 0.43, 0.56)
    LANDSAT_TM_B2 = (-26, 0.5, 0.65)
    LANDSAT_TM_B3 = (-27, 0.58, 0.74)
    LANDSAT_TM_B4 = (-28, 0.73, 0.95)
    LANDSAT_TM_B5 = (-29, 1.5025, 1.89)
    LANDSAT_TM_B7 = (-30, 1.95, 2.41)
    LANDSAT_MSS_B1 = (-31, 0.475, 0.64)
    LANDSAT_MSS_B2 = (-32, 0.58, 0.75)
    LANDSAT_MSS_B3 = (-33, 0.655, 0.855)
    LANDSAT_MSS_B4 = (-34, 0.785, 1.1)
    ER2_MAS_B1 = (-35, 0.5025, 0.5875)
    ER2_MAS_B2 = (-36, 0.6075, 0.7)
    ER2_MAS_B3 = (-37, 0.83, 0.9125)
    ER2_MAS_B4 = (-38, 0.9, 0.9975)
    ER2_MAS_B5 = (-39, 1.82, 1.9575)
    ER2_MAS_B6 = (-40, 2.0950, 2.1925)
    ER2_MAS_B7 = (-41, 3.58, 3.87)
    MODIS_B1 = (-42, 0.61, 0.685)
    MODIS_B2 = (-43, 0.82, 0.9025)
    MODIS_B3 = (-44, 0.45, 0.4825)
    MODIS_B4 = (-45, 0.54, 0.57)
    MODIS_B5 = (-46, 1.2150, 1.27)
    MODIS_B6 = (-47, 1.6, 1.665)
    MODIS_B7 = (-48, 2.0575, 2.1825)
    MODIS_B8 = (-49, 0.4025, 0.4225)
    AVHRR_NOAA12_B1 = (-50, 0.5, 1.0)
    AVHRR_NOAA12_B2 = (-51, 0.65, 1.12)
    AVHRR_NOAA14_B1 = (-52, 0.5, 1.11)
    AVHRR_NOAA14_B2 = (-53, 0.68, 1.1)
    POLDER_B1 = (-54, 0.4125, 0.4775)
    POLDER_B2 = (-55, 0.41, 0.5225)
    POLDER_B3 = (-56, 0.5325, 0.595)
    POLDER_B4 = (-57, 0.63, 0.7025)
    POLDER_B5 = (-58, 0.745, 0.78)
    POLDER_B6 = (-59, 0.7, 0.83)
    POLDER_B7 = (-60, 0.81, 0.92)
    POLDER_B8 = (-61, 0.8650, 0.94)
    SEAWIFS_B1 = (-62, 0.3825, 0.7)
    SEAWIFS_B2 = (-63, 0.38, 0.58)
    SEAWIFS_B3 = (-64, 0.38, 1.02)
    SEAWIFS_B4 = (-65, 0.38, 1.02)
    SEAWIFS_B5 = (-66, 0.3825, 1.15)
    SEAWIFS_B6 = (-67, 0.3825, 1.05)
    SEAWIFS_B7 = (-68, 0.38, 1.15)
    SEAWIFS_B8 = (-69, 0.38, 1.15)
    AATSR_B1 = (-70, 0.525, 0.5925)
    AATSR_B2 = (-71, 0.6275, 0.6975)
    AATSR_B3 = (-72, 0.8325, 0.9025)
    AATSR_B4 = (-73, 1.4475, 1.7775)
    MERIS_B1 = (-74, 0.412, 0.412 + 0.00998)
    MERIS_B2 = (-75, 0.442, 0.442 + 0.00997)
    MERIS_B3 = (-76, 0.489, 0.489 + 0.00997)
    MERIS_B4 = (-77, 0.509, 0.509 + 0.00997)
    MERIS_B5 = (-78, 0.559, 0.559 + 0.00997)
    MERIS_B6 = (-79, 0.619, 0.619 + 0.00997)
    MERIS_B7 = (-80, 0.664, 0.664 + 0.00998)
    MERIS_B8 = (-81, 0.681, 0.681 + 0.00749)
    MERIS_B9 = (-82, 0.708, 0.708 + 0.00999)
    MERIS_B10 = (-83, 0.753, 0.753 + 0.00749)
    MERIS_B11 = (-84, 0.760, 0.760 + 0.00374)
    MERIS_B12 = (-85, 0.778, 0.778 + 0.00150)
    MERIS_B13 = (-86, 0.865, 0.865 + 0.002)
    MERIS_B14 = (-87, 0.885, 0.885 + 0.001)
    MERIS_B15 = (-88, 0.9, 0.9 + 0.001)
    GLI_B1 = (-89, 0.37, 0.3925)
    GLI_B2 = (-90, 0.3875, 0.4125)
    GLI_B3 = (-91, 0.3975, 0.4275)
    GLI_B4 = (-92, 0.4475, 0.505)
    GLI_B5 = (-93, 0.4475, 0.46)
    GLI_B6 = (-94, 0.475, 0.505)
    GLI_B7 = (-95, 0.5075, 0.5325)
    GLI_B8 = (-96, 0.5265, 0.56)
    GLI_B9 = (-97, 0.5475, 0.5825)
    GLI_B10 = (-98, 0.61, 0.64)
    GLI_B11 = (-99, 0.6525, 0.6825)
    GLI_B12 = (-100, 0.665, 0.695)
    GLI_B13 = (-101, 0.6625, 0.6975)
    GLI_B14 = (-102, 0.6925, 0.7275)
    GLI_B15 = (-103, 0.6925, 0.7275)
    GLI_B16 = (-104, 0.7325, 0.7675)
    GLI_B17 = (-105, 0.75, 0.775)
    GLI_B18 = (-106, 0.84, 0.8925)
    GLI_B19 = (-107, 0.85, 0.88)
    GLI_B20 = (-108, 0.415, 0.5075)
    GLI_B21 = (-109, 0.505, 0.58)
    GLI_B22 = (-110, 0.6075, 0.715)
    GLI_B23 = (-111, 0.745, 0.9075)
    GLI_B24 = (-112, 1.03, 1.07)
    GLI_B25 = (-113, 1.085, 1.19)
    GLI_B26 = (-114, 1.22, 1.2625)
    GLI_B27 = (-115, 1.3475, 1.415)
    GLI_B28 = (-116, 1.515, 1.77)
    GLI_B29 = (-117, 2.055, 2.345)
    GLI_B30 = (-118, 3.22, 4.0)
    ALI_B1P = (-119, 0.4225, 0.4625)
    ALI_B1 = (-120, 0.4325, 0.550)
    ALI_B2 = (-121, 0.5, 0.63)
    ALI_B3 = (-122, 0.5775, 0.750)
    ALI_B4 = (-123, 0.7525, 0.8375)
    ALI_B4P = (-124, 0.8025, 0.935)
    ALI_B5P = (-125, 1.130, 1.345)
    ALI_B5 = (-126, 1.47, 1.820)
    ALI_B7 = (-127, 1.98, 2.53)
    ASTER_B1 = (-128, 0.485, 0.6425)
    ASTER_B2 = (-129, 0.590, 0.730)
    ASTER_B3N = (-130, 0.720, 0.9075)
    ASTER_B3B = (-131, 0.720, 0.9225)
    ASTER_B4 = (-132, 1.57, 1.7675)
    ASTER_B5 = (-133, 2.120, 2.2825)
    ASTER_B6 = (-134, 2.150, 2.295)
    ASTER_B7 = (-135, 2.210, 2.39)
    ASTER_B8 = (-136, 2.250, 2.244)
    ASTER_B9 = (-137, 2.2975, 2.4875)
    LANDSAT_ETM_B1 = (-138, 0.435, 0.52)
    LANDSAT_ETM_B2 = (-139, 0.5, 0.6225)
    LANDSAT_ETM_B3 = (-140, 0.615, 0.7025)
    LANDSAT_ETM_B4 = (-141, 0.74, 0.9125)
    LANDSAT_ETM_B5 = (-142, 1.51, 1.7875)
    LANDSAT_ETM_B7 = (-143, 2.015, 2.3775)
    HYPBLUE_B1 = (-144, 0.4375, 0.5)
    HYPBLUE_B2 = (-145, 0.435, 0.52)
    SPOT_VGT_B1 = (-146, 0.4175, 0.5)
    SPOT_VGT_B2 = (-147, 0.5975, 0.7675)
    SPOT_VGT_B3 = (-148, 0.7325, 0.9575)
    SPOT_VGT_B4 = (-149, 1.5225, 1.8)
    VIIRS_BM1 = (-150, 0.4025, 0.4225)
    VIIRS_BM2 = (-151, 0.4350, 0.4550)
    VIIRS_BM3 = (-152, 0.4775, 0.4975)
    VIIRS_BM4 = (-153, 0.5450, 0.565)
    VIIRS_BM5 = (-154, 0.6625, 0.6825)
    VIIRS_BM6 = (-155, 0.7375, 0.7525)
    VIIRS_BM7 = (-156, 0.8450, 0.8850)
    VIIRS_BM8 = (-157, 1.23, 1.25)
    VIIRS_BM9 = (-158, 1.37, 1.385)
    VIIRS_BM10 = (-159, 1.58, 1.64)
    VIIRS_BM11 = (-160, 2.225, 2.275)
    VIIRS_BM12 = (-161, 3.61, 3.79)
    VIIRS_BI1 = (-162, 0.6, 0.68)
    VIIRS_BI2 = (-163, 0.845, 0.885)
    VIIRS_BI3 = (-164, 1.58, 1.64)
    VIIRS_BI4 = (-165, 3.55, 3.93)
