import unittest
import os
import numpy as np
from phonopy import Phonopy
from phonopy.interface.vasp import read_vasp
from phonopy.file_IO import parse_FORCE_SETS, parse_BORN
from phonopy.phonon.random_displacements import RandomDisplacements

data_dir = os.path.dirname(os.path.abspath(__file__))

disp_str = """ 0.08604722 -0.12136884  0.2259561
-0.13041486  0.15097802  0.02631405
-0.07863048 -0.02481802  0.25070963
 0.18408689  0.07694356 -0.16558913
 0.33706965  0.00628077  0.20969844
 0.21434168  0.02912861 -0.04021545
 0.0249472   0.32642679 -0.09319597
 0.01595788  0.19935434 -0.52105534
 0.00885736  0.03454223  0.28437986
 0.33836971  0.12079372 -0.16240665
-0.10473928  0.2087231  -0.14207728
-0.08785425  0.17651202 -0.51115053
 0.35337593  0.28786038  0.00841824
-0.21919447  0.24888609 -0.42068416
 0.01601971  0.02961431 -0.14149282
-0.40903733 -0.20640984 -0.10534949
 0.12547414  0.03701679  0.222098
 0.43732633  0.05404798 -0.21346988
 0.04803753  0.2188926  -0.14176129
 0.13245449  0.15793342 -0.40829428
 0.26091131  0.09662185  0.10248825
-0.3035006   0.14364851 -0.32205807
-0.03180122  0.18306307 -0.36144391
-0.42215871  0.05674984 -0.16827307
-0.06753283  0.07419713  0.27199926
 0.56660137  0.09149754 -0.24834841
-0.07235513  0.06700648 -0.13895612
-0.14135257  0.18357592 -0.35672661
 0.23447205  0.24139625 -0.04767531
 0.02104433  0.15424369 -0.37353626
-0.01008879  0.1111799  -0.41855204
-0.42279591  0.00705917 -0.10248661
-0.40652195  0.21784334 -0.12915309
 0.21742623 -0.03333486  0.09109975
-0.17291133  0.17695929  0.03105908
 0.49976052 -0.08295726  0.03622586
 0.27168622  0.22665456 -0.07076577
 0.20052739  0.04233814  0.11868699
 0.30357113 -0.21161182  0.06214248
-0.01617524 -0.14392534  0.00242148
-0.24864539  0.00309787 -0.04601912
 0.0032162  -0.03904985 -0.06521829
-0.43430288  0.25621613 -0.11019557
-0.04498765  0.00251074  0.17407847
-0.1378953   0.06588659 -0.13111823
 0.18526224 -0.02556599  0.03865567
 0.08806443  0.20607792 -0.0402066
 0.4637113  -0.03705598  0.0784063
-0.04777629 -0.05883407 -0.03277642
-0.39592089  0.13746402 -0.08217123
-0.12889752  0.24904472 -0.14989202
-0.29222888  0.16252002  0.17399822
-0.23542401  0.18728961 -0.12993575
 0.23570141  0.16881034  0.02496769
-0.10183241  0.05345851 -0.08433899
 0.49015386 -0.15558096  0.06616878
-0.28644251 -0.14249684  0.02677136
-0.12587733  0.02733008 -0.12848019
-0.30201549  0.26556372 -0.16348007
 0.00981878  0.16449613  0.10119801
 0.03282114  0.01737568 -0.11360569
 0.17547094  0.18202549  0.01324873
 0.13272966  0.096398   -0.10436066
 0.40076674 -0.10197913  0.05464131"""

randn_ii_str = """-1.7497654731  0.3426804033  1.1530358026 -0.2524360365  0.9813207870  0.5142188414
 0.2211796692 -1.0700433306 -0.1894958308  0.2550014443 -0.4580269855  0.4351634881
-0.5835950503  0.8168470717  0.6727208057 -0.1044111434 -0.5312803769  1.0297326851
-0.4381356227 -1.1183182463  1.6189816607  1.5416051745 -0.2518791392 -0.8424357383
 0.1845186906  0.9370822011  0.7310003438  1.3615561251 -0.3262380592  0.0556760149
 0.2223996086 -1.4432169952 -0.7563523056  0.8164540110  0.7504447615 -0.4559469275
 1.1896222680 -1.6906168264 -1.3563990489 -1.2324345139 -0.5444391617 -0.6681717368
 0.0073145632 -0.6129387355  1.2997480748 -1.7330956237 -0.9833100991  0.3575077532"""

randn_ij_str_1 = """-1.6135785028  1.4707138666 -1.1880175973 -0.5497461935 -0.9400461615 -0.8279323644
-0.8817983895  0.0186389495  0.2378446219  0.0135485486 -1.6355293994 -1.0442098777
-0.3317771351 -0.6892179781  2.0346075615 -0.5507144119  0.7504533303 -1.3069923391
 0.7788223993  0.4282328706  0.1088719899  0.0282836348 -0.5788258248 -1.1994511992
-0.0760234657  0.0039575940 -0.1850141109 -2.4871515352 -1.7046512058 -1.1362610068
 0.3173679759 -0.7524141777 -1.2963918072  0.0951394436 -0.4237150999 -1.1859835649
-1.5406160246  2.0467139685 -1.3969993450 -1.0971719846 -0.2387128693 -1.4290668984
 1.2962625864  0.9522756261 -1.2172541306 -0.1572651674 -1.5075851603  0.1078841308
 2.0747931679 -0.3432976822 -0.6166293717  0.7631836461  0.1929171918 -0.3484589307
 1.7036239881 -0.7221507701  1.0936866497 -0.2295177532 -0.0088986633 -0.5431980084
-2.0151887171 -0.0795405869  0.3010494638 -1.6848999617  0.2223908094 -0.6849217352
-0.5144298914 -0.2160601200  0.4223802204 -1.0940429310  1.2369078852 -0.2302846784"""

randn_ij_str_2 = """ 0.1088634678  0.5078095905 -0.8622273465  1.2494697427 -0.0796112459 -0.8897314813
 0.6130388817  0.7362052133  1.0269214394 -1.4321906111 -1.8411883002  0.3660932262
 0.5805733358 -1.1045230927  0.6901214702  0.6868900661 -1.5666875296  0.9049741215
-1.7059520057  0.3691639571  1.8765734270 -0.3769033502  1.8319360818  0.0030174340
-2.9733154741  0.0333172781 -0.2488886671 -0.4501764350  0.1324278011  0.0222139280
-0.3654619927 -1.2710230408  1.5861709384  0.6933906585 -1.9580812342 -0.1348013120
 0.9490047765 -0.0193975860  0.8945977058  0.7596931199 -1.4977203811 -1.1938859768
 0.7470556551  0.4296764359 -1.4150429209 -0.6407599230  0.7796263037 -0.4381209163
 2.2986539407 -0.1652095526  0.4662993684  0.2699872386 -0.3198310471 -1.1477415999
 0.7530621877 -1.6094388962  1.9432622634 -1.4474361123  0.1302484554  0.9493608647
-0.1262011837  1.9902736498  0.5229978045 -0.0163454028 -0.4158163358 -1.3585029368
-0.7044181997 -0.5913751211  0.7369951690  0.4358672525  1.7759935855  0.5130743788"""


class TestRandomDisplacements(unittest.TestCase):
    def setUp(self):
        pass

    def tearDown(self):
        pass

    def _get_phonon_NaCl(self):
        cell = read_vasp(os.path.join(data_dir, "..", "POSCAR_NaCl"))
        phonon = Phonopy(cell,
                         np.diag([2, 2, 2]),
                         primitive_matrix=[[0, 0.5, 0.5],
                                           [0.5, 0, 0.5],
                                           [0.5, 0.5, 0]])
        filename = os.path.join(data_dir, "..", "FORCE_SETS_NaCl")
        force_sets = parse_FORCE_SETS(filename=filename)
        phonon.set_displacement_dataset(force_sets)
        phonon.produce_force_constants()
        phonon.symmetrize_force_constants()
        filename_born = os.path.join(data_dir, "..", "BORN_NaCl")
        nac_params = parse_BORN(phonon.get_primitive(), filename=filename_born)
        phonon.set_nac_params(nac_params)
        return phonon

    def test_NaCl(self):
        """Test by fixed random numbers of np.random.normal

        randn_ii and randn_ij were created by

            np.random.seed(seed=100)
            randn_ii = np.random.normal(size=(N_ii, 1, num_band))
            randn_ij = np.random.normal(size=(N_ij, 2, 1, num_band)).

        numpy v1.16.4 (py37h6b0580a_0) on macOS installed from conda-forge
        was used.

        """

        phonon = self._get_phonon_NaCl()
        rd = RandomDisplacements(phonon.supercell,
                                 phonon.primitive,
                                 phonon.force_constants,
                                 cutoff_frequency=0.01)
        num_band = len(phonon.primitive) * 3
        N = int(np.rint(phonon.supercell.volume / phonon.primitive.volume))
        N_ij = N - len(rd.qpoints)
        N_ii = N - N_ij * 2
        shape_ii = (N_ii, 1, num_band)
        randn_ii = np.fromstring(randn_ii_str.replace('\n', ' '),
                                 dtype=float, sep=' ').reshape(shape_ii)
        shape_ij = (N_ij, 2, 1, num_band)
        randn_ij = np.zeros(shape_ij, dtype=float)
        randn_ij[:, 0, 0, :] = np.fromstring(
            randn_ij_str_1.replace('\n', ' '),
            dtype=float, sep=' ').reshape(N_ij, num_band)
        randn_ij[:, 1, 0, :] = np.fromstring(
            randn_ij_str_2.replace('\n', ' '),
            dtype=float, sep=' ').reshape(N_ij, num_band)

        rd.run(500, randn=(randn_ii, randn_ij))

        data = np.fromstring(disp_str.replace('\n', ' '), dtype=float, sep=' ')
        np.testing.assert_allclose(data, rd.u.ravel(), atol=1e-5)


if __name__ == '__main__':
    suite = unittest.TestLoader().loadTestsFromTestCase(
        TestRandomDisplacements)
    unittest.TextTestRunner(verbosity=2).run(suite)
