# -*- coding: iso-8859-15 -*-
# This program is public domain
# Author: Paul Kienzle
u"""

Adds average mass for the elements:

*    mass, mass_units (u)
         The molar mass averaged over natural isotope abundance.

Adds mass and abundance information for isotopes:

*    mass, mass_units (u)
         The molar mass of the individual isotope.

*    abundance, abundance_units (%)
         Natural abundance for the isotope.

Atomic Weights and Isotopic Composition [#Coursey]_.

The atomic weights are available for elements 1 through 112, 114, & 116 and
isotopic compositions or abundances are given when appropriate. The atomic
weights data were published by Coplen [#Coplen]_ in Atomic Weights of the
Elements 1999, (and include changes reported from the 2001 review in
Chem. Int., 23, 179 (2001)) and the isotopic compositions data were
published by Rosman [#Rosman]_ and Taylor [#Taylor]_ in Isotopic Compositions
of the Elements 1997.  The relative atomic masses of the isotopes data were
published by Audi [#Audi]_ and Wapstra [#Wapstra]_ in the 1995 Update To The
Atomic Mass Evaluation.

This data has been compiled from the above sources for the user's convenience
and does not represent a critical evaluation by the NIST Physics Laboratory.
http://physics.nist.gov/PhysRefData/Compositions/

Neutron mass from NIST Reference on Constants, Units, and Uncertainty
http://physics.nist.gov/cuu/index.html

.. [#Coursey] Coursey. J. S., Schwab. D. J., and Dragoset. R. A., NIST, Physics Laboratory,
       Office of Electronic Commerce in Scientific and Engineering Data.
.. [#Coplen] Coplen. T. B. : U.S. Geological Survey, Reston, Virginia, USA.
.. [#Rosman] Rosman. K. J. R. : Department of Applied Physics, Curtin University of
       Technology, Australia.
.. [#Taylor] Taylor. P. D. P. : Institute for Reference Materials and Measurements,
       European Commission, Belgium.
.. [#Audi] Audi. G. : Centre de Spectromtrie Nuclaire et de Spectromtrie de Masse,
       Orsay Campus, France.
.. [#Wapstra] Wapstra. A. H. : National Institute of Nuclear Physics and High-Energy Physics,
       Amsterdam, The Netherlands.
"""

from .core import Element, Isotope
from .constants import neutron_mass

#__all__ = ['init']

def getval(str):
    idx = str.find('(')
    if idx > 0: # value(uncertainty)
        return float(str[:idx])
    if str.startswith('['): # [nominal]
        return int(str[1:-1])
    if str == "": # missing
        return 0
    return float(str)

def mass(isotope):
    """
    Atomic weight.

    :Parameters:
        *isotope* : Isotope

    :Returns:
        *mass* : float | u
            Atomic weight of the element.

    Reference:
        *Coursey. J. S., Schwab. D. J, and Dragoset. R. A., NIST Atomic Weights and Isotopic Composition
        Database.*
    """
    return isotope._mass

def abundance(isotope):
    """
    Natural abundance.

    :Parameters:
        *isotope* : Isotope

    :Returns:
        *abundance* : float | %

    Reference:
        *Coursey. J. S., Schwab. D. J, and Dragoset. R. A., NIST Atomic
        Weights and Isotopic Composition Database.*
    """
    return isotope._abundance

def init(table, reload=False):
    if 'mass' in table.properties and not reload: return
    table.properties.append('mass')
    Element.mass = property(mass,doc=mass.__doc__)
    Isotope.mass = property(mass,doc=mass.__doc__)
    Isotope.abundance = property(abundance,doc=abundance.__doc__)
    Element.mass_units = "u"
    Element.abundance_units = "%"

    for line in massdata.split('\n'):
        isotope,m,p,avg = line.split(',')
        el,sym,iso = isotope.split('-')
        el = table[int(el)]
        assert el.symbol == sym, \
            "Symbol %s does not match %s"%(sym,el.symbol)
        iso = el.add_isotope(int(iso))
        el._mass = getval(avg)
        iso._mass = getval(m)
        iso._abundance = getval(p)

    # A single neutron is an isotope of element 0
    el = table[0]
    el._mass = neutron_mass
    iso = el.add_isotope(1)
    iso._mass = neutron_mass
    iso._abundance = 100

massdata="""\
1-H-1,1.0078250321(4),99.9885(70),1.00794(7)
1-H-2,2.0141017780(4),0.0115(70),1.00794(7)
1-H-3,3.0160492675(11),,1.00794(7)
1-H-4,4.02783(12),,1.00794(7)
1-H-5,5.03954(102),,1.00794(7)
1-H-6,6.04494(28),,1.00794(7)
2-He-3,3.0160293097(9),0.000137(3),4.002602(2)
2-He-4,4.0026032497(10),99.999863(3),4.002602(2)
2-He-5,5.012220(50),,4.002602(2)
2-He-6,6.0188881(11),,4.002602(2)
2-He-7,7.028030(30),,4.002602(2)
2-He-8,8.033922(8),,4.002602(2)
2-He-9,9.043820(70),,4.002602(2)
2-He-10,10.052400(80),,4.002602(2)
3-Li-4,4.02718(23),,6.941(2)
3-Li-5,5.012540(50),,6.941(2)
3-Li-6,6.0151223(5),7.59(4),6.941(2)
3-Li-7,7.0160040(5),92.41(4),6.941(2)
3-Li-8,8.0224867(5),,6.941(2)
3-Li-9,9.0267891(21),,6.941(2)
3-Li-10,10.035481(16),,6.941(2)
3-Li-11,11.043796(29),,6.941(2)
3-Li-12,12.05378(107)#,,6.941(2)
4-Be-5,5.04079(429)#,,9.012182(3)
4-Be-6,6.019726(6),,9.012182(3)
4-Be-7,7.0169292(5),,9.012182(3)
4-Be-8,8.00530509(4),,9.012182(3)
4-Be-9,9.0121821(4),100,9.012182(3)
4-Be-10,10.0135337(4),,9.012182(3)
4-Be-11,11.021658(7),,9.012182(3)
4-Be-12,12.026921(16),,9.012182(3)
4-Be-13,13.03613(54),,9.012182(3)
4-Be-14,14.04282(12),,9.012182(3)
5-B-7,7.029920(80),,10.811(7)
5-B-8,8.0246067(12),,10.811(7)
5-B-9,9.0133288(11),,10.811(7)
5-B-10,10.0129370(4),19.9(7),10.811(7)
5-B-11,11.0093055(5),80.1(7),10.811(7)
5-B-12,12.0143521(15),,10.811(7)
5-B-13,13.0177803(12),,10.811(7)
5-B-14,14.025404(23),,10.811(7)
5-B-15,15.031097(24),,10.811(7)
5-B-16,16.039810(60),,10.811(7)
5-B-17,17.04693(15),,10.811(7)
5-B-18,18.05617(86)#,,10.811(7)
5-B-19,19.06373(43)#,,10.811(7)
6-C-8,8.037675(25),,12.0107(8)
6-C-9,9.0310401(23),,12.0107(8)
6-C-10,10.0168531(4),,12.0107(8)
6-C-11,11.0114338(10),,12.0107(8)
6-C-12,12.0000000(0),98.93(8),12.0107(8)
6-C-13,13.0033548378(10),1.07(8),12.0107(8)
6-C-14,14.003241988(4),,12.0107(8)
6-C-15,15.0105993(9),,12.0107(8)
6-C-16,16.014701(4),,12.0107(8)
6-C-17,17.022584(19),,12.0107(8)
6-C-18,18.026760(30),,12.0107(8)
6-C-19,19.03525(12),,12.0107(8)
6-C-20,20.04032(22),,12.0107(8)
6-C-21,21.04934(54)#,,12.0107(8)
6-C-22,22.05645(97)#,,12.0107(8)
7-N-10,10.04262(43)#,,14.0067(2)
7-N-11,11.02680(19),,14.0067(2)
7-N-12,12.0186132(11),,14.0067(2)
7-N-13,13.00573858(29),,14.0067(2)
7-N-14,14.0030740052(9),99.632(7),14.0067(2)
7-N-15,15.0001088984(9),0.368(7),14.0067(2)
7-N-16,16.0061014(28),,14.0067(2)
7-N-17,17.008450(16),,14.0067(2)
7-N-18,18.014082(21),,14.0067(2)
7-N-19,19.017027(18),,14.0067(2)
7-N-20,20.023370(60),,14.0067(2)
7-N-21,21.02709(10),,14.0067(2)
7-N-22,22.03444(21),,14.0067(2)
7-N-23,23.04051(76)#,,14.0067(2)
7-N-24,24.05050(54)#,,14.0067(2)
8-O-12,12.034405(20),,15.9994(3)
8-O-13,13.024810(10),,15.9994(3)
8-O-14,14.00859529(8),,15.9994(3)
8-O-15,15.0030654(5),,15.9994(3)
8-O-16,15.9949146221(15),99.757(16),15.9994(3)
8-O-17,16.99913150(22),0.038(1),15.9994(3)
8-O-18,17.9991604(9),0.205(14),15.9994(3)
8-O-19,19.003579(3),,15.9994(3)
8-O-20,20.0040762(13),,15.9994(3)
8-O-21,21.008655(13),,15.9994(3)
8-O-22,22.009970(60),,15.9994(3)
8-O-23,23.01569(11),,15.9994(3)
8-O-24,24.02037(33),,15.9994(3)
8-O-25,25.02914(40)#,,15.9994(3)
8-O-26,26.03775(46)#,,15.9994(3)
9-F-14,14.03608(43)#,,18.9984032(5)
9-F-15,15.01801(14),,18.9984032(5)
9-F-16,16.011466(9),,18.9984032(5)
9-F-17,17.00209524(27),,18.9984032(5)
9-F-18,18.0009377(6),,18.9984032(5)
9-F-19,18.99840320(7),100,18.9984032(5)
9-F-20,19.99998132(9),,18.9984032(5)
9-F-21,20.9999489(19),,18.9984032(5)
9-F-22,22.002999(13),,18.9984032(5)
9-F-23,23.003570(90),,18.9984032(5)
9-F-24,24.008100(70),,18.9984032(5)
9-F-25,25.012090(80),,18.9984032(5)
9-F-26,26.01963(13),,18.9984032(5)
9-F-27,27.02689(45),,18.9984032(5)
9-F-28,28.03567(55)#,,18.9984032(5)
9-F-29,29.04326(62)#,,18.9984032(5)
10-Ne-16,16.025757(22),,20.1797(6)
10-Ne-17,17.017700(50),,20.1797(6)
10-Ne-18,18.0056971(16),,20.1797(6)
10-Ne-19,19.0018798(6),,20.1797(6)
10-Ne-20,19.9924401759(20),90.48(3),20.1797(6)
10-Ne-21,20.99384674(4),0.27(1),20.1797(6)
10-Ne-22,21.99138551(23),9.25(3),20.1797(6)
10-Ne-23,22.99446734(26),,20.1797(6)
10-Ne-24,23.993615(11),,20.1797(6)
10-Ne-25,24.997790(50),,20.1797(6)
10-Ne-26,26.000460(60),,20.1797(6)
10-Ne-27,27.00762(10),,20.1797(6)
10-Ne-28,28.01211(12),,20.1797(6)
10-Ne-29,29.01935(32),,20.1797(6)
10-Ne-30,30.02387(88),,20.1797(6)
10-Ne-31,31.03311(97)#,,20.1797(6)
10-Ne-32,32.03991(94)#,,20.1797(6)
11-Na-18,18.02718(43)#,,22.989770(2)
11-Na-19,19.013879(13),,22.989770(2)
11-Na-20,20.007348(7),,22.989770(2)
11-Na-21,20.9976551(8),,22.989770(2)
11-Na-22,21.9944368(5),,22.989770(2)
11-Na-23,22.98976967(23),100,22.989770(2)
11-Na-24,23.99096333(23),,22.989770(2)
11-Na-25,24.9899544(13),,22.989770(2)
11-Na-26,25.992590(15),,22.989770(2)
11-Na-27,26.994010(40),,22.989770(2)
11-Na-28,27.998890(80),,22.989770(2)
11-Na-29,29.00281(10),,22.989770(2)
11-Na-30,30.00923(10),,22.989770(2)
11-Na-31,31.01360(18),,22.989770(2)
11-Na-32,32.01965(52),,22.989770(2)
11-Na-33,33.02739(160),,22.989770(2)
11-Na-34,34.03490(113)#,,22.989770(2)
11-Na-35,35.04418(166)#,,22.989770(2)
12-Mg-20,20.018863(29),,24.3050(6)
12-Mg-21,21.011714(18),,24.3050(6)
12-Mg-22,21.9995741(15),,24.3050(6)
12-Mg-23,22.9941249(13),,24.3050(6)
12-Mg-24,23.98504190(20),78.99(4),24.3050(6)
12-Mg-25,24.98583702(20),10.00(1),24.3050(6)
12-Mg-26,25.98259304(21),11.01(3),24.3050(6)
12-Mg-27,26.98434074(21),,24.3050(6)
12-Mg-28,27.9838767(22),,24.3050(6)
12-Mg-29,28.988550(30),,24.3050(6)
12-Mg-30,29.990460(70),,24.3050(6)
12-Mg-31,30.996550(80),,24.3050(6)
12-Mg-32,31.99915(10),,24.3050(6)
12-Mg-33,33.00559(16),,24.3050(6)
12-Mg-34,34.00907(28),,24.3050(6)
12-Mg-35,35.01749(47)#,,24.3050(6)
12-Mg-36,36.02245(97)#,,24.3050(6)
12-Mg-37,37.03124(97)#,,24.3050(6)
13-Al-21,21.02804(32)#,,26.981538(2)
13-Al-22,22.01952(10)#,,26.981538(2)
13-Al-23,23.007265(27),,26.981538(2)
13-Al-24,23.999941(4),,26.981538(2)
13-Al-25,24.9904286(7),,26.981538(2)
13-Al-26,25.98689166(21),,26.981538(2)
13-Al-27,26.98153844(14),100,26.981538(2)
13-Al-28,27.98191018(15),,26.981538(2)
13-Al-29,28.9804448(13),,26.981538(2)
13-Al-30,29.982960(15),,26.981538(2)
13-Al-31,30.983946(22),,26.981538(2)
13-Al-32,31.988120(90),,26.981538(2)
13-Al-33,32.990870(70),,26.981538(2)
13-Al-34,33.99693(10),,26.981538(2)
13-Al-35,34.99994(15),,26.981538(2)
13-Al-36,36.00635(29),,26.981538(2)
13-Al-37,37.01031(58),,26.981538(2)
13-Al-38,38.01690(60)#,,26.981538(2)
13-Al-39,39.02190(64)#,,26.981538(2)
14-Si-22,22.03453(22)#,,28.0855(3)
14-Si-23,23.02552(21)#,,28.0855(3)
14-Si-24,24.011546(21),,28.0855(3)
14-Si-25,25.004107(11),,28.0855(3)
14-Si-26,25.992330(3),,28.0855(3)
14-Si-27,26.98670476(17),,28.0855(3)
14-Si-28,27.9769265327(20),92.2297(7),28.0855(3)
14-Si-29,28.97649472(3),4.6832(5),28.0855(3)
14-Si-30,29.97377022(5),3.0872(5),28.0855(3)
14-Si-31,30.97536327(7),,28.0855(3)
14-Si-32,31.9741481(23),,28.0855(3)
14-Si-33,32.978001(17),,28.0855(3)
14-Si-34,33.978576(15),,28.0855(3)
14-Si-35,34.984580(40),,28.0855(3)
14-Si-36,35.98669(11),,28.0855(3)
14-Si-37,36.99300(13),,28.0855(3)
14-Si-38,37.99598(29),,28.0855(3)
14-Si-39,39.00230(43)#,,28.0855(3)
14-Si-40,40.00580(54)#,,28.0855(3)
14-Si-41,41.01270(64)#,,28.0855(3)
14-Si-42,42.01610(75)#,,28.0855(3)
15-P-24,24.03435(54)#,,30.973761(2)
15-P-25,25.02026(21)#,,30.973761(2)
15-P-26,26.01178(21)#,,30.973761(2)
15-P-27,26.999190(40),,30.973761(2)
15-P-28,27.992312(4),,30.973761(2)
15-P-29,28.9818014(8),,30.973761(2)
15-P-30,29.9783138(4),,30.973761(2)
15-P-31,30.97376151(20),100,30.973761(2)
15-P-32,31.97390716(20),,30.973761(2)
15-P-33,32.9717253(12),,30.973761(2)
15-P-34,33.973636(5),,30.973761(2)
15-P-35,34.9733142(20),,30.973761(2)
15-P-36,35.978260(14),,30.973761(2)
15-P-37,36.979610(40),,30.973761(2)
15-P-38,37.98447(15),,30.973761(2)
15-P-39,38.98642(16),,30.973761(2)
15-P-40,39.99105(21),,30.973761(2)
15-P-41,40.99480(50),,30.973761(2)
15-P-42,42.00009(54)#,,30.973761(2)
15-P-43,43.00331(54)#,,30.973761(2)
15-P-44,44.00988(75)#,,30.973761(2)
15-P-45,45.01514(86)#,,30.973761(2)
15-P-46,46.02383(97)#,,30.973761(2)
16-S-26,26.02788(32)#,,32.065(5)
16-S-27,27.01880(22)#,,32.065(5)
16-S-28,28.00437(17),,32.065(5)
16-S-29,28.996610(50),,32.065(5)
16-S-30,29.984903(3),,32.065(5)
16-S-31,30.9795544(16),,32.065(5)
16-S-32,31.97207069(12),94.93(31),32.065(5)
16-S-33,32.97145850(12),0.76(2),32.065(5)
16-S-34,33.96786683(11),4.29(28),32.065(5)
16-S-35,34.96903214(10),,32.065(5)
16-S-36,35.96708088(25),0.02(1),32.065(5)
16-S-37,36.97112572(27),,32.065(5)
16-S-38,37.971163(8),,32.065(5)
16-S-39,38.975140(50),,32.065(5)
16-S-40,39.97547(25),,32.065(5)
16-S-41,40.98003(23),,32.065(5)
16-S-42,41.98149(35),,32.065(5)
16-S-43,42.98660(90),,32.065(5)
16-S-44,43.98832(54)#,,32.065(5)
16-S-45,44.99482(64)#,,32.065(5)
16-S-46,45.99957(75)#,,32.065(5)
16-S-47,47.00762(86)#,,32.065(5)
16-S-48,48.01299(97)#,,32.065(5)
16-S-49,49.02201(107)#,,32.065(5)
17-Cl-28,28.02851(54)#,,35.453(2)
17-Cl-29,29.01411(21)#,,35.453(2)
17-Cl-30,30.00477(21)#,,35.453(2)
17-Cl-31,30.992420(50),,35.453(2)
17-Cl-32,31.985689(7),,35.453(2)
17-Cl-33,32.9774518(6),,35.453(2)
17-Cl-34,33.97376197(13),,35.453(2)
17-Cl-35,34.96885271(4),75.78(4),35.453(2)
17-Cl-36,35.96830695(8),,35.453(2)
17-Cl-37,36.96590260(5),24.22(4),35.453(2)
17-Cl-38,37.96801055(12),,35.453(2)
17-Cl-39,38.9680077(19),,35.453(2)
17-Cl-40,39.970420(30),,35.453(2)
17-Cl-41,40.970650(70),,35.453(2)
17-Cl-42,41.97317(12),,35.453(2)
17-Cl-43,42.97420(17),,35.453(2)
17-Cl-44,43.97854(24),,35.453(2)
17-Cl-45,44.97970(70),,35.453(2)
17-Cl-46,45.98412(54)#,,35.453(2)
17-Cl-47,46.98795(64)#,,35.453(2)
17-Cl-48,47.99485(75)#,,35.453(2)
17-Cl-49,48.99989(86)#,,35.453(2)
17-Cl-50,50.00773(97)#,,35.453(2)
17-Cl-51,51.01353(107)#,,35.453(2)
18-Ar-30,30.02156(32)#,,39.948(1)
18-Ar-31,31.01213(22)#,,39.948(1)
18-Ar-32,31.997660(50),,39.948(1)
18-Ar-33,32.989930(30),,39.948(1)
18-Ar-34,33.980270(3),,39.948(1)
18-Ar-35,34.9752567(8),,39.948(1)
18-Ar-36,35.96754628(27),0.3365(30),39.948(1)
18-Ar-37,36.9667759(3),,39.948(1)
18-Ar-38,37.9627322(5),0.0632(5),39.948(1)
18-Ar-39,38.964313(5),,39.948(1)
18-Ar-40,39.962383123(3),99.6003(30),39.948(1)
18-Ar-41,40.9645008(7),,39.948(1)
18-Ar-42,41.963050(40),,39.948(1)
18-Ar-43,42.965670(80),,39.948(1)
18-Ar-44,43.965365(22),,39.948(1)
18-Ar-45,44.968090(60),,39.948(1)
18-Ar-46,45.968090(40),,39.948(1)
18-Ar-47,46.97219(11),,39.948(1)
18-Ar-48,47.97507(32)#,,39.948(1)
18-Ar-49,48.98218(54)#,,39.948(1)
18-Ar-50,49.98594(75)#,,39.948(1)
18-Ar-51,50.99324(75)#,,39.948(1)
18-Ar-52,51.99817(97)#,,39.948(1)
18-Ar-53,53.00623(107)#,,39.948(1)
19-K-32,32.02192(54)#,,39.0983(1)
19-K-33,33.00726(21)#,,39.0983(1)
19-K-34,33.99841(32)#,,39.0983(1)
19-K-35,34.988012(21),,39.0983(1)
19-K-36,35.981293(8),,39.0983(1)
19-K-37,36.97337691(29),,39.0983(1)
19-K-38,37.9690801(8),,39.0983(1)
19-K-39,38.9637069(3),93.2581(44),39.0983(1)
19-K-40,39.96399867(29),0.0117(1),39.0983(1)
19-K-41,40.96182597(28),6.7302(44),39.0983(1)
19-K-42,41.9624031(3),,39.0983(1)
19-K-43,42.960716(10),,39.0983(1)
19-K-44,43.961560(40),,39.0983(1)
19-K-45,44.960700(11),,39.0983(1)
19-K-46,45.961976(17),,39.0983(1)
19-K-47,46.961678(9),,39.0983(1)
19-K-48,47.965513(26),,39.0983(1)
19-K-49,48.967450(80),,39.0983(1)
19-K-50,49.97278(30),,39.0983(1)
19-K-51,50.97638(54)#,,39.0983(1)
19-K-52,51.98261(75)#,,39.0983(1)
19-K-53,52.98712(75)#,,39.0983(1)
19-K-54,53.99399(97)#,,39.0983(1)
19-K-55,54.99939(107)#,,39.0983(1)
20-Ca-34,34.01412(32)#,,40.078(4)
20-Ca-35,35.004770(70)#,,40.078(4)
20-Ca-36,35.993090(40),,40.078(4)
20-Ca-37,36.985872(24),,40.078(4)
20-Ca-38,37.976319(5),,40.078(4)
20-Ca-39,38.9707177(19),,40.078(4)
20-Ca-40,39.9625912(3),96.941(156),40.078(4)
20-Ca-41,40.9622783(4),,40.078(4)
20-Ca-42,41.9586183(4),0.647(23),40.078(4)
20-Ca-43,42.9587668(5),0.135(10),40.078(4)
20-Ca-44,43.9554811(9),2.086(110),40.078(4)
20-Ca-45,44.9561859(10),,40.078(4)
20-Ca-46,45.9536928(25),0.004(3),40.078(4)
20-Ca-47,46.9545465(25),,40.078(4)
20-Ca-48,47.952534(4),0.187(21),40.078(4)
20-Ca-49,48.955673(4),,40.078(4)
20-Ca-50,49.957518(10),,40.078(4)
20-Ca-51,50.96147(10),,40.078(4)
20-Ca-52,51.96510(50),,40.078(4)
20-Ca-53,52.97005(54)#,,40.078(4)
20-Ca-54,53.97468(75)#,,40.078(4)
20-Ca-55,54.98055(75)#,,40.078(4)
20-Ca-56,55.98579(97)#,,40.078(4)
20-Ca-57,56.99236(107)#,,40.078(4)
21-Sc-36,36.01492(54)#,,44.955910(8)
21-Sc-37,37.00305(32)#,,44.955910(8)
21-Sc-38,37.99470(32)#,,44.955910(8)
21-Sc-39,38.984790(26),,44.955910(8)
21-Sc-40,39.977964(4),,44.955910(8)
21-Sc-41,40.9692513(3),,44.955910(8)
21-Sc-42,41.9655168(4),,44.955910(8)
21-Sc-43,42.9611510(20),,44.955910(8)
21-Sc-44,43.9594030(19),,44.955910(8)
21-Sc-45,44.9559102(12),100,44.955910(8)
21-Sc-46,45.9551703(12),,44.955910(8)
21-Sc-47,46.9524080(22),,44.955910(8)
21-Sc-48,47.952235(6),,44.955910(8)
21-Sc-49,48.950024(4),,44.955910(8)
21-Sc-50,49.952187(17),,44.955910(8)
21-Sc-51,50.953603(22),,44.955910(8)
21-Sc-52,51.95665(25),,44.955910(8)
21-Sc-53,52.95924(32)#,,44.955910(8)
21-Sc-54,53.96300(50),,44.955910(8)
21-Sc-55,54.96743(110)#,,44.955910(8)
21-Sc-56,55.97266(75)#,,44.955910(8)
21-Sc-57,56.97704(75)#,,44.955910(8)
21-Sc-58,57.98307(86)#,,44.955910(8)
21-Sc-59,58.98804(97)#,,44.955910(8)
22-Ti-38,38.00977(27)#,,47.867(1)
22-Ti-39,39.00132(11)#,,47.867(1)
22-Ti-40,39.99050(17),,47.867(1)
22-Ti-41,40.983130(40)#,,47.867(1)
22-Ti-42,41.973032(6),,47.867(1)
22-Ti-43,42.968523(7),,47.867(1)
22-Ti-44,43.9596902(8),,47.867(1)
22-Ti-45,44.9581243(13),,47.867(1)
22-Ti-46,45.9526295(12),8.25(3),47.867(1)
22-Ti-47,46.9517638(10),7.44(2),47.867(1)
22-Ti-48,47.9479471(10),73.72(3),47.867(1)
22-Ti-49,48.9478708(10),5.41(2),47.867(1)
22-Ti-50,49.9447921(11),5.18(2),47.867(1)
22-Ti-51,50.9466160(14),,47.867(1)
22-Ti-52,51.946898(8),,47.867(1)
22-Ti-53,52.94973(11),,47.867(1)
22-Ti-54,53.95087(25),,47.867(1)
22-Ti-55,54.95512(26),,47.867(1)
22-Ti-56,55.95799(30),,47.867(1)
22-Ti-57,56.96290(100)#,,47.867(1)
22-Ti-58,57.96611(75)#,,47.867(1)
22-Ti-59,58.97196(75)#,,47.867(1)
22-Ti-60,59.97564(86)#,,47.867(1)
22-Ti-61,60.98202(97)#,,47.867(1)
23-V-40,40.01109(54)#,,50.9415(1)
23-V-41,40.99974(27)#,,50.9415(1)
23-V-42,41.99123(21)#,,50.9415(1)
23-V-43,42.98065(25)#,,50.9415(1)
23-V-44,43.974400(90)#,,50.9415(1)
23-V-45,44.965782(18),,50.9415(1)
23-V-46,45.9601995(16),,50.9415(1)
23-V-47,46.9549069(12),,50.9415(1)
23-V-48,47.9522545(28),,50.9415(1)
23-V-49,48.9485169(14),,50.9415(1)
23-V-50,49.9471628(14),0.250(4),50.9415(1)
23-V-51,50.9439637(14),99.750(4),50.9415(1)
23-V-52,51.9447797(14),,50.9415(1)
23-V-53,52.944343(4),,50.9415(1)
23-V-54,53.946444(16),,50.9415(1)
23-V-55,54.94724(11),,50.9415(1)
23-V-56,55.95036(26),,50.9415(1)
23-V-57,56.95236(27),,50.9415(1)
23-V-58,57.95665(28),,50.9415(1)
23-V-59,58.95930(35),,50.9415(1)
23-V-60,59.96450(60),,50.9415(1)
23-V-61,60.96741(75)#,,50.9415(1)
23-V-62,61.97314(75)#,,50.9415(1)
23-V-63,62.97675(97)#,,50.9415(1)
24-Cr-42,42.00643(32)#,,51.9961(6)
24-Cr-43,42.997710(90)#,,51.9961(6)
24-Cr-44,43.98547(14)#,,51.9961(6)
24-Cr-45,44.97916(11)#,,51.9961(6)
24-Cr-46,45.968362(22),,51.9961(6)
24-Cr-47,46.962907(15),,51.9961(6)
24-Cr-48,47.954036(8),,51.9961(6)
24-Cr-49,48.9513411(28),,51.9961(6)
24-Cr-50,49.9460496(14),4.345(13),51.9961(6)
24-Cr-51,50.9447718(14),,51.9961(6)
24-Cr-52,51.9405119(15),83.789(18),51.9961(6)
24-Cr-53,52.9406538(15),9.501(17),51.9961(6)
24-Cr-54,53.9388849(15),2.365(7),51.9961(6)
24-Cr-55,54.9408442(16),,51.9961(6)
24-Cr-56,55.940645(10),,51.9961(6)
24-Cr-57,56.94375(10),,51.9961(6)
24-Cr-58,57.94425(26),,51.9961(6)
24-Cr-59,58.94863(27),,51.9961(6)
24-Cr-60,59.94973(28),,51.9961(6)
24-Cr-61,60.95409(30),,51.9961(6)
24-Cr-62,61.95580(40),,51.9961(6)
24-Cr-63,62.96186(75)#,,51.9961(6)
24-Cr-64,63.96420(75)#,,51.9961(6)
24-Cr-65,64.97037(97)#,,51.9961(6)
25-Mn-44,44.00687(54)#,,54.938049(9)
25-Mn-45,44.99451(32)#,,54.938049(9)
25-Mn-46,45.98672(12)#,,54.938049(9)
25-Mn-47,46.97610(17)#,,54.938049(9)
25-Mn-48,47.968870(80)#,,54.938049(9)
25-Mn-49,48.959623(26),,54.938049(9)
25-Mn-50,49.9542440(15),,54.938049(9)
25-Mn-51,50.9482155(14),,54.938049(9)
25-Mn-52,51.9455701(25),,54.938049(9)
25-Mn-53,52.9412947(15),,54.938049(9)
25-Mn-54,53.9403632(18),,54.938049(9)
25-Mn-55,54.9380496(14),100,54.938049(9)
25-Mn-56,55.9389094(15),,54.938049(9)
25-Mn-57,56.938287(4),,54.938049(9)
25-Mn-58,57.939990(30),,54.938049(9)
25-Mn-59,58.940450(30),,54.938049(9)
25-Mn-60,59.94319(29),,54.938049(9)
25-Mn-61,60.94446(28),,54.938049(9)
25-Mn-62,61.94797(28),,54.938049(9)
25-Mn-63,62.94981(30),,54.938049(9)
25-Mn-64,63.95373(35),,54.938049(9)
25-Mn-65,64.95610(60),,54.938049(9)
25-Mn-66,65.96082(75)#,,54.938049(9)
25-Mn-67,66.96382(86)#,,54.938049(9)
26-Fe-45,45.01456(43)#,,55.845(2)
26-Fe-46,46.00081(38)#,,55.845(2)
26-Fe-47,46.99289(28)#,,55.845(2)
26-Fe-48,47.98056(11)#,,55.845(2)
26-Fe-49,48.97361(17)#,,55.845(2)
26-Fe-50,49.962990(60),,55.845(2)
26-Fe-51,50.956825(16),,55.845(2)
26-Fe-52,51.948117(11),,55.845(2)
26-Fe-53,52.9453123(23),,55.845(2)
26-Fe-54,53.9396148(14),5.845(35),55.845(2)
26-Fe-55,54.9382980(14),,55.845(2)
26-Fe-56,55.9349421(15),91.754(36),55.845(2)
26-Fe-57,56.9353987(15),2.119(10),55.845(2)
26-Fe-58,57.9332805(15),0.282(4),55.845(2)
26-Fe-59,58.9348805(15),,55.845(2)
26-Fe-60,59.934077(4),,55.845(2)
26-Fe-61,60.936749(22),,55.845(2)
26-Fe-62,61.936770(16),,55.845(2)
26-Fe-63,62.94012(20),,55.845(2)
26-Fe-64,63.94087(30),,55.845(2)
26-Fe-65,64.94494(30),,55.845(2)
26-Fe-66,65.94598(35),,55.845(2)
26-Fe-67,66.95000(50),,55.845(2)
26-Fe-68,67.95251(75)#,,55.845(2)
26-Fe-69,68.95770(86)#,,55.845(2)
27-Co-48,48.00176(43)#,,58.933200(9)
27-Co-49,48.98972(28)#,,58.933200(9)
27-Co-50,49.98154(18)#,,58.933200(9)
27-Co-51,50.97072(16)#,,58.933200(9)
27-Co-52,51.963590(70)#,,58.933200(9)
27-Co-53,52.954225(19),,58.933200(9)
27-Co-54,53.9484641(14),,58.933200(9)
27-Co-55,54.9420031(15),,58.933200(9)
27-Co-56,55.9398439(26),,58.933200(9)
27-Co-57,56.9362962(15),,58.933200(9)
27-Co-58,57.9357576(19),,58.933200(9)
27-Co-59,58.9332002(15),100,58.933200(9)
27-Co-60,59.9338222(15),,58.933200(9)
27-Co-61,60.9324794(17),,58.933200(9)
27-Co-62,61.934054(22),,58.933200(9)
27-Co-63,62.933615(22),,58.933200(9)
27-Co-64,63.935814(22),,58.933200(9)
27-Co-65,64.936485(14),,58.933200(9)
27-Co-66,65.93983(29),,58.933200(9)
27-Co-67,66.94061(30),,58.933200(9)
27-Co-68,67.94436(35),,58.933200(9)
27-Co-69,68.94520(40),,58.933200(9)
27-Co-70,69.94981(75)#,,58.933200(9)
27-Co-71,70.95173(86)#,,58.933200(9)
27-Co-72,71.95641(86)#,,58.933200(9)
28-Ni-50,49.99593(28)#,,58.6934(2)
28-Ni-51,50.98772(28)#,,58.6934(2)
28-Ni-52,51.975680(90)#,,58.6934(2)
28-Ni-53,52.96846(17)#,,58.6934(2)
28-Ni-54,53.957910(50),,58.6934(2)
28-Ni-55,54.951336(12),,58.6934(2)
28-Ni-56,55.942136(12),,58.6934(2)
28-Ni-57,56.939800(3),,58.6934(2)
28-Ni-58,57.9353479(15),68.0769(89),58.6934(2)
28-Ni-59,58.9343516(15),,58.6934(2)
28-Ni-60,59.9307906(15),26.2231(77),58.6934(2)
28-Ni-61,60.9310604(15),1.1399(6),58.6934(2)
28-Ni-62,61.9283488(15),3.6345(17),58.6934(2)
28-Ni-63,62.9296729(15),,58.6934(2)
28-Ni-64,63.9279696(16),0.9256(9),58.6934(2)
28-Ni-65,64.9300880(16),,58.6934(2)
28-Ni-66,65.929115(17),,58.6934(2)
28-Ni-67,66.931570(20),,58.6934(2)
28-Ni-68,67.931845(18),,58.6934(2)
28-Ni-69,68.93518(15),,58.6934(2)
28-Ni-70,69.93614(35),,58.6934(2)
28-Ni-71,70.94000(40),,58.6934(2)
28-Ni-72,71.94130(50),,58.6934(2)
28-Ni-73,72.94608(64)#,,58.6934(2)
28-Ni-74,73.94791(75)#,,58.6934(2)
28-Ni-75,74.95297(86)#,,58.6934(2)
28-Ni-76,75.95533(97)#,,58.6934(2)
28-Ni-77,76.96083(107)#,,58.6934(2)
28-Ni-78,77.96380(118)#,,58.6934(2)
29-Cu-52,51.99718(28)#,,63.546(3)
29-Cu-53,52.98555(28)#,,63.546(3)
29-Cu-54,53.97671(23)#,,63.546(3)
29-Cu-55,54.96605(32)#,,63.546(3)
29-Cu-56,55.95856(15)#,,63.546(3)
29-Cu-57,56.949216(17),,63.546(3)
29-Cu-58,57.9445407(27),,63.546(3)
29-Cu-59,58.9395041(18),,63.546(3)
29-Cu-60,59.9373681(27),,63.546(3)
29-Cu-61,60.9334622(19),,63.546(3)
29-Cu-62,61.932587(4),,63.546(3)
29-Cu-63,62.9296011(15),69.17(3),63.546(3)
29-Cu-64,63.9297679(15),,63.546(3)
29-Cu-65,64.9277937(19),30.83(3),63.546(3)
29-Cu-66,65.9288730(19),,63.546(3)
29-Cu-67,66.927750(9),,63.546(3)
29-Cu-68,67.929640(50),,63.546(3)
29-Cu-69,68.929425(9),,63.546(3)
29-Cu-70,69.932409(16),,63.546(3)
29-Cu-71,70.932620(40),,63.546(3)
29-Cu-72,71.93552(21)#,,63.546(3)
29-Cu-73,72.93649(32)#,,63.546(3)
29-Cu-74,73.94020(43)#,,63.546(3)
29-Cu-75,74.94170(54)#,,63.546(3)
29-Cu-76,75.94599(64)#,,63.546(3)
29-Cu-77,76.94795(75)#,,63.546(3)
29-Cu-78,77.95281(86)#,,63.546(3)
29-Cu-79,78.95528(97)#,,63.546(3)
29-Cu-80,79.96189(97)#,,63.546(3)
30-Zn-54,53.99295(43)#,,65.409(4)
30-Zn-55,54.98398(27)#,,65.409(4)
30-Zn-56,55.97238(28)#,,65.409(4)
30-Zn-57,56.96491(15)#,,65.409(4)
30-Zn-58,57.954600(50),,65.409(4)
30-Zn-59,58.949270(40),,65.409(4)
30-Zn-60,59.941832(11),,65.409(4)
30-Zn-61,60.939514(18),,65.409(4)
30-Zn-62,61.934334(11),,65.409(4)
30-Zn-63,62.9332156(23),,65.409(4)
30-Zn-64,63.9291466(18),48.63(60),65.409(4)
30-Zn-65,64.9292451(18),,65.409(4)
30-Zn-66,65.9260368(16),27.90(27),65.409(4)
30-Zn-67,66.9271309(17),4.10(13),65.409(4)
30-Zn-68,67.9248476(17),18.75(51),65.409(4)
30-Zn-69,68.9265535(18),,65.409(4)
30-Zn-70,69.925325(4),0.62(3),65.409(4)
30-Zn-71,70.927727(11),,65.409(4)
30-Zn-72,71.926861(7),,65.409(4)
30-Zn-73,72.929780(40),,65.409(4)
30-Zn-74,73.929460(50),,65.409(4)
30-Zn-75,74.932940(80),,65.409(4)
30-Zn-76,75.93339(13),,65.409(4)
30-Zn-77,76.93709(14),,65.409(4)
30-Zn-78,77.93857(17),,65.409(4)
30-Zn-79,78.94268(29)#,,65.409(4)
30-Zn-80,79.94441(18),,65.409(4)
30-Zn-81,80.95048(43)#,,65.409(4)
30-Zn-82,81.95484(43)#,,65.409(4)
31-Ga-56,55.99491(28)#,,69.723(1)
31-Ga-57,56.98293(28)#,,69.723(1)
31-Ga-58,57.97425(23)#,,69.723(1)
31-Ga-59,58.96337(18)#,,69.723(1)
31-Ga-60,59.95706(12)#,,69.723(1)
31-Ga-61,60.94917(21)#,,69.723(1)
31-Ga-62,61.944180(30),,69.723(1)
31-Ga-63,62.93914(11),,69.723(1)
31-Ga-64,63.936838(4),,69.723(1)
31-Ga-65,64.9327393(19),,69.723(1)
31-Ga-66,65.931592(4),,69.723(1)
31-Ga-67,66.9282049(19),,69.723(1)
31-Ga-68,67.9279835(22),,69.723(1)
31-Ga-69,68.925581(3),60.108(9),69.723(1)
31-Ga-70,69.926028(3),,69.723(1)
31-Ga-71,70.9247050(19),39.892(9),69.723(1)
31-Ga-72,71.9263694(22),,69.723(1)
31-Ga-73,72.925170(7),,69.723(1)
31-Ga-74,73.926940(80),,69.723(1)
31-Ga-75,74.926501(7),,69.723(1)
31-Ga-76,75.92893(10),,69.723(1)
31-Ga-77,76.929280(60),,69.723(1)
31-Ga-78,77.931660(90),,69.723(1)
31-Ga-79,78.93292(13),,69.723(1)
31-Ga-80,79.93659(13),,69.723(1)
31-Ga-81,80.93775(21),,69.723(1)
31-Ga-82,81.94316(32)#,,69.723(1)
31-Ga-83,82.94687(54)#,,69.723(1)
31-Ga-84,83.95234(64)#,,69.723(1)
32-Ge-58,57.99101(34)#,,72.64(1)
32-Ge-59,58.98175(30)#,,72.64(1)
32-Ge-60,59.97019(25)#,,72.64(1)
32-Ge-61,60.96379(32)#,,72.64(1)
32-Ge-62,61.95465(15)#,,72.64(1)
32-Ge-63,62.94964(21)#,,72.64(1)
32-Ge-64,63.94157(27),,72.64(1)
32-Ge-65,64.93944(11),,72.64(1)
32-Ge-66,65.933850(30),,72.64(1)
32-Ge-67,66.932738(5),,72.64(1)
32-Ge-68,67.928097(7),,72.64(1)
32-Ge-69,68.927972(3),,72.64(1)
32-Ge-70,69.9242504(19),20.84(87),72.64(1)
32-Ge-71,70.9249540(19),,72.64(1)
32-Ge-72,71.9220762(16),27.54(34),72.64(1)
32-Ge-73,72.9234594(16),7.73(5),72.64(1)
32-Ge-74,73.9211782(16),36.28(73),72.64(1)
32-Ge-75,74.9228595(16),,72.64(1)
32-Ge-76,75.9214027(16),7.61(38),72.64(1)
32-Ge-77,76.9235485(20),,72.64(1)
32-Ge-78,77.922853(4),,72.64(1)
32-Ge-79,78.92540(10),,72.64(1)
32-Ge-80,79.925445(25),,72.64(1)
32-Ge-81,80.92882(13),,72.64(1)
32-Ge-82,81.92955(26),,72.64(1)
32-Ge-83,82.93451(32)#,,72.64(1)
32-Ge-84,83.93731(43)#,,72.64(1)
32-Ge-85,84.94269(54)#,,72.64(1)
32-Ge-86,85.94627(64)#,,72.64(1)
33-As-60,59.99313(64)#,,74.92160(2)
33-As-61,60.98062(64)#,,74.92160(2)
33-As-62,61.97320(32)#,,74.92160(2)
33-As-63,62.96369(54)#,,74.92160(2)
33-As-64,63.95757(38)#,,74.92160(2)
33-As-65,64.94948(42)#,,74.92160(2)
33-As-66,65.94437(22)#,,74.92160(2)
33-As-67,66.93919(11),,74.92160(2)
33-As-68,67.93679(11),,74.92160(2)
33-As-69,68.932280(30),,74.92160(2)
33-As-70,69.930930(50),,74.92160(2)
33-As-71,70.927115(5),,74.92160(2)
33-As-72,71.926753(5),,74.92160(2)
33-As-73,72.923825(4),,74.92160(2)
33-As-74,73.9239291(24),,74.92160(2)
33-As-75,74.9215964(18),100,74.92160(2)
33-As-76,75.9223939(18),,74.92160(2)
33-As-77,76.9206477(23),,74.92160(2)
33-As-78,77.921829(11),,74.92160(2)
33-As-79,78.920948(6),,74.92160(2)
33-As-80,79.922578(23),,74.92160(2)
33-As-81,80.922133(6),,74.92160(2)
33-As-82,81.92450(21),,74.92160(2)
33-As-83,82.92498(24),,74.92160(2)
33-As-84,83.92906(32)#,,74.92160(2)
33-As-85,84.93181(32)#,,74.92160(2)
33-As-86,85.93623(43)#,,74.92160(2)
33-As-87,86.93958(54)#,,74.92160(2)
33-As-88,87.94456(64)#,,74.92160(2)
33-As-89,88.94923(64)#,,74.92160(2)
34-Se-65,64.96466(64)#,,78.96(3)
34-Se-66,65.95521(32)#,,78.96(3)
34-Se-67,66.95009(21)#,,78.96(3)
34-Se-68,67.94187(32)#,,78.96(3)
34-Se-69,68.939560(40),,78.96(3)
34-Se-70,69.93350(22)#,,78.96(3)
34-Se-71,70.93227(22)#,,78.96(3)
34-Se-72,71.927112(13),,78.96(3)
34-Se-73,72.926767(12),,78.96(3)
34-Se-74,73.9224766(16),0.89(4),78.96(3)
34-Se-75,74.9225236(16),,78.96(3)
34-Se-76,75.9192141(16),9.37(29),78.96(3)
34-Se-77,76.9199146(16),7.63(16),78.96(3)
34-Se-78,77.9173095(16),23.77(28),78.96(3)
34-Se-79,78.9184998(16),,78.96(3)
34-Se-80,79.9165218(20),49.61(41),78.96(3)
34-Se-81,80.9179929(21),,78.96(3)
34-Se-82,81.9167000(22),8.73(22),78.96(3)
34-Se-83,82.919119(4),,78.96(3)
34-Se-84,83.918465(16),,78.96(3)
34-Se-85,84.922240(30),,78.96(3)
34-Se-86,85.924271(17),,78.96(3)
34-Se-87,86.928520(40),,78.96(3)
34-Se-88,87.931420(50),,78.96(3)
34-Se-89,88.93602(32)#,,78.96(3)
34-Se-90,89.93942(43)#,,78.96(3)
34-Se-91,90.94537(54)#,,78.96(3)
34-Se-92,91.94933(64)#,,78.96(3)
35-Br-67,66.96479(54)#,,79.904(1)
35-Br-68,67.95825(58)#,,79.904(1)
35-Br-69,68.95018(34)#,,79.904(1)
35-Br-70,69.94462(39)#,,79.904(1)
35-Br-71,70.93925(32)#,,79.904(1)
35-Br-72,71.93650(28),,79.904(1)
35-Br-73,72.93179(14),,79.904(1)
35-Br-74,73.929891(16),,79.904(1)
35-Br-75,74.925776(15),,79.904(1)
35-Br-76,75.924542(10),,79.904(1)
35-Br-77,76.921380(3),,79.904(1)
35-Br-78,77.921146(4),,79.904(1)
35-Br-79,78.9183376(20),50.69(7),79.904(1)
35-Br-80,79.9185300(20),,79.904(1)
35-Br-81,80.916291(3),49.31(7),79.904(1)
35-Br-82,81.916805(3),,79.904(1)
35-Br-83,82.915180(5),,79.904(1)
35-Br-84,83.916504(27),,79.904(1)
35-Br-85,84.915608(21),,79.904(1)
35-Br-86,85.918797(12),,79.904(1)
35-Br-87,86.920711(19),,79.904(1)
35-Br-88,87.924070(40),,79.904(1)
35-Br-89,88.926390(60),,79.904(1)
35-Br-90,89.930630(80),,79.904(1)
35-Br-91,90.933970(80),,79.904(1)
35-Br-92,91.939260(50),,79.904(1)
35-Br-93,92.94310(32)#,,79.904(1)
35-Br-94,93.94868(43)#,,79.904(1)
36-Kr-69,68.96532(54)#,,83.798(2)
36-Kr-70,69.95601(43)#,,83.798(2)
36-Kr-71,70.95051(32)#,,83.798(2)
36-Kr-72,71.94191(29),,83.798(2)
36-Kr-73,72.93893(15),,83.798(2)
36-Kr-74,73.933260(60),,83.798(2)
36-Kr-75,74.931034(17),,83.798(2)
36-Kr-76,75.925948(11),,83.798(2)
36-Kr-77,76.924668(9),,83.798(2)
36-Kr-78,77.920386(7),0.35(1),83.798(2)
36-Kr-79,78.920083(4),,83.798(2)
36-Kr-80,79.916378(4),2.28(6),83.798(2)
36-Kr-81,80.916592(3),,83.798(2)
36-Kr-82,81.9134846(28),11.58(14),83.798(2)
36-Kr-83,82.914136(3),11.49(6),83.798(2)
36-Kr-84,83.911507(3),57.00(4),83.798(2)
36-Kr-85,84.912527(3),,83.798(2)
36-Kr-86,85.9106103(12),17.30(22),83.798(2)
36-Kr-87,86.9133543(14),,83.798(2)
36-Kr-88,87.914447(14),,83.798(2)
36-Kr-89,88.917630(60),,83.798(2)
36-Kr-90,89.919524(20),,83.798(2)
36-Kr-91,90.923440(60),,83.798(2)
36-Kr-92,91.926153(13),,83.798(2)
36-Kr-93,92.93127(11),,83.798(2)
36-Kr-94,93.93436(32)#,,83.798(2)
36-Kr-95,94.93984(43)#,,83.798(2)
36-Kr-96,95.94307(54)#,,83.798(2)
36-Kr-97,96.94856(54)#,,83.798(2)
37-Rb-71,70.96532(54)#,,85.4678(3)
37-Rb-72,71.95908(54)#,,85.4678(3)
37-Rb-73,72.95037(52)#,,85.4678(3)
37-Rb-74,73.94447(77),,85.4678(3)
37-Rb-75,74.938569(8),,85.4678(3)
37-Rb-76,75.935071(8),,85.4678(3)
37-Rb-77,76.930407(8),,85.4678(3)
37-Rb-78,77.928141(8),,85.4678(3)
37-Rb-79,78.923997(7),,85.4678(3)
37-Rb-80,79.922519(8),,85.4678(3)
37-Rb-81,80.918994(7),,85.4678(3)
37-Rb-82,81.918208(8),,85.4678(3)
37-Rb-83,82.915112(7),,85.4678(3)
37-Rb-84,83.914385(3),,85.4678(3)
37-Rb-85,84.9117893(25),72.17(2),85.4678(3)
37-Rb-86,85.9111671(25),,85.4678(3)
37-Rb-87,86.9091835(27),27.83(2),85.4678(3)
37-Rb-88,87.911319(5),,85.4678(3)
37-Rb-89,88.912280(6),,85.4678(3)
37-Rb-90,89.914809(9),,85.4678(3)
37-Rb-91,90.916534(9),,85.4678(3)
37-Rb-92,91.919725(7),,85.4678(3)
37-Rb-93,92.922033(8),,85.4678(3)
37-Rb-94,93.926407(9),,85.4678(3)
37-Rb-95,94.929319(21),,85.4678(3)
37-Rb-96,95.934284(27),,85.4678(3)
37-Rb-97,96.937340(30),,85.4678(3)
37-Rb-98,97.941700(40),,85.4678(3)
37-Rb-99,98.94542(16),,85.4678(3)
37-Rb-100,99.94987(32)#,,85.4678(3)
37-Rb-101,100.95320(18),,85.4678(3)
37-Rb-102,101.95921(54)#,,85.4678(3)
38-Sr-73,72.96597(64)#,,87.62(1)
38-Sr-74,73.95631(54)#,,87.62(1)
38-Sr-75,74.94992(32)#,,87.62(1)
38-Sr-76,75.94161(32)#,,87.62(1)
38-Sr-77,76.93776(16),,87.62(1)
38-Sr-78,77.932179(8),,87.62(1)
38-Sr-79,78.929707(9),,87.62(1)
38-Sr-80,79.924525(8),,87.62(1)
38-Sr-81,80.923213(8),,87.62(1)
38-Sr-82,81.918401(6),,87.62(1)
38-Sr-83,82.917555(9),,87.62(1)
38-Sr-84,83.913425(4),0.56(1),87.62(1)
38-Sr-85,84.912933(4),,87.62(1)
38-Sr-86,85.9092624(24),9.86(1),87.62(1)
38-Sr-87,86.9088793(24),7.00(1),87.62(1)
38-Sr-88,87.9056143(24),82.58(1),87.62(1)
38-Sr-89,88.9074529(24),,87.62(1)
38-Sr-90,89.9077376(29),,87.62(1)
38-Sr-91,90.910210(6),,87.62(1)
38-Sr-92,91.911030(7),,87.62(1)
38-Sr-93,92.914022(8),,87.62(1)
38-Sr-94,93.915360(8),,87.62(1)
38-Sr-95,94.919358(8),,87.62(1)
38-Sr-96,95.921680(26),,87.62(1)
38-Sr-97,96.926149(20),,87.62(1)
38-Sr-98,97.928471(27),,87.62(1)
38-Sr-99,98.93332(15),,87.62(1)
38-Sr-100,99.93535(14),,87.62(1)
38-Sr-101,100.94052(13),,87.62(1)
38-Sr-102,101.94302(12),,87.62(1)
38-Sr-103,102.94895(54)#,,87.62(1)
38-Sr-104,103.95233(75)#,,87.62(1)
39-Y-77,76.94962(32)#,,88.90585(2)
39-Y-78,77.94350(43)#,,88.90585(2)
39-Y-79,78.93735(48),,88.90585(2)
39-Y-80,79.93434(43)#,,88.90585(2)
39-Y-81,80.929130(70),,88.90585(2)
39-Y-82,81.92679(11),,88.90585(2)
39-Y-83,82.922350(50),,88.90585(2)
39-Y-84,83.92039(10),,88.90585(2)
39-Y-85,84.916427(27),,88.90585(2)
39-Y-86,85.914888(15),,88.90585(2)
39-Y-87,86.9108778(28),,88.90585(2)
39-Y-88,87.9095034(29),,88.90585(2)
39-Y-89,88.9058479(25),100,88.90585(2)
39-Y-90,89.9071514(25),,88.90585(2)
39-Y-91,90.907303(3),,88.90585(2)
39-Y-92,91.908947(10),,88.90585(2)
39-Y-93,92.909582(11),,88.90585(2)
39-Y-94,93.911594(8),,88.90585(2)
39-Y-95,94.912824(8),,88.90585(2)
39-Y-96,95.915898(23),,88.90585(2)
39-Y-97,96.918131(13),,88.90585(2)
39-Y-98,97.922220(26),,88.90585(2)
39-Y-99,98.924635(26),,88.90585(2)
39-Y-100,99.927760(80),,88.90585(2)
39-Y-101,100.93031(10),,88.90585(2)
39-Y-102,101.933560(90),,88.90585(2)
39-Y-103,102.93694(32)#,,88.90585(2)
39-Y-104,103.94145(43)#,,88.90585(2)
39-Y-105,104.94509(54)#,,88.90585(2)
39-Y-106,105.95022(75)#,,88.90585(2)
40-Zr-79,78.94916(43)#,,91.224(2)
40-Zr-80,79.94055(32)#,,91.224(2)
40-Zr-81,80.93682(32),,91.224(2)
40-Zr-82,81.93109(55),,91.224(2)
40-Zr-83,82.92865(10),,91.224(2)
40-Zr-84,83.92325(21)#,,91.224(2)
40-Zr-85,84.92147(11),,91.224(2)
40-Zr-86,85.916470(30),,91.224(2)
40-Zr-87,86.914817(9),,91.224(2)
40-Zr-88,87.910226(11),,91.224(2)
40-Zr-89,88.908889(4),,91.224(2)
40-Zr-90,89.9047037(23),51.45(40),91.224(2)
40-Zr-91,90.9056450(23),11.22(5),91.224(2)
40-Zr-92,91.9050401(23),17.15(8),91.224(2)
40-Zr-93,92.9064756(23),,91.224(2)
40-Zr-94,93.9063158(25),17.38(28),91.224(2)
40-Zr-95,94.9080427(25),,91.224(2)
40-Zr-96,95.908276(3),2.80(9),91.224(2)
40-Zr-97,96.910951(3),,91.224(2)
40-Zr-98,97.912746(21),,91.224(2)
40-Zr-99,98.916511(21),,91.224(2)
40-Zr-100,99.917760(40),,91.224(2)
40-Zr-101,100.921140(30),,91.224(2)
40-Zr-102,101.922980(50),,91.224(2)
40-Zr-103,102.92660(12),,91.224(2)
40-Zr-104,103.92878(43)#,,91.224(2)
40-Zr-105,104.93305(43)#,,91.224(2)
40-Zr-106,105.93591(54)#,,91.224(2)
40-Zr-107,106.94086(64)#,,91.224(2)
40-Zr-108,107.94428(75)#,,91.224(2)
41-Nb-81,80.94905(43)#,,92.90638(2)
41-Nb-82,81.94313(32)#,,92.90638(2)
41-Nb-83,82.93670(34),,92.90638(2)
41-Nb-84,83.93357(32)#,,92.90638(2)
41-Nb-85,84.92791(24),,92.90638(2)
41-Nb-86,85.925040(90),,92.90638(2)
41-Nb-87,86.920360(70),,92.90638(2)
41-Nb-88,87.91796(22)#,,92.90638(2)
41-Nb-89,88.913500(40),,92.90638(2)
41-Nb-90,89.911264(5),,92.90638(2)
41-Nb-91,90.906991(3),,92.90638(2)
41-Nb-92,91.9071932(29),,92.90638(2)
41-Nb-93,92.9063775(24),100,92.90638(2)
41-Nb-94,93.9072835(24),,92.90638(2)
41-Nb-95,94.9068352(20),,92.90638(2)
41-Nb-96,95.908100(4),,92.90638(2)
41-Nb-97,96.9080971(28),,92.90638(2)
41-Nb-98,97.910331(6),,92.90638(2)
41-Nb-99,98.911618(14),,92.90638(2)
41-Nb-100,99.914181(28),,92.90638(2)
41-Nb-101,100.915252(20),,92.90638(2)
41-Nb-102,101.918040(40),,92.90638(2)
41-Nb-103,102.919140(70),,92.90638(2)
41-Nb-104,103.92246(12),,92.90638(2)
41-Nb-105,104.92393(11),,92.90638(2)
41-Nb-106,105.92819(32)#,,92.90638(2)
41-Nb-107,106.93031(43)#,,92.90638(2)
41-Nb-108,107.93501(54)#,,92.90638(2)
41-Nb-109,108.93763(54)#,,92.90638(2)
41-Nb-110,109.94268(64)#,,92.90638(2)
42-Mo-83,82.94874(54)#,,95.94(2)
42-Mo-84,83.94009(43)#,,95.94(2)
42-Mo-85,84.93659(43)#,,95.94(2)
42-Mo-86,85.93070(47),,95.94(2)
42-Mo-87,86.92733(24),,95.94(2)
42-Mo-88,87.921953(22),,95.94(2)
42-Mo-89,88.919481(17),,95.94(2)
42-Mo-90,89.913936(7),,95.94(2)
42-Mo-91,90.911751(12),,95.94(2)
42-Mo-92,91.906810(4),14.84(35),95.94(2)
42-Mo-93,92.906812(4),,95.94(2)
42-Mo-94,93.9050876(20),9.25(12),95.94(2)
42-Mo-95,94.9058415(20),15.92(13),95.94(2)
42-Mo-96,95.9046789(20),16.68(2),95.94(2)
42-Mo-97,96.9060210(20),9.55(8),95.94(2)
42-Mo-98,97.9054078(20),24.13(31),95.94(2)
42-Mo-99,98.9077116(20),,95.94(2)
42-Mo-100,99.907477(6),9.63(23),95.94(2)
42-Mo-101,100.910347(6),,95.94(2)
42-Mo-102,101.910297(22),,95.94(2)
42-Mo-103,102.913200(70),,95.94(2)
42-Mo-104,103.913760(70),,95.94(2)
42-Mo-105,104.916970(80),,95.94(2)
42-Mo-106,105.918134(23),,95.94(2)
42-Mo-107,106.92169(17),,95.94(2)
42-Mo-108,107.92358(21)#,,95.94(2)
42-Mo-109,108.92781(32)#,,95.94(2)
42-Mo-110,109.92973(43)#,,95.94(2)
42-Mo-111,110.93451(54)#,,95.94(2)
42-Mo-112,111.93684(64)#,,95.94(2)
42-Mo-113,112.94203(64)#,,95.94(2)
43-Tc-85,84.94894(54)#,,[98]
43-Tc-86,85.94288(32)#,,[98]
43-Tc-87,86.93653(32)#,,[98]
43-Tc-88,87.93283(32)#,,[98]
43-Tc-89,88.92754(23),,[98]
43-Tc-90,89.92356(26),,[98]
43-Tc-91,90.91843(22),,[98]
43-Tc-92,91.915260(28),,[98]
43-Tc-93,92.910248(4),,[98]
43-Tc-94,93.909656(5),,[98]
43-Tc-95,94.907656(6),,[98]
43-Tc-96,95.907871(6),,[98]
43-Tc-97,96.906365(5),,[98]
43-Tc-98,97.907216(4),,[98]
43-Tc-99,98.9062546(21),,[98]
43-Tc-100,99.9076576(23),,[98]
43-Tc-101,100.907314(26),,[98]
43-Tc-102,101.909213(10),,[98]
43-Tc-103,102.909179(11),,[98]
43-Tc-104,103.911440(50),,[98]
43-Tc-105,104.911660(60),,[98]
43-Tc-106,105.914355(15),,[98]
43-Tc-107,106.91508(16),,[98]
43-Tc-108,107.91848(14),,[98]
43-Tc-109,108.91963(23)#,,[98]
43-Tc-110,109.92339(43)#,,[98]
43-Tc-111,110.92505(43)#,,[98]
43-Tc-112,111.92924(54)#,,[98]
43-Tc-113,112.93133(64)#,,[98]
43-Tc-114,113.93588(64)#,,[98]
43-Tc-115,114.93828(75)#,,[98]
44-Ru-87,86.94918(64)#,,101.07(2)
44-Ru-88,87.94042(54)#,,101.07(2)
44-Ru-89,88.93611(54)#,,101.07(2)
44-Ru-90,89.92978(43)#,,101.07(2)
44-Ru-91,90.92638(54),,101.07(2)
44-Ru-92,91.92012(32)#,,101.07(2)
44-Ru-93,92.917050(90),,101.07(2)
44-Ru-94,93.911360(14),,101.07(2)
44-Ru-95,94.910413(13),,101.07(2)
44-Ru-96,95.907598(8),5.54(14),101.07(2)
44-Ru-97,96.907555(9),,101.07(2)
44-Ru-98,97.905287(7),1.87(3),101.07(2)
44-Ru-99,98.9059393(21),12.76(14),101.07(2)
44-Ru-100,99.9042197(22),12.60(7),101.07(2)
44-Ru-101,100.9055822(22),17.06(2),101.07(2)
44-Ru-102,101.9043495(22),31.55(14),101.07(2)
44-Ru-103,102.9063237(22),,101.07(2)
44-Ru-104,103.905430(4),18.62(27),101.07(2)
44-Ru-105,104.907750(4),,101.07(2)
44-Ru-106,105.907327(8),,101.07(2)
44-Ru-107,106.90991(13),,101.07(2)
44-Ru-108,107.91019(13),,101.07(2)
44-Ru-109,108.913200(70),,101.07(2)
44-Ru-110,109.91397(25),,101.07(2)
44-Ru-111,110.91756(32)#,,101.07(2)
44-Ru-112,111.91855(58)#,,101.07(2)
44-Ru-113,112.92254(54)#,,101.07(2)
44-Ru-114,113.92400(39)#,,101.07(2)
44-Ru-115,114.92831(64)#,,101.07(2)
44-Ru-116,115.93016(75)#,,101.07(2)
44-Ru-117,116.93479(86)#,,101.07(2)
44-Ru-118,117.93703(97)#,,101.07(2)
45-Rh-89,88.94938(54)#,,102.90550(2)
45-Rh-90,89.94287(54)#,,102.90550(2)
45-Rh-91,90.93655(43)#,,102.90550(2)
45-Rh-92,91.93198(43)#,,102.90550(2)
45-Rh-93,92.92574(43)#,,102.90550(2)
45-Rh-94,93.92170(48)#,,102.90550(2)
45-Rh-95,94.91590(16),,102.90550(2)
45-Rh-96,95.914518(14),,102.90550(2)
45-Rh-97,96.911340(40),,102.90550(2)
45-Rh-98,97.910716(13),,102.90550(2)
45-Rh-99,98.908132(8),,102.90550(2)
45-Rh-100,99.908117(22),,102.90550(2)
45-Rh-101,100.906164(19),,102.90550(2)
45-Rh-102,101.906843(5),,102.90550(2)
45-Rh-103,102.905504(3),100,102.90550(2)
45-Rh-104,103.906655(3),,102.90550(2)
45-Rh-105,104.905692(5),,102.90550(2)
45-Rh-106,105.907285(8),,102.90550(2)
45-Rh-107,106.906751(13),,102.90550(2)
45-Rh-108,107.90873(11),,102.90550(2)
45-Rh-109,108.908736(13),,102.90550(2)
45-Rh-110,109.91095(24),,102.90550(2)
45-Rh-111,110.91166(22)#,,102.90550(2)
45-Rh-112,111.91461(54)#,,102.90550(2)
45-Rh-113,112.91542(43)#,,102.90550(2)
45-Rh-114,113.91885(32)#,,102.90550(2)
45-Rh-115,114.92012(54),,102.90550(2)
45-Rh-116,115.92371(54)#,,102.90550(2)
45-Rh-117,116.92535(64)#,,102.90550(2)
45-Rh-118,117.92943(75)#,,102.90550(2)
45-Rh-119,118.93136(86)#,,102.90550(2)
45-Rh-120,119.93578(86)#,,102.90550(2)
45-Rh-121,120.93808(97)#,,102.90550(2)
46-Pd-91,90.94948(64)#,,106.42(1)
46-Pd-92,91.94042(54)#,,106.42(1)
46-Pd-93,92.93591(43)#,,106.42(1)
46-Pd-94,93.92877(43)#,,106.42(1)
46-Pd-95,94.92469(43)#,,106.42(1)
46-Pd-96,95.91822(16),,106.42(1)
46-Pd-97,96.91648(32),,106.42(1)
46-Pd-98,97.912721(23),,106.42(1)
46-Pd-99,98.911768(16),,106.42(1)
46-Pd-100,99.908505(12),,106.42(1)
46-Pd-101,100.908289(19),,106.42(1)
46-Pd-102,101.905608(3),1.02(1),106.42(1)
46-Pd-103,102.906087(3),,106.42(1)
46-Pd-104,103.904035(5),11.14(8),106.42(1)
46-Pd-105,104.905084(5),22.33(8),106.42(1)
46-Pd-106,105.903483(5),27.33(3),106.42(1)
46-Pd-107,106.905128(7),,106.42(1)
46-Pd-108,107.903894(4),26.46(9),106.42(1)
46-Pd-109,108.905954(4),,106.42(1)
46-Pd-110,109.905152(12),11.72(9),106.42(1)
46-Pd-111,110.907640(40),,106.42(1)
46-Pd-112,111.907313(19),,106.42(1)
46-Pd-113,112.910150(40),,106.42(1)
46-Pd-114,113.910365(26),,106.42(1)
46-Pd-115,114.913680(70),,106.42(1)
46-Pd-116,115.914160(60),,106.42(1)
46-Pd-117,116.91784(32)#,,106.42(1)
46-Pd-118,117.91898(23),,106.42(1)
46-Pd-119,118.92268(32)#,,106.42(1)
46-Pd-120,119.92403(43)#,,106.42(1)
46-Pd-121,120.92818(54)#,,106.42(1)
46-Pd-122,121.92980(54)#,,106.42(1)
46-Pd-123,122.93426(64)#,,106.42(1)
47-Ag-94,93.94278(54)#,,107.8682(2)
47-Ag-95,94.93548(43)#,,107.8682(2)
47-Ag-96,95.93068(43)#,,107.8682(2)
47-Ag-97,96.92400(43)#,,107.8682(2)
47-Ag-98,97.92176(16),,107.8682(2)
47-Ag-99,98.91760(16),,107.8682(2)
47-Ag-100,99.916070(80),,107.8682(2)
47-Ag-101,100.91280(11),,107.8682(2)
47-Ag-102,101.912000(80),,107.8682(2)
47-Ag-103,102.908972(18),,107.8682(2)
47-Ag-104,103.908628(7),,107.8682(2)
47-Ag-105,104.906528(12),,107.8682(2)
47-Ag-106,105.906666(6),,107.8682(2)
47-Ag-107,106.905093(6),51.839(8),107.8682(2)
47-Ag-108,107.905954(6),,107.8682(2)
47-Ag-109,108.904756(3),48.161(8),107.8682(2)
47-Ag-110,109.906110(3),,107.8682(2)
47-Ag-111,110.905295(4),,107.8682(2)
47-Ag-112,111.907004(18),,107.8682(2)
47-Ag-113,112.906566(18),,107.8682(2)
47-Ag-114,113.908808(28),,107.8682(2)
47-Ag-115,114.908760(40),,107.8682(2)
47-Ag-116,115.911360(50),,107.8682(2)
47-Ag-117,116.911680(50),,107.8682(2)
47-Ag-118,117.914580(70),,107.8682(2)
47-Ag-119,118.91567(10),,107.8682(2)
47-Ag-120,119.918790(80),,107.8682(2)
47-Ag-121,120.91985(16),,107.8682(2)
47-Ag-122,121.92332(22)#,,107.8682(2)
47-Ag-123,122.92490(32)#,,107.8682(2)
47-Ag-124,123.92853(43)#,,107.8682(2)
47-Ag-125,124.93054(43)#,,107.8682(2)
47-Ag-126,125.93450(43)#,,107.8682(2)
47-Ag-127,126.93688(54)#,,107.8682(2)
48-Cd-96,95.93977(54)#,,112.411(8)
48-Cd-97,96.93494(43)#,,112.411(8)
48-Cd-98,97.92758(22)#,,112.411(8)
48-Cd-99,98.92501(22)#,,112.411(8)
48-Cd-100,99.92023(10),,112.411(8)
48-Cd-101,100.91868(16),,112.411(8)
48-Cd-102,101.914780(80),,112.411(8)
48-Cd-103,102.913419(17),,112.411(8)
48-Cd-104,103.909848(10),,112.411(8)
48-Cd-105,104.909468(12),,112.411(8)
48-Cd-106,105.906458(6),1.25(6),112.411(8)
48-Cd-107,106.906614(7),,112.411(8)
48-Cd-108,107.904183(6),0.89(3),112.411(8)
48-Cd-109,108.904986(4),,112.411(8)
48-Cd-110,109.903006(3),12.49(18),112.411(8)
48-Cd-111,110.904182(3),12.80(12),112.411(8)
48-Cd-112,111.9027572(30),24.13(21),112.411(8)
48-Cd-113,112.9044009(30),12.22(12),112.411(8)
48-Cd-114,113.9033581(30),28.73(42),112.411(8)
48-Cd-115,114.905431(3),,112.411(8)
48-Cd-116,115.904755(3),7.49(18),112.411(8)
48-Cd-117,116.907218(4),,112.411(8)
48-Cd-118,117.906914(22),,112.411(8)
48-Cd-119,118.909920(90),,112.411(8)
48-Cd-120,119.909851(20),,112.411(8)
48-Cd-121,120.912980(90),,112.411(8)
48-Cd-122,121.91350(22)#,,112.411(8)
48-Cd-123,122.917000(40),,112.411(8)
48-Cd-124,123.917650(70),,112.411(8)
48-Cd-125,124.921250(70),,112.411(8)
48-Cd-126,125.922350(60),,112.411(8)
48-Cd-127,126.926430(80),,112.411(8)
48-Cd-128,127.92776(32),,112.411(8)
48-Cd-129,128.93226(43)#,,112.411(8)
48-Cd-130,129.93398(43)#,,112.411(8)
49-In-98,97.94224(54)#,,114.818(3)
49-In-99,98.93461(54)#,,114.818(3)
49-In-100,99.93115(41),,114.818(3)
49-In-101,100.92656(32)#,,114.818(3)
49-In-102,101.92471(41),,114.818(3)
49-In-103,102.919914(27),,114.818(3)
49-In-104,103.91834(15),,114.818(3)
49-In-105,104.914673(19),,114.818(3)
49-In-106,105.913461(15),,114.818(3)
49-In-107,106.910292(14),,114.818(3)
49-In-108,107.909720(40),,114.818(3)
49-In-109,108.907154(6),,114.818(3)
49-In-110,109.907169(13),,114.818(3)
49-In-111,110.905111(6),,114.818(3)
49-In-112,111.905533(6),,114.818(3)
49-In-113,112.904061(4),4.29(5),114.818(3)
49-In-114,113.904917(3),,114.818(3)
49-In-115,114.903878(5),95.71(5),114.818(3)
49-In-116,115.905260(5),,114.818(3)
49-In-117,116.904516(6),,114.818(3)
49-In-118,117.906355(9),,114.818(3)
49-In-119,118.905846(8),,114.818(3)
49-In-120,119.907960(40),,114.818(3)
49-In-121,120.907849(29),,114.818(3)
49-In-122,121.910280(50),,114.818(3)
49-In-123,122.910439(26),,114.818(3)
49-In-124,123.913180(50),,114.818(3)
49-In-125,124.913600(30),,114.818(3)
49-In-126,125.916460(40),,114.818(3)
49-In-127,126.917340(40),,114.818(3)
49-In-128,127.920170(50),,114.818(3)
49-In-129,128.92166(14),,114.818(3)
49-In-130,129.924850(50),,114.818(3)
49-In-131,130.926770(80),,114.818(3)
49-In-132,131.932920(70),,114.818(3)
49-In-133,132.93834(43)#,,114.818(3)
49-In-134,133.94466(54)#,,114.818(3)
50-Sn-100,99.93895(46)#,,118.710(7)
50-Sn-101,100.93606(54)#,,118.710(7)
50-Sn-102,101.93049(43)#,,118.710(7)
50-Sn-103,102.92813(32)#,,118.710(7)
50-Sn-104,103.92319(16),,118.710(7)
50-Sn-105,104.92139(10),,118.710(7)
50-Sn-106,105.916880(50),,118.710(7)
50-Sn-107,106.915670(90),,118.710(7)
50-Sn-108,107.911970(50),,118.710(7)
50-Sn-109,108.911287(11),,118.710(7)
50-Sn-110,109.907853(17),,118.710(7)
50-Sn-111,110.907735(8),,118.710(7)
50-Sn-112,111.904821(5),0.97(1),118.710(7)
50-Sn-113,112.905173(4),,118.710(7)
50-Sn-114,113.902782(3),0.66(1),118.710(7)
50-Sn-115,114.903346(3),0.34(1),118.710(7)
50-Sn-116,115.901744(3),14.54(9),118.710(7)
50-Sn-117,116.902954(3),7.68(7),118.710(7)
50-Sn-118,117.901606(3),24.22(9),118.710(7)
50-Sn-119,118.903309(3),8.59(4),118.710(7)
50-Sn-120,119.9021966(27),32.58(9),118.710(7)
50-Sn-121,120.9042369(27),,118.710(7)
50-Sn-122,121.9034401(29),4.63(3),118.710(7)
50-Sn-123,122.9057219(29),,118.710(7)
50-Sn-124,123.9052746(15),5.79(5),118.710(7)
50-Sn-125,124.9077849(16),,118.710(7)
50-Sn-126,125.907654(11),,118.710(7)
50-Sn-127,126.910351(27),,118.710(7)
50-Sn-128,127.910535(29),,118.710(7)
50-Sn-129,128.91344(13),,118.710(7)
50-Sn-130,129.913850(30),,118.710(7)
50-Sn-131,130.916920(80),,118.710(7)
50-Sn-132,131.917744(28),,118.710(7)
50-Sn-133,132.923810(90),,118.710(7)
50-Sn-134,133.92846(11),,118.710(7)
50-Sn-135,134.93473(43)#,,118.710(7)
50-Sn-136,135.93934(54)#,,118.710(7)
50-Sn-137,136.94579(64)#,,118.710(7)
51-Sb-103,102.94012(54)#,,121.760(1)
51-Sb-104,103.93629(39)#,,121.760(1)
51-Sb-105,104.93153(17),,121.760(1)
51-Sb-106,105.92876(34)#,,121.760(1)
51-Sb-107,106.92415(32)#,,121.760(1)
51-Sb-108,107.92216(22)#,,121.760(1)
51-Sb-109,108.918136(20),,121.760(1)
51-Sb-110,109.91676(22)#,,121.760(1)
51-Sb-111,110.91321(22)#,,121.760(1)
51-Sb-112,111.912395(25),,121.760(1)
51-Sb-113,112.909378(24),,121.760(1)
51-Sb-114,113.90910(22),,121.760(1)
51-Sb-115,114.906599(22),,121.760(1)
51-Sb-116,115.906797(6),,121.760(1)
51-Sb-117,116.904840(10),,121.760(1)
51-Sb-118,117.905532(4),,121.760(1)
51-Sb-119,118.903946(9),,121.760(1)
51-Sb-120,119.905074(8),,121.760(1)
51-Sb-121,120.9038180(24),57.21(5),121.760(1)
51-Sb-122,121.9051754(24),,121.760(1)
51-Sb-123,122.9042157(22),42.79(5),121.760(1)
51-Sb-124,123.9059375(22),,121.760(1)
51-Sb-125,124.905248(3),,121.760(1)
51-Sb-126,125.907250(30),,121.760(1)
51-Sb-127,126.906915(6),,121.760(1)
51-Sb-128,127.909167(27),,121.760(1)
51-Sb-129,128.909150(23),,121.760(1)
51-Sb-130,129.911546(27),,121.760(1)
51-Sb-131,130.911950(80),,121.760(1)
51-Sb-132,131.914413(25),,121.760(1)
51-Sb-133,132.915240(80),,121.760(1)
51-Sb-134,133.920550(60),,121.760(1)
51-Sb-135,134.92517(11),,121.760(1)
51-Sb-136,135.93066(32)#,,121.760(1)
51-Sb-137,136.93531(43)#,,121.760(1)
51-Sb-138,137.94096(54)#,,121.760(1)
51-Sb-139,138.94571(64)#,,121.760(1)
52-Te-106,105.93770(43)#,,127.60(3)
52-Te-107,106.93504(32)#,,127.60(3)
52-Te-108,107.92949(16),,127.60(3)
52-Te-109,108.927460(80),,127.60(3)
52-Te-110,109.922410(60),,127.60(3)
52-Te-111,110.921120(80),,127.60(3)
52-Te-112,111.91706(18),,127.60(3)
52-Te-113,112.91593(22)#,,127.60(3)
52-Te-114,113.91206(22)#,,127.60(3)
52-Te-115,114.91158(11),,127.60(3)
52-Te-116,115.90842(10),,127.60(3)
52-Te-117,116.908634(20),,127.60(3)
52-Te-118,117.905825(17),,127.60(3)
52-Te-119,118.906408(9),,127.60(3)
52-Te-120,119.904020(11),0.09(1),127.60(3)
52-Te-121,120.904930(27),,127.60(3)
52-Te-122,121.9030471(20),2.55(12),127.60(3)
52-Te-123,122.9042730(19),0.89(3),127.60(3)
52-Te-124,123.9028195(16),4.74(14),127.60(3)
52-Te-125,124.9044247(20),7.07(15),127.60(3)
52-Te-126,125.9033055(20),18.84(25),127.60(3)
52-Te-127,126.905217(4),,127.60(3)
52-Te-128,127.9044614(19),31.74(8),127.60(3)
52-Te-129,128.906596(3),,127.60(3)
52-Te-130,129.9062228(21),34.08(62),127.60(3)
52-Te-131,130.9085219(22),,127.60(3)
52-Te-132,131.908524(12),,127.60(3)
52-Te-133,132.910940(80),,127.60(3)
52-Te-134,133.911540(40),,127.60(3)
52-Te-135,134.91645(10),,127.60(3)
52-Te-136,135.920100(50),,127.60(3)
52-Te-137,136.92532(13),,127.60(3)
52-Te-138,137.92922(22)#,,127.60(3)
52-Te-139,138.93473(43)#,,127.60(3)
52-Te-140,139.93870(54)#,,127.60(3)
52-Te-141,140.94439(54)#,,127.60(3)
52-Te-142,141.94850(64)#,,127.60(3)
53-I-108,107.94329(39)#,,126.90447(3)
53-I-109,108.93819(16),,126.90447(3)
53-I-110,109.93521(33)#,,126.90447(3)
53-I-111,110.93028(32)#,,126.90447(3)
53-I-112,111.92797(23)#,,126.90447(3)
53-I-113,112.923640(60),,126.90447(3)
53-I-114,113.92185(32)#,,126.90447(3)
53-I-115,114.91792(50)#,,126.90447(3)
53-I-116,115.91674(15),,126.90447(3)
53-I-117,116.913650(80),,126.90447(3)
53-I-118,117.913380(80),,126.90447(3)
53-I-119,118.910180(70),,126.90447(3)
53-I-120,119.910048(20),,126.90447(3)
53-I-121,120.907366(12),,126.90447(3)
53-I-122,121.907592(6),,126.90447(3)
53-I-123,122.905598(4),,126.90447(3)
53-I-124,123.9062114(26),,126.90447(3)
53-I-125,124.9046241(20),,126.90447(3)
53-I-126,125.905619(4),,126.90447(3)
53-I-127,126.904468(4),100,126.90447(3)
53-I-128,127.905805(4),,126.90447(3)
53-I-129,128.904987(4),,126.90447(3)
53-I-130,129.906674(4),,126.90447(3)
53-I-131,130.9061242(12),,126.90447(3)
53-I-132,131.907995(11),,126.90447(3)
53-I-133,132.907806(28),,126.90447(3)
53-I-134,133.909877(16),,126.90447(3)
53-I-135,134.910050(25),,126.90447(3)
53-I-136,135.914660(50),,126.90447(3)
53-I-137,136.917873(30),,126.90447(3)
53-I-138,137.922380(90),,126.90447(3)
53-I-139,138.926090(30),,126.90447(3)
53-I-140,139.93121(23)#,,126.90447(3)
53-I-141,140.93483(32)#,,126.90447(3)
53-I-142,141.94018(43)#,,126.90447(3)
53-I-143,142.94407(43)#,,126.90447(3)
53-I-144,143.94961(54)#,,126.90447(3)
54-Xe-110,109.94448(43)#,,131.293(6)
54-Xe-111,110.94163(33)#,,131.293(6)
54-Xe-112,111.93567(16),,131.293(6)
54-Xe-113,112.93338(10),,131.293(6)
54-Xe-114,113.92815(22)#,,131.293(6)
54-Xe-115,114.92654(26)#,,131.293(6)
54-Xe-116,115.92174(26)#,,131.293(6)
54-Xe-117,116.92056(19),,131.293(6)
54-Xe-118,117.91657(107),,131.293(6)
54-Xe-119,118.91555(13),,131.293(6)
54-Xe-120,119.912150(50),,131.293(6)
54-Xe-121,120.911386(26),,131.293(6)
54-Xe-122,121.908550(90),,131.293(6)
54-Xe-123,122.908471(17),,131.293(6)
54-Xe-124,123.9058958(21),0.09(1),131.293(6)
54-Xe-125,124.9063982(21),,131.293(6)
54-Xe-126,125.904269(7),0.09(1),131.293(6)
54-Xe-127,126.905180(4),,131.293(6)
54-Xe-128,127.9035304(15),1.92(3),131.293(6)
54-Xe-129,128.9047795(9),26.44(24),131.293(6)
54-Xe-130,129.9035079(10),4.08(2),131.293(6)
54-Xe-131,130.9050819(10),21.18(3),131.293(6)
54-Xe-132,131.9041545(12),26.89(6),131.293(6)
54-Xe-133,132.905906(4),,131.293(6)
54-Xe-134,133.9053945(9),10.44(10),131.293(6)
54-Xe-135,134.907207(11),,131.293(6)
54-Xe-136,135.907220(8),8.87(16),131.293(6)
54-Xe-137,136.911563(8),,131.293(6)
54-Xe-138,137.913990(40),,131.293(6)
54-Xe-139,138.918787(23),,131.293(6)
54-Xe-140,139.921640(70),,131.293(6)
54-Xe-141,140.92665(10),,131.293(6)
54-Xe-142,141.92970(11),,131.293(6)
54-Xe-143,142.93489(24)#,,131.293(6)
54-Xe-144,143.93823(34)#,,131.293(6)
54-Xe-145,144.94367(43)#,,131.293(6)
54-Xe-146,145.94730(43)#,,131.293(6)
54-Xe-147,146.95301(54)#,,131.293(6)
55-Cs-112,111.95033(33)#,,132.90545(2)
55-Cs-113,112.94454(16),,132.90545(2)
55-Cs-114,113.94142(33)#,,132.90545(2)
55-Cs-115,114.93594(46)#,,132.90545(2)
55-Cs-116,115.93291(38),,132.90545(2)
55-Cs-117,116.928640(60),,132.90545(2)
55-Cs-118,117.926555(14),,132.90545(2)
55-Cs-119,118.922371(15),,132.90545(2)
55-Cs-120,119.920678(11),,132.90545(2)
55-Cs-121,120.917184(15),,132.90545(2)
55-Cs-122,121.916122(18),,132.90545(2)
55-Cs-123,122.912990(13),,132.90545(2)
55-Cs-124,123.912246(13),,132.90545(2)
55-Cs-125,124.909725(8),,132.90545(2)
55-Cs-126,125.909448(13),,132.90545(2)
55-Cs-127,126.907418(9),,132.90545(2)
55-Cs-128,127.907748(6),,132.90545(2)
55-Cs-129,128.906063(5),,132.90545(2)
55-Cs-130,129.906706(9),,132.90545(2)
55-Cs-131,130.905460(6),,132.90545(2)
55-Cs-132,131.906430(4),,132.90545(2)
55-Cs-133,132.905447(3),100,132.90545(2)
55-Cs-134,133.906713(3),,132.90545(2)
55-Cs-135,134.905972(3),,132.90545(2)
55-Cs-136,135.907306(4),,132.90545(2)
55-Cs-137,136.907084(3),,132.90545(2)
55-Cs-138,137.911011(10),,132.90545(2)
55-Cs-139,138.913358(5),,132.90545(2)
55-Cs-140,139.917277(9),,132.90545(2)
55-Cs-141,140.920044(11),,132.90545(2)
55-Cs-142,141.924292(11),,132.90545(2)
55-Cs-143,142.927330(24),,132.90545(2)
55-Cs-144,143.932030(30),,132.90545(2)
55-Cs-145,144.935390(50),,132.90545(2)
55-Cs-146,145.940160(90),,132.90545(2)
55-Cs-147,146.94386(16),,132.90545(2)
55-Cs-148,147.94890(63),,132.90545(2)
55-Cs-149,148.95272(32)#,,132.90545(2)
55-Cs-150,149.95797(54)#,,132.90545(2)
55-Cs-151,150.96200(75)#,,132.90545(2)
56-Ba-114,113.95094(48)#,,137.327(7)
56-Ba-115,114.94771(64)#,,137.327(7)
56-Ba-116,115.94168(54)#,,137.327(7)
56-Ba-117,116.93886(70)#,,137.327(7)
56-Ba-118,117.93344(54)#,,137.327(7)
56-Ba-119,118.93105(109),,137.327(7)
56-Ba-120,119.92605(32),,137.327(7)
56-Ba-121,120.92449(33),,137.327(7)
56-Ba-122,121.92026(32)#,,137.327(7)
56-Ba-123,122.91885(32)#,,137.327(7)
56-Ba-124,123.915088(15),,137.327(7)
56-Ba-125,124.91462(27),,137.327(7)
56-Ba-126,125.911244(15),,137.327(7)
56-Ba-127,126.91112(11),,137.327(7)
56-Ba-128,127.908309(12),,137.327(7)
56-Ba-129,128.908674(12),,137.327(7)
56-Ba-130,129.906310(7),0.106(1),137.327(7)
56-Ba-131,130.906931(7),,137.327(7)
56-Ba-132,131.905056(3),0.101(1),137.327(7)
56-Ba-133,132.906002(3),,137.327(7)
56-Ba-134,133.904503(3),2.417(18),137.327(7)
56-Ba-135,134.905683(3),6.592(12),137.327(7)
56-Ba-136,135.904570(3),7.854(24),137.327(7)
56-Ba-137,136.905821(3),11.232(24),137.327(7)
56-Ba-138,137.905241(3),71.698(42),137.327(7)
56-Ba-139,138.908835(3),,137.327(7)
56-Ba-140,139.910599(9),,137.327(7)
56-Ba-141,140.914406(9),,137.327(7)
56-Ba-142,141.916448(7),,137.327(7)
56-Ba-143,142.920617(14),,137.327(7)
56-Ba-144,143.922940(15),,137.327(7)
56-Ba-145,144.926920(60),,137.327(7)
56-Ba-146,145.930110(80),,137.327(7)
56-Ba-147,146.93399(10),,137.327(7)
56-Ba-148,147.93768(15),,137.327(7)
56-Ba-149,148.94246(43)#,,137.327(7)
56-Ba-150,149.94562(54)#,,137.327(7)
56-Ba-151,150.95070(64)#,,137.327(7)
56-Ba-152,151.95416(75)#,,137.327(7)
56-Ba-153,152.95961(97)#,,137.327(7)
57-La-117,116.95001(96)#,,138.9055(2)
57-La-118,117.94657(86)#,,138.9055(2)
57-La-119,118.94099(75)#,,138.9055(2)
57-La-120,119.93807(64)#,,138.9055(2)
57-La-121,120.93301(54)#,,138.9055(2)
57-La-122,121.93071(54)#,,138.9055(2)
57-La-123,122.92624(43)#,,138.9055(2)
57-La-124,123.92453(32)#,,138.9055(2)
57-La-125,124.92067(32)#,,138.9055(2)
57-La-126,125.91937(32)#,,138.9055(2)
57-La-127,126.91616(24)#,,138.9055(2)
57-La-128,127.91545(43),,138.9055(2)
57-La-129,128.912670(60),,138.9055(2)
57-La-130,129.91232(22)#,,138.9055(2)
57-La-131,130.91011(11),,138.9055(2)
57-La-132,131.910110(50),,138.9055(2)
57-La-133,132.90840(21),,138.9055(2)
57-La-134,133.908490(28),,138.9055(2)
57-La-135,134.906971(11),,138.9055(2)
57-La-136,135.907650(80),,138.9055(2)
57-La-137,136.906470(50),,138.9055(2)
57-La-138,137.907107(4),0.090(1),138.9055(2)
57-La-139,138.906348(3),99.910(1),138.9055(2)
57-La-140,139.909473(3),,138.9055(2)
57-La-141,140.910957(5),,138.9055(2)
57-La-142,141.914074(6),,138.9055(2)
57-La-143,142.916059(16),,138.9055(2)
57-La-144,143.919590(60),,138.9055(2)
57-La-145,144.921640(70),,138.9055(2)
57-La-146,145.925700(80),,138.9055(2)
57-La-147,146.927820(80),,138.9055(2)
57-La-148,147.93219(14),,138.9055(2)
57-La-149,148.93437(32)#,,138.9055(2)
57-La-150,149.93857(43)#,,138.9055(2)
57-La-151,150.94156(54)#,,138.9055(2)
57-La-152,151.94611(64)#,,138.9055(2)
57-La-153,152.94945(75)#,,138.9055(2)
57-La-154,153.95440(86)#,,138.9055(2)
57-La-155,154.95813(97)#,,138.9055(2)
58-Ce-119,118.95276(97)#,,140.116(1)
58-Ce-120,119.94664(86)#,,140.116(1)
58-Ce-121,120.94367(75)#,,140.116(1)
58-Ce-122,121.93801(64)#,,140.116(1)
58-Ce-123,122.93551(54)#,,140.116(1)
58-Ce-124,123.93052(54)#,,140.116(1)
58-Ce-125,124.92854(43)#,,140.116(1)
58-Ce-126,125.92410(43)#,,140.116(1)
58-Ce-127,126.92275(32)#,,140.116(1)
58-Ce-128,127.91887(32)#,,140.116(1)
58-Ce-129,128.91809(22)#,,140.116(1)
58-Ce-130,129.91469(66)#,,140.116(1)
58-Ce-131,130.91442(44),,140.116(1)
58-Ce-132,131.91149(21)#,,140.116(1)
58-Ce-133,132.91155(21)#,,140.116(1)
58-Ce-134,133.90903(22),,140.116(1)
58-Ce-135,134.909146(12),,140.116(1)
58-Ce-136,135.907140(50),0.185(2),140.116(1)
58-Ce-137,136.907780(50),,140.116(1)
58-Ce-138,137.905986(11),0.251(2),140.116(1)
58-Ce-139,138.906647(8),,140.116(1)
58-Ce-140,139.905434(3),88.450(51),140.116(1)
58-Ce-141,140.908271(3),,140.116(1)
58-Ce-142,141.909240(4),11.114(51),140.116(1)
58-Ce-143,142.912381(4),,140.116(1)
58-Ce-144,143.913643(4),,140.116(1)
58-Ce-145,144.917230(40),,140.116(1)
58-Ce-146,145.918690(70),,140.116(1)
58-Ce-147,146.922510(60),,140.116(1)
58-Ce-148,147.92439(13),,140.116(1)
58-Ce-149,148.928290(80),,140.116(1)
58-Ce-150,149.93023(13),,140.116(1)
58-Ce-151,150.93404(32)#,,140.116(1)
58-Ce-152,151.93638(43)#,,140.116(1)
58-Ce-153,152.94058(54)#,,140.116(1)
58-Ce-154,153.94332(64)#,,140.116(1)
58-Ce-155,154.94804(75)#,,140.116(1)
58-Ce-156,155.95126(86)#,,140.116(1)
58-Ce-157,156.95634(97)#,,140.116(1)
59-Pr-121,120.95536(86)#,,140.90765(2)
59-Pr-122,121.95165(86)#,,140.90765(2)
59-Pr-123,122.94596(75)#,,140.90765(2)
59-Pr-124,123.94296(64)#,,140.90765(2)
59-Pr-125,124.93783(54)#,,140.90765(2)
59-Pr-126,125.93531(54)#,,140.90765(2)
59-Pr-127,126.93083(43)#,,140.90765(2)
59-Pr-128,127.92880(43)#,,140.90765(2)
59-Pr-129,128.92486(32)#,,140.90765(2)
59-Pr-130,129.92338(32)#,,140.90765(2)
59-Pr-131,130.92006(47),,140.90765(2)
59-Pr-132,131.91912(21)#,,140.90765(2)
59-Pr-133,132.91620(21)#,,140.90765(2)
59-Pr-134,133.91567(32)#,,140.90765(2)
59-Pr-135,134.91314(16),,140.90765(2)
59-Pr-136,135.912650(50),,140.90765(2)
59-Pr-137,136.910680(50),,140.90765(2)
59-Pr-138,137.910749(16),,140.90765(2)
59-Pr-139,138.908932(9),,140.90765(2)
59-Pr-140,139.909071(7),,140.90765(2)
59-Pr-141,140.907648(3),100,140.90765(2)
59-Pr-142,141.910040(3),,140.90765(2)
59-Pr-143,142.910812(3),,140.90765(2)
59-Pr-144,143.913301(4),,140.90765(2)
59-Pr-145,144.914507(8),,140.90765(2)
59-Pr-146,145.917590(60),,140.90765(2)
59-Pr-147,146.918980(40),,140.90765(2)
59-Pr-148,147.92218(10),,140.90765(2)
59-Pr-149,148.923791(11),,140.90765(2)
59-Pr-150,149.927000(90),,140.90765(2)
59-Pr-151,150.928230(40),,140.90765(2)
59-Pr-152,151.93160(32)#,,140.90765(2)
59-Pr-153,152.93365(32)#,,140.90765(2)
59-Pr-154,153.93739(43)#,,140.90765(2)
59-Pr-155,154.93999(54)#,,140.90765(2)
59-Pr-156,155.94412(64)#,,140.90765(2)
59-Pr-157,156.94717(75)#,,140.90765(2)
59-Pr-158,157.95178(86)#,,140.90765(2)
59-Pr-159,158.95523(97)#,,140.90765(2)
60-Nd-126,125.94307(75)#,,144.24(3)
60-Nd-127,126.94050(64)#,,144.24(3)
60-Nd-128,127.93539(64)#,,144.24(3)
60-Nd-129,128.93325(39)#,,144.24(3)
60-Nd-130,129.92878(54)#,,144.24(3)
60-Nd-131,130.92710(50),,144.24(3)
60-Nd-132,131.92312(32)#,,144.24(3)
60-Nd-133,132.92221(32)#,,144.24(3)
60-Nd-134,133.91865(36)#,,144.24(3)
60-Nd-135,134.91824(22)#,,144.24(3)
60-Nd-136,135.915020(60),,144.24(3)
60-Nd-137,136.914640(80),,144.24(3)
60-Nd-138,137.91193(22)#,,144.24(3)
60-Nd-139,138.911920(50),,144.24(3)
60-Nd-140,139.909310(21),,144.24(3)
60-Nd-141,140.909605(4),,144.24(3)
60-Nd-142,141.907719(3),27.2(5),144.24(3)
60-Nd-143,142.909810(3),12.2(2),144.24(3)
60-Nd-144,143.910083(3),23.8(3),144.24(3)
60-Nd-145,144.912569(3),8.3(1),144.24(3)
60-Nd-146,145.913112(3),17.2(3),144.24(3)
60-Nd-147,146.916096(3),,144.24(3)
60-Nd-148,147.916889(3),5.7(1),144.24(3)
60-Nd-149,148.920144(3),,144.24(3)
60-Nd-150,149.920887(4),5.6(2),144.24(3)
60-Nd-151,150.923825(4),,144.24(3)
60-Nd-152,151.924680(30),,144.24(3)
60-Nd-153,152.927695(29),,144.24(3)
60-Nd-154,153.92948(12),,144.24(3)
60-Nd-155,154.93263(16),,144.24(3)
60-Nd-156,155.93520(43)#,,144.24(3)
60-Nd-157,156.93927(54)#,,144.24(3)
60-Nd-158,157.94187(64)#,,144.24(3)
60-Nd-159,158.94639(75)#,,144.24(3)
60-Nd-160,159.94939(86)#,,144.24(3)
60-Nd-161,160.95433(97)#,,144.24(3)
61-Pm-128,127.94826(97)#,,[145]
61-Pm-129,128.94316(86)#,,[145]
61-Pm-130,129.94045(75)#,,[145]
61-Pm-131,130.93580(64)#,,[145]
61-Pm-132,131.93375(54)#,,[145]
61-Pm-133,132.92972(54)#,,[145]
61-Pm-134,133.92849(42)#,,[145]
61-Pm-135,134.92462(35)#,,[145]
61-Pm-136,135.92345(22),,[145]
61-Pm-137,136.92071(15)#,,[145]
61-Pm-138,137.91945(34)#,,[145]
61-Pm-139,138.916760(60),,[145]
61-Pm-140,139.915800(30),,[145]
61-Pm-141,140.913607(29),,[145]
61-Pm-142,141.912950(50),,[145]
61-Pm-143,142.910928(4),,[145]
61-Pm-144,143.912586(4),,[145]
61-Pm-145,144.912744(4),,[145]
61-Pm-146,145.914692(5),,[145]
61-Pm-147,146.915134(3),,[145]
61-Pm-148,147.917468(7),,[145]
61-Pm-149,148.918329(5),,[145]
61-Pm-150,149.920979(22),,[145]
61-Pm-151,150.921203(6),,[145]
61-Pm-152,151.923490(80),,[145]
61-Pm-153,152.924113(12),,[145]
61-Pm-154,153.926550(80),,[145]
61-Pm-155,154.928100(30),,[145]
61-Pm-156,155.931060(40),,[145]
61-Pm-157,156.93320(32)#,,[145]
61-Pm-158,157.93669(43)#,,[145]
61-Pm-159,158.93913(54)#,,[145]
61-Pm-160,159.94299(64)#,,[145]
61-Pm-161,160.94586(75)#,,[145]
61-Pm-162,161.95029(86)#,,[145]
61-Pm-163,162.95352(97)#,,[145]
62-Sm-130,129.94863(97)#,,150.36(3)
62-Sm-131,130.94589(97)#,,150.36(3)
62-Sm-132,131.94082(75)#,,150.36(3)
62-Sm-133,132.93873(64)#,,150.36(3)
62-Sm-134,133.93402(54)#,,150.36(3)
62-Sm-135,134.93235(54)#,,150.36(3)
62-Sm-136,135.92830(43)#,,150.36(3)
62-Sm-137,136.92705(12),,150.36(3)
62-Sm-138,137.92354(32)#,,150.36(3)
62-Sm-139,138.922302(16),,150.36(3)
62-Sm-140,139.918991(16),,150.36(3)
62-Sm-141,140.918469(13),,150.36(3)
62-Sm-142,141.915193(11),,150.36(3)
62-Sm-143,142.914624(4),,150.36(3)
62-Sm-144,143.911995(4),3.07(7),150.36(3)
62-Sm-145,144.913406(4),,150.36(3)
62-Sm-146,145.913037(4),,150.36(3)
62-Sm-147,146.914893(3),14.99(18),150.36(3)
62-Sm-148,147.914818(3),11.24(10),150.36(3)
62-Sm-149,148.917180(3),13.82(7),150.36(3)
62-Sm-150,149.917271(3),7.38(1),150.36(3)
62-Sm-151,150.919928(3),,150.36(3)
62-Sm-152,151.919728(3),26.75(16),150.36(3)
62-Sm-153,152.922094(3),,150.36(3)
62-Sm-154,153.922205(3),22.75(29),150.36(3)
62-Sm-155,154.924636(3),,150.36(3)
62-Sm-156,155.925526(10),,150.36(3)
62-Sm-157,156.928350(50),,150.36(3)
62-Sm-158,157.929990(80),,150.36(3)
62-Sm-159,158.93320(32)#,,150.36(3)
62-Sm-160,159.93514(43)#,,150.36(3)
62-Sm-161,160.93883(54)#,,150.36(3)
62-Sm-162,161.94122(64)#,,150.36(3)
62-Sm-163,162.94536(75)#,,150.36(3)
62-Sm-164,163.94828(86)#,,150.36(3)
62-Sm-165,164.95298(97)#,,150.36(3)
63-Eu-132,131.95416(97)#,,151.964(1)
63-Eu-133,132.94890(97)#,,151.964(1)
63-Eu-134,133.94632(75)#,,151.964(1)
63-Eu-135,134.94172(64)#,,151.964(1)
63-Eu-136,135.93950(54)#,,151.964(1)
63-Eu-137,136.93521(54)#,,151.964(1)
63-Eu-138,137.93345(43)#,,151.964(1)
63-Eu-139,138.92984(16)#,,151.964(1)
63-Eu-140,139.928080(60),,151.964(1)
63-Eu-141,140.924890(30),,151.964(1)
63-Eu-142,141.923400(30),,151.964(1)
63-Eu-143,142.920287(14),,151.964(1)
63-Eu-144,143.918774(19),,151.964(1)
63-Eu-145,144.916261(5),,151.964(1)
63-Eu-146,145.917200(8),,151.964(1)
63-Eu-147,146.916741(4),,151.964(1)
63-Eu-148,147.918154(19),,151.964(1)
63-Eu-149,148.917926(5),,151.964(1)
63-Eu-150,149.919698(8),,151.964(1)
63-Eu-151,150.919846(3),47.81(3),151.964(1)
63-Eu-152,151.921740(3),,151.964(1)
63-Eu-153,152.921226(3),52.19(3),151.964(1)
63-Eu-154,153.922975(3),,151.964(1)
63-Eu-155,154.922889(3),,151.964(1)
63-Eu-156,155.924751(6),,151.964(1)
63-Eu-157,156.925419(7),,151.964(1)
63-Eu-158,157.927840(80),,151.964(1)
63-Eu-159,158.929084(9),,151.964(1)
63-Eu-160,159.93197(22)#,,151.964(1)
63-Eu-161,160.93368(32)#,,151.964(1)
63-Eu-162,161.93704(43)#,,151.964(1)
63-Eu-163,162.93921(54)#,,151.964(1)
63-Eu-164,163.94299(64)#,,151.964(1)
63-Eu-165,164.94572(75)#,,151.964(1)
63-Eu-166,165.94997(86)#,,151.964(1)
63-Eu-167,166.95305(97)#,,151.964(1)
64-Gd-136,135.94707(75)#,,157.25(3)
64-Gd-137,136.94465(64)#,,157.25(3)
64-Gd-138,137.93997(54)#,,157.25(3)
64-Gd-139,138.93808(54)#,,157.25(3)
64-Gd-140,139.93395(43)#,,157.25(3)
64-Gd-141,140.93221(32)#,,157.25(3)
64-Gd-142,141.92823(32)#,,157.25(3)
64-Gd-143,142.92674(22),,157.25(3)
64-Gd-144,143.92279(22)#,,157.25(3)
64-Gd-145,144.921690(40),,157.25(3)
64-Gd-146,145.918305(6),,157.25(3)
64-Gd-147,146.919089(4),,157.25(3)
64-Gd-148,147.918110(4),,157.25(3)
64-Gd-149,148.919336(5),,157.25(3)
64-Gd-150,149.918655(7),,157.25(3)
64-Gd-151,150.920344(4),,157.25(3)
64-Gd-152,151.919788(3),0.20(1),157.25(3)
64-Gd-153,152.921746(3),,157.25(3)
64-Gd-154,153.920862(3),2.18(3),157.25(3)
64-Gd-155,154.922619(3),14.80(12),157.25(3)
64-Gd-156,155.922120(3),20.47(9),157.25(3)
64-Gd-157,156.923957(3),15.65(2),157.25(3)
64-Gd-158,157.924101(3),24.84(7),157.25(3)
64-Gd-159,158.926385(3),,157.25(3)
64-Gd-160,159.927051(3),21.86(19),157.25(3)
64-Gd-161,160.929666(3),,157.25(3)
64-Gd-162,161.930981(5),,157.25(3)
64-Gd-163,162.93399(32)#,,157.25(3)
64-Gd-164,163.93586(43)#,,157.25(3)
64-Gd-165,164.93938(54)#,,157.25(3)
64-Gd-166,165.94160(64)#,,157.25(3)
64-Gd-167,166.94557(64)#,,157.25(3)
64-Gd-168,167.94836(75)#,,157.25(3)
64-Gd-169,168.95287(86)#,,157.25(3)
65-Tb-138,137.95287(86)#,,158.92534(2)
65-Tb-139,138.94803(75)#,,158.92534(2)
65-Tb-140,139.94554(96)#,,158.92534(2)
65-Tb-141,140.94116(64)#,,158.92534(2)
65-Tb-142,141.93886(82)#,,158.92534(2)
65-Tb-143,142.93475(43)#,,158.92534(2)
65-Tb-144,143.93253(32)#,,158.92534(2)
65-Tb-145,144.92888(24)#,,158.92534(2)
65-Tb-146,145.927180(50),,158.92534(2)
65-Tb-147,146.924037(13),,158.92534(2)
65-Tb-148,147.924300(30),,158.92534(2)
65-Tb-149,148.923242(5),,158.92534(2)
65-Tb-150,149.923654(9),,158.92534(2)
65-Tb-151,150.923098(5),,158.92534(2)
65-Tb-152,151.924070(40),,158.92534(2)
65-Tb-153,152.923431(5),,158.92534(2)
65-Tb-154,153.924690(50),,158.92534(2)
65-Tb-155,154.923500(13),,158.92534(2)
65-Tb-156,155.924744(5),,158.92534(2)
65-Tb-157,156.924021(3),,158.92534(2)
65-Tb-158,157.925410(3),,158.92534(2)
65-Tb-159,158.925343(3),100,158.92534(2)
65-Tb-160,159.927164(3),,158.92534(2)
65-Tb-161,160.927566(3),,158.92534(2)
65-Tb-162,161.929480(40),,158.92534(2)
65-Tb-163,162.930644(5),,158.92534(2)
65-Tb-164,163.93335(11),,158.92534(2)
65-Tb-165,164.93488(21)#,,158.92534(2)
65-Tb-166,165.93805(32)#,,158.92534(2)
65-Tb-167,166.94005(43)#,,158.92534(2)
65-Tb-168,167.94364(54)#,,158.92534(2)
65-Tb-169,168.94622(64)#,,158.92534(2)
65-Tb-170,169.95025(75)#,,158.92534(2)
65-Tb-171,170.95330(86)#,,158.92534(2)
66-Dy-140,139.95379(97)#,,162.500(1)
66-Dy-141,140.95119(75)#,,162.500(1)
66-Dy-142,141.94627(85)#,,162.500(1)
66-Dy-143,142.94383(54)#,,162.500(1)
66-Dy-144,143.93907(43)#,,162.500(1)
66-Dy-145,144.93695(32)#,,162.500(1)
66-Dy-146,145.93272(12),,162.500(1)
66-Dy-147,146.930880(60),,162.500(1)
66-Dy-148,147.927180(30),,162.500(1)
66-Dy-149,148.927334(12),,162.500(1)
66-Dy-150,149.925580(6),,162.500(1)
66-Dy-151,150.926180(5),,162.500(1)
66-Dy-152,151.924714(6),,162.500(1)
66-Dy-153,152.925761(5),,162.500(1)
66-Dy-154,153.924422(9),,162.500(1)
66-Dy-155,154.925749(13),,162.500(1)
66-Dy-156,155.924278(7),0.06(1),162.500(1)
66-Dy-157,156.925461(7),,162.500(1)
66-Dy-158,157.924405(4),0.10(1),162.500(1)
66-Dy-159,158.925736(3),,162.500(1)
66-Dy-160,159.925194(3),2.34(8),162.500(1)
66-Dy-161,160.926930(3),18.91(24),162.500(1)
66-Dy-162,161.926795(3),25.51(26),162.500(1)
66-Dy-163,162.928728(3),24.90(16),162.500(1)
66-Dy-164,163.929171(3),28.18(37),162.500(1)
66-Dy-165,164.931700(3),,162.500(1)
66-Dy-166,165.932803(3),,162.500(1)
66-Dy-167,166.935650(60),,162.500(1)
66-Dy-168,167.93723(32)#,,162.500(1)
66-Dy-169,168.94030(32),,162.500(1)
66-Dy-170,169.94267(43)#,,162.500(1)
66-Dy-171,170.94648(54)#,,162.500(1)
66-Dy-172,171.94911(64)#,,162.500(1)
66-Dy-173,172.95344(75)#,,162.500(1)
67-Ho-142,141.95986(107)#,,164.93032(2)
67-Ho-143,142.95469(75)#,,164.93032(2)
67-Ho-144,143.95164(64)#,,164.93032(2)
67-Ho-145,144.94688(64)#,,164.93032(2)
67-Ho-146,145.94410(54)#,,164.93032(2)
67-Ho-147,146.93984(43)#,,164.93032(2)
67-Ho-148,147.93727(29)#,,164.93032(2)
67-Ho-149,148.933790(23),,164.93032(2)
67-Ho-150,149.93335(11)#,,164.93032(2)
67-Ho-151,150.931681(13),,164.93032(2)
67-Ho-152,151.931740(30),,164.93032(2)
67-Ho-153,152.930195(6),,164.93032(2)
67-Ho-154,153.930596(10),,164.93032(2)
67-Ho-155,154.929079(25),,164.93032(2)
67-Ho-156,155.92971(22)#,,164.93032(2)
67-Ho-157,156.928190(50),,164.93032(2)
67-Ho-158,157.928950(30),,164.93032(2)
67-Ho-159,158.927709(4),,164.93032(2)
67-Ho-160,159.928726(16),,164.93032(2)
67-Ho-161,160.927852(4),,164.93032(2)
67-Ho-162,161.929092(5),,164.93032(2)
67-Ho-163,162.928730(3),,164.93032(2)
67-Ho-164,163.930231(3),,164.93032(2)
67-Ho-165,164.930319(3),100,164.93032(2)
67-Ho-166,165.932281(3),,164.93032(2)
67-Ho-167,166.933126(6),,164.93032(2)
67-Ho-168,167.935500(30),,164.93032(2)
67-Ho-169,168.936868(22),,164.93032(2)
67-Ho-170,169.939610(50),,164.93032(2)
67-Ho-171,170.94146(64),,164.93032(2)
67-Ho-172,171.94482(43)#,,164.93032(2)
67-Ho-173,172.94729(43)#,,164.93032(2)
67-Ho-174,173.95115(54)#,,164.93032(2)
67-Ho-175,174.95405(64)#,,164.93032(2)
68-Er-144,143.96059(86)#,,167.259(3)
68-Er-145,144.95746(75)#,,167.259(3)
68-Er-146,145.95212(64)#,,167.259(3)
68-Er-147,146.94931(54)#,,167.259(3)
68-Er-148,147.94444(43)#,,167.259(3)
68-Er-149,148.94217(51)#,,167.259(3)
68-Er-150,149.93776(11)#,,167.259(3)
68-Er-151,150.93746(32)#,,167.259(3)
68-Er-152,151.935080(30),,167.259(3)
68-Er-153,152.935093(12),,167.259(3)
68-Er-154,153.932777(6),,167.259(3)
68-Er-155,154.933200(50),,167.259(3)
68-Er-156,155.931020(80),,167.259(3)
68-Er-157,156.931950(90),,167.259(3)
68-Er-158,157.92991(11)#,,167.259(3)
68-Er-159,158.930681(5),,167.259(3)
68-Er-160,159.929080(50),,167.259(3)
68-Er-161,160.930001(10),,167.259(3)
68-Er-162,161.928775(4),0.14(1),167.259(3)
68-Er-163,162.930029(6),,167.259(3)
68-Er-164,163.929197(4),1.61(3),167.259(3)
68-Er-165,164.930723(4),,167.259(3)
68-Er-166,165.930290(3),33.61(35),167.259(3)
68-Er-167,166.932045(3),22.93(17),167.259(3)
68-Er-168,167.932368(3),26.78(26),167.259(3)
68-Er-169,168.934588(3),,167.259(3)
68-Er-170,169.935460(3),14.93(27),167.259(3)
68-Er-171,170.938026(3),,167.259(3)
68-Er-172,171.939352(5),,167.259(3)
68-Er-173,172.94240(21)#,,167.259(3)
68-Er-174,173.94434(32)#,,167.259(3)
68-Er-175,174.94793(43)#,,167.259(3)
68-Er-176,175.95029(43)#,,167.259(3)
68-Er-177,176.95437(64)#,,167.259(3)
69-Tm-146,145.96650(75)#,,168.93421(2)
69-Tm-147,146.96108(64)#,,168.93421(2)
69-Tm-148,147.95755(75)#,,168.93421(2)
69-Tm-149,148.95265(64)#,,168.93421(2)
69-Tm-150,149.94967(54)#,,168.93421(2)
69-Tm-151,150.94543(15)#,,168.93421(2)
69-Tm-152,151.94430(32)#,,168.93421(2)
69-Tm-153,152.942028(23),,168.93421(2)
69-Tm-154,153.94142(12)#,,168.93421(2)
69-Tm-155,154.939192(14),,168.93421(2)
69-Tm-156,155.939010(60),,168.93421(2)
69-Tm-157,156.93676(11),,168.93421(2)
69-Tm-158,157.93700(13)#,,168.93421(2)
69-Tm-159,158.934810(70),,168.93421(2)
69-Tm-160,159.93509(33),,168.93421(2)
69-Tm-161,160.93340(10),,168.93421(2)
69-Tm-162,161.933970(30),,168.93421(2)
69-Tm-163,162.932648(7),,168.93421(2)
69-Tm-164,163.933451(20),,168.93421(2)
69-Tm-165,164.932432(4),,168.93421(2)
69-Tm-166,165.933553(12),,168.93421(2)
69-Tm-167,166.932849(3),,168.93421(2)
69-Tm-168,167.934170(4),,168.93421(2)
69-Tm-169,168.934211(3),100,168.93421(2)
69-Tm-170,169.935798(3),,168.93421(2)
69-Tm-171,170.936426(3),,168.93421(2)
69-Tm-172,171.938396(7),,168.93421(2)
69-Tm-173,172.939600(6),,168.93421(2)
69-Tm-174,173.942160(50),,168.93421(2)
69-Tm-175,174.943830(50),,168.93421(2)
69-Tm-176,175.94699(11),,168.93421(2)
69-Tm-177,176.94904(32)#,,168.93421(2)
69-Tm-178,177.95264(43)#,,168.93421(2)
69-Tm-179,178.95534(54)#,,168.93421(2)
70-Yb-148,147.96676(86)#,,173.04(3)
70-Yb-149,148.96348(75)#,,173.04(3)
70-Yb-150,149.95799(64)#,,173.04(3)
70-Yb-151,150.95525(34)#,,173.04(3)
70-Yb-152,151.95017(38)#,,173.04(3)
70-Yb-153,152.94921(32)#,,173.04(3)
70-Yb-154,153.94624(11)#,,173.04(3)
70-Yb-155,154.94579(32)#,,173.04(3)
70-Yb-156,155.942850(40),,173.04(3)
70-Yb-157,156.942660(60),,173.04(3)
70-Yb-158,157.939858(11),,173.04(3)
70-Yb-159,158.94015(10),,173.04(3)
70-Yb-160,159.93756(22)#,,173.04(3)
70-Yb-161,160.93785(24)#,,173.04(3)
70-Yb-162,161.93575(22)#,,173.04(3)
70-Yb-163,162.93627(11),,173.04(3)
70-Yb-164,163.93452(11)#,,173.04(3)
70-Yb-165,164.935398(22),,173.04(3)
70-Yb-166,165.933880(9),,173.04(3)
70-Yb-167,166.934947(5),,173.04(3)
70-Yb-168,167.933894(5),0.13(1),173.04(3)
70-Yb-169,168.935187(5),,173.04(3)
70-Yb-170,169.934759(3),3.04(15),173.04(3)
70-Yb-171,170.936322(3),14.28(57),173.04(3)
70-Yb-172,171.9363777(30),21.83(67),173.04(3)
70-Yb-173,172.9382068(30),16.13(27),173.04(3)
70-Yb-174,173.9388581(30),31.83(92),173.04(3)
70-Yb-175,174.9412725(30),,173.04(3)
70-Yb-176,175.942568(3),12.76(41),173.04(3)
70-Yb-177,176.945257(3),,173.04(3)
70-Yb-178,177.946643(11),,173.04(3)
70-Yb-179,178.95017(32)#,,173.04(3)
70-Yb-180,179.95233(43)#,,173.04(3)
70-Yb-181,180.95615(43)#,,173.04(3)
71-Lu-150,149.97267(75)#,,174.967(1)
71-Lu-151,150.96715(65)#,,174.967(1)
71-Lu-152,151.96361(75)#,,174.967(1)
71-Lu-153,152.95869(64)#,,174.967(1)
71-Lu-154,153.95710(54)#,,174.967(1)
71-Lu-155,154.95423(14)#,,174.967(1)
71-Lu-156,155.95291(32)#,,174.967(1)
71-Lu-157,156.950102(23),,174.967(1)
71-Lu-158,157.94917(13)#,,174.967(1)
71-Lu-159,158.946620(50),,174.967(1)
71-Lu-160,159.94602(25)#,,174.967(1)
71-Lu-161,160.94354(26)#,,174.967(1)
71-Lu-162,161.94322(24)#,,174.967(1)
71-Lu-163,162.94120(24),,174.967(1)
71-Lu-164,163.94122(13)#,,174.967(1)
71-Lu-165,164.939610(90),,174.967(1)
71-Lu-166,165.93976(17),,174.967(1)
71-Lu-167,166.93831(11),,174.967(1)
71-Lu-168,167.938700(90),,174.967(1)
71-Lu-169,168.937649(6),,174.967(1)
71-Lu-170,169.938472(20),,174.967(1)
71-Lu-171,170.937910(3),,174.967(1)
71-Lu-172,171.939082(4),,174.967(1)
71-Lu-173,172.938927(3),,174.967(1)
71-Lu-174,173.9403335(30),,174.967(1)
71-Lu-175,174.9407679(28),97.41(2),174.967(1)
71-Lu-176,175.9426824(28),2.59(2),174.967(1)
71-Lu-177,176.9437550(28),,174.967(1)
71-Lu-178,177.945951(3),,174.967(1)
71-Lu-179,178.947324(6),,174.967(1)
71-Lu-180,179.949880(80),,174.967(1)
71-Lu-181,180.95197(32)#,,174.967(1)
71-Lu-182,181.95521(32)#,,174.967(1)
71-Lu-183,182.95757(32)#,,174.967(1)
71-Lu-184,183.96117(43)#,,174.967(1)
72-Hf-154,153.96425(75)#,,178.49(2)
72-Hf-155,154.96276(64)#,,178.49(2)
72-Hf-156,155.95925(38)#,,178.49(2)
72-Hf-157,156.95813(32)#,,178.49(2)
72-Hf-158,157.95465(11)#,,178.49(2)
72-Hf-159,158.95400(32)#,,178.49(2)
72-Hf-160,159.950710(40),,178.49(2)
72-Hf-161,160.950330(80),,178.49(2)
72-Hf-162,161.947203(12),,178.49(2)
72-Hf-163,162.94706(34)#,,178.49(2)
72-Hf-164,163.94442(21)#,,178.49(2)
72-Hf-165,164.94454(40)#,,178.49(2)
72-Hf-166,165.94225(32)#,,178.49(2)
72-Hf-167,166.94260(22)#,,178.49(2)
72-Hf-168,167.94063(11)#,,178.49(2)
72-Hf-169,168.941160(90),,178.49(2)
72-Hf-170,169.93965(21)#,,178.49(2)
72-Hf-171,170.94049(21)#,,178.49(2)
72-Hf-172,171.939460(50),,178.49(2)
72-Hf-173,172.94065(11)#,,178.49(2)
72-Hf-174,173.940040(3),0.16(1),178.49(2)
72-Hf-175,174.941503(3),,178.49(2)
72-Hf-176,175.9414018(29),5.26(7),178.49(2)
72-Hf-177,176.9432200(27),18.60(9),178.49(2)
72-Hf-178,177.9436977(27),27.28(7),178.49(2)
72-Hf-179,178.9458151(27),13.62(2),178.49(2)
72-Hf-180,179.9465488(27),35.08(16),178.49(2)
72-Hf-181,180.9490991(28),,178.49(2)
72-Hf-182,181.950553(7),,178.49(2)
72-Hf-183,182.953530(30),,178.49(2)
72-Hf-184,183.955450(40),,178.49(2)
72-Hf-185,184.95878(32)#,,178.49(2)
72-Hf-186,185.96092(32)#,,178.49(2)
73-Ta-156,155.97169(64)#,,180.9479(1)
73-Ta-157,156.96815(64)#,,180.9479(1)
73-Ta-158,157.96637(54)#,,180.9479(1)
73-Ta-159,158.96291(13)#,,180.9479(1)
73-Ta-160,159.96136(33)#,,180.9479(1)
73-Ta-161,160.958370(60),,180.9479(1)
73-Ta-162,161.95715(14)#,,180.9479(1)
73-Ta-163,162.954320(80),,180.9479(1)
73-Ta-164,163.95357(43)#,,180.9479(1)
73-Ta-165,164.95082(24)#,,180.9479(1)
73-Ta-166,165.95047(32)#,,180.9479(1)
73-Ta-167,166.94797(46)#,,180.9479(1)
73-Ta-168,167.94779(39)#,,180.9479(1)
73-Ta-169,168.94592(22)#,,180.9479(1)
73-Ta-170,169.94609(21)#,,180.9479(1)
73-Ta-171,170.94446(22)#,,180.9479(1)
73-Ta-172,171.94474(20),,180.9479(1)
73-Ta-173,172.94354(24)#,,180.9479(1)
73-Ta-174,173.944170(90),,180.9479(1)
73-Ta-175,174.94365(11)#,,180.9479(1)
73-Ta-176,175.94474(11),,180.9479(1)
73-Ta-177,176.944472(4),,180.9479(1)
73-Ta-178,177.94575(11),,180.9479(1)
73-Ta-179,178.945934(6),,180.9479(1)
73-Ta-180,179.947466(3),0.012(2),180.9479(1)
73-Ta-181,180.947996(3),99.988(2),180.9479(1)
73-Ta-182,181.950152(3),,180.9479(1)
73-Ta-183,182.951373(3),,180.9479(1)
73-Ta-184,183.954009(28),,180.9479(1)
73-Ta-185,184.955559(15),,180.9479(1)
73-Ta-186,185.958550(60),,180.9479(1)
73-Ta-187,186.96041(32)#,,180.9479(1)
73-Ta-188,187.96371(32)#,,180.9479(1)
74-W-158,157.97394(75)#,,183.84(1)
74-W-159,158.97228(64)#,,183.84(1)
74-W-160,159.96837(38)#,,183.84(1)
74-W-161,160.96709(33)#,,183.84(1)
74-W-162,161.96334(11)#,,183.84(1)
74-W-163,162.96253(33)#,,183.84(1)
74-W-164,163.958980(40),,183.84(1)
74-W-165,164.958340(90),,183.84(1)
74-W-166,165.955020(13),,183.84(1)
74-W-167,166.95467(33)#,,183.84(1)
74-W-168,167.95186(21)#,,183.84(1)
74-W-169,168.95176(34)#,,183.84(1)
74-W-170,169.94929(51)#,,183.84(1)
74-W-171,170.94946(30)#,,183.84(1)
74-W-172,171.94742(29)#,,183.84(1)
74-W-173,172.94783(40)#,,183.84(1)
74-W-174,173.94616(32)#,,183.84(1)
74-W-175,174.94677(21)#,,183.84(1)
74-W-176,175.94559(21)#,,183.84(1)
74-W-177,176.94662(32)#,,183.84(1)
74-W-178,177.94585(11),,183.84(1)
74-W-179,178.947072(17),,183.84(1)
74-W-180,179.946706(5),0.12(1),183.84(1)
74-W-181,180.948198(6),,183.84(1)
74-W-182,181.948206(3),26.50(16),183.84(1)
74-W-183,182.9502245(29),14.31(4),183.84(1)
74-W-184,183.9509326(29),30.64(2),183.84(1)
74-W-185,184.9534206(30),,183.84(1)
74-W-186,185.954362(3),28.43(19),183.84(1)
74-W-187,186.957158(3),,183.84(1)
74-W-188,187.958487(4),,183.84(1)
74-W-189,188.96191(21),,183.84(1)
74-W-190,189.96318(24),,183.84(1)
75-Re-160,159.98149(64)#,,186.207(1)
75-Re-161,160.97766(64)#,,186.207(1)
75-Re-162,161.97571(55)#,,186.207(1)
75-Re-163,162.97197(12)#,,186.207(1)
75-Re-164,163.97032(33)#,,186.207(1)
75-Re-165,164.967050(80),,186.207(1)
75-Re-166,165.96580(15)#,,186.207(1)
75-Re-167,166.96256(14)#,,186.207(1)
75-Re-168,167.96161(43)#,,186.207(1)
75-Re-169,168.95883(22)#,,186.207(1)
75-Re-170,169.95816(43)#,,186.207(1)
75-Re-171,170.95555(37)#,,186.207(1)
75-Re-172,171.95529(33)#,,186.207(1)
75-Re-173,172.95306(48)#,,186.207(1)
75-Re-174,173.95311(44)#,,186.207(1)
75-Re-175,174.95139(48)#,,186.207(1)
75-Re-176,175.95157(21)#,,186.207(1)
75-Re-177,176.95027(21)#,,186.207(1)
75-Re-178,177.95085(22),,186.207(1)
75-Re-179,178.949980(60),,186.207(1)
75-Re-180,179.950790(40),,186.207(1)
75-Re-181,180.950065(15),,186.207(1)
75-Re-182,181.95121(11),,186.207(1)
75-Re-183,182.950821(9),,186.207(1)
75-Re-184,183.952524(6),,186.207(1)
75-Re-185,184.9529557(30),37.40(2),186.207(1)
75-Re-186,185.954987(3),,186.207(1)
75-Re-187,186.9557508(30),62.60(2),186.207(1)
75-Re-188,187.9581123(30),,186.207(1)
75-Re-189,188.959228(9),,186.207(1)
75-Re-190,189.96182(23),,186.207(1)
75-Re-191,190.963124(11),,186.207(1)
75-Re-192,191.96596(21)#,,186.207(1)
76-Os-162,161.98382(75)#,,190.23(3)
76-Os-163,162.98205(64)#,,190.23(3)
76-Os-164,163.97793(38)#,,190.23(3)
76-Os-165,164.97648(33)#,,190.23(3)
76-Os-166,165.97253(11)#,,190.23(3)
76-Os-167,166.97155(33)#,,190.23(3)
76-Os-168,167.967830(40),,190.23(3)
76-Os-169,168.967080(90),,190.23(3)
76-Os-170,169.963570(14),,190.23(3)
76-Os-171,170.96304(33)#,,190.23(3)
76-Os-172,171.96008(21)#,,190.23(3)
76-Os-173,172.95979(33)#,,190.23(3)
76-Os-174,173.95712(50)#,,190.23(3)
76-Os-175,174.95708(32)#,,190.23(3)
76-Os-176,175.95495(22)#,,190.23(3)
76-Os-177,176.95505(30)#,,190.23(3)
76-Os-178,177.95335(22),,190.23(3)
76-Os-179,178.95395(25)#,,190.23(3)
76-Os-180,179.95235(20)#,,190.23(3)
76-Os-181,180.95327(22),,190.23(3)
76-Os-182,181.952186(27),,190.23(3)
76-Os-183,182.95311(11)#,,190.23(3)
76-Os-184,183.952491(3),0.02(1),190.23(3)
76-Os-185,184.954043(3),,190.23(3)
76-Os-186,185.953838(3),1.59(3),190.23(3)
76-Os-187,186.9557479(30),1.96(2),190.23(3)
76-Os-188,187.9558360(30),13.24(8),190.23(3)
76-Os-189,188.9581449(30),16.15(5),190.23(3)
76-Os-190,189.958445(3),26.26(2),190.23(3)
76-Os-191,190.960928(3),,190.23(3)
76-Os-192,191.961479(4),40.78(19),190.23(3)
76-Os-193,192.964148(4),,190.23(3)
76-Os-194,193.965179(4),,190.23(3)
76-Os-195,194.96812(54),,190.23(3)
76-Os-196,195.969620(40),,190.23(3)
77-Ir-165,164.98758(43)#,,192.217(3)
77-Ir-166,165.98551(55)#,,192.217(3)
77-Ir-167,166.98154(11)#,,192.217(3)
77-Ir-168,167.97997(35)#,,192.217(3)
77-Ir-169,168.97639(10),,192.217(3)
77-Ir-170,169.97503(16)#,,192.217(3)
77-Ir-171,170.97178(14)#,,192.217(3)
77-Ir-172,171.97064(43)#,,192.217(3)
77-Ir-173,172.96771(25)#,,192.217(3)
77-Ir-174,173.96680(43)#,,192.217(3)
77-Ir-175,174.96428(37)#,,192.217(3)
77-Ir-176,175.96351(33)#,,192.217(3)
77-Ir-177,176.96117(48)#,,192.217(3)
77-Ir-178,177.96108(39)#,,192.217(3)
77-Ir-179,178.95915(43)#,,192.217(3)
77-Ir-180,179.95925(21)#,,192.217(3)
77-Ir-181,180.95764(23),,192.217(3)
77-Ir-182,181.95813(15),,192.217(3)
77-Ir-183,182.95681(15)#,,192.217(3)
77-Ir-184,183.95739(29),,192.217(3)
77-Ir-185,184.95659(21)#,,192.217(3)
77-Ir-186,185.957951(22),,192.217(3)
77-Ir-187,186.957361(7),,192.217(3)
77-Ir-188,187.958852(8),,192.217(3)
77-Ir-189,188.958716(14),,192.217(3)
77-Ir-190,189.96059(21),,192.217(3)
77-Ir-191,190.960591(3),37.3(2),192.217(3)
77-Ir-192,191.962602(3),,192.217(3)
77-Ir-193,192.962924(3),62.7(2),192.217(3)
77-Ir-194,193.965076(3),,192.217(3)
77-Ir-195,194.965977(3),,192.217(3)
77-Ir-196,195.968380(40),,192.217(3)
77-Ir-197,196.969636(22),,192.217(3)
77-Ir-198,197.97228(21)#,,192.217(3)
77-Ir-199,198.973790(40),,192.217(3)
78-Pt-168,167.98804(38)#,,195.078(2)
78-Pt-169,168.98642(34)#,,195.078(2)
78-Pt-170,169.98233(11)#,,195.078(2)
78-Pt-171,170.98125(34)#,,195.078(2)
78-Pt-172,171.977380(40),,195.078(2)
78-Pt-173,172.97650(11),,195.078(2)
78-Pt-174,173.972811(14),,195.078(2)
78-Pt-175,174.97228(33)#,,195.078(2)
78-Pt-176,175.96900(21)#,,195.078(2)
78-Pt-177,176.96845(33)#,,195.078(2)
78-Pt-178,177.96571(50)#,,195.078(2)
78-Pt-179,178.96548(32)#,,195.078(2)
78-Pt-180,179.96322(22)#,,195.078(2)
78-Pt-181,180.96318(30)#,,195.078(2)
78-Pt-182,181.96127(22),,195.078(2)
78-Pt-183,182.96173(25)#,,195.078(2)
78-Pt-184,183.95990(20)#,,195.078(2)
78-Pt-185,184.96075(22),,195.078(2)
78-Pt-186,185.959430(30),,195.078(2)
78-Pt-187,186.96056(20)#,,195.078(2)
78-Pt-188,187.959396(6),,195.078(2)
78-Pt-189,188.960832(12),,195.078(2)
78-Pt-190,189.959930(7),0.014(1),195.078(2)
78-Pt-191,190.961685(5),,195.078(2)
78-Pt-192,191.961035(4),0.782(7),195.078(2)
78-Pt-193,192.962985(3),,195.078(2)
78-Pt-194,193.962664(3),32.967(99),195.078(2)
78-Pt-195,194.964774(3),33.832(10),195.078(2)
78-Pt-196,195.964935(3),25.242(41),195.078(2)
78-Pt-197,196.967323(3),,195.078(2)
78-Pt-198,197.967876(4),7.163(55),195.078(2)
78-Pt-199,198.970576(5),,195.078(2)
78-Pt-200,199.971424(22),,195.078(2)
78-Pt-201,200.974500(50),,195.078(2)
78-Pt-202,201.97574(32)#,,195.078(2)
79-Au-171,170.99177(27)#,,196.96655(2)
79-Au-172,171.99011(36)#,,196.96655(2)
79-Au-173,172.98640(11),,196.96655(2)
79-Au-174,173.98492(16)#,,196.96655(2)
79-Au-175,174.98155(26)#,,196.96655(2)
79-Au-176,175.98027(43)#,,196.96655(2)
79-Au-177,176.97722(25)#,,196.96655(2)
79-Au-178,177.97598(43)#,,196.96655(2)
79-Au-179,178.97341(37)#,,196.96655(2)
79-Au-180,179.97240(32)#,,196.96655(2)
79-Au-181,180.96995(48)#,,196.96655(2)
79-Au-182,181.96962(39)#,,196.96655(2)
79-Au-183,182.96762(43)#,,196.96655(2)
79-Au-184,183.96747(21)#,,196.96655(2)
79-Au-185,184.96581(23),,196.96655(2)
79-Au-186,185.96600(15),,196.96655(2)
79-Au-187,186.96456(16)#,,196.96655(2)
79-Au-188,187.96509(11)#,,196.96655(2)
79-Au-189,188.96389(22)#,,196.96655(2)
79-Au-190,189.964699(17),,196.96655(2)
79-Au-191,190.963650(50),,196.96655(2)
79-Au-192,191.964810(17),,196.96655(2)
79-Au-193,192.964132(10),,196.96655(2)
79-Au-194,193.965339(12),,196.96655(2)
79-Au-195,194.965018(3),,196.96655(2)
79-Au-196,195.966551(4),,196.96655(2)
79-Au-197,196.966552(3),100,196.96655(2)
79-Au-198,197.968225(3),,196.96655(2)
79-Au-199,198.968748(3),,196.96655(2)
79-Au-200,199.970720(60),,196.96655(2)
79-Au-201,200.971641(5),,196.96655(2)
79-Au-202,201.97379(18),,196.96655(2)
79-Au-203,202.975137(5),,196.96655(2)
79-Au-204,203.97771(22)#,,196.96655(2)
79-Au-205,204.97961(32)#,,196.96655(2)
80-Hg-175,174.99141(34)#,,200.59(2)
80-Hg-176,175.987410(40),,200.59(2)
80-Hg-177,176.98634(12),,200.59(2)
80-Hg-178,177.982476(16),,200.59(2)
80-Hg-179,178.98178(33)#,,200.59(2)
80-Hg-180,179.97832(21)#,,200.59(2)
80-Hg-181,180.97781(33)#,,200.59(2)
80-Hg-182,181.97475(50)#,,200.59(2)
80-Hg-183,182.97456(32)#,,200.59(2)
80-Hg-184,183.97190(22)#,,200.59(2)
80-Hg-185,184.97198(30)#,,200.59(2)
80-Hg-186,185.96946(22),,200.59(2)
80-Hg-187,186.96979(26)#,,200.59(2)
80-Hg-188,187.96756(19)#,,200.59(2)
80-Hg-189,188.96813(30)#,,200.59(2)
80-Hg-190,189.96628(16)#,,200.59(2)
80-Hg-191,190.967060(90),,200.59(2)
80-Hg-192,191.96557(30)#,,200.59(2)
80-Hg-193,192.966644(21),,200.59(2)
80-Hg-194,193.965382(25),,200.59(2)
80-Hg-195,194.966640(50),,200.59(2)
80-Hg-196,195.965815(4),0.15(1),200.59(2)
80-Hg-197,196.967195(5),,200.59(2)
80-Hg-198,197.966752(3),9.97(20),200.59(2)
80-Hg-199,198.968262(3),16.87(22),200.59(2)
80-Hg-200,199.968309(3),23.10(19),200.59(2)
80-Hg-201,200.970285(3),13.18(9),200.59(2)
80-Hg-202,201.970626(3),29.86(26),200.59(2)
80-Hg-203,202.972857(3),,200.59(2)
80-Hg-204,203.973476(3),6.87(15),200.59(2)
80-Hg-205,204.976056(5),,200.59(2)
80-Hg-206,205.977499(22),,200.59(2)
80-Hg-207,206.98258(16),,200.59(2)
80-Hg-208,207.98594(32)#,,200.59(2)
81-Tl-177,176.99688(24)#,,204.3833(2)
81-Tl-178,177.99523(23)#,,204.3833(2)
81-Tl-179,178.99147(15)#,,204.3833(2)
81-Tl-180,179.99019(48)#,,204.3833(2)
81-Tl-181,180.98690(41)#,,204.3833(2)
81-Tl-182,181.98561(43)#,,204.3833(2)
81-Tl-183,182.98270(42)#,,204.3833(2)
81-Tl-184,183.98176(32)#,,204.3833(2)
81-Tl-185,184.97910(43)#,,204.3833(2)
81-Tl-186,185.97855(39)#,,204.3833(2)
81-Tl-187,186.97617(43)#,,204.3833(2)
81-Tl-188,187.97592(24)#,,204.3833(2)
81-Tl-189,188.97369(37)#,,204.3833(2)
81-Tl-190,189.97379(46)#,,204.3833(2)
81-Tl-191,190.97189(23)#,,204.3833(2)
81-Tl-192,191.97214(22)#,,204.3833(2)
81-Tl-193,192.97055(27)#,,204.3833(2)
81-Tl-194,193.97105(22)#,,204.3833(2)
81-Tl-195,194.96965(14)#,,204.3833(2)
81-Tl-196,195.97052(15)#,,204.3833(2)
81-Tl-197,196.969540(30),,204.3833(2)
81-Tl-198,197.970470(90),,204.3833(2)
81-Tl-199,198.96981(11),,204.3833(2)
81-Tl-200,199.970945(7),,204.3833(2)
81-Tl-201,200.970804(16),,204.3833(2)
81-Tl-202,201.972091(16),,204.3833(2)
81-Tl-203,202.972329(3),29.524(14),204.3833(2)
81-Tl-204,203.973849(3),,204.3833(2)
81-Tl-205,204.974412(3),70.476(14),204.3833(2)
81-Tl-206,205.976095(3),,204.3833(2)
81-Tl-207,206.977408(6),,204.3833(2)
81-Tl-208,207.982005(3),,204.3833(2)
81-Tl-209,208.985349(10),,204.3833(2)
81-Tl-210,209.990066(12),,204.3833(2)
82-Pb-181,180.99671(17)#,,207.2(1)
82-Pb-182,181.992676(18),,207.2(1)
82-Pb-183,182.99193(33)#,,207.2(1)
82-Pb-184,183.98820(21)#,,207.2(1)
82-Pb-185,184.98758(33)#,,207.2(1)
82-Pb-186,185.98430(50)#,,207.2(1)
82-Pb-187,186.98403(32)#,,207.2(1)
82-Pb-188,187.98106(22)#,,207.2(1)
82-Pb-189,188.98088(29)#,,207.2(1)
82-Pb-190,189.97818(22),,207.2(1)
82-Pb-191,190.97820(23)#,,207.2(1)
82-Pb-192,191.97576(19)#,,207.2(1)
82-Pb-193,192.97608(20)#,,207.2(1)
82-Pb-194,193.97397(16)#,,207.2(1)
82-Pb-195,194.97447(44)#,,207.2(1)
82-Pb-196,195.97271(15)#,,207.2(1)
82-Pb-197,196.97338(11)#,,207.2(1)
82-Pb-198,197.97198(10)#,,207.2(1)
82-Pb-199,198.972910(70),,207.2(1)
82-Pb-200,199.971816(14),,207.2(1)
82-Pb-201,200.972850(30),,207.2(1)
82-Pb-202,201.972144(11),,207.2(1)
82-Pb-203,202.973375(7),,207.2(1)
82-Pb-204,203.973029(3),1.4(1),207.2(1)
82-Pb-205,204.974467(3),,207.2(1)
82-Pb-206,205.974449(3),24.1(1),207.2(1)
82-Pb-207,206.975881(3),22.1(1),207.2(1)
82-Pb-208,207.976636(3),52.4(1),207.2(1)
82-Pb-209,208.981075(3),,207.2(1)
82-Pb-210,209.984173(3),,207.2(1)
82-Pb-211,210.988731(3),,207.2(1)
82-Pb-212,211.9918875(29),,207.2(1)
82-Pb-213,212.99650(11)#,,207.2(1)
82-Pb-214,213.9997981(27),,207.2(1)
83-Bi-185,184.99771(24)#,,208.98038(2)
83-Bi-186,185.99648(48)#,,208.98038(2)
83-Bi-187,186.99346(41)#,,208.98038(2)
83-Bi-188,187.99217(32)#,,208.98038(2)
83-Bi-189,188.98951(43)#,,208.98038(2)
83-Bi-190,189.98852(39)#,,208.98038(2)
83-Bi-191,190.98605(43)#,,208.98038(2)
83-Bi-192,191.98537(24)#,,208.98038(2)
83-Bi-193,192.98306(37)#,,208.98038(2)
83-Bi-194,193.98275(46)#,,208.98038(2)
83-Bi-195,194.98075(23)#,,208.98038(2)
83-Bi-196,195.98061(22)#,,208.98038(2)
83-Bi-197,196.97893(26),,208.98038(2)
83-Bi-198,197.97902(19),,208.98038(2)
83-Bi-199,198.97758(13),,208.98038(2)
83-Bi-200,199.97814(10),,208.98038(2)
83-Bi-201,200.976970(30),,208.98038(2)
83-Bi-202,201.977670(60),,208.98038(2)
83-Bi-203,202.976868(23),,208.98038(2)
83-Bi-204,203.977805(28),,208.98038(2)
83-Bi-205,204.977375(8),,208.98038(2)
83-Bi-206,205.978483(9),,208.98038(2)
83-Bi-207,206.978455(4),,208.98038(2)
83-Bi-208,207.979727(4),,208.98038(2)
83-Bi-209,208.980383(3),100,208.98038(2)
83-Bi-210,209.984105(3),,208.98038(2)
83-Bi-211,210.987258(6),,208.98038(2)
83-Bi-212,211.991272(3),,208.98038(2)
83-Bi-213,212.994375(8),,208.98038(2)
83-Bi-214,213.998699(12),,208.98038(2)
83-Bi-215,215.00183(10),,208.98038(2)
83-Bi-216,216.00620(11)#,,208.98038(2)
84-Po-190,189.99511(51)#,,[209]
84-Po-191,190.99465(32)#,,[209]
84-Po-192,191.99152(22)#,,[209]
84-Po-193,192.99110(30)#,,[209]
84-Po-194,193.98828(22),,[209]
84-Po-195,194.98805(24)#,,[209]
84-Po-196,195.98551(19)#,,[209]
84-Po-197,196.98557(21)#,,[209]
84-Po-198,197.98334(16)#,,[209]
84-Po-199,198.98360(44)#,,[209]
84-Po-200,199.98174(15)#,,[209]
84-Po-201,200.98221(11)#,,[209]
84-Po-202,201.98070(10)#,,[209]
84-Po-203,202.981410(70),,[209]
84-Po-204,203.980307(14),,[209]
84-Po-205,204.981170(30),,[209]
84-Po-206,205.980465(11),,[209]
84-Po-207,206.981578(8),,[209]
84-Po-208,207.981231(3),,[209]
84-Po-209,208.982416(3),,[209]
84-Po-210,209.982857(3),,[209]
84-Po-211,210.986637(3),,[209]
84-Po-212,211.988852(3),,[209]
84-Po-213,212.992843(4),,[209]
84-Po-214,213.995186(3),,[209]
84-Po-215,214.999415(3),,[209]
84-Po-216,216.0019052(29),,[209]
84-Po-217,217.00625(11)#,,[209]
84-Po-218,218.0089658(27),,[209]
85-At-193,193.00019(43)#,,[210]
85-At-194,193.99897(43)#,,[210]
85-At-195,194.99655(43)#,,[210]
85-At-196,195.99570(25)#,,[210]
85-At-197,196.99329(38)#,,[210]
85-At-198,197.99275(46)#,,[210]
85-At-199,198.99063(24)#,,[210]
85-At-200,199.99029(22)#,,[210]
85-At-201,200.98849(26),,[210]
85-At-202,201.98845(19),,[210]
85-At-203,202.98685(13),,[210]
85-At-204,203.98726(10),,[210]
85-At-205,204.986040(30),,[210]
85-At-206,205.986600(60),,[210]
85-At-207,206.985776(23),,[210]
85-At-208,207.986583(28),,[210]
85-At-209,208.986159(9),,[210]
85-At-210,209.987131(9),,[210]
85-At-211,210.987481(4),,[210]
85-At-212,211.990735(4),,[210]
85-At-213,212.992921(6),,[210]
85-At-214,213.996356(5),,[210]
85-At-215,214.998641(8),,[210]
85-At-216,216.002409(5),,[210]
85-At-217,217.004710(8),,[210]
85-At-218,218.008681(13),,[210]
85-At-219,219.011300(90),,[210]
85-At-220,220.01530(12)#,,[210]
85-At-221,221.01814(32)#,,[210]
85-At-222,222.02233(32)#,,[210]
85-At-223,223.02534(43)#,,[210]
86-Rn-196,196.00231(22)#,,[222]
86-Rn-197,197.00166(30)#,,[222]
86-Rn-198,197.99878(22),,[222]
86-Rn-199,198.99831(24)#,,[222]
86-Rn-200,199.99568(19)#,,[222]
86-Rn-201,200.99554(21)#,,[222]
86-Rn-202,201.99322(16)#,,[222]
86-Rn-203,202.99332(44)#,,[222]
86-Rn-204,203.99137(15)#,,[222]
86-Rn-205,204.99167(12)#,,[222]
86-Rn-206,205.99016(10)#,,[222]
86-Rn-207,206.990730(70),,[222]
86-Rn-208,207.989631(14),,[222]
86-Rn-209,208.990380(30),,[222]
86-Rn-210,209.989680(11),,[222]
86-Rn-211,210.990585(8),,[222]
86-Rn-212,211.990689(4),,[222]
86-Rn-213,212.993868(8),,[222]
86-Rn-214,213.995346(10),,[222]
86-Rn-215,214.998729(9),,[222]
86-Rn-216,216.000258(8),,[222]
86-Rn-217,217.003915(5),,[222]
86-Rn-218,218.005586(4),,[222]
86-Rn-219,219.009475(3),,[222]
86-Rn-220,220.0113841(29),,[222]
86-Rn-221,221.01546(11)#,,[222]
86-Rn-222,222.0175705(27),,[222]
86-Rn-223,223.02179(32)#,,[222]
86-Rn-224,224.02409(32)#,,[222]
86-Rn-225,225.02844(32)#,,[222]
86-Rn-226,226.03089(43)#,,[222]
86-Rn-227,227.03541(45)#,,[222]
86-Rn-228,228.03808(50)#,,[222]
87-Fr-200,200.00650(25)#,,[223]
87-Fr-201,201.00399(38)#,,[223]
87-Fr-202,202.00329(46)#,,[223]
87-Fr-203,203.00105(25)#,,[223]
87-Fr-204,204.00059(22)#,,[223]
87-Fr-205,204.99866(25),,[223]
87-Fr-206,205.99849(19),,[223]
87-Fr-207,206.99686(12),,[223]
87-Fr-208,207.997130(80),,[223]
87-Fr-209,208.995920(30),,[223]
87-Fr-210,209.996398(23),,[223]
87-Fr-211,210.995529(22),,[223]
87-Fr-212,211.996195(28),,[223]
87-Fr-213,212.996175(9),,[223]
87-Fr-214,213.998955(10),,[223]
87-Fr-215,215.000326(8),,[223]
87-Fr-216,216.003188(14),,[223]
87-Fr-217,217.004616(8),,[223]
87-Fr-218,218.007563(6),,[223]
87-Fr-219,219.009241(8),,[223]
87-Fr-220,220.012313(5),,[223]
87-Fr-221,221.014246(8),,[223]
87-Fr-222,222.017544(23),,[223]
87-Fr-223,223.0197307(29),,[223]
87-Fr-224,224.023240(50),,[223]
87-Fr-225,225.025607(11),,[223]
87-Fr-226,226.02934(10),,[223]
87-Fr-227,227.03183(10),,[223]
87-Fr-228,228.03572(22)#,,[223]
87-Fr-229,229.03843(39)#,,[223]
87-Fr-230,230.04251(48)#,,[223]
87-Fr-231,231.04541(56)#,,[223]
87-Fr-232,232.04965(69)#,,[223]
88-Ra-203,203.00921(25)#,,[226]
88-Ra-204,204.00648(19)#,,[226]
88-Ra-205,205.00619(22)#,,[226]
88-Ra-206,206.00378(16)#,,[226]
88-Ra-207,207.00373(45)#,,[226]
88-Ra-208,208.00178(15)#,,[226]
88-Ra-209,209.00194(13)#,,[226]
88-Ra-210,210.00045(10)#,,[226]
88-Ra-211,211.000890(70),,[226]
88-Ra-212,211.999783(15),,[226]
88-Ra-213,213.000350(30),,[226]
88-Ra-214,214.000091(12),,[226]
88-Ra-215,215.002704(9),,[226]
88-Ra-216,216.003518(10),,[226]
88-Ra-217,217.006306(10),,[226]
88-Ra-218,218.007124(12),,[226]
88-Ra-219,219.010069(9),,[226]
88-Ra-220,220.011015(11),,[226]
88-Ra-221,221.013908(8),,[226]
88-Ra-222,222.015362(5),,[226]
88-Ra-223,223.018497(3),,[226]
88-Ra-224,224.0202020(29),,[226]
88-Ra-225,225.023604(3),,[226]
88-Ra-226,226.0254026(27),,[226]
88-Ra-227,227.0291707(27),,[226]
88-Ra-228,228.0310641(27),,[226]
88-Ra-229,229.034820(70),,[226]
88-Ra-230,230.037080(40),,[226]
88-Ra-231,231.04122(32)#,,[226]
88-Ra-232,232.04369(39)#,,[226]
88-Ra-233,233.04800(50)#,,[226]
88-Ra-234,234.05055(58)#,,[226]
89-Ac-207,207.01209(25)#,,[227]
89-Ac-208,208.01149(22)#,,[227]
89-Ac-209,209.00957(26),,[227]
89-Ac-210,210.00926(20),,[227]
89-Ac-211,211.00765(14),,[227]
89-Ac-212,212.00781(10),,[227]
89-Ac-213,213.006570(60),,[227]
89-Ac-214,214.006890(60),,[227]
89-Ac-215,215.006450(60),,[227]
89-Ac-216,216.008721(29),,[227]
89-Ac-217,217.009333(14),,[227]
89-Ac-218,218.011630(50),,[227]
89-Ac-219,219.012400(50),,[227]
89-Ac-220,220.014750(60),,[227]
89-Ac-221,221.015580(50),,[227]
89-Ac-222,222.017829(6),,[227]
89-Ac-223,223.019126(8),,[227]
89-Ac-224,224.021708(5),,[227]
89-Ac-225,225.023221(8),,[227]
89-Ac-226,226.026090(4),,[227]
89-Ac-227,227.0277470(29),,[227]
89-Ac-228,228.0310148(28),,[227]
89-Ac-229,229.032930(50),,[227]
89-Ac-230,230.03603(11),,[227]
89-Ac-231,231.03855(11),,[227]
89-Ac-232,232.04202(11),,[227]
89-Ac-233,233.04455(32)#,,[227]
89-Ac-234,234.04842(43)#,,[227]
89-Ac-235,235.05110(45)#,,[227]
89-Ac-236,236.05518(54)#,,[227]
90-Th-210,210.01503(17)#,,232.0381(1)
90-Th-211,211.01486(45)#,,232.0381(1)
90-Th-212,212.01292(15)#,,232.0381(1)
90-Th-213,213.01296(14)#,,232.0381(1)
90-Th-214,214.01145(10)#,,232.0381(1)
90-Th-215,215.011730(70),,232.0381(1)
90-Th-216,216.011051(17),,232.0381(1)
90-Th-217,217.013070(30),,232.0381(1)
90-Th-218,218.013268(15),,232.0381(1)
90-Th-219,219.015520(50),,232.0381(1)
90-Th-220,220.015733(24),,232.0381(1)
90-Th-221,221.018171(11),,232.0381(1)
90-Th-222,222.018454(14),,232.0381(1)
90-Th-223,223.020795(10),,232.0381(1)
90-Th-224,224.021459(13),,232.0381(1)
90-Th-225,225.023941(8),,232.0381(1)
90-Th-226,226.024891(5),,232.0381(1)
90-Th-227,227.027699(3),,232.0381(1)
90-Th-228,228.0287313(29),,232.0381(1)
90-Th-229,229.031755(3),,232.0381(1)
90-Th-230,230.0331266(22),,232.0381(1)
90-Th-231,231.0362971(22),,232.0381(1)
90-Th-232,232.0380504(22),100,232.0381(1)
90-Th-233,233.0415769(22),,232.0381(1)
90-Th-234,234.043595(4),,232.0381(1)
90-Th-235,235.047500(50),,232.0381(1)
90-Th-236,236.04971(32)#,,232.0381(1)
90-Th-237,237.05389(39)#,,232.0381(1)
90-Th-238,238.05624(39)#,,232.0381(1)
91-Pa-213,213.02118(27),,231.03588(2)
91-Pa-214,214.02074(21),,231.03588(2)
91-Pa-215,215.01910(15),,231.03588(2)
91-Pa-216,216.01911(11),,231.03588(2)
91-Pa-217,217.018290(80),,231.03588(2)
91-Pa-218,218.020010(80),,231.03588(2)
91-Pa-219,219.019880(80),,231.03588(2)
91-Pa-220,220.021880(60),,231.03588(2)
91-Pa-221,221.021860(60),,231.03588(2)
91-Pa-222,222.023730(80)#,,231.03588(2)
91-Pa-223,223.023960(80),,231.03588(2)
91-Pa-224,224.025610(60),,231.03588(2)
91-Pa-225,225.026120(80),,231.03588(2)
91-Pa-226,226.027933(13),,231.03588(2)
91-Pa-227,227.028793(8),,231.03588(2)
91-Pa-228,228.031037(5),,231.03588(2)
91-Pa-229,229.032089(10),,231.03588(2)
91-Pa-230,230.034533(4),,231.03588(2)
91-Pa-231,231.0358789(28),100,231.03588(2)
91-Pa-232,232.038582(8),,231.03588(2)
91-Pa-233,233.0402402(24),,231.03588(2)
91-Pa-234,234.043302(5),,231.03588(2)
91-Pa-235,235.045440(50),,231.03588(2)
91-Pa-236,236.04868(21),,231.03588(2)
91-Pa-237,237.05114(11),,231.03588(2)
91-Pa-238,238.054500(60),,231.03588(2)
91-Pa-239,239.05713(32)#,,231.03588(2)
91-Pa-240,240.06098(32)#,,231.03588(2)
92-U-218,218.02349(10)#,,238.02891(3)
92-U-219,219.024920(90),,238.02891(3)
92-U-220,220.02471(22)#,,238.02891(3)
92-U-221,221.02635(11)#,,238.02891(3)
92-U-222,222.02607(11)#,,238.02891(3)
92-U-223,223.027720(80),,238.02891(3)
92-U-224,224.027590(27),,238.02891(3)
92-U-225,225.029380(50),,238.02891(3)
92-U-226,226.029340(20),,238.02891(3)
92-U-227,227.031140(18),,238.02891(3)
92-U-228,228.031366(17),,238.02891(3)
92-U-229,229.033496(9),,238.02891(3)
92-U-230,230.033927(5),,238.02891(3)
92-U-231,231.036289(4),,238.02891(3)
92-U-232,232.0371463(29),,238.02891(3)
92-U-233,233.039628(3),,238.02891(3)
92-U-234,234.0409456(21),0.0055(2),238.02891(3)
92-U-235,235.0439231(21),0.7200(51),238.02891(3)
92-U-236,236.0455619(21),,238.02891(3)
92-U-237,237.0487240(21),,238.02891(3)
92-U-238,238.0507826(21),99.2745(106),238.02891(3)
92-U-239,239.0542878(21),,238.02891(3)
92-U-240,240.056586(6),,238.02891(3)
92-U-241,241.06033(32)#,,238.02891(3)
92-U-242,242.06293(22)#,,238.02891(3)
93-Np-225,225.033900(80),,[237]
93-Np-226,226.03513(10)#,,[237]
93-Np-227,227.034960(80),,[237]
93-Np-228,228.03618(21)#,,[237]
93-Np-229,229.036250(90),,[237]
93-Np-230,230.037810(60),,[237]
93-Np-231,231.038230(50),,[237]
93-Np-232,232.04010(11)#,,[237]
93-Np-233,233.040730(60),,[237]
93-Np-234,234.042889(9),,[237]
93-Np-235,235.0440559(23),,[237]
93-Np-236,236.046560(50),,[237]
93-Np-237,237.0481673(21),,[237]
93-Np-238,238.0509405(21),,[237]
93-Np-239,239.0529314(23),,[237]
93-Np-240,240.056169(16),,[237]
93-Np-241,241.058250(80),,[237]
93-Np-242,242.06164(22)#,,[237]
93-Np-243,243.064270(30)#,,[237]
93-Np-244,244.06785(32)#,,[237]
94-Pu-228,228.038730(30),,[244]
94-Pu-229,229.040140(80),,[244]
94-Pu-230,230.039646(26),,[244]
94-Pu-231,231.04126(11)#,,[244]
94-Pu-232,232.041179(20),,[244]
94-Pu-233,233.042990(50),,[244]
94-Pu-234,234.043305(8),,[244]
94-Pu-235,235.045282(22),,[244]
94-Pu-236,236.0460481(29),,[244]
94-Pu-237,237.0484038(25),,[244]
94-Pu-238,238.0495534(21),,[244]
94-Pu-239,239.0521565(21),,[244]
94-Pu-240,240.0538075(21),,[244]
94-Pu-241,241.0568453(21),,[244]
94-Pu-242,242.0587368(21),,[244]
94-Pu-243,243.061997(3),,[244]
94-Pu-244,244.064198(5),,[244]
94-Pu-245,245.067739(15),,[244]
94-Pu-246,246.070198(16),,[244]
94-Pu-247,247.07407(32)#,,[244]
95-Am-231,231.04556(32)#,,[243]
95-Am-232,232.04659(32)#,,[243]
95-Am-233,233.04647(23)#,,[243]
95-Am-234,234.04779(22)#,,[243]
95-Am-235,235.04803(22)#,,[243]
95-Am-236,236.04957(11)#,,[243]
95-Am-237,237.049970(60),,[243]
95-Am-238,238.051980(50),,[243]
95-Am-239,239.053018(3),,[243]
95-Am-240,240.055288(15),,[243]
95-Am-241,241.0568229(21),,[243]
95-Am-242,242.0595430(21),,[243]
95-Am-243,243.0613727(23),,[243]
95-Am-244,244.0642794(23),,[243]
95-Am-245,245.066445(4),,[243]
95-Am-246,246.069768(20),,[243]
95-Am-247,247.07209(11)#,,[243]
95-Am-248,248.07575(22)#,,[243]
95-Am-249,249.07848(32)#,,[243]
96-Cm-233,233.05080(43)#,,[247]
96-Cm-234,234.05024(32)#,,[247]
96-Cm-235,235.05159(24)#,,[247]
96-Cm-236,236.05141(22)#,,[247]
96-Cm-237,237.05289(22)#,,[247]
96-Cm-238,238.053020(40),,[247]
96-Cm-239,239.05495(11)#,,[247]
96-Cm-240,240.0555190(29),,[247]
96-Cm-241,241.0576467(24),,[247]
96-Cm-242,242.0588293(21),,[247]
96-Cm-243,243.0613822(24),,[247]
96-Cm-244,244.0627463(21),,[247]
96-Cm-245,245.0654856(29),,[247]
96-Cm-246,246.0672176(24),,[247]
96-Cm-247,247.070347(5),,[247]
96-Cm-248,248.072342(5),,[247]
96-Cm-249,249.075947(5),,[247]
96-Cm-250,250.078351(12),,[247]
96-Cm-251,251.082278(24),,[247]
96-Cm-252,252.08487(32)#,,[247]
97-Bk-235,235.05658(43)#,,[247]
97-Bk-236,236.05733(43)#,,[247]
97-Bk-237,237.05713(32)#,,[247]
97-Bk-238,238.05827(31)#,,[247]
97-Bk-239,239.05836(31)#,,[247]
97-Bk-240,240.05975(16)#,,[247]
97-Bk-241,241.06022(22)#,,[247]
97-Bk-242,242.06205(22)#,,[247]
97-Bk-243,243.063002(5),,[247]
97-Bk-244,244.065168(16),,[247]
97-Bk-245,245.0663554(26),,[247]
97-Bk-246,246.068670(60),,[247]
97-Bk-247,247.070299(6),,[247]
97-Bk-248,248.073080(80)#,,[247]
97-Bk-249,249.074980(3),,[247]
97-Bk-250,250.078311(4),,[247]
97-Bk-251,251.080753(12),,[247]
97-Bk-252,252.08430(22)#,,[247]
97-Bk-253,253.08688(39)#,,[247]
97-Bk-254,254.09060(32)#,,[247]
98-Cf-237,237.06207(54)#,,[251]
98-Cf-238,238.06141(43)#,,[251]
98-Cf-239,239.06258(25)#,,[251]
98-Cf-240,240.06230(22)#,,[251]
98-Cf-241,241.06372(27)#,,[251]
98-Cf-242,242.063690(40),,[251]
98-Cf-243,243.06542(15)#,,[251]
98-Cf-244,244.065990(4),,[251]
98-Cf-245,245.06804(11)#,,[251]
98-Cf-246,246.0687988(24),,[251]
98-Cf-247,247.070992(9),,[251]
98-Cf-248,248.072178(6),,[251]
98-Cf-249,249.074847(3),,[251]
98-Cf-250,250.0764000(24),,[251]
98-Cf-251,251.079580(5),,[251]
98-Cf-252,252.081620(5),,[251]
98-Cf-253,253.085127(7),,[251]
98-Cf-254,254.087316(13),,[251]
98-Cf-255,255.09104(22)#,,[251]
98-Cf-256,256.09344(32)#,,[251]
99-Es-240,240.06892(43)#,,[252]
99-Es-241,241.06866(32)#,,[252]
99-Es-242,242.06970(35)#,,[252]
99-Es-243,243.06963(31)#,,[252]
99-Es-244,244.07097(20)#,,[252]
99-Es-245,245.07132(22)#,,[252]
99-Es-246,246.07297(24)#,,[252]
99-Es-247,247.073650(30)#,,[252]
99-Es-248,248.075460(60)#,,[252]
99-Es-249,249.076410(30)#,,[252]
99-Es-250,250.07865(11)#,,[252]
99-Es-251,251.079984(7),,[252]
99-Es-252,252.082970(50),,[252]
99-Es-253,253.084818(3),,[252]
99-Es-254,254.088016(5),,[252]
99-Es-255,255.090266(12),,[252]
99-Es-256,256.09359(11)#,,[252]
99-Es-257,257.09598(44)#,,[252]
100-Fm-242,242.07343(43)#,,[257]
100-Fm-243,243.07451(25)#,,[257]
100-Fm-244,244.07408(31)#,,[257]
100-Fm-245,245.07538(30)#,,[257]
100-Fm-246,246.075280(40),,[257]
100-Fm-247,247.07682(16)#,,[257]
100-Fm-248,248.077184(13),,[257]
100-Fm-249,249.07902(15)#,,[257]
100-Fm-250,250.079515(13),,[257]
100-Fm-251,251.081566(9),,[257]
100-Fm-252,252.082460(6),,[257]
100-Fm-253,253.085176(5),,[257]
100-Fm-254,254.086848(3),,[257]
100-Fm-255,255.089955(5),,[257]
100-Fm-256,256.091767(8),,[257]
100-Fm-257,257.095099(7),,[257]
100-Fm-258,258.09707(22)#,,[257]
100-Fm-259,259.10059(30)#,,[257]
101-Md-245,245.08102(40)#,,[258]
101-Md-246,246.08193(41)#,,[258]
101-Md-247,247.08180(39)#,,[258]
101-Md-248,248.08291(26)#,,[258]
101-Md-249,249.08300(24)#,,[258]
101-Md-250,250.08449(32)#,,[258]
101-Md-251,251.08492(22)#,,[258]
101-Md-252,252.08663(21)#,,[258]
101-Md-253,253.08728(22)#,,[258]
101-Md-254,254.08973(11)#,,[258]
101-Md-255,255.091075(8),,[258]
101-Md-256,256.094050(60),,[258]
101-Md-257,257.095535(4),,[258]
101-Md-258,258.098425(5),,[258]
101-Md-259,259.10050(22)#,,[258]
101-Md-260,260.10365(34)#,,[258]
102-No-249,249.08782(37)#,,[259]
102-No-250,250.08749(22)#,,[259]
102-No-251,251.08896(19)#,,[259]
102-No-252,252.088966(14),,[259]
102-No-253,253.09065(26)#,,[259]
102-No-254,254.090949(19),,[259]
102-No-255,255.093232(13),,[259]
102-No-256,256.094276(9),,[259]
102-No-257,257.096850(30),,[259]
102-No-258,258.09820(22)#,,[259]
102-No-259,259.10102(11)#,,[259]
102-No-260,260.10264(22)#,,[259]
102-No-261,261.10574(32)#,,[259]
102-No-262,262.10752(58)#,,[259]
103-Lr-251,251.09436(32)#,,[262]
103-Lr-252,252.09533(32)#,,[262]
103-Lr-253,253.09526(24)#,,[262]
103-Lr-254,254.09659(36)#,,[262]
103-Lr-255,255.09677(22)#,,[262]
103-Lr-256,256.09876(24)#,,[262]
103-Lr-257,257.09961(22)#,,[262]
103-Lr-258,258.10188(11)#,,[262]
103-Lr-259,259.102990(80)#,,[262]
103-Lr-260,260.10557(12)#,,[262]
103-Lr-261,261.10694(22)#,,[262]
103-Lr-262,262.10969(32)#,,[262]
103-Lr-263,263.11139(39)#,,[262]
104-Rf-253,253.10068(49)#,,[261]
104-Rf-254,254.10017(31)#,,[261]
104-Rf-255,255.10149(22)#,,[261]
104-Rf-256,256.101180(29),,[261]
104-Rf-257,257.10307(29)#,,[261]
104-Rf-258,258.10357(22)#,,[261]
104-Rf-259,259.105630(80)#,,[261]
104-Rf-260,260.10643(22)#,,[261]
104-Rf-261,261.10875(11)#,,[261]
104-Rf-262,262.10992(30)#,,[261]
104-Rf-263,263.11254(20)#,,[261]
104-Rf-264,264.11398(48)#,,[261]
105-Db-255,255.10740(45)#,,[262]
105-Db-256,256.10811(39)#,,[262]
105-Db-257,257.10786(25)#,,[262]
105-Db-258,258.10944(37)#,,[262]
105-Db-259,259.10972(31)#,,[262]
105-Db-260,260.11143(25)#,,[262]
105-Db-261,261.11211(25)#,,[262]
105-Db-262,262.11415(20)#,,[262]
105-Db-263,263.11508(18)#,,[262]
105-Db-264,264.11747(25)#,,[262]
105-Db-265,265.11866(30)#,,[262]
106-Sg-258,258.11315(45)#,,[266]
106-Sg-259,259.11465(23)#,,[266]
106-Sg-260,260.114440(40),,[266]
106-Sg-261,261.11620(30)#,,[266]
106-Sg-262,262.11648(30)#,,[266]
106-Sg-263,263.11831(13)#,,[266]
106-Sg-264,264.11892(30)#,,[266]
106-Sg-265,265.12107(15)#,,[266]
106-Sg-266,266.12193(31)#,,[266]
107-Bh-260,260.12180(66)#,,[264]
107-Bh-261,261.12180(26)#,,[264]
107-Bh-262,262.12301(40)#,,[264]
107-Bh-263,263.12315(45)#,,[264]
107-Bh-264,264.12473(30)#,,[264]
107-Bh-265,265.12520(41)#,,[264]
107-Bh-266,266.12701(38)#,,[264]
107-Bh-267,267.12774(37)#,,[264]
108-Hs-263,263.12871(39)#,,[277]
108-Hs-264,264.128410(50),,[277]
108-Hs-265,265.13000(32)#,,[277]
108-Hs-266,266.13004(44)#,,[277]
108-Hs-267,267.13177(11)#,,[277]
108-Hs-268,268.13216(44)#,,[277]
108-Hs-269,269.13411(46)#,,[277]
108-Hs-277,,,[277]
109-Mt-265,265.13657(50)#,,[268]
109-Mt-266,266.13794(38)#,,[268]
109-Mt-267,267.13753(62)#,,[268]
109-Mt-268,268.13882(34)#,,[268]
109-Mt-269,269.13911(59)#,,[268]
109-Mt-270,270.14072(66)#,,[268]
109-Mt-271,271.14123(65)#,,[268]
110-Ds-267,267.14396(41)#,,[281]
110-Ds-268,268.14353(54)#,,[281]
110-Ds-269,269.14514(31)#,,[281]
110-Ds-270,270.14463(70)#,,[281]
110-Ds-271,271.14608(20)#,,[281]
110-Ds-272,272.14631(70)#,,[281]
110-Ds-273,273.14925(47)#,,[281]
110-Ds-281,,,[281]
111-Rg-272,272.15348(36)#,,[272]
112-Cn-285,,,[285]
114-Uuq-289,,,[289]
116-Uuh-292,,,[292]\
"""
