%feature("docstring") OTSubsetInverse::SubsetInverseSampling
"Subset inverse simulation.

Parameters
----------
event : :class:`~openturns.Event`
    Event we are computing the probability of. The threshold of the event is
    not used.
targetProbability : float
    The wanted final probability.
proposalRange : float, optional
    Proposal range length
conditionalProbability : float, optional
    Value of :math:`P(F_i|F_{i-1})` between successive steps

Notes
-----
The goal is to estimate the threshold of the following target probability :

.. math::

    P_f = \\\\int_{\\\\mathcal D_f} f_{\\\\uX}(\\\\ux)\\\\di{\\\\ux}\\\\\\\\
        = \\\\int_{\\\\mathbb R^{n_X}} \\\\mathbf{1}_{\\\\{g(\\\\ux,\\\\underline{d}) \\\\:\\\\leq 0\\\\: \\\\}}f_{\\\\uX}(\\\\ux)\\\\di{\\\\ux}\\\\\\\\
        = \\\\Prob {\\\\{g(\\\\uX,\\\\underline{d}) \\\\leq q\\\\}}


The idea of the subset simulation method is to replace simulating a
rare failure event in the original probability space by a sequence of
simulations of more frequent conditional events :math:`F_i` :

.. math::

    F_1 \\\\supset F_2 \\\\supset \\\\dots \\\\supset F_m = F


The original probability estimate rewrites :

.. math::

    P_f = P(F_m) = P(\\\\bigcap \\\\limits_{i=1}^m F_i) = P(F_1) \\\\prod_{i=2}^m P(F_i|F_{i-1})


And each conditional subset failure region is chosen by setting the threshold
:math:`g_i` so that :math:`P(F_i|F_{i-1})` leads to a conditional failure
probability of order :math:`0.1` :

.. math::

    F_i =\\\\Prob {\\\\{g(\\\\uX,\\\\underline{d}) \\\\leq g_i\\\\}}


The conditional samples are generated by the means of Markov Chains,
using the Metropolis Hastings algorithm.

:math:`N` being the number of simulations per subset, and :math:`p_{0i}` the
conditional probability of each subset event, and :math:`\\\\gamma_i` the
autocorrelation between Markov chain samples.

.. math::

    \\\\delta^2 = \\\\sum_{i=1}^m \\\\delta^2_i = \\\\sum_{i=1}^m (1+\\\\gamma_i) \\\\frac{1-p_{0i}}{p_{0i}N}


The first event :math:`F_1` not being conditional, :math:`\\\\delta^2_1`
expresses as the classic Monte Carlo c.o.v.

Examples
--------
>>> import openturns as ot
>>> import otsubsetinverse as otsi

>>> ot.RandomGenerator.SetSeed(0)

Create a performance function with an associated distribution.

>>> limitState = ot.SymbolicFunction(['u1', 'u2'], ['u1-u2'])
>>> dim = limitState.getInputDimension()
>>> mean = ot.Point([7., 2.])
>>> sigma = ot.Point(dim, 1.0)
>>> R = ot.IdentityMatrix(dim)
>>> myDistribution = ot.Normal(mean, sigma, R)
>>> vect = ot.RandomVector(myDistribution)
>>> output = ot.RandomVector(limitState, vect)

Create an event with a fictional threshold value which will not be used.

>>> myEvent = ot.Event(output, ot.Less(), 0.)

Define the target probability for which the threshold will be computed.

>>> targetProbability = 0.0002
>>> mySS = otsi.SubsetInverseSampling(myEvent, targetProbability)
>>> mySS.setMaximumOuterSampling(10000)
>>> mySS.run()

Get some results.

>>> ResultSS = mySS.getResult()
>>> print('Pf = {}'.format(ResultSS.getProbabilityEstimate()))
Pf = 0.0002
>>> print('Threshold = {}'.format(mySS.getThresholdPerStep()[-1]))
Threshold = 0.0202715501987
>>> print('Threshold confidence length = {}'.format(
...                         mySS.getThresholdConfidenceLength(0.90)))
Threshold confidence length = 0.0244117131807

See also
--------
openturns.Simulation"

// ---------------------------------------------------------------------------

%feature("docstring") OTSubsetInverse::SubsetInverseSampling::getCoefficientOfVariationPerStep
"Coefficient of variation per step accessor.

Returns
-------
coef : :py:class:`openturns.Point`
    Coefficient of variation at each subset step."

// ---------------------------------------------------------------------------

%feature("docstring") OTSubsetInverse::SubsetInverseSampling::getThresholdCoefficientOfVariationPerStep
"Threshold coefficient of variation per step accessor.

Returns
-------
coef : :py:class:`openturns.Point`
    Coefficient of variation at each subset step."
// ---------------------------------------------------------------------------

%feature("docstring") OTSubsetInverse::SubsetInverseSampling::setConditionalProbability
"Conditional probability accessor.

Value of :math:`P(F_i|F_{i-1})` between successive steps.

Parameters
----------
prob : float
    Conditional probability value."

// ---------------------------------------------------------------------------

%feature("docstring") OTSubsetInverse::SubsetInverseSampling::getConditionalProbability
"Conditional probability accessor.

Value of :math:`P(F_i|F_{i-1})` between successive steps.

Returns
-------
prob : float
    Conditional probability value."

// ---------------------------------------------------------------------------

%feature("docstring") OTSubsetInverse::SubsetInverseSampling::setTargetProbability
"Final target probability accessor.

Value of :math:`P(F_m)`.

Parameters
----------
prob : float
    Final target probability value."

// ---------------------------------------------------------------------------

%feature("docstring") OTSubsetInverse::SubsetInverseSampling::getThresholdConfidenceLength
"Accessor to the confidence length of the threshold.

Parameters
----------
level : float, :math:`level \\\\in ]0, 1[`
    Confidence level. By default, it is :math:`0.95`.

Returns
-------
confidenceLength : float
    Length of the confidence interval at the confidence level *level*."

// ---------------------------------------------------------------------------

%feature("docstring") OTSubsetInverse::SubsetInverseSampling::getTargetProbability
"Final target probability accessor.

Value of :math:`P(F_m)`.

Returns
-------
prob : float
    Final target probability value."

// ---------------------------------------------------------------------------

%feature("docstring") OTSubsetInverse::SubsetInverseSampling::setKeepEventSample
"Sample storage accessor.

Parameters
----------
prob : bool
    Whether to keep the event samples."

// ---------------------------------------------------------------------------

%feature("docstring") OTSubsetInverse::SubsetInverseSampling::getEventInputSample
"Input sample accessor.

Returns
-------
inputSample : :py:class:`openturns.Sample`
    Input sample."

// ---------------------------------------------------------------------------

%feature("docstring") OTSubsetInverse::SubsetInverseSampling::getEventOutputSample
"Output sample accessor.

Returns
-------
outputSample : :py:class:`openturns.Sample`
    Ouput sample."

// ---------------------------------------------------------------------------

%feature("docstring") OTSubsetInverse::SubsetInverseSampling::getInputSample
"All input sample accessor.

Returns
-------
inputSample : :py:class:`openturns.Sample`
    Input sample."

// ---------------------------------------------------------------------------

%feature("docstring") OTSubsetInverse::SubsetInverseSampling::getOutputSample
"All output sample accessor.

Returns
-------
outputSample : :py:class:`openturns.Sample`
    Output sample."

// ---------------------------------------------------------------------------

%feature("docstring") OTSubsetInverse::SubsetInverseSampling::getThresholdPerStep
"Threshold accessor.

Returns
-------
threshold : :py:class:`openturns.Point`
    Threshold values."

// ---------------------------------------------------------------------------

%feature("docstring") OTSubsetInverse::SubsetInverseSampling::getGammaPerStep
"Autocorrelation accessor.

Returns
-------
prob : :py:class:`openturns.Point`
    Autocorrelation values."

// ---------------------------------------------------------------------------

%feature("docstring") OTSubsetInverse::SubsetInverseSampling::getProbabilityEstimatePerStep
"Probability estimate accessor.

Returns
-------
prob : :py:class:`openturns.Point`
    Probability estimate values."

// ---------------------------------------------------------------------------

%feature("docstring") OTSubsetInverse::SubsetInverseSampling::getNumberOfSteps
"Subset steps number accesor.

Returns
-------
n : int
    Number of subset steps."

// ---------------------------------------------------------------------------

%feature("docstring") OTSubsetInverse::SubsetInverseSampling::setProposalRange
"Proposal range length accessor.

Parameters
----------
range : float
    Range length."

// ---------------------------------------------------------------------------

%feature("docstring") OTSubsetInverse::SubsetInverseSampling::getProposalRange
"Proposal range length accessor.

Returns
-------
range : float
    Range length."

// ---------------------------------------------------------------------------

%feature("docstring") OTSubsetInverse::SubsetInverseSampling::setBetaMin
"Radius of the hypershere accessor.

Parameters
----------
beta : float
    Radius value of the exclusion hypershere when the conditional simulation is activated."

// ---------------------------------------------------------------------------

%feature("docstring") OTSubsetInverse::SubsetInverseSampling::setISubset
"Conditonal simulation activation accessor.

Parameters
----------
isubset : bool
    Activate or not the conditional simulation for the first step of the 
    simulation."