%feature("docstring") OTLM::LinearModelAnalysis
"Class used to analyse a linear model.

**Available usages**:

    LinearModelAnalysis(linearModelResult)

See Also
--------
LinearModelResult

Notes
-----
This class is used in order to create a linear model from numerical samples. The
linear regression model between the scalar variable :math:`Y` and the :math:`n`
-dimensional one :math:`\\\\vect{X} = (X_i)_{i \\\\leq n}` writes as follows:

.. math::

    \\\\tilde{Y} = a_0 + \\\\sum_{i=1}^n a_i X_i + \\\\epsilon

where :math:`\\\\epsilon` is the residual, supposed to follow the standard Normal
distribution.

Each coefficient :math:`a_i` is evaluated from both samples :math:`Ysample` and
:math:`Xsample` and is accompagnied by a confidence interval and a p-value (which
tests if they are significantly different from 0.0).

This class enables to test the quality of the model. It provides only numerical
tests. If :math:`X` is scalar, a graphical validation test exists, that draws
the residual couples :math:`(\\\\epsilon_i, \\\\epsilon_{i+1})`, where the residual
:math:`\\\\epsilon_i` is evaluated on the samples :math:`(Xsample, Ysample)`:
:math:`\\\\epsilon_i  = Ysample_i - \\\\tilde{Y}_i` with
:math:`\\\\tilde{Y}_i = a_0 + \\\\sum_{j=1}^{n} a_j \\\\ (Xsample)_{ij}`. The OpenTURNS method is
:class:`~openturns.VisualTest_DrawLinearModelResidual`.

By default, labels of the 3 most significant points are displayed on graphs.
This number can be changed by modifying the ResourceMap key
(``LinearModelAnalysis-Identifiers``).

Examples
--------
>>> import openturns as ot
>>> import otlm
>>> ot.RandomGenerator.SetSeed(0)
>>> distribution = ot.Normal()
>>> Xsample = distribution.getSample(30)
>>> func = ot.SymbolicFunction(['x'], ['2 * x + 1'])
>>> Ysample = func(Xsample) + ot.Normal().getSample(30)
>>> algo = otlm.LinearModelAlgorithm(Ysample, Xsample)
>>> result = algo.getResult()
>>> analysis = otlm.LinearModelAnalysis(result)
"

// ---------------------------------------------------------------------

%feature("docstring") OTLM::LinearModelAnalysis::getLinearModelResult
"Accessor to the linear model result.

Returns
-------
linearModelResult : :class:`~openturns.LinearModelResult`
    The  linear model result which had been passed to the constructor."


// ---------------------------------------------------------------------

%feature("docstring") OTLM::LinearModelAnalysis::getFormula
"Accessor to the formula.

Returns
-------
condensedFormula : :class:`~openturns.String`
"

// ---------------------------------------------------------------------

%feature("docstring") OTLM::LinearModelAnalysis::getCoefficientsNames
"Accessor to the coefficients names.

Returns
-------
coefficientsNames : :class:`~openturns.Description`
"

// ---------------------------------------------------------------------

%feature("docstring") OTLM::LinearModelAnalysis::getResiduals
"Accessor to the residuals.

Returns
-------
sampleResiduals : :class:`~openturns.Sample`
"

// ---------------------------------------------------------------------

%feature("docstring") OTLM::LinearModelAnalysis::getStandardizedResiduals
"Accessor to the standardized residuals.

Returns
-------
standardizedResiduals : :class:`~openturns.Sample`
"

// ---------------------------------------------------------------------

%feature("docstring") OTLM::LinearModelAnalysis::getCoefficientsEstimates
"Accessor to the coefficients of the linear model of the trend.

Returns
-------
beta : :class:`~openturns.Sample`
"

// ---------------------------------------------------------------------

%feature("docstring") OTLM::LinearModelAnalysis::getCoefficientsStandardErrors
"Accessor to the coefficients of standard error.

Returns
-------
standardErrors : :class:`~openturns.Sample`
"


// ---------------------------------------------------------------------

%feature("docstring") OTLM::LinearModelAnalysis::getCoefficientsTScores
"Accessor to the coefficients of linear expansion over their standard error.

Returns
-------
tScores : :class:`~openturns.Sample`
"

// ---------------------------------------------------------------------

%feature("docstring") OTLM::LinearModelAnalysis::getCoefficientsPValues
"Accessor to the coefficients of the p values.

Returns
-------
pValues : :class:`~openturns.Sample`
"

// ---------------------------------------------------------------------

%feature("docstring") OTLM::LinearModelAnalysis::getLeverages
"Accessor to the leverages.

Returns
-------
leverages : :class:`~openturns.Point`
"


// ---------------------------------------------------------------------

%feature("docstring") OTLM::LinearModelAnalysis::getCookDistances
"Accessor to the cook's distances.

Returns
-------
cookDistances : :class:`~openturns.Point`
"

// ---------------------------------------------------------------------

%feature("docstring") OTLM::LinearModelAnalysis::getDegreesOfFreedom
"Accessor to the number of degrees of freedom.

Returns
-------
dof : :class:`~openturns.UnsignedInteger`
"

// ---------------------------------------------------------------------

%feature("docstring") OTLM::LinearModelAnalysis::getRSquared
"Accessor to the R-squared test.

Returns
-------
rSquared : :class:`~openturns.NumericalScalar`
"

// ---------------------------------------------------------------------

%feature("docstring") OTLM::LinearModelAnalysis::getAdjustedRSquared
"Accessor to the Adjusted R-squared test.

Returns
-------
adjustedRSquared : :class:`~openturns.NumericalScalar`
"

// ---------------------------------------------------------------------

%feature("docstring") OTLM::LinearModelAnalysis::getFisherScore
"Accessor to the Fisher test.

Returns
-------
fisherScore : :class:`~openturns.NumericalScalar`
"

// ---------------------------------------------------------------------

%feature("docstring") OTLM::LinearModelAnalysis::getFisherPValue
"Accessor to the Fisher p value.

Returns
-------
fisherPValue : :class:`~openturns.NumericalScalar`
"

// ---------------------------------------------------------------------

%feature("docstring") OTLM::LinearModelAnalysis::getNormalityTestResultChiSquared
"Performs Pearson Chi-2 statistical test to check Gaussian assumption of the model

Returns
-------
testResult : :class:`~openturns.TestResult`
    Test result class.

Notes
-----
The Chi-Square test is a goodness of fit test which objective is to check the normality assumption (null hypothesis) of residuals (and thus the model).
It is accepted when p-value is low.

From a dataset :math:`\\\\epsilon=(\\\\epsilon_1,...,\\\\epsilon_k)` of size :math:`k` (residual set), we define :math:`nClasses` classes
where :math:`nClasses := 2\\\\ k^{\\\\frac{2}{5}}`.

Under the normality hypothesis, we compute cumulative function of the :math:`\\\\epsilon` set (mean and standard
deviation of the underlying normal distribution derived from the dataset). The interval :math:`[0, 1]` is then splitted
onto :math:`nClasses` intervals.

For the Chi-Square test, two major elements are to be considered:

 - :math:`C_{i}` is the number of counted observations in the class :math:`i`
 - :math:`E_{i}` is the number of expected observations in the class :math:`i`

Using the cumulative information, we count number of elements in each class (:math:`C_{i}`).
Also, under the null hypothesis, the different classes are equiprobable ( :math:`E_{i}\\\\ =\\\\ E\\\\ =\\\\frac{k}{nClasses}`.

From the previous values per class, we compute the Pearson test statistic :math:`P=\\\\sum_{i=1}^{nClasses} \\\\frac{C_{i} - E_{i})^{2}}{E_{i}}`.

Finally, the p-value is computed thanks to a chi-square distribution with :math:`dof` degrees of freedom, where
:math:`dof` is adjusted thanks to the boolean ResourceMap key (``LinearModelAnalysis-ChiSquareAdjust``).
If key is set to *True*, degree of freedom is :math:`nClasses - 3`. Otherwise it is :math:`nClasses - 1`
It is recommended to test with both *True* and *False* values."

// ---------------------------------------------------------------------

%feature("docstring") OTLM::LinearModelAnalysis::getNormalityTestResultKolmogorovSmirnov
"Performs Kolmogorov test to check Gaussian assumption of the model (null hypothesis).
It is accepted when p-value is low.

Returns
-------
testResult : :class:`~openturns.TestResult`
    Test result class.


Notes
-----
We check that the residual is Gaussian thanks to :class:`~openturns.FittingTest::Kolmogorov`."

// ---------------------------------------------------------------------

%feature("docstring") OTLM::LinearModelAnalysis::getNormalityTestResultAndersonDarling
"Performs Anderson-Darling test to check Gaussian assumption of the model (null hypothesis).
It is accepted when p-value is low.

Returns
-------
testResult : :class:`~openturns.TestResult`
    Test result class.


Notes
-----
We check that the residual is Gaussian thanks to :class:`~openturns.NormalityTest::AndersonDarling`."

// ---------------------------------------------------------------------

%feature("docstring") OTLM::LinearModelAnalysis::drawResidualsVsFitted
"Accessor to plot of residuals versus fitted values.

Returns
-------
graph : :class:`~openturns.Graph`
"

// ---------------------------------------------------------------------

%feature("docstring") OTLM::LinearModelAnalysis::drawScaleLocation
"Accessor to a Scale-Location plot of sqrt(abs(residuals)) versus fitted values.

Returns
-------
graph : :class:`~openturns.Graph`
"

// ---------------------------------------------------------------------

%feature("docstring") OTLM::LinearModelAnalysis::drawQQplot
"Accessor to plot a Normal quantiles-quantiles plot of standardized residuals.

Returns
-------
graph : :class:`~openturns.Graph`
"

// ---------------------------------------------------------------------

%feature("docstring") OTLM::LinearModelAnalysis::drawCookDistance
"Accessor to plot of Cook's distances versus row labels.

Returns
-------
graph : :class:`~openturns.Graph`
"

// ---------------------------------------------------------------------

%feature("docstring") OTLM::LinearModelAnalysis::drawResidualsVsLeverages
"Accessor to plot of residuals versus leverages that adds bands corresponding to Cook's distances of 0.5 and 1.

Returns
-------
graph : :class:`~openturns.Graph`
"

// ---------------------------------------------------------------------

%feature("docstring") OTLM::LinearModelAnalysis::drawCookVsLeverages
"Accessor to plot of Cook's distances versus leverage/(1-leverage). 

Returns
-------
graph : :class:`~openturns.Graph`
"

