/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.files.sp3;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Locale;
import java.util.Optional;
import java.util.Scanner;
import java.util.function.Function;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import org.hipparchus.exception.Localizable;
import org.hipparchus.geometry.euclidean.threed.Vector3D;
import org.orekit.errors.OrekitException;
import org.orekit.errors.OrekitMessages;
import org.orekit.files.general.EphemerisFileParser;
import org.orekit.files.sp3.SP3File;
import org.orekit.frames.Frame;
import org.orekit.frames.FramesFactory;
import org.orekit.time.AbsoluteDate;
import org.orekit.time.TimeScale;
import org.orekit.time.TimeScalesFactory;
import org.orekit.utils.CartesianDerivativesFilter;
import org.orekit.utils.IERSConventions;

public class SP3Parser
implements EphemerisFileParser {
    private static final String SPACES = "\\s+";
    private static final double MILLIMETER = 0.001;
    private final double mu;
    private final int interpolationSamples;
    private final Function<? super String, ? extends Frame> frameBuilder;

    public SP3Parser() {
        this(3.986004415E14, 7, SP3Parser::guessFrame);
    }

    public SP3Parser(double mu, int interpolationSamples, Function<? super String, ? extends Frame> frameBuilder) {
        this.mu = mu;
        this.interpolationSamples = interpolationSamples;
        this.frameBuilder = frameBuilder;
    }

    private static Frame guessFrame(String name) {
        return FramesFactory.getITRF(IERSConventions.IERS_2010, false);
    }

    public SP3File parse(InputStream stream) throws IOException {
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(stream, StandardCharsets.UTF_8));){
            SP3File sP3File = this.parse(reader, stream.toString());
            return sP3File;
        }
    }

    @Override
    public SP3File parse(String fileName) throws IOException, OrekitException {
        try (BufferedReader reader = Files.newBufferedReader(Paths.get(fileName, new String[0]), StandardCharsets.UTF_8);){
            SP3File sP3File = this.parse(reader, fileName);
            return sP3File;
        }
    }

    @Override
    public SP3File parse(BufferedReader reader, String fileName) throws IOException {
        ParseInfo pi = new ParseInfo();
        int lineNumber = 0;
        Stream<LineParser> candidateParsers = Stream.of(LineParser.HEADER_VERSION);
        String line = reader.readLine();
        while (line != null) {
            ++lineNumber;
            String l = line;
            Optional<LineParser> selected = candidateParsers.filter(p -> p.canHandle(l)).findFirst();
            if (selected.isPresent()) {
                try {
                    selected.get().parse(line, pi);
                }
                catch (NumberFormatException | StringIndexOutOfBoundsException e) {
                    throw new OrekitException(e, OrekitMessages.UNABLE_TO_PARSE_LINE_IN_FILE, lineNumber, fileName, line);
                }
            } else {
                throw new OrekitException((Localizable)OrekitMessages.UNABLE_TO_PARSE_LINE_IN_FILE, lineNumber, fileName, line);
            }
            candidateParsers = selected.get().allowedNext();
            if (pi.done) {
                if (pi.nbEpochs != pi.file.getNumberOfEpochs()) {
                    throw new OrekitException((Localizable)OrekitMessages.SP3_NUMBER_OF_EPOCH_MISMATCH, pi.nbEpochs, fileName, pi.file.getNumberOfEpochs());
                }
                return pi.file;
            }
            line = reader.readLine();
        }
        throw new OrekitException((Localizable)OrekitMessages.SP3_UNEXPECTED_END_OF_FILE, lineNumber);
    }

    private static SP3File.SP3FileType getFileType(String fileType) {
        SP3File.SP3FileType type = SP3File.SP3FileType.UNDEFINED;
        if ("G".equalsIgnoreCase(fileType)) {
            type = SP3File.SP3FileType.GPS;
        } else if ("M".equalsIgnoreCase(fileType)) {
            type = SP3File.SP3FileType.MIXED;
        } else if ("R".equalsIgnoreCase(fileType)) {
            type = SP3File.SP3FileType.GLONASS;
        } else if ("L".equalsIgnoreCase(fileType)) {
            type = SP3File.SP3FileType.LEO;
        } else if ("E".equalsIgnoreCase(fileType)) {
            type = SP3File.SP3FileType.GALILEO;
        } else if ("C".equalsIgnoreCase(fileType)) {
            type = SP3File.SP3FileType.COMPASS;
        } else if ("J".equalsIgnoreCase(fileType)) {
            type = SP3File.SP3FileType.QZSS;
        }
        return type;
    }

    private static enum LineParser {
        HEADER_VERSION("^#[a-z].*"){

            @Override
            public void parse(String line, ParseInfo pi) {
                try (Scanner s1 = new Scanner(line);
                     Scanner s2 = s1.useDelimiter(SP3Parser.SPACES);
                     Scanner scanner = s2.useLocale(Locale.US);){
                    scanner.skip("#");
                    String v = scanner.next();
                    char version = v.substring(0, 1).toLowerCase().charAt(0);
                    if (version != 'a' && version != 'b' && version != 'c' && version != 'd') {
                        throw new OrekitException((Localizable)OrekitMessages.SP3_UNSUPPORTED_VERSION, Character.valueOf(version));
                    }
                    pi.hasVelocityEntries = "V".equals(v.substring(1, 2));
                    pi.file.setFilter(pi.hasVelocityEntries ? CartesianDerivativesFilter.USE_PV : CartesianDerivativesFilter.USE_P);
                    int year = Integer.parseInt(v.substring(2));
                    int month = scanner.nextInt();
                    int day = scanner.nextInt();
                    int hour = scanner.nextInt();
                    int minute = scanner.nextInt();
                    double second = scanner.nextDouble();
                    AbsoluteDate epoch = new AbsoluteDate(year, month, day, hour, minute, second, (TimeScale)TimeScalesFactory.getGPS());
                    pi.file.setEpoch(epoch);
                    int numEpochs = scanner.nextInt();
                    pi.file.setNumberOfEpochs(numEpochs);
                    pi.file.setDataUsed(scanner.next());
                    pi.file.setCoordinateSystem(scanner.next());
                    pi.file.setOrbitTypeKey(scanner.next());
                    pi.file.setAgency(scanner.next());
                }
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(HEADER_DATE_TIME_REFERENCE);
            }
        }
        ,
        HEADER_DATE_TIME_REFERENCE("^##.*"){

            @Override
            public void parse(String line, ParseInfo pi) {
                try (Scanner s1 = new Scanner(line);
                     Scanner s2 = s1.useDelimiter(SP3Parser.SPACES);
                     Scanner scanner = s2.useLocale(Locale.US);){
                    scanner.skip("##");
                    pi.file.setGpsWeek(scanner.nextInt());
                    pi.file.setSecondsOfWeek(scanner.nextDouble());
                    pi.file.setEpochInterval(scanner.nextDouble());
                    pi.file.setJulianDay(scanner.nextInt());
                    pi.file.setDayFraction(scanner.nextDouble());
                }
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(HEADER_SAT_IDS);
            }
        }
        ,
        HEADER_SAT_IDS("^\\+ .*"){

            @Override
            public void parse(String line, ParseInfo pi) {
                if (pi.maxSatellites == 0) {
                    pi.maxSatellites = Integer.parseInt(line.substring(3, 6).trim());
                }
                int lineLength = line.length();
                int count = pi.file.getSatelliteCount();
                int startIdx = 9;
                while (count++ < pi.maxSatellites && startIdx + 3 <= lineLength) {
                    String satId = line.substring(startIdx, startIdx + 3).trim();
                    if (satId.length() > 0) {
                        pi.file.addSatellite(satId);
                    }
                    startIdx += 3;
                }
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(HEADER_SAT_IDS, HEADER_ACCURACY);
            }
        }
        ,
        HEADER_ACCURACY("^\\+\\+.*"){

            @Override
            public void parse(String line, ParseInfo pi) {
                int lineLength = line.length();
                int startIdx = 9;
                while (pi.nbAccuracies < pi.maxSatellites && startIdx + 3 <= lineLength) {
                    String sub = line.substring(startIdx, startIdx + 3).trim();
                    if (sub.length() > 0) {
                        int exponent = Integer.parseInt(sub);
                        pi.file.setAccuracy(pi.nbAccuracies++, (double)(2 << exponent) * 0.001);
                    }
                    startIdx += 3;
                }
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(HEADER_ACCURACY, HEADER_TIME_SYSTEM);
            }
        }
        ,
        HEADER_TIME_SYSTEM("^%c.*"){

            @Override
            public void parse(String line, ParseInfo pi) {
                if (pi.file.getType() == null) {
                    pi.file.setType(SP3Parser.getFileType(line.substring(3, 5).trim()));
                    String tsStr = line.substring(9, 12).trim();
                    pi.file.setTimeScaleString(tsStr);
                    SP3File.TimeSystem ts = tsStr.equalsIgnoreCase("ccc") ? SP3File.TimeSystem.GPS : SP3File.TimeSystem.valueOf(tsStr);
                    pi.file.setTimeSystem(ts);
                    switch (ts) {
                        case GPS: {
                            pi.timeScale = TimeScalesFactory.getGPS();
                            break;
                        }
                        case GAL: {
                            pi.timeScale = TimeScalesFactory.getGST();
                            break;
                        }
                        case GLO: {
                            pi.timeScale = TimeScalesFactory.getGLONASS();
                            break;
                        }
                        case QZS: {
                            pi.timeScale = TimeScalesFactory.getQZSS();
                            break;
                        }
                        case TAI: {
                            pi.timeScale = TimeScalesFactory.getTAI();
                            break;
                        }
                        case UTC: {
                            pi.timeScale = TimeScalesFactory.getUTC();
                            break;
                        }
                        default: {
                            pi.timeScale = TimeScalesFactory.getGPS();
                        }
                    }
                    pi.file.setTimeScale(pi.timeScale);
                }
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(HEADER_TIME_SYSTEM, HEADER_STANDARD_DEVIATIONS);
            }
        }
        ,
        HEADER_STANDARD_DEVIATIONS("^%f.*"){

            @Override
            public void parse(String line, ParseInfo pi) {
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(HEADER_STANDARD_DEVIATIONS, HEADER_CUSTOM_PARAMETERS);
            }
        }
        ,
        HEADER_CUSTOM_PARAMETERS("^%i.*"){

            @Override
            public void parse(String line, ParseInfo pi) {
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(HEADER_CUSTOM_PARAMETERS, HEADER_COMMENTS);
            }
        }
        ,
        HEADER_COMMENTS("^/\\*.*"){

            @Override
            public void parse(String line, ParseInfo pi) {
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(HEADER_COMMENTS, DATA_EPOCH);
            }
        }
        ,
        DATA_EPOCH("^\\* .*"){

            @Override
            public void parse(String line, ParseInfo pi) {
                int year = Integer.parseInt(line.substring(3, 7).trim());
                int month = Integer.parseInt(line.substring(8, 10).trim());
                int day = Integer.parseInt(line.substring(11, 13).trim());
                int hour = Integer.parseInt(line.substring(14, 16).trim());
                int minute = Integer.parseInt(line.substring(17, 19).trim());
                double second = Double.parseDouble(line.substring(20, 31).trim());
                pi.latestEpoch = new AbsoluteDate(year, month, day, hour, minute, second, pi.timeScale);
                pi.nbEpochs++;
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(DATA_POSITION);
            }
        }
        ,
        DATA_POSITION("^P.*"){

            @Override
            public void parse(String line, ParseInfo pi) {
                String satelliteId = line.substring(1, 4).trim();
                if (!pi.file.containsSatellite(satelliteId)) {
                    pi.latestPosition = null;
                } else {
                    double x = Double.parseDouble(line.substring(4, 18).trim());
                    double y = Double.parseDouble(line.substring(18, 32).trim());
                    double z = Double.parseDouble(line.substring(32, 46).trim());
                    pi.latestPosition = new Vector3D(x * 1000.0, y * 1000.0, z * 1000.0);
                    pi.latestClock = Double.parseDouble(line.substring(46, 60).trim()) * 1000000.0;
                    if (!pi.hasVelocityEntries) {
                        SP3File.SP3Coordinate coord = new SP3File.SP3Coordinate(pi.latestEpoch, pi.latestPosition, pi.latestClock);
                        pi.file.addSatelliteCoordinate(satelliteId, coord);
                    }
                }
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(DATA_EPOCH, DATA_POSITION, DATA_POSITION_CORRELATION, DATA_VELOCITY, EOF);
            }
        }
        ,
        DATA_POSITION_CORRELATION("^EP.*"){

            @Override
            public void parse(String line, ParseInfo pi) {
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(DATA_EPOCH, DATA_POSITION, DATA_VELOCITY, EOF);
            }
        }
        ,
        DATA_VELOCITY("^V.*"){

            @Override
            public void parse(String line, ParseInfo pi) {
                String satelliteId = line.substring(1, 4).trim();
                if (pi.file.containsSatellite(satelliteId)) {
                    double xv = Double.parseDouble(line.substring(4, 18).trim());
                    double yv = Double.parseDouble(line.substring(18, 32).trim());
                    double zv = Double.parseDouble(line.substring(32, 46).trim());
                    Vector3D velocity = new Vector3D(xv / 10.0, yv / 10.0, zv / 10.0);
                    double clockRateChange = Double.parseDouble(line.substring(46, 60).trim()) * 1.0E10;
                    SP3File.SP3Coordinate coord = new SP3File.SP3Coordinate(pi.latestEpoch, pi.latestPosition, velocity, pi.latestClock, clockRateChange);
                    pi.file.addSatelliteCoordinate(satelliteId, coord);
                }
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(DATA_EPOCH, DATA_POSITION, DATA_VELOCITY_CORRELATION, EOF);
            }
        }
        ,
        DATA_VELOCITY_CORRELATION("^EV.*"){

            @Override
            public void parse(String line, ParseInfo pi) {
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(DATA_EPOCH, DATA_POSITION, EOF);
            }
        }
        ,
        EOF("^[eE][oO][fF]\\s*$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                pi.done = true;
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(EOF);
            }
        };

        private final Pattern pattern;

        private LineParser(String lineRegexp) {
            this.pattern = Pattern.compile(lineRegexp);
        }

        public abstract void parse(String var1, ParseInfo var2);

        public abstract Stream<LineParser> allowedNext();

        public boolean canHandle(String line) {
            return this.pattern.matcher(line).matches();
        }
    }

    private class ParseInfo {
        private SP3File file;
        private AbsoluteDate latestEpoch;
        private Vector3D latestPosition;
        private double latestClock;
        private boolean hasVelocityEntries;
        private TimeScale timeScale;
        private int maxSatellites;
        private int nbAccuracies;
        private int nbEpochs;
        private boolean done;

        protected ParseInfo() {
            this.file = new SP3File(SP3Parser.this.mu, SP3Parser.this.interpolationSamples, SP3Parser.this.frameBuilder);
            this.latestEpoch = null;
            this.latestPosition = null;
            this.latestClock = 0.0;
            this.hasVelocityEntries = false;
            this.timeScale = TimeScalesFactory.getGPS();
            this.maxSatellites = 0;
            this.nbAccuracies = 0;
            this.nbEpochs = 0;
            this.done = false;
        }
    }
}

