/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.models.earth;

import org.hipparchus.exception.Localizable;
import org.hipparchus.geometry.euclidean.threed.Vector3D;
import org.hipparchus.util.FastMath;
import org.orekit.bodies.GeodeticPoint;
import org.orekit.errors.OrekitException;
import org.orekit.errors.OrekitMessages;
import org.orekit.models.earth.GeoMagneticElements;

public class GeoMagneticField {
    private static double a = 6378.137;
    private static double epssq = 0.006694379990141317;
    private static double ellipsoidRadius = 6371.2;
    private String modelName;
    private double epoch;
    private double[] g;
    private double[] h;
    private double[] dg;
    private double[] dh;
    private int maxN;
    private int maxNSec;
    private double validityStart;
    private double validityEnd;
    private double[] schmidtQuasiNorm;

    protected GeoMagneticField(String modelName, double epoch, int maxN, int maxNSec, double validityStart, double validityEnd) {
        this.modelName = modelName;
        this.epoch = epoch;
        this.maxN = maxN;
        this.maxNSec = maxNSec;
        this.validityStart = validityStart;
        this.validityEnd = validityEnd;
        int maxMainFieldTerms = (maxN + 1) * (maxN + 2) / 2;
        this.g = new double[maxMainFieldTerms];
        this.h = new double[maxMainFieldTerms];
        int maxSecularFieldTerms = (maxNSec + 1) * (maxNSec + 2) / 2;
        this.dg = new double[maxSecularFieldTerms];
        this.dh = new double[maxSecularFieldTerms];
        this.schmidtQuasiNorm = new double[maxMainFieldTerms + 1];
        this.schmidtQuasiNorm[0] = 1.0;
        for (int n = 1; n <= maxN; ++n) {
            int index = n * (n + 1) / 2;
            int index1 = (n - 1) * n / 2;
            this.schmidtQuasiNorm[index] = this.schmidtQuasiNorm[index1] * (double)(2 * n - 1) / (double)n;
            for (int m = 1; m <= n; ++m) {
                index = n * (n + 1) / 2 + m;
                index1 = n * (n + 1) / 2 + m - 1;
                this.schmidtQuasiNorm[index] = this.schmidtQuasiNorm[index1] * FastMath.sqrt((double)((double)((n - m + 1) * (m == 1 ? 2 : 1)) / (double)(n + m)));
            }
        }
    }

    public double getEpoch() {
        return this.epoch;
    }

    public String getModelName() {
        return this.modelName;
    }

    public double validFrom() {
        return this.validityStart;
    }

    public double validTo() {
        return this.validityEnd;
    }

    public boolean supportsTimeTransform() {
        return this.maxNSec > 0;
    }

    protected void setMainFieldCoefficients(int n, int m, double gnm, double hnm) {
        int index = n * (n + 1) / 2 + m;
        this.g[index] = gnm;
        this.h[index] = hnm;
    }

    protected void setSecularVariationCoefficients(int n, int m, double dgnm, double dhnm) {
        int index = n * (n + 1) / 2 + m;
        this.dg[index] = dgnm;
        this.dh[index] = dhnm;
    }

    public GeoMagneticElements calculateField(double latitude, double longitude, double height) {
        GeodeticPoint gp = new GeodeticPoint(FastMath.toRadians((double)latitude), FastMath.toRadians((double)longitude), height * 1000.0);
        SphericalCoordinates sph = this.transformToSpherical(gp);
        SphericalHarmonicVars vars = new SphericalHarmonicVars(sph);
        LegendreFunction legendre = new LegendreFunction(FastMath.sin((double)sph.phi));
        Vector3D magFieldSph = this.summation(sph, vars, legendre);
        Vector3D magFieldGeo = this.rotateMagneticVector(sph, gp, magFieldSph);
        return new GeoMagneticElements(magFieldGeo);
    }

    public GeoMagneticField transformModel(double year) throws OrekitException {
        if (!this.supportsTimeTransform()) {
            throw new OrekitException((Localizable)OrekitMessages.UNSUPPORTED_TIME_TRANSFORM, this.modelName, String.valueOf(this.epoch));
        }
        if (year < this.validityStart || year > this.validityEnd) {
            throw new OrekitException((Localizable)OrekitMessages.OUT_OF_RANGE_TIME_TRANSFORM, this.modelName, String.valueOf(this.epoch), year, this.validityStart, this.validityEnd);
        }
        double dt = year - this.epoch;
        int maxSecIndex = this.maxNSec * (this.maxNSec + 1) / 2 + this.maxNSec;
        GeoMagneticField transformed = new GeoMagneticField(this.modelName, year, this.maxN, this.maxNSec, this.validityStart, this.validityEnd);
        for (int n = 1; n <= this.maxN; ++n) {
            for (int m = 0; m <= n; ++m) {
                int index = n * (n + 1) / 2 + m;
                if (index <= maxSecIndex) {
                    transformed.h[index] = this.h[index] + dt * this.dh[index];
                    transformed.g[index] = this.g[index] + dt * this.dg[index];
                    transformed.dh[index] = this.dh[index];
                    transformed.dg[index] = this.dg[index];
                    continue;
                }
                transformed.h[index] = this.h[index];
                transformed.g[index] = this.g[index];
            }
        }
        return transformed;
    }

    public GeoMagneticField transformModel(GeoMagneticField otherModel, double year) throws OrekitException {
        if (year < this.validityStart || year > this.validityEnd) {
            throw new OrekitException((Localizable)OrekitMessages.OUT_OF_RANGE_TIME_TRANSFORM, this.modelName, String.valueOf(this.epoch), year, this.validityStart, this.validityEnd);
        }
        double factor = (year - this.epoch) / (otherModel.epoch - this.epoch);
        int maxNCommon = FastMath.min((int)this.maxN, (int)otherModel.maxN);
        int maxNCommonIndex = maxNCommon * (maxNCommon + 1) / 2 + maxNCommon;
        int newMaxN = FastMath.max((int)this.maxN, (int)otherModel.maxN);
        GeoMagneticField transformed = new GeoMagneticField(this.modelName, year, newMaxN, 0, this.validityStart, this.validityEnd);
        for (int n = 1; n <= newMaxN; ++n) {
            for (int m = 0; m <= n; ++m) {
                int index = n * (n + 1) / 2 + m;
                if (index <= maxNCommonIndex) {
                    transformed.h[index] = this.h[index] + factor * (otherModel.h[index] - this.h[index]);
                    transformed.g[index] = this.g[index] + factor * (otherModel.g[index] - this.g[index]);
                    continue;
                }
                if (this.maxN < otherModel.maxN) {
                    transformed.h[index] = factor * otherModel.h[index];
                    transformed.g[index] = factor * otherModel.g[index];
                    continue;
                }
                transformed.h[index] = this.h[index] + factor * -this.h[index];
                transformed.g[index] = this.g[index] + factor * -this.g[index];
            }
        }
        return transformed;
    }

    public static double getDecimalYear(int day, int month, int year) {
        int[] days = new int[]{0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334};
        int leapYear = year % 4 == 0 && (year % 100 != 0 || year % 400 == 0) ? 1 : 0;
        double dayInYear = days[month - 1] + (day - 1) + (month > 2 ? leapYear : 0);
        return (double)year + dayInYear / (365.0 + (double)leapYear);
    }

    private SphericalCoordinates transformToSpherical(GeodeticPoint gp) {
        double lat = gp.getLatitude();
        double heightAboveEllipsoid = gp.getAltitude() / 1000.0;
        double sinLat = FastMath.sin((double)lat);
        double rc = a / FastMath.sqrt((double)(1.0 - epssq * sinLat * sinLat));
        double xp = (rc + heightAboveEllipsoid) * FastMath.cos((double)lat);
        double zp = (rc * (1.0 - epssq) + heightAboveEllipsoid) * sinLat;
        double r = FastMath.hypot((double)xp, (double)zp);
        return new SphericalCoordinates(r, gp.getLongitude(), FastMath.asin((double)(zp / r)));
    }

    private Vector3D rotateMagneticVector(SphericalCoordinates sph, GeodeticPoint gp, Vector3D field) {
        double psi = sph.phi - gp.getLatitude();
        double Bz = field.getX() * FastMath.sin((double)psi) + field.getZ() * FastMath.cos((double)psi);
        double Bx = field.getX() * FastMath.cos((double)psi) - field.getZ() * FastMath.sin((double)psi);
        double By = field.getY();
        return new Vector3D(Bx, By, Bz);
    }

    private Vector3D summation(SphericalCoordinates sph, SphericalHarmonicVars vars, LegendreFunction legendre) {
        double Bx = 0.0;
        double By = 0.0;
        double Bz = 0.0;
        for (int n = 1; n <= this.maxN; ++n) {
            for (int m = 0; m <= n; ++m) {
                int index = n * (n + 1) / 2 + m;
                Bz -= vars.relativeRadiusPower[n] * (this.g[index] * vars.cmLambda[m] + this.h[index] * vars.smLambda[m]) * (1.0 + (double)n) * legendre.mP[index];
                By += vars.relativeRadiusPower[n] * (this.g[index] * vars.smLambda[m] - this.h[index] * vars.cmLambda[m]) * (double)m * legendre.mP[index];
                Bx -= vars.relativeRadiusPower[n] * (this.g[index] * vars.cmLambda[m] + this.h[index] * vars.smLambda[m]) * legendre.mPDeriv[index];
            }
        }
        double cosPhi = FastMath.cos((double)sph.phi);
        By = FastMath.abs((double)cosPhi) > 1.0E-10 ? (By /= cosPhi) : this.summationSpecial(sph, vars);
        return new Vector3D(Bx, By, Bz);
    }

    private double summationSpecial(SphericalCoordinates sph, SphericalHarmonicVars vars) {
        double sinPhi = FastMath.sin((double)sph.phi);
        double[] mPcupS = new double[this.maxN + 1];
        mPcupS[0] = 1.0;
        double By = 0.0;
        for (int n = 1; n <= this.maxN; ++n) {
            int index = n * (n + 1) / 2 + 1;
            if (n == 1) {
                mPcupS[n] = mPcupS[n - 1];
            } else {
                double k = (double)((n - 1) * (n - 1) - 1) / (double)((2 * n - 1) * (2 * n - 3));
                mPcupS[n] = sinPhi * mPcupS[n - 1] - k * mPcupS[n - 2];
            }
            By += vars.relativeRadiusPower[n] * (this.g[index] * vars.smLambda[1] - this.h[index] * vars.cmLambda[1]) * mPcupS[n] * this.schmidtQuasiNorm[index];
        }
        return By;
    }

    private class LegendreFunction {
        private double[] mP;
        private double[] mPDeriv;

        private LegendreFunction(double x) {
            int index;
            int m;
            int n;
            int numTerms = (GeoMagneticField.this.maxN + 1) * (GeoMagneticField.this.maxN + 2) / 2;
            this.mP = new double[numTerms + 1];
            this.mPDeriv = new double[numTerms + 1];
            this.mP[0] = 1.0;
            this.mPDeriv[0] = 0.0;
            double z = FastMath.sqrt((double)((1.0 - x) * (1.0 + x)));
            for (n = 1; n <= GeoMagneticField.this.maxN; ++n) {
                for (m = 0; m <= n; ++m) {
                    int index1;
                    index = n * (n + 1) / 2 + m;
                    if (n == m) {
                        index1 = (n - 1) * n / 2 + m - 1;
                        this.mP[index] = z * this.mP[index1];
                        this.mPDeriv[index] = z * this.mPDeriv[index1] + x * this.mP[index1];
                        continue;
                    }
                    if (n == 1 && m == 0) {
                        index1 = (n - 1) * n / 2 + m;
                        this.mP[index] = x * this.mP[index1];
                        this.mPDeriv[index] = x * this.mPDeriv[index1] - z * this.mP[index1];
                        continue;
                    }
                    if (n <= 1 || n == m) continue;
                    index1 = (n - 2) * (n - 1) / 2 + m;
                    int index2 = (n - 1) * n / 2 + m;
                    if (m > n - 2) {
                        this.mP[index] = x * this.mP[index2];
                        this.mPDeriv[index] = x * this.mPDeriv[index2] - z * this.mP[index2];
                        continue;
                    }
                    double k = (double)((n - 1) * (n - 1) - m * m) / (double)((2 * n - 1) * (2 * n - 3));
                    this.mP[index] = x * this.mP[index2] - k * this.mP[index1];
                    this.mPDeriv[index] = x * this.mPDeriv[index2] - z * this.mP[index2] - k * this.mPDeriv[index1];
                }
            }
            for (n = 1; n <= GeoMagneticField.this.maxN; ++n) {
                for (m = 0; m <= n; ++m) {
                    index = n * (n + 1) / 2 + m;
                    this.mP[index] = this.mP[index] * GeoMagneticField.this.schmidtQuasiNorm[index];
                    this.mPDeriv[index] = -this.mPDeriv[index] * GeoMagneticField.this.schmidtQuasiNorm[index];
                }
            }
        }
    }

    private class SphericalHarmonicVars {
        private double[] relativeRadiusPower;
        private double[] cmLambda;
        private double[] smLambda;

        private SphericalHarmonicVars(SphericalCoordinates sph) {
            this.relativeRadiusPower = new double[GeoMagneticField.this.maxN + 1];
            double p = ellipsoidRadius / sph.r;
            this.relativeRadiusPower[0] = p * p;
            for (int n = 1; n <= GeoMagneticField.this.maxN; ++n) {
                this.relativeRadiusPower[n] = this.relativeRadiusPower[n - 1] * (ellipsoidRadius / sph.r);
            }
            this.cmLambda = new double[GeoMagneticField.this.maxN + 1];
            this.smLambda = new double[GeoMagneticField.this.maxN + 1];
            this.cmLambda[0] = 1.0;
            this.smLambda[0] = 0.0;
            double cosLambda = FastMath.cos((double)sph.lambda);
            double sinLambda = FastMath.sin((double)sph.lambda);
            this.cmLambda[1] = cosLambda;
            this.smLambda[1] = sinLambda;
            for (int m = 2; m <= GeoMagneticField.this.maxN; ++m) {
                this.cmLambda[m] = this.cmLambda[m - 1] * cosLambda - this.smLambda[m - 1] * sinLambda;
                this.smLambda[m] = this.cmLambda[m - 1] * sinLambda + this.smLambda[m - 1] * cosLambda;
            }
        }
    }

    private static class SphericalCoordinates {
        private double r;
        private double lambda;
        private double phi;

        private SphericalCoordinates(double r, double lambda, double phi) {
            this.r = r;
            this.lambda = lambda;
            this.phi = phi;
        }
    }
}

