/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.files.sp3;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Locale;
import java.util.Scanner;
import java.util.function.Function;
import org.hipparchus.exception.Localizable;
import org.hipparchus.geometry.euclidean.threed.Vector3D;
import org.hipparchus.util.FastMath;
import org.orekit.errors.OrekitException;
import org.orekit.errors.OrekitMessages;
import org.orekit.files.general.EphemerisFileParser;
import org.orekit.files.sp3.SP3File;
import org.orekit.frames.Frame;
import org.orekit.frames.FramesFactory;
import org.orekit.time.AbsoluteDate;
import org.orekit.time.TimeScale;
import org.orekit.time.TimeScalesFactory;
import org.orekit.utils.CartesianDerivativesFilter;
import org.orekit.utils.IERSConventions;

public class SP3Parser
implements EphemerisFileParser {
    private final double mu;
    private final int interpolationSamples;
    private final Function<? super String, ? extends Frame> frameBuilder;

    public SP3Parser() {
        this(3.986004415E14, 7, SP3Parser::guessFrame);
    }

    public SP3Parser(double mu, int interpolationSamples, Function<? super String, ? extends Frame> frameBuilder) {
        this.mu = mu;
        this.interpolationSamples = interpolationSamples;
        this.frameBuilder = frameBuilder;
    }

    private static Frame guessFrame(String name) {
        try {
            return FramesFactory.getITRF(IERSConventions.IERS_2010, true);
        }
        catch (OrekitException e) {
            throw new RuntimeException(e);
        }
    }

    public SP3File parse(InputStream stream) throws OrekitException, IOException {
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(stream, StandardCharsets.UTF_8));){
            SP3File sP3File = this.parse(reader, stream.toString());
            return sP3File;
        }
    }

    @Override
    public SP3File parse(String fileName) throws IOException, OrekitException {
        try (BufferedReader reader = Files.newBufferedReader(Paths.get(fileName, new String[0]), StandardCharsets.UTF_8);){
            SP3File sP3File = this.parse(reader, fileName);
            return sP3File;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public SP3File parse(BufferedReader reader, String fileName) throws OrekitException, IOException {
        ParseInfo pi = new ParseInfo();
        String line = null;
        int lineNumber = 1;
        try {
            while (lineNumber < 23) {
                line = reader.readLine();
                if (line == null) {
                    throw new OrekitException((Localizable)OrekitMessages.SP3_UNEXPECTED_END_OF_FILE, lineNumber - 1);
                }
                this.parseHeaderLine(lineNumber++, line, pi);
            }
            boolean done = false;
            do {
                if ((line = reader.readLine()) == null || "EOF".equalsIgnoreCase(line.trim())) {
                    done = true;
                    continue;
                }
                if (line.length() <= 0) continue;
                this.parseContentLine(line, pi);
            } while (!done);
        }
        finally {
            try {
                reader.close();
            }
            catch (IOException iOException) {}
        }
        return pi.file;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private void parseHeaderLine(int lineNumber, String line, ParseInfo pi) throws OrekitException {
        SP3File file = pi.file;
        try (Scanner s1 = new Scanner(line);
             Scanner s2 = s1.useDelimiter("\\s+");
             Scanner scanner = s2.useLocale(Locale.US);){
            switch (lineNumber) {
                case 1: {
                    scanner.skip("#");
                    String v = scanner.next();
                    char version = v.substring(0, 1).toLowerCase().charAt(0);
                    if (version != 'a' && version != 'b' && version != 'c') {
                        throw new OrekitException((Localizable)OrekitMessages.SP3_UNSUPPORTED_VERSION, Character.valueOf(version));
                    }
                    pi.hasVelocityEntries = "V".equals(v.substring(1, 2));
                    file.setFilter(pi.hasVelocityEntries ? CartesianDerivativesFilter.USE_PV : CartesianDerivativesFilter.USE_P);
                    int year = Integer.parseInt(v.substring(2));
                    int month = scanner.nextInt();
                    int day = scanner.nextInt();
                    int hour = scanner.nextInt();
                    int minute = scanner.nextInt();
                    double second = scanner.nextDouble();
                    AbsoluteDate epoch = new AbsoluteDate(year, month, day, hour, minute, second, (TimeScale)TimeScalesFactory.getGPS());
                    file.setEpoch(epoch);
                    int numEpochs = scanner.nextInt();
                    file.setNumberOfEpochs(numEpochs);
                    file.setDataUsed(scanner.next());
                    file.setCoordinateSystem(scanner.next());
                    file.setOrbitType(SP3File.SP3OrbitType.parseType(scanner.next()));
                    file.setAgency(scanner.next());
                    return;
                }
                case 2: {
                    scanner.skip("##");
                    file.setGpsWeek(scanner.nextInt());
                    file.setSecondsOfWeek(scanner.nextDouble());
                    file.setEpochInterval(scanner.nextDouble());
                    file.setJulianDay(scanner.nextInt());
                    file.setDayFraction(scanner.nextDouble());
                    return;
                }
                case 3: {
                    pi.maxSatellites = Integer.parseInt(line.substring(4, 6).trim());
                }
                case 4: 
                case 5: 
                case 6: 
                case 7: {
                    int lineLength = line.length();
                    int count = file.getSatelliteCount();
                    int startIdx = 9;
                    while (count++ < pi.maxSatellites) {
                        if (startIdx + 3 > lineLength) return;
                        String satId = line.substring(startIdx, startIdx + 3).trim();
                        file.addSatellite(satId);
                        startIdx += 3;
                    }
                    return;
                }
                case 8: 
                case 9: 
                case 10: 
                case 11: 
                case 12: {
                    int lineLength = line.length();
                    int satIdx = (lineNumber - 8) * 17;
                    int startIdx = 9;
                    while (satIdx < pi.maxSatellites) {
                        if (startIdx + 3 > lineLength) return;
                        int exponent = Integer.parseInt(line.substring(startIdx, startIdx + 3).trim());
                        file.setAccuracy(satIdx++, FastMath.pow((double)2.0, (int)exponent));
                        startIdx += 3;
                    }
                    return;
                }
                case 13: {
                    file.setType(this.getFileType(line.substring(3, 5).trim()));
                    String tsStr = line.substring(9, 12).trim();
                    file.setTimeScaleString(tsStr);
                    SP3File.TimeSystem ts = tsStr.equalsIgnoreCase("ccc") ? SP3File.TimeSystem.GPS : SP3File.TimeSystem.valueOf(tsStr);
                    file.setTimeSystem(ts);
                    switch (ts) {
                        case GPS: {
                            pi.timeScale = TimeScalesFactory.getGPS();
                            break;
                        }
                        case GAL: {
                            pi.timeScale = TimeScalesFactory.getGST();
                            break;
                        }
                        case GLO: {
                            pi.timeScale = TimeScalesFactory.getGLONASS();
                            break;
                        }
                        case QZS: {
                            pi.timeScale = TimeScalesFactory.getQZSS();
                        }
                        case TAI: {
                            pi.timeScale = TimeScalesFactory.getTAI();
                            break;
                        }
                        case UTC: {
                            pi.timeScale = TimeScalesFactory.getUTC();
                            break;
                        }
                        default: {
                            pi.timeScale = TimeScalesFactory.getGPS();
                        }
                    }
                    file.setTimeScale(pi.timeScale);
                    return;
                }
                case 14: {
                    return;
                }
                case 15: {
                    return;
                }
                case 16: 
                case 17: 
                case 18: {
                    return;
                }
                case 19: 
                case 20: 
                case 21: 
                case 22: {
                    return;
                }
            }
            return;
        }
    }

    private void parseContentLine(String line, ParseInfo pi) {
        SP3File file = pi.file;
        switch (line.charAt(0)) {
            case '*': {
                int year = Integer.parseInt(line.substring(3, 7).trim());
                int month = Integer.parseInt(line.substring(8, 10).trim());
                int day = Integer.parseInt(line.substring(11, 13).trim());
                int hour = Integer.parseInt(line.substring(14, 16).trim());
                int minute = Integer.parseInt(line.substring(17, 19).trim());
                double second = Double.parseDouble(line.substring(20, 31).trim());
                pi.latestEpoch = new AbsoluteDate(year, month, day, hour, minute, second, pi.timeScale);
                break;
            }
            case 'P': {
                String satelliteId = line.substring(1, 4).trim();
                if (!file.containsSatellite(satelliteId)) {
                    pi.latestPosition = null;
                    break;
                }
                double x = Double.parseDouble(line.substring(4, 18).trim());
                double y = Double.parseDouble(line.substring(18, 32).trim());
                double z = Double.parseDouble(line.substring(32, 46).trim());
                pi.latestPosition = new Vector3D(x * 1000.0, y * 1000.0, z * 1000.0);
                pi.latestClock = Double.parseDouble(line.substring(46, 60).trim()) * 1000000.0;
                if (pi.hasVelocityEntries) break;
                SP3File.SP3Coordinate coord = new SP3File.SP3Coordinate(pi.latestEpoch, pi.latestPosition, pi.latestClock);
                file.addSatelliteCoordinate(satelliteId, coord);
                break;
            }
            case 'V': {
                String satelliteId = line.substring(1, 4).trim();
                if (!file.containsSatellite(satelliteId)) break;
                double xv = Double.parseDouble(line.substring(4, 18).trim());
                double yv = Double.parseDouble(line.substring(18, 32).trim());
                double zv = Double.parseDouble(line.substring(32, 46).trim());
                Vector3D velocity = new Vector3D(xv / 10.0, yv / 10.0, zv / 10.0);
                double clockRateChange = Double.parseDouble(line.substring(46, 60).trim()) * 1.0E10;
                SP3File.SP3Coordinate coord = new SP3File.SP3Coordinate(pi.latestEpoch, pi.latestPosition, velocity, pi.latestClock, clockRateChange);
                file.addSatelliteCoordinate(satelliteId, coord);
                break;
            }
        }
    }

    private SP3File.SP3FileType getFileType(String fileType) {
        SP3File.SP3FileType type = SP3File.SP3FileType.UNDEFINED;
        if ("G".equalsIgnoreCase(fileType)) {
            type = SP3File.SP3FileType.GPS;
        } else if ("M".equalsIgnoreCase(fileType)) {
            type = SP3File.SP3FileType.MIXED;
        } else if ("R".equalsIgnoreCase(fileType)) {
            type = SP3File.SP3FileType.GLONASS;
        } else if ("L".equalsIgnoreCase(fileType)) {
            type = SP3File.SP3FileType.LEO;
        } else if ("E".equalsIgnoreCase(fileType)) {
            type = SP3File.SP3FileType.GALILEO;
        } else if ("C".equalsIgnoreCase(fileType)) {
            type = SP3File.SP3FileType.COMPASS;
        } else if ("J".equalsIgnoreCase(fileType)) {
            type = SP3File.SP3FileType.QZSS;
        }
        return type;
    }

    private class ParseInfo {
        private SP3File file;
        private AbsoluteDate latestEpoch;
        private Vector3D latestPosition;
        private double latestClock;
        private boolean hasVelocityEntries;
        private TimeScale timeScale;
        private int maxSatellites;

        protected ParseInfo() {
            this.file = new SP3File(SP3Parser.this.mu, SP3Parser.this.interpolationSamples, SP3Parser.this.frameBuilder);
            this.latestEpoch = null;
            this.latestPosition = null;
            this.latestClock = 0.0;
            this.hasVelocityEntries = false;
            this.timeScale = TimeScalesFactory.getGPS();
            this.maxSatellites = 0;
        }
    }
}

