/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.gnss.clock;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.InputMismatchException;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.Scanner;
import java.util.function.Function;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import org.hipparchus.exception.Localizable;
import org.orekit.annotation.DefaultDataContext;
import org.orekit.data.DataContext;
import org.orekit.errors.OrekitException;
import org.orekit.errors.OrekitMessages;
import org.orekit.frames.Frame;
import org.orekit.gnss.AppliedDCBS;
import org.orekit.gnss.AppliedPCVS;
import org.orekit.gnss.ObservationType;
import org.orekit.gnss.SatelliteSystem;
import org.orekit.gnss.TimeSystem;
import org.orekit.gnss.clock.RinexClock;
import org.orekit.time.AbsoluteDate;
import org.orekit.time.DateComponents;
import org.orekit.time.TimeComponents;
import org.orekit.time.TimeScale;
import org.orekit.time.TimeScales;
import org.orekit.utils.IERSConventions;

public class RinexClockParser {
    private static final List<Double> HANDLED_VERSIONS = Arrays.asList(2.0, 3.0, 3.01, 3.02, 3.04);
    private static final Pattern DATE_PATTERN_1 = Pattern.compile("^[0-9]{4}-[0-9]{2}-[0-9]{2} [0-9]{2}:[0-9]{2}.*$");
    private static final Pattern DATE_PATTERN_2 = Pattern.compile("^[0-9]{8}\\s{1,2}[0-9]{6}.*$");
    private static final Pattern DATE_PATTERN_3 = Pattern.compile("^[0-9]{1,2}-[a-z,A-Z]{3}-[0-9]{4} [0-9]{2}:[0-9]{2}.*$");
    private static final Pattern DATE_PATTERN_4 = Pattern.compile("^[0-9]{1,2}-[a-z,A-Z]{3}-[0-9]{2} [0-9]{2}:[0-9]{2}.*$");
    private static final Pattern DATE_PATTERN_5 = Pattern.compile("^[0-9]{4} [a-z,A-Z]{3} [0-9]{1,2} [0-9]{2}:[0-9]{2}:[0-9]{2}.*$");
    private static final String SPACES = "\\s+";
    private static final String SYS = "SYS";
    private static final double MILLIMETER = 0.001;
    private final Function<? super String, ? extends Frame> frameBuilder;
    private final TimeScales timeScales;

    @DefaultDataContext
    public RinexClockParser() {
        this(RinexClockParser::guessFrame);
    }

    @DefaultDataContext
    public RinexClockParser(Function<? super String, ? extends Frame> frameBuilder) {
        this(frameBuilder, DataContext.getDefault().getTimeScales());
    }

    public RinexClockParser(Function<? super String, ? extends Frame> frameBuilder, TimeScales timeScales) {
        this.frameBuilder = frameBuilder;
        this.timeScales = timeScales;
    }

    @DefaultDataContext
    private static Frame guessFrame(String name) {
        if (name.equals("ITRF96")) {
            return DataContext.getDefault().getFrames().getITRF(IERSConventions.IERS_1996, false);
        }
        return DataContext.getDefault().getFrames().getITRF(IERSConventions.IERS_2010, false);
    }

    public RinexClock parse(InputStream stream) throws IOException {
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(stream, StandardCharsets.UTF_8));){
            RinexClock rinexClock = this.parse(reader, stream.toString());
            return rinexClock;
        }
    }

    public RinexClock parse(String fileName) throws IOException {
        try (BufferedReader reader = Files.newBufferedReader(Paths.get(fileName, new String[0]), StandardCharsets.UTF_8);){
            RinexClock rinexClock = this.parse(reader, fileName);
            return rinexClock;
        }
    }

    public RinexClock parse(BufferedReader reader, String fileName) throws IOException {
        ParseInfo pi = new ParseInfo();
        int lineNumber = 0;
        Stream<LineParser> candidateParsers = Stream.of(LineParser.HEADER_VERSION);
        String line = reader.readLine();
        while (line != null) {
            ++lineNumber;
            String l = line;
            Optional<LineParser> selected = candidateParsers.filter(p -> p.canHandle(l)).findFirst();
            if (selected.isPresent()) {
                try {
                    selected.get().parse(line, pi);
                }
                catch (NumberFormatException | StringIndexOutOfBoundsException | InputMismatchException e) {
                    throw new OrekitException(e, OrekitMessages.UNABLE_TO_PARSE_LINE_IN_FILE, lineNumber, fileName, line);
                }
            } else {
                throw new OrekitException((Localizable)OrekitMessages.UNABLE_TO_PARSE_LINE_IN_FILE, lineNumber, fileName, line);
            }
            candidateParsers = selected.get().allowedNext();
            line = reader.readLine();
        }
        return pi.file;
    }

    private static enum LineParser {
        HEADER_VERSION("^.+RINEX VERSION / TYPE( )*$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                try (Scanner s1 = new Scanner(line);
                     Scanner s2 = s1.useDelimiter(RinexClockParser.SPACES);
                     Scanner scanner = s2.useLocale(Locale.US);){
                    double version = scanner.nextDouble();
                    if (!HANDLED_VERSIONS.contains(version)) {
                        throw new OrekitException((Localizable)OrekitMessages.CLOCK_FILE_UNSUPPORTED_VERSION, version);
                    }
                    pi.file.setFormatVersion(version);
                    String satelliteSystemString = line.substring(40, 45).trim();
                    if (!satelliteSystemString.equals("")) {
                        SatelliteSystem satelliteSystem = SatelliteSystem.parseSatelliteSystem(satelliteSystemString);
                        pi.file.setSatelliteSystem(satelliteSystem);
                        pi.file.setTimeScale(satelliteSystem.getDefaultTimeSystem(pi.timeScales));
                    }
                    if (pi.file.getTimeScale() == null) {
                        pi.file.setTimeScale(pi.timeScales.getUTC());
                    }
                }
            }
        }
        ,
        HEADER_PROGRAM("^.+PGM / RUN BY / DATE( )*$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                String programName = line.substring(0, 20).trim();
                pi.file.setProgramName(programName);
                String agencyName = line.substring(20, 40).trim();
                pi.file.setAgencyName(agencyName);
                String dateString = "";
                dateString = pi.file.getFormatVersion() < 3.04 ? line.substring(40, 60) : line.substring(42, 65);
                LineParser.parseDateTimeZone(dateString, pi);
            }
        }
        ,
        HEADER_COMMENT("^.+COMMENT( )*$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                if (pi.file.getFormatVersion() < 3.04) {
                    pi.file.addComment(line.substring(0, 60).trim());
                } else {
                    pi.file.addComment(line.substring(0, 65).trim());
                }
            }
        }
        ,
        HEADER_SYSTEM_OBS("^[A-Z] .*SYS / # / OBS TYPES( )*$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                try (Scanner s1 = new Scanner(line);
                     Scanner s2 = s1.useDelimiter(RinexClockParser.SPACES);
                     Scanner scanner = s2.useLocale(Locale.US);){
                    SatelliteSystem satelliteSystem = SatelliteSystem.parseSatelliteSystem(scanner.next());
                    pi.currentSatelliteSystem = satelliteSystem;
                    scanner.nextInt();
                    String currentObsType = scanner.next();
                    while (!currentObsType.equals(RinexClockParser.SYS)) {
                        ObservationType obsType = ObservationType.valueOf(currentObsType);
                        pi.file.addSystemObservationType(satelliteSystem, obsType);
                        currentObsType = scanner.next();
                    }
                }
            }
        }
        ,
        HEADER_SYSTEM_OBS_CONTINUATION("^ .*SYS / # / OBS TYPES( )*$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                try (Scanner s1 = new Scanner(line);
                     Scanner s2 = s1.useDelimiter(RinexClockParser.SPACES);
                     Scanner scanner = s2.useLocale(Locale.US);){
                    String currentObsType = scanner.next();
                    while (!currentObsType.equals(RinexClockParser.SYS)) {
                        ObservationType obsType = ObservationType.valueOf(currentObsType);
                        pi.file.addSystemObservationType(pi.currentSatelliteSystem, obsType);
                        currentObsType = scanner.next();
                    }
                }
            }
        }
        ,
        HEADER_TIME_SYSTEM("^.+TIME SYSTEM ID( )*$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                try (Scanner s1 = new Scanner(line);
                     Scanner s2 = s1.useDelimiter(RinexClockParser.SPACES);
                     Scanner scanner = s2.useLocale(Locale.US);){
                    TimeSystem timeSystem = TimeSystem.parseTimeSystem(scanner.next());
                    TimeScale timeScale = timeSystem.getTimeScale(pi.timeScales);
                    pi.file.setTimeSystem(timeSystem);
                    pi.file.setTimeScale(timeScale);
                }
            }
        }
        ,
        HEADER_LEAP_SECONDS("^.+LEAP SECONDS( )*$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                try (Scanner s1 = new Scanner(line);
                     Scanner s2 = s1.useDelimiter(RinexClockParser.SPACES);
                     Scanner scanner = s2.useLocale(Locale.US);){
                    int numberOfLeapSeconds = scanner.nextInt();
                    pi.file.setNumberOfLeapSeconds(numberOfLeapSeconds);
                }
            }
        }
        ,
        HEADER_LEAP_SECONDS_GNSS("^.+LEAP SECONDS GNSS( )*$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                try (Scanner s1 = new Scanner(line);
                     Scanner s2 = s1.useDelimiter(RinexClockParser.SPACES);
                     Scanner scanner = s2.useLocale(Locale.US);){
                    int numberOfLeapSecondsGNSS = scanner.nextInt();
                    pi.file.setNumberOfLeapSecondsGNSS(numberOfLeapSecondsGNSS);
                }
            }
        }
        ,
        HEADER_DCBS("^.+SYS / DCBS APPLIED( )*$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                SatelliteSystem satelliteSystem = SatelliteSystem.parseSatelliteSystem(line.substring(0, 1));
                String progDCBS = line.substring(2, 20).trim();
                String sourceDCBS = "";
                sourceDCBS = pi.file.getFormatVersion() < 3.04 ? line.substring(19, 60).trim() : line.substring(22, 65).trim();
                if (!progDCBS.equals("")) {
                    pi.file.addAppliedDCBS(new AppliedDCBS(satelliteSystem, progDCBS, sourceDCBS));
                }
            }
        }
        ,
        HEADER_PCVS("^.+SYS / PCVS APPLIED( )*$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                SatelliteSystem satelliteSystem = SatelliteSystem.parseSatelliteSystem(line.substring(0, 1));
                String progPCVS = line.substring(2, 20).trim();
                String sourcePCVS = "";
                sourcePCVS = pi.file.getFormatVersion() < 3.04 ? line.substring(19, 60).trim() : line.substring(22, 65).trim();
                if (!progPCVS.equals("") || !sourcePCVS.equals("")) {
                    pi.file.addAppliedPCVS(new AppliedPCVS(satelliteSystem, progPCVS, sourcePCVS));
                }
            }
        }
        ,
        HEADER_TYPES_OF_DATA("^.+# / TYPES OF DATA( )*$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                try (Scanner s1 = new Scanner(line);
                     Scanner s2 = s1.useDelimiter(RinexClockParser.SPACES);
                     Scanner scanner = s2.useLocale(Locale.US);){
                    int numberOfDifferentDataTypes = scanner.nextInt();
                    for (int i = 0; i < numberOfDifferentDataTypes; ++i) {
                        RinexClock.ClockDataType dataType = RinexClock.ClockDataType.parseClockDataType(scanner.next());
                        pi.file.addClockDataType(dataType);
                    }
                }
            }
        }
        ,
        HEADER_STATIONS_NAME("^.+STATION NAME / NUM( )*$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                try (Scanner s1 = new Scanner(line);
                     Scanner s2 = s1.useDelimiter(RinexClockParser.SPACES);
                     Scanner scanner = s2.useLocale(Locale.US);){
                    String stationName = scanner.next();
                    pi.file.setStationName(stationName);
                    String stationIdentifier = scanner.next();
                    pi.file.setStationIdentifier(stationIdentifier);
                }
            }
        }
        ,
        HEADER_STATION_CLOCK_REF("^.+STATION CLK REF( )*$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                if (pi.file.getFormatVersion() < 3.04) {
                    pi.file.setExternalClockReference(line.substring(0, 60).trim());
                } else {
                    pi.file.setExternalClockReference(line.substring(0, 65).trim());
                }
            }
        }
        ,
        HEADER_ANALYSIS_CENTER("^.+ANALYSIS CENTER( )*$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                String analysisCenterID = line.substring(0, 3).trim();
                pi.file.setAnalysisCenterID(analysisCenterID);
                String analysisCenterName = "";
                analysisCenterName = pi.file.getFormatVersion() < 3.04 ? line.substring(5, 60).trim() : line.substring(5, 65).trim();
                pi.file.setAnalysisCenterName(analysisCenterName);
            }
        }
        ,
        HEADER_NUMBER_OF_CLOCK_REF("^.+# OF CLK REF( )*$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                try (Scanner s1 = new Scanner(line);
                     Scanner s2 = s1.useDelimiter(RinexClockParser.SPACES);
                     Scanner scanner = s2.useLocale(Locale.US);){
                    pi.currentReferenceClocks = new ArrayList();
                    scanner.nextInt();
                    if (scanner.hasNextInt()) {
                        int startYear = scanner.nextInt();
                        int startMonth = scanner.nextInt();
                        int startDay = scanner.nextInt();
                        int startHour = scanner.nextInt();
                        int startMin = scanner.nextInt();
                        double startSec = scanner.nextDouble();
                        AbsoluteDate startEpoch = new AbsoluteDate(startYear, startMonth, startDay, startHour, startMin, startSec, pi.file.getTimeScale());
                        pi.referenceClockStartDate = startEpoch;
                        int endYear = scanner.nextInt();
                        int endMonth = scanner.nextInt();
                        int endDay = scanner.nextInt();
                        int endHour = scanner.nextInt();
                        int endMin = scanner.nextInt();
                        double endSec = 0.0;
                        endSec = pi.file.getFormatVersion() < 3.04 ? Double.parseDouble(line.substring(51, 60)) : scanner.nextDouble();
                        AbsoluteDate endEpoch = new AbsoluteDate(endYear, endMonth, endDay, endHour, endMin, endSec, pi.file.getTimeScale());
                        pi.referenceClockEndDate = endEpoch;
                    } else {
                        pi.referenceClockStartDate = AbsoluteDate.PAST_INFINITY;
                        pi.referenceClockEndDate = AbsoluteDate.FUTURE_INFINITY;
                    }
                }
            }
        }
        ,
        HEADER_ANALYSIS_CLOCK_REF("^.+ANALYSIS CLK REF( )*$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                try (Scanner s1 = new Scanner(line);
                     Scanner s2 = s1.useDelimiter(RinexClockParser.SPACES);
                     Scanner scanner = s2.useLocale(Locale.US);){
                    String referenceName = scanner.next();
                    String clockID = scanner.next();
                    double clockConstraint = 0.0;
                    if (scanner.hasNextDouble()) {
                        clockConstraint = scanner.nextDouble();
                    }
                    RinexClock.ReferenceClock referenceClock = new RinexClock.ReferenceClock(referenceName, clockID, clockConstraint, pi.referenceClockStartDate, pi.referenceClockEndDate);
                    pi.currentReferenceClocks.add(referenceClock);
                    pi.file.addReferenceClockList(pi.currentReferenceClocks, pi.referenceClockStartDate);
                }
            }
        }
        ,
        HEADER_NUMBER_OF_SOLN_STATIONS("^.+SOLN STA / TRF( )*$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                try (Scanner s1 = new Scanner(line);
                     Scanner s2 = s1.useDelimiter(RinexClockParser.SPACES);
                     Scanner scanner = s2.useLocale(Locale.US);){
                    scanner.nextInt();
                    String frameString = scanner.next();
                    pi.file.setFrameName(frameString);
                }
            }
        }
        ,
        HEADER_SOLN_STATIONS("^.+SOLN STA NAME / NUM( )*$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                String designator = line.substring(0, 10).trim();
                String receiverIdentifier = line.substring(10, 30).trim();
                String xString = "";
                String yString = "";
                String zString = "";
                if (pi.file.getFormatVersion() < 3.04) {
                    designator = line.substring(0, 4).trim();
                    receiverIdentifier = line.substring(5, 25).trim();
                    xString = line.substring(25, 36).trim();
                    yString = line.substring(37, 48).trim();
                    zString = line.substring(49, 60).trim();
                } else {
                    designator = line.substring(0, 10).trim();
                    receiverIdentifier = line.substring(10, 30).trim();
                    xString = line.substring(30, 41).trim();
                    yString = line.substring(42, 53).trim();
                    zString = line.substring(54, 65).trim();
                }
                double x = 0.001 * Double.parseDouble(xString);
                double y = 0.001 * Double.parseDouble(yString);
                double z = 0.001 * Double.parseDouble(zString);
                RinexClock.Receiver receiver = new RinexClock.Receiver(designator, receiverIdentifier, x, y, z);
                pi.file.addReceiver(receiver);
            }
        }
        ,
        HEADER_NUMBER_OF_SOLN_SATS("^.+# OF SOLN SATS( )*$"){

            @Override
            public void parse(String line, ParseInfo pi) {
            }
        }
        ,
        HEADER_PRN_LIST("^.+PRN LIST( )*$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                try (Scanner s1 = new Scanner(line);
                     Scanner s2 = s1.useDelimiter(RinexClockParser.SPACES);
                     Scanner scanner = s2.useLocale(Locale.US);){
                    String prn = scanner.next();
                    while (!prn.equals("PRN")) {
                        pi.file.addSatellite(prn);
                        prn = scanner.next();
                    }
                }
            }
        }
        ,
        HEADER_END("^.+END OF HEADER( )*$"){

            @Override
            public void parse(String line, ParseInfo pi) {
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(CLOCK_DATA);
            }
        }
        ,
        CLOCK_DATA("(^AR |^AS |^CR |^DR |^MS ).+$"){

            @Override
            public void parse(String line, ParseInfo pi) {
                try (Scanner s1 = new Scanner(line);
                     Scanner s2 = s1.useDelimiter(RinexClockParser.SPACES);
                     Scanner scanner = s2.useLocale(Locale.US);){
                    ParseInfo.access$1102(pi, new double[6]);
                    pi.currentDataType = RinexClock.ClockDataType.parseClockDataType(scanner.next());
                    pi.currentName = scanner.next();
                    int year = scanner.nextInt();
                    int month = scanner.nextInt();
                    int day = scanner.nextInt();
                    int hour = scanner.nextInt();
                    int min = scanner.nextInt();
                    double sec = scanner.nextDouble();
                    pi.currentDateComponents = new DateComponents(year, month, day);
                    pi.currentTimeComponents = new TimeComponents(hour, min, sec);
                    pi.currentNumberOfValues = scanner.nextInt();
                    int i = 0;
                    while (scanner.hasNextDouble()) {
                        ((ParseInfo)pi).currentDataValues[i++] = scanner.nextDouble();
                    }
                    if (pi.currentNumberOfValues <= 2) {
                        RinexClock rinexClock = pi.file;
                        String string = pi.currentName;
                        RinexClock rinexClock2 = pi.file;
                        Objects.requireNonNull(rinexClock2);
                        rinexClock.addClockData(string, rinexClock2.new RinexClock.ClockDataLine(pi.currentDataType, pi.currentName, pi.currentDateComponents, pi.currentTimeComponents, pi.currentNumberOfValues, pi.currentDataValues[0], pi.currentDataValues[1], 0.0, 0.0, 0.0, 0.0));
                    }
                }
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(CLOCK_DATA, CLOCK_DATA_CONTINUATION);
            }
        }
        ,
        CLOCK_DATA_CONTINUATION("^   .+"){

            @Override
            public void parse(String line, ParseInfo pi) {
                try (Scanner s1 = new Scanner(line);
                     Scanner s2 = s1.useDelimiter(RinexClockParser.SPACES);
                     Scanner scanner = s2.useLocale(Locale.US);){
                    int i = 2;
                    while (scanner.hasNextDouble()) {
                        ((ParseInfo)pi).currentDataValues[i++] = scanner.nextDouble();
                    }
                    RinexClock rinexClock = pi.file;
                    String string = pi.currentName;
                    RinexClock rinexClock2 = pi.file;
                    Objects.requireNonNull(rinexClock2);
                    rinexClock.addClockData(string, rinexClock2.new RinexClock.ClockDataLine(pi.currentDataType, pi.currentName, pi.currentDateComponents, pi.currentTimeComponents, pi.currentNumberOfValues, pi.currentDataValues[0], pi.currentDataValues[1], pi.currentDataValues[2], pi.currentDataValues[3], pi.currentDataValues[4], pi.currentDataValues[5]));
                }
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(CLOCK_DATA);
            }
        };

        private final Pattern pattern;

        private LineParser(String lineRegexp) {
            this.pattern = Pattern.compile(lineRegexp);
        }

        public abstract void parse(String var1, ParseInfo var2);

        public Stream<LineParser> allowedNext() {
            return Stream.of(HEADER_PROGRAM, HEADER_COMMENT, HEADER_SYSTEM_OBS, HEADER_SYSTEM_OBS_CONTINUATION, HEADER_TIME_SYSTEM, HEADER_LEAP_SECONDS, HEADER_LEAP_SECONDS_GNSS, HEADER_DCBS, HEADER_PCVS, HEADER_TYPES_OF_DATA, HEADER_STATIONS_NAME, HEADER_STATION_CLOCK_REF, HEADER_ANALYSIS_CENTER, HEADER_NUMBER_OF_CLOCK_REF, HEADER_ANALYSIS_CLOCK_REF, HEADER_NUMBER_OF_SOLN_STATIONS, HEADER_SOLN_STATIONS, HEADER_NUMBER_OF_SOLN_SATS, HEADER_PRN_LIST, HEADER_END);
        }

        public boolean canHandle(String line) {
            return this.pattern.matcher(line).matches();
        }

        private static void parseDateTimeZone(String dateString, ParseInfo pi) {
            String date = "";
            String time = "";
            String zone = "";
            DateComponents dateComponents = null;
            TimeComponents timeComponents = null;
            if (DATE_PATTERN_1.matcher(dateString).matches()) {
                date = dateString.substring(0, 10).trim();
                time = dateString.substring(11, 16).trim();
                zone = dateString.substring(16).trim();
            } else if (DATE_PATTERN_2.matcher(dateString).matches()) {
                date = dateString.substring(0, 8).trim();
                time = dateString.substring(9, 16).trim();
                zone = dateString.substring(16).trim();
                if (!zone.equals("")) {
                    dateComponents = new DateComponents(Integer.parseInt(date.substring(0, 4)), Integer.parseInt(date.substring(4, 6)), Integer.parseInt(date.substring(6, 8)));
                    timeComponents = new TimeComponents(Integer.parseInt(time.substring(0, 2)), Integer.parseInt(time.substring(2, 4)), Integer.parseInt(time.substring(4, 6)));
                }
            } else if (DATE_PATTERN_3.matcher(dateString).matches()) {
                date = dateString.substring(0, 11).trim();
                time = dateString.substring(11, 17).trim();
                zone = dateString.substring(17).trim();
            } else if (DATE_PATTERN_4.matcher(dateString).matches()) {
                date = dateString.substring(0, 9).trim();
                time = dateString.substring(9, 15).trim();
                zone = dateString.substring(15).trim();
            } else if (DATE_PATTERN_5.matcher(dateString).matches()) {
                date = dateString.substring(0, 11).trim();
                time = dateString.substring(11, 20).trim();
            }
            pi.file.setCreationDateString(date);
            pi.file.setCreationTimeString(time);
            pi.file.setCreationTimeZoneString(zone);
            if (dateComponents != null) {
                pi.file.setCreationDate(new AbsoluteDate(dateComponents, timeComponents, TimeSystem.parseTimeSystem(zone).getTimeScale(pi.timeScales)));
            }
        }
    }

    private class ParseInfo {
        private final TimeScales timeScales;
        private RinexClock file;
        private SatelliteSystem currentSatelliteSystem;
        private AbsoluteDate referenceClockStartDate;
        private AbsoluteDate referenceClockEndDate;
        private List<RinexClock.ReferenceClock> currentReferenceClocks;
        private RinexClock.ClockDataType currentDataType;
        private String currentName;
        private DateComponents currentDateComponents;
        private TimeComponents currentTimeComponents;
        private int currentNumberOfValues;
        private double[] currentDataValues;

        protected ParseInfo() {
            this.timeScales = RinexClockParser.this.timeScales;
            this.file = new RinexClock(RinexClockParser.this.frameBuilder);
        }

        static /* synthetic */ double[] access$1102(ParseInfo x0, double[] x1) {
            x0.currentDataValues = x1;
            return x1;
        }
    }
}

