/*
 * Decompiled with CFR 0.152.
 */
package org.hipparchus.fraction;

import java.io.Serializable;
import java.math.BigInteger;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Stream;
import org.hipparchus.FieldElement;
import org.hipparchus.exception.LocalizedCoreFormats;
import org.hipparchus.exception.MathIllegalStateException;
import org.hipparchus.exception.MathRuntimeException;
import org.hipparchus.fraction.ConvergentsIterator;
import org.hipparchus.fraction.FractionField;
import org.hipparchus.util.ArithmeticUtils;
import org.hipparchus.util.FastMath;
import org.hipparchus.util.MathUtils;
import org.hipparchus.util.Pair;
import org.hipparchus.util.Precision;

public class Fraction
extends Number
implements FieldElement<Fraction>,
Comparable<Fraction>,
Serializable {
    public static final Fraction TWO = new Fraction(2, 1);
    public static final Fraction ONE = new Fraction(1, 1);
    public static final Fraction ZERO = new Fraction(0, 1);
    public static final Fraction FOUR_FIFTHS = new Fraction(4, 5);
    public static final Fraction ONE_FIFTH = new Fraction(1, 5);
    public static final Fraction ONE_HALF = new Fraction(1, 2);
    public static final Fraction ONE_QUARTER = new Fraction(1, 4);
    public static final Fraction ONE_THIRD = new Fraction(1, 3);
    public static final Fraction THREE_FIFTHS = new Fraction(3, 5);
    public static final Fraction THREE_QUARTERS = new Fraction(3, 4);
    public static final Fraction TWO_FIFTHS = new Fraction(2, 5);
    public static final Fraction TWO_QUARTERS = new Fraction(2, 4);
    public static final Fraction TWO_THIRDS = new Fraction(2, 3);
    public static final Fraction MINUS_ONE = new Fraction(-1, 1);
    private static final long serialVersionUID = 3698073679419233275L;
    private static final double DEFAULT_EPSILON = 1.0E-5;
    private static final Function<ConvergentsIterator.ConvergenceStep, Fraction> STEP_TO_FRACTION = s -> new Fraction((int)s.getNumerator(), (int)s.getDenominator());
    private final int denominator;
    private final int numerator;

    public Fraction(double value) throws MathIllegalStateException {
        this(value, 1.0E-5, 100);
    }

    public Fraction(double value, double epsilon, int maxIterations) throws MathIllegalStateException {
        ConvergentsIterator.ConvergenceStep converged = Fraction.convergent(value, maxIterations, (ConvergentsIterator.ConvergenceStep s) -> {
            double quotient = s.getFractionValue();
            return Precision.equals(quotient, value, 1) || FastMath.abs(quotient - value) < epsilon;
        }).getKey();
        if (!(FastMath.abs(converged.getFractionValue() - value) < epsilon)) {
            throw new MathIllegalStateException(LocalizedCoreFormats.FAILED_FRACTION_CONVERSION, value, maxIterations);
        }
        this.numerator = (int)converged.getNumerator();
        this.denominator = (int)converged.getDenominator();
    }

    public Fraction(double value, int maxDenominator) throws MathIllegalStateException {
        int maxIterations = 100;
        ConvergentsIterator.ConvergenceStep[] lastValid = new ConvergentsIterator.ConvergenceStep[1];
        try {
            Fraction.convergent(value, 100, (ConvergentsIterator.ConvergenceStep s) -> {
                if (s.getDenominator() < (long)maxDenominator) {
                    lastValid[0] = s;
                }
                return Precision.equals(s.getFractionValue(), value, 1);
            });
        }
        catch (MathIllegalStateException mathIllegalStateException) {
            // empty catch block
        }
        if (lastValid[0] == null) {
            throw new MathIllegalStateException(LocalizedCoreFormats.FAILED_FRACTION_CONVERSION, value, 100);
        }
        this.numerator = (int)lastValid[0].getNumerator();
        this.denominator = (int)lastValid[0].getDenominator();
    }

    public Fraction(int num) {
        this(num, 1);
    }

    public Fraction(int num, int den) {
        int d;
        if (den == 0) {
            throw new MathRuntimeException(LocalizedCoreFormats.ZERO_DENOMINATOR_IN_FRACTION, num, den);
        }
        if (den < 0) {
            if (num == Integer.MIN_VALUE || den == Integer.MIN_VALUE) {
                throw new MathRuntimeException(LocalizedCoreFormats.OVERFLOW_IN_FRACTION, num, den);
            }
            num = -num;
            den = -den;
        }
        if ((d = ArithmeticUtils.gcd(num, den)) > 1) {
            num /= d;
            den /= d;
        }
        if (den < 0) {
            num = -num;
            den = -den;
        }
        this.numerator = num;
        this.denominator = den;
    }

    public static Stream<Fraction> convergents(double value, int maxConvergents) {
        if (FastMath.abs(value) > 2.147483647E9) {
            throw new MathIllegalStateException(LocalizedCoreFormats.FRACTION_CONVERSION_OVERFLOW, value, value, 1L);
        }
        return ConvergentsIterator.convergents(value, maxConvergents).map(STEP_TO_FRACTION);
    }

    public static Pair<Fraction, Boolean> convergent(double value, int maxConvergents, ConvergenceTest convergenceTest) {
        Pair<ConvergentsIterator.ConvergenceStep, Boolean> converged = Fraction.convergent(value, maxConvergents, (ConvergentsIterator.ConvergenceStep s) -> {
            Fraction.assertNoIntegerOverflow(s, value);
            return convergenceTest.test((int)s.getNumerator(), (int)s.getDenominator());
        });
        return Pair.create(STEP_TO_FRACTION.apply(converged.getKey()), converged.getValue());
    }

    private static Pair<ConvergentsIterator.ConvergenceStep, Boolean> convergent(double value, int maxConvergents, Predicate<ConvergentsIterator.ConvergenceStep> convergenceTests) {
        if (FastMath.abs(value) > 2.147483647E9) {
            throw new MathIllegalStateException(LocalizedCoreFormats.FRACTION_CONVERSION_OVERFLOW, value, value, 1L);
        }
        return ConvergentsIterator.convergent(value, maxConvergents, (ConvergentsIterator.ConvergenceStep s) -> {
            Fraction.assertNoIntegerOverflow(s, value);
            return convergenceTests.test((ConvergentsIterator.ConvergenceStep)s);
        });
    }

    private static void assertNoIntegerOverflow(ConvergentsIterator.ConvergenceStep s, double value) {
        if (s.getNumerator() > Integer.MAX_VALUE || s.getDenominator() > Integer.MAX_VALUE) {
            throw new MathIllegalStateException(LocalizedCoreFormats.FRACTION_CONVERSION_OVERFLOW, value, s.getNumerator(), s.getDenominator());
        }
    }

    @Override
    public double getReal() {
        return this.doubleValue();
    }

    public boolean isInteger() {
        return this.denominator == 1;
    }

    public int signum() {
        return Integer.signum(this.numerator);
    }

    public Fraction abs() {
        Fraction ret = this.numerator >= 0 ? this : this.negate();
        return ret;
    }

    @Override
    public int compareTo(Fraction object) {
        long nOd = (long)this.numerator * (long)object.denominator;
        long dOn = (long)this.denominator * (long)object.numerator;
        return Long.compare(nOd, dOn);
    }

    @Override
    public double doubleValue() {
        return (double)this.numerator / (double)this.denominator;
    }

    public boolean equals(Object other) {
        if (this == other) {
            return true;
        }
        if (other instanceof Fraction) {
            Fraction rhs = (Fraction)other;
            return this.numerator == rhs.numerator && this.denominator == rhs.denominator;
        }
        return false;
    }

    @Override
    public float floatValue() {
        return (float)this.doubleValue();
    }

    public int getDenominator() {
        return this.denominator;
    }

    public int getNumerator() {
        return this.numerator;
    }

    public int hashCode() {
        return 37 * (629 + this.numerator) + this.denominator;
    }

    @Override
    public int intValue() {
        return (int)this.doubleValue();
    }

    @Override
    public long longValue() {
        return (long)this.doubleValue();
    }

    @Override
    public Fraction negate() {
        if (this.numerator == Integer.MIN_VALUE) {
            throw new MathRuntimeException(LocalizedCoreFormats.OVERFLOW_IN_FRACTION, this.numerator, this.denominator);
        }
        return new Fraction(-this.numerator, this.denominator);
    }

    @Override
    public Fraction reciprocal() {
        return new Fraction(this.denominator, this.numerator);
    }

    @Override
    public Fraction add(Fraction fraction) {
        return this.addSub(fraction, true);
    }

    @Override
    public Fraction add(int i) {
        return new Fraction(this.numerator + i * this.denominator, this.denominator);
    }

    @Override
    public Fraction subtract(Fraction fraction) {
        return this.addSub(fraction, false);
    }

    @Override
    public Fraction subtract(int i) {
        return new Fraction(this.numerator - i * this.denominator, this.denominator);
    }

    private Fraction addSub(Fraction fraction, boolean isAdd) {
        int tmodd1;
        int d2;
        MathUtils.checkNotNull(fraction, LocalizedCoreFormats.FRACTION, new Object[0]);
        if (this.numerator == 0) {
            return isAdd ? fraction : fraction.negate();
        }
        if (fraction.numerator == 0) {
            return this;
        }
        int d1 = ArithmeticUtils.gcd(this.denominator, fraction.denominator);
        if (d1 == 1) {
            int uvp = ArithmeticUtils.mulAndCheck(this.numerator, fraction.denominator);
            int upv = ArithmeticUtils.mulAndCheck(fraction.numerator, this.denominator);
            return new Fraction(isAdd ? ArithmeticUtils.addAndCheck(uvp, upv) : ArithmeticUtils.subAndCheck(uvp, upv), ArithmeticUtils.mulAndCheck(this.denominator, fraction.denominator));
        }
        BigInteger uvp = BigInteger.valueOf(this.numerator).multiply(BigInteger.valueOf(fraction.denominator / d1));
        BigInteger upv = BigInteger.valueOf(fraction.numerator).multiply(BigInteger.valueOf(this.denominator / d1));
        BigInteger t = isAdd ? uvp.add(upv) : uvp.subtract(upv);
        BigInteger w = t.divide(BigInteger.valueOf(d2 = (tmodd1 = t.mod(BigInteger.valueOf(d1)).intValue()) == 0 ? d1 : ArithmeticUtils.gcd(tmodd1, d1)));
        if (w.bitLength() > 31) {
            throw new MathRuntimeException(LocalizedCoreFormats.NUMERATOR_OVERFLOW_AFTER_MULTIPLY, w);
        }
        return new Fraction(w.intValue(), ArithmeticUtils.mulAndCheck(this.denominator / d1, fraction.denominator / d2));
    }

    @Override
    public Fraction multiply(Fraction fraction) {
        MathUtils.checkNotNull(fraction, LocalizedCoreFormats.FRACTION, new Object[0]);
        if (this.numerator == 0 || fraction.numerator == 0) {
            return ZERO;
        }
        int d1 = ArithmeticUtils.gcd(this.numerator, fraction.denominator);
        int d2 = ArithmeticUtils.gcd(fraction.numerator, this.denominator);
        return Fraction.getReducedFraction(ArithmeticUtils.mulAndCheck(this.numerator / d1, fraction.numerator / d2), ArithmeticUtils.mulAndCheck(this.denominator / d2, fraction.denominator / d1));
    }

    @Override
    public Fraction multiply(int i) {
        return this.multiply(new Fraction(i));
    }

    @Override
    public Fraction divide(Fraction fraction) {
        MathUtils.checkNotNull(fraction, LocalizedCoreFormats.FRACTION, new Object[0]);
        if (fraction.numerator == 0) {
            throw new MathRuntimeException(LocalizedCoreFormats.ZERO_FRACTION_TO_DIVIDE_BY, fraction.numerator, fraction.denominator);
        }
        return this.multiply(fraction.reciprocal());
    }

    @Override
    public Fraction divide(int i) {
        return this.divide(new Fraction(i));
    }

    public double percentageValue() {
        return 100.0 * this.doubleValue();
    }

    public static Fraction getReducedFraction(int numerator, int denominator) {
        if (denominator == 0) {
            throw new MathRuntimeException(LocalizedCoreFormats.ZERO_DENOMINATOR_IN_FRACTION, numerator, denominator);
        }
        if (numerator == 0) {
            return ZERO;
        }
        if (denominator == Integer.MIN_VALUE && (numerator & 1) == 0) {
            numerator /= 2;
            denominator /= 2;
        }
        if (denominator < 0) {
            if (numerator == Integer.MIN_VALUE || denominator == Integer.MIN_VALUE) {
                throw new MathRuntimeException(LocalizedCoreFormats.OVERFLOW_IN_FRACTION, numerator, denominator);
            }
            numerator = -numerator;
            denominator = -denominator;
        }
        int gcd = ArithmeticUtils.gcd(numerator, denominator);
        return new Fraction(numerator /= gcd, denominator /= gcd);
    }

    public String toString() {
        if (this.denominator == 1) {
            return Integer.toString(this.numerator);
        }
        if (this.numerator == 0) {
            return "0";
        }
        return this.numerator + " / " + this.denominator;
    }

    public FractionField getField() {
        return FractionField.getInstance();
    }

    @FunctionalInterface
    public static interface ConvergenceTest {
        public boolean test(int var1, int var2);
    }
}

