/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.models.earth.atmosphere;

import org.hipparchus.CalculusFieldElement;
import org.hipparchus.Field;
import org.hipparchus.FieldElement;
import org.hipparchus.exception.Localizable;
import org.hipparchus.geometry.euclidean.threed.FieldVector3D;
import org.hipparchus.geometry.euclidean.threed.Vector3D;
import org.hipparchus.util.FastMath;
import org.hipparchus.util.FieldSinCos;
import org.hipparchus.util.MathArrays;
import org.hipparchus.util.SinCos;
import org.orekit.annotation.DefaultDataContext;
import org.orekit.bodies.BodyShape;
import org.orekit.bodies.FieldGeodeticPoint;
import org.orekit.bodies.GeodeticPoint;
import org.orekit.data.DataContext;
import org.orekit.errors.OrekitException;
import org.orekit.errors.OrekitMessages;
import org.orekit.frames.Frame;
import org.orekit.models.earth.atmosphere.Atmosphere;
import org.orekit.models.earth.atmosphere.JB2008InputParameters;
import org.orekit.time.AbsoluteDate;
import org.orekit.time.DateTimeComponents;
import org.orekit.time.FieldAbsoluteDate;
import org.orekit.time.TimeScale;
import org.orekit.utils.PVCoordinatesProvider;

public class JB2008
implements Atmosphere {
    private static final long serialVersionUID = -4201270765122160831L;
    private static final double ALT_MIN = 90000.0;
    private static final double EARTH_RADIUS = 6356.766;
    private static final double LOG10 = FastMath.log((double)10.0);
    private static final double AVOGAD = 6.02257E26;
    private static final double RSTAR = 8.31432;
    private static final double[] ALPHA = new double[]{0.0, 0.0, 0.0, 0.0, -0.38};
    private static final double[] AMW = new double[]{28.0134, 31.9988, 15.9994, 39.948, 4.0026, 1.00797};
    private static final double[] FRAC = new double[]{0.7811, 0.20955, 0.00934, 1.289E-5};
    private static final double R1 = 0.01;
    private static final double R2 = 0.025;
    private static final double R3 = 0.075;
    private static final double[] WT = new double[]{0.311111111111111, 1.422222222222222, 0.533333333333333, 1.422222222222222, 0.311111111111111};
    private static final double[] CHT = new double[]{0.22, -0.002, 0.00115, -2.11E-6};
    private static final double[] FZM = new double[]{0.2689, -0.01176, 0.02782, -0.02782, 3.47E-4};
    private static final double[] GTM = new double[]{-0.3633, 0.08506, 0.2401, -0.1897, -0.2554, -0.0179, 5.65E-4, -6.407E-4, -0.003418, -0.001252};
    private static final double[] CMB = new double[]{28.15204, -0.085586, 1.284E-4, -1.0056E-5, -1.021E-5, 1.5044E-6, 9.9826E-8};
    private static final double[] BDTC = new double[]{-4.57512297, -5.12114909, -69.3003609, 203.716701, 703.316291, -1943.49234, 1106.51308, -174.378996, 1885.94601, -7093.71517, 9224.54523, -3845.08073, -6.45841789, 40.9703319, -482.00656, 1818.70931, -2373.89204, 996.703815, 36.1416936};
    private static final double[] CDTC = new double[]{-15.5986211, -5.12114909, -69.3003609, 203.716701, 703.316291, -1943.49234, 1106.51308, -220.835117, 1432.56989, -3184.81844, 3289.81513, -1353.32119, 19.9956489, -12.7093998, 21.2825156, -2.75555432, 11.0234982, 148.881951, -751.640284, 637.876542, 12.7093998, -21.2825156, 2.75555432};
    private PVCoordinatesProvider sun;
    private JB2008InputParameters inputParams;
    private BodyShape earth;
    private final TimeScale utc;

    @DefaultDataContext
    public JB2008(JB2008InputParameters parameters, PVCoordinatesProvider sun, BodyShape earth) {
        this(parameters, sun, earth, DataContext.getDefault().getTimeScales().getUTC());
    }

    public JB2008(JB2008InputParameters parameters, PVCoordinatesProvider sun, BodyShape earth, TimeScale utc) {
        this.earth = earth;
        this.sun = sun;
        this.inputParams = parameters;
        this.utc = utc;
    }

    @Override
    public Frame getFrame() {
        return this.earth.getBodyFrame();
    }

    public double getDensity(double dateMJD, double sunRA, double sunDecli, double satLon, double satLat, double satAlt, double f10, double f10B, double s10, double s10B, double xm10, double xm10B, double y10, double y10B, double dstdtc) {
        double[] tc;
        double solarTime;
        if (satAlt < 90000.0) {
            throw new OrekitException((Localizable)OrekitMessages.ALTITUDE_BELOW_ALLOWED_THRESHOLD, satAlt, 90000.0);
        }
        double altKm = satAlt / 1000.0;
        double fn = FastMath.min((double)1.0, (double)FastMath.pow((double)(f10B / 240.0), (double)0.25));
        double fsb = f10B * fn + s10B * (1.0 - fn);
        double tsubc = 392.4 + 3.227 * fsb + 0.298 * (f10 - f10B) + 2.259 * (s10 - s10B) + 0.312 * (xm10 - xm10B) + 0.178 * (y10 - y10B);
        double eta = 0.5 * FastMath.abs((double)(satLat - sunDecli));
        double theta = 0.5 * FastMath.abs((double)(satLat + sunDecli));
        double h = satLon - sunRA;
        double tau = h - 0.64577182 + 0.10471976 * FastMath.sin((double)(h + 0.75049158));
        for (solarTime = FastMath.toDegrees((double)(h + Math.PI)) / 15.0; solarTime >= 24.0; solarTime -= 24.0) {
        }
        while (solarTime < 0.0) {
            solarTime += 24.0;
        }
        double cosEta = FastMath.pow((double)FastMath.cos((double)eta), (double)2.5);
        double sinTeta = FastMath.pow((double)FastMath.sin((double)theta), (double)2.5);
        double cosTau = FastMath.abs((double)FastMath.cos((double)(0.5 * tau)));
        double df = sinTeta + (cosEta - sinTeta) * cosTau * cosTau * cosTau;
        double tsubl = tsubc * (1.0 + 0.31 * df);
        double dtclst = JB2008.dTc(f10, solarTime, satLat, altKm);
        double[] temp = new double[2];
        temp[0] = tsubl + dstdtc;
        double tinf = temp[0] + dtclst;
        double tsubx = 444.3807 + 0.02385 * tinf - 392.8292 * FastMath.exp((double)(-0.0021357 * tinf));
        double gsubx = 0.054285714 * (tsubx - 183.0);
        tc = new double[]{tsubx, gsubx, (tinf - tsubx) * 2.0 / Math.PI, gsubx / tc[2]};
        double z1 = 90.0;
        double z2 = FastMath.min((double)altKm, (double)105.0);
        double al = FastMath.log((double)(z2 / 90.0));
        int n = 1 + (int)FastMath.floor((double)(al / 0.01));
        double zr = FastMath.exp((double)(al / (double)n));
        double mb1 = JB2008.mBar(90.0);
        double tloc1 = JB2008.localTemp(90.0, tc);
        double zend = 90.0;
        double sub2 = 0.0;
        double ain = mb1 * JB2008.gravity(90.0) / tloc1;
        double mb2 = 0.0;
        double tloc2 = 0.0;
        double z = 0.0;
        double gravl = 0.0;
        for (int i = 0; i < n; ++i) {
            z = zend;
            zend = zr * z;
            double dz = 0.25 * (zend - z);
            double sum1 = WT[0] * ain;
            for (int j = 1; j < 5; ++j) {
                mb2 = JB2008.mBar(z += dz);
                tloc2 = JB2008.localTemp(z, tc);
                gravl = JB2008.gravity(z);
                ain = mb2 * gravl / tloc2;
                sum1 += WT[j] * ain;
            }
            sub2 += dz * sum1;
        }
        double rho = 3.46E-6 * mb2 * tloc1 / FastMath.exp((double)(sub2 / 8.31432)) / (mb1 * tloc2);
        double anm = 6.02257E26 * rho;
        double an = anm / mb2;
        double fact2 = anm / 28.96;
        double[] aln = new double[6];
        aln[0] = FastMath.log((double)(FRAC[0] * fact2));
        aln[3] = FastMath.log((double)(FRAC[2] * fact2));
        aln[4] = FastMath.log((double)(FRAC[3] * fact2));
        aln[1] = FastMath.log((double)(fact2 * (1.0 + FRAC[1]) - an));
        aln[2] = FastMath.log((double)(2.0 * (an - fact2)));
        if (altKm <= 105.0) {
            temp[1] = tloc2;
            aln[5] = aln[4] - 25.0;
        } else {
            double hSign;
            double altr;
            al = FastMath.log((double)(FastMath.min((double)altKm, (double)500.0) / z));
            n = 1 + (int)FastMath.floor((double)(al / 0.025));
            zr = FastMath.exp((double)(al / (double)n));
            sub2 = 0.0;
            ain = gravl / tloc2;
            double tloc3 = 0.0;
            for (int i = 0; i < n; ++i) {
                z = zend;
                zend = zr * z;
                double dz = 0.25 * (zend - z);
                double sum1 = WT[0] * ain;
                for (int j = 1; j < 5; ++j) {
                    tloc3 = JB2008.localTemp(z += dz, tc);
                    gravl = JB2008.gravity(z);
                    ain = gravl / tloc3;
                    sum1 += WT[j] * ain;
                }
                sub2 += dz * sum1;
            }
            al = FastMath.log((double)(FastMath.max((double)altKm, (double)500.0) / z));
            double r = altKm > 500.0 ? 0.075 : 0.025;
            n = 1 + (int)FastMath.floor((double)(al / r));
            zr = FastMath.exp((double)(al / (double)n));
            double sum3 = 0.0;
            double tloc4 = 0.0;
            for (int i = 0; i < n; ++i) {
                z = zend;
                zend = zr * z;
                double dz = 0.25 * (zend - z);
                double sum1 = WT[0] * ain;
                for (int j = 1; j < 5; ++j) {
                    tloc4 = JB2008.localTemp(z += dz, tc);
                    gravl = JB2008.gravity(z);
                    ain = gravl / tloc4;
                    sum1 += WT[j] * ain;
                }
                sum3 += dz * sum1;
            }
            if (altKm <= 500.0) {
                temp[1] = tloc3;
                altr = FastMath.log((double)(tloc3 / tloc2));
                fact2 = sub2 / 8.31432;
                hSign = 1.0;
            } else {
                temp[1] = tloc4;
                altr = FastMath.log((double)(tloc4 / tloc2));
                fact2 = (sub2 + sum3) / 8.31432;
                hSign = -1.0;
            }
            for (int i = 0; i < 5; ++i) {
                aln[i] = aln[i] - (1.0 + ALPHA[i]) * altr - fact2 * AMW[i];
            }
            double al10t5 = FastMath.log10((double)tinf);
            double alnh5 = (5.5 * al10t5 - 39.4) * al10t5 + 73.13;
            aln[5] = LOG10 * (alnh5 + 6.0) + hSign * (FastMath.log((double)(tloc4 / tloc3)) + sum3 * AMW[5] / 8.31432);
        }
        double capPhi = (dateMJD - 36204.0) / 365.2422 % 1.0;
        int signum = satLat >= 0.0 ? 1 : -1;
        double sinLat = FastMath.sin((double)satLat);
        double hm90 = altKm - 90.0;
        double dlrsl = 0.02 * hm90 * FastMath.exp((double)(-0.045 * hm90)) * (double)signum * FastMath.sin((double)(Math.PI * 2 * capPhi + 1.72)) * sinLat * sinLat;
        double dlrsa = 0.0;
        if (z < 2000.0) {
            dlrsa = JB2008.semian08(JB2008.dayOfYear(dateMJD), altKm, f10B, s10B, xm10B);
        }
        double dlr = LOG10 * (dlrsl + dlrsa);
        int i = 0;
        while (i < 6) {
            int n2 = i++;
            aln[n2] = aln[n2] + dlr;
        }
        double sumnm = 0.0;
        for (int i2 = 0; i2 < 6; ++i2) {
            sumnm += FastMath.exp((double)aln[i2]) * AMW[i2];
        }
        rho = sumnm / 6.02257E26;
        double fex = 1.0;
        if (altKm >= 1000.0 && altKm < 1500.0) {
            double zeta = (altKm - 1000.0) * 0.002;
            double f15c = CHT[0] + CHT[1] * f10B + (CHT[2] + CHT[3] * f10B) * 1500.0;
            double f15cZeta = (CHT[2] + CHT[3] * f10B) * 500.0;
            double fex2 = 3.0 * f15c - f15cZeta - 3.0;
            double fex3 = f15cZeta - 2.0 * f15c + 2.0;
            fex += zeta * zeta * (fex2 + fex3 * zeta);
        } else if (altKm >= 1500.0) {
            fex = CHT[0] + CHT[1] * f10B + CHT[2] * altKm + CHT[3] * f10B * altKm;
        }
        return rho *= fex;
    }

    public <T extends CalculusFieldElement<T>> T getDensity(T dateMJD, T sunRA, T sunDecli, T satLon, T satLat, T satAlt, double f10, double f10B, double s10, double s10B, double xm10, double xm10B, double y10, double y10B, double dstdtc) {
        if (satAlt.getReal() < 90000.0) {
            throw new OrekitException((Localizable)OrekitMessages.ALTITUDE_BELOW_ALLOWED_THRESHOLD, satAlt.getReal(), 90000.0);
        }
        Field field = satAlt.getField();
        CalculusFieldElement pi = (CalculusFieldElement)((CalculusFieldElement)field.getOne()).getPi();
        CalculusFieldElement altKm = (CalculusFieldElement)satAlt.divide(1000.0);
        double fn = FastMath.min((double)1.0, (double)FastMath.pow((double)(f10B / 240.0), (double)0.25));
        double fsb = f10B * fn + s10B * (1.0 - fn);
        double tsubc = 392.4 + 3.227 * fsb + 0.298 * (f10 - f10B) + 2.259 * (s10 - s10B) + 0.312 * (xm10 - xm10B) + 0.178 * (y10 - y10B);
        CalculusFieldElement eta = (CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)satLat.subtract(sunDecli)).abs()).multiply(0.5);
        CalculusFieldElement theta = (CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)satLat.add(sunDecli)).abs()).multiply(0.5);
        CalculusFieldElement h = (CalculusFieldElement)satLon.subtract(sunRA);
        CalculusFieldElement tau = (CalculusFieldElement)((CalculusFieldElement)h.subtract(0.64577182)).add((FieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)h.add(0.75049158)).sin()).multiply(0.10471976)));
        CalculusFieldElement solarTime = (CalculusFieldElement)FastMath.toDegrees((CalculusFieldElement)((CalculusFieldElement)h.add((FieldElement)pi))).divide(15.0);
        while (solarTime.getReal() >= 24.0) {
            solarTime = (CalculusFieldElement)solarTime.subtract(24.0);
        }
        while (solarTime.getReal() < 0.0) {
            solarTime = (CalculusFieldElement)solarTime.add(24.0);
        }
        CalculusFieldElement cos = (CalculusFieldElement)eta.cos();
        CalculusFieldElement cosEta = (CalculusFieldElement)((CalculusFieldElement)cos.multiply((FieldElement)cos)).multiply((FieldElement)((CalculusFieldElement)cos.sqrt()));
        CalculusFieldElement sin = (CalculusFieldElement)theta.sin();
        CalculusFieldElement sinTeta = (CalculusFieldElement)((CalculusFieldElement)sin.multiply((FieldElement)sin)).multiply((FieldElement)((CalculusFieldElement)sin.sqrt()));
        CalculusFieldElement cosTau = (CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)tau.multiply(0.5)).cos()).abs();
        CalculusFieldElement df = (CalculusFieldElement)sinTeta.add((FieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)cosEta.subtract((FieldElement)sinTeta)).multiply((FieldElement)cosTau)).multiply((FieldElement)cosTau)).multiply((FieldElement)cosTau)));
        CalculusFieldElement tsubl = (CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)df.multiply(0.31)).add(1.0)).multiply(tsubc);
        CalculusFieldElement dtclst = JB2008.dTc(f10, solarTime, satLat, altKm);
        CalculusFieldElement[] temp = (CalculusFieldElement[])MathArrays.buildArray((Field)field, (int)2);
        temp[0] = (CalculusFieldElement)tsubl.add(dstdtc);
        CalculusFieldElement tinf = (CalculusFieldElement)temp[0].add((FieldElement)dtclst);
        CalculusFieldElement tsubx = (CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)tinf.multiply(0.02385)).add(444.3807)).subtract((FieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)tinf.multiply(-0.0021357)).exp()).multiply(392.8292)));
        CalculusFieldElement gsubx = (CalculusFieldElement)((CalculusFieldElement)tsubx.subtract(183.0)).multiply(0.054285714);
        CalculusFieldElement[] tc = (CalculusFieldElement[])MathArrays.buildArray((Field)field, (int)4);
        tc[0] = tsubx;
        tc[1] = gsubx;
        tc[2] = (CalculusFieldElement)((CalculusFieldElement)tinf.subtract((FieldElement)tsubx)).multiply((FieldElement)((CalculusFieldElement)((CalculusFieldElement)pi.reciprocal()).multiply(2.0)));
        tc[3] = (CalculusFieldElement)gsubx.divide((FieldElement)tc[2]);
        CalculusFieldElement z1 = (CalculusFieldElement)((CalculusFieldElement)field.getZero()).add(90.0);
        CalculusFieldElement z2 = this.min(105.0, altKm);
        CalculusFieldElement al = (CalculusFieldElement)((CalculusFieldElement)z2.divide((FieldElement)z1)).log();
        int n = 1 + (int)FastMath.floor((double)(al.getReal() / 0.01));
        CalculusFieldElement zr = (CalculusFieldElement)((CalculusFieldElement)al.divide((double)n)).exp();
        CalculusFieldElement mb1 = JB2008.mBar(z1);
        CalculusFieldElement tloc1 = JB2008.localTemp((CalculusFieldElement)z1, (CalculusFieldElement[])tc);
        CalculusFieldElement zend = z1;
        CalculusFieldElement sub2 = (CalculusFieldElement)field.getZero();
        CalculusFieldElement ain = (CalculusFieldElement)((CalculusFieldElement)mb1.multiply((FieldElement)JB2008.gravity(z1))).divide((FieldElement)tloc1);
        CalculusFieldElement mb2 = (CalculusFieldElement)field.getZero();
        CalculusFieldElement tloc2 = (CalculusFieldElement)field.getZero();
        CalculusFieldElement z = (CalculusFieldElement)field.getZero();
        CalculusFieldElement gravl = (CalculusFieldElement)field.getZero();
        for (int i = 0; i < n; ++i) {
            z = zend;
            zend = (CalculusFieldElement)zr.multiply((FieldElement)z);
            CalculusFieldElement dz = (CalculusFieldElement)((CalculusFieldElement)zend.subtract((FieldElement)z)).multiply(0.25);
            CalculusFieldElement sum1 = (CalculusFieldElement)ain.multiply(WT[0]);
            for (int j = 1; j < 5; ++j) {
                z = (CalculusFieldElement)z.add((FieldElement)dz);
                mb2 = JB2008.mBar(z);
                tloc2 = JB2008.localTemp((CalculusFieldElement)z, (CalculusFieldElement[])tc);
                gravl = JB2008.gravity(z);
                ain = (CalculusFieldElement)((CalculusFieldElement)mb2.multiply((FieldElement)gravl)).divide((FieldElement)tloc2);
                sum1 = (CalculusFieldElement)sum1.add((FieldElement)((CalculusFieldElement)ain.multiply(WT[j])));
            }
            sub2 = (CalculusFieldElement)sub2.add((FieldElement)((CalculusFieldElement)dz.multiply((FieldElement)sum1)));
        }
        CalculusFieldElement rho = (CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)mb2.multiply(3.46E-6)).multiply((FieldElement)tloc1)).divide((FieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)sub2.divide(8.31432)).exp()).multiply((FieldElement)((CalculusFieldElement)mb1.multiply((FieldElement)tloc2)))));
        CalculusFieldElement anm = (CalculusFieldElement)rho.multiply(6.02257E26);
        CalculusFieldElement an = (CalculusFieldElement)anm.divide((FieldElement)mb2);
        CalculusFieldElement fact2 = (CalculusFieldElement)anm.divide(28.96);
        CalculusFieldElement[] aln = (CalculusFieldElement[])MathArrays.buildArray((Field)field, (int)6);
        aln[0] = (CalculusFieldElement)((CalculusFieldElement)fact2.multiply(FRAC[0])).log();
        aln[3] = (CalculusFieldElement)((CalculusFieldElement)fact2.multiply(FRAC[2])).log();
        aln[4] = (CalculusFieldElement)((CalculusFieldElement)fact2.multiply(FRAC[3])).log();
        aln[1] = (CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)fact2.multiply(1.0 + FRAC[1])).subtract((FieldElement)an)).log();
        aln[2] = (CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)an.subtract((FieldElement)fact2)).multiply(2)).log();
        if (altKm.getReal() <= 105.0) {
            temp[1] = tloc2;
            aln[5] = (CalculusFieldElement)aln[4].subtract(25.0);
        } else {
            double hSign;
            CalculusFieldElement altr;
            al = (CalculusFieldElement)((CalculusFieldElement)this.min(500.0, altKm).divide((FieldElement)z)).log();
            n = 1 + (int)FastMath.floor((double)(al.getReal() / 0.025));
            zr = (CalculusFieldElement)((CalculusFieldElement)al.divide((double)n)).exp();
            sub2 = (CalculusFieldElement)field.getZero();
            ain = (CalculusFieldElement)gravl.divide((FieldElement)tloc2);
            CalculusFieldElement tloc3 = (CalculusFieldElement)field.getZero();
            for (int i = 0; i < n; ++i) {
                z = zend;
                zend = (CalculusFieldElement)zr.multiply((FieldElement)z);
                CalculusFieldElement dz = (CalculusFieldElement)((CalculusFieldElement)zend.subtract((FieldElement)z)).multiply(0.25);
                CalculusFieldElement sum1 = (CalculusFieldElement)ain.multiply(WT[0]);
                for (int j = 1; j < 5; ++j) {
                    z = (CalculusFieldElement)z.add((FieldElement)dz);
                    tloc3 = JB2008.localTemp((CalculusFieldElement)z, (CalculusFieldElement[])tc);
                    gravl = JB2008.gravity(z);
                    ain = (CalculusFieldElement)gravl.divide((FieldElement)tloc3);
                    sum1 = (CalculusFieldElement)sum1.add((FieldElement)((CalculusFieldElement)ain.multiply(WT[j])));
                }
                sub2 = (CalculusFieldElement)sub2.add((FieldElement)((CalculusFieldElement)dz.multiply((FieldElement)sum1)));
            }
            al = (CalculusFieldElement)((CalculusFieldElement)this.max(500.0, altKm).divide((FieldElement)z)).log();
            double r = altKm.getReal() > 500.0 ? 0.075 : 0.025;
            n = 1 + (int)FastMath.floor((double)(al.getReal() / r));
            zr = (CalculusFieldElement)((CalculusFieldElement)al.divide((double)n)).exp();
            CalculusFieldElement sum3 = (CalculusFieldElement)field.getZero();
            CalculusFieldElement tloc4 = (CalculusFieldElement)field.getZero();
            for (int i = 0; i < n; ++i) {
                z = zend;
                zend = (CalculusFieldElement)zr.multiply((FieldElement)z);
                CalculusFieldElement dz = (CalculusFieldElement)((CalculusFieldElement)zend.subtract((FieldElement)z)).multiply(0.25);
                CalculusFieldElement sum1 = (CalculusFieldElement)ain.multiply(WT[0]);
                for (int j = 1; j < 5; ++j) {
                    z = (CalculusFieldElement)z.add((FieldElement)dz);
                    tloc4 = JB2008.localTemp((CalculusFieldElement)z, (CalculusFieldElement[])tc);
                    gravl = JB2008.gravity(z);
                    ain = (CalculusFieldElement)gravl.divide((FieldElement)tloc4);
                    sum1 = (CalculusFieldElement)sum1.add((FieldElement)((CalculusFieldElement)ain.multiply(WT[j])));
                }
                sum3 = (CalculusFieldElement)sum3.add((FieldElement)((CalculusFieldElement)dz.multiply((FieldElement)sum1)));
            }
            if (altKm.getReal() <= 500.0) {
                temp[1] = tloc3;
                altr = (CalculusFieldElement)((CalculusFieldElement)tloc3.divide((FieldElement)tloc2)).log();
                fact2 = (CalculusFieldElement)sub2.divide(8.31432);
                hSign = 1.0;
            } else {
                temp[1] = tloc4;
                altr = (CalculusFieldElement)((CalculusFieldElement)tloc4.divide((FieldElement)tloc2)).log();
                fact2 = (CalculusFieldElement)((CalculusFieldElement)sub2.add((FieldElement)sum3)).divide(8.31432);
                hSign = -1.0;
            }
            for (int i = 0; i < 5; ++i) {
                aln[i] = (CalculusFieldElement)((CalculusFieldElement)aln[i].subtract((FieldElement)((CalculusFieldElement)altr.multiply(1.0 + ALPHA[i])))).subtract((FieldElement)((CalculusFieldElement)fact2.multiply(AMW[i])));
            }
            CalculusFieldElement al10t5 = (CalculusFieldElement)tinf.log10();
            CalculusFieldElement alnh5 = (CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)al10t5.multiply(5.5)).subtract(39.4)).multiply((FieldElement)al10t5)).add(73.13);
            aln[5] = (CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)alnh5.add(6.0)).multiply(LOG10)).add((FieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)tloc4.divide((FieldElement)tloc3)).log()).add((FieldElement)((CalculusFieldElement)sum3.multiply(AMW[5] / 8.31432)))).multiply(hSign)));
        }
        CalculusFieldElement capPhi = (CalculusFieldElement)((CalculusFieldElement)dateMJD.subtract(36204.0)).divide(365.2422);
        capPhi = (CalculusFieldElement)capPhi.subtract(FastMath.floor((double)capPhi.getReal()));
        int signum = satLat.getReal() >= 0.0 ? 1 : -1;
        CalculusFieldElement sinLat = (CalculusFieldElement)satLat.sin();
        CalculusFieldElement hm90 = (CalculusFieldElement)altKm.subtract(90.0);
        CalculusFieldElement dlrsl = (CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)hm90.multiply(0.02)).multiply((FieldElement)((CalculusFieldElement)((CalculusFieldElement)hm90.multiply(-0.045)).exp()))).multiply((FieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)capPhi.multiply(Math.PI * 2)).add(1.72)).sin()))).multiply(signum)).multiply((FieldElement)sinLat)).multiply((FieldElement)sinLat);
        CalculusFieldElement dlrsa = (CalculusFieldElement)field.getZero();
        if (z.getReal() < 2000.0) {
            dlrsa = JB2008.semian08(JB2008.dayOfYear(dateMJD), altKm, f10B, s10B, xm10B);
        }
        CalculusFieldElement dlr = (CalculusFieldElement)((CalculusFieldElement)dlrsl.add((FieldElement)dlrsa)).multiply(LOG10);
        for (int i = 0; i < 6; ++i) {
            aln[i] = (CalculusFieldElement)aln[i].add((FieldElement)dlr);
        }
        CalculusFieldElement sumnm = (CalculusFieldElement)field.getZero();
        for (int i = 0; i < 6; ++i) {
            sumnm = (CalculusFieldElement)sumnm.add((FieldElement)((CalculusFieldElement)((CalculusFieldElement)aln[i].exp()).multiply(AMW[i])));
        }
        rho = (CalculusFieldElement)sumnm.divide(6.02257E26);
        CalculusFieldElement fex = (CalculusFieldElement)field.getOne();
        if (altKm.getReal() >= 1000.0 && altKm.getReal() < 1500.0) {
            CalculusFieldElement zeta = (CalculusFieldElement)((CalculusFieldElement)altKm.subtract(1000.0)).multiply(0.002);
            double f15c = CHT[0] + CHT[1] * f10B + (CHT[2] + CHT[3] * f10B) * 1500.0;
            double f15cZeta = (CHT[2] + CHT[3] * f10B) * 500.0;
            double fex2 = 3.0 * f15c - f15cZeta - 3.0;
            double fex3 = f15cZeta - 2.0 * f15c + 2.0;
            fex = (CalculusFieldElement)fex.add((FieldElement)((CalculusFieldElement)((CalculusFieldElement)zeta.multiply((FieldElement)zeta)).multiply((FieldElement)((CalculusFieldElement)((CalculusFieldElement)zeta.multiply(fex3)).add(fex2)))));
        } else if (altKm.getReal() >= 1500.0) {
            fex = (CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)altKm.multiply(CHT[3] * f10B)).add((FieldElement)((CalculusFieldElement)altKm.multiply(CHT[2])))).add(CHT[0] + CHT[1] * f10B);
        }
        rho = (CalculusFieldElement)rho.multiply((FieldElement)fex);
        return (T)rho;
    }

    private static double dTc(double f10, double solarTime, double satLat, double satAlt) {
        double dTc = 0.0;
        double st = solarTime / 24.0;
        double cs = FastMath.cos((double)satLat);
        double fs = (f10 - 100.0) / 100.0;
        if (satAlt >= 120.0 && satAlt <= 200.0) {
            double dtc200 = JB2008.poly2CDTC(fs, st, cs);
            double dtc200dz = JB2008.poly1CDTC(fs, st, cs);
            double cc = 3.0 * dtc200 - dtc200dz;
            double dd = dtc200 - cc;
            double zp = (satAlt - 120.0) / 80.0;
            dTc = zp * zp * (cc + dd * zp);
        } else if (satAlt > 200.0 && satAlt <= 240.0) {
            double h = (satAlt - 200.0) / 50.0;
            dTc = JB2008.poly1CDTC(fs, st, cs) * h + JB2008.poly2CDTC(fs, st, cs);
        } else if (satAlt > 240.0 && satAlt <= 300.0) {
            double h = 0.8;
            double bb = JB2008.poly1CDTC(fs, st, cs);
            double aa = bb * 0.8 + JB2008.poly2CDTC(fs, st, cs);
            double p2BDT = JB2008.poly2BDTC(st);
            double dtc300 = JB2008.poly1BDTC(fs, st, cs, 3.0 * p2BDT);
            double dtc300dz = cs * p2BDT;
            double cc = 3.0 * dtc300 - dtc300dz - 3.0 * aa - 2.0 * bb;
            double dd = dtc300 - aa - bb - cc;
            double zp = (satAlt - 240.0) / 60.0;
            dTc = aa + zp * (bb + zp * (cc + zp * dd));
        } else if (satAlt > 300.0 && satAlt <= 600.0) {
            double h = satAlt / 100.0;
            dTc = JB2008.poly1BDTC(fs, st, cs, h * JB2008.poly2BDTC(st));
        } else if (satAlt > 600.0 && satAlt <= 800.0) {
            double poly2 = JB2008.poly2BDTC(st);
            double aa = JB2008.poly1BDTC(fs, st, cs, 6.0 * poly2);
            double bb = cs * poly2;
            double cc = -(3.0 * aa + 4.0 * bb) / 4.0;
            double dd = (aa + bb) / 4.0;
            double zp = (satAlt - 600.0) / 100.0;
            dTc = aa + zp * (bb + zp * (cc + zp * dd));
        }
        return dTc;
    }

    private static <T extends CalculusFieldElement<T>> T dTc(double f10, T solarTime, T satLat, T satAlt) {
        CalculusFieldElement dTc = (CalculusFieldElement)solarTime.getField().getZero();
        CalculusFieldElement st = (CalculusFieldElement)solarTime.divide(24.0);
        CalculusFieldElement cs = (CalculusFieldElement)satLat.cos();
        double fs = (f10 - 100.0) / 100.0;
        if (satAlt.getReal() >= 120.0 && satAlt.getReal() <= 200.0) {
            CalculusFieldElement dtc200 = JB2008.poly2CDTC(fs, st, cs);
            CalculusFieldElement dtc200dz = JB2008.poly1CDTC(fs, st, cs);
            CalculusFieldElement cc = (CalculusFieldElement)((CalculusFieldElement)dtc200.multiply(3)).subtract((FieldElement)dtc200dz);
            CalculusFieldElement dd = (CalculusFieldElement)dtc200.subtract((FieldElement)cc);
            CalculusFieldElement zp = (CalculusFieldElement)((CalculusFieldElement)satAlt.subtract(120.0)).divide(80.0);
            dTc = (CalculusFieldElement)((CalculusFieldElement)zp.multiply((FieldElement)zp)).multiply((FieldElement)((CalculusFieldElement)cc.add((FieldElement)((CalculusFieldElement)dd.multiply((FieldElement)zp)))));
        } else if (satAlt.getReal() > 200.0 && satAlt.getReal() <= 240.0) {
            CalculusFieldElement h = (CalculusFieldElement)((CalculusFieldElement)satAlt.subtract(200.0)).divide(50.0);
            dTc = (CalculusFieldElement)((CalculusFieldElement)JB2008.poly1CDTC(fs, st, cs).multiply((FieldElement)h)).add((FieldElement)JB2008.poly2CDTC(fs, st, cs));
        } else if (satAlt.getReal() > 240.0 && satAlt.getReal() <= 300.0) {
            CalculusFieldElement h = (CalculusFieldElement)((CalculusFieldElement)solarTime.getField().getZero()).add(0.8);
            CalculusFieldElement bb = JB2008.poly1CDTC(fs, st, cs);
            CalculusFieldElement aa = (CalculusFieldElement)((CalculusFieldElement)bb.multiply((FieldElement)h)).add((FieldElement)JB2008.poly2CDTC(fs, st, cs));
            CalculusFieldElement p2BDT = JB2008.poly2BDTC(st);
            CalculusFieldElement dtc300 = JB2008.poly1BDTC(fs, st, cs, (CalculusFieldElement)p2BDT.multiply(3));
            CalculusFieldElement dtc300dz = (CalculusFieldElement)cs.multiply((FieldElement)p2BDT);
            CalculusFieldElement cc = (CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)dtc300.multiply(3)).subtract((FieldElement)dtc300dz)).subtract((FieldElement)((CalculusFieldElement)aa.multiply(3)))).subtract((FieldElement)((CalculusFieldElement)bb.multiply(2)));
            CalculusFieldElement dd = (CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)dtc300.subtract((FieldElement)aa)).subtract((FieldElement)bb)).subtract((FieldElement)cc);
            CalculusFieldElement zp = (CalculusFieldElement)((CalculusFieldElement)satAlt.subtract(240.0)).divide(60.0);
            dTc = (CalculusFieldElement)aa.add((FieldElement)((CalculusFieldElement)zp.multiply((FieldElement)((CalculusFieldElement)bb.add((FieldElement)((CalculusFieldElement)zp.multiply((FieldElement)((CalculusFieldElement)cc.add((FieldElement)((CalculusFieldElement)zp.multiply((FieldElement)dd)))))))))));
        } else if (satAlt.getReal() > 300.0 && satAlt.getReal() <= 600.0) {
            CalculusFieldElement h = (CalculusFieldElement)satAlt.divide(100.0);
            dTc = JB2008.poly1BDTC(fs, st, cs, (CalculusFieldElement)h.multiply((FieldElement)JB2008.poly2BDTC(st)));
        } else if (satAlt.getReal() > 600.0 && satAlt.getReal() <= 800.0) {
            CalculusFieldElement poly2 = JB2008.poly2BDTC(st);
            CalculusFieldElement aa = JB2008.poly1BDTC(fs, st, cs, (CalculusFieldElement)poly2.multiply(6));
            CalculusFieldElement bb = (CalculusFieldElement)cs.multiply((FieldElement)poly2);
            CalculusFieldElement cc = (CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)aa.multiply(3)).add((FieldElement)((CalculusFieldElement)bb.multiply(4)))).divide(-4.0);
            CalculusFieldElement dd = (CalculusFieldElement)((CalculusFieldElement)aa.add((FieldElement)bb)).divide(4.0);
            CalculusFieldElement zp = (CalculusFieldElement)((CalculusFieldElement)satAlt.subtract(600.0)).divide(100.0);
            dTc = (CalculusFieldElement)aa.add((FieldElement)((CalculusFieldElement)zp.multiply((FieldElement)((CalculusFieldElement)bb.add((FieldElement)((CalculusFieldElement)zp.multiply((FieldElement)((CalculusFieldElement)cc.add((FieldElement)((CalculusFieldElement)zp.multiply((FieldElement)dd)))))))))));
        }
        return (T)dTc;
    }

    private static double poly1CDTC(double fs, double st, double cs) {
        return CDTC[0] + fs * (CDTC[1] + st * (CDTC[2] + st * (CDTC[3] + st * (CDTC[4] + st * (CDTC[5] + st * CDTC[6]))))) + cs * st * (CDTC[7] + st * (CDTC[8] + st * (CDTC[9] + st * (CDTC[10] + st * CDTC[11])))) + cs * (CDTC[12] + fs * (CDTC[13] + st * (CDTC[14] + st * CDTC[15])));
    }

    private static <T extends CalculusFieldElement<T>> T poly1CDTC(double fs, T st, T cs) {
        return (T)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)st.multiply(CDTC[6])).add(CDTC[5])).multiply(st)).add(CDTC[4])).multiply(st)).add(CDTC[3])).multiply(st)).add(CDTC[2])).multiply(st)).add(CDTC[1])).multiply(fs)).add((FieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)st.multiply(CDTC[11])).add(CDTC[10])).multiply(st)).add(CDTC[9])).multiply(st)).add(CDTC[8])).multiply(st)).add(CDTC[7])).multiply(st)).multiply(cs)))).add((FieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)st.multiply(CDTC[15])).add(CDTC[14])).multiply(st)).add(CDTC[13])).multiply(fs)).add(CDTC[12])).multiply(cs)))).add(CDTC[0]));
    }

    private static double poly2CDTC(double fs, double st, double cs) {
        return CDTC[16] + st * cs * (CDTC[17] + st * (CDTC[18] + st * CDTC[19])) + fs * cs * (CDTC[20] + st * (CDTC[21] + st * CDTC[22]));
    }

    private static <T extends CalculusFieldElement<T>> T poly2CDTC(double fs, T st, T cs) {
        return (T)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)st.multiply(CDTC[19])).add(CDTC[18])).multiply(st)).add(CDTC[17])).multiply(cs)).multiply(st)).add((FieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)st.multiply(CDTC[22])).add(CDTC[21])).multiply(st)).add(CDTC[20])).multiply(cs)).multiply(fs)))).add(CDTC[16]));
    }

    private static double poly1BDTC(double fs, double st, double cs, double hp) {
        return BDTC[0] + fs * (BDTC[1] + st * (BDTC[2] + st * (BDTC[3] + st * (BDTC[4] + st * (BDTC[5] + st * BDTC[6]))))) + cs * (st * (BDTC[7] + st * (BDTC[8] + st * (BDTC[9] + st * (BDTC[10] + st * BDTC[11])))) + hp + BDTC[18]);
    }

    private static <T extends CalculusFieldElement<T>> T poly1BDTC(double fs, T st, T cs, T hp) {
        return (T)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)st.multiply(BDTC[6])).add(BDTC[5])).multiply(st)).add(BDTC[4])).multiply(st)).add(BDTC[3])).multiply(st)).add(BDTC[2])).multiply(st)).add(BDTC[1])).multiply(fs)).add((FieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)st.multiply(BDTC[11])).add(BDTC[10])).multiply(st)).add(BDTC[9])).multiply(st)).add(BDTC[8])).multiply(st)).add(BDTC[7])).multiply(st)).add(hp)).add(BDTC[18])).multiply(cs)))).add(BDTC[0]));
    }

    private static double poly2BDTC(double st) {
        return BDTC[12] + st * (BDTC[13] + st * (BDTC[14] + st * (BDTC[15] + st * (BDTC[16] + st * BDTC[17]))));
    }

    private static <T extends CalculusFieldElement<T>> T poly2BDTC(T st) {
        return (T)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)st.multiply(BDTC[17])).add(BDTC[16])).multiply(st)).add(BDTC[15])).multiply(st)).add(BDTC[14])).multiply(st)).add(BDTC[13])).multiply(st)).add(BDTC[12]));
    }

    private static double mBar(double z) {
        double dz = z - 100.0;
        double amb = CMB[6];
        for (int i = 5; i >= 0; --i) {
            amb = dz * amb + CMB[i];
        }
        return amb;
    }

    private static <T extends CalculusFieldElement<T>> T mBar(T z) {
        CalculusFieldElement dz = (CalculusFieldElement)z.subtract(100.0);
        CalculusFieldElement amb = (CalculusFieldElement)((CalculusFieldElement)z.getField().getZero()).add(CMB[6]);
        for (int i = 5; i >= 0; --i) {
            amb = (CalculusFieldElement)((CalculusFieldElement)dz.multiply((FieldElement)amb)).add(CMB[i]);
        }
        return (T)amb;
    }

    private static double localTemp(double z, double[] tc) {
        double dz = z - 125.0;
        if (dz <= 0.0) {
            return ((-9.8204695E-6 * dz - 7.3039742E-4) * dz * dz + 1.0) * dz * tc[1] + tc[0];
        }
        return tc[0] + tc[2] * FastMath.atan((double)(tc[3] * dz * (1.0 + 4.5E-6 * FastMath.pow((double)dz, (double)2.5))));
    }

    private static <T extends CalculusFieldElement<T>> T localTemp(T z, T[] tc) {
        CalculusFieldElement dz = (CalculusFieldElement)z.subtract(125.0);
        if (dz.getReal() <= 0.0) {
            return (T)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)dz.multiply(-9.8204695E-6)).subtract(7.3039742E-4)).multiply((FieldElement)dz)).multiply((FieldElement)dz)).add(1.0)).multiply((FieldElement)dz)).multiply(tc[1])).add(tc[0]));
        }
        return (T)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)dz.multiply((FieldElement)dz)).multiply((FieldElement)((CalculusFieldElement)dz.sqrt()))).multiply(4.5E-6)).add(1.0)).multiply((FieldElement)dz)).multiply(tc[3])).atan()).multiply(tc[2])).add(tc[0]));
    }

    private static double gravity(double z) {
        double tmp = 1.0 + z / 6356.766;
        return 9.80665 / (tmp * tmp);
    }

    private static <T extends CalculusFieldElement<T>> T gravity(T z) {
        CalculusFieldElement tmp = (CalculusFieldElement)((CalculusFieldElement)z.divide(6356.766)).add(1.0);
        return (T)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)tmp.multiply((FieldElement)tmp)).reciprocal()).multiply(9.80665));
    }

    private static double semian08(double doy, double alt, double f10B, double s10B, double xm10B) {
        double htz = alt / 1000.0;
        double fsmb = f10B - 0.7 * s10B - 0.04 * xm10B;
        double fzz = FZM[0] + fsmb * (FZM[1] + htz * (FZM[2] + FZM[3] * htz + FZM[4] * fsmb));
        fsmb = f10B - 0.75 * s10B - 0.37 * xm10B;
        double tau = Math.PI * 2 * (doy - 1.0) / 365.0;
        SinCos sc1P = FastMath.sinCos((double)tau);
        SinCos sc2P = SinCos.sum((SinCos)sc1P, (SinCos)sc1P);
        double gtz = GTM[0] + GTM[1] * sc1P.sin() + GTM[2] * sc1P.cos() + GTM[3] * sc2P.sin() + GTM[4] * sc2P.cos() + fsmb * (GTM[5] + GTM[6] * sc1P.sin() + GTM[7] * sc1P.cos() + GTM[8] * sc2P.sin() + GTM[9] * sc2P.cos());
        return FastMath.max((double)1.0E-6, (double)fzz) * gtz;
    }

    private static <T extends CalculusFieldElement<T>> T semian08(T doy, T alt, double f10B, double s10B, double xm10B) {
        CalculusFieldElement htz = (CalculusFieldElement)alt.divide(1000.0);
        double fsmb = f10B - 0.7 * s10B - 0.04 * xm10B;
        CalculusFieldElement fzz = (CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)htz.multiply(FZM[3])).add(FZM[2] + FZM[4] * fsmb)).multiply((FieldElement)htz)).add(FZM[1])).multiply(fsmb)).add(FZM[0]);
        fsmb = f10B - 0.75 * s10B - 0.37 * xm10B;
        CalculusFieldElement tau = (CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)doy.subtract(1.0)).divide(365.0)).multiply(Math.PI * 2);
        FieldSinCos sc1P = FastMath.sinCos((CalculusFieldElement)tau);
        FieldSinCos sc2P = FieldSinCos.sum((FieldSinCos)sc1P, (FieldSinCos)sc1P);
        CalculusFieldElement gtz = (CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)sc2P.cos()).multiply(GTM[9])).add((FieldElement)((CalculusFieldElement)((CalculusFieldElement)sc2P.sin()).multiply(GTM[8])))).add((FieldElement)((CalculusFieldElement)((CalculusFieldElement)sc1P.cos()).multiply(GTM[7])))).add((FieldElement)((CalculusFieldElement)((CalculusFieldElement)sc1P.sin()).multiply(GTM[6])))).add(GTM[5])).multiply(fsmb)).add((FieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)sc2P.cos()).multiply(GTM[4])).add((FieldElement)((CalculusFieldElement)((CalculusFieldElement)sc2P.sin()).multiply(GTM[3])))).add((FieldElement)((CalculusFieldElement)((CalculusFieldElement)sc1P.cos()).multiply(GTM[2])))).add((FieldElement)((CalculusFieldElement)((CalculusFieldElement)sc1P.sin()).multiply(GTM[1])))).add(GTM[0])));
        return (T)(fzz.getReal() > 1.0E-6 ? (CalculusFieldElement)gtz.multiply((FieldElement)fzz) : (CalculusFieldElement)gtz.multiply(1.0E-6));
    }

    private static double dayOfYear(double dateMJD) {
        double d1950 = dateMJD - 33281.0;
        int iyday = (int)d1950;
        double frac = d1950 - (double)iyday;
        int itemp = (iyday += 364) / 1461;
        if ((itemp = (iyday -= itemp * 1461) / 365) >= 3) {
            itemp = 3;
        }
        iyday = iyday - 365 * itemp + 1;
        return (double)iyday + frac;
    }

    private static <T extends CalculusFieldElement<T>> T dayOfYear(T dateMJD) {
        CalculusFieldElement d1950 = (CalculusFieldElement)dateMJD.subtract(33281.0);
        int iyday = (int)d1950.getReal();
        CalculusFieldElement frac = (CalculusFieldElement)d1950.subtract((double)iyday);
        int itemp = (iyday += 364) / 1461;
        if ((itemp = (iyday -= itemp * 1461) / 365) >= 3) {
            itemp = 3;
        }
        iyday = iyday - 365 * itemp + 1;
        return (T)((CalculusFieldElement)frac.add((double)iyday));
    }

    private <T extends CalculusFieldElement<T>> T min(double d, T f) {
        return (T)(f.getReal() > d ? (CalculusFieldElement)((CalculusFieldElement)f.getField().getZero()).add(d) : f);
    }

    private <T extends CalculusFieldElement<T>> T max(double d, T f) {
        return (T)(f.getReal() <= d ? (CalculusFieldElement)((CalculusFieldElement)f.getField().getZero()).add(d) : f);
    }

    @Override
    public double getDensity(AbsoluteDate date, Vector3D position, Frame frame) {
        if (date.compareTo(this.inputParams.getMaxDate()) > 0 || date.compareTo(this.inputParams.getMinDate()) < 0) {
            throw new OrekitException((Localizable)OrekitMessages.NO_SOLAR_ACTIVITY_AT_DATE, date, this.inputParams.getMinDate(), this.inputParams.getMaxDate());
        }
        DateTimeComponents dt = date.getComponents(this.utc);
        double dateMJD = (double)dt.getDate().getMJD() + dt.getTime().getSecondsInLocalDay() / 86400.0;
        GeodeticPoint inBody = this.earth.transform(position, frame, date);
        Frame ecef = this.earth.getBodyFrame();
        Vector3D sunPos = this.sun.getPVCoordinates(date, ecef).getPosition();
        GeodeticPoint sunInBody = this.earth.transform(sunPos, ecef, date);
        return this.getDensity(dateMJD, sunInBody.getLongitude(), sunInBody.getLatitude(), inBody.getLongitude(), inBody.getLatitude(), inBody.getAltitude(), this.inputParams.getF10(date), this.inputParams.getF10B(date), this.inputParams.getS10(date), this.inputParams.getS10B(date), this.inputParams.getXM10(date), this.inputParams.getXM10B(date), this.inputParams.getY10(date), this.inputParams.getY10B(date), this.inputParams.getDSTDTC(date));
    }

    @Override
    public <T extends CalculusFieldElement<T>> T getDensity(FieldAbsoluteDate<T> date, FieldVector3D<T> position, Frame frame) {
        AbsoluteDate dateD = date.toAbsoluteDate();
        if (dateD.compareTo(this.inputParams.getMaxDate()) > 0 || dateD.compareTo(this.inputParams.getMinDate()) < 0) {
            throw new OrekitException((Localizable)OrekitMessages.NO_SOLAR_ACTIVITY_AT_DATE, dateD, this.inputParams.getMinDate(), this.inputParams.getMaxDate());
        }
        DateTimeComponents components = date.getComponents(this.utc);
        CalculusFieldElement dateMJD = (CalculusFieldElement)((CalculusFieldElement)((CalculusFieldElement)date.durationFrom(new FieldAbsoluteDate<T>(date.getField(), components, this.utc)).add(components.getTime().getSecondsInLocalDay())).divide(86400.0)).add((double)components.getDate().getMJD());
        FieldGeodeticPoint<T> inBody = this.earth.transform(position, frame, date);
        Frame ecef = this.earth.getBodyFrame();
        FieldVector3D sunPos = new FieldVector3D(date.getField(), this.sun.getPVCoordinates(dateD, ecef).getPosition());
        FieldGeodeticPoint<T> sunInBody = this.earth.transform(sunPos, ecef, date);
        return (T)this.getDensity((T)dateMJD, sunInBody.getLongitude(), sunInBody.getLatitude(), inBody.getLongitude(), inBody.getLatitude(), inBody.getAltitude(), this.inputParams.getF10(dateD), this.inputParams.getF10B(dateD), this.inputParams.getS10(dateD), this.inputParams.getS10B(dateD), this.inputParams.getXM10(dateD), this.inputParams.getXM10B(dateD), this.inputParams.getY10(dateD), this.inputParams.getY10B(dateD), this.inputParams.getDSTDTC(dateD));
    }
}

