/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.files.ilrs;

import java.io.BufferedReader;
import java.io.IOException;
import java.util.Optional;
import java.util.regex.Pattern;
import java.util.stream.Stream;
import org.hipparchus.exception.Localizable;
import org.hipparchus.util.FastMath;
import org.orekit.annotation.DefaultDataContext;
import org.orekit.data.DataContext;
import org.orekit.data.DataSource;
import org.orekit.errors.OrekitException;
import org.orekit.errors.OrekitMessages;
import org.orekit.files.ilrs.CRD;
import org.orekit.files.ilrs.CRDConfiguration;
import org.orekit.files.ilrs.CRDHeader;
import org.orekit.time.AbsoluteDate;
import org.orekit.time.DateComponents;
import org.orekit.time.TimeComponents;
import org.orekit.time.TimeScale;
import org.orekit.utils.units.Unit;
import org.orekit.utils.units.UnitsConverter;

public class CRDParser {
    public static final String DEFAULT_CRD_SUPPORTED_NAMES = "^(?!0+$)\\w{1,12}\\_\\d{6,8}.\\w{3}$";
    private static final Unit NM = Unit.parse("nm");
    private static final Unit KHZ = Unit.parse("kHz");
    private static final Unit US = Unit.parse("\u00b5s");
    private static final UnitsConverter MBAR_TO_BAR = new UnitsConverter(Unit.parse("mbar"), Unit.parse("bar"));
    private static final String FILE_FORMAT = "CRD";
    private static final Pattern SEPARATOR = Pattern.compile("\\s+");
    private static final Pattern COMMA = Pattern.compile(",");
    private final TimeScale timeScale;

    @DefaultDataContext
    public CRDParser() {
        this(DataContext.getDefault().getTimeScales().getUTC());
    }

    public CRDParser(TimeScale utc) {
        this.timeScale = utc;
    }

    public TimeScale getTimeScale() {
        return this.timeScale;
    }

    public CRD parse(DataSource source) throws IOException {
        ParseInfo pi = new ParseInfo();
        int lineNumber = 0;
        Stream<LineParser> crdParsers = Stream.of(LineParser.H1);
        try (BufferedReader reader = new BufferedReader(source.getOpener().openReaderOnce());){
            String line = reader.readLine();
            while (line != null) {
                ++lineNumber;
                String l = line;
                Optional<LineParser> selected = crdParsers.filter(p -> p.canHandle(l)).findFirst();
                if (selected.isPresent()) {
                    try {
                        selected.get().parse(line, pi);
                    }
                    catch (NumberFormatException | StringIndexOutOfBoundsException e) {
                        throw new OrekitException(e, OrekitMessages.UNABLE_TO_PARSE_LINE_IN_FILE, lineNumber, source.getName(), line);
                    }
                    crdParsers = selected.get().allowedNext();
                }
                if (pi.done) {
                    CRD cRD = pi.file;
                    return cRD;
                }
                line = reader.readLine();
            }
        }
        throw new OrekitException((Localizable)OrekitMessages.CRD_UNEXPECTED_END_OF_FILE, lineNumber);
    }

    private static int checkRollover(double lastSecOfDay, double secOfDay) {
        return secOfDay > lastSecOfDay ? 0 : 1;
    }

    private static enum LineParser {
        H1(new String[]{"H1", "h1"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] values = SEPARATOR.split(line);
                String format = values[1];
                pi.version = Integer.parseInt(values[2]);
                if (!format.equalsIgnoreCase(CRDParser.FILE_FORMAT)) {
                    throw new OrekitException((Localizable)OrekitMessages.UNEXPECTED_FORMAT_FOR_ILRS_FILE, CRDParser.FILE_FORMAT, format);
                }
                pi.header.setFormat(format);
                pi.header.setVersion(pi.version);
                int year = Integer.parseInt(values[3]);
                int month = Integer.parseInt(values[4]);
                int day = Integer.parseInt(values[5]);
                pi.header.setProductionEpoch(new DateComponents(year, month, day));
                pi.header.setProductionHour(Integer.parseInt(values[6]));
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(H2, COMMENTS);
            }
        }
        ,
        H2(new String[]{"H2", "h2"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] values = SEPARATOR.split(line);
                pi.header.setStationName(values[1]);
                pi.header.setSystemIdentifier(Integer.parseInt(values[2]));
                pi.header.setSystemNumber(Integer.parseInt(values[3]));
                pi.header.setSystemOccupancy(Integer.parseInt(values[4]));
                pi.header.setEpochIdentifier(Integer.parseInt(values[5]));
                if (pi.version == 2) {
                    pi.header.setStationNetword(values[6]);
                }
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(H3, COMMENTS);
            }
        }
        ,
        H3(new String[]{"H3", "h3"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] values = SEPARATOR.split(line);
                pi.header.setName(values[1]);
                pi.header.setIlrsSatelliteId(values[2]);
                pi.header.setSic(values[3]);
                pi.header.setNoradId(values[4]);
                pi.header.setSpacecraftEpochTimeScale(Integer.parseInt(values[5]));
                pi.header.setTargetClass(Integer.parseInt(values[6]));
                if (pi.version == 2) {
                    pi.header.setTargetLocation(Integer.parseInt(values[7]));
                }
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(H4, COMMENTS);
            }
        }
        ,
        H4(new String[]{"H4", "h4"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] values = SEPARATOR.split(line);
                pi.header.setDataType(Integer.parseInt(values[1]));
                int yearS = Integer.parseInt(values[2]);
                int monthS = Integer.parseInt(values[3]);
                int dayS = Integer.parseInt(values[4]);
                int hourS = Integer.parseInt(values[5]);
                int minuteS = Integer.parseInt(values[6]);
                double secondS = Integer.parseInt(values[7]);
                pi.startEpoch = new DateComponents(yearS, monthS, dayS);
                pi.header.setStartEpoch(new AbsoluteDate(yearS, monthS, dayS, hourS, minuteS, secondS, pi.timeScale));
                int yearE = Integer.parseInt(values[8]);
                int monthE = Integer.parseInt(values[9]);
                int dayE = Integer.parseInt(values[10]);
                int hourE = Integer.parseInt(values[11]);
                int minuteE = Integer.parseInt(values[12]);
                double secondE = Integer.parseInt(values[13]);
                pi.header.setEndEpoch(new AbsoluteDate(yearE, monthE, dayE, hourE, minuteE, secondE, pi.timeScale));
                pi.header.setDataReleaseFlag(Integer.parseInt(values[14]));
                pi.header.setIsTroposphericRefractionApplied(LineParser.readBoolean(values[15]));
                pi.header.setIsCenterOfMassCorrectionApplied(LineParser.readBoolean(values[16]));
                pi.header.setIsReceiveAmplitudeCorrectionApplied(LineParser.readBoolean(values[17]));
                pi.header.setIsStationSystemDelayApplied(LineParser.readBoolean(values[18]));
                pi.header.setIsTransponderDelayApplied(LineParser.readBoolean(values[19]));
                pi.header.setRangeType(Integer.parseInt(values[20]));
                pi.header.setQualityIndicator(Integer.parseInt(values[21]));
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(H5, C0, COMMENTS);
            }
        }
        ,
        H5(new String[]{"H5", "h5"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] values = SEPARATOR.split(line);
                pi.header.setPredictionType(Integer.parseInt(values[1]));
                pi.header.setYearOfCentury(Integer.parseInt(values[2]));
                pi.header.setDateAndTime(values[3]);
                pi.header.setPredictionProvider(values[4]);
                pi.header.setSequenceNumber(Integer.parseInt(values[5]));
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(C0, COMMENTS);
            }
        }
        ,
        C0(new String[]{"C0", "c0"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                CRDConfiguration.SystemConfiguration systemRecord = new CRDConfiguration.SystemConfiguration();
                String[] values = SEPARATOR.split(line);
                systemRecord.setWavelength(NM.toSI(Double.parseDouble(values[2])));
                systemRecord.setSystemId(values[3]);
                pi.configurationRecords.setSystemRecord(systemRecord);
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(C1, C2, C3, C4, C5, C6, C7, TEN, ELEVEN, METEO, ANGLES, CALIB, STAT, COMPATIBILITY, COMMENTS);
            }
        }
        ,
        C1(new String[]{"C1", "c1"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                CRDConfiguration.LaserConfiguration laserRecord = new CRDConfiguration.LaserConfiguration();
                String[] values = SEPARATOR.split(line);
                laserRecord.setLaserId(values[2]);
                laserRecord.setLaserType(values[3]);
                laserRecord.setPrimaryWavelength(NM.toSI(Double.parseDouble(values[4])));
                laserRecord.setNominalFireRate(Double.parseDouble(values[5]));
                laserRecord.setPulseEnergy(Double.parseDouble(values[6]));
                laserRecord.setPulseWidth(Double.parseDouble(values[7]));
                laserRecord.setBeamDivergence(Double.parseDouble(values[8]));
                laserRecord.setPulseInOutgoingSemiTrain(Integer.parseInt(values[9]));
                pi.configurationRecords.setLaserRecord(laserRecord);
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(C2, C3, C4, C5, C6, C7, TEN, ELEVEN, METEO, ANGLES, CALIB, STAT, COMPATIBILITY, COMMENTS);
            }
        }
        ,
        C2(new String[]{"C2", "c2"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                CRDConfiguration.DetectorConfiguration detectorRecord = new CRDConfiguration.DetectorConfiguration();
                String[] values = SEPARATOR.split(line);
                detectorRecord.setDetectorId(values[2]);
                detectorRecord.setDetectorType(values[3]);
                detectorRecord.setApplicableWavelength(NM.toSI(Double.parseDouble(values[4])));
                detectorRecord.setQuantumEfficiency(Double.parseDouble(values[5]));
                detectorRecord.setAppliedVoltage(Double.parseDouble(values[6]));
                detectorRecord.setDarkCount(KHZ.toSI(Double.parseDouble(values[7])));
                detectorRecord.setOutputPulseType(values[8]);
                detectorRecord.setOutputPulseWidth(Double.parseDouble(values[9]));
                detectorRecord.setSpectralFilter(NM.toSI(Double.parseDouble(values[10])));
                detectorRecord.setTransmissionOfSpectralFilter(Double.parseDouble(values[11]));
                detectorRecord.setSpatialFilter(Double.parseDouble(values[12]));
                detectorRecord.setExternalSignalProcessing(values[13]);
                if (pi.version == 2) {
                    detectorRecord.setAmplifierGain(Double.parseDouble(values[14]));
                    detectorRecord.setAmplifierBandwidth(KHZ.toSI(Double.parseDouble(values[15])));
                    detectorRecord.setAmplifierInUse(values[16]);
                }
                pi.configurationRecords.setDetectorRecord(detectorRecord);
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(C3, C4, C5, C6, C7, TEN, ELEVEN, METEO, ANGLES, CALIB, STAT, COMPATIBILITY, COMMENTS);
            }
        }
        ,
        C3(new String[]{"C3", "c3"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                CRDConfiguration.TimingSystemConfiguration timingRecord = new CRDConfiguration.TimingSystemConfiguration();
                String[] values = SEPARATOR.split(line);
                timingRecord.setLocalTimingId(values[2]);
                timingRecord.setTimeSource(values[3]);
                timingRecord.setFrequencySource(values[4]);
                timingRecord.setTimer(values[5]);
                timingRecord.setTimerSerialNumber(values[6]);
                timingRecord.setEpochDelayCorrection(US.toSI(Double.parseDouble(values[7])));
                pi.configurationRecords.setTimingRecord(timingRecord);
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(C4, C5, C6, C7, TEN, ELEVEN, METEO, ANGLES, CALIB, STAT, COMPATIBILITY, COMMENTS);
            }
        }
        ,
        C4(new String[]{"C4", "c4"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                CRDConfiguration.TransponderConfiguration transponderRecord = new CRDConfiguration.TransponderConfiguration();
                String[] values = SEPARATOR.split(line);
                transponderRecord.setTransponderId(values[2]);
                transponderRecord.setStationUTCOffset(NM.toSI(Double.parseDouble(values[3])));
                transponderRecord.setStationOscDrift(Double.parseDouble(values[4]));
                transponderRecord.setTranspUTCOffset(NM.toSI(Double.parseDouble(values[5])));
                transponderRecord.setTranspOscDrift(Double.parseDouble(values[6]));
                transponderRecord.setTranspClkRefTime(Double.parseDouble(values[7]));
                transponderRecord.setStationClockAndDriftApplied(Integer.parseInt(values[8]));
                transponderRecord.setSpacecraftClockAndDriftApplied(Integer.parseInt(values[9]));
                transponderRecord.setIsSpacecraftTimeSimplified(LineParser.readBoolean(values[10]));
                pi.configurationRecords.setTransponderRecord(transponderRecord);
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(C5, C6, C7, TEN, ELEVEN, METEO, ANGLES, CALIB, STAT, COMPATIBILITY, COMMENTS);
            }
        }
        ,
        C5(new String[]{"C5", "c5"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                CRDConfiguration.SoftwareConfiguration softwareRecord = new CRDConfiguration.SoftwareConfiguration();
                String[] values = SEPARATOR.split(line);
                softwareRecord.setSoftwareId(values[2]);
                softwareRecord.setTrackingSoftwares(COMMA.split(values[3]));
                softwareRecord.setTrackingSoftwareVersions(COMMA.split(values[4]));
                softwareRecord.setProcessingSoftwares(COMMA.split(values[5]));
                softwareRecord.setProcessingSoftwareVersions(COMMA.split(values[6]));
                pi.configurationRecords.setSoftwareRecord(softwareRecord);
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(C6, C7, TEN, ELEVEN, METEO, ANGLES, CALIB, STAT, COMPATIBILITY, COMMENTS);
            }
        }
        ,
        C6(new String[]{"C6", "c6"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                CRDConfiguration.MeteorologicalConfiguration meteoRecord = new CRDConfiguration.MeteorologicalConfiguration();
                String[] values = SEPARATOR.split(line);
                meteoRecord.setMeteorologicalId(values[2]);
                meteoRecord.setPressSensorManufacturer(values[3]);
                meteoRecord.setPressSensorModel(values[4]);
                meteoRecord.setPressSensorSerialNumber(values[5]);
                meteoRecord.setTempSensorManufacturer(values[6]);
                meteoRecord.setTempSensorModel(values[7]);
                meteoRecord.setTempSensorSerialNumber(values[8]);
                meteoRecord.setHumiSensorManufacturer(values[9]);
                meteoRecord.setHumiSensorModel(values[10]);
                meteoRecord.setHumiSensorSerialNumber(values[11]);
                pi.configurationRecords.setMeteorologicalRecord(meteoRecord);
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(C7, TEN, ELEVEN, METEO, ANGLES, CALIB, STAT, COMPATIBILITY, COMMENTS);
            }
        }
        ,
        C7(new String[]{"C7", "c7"}){

            @Override
            public void parse(String line, ParseInfo pi) {
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(TEN, ELEVEN, METEO, ANGLES, CALIB, STAT, COMPATIBILITY, COMMENTS);
            }
        }
        ,
        TEN(new String[]{"10"}){
            private double lastSecOfDay = 0.0;
            private int dayShift = 0;

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] values = SEPARATOR.split(line);
                double secOfDay = Double.parseDouble(values[1]);
                double timeOfFlight = Double.parseDouble(values[2]);
                int epochEvent = Integer.parseInt(values[4]);
                this.dayShift += CRDParser.checkRollover(this.lastSecOfDay, secOfDay);
                AbsoluteDate epoch = new AbsoluteDate(pi.startEpoch, new TimeComponents(secOfDay), pi.timeScale).shiftedBy(this.dayShift * 86400);
                CRD.RangeMeasurement range = new CRD.RangeMeasurement(epoch, timeOfFlight, epochEvent);
                pi.dataBlock.addRangeData(range);
                this.lastSecOfDay = secOfDay;
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(H8, TEN, TWELVE, METEO, ANGLES, CALIB, STAT, COMPATIBILITY, COMMENTS);
            }
        }
        ,
        ELEVEN(new String[]{"11"}){
            private double lastSecOfDay = 0.0;
            private int dayShift = 0;

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] values = SEPARATOR.split(line);
                double secOfDay = Double.parseDouble(values[1]);
                double timeOfFlight = Double.parseDouble(values[2]);
                int epochEvent = Integer.parseInt(values[4]);
                double snr = pi.version == 2 ? Double.parseDouble(values[13]) : Double.NaN;
                this.dayShift += CRDParser.checkRollover(this.lastSecOfDay, secOfDay);
                AbsoluteDate epoch = new AbsoluteDate(pi.startEpoch, new TimeComponents(secOfDay), pi.timeScale).shiftedBy(this.dayShift * 86400);
                CRD.RangeMeasurement range = new CRD.RangeMeasurement(epoch, timeOfFlight, epochEvent, snr);
                pi.dataBlock.addRangeData(range);
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(H8, ELEVEN, TWELVE, METEO, ANGLES, CALIB, STAT, COMPATIBILITY, COMMENTS);
            }
        }
        ,
        TWELVE(new String[]{"12"}){

            @Override
            public void parse(String line, ParseInfo pi) {
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(H8, TEN, ELEVEN, TWELVE, METEO, ANGLES, CALIB, STAT, COMPATIBILITY, COMMENTS);
            }
        }
        ,
        METEO(new String[]{"20"}){
            private double lastSecOfDay = 0.0;
            private int dayShift = 0;

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] values = SEPARATOR.split(line);
                double secOfDay = Double.parseDouble(values[1]);
                double pressure = MBAR_TO_BAR.convert(Double.parseDouble(values[2]));
                double temperature = Double.parseDouble(values[3]);
                double humidity = Double.parseDouble(values[4]);
                this.dayShift += CRDParser.checkRollover(this.lastSecOfDay, secOfDay);
                AbsoluteDate epoch = new AbsoluteDate(pi.startEpoch, new TimeComponents(secOfDay), pi.timeScale).shiftedBy(this.dayShift * 86400);
                CRD.MeteorologicalMeasurement meteo = new CRD.MeteorologicalMeasurement(epoch, pressure, temperature, humidity);
                pi.dataBlock.addMeteoData(meteo);
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(H8, METEO, METEO_SUPP, TEN, ELEVEN, TWELVE, ANGLES, CALIB, STAT, COMPATIBILITY, COMMENTS);
            }
        }
        ,
        METEO_SUPP(new String[]{"21"}){

            @Override
            public void parse(String line, ParseInfo pi) {
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(H8, METEO, METEO_SUPP, TEN, ELEVEN, TWELVE, ANGLES, CALIB, STAT, COMPATIBILITY, COMMENTS);
            }
        }
        ,
        ANGLES(new String[]{"30"}){
            private double lastSecOfDay = 0.0;
            private int dayShift = 0;

            @Override
            public void parse(String line, ParseInfo pi) {
                String[] values = SEPARATOR.split(line);
                double secOfDay = Double.parseDouble(values[1]);
                double azmiuth = FastMath.toRadians((double)Double.parseDouble(values[2]));
                double elevation = FastMath.toRadians((double)Double.parseDouble(values[3]));
                int directionFlag = Integer.parseInt(values[4]);
                int orginFlag = Integer.parseInt(values[5]);
                boolean isRefractionCorrected = LineParser.readBoolean(values[6]);
                double azimuthRate = Double.NaN;
                double elevationRate = Double.NaN;
                if (pi.version == 2) {
                    azimuthRate = LineParser.readDoubleWithNaN(values[7]);
                    elevationRate = LineParser.readDoubleWithNaN(values[8]);
                }
                this.dayShift += CRDParser.checkRollover(this.lastSecOfDay, secOfDay);
                AbsoluteDate epoch = new AbsoluteDate(pi.startEpoch, new TimeComponents(secOfDay), pi.timeScale);
                CRD.AnglesMeasurement angles = new CRD.AnglesMeasurement(epoch, azmiuth, elevation, directionFlag, orginFlag, isRefractionCorrected, azimuthRate, elevationRate);
                pi.dataBlock.addAnglesData(angles);
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(H8, METEO, TEN, ELEVEN, ANGLES, CALIB, STAT, COMPATIBILITY, COMMENTS);
            }
        }
        ,
        CALIB(new String[]{"40"}){

            @Override
            public void parse(String line, ParseInfo pi) {
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(H8, METEO, CALIB, CALIB_DETAILS, CALIB_SHOT, TEN, ELEVEN, TWELVE, ANGLES, STAT, COMPATIBILITY, COMMENTS);
            }
        }
        ,
        CALIB_DETAILS(new String[]{"41"}){

            @Override
            public void parse(String line, ParseInfo pi) {
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(H8, METEO, CALIB, CALIB_DETAILS, CALIB_SHOT, TEN, ELEVEN, TWELVE, ANGLES, STAT, COMPATIBILITY, COMMENTS);
            }
        }
        ,
        CALIB_SHOT(new String[]{"42"}){

            @Override
            public void parse(String line, ParseInfo pi) {
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(H8, METEO, CALIB, CALIB_DETAILS, CALIB_SHOT, TEN, ELEVEN, TWELVE, ANGLES, STAT, COMPATIBILITY, COMMENTS);
            }
        }
        ,
        STAT(new String[]{"50"}){

            @Override
            public void parse(String line, ParseInfo pi) {
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(H8, METEO, CALIB, CALIB_DETAILS, CALIB_SHOT, TEN, ELEVEN, TWELVE, ANGLES, STAT, COMPATIBILITY, H8, COMMENTS);
            }
        }
        ,
        COMPATIBILITY(new String[]{"60"}){

            @Override
            public void parse(String line, ParseInfo pi) {
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(H8, METEO, CALIB, CALIB_DETAILS, CALIB_SHOT, TEN, ELEVEN, TWELVE, ANGLES, STAT, COMPATIBILITY, COMMENTS);
            }
        }
        ,
        COMMENTS(new String[]{"00"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                String comment = line.split(this.getFirstIdentifier())[1].trim();
                pi.file.getComments().add(comment);
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(H1, H2, H3, H4, H5, H8, H9, C0, C1, C2, C3, C4, C5, C6, C7, TEN, ELEVEN, TWELVE, METEO, METEO_SUPP, ANGLES, CALIB, CALIB_DETAILS, CALIB_SHOT, STAT, COMPATIBILITY, COMMENTS);
            }
        }
        ,
        H8(new String[]{"H8", "h8"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                pi.dataBlock.setHeader(pi.header);
                pi.dataBlock.setConfigurationRecords(pi.configurationRecords);
                pi.file.addDataBlock(pi.dataBlock);
                pi.startEpoch = DateComponents.J2000_EPOCH;
                pi.header = new CRDHeader();
                pi.configurationRecords = new CRDConfiguration();
                pi.dataBlock = new CRD.CRDDataBlock();
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(H1, H9, COMMENTS);
            }
        }
        ,
        H9(new String[]{"H9", "h9"}){

            @Override
            public void parse(String line, ParseInfo pi) {
                pi.done = true;
            }

            @Override
            public Stream<LineParser> allowedNext() {
                return Stream.of(H9);
            }
        };

        private final Pattern[] patterns;
        private final String[] identifiers;

        private LineParser(String ... identifier) {
            this.identifiers = identifier;
            this.patterns = new Pattern[this.identifiers.length];
            for (int index = 0; index < this.patterns.length; ++index) {
                this.patterns[index] = Pattern.compile(this.identifiers[index]);
            }
        }

        public String getFirstIdentifier() {
            return this.identifiers[0];
        }

        public abstract void parse(String var1, ParseInfo var2);

        public abstract Stream<LineParser> allowedNext();

        public boolean canHandle(String line) {
            String lineId = SEPARATOR.split(line)[0];
            for (Pattern pattern : this.patterns) {
                if (!pattern.matcher(lineId).matches()) continue;
                return true;
            }
            return false;
        }

        private static boolean readBoolean(String value) {
            return Integer.parseInt(value) == 1;
        }

        private static double readDoubleWithNaN(String value) {
            return "na".equals(value) ? Double.NaN : Double.parseDouble(value);
        }
    }

    private class ParseInfo {
        private CRD file;
        private int version = 1;
        private CRD.CRDDataBlock dataBlock;
        private CRDHeader header;
        private CRDConfiguration configurationRecords;
        private TimeScale timeScale;
        private DateComponents startEpoch = DateComponents.J2000_EPOCH;
        private boolean done = false;

        protected ParseInfo() {
            this.file = new CRD();
            this.header = new CRDHeader();
            this.configurationRecords = new CRDConfiguration();
            this.dataBlock = new CRD.CRDDataBlock();
            this.timeScale = CRDParser.this.timeScale;
        }
    }
}

