import java.io
import java.util
import org
import org.hipparchus.distribution
import org.hipparchus.linear
import typing



class NormalizedRandomGenerator:
    """
    public interface NormalizedRandomGenerator
    
        This interface represent a normalized random generator for scalars. Normalized generator provide null mean and unit
        standard deviation scalars.
    """
    def nextNormalizedDouble(self) -> float:
        """
            Generate a random scalar with null mean and unit standard deviation.
        
            This method does **not** specify the shape of the distribution, it is the implementing class that provides it. The only
            contract here is to generate numbers with null mean and unit standard deviation.
        
            Returns:
                a random scalar with null mean and unit standard deviation
        
        
        """
        ...

class RandomGenerator:
    """
    public interface RandomGenerator
    
        Interface for generators of random number sequences.
    """
    def nextBoolean(self) -> bool:
        """
            Returns the next pseudorandom, uniformly distributed :code:`boolean` value from this random number generator's sequence.
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`boolean` value from this random number generator's sequence
        
        
        """
        ...
    @typing.overload
    def nextBytes(self, byteArray: typing.List[int]) -> None:
        """
            Generates random bytes and places them into a user-supplied byte array. The number of random bytes produced is equal to
            the length of the byte array.
        
            Parameters:
                bytes (byte[]): the non-null byte array in which to put the random bytes
        
        """
        ...
    @typing.overload
    def nextBytes(self, byteArray: typing.List[int], int: int, int2: int) -> None:
        """
            Generates random bytes and places them into a user-supplied byte array.
        
            Parameters:
                bytes (byte[]): the non-null byte array in which to put the random bytes
                offset (int): the starting index for inserting the generated bytes into the array
                len (int): the number of bytes to generate
        
            Raises:
                :class:`~org.hipparchus.exception.MathIllegalArgumentException`: if :code:`offset < 0` or :code:`offset + len >= bytes.length`
        
        
        """
        ...
    def nextDouble(self) -> float:
        """
            Returns the next pseudorandom, uniformly distributed :code:`double` value between :code:`0.0` and :code:`1.0` from this
            random number generator's sequence.
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`double` value between :code:`0.0` and :code:`1.0` from this random
                number generator's sequence
        
        
        """
        ...
    def nextFloat(self) -> float:
        """
            Returns the next pseudorandom, uniformly distributed :code:`float` value between :code:`0.0` and :code:`1.0` from this
            random number generator's sequence.
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`float` value between :code:`0.0` and :code:`1.0` from this random
                number generator's sequence
        
        
        """
        ...
    def nextGaussian(self) -> float:
        """
            Returns the next pseudorandom, Gaussian ("normally") distributed :code:`double` value with mean :code:`0.0` and standard
            deviation :code:`1.0` from this random number generator's sequence.
        
            Returns:
                the next pseudorandom, Gaussian ("normally") distributed :code:`double` value with mean :code:`0.0` and standard
                deviation :code:`1.0` from this random number generator's sequence
        
        
        """
        ...
    @typing.overload
    def nextInt(self) -> int:
        """
            Returns the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence.
        
            All 2 :sup:`32` possible :code:`int` values should be produced with (approximately) equal probability.
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence
        
        """
        ...
    @typing.overload
    def nextInt(self, int: int) -> int:
        """
            Returns a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and the specified value
            (exclusive), drawn from this random number generator's sequence.
        
            Parameters:
                n (int): the bound on the random number to be returned. Must be positive.
        
            Returns:
                a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and n (exclusive).
        
            Raises:
                :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.IllegalArgumentException?is`: if n is not positive.
        
        
        """
        ...
    @typing.overload
    def nextLong(self) -> int:
        """
            Returns the next pseudorandom, uniformly distributed :code:`long` value from this random number generator's sequence.
            All 2 :sup:`64` possible :code:`long` values should be produced with (approximately) equal probability.
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`long` value from this random number generator's sequence
        
        """
        ...
    @typing.overload
    def nextLong(self, long: int) -> int:
        """
            Returns a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and the specified value
            (exclusive), drawn from this random number generator's sequence.
        
            Parameters:
                n (long): the bound on the random number to be returned. Must be positive.
        
            Returns:
                a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and n (exclusive).
        
            Raises:
                :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.IllegalArgumentException?is`: if n is not positive.
        
        
        """
        ...
    @typing.overload
    def setSeed(self, int: int) -> None:
        """
            Sets the seed of the underlying random number generator using an :code:`int` seed.
        
            Sequences of values generated starting with the same seeds should be identical.
        
            Parameters:
                seed (int): the seed value
        
            Sets the seed of the underlying random number generator using an :code:`int` array seed.
        
            Sequences of values generated starting with the same seeds should be identical.
        
            Parameters:
                seed (int[]): the seed value
        
            Sets the seed of the underlying random number generator using a :code:`long` seed.
        
            Sequences of values generated starting with the same seeds should be identical.
        
            Parameters:
                seed (long): the seed value
        
        
        """
        ...
    @typing.overload
    def setSeed(self, intArray: typing.List[int]) -> None: ...
    @typing.overload
    def setSeed(self, long: int) -> None: ...

class RandomVectorGenerator:
    """
    public interface RandomVectorGenerator
    
        This interface represents a random generator for whole vectors.
    """
    def nextVector(self) -> typing.List[float]:
        """
            Generate a random vector.
        
            Returns:
                a random vector as an array of double.
        
        
        """
        ...

class CorrelatedRandomVectorGenerator(RandomVectorGenerator):
    """
    public class CorrelatedRandomVectorGenerator extends :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.Object?is` implements :class:`~org.hipparchus.random.RandomVectorGenerator`
    
        A :class:`~org.hipparchus.random.RandomVectorGenerator` that generates vectors with with correlated components.
    
        Random vectors with correlated components are built by combining the uncorrelated components of another random vector in
        such a way that the resulting correlations are the ones specified by a positive definite covariance matrix.
    
        The main use for correlated random vector generation is for Monte-Carlo simulation of physical problems with several
        variables, for example to generate error vectors to be added to a nominal vector. A particularly interesting case is
        when the generated vector should be drawn from a ` Multivariate Normal Distribution
        <http://en.wikipedia.org/wiki/Multivariate_normal_distribution>`. The approach using a Cholesky decomposition is quite
        usual in this case. However, it can be extended to other cases as long as the underlying random generator provides
        :class:`~org.hipparchus.random.NormalizedRandomGenerator` like :class:`~org.hipparchus.random.GaussianRandomGenerator`
        or :class:`~org.hipparchus.random.UniformRandomGenerator`.
    
        Sometimes, the covariance matrix for a given simulation is not strictly positive definite. This means that the
        correlations are not all independent from each other. In this case, however, the non strictly positive elements found
        during the Cholesky decomposition of the covariance matrix should not be negative either, they should be null. Another
        non-conventional extension handling this case is used here. Rather than computing :code:`C = U :sup:`T` .U` where
        :code:`C` is the covariance matrix and :code:`U` is an upper-triangular matrix, we compute :code:`C = B.B :sup:`T``
        where :code:`B` is a rectangular matrix having more rows than columns. The number of columns of :code:`B` is the rank of
        the covariance matrix, and it is the dimension of the uncorrelated random vector that is needed to compute the component
        of the correlated vector. This class handles this situation automatically.
    """
    @typing.overload
    def __init__(self, doubleArray: typing.List[float], realMatrix: org.hipparchus.linear.RealMatrix, double2: float, normalizedRandomGenerator: NormalizedRandomGenerator): ...
    @typing.overload
    def __init__(self, realMatrix: org.hipparchus.linear.RealMatrix, double: float, normalizedRandomGenerator: NormalizedRandomGenerator): ...
    def getGenerator(self) -> NormalizedRandomGenerator:
        """
            Get the underlying normalized components generator.
        
            Returns:
                underlying uncorrelated components generator
        
        
        """
        ...
    def getRank(self) -> int:
        """
            Get the rank of the covariance matrix. The rank is the number of independent rows in the covariance matrix, it is also
            the number of columns of the root matrix.
        
            Returns:
                rank of the square matrix.
        
            Also see:
                :meth:`~org.hipparchus.random.CorrelatedRandomVectorGenerator.getRootMatrix`
        
        
        """
        ...
    def getRootMatrix(self) -> org.hipparchus.linear.RealMatrix:
        """
            Get the root of the covariance matrix. The root is the rectangular matrix :code:`B` such that the covariance matrix is
            equal to :code:`B.B :sup:`T``
        
            Returns:
                root of the square matrix
        
            Also see:
                :meth:`~org.hipparchus.random.CorrelatedRandomVectorGenerator.getRank`
        
        
        """
        ...
    def nextVector(self) -> typing.List[float]:
        """
            Generate a correlated random vector.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomVectorGenerator.nextVector` in
                interface :class:`~org.hipparchus.random.RandomVectorGenerator`
        
            Returns:
                a random vector as an array of double. The returned array is created at each call, the caller can do what it wants with
                it.
        
        
        """
        ...

class GaussianRandomGenerator(NormalizedRandomGenerator):
    """
    public class GaussianRandomGenerator extends :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.Object?is` implements :class:`~org.hipparchus.random.NormalizedRandomGenerator`
    
        This class is a gaussian normalized random generator for scalars.
    
        This class is a simple wrapper around the :meth:`~org.hipparchus.random.RandomGenerator.nextGaussian` method.
    """
    def __init__(self, randomGenerator: RandomGenerator): ...
    def nextNormalizedDouble(self) -> float:
        """
            Generate a random scalar with null mean and unit standard deviation.
        
            Specified by:
                :meth:`~org.hipparchus.random.NormalizedRandomGenerator.nextNormalizedDouble` in
                interface :class:`~org.hipparchus.random.NormalizedRandomGenerator`
        
            Returns:
                a random scalar with null mean and unit standard deviation
        
        
        """
        ...

class HaltonSequenceGenerator(RandomVectorGenerator):
    """
    public class HaltonSequenceGenerator extends :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.Object?is` implements :class:`~org.hipparchus.random.RandomVectorGenerator`
    
        Implementation of a Halton sequence.
    
        A Halton sequence is a low-discrepancy sequence generating points in the interval [0, 1] according to
    
        .. code-block: java
        
           H(n) = d_0 / b + d_1 / b^2 .... d_j / b^j+1
        
           with
        
           n = d_j * b^j-1 + ... d_1 * b + d_0 * b^0
         
        For higher dimensions, subsequent prime numbers are used as base, e.g. { 2, 3, 5 } for a Halton sequence in R^3.
    
        Halton sequences are known to suffer from linear correlation for larger prime numbers, thus the individual digits are
        usually scrambled. This implementation already comes with support for up to 40 dimensions with optimal weight numbers
        from ` H. Chi: Scrambled quasirandom sequences and their applications
        <http://etd.lib.fsu.edu/theses/available/etd-07062004-140409/unrestricted/dissertation1.pdf>`.
    
        The generator supports two modes:
    
          - sequential generation of points: :meth:`~org.hipparchus.random.HaltonSequenceGenerator.nextVector`
          - random access to the i-th point in the sequence: :meth:`~org.hipparchus.random.HaltonSequenceGenerator.skipTo`
    
    
        Also see:
            `Halton sequence (Wikipedia) <http://en.wikipedia.org/wiki/Halton_sequence>`,
            :class:`~org.hipparchus.random.https:.lirias.kuleuven.be.bitstream.123456789.131168.1.mcm2005_bartv.pdf`
    """
    @typing.overload
    def __init__(self, int: int): ...
    @typing.overload
    def __init__(self, int: int, intArray: typing.List[int], intArray2: typing.List[int]): ...
    def getNextIndex(self) -> int:
        """
            Returns the index i of the next point in the Halton sequence that will be returned by calling
            :meth:`~org.hipparchus.random.HaltonSequenceGenerator.nextVector`.
        
            Returns:
                the index of the next point
        
        
        """
        ...
    def nextVector(self) -> typing.List[float]:
        """
            Generate a random vector.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomVectorGenerator.nextVector` in
                interface :class:`~org.hipparchus.random.RandomVectorGenerator`
        
            Returns:
                a random vector as an array of double.
        
        
        """
        ...
    def skipTo(self, int: int) -> typing.List[float]: ...

class RandomAdaptor(java.util.Random, RandomGenerator):
    """
    public class RandomAdaptor extends :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.util.Random?is` implements :class:`~org.hipparchus.random.RandomGenerator`
    
        Extension of :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.util.Random?is` wrapping a
        :class:`~org.hipparchus.random.RandomGenerator`.
    
        Also see:
            :meth:`~serialized`
    """
    def __init__(self, randomGenerator: RandomGenerator): ...
    def nextBoolean(self) -> bool:
        """
            Returns the next pseudorandom, uniformly distributed :code:`boolean` value from this random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextBoolean` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Overrides:
                :meth:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.util.Random.html?is` in
                class :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.util.Random?is`
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`boolean` value from this random number generator's sequence
        
        
        """
        ...
    @typing.overload
    def nextBytes(self, byteArray: typing.List[int]) -> None:
        """
            Generates random bytes and places them into a user-supplied byte array. The number of random bytes produced is equal to
            the length of the byte array.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextBytes` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Overrides:
                :meth:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.util.Random.html?is` in
                class :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.util.Random?is`
        
            Parameters:
                bytes (byte[]): the non-null byte array in which to put the random bytes
        
        """
        ...
    @typing.overload
    def nextBytes(self, byteArray: typing.List[int], int: int, int2: int) -> None:
        """
            Generates random bytes and places them into a user-supplied byte array.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextBytes` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                bytes (byte[]): the non-null byte array in which to put the random bytes
                offset (int): the starting index for inserting the generated bytes into the array
                len (int): the number of bytes to generate
        
        
        """
        ...
    def nextDouble(self) -> float:
        """
            Returns the next pseudorandom, uniformly distributed :code:`double` value between :code:`0.0` and :code:`1.0` from this
            random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextDouble` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Overrides:
                :meth:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.util.Random.html?is` in
                class :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.util.Random?is`
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`double` value between :code:`0.0` and :code:`1.0` from this random
                number generator's sequence
        
        
        """
        ...
    def nextFloat(self) -> float:
        """
            Returns the next pseudorandom, uniformly distributed :code:`float` value between :code:`0.0` and :code:`1.0` from this
            random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextFloat` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Overrides:
                :meth:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.util.Random.html?is` in
                class :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.util.Random?is`
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`float` value between :code:`0.0` and :code:`1.0` from this random
                number generator's sequence
        
        
        """
        ...
    def nextGaussian(self) -> float:
        """
            Returns the next pseudorandom, Gaussian ("normally") distributed :code:`double` value with mean :code:`0.0` and standard
            deviation :code:`1.0` from this random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextGaussian` in
                interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Overrides:
                :meth:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.util.Random.html?is` in
                class :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.util.Random?is`
        
            Returns:
                the next pseudorandom, Gaussian ("normally") distributed :code:`double` value with mean :code:`0.0` and standard
                deviation :code:`1.0` from this random number generator's sequence
        
        
        """
        ...
    @typing.overload
    def nextInt(self) -> int:
        """
            Returns the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence. All
            2:sup:`32` possible :code:`int` values should be produced with (approximately) equal probability.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextInt` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Overrides:
                :meth:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.util.Random.html?is` in
                class :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.util.Random?is`
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence
        
        """
        ...
    @typing.overload
    def nextInt(self, int: int) -> int:
        """
            Returns a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and the specified value
            (exclusive), drawn from this random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextInt` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Overrides:
                :meth:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.util.Random.html?is` in
                class :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.util.Random?is`
        
            Parameters:
                n (int): the bound on the random number to be returned. Must be positive.
        
            Returns:
                a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and n (exclusive).
        
            Raises:
                :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.IllegalArgumentException?is`: if n is not positive.
        
        
        """
        ...
    @typing.overload
    def nextLong(self) -> int:
        """
            Returns the next pseudorandom, uniformly distributed :code:`long` value from this random number generator's sequence.
            All 2:sup:`64` possible :code:`long` values should be produced with (approximately) equal probability.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextLong` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Overrides:
                :meth:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.util.Random.html?is` in
                class :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.util.Random?is`
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`long` value from this random number generator's sequence
        
        """
        ...
    @typing.overload
    def nextLong(self, long: int) -> int:
        """
            Returns a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and the specified value
            (exclusive), drawn from this random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextLong` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                n (long): the bound on the random number to be returned. Must be positive.
        
            Returns:
                a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and n (exclusive).
        
        
        """
        ...
    @staticmethod
    def of(randomGenerator: RandomGenerator) -> java.util.Random:
        """
            Factory method to create a :code:`Random` using the supplied :code:`RandomGenerator`.
        
            Parameters:
                randomGenerator (:class:`~org.hipparchus.random.RandomGenerator`): wrapped RandomGenerator instance
        
            Returns:
                a Random instance wrapping the RandomGenerator
        
        
        """
        ...
    @typing.overload
    def setSeed(self, int: int) -> None:
        """
            Sets the seed of the underlying random number generator using an :code:`int` seed.
        
            Sequences of values generated starting with the same seeds should be identical.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.setSeed` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                seed (int): the seed value
        
            Sets the seed of the underlying random number generator using an :code:`int` array seed.
        
            Sequences of values generated starting with the same seeds should be identical.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.setSeed` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                seed (int[]): the seed value
        
            Sets the seed of the underlying random number generator using a :code:`long` seed.
        
            Sequences of values generated starting with the same seeds should be identical.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.setSeed` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Overrides:
                :meth:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.util.Random.html?is` in
                class :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.util.Random?is`
        
            Parameters:
                seed (long): the seed value
        
        
        """
        ...
    @typing.overload
    def setSeed(self, intArray: typing.List[int]) -> None: ...
    @typing.overload
    def setSeed(self, long: int) -> None: ...

class SobolSequenceGenerator(RandomVectorGenerator):
    """
    public class SobolSequenceGenerator extends :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.Object?is` implements :class:`~org.hipparchus.random.RandomVectorGenerator`
    
        Implementation of a Sobol sequence.
    
        A Sobol sequence is a low-discrepancy sequence with the property that for all values of N, its subsequence (x1, ... xN)
        has a low discrepancy. It can be used to generate pseudo-random points in a space S, which are equi-distributed.
    
        The implementation already comes with support for up to 1000 dimensions with direction numbers calculated from `Stephen
        Joe and Frances Kuo <http://web.maths.unsw.edu.au/~fkuo/sobol/>`.
    
        The generator supports two modes:
    
          - sequential generation of points: :meth:`~org.hipparchus.random.SobolSequenceGenerator.nextVector`
          - random access to the i-th point in the sequence: :meth:`~org.hipparchus.random.SobolSequenceGenerator.skipTo`
    
    
        Also see:
            `Sobol sequence (Wikipedia) <http://en.wikipedia.org/wiki/Sobol_sequence>`, `Sobol sequence direction numbers
            <http://web.maths.unsw.edu.au/~fkuo/sobol/>`
    """
    @typing.overload
    def __init__(self, int: int): ...
    @typing.overload
    def __init__(self, int: int, inputStream: java.io.InputStream): ...
    def getNextIndex(self) -> int:
        """
            Returns the index i of the next point in the Sobol sequence that will be returned by calling
            :meth:`~org.hipparchus.random.SobolSequenceGenerator.nextVector`.
        
            Returns:
                the index of the next point
        
        
        """
        ...
    def nextVector(self) -> typing.List[float]:
        """
            Generate a random vector.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomVectorGenerator.nextVector` in
                interface :class:`~org.hipparchus.random.RandomVectorGenerator`
        
            Returns:
                a random vector as an array of double.
        
        
        """
        ...
    def skipTo(self, int: int) -> typing.List[float]: ...

class StableRandomGenerator(NormalizedRandomGenerator):
    """
    public class StableRandomGenerator extends :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.Object?is` implements :class:`~org.hipparchus.random.NormalizedRandomGenerator`
    
    
        This class provides a stable normalized random generator. It samples from a stable distribution with location parameter
        0 and scale 1.
    
        The implementation uses the Chambers-Mallows-Stuck method as described in *Handbook of computational statistics:
        concepts and methods* by James E. Gentle, Wolfgang Härdle, Yuichi Mori.
    """
    def __init__(self, randomGenerator: RandomGenerator, double: float, double2: float): ...
    def nextNormalizedDouble(self) -> float:
        """
            Generate a random scalar with zero location and unit scale.
        
            Specified by:
                :meth:`~org.hipparchus.random.NormalizedRandomGenerator.nextNormalizedDouble` in
                interface :class:`~org.hipparchus.random.NormalizedRandomGenerator`
        
            Returns:
                a random scalar with zero location and unit scale
        
        
        """
        ...

class SynchronizedRandomGenerator(RandomGenerator):
    """
    public class SynchronizedRandomGenerator extends :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.Object?is` implements :class:`~org.hipparchus.random.RandomGenerator`
    
        Any :class:`~org.hipparchus.random.RandomGenerator` implementation can be thread-safe if it is used through an instance
        of this class. This is achieved by enclosing calls to the methods of the actual generator inside the overridden
        :code:`synchronized` methods of this class.
    """
    def __init__(self, randomGenerator: RandomGenerator): ...
    def nextBoolean(self) -> bool:
        """
            Returns the next pseudorandom, uniformly distributed :code:`boolean` value from this random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextBoolean` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`boolean` value from this random number generator's sequence
        
        
        """
        ...
    @typing.overload
    def nextBytes(self, byteArray: typing.List[int]) -> None:
        """
            Generates random bytes and places them into a user-supplied byte array. The number of random bytes produced is equal to
            the length of the byte array.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextBytes` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                bytes (byte[]): the non-null byte array in which to put the random bytes
        
        """
        ...
    @typing.overload
    def nextBytes(self, byteArray: typing.List[int], int: int, int2: int) -> None:
        """
            Generates random bytes and places them into a user-supplied byte array.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextBytes` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                bytes (byte[]): the non-null byte array in which to put the random bytes
                offset (int): the starting index for inserting the generated bytes into the array
                len (int): the number of bytes to generate
        
        
        """
        ...
    def nextDouble(self) -> float:
        """
            Returns the next pseudorandom, uniformly distributed :code:`double` value between :code:`0.0` and :code:`1.0` from this
            random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextDouble` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`double` value between :code:`0.0` and :code:`1.0` from this random
                number generator's sequence
        
        
        """
        ...
    def nextFloat(self) -> float:
        """
            Returns the next pseudorandom, uniformly distributed :code:`float` value between :code:`0.0` and :code:`1.0` from this
            random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextFloat` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`float` value between :code:`0.0` and :code:`1.0` from this random
                number generator's sequence
        
        
        """
        ...
    def nextGaussian(self) -> float:
        """
            Returns the next pseudorandom, Gaussian ("normally") distributed :code:`double` value with mean :code:`0.0` and standard
            deviation :code:`1.0` from this random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextGaussian` in
                interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Returns:
                the next pseudorandom, Gaussian ("normally") distributed :code:`double` value with mean :code:`0.0` and standard
                deviation :code:`1.0` from this random number generator's sequence
        
        
        """
        ...
    @typing.overload
    def nextInt(self) -> int:
        """
            Returns the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence.
        
            All 2 :sup:`32` possible :code:`int` values should be produced with (approximately) equal probability.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextInt` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence
        
        """
        ...
    @typing.overload
    def nextInt(self, int: int) -> int:
        """
            Returns a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and the specified value
            (exclusive), drawn from this random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextInt` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                n (int): the bound on the random number to be returned. Must be positive.
        
            Returns:
                a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and n (exclusive).
        
        
        """
        ...
    @typing.overload
    def nextLong(self) -> int:
        """
            Returns the next pseudorandom, uniformly distributed :code:`long` value from this random number generator's sequence.
            All 2 :sup:`64` possible :code:`long` values should be produced with (approximately) equal probability.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextLong` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`long` value from this random number generator's sequence
        
        """
        ...
    @typing.overload
    def nextLong(self, long: int) -> int:
        """
            Returns a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and the specified value
            (exclusive), drawn from this random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextLong` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                n (long): the bound on the random number to be returned. Must be positive.
        
            Returns:
                a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and n (exclusive).
        
        
        """
        ...
    @typing.overload
    def setSeed(self, int: int) -> None:
        """
            Sets the seed of the underlying random number generator using an :code:`int` seed.
        
            Sequences of values generated starting with the same seeds should be identical.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.setSeed` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                seed (int): the seed value
        
            Sets the seed of the underlying random number generator using an :code:`int` array seed.
        
            Sequences of values generated starting with the same seeds should be identical.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.setSeed` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                seed (int[]): the seed value
        
            Sets the seed of the underlying random number generator using a :code:`long` seed.
        
            Sequences of values generated starting with the same seeds should be identical.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.setSeed` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                seed (long): the seed value
        
        
        """
        ...
    @typing.overload
    def setSeed(self, intArray: typing.List[int]) -> None: ...
    @typing.overload
    def setSeed(self, long: int) -> None: ...

class UncorrelatedRandomVectorGenerator(RandomVectorGenerator):
    """
    public class UncorrelatedRandomVectorGenerator extends :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.Object?is` implements :class:`~org.hipparchus.random.RandomVectorGenerator`
    
        A :class:`~org.hipparchus.random.RandomVectorGenerator` that generates vectors with uncorrelated components.
    
        Components of generated vectors follow (independent) Gaussian distributions, with parameters supplied in the
        constructor.
    """
    @typing.overload
    def __init__(self, doubleArray: typing.List[float], doubleArray2: typing.List[float], normalizedRandomGenerator: NormalizedRandomGenerator): ...
    @typing.overload
    def __init__(self, int: int, normalizedRandomGenerator: NormalizedRandomGenerator): ...
    def nextVector(self) -> typing.List[float]:
        """
            Generate an uncorrelated random vector.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomVectorGenerator.nextVector` in
                interface :class:`~org.hipparchus.random.RandomVectorGenerator`
        
            Returns:
                a random vector as a newly built array of double
        
        
        """
        ...

class UniformRandomGenerator(NormalizedRandomGenerator):
    def __init__(self, randomGenerator: RandomGenerator): ...
    def nextNormalizedDouble(self) -> float: ...

class UnitSphereRandomVectorGenerator(RandomVectorGenerator):
    """
    public class UnitSphereRandomVectorGenerator extends :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.Object?is` implements :class:`~org.hipparchus.random.RandomVectorGenerator`
    
        Generate random vectors isotropically located on the surface of a sphere.
    """
    @typing.overload
    def __init__(self, int: int): ...
    @typing.overload
    def __init__(self, int: int, randomGenerator: RandomGenerator): ...
    def nextVector(self) -> typing.List[float]:
        """
            Generate a random vector.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomVectorGenerator.nextVector` in
                interface :class:`~org.hipparchus.random.RandomVectorGenerator`
        
            Returns:
                a random vector as an array of double.
        
        
        """
        ...

class AbstractWell(org.hipparchus.random.IntRandomGenerator, java.io.Serializable):
    """
    public abstract class AbstractWell extends :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.Object?is` implements :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.io.Serializable?is`
    
        This abstract class implements the WELL class of pseudo-random number generator from François Panneton, Pierre L'Ecuyer
        and Makoto Matsumoto.
    
        This generator is described in a paper by François Panneton, Pierre L'Ecuyer and Makoto Matsumoto ` Improved
        Long-Period Generators Based on Linear Recurrences Modulo 2
        <http://www.iro.umontreal.ca/~lecuyer/myftp/papers/wellrng.pdf>` ACM Transactions on Mathematical Software, 32, 1
        (2006). The errata for the paper are in ` wellrng-errata.txt
        <http://www.iro.umontreal.ca/~lecuyer/myftp/papers/wellrng-errata.txt>`.
    
        Also see:
            `WELL Random number generator <http://www.iro.umontreal.ca/~panneton/WELLRNG.html>`, :meth:`~serialized`
    """
    def nextInt(self) -> int:
        """
            Returns the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence.
        
            All 2 :sup:`32` possible :code:`int` values should be produced with (approximately) equal probability.
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence
        
        public int nextInt(int n) throws :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.IllegalArgumentException?is`
        
            Returns a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and the specified value
            (exclusive), drawn from this random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextInt` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                n (int): the bound on the random number to be returned. Must be positive.
        
            Returns:
                a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and n (exclusive).
        
            Raises:
                :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.IllegalArgumentException?is`: if n is not positive.
        
        
        """
        ...
    def setSeed(self, intArray: typing.List[int]) -> None:
        """
            Reinitialize the generator as if just built with the given int array seed.
        
            The state of the generator is exactly the same as a new generator built with the same seed.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.setSeed` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                seed (int[]): the initial seed (32 bits integers array). If null the seed of the generator will be the system time plus the system
                    identity hash code of the instance.
        
            Sets the seed of the underlying random number generator using an :code:`int` seed.
        
            Sequences of values generated starting with the same seeds should be identical.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.setSeed` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                seed (int): the seed value
        
            Sets the seed of the underlying random number generator using a :code:`long` seed.
        
            Sequences of values generated starting with the same seeds should be identical.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.setSeed` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                seed (long): the seed value
        
        
        """
        ...

class ISAACRandom(org.hipparchus.random.IntRandomGenerator, java.io.Serializable):
    @typing.overload
    def __init__(self): ...
    @typing.overload
    def __init__(self, intArray: typing.List[int]): ...
    @typing.overload
    def __init__(self, long: int): ...
    def nextInt(self) -> int: ...
    def setSeed(self, intArray: typing.List[int]) -> None: ...

class JDKRandomGenerator(org.hipparchus.random.IntRandomGenerator, java.io.Serializable):
    """
    public class JDKRandomGenerator extends :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.Object?is` implements :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.io.Serializable?is`
    
        A :class:`~org.hipparchus.random.RandomGenerator` adapter that delegates the random number generation to the standard
        :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.util.Random?is` class.
    
        Also see:
            :meth:`~serialized`
    """
    @typing.overload
    def __init__(self): ...
    @typing.overload
    def __init__(self, random: java.util.Random): ...
    @typing.overload
    def __init__(self, long: int): ...
    def nextBoolean(self) -> bool:
        """
            Returns the next pseudorandom, uniformly distributed :code:`boolean` value from this random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextBoolean` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`boolean` value from this random number generator's sequence
        
        
        """
        ...
    def nextBytes(self, byteArray: typing.List[int]) -> None:
        """
            Generates random bytes and places them into a user-supplied byte array. The number of random bytes produced is equal to
            the length of the byte array.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextBytes` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                bytes (byte[]): the non-null byte array in which to put the random bytes
        
        public void nextBytes(byte[] bytes, int start, int len)
        
            Generates random bytes and places them into a user-supplied byte array.
        
            Parameters:
                bytes (byte[]): the non-null byte array in which to put the random bytes
                start (int): the starting index for inserting the generated bytes into the array
                len (int): the number of bytes to generate
        
        
        """
        ...
    def nextDouble(self) -> float:
        """
            Returns the next pseudorandom, uniformly distributed :code:`double` value between :code:`0.0` and :code:`1.0` from this
            random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextDouble` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`double` value between :code:`0.0` and :code:`1.0` from this random
                number generator's sequence
        
        
        """
        ...
    def nextFloat(self) -> float:
        """
            Returns the next pseudorandom, uniformly distributed :code:`float` value between :code:`0.0` and :code:`1.0` from this
            random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextFloat` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`float` value between :code:`0.0` and :code:`1.0` from this random
                number generator's sequence
        
        
        """
        ...
    def nextGaussian(self) -> float:
        """
            Returns the next pseudorandom, Gaussian ("normally") distributed :code:`double` value with mean :code:`0.0` and standard
            deviation :code:`1.0` from this random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextGaussian` in
                interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Returns:
                the next pseudorandom, Gaussian ("normally") distributed :code:`double` value with mean :code:`0.0` and standard
                deviation :code:`1.0` from this random number generator's sequence
        
        
        """
        ...
    @typing.overload
    def nextInt(self) -> int:
        """
            Returns the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence.
        
            All 2 :sup:`32` possible :code:`int` values should be produced with (approximately) equal probability.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextInt` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence
        
        """
        ...
    @typing.overload
    def nextInt(self, int: int) -> int:
        """
            Returns a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and the specified value
            (exclusive), drawn from this random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextInt` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                n (int): the bound on the random number to be returned. Must be positive.
        
            Returns:
                a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and n (exclusive).
        
        
        """
        ...
    def nextLong(self) -> int:
        """
            Returns the next pseudorandom, uniformly distributed :code:`long` value from this random number generator's sequence.
            All 2 :sup:`64` possible :code:`long` values should be produced with (approximately) equal probability.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextLong` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`long` value from this random number generator's sequence
        
        public long nextLong(long n)
        
            Returns a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and the specified value
            (exclusive), drawn from this random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextLong` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                n (long): the bound on the random number to be returned. Must be positive.
        
            Returns:
                a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and n (exclusive).
        
        
        """
        ...
    @typing.overload
    def setSeed(self, int: int) -> None:
        """
            Sets the seed of the underlying random number generator using an :code:`int` seed.
        
            Sequences of values generated starting with the same seeds should be identical.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.setSeed` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                seed (int): the seed value
        
            Sets the seed of the underlying random number generator using a :code:`long` seed.
        
            Sequences of values generated starting with the same seeds should be identical.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.setSeed` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                seed (long): the seed value
        
            Sets the seed of the underlying random number generator using an :code:`int` array seed.
        
            Sequences of values generated starting with the same seeds should be identical.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.setSeed` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                seed (int[]): the seed value
        
        
        """
        ...
    @typing.overload
    def setSeed(self, intArray: typing.List[int]) -> None: ...
    @typing.overload
    def setSeed(self, long: int) -> None: ...

class MersenneTwister(org.hipparchus.random.IntRandomGenerator, java.io.Serializable):
    """
    public class MersenneTwister extends :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.Object?is` implements :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.io.Serializable?is`
    
        This class implements a powerful pseudo-random number generator developed by Makoto Matsumoto and Takuji Nishimura
        during 1996-1997.
    
        **Caveat:** It is recommended to use one of WELL generators rather than the MersenneTwister generator (see ` this paper
        <http://www.iro.umontreal.ca/~panneton/WELLRNG.html>` for more information).
    
        This generator features an extremely long period (2 :sup:`19937` -1) and 623-dimensional equidistribution up to 32 bits
        accuracy. The home page for this generator is located at ` http://www.math.sci.hiroshima-u.ac.jp/~m-mat/MT/emt.html
        <http://www.math.sci.hiroshima-u.ac.jp/~m-mat/MT/emt.html>`.
    
        This generator is described in a paper by Makoto Matsumoto and Takuji Nishimura in 1998: `Mersenne Twister: A
        623-Dimensionally Equidistributed Uniform Pseudo-Random Number Generator
        <http://www.math.sci.hiroshima-u.ac.jp/~m-mat/MT/ARTICLES/mt.pdf>`, ACM Transactions on Modeling and Computer
        Simulation, Vol. 8, No. 1, January 1998, pp 3--30.
    
        This class is mainly a Java port of the 2002-01-26 version of the generator written in C by Makoto Matsumoto and Takuji
        Nishimura. Here is their original copyright:
    
    
        Also see:
            :meth:`~serialized`
    """
    @typing.overload
    def __init__(self): ...
    @typing.overload
    def __init__(self, int: int): ...
    @typing.overload
    def __init__(self, intArray: typing.List[int]): ...
    @typing.overload
    def __init__(self, long: int): ...
    def nextInt(self) -> int:
        """
            Returns the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence.
        
            All 2 :sup:`32` possible :code:`int` values should be produced with (approximately) equal probability.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextInt` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence
        
        public int nextInt(int n) throws :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.IllegalArgumentException?is`
        
            Returns a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and the specified value
            (exclusive), drawn from this random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextInt` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                n (int): the bound on the random number to be returned. Must be positive.
        
            Returns:
                a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and n (exclusive).
        
            Raises:
                :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.IllegalArgumentException?is`: if n is not positive.
        
        
        """
        ...
    @typing.overload
    def setSeed(self, int: int) -> None:
        """
            Reinitialize the generator as if just built with the given int seed.
        
            The state of the generator is exactly the same as a new generator built with the same seed.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.setSeed` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                seed (int): the initial seed (32 bits integer)
        
            Reinitialize the generator as if just built with the given int array seed.
        
            The state of the generator is exactly the same as a new generator built with the same seed.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.setSeed` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                seed (int[]): the initial seed (32 bits integers array), if null the seed of the generator will be the current system time plus the
                    system identity hash code of this instance
        
            Sets the seed of the underlying random number generator using a :code:`long` seed.
        
            Sequences of values generated starting with the same seeds should be identical.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.setSeed` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                seed (long): the seed value
        
        
        """
        ...
    @typing.overload
    def setSeed(self, intArray: typing.List[int]) -> None: ...

class RandomDataGenerator(org.hipparchus.random.ForwardingRandomGenerator, RandomGenerator, java.io.Serializable):
    """
    public class RandomDataGenerator extends :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.Object?is` implements :class:`~org.hipparchus.random.RandomGenerator`, :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.io.Serializable?is`
    
        A class for generating random data.
    
        Also see:
            :meth:`~serialized`
    """
    @typing.overload
    def __init__(self): ...
    @typing.overload
    def __init__(self, long: int): ...
    def nextBeta(self, double: float, double2: float) -> float:
        """
            Returns the next pseudo-random beta-distributed value with the given shape and scale parameters.
        
            Parameters:
                alpha (double): First shape parameter (must be positive).
                beta (double): Second shape parameter (must be positive).
        
            Returns:
                beta-distributed random deviate
        
        
        """
        ...
    @typing.overload
    def nextDeviate(self, realDistribution: org.hipparchus.distribution.RealDistribution) -> float:
        """
            Returns a random deviate from the given distribution.
        
            Parameters:
                dist (:class:`~org.hipparchus.distribution.RealDistribution`): the distribution to sample from
        
            Returns:
                a random value following the given distribution
        
            Returns a random deviate from the given distribution.
        
            Parameters:
                dist (:class:`~org.hipparchus.distribution.IntegerDistribution`): the distribution to sample from
        
            Returns:
                a random value following the given distribution
        
        
        """
        ...
    @typing.overload
    def nextDeviate(self, integerDistribution: org.hipparchus.distribution.IntegerDistribution) -> int: ...
    @typing.overload
    def nextDeviates(self, realDistribution: org.hipparchus.distribution.RealDistribution, int: int) -> typing.List[float]:
        """
            Returns an array of random deviates from the given distribution.
        
            Parameters:
                dist (:class:`~org.hipparchus.distribution.RealDistribution`): the distribution to sample from
                size (int): the number of values to return
        
            Returns:
                an array of :code:`size` values following the given distribution
        
            Returns an array of random deviates from the given distribution.
        
            Parameters:
                dist (:class:`~org.hipparchus.distribution.IntegerDistribution`): the distribution to sample from
                size (int): the number of values to return
        
            Returns:
                an array of :code:`size`values following the given distribution
        
        
        """
        ...
    @typing.overload
    def nextDeviates(self, integerDistribution: org.hipparchus.distribution.IntegerDistribution, int2: int) -> typing.List[int]: ...
    def nextExponential(self, double: float) -> float:
        """
            Returns the next pseudo-random, exponentially distributed deviate.
        
            Parameters:
                mean (double): mean of the exponential distribution
        
            Returns:
                exponentially distributed deviate about the given mean
        
        
        """
        ...
    def nextGamma(self, double: float, double2: float) -> float:
        """
            Returns the next pseudo-random gamma-distributed value with the given shape and scale parameters.
        
            Parameters:
                shape (double): shape parameter of the distribution
                scale (double): scale parameter of the distribution
        
            Returns:
                gamma-distributed random deviate
        
        
        """
        ...
    def nextHexString(self, int: int) -> str: ...
    def nextInt(self, int: int, int2: int) -> int:
        """
            Returns a uniformly distributed random integer between lower and upper (inclusive).
        
            Parameters:
                lower (int): lower bound for the generated value
                upper (int): upper bound for the generated value
        
            Returns:
                a random integer value within the given bounds
        
            Raises:
                :class:`~org.hipparchus.exception.MathIllegalArgumentException`: if lower is not strictly less than or equal to upper
        
        public int nextInt()
        
            Returns the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence.
        
            All 2 :sup:`32` possible :code:`int` values should be produced with (approximately) equal probability.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextInt` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence
        
        public int nextInt(int n)
        
            Returns a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and the specified value
            (exclusive), drawn from this random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextInt` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                n (int): the bound on the random number to be returned. Must be positive.
        
            Returns:
                a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and n (exclusive).
        
        
        """
        ...
    def nextLogNormal(self, double: float, double2: float) -> float:
        """
            Returns the next log-normally-distributed pseudo-random deviate.
        
            Parameters:
                shape (double): shape parameter of the log-normal distribution
                scale (double): scale parameter of the log-normal distribution
        
            Returns:
                a random value, normally distributed with the given mean and standard deviation
        
        
        """
        ...
    def nextLong(self, long: int, long2: int) -> int: ...
    def nextNormal(self, double: float, double2: float) -> float:
        """
            Returns the next normally-distributed pseudo-random deviate.
        
            Parameters:
                mean (double): mean of the normal distribution
                standardDeviation (double): standard deviation of the normal distribution
        
            Returns:
                a random value, normally distributed with the given mean and standard deviation
        
        
        """
        ...
    def nextPermutation(self, int: int, int2: int) -> typing.List[int]: ...
    def nextPoisson(self, double: float) -> int:
        """
            Returns a poisson-distributed deviate with the given mean.
        
            Parameters:
                mean (double): expected value
        
            Returns:
                poisson deviate
        
            Raises:
                :class:`~org.hipparchus.exception.MathIllegalArgumentException`: if mean is not strictly positive
        
        
        """
        ...
    @typing.overload
    def nextSample(self, doubleArray: typing.List[float], int: int) -> typing.List[float]: ...
    @typing.overload
    def nextSample(self, collection: typing.Union[java.util.Collection[typing.Any], typing.Sequence[typing.Any], typing.Set[typing.Any]], int: int) -> typing.List[typing.Any]: ...
    def nextSampleWithReplacement(self, int: int, doubleArray: typing.List[float]) -> typing.List[int]:
        """
            Generates a random sample of size sampleSize from {0, 1, ... , weights.length - 1}, using weights as probabilities.
        
            For 0 < i < weights.length, the probability that i is selected (on any draw) is weights[i]. If necessary, the weights
            array is normalized to sum to 1 so that weights[i] is a probability and the array sums to 1.
        
            Weights can be 0, but must not be negative, infinite or NaN. At least one weight must be positive.
        
            Parameters:
                sampleSize (int): size of sample to generate
                weights (double[]): probability sampling weights
        
            Returns:
                an array of integers between 0 and weights.length - 1
        
            Raises:
                :class:`~org.hipparchus.exception.MathIllegalArgumentException`: if weights contains negative, NaN or infinite values or only 0s or sampleSize is less than 0
        
        
        """
        ...
    def nextUniform(self, double: float, double2: float) -> float:
        """
            Returns a double value uniformly distributed over [lower, upper]
        
            Parameters:
                lower (double): lower bound
                upper (double): upper bound
        
            Returns:
                uniform deviate
        
            Raises:
                :class:`~org.hipparchus.exception.MathIllegalArgumentException`: if upper is less than or equal to upper
        
        
        """
        ...
    def nextZipf(self, int: int, double: float) -> int:
        """
            Returns an integer value following a Zipf distribution with the given parameter.
        
            Parameters:
                numberOfElements (int): number of elements of the distribution
                exponent (double): exponent of the distribution
        
            Returns:
                random Zipf value
        
        
        """
        ...
    @staticmethod
    def of(randomGenerator: RandomGenerator) -> 'RandomDataGenerator':
        """
            Factory method to create a :code:`RandomData` instance using the supplied :code:`RandomGenerator`.
        
            Parameters:
                randomGenerator (:class:`~org.hipparchus.random.RandomGenerator`): source of random bits
        
            Returns:
                a RandomData using the given RandomGenerator to source bits
        
            Raises:
                :class:`~org.hipparchus.exception.MathIllegalArgumentException`: if randomGenerator is null
        
        
        """
        ...

class Well1024a(AbstractWell):
    """
    public class Well1024a extends :class:`~org.hipparchus.random.AbstractWell`
    
        This class implements the WELL1024a pseudo-random number generator from François Panneton, Pierre L'Ecuyer and Makoto
        Matsumoto.
    
        This generator is described in a paper by François Panneton, Pierre L'Ecuyer and Makoto Matsumoto `Improved Long-Period
        Generators Based on Linear Recurrences Modulo 2 <http://www.iro.umontreal.ca/~lecuyer/myftp/papers/wellrng.pdf>` ACM
        Transactions on Mathematical Software, 32, 1 (2006). The errata for the paper are in ` wellrng-errata.txt
        <http://www.iro.umontreal.ca/~lecuyer/myftp/papers/wellrng-errata.txt>`.
    
        Also see:
            `WELL Random number generator <http://www.iro.umontreal.ca/~panneton/WELLRNG.html>`, :meth:`~serialized`
    """
    @typing.overload
    def __init__(self): ...
    @typing.overload
    def __init__(self, int: int): ...
    @typing.overload
    def __init__(self, intArray: typing.List[int]): ...
    @typing.overload
    def __init__(self, long: int): ...
    def nextInt(self) -> int:
        """
            Returns the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence.
        
            All 2 :sup:`32` possible :code:`int` values should be produced with (approximately) equal probability.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextInt` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence
        
        public int nextInt(int n) throws :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.IllegalArgumentException?is`
        
            Returns a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and the specified value
            (exclusive), drawn from this random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextInt` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                n (int): the bound on the random number to be returned. Must be positive.
        
            Returns:
                a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and n (exclusive).
        
            Raises:
                :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.IllegalArgumentException?is`: if n is not positive.
        
        
        """
        ...

class Well19937a(AbstractWell):
    """
    public class Well19937a extends :class:`~org.hipparchus.random.AbstractWell`
    
        This class implements the WELL19937a pseudo-random number generator from François Panneton, Pierre L'Ecuyer and Makoto
        Matsumoto.
    
        This generator is described in a paper by François Panneton, Pierre L'Ecuyer and Makoto Matsumoto `Improved Long-Period
        Generators Based on Linear Recurrences Modulo 2 <http://www.iro.umontreal.ca/~lecuyer/myftp/papers/wellrng.pdf>` ACM
        Transactions on Mathematical Software, 32, 1 (2006). The errata for the paper are in ` wellrng-errata.txt
        <http://www.iro.umontreal.ca/~lecuyer/myftp/papers/wellrng-errata.txt>`.
    
        Also see:
            `WELL Random number generator <http://www.iro.umontreal.ca/~panneton/WELLRNG.html>`, :meth:`~serialized`
    """
    @typing.overload
    def __init__(self): ...
    @typing.overload
    def __init__(self, int: int): ...
    @typing.overload
    def __init__(self, intArray: typing.List[int]): ...
    @typing.overload
    def __init__(self, long: int): ...
    def nextInt(self) -> int:
        """
            Returns the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence.
        
            All 2 :sup:`32` possible :code:`int` values should be produced with (approximately) equal probability.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextInt` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence
        
        public int nextInt(int n) throws :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.IllegalArgumentException?is`
        
            Returns a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and the specified value
            (exclusive), drawn from this random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextInt` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                n (int): the bound on the random number to be returned. Must be positive.
        
            Returns:
                a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and n (exclusive).
        
            Raises:
                :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.IllegalArgumentException?is`: if n is not positive.
        
        
        """
        ...

class Well19937c(AbstractWell):
    """
    public class Well19937c extends :class:`~org.hipparchus.random.AbstractWell`
    
        This class implements the WELL19937c pseudo-random number generator from François Panneton, Pierre L'Ecuyer and Makoto
        Matsumoto.
    
        This generator is described in a paper by François Panneton, Pierre L'Ecuyer and Makoto Matsumoto `Improved Long-Period
        Generators Based on Linear Recurrences Modulo 2 <http://www.iro.umontreal.ca/~lecuyer/myftp/papers/wellrng.pdf>` ACM
        Transactions on Mathematical Software, 32, 1 (2006). The errata for the paper are in ` wellrng-errata.txt
        <http://www.iro.umontreal.ca/~lecuyer/myftp/papers/wellrng-errata.txt>`.
    
        Also see:
            `WELL Random number generator <http://www.iro.umontreal.ca/~panneton/WELLRNG.html>`, :meth:`~serialized`
    """
    @typing.overload
    def __init__(self): ...
    @typing.overload
    def __init__(self, int: int): ...
    @typing.overload
    def __init__(self, intArray: typing.List[int]): ...
    @typing.overload
    def __init__(self, long: int): ...
    def nextInt(self) -> int:
        """
            Returns the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence.
        
            All 2 :sup:`32` possible :code:`int` values should be produced with (approximately) equal probability.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextInt` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence
        
        public int nextInt(int n) throws :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.IllegalArgumentException?is`
        
            Returns a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and the specified value
            (exclusive), drawn from this random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextInt` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                n (int): the bound on the random number to be returned. Must be positive.
        
            Returns:
                a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and n (exclusive).
        
            Raises:
                :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.IllegalArgumentException?is`: if n is not positive.
        
        
        """
        ...

class Well44497a(AbstractWell):
    """
    public class Well44497a extends :class:`~org.hipparchus.random.AbstractWell`
    
        This class implements the WELL44497a pseudo-random number generator from François Panneton, Pierre L'Ecuyer and Makoto
        Matsumoto.
    
        This generator is described in a paper by François Panneton, Pierre L'Ecuyer and Makoto Matsumoto `Improved Long-Period
        Generators Based on Linear Recurrences Modulo 2 <http://www.iro.umontreal.ca/~lecuyer/myftp/papers/wellrng.pdf>` ACM
        Transactions on Mathematical Software, 32, 1 (2006). The errata for the paper are in ` wellrng-errata.txt
        <http://www.iro.umontreal.ca/~lecuyer/myftp/papers/wellrng-errata.txt>`.
    
        Also see:
            `WELL Random number generator <http://www.iro.umontreal.ca/~panneton/WELLRNG.html>`, :meth:`~serialized`
    """
    @typing.overload
    def __init__(self): ...
    @typing.overload
    def __init__(self, int: int): ...
    @typing.overload
    def __init__(self, intArray: typing.List[int]): ...
    @typing.overload
    def __init__(self, long: int): ...
    def nextInt(self) -> int:
        """
            Returns the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence.
        
            All 2 :sup:`32` possible :code:`int` values should be produced with (approximately) equal probability.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextInt` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence
        
        public int nextInt(int n) throws :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.IllegalArgumentException?is`
        
            Returns a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and the specified value
            (exclusive), drawn from this random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextInt` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                n (int): the bound on the random number to be returned. Must be positive.
        
            Returns:
                a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and n (exclusive).
        
            Raises:
                :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.IllegalArgumentException?is`: if n is not positive.
        
        
        """
        ...

class Well44497b(AbstractWell):
    """
    public class Well44497b extends :class:`~org.hipparchus.random.AbstractWell`
    
        This class implements the WELL44497b pseudo-random number generator from François Panneton, Pierre L'Ecuyer and Makoto
        Matsumoto.
    
        This generator is described in a paper by François Panneton, Pierre L'Ecuyer and Makoto Matsumoto `Improved Long-Period
        Generators Based on Linear Recurrences Modulo 2 <http://www.iro.umontreal.ca/~lecuyer/myftp/papers/wellrng.pdf>` ACM
        Transactions on Mathematical Software, 32, 1 (2006). The errata for the paper are in ` wellrng-errata.txt
        <http://www.iro.umontreal.ca/~lecuyer/myftp/papers/wellrng-errata.txt>`.
    
        Also see:
            `WELL Random number generator <http://www.iro.umontreal.ca/~panneton/WELLRNG.html>`, :meth:`~serialized`
    """
    @typing.overload
    def __init__(self): ...
    @typing.overload
    def __init__(self, int: int): ...
    @typing.overload
    def __init__(self, intArray: typing.List[int]): ...
    @typing.overload
    def __init__(self, long: int): ...
    def nextInt(self) -> int:
        """
            Returns the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence.
        
            All 2 :sup:`32` possible :code:`int` values should be produced with (approximately) equal probability.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextInt` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence
        
        public int nextInt(int n) throws :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.IllegalArgumentException?is`
        
            Returns a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and the specified value
            (exclusive), drawn from this random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextInt` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                n (int): the bound on the random number to be returned. Must be positive.
        
            Returns:
                a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and n (exclusive).
        
            Raises:
                :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.IllegalArgumentException?is`: if n is not positive.
        
        
        """
        ...

class Well512a(AbstractWell):
    """
    public class Well512a extends :class:`~org.hipparchus.random.AbstractWell`
    
        This class implements the WELL512a pseudo-random number generator from François Panneton, Pierre L'Ecuyer and Makoto
        Matsumoto.
    
        This generator is described in a paper by François Panneton, Pierre L'Ecuyer and Makoto Matsumoto `Improved Long-Period
        Generators Based on Linear Recurrences Modulo 2 <http://www.iro.umontreal.ca/~lecuyer/myftp/papers/wellrng.pdf>` ACM
        Transactions on Mathematical Software, 32, 1 (2006). The errata for the paper are in ` wellrng-errata.txt
        <http://www.iro.umontreal.ca/~lecuyer/myftp/papers/wellrng-errata.txt>`.
    
        Also see:
            `WELL Random number generator <http://www.iro.umontreal.ca/~panneton/WELLRNG.html>`, :meth:`~serialized`
    """
    @typing.overload
    def __init__(self): ...
    @typing.overload
    def __init__(self, int: int): ...
    @typing.overload
    def __init__(self, intArray: typing.List[int]): ...
    @typing.overload
    def __init__(self, long: int): ...
    def nextInt(self) -> int:
        """
            Returns the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence.
        
            All 2 :sup:`32` possible :code:`int` values should be produced with (approximately) equal probability.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextInt` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Returns:
                the next pseudorandom, uniformly distributed :code:`int` value from this random number generator's sequence
        
        public int nextInt(int n) throws :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.IllegalArgumentException?is`
        
            Returns a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and the specified value
            (exclusive), drawn from this random number generator's sequence.
        
            Specified by:
                :meth:`~org.hipparchus.random.RandomGenerator.nextInt` in interface :class:`~org.hipparchus.random.RandomGenerator`
        
            Parameters:
                n (int): the bound on the random number to be returned. Must be positive.
        
            Returns:
                a pseudorandom, uniformly distributed :code:`int` value between 0 (inclusive) and n (exclusive).
        
            Raises:
                :class:`~org.hipparchus.random.https:.docs.oracle.com.javase.8.docs.api.java.lang.IllegalArgumentException?is`: if n is not positive.
        
        
        """
        ...

class ForwardingRandomGenerator: ...

class IntRandomGenerator: ...


class __module_protocol__(typing.Protocol):
    # A module protocol which reflects the result of ``jp.JPackage("org.hipparchus.random")``.

    AbstractWell: typing.Type[AbstractWell]
    CorrelatedRandomVectorGenerator: typing.Type[CorrelatedRandomVectorGenerator]
    ForwardingRandomGenerator: typing.Type[ForwardingRandomGenerator]
    GaussianRandomGenerator: typing.Type[GaussianRandomGenerator]
    HaltonSequenceGenerator: typing.Type[HaltonSequenceGenerator]
    ISAACRandom: typing.Type[ISAACRandom]
    IntRandomGenerator: typing.Type[IntRandomGenerator]
    JDKRandomGenerator: typing.Type[JDKRandomGenerator]
    MersenneTwister: typing.Type[MersenneTwister]
    NormalizedRandomGenerator: typing.Type[NormalizedRandomGenerator]
    RandomAdaptor: typing.Type[RandomAdaptor]
    RandomDataGenerator: typing.Type[RandomDataGenerator]
    RandomGenerator: typing.Type[RandomGenerator]
    RandomVectorGenerator: typing.Type[RandomVectorGenerator]
    SobolSequenceGenerator: typing.Type[SobolSequenceGenerator]
    StableRandomGenerator: typing.Type[StableRandomGenerator]
    SynchronizedRandomGenerator: typing.Type[SynchronizedRandomGenerator]
    UncorrelatedRandomVectorGenerator: typing.Type[UncorrelatedRandomVectorGenerator]
    UniformRandomGenerator: typing.Type[UniformRandomGenerator]
    UnitSphereRandomVectorGenerator: typing.Type[UnitSphereRandomVectorGenerator]
    Well1024a: typing.Type[Well1024a]
    Well19937a: typing.Type[Well19937a]
    Well19937c: typing.Type[Well19937c]
    Well44497a: typing.Type[Well44497a]
    Well44497b: typing.Type[Well44497b]
    Well512a: typing.Type[Well512a]
