/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.models.earth.ionosphere;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.List;
import java.util.regex.Pattern;
import org.hipparchus.Field;
import org.hipparchus.RealFieldElement;
import org.hipparchus.exception.Localizable;
import org.hipparchus.util.FastMath;
import org.hipparchus.util.FieldSinCos;
import org.hipparchus.util.MathArrays;
import org.hipparchus.util.SinCos;
import org.orekit.annotation.DefaultDataContext;
import org.orekit.bodies.BodyShape;
import org.orekit.bodies.FieldGeodeticPoint;
import org.orekit.bodies.GeodeticPoint;
import org.orekit.data.DataContext;
import org.orekit.errors.OrekitException;
import org.orekit.errors.OrekitMessages;
import org.orekit.frames.TopocentricFrame;
import org.orekit.models.earth.ionosphere.FieldNeQuickParameters;
import org.orekit.models.earth.ionosphere.IonosphericModel;
import org.orekit.models.earth.ionosphere.NeQuickParameters;
import org.orekit.propagation.FieldSpacecraftState;
import org.orekit.propagation.SpacecraftState;
import org.orekit.time.AbsoluteDate;
import org.orekit.time.DateComponents;
import org.orekit.time.DateTimeComponents;
import org.orekit.time.FieldAbsoluteDate;
import org.orekit.time.TimeScale;
import org.orekit.utils.ParameterDriver;
import org.orekit.utils.TimeStampedFieldPVCoordinates;
import org.orekit.utils.TimeStampedPVCoordinates;

public class NeQuickModel
implements IonosphericModel {
    private static final String NEQUICK_BASE = "/assets/org/orekit/nequick/";
    private static final long serialVersionUID = 201928051L;
    private static final Pattern SEPARATOR = Pattern.compile("\\s+");
    private static final double RE = 6371200.0;
    private static final double M_TO_KM = 0.001;
    private static final double DENSITY_FACTOR = 1.0E11;
    private static final double DELAY_FACTOR = 4.03E17;
    private final double[] alpha;
    private final double[][] stModip;
    private int month = 0;
    private double[][][] f2 = null;
    private double[][][] fm3 = null;
    private final TimeScale utc;

    @DefaultDataContext
    public NeQuickModel(double[] alpha) {
        this(alpha, DataContext.getDefault().getTimeScales().getUTC());
    }

    public NeQuickModel(double[] alpha, TimeScale utc) {
        MODIPLoader parser = new MODIPLoader();
        parser.loadMODIPGrid();
        this.stModip = parser.getMODIPGrid();
        this.alpha = (double[])alpha.clone();
        this.utc = utc;
    }

    @Override
    public double pathDelay(SpacecraftState state, TopocentricFrame baseFrame, double frequency, double[] parameters) {
        GeodeticPoint recPoint = baseFrame.getPoint();
        AbsoluteDate date = state.getDate();
        BodyShape ellipsoid = baseFrame.getParentShape();
        TimeStampedPVCoordinates pv = state.getPVCoordinates(ellipsoid.getBodyFrame());
        GeodeticPoint satPoint = ellipsoid.transform(pv.getPosition(), ellipsoid.getBodyFrame(), state.getDate());
        double tec = this.stec(date, recPoint, satPoint);
        double factor = 4.03E17 / (frequency * frequency);
        return factor * tec;
    }

    @Override
    public <T extends RealFieldElement<T>> T pathDelay(FieldSpacecraftState<T> state, TopocentricFrame baseFrame, double frequency, T[] parameters) {
        FieldAbsoluteDate<T> date = state.getDate();
        FieldGeodeticPoint<T> recPoint = baseFrame.getPoint(date.getField());
        BodyShape ellipsoid = baseFrame.getParentShape();
        TimeStampedFieldPVCoordinates<T> pv = state.getPVCoordinates(ellipsoid.getBodyFrame());
        FieldGeodeticPoint satPoint = ellipsoid.transform(pv.getPosition(), ellipsoid.getBodyFrame(), state.getDate());
        T tec = this.stec(date, recPoint, satPoint);
        double factor = 4.03E17 / (frequency * frequency);
        return (T)((RealFieldElement)tec.multiply(factor));
    }

    @Override
    public List<ParameterDriver> getParametersDrivers() {
        return Collections.emptyList();
    }

    public double stec(AbsoluteDate date, GeodeticPoint recP, GeodeticPoint satP) {
        int count;
        Ray ray = new Ray(recP, satP);
        DateTimeComponents dateTime = date.getComponents(this.utc);
        this.loadsIfNeeded(dateTime.getDate());
        double h1 = recP.getAltitude();
        double tolerance = h1 < 1000000.0 ? 0.001 : 0.01;
        int n = 8;
        Segment seg1 = new Segment(n, ray);
        double gn1 = this.stecIntegration(seg1, dateTime);
        Segment seg2 = new Segment(n *= 2, ray);
        double gn2 = this.stecIntegration(seg2, dateTime);
        for (count = 1; FastMath.abs((double)(gn2 - gn1)) > tolerance * FastMath.abs((double)gn1) && count < 20; ++count) {
            gn1 = gn2;
            Segment seg = new Segment(n *= 2, ray);
            gn2 = this.stecIntegration(seg, dateTime);
        }
        if (count == 20) {
            throw new OrekitException((Localizable)OrekitMessages.STEC_INTEGRATION_DID_NOT_CONVERGE, new Object[0]);
        }
        return (gn2 + (gn2 - gn1) / 15.0) * 1.0E-16;
    }

    public <T extends RealFieldElement<T>> T stec(FieldAbsoluteDate<T> date, FieldGeodeticPoint<T> recP, FieldGeodeticPoint<T> satP) {
        int count;
        Field<T> field = date.getField();
        FieldRay<T> ray = new FieldRay<T>(field, recP, satP);
        DateTimeComponents dateTime = date.getComponents(this.utc);
        this.loadsIfNeeded(dateTime.getDate());
        T h1 = recP.getAltitude();
        double tolerance = h1.getReal() < 1000000.0 ? 0.001 : 0.01;
        int n = 8;
        FieldSegment<T> seg1 = new FieldSegment<T>(field, n, ray);
        T gn1 = this.stecIntegration(field, seg1, dateTime);
        FieldSegment<T> seg2 = new FieldSegment<T>(field, n *= 2, ray);
        T gn2 = this.stecIntegration(field, seg2, dateTime);
        for (count = 1; FastMath.abs((RealFieldElement)((RealFieldElement)gn2.subtract(gn1))).getReal() > ((RealFieldElement)FastMath.abs(gn1).multiply(tolerance)).getReal() && count < 20; ++count) {
            gn1 = gn2;
            FieldSegment<T> seg = new FieldSegment<T>(field, n *= 2, ray);
            gn2 = this.stecIntegration(field, seg, dateTime);
        }
        if (count == 20) {
            throw new OrekitException((Localizable)OrekitMessages.STEC_INTEGRATION_DID_NOT_CONVERGE, new Object[0]);
        }
        return (T)((RealFieldElement)((RealFieldElement)gn2.add(((RealFieldElement)gn2.subtract(gn1)).divide(15.0))).multiply(1.0E-16));
    }

    private double stecIntegration(Segment seg, DateTimeComponents dateTime) {
        double[] heightS = seg.getHeights();
        double[] latitudeS = seg.getLatitudes();
        double[] longitudeS = seg.getLongitudes();
        double density = 0.0;
        for (int i = 0; i < heightS.length; ++i) {
            NeQuickParameters parameters = new NeQuickParameters(dateTime, this.f2, this.fm3, latitudeS[i], longitudeS[i], this.alpha, this.stModip);
            density += this.electronDensity(heightS[i], parameters);
        }
        return 0.5 * seg.getInterval() * density;
    }

    private <T extends RealFieldElement<T>> T stecIntegration(Field<T> field, FieldSegment<T> seg, DateTimeComponents dateTime) {
        RealFieldElement[] heightS = seg.getHeights();
        RealFieldElement[] latitudeS = seg.getLatitudes();
        RealFieldElement[] longitudeS = seg.getLongitudes();
        RealFieldElement density = (RealFieldElement)field.getZero();
        for (int i = 0; i < heightS.length; ++i) {
            FieldNeQuickParameters<RealFieldElement> parameters = new FieldNeQuickParameters<RealFieldElement>(field, dateTime, this.f2, this.fm3, latitudeS[i], longitudeS[i], this.alpha, this.stModip);
            density = (RealFieldElement)density.add((Object)this.electronDensity(field, heightS[i], parameters));
        }
        return (T)((RealFieldElement)((RealFieldElement)seg.getInterval().multiply((Object)density)).multiply(0.5));
    }

    private double electronDensity(double h, NeQuickParameters parameters) {
        double hInKm = h * 0.001;
        double n = hInKm <= parameters.getHmF2() ? this.bottomElectronDensity(hInKm, parameters) : this.topElectronDensity(hInKm, parameters);
        return n;
    }

    private <T extends RealFieldElement<T>> T electronDensity(Field<T> field, T h, FieldNeQuickParameters<T> parameters) {
        RealFieldElement hInKm = (RealFieldElement)h.multiply(0.001);
        RealFieldElement n = hInKm.getReal() <= parameters.getHmF2().getReal() ? this.bottomElectronDensity(field, hInKm, parameters) : this.topElectronDensity(field, hInKm, parameters);
        return (T)n;
    }

    private double bottomElectronDensity(double h, NeQuickParameters parameters) {
        double be = h > parameters.getHmE() ? parameters.getBETop() : parameters.getBEBot();
        double bf1 = h > parameters.getHmF1() ? parameters.getB1Top() : parameters.getB1Bot();
        double bf2 = parameters.getB2Bot();
        double[] ct = new double[]{1.0 / bf2, 1.0 / bf1, 1.0 / be};
        double hTemp = FastMath.max((double)100.0, (double)h);
        double exp = this.clipExp(10.0 / (1.0 + FastMath.abs((double)(hTemp - parameters.getHmF2()))));
        double[] arguments = new double[]{(hTemp - parameters.getHmF2()) / bf2, (hTemp - parameters.getHmF1()) / bf1 * exp, (hTemp - parameters.getHmE()) / be * exp};
        double[] s = new double[3];
        double[] ds = new double[3];
        double[] amplitudes = parameters.getLayerAmplitudes();
        for (int i = 0; i < 3; ++i) {
            if (FastMath.abs((double)arguments[i]) > 25.0) {
                s[i] = 0.0;
                ds[i] = 0.0;
                continue;
            }
            double expA = this.clipExp(arguments[i]);
            double opExpA = 1.0 + expA;
            s[i] = amplitudes[i] * (expA / (opExpA * opExpA));
            ds[i] = ct[i] * ((1.0 - expA) / (1.0 + expA));
        }
        double aNo = MathArrays.linearCombination((double)s[0], (double)1.0, (double)s[1], (double)1.0, (double)s[2], (double)1.0);
        if (h >= 100.0) {
            return aNo * 1.0E11;
        }
        double bc = 1.0 - 10.0 * (MathArrays.linearCombination((double)s[0], (double)ds[0], (double)s[1], (double)ds[1], (double)s[2], (double)ds[2]) / aNo);
        double z = 0.1 * (h - 100.0);
        return aNo * this.clipExp(1.0 - bc * z - this.clipExp(-z)) * 1.0E11;
    }

    private <T extends RealFieldElement<T>> T bottomElectronDensity(Field<T> field, T h, FieldNeQuickParameters<T> parameters) {
        RealFieldElement zero = (RealFieldElement)field.getZero();
        RealFieldElement one = (RealFieldElement)field.getOne();
        T be = h.getReal() > parameters.getHmE().getReal() ? parameters.getBETop() : parameters.getBEBot();
        T bf1 = h.getReal() > parameters.getHmF1().getReal() ? parameters.getB1Top() : parameters.getB1Bot();
        T bf2 = parameters.getB2Bot();
        RealFieldElement[] ct = (RealFieldElement[])MathArrays.buildArray(field, (int)3);
        ct[0] = (RealFieldElement)bf2.reciprocal();
        ct[1] = (RealFieldElement)bf1.reciprocal();
        ct[2] = (RealFieldElement)be.reciprocal();
        RealFieldElement hTemp = FastMath.max((RealFieldElement)((RealFieldElement)zero.add(100.0)), h);
        RealFieldElement exp = this.clipExp(field, (RealFieldElement)((RealFieldElement)((RealFieldElement)FastMath.abs((RealFieldElement)((RealFieldElement)hTemp.subtract(parameters.getHmF2()))).add(1.0)).divide(10.0)).reciprocal());
        RealFieldElement[] arguments = (RealFieldElement[])MathArrays.buildArray(field, (int)3);
        arguments[0] = (RealFieldElement)((RealFieldElement)hTemp.subtract(parameters.getHmF2())).divide(bf2);
        arguments[1] = (RealFieldElement)((RealFieldElement)((RealFieldElement)hTemp.subtract(parameters.getHmF1())).divide(bf1)).multiply((Object)exp);
        arguments[2] = (RealFieldElement)((RealFieldElement)((RealFieldElement)hTemp.subtract(parameters.getHmE())).divide(be)).multiply((Object)exp);
        RealFieldElement[] s = (RealFieldElement[])MathArrays.buildArray(field, (int)3);
        RealFieldElement[] ds = (RealFieldElement[])MathArrays.buildArray(field, (int)3);
        RealFieldElement[] amplitudes = parameters.getLayerAmplitudes();
        for (int i = 0; i < 3; ++i) {
            if (FastMath.abs((RealFieldElement)arguments[i]).getReal() > 25.0) {
                s[i] = zero;
                ds[i] = zero;
                continue;
            }
            RealFieldElement expA = this.clipExp(field, arguments[i]);
            RealFieldElement opExpA = (RealFieldElement)expA.add(1.0);
            s[i] = (RealFieldElement)amplitudes[i].multiply(expA.divide(opExpA.multiply((Object)opExpA)));
            ds[i] = (RealFieldElement)ct[i].multiply(((RealFieldElement)((RealFieldElement)expA.negate()).add(1.0)).divide(expA.add(1.0)));
        }
        RealFieldElement aNo = (RealFieldElement)one.linearCombination((Object)s[0], (Object)one, (Object)s[1], (Object)one, (Object)s[2], (Object)one);
        if (h.getReal() >= 100.0) {
            return (T)((RealFieldElement)aNo.multiply(1.0E11));
        }
        RealFieldElement bc = (RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)s[0].multiply((Object)ds[0])).add(one.linearCombination((Object)s[0], (Object)ds[0], (Object)s[1], (Object)ds[1], (Object)s[2], (Object)ds[2]))).divide((Object)aNo)).multiply(10.0)).negate()).add(1.0);
        RealFieldElement z = (RealFieldElement)((RealFieldElement)h.subtract(100.0)).multiply(0.1);
        return (T)((RealFieldElement)((RealFieldElement)aNo.multiply((Object)this.clipExp(field, (RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)bc.multiply((Object)z)).add((Object)this.clipExp(field, (RealFieldElement)z.negate()))).negate()).add(1.0)))).multiply(1.0E11));
    }

    private double topElectronDensity(double h, NeQuickParameters parameters) {
        double g = 0.125;
        double r = 100.0;
        double deltaH = h - parameters.getHmF2();
        double z = deltaH / (parameters.getH0() * (1.0 + 12.5 * deltaH / (100.0 * parameters.getH0() + 0.125 * deltaH)));
        double ee = this.clipExp(z);
        if (ee > 1.0E11) {
            return 4.0 * parameters.getNmF2() / ee * 1.0E11;
        }
        double opExpZ = 1.0 + ee;
        return 4.0 * parameters.getNmF2() * ee / (opExpZ * opExpZ) * 1.0E11;
    }

    private <T extends RealFieldElement<T>> T topElectronDensity(Field<T> field, T h, FieldNeQuickParameters<T> parameters) {
        double g = 0.125;
        double r = 100.0;
        RealFieldElement deltaH = (RealFieldElement)h.subtract(parameters.getHmF2());
        RealFieldElement z = (RealFieldElement)deltaH.divide(parameters.getH0().multiply(((RealFieldElement)((RealFieldElement)((RealFieldElement)deltaH.multiply(100.0)).multiply(0.125)).divide(((RealFieldElement)parameters.getH0().multiply(100.0)).add(deltaH.multiply(0.125)))).add(1.0)));
        RealFieldElement ee = this.clipExp(field, z);
        if (ee.getReal() > 1.0E11) {
            return (T)((RealFieldElement)((RealFieldElement)((RealFieldElement)parameters.getNmF2().multiply(4.0)).divide((Object)ee)).multiply(1.0E11));
        }
        RealFieldElement opExpZ = (RealFieldElement)ee.add(field.getOne());
        return (T)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)parameters.getNmF2().multiply(4.0)).multiply((Object)ee)).divide(opExpZ.multiply((Object)opExpZ))).multiply(1.0E11));
    }

    private void loadsIfNeeded(DateComponents date) {
        int currentMonth = date.getMonth();
        if (currentMonth != this.month || this.f2 == null || this.fm3 == null) {
            this.month = currentMonth;
            CCIRLoader loader = new CCIRLoader();
            loader.loadCCIRCoefficients(date);
            this.f2 = loader.getF2();
            this.fm3 = loader.getFm3();
        }
    }

    private double clipExp(double power) {
        if (power > 80.0) {
            return 5.5406E34;
        }
        if (power < -80.0) {
            return 1.8049E-35;
        }
        return FastMath.exp((double)power);
    }

    private <T extends RealFieldElement<T>> T clipExp(Field<T> field, T power) {
        RealFieldElement zero = (RealFieldElement)field.getZero();
        if (power.getReal() > 80.0) {
            return (T)((RealFieldElement)zero.add(5.5406E34));
        }
        if (power.getReal() < -80.0) {
            return (T)((RealFieldElement)zero.add(1.8049E-35));
        }
        return (T)FastMath.exp(power);
    }

    private static InputStream getStream(String name) {
        return NeQuickModel.class.getResourceAsStream(name);
    }

    private static class FieldSegment<T extends RealFieldElement<T>> {
        private final T[] latitudes;
        private final T[] longitudes;
        private final T[] heights;
        private final T deltaN;

        FieldSegment(Field<T> field, int n, FieldRay<T> ray) {
            T s1 = ray.getS1();
            T s2 = ray.getS2();
            this.deltaN = (RealFieldElement)((RealFieldElement)s2.subtract(s1)).divide((double)n);
            RealFieldElement[] s = this.getSegments((Field)field, n, (RealFieldElement)s1, (RealFieldElement)s2);
            T rp = ray.getRadius();
            FieldSinCos scLatP = FastMath.sinCos(ray.getLatitude());
            int size = s.length;
            this.heights = (RealFieldElement[])MathArrays.buildArray(field, (int)size);
            this.latitudes = (RealFieldElement[])MathArrays.buildArray(field, (int)size);
            this.longitudes = (RealFieldElement[])MathArrays.buildArray(field, (int)size);
            for (int i = 0; i < size; ++i) {
                this.heights[i] = (RealFieldElement)FastMath.sqrt((RealFieldElement)((RealFieldElement)((RealFieldElement)s[i].multiply((Object)s[i])).add(rp.multiply(rp)))).subtract(6371200.0);
                RealFieldElement tanDs = (RealFieldElement)s[i].divide(rp);
                RealFieldElement cosDs = (RealFieldElement)FastMath.sqrt((RealFieldElement)((RealFieldElement)((RealFieldElement)tanDs.multiply((Object)tanDs)).add(1.0))).reciprocal();
                RealFieldElement sinDs = (RealFieldElement)tanDs.multiply((Object)cosDs);
                RealFieldElement sinLatS = (RealFieldElement)((RealFieldElement)((RealFieldElement)scLatP.sin()).multiply((Object)cosDs)).add(((RealFieldElement)((RealFieldElement)scLatP.cos()).multiply((Object)sinDs)).multiply(ray.getCosineAz()));
                RealFieldElement cosLatS = FastMath.sqrt((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)sinLatS.multiply((Object)sinLatS)).negate()).add(1.0)));
                this.latitudes[i] = FastMath.atan2((RealFieldElement)sinLatS, (RealFieldElement)cosLatS);
                RealFieldElement sinLonS = (RealFieldElement)((RealFieldElement)sinDs.multiply(ray.getSineAz())).multiply(scLatP.cos());
                RealFieldElement cosLonS = (RealFieldElement)cosDs.subtract(((RealFieldElement)scLatP.sin()).multiply((Object)sinLatS));
                this.longitudes[i] = (RealFieldElement)FastMath.atan2((RealFieldElement)sinLonS, (RealFieldElement)cosLonS).add(ray.getLongitude());
            }
        }

        private T[] getSegments(Field<T> field, int n, T s1, T s2) {
            RealFieldElement g = (RealFieldElement)this.deltaN.multiply(0.5773502691896);
            RealFieldElement y = (RealFieldElement)s1.add(((RealFieldElement)this.deltaN.subtract((Object)g)).multiply(0.5));
            RealFieldElement[] segments = (RealFieldElement[])MathArrays.buildArray(field, (int)(2 * n));
            int index = 0;
            for (int i = 0; i < n; ++i) {
                segments[index] = (RealFieldElement)y.add(this.deltaN.multiply(i));
                segments[++index] = (RealFieldElement)((RealFieldElement)y.add(this.deltaN.multiply(i))).add((Object)g);
                ++index;
            }
            return segments;
        }

        public T[] getLatitudes() {
            return this.latitudes;
        }

        public T[] getLongitudes() {
            return this.longitudes;
        }

        public T[] getHeights() {
            return this.heights;
        }

        public T getInterval() {
            return this.deltaN;
        }
    }

    private static class Segment {
        private final double[] latitudes;
        private final double[] longitudes;
        private final double[] heights;
        private final double deltaN;

        Segment(int n, Ray ray) {
            double s1 = ray.getS1();
            double s2 = ray.getS2();
            this.deltaN = (s2 - s1) / (double)n;
            double[] s = this.getSegments(n, s1, s2);
            double rp = ray.getRadius();
            SinCos scLatP = FastMath.sinCos((double)ray.getLatitude());
            int size = s.length;
            this.heights = new double[size];
            this.latitudes = new double[size];
            this.longitudes = new double[size];
            for (int i = 0; i < size; ++i) {
                this.heights[i] = FastMath.sqrt((double)(s[i] * s[i] + rp * rp)) - 6371200.0;
                double tanDs = s[i] / rp;
                double cosDs = 1.0 / FastMath.sqrt((double)(1.0 + tanDs * tanDs));
                double sinDs = tanDs * cosDs;
                double sinLatS = scLatP.sin() * cosDs + scLatP.cos() * sinDs * ray.getCosineAz();
                double cosLatS = FastMath.sqrt((double)(1.0 - sinLatS * sinLatS));
                this.latitudes[i] = FastMath.atan2((double)sinLatS, (double)cosLatS);
                double sinLonS = sinDs * ray.getSineAz() * scLatP.cos();
                double cosLonS = cosDs - scLatP.sin() * sinLatS;
                this.longitudes[i] = FastMath.atan2((double)sinLonS, (double)cosLonS) + ray.getLongitude();
            }
        }

        private double[] getSegments(int n, double s1, double s2) {
            double g = 0.5773502691896 * this.deltaN;
            double y = s1 + (this.deltaN - g) * 0.5;
            double[] segments = new double[2 * n];
            int index = 0;
            for (int i = 0; i < n; ++i) {
                segments[index] = y + (double)i * this.deltaN;
                segments[++index] = y + (double)i * this.deltaN + g;
                ++index;
            }
            return segments;
        }

        public double[] getLatitudes() {
            return this.latitudes;
        }

        public double[] getLongitudes() {
            return this.longitudes;
        }

        public double[] getHeights() {
            return this.heights;
        }

        public double getInterval() {
            return this.deltaN;
        }
    }

    private static class FieldRay<T extends RealFieldElement<T>> {
        private static final double THRESHOLD = 1.0E-10;
        private final T s1;
        private final T s2;
        private final T rp;
        private final T latP;
        private final T lonP;
        private final FieldSinCos<T> scLatP;
        private final T sinAzP;
        private final T cosAzP;

        FieldRay(Field<T> field, FieldGeodeticPoint<T> recP, FieldGeodeticPoint<T> satP) {
            RealFieldElement r1 = (RealFieldElement)recP.getAltitude().add(6371200.0);
            RealFieldElement r2 = (RealFieldElement)satP.getAltitude().add(6371200.0);
            T lat1 = recP.getLatitude();
            T lat2 = satP.getLatitude();
            T lon1 = recP.getLongitude();
            T lon2 = satP.getLongitude();
            FieldSinCos scLatSat = FastMath.sinCos(lat2);
            FieldSinCos scLatRec = FastMath.sinCos(lat1);
            RealFieldElement cosD = (RealFieldElement)((RealFieldElement)((RealFieldElement)scLatRec.sin()).multiply(scLatSat.sin())).add(((RealFieldElement)((RealFieldElement)scLatRec.cos()).multiply(scLatSat.cos())).multiply((Object)FastMath.cos((RealFieldElement)((RealFieldElement)lon2.subtract(lon1)))));
            RealFieldElement sinD = FastMath.sqrt((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)cosD.multiply((Object)cosD)).negate()).add(1.0)));
            RealFieldElement z = FastMath.atan2((RealFieldElement)sinD, (RealFieldElement)((RealFieldElement)cosD.subtract(r1.divide((Object)r2))));
            this.rp = (RealFieldElement)r1.multiply((Object)FastMath.sin((RealFieldElement)z));
            if (FastMath.abs((double)(FastMath.abs(lat1).getReal() - 1.5707963267948966)) < 1.0E-10) {
                this.latP = lat1.getReal() < 0.0 ? (RealFieldElement)z.negate() : z;
                this.lonP = z.getReal() < 0.0 ? lon2 : (RealFieldElement)lon2.add(Math.PI);
            } else {
                RealFieldElement deltaP = (RealFieldElement)((RealFieldElement)z.negate()).add(1.5707963267948966);
                FieldSinCos scDeltaP = FastMath.sinCos((RealFieldElement)deltaP);
                RealFieldElement sinAz = (RealFieldElement)((RealFieldElement)FastMath.sin((RealFieldElement)((RealFieldElement)lon2.subtract(lon1))).multiply(scLatSat.cos())).divide((Object)sinD);
                RealFieldElement cosAz = (RealFieldElement)((RealFieldElement)((RealFieldElement)scLatSat.sin()).subtract(cosD.multiply(scLatRec.sin()))).divide(sinD.multiply(scLatRec.cos()));
                RealFieldElement sinLatP = (RealFieldElement)((RealFieldElement)((RealFieldElement)scLatRec.sin()).multiply(scDeltaP.cos())).subtract(((RealFieldElement)((RealFieldElement)scLatRec.cos()).multiply(scDeltaP.sin())).multiply((Object)cosAz));
                RealFieldElement cosLatP = FastMath.sqrt((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)sinLatP.multiply((Object)sinLatP)).negate()).add(1.0)));
                this.latP = FastMath.atan2((RealFieldElement)sinLatP, (RealFieldElement)cosLatP);
                RealFieldElement sinLonP = (RealFieldElement)((RealFieldElement)((RealFieldElement)sinAz.negate()).multiply(scDeltaP.sin())).divide((Object)cosLatP);
                RealFieldElement cosLonP = (RealFieldElement)((RealFieldElement)((RealFieldElement)scDeltaP.cos()).subtract(((RealFieldElement)scLatRec.sin()).multiply((Object)sinLatP))).divide(((RealFieldElement)scLatRec.cos()).multiply((Object)cosLatP));
                this.lonP = (RealFieldElement)FastMath.atan2((RealFieldElement)sinLonP, (RealFieldElement)cosLonP).add(lon1);
            }
            this.scLatP = FastMath.sinCos(this.latP);
            FieldSinCos scLon = FastMath.sinCos((RealFieldElement)((RealFieldElement)lon2.subtract(this.lonP)));
            T psi = this.greatCircleAngle(scLatSat, scLon);
            FieldSinCos scPsi = FastMath.sinCos(psi);
            if (FastMath.abs((double)(FastMath.abs(this.latP).getReal() - 1.5707963267948966)) < 1.0E-10) {
                this.sinAzP = (RealFieldElement)field.getZero();
                this.cosAzP = this.latP.getReal() < 0.0 ? (RealFieldElement)field.getOne() : (RealFieldElement)((RealFieldElement)field.getOne()).negate();
            } else {
                this.sinAzP = (RealFieldElement)((RealFieldElement)((RealFieldElement)scLatSat.cos()).multiply(scLon.sin())).divide(scPsi.sin());
                this.cosAzP = (RealFieldElement)((RealFieldElement)((RealFieldElement)scLatSat.sin()).subtract(((RealFieldElement)this.scLatP.sin()).multiply(scPsi.cos()))).divide(((RealFieldElement)this.scLatP.cos()).multiply(scPsi.sin()));
            }
            this.s1 = FastMath.sqrt((RealFieldElement)((RealFieldElement)((RealFieldElement)r1.multiply((Object)r1)).subtract(this.rp.multiply(this.rp))));
            this.s2 = FastMath.sqrt((RealFieldElement)((RealFieldElement)((RealFieldElement)r2.multiply((Object)r2)).subtract(this.rp.multiply(this.rp))));
        }

        public T getS1() {
            return this.s1;
        }

        public T getS2() {
            return this.s2;
        }

        public T getRadius() {
            return this.rp;
        }

        public T getLatitude() {
            return this.latP;
        }

        public T getLongitude() {
            return this.lonP;
        }

        public T getSineAz() {
            return this.sinAzP;
        }

        public T getCosineAz() {
            return this.cosAzP;
        }

        private T greatCircleAngle(FieldSinCos<T> scLat, FieldSinCos<T> scLon) {
            if (FastMath.abs((double)(FastMath.abs(this.latP).getReal() - 1.5707963267948966)) < 1.0E-10) {
                return (T)FastMath.abs((RealFieldElement)((RealFieldElement)FastMath.asin((RealFieldElement)((RealFieldElement)scLat.sin())).subtract(this.latP)));
            }
            RealFieldElement cosPhi = (RealFieldElement)((RealFieldElement)((RealFieldElement)this.scLatP.sin()).multiply(scLat.sin())).add(((RealFieldElement)((RealFieldElement)this.scLatP.cos()).multiply(scLat.cos())).multiply(scLon.cos()));
            RealFieldElement sinPhi = FastMath.sqrt((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)cosPhi.multiply((Object)cosPhi)).negate()).add(1.0)));
            return (T)FastMath.atan2((RealFieldElement)sinPhi, (RealFieldElement)cosPhi);
        }
    }

    private static class Ray {
        private static final double THRESHOLD = 1.0E-10;
        private final double s1;
        private final double s2;
        private final double rp;
        private final double latP;
        private final double lonP;
        private final SinCos scLatP;
        private final double sinAzP;
        private final double cosAzP;

        Ray(GeodeticPoint recP, GeodeticPoint satP) {
            double r1 = 6371200.0 + recP.getAltitude();
            double r2 = 6371200.0 + satP.getAltitude();
            double lat1 = recP.getLatitude();
            double lat2 = satP.getLatitude();
            double lon1 = recP.getLongitude();
            double lon2 = satP.getLongitude();
            SinCos scLatSat = FastMath.sinCos((double)lat2);
            SinCos scLatRec = FastMath.sinCos((double)lat1);
            SinCos scLon21 = FastMath.sinCos((double)(lon2 - lon1));
            double cosD = scLatRec.sin() * scLatSat.sin() + scLatRec.cos() * scLatSat.cos() * scLon21.cos();
            double sinD = FastMath.sqrt((double)(1.0 - cosD * cosD));
            double z = FastMath.atan2((double)sinD, (double)(cosD - r1 / r2));
            this.rp = r1 * FastMath.sin((double)z);
            if (FastMath.abs((double)(FastMath.abs((double)lat1) - 1.5707963267948966)) < 1.0E-10) {
                this.latP = lat1 < 0.0 ? -z : z;
                this.lonP = z < 0.0 ? lon2 : lon2 + Math.PI;
            } else {
                double deltaP = 1.5707963267948966 - z;
                SinCos scDeltaP = FastMath.sinCos((double)deltaP);
                double sinAz = scLon21.sin() * scLatSat.cos() / sinD;
                double cosAz = (scLatSat.sin() - cosD * scLatRec.sin()) / (sinD * scLatRec.cos());
                double sinLatP = scLatRec.sin() * scDeltaP.cos() - scLatRec.cos() * scDeltaP.sin() * cosAz;
                double cosLatP = FastMath.sqrt((double)(1.0 - sinLatP * sinLatP));
                this.latP = FastMath.atan2((double)sinLatP, (double)cosLatP);
                double sinLonP = -sinAz * scDeltaP.sin() / cosLatP;
                double cosLonP = (scDeltaP.cos() - scLatRec.sin() * sinLatP) / (scLatRec.cos() * cosLatP);
                this.lonP = FastMath.atan2((double)sinLonP, (double)cosLonP) + lon1;
            }
            this.scLatP = FastMath.sinCos((double)this.latP);
            SinCos scLon = FastMath.sinCos((double)(lon2 - this.lonP));
            double psi = this.greatCircleAngle(scLatSat, scLon);
            SinCos scPsi = FastMath.sinCos((double)psi);
            if (FastMath.abs((double)(FastMath.abs((double)this.latP) - 1.5707963267948966)) < 1.0E-10) {
                this.sinAzP = 0.0;
                this.cosAzP = this.latP < 0.0 ? 1.0 : -1.0;
            } else {
                this.sinAzP = scLatSat.cos() * scLon.sin() / scPsi.sin();
                this.cosAzP = (scLatSat.sin() - this.scLatP.sin() * scPsi.cos()) / (this.scLatP.cos() * scPsi.sin());
            }
            this.s1 = FastMath.sqrt((double)(r1 * r1 - this.rp * this.rp));
            this.s2 = FastMath.sqrt((double)(r2 * r2 - this.rp * this.rp));
        }

        public double getS1() {
            return this.s1;
        }

        public double getS2() {
            return this.s2;
        }

        public double getRadius() {
            return this.rp;
        }

        public double getLatitude() {
            return this.latP;
        }

        public double getLongitude() {
            return this.lonP;
        }

        public double getSineAz() {
            return this.sinAzP;
        }

        public double getCosineAz() {
            return this.cosAzP;
        }

        private double greatCircleAngle(SinCos scLat, SinCos scLon) {
            if (FastMath.abs((double)(FastMath.abs((double)this.latP) - 1.5707963267948966)) < 1.0E-10) {
                return FastMath.abs((double)(FastMath.asin((double)scLat.sin()) - this.latP));
            }
            double cosPhi = this.scLatP.sin() * scLat.sin() + this.scLatP.cos() * scLat.cos() * scLon.cos();
            double sinPhi = FastMath.sqrt((double)(1.0 - cosPhi * cosPhi));
            return FastMath.atan2((double)sinPhi, (double)cosPhi);
        }
    }

    private static class CCIRLoader {
        public static final String DEFAULT_SUPPORTED_NAME = "ccir**.asc";
        private static final int NUMBER_F2_COEFFICIENTS = 1976;
        private static final int ROWS = 2;
        private static final int TOTAL_COLUMNS_F2 = 76;
        private static final int TOTAL_COLUMNS_FM3 = 49;
        private static final int DEPTH_F2 = 13;
        private static final int DEPTH_FM3 = 9;
        private String supportedName = "ccir**.asc";
        private double[][][] f2Loader = null;
        private double[][][] fm3Loader = null;

        CCIRLoader() {
        }

        public double[][][] getF2() {
            return (double[][][])this.f2Loader.clone();
        }

        public double[][][] getFm3() {
            return (double[][][])this.fm3Loader.clone();
        }

        public void loadCCIRCoefficients(DateComponents dateComponents) {
            int currentMonth = dateComponents.getMonth();
            this.supportedName = NeQuickModel.NEQUICK_BASE + String.format("ccir%02d.asc", currentMonth + 10);
            try (InputStream in = NeQuickModel.getStream(this.supportedName);){
                this.loadData(in, this.supportedName);
            }
            catch (IOException e) {
                throw new OrekitException((Localizable)OrekitMessages.INTERNAL_ERROR, (Throwable)e);
            }
            if (this.f2Loader == null || this.fm3Loader == null) {
                throw new OrekitException((Localizable)OrekitMessages.NEQUICK_F2_FM3_NOT_LOADED, this.supportedName);
            }
        }

        public void loadData(InputStream input, String name) throws IOException {
            double[][][] f2Temp = new double[2][76][13];
            double[][][] fm3Temp = new double[2][49][9];
            int lineNumber = 0;
            int index = 0;
            int currentRowF2 = 0;
            int currentColumnF2 = 0;
            int currentDepthF2 = 0;
            int currentRowFm3 = 0;
            int currentColumnFm3 = 0;
            int currentDepthFm3 = 0;
            String line = null;
            try (InputStreamReader isr = new InputStreamReader(input, StandardCharsets.UTF_8);
                 BufferedReader br = new BufferedReader(isr);){
                line = br.readLine();
                while (line != null) {
                    ++lineNumber;
                    if ((line = line.trim()).length() > 0) {
                        String[] ccir_line = SEPARATOR.split(line);
                        for (int i = 0; i < ccir_line.length; ++i) {
                            if (index < 1976) {
                                if (currentDepthF2 >= 13 && currentColumnF2 < 75) {
                                    currentDepthF2 = 0;
                                    ++currentColumnF2;
                                } else if (currentDepthF2 >= 13 && currentColumnF2 >= 75) {
                                    currentDepthF2 = 0;
                                    currentColumnF2 = 0;
                                    ++currentRowF2;
                                }
                                f2Temp[currentRowF2][currentColumnF2][currentDepthF2++] = Double.valueOf(ccir_line[i]);
                                ++index;
                                continue;
                            }
                            if (currentDepthFm3 >= 9 && currentColumnFm3 < 48) {
                                currentDepthFm3 = 0;
                                ++currentColumnFm3;
                            } else if (currentDepthFm3 >= 9 && currentColumnFm3 >= 48) {
                                currentDepthFm3 = 0;
                                currentColumnFm3 = 0;
                                ++currentRowFm3;
                            }
                            fm3Temp[currentRowFm3][currentColumnFm3][currentDepthFm3++] = Double.valueOf(ccir_line[i]);
                            ++index;
                        }
                    }
                    line = br.readLine();
                }
            }
            catch (NumberFormatException nfe) {
                throw new OrekitException((Localizable)OrekitMessages.UNABLE_TO_PARSE_LINE_IN_FILE, lineNumber, name, line);
            }
            this.f2Loader = (double[][][])f2Temp.clone();
            this.fm3Loader = (double[][][])fm3Temp.clone();
        }
    }

    private static class MODIPLoader {
        private static final String SUPPORTED_NAME = "/assets/org/orekit/nequick/modip.txt";
        private double[][] grid = null;

        MODIPLoader() {
        }

        public double[][] getMODIPGrid() {
            return (double[][])this.grid.clone();
        }

        public void loadMODIPGrid() {
            try (InputStream in = NeQuickModel.getStream(SUPPORTED_NAME);){
                this.loadData(in, SUPPORTED_NAME);
            }
            catch (IOException e) {
                throw new OrekitException((Localizable)OrekitMessages.INTERNAL_ERROR, (Throwable)e);
            }
            if (this.grid == null) {
                throw new OrekitException((Localizable)OrekitMessages.MODIP_GRID_NOT_LOADED, SUPPORTED_NAME);
            }
        }

        public void loadData(InputStream input, String name) throws IOException {
            int size = 39;
            double[][] array = new double[39][39];
            int lineNumber = 0;
            String line = null;
            try (InputStreamReader isr = new InputStreamReader(input, StandardCharsets.UTF_8);
                 BufferedReader br = new BufferedReader(isr);){
                line = br.readLine();
                while (line != null) {
                    ++lineNumber;
                    if ((line = line.trim()).length() > 0) {
                        String[] modip_line = SEPARATOR.split(line);
                        for (int column = 0; column < modip_line.length; ++column) {
                            array[lineNumber - 1][column] = Double.valueOf(modip_line[column]);
                        }
                    }
                    line = br.readLine();
                }
            }
            catch (NumberFormatException nfe) {
                throw new OrekitException((Localizable)OrekitMessages.UNABLE_TO_PARSE_LINE_IN_FILE, lineNumber, name, line);
            }
            this.grid = (double[][])array.clone();
        }
    }
}

