/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.gnss.antenna;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Optional;
import java.util.regex.Pattern;
import org.hipparchus.exception.Localizable;
import org.hipparchus.geometry.euclidean.threed.Vector3D;
import org.hipparchus.util.FastMath;
import org.orekit.annotation.DefaultDataContext;
import org.orekit.data.DataContext;
import org.orekit.data.DataLoader;
import org.orekit.data.DataProvidersManager;
import org.orekit.errors.OrekitException;
import org.orekit.errors.OrekitIllegalArgumentException;
import org.orekit.errors.OrekitMessages;
import org.orekit.gnss.Frequency;
import org.orekit.gnss.SatelliteSystem;
import org.orekit.gnss.antenna.FrequencyPattern;
import org.orekit.gnss.antenna.OneDVariation;
import org.orekit.gnss.antenna.PhaseCenterVariationFunction;
import org.orekit.gnss.antenna.ReceiverAntenna;
import org.orekit.gnss.antenna.SatelliteAntenna;
import org.orekit.gnss.antenna.SatelliteType;
import org.orekit.gnss.antenna.TwoDVariation;
import org.orekit.time.AbsoluteDate;
import org.orekit.time.TimeScale;
import org.orekit.utils.TimeSpanMap;

public class AntexLoader {
    public static final String DEFAULT_ANTEX_SUPPORTED_NAMES = "^\\w{5}(?:_\\d{4})?\\.atx$";
    private static final Pattern SEPARATOR = Pattern.compile("\\s+");
    private final List<TimeSpanMap<SatelliteAntenna>> satellitesAntennas;
    private final List<ReceiverAntenna> receiversAntennas;
    private final TimeScale gps;

    @DefaultDataContext
    public AntexLoader(String supportedNames) {
        this(supportedNames, DataContext.getDefault().getDataProvidersManager(), DataContext.getDefault().getTimeScales().getGPS());
    }

    public AntexLoader(String supportedNames, DataProvidersManager dataProvidersManager, TimeScale gps) {
        this.gps = gps;
        this.satellitesAntennas = new ArrayList<TimeSpanMap<SatelliteAntenna>>();
        this.receiversAntennas = new ArrayList<ReceiverAntenna>();
        dataProvidersManager.feed(supportedNames, new Parser());
    }

    private void addSatelliteAntenna(SatelliteAntenna antenna) {
        try {
            TimeSpanMap<SatelliteAntenna> existing = this.findSatelliteAntenna(antenna.getSatelliteSystem(), antenna.getPrnNumber());
            existing.addValidAfter(antenna, antenna.getValidFrom());
        }
        catch (OrekitException oe) {
            this.satellitesAntennas.add(new TimeSpanMap<SatelliteAntenna>(antenna));
        }
    }

    public List<TimeSpanMap<SatelliteAntenna>> getSatellitesAntennas() {
        return Collections.unmodifiableList(this.satellitesAntennas);
    }

    public TimeSpanMap<SatelliteAntenna> findSatelliteAntenna(SatelliteSystem satelliteSystem, int prnNumber) {
        Optional<TimeSpanMap> existing = this.satellitesAntennas.stream().filter(m -> {
            SatelliteAntenna first = (SatelliteAntenna)((TimeSpanMap.Transition)m.getTransitions().first()).getBefore();
            return first.getSatelliteSystem() == satelliteSystem && first.getPrnNumber() == prnNumber;
        }).findFirst();
        if (existing.isPresent()) {
            return existing.get();
        }
        throw new OrekitException((Localizable)OrekitMessages.CANNOT_FIND_SATELLITE_IN_SYSTEM, new Object[]{prnNumber, satelliteSystem});
    }

    private void addReceiverAntenna(ReceiverAntenna antenna) {
        this.receiversAntennas.add(antenna);
    }

    public List<ReceiverAntenna> getReceiversAntennas() {
        return Collections.unmodifiableList(this.receiversAntennas);
    }

    private class Parser
    implements DataLoader {
        private static final int LABEL_START = 60;
        private static final double FORMAT_VERSION = 1.4;
        private static final double MM_TO_M = 0.001;

        private Parser() {
        }

        @Override
        public boolean stillAcceptsData() {
            return true;
        }

        @Override
        public void loadData(InputStream input, String name) throws IOException, OrekitException {
            int lineNumber = 0;
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(input, StandardCharsets.UTF_8));){
                SatelliteSystem satelliteSystem = null;
                String antennaType = null;
                SatelliteType satelliteType = null;
                String serialNumber = null;
                int prnNumber = -1;
                int satelliteCode = -1;
                String cosparID = null;
                AbsoluteDate validFrom = AbsoluteDate.PAST_INFINITY;
                AbsoluteDate validUntil = AbsoluteDate.FUTURE_INFINITY;
                String sinexCode = null;
                double azimuthStep = Double.NaN;
                double polarStart = Double.NaN;
                double polarStop = Double.NaN;
                double polarStep = Double.NaN;
                double[] grid1D = null;
                double[][] grid2D = null;
                Vector3D eccentricities = Vector3D.ZERO;
                int nbFrequencies = -1;
                Frequency frequency = null;
                HashMap<Frequency, FrequencyPattern> patterns = null;
                boolean inFrequency = false;
                boolean inRMS = false;
                String line = reader.readLine();
                while (line != null) {
                    ++lineNumber;
                    switch (line.substring(60).trim()) {
                        case "COMMENT": {
                            break;
                        }
                        case "ANTEX VERSION / SYST": {
                            if (FastMath.abs((double)(this.parseDouble(line, 0, 8) - 1.4)) > 0.001) {
                                throw new OrekitException((Localizable)OrekitMessages.UNSUPPORTED_FILE_FORMAT, name);
                            }
                            SatelliteSystem.parseSatelliteSystem(this.parseString(line, 20, 1));
                            break;
                        }
                        case "PCV TYPE / REFANT": {
                            break;
                        }
                        case "END OF HEADER": {
                            break;
                        }
                        case "START OF ANTENNA": {
                            satelliteSystem = null;
                            antennaType = null;
                            satelliteType = null;
                            serialNumber = null;
                            prnNumber = -1;
                            satelliteCode = -1;
                            cosparID = null;
                            validFrom = AbsoluteDate.PAST_INFINITY;
                            validUntil = AbsoluteDate.FUTURE_INFINITY;
                            sinexCode = null;
                            azimuthStep = Double.NaN;
                            polarStart = Double.NaN;
                            polarStop = Double.NaN;
                            polarStep = Double.NaN;
                            grid1D = null;
                            grid2D = null;
                            eccentricities = Vector3D.ZERO;
                            nbFrequencies = -1;
                            frequency = null;
                            patterns = null;
                            inFrequency = false;
                            inRMS = false;
                            break;
                        }
                        case "TYPE / SERIAL NO": {
                            antennaType = this.parseString(line, 0, 20);
                            try {
                                satelliteType = SatelliteType.parseSatelliteType(antennaType);
                                String satField = this.parseString(line, 20, 20);
                                if (satField.length() <= 0) break;
                                satelliteSystem = SatelliteSystem.parseSatelliteSystem(satField);
                                int n = this.parseInt(satField, 1, 19);
                                switch (satelliteSystem) {
                                    case GPS: 
                                    case GLONASS: 
                                    case GALILEO: 
                                    case BEIDOU: 
                                    case IRNSS: {
                                        prnNumber = n;
                                        break;
                                    }
                                    case QZSS: {
                                        prnNumber = n + 192;
                                        break;
                                    }
                                    case SBAS: {
                                        prnNumber = n + 100;
                                        break;
                                    }
                                    default: {
                                        throw new OrekitException((Localizable)OrekitMessages.UNABLE_TO_PARSE_LINE_IN_FILE, lineNumber, name, line);
                                    }
                                }
                                satelliteCode = this.parseInt(line, 41, 9);
                                cosparID = this.parseString(line, 50, 10);
                            }
                            catch (OrekitIllegalArgumentException oiae) {
                                serialNumber = this.parseString(line, 20, 20);
                            }
                            break;
                        }
                        case "METH / BY / # / DATE": {
                            break;
                        }
                        case "DAZI": {
                            azimuthStep = FastMath.toRadians((double)this.parseDouble(line, 2, 6));
                            break;
                        }
                        case "ZEN1 / ZEN2 / DZEN": {
                            polarStart = FastMath.toRadians((double)this.parseDouble(line, 2, 6));
                            polarStop = FastMath.toRadians((double)this.parseDouble(line, 8, 6));
                            polarStep = FastMath.toRadians((double)this.parseDouble(line, 14, 6));
                            break;
                        }
                        case "# OF FREQUENCIES": {
                            nbFrequencies = this.parseInt(line, 0, 6);
                            patterns = new HashMap<Frequency, FrequencyPattern>(nbFrequencies);
                            break;
                        }
                        case "VALID FROM": {
                            validFrom = new AbsoluteDate(this.parseInt(line, 0, 6), this.parseInt(line, 6, 6), this.parseInt(line, 12, 6), this.parseInt(line, 18, 6), this.parseInt(line, 24, 6), this.parseDouble(line, 30, 13), AntexLoader.this.gps);
                            break;
                        }
                        case "VALID UNTIL": {
                            validUntil = new AbsoluteDate(this.parseInt(line, 0, 6), this.parseInt(line, 6, 6), this.parseInt(line, 12, 6), this.parseInt(line, 18, 6), this.parseInt(line, 24, 6), this.parseDouble(line, 30, 13), AntexLoader.this.gps);
                            break;
                        }
                        case "SINEX CODE": {
                            sinexCode = this.parseString(line, 0, 10);
                            break;
                        }
                        case "START OF FREQUENCY": {
                            try {
                                frequency = Frequency.valueOf(this.parseString(line, 3, 3));
                                grid1D = new double[1 + (int)FastMath.round((double)((polarStop - polarStart) / polarStep))];
                                if (azimuthStep > 0.001) {
                                    grid2D = new double[1 + (int)FastMath.round((double)(Math.PI * 2 / azimuthStep))][grid1D.length];
                                }
                            }
                            catch (IllegalArgumentException iae) {
                                throw new OrekitException(iae, OrekitMessages.UNKNOWN_RINEX_FREQUENCY, this.parseString(line, 3, 3), name, lineNumber);
                            }
                            inFrequency = true;
                            break;
                        }
                        case "NORTH / EAST / UP": {
                            if (inRMS) break;
                            eccentricities = new Vector3D(this.parseDouble(line, 0, 10) * 0.001, this.parseDouble(line, 10, 10) * 0.001, this.parseDouble(line, 20, 10) * 0.001);
                            break;
                        }
                        case "END OF FREQUENCY": {
                            PhaseCenterVariationFunction phaseCenterVariation;
                            String endFrequency = this.parseString(line, 3, 3);
                            if (frequency == null || !frequency.toString().equals(endFrequency)) {
                                throw new OrekitException((Localizable)OrekitMessages.MISMATCHED_FREQUENCIES, new Object[]{name, lineNumber, frequency, endFrequency});
                            }
                            if (patterns == null) {
                                throw new OrekitException((Localizable)OrekitMessages.UNABLE_TO_PARSE_LINE_IN_FILE, lineNumber, name, line);
                            }
                            if (grid2D == null) {
                                double max = 0.0;
                                for (double v : grid1D) {
                                    max = FastMath.max((double)max, (double)FastMath.abs((double)v));
                                }
                                phaseCenterVariation = max == 0.0 ? (polarAngle, azimuthAngle) -> 0.0 : new OneDVariation(polarStart, polarStep, grid1D);
                            } else {
                                phaseCenterVariation = new TwoDVariation(polarStart, polarStep, azimuthStep, grid2D);
                            }
                            patterns.put(frequency, new FrequencyPattern(eccentricities, phaseCenterVariation));
                            frequency = null;
                            grid1D = null;
                            grid2D = null;
                            inFrequency = false;
                            break;
                        }
                        case "START OF FREQ RMS": {
                            inRMS = true;
                            break;
                        }
                        case "END OF FREQ RMS": {
                            inRMS = false;
                            break;
                        }
                        case "END OF ANTENNA": {
                            if (satelliteType == null) {
                                AntexLoader.this.addReceiverAntenna(new ReceiverAntenna(antennaType, sinexCode, patterns, serialNumber));
                                break;
                            }
                            AntexLoader.this.addSatelliteAntenna(new SatelliteAntenna(antennaType, sinexCode, patterns, satelliteSystem, prnNumber, satelliteType, satelliteCode, cosparID, validFrom, validUntil));
                            break;
                        }
                        default: {
                            if (inFrequency) {
                                String[] fields = SEPARATOR.split(line.trim());
                                if (fields.length != grid1D.length + 1) {
                                    throw new OrekitException((Localizable)OrekitMessages.WRONG_COLUMNS_NUMBER, name, lineNumber, grid1D.length + 1, fields.length);
                                }
                                if ("NOAZI".equals(fields[0])) {
                                    for (int i = 0; i < grid1D.length; ++i) {
                                        grid1D[i] = Double.parseDouble(fields[i + 1]) * 0.001;
                                    }
                                } else {
                                    int k = (int)FastMath.round((double)(FastMath.toRadians((double)Double.parseDouble(fields[0])) / azimuthStep));
                                    for (int i = 0; i < grid2D[k].length; ++i) {
                                        grid2D[k][i] = Double.parseDouble(fields[i + 1]) * 0.001;
                                    }
                                }
                                break;
                            }
                            if (inRMS) break;
                            throw new OrekitException((Localizable)OrekitMessages.UNABLE_TO_PARSE_LINE_IN_FILE, lineNumber, name, line);
                        }
                    }
                    line = reader.readLine();
                }
            }
            catch (NumberFormatException nfe) {
                throw new OrekitException((Localizable)OrekitMessages.UNABLE_TO_PARSE_LINE_IN_FILE, lineNumber, name, "tot");
            }
        }

        private String parseString(String line, int start, int length) {
            return line.substring(start, FastMath.min((int)line.length(), (int)(start + length))).trim();
        }

        private int parseInt(String line, int start, int length) {
            return Integer.parseInt(this.parseString(line, start, length));
        }

        private double parseDouble(String line, int start, int length) {
            return Double.parseDouble(this.parseString(line, start, length));
        }
    }
}

