/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.propagation.numerical;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import org.hipparchus.exception.Localizable;
import org.hipparchus.geometry.euclidean.threed.Vector3D;
import org.hipparchus.ode.ODEIntegrator;
import org.hipparchus.util.FastMath;
import org.orekit.annotation.DefaultDataContext;
import org.orekit.attitudes.Attitude;
import org.orekit.attitudes.AttitudeProvider;
import org.orekit.data.DataContext;
import org.orekit.errors.OrekitException;
import org.orekit.errors.OrekitIllegalArgumentException;
import org.orekit.errors.OrekitInternalError;
import org.orekit.errors.OrekitMessages;
import org.orekit.forces.ForceModel;
import org.orekit.forces.gravity.NewtonianAttraction;
import org.orekit.forces.inertia.InertialForces;
import org.orekit.frames.Frame;
import org.orekit.orbits.Orbit;
import org.orekit.orbits.OrbitType;
import org.orekit.orbits.PositionAngle;
import org.orekit.propagation.PropagationType;
import org.orekit.propagation.Propagator;
import org.orekit.propagation.SpacecraftState;
import org.orekit.propagation.integration.AbstractIntegratedPropagator;
import org.orekit.propagation.integration.StateMapper;
import org.orekit.propagation.numerical.TimeDerivativesEquations;
import org.orekit.time.AbsoluteDate;
import org.orekit.utils.AbsolutePVCoordinates;
import org.orekit.utils.ParameterDriver;
import org.orekit.utils.ParameterObserver;
import org.orekit.utils.TimeStampedPVCoordinates;

public class NumericalPropagator
extends AbstractIntegratedPropagator {
    private final List<ForceModel> forceModels = new ArrayList<ForceModel>();
    private boolean ignoreCentralAttraction = false;

    @DefaultDataContext
    public NumericalPropagator(ODEIntegrator integrator) {
        this(integrator, Propagator.getDefaultLaw(DataContext.getDefault().getFrames()));
    }

    public NumericalPropagator(ODEIntegrator integrator, AttitudeProvider attitudeProvider) {
        super(integrator, PropagationType.MEAN);
        this.initMapper();
        this.setAttitudeProvider(attitudeProvider);
        this.setSlaveMode();
        this.setOrbitType(OrbitType.EQUINOCTIAL);
        this.setPositionAngleType(PositionAngle.TRUE);
    }

    public void setIgnoreCentralAttraction(boolean ignoreCentralAttraction) {
        this.ignoreCentralAttraction = ignoreCentralAttraction;
    }

    @Override
    public void setMu(double mu) {
        if (this.ignoreCentralAttraction) {
            this.superSetMu(mu);
        } else {
            this.addForceModel(new NewtonianAttraction(mu));
        }
    }

    private void superSetMu(double mu) {
        super.setMu(mu);
    }

    private boolean hasNewtonianAttraction() {
        int last = this.forceModels.size() - 1;
        return last >= 0 && this.forceModels.get(last) instanceof NewtonianAttraction;
    }

    public void addForceModel(ForceModel model) {
        if (model instanceof NewtonianAttraction) {
            try {
                model.getParametersDrivers()[0].addObserver(new ParameterObserver(){

                    @Override
                    public void valueChanged(double previousValue, ParameterDriver driver) {
                        NumericalPropagator.this.superSetMu(driver.getValue());
                    }
                });
            }
            catch (OrekitException oe) {
                throw new OrekitInternalError(oe);
            }
            if (this.hasNewtonianAttraction()) {
                this.forceModels.set(this.forceModels.size() - 1, model);
            } else {
                this.forceModels.add(model);
            }
        } else if (this.hasNewtonianAttraction()) {
            this.forceModels.add(this.forceModels.size() - 1, model);
        } else {
            this.forceModels.add(model);
        }
    }

    public void removeForceModels() {
        int last = this.forceModels.size() - 1;
        if (this.hasNewtonianAttraction()) {
            ForceModel newton = this.forceModels.get(last);
            this.forceModels.clear();
            this.forceModels.add(newton);
        } else {
            this.forceModels.clear();
        }
    }

    public List<ForceModel> getAllForceModels() {
        return Collections.unmodifiableList(this.forceModels);
    }

    @Override
    public void setOrbitType(OrbitType orbitType) {
        super.setOrbitType(orbitType);
    }

    @Override
    public OrbitType getOrbitType() {
        return super.getOrbitType();
    }

    @Override
    public void setPositionAngleType(PositionAngle positionAngleType) {
        super.setPositionAngleType(positionAngleType);
    }

    @Override
    public PositionAngle getPositionAngleType() {
        return super.getPositionAngleType();
    }

    public void setInitialState(SpacecraftState initialState) {
        this.resetInitialState(initialState);
    }

    @Override
    public void resetInitialState(SpacecraftState state) {
        super.resetInitialState(state);
        if (!this.hasNewtonianAttraction()) {
            this.setMu(state.getMu());
        }
        this.setStartDate(state.getDate());
    }

    @Override
    public TimeStampedPVCoordinates getPVCoordinates(AbsoluteDate date, Frame frame) {
        return this.propagate(date).getPVCoordinates(frame);
    }

    @Override
    protected StateMapper createMapper(AbsoluteDate referenceDate, double mu, OrbitType orbitType, PositionAngle positionAngleType, AttitudeProvider attitudeProvider, Frame frame) {
        return new OsculatingMapper(referenceDate, mu, orbitType, positionAngleType, attitudeProvider, frame);
    }

    @Override
    protected AbstractIntegratedPropagator.MainStateEquations getMainStateEquations(ODEIntegrator integrator) {
        return new Main(integrator);
    }

    public static double[][] tolerances(double dP, AbsolutePVCoordinates absPva) {
        double relative = dP / absPva.getPosition().getNorm();
        double dV = relative * absPva.getVelocity().getNorm();
        double[] absTol = new double[7];
        double[] relTol = new double[7];
        absTol[0] = dP;
        absTol[1] = dP;
        absTol[2] = dP;
        absTol[3] = dV;
        absTol[4] = dV;
        absTol[5] = dV;
        absTol[6] = 1.0E-6;
        Arrays.fill(relTol, relative);
        return new double[][]{absTol, relTol};
    }

    public static double[][] tolerances(double dP, Orbit orbit, OrbitType type) {
        TimeStampedPVCoordinates pv = orbit.getPVCoordinates();
        double r2 = pv.getPosition().getNormSq();
        double v = pv.getVelocity().getNorm();
        double dV = orbit.getMu() * dP / (v * r2);
        return NumericalPropagator.tolerances(dP, dV, orbit, type);
    }

    public static double[][] tolerances(double dP, double dV, Orbit orbit, OrbitType type) {
        double[] absTol = new double[7];
        double[] relTol = new double[7];
        absTol[6] = 1.0E-6;
        if (type == OrbitType.CARTESIAN) {
            absTol[0] = dP;
            absTol[1] = dP;
            absTol[2] = dP;
            absTol[3] = dV;
            absTol[4] = dV;
            absTol[5] = dV;
        } else {
            double[][] jacobian = new double[6][6];
            Orbit converted = type.convertType(orbit);
            converted.getJacobianWrtCartesian(PositionAngle.TRUE, jacobian);
            for (int i = 0; i < 6; ++i) {
                double[] row = jacobian[i];
                absTol[i] = FastMath.abs((double)row[0]) * dP + FastMath.abs((double)row[1]) * dP + FastMath.abs((double)row[2]) * dP + FastMath.abs((double)row[3]) * dV + FastMath.abs((double)row[4]) * dV + FastMath.abs((double)row[5]) * dV;
                if (!Double.isNaN(absTol[i])) continue;
                throw new OrekitException((Localizable)OrekitMessages.SINGULAR_JACOBIAN_FOR_ORBIT_TYPE, new Object[]{type});
            }
        }
        Arrays.fill(relTol, dP / FastMath.sqrt((double)orbit.getPVCoordinates().getPosition().getNormSq()));
        return new double[][]{absTol, relTol};
    }

    @Override
    protected void beforeIntegration(SpacecraftState initialState, AbsoluteDate tEnd) {
        if (!this.getFrame().isPseudoInertial()) {
            for (ForceModel force : this.forceModels) {
                if (!(force instanceof InertialForces)) continue;
                return;
            }
            throw new OrekitException((Localizable)OrekitMessages.INERTIAL_FORCE_MODEL_MISSING, this.getFrame().getName());
        }
    }

    private class Main
    implements AbstractIntegratedPropagator.MainStateEquations,
    TimeDerivativesEquations {
        private final double[] yDot = new double[7];
        private SpacecraftState currentState;
        private double[][] jacobian = new double[6][6];

        Main(ODEIntegrator integrator) {
            for (ForceModel forceModel : NumericalPropagator.this.forceModels) {
                forceModel.getEventsDetectors().forEach(detector -> NumericalPropagator.this.setUpEventDetector(integrator, detector));
            }
            if (NumericalPropagator.this.getOrbitType() == null) {
                for (int i = 0; i < this.jacobian.length; ++i) {
                    Arrays.fill(this.jacobian[i], 0.0);
                    this.jacobian[i][i] = 1.0;
                }
            }
        }

        @Override
        public void init(SpacecraftState initialState, AbsoluteDate target) {
            for (ForceModel forceModel : NumericalPropagator.this.forceModels) {
                forceModel.init(initialState, target);
            }
        }

        @Override
        public double[] computeDerivatives(SpacecraftState state) {
            this.currentState = state;
            Arrays.fill(this.yDot, 0.0);
            if (NumericalPropagator.this.getOrbitType() != null) {
                this.currentState.getOrbit().getJacobianWrtCartesian(NumericalPropagator.this.getPositionAngleType(), this.jacobian);
            }
            for (ForceModel forceModel : NumericalPropagator.this.forceModels) {
                forceModel.addContribution(state, this);
            }
            if (NumericalPropagator.this.getOrbitType() == null) {
                Vector3D velocity = this.currentState.getPVCoordinates().getVelocity();
                this.yDot[0] = this.yDot[0] + velocity.getX();
                this.yDot[1] = this.yDot[1] + velocity.getY();
                this.yDot[2] = this.yDot[2] + velocity.getZ();
            }
            return (double[])this.yDot.clone();
        }

        @Override
        public void addKeplerContribution(double mu) {
            if (NumericalPropagator.this.getOrbitType() == null) {
                if (mu > 0.0) {
                    Vector3D position = this.currentState.getPVCoordinates().getPosition();
                    double r2 = position.getNormSq();
                    double coeff = -mu / (r2 * FastMath.sqrt((double)r2));
                    this.yDot[3] = this.yDot[3] + coeff * position.getX();
                    this.yDot[4] = this.yDot[4] + coeff * position.getY();
                    this.yDot[5] = this.yDot[5] + coeff * position.getZ();
                }
            } else {
                this.currentState.getOrbit().addKeplerContribution(NumericalPropagator.this.getPositionAngleType(), mu, this.yDot);
            }
        }

        @Override
        public void addNonKeplerianAcceleration(Vector3D gamma) {
            int i = 0;
            while (i < 6) {
                double[] jRow = this.jacobian[i];
                int n = i++;
                this.yDot[n] = this.yDot[n] + (jRow[3] * gamma.getX() + jRow[4] * gamma.getY() + jRow[5] * gamma.getZ());
            }
        }

        @Override
        public void addMassDerivative(double q) {
            if (q > 0.0) {
                throw new OrekitIllegalArgumentException(OrekitMessages.POSITIVE_FLOW_RATE, q);
            }
            this.yDot[6] = this.yDot[6] + q;
        }
    }

    private static class OsculatingMapper
    extends StateMapper {
        OsculatingMapper(AbsoluteDate referenceDate, double mu, OrbitType orbitType, PositionAngle positionAngleType, AttitudeProvider attitudeProvider, Frame frame) {
            super(referenceDate, mu, orbitType, positionAngleType, attitudeProvider, frame);
        }

        @Override
        public SpacecraftState mapArrayToState(AbsoluteDate date, double[] y, double[] yDot, PropagationType type) {
            double mass = y[6];
            if (mass <= 0.0) {
                throw new OrekitException((Localizable)OrekitMessages.SPACECRAFT_MASS_BECOMES_NEGATIVE, mass);
            }
            if (this.getOrbitType() == null) {
                AbsolutePVCoordinates absPva;
                Vector3D p = new Vector3D(y[0], y[1], y[2]);
                Vector3D v = new Vector3D(y[3], y[4], y[5]);
                if (yDot == null) {
                    absPva = new AbsolutePVCoordinates(this.getFrame(), new TimeStampedPVCoordinates(date, p, v));
                } else {
                    Vector3D a = new Vector3D(yDot[3], yDot[4], yDot[5]);
                    absPva = new AbsolutePVCoordinates(this.getFrame(), new TimeStampedPVCoordinates(date, p, v, a));
                }
                Attitude attitude = this.getAttitudeProvider().getAttitude(absPva, date, this.getFrame());
                return new SpacecraftState(absPva, attitude, mass);
            }
            Orbit orbit = this.getOrbitType().mapArrayToOrbit(y, yDot, this.getPositionAngleType(), date, this.getMu(), this.getFrame());
            Attitude attitude = this.getAttitudeProvider().getAttitude(orbit, date, this.getFrame());
            return new SpacecraftState(orbit, attitude, mass);
        }

        @Override
        public void mapStateToArray(SpacecraftState state, double[] y, double[] yDot) {
            if (this.getOrbitType() == null) {
                Vector3D p = state.getAbsPVA().getPosition();
                Vector3D v = state.getAbsPVA().getVelocity();
                y[0] = p.getX();
                y[1] = p.getY();
                y[2] = p.getZ();
                y[3] = v.getX();
                y[4] = v.getY();
                y[5] = v.getZ();
                y[6] = state.getMass();
            } else {
                this.getOrbitType().mapOrbitToArray(state.getOrbit(), this.getPositionAngleType(), y, yDot);
                y[6] = state.getMass();
            }
        }
    }
}

