/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.files.ccsds;

import java.io.IOException;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import org.hipparchus.exception.Localizable;
import org.hipparchus.exception.LocalizedCoreFormats;
import org.hipparchus.linear.RealMatrix;
import org.orekit.errors.OrekitException;
import org.orekit.errors.OrekitMessages;
import org.orekit.files.ccsds.CCSDSFrame;
import org.orekit.files.ccsds.CcsdsModifiedFrame;
import org.orekit.files.ccsds.Keyword;
import org.orekit.files.ccsds.OEMFile;
import org.orekit.frames.FactoryManagedFrame;
import org.orekit.frames.Frame;
import org.orekit.frames.LOFType;
import org.orekit.frames.Predefined;
import org.orekit.frames.VersionedITRF;
import org.orekit.propagation.SpacecraftState;
import org.orekit.propagation.sampling.OrekitFixedStepHandler;
import org.orekit.time.AbsoluteDate;
import org.orekit.time.DateTimeComponents;
import org.orekit.time.TimeComponents;
import org.orekit.time.TimeScale;
import org.orekit.utils.TimeStampedPVCoordinates;

public class StreamingOemWriter {
    public static final String CCSDS_OEM_VERS = "2.0";
    public static final String DEFAULT_ORIGINATOR = "OREKIT";
    private static final String NEW_LINE = "\n";
    private static final Locale STANDARDIZED_LOCALE = Locale.US;
    private static final String KV_FORMAT = "%s = %s%n";
    private static final double M_TO_KM = 0.001;
    private static final String INERTIAL_FRAME_SUFFIX = "/inertial";
    private final Appendable writer;
    private final Map<Keyword, String> metadata;
    private final TimeScale timeScale;

    public StreamingOemWriter(Appendable writer, TimeScale timeScale, Map<Keyword, String> metadata) {
        this.writer = writer;
        this.timeScale = timeScale;
        this.metadata = new LinkedHashMap<Keyword, String>(metadata);
        this.metadata.putIfAbsent(Keyword.CCSDS_OEM_VERS, CCSDS_OEM_VERS);
        this.metadata.putIfAbsent(Keyword.CREATION_DATE, ZonedDateTime.now(ZoneOffset.UTC).format(DateTimeFormatter.ISO_INSTANT));
        this.metadata.putIfAbsent(Keyword.ORIGINATOR, DEFAULT_ORIGINATOR);
        this.metadata.putIfAbsent(Keyword.TIME_SYSTEM, timeScale.getName());
    }

    static String guessFrame(Frame frame) {
        String tod = "TOD";
        String itrf = "ITRF";
        String name = frame.getName();
        if (Arrays.stream(CCSDSFrame.values()).map(Enum::name).anyMatch(name::equals)) {
            return name;
        }
        if (frame instanceof CcsdsModifiedFrame) {
            return ((CcsdsModifiedFrame)frame).getRefFrame();
        }
        if ("Mars/inertial".equals(name)) {
            return "MCI";
        }
        if ("solar system barycenter/inertial".equals(name)) {
            return "ICRF";
        }
        if (name.contains("GTOD")) {
            return "TDR";
        }
        if (name.contains("TOD")) {
            return "TOD";
        }
        if (name.contains("Equinox") && name.contains("ITRF")) {
            return "GRC";
        }
        if (frame instanceof VersionedITRF) {
            return ((VersionedITRF)frame).getITRFVersion().getName().replace("-", "");
        }
        if (name.contains("CIO") && name.contains("ITRF")) {
            return "ITRF2014";
        }
        return name;
    }

    static String guessCenter(Frame frame) {
        String name = frame.getName();
        if (name.endsWith(INERTIAL_FRAME_SUFFIX) || name.endsWith("/rotating")) {
            return name.substring(0, name.length() - 9).toUpperCase(STANDARDIZED_LOCALE);
        }
        if (frame instanceof CcsdsModifiedFrame) {
            return ((CcsdsModifiedFrame)frame).getCenterName();
        }
        if (frame.getName().equals(Predefined.ICRF.getName())) {
            return "solar system barycenter".toUpperCase(STANDARDIZED_LOCALE);
        }
        if (frame.getDepth() == 0 || frame instanceof FactoryManagedFrame) {
            return "EARTH";
        }
        return "UNKNOWN";
    }

    private void writeKeyValue(Keyword key, String value) throws IOException {
        this.writer.append(String.format(STANDARDIZED_LOCALE, KV_FORMAT, key.toString(), value));
    }

    public void writeHeader() throws IOException {
        this.writeKeyValue(Keyword.CCSDS_OEM_VERS, this.metadata.get((Object)Keyword.CCSDS_OEM_VERS));
        String comment = this.metadata.get((Object)Keyword.COMMENT);
        if (comment != null) {
            this.writeKeyValue(Keyword.COMMENT, comment);
        }
        this.writeKeyValue(Keyword.CREATION_DATE, this.metadata.get((Object)Keyword.CREATION_DATE));
        this.writeKeyValue(Keyword.ORIGINATOR, this.metadata.get((Object)Keyword.ORIGINATOR));
        this.writer.append(NEW_LINE);
    }

    public Segment newSegment(Frame frame, Map<Keyword, String> segmentMetadata) {
        LinkedHashMap<Keyword, String> meta = new LinkedHashMap<Keyword, String>(this.metadata);
        meta.putAll(segmentMetadata);
        if (!meta.containsKey((Object)Keyword.REF_FRAME)) {
            meta.put(Keyword.REF_FRAME, StreamingOemWriter.guessFrame(frame));
        }
        if (!meta.containsKey((Object)Keyword.CENTER_NAME)) {
            meta.put(Keyword.CENTER_NAME, StreamingOemWriter.guessCenter(frame));
        }
        return new Segment(frame, meta);
    }

    static String dateToString(DateTimeComponents components) {
        TimeComponents time = components.getTime();
        int hour = time.getHour();
        int minute = time.getMinute();
        double second = time.getSecond();
        DecimalFormatSymbols locale = new DecimalFormatSymbols(STANDARDIZED_LOCALE);
        DecimalFormat twoDigits = new DecimalFormat("00", locale);
        DecimalFormat precise = new DecimalFormat("00.0########", locale);
        return components.getDate().toString() + "T" + twoDigits.format(hour) + ":" + twoDigits.format(minute) + ":" + precise.format(second);
    }

    public class Segment
    implements OrekitFixedStepHandler {
        private final Frame frame;
        private final Map<Keyword, String> metadata;

        private Segment(Frame frame, Map<Keyword, String> metadata) {
            this.frame = frame;
            this.metadata = metadata;
        }

        public void writeMetadata() throws IOException {
            String interpolationDegree;
            String interpolation;
            StreamingOemWriter.this.writer.append("META_START").append(StreamingOemWriter.NEW_LINE);
            if (this.frame != null) {
                StreamingOemWriter.this.writer.append("COMMENT ").append("Orekit frame: ").append(this.frame.toString()).append(StreamingOemWriter.NEW_LINE);
            }
            StreamingOemWriter.this.writeKeyValue(Keyword.OBJECT_NAME, this.metadata.get((Object)Keyword.OBJECT_NAME));
            StreamingOemWriter.this.writeKeyValue(Keyword.OBJECT_ID, this.metadata.get((Object)Keyword.OBJECT_ID));
            StreamingOemWriter.this.writeKeyValue(Keyword.CENTER_NAME, this.metadata.get((Object)Keyword.CENTER_NAME));
            StreamingOemWriter.this.writeKeyValue(Keyword.REF_FRAME, this.metadata.get((Object)Keyword.REF_FRAME));
            String refFrameEpoch = this.metadata.get((Object)Keyword.REF_FRAME_EPOCH);
            if (refFrameEpoch != null) {
                StreamingOemWriter.this.writeKeyValue(Keyword.REF_FRAME_EPOCH, refFrameEpoch);
            }
            StreamingOemWriter.this.writeKeyValue(Keyword.TIME_SYSTEM, this.metadata.get((Object)Keyword.TIME_SYSTEM));
            StreamingOemWriter.this.writeKeyValue(Keyword.START_TIME, this.metadata.get((Object)Keyword.START_TIME));
            String usableStartTime = this.metadata.get((Object)Keyword.USEABLE_START_TIME);
            if (usableStartTime != null) {
                StreamingOemWriter.this.writeKeyValue(Keyword.USEABLE_START_TIME, usableStartTime);
            }
            StreamingOemWriter.this.writeKeyValue(Keyword.STOP_TIME, this.metadata.get((Object)Keyword.STOP_TIME));
            String usableStopTime = this.metadata.get((Object)Keyword.USEABLE_STOP_TIME);
            if (usableStopTime != null) {
                StreamingOemWriter.this.writeKeyValue(Keyword.USEABLE_STOP_TIME, usableStopTime);
            }
            if ((interpolation = this.metadata.get((Object)Keyword.INTERPOLATION)) != null) {
                StreamingOemWriter.this.writeKeyValue(Keyword.INTERPOLATION, interpolation);
            }
            if ((interpolationDegree = this.metadata.get((Object)Keyword.INTERPOLATION_DEGREE)) != null) {
                StreamingOemWriter.this.writeKeyValue(Keyword.INTERPOLATION_DEGREE, interpolationDegree);
            }
            StreamingOemWriter.this.writer.append("META_STOP").append(StreamingOemWriter.NEW_LINE).append(StreamingOemWriter.NEW_LINE);
        }

        public void writeEphemerisLine(TimeStampedPVCoordinates pv) throws IOException {
            String epoch = StreamingOemWriter.dateToString(pv.getDate().getComponents(StreamingOemWriter.this.timeScale));
            StreamingOemWriter.this.writer.append(epoch).append(" ");
            StreamingOemWriter.this.writer.append(Double.toString(pv.getPosition().getX() * 0.001)).append(" ");
            StreamingOemWriter.this.writer.append(Double.toString(pv.getPosition().getY() * 0.001)).append(" ");
            StreamingOemWriter.this.writer.append(Double.toString(pv.getPosition().getZ() * 0.001)).append(" ");
            StreamingOemWriter.this.writer.append(Double.toString(pv.getVelocity().getX() * 0.001)).append(" ");
            StreamingOemWriter.this.writer.append(Double.toString(pv.getVelocity().getY() * 0.001)).append(" ");
            StreamingOemWriter.this.writer.append(Double.toString(pv.getVelocity().getZ() * 0.001));
            StreamingOemWriter.this.writer.append(StreamingOemWriter.NEW_LINE);
        }

        public void writeCovarianceMatrices(List<OEMFile.CovarianceMatrix> covarianceMatrices) throws IOException {
            StreamingOemWriter.this.writer.append("COVARIANCE_START").append(StreamingOemWriter.NEW_LINE);
            Collections.sort(covarianceMatrices, (mat1, mat2) -> mat1.getEpoch().compareTo(mat2.getEpoch()));
            for (OEMFile.CovarianceMatrix covarianceMatrix : covarianceMatrices) {
                String epoch = StreamingOemWriter.dateToString(covarianceMatrix.getEpoch().getComponents(StreamingOemWriter.this.timeScale));
                StreamingOemWriter.this.writeKeyValue(Keyword.EPOCH, epoch);
                if (covarianceMatrix.getFrame() != null) {
                    StreamingOemWriter.this.writeKeyValue(Keyword.COV_REF_FRAME, StreamingOemWriter.guessFrame(covarianceMatrix.getFrame()));
                } else if (covarianceMatrix.getLofType() != null) {
                    if (covarianceMatrix.getLofType() == LOFType.QSW) {
                        StreamingOemWriter.this.writeKeyValue(Keyword.COV_REF_FRAME, "RTN");
                    } else if (covarianceMatrix.getLofType() == LOFType.TNW) {
                        StreamingOemWriter.this.writeKeyValue(Keyword.COV_REF_FRAME, covarianceMatrix.getLofType().name());
                    } else {
                        throw new OrekitException((Localizable)OrekitMessages.CCSDS_INVALID_FRAME, this.toString());
                    }
                }
                RealMatrix covRealMatrix = covarianceMatrix.getMatrix();
                for (int i = 0; i < covRealMatrix.getRowDimension(); ++i) {
                    StreamingOemWriter.this.writer.append(Double.toString(covRealMatrix.getEntry(i, 0)));
                    for (int j = 1; j < i + 1; ++j) {
                        StreamingOemWriter.this.writer.append(" ").append(Double.toString(covRealMatrix.getEntry(i, j)));
                    }
                    StreamingOemWriter.this.writer.append(StreamingOemWriter.NEW_LINE);
                }
            }
            StreamingOemWriter.this.writer.append("COVARIANCE_STOP").append(StreamingOemWriter.NEW_LINE).append(StreamingOemWriter.NEW_LINE);
        }

        @Override
        public void init(SpacecraftState s0, AbsoluteDate t, double step) {
            try {
                String start = StreamingOemWriter.dateToString(s0.getDate().getComponents(StreamingOemWriter.this.timeScale));
                String stop = StreamingOemWriter.dateToString(t.getComponents(StreamingOemWriter.this.timeScale));
                this.metadata.putIfAbsent(Keyword.START_TIME, start);
                this.metadata.putIfAbsent(Keyword.STOP_TIME, stop);
                this.writeMetadata();
            }
            catch (IOException e) {
                throw new OrekitException(e, (Localizable)LocalizedCoreFormats.SIMPLE_MESSAGE, e.getLocalizedMessage());
            }
        }

        @Override
        public void handleStep(SpacecraftState s, boolean isLast) {
            try {
                this.writeEphemerisLine(s.getPVCoordinates(this.frame));
            }
            catch (IOException e) {
                throw new OrekitException(e, (Localizable)LocalizedCoreFormats.SIMPLE_MESSAGE, e.getLocalizedMessage());
            }
        }
    }
}

