/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.orbits;

import java.io.Serializable;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.hipparchus.analysis.differentiation.UnivariateDerivative1;
import org.hipparchus.analysis.interpolation.HermiteInterpolator;
import org.hipparchus.geometry.Vector;
import org.hipparchus.geometry.euclidean.threed.Vector3D;
import org.hipparchus.util.FastMath;
import org.hipparchus.util.MathUtils;
import org.orekit.annotation.DefaultDataContext;
import org.orekit.data.DataContext;
import org.orekit.errors.OrekitIllegalArgumentException;
import org.orekit.errors.OrekitInternalError;
import org.orekit.errors.OrekitMessages;
import org.orekit.frames.Frame;
import org.orekit.orbits.FieldCircularOrbit;
import org.orekit.orbits.Orbit;
import org.orekit.orbits.OrbitType;
import org.orekit.orbits.PositionAngle;
import org.orekit.time.AbsoluteDate;
import org.orekit.utils.PVCoordinates;
import org.orekit.utils.TimeStampedPVCoordinates;

public class CircularOrbit
extends Orbit {
    private static final long serialVersionUID = 20170414L;
    private final double a;
    private final double ex;
    private final double ey;
    private final double i;
    private final double raan;
    private final double alphaV;
    private final double aDot;
    private final double exDot;
    private final double eyDot;
    private final double iDot;
    private final double raanDot;
    private final double alphaVDot;
    private final boolean serializePV;
    private transient PVCoordinates partialPV;

    public CircularOrbit(double a, double ex, double ey, double i, double raan, double alpha, PositionAngle type, Frame frame, AbsoluteDate date, double mu) throws IllegalArgumentException {
        this(a, ex, ey, i, raan, alpha, Double.NaN, Double.NaN, Double.NaN, Double.NaN, Double.NaN, Double.NaN, type, frame, date, mu);
    }

    public CircularOrbit(double a, double ex, double ey, double i, double raan, double alpha, double aDot, double exDot, double eyDot, double iDot, double raanDot, double alphaDot, PositionAngle type, Frame frame, AbsoluteDate date, double mu) throws IllegalArgumentException {
        super(frame, date, mu);
        if (ex * ex + ey * ey >= 1.0) {
            throw new OrekitIllegalArgumentException(OrekitMessages.HYPERBOLIC_ORBIT_NOT_HANDLED_AS, this.getClass().getName());
        }
        this.a = a;
        this.aDot = aDot;
        this.ex = ex;
        this.exDot = exDot;
        this.ey = ey;
        this.eyDot = eyDot;
        this.i = i;
        this.iDot = iDot;
        this.raan = raan;
        this.raanDot = raanDot;
        if (this.hasDerivatives()) {
            UnivariateDerivative1 alphavUD;
            UnivariateDerivative1 exUD = new UnivariateDerivative1(ex, exDot);
            UnivariateDerivative1 eyUD = new UnivariateDerivative1(ey, eyDot);
            UnivariateDerivative1 alphaUD = new UnivariateDerivative1(alpha, alphaDot);
            switch (type) {
                case MEAN: {
                    alphavUD = FieldCircularOrbit.eccentricToTrue(FieldCircularOrbit.meanToEccentric(alphaUD, exUD, eyUD), exUD, eyUD);
                    break;
                }
                case ECCENTRIC: {
                    alphavUD = FieldCircularOrbit.eccentricToTrue(alphaUD, exUD, eyUD);
                    break;
                }
                case TRUE: {
                    alphavUD = alphaUD;
                    break;
                }
                default: {
                    throw new OrekitInternalError(null);
                }
            }
            this.alphaV = alphavUD.getValue();
            this.alphaVDot = alphavUD.getDerivative(1);
        } else {
            switch (type) {
                case MEAN: {
                    this.alphaV = CircularOrbit.eccentricToTrue(CircularOrbit.meanToEccentric(alpha, ex, ey), ex, ey);
                    break;
                }
                case ECCENTRIC: {
                    this.alphaV = CircularOrbit.eccentricToTrue(alpha, ex, ey);
                    break;
                }
                case TRUE: {
                    this.alphaV = alpha;
                    break;
                }
                default: {
                    throw new OrekitInternalError(null);
                }
            }
            this.alphaVDot = Double.NaN;
        }
        this.serializePV = false;
        this.partialPV = null;
    }

    private CircularOrbit(double a, double ex, double ey, double i, double raan, double alphaV, double aDot, double exDot, double eyDot, double iDot, double raanDot, double alphaVDot, TimeStampedPVCoordinates pvCoordinates, Frame frame, double mu) throws IllegalArgumentException {
        super(pvCoordinates, frame, mu);
        this.a = a;
        this.aDot = aDot;
        this.ex = ex;
        this.exDot = exDot;
        this.ey = ey;
        this.eyDot = eyDot;
        this.i = i;
        this.iDot = iDot;
        this.raan = raan;
        this.raanDot = raanDot;
        this.alphaV = alphaV;
        this.alphaVDot = alphaVDot;
        this.serializePV = true;
        this.partialPV = null;
    }

    public CircularOrbit(TimeStampedPVCoordinates pvCoordinates, Frame frame, double mu) throws IllegalArgumentException {
        super(pvCoordinates, frame, mu);
        Vector3D pvP = pvCoordinates.getPosition();
        Vector3D pvV = pvCoordinates.getVelocity();
        Vector3D pvA = pvCoordinates.getAcceleration();
        double r2 = pvP.getNormSq();
        double r = FastMath.sqrt((double)r2);
        double V2 = pvV.getNormSq();
        double rV2OnMu = r * V2 / mu;
        if (rV2OnMu > 2.0) {
            throw new OrekitIllegalArgumentException(OrekitMessages.HYPERBOLIC_ORBIT_NOT_HANDLED_AS, this.getClass().getName());
        }
        this.a = r / (2.0 - rV2OnMu);
        Vector3D momentum = pvCoordinates.getMomentum();
        this.i = Vector3D.angle((Vector3D)momentum, (Vector3D)Vector3D.PLUS_K);
        Vector3D node = Vector3D.crossProduct((Vector3D)Vector3D.PLUS_K, (Vector3D)momentum);
        this.raan = FastMath.atan2((double)node.getY(), (double)node.getX());
        double cosRaan = FastMath.cos((double)this.raan);
        double sinRaan = FastMath.sin((double)this.raan);
        double cosI = FastMath.cos((double)this.i);
        double sinI = FastMath.sin((double)this.i);
        double xP = pvP.getX();
        double yP = pvP.getY();
        double zP = pvP.getZ();
        double x2 = (xP * cosRaan + yP * sinRaan) / this.a;
        double y2 = ((yP * cosRaan - xP * sinRaan) * cosI + zP * sinI) / this.a;
        double eSE = Vector3D.dotProduct((Vector3D)pvP, (Vector3D)pvV) / FastMath.sqrt((double)(mu * this.a));
        double eCE = rV2OnMu - 1.0;
        double e2 = eCE * eCE + eSE * eSE;
        double f = eCE - e2;
        double g = FastMath.sqrt((double)(1.0 - e2)) * eSE;
        double aOnR = this.a / r;
        double a2OnR2 = aOnR * aOnR;
        this.ex = a2OnR2 * (f * x2 + g * y2);
        this.ey = a2OnR2 * (f * y2 - g * x2);
        double beta = 1.0 / (1.0 + FastMath.sqrt((double)(1.0 - this.ex * this.ex - this.ey * this.ey)));
        this.alphaV = CircularOrbit.eccentricToTrue(FastMath.atan2((double)(y2 + this.ey + eSE * beta * this.ex), (double)(x2 + this.ex - eSE * beta * this.ey)), this.ex, this.ey);
        this.partialPV = pvCoordinates;
        if (CircularOrbit.hasNonKeplerianAcceleration(pvCoordinates, mu)) {
            double[][] jacobian = new double[6][6];
            this.getJacobianWrtCartesian(PositionAngle.MEAN, jacobian);
            Vector3D keplerianAcceleration = new Vector3D(-mu / (r * r2), pvP);
            Vector3D nonKeplerianAcceleration = pvA.subtract((Vector)keplerianAcceleration);
            double aX = nonKeplerianAcceleration.getX();
            double aY = nonKeplerianAcceleration.getY();
            double aZ = nonKeplerianAcceleration.getZ();
            this.aDot = jacobian[0][3] * aX + jacobian[0][4] * aY + jacobian[0][5] * aZ;
            this.exDot = jacobian[1][3] * aX + jacobian[1][4] * aY + jacobian[1][5] * aZ;
            this.eyDot = jacobian[2][3] * aX + jacobian[2][4] * aY + jacobian[2][5] * aZ;
            this.iDot = jacobian[3][3] * aX + jacobian[3][4] * aY + jacobian[3][5] * aZ;
            this.raanDot = jacobian[4][3] * aX + jacobian[4][4] * aY + jacobian[4][5] * aZ;
            double alphaMDot = this.getKeplerianMeanMotion() + jacobian[5][3] * aX + jacobian[5][4] * aY + jacobian[5][5] * aZ;
            UnivariateDerivative1 exUD = new UnivariateDerivative1(this.ex, this.exDot);
            UnivariateDerivative1 eyUD = new UnivariateDerivative1(this.ey, this.eyDot);
            UnivariateDerivative1 alphaMUD = new UnivariateDerivative1(this.getAlphaM(), alphaMDot);
            UnivariateDerivative1 alphavUD = FieldCircularOrbit.eccentricToTrue(FieldCircularOrbit.meanToEccentric(alphaMUD, exUD, eyUD), exUD, eyUD);
            this.alphaVDot = alphavUD.getDerivative(1);
        } else {
            this.aDot = Double.NaN;
            this.exDot = Double.NaN;
            this.eyDot = Double.NaN;
            this.iDot = Double.NaN;
            this.raanDot = Double.NaN;
            this.alphaVDot = Double.NaN;
        }
        this.serializePV = true;
    }

    public CircularOrbit(PVCoordinates pvCoordinates, Frame frame, AbsoluteDate date, double mu) throws IllegalArgumentException {
        this(new TimeStampedPVCoordinates(date, pvCoordinates), frame, mu);
    }

    public CircularOrbit(Orbit op) {
        super(op.getFrame(), op.getDate(), op.getMu());
        this.a = op.getA();
        this.i = op.getI();
        double hx = op.getHx();
        double hy = op.getHy();
        double h2 = hx * hx + hy * hy;
        double h = FastMath.sqrt((double)h2);
        this.raan = FastMath.atan2((double)hy, (double)hx);
        double cosRaan = h == 0.0 ? FastMath.cos((double)this.raan) : hx / h;
        double sinRaan = h == 0.0 ? FastMath.sin((double)this.raan) : hy / h;
        double equiEx = op.getEquinoctialEx();
        double equiEy = op.getEquinoctialEy();
        this.ex = equiEx * cosRaan + equiEy * sinRaan;
        this.ey = equiEy * cosRaan - equiEx * sinRaan;
        this.alphaV = op.getLv() - this.raan;
        if (op.hasDerivatives()) {
            this.aDot = op.getADot();
            double hxDot = op.getHxDot();
            double hyDot = op.getHyDot();
            this.iDot = 2.0 * (cosRaan * hxDot + sinRaan * hyDot) / (1.0 + h2);
            this.raanDot = (hx * hyDot - hy * hxDot) / h2;
            double equiExDot = op.getEquinoctialExDot();
            double equiEyDot = op.getEquinoctialEyDot();
            this.exDot = (equiExDot + equiEy * this.raanDot) * cosRaan + (equiEyDot - equiEx * this.raanDot) * sinRaan;
            this.eyDot = (equiEyDot - equiEx * this.raanDot) * cosRaan - (equiExDot + equiEy * this.raanDot) * sinRaan;
            this.alphaVDot = op.getLvDot() - this.raanDot;
        } else {
            this.aDot = Double.NaN;
            this.exDot = Double.NaN;
            this.eyDot = Double.NaN;
            this.iDot = Double.NaN;
            this.raanDot = Double.NaN;
            this.alphaVDot = Double.NaN;
        }
        this.serializePV = false;
        this.partialPV = null;
    }

    @Override
    public OrbitType getType() {
        return OrbitType.CIRCULAR;
    }

    @Override
    public double getA() {
        return this.a;
    }

    @Override
    public double getADot() {
        return this.aDot;
    }

    @Override
    public double getEquinoctialEx() {
        double cosRaan = FastMath.cos((double)this.raan);
        double sinRaan = FastMath.sin((double)this.raan);
        return this.ex * cosRaan - this.ey * sinRaan;
    }

    @Override
    public double getEquinoctialExDot() {
        double cosRaan = FastMath.cos((double)this.raan);
        double sinRaan = FastMath.sin((double)this.raan);
        return (this.exDot - this.ey * this.raanDot) * cosRaan - (this.eyDot + this.ex * this.raanDot) * sinRaan;
    }

    @Override
    public double getEquinoctialEy() {
        double cosRaan = FastMath.cos((double)this.raan);
        double sinRaan = FastMath.sin((double)this.raan);
        return this.ey * cosRaan + this.ex * sinRaan;
    }

    @Override
    public double getEquinoctialEyDot() {
        double cosRaan = FastMath.cos((double)this.raan);
        double sinRaan = FastMath.sin((double)this.raan);
        return (this.eyDot + this.ex * this.raanDot) * cosRaan + (this.exDot - this.ey * this.raanDot) * sinRaan;
    }

    public double getCircularEx() {
        return this.ex;
    }

    public double getCircularExDot() {
        return this.exDot;
    }

    public double getCircularEy() {
        return this.ey;
    }

    public double getCircularEyDot() {
        return this.eyDot;
    }

    @Override
    public double getHx() {
        if (FastMath.abs((double)(this.i - Math.PI)) < 1.0E-10) {
            return Double.NaN;
        }
        return FastMath.cos((double)this.raan) * FastMath.tan((double)(this.i / 2.0));
    }

    @Override
    public double getHxDot() {
        if (FastMath.abs((double)(this.i - Math.PI)) < 1.0E-10) {
            return Double.NaN;
        }
        double cosRaan = FastMath.cos((double)this.raan);
        double sinRaan = FastMath.sin((double)this.raan);
        double tan = FastMath.tan((double)(0.5 * this.i));
        return 0.5 * cosRaan * (1.0 + tan * tan) * this.iDot - sinRaan * tan * this.raanDot;
    }

    @Override
    public double getHy() {
        if (FastMath.abs((double)(this.i - Math.PI)) < 1.0E-10) {
            return Double.NaN;
        }
        return FastMath.sin((double)this.raan) * FastMath.tan((double)(this.i / 2.0));
    }

    @Override
    public double getHyDot() {
        if (FastMath.abs((double)(this.i - Math.PI)) < 1.0E-10) {
            return Double.NaN;
        }
        double cosRaan = FastMath.cos((double)this.raan);
        double sinRaan = FastMath.sin((double)this.raan);
        double tan = FastMath.tan((double)(0.5 * this.i));
        return 0.5 * sinRaan * (1.0 + tan * tan) * this.iDot + cosRaan * tan * this.raanDot;
    }

    public double getAlphaV() {
        return this.alphaV;
    }

    public double getAlphaVDot() {
        return this.alphaVDot;
    }

    public double getAlphaE() {
        return CircularOrbit.trueToEccentric(this.alphaV, this.ex, this.ey);
    }

    public double getAlphaEDot() {
        UnivariateDerivative1 alphaVUD = new UnivariateDerivative1(this.alphaV, this.alphaVDot);
        UnivariateDerivative1 exUD = new UnivariateDerivative1(this.ex, this.exDot);
        UnivariateDerivative1 eyUD = new UnivariateDerivative1(this.ey, this.eyDot);
        UnivariateDerivative1 alphaEUD = FieldCircularOrbit.trueToEccentric(alphaVUD, exUD, eyUD);
        return alphaEUD.getDerivative(1);
    }

    public double getAlphaM() {
        return CircularOrbit.eccentricToMean(CircularOrbit.trueToEccentric(this.alphaV, this.ex, this.ey), this.ex, this.ey);
    }

    public double getAlphaMDot() {
        UnivariateDerivative1 alphaVUD = new UnivariateDerivative1(this.alphaV, this.alphaVDot);
        UnivariateDerivative1 exUD = new UnivariateDerivative1(this.ex, this.exDot);
        UnivariateDerivative1 eyUD = new UnivariateDerivative1(this.ey, this.eyDot);
        UnivariateDerivative1 alphaMUD = FieldCircularOrbit.eccentricToMean(FieldCircularOrbit.trueToEccentric(alphaVUD, exUD, eyUD), exUD, eyUD);
        return alphaMUD.getDerivative(1);
    }

    public double getAlpha(PositionAngle type) {
        return type == PositionAngle.MEAN ? this.getAlphaM() : (type == PositionAngle.ECCENTRIC ? this.getAlphaE() : this.getAlphaV());
    }

    public double getAlphaDot(PositionAngle type) {
        return type == PositionAngle.MEAN ? this.getAlphaMDot() : (type == PositionAngle.ECCENTRIC ? this.getAlphaEDot() : this.getAlphaVDot());
    }

    public static double eccentricToTrue(double alphaE, double ex, double ey) {
        double epsilon = FastMath.sqrt((double)(1.0 - ex * ex - ey * ey));
        double cosAlphaE = FastMath.cos((double)alphaE);
        double sinAlphaE = FastMath.sin((double)alphaE);
        return alphaE + 2.0 * FastMath.atan((double)((ex * sinAlphaE - ey * cosAlphaE) / (epsilon + 1.0 - ex * cosAlphaE - ey * sinAlphaE)));
    }

    public static double trueToEccentric(double alphaV, double ex, double ey) {
        double epsilon = FastMath.sqrt((double)(1.0 - ex * ex - ey * ey));
        double cosAlphaV = FastMath.cos((double)alphaV);
        double sinAlphaV = FastMath.sin((double)alphaV);
        return alphaV + 2.0 * FastMath.atan((double)((ey * cosAlphaV - ex * sinAlphaV) / (epsilon + 1.0 + ex * cosAlphaV + ey * sinAlphaV)));
    }

    public static double meanToEccentric(double alphaM, double ex, double ey) {
        double alphaE = alphaM;
        double shift = 0.0;
        double alphaEMalphaM = 0.0;
        double cosAlphaE = FastMath.cos((double)alphaE);
        double sinAlphaE = FastMath.sin((double)alphaE);
        int iter = 0;
        do {
            double f2 = ex * sinAlphaE - ey * cosAlphaE;
            double f1 = 1.0 - ex * cosAlphaE - ey * sinAlphaE;
            double f0 = alphaEMalphaM - f2;
            double f12 = 2.0 * f1;
            shift = f0 * f12 / (f1 * f12 - f0 * f2);
            alphaE = alphaM + (alphaEMalphaM -= shift);
            cosAlphaE = FastMath.cos((double)alphaE);
            sinAlphaE = FastMath.sin((double)alphaE);
        } while (++iter < 50 && FastMath.abs((double)shift) > 1.0E-12);
        return alphaE;
    }

    public static double eccentricToMean(double alphaE, double ex, double ey) {
        return alphaE + (ey * FastMath.cos((double)alphaE) - ex * FastMath.sin((double)alphaE));
    }

    @Override
    public double getE() {
        return FastMath.sqrt((double)(this.ex * this.ex + this.ey * this.ey));
    }

    @Override
    public double getEDot() {
        return (this.ex * this.exDot + this.ey * this.eyDot) / this.getE();
    }

    @Override
    public double getI() {
        return this.i;
    }

    @Override
    public double getIDot() {
        return this.iDot;
    }

    public double getRightAscensionOfAscendingNode() {
        return this.raan;
    }

    public double getRightAscensionOfAscendingNodeDot() {
        return this.raanDot;
    }

    @Override
    public double getLv() {
        return this.alphaV + this.raan;
    }

    @Override
    public double getLvDot() {
        return this.alphaVDot + this.raanDot;
    }

    @Override
    public double getLE() {
        return this.getAlphaE() + this.raan;
    }

    @Override
    public double getLEDot() {
        return this.getAlphaEDot() + this.raanDot;
    }

    @Override
    public double getLM() {
        return this.getAlphaM() + this.raan;
    }

    @Override
    public double getLMDot() {
        return this.getAlphaMDot() + this.raanDot;
    }

    private void computePVWithoutA() {
        if (this.partialPV != null) {
            return;
        }
        double equEx = this.getEquinoctialEx();
        double equEy = this.getEquinoctialEy();
        double hx = this.getHx();
        double hy = this.getHy();
        double lE = this.getLE();
        double hx2 = hx * hx;
        double hy2 = hy * hy;
        double factH = 1.0 / (1.0 + hx2 + hy2);
        double ux = (1.0 + hx2 - hy2) * factH;
        double uy = 2.0 * hx * hy * factH;
        double uz = -2.0 * hy * factH;
        double vx = uy;
        double vy = (1.0 - hx2 + hy2) * factH;
        double vz = 2.0 * hx * factH;
        double exey = equEx * equEy;
        double ex2 = equEx * equEx;
        double ey2 = equEy * equEy;
        double e2 = ex2 + ey2;
        double eta = 1.0 + FastMath.sqrt((double)(1.0 - e2));
        double beta = 1.0 / eta;
        double cLe = FastMath.cos((double)lE);
        double sLe = FastMath.sin((double)lE);
        double exCeyS = equEx * cLe + equEy * sLe;
        double x = this.a * ((1.0 - beta * ey2) * cLe + beta * exey * sLe - equEx);
        double y = this.a * ((1.0 - beta * ex2) * sLe + beta * exey * cLe - equEy);
        double factor = FastMath.sqrt((double)(this.getMu() / this.a)) / (1.0 - exCeyS);
        double xdot = factor * (-sLe + beta * equEy * exCeyS);
        double ydot = factor * (cLe - beta * equEx * exCeyS);
        Vector3D position = new Vector3D(x * ux + y * vx, x * uy + y * vy, x * uz + y * vz);
        Vector3D velocity = new Vector3D(xdot * ux + ydot * vx, xdot * uy + ydot * vy, xdot * uz + ydot * vz);
        this.partialPV = new PVCoordinates(position, velocity);
    }

    private Vector3D nonKeplerianAcceleration() {
        double[][] dCdP = new double[6][6];
        this.getJacobianWrtParameters(PositionAngle.MEAN, dCdP);
        double nonKeplerianMeanMotion = this.getAlphaMDot() - this.getKeplerianMeanMotion();
        double nonKeplerianAx = dCdP[3][0] * this.aDot + dCdP[3][1] * this.exDot + dCdP[3][2] * this.eyDot + dCdP[3][3] * this.iDot + dCdP[3][4] * this.raanDot + dCdP[3][5] * nonKeplerianMeanMotion;
        double nonKeplerianAy = dCdP[4][0] * this.aDot + dCdP[4][1] * this.exDot + dCdP[4][2] * this.eyDot + dCdP[4][3] * this.iDot + dCdP[4][4] * this.raanDot + dCdP[4][5] * nonKeplerianMeanMotion;
        double nonKeplerianAz = dCdP[5][0] * this.aDot + dCdP[5][1] * this.exDot + dCdP[5][2] * this.eyDot + dCdP[5][3] * this.iDot + dCdP[5][4] * this.raanDot + dCdP[5][5] * nonKeplerianMeanMotion;
        return new Vector3D(nonKeplerianAx, nonKeplerianAy, nonKeplerianAz);
    }

    @Override
    protected TimeStampedPVCoordinates initPVCoordinates() {
        this.computePVWithoutA();
        double r2 = this.partialPV.getPosition().getNormSq();
        Vector3D keplerianAcceleration = new Vector3D(-this.getMu() / (r2 * FastMath.sqrt((double)r2)), this.partialPV.getPosition());
        Vector3D acceleration = this.hasDerivatives() ? keplerianAcceleration.add((Vector)this.nonKeplerianAcceleration()) : keplerianAcceleration;
        return new TimeStampedPVCoordinates(this.getDate(), this.partialPV.getPosition(), this.partialPV.getVelocity(), acceleration);
    }

    @Override
    public CircularOrbit shiftedBy(double dt) {
        CircularOrbit keplerianShifted = new CircularOrbit(this.a, this.ex, this.ey, this.i, this.raan, this.getAlphaM() + this.getKeplerianMeanMotion() * dt, PositionAngle.MEAN, this.getFrame(), this.getDate().shiftedBy(dt), this.getMu());
        if (this.hasDerivatives()) {
            Vector3D nonKeplerianAcceleration = this.nonKeplerianAcceleration();
            keplerianShifted.computePVWithoutA();
            Vector3D fixedP = new Vector3D(1.0, keplerianShifted.partialPV.getPosition(), 0.5 * dt * dt, nonKeplerianAcceleration);
            double fixedR2 = fixedP.getNormSq();
            double fixedR = FastMath.sqrt((double)fixedR2);
            Vector3D fixedV = new Vector3D(1.0, keplerianShifted.partialPV.getVelocity(), dt, nonKeplerianAcceleration);
            Vector3D fixedA = new Vector3D(-this.getMu() / (fixedR2 * fixedR), keplerianShifted.partialPV.getPosition(), 1.0, nonKeplerianAcceleration);
            return new CircularOrbit(new TimeStampedPVCoordinates(keplerianShifted.getDate(), fixedP, fixedV, fixedA), keplerianShifted.getFrame(), keplerianShifted.getMu());
        }
        return keplerianShifted;
    }

    @Override
    public CircularOrbit interpolate(AbsoluteDate date, Stream<Orbit> sample) {
        List list = sample.collect(Collectors.toList());
        boolean useDerivatives = true;
        for (Orbit orbit : list) {
            useDerivatives = useDerivatives && orbit.hasDerivatives();
        }
        HermiteInterpolator interpolator = new HermiteInterpolator();
        AbsoluteDate previousDate = null;
        double previousRAAN = Double.NaN;
        double previousAlphaM = Double.NaN;
        for (Orbit orbit : list) {
            double continuousAlphaM;
            double continuousRAAN;
            CircularOrbit circ = (CircularOrbit)OrbitType.CIRCULAR.convertType(orbit);
            if (previousDate == null) {
                continuousRAAN = circ.getRightAscensionOfAscendingNode();
                continuousAlphaM = circ.getAlphaM();
            } else {
                double dt = circ.getDate().durationFrom(previousDate);
                double keplerAM = previousAlphaM + circ.getKeplerianMeanMotion() * dt;
                continuousRAAN = MathUtils.normalizeAngle((double)circ.getRightAscensionOfAscendingNode(), (double)previousRAAN);
                continuousAlphaM = MathUtils.normalizeAngle((double)circ.getAlphaM(), (double)keplerAM);
            }
            previousDate = circ.getDate();
            previousRAAN = continuousRAAN;
            previousAlphaM = continuousAlphaM;
            if (useDerivatives) {
                interpolator.addSamplePoint(circ.getDate().durationFrom(date), (double[][])new double[][]{{circ.getA(), circ.getCircularEx(), circ.getCircularEy(), circ.getI(), continuousRAAN, continuousAlphaM}, {circ.getADot(), circ.getCircularExDot(), circ.getCircularEyDot(), circ.getIDot(), circ.getRightAscensionOfAscendingNodeDot(), circ.getAlphaMDot()}});
                continue;
            }
            interpolator.addSamplePoint(circ.getDate().durationFrom(date), (double[][])new double[][]{{circ.getA(), circ.getCircularEx(), circ.getCircularEy(), circ.getI(), continuousRAAN, continuousAlphaM}});
        }
        double[][] interpolated = interpolator.derivatives(0.0, 1);
        return new CircularOrbit(interpolated[0][0], interpolated[0][1], interpolated[0][2], interpolated[0][3], interpolated[0][4], interpolated[0][5], interpolated[1][0], interpolated[1][1], interpolated[1][2], interpolated[1][3], interpolated[1][4], interpolated[1][5], PositionAngle.MEAN, this.getFrame(), date, this.getMu());
    }

    @Override
    protected double[][] computeJacobianMeanWrtCartesian() {
        double[][] jacobian = new double[6][6];
        this.computePVWithoutA();
        Vector3D position = this.partialPV.getPosition();
        Vector3D velocity = this.partialPV.getVelocity();
        double x = position.getX();
        double y = position.getY();
        double z = position.getZ();
        double vx = velocity.getX();
        double vy = velocity.getY();
        double vz = velocity.getZ();
        double pv = Vector3D.dotProduct((Vector3D)position, (Vector3D)velocity);
        double r2 = position.getNormSq();
        double r = FastMath.sqrt((double)r2);
        double v2 = velocity.getNormSq();
        double mu = this.getMu();
        double oOsqrtMuA = 1.0 / FastMath.sqrt((double)(mu * this.a));
        double rOa = r / this.a;
        double aOr = this.a / r;
        double aOr2 = this.a / r2;
        double a2 = this.a * this.a;
        double ex2 = this.ex * this.ex;
        double ey2 = this.ey * this.ey;
        double e2 = ex2 + ey2;
        double epsilon = FastMath.sqrt((double)(1.0 - e2));
        double beta = 1.0 / (1.0 + epsilon);
        double eCosE = 1.0 - rOa;
        double eSinE = pv * oOsqrtMuA;
        double cosI = FastMath.cos((double)this.i);
        double sinI = FastMath.sin((double)this.i);
        double cosRaan = FastMath.cos((double)this.raan);
        double sinRaan = FastMath.sin((double)this.raan);
        CircularOrbit.fillHalfRow(2.0 * aOr * aOr2, position, jacobian[0], 0);
        CircularOrbit.fillHalfRow(2.0 * a2 / mu, velocity, jacobian[0], 3);
        Vector3D danP = new Vector3D(v2, position, -pv, velocity);
        Vector3D danV = new Vector3D(r2, velocity, -pv, position);
        double recip = 1.0 / this.partialPV.getMomentum().getNorm();
        double recip2 = recip * recip;
        Vector3D dwXP = new Vector3D(recip, new Vector3D(0.0, vz, -vy), -recip2 * sinRaan * sinI, danP);
        Vector3D dwYP = new Vector3D(recip, new Vector3D(-vz, 0.0, vx), recip2 * cosRaan * sinI, danP);
        Vector3D dwZP = new Vector3D(recip, new Vector3D(vy, -vx, 0.0), -recip2 * cosI, danP);
        Vector3D dwXV = new Vector3D(recip, new Vector3D(0.0, -z, y), -recip2 * sinRaan * sinI, danV);
        Vector3D dwYV = new Vector3D(recip, new Vector3D(z, 0.0, -x), recip2 * cosRaan * sinI, danV);
        Vector3D dwZV = new Vector3D(recip, new Vector3D(-y, x, 0.0), -recip2 * cosI, danV);
        CircularOrbit.fillHalfRow(sinRaan * cosI, dwXP, -cosRaan * cosI, dwYP, -sinI, dwZP, jacobian[3], 0);
        CircularOrbit.fillHalfRow(sinRaan * cosI, dwXV, -cosRaan * cosI, dwYV, -sinI, dwZV, jacobian[3], 3);
        CircularOrbit.fillHalfRow(sinRaan / sinI, dwYP, cosRaan / sinI, dwXP, jacobian[4], 0);
        CircularOrbit.fillHalfRow(sinRaan / sinI, dwYV, cosRaan / sinI, dwXV, jacobian[4], 3);
        double u = x * cosRaan + y * sinRaan;
        double cv = -x * sinRaan + y * cosRaan;
        double v = cv * cosI + z * sinI;
        Vector3D duP = new Vector3D(cv * cosRaan / sinI, dwXP, cv * sinRaan / sinI, dwYP, 1.0, new Vector3D(cosRaan, sinRaan, 0.0));
        Vector3D duV = new Vector3D(cv * cosRaan / sinI, dwXV, cv * sinRaan / sinI, dwYV);
        Vector3D dvP = new Vector3D(-u * cosRaan * cosI / sinI + sinRaan * z, dwXP, -u * sinRaan * cosI / sinI - cosRaan * z, dwYP, cv, dwZP, 1.0, new Vector3D(-sinRaan * cosI, cosRaan * cosI, sinI));
        Vector3D dvV = new Vector3D(-u * cosRaan * cosI / sinI + sinRaan * z, dwXV, -u * sinRaan * cosI / sinI - cosRaan * z, dwYV, cv, dwZV);
        Vector3D dc1P = new Vector3D(aOr2 * (2.0 * eSinE * eSinE + 1.0 - eCosE) / r2, position, -2.0 * aOr2 * eSinE * oOsqrtMuA, velocity);
        Vector3D dc1V = new Vector3D(-2.0 * aOr2 * eSinE * oOsqrtMuA, position, 2.0 / mu, velocity);
        Vector3D dc2P = new Vector3D(aOr2 * eSinE * (eSinE * eSinE - (1.0 - e2)) / (r2 * epsilon), position, aOr2 * (1.0 - e2 - eSinE * eSinE) * oOsqrtMuA / epsilon, velocity);
        Vector3D dc2V = new Vector3D(aOr2 * (1.0 - e2 - eSinE * eSinE) * oOsqrtMuA / epsilon, position, eSinE / (mu * epsilon), velocity);
        double cof1 = aOr2 * (eCosE - e2);
        double cof2 = aOr2 * epsilon * eSinE;
        Vector3D dexP = new Vector3D(u, dc1P, v, dc2P, cof1, duP, cof2, dvP);
        Vector3D dexV = new Vector3D(u, dc1V, v, dc2V, cof1, duV, cof2, dvV);
        Vector3D deyP = new Vector3D(v, dc1P, -u, dc2P, cof1, dvP, -cof2, duP);
        Vector3D deyV = new Vector3D(v, dc1V, -u, dc2V, cof1, dvV, -cof2, duV);
        CircularOrbit.fillHalfRow(1.0, dexP, jacobian[1], 0);
        CircularOrbit.fillHalfRow(1.0, dexV, jacobian[1], 3);
        CircularOrbit.fillHalfRow(1.0, deyP, jacobian[2], 0);
        CircularOrbit.fillHalfRow(1.0, deyV, jacobian[2], 3);
        double cle = u / this.a + this.ex - eSinE * beta * this.ey;
        double sle = v / this.a + this.ey + eSinE * beta * this.ex;
        double m1 = beta * eCosE;
        double m2 = 1.0 - m1 * eCosE;
        double m3 = u * this.ey - v * this.ex + eSinE * beta * (u * this.ex + v * this.ey);
        double m4 = -sle + cle * eSinE * beta;
        double m5 = cle + sle * eSinE * beta;
        CircularOrbit.fillHalfRow((2.0 * m3 / r + aOr * eSinE + m1 * eSinE * (1.0 + m1 - (1.0 + aOr) * m2) / epsilon) / r2, position, (m1 * m2 / epsilon - 1.0) * oOsqrtMuA, velocity, m4, dexP, m5, deyP, -sle / this.a, duP, cle / this.a, dvP, jacobian[5], 0);
        CircularOrbit.fillHalfRow((m1 * m2 / epsilon - 1.0) * oOsqrtMuA, position, (2.0 * m3 + eSinE * this.a + m1 * eSinE * r * (eCosE * beta * 2.0 - aOr * m2) / epsilon) / mu, velocity, m4, dexV, m5, deyV, -sle / this.a, duV, cle / this.a, dvV, jacobian[5], 3);
        return jacobian;
    }

    @Override
    protected double[][] computeJacobianEccentricWrtCartesian() {
        double[][] jacobian = this.computeJacobianMeanWrtCartesian();
        double alphaE = this.getAlphaE();
        double cosAe = FastMath.cos((double)alphaE);
        double sinAe = FastMath.sin((double)alphaE);
        double aOr = 1.0 / (1.0 - this.ex * cosAe - this.ey * sinAe);
        double[] rowEx = jacobian[1];
        double[] rowEy = jacobian[2];
        double[] rowL = jacobian[5];
        for (int j = 0; j < 6; ++j) {
            rowL[j] = aOr * (rowL[j] + sinAe * rowEx[j] - cosAe * rowEy[j]);
        }
        return jacobian;
    }

    @Override
    protected double[][] computeJacobianTrueWrtCartesian() {
        double[][] jacobian = this.computeJacobianEccentricWrtCartesian();
        double alphaE = this.getAlphaE();
        double cosAe = FastMath.cos((double)alphaE);
        double sinAe = FastMath.sin((double)alphaE);
        double eSinE = this.ex * sinAe - this.ey * cosAe;
        double ecosE = this.ex * cosAe + this.ey * sinAe;
        double e2 = this.ex * this.ex + this.ey * this.ey;
        double epsilon = FastMath.sqrt((double)(1.0 - e2));
        double onePeps = 1.0 + epsilon;
        double d = onePeps - ecosE;
        double cT = (d * d + eSinE * eSinE) / 2.0;
        double cE = ecosE * onePeps - e2;
        double cX = this.ex * eSinE / epsilon - this.ey + sinAe * onePeps;
        double cY = this.ey * eSinE / epsilon + this.ex - cosAe * onePeps;
        double factorLe = (cT + cE) / cT;
        double factorEx = cX / cT;
        double factorEy = cY / cT;
        double[] rowEx = jacobian[1];
        double[] rowEy = jacobian[2];
        double[] rowA = jacobian[5];
        for (int j = 0; j < 6; ++j) {
            rowA[j] = factorLe * rowA[j] + factorEx * rowEx[j] + factorEy * rowEy[j];
        }
        return jacobian;
    }

    @Override
    public void addKeplerContribution(PositionAngle type, double gm, double[] pDot) {
        double n = FastMath.sqrt((double)(gm / this.a)) / this.a;
        switch (type) {
            case MEAN: {
                pDot[5] = pDot[5] + n;
                break;
            }
            case ECCENTRIC: {
                double oMe2 = 1.0 - this.ex * this.ex - this.ey * this.ey;
                double ksi = 1.0 + this.ex * FastMath.cos((double)this.alphaV) + this.ey * FastMath.sin((double)this.alphaV);
                pDot[5] = pDot[5] + n * ksi / oMe2;
                break;
            }
            case TRUE: {
                double oMe2 = 1.0 - this.ex * this.ex - this.ey * this.ey;
                double ksi = 1.0 + this.ex * FastMath.cos((double)this.alphaV) + this.ey * FastMath.sin((double)this.alphaV);
                pDot[5] = pDot[5] + n * ksi * ksi / (oMe2 * FastMath.sqrt((double)oMe2));
                break;
            }
            default: {
                throw new OrekitInternalError(null);
            }
        }
    }

    public String toString() {
        return new StringBuffer().append("circular parameters: ").append('{').append("a: ").append(this.a).append(", ex: ").append(this.ex).append(", ey: ").append(this.ey).append(", i: ").append(FastMath.toDegrees((double)this.i)).append(", raan: ").append(FastMath.toDegrees((double)this.raan)).append(", alphaV: ").append(FastMath.toDegrees((double)this.alphaV)).append(";}").toString();
    }

    @DefaultDataContext
    private Object writeReplace() {
        return new DTO(this);
    }

    @DefaultDataContext
    private static class DTO
    implements Serializable {
        private static final long serialVersionUID = 20170414L;
        private double[] d;
        private final Frame frame;

        private DTO(CircularOrbit orbit) {
            AbsoluteDate date = orbit.getDate();
            AbsoluteDate j2000Epoch = DataContext.getDefault().getTimeScales().getJ2000Epoch();
            double epoch = FastMath.floor((double)date.durationFrom(j2000Epoch));
            double offset = date.durationFrom(j2000Epoch.shiftedBy(epoch));
            if (orbit.serializePV) {
                TimeStampedPVCoordinates pv = orbit.getPVCoordinates();
                this.d = orbit.hasDerivatives() ? new double[]{epoch, offset, orbit.getMu(), orbit.a, orbit.ex, orbit.ey, orbit.i, orbit.raan, orbit.alphaV, orbit.aDot, orbit.exDot, orbit.eyDot, orbit.iDot, orbit.raanDot, orbit.alphaVDot, pv.getPosition().getX(), pv.getPosition().getY(), pv.getPosition().getZ(), pv.getVelocity().getX(), pv.getVelocity().getY(), pv.getVelocity().getZ(), pv.getAcceleration().getX(), pv.getAcceleration().getY(), pv.getAcceleration().getZ()} : new double[]{epoch, offset, orbit.getMu(), orbit.a, orbit.ex, orbit.ey, orbit.i, orbit.raan, orbit.alphaV, pv.getPosition().getX(), pv.getPosition().getY(), pv.getPosition().getZ(), pv.getVelocity().getX(), pv.getVelocity().getY(), pv.getVelocity().getZ(), pv.getAcceleration().getX(), pv.getAcceleration().getY(), pv.getAcceleration().getZ()};
            } else {
                this.d = orbit.hasDerivatives() ? new double[]{epoch, offset, orbit.getMu(), orbit.a, orbit.ex, orbit.ey, orbit.i, orbit.raan, orbit.alphaV, orbit.aDot, orbit.exDot, orbit.eyDot, orbit.iDot, orbit.raanDot, orbit.alphaVDot} : new double[]{epoch, offset, orbit.getMu(), orbit.a, orbit.ex, orbit.ey, orbit.i, orbit.raan, orbit.alphaV};
            }
            this.frame = orbit.getFrame();
        }

        private Object readResolve() {
            AbsoluteDate j2000Epoch = DataContext.getDefault().getTimeScales().getJ2000Epoch();
            switch (this.d.length) {
                case 24: {
                    return new CircularOrbit(this.d[3], this.d[4], this.d[5], this.d[6], this.d[7], this.d[8], this.d[9], this.d[10], this.d[11], this.d[12], this.d[13], this.d[14], new TimeStampedPVCoordinates(j2000Epoch.shiftedBy(this.d[0]).shiftedBy(this.d[1]), new Vector3D(this.d[15], this.d[16], this.d[17]), new Vector3D(this.d[18], this.d[19], this.d[20]), new Vector3D(this.d[21], this.d[22], this.d[23])), this.frame, this.d[2]);
                }
                case 18: {
                    return new CircularOrbit(this.d[3], this.d[4], this.d[5], this.d[6], this.d[7], this.d[8], Double.NaN, Double.NaN, Double.NaN, Double.NaN, Double.NaN, Double.NaN, new TimeStampedPVCoordinates(j2000Epoch.shiftedBy(this.d[0]).shiftedBy(this.d[1]), new Vector3D(this.d[9], this.d[10], this.d[11]), new Vector3D(this.d[12], this.d[13], this.d[14]), new Vector3D(this.d[15], this.d[16], this.d[17])), this.frame, this.d[2]);
                }
                case 15: {
                    return new CircularOrbit(this.d[3], this.d[4], this.d[5], this.d[6], this.d[7], this.d[8], this.d[9], this.d[10], this.d[11], this.d[12], this.d[13], this.d[14], PositionAngle.TRUE, this.frame, j2000Epoch.shiftedBy(this.d[0]).shiftedBy(this.d[1]), this.d[2]);
                }
            }
            return new CircularOrbit(this.d[3], this.d[4], this.d[5], this.d[6], this.d[7], this.d[8], PositionAngle.TRUE, this.frame, j2000Epoch.shiftedBy(this.d[0]).shiftedBy(this.d[1]), this.d[2]);
        }
    }
}

