/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.files.ccsds;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.hipparchus.exception.Localizable;
import org.hipparchus.util.FastMath;
import org.orekit.errors.OrekitException;
import org.orekit.errors.OrekitMessages;
import org.orekit.files.ccsds.CCSDSFrame;
import org.orekit.files.ccsds.CcsdsTimeScale;
import org.orekit.files.ccsds.CenterName;
import org.orekit.files.ccsds.KeyValue;
import org.orekit.files.ccsds.Keyword;
import org.orekit.files.ccsds.ODMFile;
import org.orekit.files.ccsds.ODMMetaData;
import org.orekit.files.ccsds.OGMFile;
import org.orekit.time.AbsoluteDate;
import org.orekit.time.TimeScale;
import org.orekit.time.TimeScalesFactory;
import org.orekit.utils.IERSConventions;

public abstract class ODMParser {
    private static final Pattern INTERNATIONAL_DESIGNATOR = Pattern.compile("(\\p{Digit}{4})-(\\p{Digit}{3})(\\p{Upper}{1,3})");
    private final AbsoluteDate missionReferenceDate;
    private final double mu;
    private final IERSConventions conventions;
    private final boolean simpleEOP;
    private int launchYear;
    private int launchNumber;
    private String launchPiece;

    protected ODMParser(AbsoluteDate missionReferenceDate, double mu, IERSConventions conventions, boolean simpleEOP, int launchYear, int launchNumber, String launchPiece) {
        this.missionReferenceDate = missionReferenceDate;
        this.mu = mu;
        this.conventions = conventions;
        this.simpleEOP = simpleEOP;
        this.launchYear = launchYear;
        this.launchNumber = launchNumber;
        this.launchPiece = launchPiece;
    }

    public abstract ODMParser withMissionReferenceDate(AbsoluteDate var1);

    public AbsoluteDate getMissionReferenceDate() {
        return this.missionReferenceDate;
    }

    public abstract ODMParser withMu(double var1);

    public double getMu() {
        return this.mu;
    }

    public abstract ODMParser withConventions(IERSConventions var1);

    public IERSConventions getConventions() {
        return this.conventions;
    }

    public abstract ODMParser withSimpleEOP(boolean var1);

    public boolean isSimpleEOP() {
        return this.simpleEOP;
    }

    public abstract ODMParser withInternationalDesignator(int var1, int var2, String var3);

    public int getLaunchYear() {
        return this.launchYear;
    }

    public int getLaunchNumber() {
        return this.launchNumber;
    }

    public String getLaunchPiece() {
        return this.launchPiece;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public ODMFile parse(String fileName) {
        try (FileInputStream stream = new FileInputStream(fileName);){
            ODMFile oDMFile = this.parse(stream, fileName);
            return oDMFile;
        }
        catch (IOException e) {
            throw new OrekitException((Localizable)OrekitMessages.UNABLE_TO_FIND_FILE, fileName);
        }
    }

    public ODMFile parse(InputStream stream) {
        return this.parse(stream, "<unknown>");
    }

    public abstract ODMFile parse(InputStream var1, String var2);

    protected boolean parseComment(KeyValue keyValue, List<String> comment) {
        if (keyValue.getKeyword() == Keyword.COMMENT) {
            comment.add(keyValue.getValue());
            return true;
        }
        return false;
    }

    protected boolean parseHeaderEntry(KeyValue keyValue, ODMFile odmFile, List<String> comment) {
        switch (keyValue.getKeyword()) {
            case CREATION_DATE: {
                if (!comment.isEmpty()) {
                    odmFile.setHeaderComment(comment);
                    comment.clear();
                }
                odmFile.setCreationDate(new AbsoluteDate(keyValue.getValue(), (TimeScale)TimeScalesFactory.getUTC()));
                return true;
            }
            case ORIGINATOR: {
                odmFile.setOriginator(keyValue.getValue());
                return true;
            }
        }
        return false;
    }

    protected boolean parseMetaDataEntry(KeyValue keyValue, ODMMetaData metaData, List<String> comment) {
        switch (keyValue.getKeyword()) {
            case OBJECT_NAME: {
                if (!comment.isEmpty()) {
                    metaData.setComment(comment);
                    comment.clear();
                }
                metaData.setObjectName(keyValue.getValue());
                return true;
            }
            case OBJECT_ID: {
                metaData.setObjectID(keyValue.getValue());
                Matcher matcher = INTERNATIONAL_DESIGNATOR.matcher(keyValue.getValue());
                if (matcher.matches()) {
                    metaData.setLaunchYear(Integer.parseInt(matcher.group(1)));
                    metaData.setLaunchNumber(Integer.parseInt(matcher.group(2)));
                    metaData.setLaunchPiece(matcher.group(3));
                }
                return true;
            }
            case CENTER_NAME: {
                metaData.setCenterName(keyValue.getValue());
                String canonicalValue = keyValue.getValue().equals("SOLAR SYSTEM BARYCENTER") || keyValue.getValue().equals("SSB") ? "SOLAR_SYSTEM_BARYCENTER" : (keyValue.getValue().equals("EARTH MOON BARYCENTER") || keyValue.getValue().equals("EARTH-MOON BARYCENTER") || keyValue.getValue().equals("EARTH BARYCENTER") || keyValue.getValue().equals("EMB") ? "EARTH_MOON" : keyValue.getValue());
                for (CenterName c : CenterName.values()) {
                    if (!c.name().equals(canonicalValue)) continue;
                    metaData.setHasCreatableBody(true);
                    metaData.setCenterBody(c.getCelestialBody());
                    metaData.getODMFile().setMuCreated(c.getCelestialBody().getGM());
                }
                return true;
            }
            case REF_FRAME: {
                metaData.setFrameString(keyValue.getValue());
                metaData.setRefFrame(this.parseCCSDSFrame(keyValue.getValue()).getFrame(this.getConventions(), this.isSimpleEOP()));
                return true;
            }
            case REF_FRAME_EPOCH: {
                metaData.setFrameEpochString(keyValue.getValue());
                return true;
            }
            case TIME_SYSTEM: {
                if (!CcsdsTimeScale.contains(keyValue.getValue())) {
                    throw new OrekitException((Localizable)OrekitMessages.CCSDS_TIME_SYSTEM_NOT_IMPLEMENTED, keyValue.getValue());
                }
                CcsdsTimeScale timeSystem = CcsdsTimeScale.valueOf(keyValue.getValue());
                metaData.setTimeSystem(timeSystem);
                if (metaData.getFrameEpochString() != null) {
                    metaData.setFrameEpoch(this.parseDate(metaData.getFrameEpochString(), timeSystem));
                }
                return true;
            }
        }
        return false;
    }

    protected boolean parseGeneralStateDataEntry(KeyValue keyValue, OGMFile general, List<String> comment) {
        switch (keyValue.getKeyword()) {
            case EPOCH: {
                general.setEpochComment(comment);
                comment.clear();
                general.setEpoch(this.parseDate(keyValue.getValue(), general.getMetaData().getTimeSystem()));
                return true;
            }
            case SEMI_MAJOR_AXIS: {
                general.setKeplerianElementsComment(comment);
                comment.clear();
                general.setA(keyValue.getDoubleValue() * 1000.0);
                general.setHasKeplerianElements(true);
                return true;
            }
            case ECCENTRICITY: {
                general.setE(keyValue.getDoubleValue());
                return true;
            }
            case INCLINATION: {
                general.setI(FastMath.toRadians((double)keyValue.getDoubleValue()));
                return true;
            }
            case RA_OF_ASC_NODE: {
                general.setRaan(FastMath.toRadians((double)keyValue.getDoubleValue()));
                return true;
            }
            case ARG_OF_PERICENTER: {
                general.setPa(FastMath.toRadians((double)keyValue.getDoubleValue()));
                return true;
            }
            case TRUE_ANOMALY: {
                general.setAnomalyType("TRUE");
                general.setAnomaly(FastMath.toRadians((double)keyValue.getDoubleValue()));
                return true;
            }
            case MEAN_ANOMALY: {
                general.setAnomalyType("MEAN");
                general.setAnomaly(FastMath.toRadians((double)keyValue.getDoubleValue()));
                return true;
            }
            case GM: {
                general.setMuParsed(keyValue.getDoubleValue() * 1.0E9);
                return true;
            }
            case MASS: {
                comment.addAll(0, general.getSpacecraftComment());
                general.setSpacecraftComment(comment);
                comment.clear();
                general.setMass(keyValue.getDoubleValue());
                return true;
            }
            case SOLAR_RAD_AREA: {
                comment.addAll(0, general.getSpacecraftComment());
                general.setSpacecraftComment(comment);
                comment.clear();
                general.setSolarRadArea(keyValue.getDoubleValue());
                return true;
            }
            case SOLAR_RAD_COEFF: {
                comment.addAll(0, general.getSpacecraftComment());
                general.setSpacecraftComment(comment);
                comment.clear();
                general.setSolarRadCoeff(keyValue.getDoubleValue());
                return true;
            }
            case DRAG_AREA: {
                comment.addAll(0, general.getSpacecraftComment());
                general.setSpacecraftComment(comment);
                comment.clear();
                general.setDragArea(keyValue.getDoubleValue());
                return true;
            }
            case DRAG_COEFF: {
                comment.addAll(0, general.getSpacecraftComment());
                general.setSpacecraftComment(comment);
                comment.clear();
                general.setDragCoeff(keyValue.getDoubleValue());
                return true;
            }
            case COV_REF_FRAME: {
                general.setCovarianceComment(comment);
                comment.clear();
                CCSDSFrame covFrame = this.parseCCSDSFrame(keyValue.getValue());
                if (covFrame.isLof()) {
                    general.setCovRefLofType(covFrame.getLofType());
                } else {
                    general.setCovRefFrame(covFrame.getFrame(this.getConventions(), this.isSimpleEOP()));
                }
                return true;
            }
            case CX_X: {
                general.createCovarianceMatrix();
                general.setCovarianceMatrixEntry(0, 0, keyValue.getDoubleValue() * 1000000.0);
                return true;
            }
            case CY_X: {
                general.setCovarianceMatrixEntry(0, 1, keyValue.getDoubleValue() * 1000000.0);
                return true;
            }
            case CY_Y: {
                general.setCovarianceMatrixEntry(1, 1, keyValue.getDoubleValue() * 1000000.0);
                return true;
            }
            case CZ_X: {
                general.setCovarianceMatrixEntry(0, 2, keyValue.getDoubleValue() * 1000000.0);
                return true;
            }
            case CZ_Y: {
                general.setCovarianceMatrixEntry(1, 2, keyValue.getDoubleValue() * 1000000.0);
                return true;
            }
            case CZ_Z: {
                general.setCovarianceMatrixEntry(2, 2, keyValue.getDoubleValue() * 1000000.0);
                return true;
            }
            case CX_DOT_X: {
                general.setCovarianceMatrixEntry(0, 3, keyValue.getDoubleValue() * 1000000.0);
                return true;
            }
            case CX_DOT_Y: {
                general.setCovarianceMatrixEntry(1, 3, keyValue.getDoubleValue() * 1000000.0);
                return true;
            }
            case CX_DOT_Z: {
                general.setCovarianceMatrixEntry(2, 3, keyValue.getDoubleValue() * 1000000.0);
                return true;
            }
            case CX_DOT_X_DOT: {
                general.setCovarianceMatrixEntry(3, 3, keyValue.getDoubleValue() * 1000000.0);
                return true;
            }
            case CY_DOT_X: {
                general.setCovarianceMatrixEntry(0, 4, keyValue.getDoubleValue() * 1000000.0);
                return true;
            }
            case CY_DOT_Y: {
                general.setCovarianceMatrixEntry(1, 4, keyValue.getDoubleValue() * 1000000.0);
                return true;
            }
            case CY_DOT_Z: {
                general.setCovarianceMatrixEntry(2, 4, keyValue.getDoubleValue() * 1000000.0);
                return true;
            }
            case CY_DOT_X_DOT: {
                general.setCovarianceMatrixEntry(3, 4, keyValue.getDoubleValue() * 1000000.0);
                return true;
            }
            case CY_DOT_Y_DOT: {
                general.setCovarianceMatrixEntry(4, 4, keyValue.getDoubleValue() * 1000000.0);
                return true;
            }
            case CZ_DOT_X: {
                general.setCovarianceMatrixEntry(0, 5, keyValue.getDoubleValue() * 1000000.0);
                return true;
            }
            case CZ_DOT_Y: {
                general.setCovarianceMatrixEntry(1, 5, keyValue.getDoubleValue() * 1000000.0);
                return true;
            }
            case CZ_DOT_Z: {
                general.setCovarianceMatrixEntry(2, 5, keyValue.getDoubleValue() * 1000000.0);
                return true;
            }
            case CZ_DOT_X_DOT: {
                general.setCovarianceMatrixEntry(3, 5, keyValue.getDoubleValue() * 1000000.0);
                return true;
            }
            case CZ_DOT_Y_DOT: {
                general.setCovarianceMatrixEntry(4, 5, keyValue.getDoubleValue() * 1000000.0);
                return true;
            }
            case CZ_DOT_Z_DOT: {
                general.setCovarianceMatrixEntry(5, 5, keyValue.getDoubleValue() * 1000000.0);
                return true;
            }
            case USER_DEFINED_X: {
                general.setUserDefinedParameters(keyValue.getKey(), keyValue.getValue());
                return true;
            }
        }
        return false;
    }

    protected CCSDSFrame parseCCSDSFrame(String frameName) {
        return CCSDSFrame.valueOf(frameName.replaceAll("-", ""));
    }

    protected AbsoluteDate parseDate(String date, CcsdsTimeScale timeSystem) {
        return timeSystem.parseDate(date, this.conventions, this.missionReferenceDate);
    }
}

