# -*- coding: utf-8 -*-
# vim: tabstop=4 shiftwidth=4 softtabstop=4
#
# Copyright (C) 2014-2017 GEM Foundation
#
# OpenQuake is free software: you can redistribute it and/or modify it
# under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenQuake is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with OpenQuake. If not, see <http://www.gnu.org/licenses/>.

import numpy

from openquake.baselib.general import AccumDict
from openquake.hazardlib.stats import compute_stats2
from openquake.calculators import base, classical_risk

F32 = numpy.float32

bcr_dt = numpy.dtype([('annual_loss_orig', F32), ('annual_loss_retro', F32),
                      ('bcr', F32)])


def classical_bcr(riskinput, riskmodel, param, monitor):
    """
    Compute and return the average losses for each asset.

    :param riskinput:
        a :class:`openquake.risklib.riskinput.RiskInput` object
    :param riskmodel:
        a :class:`openquake.risklib.riskinput.CompositeRiskModel` instance
    :param param:
        dictionary of extra parameters
    :param monitor:
        :class:`openquake.baselib.performance.Monitor` instance
    """
    R = riskinput.hazard_getter.num_rlzs
    result = AccumDict(accum=numpy.zeros((R, 3), F32))
    for outputs in riskmodel.gen_outputs(riskinput, monitor):
        assets = outputs.assets
        for out in outputs:
            for asset, (eal_orig, eal_retro, bcr) in zip(assets, out):
                aval = asset.value('structural')
                result[asset.ordinal][outputs.rlzi] = numpy.array([
                    eal_orig * aval, eal_retro * aval, bcr])
    return result


@base.calculators.add('classical_bcr')
class ClassicalBCRCalculator(classical_risk.ClassicalRiskCalculator):
    """
    Classical BCR Risk calculator
    """
    core_task = classical_bcr

    def post_execute(self, result):
        # NB: defined only for loss_type = 'structural'
        bcr_data = numpy.zeros((self.A, self.R), bcr_dt)
        for aid, data in result.items():
            bcr_data[aid]['annual_loss_orig'] = data[:, 0]
            bcr_data[aid]['annual_loss_retro'] = data[:, 1]
            bcr_data[aid]['bcr'] = data[:, 2]
        self.datastore['bcr-rlzs'] = bcr_data
        weights = [rlz.weight for rlz in self.rlzs_assoc.realizations]
        if len(weights) > 1:
            snames, sfuncs = zip(*self.oqparam.risk_stats())
            bcr_stats = numpy.zeros((self.A, len(sfuncs)), bcr_dt)
            bcr_stats = compute_stats2(bcr_data, sfuncs, weights)
            self.datastore['bcr-stats'] = bcr_stats
