# FILE: options.tcl
#
# A resource file for the OOMMF project.
#
# Configurable options of Oc_Classes may be set in this file via calls
# to [Oc_Option Add].
#
# Each block below defines the default value of one configurable option
# in OOMMF (the value which would prevail if this file didn't exist).
# Some blocks include alternatives, commented out.  You may edit this
# file to configure OOMMF to use different option values.  However,
# future OOMMF distributions may overwrite this file.  OOMMF also
# queries the file ./local/options.tcl for configuraton information, if
# it exists, and that file will not be overwritten by future
# distributions of OOMMF.  To make permanent configuration changes to
# OOMMF, copy this file to ./local/options.tcl and edit it.
#
# The signature for the Oc_Option Add command is
#
#    Oc_Option Add {app className option value}
#
# In particular, the import 'app' selects, via a case-insensitive glob
# match, which applications have access to that option. For example,
#
#   Oc_Option Add mmGraph Ow_GraphWin default_curve_width 2
#
# is only accessible inside the mmGraph application. Use '*' as the
# appname for options available to all applications, e.g.,
#
#   Oc_Option Add * Net_Host port 15136
#

########################################################################
# The port on which the OOMMF host service directory application
# listens for connections.  The default port number is 15136, but
# this may be overridden with the environment variable OOMMF_HOSTPORT.
global env
if {[info exists env(OOMMF_HOSTPORT)]} {
   Oc_Option Add * Net_Host port $env(OOMMF_HOSTPORT)
} else {
   Oc_Option Add * Net_Host port 15136
}

########################################################################
# Socket access options.  Net_Server setting controls user id checks
# on server sockets, Net_Link setting controls user id checks on
# client sockets.  Set to 0 to disable checks, 2 to force checks, and
# 1 to use checks if possible.
#
Oc_Option Add * Net_Server checkUserIdentities 0
Oc_Option Add * Net_Link checkUserIdentities 0

########################################################################
# The (partial) Tcl command used to launch a browser for HTML files.
#
# When an OOMMF application needs to display the HTML contents
# located by an URL, it invokes this command with an additional
# argument: the absolute URL locating the HTML to display.
# By default, this command invokes mmHelp, OOMMF's own HTML
# browsing application.  To use a different HTML browser,
# supply the appropriate partial Tcl command for launching
# the browser of your choice.  (If you choose Microsoft's
# Internet Explorer, see also the following configurable
# option.)
#
# OOMMF's native HTML browser, mmHelp:
Oc_Option Add * {} htmlBrowserCmd {Oc_Application Exec mmHelp}
#
# Netscape's Navigator/Communicator -- may need full path
#Oc_Option Add * {} htmlBrowserCmd \
#    {exec {C:\Program Files\Netscape\Communicator\Program\netscape.exe}}
#
# Microsoft's Internet Explorer -- may need full path.  Also
# if your release of Internet Explorer is less than 4.0,
# be sure to select MSIE compatibile file URLs below
# with "Oc_Option Add * Oc_Url msieCompatible 1".
#Oc_Option Add * {} htmlBrowserCmd \
#    {exec {C:\Program Files\Internet Explorer/iexplore.exe}}
########################################################################
# Formatting of file: URLs
#
# The file: URLs generated by the Oc_Class Oc_Url follow RFC 1738 and
# RFC 1808 by default, so that they look like
# file://localhost/path/to/file.html .  That format is not acceptable
# to releases of Microsoft's Internet Explorer before 4.0.  They
# insists on the format: file:/path/to/file.html .  Set the value
# below to 1 to enable compatibility with pre-4.0 releases of Microsoft
# Internet Explorer.
#
Oc_Option Add * Oc_Url msieCompatible 1
########################################################################
# Base font size for mmHelp
Oc_Option Add mmHelp Font baseSize 16
########################################################################
# Color database for Nb_GetColor
#
# List of Tcl files to source.  Each file should fill the
# nbcolordatabase array with rgb triplets, 8-bits per color.
# File names should be absolute paths.
Oc_Option Add * Color filename [list \
  [file join [file dirname [Oc_DirectPathname [info script]]] colors.config]]
########################################################################
# OOMMF color scheme: default (default), light, or dark
#Oc_Option Add * {} ColorScheme light
########################################################################
# Input filter programs for mmDisp and mmGraph
#
# The value for each is a list with an even number of elements.  The
# first element of each pair is a list of file name extensions that
# match files containing a certain data format.  The second element of
# each pair is a program that translates from that format into one
# which the OOMMF application can read.
#   The main option here is "decompress", used by the Nb_InputFilter
# class to launch programs like gzip and unzip to decompress files.  The
# ovf and odt options provide support for converting from other data
# formats to OVF (for mmDisp) and ODT (for mmGraph).  These are not
# enabled by default, but are supplied as hooks for end-user or third
# party use.  Files are sent first through the decompression filters, if
# applicable.
#   NOTE: These filters may be  used by a variety of applications (e.g.,
# mmDisp, avf2ppm, avf2ps), so the proper setting for the "app" field
# of the Oc_Option Add command is '*', not mmDisp, mmGraph, or similar.
Oc_Option Add * Nb_InputFilter decompress {{.gz .z .zip} {gzip -dc}}
Oc_Option Add * Nb_InputFilter ovf {}
Oc_Option Add * Nb_InputFilter odt {}
#
# Example extended decompress filter setting:
#   Oc_Option Add * Nb_InputFilter decompress \
#      { {.gz .z} {gzip -dc}   .bz2 bunzip2   .zip funzip }
#
#
########################################################################
# Default curve line widths in mmGraph.  This should be an non-negative
# integer.  Performace on Windows can be improved by setting this to 1.
global tcl_platform
if {[string match windows $tcl_platform(platform)]} {
   Oc_Option Add mmGraph Ow_GraphWin default_curve_width 1
} else {
   Oc_Option Add mmGraph Ow_GraphWin default_curve_width 2
}
#
########################################################################
# Default symbol frequency and size in mmGraph.  These should be
# non-negative integers.  To default to no symbols, set
# default_symbol_freq to 0.  Symbol size is in pixels.
Oc_Option Add mmGraph Ow_GraphWin default_symbol_freq  0
Oc_Option Add mmGraph Ow_GraphWin default_symbol_size 10
#
########################################################################
# Default curve coloring selection method in mmGraph; should be either
# "curve" or "segment".  If "curve", then each curve is a different
# color.  If "segment", then each segment of each curve is a different
# color.
Oc_Option Add mmGraph Ow_GraphWin default_color_selection curve
#
########################################################################
# Default canvas background coloring for mmGraph; in order to be
# consistent with mmGraph's configuration dialog, value should be either
# "white", "#042" (which is a dark green), or "auto", but any valid
# color is accepted.
Oc_Option Add mmGraph Ow_GraphWin default_canvas_color auto
#
########################################################################
# Icon settings.  owWindowIconType should be one of color, b/w, or none.
# owWindowIconSize should be either 16 or 32.  The default setting of 16
# selects the smaller 16x16 pixel icons; change this to 32 if you want
# the larger 32x32 pixel icons.
#
Oc_Option Add * {} owWindowIconType color
Oc_Option Add * {} owWindowIconSize 16
if {[string match windows $tcl_platform(platform)]} {
   # There is a Tk bug on Windows that swaps the red and blue color
   # planes in color icons.  Setting this option to 1 tells the OOMMF
   # icon code to work around this bug.  This bug is fixed in Tk 8.4.16.
   foreach {major minor pl} [split [info patchlevel] .] {break}
   if {$major>8 || ($major==8 && $minor>4) || \
          ($major==8 && $minor==4 && $pl>15)} {
      Oc_Option Add * {} owWindowIconSwapRB 0
   } else {
      Oc_Option Add * {} owWindowIconSwapRB 1
   }
} else {
   Oc_Option Add * {} owWindowIconSwapRB 0
}

########################################################################
# Scaling factor of Tk display. Adjust if necessary
#if {[llength [info commands tk]]==1} {
#   tk scaling 0.9
#}
#
########################################################################
# Windowing system window decoration width and maximum default title
# display width. If the settings are too small, then window titles will
# be ellipsized. If too generous, then windows may be much wider than
# needed. Tweak as necessary. (NB: As explained in the file header, it
# is best to create a ./local/options.tcl file and override these
# defaults there, rather than directly editing this file.)
if {![catch {package present Tk}]} {
   # If TkCaptionFont is not identical to the font used by the window
   # manager to set window titles, then pick a font as close as possible
   # and make minor adjustments via owTitleScaling. The created font
   # owTitleFont is used ONLY to estimate the width of the title text;
   # OOMMF does not set text with it.
   if {[lsearch -exact [font names] owTitleFont]<0} {
      font create owTitleFont {*}[font configure TkCaptionFont]
   }
   set owTitleScaling 1.0     ;# Relative scaling
   # owDecorationWidth is the width of the minimize, maximize, close,
   # and any other title bar decorations, in pixels. It is usually
   # best to pad this by a few pixels to protect against round off.
   set owDecorationWidth 128
   global tcl_platform env
   if {[Oc_IsCygwin64Platform] || [Oc_IsCygwinPlatform]} {
      font configure owTitleFont -family {Segoe UI} -size 9 -weight normal
      set owTitleScaling 1.007
   } elseif {[string match windows $tcl_platform(platform)]} {
      # font configure owTitleFont -family {Segoe UI} -size 9 -weight normal
      ## This should match TkCaptionFont
   } elseif {[string match darwin $tcl_platform(platform)]} {
      # font configure owTitleFont -family {.SF NS Display} -size 15
      ## The macOS font varies with version, but TkCaptionFont is
      ## usually a pretty good match. You can adjust this if necessary,
      ## however. The San Francisco (.SF) typeface is used in El Capitan
      ## and later.  Older macOS versions used Helvetica Neue (Yosemite)
      ## and Lucida Grande.
      set owDecorationWidth 80
   } else { # Assume unix
      font configure owTitleFont {*}[font actual TkHeadingFont]
      ## For some reason on X11 TkHeadingFont tends to be a closer
      ## match to the system font than TkCaptionFont. YMMV.
      font configure owTitleFont -size 9
      set owTitleScaling 0.99
      # Different desktops provide different decorations. Gnome
      # in particular provides a very minimal set.
      if {[info exists env(XDG_CURRENT_DESKTOP)] && \
           [string compare -nocase gnome $env(XDG_CURRENT_DESKTOP)]==0} {
         set owDecorationWidth 28
      } else {
         set owDecorationWidth 98
      }
   }
   set owDecorationWidth [expr {int(ceil($owDecorationWidth*[tk scaling]))}]
   Oc_Option Add * {} owDecorationWidth $owDecorationWidth
   Oc_Option Add * {} owTitleScaling $owTitleScaling
   unset owDecorationWidth owTitleScaling
   Oc_Option Add * {} owMaxTitleWidth 25   ;# Units are "M" width in
                                           ## owTitleFont.
}

########################################################################
# Control "watch" cursor display in Ow library routines
# Performace on Windows can be improved by setting this to 1.
# Oc_Option Add * {} owDisableWatchCursor 0
#
########################################################################
# Show (1) or hide (0) console option on app menus.
Oc_Option Add * Menu show_console_option 1
#
########################################################################
# Directory for Oxsii and Boxsi log/error files.  Default is the OOMMF
# root directory.  (File names are oxsii.errors and boxsi.errors,
# respectively.)
if {[info exists env(OOMMF_OXSLOGDIR)]} {
   Oc_Option Add * OxsLogs directory $env(OOMMF_OXSLOGDIR)
} else {
   Oc_Option Add * OxsLogs directory [Oc_Main GetOOMMFRootDir]
}
#
########################################################################
# Slave MIF interpreter control in Oxs
#
# Value should be one of safe, custom or unsafe
Oc_Option Add * MIFinterp safety custom
#
########################################################################
# Number of threads to run (per process), for thread-enabled builds.
# Usually, this is set in the applicable oommf/config/platform/ file,
# but that value may be overridden here.  Additionally, the value
# here may be overridden by the OOMMF_THREADS environment variable,
# or may be set on the command line of thread-aware apps (e.g., oxsii
# and boxsi).
#   In most instances, this line should be left commented out, so that
# the generally more reliable oommf/config/platform setting is retained.
# Oc_Option Add * Threads oommf_thread_count 2
#
# If set, oommf_thread_limit sets a hard limit on the number of threads
# per process.  If unset, then there is no limit.  In parallel with
# oommf_thread_count, there are also overrides via oommf/config/platform
# settings and the OOMMF_THREADLIMIT environment variable.
# Oc_Option Add * Threads oommf_thread_limit 8
#
########################################################################
# Default numanodes setting, for numa-enabled builds.
# Oc_Option Add * Numa numanodes auto
#
########################################################################
# Platform-generic default flags for compiling
# To enable compiler warnings, add '-warn 1'
# To enable debugger support, add '-debug 1'
# To disable some slow runtime error checks, add '-def NDEBUG'
Oc_Option Add * Platform cflags {-def NDEBUG}
#Oc_Option Add * Platform cflags {-warn 1 -debug 1}
#
########################################################################
# Platform-generic default flags for linking
Oc_Option Add * Platform lflags {}
########################################################################
# Whether Oc_Classes should enforce the 'const' keyword
Oc_Option Add * Oc_Class enforceConst 0
########################################################################
# Whether Oc_Classes should enforce the 'private' keyword
Oc_Option Add * Oc_Class enforcePrivate 0
########################################################################
########################################################################
# Evalute the file ./local/options.tcl if it exists.

set fn [file join [file dirname [Oc_DirectPathname [info script]]] local \
        options.tcl]
if {[file readable $fn]} {
    if {[catch {source $fn} msg]} {
        global errorInfo errorCode
	set msg [join [split $msg \n] \n\t]
	error "Error sourcing local options file:\n    $fn:\n\t$msg" \
		$errorInfo $errorCode
    }
}
