#/usr/bin/env python
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function
from __future__ import unicode_literals

import argparse
import os

from onnx import defs
from onnx.defs import OpSchema


def display_number(v):
    if defs.OpSchema.is_infinite(v):
        return '&#8734;'
    return str(v)


def support_level_str(level):
    return \
        "<sub>experimental</sub> " if level == OpSchema.SupportType.EXPERIMENTAL else ""


def main(args):
    args.output.write('## Operator Schemas\n')
    args.output.write(
        """*This file is automatically generated from the
            [def files](/onnx/defs) via [this script](/onnx/defs/gen_doc.py).
            Do not modify directly and instead edit operator definitions.*\n""")

    sorted_ops = sorted(
        (int(schema.support_level), op_type, schema)
        for (op_type, schema) in defs.get_all_schemas().items())

    args.output.write('\n')

    # Table of contents
    for _, op_type, schema in sorted_ops:
        s = '* <a href="#{}">{}{}</a>\n'.format(
            op_type, support_level_str(schema.support_level), op_type)
        args.output.write(s)

    args.output.write('\n')

    for _, op_type, schema in sorted_ops:
        # op_type
        s = '### <a name="{}"></a><a name="{}">**{}{}**</a>\n'.format(
            op_type, op_type.lower(), support_level_str(schema.support_level),
            op_type)

        # doc
        if schema.doc:
            s += '\n'
            s += '\n'.join('  ' + line
                           for line in schema.doc.lstrip().splitlines())
            s += '\n'

        # attributes
        if schema.attributes:
            s += '  * **attribute**:\n'
            s += '    <dl>\n'
            for _, attr in sorted(schema.attributes.items()):
                s += '      <dt>{}</dt>\n'.format(attr.name)
                s += '      <dd>{}</dd>\n'.format(attr.description)
            s += '    </dl>\n'


        # inputs
        s += '  * **input**:'
        if schema.min_input != schema.max_input:
            s += '{} - {}'.format(display_number(schema.min_input),
                                  display_number(schema.max_input))
        s += '\n'
        if schema.input_desc:
            s += '    <dl>\n'
            for input_name, input_desc in schema.input_desc:
                s += '      <dt>{}</dt>\n'.format(input_name)
                s += '      <dd>{}</dd>\n'.format(input_desc)
            s += '    </dl>\n'

        # outputs
        s += '  * **output**:'
        if schema.min_output != schema.max_output:
            s += '{} - {}'.format(display_number(schema.min_output),
                                  display_number(schema.max_output))
        s += '\n'
        if schema.output_desc:
            s += '    <dl>\n'
            for output_name, output_desc in schema.output_desc:
                s += '      <dt>{}</dt>\n'.format(output_name)
                s += '      <dd>{}</dd>\n'.format(output_desc)
            s += '    </dl>\n'

        s += '\n\n'
        args.output.write(s)


if __name__ == '__main__':
    parser = argparse.ArgumentParser('gen_doc')
    parser.add_argument('-o', '--output', type=argparse.FileType('w'),
                        default=os.path.join(
                            os.path.dirname(os.path.dirname(os.path.dirname(os.path.realpath(__file__)))),
                            'docs',
                            'Operators.md'),
                        help='output path (default: %(default)s)')
    main(parser.parse_args())
