import csv
from csv import excel
import os
from collections import OrderedDict
import logging

import fiona
from shapely.geometry.geo import mapping

from ocgis import constants
from ocgis import env
from ocgis.conv.base import AbstractTabularConverter
from ocgis.util.logging_ocgis import ocgis_lh


class OcgDialect(excel):
    lineterminator = '\n'


class CsvConverter(AbstractTabularConverter):
    _ext = 'csv'

    def _build_(self, coll):
        headers = self.get_headers(coll)
        f = open(self.path, 'w')
        writer = csv.DictWriter(f, headers, dialect=OcgDialect)
        writer.writeheader()
        ret = {'file_object': f, 'csv_writer': writer}
        return ret

    def _write_coll_(self, f, coll):
        writer = f['csv_writer']

        for geom, row in self.get_iter_from_spatial_collection(coll):
            writer.writerow(row)

    def _finalize_(self, f):
        for fobj in f.itervalues():
            try:
                fobj.close()
            except:
                pass


class CsvShapefileConverter(CsvConverter):
    _add_ugeom = True

    def __init__(self, *args, **kwargs):
        CsvConverter.__init__(self, *args, **kwargs)
        if self.ops is None:
            raise ValueError('The argument "ops" may not be "None".')

    @property
    def geom_uid(self):
        geom_uid = self.ops.geom_uid or env.DEFAULT_GEOM_UID
        return geom_uid

    def _build_(self, coll):
        ret = CsvConverter._build_(self, coll)

        self._ugid_gid_store = {}

        if not self.ops.aggregate:
            fiona_path = os.path.join(self._get_or_create_shp_folder_(), self.prefix + '_gid.shp')
            archetype_field = coll._archetype_field

            try:
                fiona_crs = archetype_field.spatial.crs.value
            except AttributeError:
                if archetype_field.spatial.crs is None:
                    raise ValueError('"crs" is None. A coordinate systems is required for writing to Fiona output.')
                else:
                    raise

            fiona_schema = {'geometry': archetype_field.spatial.abstraction_geometry.geom_type,
                            'properties': OrderedDict([[constants.HEADERS.ID_DATASET.upper(), 'int'],
                                                       [self.geom_uid, 'int'],
                                                       [constants.HEADERS.ID_GEOMETRY.upper(), 'int']])}
            fiona_object = fiona.open(fiona_path, 'w', driver='ESRI Shapefile', crs=fiona_crs, schema=fiona_schema)
        else:
            ocgis_lh('creating a UGID-GID shapefile is not necessary for aggregated data. use UGID shapefile.',
                     'conv.csv-shp',
                     logging.WARN)
            fiona_object = None

        ret.update({'fiona_object': fiona_object})

        return ret

    def _write_coll_(self, f, coll):
        writer = f['csv_writer']
        file_fiona = f['fiona_object']
        rstore = self._ugid_gid_store
        is_aggregated = self.ops.aggregate

        for geom, row in self.get_iter_from_spatial_collection(coll):
            writer.writerow(row)

        if not is_aggregated:
            for ugid, field_dict in coll.iteritems():
                for field in field_dict.itervalues():
                    did = field.uid
                    for _, _, geom, gid in field.spatial.get_geom_iter():
                        try:
                            if gid in rstore[did][ugid]:
                                continue
                            else:
                                raise KeyError
                        except KeyError:
                            if did not in rstore:
                                rstore[did] = {}
                            if ugid not in rstore[did]:
                                rstore[did][ugid] = []
                            if gid not in rstore[did][ugid]:
                                rstore[did][ugid].append(gid)

                            # for multivariate calculation outputs the dataset identifier is None.
                            try:
                                converted_did = int(did)
                            except TypeError:
                                converted_did = None

                            feature = {'properties': {constants.HEADERS.ID_GEOMETRY.upper(): int(gid),
                                                      self.geom_uid: int(ugid),
                                                      constants.HEADERS.ID_DATASET.upper(): converted_did},
                                       'geometry': mapping(geom)}
                            try:
                                file_fiona.write(feature)
                            except ValueError as e:
                                if feature['geometry']['type'] != file_fiona.meta['schema']['geometry']:
                                    msg = 'Spatial abstractions do not match. You may need to override "abstraction" and/or "s_abstraction"'
                                    msg = '{0}. Original error message from Fiona is "ValueError({1})".'.format(msg,
                                                                                                                e.message)
                                    raise ValueError(msg)
                                else:
                                    raise
