import numpy as np

from ocgis.calc import base


class HeatIndex(base.AbstractMultivariateFunction):
    description = 'Heat Index following: http://en.wikipedia.org/wiki/Heat_index. If temperature is < 80F or relative humidity is < 40%, the value is masked during calculation. Output units are Fahrenheit.'
    required_variables = ['tas', 'rhs']
    required_units = {'tas': 'fahrenheit', 'rhs': 'percent'}
    key = 'heat_index'
    units = None
    standard_name = 'heat_index'
    long_name = 'Heat Index'

    def calculate(self, tas=None, rhs=None):
        c1 = -42.379
        c2 = 2.04901523
        c3 = 10.14333127
        c4 = -0.22475541
        c5 = -6.83783e-3
        c6 = -5.481717e-2
        c7 = 1.22874e-3
        c8 = 8.5282e-4
        c9 = -1.99e-6

        idx = tas < 80
        tas.mask = np.logical_or(idx, tas.mask)
        idx = rhs < 40
        rhs.mask = np.logical_or(idx, rhs.mask)

        tas_sq = np.square(tas)
        rhs_sq = np.square(rhs)

        hi = c1 + c2 * tas + c3 * rhs + c4 * tas * rhs + c5 * tas_sq + c6 * rhs_sq + c7 * tas_sq * rhs + c8 * tas * rhs_sq + c9 * tas_sq * rhs_sq

        return hi
